# from falcon_rest.db import fields, orm
# from falcon_rest import resources
import falcon_rest.schemes
from falcon_rest.logger import log
from marshmallow_sqlalchemy import ModelSchemaOpts
from falcon_rest.schemes import (CHOICES_VALIDATION_ERR_MESSAGE, BaseModelScheme as _BaseModelScheme, SuccessScheme)
from falcon_rest.responses import ObjectCreatedResponse
from falcon_rest.schemes import ObjectCreatedUuidAsPk, ObjectCreatedCompositeOrStrPk
from falcon_rest.db import get_db
from sqlalchemy import inspect, and_
from marshmallow import (
    Schema, pre_load, pre_dump, post_dump, post_load, validates_schema,
    validate, validates, fields, ValidationError
)
from marshmallow.fields import (
    Field, Raw, Nested, Dict, List, String, UUID, Number, Integer, Decimal, Boolean,
    FormattedString, Float, DateTime, LocalDateTime, Time, Date, TimeDelta, Url, URL,
    Email, Method, Function, Str, Bool, Int, Constant)
from api_dnl.base_model import DnlApiBaseModel
from api_dnl import model
from api_dnl.fields import Choice
from api_dnl.schemes.common import *
from datetime import datetime, timedelta
from dateutil.parser import parse as parse_datetime
from pytz import UTC
from email.utils import parseaddr
from api_dnl.utils.statisticapi2 import FIELD_MAP
import json

FQDN_REGEXP = "^((?=[a-z0-9-]{1,63}\.)(xn--)?[a-z0-9]+(-[a-z0-9]+)*\.)+[a-z]{2,63}$"
IP_REGEXP_DOMAIN = "^([a-zA-Z0-9]|[a-zA-Z0-9][a-zA-Z0-9\-]{0,61}[a-zA-Z0-9])" \
                   "(\.([a-zA-Z0-9]|[a-zA-Z0-9][a-zA-Z0-9\-]{0,61}[a-zA-Z0-9]))*$"
# IP_REGEXP = "^((25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\\.){3}(25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)|localhost$"
IP_REGEXP = "^((25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\\.){3}(25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)|localhost$"
DATE_REGEXP = r'^(0?[1-9]|1[012])/(0?[1-9]|[1-2][0-9]|3[01])/[12][09][0-9][0-9]$|' \
              r'^[12][09][0-9][0-9]-(0?[1-9]|1[012])-(0?[1-9]|[1-2][0-9]|3[01])$|' \
              r'^(0?[1-9]|1[012])-[12][09][0-9][0-9]-(0?[1-9]|[1-2][0-9]|3[01])$|' \
              r'^(0?[1-9]|1[012])/[12][09][0-9][0-9]/(0?[1-9]|[1-2][0-9]|3[01])$|' \
              r'^[12][09][0-9][0-9]/(0?[1-9]|1[012])/(0?[1-9]|[1-2][0-9]|3[01])$'
TIME_REGEXP = r'(24:00|2[0-3]:[0-5][0-9]|[0-1][0-9]:[0-5][0-9])'
TIMEZONE_REGEXP = r'^(?:Z|[+-](?:2[0-3]|[01][0-9]):[0-5][0-9])$'
PHONE_REGEXP = r'^\+?[0-9]+$'
DIGITS_REGEXP = r'^[0-9]*$'
PREFIX_REGEXP = r'^[A-Za-z0-9#\*\+]*$'
PREFIXES_REGEXP = r'^([A-Za-z0-9#*+]*)(,([A-Za-z0-9#*+]*))*$'
NAME_REGEXP = "^([\w _\-\.\/]?)*$"
USERNAME_REGEXP = "^(\w[ _\-\.\@]?)*$"
PASS_REGEXP = r"^[A-Za-z\d.!@$#%^&(){}\[\]:;<>,*.?/~_+-=|\\]{1,}$"  # r"^(?=.*[A-Za-z])(?=.*\d)[A-Za-z\d]{6,}$"


def valid_date(d):
    try:
        if len(str(d)) != 10:
            raise ValidationError('Invalid date "{}" must be in format "YYYY-MM-DD"'.format(d))
        parse_datetime(str(d))
        return True
    except ValueError as e:
        raise ValidationError('Invalid date {}: {}'.format(d, str(e)))


def valid_future_date(d):
    try:
        # if len(str(d)) != 10:
        #     raise ValidationError('Invalid date "{}" must be in format "YYYY-MM-DD"'.format(d))
        parse_datetime(str(d))
        if d < datetime.now(UTC):
            raise ValidationError('Invalid date "{}" must be in the future'.format(d))
        return True
    except ValueError:
        raise ValidationError('Invalid date {}'.format(d))


def _emails_validate(value):
    if ',' in value:
        value = value.replace(',', ';')
    lst = value.split(';')
    for addr in lst:
        if addr == '':
            continue
        name, email = parseaddr(addr)
        validate.Email(error='invalid emails')(email)


class Emails(String):
    def __init__(self, *args, **kwargs):
        String.__init__(self, *args, **kwargs)
        # Insert validation into self.validators so that multiple errors can be
        # stored.
        self.validators.insert(0, _emails_validate)

    def _validated(self, value):
        """Format the value or raise a :exc:`ValidationError` if an error occurs."""
        if value is None:
            return None
        _emails_validate(value)
        return value


def valid_dict(d):
    return validate.OneOf(choices=d.values(), labels=d.keys(), error=CHOICES_VALIDATION_ERR_MESSAGE)


Schemes = []


def plain(data, key):
    if key in data:
        temp = data[key]
        del data[key]
        data.update(temp)
    return data


# validate any model field
def _valid(cls_name, field, value, msg=''):
    cls = model.__dict__[cls_name]
    if type(value) == type('') or (not getattr(value, 'urn', None) is None and 'uuid' in getattr(value, 'urn', None)):
        value = "'" + str(value) + "'"
    if cls.filter(model.text_('{}={}'.format(field, value))).count() == 0:
        if msg:
            raise ValidationError(msg)
        else:
            raise ValidationError('{} {} is invalid (not exists)'.format(field, value))


def _valid_unique(cls_name, field, value, msg=''):
    cls = model.__dict__[cls_name]
    if type(value) == type('') or (not getattr(value, 'urn', None) is None and 'uuid' in getattr(value, 'urn', None)):
        value = "'" + value + "'"
    if cls.filter(model.text_('{}={}'.format(field, value))).count() > 0:
        if msg:
            raise ValidationError(msg)
        else:
            raise ValidationError('{} {} already exist (duplicate) '.format(field, value))


def _valid_unique2(cls_name, field, value, msg=''):
    from api_dnl.resources import AlreadyExists
    cls = model.__dict__[cls_name]
    if type(value) == type(''):
        value = "'" + value + "'"
    if cls.filter(model.text_('{}={}'.format(field, value))).count() > 0:
        if not msg:
            msg = '{} {} already exist (duplicate) '.format(field, value)
        raise AlreadyExists(msg)


#
from marshmallow_sqlalchemy import convert


class MyModelConverter(convert.ModelConverter):
    def _get_field_class_for_property(self, prop):
        if hasattr(prop, 'direction'):
            field_cls = convert.Related
        elif hasattr(prop, 'columns'):
            column = prop.columns[0]
            field_cls = self._get_field_class_for_column(column)
        elif hasattr(prop, '_proxied_property'):
            prop1 = prop._proxied_property
            if hasattr(prop1, 'direction'):
                field_cls = convert.Related
            elif hasattr(prop1, 'columns'):
                column = prop1.columns[0]
                field_cls = self._get_field_class_for_column(column)
            else:
                raise convert.ModelConversionError(
                    'Property and proxy field not found! {}:{}'.format(str(prop), prop.name))
        return field_cls

    def _add_column_kwargs(self, kwargs, column):
        try:
            super(MyModelConverter, self)._add_column_kwargs(kwargs, column)
        except AttributeError as e:
            kwargs['allow_none'] = True


# raise convert.ModelConversionError('Add column kwargs:! {}:{}:{}'.format(str(column), column.name,str(e)))


class BaseOpts(ModelSchemaOpts):
    def __init__(self, meta):
        super(BaseOpts, self).__init__(meta)
        self.model_converter = getattr(meta, 'model_converter', MyModelConverter)


class BaseModelScheme(_BaseModelScheme):
    OPTIONS_CLASS = BaseOpts

    @classmethod
    def get_object_created_response(cls, **kwargs):
        if 'scheme_class' in kwargs:
            scheme_class = kwargs['scheme_class']
            del kwargs['scheme_class']
        else:
            scheme_class = cls

        pk_field_name = scheme_class.Meta.model.get_model_class_primary_key()
        pk_field = scheme_class.Meta.model.get_field(pk_field_name)
        try:
            pk_type = str(pk_field.type).lower()
        except:
            pk_type = 'unknown'

        if 'int' in pk_type:
            pass
        elif 'varchar' in pk_type:
            if 'uuid' in pk_field_name:
                if 'scheme' not in kwargs:
                    kwargs['scheme'] = ObjectCreatedUuidAsPk
                if 'data_key' not in kwargs:
                    kwargs['data_key'] = 'object_uuid'
            else:
                if 'scheme' not in kwargs:
                    kwargs['scheme'] = ObjectCreatedCompositeOrStrPk
                if 'data_key' not in kwargs:
                    kwargs['data_key'] = 'object_pk'

        return ObjectCreatedResponse(**kwargs)


from falcon_rest.responses import SuccessResponse


class SystemFunctionSchemeGet(BaseModelScheme):
    class Meta:
        model = model.SystemFunction


# exclude = ('system_function_id', )


# +++ Revisions
class EntityScheme(Schema):
    name = Str()
    table_name = Str()
    fields = List(Str())


from falcon_rest.contrib.objects_history.object_revision.object_revision import ObjectRevisionSchemeGet


class ObjectRevisionRecordSchemeGet(BaseModelScheme):
    class Meta:
        model = model.ObjectRevisionRecordModel
        exclude = ('id',)


class ObjectRevisionFilterSchemeGet(BaseModelScheme):
    id = Int()
    changes = Nested(ObjectRevisionRecordSchemeGet, many=True, attribute='_changes')
    action = String(validate=validate.OneOf(choices=('create', 'update', 'delete', 'restore')))
    user_name = Str()
    entity_name = Str()
    object_name = Str()
    name = Str()
    revision_time = DateTime()
    revision_number = Int()
    module = Str()

    class Meta:
        model = model.ObjectRevisionModel
        fields = ('id', 'user_id', 'revision_time', 'user_name', 'entity_name', 'object_name', 'entity_pk', 'action',
                  'revision_number', 'changes', 'module')
        search_fields = (
            'id', 'user_id', 'user_name', 'entity_name', 'object_name', 'entity_pk', 'action', 'revision_number',
            'module')
        query_fields = ('revision_time_gt', 'revision_time_lt')


# +++ User
class UserAuthIpScheme(BaseModelScheme):
    ip = Str(validate=validate.Regexp(IP_REGEXP))

    class Meta:
        model = model.UserAuthIp
        fields = ('ip',)


class UserCodedeckAlertsScheme(BaseModelScheme):
    show_az_codedeck = Bool()
    show_us_codedeck = Bool()

    class Meta:
        model = model.UserCodedeckAlerts
        fields = ('show_az_codedeck', 'show_us_codedeck')


class UserCodedeckAlertsGetScheme(BaseModelScheme):
    show_az_codedeck = Bool()
    show_us_codedeck = Bool()
    has_updated_az_codedeck = Bool()
    has_updated_us_codedeck = Bool()

    class Meta:
        model = model.UserCodedeckAlerts
        fields = ('show_az_codedeck', 'show_us_codedeck', 'has_updated_az_codedeck', 'has_updated_us_codedeck')


class UserScheme(BaseModelScheme):
    email = Str()  # Emails()  # Email(validate=validate.Email(error='Incorrect email address'))
    passwd = Str(validate=validate.Length(max=40))
    user_type = Choice()
    role_id = Int(required=False)
    login_ip = Str(required=False, validate=validate.Regexp(IP_REGEXP))
    auth_ip = Nested(UserAuthIpScheme, many=True)
    landing_page = Str(required=False)
    client_limits = Nested('UsersLimitScheme', many=True)
    avatar_id = Int(validate=lambda value: _valid('ImportExportLogs', 'id', value))
    first_name = Str(required=False, validate=validate.Length(max=40))
    last_name = Str(required=False, validate=validate.Length(max=40))
    default_mod = Str(required=False, validate=validate.Length(max=100))
    default_mod2 = Str(required=False, validate=validate.Length(max=100))
    client_id = Int(required=False, validate=lambda value: _valid('Client', 'client_id', value))
    client = Nested('CarrierInnerUserScheme', many=False, required=False, allow_none=True)
    codedeck_alerts = Nested('UserCodedeckAlertsScheme', many=False, required=False, allow_none=True)

    @pre_load
    def check_carrier(self, data):
        if 'client' in data:
            # if not 'client_id' in data:
            # 	raise ValidationError({'client_id': ['must set  "client_id" for "client" field !']})
            if 'user_type' in data and data['user_type'] != 'client':
                raise ValidationError({'client': ['must not set when user type not = "client"!']})
        if 'codedeck_alerts' in data:
            if data['user_type'] != 'admin':
                raise ValidationError({'codedeck_alerts': ['must not set when user type not = "admin"!']})
        return data

    class Meta:
        model = model.User
        fields = (
            'active', 'all_termination', 'card_id', 'client_id', 'client_limits', 'default_mod', 'default_mod2',
            'email',
            'landing_page', 'auth_ip', 'name', 'first_name', 'last_name', 'outbound_report', 'passwd', 'report_fields',
            'report_group', 'reseller_id', 'client_id',
            'role_id', 'show_carrier_trunk_drop_only', 'user_type', 'avatar_id', 'client', 'codedeck_alerts')


class UserMinScheme(BaseModelScheme):
    name = Str(required=True, validate=validate.Regexp(NAME_REGEXP))
    passwd = Str(required=True)
    first_name = Str(required=False)
    last_name = Str(required=False)
    avatar_id = Int()

    class Meta:
        model = model.User
        fields = ('name', 'passwd', 'first_name', 'last_name', 'avatar_id')


class UserAdminScheme(UserScheme):
    passwd = Str(allow_none=False, validate=validate.Length(min=4, max=16))
    name = Str(allow_none=False, validate=[validate.Length(min=4, max=16), validate.Regexp(NAME_REGEXP)])

    class Meta:
        model = model.User
        fields = ('name', 'passwd')


class UserGetScheme(UserMinScheme):
    user_type = Choice()
    role_id = Int()
    role_name = Str(validate=validate.Regexp(NAME_REGEXP))
    client_limits = Nested('UsersLimitScheme', many=True)
    auth_ip = Nested(UserAuthIpScheme, many=True)
    carrier_name = Str()
    client_id = Int()
    client = Nested('CarrierInnerUserScheme', many=False, required=False, allow_none=True)
    default_billing_decimal = Int()
    report_count = Str()
    is_preload = Bool()
    email = Str()
    has_limit_carrier_id = Int()
    referal_key = Str()
    allowed_ingress_id = List(Str())
    allowed_egress_id = List(Str())
    codedeck_alerts = Nested('UserCodedeckAlertsGetScheme', many=False, required=False, allow_none=True)
    company_type = Str()
    soon_expired_gateways = List(Dict())

    class Meta:
        model = model.User
        # fields = ('user_id','name','fullname','email','user_type','last_login_time','active')
        exclude = ('password', 'role', 'cdr_api_token', 'cdr_expire', 'login_url', 'passwd')
        search_fields = ('name', 'user_type', 'active', 'is_online', 'role_id', 'carrier_name', 'client', 'email',
                         'has_limit_carrier_id', 'role_name', 'client_id', 'last_seen', 'company_type')
        query_fields = ('last_login_time_gt', 'last_login_time_lt', 'last_login_time_isnull',
                        'create_time_gt', 'create_time_lt', 'user_id_in')


class UserActivateScheme(UserGetScheme):
    active = Bool()

    class Meta:
        model = model.User
        fields = ('active',)
        search_fields = (
            'name', 'user_id', 'login_ip', 'email', 'user_type', 'active', 'is_online', 'role_id', 'role_name')
        query_fields = ('last_login_time_gt', 'last_login_time_lt',
                        'create_time_gt', 'create_time_lt', 'user_id_in')


class UserInfoScheme(UserMinScheme):
    passwd = Str(required=True)
    first_name = Str(required=False, allow_none=True)
    last_name = Str(required=False, allow_none=True)
    avatar_id = Int(validate=lambda value: _valid('ImportExportLogs', 'id', value))
    email = Emails()

    class Meta:
        model = model.User
        fields = ('passwd', 'first_name', 'last_name', 'avatar_id', 'email')


class ApiKeyGetScheme(BaseModelScheme):
    token = Str()
    expired_on = DateTime()

    class Meta:
        model = model.UserApiKey
        fields = ('token', 'expired_on')


class UserInfoGetScheme(UserMinScheme):
    email = Emails()
    user_type = Choice()
    role_id = Int()
    role_name = Str()
    # client_limits = Nested('UsersLimitScheme', many=True)
    report_count = Str()
    is_preload = Bool()
    referal_key = Str()
    company_type = Str()
    # api_key = Nested('ApiKeyGetScheme')
    client = Nested('CarrierInnerUserScheme', many=False, required=False, allow_none=True)

    class Meta:
        model = model.User
        # fields = ('user_id','name','fullname','email','user_type','last_login_time','active')
        exclude = ('password', 'role', 'passwd', 'client_limits', 'api_key', 'cdr_api_token')
        search_fields = ('name', 'user_type', 'active', 'is_online', 'role_id', 'role_name')
        query_fields = ('last_login_time_gt', 'last_login_time_lt',
                        'create_time_gt', 'create_time_lt')


class UserGetExtScheme(UserGetScheme):
    # cdr_token = Str()

    class Meta:
        model = model.User


class UserResetPasswordLetterScheme(BaseModelScheme):
    name = Str()

    class Meta:
        fields = ('name',)
        model = model.User


class UserResetPasswordScheme(BaseModelScheme):
    token = Str()
    passwd = Str()

    class Meta:
        fields = ('passwd',)
        model = model.User


class UsersLimitScheme(BaseModelScheme):
    # user_id  =  Int()
    client_id = Int(validate=lambda value: _valid('Client', 'client_id', value))
    client_name = Str(validate=lambda value: _valid('Client', 'client_name', value))

    class Meta:
        model = model.UsersLimit
        fields = ('client_id', 'client_name')

    # --- User


# +++WebSession
class WebSessionGetScheme(BaseModelScheme):
    create_time = DateTime()
    user_id = Int()
    host = Str()
    agent = Str()
    exp = DateTime()
    # token = Str()
    user_name = Str()

    class Meta:
        model = model.WebSession
        fields = ('user_id', 'user_name', 'agent', 'host', 'create_time', 'exp')
        search_fields = ('user_id', 'user_name', 'agent', 'host')
        query_fields = ('create_time_lt', 'create_time_gt', 'exp_lt', 'exp_gt')


# ---WebSession

# +++SystemFunction


class SystemFunctionModulesGetScheme(BaseModelScheme):
    image_name = Str()
    usage_count = Int()

    class Meta:
        fields = ('image_name', 'usage_count')
        # search_fields=('image_name','usage_count')
        model = model.SystemModule


# exclude = ('role_privilegies', )


"""

class SystemFunctionModuledScheme(BaseModelScheme):
    modules = Nested(SystemFunctionModulesGetScheme,many=True)
    class Meta:
        model = model.SystemFunction
        fields = ('modules',)
"""


class SysMainMenuScheme(BaseModelScheme):
    id = Int()
    display_name = Str()
    ordering = Int()
    enabled = Bool()

    class Meta:
        model = model.SysMainMenu
        fields = ('display_name', 'ordering', 'enabled')


class SysMainMenuGetScheme(BaseModelScheme):
    class Meta:
        fields = ('id', 'display_name', 'ordering', 'enabled')
        search_fields = ('id', 'display_name', 'ordering', 'enabled')
        model = model.SysMainMenu


class SysSubMenuScheme(BaseModelScheme):
    id = Int()
    display_name = Str()
    ordering = Int()
    enabled = Bool()
    main_menu_id = Int()
    web_url = Str()
    is_readable = Bool()
    is_editable = Bool()
    is_executable = Bool()

    class Meta:
        model = model.SysSubMenu
        fields = ('id', 'display_name', 'ordering', 'enabled', 'main_menu_id', 'web_url',
                  'is_readable', 'is_editable', 'is_executable')


class SysSubMenuGetScheme(SysSubMenuScheme):
    class Meta:
        fields = ('id', 'display_name', 'ordering', 'enabled', 'main_menu_id', 'web_url',
                  'is_readable', 'is_editable', 'is_executable')
        search_fields = ('id', 'display_name', 'ordering', 'enabled', 'main_menu_id', 'web_url')
        model = model.SysSubMenu


class SystemFunctionScheme(BaseModelScheme):
    class Meta:
        model = model.SystemFunction
        exclude = ('system_function_id', 'role_privilegies')


class SystemFunctionGetScheme(BaseModelScheme):
    func_type = Str()
    group_id = Int()
    ordering = Int()

    class Meta:
        search_fields = (
            'func_name', 'image_name', 'func_type', 'is_read', 'is_write', 'is_exe', 'system_function_id',
            'func_url', 'group_id', 'ordering', 'web_page_name')
        model = model.SystemFunction
        exclude = ('role_privilegies',)


class SystemFunctionInnerScheme(BaseModelScheme):
    system_function_id = Int(validate=lambda value: _valid('SystemFunction', 'system_function_id', value))

    class Meta:
        model = model.SystemFunction
        fields = ('system_function_id',)


class SystemFunctionGroupScheme(BaseModelScheme):
    group_name = Str(validate=[validate.Regexp(NAME_REGEXP),
                               lambda value: _valid_unique('SystemFunctionGroup', 'group_name', value)])
    functions = Nested('SystemFunctionInnerScheme', many=True)
    ordering = Int()

    class Meta:
        model = model.SystemFunctionGroup
        fields = ('group_name', 'description', 'functions', 'ordering')


class SystemFunctionGroupModifyScheme(BaseModelScheme):
    group_name = Str(validate=[validate.Regexp(NAME_REGEXP)])
    functions = Nested('SystemFunctionInnerScheme', many=True)
    ordering = Int()

    class Meta:
        model = model.SystemFunctionGroup
        fields = ('group_name', 'description', 'functions', 'ordering')


class SystemFunctionGroupGetScheme(BaseModelScheme):
    functions = Nested('SystemFunctionInnerScheme', many=True)

    class Meta:
        model = model.SystemFunctionGroup
        fields = ('id', 'group_name', 'description', 'functions', 'ordering')
        search_fields = ('id', 'group_name',)


# ---SystemFunction

# +++RolePrivilege
class RolePrivilegeScheme(BaseModelScheme):
    writable = Bool(default=True)
    readable = Bool(default=True)
    executable = Bool(default=True)

    class Meta:
        model = model.RolePrivilege
        # exclude = ('role_id', )
        fields = ('writable', 'readable', 'executable')


class RolePrivilegeNestedScheme(BaseModelScheme):
    # system_function_id = Int(validate=lambda value: _valid('SystemFunction', 'system_function_id', value))
    func_url = Str(allow_none=False, required=True, validate=lambda value: _valid('SystemFunction', 'func_url', value))
    writable = Bool(default=True)
    readable = Bool(default=True)
    executable = Bool(default=True)

    class Meta:
        model = model.RolePrivilege
        # exclude = ('role_id', )
        fields = ('func_url', 'writable', 'readable', 'executable')


class RoleSubMenuNestedScheme(BaseModelScheme):
    # system_function_id = Int(validate=lambda value: _valid('SystemFunction', 'system_function_id', value))
    submenu_id = Int(allow_none=False, required=True, validate=lambda value: _valid('SysSubMenu', 'id', value))
    writable = Bool(default=True)
    readable = Bool(default=True)
    executable = Bool(default=True)
    main_menu_id = Int()
    ordering = Int()
    enabled = Bool()

    class Meta:
        model = model.RoleSubMenu
        # exclude = ('role_id', )
        fields = ('submenu_id', 'writable', 'readable', 'executable', 'main_menu_id', 'enabled', 'ordering',
                  'web_url', 'display_name')


class RolePrivilegeSchemeMultiple(BaseModelScheme):
    writable = Bool(default=True)
    readable = Bool(default=True)
    executable = Bool(default=True)

    class Meta:
        model = model.RolePrivilege
        # exclude = ('role_id', )
        fields = ('writable', 'readable', 'executable')


class RolePrivilegeGetScheme(BaseModelScheme):
    role_name = Str()
    func_name = Str()
    image_name = Str()
    func_url = Str()

    class Meta:
        model = model.RolePrivilege
        search_fields = ('system_function_id',)
        fields = (
            'role_name', 'func_name', 'image_name', 'func_url', 'role_id', 'system_function_id', 'writable', 'readable',
            'executable', 'role_privilege_id')


class GroupPrivilegeGetScheme(BaseModelScheme):
    display_name = Str()
    description = Str()
    ordering = Int()

    class Meta:
        model = model.SystemFunctionGroup
        search_fields = ('id',)
        fields = ('display_name', 'id', 'readable', 'ordering')


# ---RolePrivilege


# +++Role
class RoleScheme(BaseModelScheme):
    landing_page = Choice()
    default_mod = Str()
    default_mod2 = Str()
    role_privilegies = Nested(RolePrivilegeNestedScheme, many=True, attribute='_role_privilegies')
    role_ui_privilegies = Nested(RoleSubMenuNestedScheme, many=True, attribute='_role_ui_privilegies')
    group_privilegies = Nested(GroupPrivilegeGetScheme, many=True)
    delete_invoice = Bool()
    delete_payment = Bool()
    delete_credit = Bool()
    reset_balance = Bool()
    modify_credit_limit = Bool()
    modify_min_profit = Bool()

    class Meta:
        model = model.Role
        # exclude = ('role_id', )
        fields = ('role_name', 'landing_page', 'default_mod', 'default_mod2', 'delete_invoice',
                  'delete_payment', 'delete_credit', 'reset_balance', 'modify_credit_limit', 'modify_min_profit',
                  'group_privilegies', 'role_ui_privilegies', 'role_privilegies')


class RoleGetScheme(BaseModelScheme):
    landing_page = Choice()
    default_mod = Str()
    default_mod2 = Str()
    role_privilegies = Nested(RolePrivilegeNestedScheme, many=True, attribute='_role_privilegies')
    role_ui_privilegies = Nested(RoleSubMenuNestedScheme, many=True)
    # group_privilegies = Nested(GroupPrivilegeGetScheme, many=True)
    user_count = Int()

    class Meta:
        model = model.Role
        fields = (
            'role_id', 'role_name', 'user_count', 'landing_page', 'default_mod', 'default_mod2',
            'delete_invoice', 'delete_payment', 'delete_credit', 'reset_balance', 'modify_credit_limit',
            'modify_min_profit', 'group_privilegies', 'role_ui_privilegies', 'role_privilegies')
        search_fields = ('role_id', 'role_name', 'user_count', 'landing_page', 'delete_invoice', 'delete_payment',
                         'delete_credit', 'reset_balance', 'modify_credit_limit', 'modify_min_profit')


class RoleFunctionGroupGetScheme(RoleGetScheme):
    class Meta:
        model = model.Role
        fields = ('group_privilegies',)
        search_fields = ()


class WebSessionScheme(BaseModelScheme):
    id = Int()
    create_time = DateTime()
    user_id = Int()
    host = Str()
    agent = Str()
    msg = Str()

    user_name = Str()

    class Meta:
        model = model.WebSession
        fields = ('id', 'create_time', 'user_id', 'host', 'agent', 'msg', 'user_name')
        search_fields = ('id', 'user_id', 'host', 'agent', 'msg')
        query_fields = ('create_time_gt', 'create_time_lt')


# ---Role


# +++Agent
class AgentScheme(BaseModelScheme):
    email = Emails()  # Email(validate=validate.Email(error='Incorrect email address'))
    # method_type = Choice(validate=valid_dict(model.Agent.METHOD_TYPE_DICT),required=True)
    method_type = Choice(required=True)
    frequency_type = Choice(required=True)
    commission = Decimal(required=True, validate=validate.Range(0, 100))
    user = Nested(UserMinScheme)
    edit_permission = Bool()
    status = Bool()
    invoice_setting_id = Int(allow_none=True, validate=lambda value: _valid('InvoiceSetting', 'id', value))
    mail_sender_id = Int(validate=lambda value: _valid('MailSender', 'id', value))
    mail_template_title = String(validate=lambda value: _valid('MailTemplate', 'title', value))

    class Meta:
        model = model.Agent
        fields = ('agent_name', 'email', 'method_type', 'frequency_type', 'commission', 'user',
                  'edit_permission', 'status', 'invoice_setting_id', 'mail_sender_id',
                  'mail_template_title')


# exclude = ('agent_id', 'user_id', 'create_on', 'update_by', 'update_on','clients' ,
# 'user',
# 'method_type',
#          'frequency_type','agent_role_id')


class AgentModifyScheme(AgentScheme):
    method_type = Choice()
    frequency_type = Choice()
    commission = Decimal()

    class Meta:
        model = model.Agent
        fields = ('agent_name', 'email', 'method_type', 'frequency_type', 'commission', 'user',
                  'edit_permission', 'status', 'invoice_setting_id', 'mail_sender_id',
                  'mail_template_title')


class AgentGetScheme(AgentScheme):
    user = Nested(UserMinScheme)
    carriers = Nested('AgentClientGetScheme', many=True)
    active = Bool()
    commission = Float()
    email = Str()
    usage_count = Int()
    referal_key = Str()
    ingress_ids = Str(dump_only=True)
    egress_ids = Str(dump_only=True)

    class Meta:
        model = model.Agent
        fields = ('agent_id', 'user_id', 'create_on', 'update_by', 'update_on', 'carriers', 'agent_name',
                  'email', 'user', 'method_type', 'frequency_type', 'commission', 'active', 'edit_permission',
                  'usage_count', 'referal_key', 'ingress_ids', 'egress_ids', 'invoice_setting_id', 'mail_sender_id',
                  'mail_template_title')
        search_fields = (
            'agent_id', 'agent_name', 'email', 'method_type', 'frequency_type', 'edit_permission', 'usage_count',
            'active', 'invoice_setting_id', 'mail_sender_id', 'update_by', 'update_on')
        query_fields = ('commission_gt', 'commission_lt', 'usage_count_gt', 'usage_count_lt')


class AgentActivateScheme(AgentGetScheme):
    class Meta:
        model = model.Agent
        fields = ('active',)
        search_fields = ('agent_name', 'email', 'method_type', 'frequency_type', 'edit_permission')
        query_fields = ('commission_gt', 'commission_lt', 'agent_id_in')


class AgentAssignProductsScheme(AgentScheme):
    products = List(
        Int(validate=lambda value: _valid('ProductRoutRateTable', 'id', value, 'Invalid product id {}'.format(value))))

    class Meta:
        model = model.Agent
        fields = ('products',)


class AgentSendLinkScheme(AgentScheme):
    emails = Emails()

    class Meta:
        model = model.Agent
        fields = ('emails',)


class AgentProductsGetScheme(BaseModelScheme):
    id = Int()
    product_id = Int()
    name = Str()
    tech_prefix = Int()
    type = Choice()
    marketplace = Choice()
    description = Str()
    usage_count = Int()
    rate_table_id = Int()
    agent_rate_download_link = Str()
    rate_table_name = Str()
    rate_table_type = Str()

    class Meta:
        model = model.ProductAgentsRef
        fields = ('id', 'product_id', 'name', 'tech_prefix', 'type', 'description', 'marketplace', 'usage_count',
                  'agent_rate_download_link', 'rate_table_id', 'rate_table_name', 'rate_table_type')
        search_fields = ('name', 'tech_prefix', 'marketplace', 'usage_count', 'type', 'rate_table_name')


class AgentCarriersAddScheme(AgentScheme):
    carriers = Nested('AgentClientScheme', many=True)

    class Meta:
        model = model.Agent
        fields = ('carriers',)


class AgentCarriersGetScheme(AgentScheme):
    carriers = Nested('AgentClientGetScheme', many=True)

    class Meta:
        model = model.Agent
        fields = ('agent_id', 'carriers')


# +++ AgentClient

class AgentClientScheme(BaseModelScheme):
    client_id = Int(required=True)
    commission = Float(required=False)  # , dump_only=True)
    method_type = Choice(required=False)

    class Meta:
        model = model.AgentClient
        fields = ('client_id', 'method_type', 'commission')


class AgentClientGetScheme(BaseModelScheme):
    # user = Nested(UserGetScheme)
    # agent = Nested(AgentGetScheme)
    client = Nested('CarrierGetMinWithIngressScheme')
    agent_id = Int()
    agent_name = Str(validate=validate.Regexp(NAME_REGEXP))
    method_type = Choice()
    client_id = Int()
    client_name = Str()
    commission = Float()
    assigned_on = DateTime()
    assigned_by = Str()
    status = Str()
    mode = Choice()
    registered_on = DateTime()
    company_type = Choice()

    class Meta:
        model = model.AgentClient
        fields = ('agent_id', 'agent_name', 'commission', 'method_type', 'client_id', 'client_name',
                  'assigned_on', 'assigned_by', 'status', 'registered_on', 'mode', 'company_type')
        # exclude = ('commission', )
        search_fields = ('agent_id', 'client_id', 'client_name', 'assigned_by', 'company_type')
        query_fields = ('assigned_on_gt', 'assigned_on_lt', 'comission_gt', 'comission_lt')


# --- AgentClient

class AgentCommissionHistoryDetailGetScheme(BaseModelScheme):
    client_id = Int()
    client_name = Str()
    agent_name = Str()
    start_time = DateTime()
    end_time = DateTime()
    total_date = Int()
    agent_id = Int()
    client_cost = Float()

    class Meta:
        model = model.AgentCommissionHistoryDetail
        fields = ('id', 'client_id', 'client_name', 'start_time', 'end_time', 'total_date',
                  'commission', 'client_cost')
        search_fields = ('agent_id', 'client_id',)
        query_fields = ('start_time_gt', 'start_time_lt', 'end_time_gt', 'end_time_lt',
                        'client_cost_gt', 'client_cost_lt')


class AgentCommissionHistoryGetScheme(BaseModelScheme):
    agent_name = Str()
    details = Nested('AgentCommissionHistoryDetailGetScheme', many=True)
    commission = Float()
    agent_id = Int()
    client_id = Int()

    class Meta:
        model = model.AgentCommissionHistory
        fields = ('history_id', 'agent_id', 'agent_name', 'create_date', 'start_time', 'end_time',
                  'total_date', 'commission', 'finished',
                  'details', 'client_id'
                  )
        search_fields = ('agent_id', 'commision', 'finished', 'client_id')
        query_fields = ('start_time_gt', 'start_time_lt', 'create_date_gt', 'create_date_lt')


# region +++AgentCommissionPayment+++
class AgentCommissionPaymentScheme(BaseModelScheme):
    id = Int()
    agent_id = Int(validate=lambda value: _valid('Agent', 'agent_id', value))
    create_on = DateTime()
    create_by = Str(validate=[validate.Length(max=100)])
    amount = Float()
    note = Str()
    start_time = DateTime()
    end_time = DateTime()
    history = Nested('AgentCommissionHistoryScheme', many=False)

    class Meta:
        model = model.AgentCommissionPayment
        fields = ('agent_id', 'amount', 'note', 'start_time', 'end_time',)


class AgentCommissionPaymentSchemeGet(AgentCommissionPaymentScheme):
    agent_name = Str()

    class Meta:
        model = model.AgentCommissionPayment
        fields = (
        'agent_id', 'create_on', 'create_by', 'amount', 'note', 'start_time', 'end_time', "agent_name", "commission")
        search_fields = ('id', 'agent_id', 'create_by', 'note', "agent_name")
        query_fields = (
            'create_on_gt', 'create_on_lt', 'amount_gt', 'amount_lt', 'start_time_gt', 'start_time_lt', 'end_time_gt',
            'end_time_lt',)


class AgentCommissionPaymentSchemeModify(AgentCommissionPaymentScheme):
    pass


# endregion ---AgentCommissionPayment---


class ClientSimpleGetScheme(BaseModelScheme):
    client_id = Int()
    name = Str()
    email = Emails()
    status = Bool()
    client_type = Choice()
    mode = Int()
    ingress_count = Int()
    egress_count = Int()
    group_id = Int()
    is_vendor = Bool()
    is_client = Bool()
    is_orig = Bool()
    is_term = Bool()

    class Meta:
        model = model.Client
        fields = (
            'client_id', 'name', 'email', 'status', 'client_type', 'mode', 'is_vendor', 'is_client', 'is_orig',
            'is_term')
        search_fields = ('client_id', 'name', 'email', 'status', 'client_type', 'mode', 'group_id', 'is_term')
        query_fields = ('ingress_count_gt', 'egress_count_gt')


class TimeProfileScheme(BaseModelScheme):
    # gen module  time-profile
    # --- Time Profile ---

    name = Str(validate=[validate.Regexp(USERNAME_REGEXP), validate.Length(1, 100)], allow_none=False)
    start_time = Time(False, allow_none=True)  # ,default=datetime.time("00:00:00") )
    end_time = Time(False, allow_none=True)  # ,default=datetime.time("23:59:59") )
    start_day_of_week = Choice(allow_none=True)
    end_day_of_week = Choice(allow_none=True)

    type_name = Choice()
    time_zone = Str(validate=[validate.Length(1, 10)], allow_none=True)

    @pre_load
    def check_end_time(self, data):
        if 'start_time' in data and 'end_time' in data:
            if data['start_time'] >= data['end_time']:
                raise ValidationError({'end_time': ['Must be more than start_time']})
        return data

    @pre_load
    def set_all_time(self, data):
        if 'type_name' in data:
            if data['type_name'] == 'all time':
                try:
                    del data['start_time']
                    del data['end_time']
                    del data['start_day_of_week']
                    del data['end_day_of_week']
                except:
                    pass
            if data['type_name'] == 'weekly':
                if not 'start_day_of_week' in data or data['start_day_of_week'] is None:
                    raise ValidationError({'start_day_of_week': ['Missing field, must be when weekly']})
                if not 'end_day_of_week' in data or data['end_day_of_week'] is None:
                    raise ValidationError({'end_day_of_week': ['Missing field, must be when weekly']})
        return data

    class Meta:
        model = model.TimeProfile
        fields = ('name', 'start_time', 'end_time', 'start_day_of_week', 'end_day_of_week', 'type_name', 'time_zone')


class TimeProfileGetScheme(TimeProfileScheme):
    type_name = Choice()
    start_time = Str()
    end_time = Str()

    class Meta:
        model = model.TimeProfile
        fields = ('time_profile_id', 'name', 'start_time', 'end_time', 'start_day_of_week', 'end_day_of_week',
                  'type_name', 'time_zone')
        search_fields = ('name', 'type_name', 'start_day_of_week', 'end_day_of_week')
        query_fields = ('start_time_gt', 'start_time_lt',
                        'end_time_gt', 'end_time_lt', 'time_profile_id_in')


class DigitTranslationScheme(BaseModelScheme):
    digit_map_name = Str(validate=validate.Regexp(NAME_REGEXP))

    class Meta:
        model = model.DigitTranslation
        fields = ('digit_map_name',)


class DigitTranslationGetScheme(BaseModelScheme):
    update_at = DateTime()
    digit_map_name = Str()

    items = Nested('TranslationItemGetScheme', many=True)
    digit_map_count = Int()

    class Meta:
        model = model.DigitTranslation
        fields = ('translation_id', 'digit_map_name', 'items', 'digit_map_count', 'update_at')
        search_fields = ('translation_id', 'digit_map_name', 'items', 'digit_map_count')
        query_fields = ('update_at_gte', 'update_at_lt', 'translation_id_in')


class TranslationItemScheme(BaseModelScheme):
    # gen   patch /digit_mapping/{digit_mapping_id}
    # Modify Digital Mapping
    # ANI_action = Str()
    ANI_action = Choice(attribute='ani_method', load_from='ANI_action', dump_to='ANI_action')
    # ANI_prefix = Str()
    ANI_prefix = Str()
    # ANI_replace_to = Str()
    ANI_replace_to = Str()  # attribute='action_ani',load_from='ANI_replace_to',dump_to='ANI_replace_to' )
    # DNIS_action = Str()
    DNIS_action = Choice(attribute='dnis_method', load_from='DNIS_action', dump_to='DNIS_action')
    # DNIS_prefix = Str()
    DNIS_prefix = Str()
    # DNIS_replace_to = Str()
    DNIS_replace_to = Str()  # attribute='action_dnis',load_from='DNIS_replace_to',dump_to='DNIS_replace_to' )

    class Meta:
        model = model.TranslationItem
        fields = ('ANI_action', 'ANI_prefix', 'ANI_replace_to', 'DNIS_action', 'DNIS_prefix', 'DNIS_replace_to')


class TranslationItemGetScheme(TranslationItemScheme):
    digit_map_name = Str()  # parens's

    class Meta:
        model = model.TranslationItem
        fields = ('ref_id', 'digit_map_name', 'ANI_action', 'ANI_prefix', 'ANI_replace_to',
                  'DNIS_action', 'DNIS_prefix', 'DNIS_replace_to')
        search_fields = ('digit_map_name', 'ANI_replace_to', 'DNIS_replace_to', 'ANI_prefix', 'DNIS_prefix')
        query_fields = ('ref_id_in')


class CurrencyScheme(BaseModelScheme):
    rate = Float()

    class Meta:
        model = model.Currency
        exclude = ('currency_id', 'update_by')


class CurrencyGetScheme(BaseModelScheme):
    rate = Float()
    count = Method("get_rate_tables_count_using_this_currency", default=0)
    count.num_type = int

    # noinspection PyMethodMayBeStatic
    def get_rate_tables_count_using_this_currency(self, obj):
        return model.Currency.get_rate_tables_count_using_currency(obj.currency_id)

    class Meta:
        model = model.Currency
        fields = ("currency_id", "code", "active", "count", "update_by", "rate")


validate_currency = lambda value: _valid('Currency', 'code', value)
validate_payment_term = lambda value: _valid('PaymentTerm', 'name', value)


class PaymentTermScheme(BaseModelScheme):
    name = Str(validate=validate.Regexp(NAME_REGEXP))
    type = Choice()
    grace_period = Int()
    days = List(Int(validate=validate.Range(0, 31)), attribute='more_days_arr')
    notify_days = Int()

    @pre_load
    def check_days(self, data):
        if 'days' in data:
            days = data['days']
            for d in days:
                if 'type' in data:
                    if data['type'] in ('Day of Month', 'Some day of month') and (int(d) == 0 or int(d) > 31):
                        raise ValidationError('Wrong day {} for {}'.format(d, data['type']))
                    if data['type'] == 'Day of Week' and (int(d) > 7 or int(d) < 1):
                        raise ValidationError('Wrong day {} for {}, allowed is {}'.format(d, data['type'],
                                                                                          model.PaymentTerm.DAYS_OF_WEEK))
        return data

    class Meta:
        model = model.PaymentTerm
        fields = ('name', 'type', 'grace_period', 'days', 'notify_days')


class PaymentTermGetScheme(PaymentTermScheme):
    # count = Method("get_clients_count_using_this_payment_term", default=0)
    usage_count = Int()
    type_name = Choice()
    cycle = Str()

    class Meta:
        model = model.PaymentTerm
        fields = ('payment_term_id', 'name', 'type_name', 'grace_period', 'days', 'notify_days', 'usage_count',
                  'cycle', 'term_usage_count', 'orig_usage_count')
        search_fields = ('payment_term_id', 'name', 'type_name', 'grace_period', 'notify_days')
        query_fields = ('payment_term_id_in', 'usage_count_gt', 'usage_count_lt')


# --- VoipGateway==switch

class SwitchProfileSipHostScheme(BaseModelScheme):
    host_name = Str(required=True)
    sip_ip = Str(validate=validate.Regexp(IP_REGEXP), required=True)
    sip_port = Int(validate=validate.Range(1000, 65535), required=True)
    cps = Int()
    cap = Int()
    self_cps = Int()
    self_cap = Int()
    support_register = Int(attribute='default_register')
    rpid = Int(attribute='support_rpid')
    oli = Int(attribute='support_oli')
    priv = Int(attribute='support_priv')
    div = Int(attribute='support_div')
    paid = Int(attribute='support_paid')
    pci = Int(attribute='support_pci')
    x_lrn = Int(attribute='support_x_lrn')
    x_header = Int(attribute='support_x_header')
    status = Choice()
    auth_register = Int()
    last_started_on = Str()
    lan_ip = Str(validate=validate.Regexp(IP_REGEXP), allow_none=True)
    lan_port = Int(validate=validate.Range(1000, 65535), allow_none=True)

    class Meta:
        model = model.SwitchProfile
        fields = ('host_name', 'sip_ip', 'sip_port', 'self_cps', 'self_cap', 'support_register',
                  'rpid', 'oli', 'priv', 'div', 'paid', 'pci', 'x_lrn', 'x_header', 'status', 'auth_register',
                  'lan_ip', 'lan_port', 'last_started_on')


class SwitchProfileSipHostGetScheme(SwitchProfileSipHostScheme):
    id = Int()
    switch_id = Int()
    server_name = Str()
    switch_name = Str()
    auth_register = Int()
    last_started_on = Str()

    class Meta:
        model = model.SwitchProfile
        fields = (
            'id', 'switch_id', 'host_name', 'sip_ip', 'sip_port', 'self_cps', 'self_cap',
            'support_register', 'lan_ip', 'lan_port',
            'rpid', 'oli', 'priv', 'div', 'paid', 'pci', 'x_lrn', 'x_header', 'status',
            'server_name', 'switch_name', 'auth_register', 'last_started_on')
        search_fields = ('host_name', 'sip_ip', 'switch_id', 'server_name', 'switch_name', 'auth_register', 'status')


class SwitchProfileSipHostGetVoipGatewayScheme(SwitchProfileSipHostGetScheme):
    class Meta(SwitchProfileSipHostGetScheme.Meta):
        fields = (
            'id', 'switch_id', 'host_name', 'sip_ip', 'sip_port', 'support_register', 'lan_ip', 'lan_port',
            'rpid', 'oli', 'priv', 'div', 'paid', 'pci', 'x_lrn', 'x_header', 'status',
            'server_name', 'switch_name', 'auth_register', 'last_started_on')


class VoipGatewayScheme(BaseModelScheme):
    # gen module  voip-gateway
    # --- Switch Instance ---
    name = Str(validate=validate.Regexp(NAME_REGEXP), allow_none=False, required=True)
    lan_ip = Str(validate=validate.Regexp(IP_REGEXP), allow_none=False, required=True)
    lan_port = Int(validate=validate.Range(1000, 65535), allow_none=False, required=True)
    active_call_ip = Str(validate=validate.Regexp(IP_REGEXP))
    active_call_port = Int(validate=validate.Range(1000, 65535))
    # connected = Str()
    user_limited_cps = Str()
    # profiles = Array(attribute='profiles',
    #    load_from='profiles',dump_to='profiles' )
    licensed_channel_limit = Str()
    user_defined_channel_limit = Str()
    licensed_cps = Str()
    paid_replace_ip = Integer()
    sip_host = Nested(SwitchProfileSipHostScheme, many=True)

    class Meta:
        model = model.VoipGateway
        fields = ('name', 'lan_ip', 'lan_port', 'paid_replace_ip', 'sip_host', 'active_call_ip', 'active_call_port', 'cps', 'cap', 'self_cps', 'self_cap',)


# fields=('lan_port','user_limited_cps','licensed_channel_limit','connected','name','user_defined_channel_limit','lan_ip','licensed_cps')


class VoipGatewayGetScheme(VoipGatewayScheme):
    # usage_count = Int(dump_only=True)
    switch_id = Int()
    active_call_ip = Str(validate=validate.Regexp(IP_REGEXP))
    active_call_port = Int(validate=validate.Range(1000, 65535))
    connected = Bool()
    expire = Int()
    uuid = Str()
    version_info = Nested('VersionInformationInnerGetScheme', many=False)

    class Meta:
        model = model.VoipGateway
        fields = (
            'switch_id', 'name', 'lan_ip', 'lan_port', 'connected', 'paid_replace_ip', 'active_call_ip',
            'active_call_port', 'expire', 'uuid', 'version_info', 'cap', 'self_cap', 'cps', 'self_cps')
        search_fields = ('switch_id', 'name', 'lan_ip', 'lan_port', 'active_call_ip', 'expire', 'uuid')


class VoipGatewayFullGetScheme(VoipGatewayScheme):
    # usage_count = Int(dump_only=True)
    switch_id = Int()
    active_call_ip = Str(validate=validate.Regexp(IP_REGEXP))
    active_call_port = Int(validate=validate.Range(1000, 65535))
    connected = Bool(attribute='_connected')
    sip_host = Nested(SwitchProfileSipHostGetVoipGatewayScheme, many=True)
    limits = Dict()
    sip_profile_show = Dict()
    get_disk_info = Dict()

    class Meta:
        model = model.VoipGateway
        fields = (
            'switch_id', 'name', 'lan_ip', 'lan_port', 'connected', 'paid_replace_ip', 'sip_host', 'active_call_ip',
            'active_call_port', 'limits', 'sip_profile_show', 'get_disk_info')
        search_fields = ('switch_id', 'name', 'lan_ip', 'lan_port', 'active_call_ip', 'connected')


from api_dnl.utils.dnl_active_calls import DnlActiveCallSession


class VoipGatewayActiveCallsScheme(BaseModelScheme):
    fields = List(Str(validate=validate.OneOf(DnlActiveCallSession.FIELDS)))
    filters = Dict()
    # first_record_number = Int(default=1, validate=validate.Range(1, 65535))
    # last_record_number = Int(default=10, validate=validate.Range(1, 65535))
    raw_values = Bool(default=False)
    page = Int(default=0)
    per_page = Int(default=10)

    class Meta:
        model = model.VoipGateway
        fields = ('fields', 'filters', 'first_record_number', 'last_record_number', 'raw_values', 'page', 'per_page')


class VoipGatewayCallStatScheme(BaseModelScheme):
    period = Str(
        validate=validate.OneOf(['last hour', 'last 24 hours', 'today', 'last week', 'last month', 'last year']))

    class Meta:
        model = model.VoipGateway
        fields = ('period',)


class VoipGatewayRateSortingScheme(BaseModelScheme):
    effective_date = Str(default=str(datetime.now(UTC).date()))
    code = Str()
    rate_table_ids = List(Int(validate=lambda value: _valid('RateTable', 'rate_table_id', value)))

    class Meta:
        model = model.VoipGateway
        fields = ('effective_date', 'code', 'rate_table_ids')


class VoipGatewaySignCallScheme(BaseModelScheme):
    ani = Str()
    dnis = Str()
    attestation_lvl = Str()

    class Meta:
        model = model.VoipGateway
        fields = ('ani', 'dnis', 'attestation_lvl')


class VoipGatewayCallStatGetScheme(Schema):
    last_revenue = Float()
    daily_margin_percentage = Float()
    last_acd = Float()
    daily_minutes = Float()
    last_total_calls = Float()
    last_non_zero_calls = Float()
    last_asr_percentage = Float()
    daily_margin = Float()
    last_profitability_percentage = Float()


class VoipGatewayTestTelnetScheme(BaseModelScheme):
    cmd_line = Str()

    class Meta:
        model = model.VoipGateway
        fields = ('cmd_line',)


class VoipGatewaySystemCallStatScheme(BaseModelScheme):
    # period = Str(validate=validate.OneOf(['last hour', 'last 24 hours', 'today']))
    switch_id = Int()
    type = Str(validate=validate.OneOf(['term', 'orig']))

    class Meta:
        model = model.VoipGateway
        fields = (' switch_id', 'type')


class VoipGatewaySystemCallStatGetScheme(Schema):
    call = Int()
    chan = Int()
    cps = Int()
    o_chan = Int()
    o_cps = Int()
    t_chan = Int()
    t_cps = Int()


class VoipGatewayLicenseLimitGetScheme(Schema):
    sys_cap = Int()
    sys_cps = Int()
    lic_cap = Int()
    lic_cps = Int()
    expire = Int()


class VoipGatewaySystemPeakStatGetScheme(Schema):
    curr_cps = Int()  # CPS    Limit
    curr_call = Int()  # Calls
    curr_chan = Int()  # Channel
    cps_24hr = Int()
    cps_7day = Int()
    cps_rece = Int()
    chan_24hr = Int()
    chan_7day = Int()
    chan_rece = Int()
    call_24hr = Int()
    call_7day = Int()
    call_rece = Int()
    by_pass_call_24hr = Int()
    by_pass_call_7day = Int()
    by_pass_call_rece = Int()
    proxy_call_24hr = Int()
    proxy_call_7day = Int()
    proxy_call_rece = Int()
    transcoding_call_24hr = Int()
    transcoding_call_7day = Int()
    transcoding_call_rece = Int()
    call_max = Function(lambda obj: max(obj.curr_call, obj.call_24hr, obj.call_7day, obj.call_rece))


# --- VoipGateway==switch

# +++ SwitchProfile
class SwitchProfileScheme(BaseModelScheme):
    status = Choice()
    switch_id = Str()
    host_name = Str()

    class Meta:
        model = model.SwitchProfile
        exclude = ('id', 'voip_gateway', 'egress', 'cli_ip', 'cli_port')


class SwitchProfileModifyScheme(BaseModelScheme):
    status = Choice()
    switch_id = Str()

    class Meta:
        model = model.SwitchProfile
        fields = ('cps', 'cap',)


class SwitchProfileMinGetScheme(BaseModelScheme):
    status = Choice()
    switch_id = Str()

    class Meta:
        model = model.SwitchProfile
        fields = ('id', 'cps', 'cap')


class SwitchProfileGetScheme(BaseModelScheme):
    switch_name = Str()
    switch_id = Str()
    cli_ip = Str(validate=validate.Regexp(IP_REGEXP))
    cli_port = Int(validate=validate.Range(1000, 65535))
    cps = Int()
    cap = Int()
    expiration_date = DateTime()
    status = Choice(attribute='_status')
    limits = Dict()

    class Meta:
        model = model.SwitchProfile
        fields = (
            'id', 'switch_id', 'switch_name', 'cli_ip', 'cli_port', 'cps', 'cap', 'expiration_date', 'status', 'limits')
        query_fields = ('id_in', 'switch_name', 'switch_id', 'status')


# -- SwitchProfile


class RouteStrategyScheme(BaseModelScheme):
    name = Str(validate=[validate.Regexp(USERNAME_REGEXP), validate.Length(min=1)])

    class Meta:
        model = model.RouteStrategy
        fields = ('name', 'is_virtual')


class RouteStrategyExScheme(BaseModelScheme):
    type = Str(required=True, allow_none=False, validate=validate.Regexp(NAME_REGEXP))
    type = Str(required=True, allow_none=False, validate=validate.OneOf(['LCR', 'Top-Down', 'Round-Robin']))
    egress_trunks = List(Int(validate=lambda value: _valid('Resource', 'resource_id', value)))

    class Meta:
        model = model.RouteStrategy
        fields = ('name', 'type', 'egress_trunks')


class RouteStrategyGetScheme(BaseModelScheme):
    usage_count = Int(dump_only=True)
    term_usage_count = Int(dump_only=True)
    orig_usage_count = Int(dump_only=True)
    route_plan_id = Int()
    dynamic_usage_count = Int(dump_only=True)
    static_usage_count = Str(dump_only=True)
    dynamic_name = Str()
    static_name = Str()
    dynamic_route_id = Int()
    static_route_id = Str()

    class Meta:
        model = model.RouteStrategy
        fields = ('route_plan_id', 'name', 'update_at', 'update_by', 'is_virtual', 'usage_count', 'dynamic_name',
                  'static_name', 'dynamic_usage_count', 'static_usage_count', 'term_usage_count', 'orig_usage_count')
        search_fields = (
            'route_plan_id', 'name', 'is_virtual', 'update_by', 'usage_count', 'dynamic_name', 'static_name',
            'dynamic_route_id', 'static_route_id', 'dynamic_usage_count', 'static_usage_count', 'term_usage_count',
            'orig_usage_count')
        query_fields = ('update_at_gt', 'update_at_lt', 'route_plan_id_in')


class RouteStrategyCopyScheme(BaseModelScheme):
    route_plan_id = Int(validate=[lambda value: _valid_unique('RouteStrategy', 'route_plan_id', value)])

    class Meta:
        model = model.RouteStrategy
        fields = ('name',)


# ?Product? view


# +++ CarrierGroupScheme
def validate_client_list(value):
    if not value or len(value) == 0:
        raise ValidationError({'clients': ["Client list must have one or more clients"]})
    for c in value:
        _valid('Client', 'client_id', c)


class CarrierGroupScheme(BaseModelScheme):
    group_name = Str(validate=validate.Regexp(NAME_REGEXP))
    clients = List(Int(), validate=validate_client_list)

    class Meta:
        model = model.CarrierGroup
        fields = ('group_name', 'clients')


class CarrierGroupGetScheme(BaseModelScheme):
    group_id = Int()
    group_name = Str()
    clients = List(Int())
    clients_detail = Nested('CarrierGetScheme', many=True)  # List(Int())
    clients_count = Int()

    class Meta:
        model = model.CarrierGroup
        fields = ('group_id', 'clients', 'clients_detail', 'clients_count', 'group_name')
        exclude = ('carriers', 'clients_rel')
        search_fields = ('group_id', 'group_name')
        query_fields = ('group_id_in')


# --- CarrierGroupScheme

class ProductRoutRateTableMinScheme(BaseModelScheme):
    class Meta:
        model = model.ProductRoutRateTable
        fields = ('id',)


class ProductRoutRateTableScheme(BaseModelScheme):
    name = Str(validate=validate.Regexp(NAME_REGEXP))
    route_plan_id = Int(validate=lambda value: _valid('RouteStrategy', 'route_strategy_id', value))
    rate_table_id = Int(validate=lambda value: _valid('RateTable', 'rate_table_id', value))
    tech_prefix = Str(validate=validate.Regexp(PREFIX_REGEXP))
    agent_id = Int(validate=lambda value: _valid('Agent', 'agent_id', value))
    type = Choice()
    agents = List(Int(validate=lambda value: _valid('Agent', 'agent_id', value)))
    clients = List(Int(validate=lambda value: _valid('Client', 'client_id', value)))
    description = Str()
    allowed_clients = List(Int(validate=lambda value: _valid('Client', 'client_id', value)))

    class Meta:
        model = model.ProductRoutRateTable
        # exclude = ('product_id',)
        fields = ('name', 'route_plan_id', 'rate_table_id', 'tech_prefix', 'agent_id',
                  'type', 'agents', 'clients', 'description', 'allowed_clients')


class ProductRoutRateTableGetScheme(BaseModelScheme):
    name = Str()
    route_plan_id = Int()
    rate_table_id = Int()
    route_plan_name = Str()
    rate_table_name = Str()
    tech_prefix = Str()
    agent_id = Int()
    type = Choice()
    agents = List(Int())
    clients = List(Int())
    trunks = Nested('ResourcePrefixScheme', many=True)  # 'ResourceInfoGetScheme'
    description = Str()
    allowed_clients = List(Int())
    update_at = DateTime()
    clients_count = Int()
    agents_count = Int()
    client_id = Str()

    class Meta:
        model = model.ProductRoutRateTable
        fields = ('id', 'name', 'route_plan_id', 'rate_table_id', 'route_plan_name',
                  'rate_table_name', 'tech_prefix', 'type', 'agents', 'clients', 'description', 'agent_id',
                  'allowed_clients', 'update_by', 'update_at', 'trunks', 'clients_count', 'agents_count', 'client_id')
        search_fields = ('name', 'route_plan_id', 'rate_table_id', 'type', 'agent_id', 'update_by', 'route_plan_name',
                         'rate_table_name', 'tech_prefix', 'client_id', 'clients_count', 'agents_count')
        query_fields = ('update_at_gt', 'update_at_lt', 'id_in')


class ProductRoutRateTablePublicGetScheme(BaseModelScheme):
    name = Str()
    tech_prefix = Int()
    type = Choice()
    description = Str()
    reference = Str()

    class Meta:
        model = model.ProductRoutRateTable
        fields = ('id', 'name', 'tech_prefix', 'type', 'description')
        search_fields = ('name', 'tech_prefix', 'reference')


class ProductRoutRateTableClientsScheme(BaseModelScheme):
    clients = List(Int())

    class Meta:
        model = model.ProductRoutRateTable
        # exclude = ('product_id',)
        fields = ('clients',)


class ProductRouteAnalysisScheme(BaseModelScheme):
    route_plan_id = Int(validate=lambda value: _valid('RouteStrategy', 'route_strategy_id', value))
    rate_table_id = Int(validate=lambda value: _valid('RateTable', 'rate_table_id', value))
    code_deck_id = Int(allow_none=True, validate=lambda value: _valid('CodeDeck', 'code_deck_id', value))
    characteristics = Choice()  # Str(validate=validate.OneOf(('All', 'No Active Egress Trunk', 'No profitable route', 'Number of profitable routes equals')))
    count_profiltable_routes = Int(allow_none=True)

    class Meta:
        model = model.ProductRouteAnalysis
        fields = ('route_plan_id', 'rate_table_id', 'characteristics', 'count_profiltable_routes', 'code_deck_id')


class ProductRouteAnalysisGetScheme(BaseModelScheme):
    uuid = UUID()
    date = DateTime()
    status = Choice()
    finish = DateTime()
    process = Str()
    rec_num = Int()

    class Meta:
        model = model.ProductRouteAnalysis
        fields = ('uuid', 'date', 'status', 'finish', 'process', 'rec_num') + ProductRouteAnalysisScheme.Meta.fields
        search_fields = ('uuid', 'status', 'process', 'rec_num') + ProductRouteAnalysisScheme.Meta.fields
        query_fields = ('date_gt', 'date_lt', 'finish_gt', 'finish_lt')


class ProductRouteAnalysisLogGetScheme(BaseModelScheme):
    id = Int()
    uuid = UUID()
    code = Str()
    code_name = Str()
    country = Str()
    rate = Float()
    trunk_count = Int()
    profitable_trunk_count = Int()

    class Meta:
        model = model.ProductRouteAnalysisLog
        fields = ('id', 'code', 'code_name', 'country', 'rate', 'trunk_count', 'profitable_trunk_count')
        search_fields = ('id', 'uuid', 'code', 'code_name', 'country', 'rate', 'trunk_count', 'profitable_trunk_count')


# ?Product" view

class ProductScheme(BaseModelScheme):
    name = Str(validate=[lambda value: _valid_unique('Product', 'name', value), validate.Regexp(USERNAME_REGEXP)])
    defined_by = Choice()
    routed_by = Choice()
    code_deck_id = Int(allow_none=True, validate=lambda value: _valid('CodeDeck', 'code_deck_id', value))

    @pre_load
    def check_code_deck(self, data):
        if 'defined_by' in data:
            if data['defined_by'] == 'Code Name':
                if 'code_deck_id' not in data or data['code_deck_id'] == None:
                    raise ValidationError({'code_deck_id': ['must present if defined_by Code Name']})
            else:
                if 'code_deck_id' in data and data['code_deck_id']:
                    raise ValidationError({'code_deck_id': ['not needed when not defined_by Code Name']})
        return data

    class Meta:
        model = model.Product
        # exclude = ('product_id',)
        fields = ('name', 'code_deck_id', 'defined_by', 'routed_by')


class ProductModifyScheme(ProductScheme):
    name = Str()

    class Meta:
        model = model.Product
        # exclude = ('product_id',)
        fields = ('name', 'code_deck_id', 'defined_by', 'routed_by')


class ProductGetScheme(BaseModelScheme):
    name = Str()
    defined_by = Choice()
    routed_by = Choice()
    code_deck_id = Int()
    code_deck_name = Str()
    static_route_id = Int()  # attribute='product_id',load_from='static_route_id',dump_to='static_route_id')
    route_count = Int()
    prefix_count = Int()
    update_at = DateTime()
    update_by = Str()
    has_trunk_id = Int()

    class Meta:
        model = model.Product
        fields = ('static_route_id', 'name', 'code_deck_id', 'code_deck_name', 'defined_by',
                  'routed_by', 'route_count', 'prefix_count', 'update_at', 'update_by')
        search_fields = (
            'static_route_id', 'name', 'code_deck_id', 'code_deck_name', 'defined_by',
            'routed_by', 'update_by', 'has_trunk_id'
        )
        query_fields = ('update_at_gt', 'update_at_lt', 'static_route_id_in')


# +++ ProductItemResource
class StaticRouteItemInnerScheme(Schema):
    prefix = Str(validate=validate.Regexp(PREFIX_REGEXP))
    static_route_id = Int(
        validate=lambda value: _valid('Product', 'product_id', value, 'invalid static_route_id {}'.format(value)))


class ProductsItemsResourceScheme(BaseModelScheme):
    trunk_id = Int(validate=lambda value: _valid('Resource', 'resource_id', value))
    by_percentage = Int(allow_none=True, required=False, validate=validate.Range(0, 100))

    class Meta:
        model = model.ProductItemsResource
        # exclude = ('item_id', 'trunks')
        fields = ('trunk_id', 'by_percentage')


class ProductsItemsResourceGetScheme(BaseModelScheme):
    trunk_id = Int()
    by_percentage = Int()
    carrier_name = Str()
    trunk_name = Str()

    class Meta:
        model = model.ProductItemsResource
        # exclude = ('item_id', 'trunks')
        fields = ('trunk_id', 'trunk_name', 'carrier_name', 'by_percentage')


# +++ ProductItem
class ProductItemsCopyScheme(BaseModelScheme):
    class Meta:
        model = model.ProductItems
        fields = ('code_name',)


class ProductItemsScheme(BaseModelScheme):
    # name=Str()
    strategy = Choice(validate=validate.OneOf(model.ProductItems.STRATEGY_CHOICES.values()),
                      default=tuple(model.ProductItems.STRATEGY_CHOICES.values())[0])
    time_profile_id = Int(allow_none=True, validate=lambda value: _valid('TimeProfile', 'time_profile_id', value))
    trunks = Nested(ProductsItemsResourceScheme, many=True, allow_none=False, required=True)
    static_route_id = Int()
    code_name = Str(validate=validate.Regexp(USERNAME_REGEXP))

    @pre_load
    def check_prefix(self, data):
        if 'trunks' in data and 'strategy' in data and data['strategy'] == 'By Percentage':
            perc = 0
            for t in data['trunks']:
                if 'by_percentage' in t:
                    if t['by_percentage'] == 0:
                        raise ValidationError(
                            {'trunks': ['Percentage of trunk must be set if strategy is "By Persentage"!']})
                    perc = perc + t['by_percentage']
                else:
                    raise ValidationError(
                        {'trunks': ['Percentage of trunk must be set if strategy is "By Persentage"!']})
            if perc > 100:
                raise ValidationError({'trunks': ['Percentage of trunks more than 100!']})
        return data

    class Meta:
        model = model.ProductItems
        exclude = ('item_id', 'static_route_id', 'product_id', 'product', 'time_profile',
                   'update_at', 'update_by', 'did', 'static_route_name', 'time_profile_name')


# fields = ('name','introduction',)


class ProductItemsModifyScheme(ProductItemsScheme):
    # name=Str()
    static_route_name = Str(attribute='_static_route_name')
    strategy = Choice()
    trunks = Nested(ProductsItemsResourceScheme, many=True)

    class Meta:
        model = model.ProductItems
        exclude = ('item_id', 'static_route_id', 'product', 'update_at', 'time_profile', 'update_by', 'did')


class ProductItemsGetScheme(ProductItemsScheme):
    product = Str(attribute='product_name')
    did = Str()
    trunks = Nested(ProductsItemsResourceGetScheme, many=True)
    time_profile_id = Int()
    time_profile_name = Str()
    static_route_id = Int()
    has_trunk_id = Int()
    static_route_name = Str()
    trunk_name = Str()

    # carrier_id = Int()
    class Meta:
        model = model.ProductItems
        exclude = ('time_profile',)
        search_fields = ('static_route_id', 'static_route_name', 'item_id', 'did', 'strategy', 'time_profile_name',
                         'update_by', 'alias', 'code_name', 'has_trunk_id', 'trunk_name')
        query_fields = ('item_id_in', 'update_at_gt', 'update_at_lt')


class ProductItemsForDidGetScheme(BaseModelScheme):
    did = Str()
    trunks = Nested(ProductsItemsResourceGetScheme, many=True)

    class Meta:
        model = model.ProductItems
        fields = ('item_id', 'did', 'trunks', 'update_at')
        search_fields = ('item_id', 'did')
        query_fields = ('update_at_gt', 'update_at_lt')


# ---


class TrunkGroupScheme(BaseModelScheme):
    group_name = Str(allow_none=False, required=True, validate=[
        lambda value: _valid_unique('TrunkGroup', 'group_name', value, 'Trunk group {} already exists!'.format(value)),
        validate.Regexp(NAME_REGEXP)])
    trunk_type = Choice()
    all_trunks = List(Int(validate=lambda value: _valid('Resource', 'resource_id', value, 'invalid trunk id')))

    class Meta:
        model = model.TrunkGroup
        fields = ('group_name', 'trunk_type', 'all_trunks')


# exclude = ('group_id','trunk_type' )


class TrunkGroupModifyScheme(BaseModelScheme):
    group_name = Str(allow_none=False, required=True, validate=validate.Regexp(NAME_REGEXP))
    trunk_type = Choice()
    all_trunks = List(Int())

    @post_load
    def check_trunks(self, obj):
        if 'all_trunks' in obj:
            if len(obj['all_trunks']) != len(list(set(obj['all_trunks']))):
                raise ValidationError('Duplicates in all_trunks!')
            for t in obj['all_trunks']:
                r = model.Resource.get(t)
                if not r:
                    raise ValidationError('Trunk {} not exists!'.format(t))
                if r.direction != obj['trunk_type']:
                    raise ValidationError('Trunk {} alias {} is not {}!'.format(t, r.alias, obj['trunk_type']))

    class Meta:
        model = model.TrunkGroup
        fields = ('group_name', 'trunk_type', 'all_trunks')


class TrunkGroupGetScheme(TrunkGroupScheme):
    trunks = Nested('ResourceInfoGetScheme', many=True)
    trunk_type = Choice()
    group_id = Int()
    group_name = Str()
    trunks_count = Int()

    class Meta:
        model = model.TrunkGroup
        fields = ('group_id', 'group_name', 'trunk_type', 'trunks', 'trunks_count')
        search_fields = ('group_id', 'group_name', 'trunk_type', 'trunks_count')
        query_fields = ('group_id_in')


class TrunkGroupAddTrunkScheme(BaseModelScheme):
    trunks = List(Int())

    class Meta:
        model = model.TrunkGroup
        fields = ('add_trunks',)


class ClientTaxesScheme(BaseModelScheme):
    tax_name = Str(validate=[validate.Regexp(NAME_REGEXP), validate.Length(max=255)])
    tax_percent = Float(validate=validate.Range(0, 100))

    class Meta:
        model = model.ClientTaxes
        fields = ('tax_name', 'tax_percent')


# Client view
class ClientScheme(BaseModelScheme):
    """
    carrier_name = field_for(ClientModel,'name',default='Acme')
    is_prepay = field_for(ClientModel,'mode')
    payment_term = field_for(ClientModel,'payment_term')
    credit_limit = field_for(ClientModel,'allowed_credit',default='0.0')
    enable_auto_invoice = field_for(ClientModel,'auto_invoicing')
    enable_low_balance = field_for(ClientModel,'notify_client_balance')
    invoice_format = field_for(ClientModel,'invoice_format')
    company_name = field_for(ClientModel,'company')
    address = field_for(ClientModel,'address')
    email = field_for(ClientModel,'email',validate=validate.Email(error='Incorrect email address'))
    username = field_for(ClientModel,'login')
    password = field_for(ClientModel,'password')
    """
    carrier_name = Str(attribute='name', load_from='carrier_name', dump_to='carrier_name',
                       validate=validate.Regexp(NAME_REGEXP))
    is_prepay = Bool(attribute='mode', load_from='is_prepay', dump_to='is_prepay')
    payment_term = Int(attribute='payment_term_id', load_from='payment_term', dump_to='payment_term')
    credit_limit = Decimal(allow_none=True,
                           required=False)  # Float(attribute='allowed_credit',load_from='credit_limit',dump_to='credit_limit')
    enable_auto_invoice = Bool(attribute='auto_invoicing', load_from='enable_auto_invoice',
                               dump_to='enable_auto_invoice')
    enable_low_balance = Bool(attribute='low_balance_notice', load_from='enable_low_balance',
                              dump_to='enable_low_balance')
    invoice_format = Choice()
    company_name = Str(attribute='company', load_from='company_name', dump_to='company_name')
    address = Str(attribute='address', load_from='address', dump_to='address')
    email = Emails(attribute='email', load_from='email', dump_to='email')
    username = Str(attribute='login', load_from='username', dump_to='username', validate=validate.Regexp(NAME_REGEXP))
    password = Str(attribute='password', load_from='password', dump_to='password')

    # email = Str(validate=validate.Email(error='Incorrect email address'))
    # currency_id = Int(required=True,default=1)
    # group_id = Int()
    # payment_term_id = Int()

    # allowed_credit = Float()
    # notify_client_balance = Float()
    # notify_admin_balance = Float()
    # invoice_past_amount = Float()
    # scc_charge = Float()
    # tax = Float()
    # Smart hyperlinking
    #    _links = ma.Hyperlinks({
    #        'self': ma.URLFor('client', id='<id>'),
    #        'collection': ma.URLFor('group')
    #    })

    class Meta:
        model = model.Client
        fields = (
            'carrier_name', 'is_prepay', 'payment_term', 'credit_limit', 'enable_auto_invoice',
            'enable_low_balance', 'invoice_format', 'company_name', 'address', 'email', 'username', 'password'
        )


#        exclude = (
#            'client_id', 'user_id', 'create_time', 'update_by', 'update_at', 'group', 'currency', 'payment_term'
#        )

class PastDueScheme(Schema):
    total = Decimal()
    past_7 = Decimal()
    past_15 = Decimal()
    past_30 = Decimal()
    past_gt_30 = Decimal()
    balance = Decimal()


class ClientGetPastDueScheme(BaseModelScheme):
    past_due = Nested(PastDueScheme)
    over_due = Decimal()

    class Meta:
        model = model.Client
        fields = ('client_id', 'name', 'past_due', 'over_due')
        search_fields = ('client_id', 'name')
        query_fields = ('over_due_gt', 'over_due_isnull')


"""
class ClientGetScheme(BaseModelScheme):
    currency = Nested(CurrencyGetScheme)
    group = Nested(CarrierGroupGetScheme)
    payment_term = Nested(PaymentTermGetScheme)

    class Meta:
        model = model.Client
        exclude = ('currency_id', 'group_id', 'payment_term_id')

"""


class CodeSchemeImport(BaseModelScheme):
    code_deck_id = Int(validate=lambda value: _valid('CodeDeck', 'code_deck_id', value))
    code_name = Str()
    code = Str(required=True)
    state = Str()
    country = Str(reqiered=True)
    city = Str()

    class Meta:
        model = model.Code
        fields = ('code_name', 'code', 'state', 'country', 'city', 'code_deck_id', 'name')


class CodeScheme(BaseModelScheme):
    code_deck_id = Int(validate=lambda value: _valid('CodeDeck', 'code_deck_id', value))
    code_name = Str()
    code = Str()
    state = Str()
    country = Str()
    city = Str()

    class Meta:
        model = model.Code
        fields = ('code_name', 'code', 'state', 'country', 'city')


class CodeCountryScheme(BaseModelScheme):
    country_code = Str(validate=[validate.Length(max=20), validate.Regexp(DIGITS_REGEXP)])
    country = Str(validate=validate.Length(max=20))
    phone_code = Str()

    class Meta:
        model = model.CodeCountry
        fields = ('country_code', 'country', 'phone_code')
        search_fields = ('country_code', 'country', 'phone_code')


class CodeGetScheme(BaseModelScheme):
    # client = Nested(client.ClientSchemeGet)
    # code_id = Int()
    code_name = Str(validate=validate.Regexp(NAME_REGEXP))

    class Meta:
        model = model.Code
        fields = ('code_name', 'code', 'state', 'country', 'city')
        search_fields = ('code_name', 'code', 'state', 'country', 'city')
        query_fileds = ('code_id_in', 'code_in')


# exclude = ('client_id', )


class CodeAllScheme(BaseModelScheme):
    code_deck_id = Int(allow_none=False, required=True,
                       validate=lambda value: _valid('CodeDeck', 'code_deck_id', value, 'invalid code_deck_id'))
    name = Str(validate=validate.Regexp(NAME_REGEXP))
    code = Str(allow_none=False, required=True, validate=validate.Regexp(DIGITS_REGEXP))
    state = Str()
    country = Str()
    city = Str()

    class Meta:
        model = model.Code
        fields = ('code_deck_id', 'name', 'code', 'state', 'country', 'city')
        search_fields = ('code_deck_id', 'name', 'code', 'state', 'country', 'city')


class CodeAllGetScheme(CodeAllScheme):
    code_id = Int()

    class Meta:
        model = model.Code
        fields = ('code_id', 'code_deck_id', 'name', 'code', 'state', 'country', 'city')
        search_fields = ('code_id', 'code_deck_id', 'name', 'code', 'state', 'country', 'city')


class CodeDestinationGetScheme(CodeAllScheme):
    destination = Str()
    country = Str()
    code = Str()
    codes_count = Int()

    class Meta:
        model = model.Code
        fields = ('destination', 'country', 'code', 'codes_count')
        search_fields = ('destination', 'country', 'code')


class CodeDeckScheme(BaseModelScheme):
    # client_id = Int(required=True)
    name = Str(validate=validate.Regexp(NAME_REGEXP))

    # codes = Nested(CodeScheme,many=True)
    class Meta:
        model = model.CodeDeck
        fields = ('name',)


class CodeDeckGetScheme(BaseModelScheme):
    # client = Nested(client.ClientSchemeGet)
    code_deck_id = Int()
    name = Str()
    # codes = Nested(CodeGetScheme,many=True)
    usage_count = Int()
    code_count = Int()
    update_by = Str()
    update_on = DateTime()

    class Meta:
        model = model.CodeDeck
        fields = ('code_deck_id', 'name', 'usage_count', 'code_count', 'update_on', 'update_by')
        # exclude = ('client_id', )
        search_fields = ('code_deck_id', 'name', 'usage_count', 'code_count', 'update_by')
        query_fields = ('update_on_gte', 'update_on_lt', 'code_deck_id_in')


class RateScheme(BaseModelScheme):
    # gen module  rate
    # --- Rate ---
    effective_date = Str(default=str(datetime.now(UTC).date()))
    inter_rate = Float(allow_none=True, validate=validate.Range(-10000.0, 10000.0))
    rate = Decimal(required=True, validate=validate.Range(-10000.0, 10000.0))
    code_name = Str(required=False, allow_none=True, validate=[validate.Length(max=100)])
    country = Str(validate=validate.Length(max=1000))
    setup_fee = Float(default='0.0', validate=validate.Range(0.0, 10000.0), allow_none=True)
    intra_rate = Float(allow_none=True, validate=validate.Range(-10000.0, 10000.0))
    end_date = Str(default=str(datetime.now(UTC).date()), allow_none=True)
    code = Str(validate=[validate.Length(max=32), validate.Regexp(PREFIX_REGEXP)])
    interval = Int(default='1', validate=validate.Range(1, 1000000))
    min_time = Int(default='60', validate=validate.Range(1, 1000000))
    time_profile_id = Int(allow_none=True, validate=lambda value: _valid('TimeProfile', 'time_profile_id', value))
    gmt = Str(validate=validate.Regexp(TIMEZONE_REGEXP))
    local_rate = Float(allow_none=True, validate=validate.Range(-10000.0, 10000.0))

    profile = Int(validate=lambda value: _valid('TimeProfile', 'time_profile_id', value))  # ???

    @pre_load
    def check_dates(self, data):
        if 'effective_date' in data and 'end_date' in data:
            try:
                d0 = parse_datetime(data['effective_date'])
                d1 = None
                if data['end_date'] == 'null':
                    data['end_date'] = None
                if data['end_date'] is not None:
                    d1 = parse_datetime(data['end_date'])
            except:
                raise ValidationError({'effective_date': ['Invalid date']})
            if d1 and d1 < d0:
                raise ValidationError({'end_date': ['Invalid end_date must be after effective_date']})
        return data

    class Meta:
        model = model.Rate
        fields = ('effective_date', 'inter_rate', 'rate', 'code_name', 'country',
                  'setup_fee', 'interval', 'intra_rate', 'end_date', 'code',
                  'min_time', 'time_profile_id', 'gmt', 'local_rate')


class RateModifyScheme(RateScheme):
    rate_table_id = Int(required=True, validate=lambda value: _valid('RateTable', 'rate_table_id', value))

    class Meta:
        model = model.Rate
        fields = ('rate_table_id', 'effective_date', 'inter_rate', 'rate', 'code_name', 'country',
                  'setup_fee', 'interval', 'intra_rate', 'end_date', 'code', 'min_time', 'time_profile_id',
                  'gmt', 'local_rate')


class RateModify2Scheme(RateScheme):
    class Meta:
        model = model.Rate
        fields = ('rate_id',) + RateModifyScheme.Meta.fields


class RateGetScheme(RateScheme):
    # rate_table = Nested(rate_table.RateTableSchemeGet)
    rate_table_id = Int()
    rate_id = Int()
    is_effective_now = Bool()
    search_type = String(validate=validate.OneOf(choices=('Exact Match', 'Start With')))#, 'Match Any')))
    # new_rate = Decimal()
    change_status = Str(validate=validate.OneOf(('increased', 'decreased', 'unchanged', 'none')))
    effective_date = Str()
    interval = Int()
    min_time = Int()
    # indeterminte = Decimal()
    indet_rate = Decimal()

    class Meta:
        model = model.Rate
        # search_fields=('rate_table_name', 'code_deck_id', 'currency_id', 'rate', 'billing_type')
        fields = ('rate_id', 'rate_table_id', 'effective_date', 'inter_rate', 'rate', 'code_name', 'country',
                  'setup_fee', 'interval', 'intra_rate', 'end_date', 'code', 'min_time', 'time_profile_id', 'gmt',
                  'local_rate', 'indet_rate', 'ij_rate', 'search_type',
                  'is_effective_now', 'change_status')
        search_fields = ('rate_table_id', 'code', 'code_name', 'country', 'time_profile_id', 'gmt', 'is_effective_now',
                         'change_status', 'effective_date', 'min_time', 'interval', 'end_date')
        query_fields = (
            'rate_id_in', 'effective_date_gt', 'effective_date_ld', 'inter_rate_gt', 'inter_rate_lt', 'local_rate_gt',
            'intra_rate_gt', 'intra_rate_lt', 'rate_gt', 'rate_lt', 'setup_fee_gt', 'end_date_lt', 'end_date_gt')


class RateSendToClientsScheme(RateScheme):
    clients = List(Int(validate=lambda value: _valid('Client', 'client_id', value)))

    class Meta:
        model = model.Rate
        fields = ('clients',)


class RateDownloadScheme(RateGetScheme):
    class Meta:
        model = model.Rate
        # search_fields=('rate_table_name', 'code_deck_id', 'currency_id', 'rate', 'billing_type')
        fields = ('effective_date', 'inter_rate', 'rate', 'code_name', 'country',
                  'interval', 'intra_rate', 'end_date', 'code', 'min_time', 'gmt',
                  'local_rate', 'change_status')


class VerifyTokenHeaderScheme(Schema):
    token_header = Str(required=True)


# +++ RateMassEdit

class RMEOpDecimalScheme(Schema):
    operation = Str(validate=validate.OneOf(('No Change', 'set to', 'increase by', 'decrease by', 'multiple by')),
                    required=True, default='No Change')
    value = Decimal(allow_none=True)


class RMEOpIntScheme(Schema):
    operation = Str(validate=validate.OneOf(('No Change', 'set to')), required=True, default='No Change')
    value = Int(allow_none=True)


class RMEOpDateScheme(Schema):
    operation = Str(validate=validate.OneOf(('No Change', 'set to')), required=True, default='No Change')
    value = DateTime(allow_none=True)


class RMEOpProfileScheme(Schema):
    operation = Str(validate=validate.OneOf(('No Change', 'set to')), required=True, default='No Change')
    value = Int(allow_none=True, validate=lambda value: _valid('TimeProfile', 'time_profile_id', value))


class RateMassEditScheme(Schema):
    rate_id_list = List(Int(validate=lambda value: _valid('Rate', 'rate_id', value)), allow_none=True)
    action_type = Str(validate=validate.OneOf(('update current rates', 'delete found rates', 'insert as new rates',
                                               'delete all rates', 'delete all future rates', 'update all rates',
                                               'update future rates with specific effective date',
                                               'delete future rates with specific effective date')), required=True, allow_none=False,
                      default='update current rates')


    filter_effective_date = DateTime(allow_none=True)
    rate = Nested(RMEOpDecimalScheme)
    inter_rate = Nested(RMEOpDecimalScheme)
    intra_rate = Nested(RMEOpDecimalScheme)
    setup_fee = Nested(RMEOpDecimalScheme)
    min_time = Nested(RMEOpIntScheme)
    interval = Nested(RMEOpIntScheme)
    effective_date = Nested(RMEOpDateScheme)
    end_date = Nested(RMEOpDateScheme)

    @pre_load
    def check(self, data):
        return data

    class Meta:
        model = model.Rate
        fields = ('rate_id_list', 'action_type', 'rate', 'inter_rate', 'intra_rate',
                  'setup_fee', 'filter_effective_date',
                  'min_time', 'interval', 'effective_date', 'end_date')


# --- RateMassEdit


# +++ RateTableAssignTrunks
class RateTableAssignTrunksInnerScheme(BaseModelScheme):
    prefix = Str(validate=validate.Regexp(PREFIX_REGEXP))
    resource_id = Int(
        validate=lambda value: _valid('Resource', 'resource_id', value, 'Invalid resource_id {}'.format(value)))

    class Meta:
        model = model.ResourcePrefix
        fields = ('prefix', 'resource_id')


class RateTableAssignTrunksScheme(BaseModelScheme):
    trunks = Nested('RateTableAssignTrunksInnerScheme', many=True)

    class Meta:
        model = model.RateTable
        fields = ('trunks',)


# --- RateTableAssignTrunks


class RateTableCodeNameScheme(BaseModelScheme):
    country = Str()
    code_name = Str()

    class Meta:
        model = model.RateTable
        search_fields = ('country', 'code_name')
        fields = ('country', 'code_name')


class RateTableEffectiveDateScheme(BaseModelScheme):
    earliest_effective_date = Int()
    last_effective_date = Int()

    class Meta:
        model = model.RateTable
        fields = ('earliest_effective_date', 'last_effective_date')


class RateManyScheme(BaseModelScheme):
    rates = List(Nested(RateScheme))

    class Meta:
        model = model.RateTable
        fields = ('rates',)

    @pre_load
    def check_duplicates(self, data):
        d = dict()
        for i in data['rates']:
            k = (i.get('code'), i.get('effective_date'))
            if k in d:
                raise ValidationError('can not create rates with same effective date and code {}'.format(k))
            else:
                d[k] = 0
        return data


class RateTableEmptyScheme(BaseModelScheme):
    code_deck_id = Int(allow_none=True)
    currency_id = Int(default=1)
    billing_method = Choice()
    rate_type_name = Choice()
    code_count = Int(dump_only=True)

    class Meta:
        model = model.RateTable
        exclude = ('rate_table_id', 'code_deck', 'currency', 'modify_time', 'create_time', 'update_at', 'update_by')


class RateTableScheme(BaseModelScheme):
    name = Str(validate=[validate.Length(1, 100), validate.Regexp(USERNAME_REGEXP)])
    code_deck_id = Int(allow_none=True)
    currency_id = Int(allow_none=True)
    jurisdiction_country_id = Int(allow_none=True)
    billing_method = Choice(default="DNIS")
    rate_type_name = Choice(default="A-Z")
    require_setup_fee = Bool()
    default_setup_fee = Float()
    require_local_rate = Bool()
    origination = Bool()

    class Meta:
        model = model.RateTable
        fields = ('name', 'code_deck_id', 'currency_id', 'rate_type_name', 'billing_method', 'jurisdiction_country_id',
                  'require_setup_fee', 'default_setup_fee', 'require_local_rate', 'origination')


class RateTableGetScheme(RateTableScheme):
    # code_deck = Nested(CodeDeckGetScheme,many=False)
    # currency = Nested(CurrencyGetScheme,many=False)
    # rates = Nested(RateGetScheme,many=True)

    billing_method = Str()  # serialize='get_billing',deserialize='set_billing',
    #        attribute='billing_method',
    #                                  validate=validate.OneOf(model.RateTable.rate_type_dict.values() ) )
    rate_type_name = Choice()  # serialize='get_type',deserialize='set_type',
    #                                 validate=validate.OneOf(model.RateTable.jur_type_dict.values()) )

    currency_name = Str(dump_only=True)
    code_deck_name = Str(dump_only=True)
    # code_count = Int(dump_only=True)
    ingress_count = Int(dump_only=True)
    egress_count = Int(dump_only=True)
    orig_count = Int(dump_only=True)
    trunks = List(Int(), dump_only=True)
    has_billing_rule = Bool()

    class Meta:
        model = model.RateTable
        fields = ('rate_table_id', 'name', 'code_deck_id', 'code_deck_name', 'currency_id', 'currency_name',
                  'billing_method', 'rate_type_name', 'update_at', 'update_by', 'code_count', 'trunks',
                  'require_setup_fee', 'default_setup_fee', 'require_local_rate',
                  'jurisdiction_country_id', 'ingress_count', 'egress_count', 'orig_count', 'origination')
        search_fields = ('rate_table_id', 'name', 'code_deck_name', 'currency_name', 'billing_method', 'rate_type_name',
                         'code_deck_id', 'require_setup_fee', 'default_setup_fee',
                         'require_local_rate', 'jurisdiction_country_id', 'origination',
                         'ingress_count', 'egress_count', 'orig_count', 'has_billing_rule')
        query_fields = ('update_at_gt', 'update_at_lt', 'code_count_gt',
                        'rate_table_id_in')


class RateTableCopyScheme(BaseModelScheme):
    name = Str(validate=[lambda value: _valid_unique('RateTable', 'name', value), validate.Regexp(NAME_REGEXP)])

    class Meta:
        model = model.RateTable
        fields = ('name',)


# ---RateTable

# +++RateGeneration
class RateGenerationTemplateDetailScheme(BaseModelScheme):
    class Meta:
        model = model.RateGenerationTemplateDetail
        exclude = ('template',)


class RateGenerationTemplateMarginScheme(BaseModelScheme):
    min_rate = Decimal()
    max_rate = Decimal()
    markup_type = Choice()
    markup_value = Decimal()

    class Meta:
        model = model.RateGenerationTemplateMargin
        exclude = ('id', 'rate_generation_template_id', 'template')


class RateGenerationCodeDeckScheme(BaseModelScheme):
    class Meta:
        model = model.RateGenerationCodeDeck
        exclude = ('id', 'rate_generation_template_id', 'template')


def valid_egress_for_generate(value):
    q = model.Resource.get(value)
    if q:
        rt = q.rate_table_id
        if rt:
            return True
        else:
            raise ValidationError(
                'Egress trunk {} has no rate_table, it must be set for rate generation!'.format(q.alias))
    else:
        raise ValidationError('Egress trunk {} not exists!'.format(value))


class RateGenerationTemplateScheme(BaseModelScheme):
    details = Nested('RateGenerationTemplateDetailScheme', many=True)
    margins = Nested('RateGenerationTemplateMarginScheme', many=True)
    # user_code_deck = Nested('RateGenerationCodeDeckScheme',many=True)
    default_rate = Decimal()
    margin_default_value = Str(allow_none=False, required=True)
    rate_table_type = Int(load_from='_rate_table_type')
    calculate_ij_based_on = Str(validate=validate.OneOf(('IJ - Inter', 'IJ - Intra', 'IJ - Max(Inter,Intra)', 'IJ - True Match')))
    margin_default_type = Choice()
    egress_trunks = List(Int(validate=valid_egress_for_generate))
    code_deck_id = Int(validate=[lambda value: _valid('CodeDeck', 'code_deck_id', value)], allow_none=True, required=False)
    min_asr = Float()
    asr_days = Int()
    ignore_zero_rates = Bool()
    user_defined_code_deck = Bool()

    # egress_trunks = List(Int(validate=[lambda value:_valid('EgressTrunk','resource_id',value),lambda value: model.Resource.get(value).rate_table]))

    class Meta:
        model = model.RateGenerationTemplate
        exclude = (
            'id', 'create_by', 'create_on', 'user_code_deck', 'egress_str', 'last_generated',
            'egress_trunks_names')


class RateGenerationTemplateGetScheme(RateGenerationTemplateScheme):
    rate_table_type = Int(attribute='_rate_table_type')
    egress_trunks_names = List(Str())

    class Meta:
        model = model.RateGenerationTemplate
        exclude = ('user_code_deck', 'egress_str')
        search_fields = ('create_by', 'name')
        query_fields = (
            'create_on_gt', 'create_on_lt', 'last_generated_gt', 'last_generated_lt', 'id_in', 'egress_trunks_names')


class RateGenerationHistoryScheme(BaseModelScheme):
    status = Choice()

    class Meta:
        model = model.RateGenerationHistory
        fields = ('status',)
        
class RateGenerationWhatIfScheme(BaseModelScheme):
    code = Str()
    code_name = Str()
    country = Str()
    current_rate = Float(allow_none=True)
    current_effective_date = DateTime(allow_none=True)
    new_rate = Float()
    new_effective_date = DateTime()
    status = Str()

    class Meta:
        fields = ('code', 'code_name', 'country', 'current_rate', 'current_effective_date', 
                 'new_rate', 'new_effective_date', 'status')

class RateGenerationHistoryGetScheme(BaseModelScheme):
    id = Int()
    is_applied = Bool(attribute='_is_applied')
    rate_generation_template_id = Int()
    status = Choice()
    finished_time = DateTime()
    rate_count = Int()
    processing_time = Int()
    progress = Str()
    create_on = DateTime()
    create_by = Str()
    rate_table_type = Int()
    rate_table_name = Str()

    class Meta:
        model = model.RateGenerationHistory
        fields = (
            'id', 'is_applied', 'rate_generation_template_id', 'status', 'finished_time', 'rate_count',
            'processing_time',
            'progress', 'create_on', 'create_by', 'rate_table_type', 'rate_table_name')
        search_fields = ('id', 'is_applied', 'rate_generation_template_id', 'status', 'create_by', 'rate_table_name')
        query_fields = ('create_on_gt', 'create_on_lt', 'finished_time_gt', 'finished_time_lt',)


# ---RateGeneration
class RateGenerationRateImportScheme(BaseModelScheme):
    class Meta:
        model = model.RateGenerationRate
        fields = (
            'generation_rate_id', 'rate_generation_history_id', 'effective_date', 'inter_rate', 'rate', 'code_name',
            'country',
            'setup_fee', 'interval', 'intra_rate', 'end_date', 'code', 'min_time', 'time_profile_id', 'local_rate')


class RateGenerationRateGetScheme(BaseModelScheme):
    rate_generation_history_id = Int()
    code = Str()
    rate = Float()
    setup_fee = Float()
    effective_date = DateTime()
    end_date = DateTime()
    interval = Int(default='1', validate=validate.Range(1, 1000000))
    min_time = Int(default='60', validate=validate.Range(1, 1000000))
    grace_time = Int()
    time_profile_id = Int()
    seconds = Int()
    code_name = Str()
    rate_type = Int()
    intra_rate = Float()
    inter_rate = Float()
    local_rate = Float()
    country = Str()
    zone = Str(validate=validate.Regexp(TIMEZONE_REGEXP))
    ocn = Str()
    lata = Str()
    lcr_rate = Str()
    lcr_intra_rate = Str()
    lcr_inter_rate = Str()
    lcr_local_rate = Str()

    class Meta:
        model = model.RateGenerationRate
        exclude = ('rate_generation_history',)
        # fields = ('')
        search_fields = ('generation_rate_id', 'code', 'code_name', 'country', 'time_profile_id', 'create_by')
        query_fields = ('create_on_gt', 'create_on_lt', 'finished_time_gt', 'finished_time_lt', 'generation_rate_id_in',
                        'effective_date_gt', 'effective_date_ld', 'inter_rate_gt', 'inter_rate_lt',
                        'intra_rate_gt', 'intra_rate_lt', 'rate_gt', 'rate_lt')


class RateGenerationApplyRateScheme(BaseModelScheme):
    rate_table_name = Str(required=False, validate=[
        # lambda value: _valid_unique('RateTable','name',value),
        validate.Regexp(NAME_REGEXP)])
    action = Str(required=True,
                 validate=validate.OneOf(
                     ('delete old rates', 'end date all records', 'end date existing records', 'none')))
    effective_date = Str(validate=validate.Regexp(DATE_REGEXP))
    code_deck_id = Int(validate=[lambda value: _valid('CodeDeck', 'code_deck_id', value)])
    end_date = Str(required=False, allow_none=True, validate=validate.Regexp(DATE_REGEXP))
    email_template_id = Int(allow_none=True, validate=lambda value: _valid('SendRateTemplate', 'id', value))
    rate_table_id = Int()
    interval = Int()
    min_time = Int()

    @pre_load
    def check_dates(self, data):
        if 'end' in data['action']:
            if not 'end_date' in data:
                raise ValidationError({'end_date': ['Must present on choosen action']})
            else:
                d = parse_datetime(data['end_date']).date()
                if d < datetime.now(UTC).date():
                    raise ValidationError({'end_date': ['End date cannot be past']})
                if 'effective_date' in data:
                    de = parse_datetime(data['effective_date']).date()
                    if de < d:
                        raise ValidationError({'effective_date': [
                            'Effective date for new records must be greater than end_date of old records!']})
        if 'effective_date' in data:
            d = parse_datetime(data['effective_date']).date()
            if d < (datetime.now(UTC) - timedelta(days=365)).date():
                raise ValidationError({'effective_date': ['Effective date date cannot be past']})
        return data

    class Meta:
        model = model.RateGenerationHistory
        fields = ('rate_table_name', 'rate_table_id', 'action', 'effective_date', 'end_date',
                  'email_template_id', 'code_deck_id', 'min_time', 'interval')


class RateGenerationHistoryDetailGetScheme(BaseModelScheme):
    id = Int()
    rate_table_name = Str()
    rate_table_id = Int()
    effective_date_new = DateTime()
    effective_date_increase = DateTime()
    effective_date_decrease = DateTime()
    end_date = Str()
    is_send_mail = Bool()
    create_on = DateTime()
    create_by = Str()
    finished_time = DateTime()
    end_date_method = Choice()
    progress = Str()
    rate_upload_task_id = Int()
    override_records = Bool()

    rate_generation_history_id = Int()
    email_template = Str(attribute='email_template_id')

    class Meta:
        model = model.RateGenerationHistoryDetail
        fields = ('id', 'rate_table_name', 'rate_table_id', 'effective_date_new', 'effective_date_increase',
                  'effective_date_decrease', 'end_date', 'is_send_mail', 'create_on', 'create_by', 'count',
                  'finished_time', 'end_date_method', 'rate_generation_history_id', 'progress', 'rate_upload_task_id',
                  'override_records', 'email_template')
        search_fields = ('id', 'rate_table_name', 'rate_table_id', 'is_send_mail', 'create_by', 'end_date_method',
                         'rate_generation_history_id', 'progress', 'rate_upload_task_id', 'email_template')
        query_fields = ('effective_date_new_gt', 'effective_date_new_lt', 'end_date_gt', 'end_date_lt',
                        'create_on_gt', 'create_on_lt', 'finished_time_gt', 'finished_time_lt')


# +++ RateSend
def valid_download_deadline(d):
    if valid_date(d):
        if parse_datetime(d).date() >= datetime.now(UTC).date():
            return True
        else:
            raise ValidationError('Date cannot be in past!')


def valid_effective_date(d):
    if valid_date(d):
        if parse_datetime(d).date() >= (datetime.now(UTC) - timedelta(days=365)).date():
            return True
        else:
            raise ValidationError('Date cannot be in past!')


class RateSendScheme(BaseModelScheme):
    rate_table_id = Int()
    format = Choice()
    download_deadline = Str(
        validate=[validate.Regexp(DATE_REGEXP, 0, 'Wrong deadline date format'), valid_download_deadline])
    effective_date = Str(
        validate=[validate.Regexp(DATE_REGEXP, 0, 'Wrong effective date format'), valid_effective_date])
    start_effective_date = Str(
        validate=[validate.Regexp(DATE_REGEXP, 0, 'Wrong effective date format')])
    is_email_alert = Bool()
    is_disable = Bool()
    trunks = List(Int(validate=lambda value: _valid('Resource', 'resource_id', value)))
    carriers = List(Int(validate=lambda value: _valid('Client', 'client_id', value)))
    send_type = Choice()
    download_method = Choice()
    send_specify_email = Emails()
    email_template_id = Int(allow_none=True, default=None,
                            validate=lambda value: _valid('SendRateTemplate', 'id', value))
    zip = Bool()
    email_direct = Nested('SendRateDirectScheme', many=False)
    sent_area = Choice()
    date_format = Choice()
    indicate_inc_dec = Bool()

    @post_load
    def bool_to_int(self, data):
        d = {True: 1, False: 0}
        for attr in ['zip']:
            if attr in data:
                v = data[attr]
                if v in d:
                    data[attr] = d[v]
        return data

    class Meta:
        model = model.RateSendLog
        fields = ('rate_table_id', 'format', 'download_deadline', 'effective_date',
                  'is_email_alert', 'is_disable', 'trunks', 'send_type',  # 'download_method',
                  'send_specify_email', 'email_template_id', 'zip', 'email_direct', 'sent_area', 'date_format',
                  'indicate_inc_dec', 'carriers', 'start_effective_date')


class RateSendCopyScheme(BaseModelScheme):
    download_deadline = Str(
        validate=[validate.Regexp(DATE_REGEXP, 0, 'Wrong deadline date format'), valid_download_deadline])
    send_specify_email = Emails()

    class Meta:
        model = model.RateSendLog
        fields = ('download_deadline', 'send_specify_email')
        exclude = ('rate_table_id', 'status', 'format')


class RateSendGetScheme(RateSendScheme):
    status = Choice()
    error = Str()
    create_time = DateTime()
    total_records = Int()
    completed_records = Int()
    file = Str()
    job_id = Int()
    send_type = Choice()
    email_direct = Nested('SendRateDirectGetScheme', many=False)
    email_count = Int()
    recipient_count = Int()
    rate_table_name = Str()
    process = Float()
    date_format = Choice()
    download_token = Str()

    class Meta:
        model = model.RateSendLog
        fields = ('job_id', 'status', 'error', 'create_time', 'total_records',
                  'completed_records', 'file', 'rate_table_id', 'format', 'download_deadline',
                  'effective_date', 'is_email_alert', 'is_disable', 'trunks', 'send_type',  # 'download_method',
                  'send_specify_email', 'email_template_id', 'zip', 'email_direct', 'sent_area',
                  'email_count', 'recipient_count', 'rate_table_name', 'process', 'date_format', 'download_token',
                  'indicate_inc_dec')
        search_fields = ('status', 'error', 'total_records',
                         'completed_records', 'file', 'rate_table_id', 'format',
                         'is_email_alert', 'is_disable', 'trunks', 'send_type',  # 'download_method',
                         'send_specify_email', 'email_template_id', 'zip', 'sent_from',
                         'email_count', 'recipient_count', 'rate_table_name', 'date_format')
        query_fields = ('job_id_in', 'create_time_gt', 'create_time_lt',
                        'download_deadline_gt', 'download_deadline_lt',
                        'effective_date_gt', 'effective_date_lt', 'process_gt', 'process_lt')


class RateSendLogDetailGetScheme(BaseModelScheme):
    id = Int()
    job_id = Int()
    parent_job = Nested(RateSendGetScheme)
    status = Choice()
    trunk_id = Int()
    send_to = Str()
    send_on = DateTime()
    download_date = Str()
    download_deadline = DateTime()
    trunk_name = Str()
    client_name = Str()
    client_id = Int()
    rate_table_name = Str()
    rate_table_id = Int()
    email_count = Int()
    recipient_count = Int()
    create_time = DateTime()
    send_type = Str()
    sent_area = Choice()
    error = Str()
    file = Str()
    process = Float()
    download_link = Str()

    class Meta:
        model = model.RateSendLogDetail
        fields = ('id', 'job_id', 'parent_job',
                  'status', 'send_to', 'trunk_id', 'trunk_name', 'client_id', 'client_name',
                  'send_on', 'download_date', 'rate_table_name', 'error',
                  'email_count', 'recipient_count', 'process',
                  'create_time', 'send_type', 'sent_area', 'file', 'download_link')
        search_fields = (
            'id', 'job_id', 'trunk_id', 'status', 'send_to', 'trunk_id', 'trunk_name', 'client_id', 'client_name',
            'rate_table_name', 'rate_table_id',
            'email_count', 'recipient_count',
            'send_type', 'error', 'sent_area', 'file')
        query_fields = ('download_date_gt', 'download_date_lt', 'download_deadline_gt', 'download_deadline_lt',
                        'create_time_gt', 'create_time_lt', 'process_gt', 'process_lt')


class RateDownloadLogNestedScheme(BaseModelScheme):
    download_time = DateTime()
    download_ip = Str()
    download_username = Str()

    class Meta:
        model = model.RateDownloadLog
        fields = ('download_time', 'download_ip', 'download_username')


class RateSendLogDetailGetResultScheme(BaseModelScheme):
    result = Str(validate=validate.OneOf(['expired', 'cancelled', 'normal']))
    download_log = Nested('RateDownloadLogNestedScheme', many=True)

    class Meta:
        model = model.RateSendLogDetail
        fields = ('result', 'download_log')


# --- RateSend

class RateTypeOverrideScheme(BaseModelScheme):
    id = Int()
    src_country = Str(validate=validate.Length(1, 4))
    dest_country = Str(validate=validate.Length(1, 2))
    rate_type = Int(validate=validate.Range(min=1, max=3))
    comment = Str()
    resource_id = Int()

    class Meta:
        model = model.RateTypeOverride
        fields = ('id', 'src_country', 'dest_country', 'rate_type', 'resource_id', 'comment')
        search_fields = ('resource_id',)


class RateTypeOverrideByTrunkModifyScheme(BaseModelScheme):
    src_country = Str(validate=validate.Length(1, 4))
    dest_country = Str(validate=validate.Length(1, 2))
    rate_type = Int(validate=validate.Range(min=1, max=3))
    comment = Str()
    resource_id = Int()

    class Meta:
        model = model.RateTypeOverride
        fields = ('src_country', 'dest_country', 'rate_type', 'comment')


class RateTypeOverrideByTrunkScheme(BaseModelScheme):
    items = Nested(RateTypeOverrideByTrunkModifyScheme, many=True)

    class Meta:
        model = model.RateTypeOverride
        fields = ('items',)


class RateTypeOverrideByTrunkScheme(BaseModelScheme):
    items = Nested(RateTypeOverrideByTrunkModifyScheme, many=True)

    class Meta:
        model = model.RateTypeOverride
        fields = ('items',)


class RateTypeOverrideEditManyScheme(BaseModelScheme):
    items = Nested(RateTypeOverrideScheme, many=True)

    class Meta:
        model = model.RateTypeOverride
        fields = ('items',)


"""
class ResourceScheme(BaseModelScheme):
    client_id = Int(required=True)
    group_id = Int(required=True)
    rate_table_id = Int(required=True)
    route_strategy_id = Int(required=True)

    class Meta:
        model = model.Resource
        exclude = (
            'resource_id', 'client', 'group', 'rate_table', 'route_strategy', 'update_by',
            'create_time', 'update_time', 'update_at'
        )
"""


class ResourceFromTemplateScheme(BaseModelScheme):
    name = Str(validate=validate.Regexp(NAME_REGEXP))
    client_id = Int()
    is_active = Boolean()
    ip = Nested('ResourceIpScheme', many=True)
    prefixes = Nested('ResourcePrefixModifyScheme', many=True, allow_none=True)
    media_type = Choice(default='Proxy Media + Transcoding')
    cps_limit = Int(allow_none=True, validate=validate.Range(min=0, max=1000000))
    call_limit = Int(allow_none=True, validate=validate.Range(min=0, max=100000000))

    class Meta:
        model = model.Resource
        fields = ('name', 'is_active', 'ip', 'media_type', 'call_limit', 'cps_limit', 'prefixes')


class ResourceSimpleScheme(BaseModelScheme):
    trunk_id = Int()  # attribute='resource_id')
    trunk_name = Str()  # attribute='name')
    client_name = Str()

    class Meta:
        model = model.Resource
        fields = ('trunk_id', 'trunk_name', 'client_name', 'client_name')


class ResourceMinScheme(BaseModelScheme):
    trunk_id = Int()  # attribute='resource_id')
    alias = Str()  # attribute='name')

    class Meta:
        model = model.Resource
        fields = ('trunk_id', 'alias')


class ResourceInfoGetScheme(BaseModelScheme):
    # client_id = Nested(ClientGetScheme)
    # group_id = Nested(TrunkGroupGetScheme)
    # rate_table_id = Nested(RateTableGetScheme)
    # route_strategy_id = Nested(RouteStrategyGetScheme)
    trunk_id = Int()  # attribute='resource_id')
    trunk_name = Str()  # attribute='name')
    type = Str(validate=validate.OneOf(('orig', 'term')))
    direction = Str(validate=validate.OneOf(('ingress', 'egress')))
    active = Bool()
    carrier = Str()
    carrier_id = Int()
    client_name = Str()

    class Meta:
        model = model.Resource
        fields = ('trunk_id', 'trunk_name', 'type', 'direction', 'active', 'carrier', 'carrier_id', 'client_name')
        search_fields = ('type', 'direction', 'active', 'carrier_id', 'trunk_id', 'trunk_name', 'client_name')


class ResourceUnclaimedGetScheme(ResourceInfoGetScheme):
    prefix = Str()
    rate_table_id = Int()
    rate_table_name = Str()
    rate_email = Str()
    update_at = DateTime()

    class Meta:
        model = model.Resource
        fields = ('trunk_id', 'trunk_name', 'type', 'direction', 'active', 'carrier', 'carrier_id',
                  'prefix', 'rate_table_id', 'rate_table_name', 'rate_email', 'update_at')
        search_fields = (
            'type', 'rate_table_id', 'direction', 'active', 'carrier_id', 'trunk_id', 'trunk_name', 'rate_table_name')
        query_fields = ('update_at_gt', 'update_at_lt')


class CodecScheme(BaseModelScheme):
    class Meta:
        model = model.Codec
        exclude = ('id',)


class CodecGetScheme(BaseModelScheme):
    class Meta:
        model = model.Codec
        fields = ('id', 'name', 'detail')
        search_fields = ('id', 'name', 'detail')


class ResourceCodecsRefScheme(BaseModelScheme):
    resource_id = Int(required=True)

    class Meta:
        model = model.ResourceCodecsRef
        exclude = ('id', 'resource')


class ResourceCodecsRefInnerScheme(BaseModelScheme):
    resource_id = Int(required=True)

    class Meta:
        model = model.ResourceCodecsRef
        exclude = ('id', 'resource', 'resource_id')


class ResourceCodecsRefGetScheme(BaseModelScheme):
    # resources = Nested(resource.ResourceSchemeGet)
    codecs = Nested(CodecGetScheme)

    class Meta:
        model = model.ResourceCodecsRef


# +++ ResourceIpLimit
class ResourceIpLimitScheme(BaseModelScheme):
    cps = Int()
    capacity = Int()
    time_profile_id = Int(validate=lambda value: _valid('TimeProfile', 'time_profile_id', value))

    class Meta:
        model = model.ResourceIpLimit
        fields = ('cps', 'capacity', 'time_profile_id')


class ResourceIpLimitGetScheme(ResourceIpLimitScheme):
    limit_id = Int()
    resource_id = Int()
    resource_ip_id = Int()
    time_profile_id = Int()
    time_profile_name = Str(dump_only=True)

    class Meta:
        model = model.ResourceIpLimit
        fields = (
            'limit_id', 'resource_ip_id', 'resource_id', 'cps', 'capacity', 'time_profile_id', 'time_profile_name')
        search_fields = (
            'limit_id', 'resource_ip_id', 'resource_id', 'cps', 'capacity', 'time_profile_id', 'time_profile_name')


class ResourceIpLimitExtGetScheme(BaseModelScheme):
    cps = Int()
    capacity = Int()
    resource_ip = Str()
    resource_id = Int()
    resource_port = Int()
    resource_active = Bool()
    resource_alias = Str()
    resource_cps_limit = Int()
    resource_capacity = Int()
    client_name = Str()
    client_id = Int()
    client_call_limit = Int()
    client_cps_limit = Int()
    client_type = Choice()
    company_type = Choice()

    class Meta:
        model = model.ResourceIpLimitQuery
        fields = ('cps', 'capacity', 'resource_ip', 'resource_id', 'resource_port', 'resource_active', 'resource_alias',
                  'resource_cps_limit', 'resource_capacity', 'client_name', 'client_call_limit', 'client_cps_limit',
                  'client_id')
        search_fields = ('cps', 'capacity', 'resource_ip', 'resource_port', 'resource_id',
                         'resource_active', 'resource_alias', 'resource_cps_limit', 'resource_capacity', 'client_name',
                         'client_call_limit', 'client_cps_limit', 'client_type', 'company_type', 'client_id')


# +++ ResourceIp
class ResourceIpIngressScheme(BaseModelScheme):
    # resource = Nested(resource.ResourceSchemeGet)
    ip = Str(validate=[validate.Regexp(IP_REGEXP)])  # ,lambda value:_valid_unique('ResourceIp','ip',value)])
    port = Int(validate=validate.Range(1000, 65535), allow_none=True)

    class Meta:
        model = model.ResourceIp
        fields = ('ip', 'port',)


class ResourceIpScheme(BaseModelScheme):
    # resource = Nested(resource.ResourceSchemeGet)
    ip = Str(validate=validate.Regexp(IP_REGEXP), allow_none=True)
    port = Int(validate=validate.Range(1000, 65535), allow_none=True)
    fqdn = Str(allow_none=True)
    addr_type = Choice()
    resource_id = Int()
    limits = Nested(ResourceIpLimitScheme, many=True)
    direction = Int(validate=validate.Range(0, 1))
    sip_rpid = Str(allow_none=True)
    trunk_name = Str(validate=lambda value: _valid('Resource', 'alias', value))
    trunk_type2 = Choice()

    class Meta:
        model = model.ResourceIp
        fields = ('ip', 'port', 'fqdn', 'addr_type')


class ResourceIpImportScheme(ResourceIpScheme):
    class Meta:
        model = model.ResourceIp
        fields = ('trunk_name', 'ip', 'port', 'fqdn', 'addr_type')

    @pre_load
    def load_ids(self, obj):
        if 'trunk_name' in obj:
            trunk = model.Resource.filter(model.Resource.alias == obj['trunk_name']).first()
            if trunk:
                obj['resource_id'] = trunk.resource_id
                del obj['trunk_name']
            else:
                raise ValidationError({'trunk_name': 'invalid trunk name'})
        return obj


class ResourceIpGetScheme(ResourceIpScheme):
    class Meta:
        model = model.ResourceIp
        fields = ('ip', 'port', 'fqdn', 'sip_rpid', 'direction', 'addr_type')


class ResourceIpAllScheme(ResourceIpScheme):
    class Meta:
        model = model.ResourceIp
        fields = ('ip', 'port', 'fqdn', 'sip_rpid', 'direction', 'addr_type')


class ResourceIpAllGetScheme(ResourceIpScheme):
    limits = Nested(ResourceIpLimitGetScheme, many=True)

    class Meta:
        model = model.ResourceIp
        fields = ('resource_ip_id', 'resource_id', 'ip', 'port', 'fqdn', 'trunk_type2',
                  'sip_rpid', 'limits', 'direction', 'addr_type', 'trunk_name')
        search_fields = ('resource_ip_id', 'resource_id', 'ip', 'port', 'trunk_type2',
                         'fqdn', 'sip_rpid', 'direction', 'addr_type')


class ResourceIpClientGetScheme(ResourceIpAllGetScheme):
    class Meta:
        model = model.ResourceIp
        fields = ('resource_ip_id', 'resource_id', 'ip', 'port', 'fqdn', 'sip_rpid', 'limits', 'direction', 'addr_type')
        search_fields = ('resource_ip_id', 'resource_id', 'ip', 'port', 'fqdn', 'sip_rpid', 'direction', 'addr_type')


class ResourceIpPortScheme(ResourceIpScheme):
    class Meta:
        model = model.ResourceIp
        fields = ('ip', 'port')
        search_fields = ('ip', 'port')


class ResourceIpLimitsScheme(ResourceIpScheme):
    class Meta:
        model = model.ResourceIp
        fields = ('resource_ip_id', 'limits')
        search_fields = ('resource_ip_id', 'resource_id', 'ip', 'port', 'fqdn', 'sip_rpid', 'addr_type')


class ResourceIpRegUserScheme(BaseModelScheme):
    # resource = Nested(resource.ResourceSchemeGet)
    username = Str(validate=[validate.Length(1, 50), validate.Regexp(NAME_REGEXP)])
    password = Str(validate=validate.Length(1, 50))  # ,load_only=True)
    profile_id = Int(validate=lambda value: _valid('SwitchProfile', 'id', value))
    reg_type = Int()
    reg_status = Choice()
    registered = Str()
    expires = Int()

    @post_load
    def set_reg(self, data):
        data.reg_type = 1

    class Meta:
        model = model.ResourceIp
        fields = ('username', 'password', 'profile_id', 'reg_status')


class ResourceIpRegUserGetScheme(ResourceIpRegUserScheme):
    id = Int()

    class Meta:
        model = model.ResourceIp
        fields = ('id', 'username', 'password', 'profile_id', 'reg_status', 'expires', 'registered', 'resource_ip_id')


class ResourceIpRegGatewayScheme(BaseModelScheme):
    # resource = Nested(resource.ResourceSchemeGet)
    username = Str(validate=validate.Length(1, 50))
    password = Str(validate=validate.Length(1, 50))  # ,load_only=True)
    reg_srv_ip = Str(validate=[validate.Regexp(IP_REGEXP)])
    reg_srv_port = Int(validate=validate.Range(1000, 65535))
    profile_id = Int(validate=lambda value: _valid('SwitchProfile', 'id', value))
    reg_type = Int()
    reg_status = Choice()
    registered = Str()
    expires = Int(validate=validate.Range(0, 1000000))

    @post_load
    def set_reg(self, data):
        data.reg_type = 2

    class Meta:
        model = model.ResourceIp
        fields = ('username', 'password', 'reg_srv_ip', 'reg_srv_port', 'profile_id', 'expires')


class ResourceIpRegGatewayGetScheme(ResourceIpRegGatewayScheme):
    id = Int()

    class Meta:
        model = model.ResourceIp
        fields = (
            'id', 'username', 'password', 'reg_srv_ip', 'reg_srv_port', 'profile_id', 'reg_status', 'expires',
            'registered')


# --- ResourceIp

# +++ ResourceDirection
class ResourceDirectionScheme(BaseModelScheme):
    resource_id = Int()
    # direction_id
    time_profile_id = Int(allow_none=True,
                          validate=lambda value: _valid('TimeProfile', 'time_profile_id', value))
    action = Choice(allow_none=True)
    direction = Choice(allow_none=True)
    digits = Str(required=True, allow_none=False, validate=validate.Regexp(PREFIX_REGEXP))
    dnis = Str(allow_none=True, validate=validate.Regexp(PREFIX_REGEXP))
    number_length = Int(allow_none=True)
    number_type = Choice(allow_none=True)
    type = Choice(allow_none=False)

    @pre_load
    def check_empty_strings(self, data):
        if 'number_type' in data:
            data['number_type'] = data['number_type'] or None
        if 'number_length' in data:
            data['number_length'] = data['number_length'] or None

        return data

    class Meta:
        model = model.ResourceDirection
        fields = ('time_profile_id', 'action', 'direction', 'digits',
                  'dnis', 'number_length', 'number_type', 'type')


class OcnBlocklistMainScheme(BaseModelScheme):
    res_id = Int(allow_none=True)
    ocn = Str()
    class Meta:
        model = model.OcnBlocklist
        fields = ('res_id', 'ocn')


class OcnBlocklistGetMainScheme(BaseModelScheme):
    id = Int()
    res_id = Int()
    ocn = Str()
    with_res_id = Bool()  
    class Meta:
        model = model.OcnBlocklist
        query_fields = ('res_id', 'ocn', 'id', 'with_res_id')


class OcnBlocklistScheme(BaseModelScheme):
    res_id = Int()
    ocn = Str()
    class Meta:
        model = model.OcnBlocklist
        fields = ('id', 'res_id', 'ocn', 'created_at', 'created_by')


class OcnBlocklistPostScheme(BaseModelScheme):
    ocn = Str()
    class Meta:
        model = model.OcnBlocklist
        fields = ('id', 'ocn', 'created_at', 'created_by')


class OcnBlocklistManyScheme(BaseModelScheme):
    ocns = List(Nested(OcnBlocklistPostScheme))

    class Meta:
        model = model.RateTable
        fields = ('ocns',)


class ResourceDirectionPrefixGetScheme(BaseModelScheme):
    tech_prefix = Str()

    class Meta:
        model = model.ResourceDirection
        fields = ('tech_prefix',)


class ResourceDirectionModifyScheme(BaseModelScheme):
    direction = Choice(allow_none=True)
    digits = Str(required=True, allow_none=True)  # , validate=validate.Regexp(PREFIX_REGEXP))
    action = Choice(allow_none=True)
    type = Choice(allow_none=False)

    class Meta:
        model = model.ResourceDirection
        fields = ('direction', 'digits', 'type', 'action')


class ResourceDirectionGetScheme(ResourceDirectionScheme):
    resource_id = Int(required=True)
    direction_id = Int(required=True)
    empty_dnis = Bool()

    class Meta:
        model = model.ResourceDirection
        fields = ('direction_id', 'resource_id',) + ResourceDirectionScheme.Meta.fields
        search_fields = ('direction_id', 'action', 'direction', 'digits', 'type', 'empty_dnis')


class ResourceWithDirectionScheme(BaseModelScheme):
    directions = Nested(ResourceDirectionScheme, many=True)

    class Meta:
        model = model.Resource
        fields = ('directions',)


# --- ResourceDirection

# +++ ResourcePrefix

class ResourcePrefixScheme(BaseModelScheme):
    tech_prefix = Str(validate=lambda value: _valid_unique('ResourcePrefix', 'tech_prefix', value))
    trunk_id = Int(validate=lambda value: _valid('Resource', 'resource_id', value))
    routing_plan_id = Int(validate=lambda value: _valid('RouteStrategy', 'route_strategy_id', value))

    class Meta:
        model = model.ResourcePrefix
        fields = (
            'trunk_id', 'tech_prefix', 'routing_plan_id', 'code_cap', 'code_cps', 'code', 'rate_table_id', 'product_id')


class ResourcePrefixModifyScheme(BaseModelScheme):
    tech_prefix = Str(allow_none=True, required=False, validate=validate.Regexp(PREFIX_REGEXP))
    trunk_id = Int(validate=lambda value: _valid('Resource', 'resource_id', value))
    routing_plan_id = Int(validate=lambda value: _valid('RouteStrategy', 'route_strategy_id', value))
    code_cap = Int(allow_none=True)
    code_cps = Int(allow_none=True)

    class Meta:
        model = model.ResourcePrefix
        fields = ('tech_prefix', 'routing_plan_id', 'code_cap', 'code_cps', 'code', 'rate_table_id', 'product_id')


class ResourcePrefixWithIPScheme(BaseModelScheme):
    trunk_id = Int()
    tech_prefix = Str()
    routing_plan_id = Int()
    ips = List(Str(), allow_none=True)
    code_cps = Int(allow_none=True)
    code_cap = Int(allow_none=True)
    product_id = Int()

    class Meta:
        model = model.ResourcePrefix
        fields = ('ips', 'tech_prefix', 'rate_table_id', 'routing_plan_id', 'code', 'code_cps',
                  'code_cap', 'product_id')


class ResourcePrefixGetScheme(BaseModelScheme):
    # resource = Nested(resource.ResourceSchemeGet)
    # ips =Nested(ResourceIpGetScheme,many=True)
    id = Int()
    routing_plan_id = Int()
    trunk_id = Int()
    product_id = Int()
    product_name = Str(dump_only=True)
    trunk_name = Str(dump_only=True)
    carrier_name = Str(dump_only=True)
    rate_table_id = Int()
    rate_table_name = Str(dump_only=True)
    ingress = Bool()

    class Meta:
        model = model.ResourcePrefix
        fields = (
            'id', 'trunk_id', 'trunk_name', 'code', 'tech_prefix', 'rate_table_id', 'rate_table_name',
            'routing_plan_id',
            'code_cps',
            'code_cap', 'product_name', 'product_id', 'carrier_name', 'carrier_id', 'ingress')
        search_fields = (
            'id', 'code', 'tech_prefix', 'rate_table_id', 'rate_table_name', 'routing_plan_id', 'code_cps', 'code_cap',
            'trunk_id', 'product_name', 'product_id', 'trunk_name', 'carrier_name', 'ingress')


class ResourceWithIpsScheme(BaseModelScheme):
    # resource = Nested(resource.ResourceSchemeGet)
    ip = Nested(ResourceIpScheme, many=True, allow_none=True, required=False)

    class Meta:
        model = model.Resource
        fields = ('ip',)


class ResourceWithIpsGetScheme(BaseModelScheme):
    # resource = Nested(resource.ResourceSchemeGet)
    ip = Nested(ResourceIpScheme, many=True)
    trunk_id = Int()

    class Meta:
        model = model.Resource
        fields = ('trunk_id', 'ip', 'rate_table_id', 'route_strategy_id')


class ResourceWithIpAndAliasGetScheme(BaseModelScheme):
    # resource = Nested(resource.ResourceSchemeGet)
    ip = Nested(ResourceIpScheme, many=True)
    trunk_id = Int()
    alias = Str()

    class Meta:
        model = model.Resource
        fields = ('trunk_id', 'ip', 'alias')


# --- ResourcePrefix

# +++ ResourceReplaceAction
class ResourceReplaceActionScheme(BaseModelScheme):
    resource_id = Int(required=True)
    ani_prefix = Str()
    ani = Str()
    ani_min_length = Int(allow_none=True)
    ani_max_length = Int(allow_none=True)
    type = Choice()
    dnis_prefix = Str()
    dnis = Str()
    dnis_min_length = Int(allow_none=True)
    dnis_max_length = Int(allow_none=True)

    @pre_load
    def check_length(self, data):
        if 'ani_min_length' in data and 'ani_max_length' in data and data['ani_min_length'] and data['ani_max_length']:
            if data['ani_max_length'] < data['ani_min_length']:
                raise ValidationError({'ani_max_length': ['ani_max_length less than ani_min_length !']})
        if 'dnis_min_length' in data and 'dnis_max_length' in data and data['dnis_min_length'] and data[
            'dnis_max_length']:
            if data['dnis_max_length'] < data['dnis_min_length']:
                raise ValidationError({'dnis_max_length': ['dnis_max_length less than dnis_min_length !']})
        return data

    class Meta:
        model = model.ResourceReplaceAction
        fields = ('ani_prefix', 'ani', 'ani_min_length', 'ani_max_length',
                  'type', 'dnis_prefix', 'dnis', 'dnis_min_length', 'dnis_max_length')


class ResourceReplaceActionGetScheme(ResourceReplaceActionScheme):
    resource_id = Int(required=True)
    direction_id = Int(required=True)

    class Meta:
        model = model.ResourceReplaceAction
        fields = ('id', 'resource_id',) + ResourceReplaceActionScheme.Meta.fields
        search_fields = ('id', 'type', 'ani', 'dnis', 'ani_prefix', 'dnis_prefix')


class ResourceWithReplaceActionScheme(BaseModelScheme):
    replace_actions = Nested(ResourceReplaceActionScheme, many=True)

    class Meta:
        model = model.Resource
        fields = ('replace_actions',)


# --- ResourceReplaceAction


class EgressProfileScheme(BaseModelScheme):
    egress_id = Int()
    profile_id = Int(validate=lambda value: _valid('SwitchProfile', 'id', value))
    ingress_id = Int(allow_none=True, required=False, validate=lambda value: _valid('Resource', 'resource_id', value))

    class Meta:
        model = model.EgressProfile
        fields = ('profile_id', 'ingress_id',)


class EgressProfileGetScheme(EgressProfileScheme):
    id = Int()
    egress_id = Int()
    profile_id = Int()
    server_name = Str(attribute='_server_name')
    switch_name = Str()
    ingress_id = Int()
    egress_name = Str()
    ingress_name = Str()
    profile_sip_ip = Str()
    profile_sip_port = Int()

    class Meta:
        model = model.EgressProfile
        fields = (
            'id', 'egress_id', 'profile_id', 'server_name', 'switch_name', 'ingress_id', 'egress_name', 'ingress_name', 'profile_sip_ip', 'profile_sip_port')
        search_fields = ('id', 'egress_id', 'profile_id', 'ingress_id', 'server_name', 'switch_name')


class ResourceCapacityScheme(BaseModelScheme):
    egress_id = Int(validate=lambda value: _valid('Resource', 'resource_id', value))
    ingress_id = Int(validate=lambda value: _valid('Resource', 'resource_id', value))
    max_cps = Int(required=False, allow_none=True)
    max_cap = Int(required=False, allow_none=True)

    class Meta:
        model = model.ResourceCapacity
        fields = ('ingress_id', 'max_cps', 'max_cap')


class ResourceCapacityGetScheme(ResourceCapacityScheme):
    egress_id = Int()
    ingress_id = Int(validate=lambda value: _valid('Resource', 'resource_id', value))
    max_cps = Int()
    max_cap = Int()
    egress_name = Str()
    ingress_name = Str()

    class Meta:
        model = model.ResourceCapacity
        fields = ('id', 'egress_id', 'ingress_id', 'egress_name', 'ingress_name', 'max_cps', 'max_cap')
        search_fields = ('id', 'egress_id', 'ingress_id')


class ResourceInnerGatewayGroupGetScheme(BaseModelScheme):
    alias = Str()
    capacity = Int()
    cps_limit = Int()
    egress = Bool()
    ingress = Bool()
    resource_id = Int()
    client_id = Int()
    client_name = Str()
    client_mode = Choice()
    # client_status = Bool()
    client_allowed_credit = Float()
    client_call_limit = Int()
    client_cps_limit = Int()
    active = Bool()

    class Meta:
        model = model.Resource
        fields = ('alias', 'capacity', 'cps_limit', 'egress', 'ingress', 'resource_id', 'active',
                  # 'client_mode',#'client_status',
                  # 'client_id','client_name','client_allowed_credit','client_call_limit','client_cps_limit',
                  )
        search_fields = (
            'alias', 'capacity', 'cps_limit', 'egress', 'ingress', 'resource_id', 'client_id', 'client_name',
            'client_mode',
            # 'client_status',
            'client_allowed_credit', 'client_call_limit', 'client_cps_limit',)
        query_fields = ('client_allowed_credit_gt', 'client_allowed_credit_lt')


class ClientGatewayGroupGetScheme(BaseModelScheme):
    resources = Nested('ResourceInnerGatewayGroupGetScheme', many=True, attribute='_resources')

    client_id = Int()
    name = Str()
    mode = Choice()
    # client_status = Bool()
    allowed_credit = Float()
    call_limit = Int()
    cps_limit = Int()
    resource_id = Int()
    alias = Str()
    client_type = Choice()
    company_type = Choice()
    status = Bool()
    payment_term_id = Int()

    class Meta:
        model = model.Client
        fields = ('mode', 'client_type', 'company_type', 'payment_term_id',  # 'client_status',
                  'client_id', 'name', 'allowed_credit', 'call_limit', 'cps_limit', 'status')
        search_fields = ('client_id', 'name', 'mode',
                         # 'client_status',
                         'allowed_credit', 'call_limit', 'cps_limit',
                         'resource_id', 'alias', 'payment_term_id',
                         'client_type', 'company_type', 'status')
        query_fields = ('allowed_credit_gt', 'allowed_credit_lt')


class ResourceActionsInnerScheme(Schema):
    match_prefix = Str()  # validate=validate.Regexp(PREFIX_REGEXP))
    min_length = Int(validate=validate.Range(0, 32))
    max_length = Int(validate=validate.Range(0, 32))
    action = Str(validate=validate.OneOf(('Add Prefix', 'Delete Prefix', 'Add Suffix', 'Delete Suffix', 'Replace')))
    num_digits = Int(validate=validate.Range(1, 15), required=False)
    new_digits = Str(validate=validate.Regexp(r'^[0-9#\*\+]*|(none)$'), required=False)
    number_length = Int(validate=validate.Range(0, 32))
    number_type = Str(validate=validate.OneOf(('all', 'greater than', 'equal to', 'less than')), allow_none=True)

    @pre_load
    def check(self, data):
        if False and 'min_length' in data and 'max_length' in data and data['min_length'] > data['max_length']:
            raise ValidationError({'max_length': ['must be less than min length {} !'.format(data['min_length'])]})
        if 'action' in data:
            t = data['action']
            if t in ('Add Prefix', 'Add Suffix', 'Replace') and not 'new_digits' in data:
                raise ValidationError({'new_digits': ['new_digits mandatory when {}!'.format(t)]})
            if t in ('Delete Prefix', 'Delete Suffix') and not 'num_digits' in data:
                raise ValidationError({'num_digits': ['num_digits mandatory when {}!'.format(t)]})
        return data


class ResourceActionsScheme(Schema):
    ani = Nested(ResourceActionsInnerScheme)
    dnis = Nested(ResourceActionsInnerScheme)


# +++ EgressTrunk
class ResourceAllSheme(BaseModelScheme):
    account_id = Str()
    amount_per_port = Decimal()
    ani_cap_limit = Int(allow_none=True)
    ani_cps_limit = Int(allow_none=True)
    auth_type = Choice(allow_none=True, required=False)
    authorization_type = Choice()
    billing_by_port = Choice()
    billing_method = Choice()
    bypass_media = Choice(default='Bypass Media')
    block_dno = Bool()
    call_limit = Int(allow_none=True)
    carrier = Str(dump_only=True)
    carrier_id = Int(dump_only=True)
    carrier_is_active = Bool()
    cli_type = Choice()
    codecs = List(Str(validate=lambda value: _valid('Codec', 'name',
                                                    value)))  # Nested('ResourceCodecsRefGetScheme',many=True,allow_none=True,)
    codes = List(Str(validate=lambda value: _valid('DigitTranslation', 'translation_name',
                                                   value)))  # Nested('ResourceTranslationRefGetScheme',many=True,allow_none=True,)
    cps_limit = Int(allow_none=True, required=False)
    created_at = DateTime(dump_only=True)
    created_by = Str(dump_only=True)
    display_name = Bool()
    div = Choice()
    dnis_cap_limit = Int(allow_none=True)
    dnis_cps_limit = Int(allow_none=True)
    enable_global_404_blocking = Bool()
    enfource_cid = Bool(allow_none=True)
    group_id = Int()  # Nested(TrunkGroupGetScheme)
    host_routing_strategy = Choice()
    ignore_ring = Bool()
    ignore_early_media = Bool()
    ignore_early_no_sdp = Bool()
    ip = Nested('ResourceIpScheme', many=True)
    is_active = Bool()
    jurisdiction_use_dnis = Choice()
    max_duration = Int(allow_none=True, required=False)
    media_timeout = Int(allow_none=True)
    media_type = Choice()
    min_duration = Int(allow_none=True, required=False, validate=validate.Range(0, 60000))
    min_profit_type = Choice()
    min_profit_value = Float(allow_none=True, attribute='_min_profit_value')
    name = Str(validate=validate.Regexp(NAME_REGEXP))
    oli = Choice()
    paid = Choice()
    paid_privacy = Choice()
    paid_uri = Str()
    paid_display = Str()
    pass_lrn = Bool()
    pass_through = Choice()
    pci = Choice()
    pani = Choice()
    pani_value = Str()
    pdd = Int(allow_none=True)
    prefix = Str(allow_none=True)
    prefixes = Nested('ResourcePrefixModifyScheme', many=True, allow_none=True)
    priv = Choice()
    product_id = Int(allow_none=True, required=False,
                     validate=lambda value: _valid('ProductRoutRateTable', 'id', value))
    rate_decimal = Int(default=6, validate=validate.Range(0, 7))
    rate_rounding = Choice()
    rate_table_id = Int(allow_none=True, required=False,
                        validate=lambda value: _valid('RateTable', 'rate_table_id', value))
    rate_table_name = Str(dump_only=True)
    rate_use_rpid = Bool()
    re_invite_interval = Int(allow_none=True)
    reg_gateway = Nested(ResourceIpRegGatewayScheme, many=True)
    reg_user = Nested(ResourceIpRegUserScheme, many=True, attribute='_reg_user')
    resource_template_id = Int(allow_none=True, required=False,
                               validate=lambda value: _valid('ResourceTemplate', 'resource_template_id', value))
    ring_timeout = Int(allow_none=True, required=False)
    route_plan_id = Int(allow_none=True, required=False,
                        validate=lambda value: _valid('RouteStrategy', 'route_strategy_id', value))
    route_plan_name = Str(dump_only=True)
    rpid = Choice()
    rpid_id_type = Choice()
    rpid_party = Choice()
    rpid_privacy = Choice()
    rpid_screen = Choice()
    service_type = Choice()
    t38 = Bool()
    transaction_fee_id = Int(allow_none=True, required=False,
                             validate=lambda value: _valid('TransactionFeeItems', 'id', value))
    trunk_id = Int(dump_only=True)
    trunk_name = Str(validate=validate.Regexp(NAME_REGEXP))
    ingress_trunk_name = Str(validate=validate.Regexp(NAME_REGEXP))
    trunk_type2 = Choice()
    us_other = Choice()
    update_at = DateTime(dump_only=True)
    update_by = Str(dump_only=True)
    usage_count = Int(dump_only=True)
    static_count = Int(dump_only=True)
    dynamic_count = Int(dump_only=True)
    ip_count = Int(dump_only=True)
    ym_fraud_prob = Float(validate=validate.Range(0, 1), allow_none=True)
    ym_spam_score = Float(validate=validate.Range(-1, 1), allow_none=True)
    ym_tcpa_prob = Float(validate=validate.Range(0, 1), allow_none=True)

    shaken_sign_policy = Int()#Choice(required=False)
    shaken_vfy_policy = Int(validate=validate.OneOf([0, 1, 2, 3, 4]), required=False, allow_none=True)
    shaken_vfy_countries = Str(allow_none=True)
    vfy_policy = Int(validate=validate.OneOf([0, 1, 2, 3]), required=False)
    shaken_allow_resign = Bool(required=False)
    shaken_ani_group_list_id = Int(allow_none=True, required=False,
                                   validate=lambda value: _valid('ShakenAniGroupList', 'id', value))
    shaken_default_attest_lvl = Str(required=False, validate=validate.OneOf(['A', 'B', 'C']), default='A')
    shaken_p_headers = Int(validate=validate.OneOf([0, 1, 2]), default=0, allow_none=True)
    priv_value = Int()

    # non_us_ani_action = Int()
    block_dnc = Bool()
    block_wireless = Bool()
    block_dno = Str()

    block_wireless_ani = Bool()
    block_toll_free_ani = Bool()
    invalid_ani_action = Int()
    ani_country_whitelist = Str(allow_none=True)

    ftc_days = Int(allow_none=True)
    resource_block_group_id = Int()

    actions = Nested('ResourceActionsScheme', many=True)
    failover = Nested('ResourceFailoverScheme', many=True, allow_none=True)

    ani_group_id = Int(allow_none=True, required=False, validate=lambda value: _valid('RandomAniGroup', 'id', value))
    ani_group_name = Str(dump_only=True)
    shaken_force_validation = Bool()

    shaken_sign_tollfree = Bool()

    @post_load
    def bool_to_int(self, data):
        d = {True: 1, False: 0}
        for attr in ['display_name']:  # 'paid', 'oli', 'pci', 'priv', 'div',
            if attr in data:
                v = data[attr]
                if v in d:
                    data[attr] = d[v]
        return data

    class Meta:
        model = model.Resource
        fields = (
            'account_id',
            'amount_per_port',
            'ani_cap_limit',
            'ani_cps_limit',
            'auth_type',
            # 'authorization_type',
            'block_dno',
            'billing_method',
            'bypass_media',
            'call_limit',
            'cli_type',
            'codecs',
            'codes',
            'cps_limit',
            'display_name',
            'div',
            'dnis_cap_limit',
            'dnis_cps_limit',
            'enable_global_404_blocking',
            'enfource_cid',
            'host_routing_strategy',
            'ignore_ring',
            'ignore_early_media',
            'ignore_early_no_sdp',
            'ip',
            'is_active',
            'jurisdiction_use_dnis',
            'max_duration',
            'media_timeout',
            'media_type',
            'min_duration',
            'min_profit_type',
            'min_profit_value',
            # 'name',
            'oli',
            'paid',
            'paid_privacy',
            'paid_uri',
            'paid_display',
            'pass_lrn',
            'pass_through',
            'pci',
            'pani',
            'pani_value',
            'pdd',
            # 'prefix',
            # 'prefixes',
            'priv',
            'priv_value',
            # 'product_id',
            'rate_decimal',
            'rate_rounding',
            'rate_table_id',
            'rate_use_rpid',
            're_invite_interval',
            'reg_gateway',
            'reg_user',
            'ring_timeout',
            # 'route_plan_id',
            'rpid',
            'rpid',
            'rpid_id_type',
            'rpid_party',
            'rpid_privacy',
            'rpid_screen',
            'service_type',
            't38',
            'transaction_fee_id',
            'trunk_name',
            'trunk_type2',
            'us_other',
            'resource_template_id',
            'failover',
            'shaken_sign_policy',
            'shaken_vfy_policy',
            'shaken_vfy_countries',
            'shaken_allow_resign',
            'shaken_ani_group_list_id',
            'shaken_default_attest_lvl',
            'shaken_p_headers',
            'vfy_policy',
            'ym_tcpa_prob',
            'ym_spam_score',
            'ym_fraud_prob',
            'block_dnc',
            'block_wireless',
            'block_dno',
            'ani_group_id',
            'ani_group_name',
            'block_wireless_ani',
            'block_toll_free_ani',
            'invalid_ani_action',
            'ani_country_whitelist',
            'shaken_force_validation',
            'ftc_days',
            'ani_type_block',
            'dnis_type_block',
            'resource_block_group_id',
            'shaken_sign_tollfree'
        )
        get_fields = ('trunk_id', 'resource_id', 'client_id', 'carrier_id',
                      'client_name', 'email', 'rate_email', 'carrier_is_active',
                      'created_at', 'created_by', 'update_at', 'update_by',
                      'rate_table_name',
                      # 'route_plan_name',
                      'usage_count', 'enough_balance',
                      'static_count', 'dynamic_count', 'ip_count') + fields


class ResourceSchemeGet(ResourceAllSheme):
    class Meta:
        model = model.Resource
        fields = ResourceAllSheme.Meta.get_fields


class EgressTrunkActionsScheme(BaseModelScheme):
    actions = Nested('ResourceActionsScheme', many=True)

    class Meta:
        model = model.EgressTrunk
        fields = ('actions',)


class EgressTrunkScheme(ResourceAllSheme):
    # codes = List(Str())
    # codecs = List(Str())
    name = Str(validate=validate.Regexp(NAME_REGEXP))
    directions = Nested('ResourceDirectionModifyScheme', many=True, allow_none=True)
    number = Int(allow_none=True, required=False)
    counter_time = Int(allow_none=True, required=False)
    block_time = Int(allow_none=True, required=False)
    wait_ringtime180 = Int(default='60000')

    class Meta:
        model = model.EgressTrunk
        fields = ResourceAllSheme.Meta.fields + ('directions', 'number', 'counter_time', 'block_time', 'wait_ringtime180')
        # fields = ('directions',)


class EgressTrunkDynRoutesScheme(BaseModelScheme):
    dynamic_routes = Nested('DynamicRouteItemInnerScheme', many=True)

    class Meta:
        model = model.EgressTrunk
        fields = ('dynamic_routes',)


class EgressTrunkDynRoutesGetScheme(BaseModelScheme):
    dynamic_routes = Nested('DynamicRouteItemInnerGetScheme', many=True)
    trunk_id = Int()

    class Meta:
        model = model.EgressTrunk
        fields = ('dynamic_routes', 'trunk_id')


class EgressTrunkProfilesScheme(BaseModelScheme):
    profiles = Nested('EgressProfileScheme', many=True)

    class Meta:
        model = model.EgressTrunk
        fields = ('profiles',)


class EgressTrunkStaticRouteItemsScheme(BaseModelScheme):
    static_route_items = Nested('StaticRouteItemInnerScheme', many=True)

    class Meta:
        model = model.EgressTrunk
        fields = ('static_route_items',)


class EgressTrunkStaticRouteItemsGetScheme(BaseModelScheme):
    static_route_items = Nested('StaticRouteItemInnerScheme', many=True)
    trunk_id = Int()

    class Meta:
        model = model.EgressTrunk
        fields = ('static_route_items', 'trunk_id')


class EgressTrunkProfilesGetScheme(BaseModelScheme):
    dynamic_routes = Nested('DynamicRouteItemInnerScheme', many=True)
    profiles = Nested('EgressProfileScheme', many=True)
    static_route_items = Nested('StaticRouteItemInnerScheme', many=True)
    trunk_id = Int()
    client_id = Int()
    client_name = Str()

    class Meta:
        model = model.EgressTrunk
        fields = ('trunk_name', 'trunk_id', 'client_id', 'client_name', 'profiles')


# 'dynamic_routes','profiles','static_route_items',)


class EgressTrunkGetScheme(EgressTrunkScheme):
    codes = List(Str())
    codecs = List(Str())
    # codecs = Nested('ResourceCodecsRefGetScheme',many=True)
    # codes = Nested('ResourceTranslationRefGetScheme',many=True)
    ip = Nested('ResourceIpAllGetScheme', many=True)
    prefixes = Nested('ResourceDirectionPrefixGetScheme', many=True, load_from='directions', attribute='directions')
    reg_gateway = Nested(ResourceIpRegGatewayGetScheme, many=True)
    reg_user = Nested(ResourceIpRegUserGetScheme, many=True)
    dynamic_route_id = Int()
    dynamic_routes = Nested('DynamicRouteItemInnerScheme', many=True)
    profiles = Nested('EgressProfileScheme', many=True)
    static_route_items = Nested('StaticRouteItemInnerScheme', many=True)
    pdd = Int()
    client_type = Choice()
    has_ip = Str(validate=validate.Regexp(IP_REGEXP))
    is_rt_null = Bool()
    purged = Bool()
    number = Int(allow_none=True, required=False)
    counter_time = Int(allow_none=True, required=False)
    block_time = Int(allow_none=True, required=False)
    wait_ringtime180 = Int(default='60000')
    group_id = Int()

    class Meta:
        model = model.EgressTrunk
        fields = ('group_id', 'profiles', 'dynamic_routes', 'static_route_items', 'client_type', 'number', 'counter_time',
                  'block_time', 'wait_ringtime180') + ResourceAllSheme.Meta.get_fields
        search_fields = (
            'wait_ringtime180', 'resource_id', 'carrier_id', 'is_active', 'carrier_is_active', 'trunk_id', 'trunk_name',
            'resource_template_id',
            'update_by', 'static_count', 'dynamic_count', 'ip_count', 'rate_table_id', 'rate_table_name',
            'client_name', 'dynamic_route_id', 'client_type', 'trunk_type2', 'has_ip', 'is_rt_null', 'purged',
            'cps_limit', 'call_limit', 'group_id', 'pdd', 'min_profit_type', 'min_duration', 'shaken_sign_policy',
            'min_profit_value', 'shaken_vfy_policy', 'shaken_p_headers')
        query_fields = ('update_at_gt', 'update_at_lt', 'resource_id_in', 'min_profit_value_gt', 'min_profit_value_lt')


class EgressTrunkSmallGetScheme(EgressTrunkGetScheme):
    class Meta:
        model = model.EgressTrunk
        fields = ('profiles', 'client_type', 'number', 'counter_time', 'block_time') + ResourceAllSheme.Meta.get_fields
        search_fields = (
            'resource_id', 'carrier_id', 'is_active', 'carrier_is_active', 'trunk_id', 'trunk_name',
            'resource_template_id',
            'update_by', 'static_count', 'dynamic_count', 'ip_count', 'rate_table_id', 'rate_table_name',
            'client_name', 'dynamic_route_id', 'client_type', 'trunk_type2', 'has_ip', 'is_rt_null', 'purged',
            'cps_limit', 'call_limit')
        query_fields = ('update_at_gt', 'update_at_lt', 'resource_id_in', 'min_profit_value_gt', 'min_profit_value_lt')


class EgressTrunkGetSmallScheme(EgressTrunkGetScheme):
    class Meta:
        model = model.EgressTrunk
        fields = ('profiles', 'dynamic_routes', 'client_type', 'number', 'counter_time', 'block_time') + ResourceAllSheme.Meta.get_fields
        search_fields = (
            'resource_id', 'carrier_id', 'is_active', 'carrier_is_active', 'trunk_id', 'trunk_name',
            'resource_template_id',
            'update_by', 'static_count', 'dynamic_count', 'ip_count', 'rate_table_id', 'rate_table_name',
            'client_name', 'dynamic_route_id', 'client_type', 'trunk_type2', 'has_ip', 'is_rt_null')
        query_fields = ('update_at_gt', 'update_at_lt', 'resource_id_in', 'min_profit_value_gt', 'min_profit_value_lt')


class EgressTrunkOriginationGetScheme(EgressTrunkGetScheme):
    class Meta:
        model = model.EgressTrunk
        fields = ('profiles', 'dynamic_routes', 'client_type') + ResourceAllSheme.Meta.get_fields


class EgressTrunkActivateScheme(EgressTrunkGetScheme):
    active = Bool()

    class Meta:
        model = model.EgressTrunk
        fields = ('active',)
        search_fields = ('carrier_id', 'is_active', 'trunk_name', 'trunk_id', 'update_by',
                         'call_limit', 'cps_limit', 'resource_template_id')
        query_fields = ('update_at_gt', 'update_at_lt', 'resource_id_in')


class EgressTrunkPassScheme(EgressTrunkScheme):
    class Meta:
        model = model.EgressTrunk
        fields = ('rpid', 'paid', 'oli', 'pci', 'priv', 'div', 'rpid_id_type', 'rpid_privacy', 'display_name',
                  'rpid_party', 'rpid_screen', 'cli_type', 'pass_through',
                  'rate_rounding', 'rate_decimal', 'jurisdiction_use_dnis')


class IngressTrunkActionsScheme(BaseModelScheme):
    actions = Nested('ResourceActionsScheme', many=True)

    class Meta:
        model = model.IngressTrunk
        fields = ('actions',)


class IngressTrunkScheme(ResourceAllSheme):
    # rate_table_id = Int(allow_none=False, required=True,
    # 					validate=lambda value: _valid('RateTable', 'rate_table_id', value))
    number = Int(allow_none=True, required=False)
    counter_time = Int(allow_none=True, required=False)
    block_time = Int(allow_none=True, required=False)

    @pre_load
    def check_prefix(self, data):
        if 'prefixes' in data:
            if 'prefix' in data:
                raise ValidationError('You must set or "prefix" or "prefixes" only!')
            d = {}
            for p in data['prefixes']:
                if 'tech_prefix' in p:
                    if p['tech_prefix'] in d and False:
                        raise ValidationError({'prefixes': ['duplicate "tech_prefix" in "prefixes"!']})
                    else:
                        d[p['tech_prefix']] = 1

        return data

    @pre_load
    def check_ip(self, data):
        ip_port = {}
        if 'ip' in data and len(data['ip']) > 1:
            for ip in data['ip']:
                item = (ip['ip'] if 'ip' in ip else None, ip['port'] if 'port' in ip else None,
                        ip['fqdn'] if 'fqdn' in ip else None)
                if item in ip_port:
                    raise ValidationError({'ip': ['duplicate ip']})
                ip_port[item] = True

        if False and 'ip' in data and len(data['ip']) > 1:  # disabled by shuva request
            if (not 'prefixes' in data) and \
                    not ('prefix' in data and data['prefix']) and \
                    not ('prefixes' in data and data['prefixes'] and len(data['prefixes']) > 0):
                raise ValidationError({'prefixes': ['Cannot add more than one IP when no prefix!']})

    @pre_load
    def check_empty_prefix(self, data):
        if 'prefixes' in data and data['prefixes'] and len(data['prefixes']) > 1:
            if ''.join([p['tech_prefix'] for p in data['prefixes'] if
                        'tech_prefix' in p and not p['tech_prefix'] == None and not p[
                                                                                        'tech_prefix'].lower() == 'null']) != '':
                for p in data['prefixes']:
                    if (not 'tech_prefix' in p) or p['tech_prefix'] == '' or p['tech_prefix'] == None or p[
                        'tech_prefix'].lower() == 'null':
                        raise ValidationError({'prefixes': [
                            'Since you have more than one product for this trunk, you must define prefix for each']})

    class Meta:
        model = model.IngressTrunk
        fields = ResourceAllSheme.Meta.fields + ('number', 'counter_time', 'block_time')


class IngressTrunkModifyScheme(IngressTrunkScheme):
    prefixes = Nested('ResourcePrefixModifyScheme', many=True, allow_none=True)
    ip = Nested('ResourceIpScheme', many=True)

    class Meta:
        model = model.IngressTrunk
        fields = ResourceAllSheme.Meta.fields + ('prefixes', 'number', 'counter_time', 'block_time')


class ShakenAniGroupListRelScheme(BaseModelScheme):
    id = Int()
    ani_group_list_id = Int()
    ani_group_id = Int()
    attest_lvl = Str(validate=[validate.Length(max=1)])

    class Meta:
        model = model.ShakenAniGroupListRel
        fields = ('ani_group_list_id', 'attest_lvl', 'ani_group_id', 'id')


class IngressTrunkGetScheme(IngressTrunkScheme):
    ip = Nested('ResourceIpAllGetScheme', many=True)
    prefixes = Nested('ResourcePrefixGetScheme', many=True)
    carrier_id = Int()
    carrier = Str(dump_only=True)
    email = Str(dump_only=True)
    trunk_id = Int()  # attribute='resource_id')
    trunk_name = Str()  # attribute='name')
    trunk_count = Int(dump_only=True)
    resource_template_id = Int()
    reg_gateway = Nested(ResourceIpRegGatewayGetScheme, many=True)
    reg_user = Nested(ResourceIpRegUserGetScheme, many=True)
    routing_plan_id = Int()
    product_id = Int()
    client_type = Choice()
    has_ip = Str(validate=validate.Regexp(IP_REGEXP))
    attestation_settings = Nested(ShakenAniGroupListRelScheme, many=True)
    allowed_sendto_ips = Nested('AllowedSendToIpGetScheme', many=True)
    purged = Bool()
    cid_block_config = Nested('ResourceCidBlockConfigSchemeGet')
    client_id = Int()
    ip_count = Str()
    shaken_force_validation = Bool()
    ftc_days = Int(allow_none=True, required=False)
    group_id = Int()

    class Meta:
        model = model.IngressTrunk
        fields = ('group_id', 'client_type', 'trunk_type2', 'attestation_settings',
                  'cid_block_config', 'client_id', 'allowed_sendto_ips',
                  'number', 'counter_time', 'block_time') + ResourceAllSheme.Meta.get_fields
        search_fields = (
            'resource_id', 'carrier_id', 'is_active', 'carrier_is_active', 'trunk_id', 'trunk_name', 'update_by',
            'client_name', 'email', 'call_limit', 'cps_limit', 'resource_template_id', 'static_count', 'dynamic_count',
            'ip_count', 'rate_table_id', 'rate_table_name', 'routing_plan_id', 'product_id', 'client_type',
            'trunk_type2', 'has_ip', 'attestation_settings', 'purged', 'client_id', 'ftc_days', 'group_id', 'pdd',
            'min_profit_type', 'shaken_default_attest_lvl', 'min_duration', 'shaken_sign_policy', 'min_profit_value',
            'shaken_vfy_policy')
        query_fields = ('update_at_gt', 'update_at_lt', 'resource_id_in', 'min_profit_value_gt', 'min_profit_value_lt')


class IngressTrunkActivateScheme(IngressTrunkGetScheme):
    active = Bool()

    class Meta:
        model = model.IngressTrunk
        fields = ('active',)
        search_fields = ('carrier_id', 'is_active', 'trunk_id', 'trunk_name', 'update_by',
                         'call_limit', 'cps_limit', 'resource_template_id')
        query_fields = ('update_at_gt', 'update_at_lt', 'resource_id_in')


class IngressTrunkRateSummaryGetScheme(BaseModelScheme):
    trunk_id = Int()
    carrier_id = Int()
    carrier_name = Str(dump_only=True)
    trunk_name = Str()
    rate_table_id = Int()
    rate_table_name = Str(dump_only=True)
    code_deck_name = Str(dump_only=True)
    prefix = Str()
    rate_email = Str()
    tech_prefix = Str()

    class Meta:
        model = model.ResourcePrefix
        fields = ('trunk_id', 'carrier_id', 'carrier_name', 'trunk_name', 'prefix', 'rate_email', 'rate_table_name',
                  'rate_table_id', 'code_deck_name', 'trunk_type2')
        search_fields = (
            'trunk_name', 'carrier_name', 'rate_table_id', 'rate_table_name', 'prefix', 'code_deck_name', 'rate_email',
            'trunk_type2')
        query_fields = ('prefix_isnull',)


class IngressTrunkRateSummaryUniqueGetScheme(BaseModelScheme):
    trunk_id = Int()
    carrier_id = Int()
    carrier_name = Str(dump_only=True)
    trunk_name = Str()
    prefix = Str()
    is_active = Bool()

    class Meta:
        model = model.ResourcePrefix
        fields = ('trunk_id', 'carrier_id', 'carrier_name', 'trunk_name', 'prefix')
        search_fields = (
            'trunk_name', 'carrier_name', 'prefix', 'trunk_type2', 'is_active')
        query_fields = ('prefix_isnull',)


# -- IngressTrunk

# +++ ProductCodeName

class ProductCodeNameScheme(BaseModelScheme):
    item_id = Int()
    product_id = Int()
    code_name = Str()
    resources = List(Int())

    class Meta:
        model = model.ProductCodeName
        fields = ('product_id', 'code_name', 'resources')


class ProductCodeNameGetScheme(ProductCodeNameScheme):
    class Meta:
        model = model.ProductCodeName
        fields = ('item_id', 'product_id', 'code_name', 'resources')
        search_fields = ('item_id', 'product_id', 'code_name')


class ProductCodeNameResourceScheme(BaseModelScheme):
    item_id = Int()
    resource_id = Int()

    class Meta:
        model = model.ProductCodeNameResource
        fields = ('resource_id',)


class ProductCodeNameResourceGetScheme(ProductCodeNameResourceScheme):
    class Meta:
        model = model.ProductCodeNameResource
        fields = ('item_id', 'resource_id')
# -- ProductCodeName

# +++ ClientPayment---

class ClientPaymentMinimalSheme(BaseModelScheme):
    # client_id = Int(attribute='client_id',load_from='client_id',dump_to='client_id' )
    # client_name = Str(attribute='client_name',load_from='client_name',dump_to='client_name' )
    amount = Float()

    # paypal_transaction_id = Str(attribute='paypal_transaction_id',load_from='paypal_transaction_id',dump_to='paypal_transaction_id' )
    # paypal_fee = Number(attribute='paypal_fee',load_from='paypal_fee',dump_to='paypal_fee' )
    # strip_transaction_id = Str(attribute='strip_transaction_id',load_from='strip_transaction_id',dump_to='strip_transaction_id' )
    # type = Int(attribute='payment_type',load_from='type',dump_to='type' )
    # actual_received = Number(attribute='actual_received',load_from='actual_received',dump_to='actual_received' )
    # strip_id = Str(attribute='strip_id',load_from='strip_id',dump_to='strip_id' )
    # paid_on = Str(attribute='paid_on',load_from='paid_on',dump_to='paid_on' )
    # paypal_id = Number(attribute='paypal_id',load_from='paypal_id',dump_to='paypal_id' )
    # entered_on = Str(attribute='entered_on',load_from='entered_on',dump_to='entered_on' )
    class Meta:
        model = model.ClientPayment
        fields = ('amount',)

    @pre_load
    def before_create(self, data):  # not needed
        data['payment_time'] = datetime.now()
        return data


class ClientPaymentScheme(BaseModelScheme):
    # gen   post /carrier/{carrier_id}/payment
    # Add Payment
    amount = Float(default='0.0', validate=validate.Range(-10000000, 10000000))
    balance = Float()
    # egress_amount = Float(default='0.0')
    note = Str(validate=validate.Length(max=500), allow_none=True)
    paid_on = DateTime()
    payment_type = Str(allow_none=False, required=True, validate=validate.OneOf(model.ClientPayment.USER_PAYMENT_TYPES))
    invoice_number = Str(allow_none=True, required=False,
                         validate=lambda value: _valid('Invoice', 'invoice_number', value))

    class Meta:
        model = model.ClientPayment
        fields = ('amount', 'note', 'paid_on', 'payment_type', 'invoice_number')  # ,'egress_amount')

    @pre_load
    def before_create(self, data):  # not needed
        if 'payment_type' in data and data['payment_type'] == 'invoice payment received':
            if not 'invoice_number' in data:
                raise ValidationError(
                    {'invoice_number': ['must be set on this payment type: {}'.format(data['payment_type'])]})
            if model.Invoice.filter(model.Invoice.invoice_number == data['invoice_number']).one().state == 'void':
                raise ValidationError(
                    {'invoice_number': ['invoice has "void" state '.format(data['invoice_number'])]})
        return data


class ClientPaymentExtraScheme(ClientPaymentScheme):
    # gen   post /carrier/{carrier_id}/payment
    # Add Payment
    amount = Float(default='0.0', validate=validate.Range(-10000000, 10000000))
    charge_name = Str(validate=validate.Length(max=50))
    description = Str(validate=validate.Length(max=500), attribute='note')

    class Meta:
        model = model.ClientPayment
        fields = ('amount', 'description', 'charge_name')


class ClientPaymentExtraGetScheme(ClientPaymentExtraScheme):
    class Meta:
        model = model.ClientPayment
        fields = ('amount', 'description', 'client_name', 'payment_time', 'charge_name')
        search_fields = ('amount', 'description', 'client_name', 'payment_time', 'charge_name')


class ClientPaymentNoteScheme(BaseModelScheme):
    note = Str(validate=validate.Length(max=500))

    class Meta:
        model = model.ClientPayment
        fields = ('note',)


class ClientPaymentAmountNoteScheme(BaseModelScheme):
    amount = Float(default='0.0', validate=validate.Range(0, 10000000))
    note = Str(validate=validate.Length(max=500))

    class Meta:
        model = model.ClientPayment
        fields = ('note', 'amount')


class ClientPaymentGetScheme(ClientPaymentScheme):
    client_name = Str()
    client_id = Int()
    payment_type_name = Choice()
    invoice_total_amount = Float()
    invoice_paid_amount = Float()
    invoice_unpaid_amount = Float()
    invoice_due_date = Str()
    invoice_period = Str()

    class Meta:
        model = model.ClientPayment
        fields = ('client_payment_id', 'client_id', 'client_name', 'amount', 'egress_amount',
                  'note', 'paid_on', 'update_by', 'payment_time', 'payment_type_name', 'invoice_number',
                  'invoice_total_amount', 'balance',
                  'invoice_paid_amount', 'invoice_unpaid_amount', 'invoice_due_date', 'invoice_period')
        search_fields = (
            'client_payment_id', 'client_id', 'client_name', 'invoice_number', 'invoice_due_date', 'payment_type_name',
            'update_by')
        query_fields = ('paid_on_gt', 'paid_on_lt', 'amount_gt', 'amount_lt',
                        'invoice_total_amount_gt', 'invoice_total_amount_lt',
                        'invoice_paid_amount_gt', 'invoice_paid_amount_lt',
                        'invoice_unpaid_amount_gt', 'invoice_unpaid_amount_lt', 'payment_time_gt', 'payment_time_lt')


class ClientPaymentTransactGetScheme(ClientPaymentScheme):
    date = DateTime()
    transaction_type = Choice()
    current_balance = Decimal()
    client_name = Str()
    receiving_time = DateTime()

    class Meta:
        model = model.ClientPayment
        fields = ('date', 'transaction_type', 'amount', 'note', 'current_balance', 'client_name', 'receiving_time')
        search_fields = ('transaction_type', 'amount', 'client_name')
        query_fields = ('date_gt', 'date_lt', 'receiving_time_gt', 'receiving_time_lt')


class ClientPaymentSentScheme(BaseModelScheme):
    payment_time = DateTime()
    receiving_time = DateTime()
    payment_type = Str(allow_none=False, required=True, validate=validate.OneOf(model.ClientPayment.USER_PAYMENT_TYPES))
    client_id = Int()
    amount = Decimal()
    note = Str(validate=validate.Length(max=500))

    class Meta:
        model = model.ClientPayment
        fields = ('payment_time', 'receiving_time', 'payment_type', 'amount', 'note', 'client_id')


class ClientPaymentSentGetScheme(BaseModelScheme):
    payment_time = DateTime()
    receiving_time = DateTime()
    payment_type = Choice()
    client_id = Int()
    client_name = Str()
    update_by = Str()
    amount = Decimal()
    note = Str(validate=validate.Length(max=500))
    tz = Int()

    class Meta:
        model = model.ClientPayment
        fields = ('client_payment_id', 'payment_time', 'receiving_time', 'payment_type', 'amount', 'note',
                  'client_id', 'client_name', 'update_by')
        search_fields = ('client_payment_id', 'client_id', 'update_by', 'payment_type', 'client_name')
        query_fields = (
            'payment_time_gt', 'payment_time_lt', 'amount_gt', 'amount_lt', 'receiving_time_gt', 'receiving_time_lt',
            'tz')


class ClientPaymentAllGetScheme(ClientPaymentSentGetScheme):
    invoice_number = Str()
    invoice_id = Int()
    current_balance = Float()

    class Meta:
        model = model.ClientPayment
        fields = ('invoice_id', 'invoice_number', 'current_balance') + ClientPaymentSentGetScheme.Meta.fields
        search_fields = ('invoice_id', 'invoice_number') + ClientPaymentSentGetScheme.Meta.search_fields
        query_fields = ClientPaymentSentGetScheme.Meta.query_fields


# --- ClientPayment---

# ---PaymentInvoice---
class PaymentInvoiceScheme(BaseModelScheme):
    id = Int()
    payment_id = Int()
    invoice_id = Int()
    amount = Decimal()
    client_id = Int()
    invoice = Nested('InvoiceGetScheme')
    payment = Nested('ClientPaymentGetScheme')

    class Meta:
        model = model.PaymentInvoice


# +++ PaymentInvoice


# +++ TransactionFeeItems

class TransactionFeeItemsScheme(BaseModelScheme):
    date = DateTime()
    transaction_type = Choice()
    amount = Float()
    note = Str()

    class Meta:
        model = model.TransactionFeeItems
        fields = ('date', 'transaction_type', 'amount', 'note')
        search_fields = ('date')


# --- TransactionFeeItems


class ObjectCreatedMultipleScheme(SuccessScheme):
    object_ids = List(Int())


# +++ BalanceHistory
class BalanceHistoryGetScheme(BaseModelScheme):
    date = Str()  # DateTime()
    payment_sent = Decimal()
    payment_received = Decimal()
    invoice_sent = Decimal()
    invoice_received = Decimal()
    credit_sent = Decimal()
    credit_received = Decimal()
    mutual_balance = Decimal()
    client_name = Int()
    client_name = Str()

    class Meta:
        model = model.BalanceHistory
        fields = ('client_id', 'date', 'payment_sent', 'invoice_received', 'payment_received', 'invoice_sent',
                  'credit_received', 'credit_sent', 'mutual_balance', 'client_name')
        search_fields = ('date', 'client_id', 'client_name')
        query_fields = ('date_gte', 'date_lte',)


class ActualBalanceHistoryGetScheme(BaseModelScheme):
    date = Str()  # DateTime()
    payment_sent = Decimal()
    payment_received = Decimal()

    # invoice_received = Decimal(allow_none=True,allow_nan=True)
    # invoice_sent = Decimal()
    short_charges = Decimal()

    credit_received = Decimal()
    credit_sent = Decimal()

    incoming_traffic = Decimal()
    outgoing_traffic = Decimal()
    actual_balance = Decimal()
    mrc = Decimal()
    nrc = Decimal()
    # port_fee = Decimal()
    sms = Decimal()

    client_id = Int()
    client_name = Str()
    id = Int()

    class Meta:
        model = model.BalanceHistoryActual
        fields = ('id', 'client_id', 'date', 'payment_sent', 'payment_received', 'short_charges',
                  # 'invoice_received','invoice_sent',
                  'mrc', 'nrc', 'sms',
                  'credit_received', 'credit_sent', 'incoming_traffic', 'outgoing_traffic',
                  'actual_balance', 'client_name')
        search_fields = ('id', 'date', 'client_id', 'client_name')
        query_fields = ('date_gte', 'date_lte',
                        'payment_sent_gte', 'payment_received_gte', 'short_charges_gte', 'credit_received_gte',
                        'credit_sent_gte', 'incoming_traffic_gte', 'outgoing_traffic_gte', 'actual_balance_gte',
                        'payment_sent_lte', 'payment_received_lte', 'short_charges_lte', 'credit_received_lte',
                        'credit_sent_lte', 'incoming_traffic_lte', 'outgoing_traffic_lte', 'actual_balance_lte',
                        )


class ActualBalanceHistoryLogScheme(BaseModelScheme):
    date = Str()
    client_id = Int()
    actual_balance = Decimal()
    client_name = Str()

    class Meta:
        model = model.BalanceHistoryActual
        fields = ('date', 'client_id', 'client_name', 'actual_balance')
        search_fields = ('client_id',)
        query_fields = ('date_gte', 'date_lte')


# --- low balanse config
# +++ClientDid

class ResourceIpDidGetScheme(BaseModelScheme):
    # resource = Nested(resource.ResourceSchemeGet)
    class Meta:
        model = model.ResourceIp
        fields = ('resource_ip_id', 'ip', 'port', 'fqdn')


class ClientResourceDidScheme(BaseModelScheme):
    media_type = Choice()  #
    # ip = Nested('ResourceIpScheme', many=True,allow_none=False,required=True)
    ip = Nested('ResourceIpScheme', many=True)
    billing_by_port = Choice()
    profit_type = Choice()
    allowed_ports = Int()
    rate_decimal = Int(validate=validate.Range(0, 7))
    t38 = Bool()
    price_per_max_channel = Float(validate=validate.Range(0.0, 10000.0))
    price_per_actual_channel = Float(validate=validate.Range(0.0, 10000.0))
    cost_per_port = Float(validate=validate.Range(0.0, 10000.0))
    reg_user = Nested(ResourceIpRegUserScheme, many=True)
    capacity = Int()
    dnis_cap_limit = Int()
    ani_cap_limit = Int()
    auth_type = Choice(allow_none=True, required=False)
    rate_table_id = Int(validate=[lambda value: _valid('RateTable', 'rate_table_id', value)])

    class Meta:
        model = model.Resource
        fields = ('media_type', 'ip', 'billing_by_port', 'profit_type', 'allowed_ports', 'rate_decimal', 't38',
                  'price_per_max_channel', 'price_per_actual_channel', 'cost_per_port', 'capacity',
                  'ani_cap_limit', 'rate_table_id', 'auth_type', 'reg_user', 'dnis_cap_limit')


class ClientResourceDidGetScheme(ClientResourceDidScheme):
    media_type = Choice()
    resource_id = Int()
    rate_decimal = Int()
    ip = Nested('ResourceIpDidGetScheme', many=True)
    billing_by_port = Choice()
    allowed_ports = Int()

    class Meta:
        model = model.Resource
        fields = ('resource_id',) + ClientResourceDidScheme.Meta.fields


class ClientDidScheme(BaseModelScheme):
    name = Str(validate=validate.Regexp(NAME_REGEXP))
    company = Str(validate=validate.Length(min=2, max=500))
    username = Str(validate=validate.Regexp(NAME_REGEXP))
    password = Str()
    main_emails = Emails()
    billing_email = Emails()
    noc_email = Emails()
    rate_email = Emails()
    address = Str(missing='')
    billing_mode = Str(validate=validate.OneOf(['prepay', 'postpay']))
    test_credit = Decimal(allow_none=True, validate=validate.Range(0, 1000000))
    billing_method = Choice()
    # credit_limit = Float()
    call_limit = Int(allow_none=True, missing='0')
    # cps_limit = Int(missing='0')

    auto_invoice_type = Choice()
    # resource = Nested(ClientResourceDidScheme, many=False,allow_none=False,required=True)
    resource = Nested(ClientResourceDidScheme, many=False)

    is_panel_accountsummary = Bool()
    is_panel_block_list = Bool()
    is_panel_ratetable = Bool()
    is_panel_trunks = Bool()
    is_panel_products = Bool()
    is_panel_balance = Bool()
    is_panel_paymenthistory = Bool()
    is_panel_invoices = Bool()
    is_panel_cdrslist = Bool()
    is_panel_summaryreport = Bool()
    is_panel_mydid = Bool()
    is_panel_didrequest = Bool()
    is_panel_cid_blocking = Bool()
    is_panel_onlinepayment = Bool()
    enable_client_portal = Bool()
    enable_paypal = Bool()
    enable_strip = Bool()
    enable_trunk_edit = Bool()
    enable_trunk_view = Bool()
    active = Bool()
    allowed_credit = Decimal(attribute='_credit_limit')
    unlimited_credit = Bool()
    payment_term_id = Int(validate=lambda value: _valid('PaymentTerm', 'payment_term_id', value))
    auto_invoicing = Bool()
    # auto_send_invoice = Bool()
    email_invoice = Bool()
    include_tax = Bool()
    tax = Decimal()
    client_taxes = Nested('ClientTaxesScheme', many=True)
    enable_notification = Bool()
    enable_sign_in_notification = Bool()
    enable_short_dur_penalty = Bool()

    did_billing_by = Choice()
    # fee_per_port = Float()
    did_product = Nested('ClientDidProductScheme')
    auth_ip = Nested(UserAuthIpScheme, many=True)

    ani_cap_limit = Int(allow_none=True)
    dnis_cap_limit = Int(allow_none=True)

    allowed_ports = Int(allow_none=True)

    mask_did_start_post = Int()
    mask_did_end_post = Int()
    break_call_on_no_balance = Bool(default=True)

    class Meta:
        model = model.Client
        fields = ('password', 'name', 'company', 'username', 'main_email', 'billing_email', 'noc_email',
                  'rate_email', 'address', 'test_credit', 'billing_mode',
                  'call_limit', 'auto_invoicing', 'email_invoice',  # 'auto_send_invoice',
                  'resource', 'enable_notification', 'enable_short_dur_penalty', 'auth_ip',
                  'is_panel_accountsummary', 'is_panel_ratetable', 'is_panel_trunks', 'is_panel_products',
                  'is_panel_balance', 'is_panel_paymenthistory', 'is_panel_invoices', 'is_panel_cdrslist',
                  'is_panel_summaryreport', 'is_panel_mydid', 'is_panel_didrequest', 'is_panel_block_list',
                  'is_panel_onlinepayment', 'is_panel_cid_blocking', 'enable_paypal', 'enable_strip', 'active',
                  'allowed_credit', 'unlimited_credit', 'payment_term_id', 'include_tax', 'tax', 'client_taxes',
                  'did_billing_by', 'did_product', 'enable_client_portal',
                  'enable_sign_in_notification', 'enable_trunk_view', 'enable_trunk_edit',
                  'ani_cap_limit', 'dnis_cap_limit', 'allowed_ports', 'mask_did_start_post', 'mask_did_end_post',
                  'break_call_on_no_balance')


class ClientDidGetScheme(ClientDidScheme):
    resource = Nested(ClientResourceDidGetScheme, many=False)
    balance = Decimal()
    allowed_credit = Decimal()
    active = Bool()
    did_product = Nested('ClientDidProductSchemeGet')
    did_product_id = Int()
    # assigned_did_count = Int()

    # Price Per Actual Channel
    # Price Per Max Channel
    class Meta:
        model = model.Client
        fields = ('client_id', 'balance', 'allowed_credit', 'update_at', 'update_by',) + ClientDidScheme.Meta.fields
        exclude = ('resource',)
        search_fields = (
        'client_id', 'name', 'enable_notification', 'active', 'did_billing_by', 'fee_per_port', 'did_product_id')
        query_fields = ('update_at_gt', 'update_at_lt')


# ---- ClientDid
# region +++DidNumberUpload+++
class DidNumberUploadScheme(BaseModelScheme):
    id = Int()
    file = Str(validate=[validate.Length(max=512)])
    created_on = DateTime()
    created_by = Str(validate=[validate.Length(max=255)])
    finished_on = DateTime()
    status = Int()
    result = Str()
    num_records = Int()
    num_imported = Int()
    num_skiped = Int()
    duplicates_action = Choice()
    client_id = Int()
    vendor_id = Int()
    client_billing_rule_id = Int()
    vendor_billing_rule_id = Int()

    class Meta:
        model = model.DidNumberUpload
        fields = ('file', 'created_on', 'created_by', 'finished_on', 'status', 'result', 'num_records', 'num_imported',
                  'num_skiped', 'duplicates_action', 'client_id', 'vendor_id', 'client_billing_rule_id',
                  'vendor_billing_rule_id',)


class DidNumberUploadSchemeGet(DidNumberUploadScheme):
    class Meta:
        model = model.DidNumberUpload
        search_fields = ('id', 'file', 'created_by', 'status', 'result', 'num_records', 'num_imported', 'num_skiped',
                         'duplicates_action', 'client_id', 'vendor_id', 'client_billing_rule_id',
                         'vendor_billing_rule_id',)
        query_fields = ('created_on_gt', 'created_on_lt', 'finished_on_gt', 'finished_on_lt',)


class DidNumberUploadSchemeModify(DidNumberUploadScheme):
    pass


# endregion ---DidNumberUpload---
# region +++DidNumberDeleteTask+++
class DidNumberDeleteTaskScheme(BaseModelScheme):
    id = Int()
    operator_user = Str(validate=[validate.Length(max=40)])
    upload_file_path = Str(validate=[validate.Length(max=256)])
    upload_orig_file = Str(validate=[validate.Length(max=100)])
    upload_format_file = Str(validate=[validate.Length(max=100)])
    orig_name = Str(validate=[validate.Length(max=256)])
    repeated_action = Int()
    status = Choice()
    progress = Str(validate=[validate.Length(max=200)])
    create_time = DateTime()
    start_time = DateTime()
    end_time = DateTime()
    import_export_logs_id = Int()
    op_method = Choice()
    client_id = Int()
    vendor_id = Int()

    class Meta:
        model = model.DidNumberDeleteTask
        fields = ('operator_user', 'upload_file_path', 'upload_orig_file', 'upload_format_file', 'orig_name',
                  'repeated_action', 'status', 'progress', 'create_time', 'start_time', 'end_time',
                  'import_export_logs_id', 'op_method', 'client_id', 'vendor_id')


class DidNumberDeleteTaskSchemeGet(DidNumberDeleteTaskScheme):
    class Meta:
        model = model.DidNumberDeleteTask
        fields = ('operator_user', 'upload_file_path', 'upload_orig_file', 'upload_format_file', 'orig_name',
                  'repeated_action', 'status', 'progress', 'create_time', 'start_time', 'end_time',
                  'import_export_logs_id', 'op_method', 'client_id', 'vendor_id', 'fail', 'success', 'total', 'id')
        search_fields = (
            'id', 'operator_user', 'upload_file_path', 'upload_orig_file', 'upload_format_file', 'result_file_path',
            'repeated_action', 'status', 'progress', 'import_export_logs_id', 'op_method', 'client_id', 'vendor_id')
        query_fields = (
            'create_time_gt', 'create_time_lt', 'start_time_gt', 'start_time_lt', 'end_time_gt', 'end_time_lt',)


class DidNumberDeleteTaskSchemeModify(DidNumberDeleteTaskScheme):
    pass


# endregion ---DidNumberDeleteTask---
# region +++DidNumberAssignTask+++
class DidNumberAssignTaskScheme(BaseModelScheme):
    id = Int()
    operator_user = Str(validate=[validate.Length(max=40)])
    upload_file_path = Str(validate=[validate.Length(max=256)])
    upload_orig_file = Str(validate=[validate.Length(max=100)])
    upload_format_file = Str(validate=[validate.Length(max=100)])
    orig_name = Str(validate=[validate.Length(max=256)])
    repeated_action = Int()
    status = Int()
    progress = Str(validate=[validate.Length(max=200)])
    create_time = DateTime()
    start_time = DateTime()
    end_time = DateTime()
    import_export_logs_id = Int()
    op_method = Choice()
    client_id = Int()
    client_billing_rule_id = Int()

    class Meta:
        model = model.DidNumberAssignTask
        fields = ('operator_user', 'upload_file_path', 'upload_orig_file', 'upload_format_file', 'orig_name',
                  'repeated_action', 'status', 'progress', 'create_time', 'start_time', 'end_time',
                  'import_export_logs_id', 'op_method', 'client_id', 'client_billing_rule_id')


class DidNumberAssignTaskSchemeGet(DidNumberAssignTaskScheme):
    class Meta:
        model = model.DidNumberAssignTask
        fields = ('operator_user', 'upload_file_path', 'upload_orig_file', 'upload_format_file', 'orig_name',
                  'repeated_action', 'status', 'progress', 'create_time', 'start_time', 'end_time',
                  'import_export_logs_id', 'op_method', 'client_id', 'client_billing_rule_id')
        search_fields = (
            'id', 'operator_user', 'upload_file_path', 'upload_orig_file', 'upload_format_file', 'result_file_path',
            'repeated_action', 'status', 'progress', 'import_export_logs_id', 'op_method', 'client_id', 'vendor_id')
        query_fields = (
            'create_time_gt', 'create_time_lt', 'start_time_gt', 'start_time_lt', 'end_time_gt', 'end_time_lt',)


class DidNumberAssignTaskSchemeModify(DidNumberAssignTaskScheme):
    pass


# endregion ---DidNumberAssignTask---

# region ---BlockNumberImportTask---
class BlockNumberImportTaskScheme(BaseModelScheme):
    id = Int()
    operator_user = Str(validate=[validate.Length(max=40)])
    import_file_path = Str(validate=[validate.Length(max=256)])
    orig_import_file = Str(validate=[validate.Length(max=100)])
    format_import_file = Str(validate=[validate.Length(max=100)])
    import_log_path = Str(validate=[validate.Length(max=256)])
    redup_in_block_rable_action = Choice()
    redup_in_file_action = Choice()
    status = Choice()
    progress = Str(validate=[validate.Length(max=200)])
    create_time = DateTime()
    start_time = DateTime()
    end_time = DateTime()
    op_method = Choice()
    egress_client_id = Int()
    egress_trunk_id = Int()
    ingress_client_id = Int()
    ingress_trunk_id = Int()

    class Meta:
        model = model.BlockNumberImportTask
        fields = ('id', 'operator_user', 'import_file_path', 'orig_import_file', 'format_import_file',
                  'import_log_path', 'redup_in_block_rable_action', 'redup_in_file_action', 'status', 'progress',
                  'create_time', 'start_time', 'end_time', 'op_method',)
        # , 'egress_client_id', 'egress_trunk_id', 'ingress_client_id', 'ingress_trunk_id')
        search_fields = ('id', 'status', 'create_time', 'op_method', 'operator_user',)
        query_fields = ('create_time_lt', 'create_time_gt',)


# endregion ---BlockNumberImportTask---

# region +++DidNumberUploadTask+++
class DidNumberUploadTaskScheme(BaseModelScheme):
    id = Int()
    operator_user = Str(validate=[validate.Length(max=40)])
    upload_file_path = Str(validate=[validate.Length(max=256)])
    upload_orig_file = Str(validate=[validate.Length(max=100)])
    upload_format_file = Str(validate=[validate.Length(max=100)])
    result_file_path = Str(validate=[validate.Length(max=256)])
    repeated_action = Choice()
    status = Choice()
    progress = Str(validate=[validate.Length(max=200)])
    create_time = DateTime()
    start_time = DateTime()
    end_time = DateTime()
    import_export_logs_id = Int()
    op_method = Choice()
    enable_for_clients = Choice()

    class Meta:
        model = model.DidNumberUploadTask
        fields = ('operator_user', 'upload_file_path', 'upload_orig_file', 'upload_format_file', 'result_file_path',
                  'repeated_action', 'status', 'progress', 'create_time', 'start_time', 'end_time',
                  'import_export_logs_id', 'op_method', 'enable_for_clients')


class DidNumberUploadTaskSchemeGet(DidNumberUploadTaskScheme):
    error_cause = Str()
    error_file_url = Str()

    class Meta:
        model = model.DidNumberUploadTask
        fields = (
            'operator_user', 'upload_file_path', 'upload_orig_file', 'upload_format_file', 'result_file_path',
            'repeated_action', 'status', 'progress', 'create_time', 'start_time', 'end_time', 'id',
            'import_export_logs_id', 'op_method', 'enable_for_clients', 'total', 'fail', 'success',
            'error_cause', 'error_file_url')
        search_fields = (
            'id', 'operator_user', 'upload_file_path', 'upload_orig_file', 'upload_format_file', 'result_file_path',
            'repeated_action', 'status', 'progress', 'import_export_logs_id', 'op_method', 'enable_for_clients',
            'error_cause')
        query_fields = (
            'create_time_gt', 'create_time_lt', 'start_time_gt', 'start_time_lt', 'end_time_gt', 'end_time_lt',)


class DidNumberUploadTaskSchemeModify(DidNumberUploadTaskScheme):
    pass


# endregion ---DidNumberUploadTask---

# region +++CodeDeckImportTask+++
class CodeDeckImportTaskScheme(BaseModelScheme):
    id = Int()
    operator_user = Str(validate=[validate.Length(max=40)])
    orig_import_filename = Str(validate=[validate.Length(max=256)])
    format_import_filename = Str(validate=[validate.Length(max=256)])
    import_log_filename = Str(validate=[validate.Length(max=256)])
    redup_in_code_deck_table_action = Choice()
    redup_in_file_action = Choice()
    delete_all = Str(validate=validate.OneOf(['Yes', 'No']))
    code_deck_id = Int()
    status = Choice()
    progress = Str(validate=[validate.Length(max=1024)])
    start_time = DateTime()
    end_time = DateTime()
    create_time = DateTime()

    class Meta:
        model = model.CodeDeckImportTask
        fields = ('operator_user', 'orig_import_filename', 'format_import_filename', 'import_log_filename',
                  'redup_in_code_deck_table_action', 'redup_in_file_action', 'delete_all', 'code_deck_id', 'status',
                  'progress', 'start_time', 'end_time', 'create_time',)


class CodeDeckImportTaskSchemeGet(CodeDeckImportTaskScheme):
    delete_all = Str(attribute='d_all', validate=validate.OneOf(['Yes', 'No']))

    class Meta:
        model = model.CodeDeckImportTask
        fields = ('operator_user', 'orig_import_filename', 'format_import_filename', 'import_log_filename',
                  'redup_in_code_deck_table_action', 'redup_in_file_action', 'delete_all', 'code_deck_id', 'status',
                  'progress', 'start_time', 'end_time', 'create_time',)
        search_fields = ('id', 'operator_user', 'orig_import_filename', 'format_import_filename', 'import_log_filename',
                         'redup_in_code_deck_table_action', 'redup_in_file_action', 'delete_all', 'code_deck_id',
                         'status', 'progress',)
        query_fields = (
            'start_time_gt', 'start_time_lt', 'end_time_gt', 'end_time_lt', 'create_time_gt', 'create_time_lt',)


class CodeDeckImportTaskSchemeModify(CodeDeckImportTaskScheme):
    pass


# endregion ---CodeDeckImportTask---

# region +++Watchdog+++

class WatchdogServiceActionScheme(BaseModelScheme):
    action = Str()

    class Meta:
        fields = ('action',)


# endregion ---Watchdog---

def valid_free_did(value):
    cls = model.DidBillingRel
    q = cls.filter(and_(cls.did == str(value), cls.end_date.is_(None))).first()
    if not q:
        raise ValidationError('invalid did {}'.format(value))


class ClientDidAssignDidsScheme(ClientDidScheme):
    dids = List(Int(validate=valid_free_did))
    force_reassign = Bool()

    class Meta:
        model = model.Client
        fields = ('dids', 'force_reassign')


class AvailableCountrySearchScheme(Schema):
    country = Str(validate=[validate.Length(min=3, max=3)])


class AvailableStateSearchScheme(Schema):
    state = Str(validate=[validate.Length(min=3, max=3)])
    country = Str(validate=[validate.Length(min=3, max=3)])


class AvailableLataSearchScheme(Schema):
    lata = Str(validate=[validate.Length(min=3, max=3)])
    country = Str(validate=[validate.Length(min=3, max=3)])


class AvailableNpaSearchScheme(Schema):
    npa = Str(validate=[validate.Length(min=3, max=3)])


# +++VendorDid
class VendorDidBillingRelScheme(BaseModelScheme):
    did = Str(validate=validate.Regexp(PREFIX_REGEXP))
    vendor_billing_rule_id = Int(validate=lambda value: _valid('DidBillingPlan', 'id', value))
    client_res_id = Int(allow_none=True, required=False,
                        validate=lambda value: _valid('Resource', 'resource_id', value))
    client_billing_rule_id = Int(allow_none=True, required=False,
                                 validate=lambda value: _valid('DidBillingPlan', 'id', value))

    class Meta:
        model = model.DidBillingRel
        fields = ('did', 'vendor_billing_rule_id', 'client_res_id', 'client_billing_rule_id')


class VendorResourceDidScheme(BaseModelScheme):
    media_type = Choice()  #
    ip = Nested('ResourceIpScheme', many=True)
    billing_by_port = Choice()
    t38 = Bool()
    rate_decimal = Int(validate=validate.Range(0, 7))
    rate_rounding = Choice()
    vendor_tech_prefix = Str(allow_none=True, required=False, validate=validate.Regexp(PREFIX_REGEXP))
    vendor_dids = Nested('VendorDidBillingRelScheme', many=True, required=False, allow_none=True)
    rate_table_id = Int(validate=[lambda value: _valid('RateTable', 'rate_table_id', value)])

    @pre_load
    def fix_dids(self, data):
        if 'vendor_dids' in data:
            nd = []
            for d in data['vendor_dids']:
                if ',' in d['did']:
                    for n in d['did'].split(','):
                        if 'vendor_billing_rule_id' in d:
                            nd.append({'did': n, 'vendor_billing_rule_id': d['vendor_billing_rule_id']})
                        else:
                            nd.append({'did': n})
                else:
                    nd.append(d)
            data['vendor_dids'] = nd
        return data

    class Meta:
        model = model.Resource
        fields = ('media_type', 'ip', 'billing_by_port', 't38', 'vendor_tech_prefix', 'vendor_dids', 'rate_decimal',
                  'rate_rounding', 'rate_table_id')


class VendorResourceDidGetScheme(VendorResourceDidScheme):
    resource_id = Int()
    rate_decimal = Int()
    vendor_dids = Nested('DidBillingRelGetScheme', many=True)

    class Meta:
        model = model.Resource
        fields = ('resource_id',) + VendorResourceDidScheme.Meta.fields
        query_fields = ('resource_id_in',)


class VendorResourceDidGetInnerScheme(VendorResourceDidScheme):
    resource_id = Int()
    rate_decimal = Int()

    class Meta:
        model = model.Resource
        fields = ('resource_id', 'media_type', 'ip', 'billing_by_port', 't38', 'vendor_tech_prefix', 'rate_decimal',
                  'rate_rounding', 'rate_table_id')
        query_fields = ('resource_id_in',)


class VendorResourceDidModifyScheme(VendorResourceDidScheme):
    class Meta:
        model = model.Resource
        fields = ('media_type', 'ip', 'billing_by_port', 't38', 'vendor_tech_prefix', 'rate_decimal',
                  'rate_rounding')


class VendorDidScheme(BaseModelScheme):
    # FROM MODEL
    # {"errors": [{"call_limit": ["Field may not be null."], "address": ["Field may not be null."], "account_detail": ["Field may not be null."], "short_call_charge": ["Field may not be null."], "cps_limit": ["Field may not be null."]}], "error_type": "validation_error", "success": false}

    name = Str(required=True, allow_none=False, validate=validate.Regexp(NAME_REGEXP))
    resource = Nested(VendorResourceDidScheme, many=False, required=True, allow_none=False)
    call_limit = Int(missing='0', allow_none=True)
    billing_method = Choice()
    status = Bool()
    # vendor_billing_rule_id = Int(allow_none=True, required=False, validate=lambda value: _valid('DidBillingPlan', 'id', value))
    vendor_api = Nested('DidVendorScheme', allow_none=True)

    class Meta:
        model = model.Client
        fields = ('name', 'resource', 'call_limit', 'billing_method', 'status', 'vendor_api')

    @pre_load
    def fix_data(self, data):
        if 'call_limit' in data and data['call_limit'] in ('null', ''):
            data['call_limit'] = None
        return data


class VendorDidGetScheme(VendorDidScheme):
    balance = Decimal()
    resource = Nested(VendorResourceDidGetInnerScheme, many=False)
    vendor_res_id = Int(dump_only=True)
    has_api = Bool()
    vendor_tech_prefix = Str()
    vendor_api = Nested('DidVendorSchemeGet')

    class Meta:
        model = model.Client
        fields = ('client_id', 'has_api', 'update_at', 'update_by', 'balance') + VendorDidScheme.Meta.fields
        search_fields = ('client_id', 'has_api', 'name', 'status', 'vendor_res_id', 'vendor_tech_prefix')
        query_fields = ('update_at_gt', 'update_at_lt')


class VendorDidModifyScheme(VendorDidScheme):
    resource = Nested(VendorResourceDidModifyScheme, many=False, required=True, allow_none=False)


# ---- VendorDid

class DidRepositoryScheme(BaseModelScheme):
    id = Int()
    did = Str(validate=validate.Regexp(PREFIX_REGEXP))  # validate=lambda value : _valid('Rate','code',value))#
    vendor_trunk_id = Int(validate=lambda value: _valid('Resource', 'resource_id', value))
    # client_trunk_id = Int(validate=lambda value: _valid('Resource', 'resource_id', value))
    vendor_billing_plan_id = Int(validate=lambda value: _valid('DidBillingPlan', 'id', value))
    vendor_billing_rule_name = Str(validate=lambda value: _valid('DidBillingPlan', 'name', value))
    did_vendor_name = Str()
    created_at = DateTime()
    created_by = Int()
    vendor_id = Int()

    class Meta:
        model = model.DidRepository
        fields = (
            'did', 'did_vendor_name',
            'vendor_billing_rule_name',
            'vendor_billing_plan_id', 'vendor_trunk_id',
            'vendor_id'
        )


class DidRepositoryModifyScheme(DidRepositoryScheme):
    actions = Nested('DidBillingRelActionsScheme')
    class Meta:
        model = model.DidRepository
        fields = ('did', 'vendor_trunk_id', 'vendor_billing_plan_id', 'did_vendor_name', 'vendor_billing_rule_name', 'actions')


class DidRepositoryGetScheme(DidRepositoryScheme):
    did_vendor_id = Int(validate=lambda value: _valid('Resource', 'resource_id', value))
    assigned = Boolean(required=False, dump_only=False)

    class Meta:
        model = model.DidRepository
        fields = ('id', 'did', 'vendor_trunk_id', 'vendor_billing_plan_id', 'created_at',
                  'created_by', 'did_vendor_id', 'did_vendor_name', 'vendor_billing_rule_name',
                  'client_id', 'did_client_name', 'client_billing_rule_name', 'client_trunk_id')
        search_fields = ('id', 'did', 'vendor_trunk_id', 'vendor_billing_plan_id', 'created_at',
                  'created_by', 'did_vendor_name', 'vendor_billing_rule_name',
                  'client_id', 'client_billing_plan_id', 'fallback_id', 'client_trunk_id',
                  'client_billing_rule_name', 'did_client_name', 'country', 'did_client_name_is_null',
                  'state', 'is_available', 'is_assigned', 'active', 'status', 'vendor_id',
                  'client_id', 'did_client_name', 'client_billing_rule_name', 'assigned'
                  )

class DidAssignmentsGetScheme(BaseModelScheme):
    id = Int()
    did = Str(validate=validate.Regexp(PREFIX_REGEXP))
    vendor_trunk_id = Int(validate=lambda value: _valid('Resource', 'resource_id', value))
    client_trunk_id = Int(validate=lambda value: _valid('Resource', 'resource_id', value))
    vendor_billing_plan_id = Int(validate=lambda value: _valid('DidBillingPlan', 'id', value))
    vendor_id = Int()
    client_id = Int()

    class Meta:
        model = model.DidAssignments
        fields = (
            'did', 'vendor_trunk_id', 'client_trunk_id', 'vendor_billing_plan_id',
            'vendor_billing_plan_id', 'vendor_id', 'client_id'
        )
        search_fields = (
            'did', 'vendor_trunk_id', 'client_trunk_id', 'vendor_billing_plan_id',
            'vendor_billing_plan_id', 'vendor_id', 'client_id'
        )

class DidBillingRelScheme(BaseModelScheme):
    did = Str(validate=validate.Regexp(PREFIX_REGEXP))  # validate=lambda value : _valid('Rate','code',value))#
    client_res_id = Int(allow_none=True, validate=lambda value: _valid('Resource', 'resource_id', value))
    client_billing_rule_id = Int(validate=lambda value: _valid('DidBillingPlan', 'id', value))
    vendor_res_id = Int(validate=lambda value: _valid('Resource', 'resource_id', value))
    vendor_billing_rule_id = Int(validate=lambda value: _valid('DidBillingPlan', 'id', value))
    assigned_date = DateTime(required=False)
    end_date = DateTime(required=False)
    did_vendor_name = Str()
    did_client_name = Str()
    client_id = Int(allow_none=True, validate=lambda value: _valid('Client', 'client_id', value))
    vendor_id = Int(allow_none=True, validate=lambda value: _valid('Client', 'client_id', value))
    is_available = Bool()
    is_effective = Bool(dump_only=True)
    is_assigned = Bool(dump_only=True)
    assigned_client_id = Int(allow_none=True, validate=lambda value: _valid('Client', 'client_id', value))
    ip = Str(allow_none=True, validate=validate.Regexp(IP_REGEXP))
    note = Str()
    status = Bool(attribute='_is_available')

    @pre_load
    def fix_empty(self, data):
        for f in ('client_id', 'client_res_id', 'client_billing_rule_id'):
            if f in data and data[f] == '':
                data[f] = None
        return data

    class Meta:
        model = model.DidBillingRel
        fields = ('did', 'client_id', 'client_billing_rule_id', 'vendor_res_id', 'assigned_client_id',
                  'vendor_billing_rule_id', 'is_available', 'ip', 'note', 'vendor_id'  # ,'start_date','end_date'
                  )


class DidBillingRelImportScheme(DidBillingRelScheme):
    client_billing_rule_name = Str()
    vendor_billing_rule_name = Str()
    assigned_client_id = Int()

    class Meta:
        model = model.DidBillingRel
        fields = ('did', 'client_id', 'client_billing_rule_id', 'vendor_res_id',
                  'vendor_billing_rule_id', 'is_available', 'client_billing_rule_name', 'vendor_billing_rule_name',
                  'did_client_name', 'did_vendor_name', 'assigned_client_id'
                  )

    def validate_ids(self, data):
        for field in ['client_billing_rule', 'vendor_billing_rule']:
            if field + '_name' in data:
                if field + '_id' in data:
                    raise ValidationError('invalid input! Both {}_name and {}_id!'.format(field, field))
        if 'vendor_res_id' in data and 'did_vendor_name' in data:
            raise ValidationError('invalid input! Both {}_name and {}_id!'.format('vendor_res_id', 'did_vendor_name'))
        if 'client_res_id' in data and 'did_client_name' in data:
            raise ValidationError('invalid input! Both {}_name and {}_id!'.format('client_res_id', 'did_client_name'))
        return data

    @pre_load
    def load_ids(self, obj):
        self.validate_ids(obj)
        for field in ['client_billing_rule', 'vendor_billing_rule', 'did_client', 'did_vendor']:
            name = field + '_name'
            if name in obj:
                if 'billing_rule' in field:
                    obj[field + '_id'] = model._ref_id('DidBillingPlan', 'name', obj[name])
                    del obj[name]
                    continue
                if 'did_vendor' in field:
                    client_id = model._ref_id('Client', 'name', obj[name])
                    res_id = model.Client.get(client_id).resource.resource_id
                    obj[field[4:] + '_res_id'] = res_id
                    del obj[name]
                    continue
                if 'did_client' in field:
                    client_id = model._ref_id('Client', 'name', obj[name])
                    res_id = model.Client.get(client_id).resource.resource_id
                    obj[field[4:] + '_res_id'] = res_id
                    # obj['client_id'] = client_id
                    del obj[name]
                    continue

        return obj


class DidBillingRelActionsInnerScheme(Schema):
    type = Str(validate=validate.OneOf(('Add Prefix', 'Remove Prefix', 'Replace')))
    digits = Str(validate=validate.Regexp(PREFIX_REGEXP))
    num_digits = Int(validate=validate.Range(1, 15))
    new_number = Str(validate=validate.Regexp(PREFIX_REGEXP))

    @pre_load
    def check(self, data):
        if 'type' in data:
            t = data['type']
            if t == 'Add Prefix' and not 'digits' in data:
                raise ValidationError('digits mandatory when add prefix!')
            if t == 'Remove Prefix' and not 'num_digits' in data:
                raise ValidationError('num_digits mandatory when remove prefix!')
            if t == 'Replace' and not 'new_number' in data:
                raise ValidationError('new_number mandatory when replace!')
        return data


class DidBillingRelActionsScheme(Schema):
    ani = Nested(DidBillingRelActionsInnerScheme, allow_none=True, required=False)
    dnis = Nested(DidBillingRelActionsInnerScheme, allow_none=True, required=False)


class DidBillingRelModifyScheme(BaseModelScheme):
    did = Str()  # validate=lambda value : _valid('Rate','code',value))#
    client_res_id = Int(allow_none=True, validate=lambda value: _valid('Resource', 'resource_id', value))
    client_billing_rule_id = Int(allow_none=True, validate=lambda value: _valid('DidBillingPlan', 'id', value))
    vendor_res_id = Int(allow_none=True, validate=lambda value: _valid('Resource', 'resource_id', value))
    vendor_billing_rule_id = Int(allow_none=True, validate=lambda value: _valid('DidBillingPlan', 'id', value))
    assigned_date = DateTime(required=False)
    end_date = DateTime(required=False, validate=valid_future_date)
    actions = Nested(DidBillingRelActionsScheme)
    client_id = Int(allow_none=True, validate=lambda value: _valid('Client', 'client_id', value),
                    attribute='_client_id')
    vendor_id = Int(allow_none=True, validate=lambda value: _valid('Client', 'client_id', value))
    is_available = Bool()
    is_effective = Bool(dump_only=True)
    is_assigned = Bool(dump_only=True)
    assigned_client_id = Int(allow_none=True, validate=lambda value: _valid('Client', 'client_id', value))
    active = Bool()

    class Meta:
        model = model.DidBillingRel
        fields = ('client_id', 'client_billing_rule_id', 'vendor_res_id',
                  'vendor_billing_rule_id', 'actions', 'is_available', 'assigned_client_id', 'ip', 'note',
                  'vendor_id', 'active'
                  # , 'start_date', 'end_date'
                  )


class DidBillingRelGetScheme(DidBillingRelScheme):
    id = Int()  #
    assigned_date = Str()
    end_date = Str()
    actions = Dict()
    client_id = Int(dump_only=True)
    vendor_id = Int(dump_only=True)
    did_client_name = Str(dump_only=True)
    did_vendor_name = Str(dump_only=True)
    did_client_res_name = Str(dump_only=True)
    did_vendor_res_name = Str(dump_only=True)
    vendor_billing_rule_name = Str(dump_only=True)
    client_billing_rule_name = Str(dump_only=True)
    is_effective = Bool(dump_only=True)
    is_assigned = Bool(dump_only=True)
    is_available = Bool(dump_only=True)
    did_type = Str()
    # status = Choice()
    is_sms = Bool()
    assigned_client_id = Int()
    assigned_client_name = Str()
    ip = Str()
    note = Str()
    vendor_tech_prefix = Str()
    ocn = Str()
    lata = Str()
    country = Str()
    state = Str()
    clec = Str()
    rate_type = Str()
    company = Str()
    switch = Str()
    active = Bool()

    class Meta:
        model = model.DidBillingRel
        fields = ('id', 'did', 'client_res_id', 'client_billing_rule_id', 'vendor_res_id', 'vendor_billing_rule_id',
                  ###'create_time',
                  'did_vendor_res_name', 'did_client_res_name', 'assigned_client_id', 'assigned_client_name',
                  'did_vendor_name', 'did_client_name', 'assigned_date', 'end_date', 'actions', 'is_available',
                  'client_id', 'vendor_id', 'vendor_billing_rule_name', 'client_billing_rule_name', 'is_effective',
                  'is_assigned', 'did_type', 'status', 'is_sms', 'ip', 'vendor_tech_prefix', 'note',
                  'ocn', 'lata', 'country', 'state', 'rate_type', 'company', 'switch', 'clec', 'active')
        search_fields = ('id', 'did', 'client_res_id', 'client_billing_rule_id', 'vendor_res_id',
                         'vendor_billing_rule_id', 'is_available', 'client_id', 'vendor_id', 'did_client_name',
                         'did_vendor_name', 'assigned_client_id', 'assigned_client_name',
                         'vendor_billing_rule_name', 'client_billing_rule_name', 'did_vendor_res_name',
                         'did_client_res_name', 'is_effective', 'is_assigned', 'did_type', 'status', 'is_sms', 'ip',
                         'vendor_tech_prefix', 'note', 'active',
                         'ocn', 'lata', 'country', 'state', 'rate_type', 'company', 'switch', 'clec')
        query_fields = ('assigned_date_gt', 'assigned_date_lt', 'end_date_gt', 'end_date_lt', 'end_date_isnull',
                        'client_res_id_isnull', 'client_billing_rule_id_isnull', 'vendor_billing_rule_id_isnull')


# query_fields = ()#'created_time_gt', 'created_time_lt','assigned_time_gt', 'assigned_time_lt')


class DidBillingRelMassAssignScheme(BaseModelScheme):
    dids = List(Str(validate=lambda value: _valid('DidRepository', 'did', value)))
    client_id = Int(allow_none=True, validate=lambda value: _valid('Client', 'client_id', value))
    client_billing_rule_id = Int(allow_none=True, validate=lambda value: _valid('DidBillingPlan', 'id', value))
    assigned_client_id = Int(allow_none=True, validate=lambda value: _valid('Client', 'client_id', value))
    src_client_id = Int(allow_none=True, validate=lambda value: _valid('Client', 'client_id', value))

    class Meta:
        model = model.DidRepository
        fields = (
            'dids',
            'client_id',
            'client_billing_rule_id',
            'assigned_client_id',
            'src_client_id'
        )


class DidBillingRelAvailableGetScheme(DidBillingRelScheme):
    id = Int()  #
    create_time = DateTime()
    assign_time = DateTime()
    assigned_date = DateTime()
    actions = Dict()
    vendor_billing_rule = Nested('DidBillingPlanGetScheme')

    class Meta:
        model = model.DidBillingRel
        fields = ('id', 'did',  # 'client_res_id', 'client_billing_rule_id',
                  'vendor_res_id', 'vendor_billing_rule_id', 'vendor_billing_rule',
                  ###'create_time',
                  'did_vendor_name',
                  # 'did_client_name','assigned_date','end_date',
                  'actions', 'is_available',
                  )
        search_fields = ('id', 'did', 'vendor_res_id', 'vendor_billing_rule_id')


# ---ClientDid

# +++ Client Credit Management
class ClientCreditScheme(BaseModelScheme):
    credit_limit = Decimal(allow_none=True, attribute='_credit_limit')
    call_limit = Int(missing='0', allow_none=True)
    cps_limit = Int(missing='0', allow_none=True)
    active = Bool()
    payment_term_id = Int(validate=lambda value: _valid('PaymentTerm', 'payment_term_id', value))
    unlimited_credit = Bool()

    class Meta:
        model = model.Client
        fields = ('active', 'credit_limit', 'call_limit', 'cps_limit', 'payment_term_id', 'unlimited_credit')


class ClientCreditSchemeGet(BaseModelScheme):
    name = Str()
    credit_limit = Decimal()
    call_limit = Int(missing='0')
    cps_limit = Int(missing='0')
    billing_mode = Str()
    payment_term_name = Str()
    last_payment = Nested('ClientPaymentScheme')
    active = Bool()
    unlimited_credit = Bool()
    actual_balance = Float()
    last_invoice_amount = Float()
    last_invoice_total_charge = Float()
    last_invoice_number = Str()

    update_at = Str()
    update_by = Str()
    last_payment_date = DateTime()
    last_payment_amount = Float()

    class Meta:
        model = model.Client
        fields = ('client_id', 'name', 'active', 'credit_limit', 'call_limit', 'cps_limit', 'billing_mode',
                  'payment_term_name', 'last_payment', 'update_at', 'update_by', 'unlimited_credit', 'actual_balance',
                  'last_invoice_amount', 'last_invoice_number', 'last_invoice_total_charge')
        search_fields = ('active', 'name', 'call_limit', 'cps_limit', 'billing_mode', 'payment_term_name', 'update_by',
                         'unlimited_credit')
        query_fields = (
            'credit_limit_gt', 'update_at_gt', 'last_payment_date_gt', 'last_payment_amount_gt', 'actual_balance_gt',
            'actual_balance_lt')


# ---

class CarrierLowBalanceConfigScheme(BaseModelScheme):
    actual_notify_balance = Float(allow_none=True)
    percentage_notify_balance = Float(allow_none=True)
    value_type = Choice()
    send_time_type = Choice()

    class Meta:
        model = model.CarrierLowBalanceConfig
        # exclude = ('client_id', 'client')
        fields = (
            'is_notify',
            'value_type',
            'actual_notify_balance',
            'percentage_notify_balance',
            'send_time_type',
            'daily_send_time',
            'duplicate_days',
            'send_to',
            'duplicate_send_days',
            'disable_trunks_days')


# --- Carrier invoice_setting
class CarrierInvoiceSettingScheme(BaseModelScheme):
    # gen module  carrier-auto-invoice
    is_breakdown_by_rate_table = Bool()
    show_trunk_summary = Bool(attribute='is_show_total_trunk', load_from='show_trunk_summary',
                              dump_to='show_trunk_summary')
    decimal = Int(attribute='decimal_place', load_from='decimal', dump_to='decimal', default=5)
    show_country_summary = Bool(attribute='is_show_country', load_from='show_country_summary',
                                dump_to='show_country_summary')
    inlcude_cdr_in_email = Bool(attribute='attach_cdrs_list', load_from='inlcude_cdr_in_email',
                                dump_to='inlcude_cdr_in_email')
    show_payment_summary = Bool(attribute='invoice_include_payment', load_from='show_payment_summary',
                                dump_to='show_payment_summary')
    send_invoice_as_link = Bool()
    auto_send_invoice = Bool(attribute='email_invoice', load_from='email_invoice', dump_to='email_invoice')
    show_daily_usage = Bool(attribute='is_show_daily_usage')
    payment_term = Str(attribute='pt_name', validate=validate_payment_term)
    include_tax = Bool(attribute='include_tax', load_from='include_tax', dump_to='include_tax')
    tax = Float(validate=validate.Range(0.01, 99.99))
    show_account_summary = Bool(attribute='is_invoice_account_summary', load_from='show_account_summary',
                                dump_to='show_account_summary')
    rate_value = Choice()
    show_jurisdiction_detail = Bool(attribute='invoice_jurisdictional_detail', load_from='show_jurisdiction_detail',
                                    dump_to='show_jurisdiction_detail')
    enable_auto_invoice = Bool(attribute='auto_invoicing', default=False, allow_none=True)
    cdr_format = Choice()
    show_code_name_summary = Bool(attribute='is_show_code_name', load_from='show_code_name_summary',
                                  dump_to='show_code_name_summary')
    time_zone = Str(attribute='invoice_zone', load_from='time_zone', dump_to='time_zone',
                    validate=validate.Regexp(TIMEZONE_REGEXP))
    format = Choice()
    show_code_summary = Bool(attribute='is_show_code_100', load_from='show_code_summary', dump_to='show_code_summary')
    include_short_call_charge = Bool(attribute='is_short_duration_call_surcharge_detail',
                                     load_from='include_short_call_charge', dump_to='include_short_call_charge')
    non_zero_invoice_only = Bool()
    usage_fields = List(Str(validate=lambda value: _valid('DailyCdrField', 'field', value)))
    ingress_prefix = Bool(
        attribute='is_show_detail_trunk')  # show_detail_by_trunk = Bool(attribute='is_show_detail_trunk')
    show_calls_date = Bool(attribute='is_show_by_date')
    include_detail = Bool(attribute='invoice_show_details')
    client_invoice_settings_id = Int(allow_none=True, attribute='_client_invoice_settings_id',
                                     validate=lambda value: _valid('InvoiceSetting', 'id', value))

    class Meta:
        model = model.Client
        fields = (
            'is_breakdown_by_rate_table', 'show_trunk_summary', 'decimal', 'show_country_summary',
            'inlcude_cdr_in_email', 'client_invoice_settings_id', 'enable_auto_invoice',
            'show_payment_summary', 'send_invoice_as_link', 'auto_send_invoice', 'show_daily_usage',
            'payment_term', 'include_tax', 'tax', 'show_account_summary', 'rate_value',
            'show_jurisdiction_detail', 'cdr_format', 'show_code_name_summary', 'time_zone',
            'format', 'show_code_summary', 'include_short_call_charge', 'non_zero_invoice_only', 'usage_fields',
            'ingress_prefix', 'show_calls_date', 'include_detail'
        )


class CarrierAlertsScheme(BaseModelScheme):
    # gen   patch /Carrier/{client_id}/alerts
    notify_client_balance_type = Int(attribute='notify_client_balance_type')
    attach_cdrs_list = Bool(attribute='attach_cdrs_list')
    is_auto_summary = Bool(attribute='is_auto_summary')
    auto_summary_hour = Int(attribute='auto_summary_hour')
    is_send_trunk_update = Bool(attribute='is_send_trunk_update')
    low_balance_notification_time_type = Int(attribute='low_balance_notification_time_type')
    auto_summary_period = Int(attribute='auto_summary_period', default=24)
    auto_summary_group_by = Str() #Choice()  # Str(attribute='auto_summary_group_by',
    # load_from='auto_summary_group_by',dump_to='auto_summary_group_by' )
    auto_summary_not_zero = Bool(attribute='daily_usage_on_non_zero')
    low_balance_notification_time_cycle = Int(attribute='low_balance_notification_time_cycle')
    daily_cdr_generation_zone = Str(attribute='daily_cdr_generation_zone')
    daily_cdr_generation = Bool(attribute='daily_cdr_generation')
    low_balance_notice = Bool(attribute='low_balance_notice')
    zero_balance_notice = Bool(attribute='zero_balance_notice')
    is_daily_balance_notification = Bool(attribute='is_daily_balance_notification')
    is_show_daily_usage = Bool(attribute='is_show_daily_usage')

    @pre_load
    def fix_data(self, data):
        if not data['auto_summary_group_by']:
            data['auto_summary_group_by'] = None
        return data

    class Meta:
        model = model.Client
        fields = ('notify_client_balance_type', 'attach_cdrs_list', 'is_auto_summary', 'auto_summary_hour',
                  'is_send_trunk_update', 'low_balance_notification_time_type', 'auto_summary_period',
                  'auto_summary_group_by', 'auto_summary_not_zero', 'low_balance_notification_time_cycle',
                  'daily_cdr_generation_zone', 'daily_cdr_generation', 'low_balance_notice',
                  'zero_balance_notice', 'is_daily_balance_notification', 'is_show_daily_usage')


# --- alert_setting
class CarrierAllertSettingsScheme(BaseModelScheme):
    enable_trunk_update_alert = Bool(attribute='is_send_trunk_update', load_from='enable_trunk_update_alert',
                                     dump_to='enable_trunk_update_alert')
    enable_daily_usage_alert = Bool(attribute='is_show_daily_usage', load_from='enable_daily_usage_alert',
                                    dump_to='enable_daily_usage_alert')
    enable_zero_balance_alert = Bool(attribute='zero_balance_notice', load_from='enable_zero_balance_alert',
                                     dump_to='enable_zero_balance_alert')
    enable_low_balance_alert = Bool(attribute='low_balance_notice', load_from='enable_low_balance_alert',
                                    dump_to='enable_low_balance_alert')
    enable_daily_cdr_alert = Bool(attribute='daily_cdr_generation', load_from='enable_daily_cdr_alert',
                                  dump_to='enable_daily_cdr_alert')
    enable_daily_balance_alert = Bool(attribute='is_auto_summary')
    enable_payment_received_alert = Bool(attribute='payment_received_notice')
    is_show_daily_usage = Bool(attribute='is_show_daily_usage')
    daily_usage_on_non_zero = Bool(attribute='daily_usage_on_non_zero')
    daily_usage_group_by = Choice()
    auto_summary_hour = Int(validate=validate.Range(0, 23), default=0)
    auto_send_zone = Str()  # validate=validate.Regexp(r'^[\+\-][0-9][0-9]:[0-9][0-9])$'),default='+00')
    usage_fields = List(Str(validate=lambda value: _valid('DailyCdrField', 'field', value)))
    auto_daily_balance_recipient = Choice()
    enable_payment_alert = Bool()

    class Meta:
        model = model.Client
        fields = (
            'enable_trunk_update_alert', 'enable_daily_usage_alert', 'enable_zero_balance_alert',
            'enable_payment_alert',
            'enable_low_balance_alert', 'enable_daily_cdr_alert', 'enable_daily_balance_alert',
            'enable_payment_received_alert', 'is_show_daily_usage', 'daily_usage_on_non_zero',
            'daily_usage_group_by', 'auto_summary_hour', 'auto_send_zone', 'usage_fields',
            'auto_daily_balance_recipient')


# --- Carrier Short call Charges ---
class CarrierSccScheme(BaseModelScheme):
    # gen   patch /carrier/{client_id}/scc
    # Carrier Short call Charges
    scc_percent = Decimal(validate=validate.Range(0.0, 100.0))
    scc_charge = Decimal(validate=validate.Range(0.0, 100000000.0))
    scc_type = Int(attribute='scc_type')
    scc_below = Int(validate=validate.Range(0, 65535))

    class Meta:
        model = model.Client
        fields = ('scc_percent', 'scc_charge', 'scc_type', 'scc_below')


# --- Carrier portal
class CarrierPortalScheme(BaseModelScheme):
    # gen   patch /Carrier/{client_id}/portal
    # Carrier portal
    is_panelaccess = Bool()
    is_panel_ratetable = Bool()
    is_panel_paymenthistory = Bool()
    is_panel_invoices = Bool()
    is_panel_accountsummary = Bool()
    is_panel_block_list = Bool()
    is_panel_products = Bool()
    is_panel_balance = Bool()
    is_panel_trunks = Bool()
    is_panel_cdrslist = Bool()
    is_panel_sippacket = Bool()
    is_panel_summaryreport = Bool()
    is_panel_onlinepayment = Bool()
    is_panel_cid_blocking = Bool()

    is_panel_mydid = Bool()
    is_panel_didrequest = Bool()  # 'is_panel_mydid','is_panel_didrequest',

    login = Str(validate=validate.Length(0, 60))
    password = Str()
    profit_margin = Decimal()
    rate_email = Emails()

    class Meta:
        model = model.Client
        fields = ('is_panelaccess', 'is_panel_ratetable', 'is_panel_paymenthistory',
                  'is_panel_mydid', 'is_panel_didrequest',
                  'is_panel_invoices', 'is_panel_accountsummary', 'is_panel_products', 'is_panel_balance',
                  'is_panel_trunks', 'is_panel_cdrslist', 'is_panel_sippacket', 'is_panel_summaryreport',
                  'is_panel_onlinepayment', 'is_panel_cid_blocking', 'is_panel_block_list',
                  'login', 'password', 'profit_margin', 'rate_email')


class CarrierInnerUserScheme(BaseModelScheme):
    # gen   patch /Carrier/{client_id}/portal
    # Carrier portal
    is_panelaccess = Bool()
    is_panel_ratetable = Bool()
    is_panel_paymenthistory = Bool()
    is_panel_invoices = Bool()
    is_panel_accountsummary = Bool()
    is_panel_block_list = Bool()
    is_panel_products = Bool()
    is_panel_balance = Bool()
    is_panel_trunks = Bool()
    is_panel_cdrslist = Bool()
    is_panel_sippacket = Bool()
    is_panel_summaryreport = Bool()
    is_panel_onlinepayment = Bool()
    is_panel_cid_blocking = Bool()
    is_panel_mydid = Bool()
    is_panel_didrequest = Bool()  # 'is_panel_mydid','is_panel_didrequest',
    enable_trunk_edit = Bool()
    enable_trunk_view = Bool()
    enable_strip = Bool()
    enable_paypal = Bool()
    carrier_name = Str(attribute='name')
    enable_client_portal = Bool()
    client_type = Str()

    # login = Str(validate=validate.Length(0,))
    # password = Str()
    # email = Emails()
    class Meta:
        model = model.Client
        fields = ('is_panelaccess', 'is_panel_ratetable', 'is_panel_paymenthistory',
                  'is_panel_invoices', 'is_panel_accountsummary', 'is_panel_products', 'is_panel_balance',
                  'is_panel_trunks', 'is_panel_cdrslist', 'is_panel_sippacket', 'is_panel_summaryreport',
                  'is_panel_onlinepayment', 'is_panel_cid_blocking', 'is_panel_mydid', 'is_panel_didrequest',
                  'is_panel_block_list', 'enable_trunk_view', 'enable_trunk_edit', 'enable_strip',
                  'enable_paypal', 'enable_client_portal', 'carrier_name', 'client_id', 'client_type'
                  # 'login','password','email'
                  )


class CarrierContactsScheme(BaseModelScheme):
    # gen   patch /carrier/{client_id}
    # Company Details
    rate_email = Emails()
    billing_email = Emails()
    tax = Decimal(validate=validate.Range(min=0.0, max=100.0))
    noc_email = Emails()
    address = Str(validate=validate.Length(max=500))
    company_name = Str(attribute='company', validate=validate.Length(max=500))
    phone = Number()
    email = Emails()
    account_detail = Str(attribute='details', validate=validate.Length(max=1000))
    rate_delivery_email = Emails()

    class Meta:
        model = model.Client
        fields = ('rate_email', 'billing_email', 'tax', 'noc_email', 'address', 'company_name', 'phone', 'email',
                  'account_detail', 'rate_delivery_email')


class CarrierBalanceGetScheme(BaseModelScheme):
    # gen   patch /carrier/{client_id}/scc
    # Carrier Short call Charges
    actual_balance = Float()
    mutual_balance = Float()

    class Meta:
        model = model.Client
        fields = ('actual_balance', 'mutual_balance')


class CarrierPasswordScheme(BaseModelScheme):
    password = Str()

    class Meta:
        model = model.Client
        fields = ('password',)


class CarrierApplyTemplateScheme(BaseModelScheme):
    carrier_template_id = Int(validate=lambda value: _valid('CarrierTemplate', 'id', value))

    class Meta:
        model = model.Client
        fields = ('carrier_template_id',)


class CarrierFromTemplateScheme(BaseModelScheme):
    name = Str(validate=[validate.Length(max=500), validate.Regexp(NAME_REGEXP)])
    is_active = Bool()
    noc_email = Emails()
    billing_email = Emails()
    rate_email = Emails()
    rate_delivery_email = Emails()
    company = Str()
    address = Str()
    account_detail = Str(attribute='details', missing='', default='', allow_none=True)
    login = Str(allow_none=True, validate=validate.Length(0, 60))
    password = Str(allow_none=True, validate=validate.Length(0, 60))

    class Meta:
        model = model.Client
        fields = ('name', 'is_active', 'noc_email', 'billing_email', 'rate_email', 'rate_delivery_email',
                  'company', 'address', 'account_detail', 'login', 'password')


class CarrierScheme(BaseModelScheme):
    name = Str(validate=[validate.Length(max=500), validate.Regexp(NAME_REGEXP)])
    is_active = Bool()
    is_prepay = Bool()
    currency = Int(attribute='currency_id', load_from='currency', dump_to='currency')
    credit_limit = Decimal(allow_none=True, validate=validate.Range(min=-100000000.0, max=100000000.0), attribute='_credit_limit')
    cps_limit = Int(allow_none=True, validate=validate.Range(min=0, max=1000000))
    call_limit = Int(allow_none=True, validate=validate.Range(min=0, max=100000000))
    company = Str(validate=validate.Length(max=500))
    address = Str(missing='', validate=validate.Length(max=500))
    main_email = Emails()
    billing_email = Emails()
    noc_email = Emails()
    rate_email = Emails()
    finance_email_cc = Emails()
    account_detail = Str(attribute='details', missing='', default='', allow_none=True)
    daily_balance_summary = Bool()
    daily_usage_summary = Bool()
    daily_cdr_delivery = Bool()
    daily_usage_group_by = Choice()
    daily_usage_on_non_zero = Bool()  # Bool( attribute = 'auto_summary_not_zero', load_from='daily_usage_on_non_zero',dump_to='daily_usage_on_non_zero')
    report_frequency = Int(attribute='auto_summary_period', load_from='report_frequency', dump_to='report_frequency')
    one_time_report_time = Int(attribute='auto_summary_hour', load_from='one_time_report_time',
                               dump_to='one_time_report_time')
    is_send_trunk_update = Bool()
    # low_balance_alert =  #Bool(attribute='low_balance_alert')
    low_balance_alert = Bool(attribute='low_balance_notice', load_from='low_balance_alert', dump_to='low_balance_alert')
    short_call_percent = Float(attribute='scc_percent', load_from='short_call_percent', dump_to='short_call_percent')
    short_call_duration = Float(attribute='scc_bellow', load_from='short_call_duration', dump_to='short_call_duration')
    short_call_charge = Float(attribute='scc_charge', load_from='short_call_charge', dump_to='short_call_charge',
                              missing='0')
    short_call_charge_exceed_only = Int(attribute='scc_type', load_from='short_call_charge_exceed_only',
                                        dump_to='short_call_charge_exceed_only')
    test_credit = Decimal()
    currency_name = Str(validate=validate_currency)
    profit_type = Choice()
    auto_invoice_type = Choice()
    phone = Str()
    unlimited_credit = Bool(default=False, allow_none=True)
    # enough_balance = Bool(default=False,allow_none=True)
    payment_term_id = Int(allow_none=True, validate=lambda value: _valid('PaymentTerm', 'payment_term_id', value))
    profit_margin = Decimal(allow_none=True, validate=validate.Range(min=0.0, max=1000000.0))
    login = Str(allow_none=True, validate=validate.Length(0, 60))
    password = Str(allow_none=True, validate=validate.Length(0, 60))
    group_id = Int(allow_none=True, validate=lambda value: _valid('CarrierGroup', 'group_id', value))
    payment_received_notice = Bool()
    company_type = Choice()
    client_invoice_settings_id = Int(allow_none=True, attribute='_client_invoice_settings_id',
                                     validate=lambda value: _valid('InvoiceSetting', 'id', value))
    is_panelaccess = Bool(attribute='is_panelaccess',
                          load_from='is_panelaccess', dump_to='is_panelaccess')
    password = Str(attribute='password',
                   load_from='password', dump_to='password')
    is_panel_ratetable = Bool(attribute='is_panel_ratetable',
                              load_from='is_panel_ratetable', dump_to='is_panel_ratetable')
    is_panel_paymenthistory = Bool(attribute='is_panel_paymenthistory',
                                   load_from='is_panel_paymenthistory', dump_to='is_panel_paymenthistory')
    is_panel_invoices = Bool(attribute='is_panel_invoices',
                             load_from='is_panel_invoices', dump_to='is_panel_invoices')
    is_panel_accountsummary = Bool(attribute='is_panel_accountsummary',
                                   load_from='is_panel_accountsummary', dump_to='is_panel_accountsummary')
    is_panel_block_list = Bool(attribute='is_panel_block_list',
                               load_from='is_panel_block_list', dump_to='is_panel_block_list')
    is_panel_products = Bool(attribute='is_panel_products',
                             load_from='is_panel_products', dump_to='is_panel_products')
    is_panel_balance = Bool(attribute='is_panel_balance',
                            load_from='is_panel_balance', dump_to='is_panel_balance')
    is_panel_trunks = Bool(attribute='is_panel_trunks',
                           load_from='is_panel_trunks', dump_to='is_panel_trunks')
    is_panel_cdrslist = Bool(attribute='is_panel_cdrslist',
                             load_from='is_panel_cdrslist', dump_to='is_panel_cdrslist')
    is_panel_sippacket = Bool(attribute='is_panel_sippacket',
                              load_from='is_panel_sippacket', dump_to='is_panel_sippacket')
    is_panel_summaryreport = Bool(attribute='is_panel_summaryreport',
                                  load_from='is_panel_summaryreport', dump_to='is_panel_summaryreport')
    is_panel_onlinepayment = Bool(attribute='is_panel_onlinepayment',
                                  load_from='is_panel_onlinepayment', dump_to='is_panel_onlinepayment')
    is_panel_cid_blocking = Bool()
    test_credit = Decimal(allow_none=True)
    break_call_on_no_balance = Bool(default=True)

    class Meta:
        model = model.Client
        fields = ('name', 'is_active', 'is_prepay', 'currency_name', 'credit_limit', 'test_credit', 'cps_limit',
                  'call_limit', 'company', 'address', 'main_email', 'billing_email', 'noc_email', 'rate_email',
                  'finance_email_cc', 'company_type',
                  'account_detail', 'daily_balance_summary', 'daily_usage_summary', 'daily_cdr_delivery',
                  'daily_usage_group_by', 'daily_usage_on_non_zero', 'report_frequency', 'one_time_report_time',
                  'is_send_trunk_update', 'low_balance_alert', 'short_call_percent', 'short_call_duration',
                  'short_call_charge',
                  'short_call_charge_exceed_only', 'profit_type', 'auto_invoice_type', 'phone', 'unlimited_credit',
                  # 'enough_balance',
                  'payment_term_id', 'profit_margin', 'login', 'password', 'group_id', 'payment_received_notice',
                  'is_panel_ratetable', 'is_panel_paymenthistory', 'is_panel_invoices', 'is_panel_accountsummary',
                  'is_panel_products', 'is_panel_balance', 'is_panel_trunks', 'is_panel_cdrslist', 'is_panel_sippacket',
                  'is_panel_summaryreport', 'is_panel_onlinepayment', 'is_panel_cid_blocking', 'is_panel_block_list',
                  'break_call_on_no_balance')


class CarrierModifyScheme(CarrierScheme):
    name = Str(validate=[validate.Length(max=500), validate.Regexp(NAME_REGEXP)])

    class Meta:
        model = model.Client
        fields = ('name', 'is_active', 'is_prepay', 'currency_name', 'credit_limit', 'cps_limit', 'email',
                  'call_limit', 'company', 'address', 'main_email', 'billing_email', 'noc_email', 'rate_email',
                  'finance_email_cc', 'company_type',
                  'account_detail', 'daily_balance_summary', 'daily_usage_summary', 'daily_cdr_delivery',
                  'daily_usage_group_by', 'daily_usage_on_non_zero', 'report_frequency', 'one_time_report_time',
                  'is_send_trunk_update', 'low_balance_alert', 'short_call_percent', 'short_call_duration',
                  'short_call_charge',
                  'short_call_charge_exceed_only', 'profit_type', 'auto_invoice_type', 'phone', 'unlimited_credit',
                  'payment_term_id', 'profit_margin', 'login', 'password', 'group_id', 'payment_received_notice',
                  'is_show_daily_usage', 'break_call_on_no_balance')


class CarrierLongScheme(BaseModelScheme):
    carrier_name = Str(attribute='name', load_from='carrier_name', dump_to='carrier_name',
                       validate=[validate.Length(max=500)])
    is_prepay = Bool()
    credit_limit = Decimal(allow_none=True, validate=validate.Range(min=0.0, max=1000000000.0))
    # status = Bool()
    client_type = Choice()
    is_active = Bool()
    currency = Int(attribute='currency_id', load_from='currency', dump_to='currency', default="1")
    # validate=fields.validate.OneOf( model.Currency.query().session.query('currency_id').all()  ) )
    call_limit = Int(allow_none=True, default=None, validate=validate.Range(min=0, max=100000000))
    cps_limit = Int(validate.Range(min=0, max=1000000), allow_none=True)

    unlimited_credit = Bool(default=False)
    profit_margin = Float()  # validate=validate.Range(min=0.0, max=100.0))
    profit_type = Choice()
    # invoices
    is_breakdown_by_rate_table = Bool()
    show_trunk_summary = Bool(attribute='is_show_total_trunk', load_from='show_trunk_summary',
                              dump_to='show_trunk_summary')
    decimal = Int(attribute='decimal_place', load_from='decimal', dump_to='decimal',
                  validate=validate.Range(min=0, max=6))
    show_country_summary = Bool(attribute='is_show_country', load_from='show_country_summary',
                                dump_to='show_country_summary')
    inlcude_cdr_in_email = Bool(attribute='attach_cdrs_list', load_from='inlcude_cdr_in_email',
                                dump_to='inlcude_cdr_in_email')
    show_payment_summary = Bool(attribute='invoice_include_payment', load_from='show_payment_summary',
                                dump_to='show_payment_summary')
    send_invoice_as_link = Bool()
    auto_send_invoice = Bool()
    auto_send_zone = Str(validate=validate.Regexp(TIMEZONE_REGEXP))
    invoice_zone = Str(validate=validate.Regexp(TIMEZONE_REGEXP))
    invoice_hour = Int()
    show_daily_usage = Bool(attribute='is_show_daily_usage', load_from='show_daily_usage', dump_to='show_daily_usage')
    payment_term = Str(attribute='pt_name', validate=[validate_payment_term, validate.Length(max=100)])
    include_tax = Bool(attribute='include_tax', load_from='include_tax', dump_to='include_tax')
    show_account_summary = Bool(attribute='is_invoice_account_summary', load_from='show_account_summary',
                                dump_to='show_account_summary')
    rate_value = Choice()
    show_jurisdiction_detail = Bool(attribute='invoice_jurisdictional_detail', load_from='show_jurisdiction_detail',
                                    dump_to='show_jurisdiction_detail')
    cdr_format = Choice()
    show_code_name_summary = Bool(attribute='is_show_code_name', load_from='show_code_name_summary',
                                  dump_to='show_code_name_summary')
    time_zone = Str(attribute='daily_balance_send_time_zone', load_from='time_zone', dump_to='time_zone')
    format = Choice()
    show_detail_by_trunk = Bool(attribute='is_show_detail_trunk', load_from='show_detail_by_trunk',
                                dump_to='show_detail_by_trunk')
    show_code_summary = Bool(attribute='is_show_code_100', load_from='show_code_summary', dump_to='show_code_summary')
    include_short_call_charge = Bool(attribute='is_short_duration_call_surcharge_detail',
                                     load_from='include_short_call_charge', dump_to='include_short_call_charge')
    non_zero_invoice_only = Bool()
    invoice_show_details = Bool()
    # alerts
    notify_client_balance_type = Int(attribute='notify_client_balance_type',
                                     load_from='notify_client_balance_type', dump_to='notify_client_balance_type')
    attach_cdrs_list = Bool(attribute='attach_cdrs_list', load_from='attach_cdrs_list', dump_to='attach_cdrs_list')
    is_auto_summary = Bool(attribute='is_auto_summary',
                           load_from='is_auto_summary', dump_to='is_auto_summary')
    auto_summary_hour = Int(attribute='auto_summary_hour',
                            load_from='auto_summary_hour', dump_to='auto_summary_hour')
    is_send_trunk_update = Bool(attribute='is_send_trunk_update',
                                load_from='is_send_trunk_update', dump_to='is_send_trunk_update')
    low_balance_notification_time_type = Int(attribute='low_balance_notification_time_type',
                                             load_from='low_balance_notification_time_type',
                                             dump_to='low_balance_notification_time_type')
    auto_summary_period = Int(attribute='auto_summary_period',
                              load_from='auto_summary_period', dump_to='auto_summary_period')
    auto_summary_group_by = Choice()
    auto_summary_not_zero = Bool(attribute='auto_summary_not_zero_bool',
                                 load_from='auto_summary_not_zero', dump_to='auto_summary_not_zero')
    low_balance_notification_time_cycle = Str(attribute='low_balance_notification_time_cycle',
                                              load_from='low_balance_notification_time_cycle',
                                              dump_to='low_balance_notification_time_cycle')
    daily_cdr_generation_zone = Str(attribute='daily_cdr_generation_zone',
                                    load_from='daily_cdr_generation_zone', dump_to='daily_cdr_generation_zone')
    daily_cdr_generation = Bool(attribute='daily_cdr_generation',
                                load_from='daily_cdr_generation', dump_to='daily_cdr_generation')
    low_balance_notice = Bool(attribute='low_balance_notice',
                              load_from='low_balance_notice', dump_to='low_balance_notice')
    zero_balance_notice = Bool(attribute='zero_balance_notice',
                               load_from='zero_balance_notice', dump_to='zero_balance_notice')
    is_daily_balance_notification = Bool(attribute='is_daily_balance_notification',
                                         load_from='is_daily_balance_notification',
                                         dump_to='is_daily_balance_notification')
    is_show_daily_usage = Bool(attribute='is_show_daily_usage',
                               load_from='is_show_daily_usage', dump_to='is_show_daily_usage')
    payment_received_notice = Bool()
    # SCC
    scc_percent = Int(validate=validate.Range(min=0, max=100))
    scc_charge = Decimal(validate=validate.Range(min=0.0, max=1000000.0))
    scc_type = Int(validate=validate.OneOf((0, 1)))
    scc_below = Int(validate=validate.Range(min=0, max=1000000))
    # portal
    is_panelaccess = Bool(attribute='is_panelaccess',
                          load_from='is_panelaccess', dump_to='is_panelaccess')
    password = Str(attribute='password',
                   load_from='password', dump_to='password')
    is_panel_ratetable = Bool(attribute='is_panel_ratetable',
                              load_from='is_panel_ratetable', dump_to='is_panel_ratetable')
    is_panel_paymenthistory = Bool(attribute='is_panel_paymenthistory',
                                   load_from='is_panel_paymenthistory', dump_to='is_panel_paymenthistory')
    is_panel_invoices = Bool(attribute='is_panel_invoices',
                             load_from='is_panel_invoices', dump_to='is_panel_invoices')
    is_panel_accountsummary = Bool(attribute='is_panel_accountsummary',
                                   load_from='is_panel_accountsummary', dump_to='is_panel_accountsummary')
    is_panel_block_list = Bool(attribute='is_panel_block_list',
                               load_from='is_panel_block_list', dump_to='is_panel_block_list')
    is_panel_products = Bool(attribute='is_panel_products',
                             load_from='is_panel_products', dump_to='is_panel_products')
    is_panel_balance = Bool(attribute='is_panel_balance',
                            load_from='is_panel_balance', dump_to='is_panel_balance')
    is_panel_trunks = Bool(attribute='is_panel_trunks',
                           load_from='is_panel_trunks', dump_to='is_panel_trunks')
    is_panel_cdrslist = Bool(attribute='is_panel_cdrslist',
                             load_from='is_panel_cdrslist', dump_to='is_panel_cdrslist')
    is_panel_sippacket = Bool(attribute='is_panel_sippacket',
                              load_from='is_panel_sippacket', dump_to='is_panel_sippacket')
    is_panel_summaryreport = Bool(attribute='is_panel_summaryreport',
                                  load_from='is_panel_summaryreport', dump_to='is_panel_summaryreport')
    is_panel_onlinepayment = Bool(attribute='is_panel_onlinepayment',
                                  load_from='is_panel_onlinepayment', dump_to='is_panel_onlinepayment')
    is_panel_didrequest = Bool(attribute='is_panel_didrequest',
                               load_from='is_panel_didrequest', dump_to='is_panel_didrequest')
    is_panel_cid_blocking = Bool()
    login = Str(validate=validate.Length(max=60))
    # enables
    enable_auto_invoice = Bool(attribute='auto_invoicing', default=False, allow_none=True)
    enable_auto_report = Bool(default=False, allow_none=True, attribute='_enable_auto_report')
    enable_client_portal = Bool(default=False, allow_none=True)
    enable_notification = Bool(default=False, allow_none=True)
    enable_short_dur_penalty = Bool(default=False, allow_none=True)
    enable_paypal = Bool(default=True, allow_none=True)
    enable_strip = Bool(default=True, allow_none=True)
    enable_sign_in_notification = Bool(default=True, allow_none=True)
    enable_trunk_view = Bool(default=False, allow_none=True)
    enable_trunk_edit = Bool(default=False, allow_none=True)
    rate_email = Emails()
    billing_email = Emails()
    tax = Decimal(validate=validate.Range(min=0.0, max=100.0), allow_none=True)
    tax_id = Str(validate=validate.Length(max=100))
    noc_email = Emails()
    address = Str(validate=validate.Length(max=500), allow_none=True)
    company_name = Str(attribute='company', validate=[validate.Length(max=500),
                                                      lambda value: _valid_unique('Client', 'company', value)])
    phone = Number()
    email = Emails()
    account_detail = Str(attribute='details', validate=validate.Length(max=1000))
    rate_delivery_email = Emails()
    low_balance_config = Nested(CarrierLowBalanceConfigScheme, many=False)

    test_credit = Decimal(validate=validate.Range(min=0.0, max=100000000.0))
    enough_balance = Bool(default=False, allow_none=True)
    usage_fields = List(Str(validate=lambda value: _valid('DailyCdrField', 'field', value)))
    enable_payment_alert = Bool()
    auto_daily_balance_recipient = Choice()
    client_type = Choice()
    company_type = Choice()
    break_call_on_no_balance = Bool(default=True)

    ani_cap_limit = Int()
    dnis_cap_limit = Int()

    client_invoice_settings_id = Int(allow_none=True,
                                     validate=lambda value: _valid('InvoiceSetting', 'id', value))

    rate_generation_template_id = Int(validate=lambda value: _valid('RateGenerationTemplate', 'id', value), allow_none=True)

    template_name = Str(validate=lambda value: _valid('CarrierTemplate', 'template_name', value))

    auth_ip = Nested(UserAuthIpScheme, many=True)
    client_taxes = Nested('ClientTaxesScheme', many=True)

    def get_low_balance_alert(self, obj):
        return obj.low_balance_alert

    def set_low_balance_alert(self, value):
        raise Exception(str(self.data) + str(value))

    class Meta:
        model = model.Client
        fields = (  # basic
            'carrier_name', 'is_prepay', 'credit_limit', 'is_active', 'currency', 'call_limit', 'cps_limit',
            'test_credit',
            'unlimited_credit', 'profit_margin', 'profit_type', 'company_type', 'client_type',
            ##invoice
            'client_invoice_settings_id',
            'is_breakdown_by_rate_table', 'show_trunk_summary', 'decimal', 'show_country_summary',
            'inlcude_cdr_in_email',
            'show_payment_summary', 'send_invoice_as_link', 'auto_send_invoice', 'auto_send_zone', 'invoice_zone',
            'invoice_hour',
            'show_daily_usage',
            'payment_term', 'include_tax', 'show_account_summary', 'rate_value', 'show_jurisdiction_detail',
            'cdr_format',
            'show_code_name_summary', 'time_zone', 'format', 'show_detail_by_trunk', 'show_code_summary',
            'include_short_call_charge', 'non_zero_invoice_only', 'invoice_show_details',
            ###'alerts'
            'notify_client_balance_type', 'attach_cdrs_list', 'is_auto_summary', 'auto_summary_hour',
            'is_send_trunk_update', 'low_balance_notification_time_type', 'auto_summary_period',
            'auto_summary_group_by', 'auto_summary_not_zero',
            ##'low_balance_notification_time_cycle',
            'daily_cdr_generation_zone', 'daily_cdr_generation', 'low_balance_notice',
            'zero_balance_notice', 'is_daily_balance_notification', 'is_show_daily_usage',
            'payment_received_notice',
            ##'scc'
            'scc_percent', 'scc_charge', 'scc_type', 'scc_below',
            ##'portal'
            'is_panelaccess', 'password', 'is_panel_ratetable', 'is_panel_paymenthistory', 'is_panel_invoices',
            'is_panel_accountsummary', 'is_panel_products', 'is_panel_balance', 'is_panel_trunks', 'is_panel_cdrslist',
            'is_panel_sippacket', 'is_panel_summaryreport', 'is_panel_onlinepayment', 'is_panel_cid_blocking',
            'is_panel_block_list', 'is_panel_didrequest', 'login',
            ##'enables'
            'enable_auto_invoice', 'enable_auto_report', 'enable_client_portal', 'enable_notification',
            'enable_short_dur_penalty', 'enable_paypal', 'enable_strip', 'enable_sign_in_notification',
            'enable_trunk_view',
            'enable_trunk_edit',
            ##'contacts'
            'rate_email', 'billing_email', 'tax', 'tax_id', 'noc_email', 'address', 'company_name', 'phone', 'email',
            'account_detail', 'rate_delivery_email',
            # low_balance_config
            'low_balance_config',
            # 'enough_balance',
            'usage_fields',
            'enable_payment_alert',
            'auto_daily_balance_recipient',
            # addon
            'rate_generation_template_id',
            'template_name', 'auth_ip', 'client_taxes',
            'ani_cap_limit', 'dnis_cap_limit', 'break_call_on_no_balance'
        )


class CarrierLongGetScheme(CarrierLongScheme):
    client_invoice_settings_id = Int()
    is_vendor = Bool()
    is_client = Bool()
    is_orig = Bool()
    is_term = Bool()

    class Meta:
        model = model.Client
        fields = ('client_id', 'enough_balance', 'client_type', 'is_vendor', 'is_client', 'is_orig',
                  'is_term') + CarrierLongScheme.Meta.fields


class CarrierLongModifyScheme(CarrierLongScheme):
    credit_limit = Decimal(allow_none=True, validate=validate.Range(min=0.0, max=1000000000.0), attribute='_credit_limit')
    company_name = Str(attribute='company', validate=[validate.Length(max=500)])
    class Meta:
        model = model.Client
        fields = (  # basic
            'carrier_name', 'is_prepay', 'is_active', 'currency', 'call_limit', 'cps_limit', 'credit_limit',
            'unlimited_credit', 'profit_margin', 'profit_type', 'company_type', 'ani_cap_limit', 'dnis_cap_limit',
            ###'invoice'
            'client_invoice_settings_id',
            'is_breakdown_by_rate_table', 'show_trunk_summary', 'decimal', 'show_country_summary',
            'inlcude_cdr_in_email',
            'show_payment_summary', 'send_invoice_as_link', 'auto_send_invoice', 'auto_send_zone', 'invoice_zone',
            'invoice_hour',
            'show_daily_usage',
            'payment_term', 'include_tax', 'show_account_summary', 'rate_value', 'show_jurisdiction_detail',
            'cdr_format',
            'show_code_name_summary', 'time_zone', 'format', 'show_detail_by_trunk', 'show_code_summary',
            'include_short_call_charge', 'non_zero_invoice_only', 'invoice_show_details',
            ###'alerts'
            'notify_client_balance_type', 'attach_cdrs_list', 'is_auto_summary', 'auto_summary_hour',
            'is_send_trunk_update', 'low_balance_notification_time_type', 'auto_summary_period',
            'auto_summary_group_by', 'auto_summary_not_zero',
            ##'low_balance_notification_time_cycle',
            'daily_cdr_generation_zone', 'daily_cdr_generation', 'low_balance_notice',
            'zero_balance_notice', 'is_daily_balance_notification', 'is_show_daily_usage',
            'payment_received_notice',
            ##'scc'
            'scc_percent', 'scc_charge', 'scc_type', 'scc_below',
            ##'portal'
            'is_panelaccess', 'password', 'is_panel_ratetable', 'is_panel_paymenthistory', 'is_panel_invoices', 'login',
            'is_panel_accountsummary', 'is_panel_products', 'is_panel_balance', 'is_panel_trunks', 'is_panel_cdrslist',
            'is_panel_sippacket', 'is_panel_summaryreport', 'is_panel_onlinepayment', 'is_panel_cid_blocking',
            'is_panel_block_list',
            ##'enables'
            'enable_auto_invoice', 'enable_auto_report', 'enable_client_portal', 'enable_notification',
            'enable_short_dur_penalty', 'enable_paypal', 'enable_strip', 'enable_sign_in_notification',
            'enable_trunk_view',
            'enable_trunk_edit',
            ##'contacts'
            'rate_email', 'billing_email', 'tax', 'tax_id', 'noc_email', 'address', 'company_name', 'phone', 'email',
            'account_detail', 'rate_delivery_email',
            # low_balance_config
            'low_balance_config',
            # 'enough_balance',
            'usage_fields',
            'enable_payment_alert',
            'auto_daily_balance_recipient'
            ,
            # addon
            'rate_generation_template_id', 'auth_ip', 'break_call_on_no_balance'
        )


class CarrierGetScheme(CarrierLongScheme):
    client_id = Int()
    name = Str()
    company = Str()
    company_name = Str()
    actual_balance = Decimal()
    mutual_balance = Decimal()
    update_at = DateTime()
    update_by = Str()
    unlimited_credit = Bool()
    ingress_count = Int()
    egress_count = Int()
    carrier_template_id = Int()
    mode = Choice()
    group_id = Int()
    payment_term_id = Int(validate=lambda value: _valid('PaymentTerm', 'payment_term_id', value))
    usage_fields = List(Str(validate=lambda value: _valid('DailyCdrField', 'field', value)))
    is_egress_only = Boolean()
    is_ingress_only = Boolean()

    is_vendor = Bool()
    is_client = Bool()
    is_orig = Bool()
    is_term = Bool()
    client_type = Choice()
    company_type = Choice()
    enable_client_portal = Bool()
    password = Str()

    class Meta:
        model = model.Client
        fields = ('name', 'company', 'client_id', 'mode', 'ingress_count', 'egress_count', 'unlimited_credit',
                  'actual_balance',
                  'mutual_balance', 'update_at', 'update_by', 'carrier_template_id', 'payment_term_id',
                  'enough_balance', 'usage_fields', 'client_type') + CarrierLongScheme.Meta.fields
        search_fields = ('client_id', 'name', 'is_active', 'is_prepay', 'company', 'update_by', 'unlimited_credit',
                         'ingress_count', 'egress_count', 'carrier_template_id', 'mode', 'group_id',
                         'payment_received_notice', 'actual_balance',
                         'cps_limit', 'call_limit', 'payment_term_id', 'login', 'is_ingress_only', 'is_egress_only',
                         'company_type', 'client_type', 'client_invoice_settings_id',
                         'is_vendor', 'is_client', 'is_orig', 'is_term', 'company_name', 'enable_client_portal',
                         'password'
                         )
        query_fields = ('credit_limit_gt', 'credit_limit_lt', 'update_at_gt', 'update_at_lt',
                        'cps_limit_gt', 'cps_limit_lt', 'call_limit_gt', 'call_limit_lt', 'profit_margin_gt',
                        'profit_margin_lt', 'ingress_count_gt', 'egress_count_gt')


class CarrierSmallGetScheme(CarrierGetScheme):
    class Meta:
        model = model.Client
        fields = ('client_id', 'name', 'client_type', 'company_type')
        search_fields = ('client_id', 'name', 'client_type', 'company_type')


class CarrierActivateScheme(CarrierGetScheme):
    class Meta:
        model = model.Client
        fields = ('is_active',)
        search_fields = ('name', 'is_active', 'is_prepay', 'company', 'update_by', 'unlimited_credit',
                         'ingress_count', 'egress_count', 'carrier_template_id', 'mode')
        query_fields = ('credit_limit_gt', 'credit_limit_lt', 'update_at_gt', 'update_at_lt', 'client_id_in')


class CarrierGetMinScheme(CarrierScheme):
    class Meta:
        model = model.Client
        fields = ('client_id', 'name', 'is_active', 'company')
        search_fields = ('name', 'is_active', 'ingress_count', 'is_vendor', 'is_client', 'is_orig', 'is_term')


class CarrierGetMinWithIngressScheme(CarrierScheme):
    mode = Choice()
    ingress_trunks = Nested('ResourceMinScheme', many=True)

    class Meta:
        model = model.Client
        fields = ('client_id', 'name', 'is_active', 'company', 'ingress_trunks', 'mode', 'credit_limit')
        search_fields = ('name', 'is_active', 'mode')


class CarrierGetMinWithEgressScheme(CarrierScheme):
    mode = Choice()
    egress_trunks = Nested('ResourceMinScheme', many=True)

    class Meta:
        model = model.Client
        fields = ('client_id', 'name', 'is_active', 'company', 'egress_trunks', 'mode', 'credit_limit')
        search_fields = ('name', 'is_active', 'mode')


class CarrierGetUserScheme(CarrierGetMinScheme):
    user = Nested(UserMinScheme)

    class Meta:
        model = model.Client
        fields = ('user',) + CarrierGetMinScheme.Meta.fields


class CarrierGetIdsScheme(CarrierScheme):
    class Meta:
        model = model.Client
        fields = ('client_id',)


# search_fields = ('name', 'is_active')


class CarrierAutoInvoicingGetScheme(BaseModelScheme):
    name = Str()
    payment_term_id = Int(validate=lambda value: _valid('PaymentTerm', 'payment_term_id', value))
    pt_name = Str()
    auto_invoicing = Bool()
    invoice_cycle = Str()
    last_invoice_date = Str()
    next_invoice_date = Str()
    last_invoice_amount = Float()
    last_invoice_period = Str()
    ingress_count = Int()

    class Meta:
        model = model.Client
        fields = ('name', 'client_id', 'pt_name', 'auto_invoicing', 'last_invoice_date', 'last_invoice_amount',
                  'next_invoice_date',
                  'last_invoice_period', 'payment_term_id', 'invoice_cycle', 'ingress_count')
        search_fields = ('client_id', 'name', 'payment_term_id', 'pt_name')
        query_fields = ('last_invoice_date_gt', 'last_invoice_date_lt', 'next_invoice_date_gt', 'next_invoice_date_lt',
                        'ingress_count_gt')

class ShakenStiSpConfScheme(BaseModelScheme):
    priv_key = String()
    passphrase = String()
    x5u = String()
    id = Bool()
    enabled = Bool()

    class Meta:
        model = model.ShakenStiSpConf
        fields = ('priv_key', 'passphrase', 'x5u', 'id', 'enabled')
# --- DynamicRoute
class DynamicRouteItemScheme(BaseModelScheme):
    dynamic_route_id = Int(required=True)
    resource_id = Int(required=True, validate=lambda value: _valid('Resource', 'resource_id', value))

    class Meta:
        model = model.DynamicRouteItem
        fields = ('resource_id',)


class DynamicRouteItemInnerScheme(BaseModelScheme):
    dynamic_route_id = Int(required=True, validate=lambda value: _valid('DynamicRoute', 'dynamic_route_id', value))

    class Meta:
        model = model.DynamicRouteItem
        fields = ('dynamic_route_id',)


class DynamicRouteItemInnerGetScheme(BaseModelScheme):
    id = Int()
    dynamic_route_id = Int(required=True)
    route = Nested('DynamicRouteSchemeInnerGet', many=False)

    class Meta:
        model = model.DynamicRouteItem
        fields = ('id', 'dynamic_route_id', 'route')


class DynamicRouteItemSchemeGetTrunk(BaseModelScheme):
    # route = Nested(DynamicRouteSchemeGet,many=False)
    resource_id = Int(required=True)
    trunk_name = Str()
    client_id = Int()
    client_name = Str()
    is_active = Bool()

    # trunk = Nested(EgressTrunkGetScheme)

    class Meta:
        model = model.DynamicRouteItem
        # exclude = ('dynamic_route_id', 'resource_id','route')
        fields = ('resource_id', 'trunk_name', 'client_id', 'client_name', 'is_active')


class DynamicRouteItemSchemeMultiple(BaseModelScheme):
    dynamic_route_id = Int(required=True, validate=lambda value: _valid('DynamicRoute', 'dynamic_route_id', value))
    trunks = List(Int(), attribute='resource_id', load_from='trunks')

    class Meta:
        model = model.DynamicRouteItem
        fields = ('trunks',)


class DynamicRouteQosScheme(BaseModelScheme):
    dynamic_route_id = Int(required=True)
    digits = Str(validate=validate.Regexp(PREFIX_REGEXP))
    min_asr = Float()
    max_asr = Float()
    min_abr = Float()
    max_abr = Float()
    min_acd = Float()
    max_acd = Float()
    min_pdd = Int()
    max_pdd = Int()
    min_aloc = Float()
    max_aloc = Float()
    limit_price = Float()

    @pre_load
    def check_min_max(self, data):
        if 'min_asr' in data and 'max_asr' in data:
            if data['min_asr'] > data['max_asr']:
                raise ValidationError({'min_asr': ['min asr must be less max asr!']})
        if 'min_abr' in data and 'max_abr' in data:
            if data['min_abr'] > data['max_abr']:
                raise ValidationError({'min_abr': ['min abr must be less max abr!']})
        if 'min_acd' in data and 'max_acd' in data:
            if data['min_acd'] > data['max_acd']:
                raise ValidationError({'min_acd': ['min acd must be less max acd!']})
        if 'min_pdd' in data and 'max_pdd' in data:
            if data['min_pdd'] > data['max_pdd']:
                raise ValidationError({'min_pdd': ['min pdd must be less max pdd!']})
        if 'min_aloc' in data and 'max_aloc' in data:
            if data['min_aloc'] > data['max_aloc']:
                raise ValidationError({'min_aloc': ['min aloc must be less max aloc!']})
        return data

    class Meta:
        model = model.DynamicRouteQos
        fields = ('digits', 'min_asr', 'max_asr', 'min_abr', 'max_abr', 'min_acd', 'max_acd',
                  'min_pdd', 'max_pdd', 'min_aloc', 'max_aloc', 'limit_price')


class DynamicRouteQosSchemeGet(DynamicRouteQosScheme):
    id = Int()
    dynamic_route_id = Int()

    class Meta:
        model = model.DynamicRouteQos
        fields = ('id', 'dynamic_route_id') + DynamicRouteQosScheme.Meta.fields
        search_fields = ('id', 'digits', 'dynamic_route_id')
        query_fields = ('min_asr_gt', 'min_asr_lt', 'max_asr_gt', 'max_asr_lt', 'min_acd_gt', 'min_acd_lt',
                        'max_acd_gt', 'max_acd_lt', 'min_pdd_gt', 'min_pdd_lt', 'min_aloc_gt', 'min_aloc_lt',
                        'limit_price_gt', 'limit_price_lt')


class DynamicRoutePriScheme(BaseModelScheme):
    dynamic_route_id = Int(required=True)
    digits = Str(validate=validate.Regexp(DIGITS_REGEXP), allow_none=True)
    resource_id = Int()
    resource_pri = Int()

    class Meta:
        model = model.DynamicRoutePri
        fields = ('digits', 'resource_id', 'resource_pri')


class DynamicRoutePriSchemeGet(DynamicRoutePriScheme):
    id = Int()
    dynamic_route_id = Int()
    trunk_name = Str(dump_only=True)

    class Meta:
        model = model.DynamicRoutePri
        fields = ('id', 'dynamic_route_id', 'trunk_name') + DynamicRoutePriScheme.Meta.fields
        search_fields = ('id', 'digits', 'resource_id', 'trunk_name')
        query_fields = ('resource_pri_gt', 'resource_pri_lt')


class DynamicRouteOverScheme(BaseModelScheme):
    dynamic_route_id = Int(required=True)
    digits = Str(validate=validate.Regexp(PREFIX_REGEXP))
    resource_id = Int()
    percentage = Int(validate=validate.Range(0, 100))

    class Meta:
        model = model.DynamicRouteOverride
        fields = ('digits', 'resource_id', 'percentage')


class DynamicRouteOverSchemeGet(DynamicRouteOverScheme):
    id = Int()
    dynamic_route_id = Int()
    resource_name = Str()

    class Meta:
        model = model.DynamicRouteOverride
        fields = ('id', 'dynamic_route_id', 'resource_name') + DynamicRouteOverScheme.Meta.fields
        search_fields = ('id', 'digits', 'resource_id', 'percentage', 'resource_name')
        query_fields = ('percentage_gt', 'percentage_lt')


class DynamicRouteCopyScheme(BaseModelScheme):
    name = Str(validate=validate.Regexp(NAME_REGEXP))

    class Meta:
        model = model.DynamicRoute
        fields = ('name',)


class DynamicRouteScheme(BaseModelScheme):
    # gen module  dynamic-routing
    # --- Dynamic Routing ---
    # last_modified = Str(attribute='update_at',
    #    load_from='last_modified',dump_to='last_modified' )
    # modified_by = Str(attribute='update_by',
    #    load_from='modified_by',dump_to='modified_by' )
    time_profile_id = Int(validate=lambda value: _valid('TimeProfile', 'time_profile_id', value), allow_none=True)
    egress_trunks = Nested(DynamicRouteItemScheme, many=True)
    qos_cycle = Choice()
    name = Str(validate=validate.Regexp(NAME_REGEXP))
    route_rule_name = Choice(default='LCR')

    # route_rule_name = Choice()
    @pre_load
    def check_qos_cycle(self, data):
        if 'route_rule_name' in data and data['route_rule_name'] == 'LCR':
            if 'qos_cycle' in data and data['qos_cycle'] not in [None, 'null', 'not set']:
                raise ValidationError('Do not set qos_cycle when routing_rule=LCR !')

    @pre_load
    def remove_duplicates_egress_trunks(self, data):
        if 'egress_trunks' in data:
            lst = set([t['resource_id'] for t in data['egress_trunks']])
            data['egress_trunks'] = [{'resource_id': id} for id in lst]
        return data

    class Meta:
        model = model.DynamicRoute
        fields = ('time_profile_id', 'egress_trunks', 'qos_cycle', 'name', 'route_rule_name')
        search_fields = ('name', 'route_rule_name', 'qos_cycle', 'time_profile_name', 'time_profile_id',
                         'modified_by', 'usage_count', 'egress_trunk_id', 'route_plan_id')
        query_fields = ('dynamic_route_id_in',)


class DynamicRouteSchemeGet(BaseModelScheme):
    # gen module  dynamic-routing
    # --- Dynamic Routing ---
    last_modified = Str()
    modified_by = Str()
    time_profile_id = Int()
    time_profile_name = Str()
    egress_trunks = Nested(DynamicRouteItemSchemeGetTrunk, many=True)
    qos_cycle = Choice()
    name = Str()
    route_rule_name = Choice(default='LCR')
    usage_count = Int()
    # for search only
    egress_trunk_id = Int()
    route_plan_id = Int()

    class Meta:
        model = model.DynamicRoute
        fields = ('dynamic_route_id', 'last_modified', 'modified_by', 'time_profile_name', 'time_profile_id',
                  'egress_trunks', 'qos_cycle', 'name', 'route_rule_name', 'usage_count')
        search_fields = ('name', 'route_rule_name', 'qos_cycle', 'time_profile_name', 'time_profile_id',
                         'modified_by', 'usage_count', 'egress_trunk_id', 'route_plan_id', 'last_modified')
        query_fields = ('dynamic_route_id_in',)


class DynamicRouteSchemeAllModify(DynamicRouteSchemeGet):
    egress_trunks = Nested(DynamicRouteItemScheme, many=True)

    class Meta:
        model = model.DynamicRoute
        fields = ('time_profile_id', 'egress_trunks', 'qos_cycle', 'route_rule_name')
        search_fields = ('name', 'route_rule_name', 'qos_cycle', 'time_profile_name', 'time_profile_id',
                         'modified_by', 'usage_count', 'egress_trunk_id', 'route_plan_id')
        query_fields = ('dynamic_route_id_in',)


class DynamicRouteSchemeInnerGet(DynamicRouteSchemeGet):
    class Meta:
        model = model.DynamicRoute
        fields = ('dynamic_route_id', 'last_modified', 'modified_by', 'time_profile_name', 'time_profile_id',
                  'qos_cycle', 'name', 'route_rule_name', 'usage_count')


##--- QOS_TOTAL

class QosTotalReportScheme(Schema):
    interval = Str(validate=validate.OneOf(
        ('15 minute', '30 minute', '1 hour', '24 hour', '7 days', '15 days', '30 days', '60 days')))
    server_ip = Str()

    class Meta:
        model = model.QosTotal
        fields = ('interval', 'server_ip')


class QosTotalSchemeGet(BaseModelScheme):
    report_time = Str()
    call = Int()
    cps = Float()
    channels = Float()
    egress_cps = Float()
    egress_channels = Float()
    ingress_cps = Float()
    ingress_channels = Float()
    server_ip = Str()
    ingress_id = Int()
    egress_id = Int()
    ingress_client_id = Int()
    egress_client_id = Int()
    server_name = Str()

    class Meta:
        model = model.QosTotal
        fields = (
            'report_time', 'call', 'cps', 'channels', 'egress_cps', 'egress_channels', 'ingress_cps',
            'ingress_channels', 'server_ip', 'ingress_id', 'egress_id', 'ingress_client_id',
            'egress_client_id', 'server_name')


class QosClientSchemeGet(BaseModelScheme):
    report_time = Str()
    call = Int()
    inbound_cps = Int()
    inbound_chan = Int()
    outbound_cps = Int()
    outbound_chan = Int()
    server_ip = Str()
    server_port = Int()
    client_id = Int()
    client_name = Str()

    class Meta:
        model = model.QosClient
        fields = ('report_time', 'call', 'inbound_cps', 'inbound_chan', 'outbound_cps', 'outbound_chan', 'server_ip',
                  'server_port', 'client_id', 'client_name')


class QosTotalIngressSchemeGet(BaseModelScheme):
    start_date = Str()
    end_date = Str()

    class Meta:
        model = model.QosTotal
        fields = ('report_date', 'max_ingress')
        # fields = ('start_date','end_date','server_ip','ingress_channels')
        search_fields = ('start_date', 'end_date', 'server_ip')


class QosResourceSchemeGet(BaseModelScheme):
    report_time = Str()
    res_id = Int()
    res_name = Str()
    call = Int()
    cps = Float()
    channels = Float()
    direction = Choice()
    server_ip = Str()
    trunk_type2 = Str()
    res_ip = Str()
    ip_id = Str()

    class Meta:
        model = model.QosResource
        fields = (
            'report_time', 'res_id', 'res_name', 'call', 'cps', 'channels', 'direction', 'server_ip', 'trunk_type2',
            'res_ip', 'ip_id')


class QosIpSchemeGet(BaseModelScheme):
    report_time = Str()
    res_id = Int()
    res_name = Str()
    call = Int()
    cps = Float()
    channels = Float()
    direction = Choice()
    server_ip = Str()
    trunk_type2 = Str()
    res_ip = Str()
    ip_id = Int()

    class Meta:
        model = model.QosIp
        fields = (
            'report_time', 'res_id', 'res_name', 'call', 'cps', 'channels', 'direction', 'server_ip', 'trunk_type2',
            'res_ip', 'ip_id')


class QosRouteReportTop10SchemeGet(BaseModelScheme):
    ingress_name = Str()
    egress_name = Str()
    trunk_type2 = Str()
    ingress_total_calls = Int()
    ingress_call_cost = Float()
    egress_total_calls = Int()
    egress_call_cost = Float()
    minutes = Int()
    not_zero_calls = Int()
    ingress_client_name = Str()
    egress_client_name = Str()

    class Meta:
        model = model.QosTotal
        fields = (
            'ingress_name', 'ingress_client_name', 'trunk_type2', 'ingress_total_calls', 'ingress_call_cost',
            'minutes', 'not_zero_calls', 'egress_name', 'egress_client_name', 'egress_total_calls', 'egress_call_cost')


class QosRouteReportSchemeGet(BaseModelScheme):
    report_time = Str()
    alias = Str()
    call = Int()
    cost = Float()
    minutes = Int()
    total_calls = Int()
    no_zero = Int()
    asr = Float()
    qsr = Float()
    acd = Float()
    pdd = Float()
    revenue = Float()
    egress_cost = Float()
    ingress_cost = Float()
    term_profitability = Float()
    orig_profitability = Float()
    profitability = Float()

    class Meta:
        model = model.QosTotal
        fields = ()


# -- Resource Block

class ResourceBlockScheme(BaseModelScheme):
    ingress_trunk_id = Int(allow_none=True, validate=lambda value: _valid('Resource', 'resource_id', value))
    # ingress_trunk_name = Str(allow_none=True, validate=lambda value: _valid('Resource', 'alias', value))
    egress_trunk_id = Int(allow_none=True, validate=lambda value: _valid('Resource', 'resource_id', value))
    # egress_trunk_name = Str(allow_none=True, validate=lambda value: _valid('Resource', 'alias', value))

    ingress_res_id = Int(allow_none=True, validate=lambda value: _valid('Resource', 'resource_id', value))
    ingress_res_id = Int(allow_none=True, validate=lambda value: _valid('Resource', 'resource_id', value))
    engress_res_id = Int(allow_none=True, validate=lambda value: _valid('Resource', 'resource_id', value))

    ingress_group_id = Int(allow_none=True, validate=lambda value: _valid('TrunkGroup', 'group_id', value))
    # ingress_group_name = Int(allow_none=True, validate=lambda value: _valid('TrunkGroup', 'group_name', value))
    egress_group_id = Int(allow_none=True, validate=lambda value: _valid('TrunkGroup', 'group_id', value))
    # egress_group_name = Str(allow_none=True, validate=lambda value: _valid('TrunkGroup', 'group_name', value))

    ingress_client_id = Int(allow_none=True, validate=lambda value: _valid('Client', 'client_id', value))
    ingress_client_name = Str(allow_none=True, validate=lambda value: _valid('Client', 'name', value))

    egress_client_id = Int(allow_none=True, validate=lambda value: _valid('Client', 'client_id', value))
    egress_client_name = Str(allow_none=True, validate=lambda value: _valid('Client', 'name', value))

    time_profile_id = Int(allow_none=True, validate=lambda value: _valid('TimeProfile', 'time_profile_id', value))
    time_profile_name = Int(allow_none=True, validate=lambda value: _valid('TimeProfile', 'name', value))
    ANI_empty = Bool()

    ANI_prefix = Str(validate=validate.Regexp(USERNAME_REGEXP))
    ANI_min = Int(validate=validate.Range(0, 64))
    ANI_max = Int(validate=validate.Range(0, 64))
    DNIS_prefix = Str(validate=validate.Regexp(USERNAME_REGEXP))
    DNIS_min = Int(validate=validate.Range(0, 64))
    DNIS_max = Int(validate=validate.Range(0, 64))
    block_type = Str(attribute='block_type_',
                     validate=validate.OneOf(('specific group', 'specific trunk', 'specific carrier', 'all')))
    block_by = Str()
    block_at = Float(allow_none=True)
    unblock_at = Float(allow_none=True)
    unblock_after = Float(allow_none=True)

    @validates_schema
    def validate_ani_dnis(self, data):
        if 'ANI_min' in data and 'ANI_max' in data and data['ANI_min'] > data['ANI_max']:
            raise ValidationError('invalid input! ANI_min must be less than ANI_max', ['ANI_min'])
        if 'DNIS_min' in data and 'DNIS_max' in data and data['DNIS_min'] > data['DNIS_max']:
            raise ValidationError('invalid input! DNIS_min must be less than DNIS_max', ['DNIS_min'])
        for field in ['ingress_trunk', 'egress_trunk', 'ingress_client', 'egress_client', 'time_profile']:
            if field + '_name' in data:
                if field + '_id' in data:
                    raise ValidationError('invalid input! Both {}_name and {}_id!'.format(field, field))

        return data

    @pre_load
    def load_ids(self, obj):
        for field in ['ingress_trunk', 'egress_trunk', 'ingress_client', 'egress_client', 'ingress_group',
                      'egress_group', 'time_profile']:
            name = field + '_name'
            id = field + '_id'
            if name in obj:
                if id in obj:
                    raise ValidationError({name: ['cannot use both {} and {}'.format(name, id)]})
                if 'trunk' in field:
                    obj[field + '_id'] = model._ref_id('Resource', 'alias', obj[name])
                    del obj[name]
                    continue
                if 'client' in field:
                    obj[field + '_id'] = model._ref_id('Client', 'name', obj[name])
                    del obj[name]
                    continue
                if 'group' in field:
                    obj[field + '_id'] = model._ref_id('TrunkGroup', 'group_name', obj[name])
                    del obj[name]
                    continue
                if 'time_profile' in field:
                    obj[field + '_id'] = model._ref_id('TimeProfile', 'name', obj[name])
                    del obj[name]
                    continue
        
        for date_field in ['block_at', 'unblock_at', 'unblock_after']:
            if date_field in obj and obj[date_field]:
                val = obj[date_field]
                try:
                    # Convert '+00' to '+0000' and parse
                    if val.endswith('+00'):
                        val = val[:-3] + '+0000'
                    dt = datetime.strptime(val, '%Y-%m-%d %H:%M:%S%z')
                    obj[date_field] = dt.timestamp()
                except Exception:
                    raise ValidationError({date_field: ['Invalid datetime format. Expected format: YYYY-MM-DD HH:MM:SS+00']})

        return obj

    class Meta:
        model = model.ResourceBlock
        fields = ('ingress_trunk_id',
                  'egress_trunk_id',
                  'time_profile_id',
                  'ANI_empty', 'ANI_prefix',
                  'ANI_min', 'ANI_max', 'DNIS_prefix', 'DNIS_min', 'DNIS_max',
                  'ingress_group_id',
                  'egress_group_id',
                  'ingress_client_id',
                  'egress_client_id',
                  'block_type',
                  'time_profile_name',
                  'block_at',
                  'unblock_at',
                  'unblock_after',
                  )


class ResourceBlockSchemeGet(ResourceBlockScheme):
    egress_client_id = Int(attribute='fake_egress_client_id')
    ingress_client_id = Int(attribute='fake_ingress_client_id')
    time_profile_name = Str()
    ingress_trunk_name = Str()
    egress_trunk_name = Str()
    ingress_group_name = Str()
    egress_group_name = Str()
    ingress_client_name = Str()
    egress_client_name = Str()
    block_by = Str()
    update_by = Str()
    create_time = DateTime()
    block_type = Str(validate=validate.OneOf(('specific group', 'specific trunk', 'specific carrier')))
    action_type = Choice()

    class Meta:
        model = model.ResourceBlock
        fields = ('res_block_id', 'block_type', 'ingress_trunk_id', 'egress_trunk_id',
                  'ingress_trunk_name', 'egress_trunk_name', 'ingress_group_id', 'egress_group_id',
                  'ingress_group_name', 'egress_group_name', 'ingress_client_id', 'egress_client_id',
                  'ingress_client_name', 'egress_client_name', 'time_profile_id', 'time_profile_name',
                  'ANI_empty', 'ANI_prefix', 'ANI_min', 'ANI_max', 'DNIS_prefix', 'DNIS_min', 'DNIS_max',
                  'block_by', 'update_by', 'create_time', 'action_type', 'block_at', 'unblock_at',
                  'unblock_after')
        search_fields = ('ingress_trunk_id', 'egress_trunk_id', 'ingress_group_id', 'egress_group_id',
                         'ingress_client_id', 'egress_client_id', 'time_profile_name',
                         'ANI_prefix', 'DNIS_prefix', 'block_by', 'update_by',
                         'ingress_trunk_name', 'egress_trunk_name', 'ingress_group_name', 'egress_group_name',
                         'ingress_client_name', 'egress_client_name',
                         'ANI_min', 'ANI_max', 'DNIS_min', 'DNIS_max', 'block_type', 'action_type')
        query_fields = ('create_time_gt', 'create_time_lt', 'res_block_id_in')


# region +++ResourceCidBlockConfig+++
class ResourceCidBlockConfigScheme(BaseModelScheme):
    resource_id = Int()
    min_asr = Int(allow_none=True)
    min_acd = Int(allow_none=True)
    max_sdp = Int(allow_none=True)
    max_cpm = Int(allow_none=True)

    class Meta:
        model = model.ResourceCidBlockConfig
        fields = ('min_asr', 'min_acd', 'max_sdp', 'max_cpm',)

    @pre_load
    def fix_data(self, data):
        for field in ('min_asr', 'min_acd', 'max_sdp', 'max_cpm'):
            if field in data and data[field] in ('null', ''):
                data[field] = None
        return data


class ResourceCidBlockConfigSchemeGet(ResourceCidBlockConfigScheme):
    class Meta:
        model = model.ResourceCidBlockConfig
        fields = ('resource_id', 'min_asr', 'min_acd', 'max_sdp', 'max_cpm',)
        search_fields = ('resource_id', 'min_asr', 'min_acd', 'max_sdp', 'max_cpm',)


class ResourceCidBlockConfigSchemeModify(ResourceCidBlockConfigScheme):
    pass


# endregion ---ResourceCidBlockConfig---
# region +++ResourceCidBlockLog+++
class ResourceCidBlockLogScheme(BaseModelScheme):
    id = Int()
    time = DateTime()
    result = Str()
    data = Dict()
    trunk_name = Str()

    class Meta:
        model = model.ResourceCidBlockLog
        fields = ('time', 'result', 'data',)


class ResourceCidBlockLogSchemeGet(ResourceCidBlockLogScheme):
    class Meta:
        model = model.ResourceCidBlockLog
        fields = ('id', 'time', 'result', 'data', 'trunk_name')
        search_fields = ('id', 'result', 'data', 'trunk_name')
        query_fields = ('time_gt', 'time_lt',)


class ResourceCidBlockLogSchemeModify(ResourceCidBlockLogScheme):
    pass


# endregion ---ResourceCidBlockLog---
class RouteScheme(BaseModelScheme):
    # gen   patch /route/{route_id}
    # Modify Route
    ANI_prefix = Str(allow_none=True, validate=validate.Regexp(PREFIXES_REGEXP))
    DNIS_prefix = Str(allow_none=True, validate=validate.Regexp(PREFIXES_REGEXP))
    route_type_flg = Choice()
    dynamic_route_id = Int(validate=lambda value: _valid('DynamicRoute', 'dynamic_route_id', value))
    static_route_id = Int(validate=lambda value: _valid('Product', 'product_id', value))
    ANI_min = Int(default=0, validate=validate.Range(0, 64))
    ANI_max = Int(default=32, validate=validate.Range(0, 100))
    DNIS_min = Int(default=0, validate=validate.Range(0, 64))
    DNIS_max = Int(default=32, validate=validate.Range(0, 100))
    intra_static_route_id = Int(validate=lambda value: _valid('Product', 'product_id', value))
    inter_static_route_id = Int(validate=lambda value: _valid('Product', 'product_id', value))

    @validates_schema
    def validate_ani_dnis(self, data):
        if 'route_type_flg' in data and 'JD' in data['route_type_flg']:
            if not 'intra_static_route_id' in data or not data['intra_static_route_id']:
                raise ValidationError('invalid input! With JD intra_static_route_id must exists!',
                                      'intra_static_route_id')
            if not 'inter_static_route_id' in data or not data['inter_static_route_id']:
                raise ValidationError('invalid input! With JD inter_static_route_id must exists!',
                                      'inter_static_route_id')

        if 'ANI_min' in data and 'ANI_max' in data and data['ANI_min'] > data['ANI_max']:
            raise ValidationError('invalid input! ANI_min must be less than ANI_max', 'ANI_min')
        if 'DNIS_min' in data and 'DNIS_max' in data and data['DNIS_min'] > data['DNIS_max']:
            raise ValidationError('invalid input! DNIS_min must be less than DNIS_max', 'DNIS_min')
        return data

    @post_load
    def route_type(self, obj):
        if obj.route_type_flg in ['Dynamic Routing']:
            obj.route_type = 'dynamic routing'
            obj.static_route_id = None
            # if not obj.dynamic_route_id:
            #     raise ValidationError('invalid input! dynamic_route_id must exists!',
            #                           'dynamic_route_id')
        if obj.route_type_flg in ['Static Routing', 'Static Routing JD']:
            obj.route_type = 'static routing'
            obj.dynamic_route_id = None
            # if not obj.static_route_id:
            #     raise ValidationError('invalid input! static_route_id must exists!',
            #                           ['static_route_id'])
        if obj.route_type_flg in ['Dynamic Routing - Static Routing', 'Dynamic Routing - Static Routing JD']:
            obj.route_type = 'dynamic routing-static routing'
        if obj.route_type_flg in ['Static Routing - Dynamic routing', 'Static Routing JD - Dynamic routing']:
            obj.route_type = 'static routing-dynamic routing'
        return obj

    class Meta:
        model = model.Route
        fields = ('ANI_prefix', 'DNIS_prefix', 'route_type_flg', 'dynamic_route_id', 'static_route_id',
                  'ANI_min', 'ANI_max', 'DNIS_min', 'DNIS_max', 'intra_static_route_id', 'inter_static_route_id')

class RouteDeleteScheme(BaseModelScheme):
    ANI_prefix = Str(allow_none=True, validate=validate.Regexp(PREFIXES_REGEXP))
    DNIS_prefix = Str(allow_none=True, validate=validate.Regexp(PREFIXES_REGEXP))
    ANI_min = Int(validate=validate.Range(0, 64))
    ANI_max = Int(validate=validate.Range(0, 100))
    DNIS_min = Int(validate=validate.Range(0, 64))
    DNIS_max = Int(validate=validate.Range(0, 100))

    class Meta:
        model = model.Route
        fields = ('ANI_prefix', 'DNIS_prefix', 'ANI_min', 'ANI_max', 'DNIS_min', 'DNIS_max')


class MultipleRoutesDeleteScheme(BaseModelScheme):
    items = Nested(RouteDeleteScheme, many=True, required=True)

    class Meta:
        model = model.Route
        fields = ('items',)


class MultipleRoutesScheme(BaseModelScheme):
    # Array of route items for multiple route creation
    items = Nested(RouteScheme, many=True, required=True)

    @validates_schema
    def validate_items(self, data, **kwargs):
        if not data.get('items'):
            raise ValidationError('The items field must contain at least one route.')
        errors = {}
        for idx, item in enumerate(data['items']):
            if 'route_type_flg' in item and 'JD' in item['route_type_flg']:
                if not item.get('intra_static_route_id'):
                    errors[f'items[{idx}].intra_static_route_id'] = ['invalid input! With JD intra_static_route_id must exists!']
                if not item.get('inter_static_route_id'):
                    errors[f'items[{idx}].inter_static_route_id'] = ['invalid input! With JD inter_static_route_id must exists!']

            if 'ANI_min' in item and 'ANI_max' in item and item['ANI_min'] > item['ANI_max']:
                errors[f'items[{idx}].ANI_min'] = ['invalid input! ANI_min must be less than ANI_max']
            if 'DNIS_min' in item and 'DNIS_max' in item and item['DNIS_min'] > item['DNIS_max']:
                errors[f'items[{idx}].DNIS_min'] = ['invalid input! DNIS_min must be less than DNIS_max']

        if errors:
            raise ValidationError(errors)

    @post_load
    def apply_route_type(self, data):
        for item in data['items']:
            RouteScheme().route_type(item)
        return data

    class Meta:
        model = model.Route
        fields = ('items',)


class RouteGetScheme(RouteScheme):
    route_id = Int()
    route_types = Dict()
    # route_type_name = Str(validate=validate.OneOf(model.Route.route_type_dict.values()))
    route_plan_name = Str()
    dynamic_route_name = Str()
    static_route_name = Str()
    route_type = Choice()
    route_type_flg = Choice()
    update_at = DateTime()
    update_by = Str()
    route_plan_name = Str()
    route_plan_id = Int()
    intra_static_route_id = Int()
    inter_static_route_id = Int()

    class Meta:
        model = model.Route
        fields = ('route_id', 'ANI_prefix', 'DNIS_prefix', 'route_type_flg', 'route_type',
                  'dynamic_route_id', 'static_route_id', 'dynamic_route_name', 'static_route_name',
                  'intra_static_route_id', 'inter_static_route_id',
                  'ANI_min', 'ANI_max', 'DNIS_min', 'DNIS_max', 'update_at', 'update_by', 'route_plan_id',
                  'route_plan_name')
        search_fields = (
            'dynamic_route_id', 'static_route_id', 'route_type_flg', 'ANI_prefix', 'DNIS_prefix', 'ANI_min', 'ANI_max',
            'DNIS_min', 'DNIS_max',
            'intra_static_route_id', 'inter_static_route_id', 'route_plan_name', 'dynamic_route_name',
            'static_route_name', 'intra_static_route_name', 'inter_static_route_name')
        query_fields = ('ANI_prefix_gt', 'ANI_prefix_lt', 'DNIS_prefix_gt', 'DNIS_prefix_lt',
                        'update_at_gt', 'update_at_lt', 'route_id_in', 'intra_static_route_id_isnull')


class ResourceTranslationRefScheme(BaseModelScheme):
    class Meta:
        model = model.ResourceTranslationRef
        exclude = ('ref_id', 'gateway', 'translation', 'time_profile')


class ResourceTranslationRefInnerScheme(BaseModelScheme):
    translation_id = Int(validate=lambda value: _valid('TranslationItem', 'translation_id', value))
    time_profile_id = Int(validate=lambda value: _valid('TimeProfile', 'time_profile_id', value))

    class Meta:
        model = model.ResourceTranslationRef
        fields = ('translation_id', 'time_profile_id')


class ResourceTranslationRefGetScheme(BaseModelScheme):
    trunk_id = Int(attribute='resource_id', load_from='trunk_id', dump_to='trunk_id')

    class Meta:
        model = model.ResourceTranslationRef
        fields = ('ref_id', 'gateway', 'translation', 'time_profile')


# +++ MailSender

class MailSenderScheme(BaseModelScheme):
    # gen module  mail-sender
    # --- Mail Sender ---
    email = Emails(validate=_emails_validate)
    password = Str(validate=validate.Regexp(PASS_REGEXP), required=True)
    enable_auth = Bool()
    mail_server = Str()
    # protocol = Int(attribute='secure',
    #    load_from='protocol',dump_to='protocol' )
    name = Str(validate=[lambda value: _valid_unique('MailSender', 'name', value), validate.Regexp(NAME_REGEXP)])
    mail_port = Str()
    username = Str(validate=[validate.Regexp(USERNAME_REGEXP)])
    secure = Choice()
    update_by = Str()
    update_on = DateTime()

    class Meta:
        model = model.MailSender
        fields = ('email', 'password', 'enable_auth', 'mail_server', 'secure',
                  'name', 'mail_port', 'username')


class MailSenderStatusGetScheme(BaseModelScheme):
    id = Int(validate=[lambda value: _valid_unique('MailSender', 'id', value)])


class MailSenderModifyScheme(MailSenderScheme):
    name = Str(validate=validate.Regexp(NAME_REGEXP))
    username = Str(validate=validate.Regexp(USERNAME_REGEXP))


class MailSenderGetScheme(MailSenderScheme):
    password = Str(attribute='passwd', dump_only=True)

    class Meta:
        model = model.MailSender
        fields = ('id', 'email', 'password', 'enable_auth', 'mail_server', 'secure',
                  'name', 'mail_port', 'username', 'update_by', 'update_on'
                  )
        search_fields = ('id', 'email', 'mail_server', 'secure', 'name', 'username',
                         'update_by'
                         )
        query_fields = ('update_on_gt', 'update_on_lt', 'id_in')


class MailSendersStatusGetScheme(MailSenderGetScheme):
    class Meta:
        model = model.MailSender
        fields = ('status', 'id')


class MailSenderTestScheme(BaseModelScheme):
    # gen module  mail-sender
    # --- Mail Sender ---
    mail_to = Emails(required=True)
    test_subject = Str(required=True)
    test_content = Str(required=True)

    class Meta:
        model = model.MailSender
        fields = ('mail_to', 'test_subject', 'test_content')


# --- MailSender

# +++ MailTemplate
class SendRateTemplateScheme(BaseModelScheme):
    name = Str(validate=[validate.Length(1, 200), lambda value: _valid_unique('SendRateTemplate', 'name', value),
                         validate.Regexp(NAME_REGEXP)])
    subject = Str(validate=validate.Length(max=200))
    content = Str()
    download_method = Choice()
    headers = List(Str(validate=validate.OneOf(model.SendRateTemplate.HEADERS)), allow_none=True)
    sender_id = Int(validate=lambda value: _valid('MailSender', 'id', value))
    mail_cc = Emails()

    class Meta:
        model = model.SendRateTemplate
        fields = ('name', 'subject', 'content', 'download_method', 'headers', 'sender_id', 'mail_cc')


class SendRateTemplateModifyScheme(SendRateTemplateScheme):
    name = Str(validate=[validate.Length(max=200), validate.Regexp(NAME_REGEXP)])

    class Meta:
        model = model.SendRateTemplate
        fields = ('name', 'subject', 'content', 'download_method', 'headers', 'sender_id', 'mail_cc')


class SendRateTemplateGetScheme(SendRateTemplateScheme):
    mail_from = Str()

    class Meta:
        model = model.SendRateTemplate
        fields = ('id', 'name', 'subject', 'content', 'download_method', 'headers', 'sender_id', 'mail_cc', 'mail_from')
        search_fields = ('sender_id', 'mail_cc', 'mail_from', 'name', 'subject', 'download_method')
        query_fields = ('id_in',)


# +++
class SendRateDirectScheme(BaseModelScheme):
    # name = Str(validate=[validate.Length(max=200),lambda value:_valid_unique('SendRateTemplate','name',value)])
    subject = Str(validate=validate.Length(max=200))
    content = Str()
    download_method = Choice()
    headers = List(Str(validate=validate.OneOf(model.SendRateTemplate.HEADERS)), allow_none=True)
    sender_id = Int(validate=lambda value: _valid('MailSender', 'id', value))
    mail_cc = Emails()

    class Meta:
        model = model.SendRateDirect
        fields = ('subject', 'content', 'download_method', 'headers', 'sender_id', 'mail_cc')


class SendRateDirectModifyScheme(SendRateDirectScheme):
    name = Str(validate=[validate.Length(max=200), validate.Regexp(NAME_REGEXP)])

    class Meta:
        model = model.SendRateTemplate
        fields = ('subject', 'content', 'download_method', 'headers', 'sender_id', 'mail_cc')


class SendRateDirectGetScheme(SendRateDirectScheme):
    mail_from = Str()

    class Meta:
        model = model.SendRateDirect
        fields = ('subject', 'content', 'download_method', 'headers', 'sender_id', 'mail_cc', 'mail_from')
        search_fields = ('sender_id', 'mail_cc', 'mail_from', 'subject', 'download_method')
        query_fields = ('id_in',)


# ---

class MailTemplateScheme(BaseModelScheme):
    # gen module  mail-template
    # --- Mail Template ---
    TITLE_LIST = list(model.MAIL_TYPE.values())

    from_mail_id = Int(validate=lambda value: _valid('MailSender', 'id', value))
    subject = Str(validate=validate.Length(max=200))
    cc_mail = Str(validate=validate.Length(max=200))
    # title = Str(validate=validate.OneOf(TITLE_LIST))
    html_content = Str(attribute='_html_content', load_from='_html_content', dump_to='_html_content')
    to_mail = Str(allow_none=True, validate=[validate.OneOf(['', 'billing_email', 'corporate_contact_email',
                                                             'email', 'noc_email', 'rate_email'])], default='email')

    class Meta:
        model = model.MailTemplate
        fields = ('from_mail_id', 'subject', 'cc_mail', 'html_content', 'to_mail')


class AgentMailTemplateScheme(MailTemplateScheme):
    agent_id = Str()
    title = Str(validate=validate.Length(max=200))

    class Meta:
        model = model.MailTemplate
        fields = ('from_mail_id', 'subject', 'cc_mail', 'html_content', 'to_mail', 'title', 'agent_id')


class AgentMailTemplateModifyScheme(AgentMailTemplateScheme):
    class Meta:
        model = model.MailTemplate
        fields = ('from_mail_id', 'subject', 'cc_mail', 'html_content', 'to_mail', 'agent_id')


class MailTemplateGetScheme(MailTemplateScheme):
    class Meta:
        model = model.MailTemplate
        fields = ('title', 'from_mail_id', 'subject', 'cc_mail', 'html_content', 'to_mail')
        search_fields = ('title', 'from_mail_id', 'subject', 'cc_mail', 'to_mail')


# +++ EmailLog
class EmailLogGetScheme(BaseModelScheme):
    date = DateTime()
    sent_to = Str()
    type = Choice()
    subject = Str()
    sent_from = Str()
    content = Str()
    status = Choice()
    client_name = Str()
    error = Str()

    class Meta:
        model = model.EmailLog
        fields = (
            'date', 'sent_to', 'type', 'subject', 'sent_from', 'content', 'status', 'client_id', 'client_name', 'error')
        search_fields = (
            'sent_to', 'type', 'subject', 'sent_from', 'content', 'status', 'client_id', 'client_name', 'error')
        query_fields = ('date_gt', 'date_lt')


# --- EmailLog


# ---CdrReportDetail---
class CdrReportDetail(BaseModelScheme):
    class Meta:
        model = model.ClientCdr


# ---CdrReportDetail---

# ---CdrExportLog---
class CdrExportLogGetScheme(BaseModelScheme):
    id = Int()
    triggered_start_time = DateTime()
    triggered_end_time = DateTime()
    cdr_start_time = DateTime()
    cdr_end_time = DateTime()
    status = Choice()
    progress = Int()
    number_of_rows = Int()
    file_size = Int()

    class Meta:
        model = model.CdrExportLog
        fields = (
            'id', 'triggered_start_time', 'triggered_end_time', 'cdr_start_time', 'cdr_end_time', 'status', 'progress',
            'number_of_rows', 'file_size')
        search_fields = ('id', 'status', 'progress', 'number_of_rows', 'file_size')
        query_fields = (
            'triggered_start_time_gt', 'triggered_end_time_gt', 'triggered_start_time_lt', 'triggered_end_time_lt')


# ---CdrExportLog---

# --- ClientCdr
class CallApiScheme(SuccessScheme):
    payload = Dict()


class ClientCdrSimulateCallScheme(BaseModelScheme):
    server = Str(attribute='origination_source_host_name', validate=lambda value: _valid('VoipGateway', 'name', value))
    ingress_trunk = Int(attribute='ingress_id', validate=lambda value: _valid('Resource', 'resource_id', value))
    ingress_host = Str(attribute='origination_destination_host_name')
    ingress_port = Int(attribute='origination_profile_port')
    ani = Str(attribute='translation_ani', validate=validate.Length(max=16))
    dnis = Str(attribute='lrn_dnis', validate=validate.Length(max=16))
    call_time = DateTime(attribute='time')
    client_cost = Decimal(attribute='egress_cost')

    class Meta:
        model = model.ClientCdr
        fields = ('server', 'ingress_trunk', 'ingress_host', 'ingress_port', 'ani', 'dnis',)  # 'call_time', 'client_cost')


class ClientCdrGetScheme(BaseModelScheme):
    # db_fields = List[Str()]

    class Meta:
        model = model.ClientCdr
        # fields = ( 'ingress_id',)
        search_fields = (
            'ingress_id', 'ingress_client_id', 'egress_id', 'egress_client_id', 'origination_source_host_name',
            'termination_source_host_name', 'translation_ani', 'lrn_dnis')


# query_fields = ('db_fields')


# --- DailyCdrField

class DailyCdrFieldScheme(BaseModelScheme):
    db_name = Str()
    display_name = Str(validate=validate.Regexp(NAME_REGEXP))
    admin_default = Bool()
    admin_default_origination = Bool()
    client_viewable = Bool()
    vendor_viewable = Bool()
    client_cdr_delivery = Bool()
    vendor_cdr_delivery = Bool()
    client_viewable_origination = Bool()

    class Meta:
        model = model.DailyCdrField
        fields = ('db_name', 'display_name', 'admin_default', 'admin_default_origination',
                  'client_viewable', 'vendor_viewable', 'client_cdr_delivery',
                  'vendor_cdr_delivery', 'client_viewable_origination')


class DailyCdrFieldGetScheme(DailyCdrFieldScheme):
    class Meta:
        model = model.DailyCdrField
        fields = ('id',) + DailyCdrFieldScheme.Meta.fields
        search_fields = (
            'id', 'db_name', 'display_name', 'client_cdr_delivery', 'vendor_cdr_delivery', 'client_viewable',
            'vendor_viewable', 'admin_default', 'admin_default_origination', 'client_viewable_origination')


class DailyCdrFieldAutoMiniScheme(BaseModelScheme):
    id = Int(validate=lambda value: _valid('DailyCdrField', 'id', value))
    display_name = Str()
    client_cdr_delivery = Bool()
    vendor_cdr_delivery = Bool()

    class Meta:
        model = model.DailyCdrField
        fields = ('id', 'display_name', 'vendor_cdr_delivery', 'client_cdr_delivery')


class DailyCdrFieldMiniScheme(BaseModelScheme):
    id = Int(validate=lambda value: _valid('DailyCdrField', 'id', value))
    db_name = Str(validate=validate.Regexp(NAME_REGEXP))
    display_name = Str()
    admin_default = Bool()
    admin_default_origination = Bool()
    client_viewable = Bool()
    vendor_viewable = Bool()
    client_cdr_delivery = Bool()
    vendor_cdr_delivery = Bool()
    client_viewable_origination = Bool()

    class Meta:
        model = model.DailyCdrField
        fields = ('id', 'db_name', 'display_name', 'admin_default', 'client_viewable', 'vendor_viewable',
                  'admin_default_origination', 'client_viewable_origination')


class DailyCdrFieldEditManyScheme(BaseModelScheme):
    items = Nested(DailyCdrFieldMiniScheme, many=True)

    class Meta:
        model = model.DailyCdrField
        fields = ('items',)


class DailyCdrFieldAutoEditManyScheme(BaseModelScheme):
    items = Nested(DailyCdrFieldAutoMiniScheme, many=True)

    class Meta:
        model = model.DailyCdrField
        fields = ('items',)


# +++  Registration +++

class SignupIpScheme(BaseModelScheme):
    ip = Str(validate=validate.Regexp(IP_REGEXP))
    port = Int(default=5060, validate=validate.Range(1000, 65535))
    mask = Int(default=32, validate=validate.OneOf((1, 2, 8, 16, 32)))

    class Meta:
        model = model.SignupIp
        fields = ('ip', 'port', 'mask')


class SignupIpGetScheme(BaseModelScheme):
    ip = Str(validate=validate.Regexp(IP_REGEXP))
    port = Int(default=5060)
    mask = Int(default=32)

    class Meta:
        model = model.SignupIp
        fields = ('id', 'ip', 'port', 'mask')


class SignupScheme(BaseModelScheme):
    # gen module  registration
    # --- Registration ---
    billing_address = Str(validate=validate.Length(max=40))
    billing_city = Str(validate=validate.Length(max=40))
    billing_contact_name = Str(validate=validate.Length(max=40))
    billing_country = Str(validate=validate.Length(max=40))
    billing_email = Emails()
    billing_phone = Str(validate=[validate.Length(max=40), validate.Regexp(PHONE_REGEXP)])
    billing_state = Str(validate=validate.Length(max=40))
    billing_zip = Str(validate=validate.Length(max=40))
    city = Str(validate=validate.Length(max=40))
    company = Str(validate=validate.Length(max=100))
    company_detail = Str(validate=validate.Length(max=1000))
    country = Str(validate=validate.Length(max=40))
    client_name = Str(required=True, allow_none=False,
                      validate=[validate.Length(max=100), validate.Regexp(NAME_REGEXP)])
    # ,lambda value:_valid_unique2('Client','name',value,'Client {} already exists!'.format(value))])
    ip_address = Nested(SignupIpScheme, many=True)
    # ip_address_1 = Str()
    # ip_address_2 = Str()
    # ip_address_3 = Str()
    # ip_address_4 = Stri()
    # ip_address_5 = Str()
    # ip_address_6 = Str()
    main_email = Emails(required=True, allow_none=False)  #
    # ,validate=[lambda value:_valid_unique2('Signup','email',value,'Registration {} already exists!'.format(value)),
    #    lambda value:_valid_unique2('User','email',value,'User {} already exists!'.format(value))])
    modified_on = DateTime()
    noc_email = Emails()
    password = Str(validate=validate.Length(max=50))
    rate_send_from_email = Emails()
    rate_send_to_email = Emails()
    send_signup_notification = Int(validate=validate.Range(0, 1))
    signed_up_on = DateTime()
    state = Str(validate=validate.Length(max=40))
    status = Choice()
    username = Str(required=True, allow_none=False, validate=[validate.Length(max=40), validate.Regexp(NAME_REGEXP)])
    # ,lambda value:_valid_unique2('User','name',value, 'User {} already exists!'.format(value))])
    zip = Str(validate=validate.Length(max=40))
    tax_id = Str(validate=validate.Length(max=100))
    phone = Str(validate=[validate.Length(max=40), validate.Regexp(PHONE_REGEXP)])
    address = Str(validate=validate.Length(max=40))
    products = List(
        Int(validate=lambda value: _valid('ProductRoutRateTable', 'id', value, 'Invalid product id {}'.format(value))))
    referral = Str(validate=validate.Length(max=500))
    company_type = Choice()
    client_type = Choice()

    class Meta:
        model = model.Signup
        fields = ('billing_address', 'billing_city', 'billing_contact_name', 'billing_country',
                  'billing_email', 'billing_phone', 'billing_state', 'billing_zip', 'city', 'company',
                  'company_detail', 'country', 'client_name', 'ip_address',
                  # 'ip_address_1','ip_address_2','ip_address_3','ip_address_4','ip_address_5','ip_address_6',
                  'main_email',
                  # 'modified_on',
                  'noc_email', 'password', 'rate_send_from_email',
                  'rate_send_to_email', 'send_signup_notification',
                  # 'signed_up_on',
                  'state',
                  # 'status',
                  'username', 'zip', 'tax_id', 'phone', 'address', 'referral', 'products', 'company_type',
                  'client_type')


class SignupModifyScheme(SignupScheme):
    main_email = Email(required=True, allow_none=False,
                       validate=[lambda value: _valid_unique('User', 'email', value, ' in users')])


class SignupGetScheme(SignupScheme):
    ip_address = Nested(SignupIpGetScheme, many=True)
    status = Choice()
    referral_name = Str()

    class Meta:
        model = model.Signup
        fields = ('id', 'status', 'signed_up_on', 'modified_on', 'referral_name',
                  'is_finished') + SignupScheme.Meta.fields
        search_fields = ('username', 'status', 'city', 'company', 'main_email', 'referral', 'client_name',
                         'phone', 'company_type', 'client_type', 'is_finished')
        query_fields = ('signed_up_on_gt', 'signed_up_on_lt', 'modified_on_gt', 'modified_on_lt', 'id_in')


class SignupApproveScheme(BaseModelScheme):
    status = Choice()
    rate_table_id = Int()
    route_strategy_id = Int()
    product_id = Int()

    class Meta:
        model = model.Signup
        fields = ('status', 'rate_table_id', 'route_strategy_id', 'product_id')


# --- Registration ----

class AgentReferalNameScheme(BaseModelScheme):
    agent_name = Str()

    class Meta:
        model = model.Agent
        fields = ('agent_name',)


# +++ Failover
class GlobalRouteErrorScheme(BaseModelScheme):
    class Meta:
        model = model.GlobalRouteError
        # fields=('to_sip_code','to_sip_string')
        exclude = ('id',)


class GlobalRouteErrorModifyScheme(BaseModelScheme):
    class Meta:
        model = model.GlobalRouteError
        fields = ('to_sip_code', 'to_sip_string')


# exclude = ('id',)


class GlobalRouteErrorGetScheme(BaseModelScheme):
    class Meta:
        model = model.GlobalRouteError
        search_fields = ('id', 'error_code', 'default_to_sip_code', 'to_sip_code')


class GlobalRouteErrorManyScheme(BaseModelScheme):
    items = Nested(GlobalRouteErrorGetScheme)

    class Meta:
        model = model.GlobalRouteError


class FailoverScheme(BaseModelScheme):
    id = Int()
    failover_method = Choice(attribute='failover_strategy')
    match_code = Int(attribute='from_sip_code')
    return_code = Int(attribute='to_sip_code')
    return_clause = Str(attribute='to_sip_string')

    class Meta:
        model = model.TerminationFailover
        fields = ('failover_method', 'match_code', 'return_code', 'return_clause')


# class GlobalFailoverScheme(BaseModelScheme):
#     id = Int()
#     failover_method = Choice()
#     match_code = Int(validate=validate.Range(100, 999))
#     return_code = Int(validate=validate.Range(100, 999))
#     return_clause = Str(validate=validate.Length(0, 100))

#     class Meta:
#         model = model.GlobalFailover
#         fields = ('failover_method', 'match_code', 'return_code', 'return_clause')


# class GlobalFailoverModifyScheme(GlobalFailoverScheme):
#     class Meta:
#         model = model.GlobalFailover
#         fields = ('failover_method', 'return_code', 'return_clause')


# class GlobalFailoverGetScheme(GlobalFailoverScheme):
#     class Meta:
#         model = model.GlobalFailover
#         fields = ('id', 'failover_method', 'return_code', 'return_clause')
#         search_fields = ('failover_method', 'return_code', 'return_clause')


# class GlobalFailoverManyScheme(BaseModelScheme):
#     items = Nested(GlobalFailoverGetScheme, many=True)

#     class Meta:
#         model = model.GlobalFailover
#         fields = ('items',)


class TerminationFailoverScheme(BaseModelScheme):
    id = Int()
    failover_method = Choice()
    match_code = Int(validate=validate.Range(100, 999))
    return_code = Int(validate=validate.Range(100, 999))
    return_clause = Str(validate=validate.Length(0, 100))
    # defaults = Nested(GlobalFailoverScheme)

    class Meta:
        model = model.TerminationFailover
        fields = ('failover_method', 'return_code', 'return_clause')


class TerminationFailoverModifyScheme(TerminationFailoverScheme):
    class Meta:
        model = model.TerminationFailover
        fields = ('failover_method', 'return_code', 'return_clause')


class TerminationFailoverGetScheme(TerminationFailoverScheme):
    class Meta:
        model = model.TerminationFailover
        fields = ('id', 'failover_method', 'return_code', 'return_clause')
        search_fields = ('failover_method', 'return_code', 'return_clause')


class TerminationFailoverManyScheme(BaseModelScheme):
    items = Nested(TerminationFailoverGetScheme, many=True)

    class Meta:
        model = model.TerminationFailover
        fields = ('items',)


class OriginationFailoverScheme(BaseModelScheme):
    id = Int()
    failover_method = Choice()
    match_code = Int(validate=validate.Range(100, 999))
    return_code = Int(validate=validate.Range(100, 999))
    return_clause = Str(validate=validate.Length(0, 100))
    # defaults = Nested(GlobalFailoverScheme)

    class Meta:
        model = model.OriginationFailover
        fields = ('failover_method', 'return_code', 'return_clause')


class OriginationFailoverModifyScheme(OriginationFailoverScheme):
    class Meta:
        model = model.OriginationFailover
        fields = ('failover_method', 'return_code', 'return_clause')


class OriginationFailoverGetScheme(OriginationFailoverScheme):
    class Meta:
        model = model.OriginationFailover
        fields = ('id', 'failover_method', 'return_code', 'return_clause')
        search_fields = ('failover_method', 'id', 'return_code', 'return_clause')


class OriginationFailoverManyScheme(BaseModelScheme):
    items = Nested(OriginationFailoverGetScheme, many=True)

    class Meta:
        model = model.OriginationFailover
        fields = ('items',)


# --- Failover
# region +++ResourceFailover+++
class ResourceFailoverScheme(FailoverScheme):
    resource_id = Int()
    resource = Nested('ResourceScheme', many=False)

    class Meta:
        model = model.TerminationFailover
        fields = ('failover_method', 'match_code', 'return_code', 'return_clause')


class ResourceNextRouteRuleScheme(FailoverScheme):
    resource_id = Int()
    # failover_method = Int()
    # resource = Nested('ResourceScheme', many=False)

    class Meta:
        model = model.ResourceNextRouteRule
        fields = ('id', 'failover_method', 'match_code', 'return_code', 'return_clause')


class ResourceNextRouteRuleModifyScheme(ResourceNextRouteRuleScheme):
    class Meta:
        model = model.ResourceNextRouteRule
        fields = ('failover_method', 'match_code', 'return_code', 'return_clause')


class TrunkTemplateFailoverScheme(Schema):
    failover_method = Str(validate=validate.OneOf(('Fail to Next Host', 'Fail to Next Trunk', 'Stop')))
    match_code = Int(validate=validate.Range(100, 999))
    return_code = Int(validate=validate.Range(100, 999))
    return_clause = Str(validate=validate.Length(0, 100))


# class ResourceFailoverSchemeGet(ResourceFailoverScheme):
#     class Meta:
#         model = model.ResourceFailover
#         fields = ('id', 'failover_method', 'match_code', 'return_code', 'return_clause', 'resource_id')
#         search_fields = ('id', 'failover_method', 'match_code', 'return_code', 'return_clause', 'resource_id')


class ResourceFailoverSchemeModify(ResourceFailoverScheme):
    pass


class SipErrorCodeScheme(BaseModelScheme):
    resource_id = Int()
    return_code = Int()
    return_code_str = Str(validate=validate.Length(0, 100))
    switch_error_code = Int()
    switch_error_code_str = Str(validate=validate.Length(0, 100))

    class Meta:
        model = model.SipErrorCode
        fields = ('return_code', 'return_code_str', 'switch_error_code', 'switch_error_code_str')


class SipErrorCodeSchemeGet(SipErrorCodeScheme):
    class Meta:
        model = model.SipErrorCode
        fields = ('sip_error_code_id', 'return_code', 'return_code_str', 'switch_error_code', 'switch_error_code_str')
        search_fields = ('sip_error_code_id', 'return_code', 'return_code_str', 'switch_error_code', 'switch_error_code_str')


# endregion ---ResourceFailover---
# +++ FtpConf


class FtpConfScheme(BaseModelScheme):
    # gen module  ftp-configuration
    # --- FTP Job ---
    active = Bool()
    compression = Choice()
    egress_trunks = List(Int())
    file_breakdown = Choice()
    frequency = Choice(allow_none=False)
    ftp_directory = Str(allow_none=False)
    ftp_password = Str(allow_none=False)
    ftp_server_ip = Str(allow_none=False, validate=validate.Regexp(IP_REGEXP), attribute='ftp_server_ip_1')
    ftp_server_port = Str(allow_none=False, validate=validate.Regexp(DIGITS_REGEXP))
    ftp_username = Str(allow_none=False)
    include_all_egress = Bool()
    include_all_ingress = Bool()
    include_fields = List(Str(validate=validate.OneOf(list(FIELD_MAP.keys()))))
    include_headers = List(Str())
    include_header = Bool()
    ingress_trunks = List(Int())
    max_line_per_file = Int()
    name = Str()
    non_zero = Bool()
    # orig_return_code = List(Int(validate=validate.OneOf(list(model.GlobalFailover.CODES.keys()))))
    # term_return_code = List(Int(validate=validate.OneOf(list(model.GlobalFailover.CODES.keys()))))
    time = Str(validate=validate.Regexp(TIME_REGEXP))
    day_of_week = Choice()
    every_hours = Int(allow_none=True, validate=validate.Range(0, 23))

    class Meta:
        model = model.FtpConf
        fields = ('active', 'compression', 'egress_trunks', 'file_breakdown', 'frequency', 'ftp_directory',
                  'ftp_password', 'ftp_server_ip', 'ftp_server_port', 'ftp_username', 'include_all_egress',
                  'include_all_ingress', 'include_fields', 'include_header', 'ingress_trunks',
                  'max_line_per_file', 'name', 'non_zero', 'orig_return_code', 'term_return_code', 'time',
                  'day_of_week', 'every_hours')


class FtpConfGetScheme(FtpConfScheme):
    ftp_server_ip = Str()
    last_ftp_time = DateTime()
    status = Choice()

    class Meta:
        model = model.FtpConf
        fields = ('id', 'last_ftp_time', 'status') + FtpConfScheme.Meta.fields
        search_fields = ('active', 'name', 'time', 'ftp_server_ip', 'status')
        query_fields = ('time_gt', 'time_lt', 'last_ftp_time_gt', 'last_ftp_time_lt')


class FtpConfActivateScheme(FtpConfScheme):
    class Meta:
        model = model.FtpConf
        fields = ('active',)
        search_fields = ('active', 'name', 'time', 'ftp_server_ip')
        query_fields = ('time_gt', 'time_lt', 'id_in')


class FtpConfCopyScheme(BaseModelScheme):
    class Meta:
        model = model.FtpConf
        fields = ('name',)


class FtpConfManualTriggerScheme(FtpConfScheme):
    start_time = DateTime()
    end_time = DateTime()

    @validates_schema(skip_on_field_errors=True)
    def check_date(self, data):
        """
        try:
            valid_date(data['start_time'])
        except:
            raise ValidationError({'start_time': ['invalid date {}'.format(data['start_time'])]})
        try:
            valid_date(data['end_time'])
        except:
            raise ValidationError({'end_time': ['invalid date {}'.format(data['end_time'])]})
        """
        if data['start_time'] >= data['end_time']:
            raise ValidationError('"start_time" must be less than "end_time"')
        return data

    class Meta:
        model = model.FtpConf
        fields = ('start_time', 'end_time')


# --- FtpConf

# +++ FtpServerLog
class FtpServerLogGetScheme(BaseModelScheme):
    id = Int()
    time = DateTime()
    cmd = Str()
    response = Str()

    class Meta:
        model = model.FtpServerLog
        fields = ('id', 'time', 'cmd', 'responce')
        search_fields = ('id', 'cmd', 'responce')
        query_fields = ('time_gt', 'time_lt')


# ---FtpServerLog

# +++ FtpCdrLog
class FtpCdrLogGetScheme(BaseModelScheme):
    id = Int()
    ftp_start_time = DateTime()
    ftp_end_time = DateTime()
    status = Choice()
    ftp_ip = String()
    ftp_dir = String()
    client_alias = String()
    ftp_conf_id = Integer()
    cdr_start_time = DateTime()
    cdr_end_time = DateTime()
    pid = Integer()
    result = Nested('FtpCdrLogDetailGetScheme', many=True)
    status = Choice()

    class Meta:
        model = model.FtpCdrLog
        fields = ('id', 'ftp_start_time', 'ftp_end_time', 'ftp_ip', 'ftp_dir', 'client_alias',
                  'cdr_start_time', 'cdr_end_time', 'pid', 'result', 'status')
        search_fields = ('id', 'client_alias', 'status')
        query_fields = ('cdr_start_time_gt', 'cdr_end_time_lt', 'ftp_start_time_gt', 'ftp_end_time_lt')


class FtpCdrLogDetailGetScheme(BaseModelScheme):
    id = Int()
    ftp_cdr_log_id = Int()
    create_time = DateTime()
    target = String()
    ftp_ip = String()
    ftp_dir = String()
    ftp_post = Str()
    local_file_path = String()

    class Meta:
        model = model.FtpCdrLogDetail
        fields = ('id', 'target', 'ftp_post', 'local_file_path')
        search_fields = ('id', 'target', 'ftp_post', 'local_file_path')
        query_fields = ('create_time_gt', 'create_time_lt')


# ---FtpCdrLog


# region +++ClientInvoiceSettings+++
class ClientInvoiceSettingsScheme(BaseModelScheme):
    id = Int()
    invoice_name = Str(validate=[validate.Length(max=200),
                                 lambda value: _valid_unique('InvoiceSetting', 'invoice_name', value)], required=True)
    tpl_number = Choice()
    pdf_tpl = Str()
    company_info = Str(validate=[validate.Length(max=2500)], allow_none=True)
    overlap_invoice_protection = Bool()
    cdr_fields = List(Str(validate=lambda value: _valid('DailyCdrField', 'field', value)))
    invoice_send_mode = Choice()
    invoice_send_mode_cdr = Choice()
    company_info_location = Choice(allow_none=True)
    billing_info = Str(allow_none=True)
    billing_info_position = Choice()
    invoice_decimal_digits = Int()
    invoices_logo_id = Int()
    create_by = Str()
    create_on = DateTime()
    update_on = DateTime()
    used_by_client = Int()
    used_by_agent = Int()

    class Meta:
        model = model.InvoiceSetting
        fields = (
            'invoice_name', 'company_info', 'overlap_invoice_protection', 'cdr_fields',
            'invoice_send_mode', 'invoice_send_mode_cdr', 'company_info_location', 'invoice_decimal_digits',
            'billing_info', 'billing_info_position', 'invoices_logo_id')


class ClientInvoiceSettingsSchemeGet(ClientInvoiceSettingsScheme):
    class Meta:
        model = model.InvoiceSetting
        fields = (
            'id', 'invoice_name', 'company_info', 'overlap_invoice_protection',
            'cdr_fields',
            'invoice_send_mode', 'invoice_send_mode_cdr', 'company_info_location', 'invoice_decimal_digits',
            'billing_info',
            'billing_info_position', 'invoices_logo_id',
            'create_by', 'create_on',
            'update_on',
            'used_by_client', 'used_by_agent')
        search_fields = (
            'id', 'invoice_name', 'tpl_number', 'pdf_tpl', 'company_info', 'overlap_invoice_protection',
            'invoice_send_mode', 'invoice_send_mode_cdr', 'company_info_location', 'invoice_decimal_digits',
            'billing_info',
            'billing_info_position', 'invoices_logo_id', 'create_by', 'used_by_client', 'used_by_agent')
        query_fields = ('create_on_gt', 'update_on_gt', 'create_on_lt', 'update_on_lt')


class ClientInvoiceSettingsSchemeModify(ClientInvoiceSettingsScheme):
    invoice_name = Str(validate=[validate.Length(max=200)])


# endregion ---ClientInvoiceSettings---
# +++
class SystemParameterLoginPageScheme(BaseModelScheme):
    # gen module  payment-setting
    # --- Payment Setting ---
    image_url = Str(attribute='_image_url')
    fit_to_screen = Bool()
    use_captcha = Bool()
    content = Str()
    default_avatar_id = Int(allow_none=True, validate=lambda value: _valid('ImportExportLogs', 'id', value))
    logo_image_id = Int(allow_none=True, validate=lambda value: _valid('ImportExportLogs', 'id', value))
    logo_icon_id = Int(allow_none=True, validate=lambda value: _valid('ImportExportLogs', 'id', value))
    favicon_id = Int(allow_none=True, validate=lambda value: _valid('ImportExportLogs', 'id', value))
    base_url = Str(validate=validate.Length(0, 512))
    doc_url = URL(validate=validate.Length(0, 512))
    public_logo_url = Str()
    login_footer_color = Str()

    class Meta:
        model = model.SystemParameter
        fields = (
            'image_url', 'fit_to_screen', 'use_captcha', 'content', 'default_avatar_id', 'logo_image_id',
            'logo_icon_id',
            'favicon_id', 'base_url', 'doc_url', 'public_logo_url', 'login_footer_color')


class SystemParameterLoginPageSchemeGet(SystemParameterLoginPageScheme):
    pass


class SystemParameterPortalScheme(BaseModelScheme):
    enable_origination_traffic = Bool()
    enable_termination_traffic = Bool()
    portal_display_sip_ip = Bool()
    portal_enable_egress_trunks = Bool()
    portal_enable_ingress_trunks = Bool()
    portal_enable_ip_change = Bool()
    portal_enable_public_products = Bool()
    portal_offset = Bool()

    class Meta:
        model = model.SystemParameter
        fields = ('portal_display_sip_ip', 'portal_enable_egress_trunks', 'portal_enable_ingress_trunks',
                  'portal_enable_ip_change', 'portal_enable_public_products', 'portal_offset',
                  'enable_origination_traffic', 'enable_termination_traffic')


class SystemParameterPortalSchemeGet(SystemParameterPortalScheme):
    pass


class SystemParameterPaymentSettingScheme(BaseModelScheme):
    # gen module  payment-setting
    # --- Payment Setting ---
    cc_email = Str()
    credit_full_payment = Bool()
    emails = Str()
    enable_email_notification = Bool()
    paypal_account = Str()
    paypal_skey = Str()
    paypal_fee = Int()
    paypal_test_mode = Boolean()
    stripe_fee = Int()
    stripe_publisher_key = Str()
    stripe_secret_key = Str()
    # stripe_test_mode = Boolean()
    payment_received_confirmation = Bool()
    auto_carrier_notification = Bool()
    paypal_charge_type = Choice()
    stripe_charge_type = Choice()

    class Meta:
        model = model.SystemParameter
        fields = ('cc_email',
                  # 'credit_full_payment',
                  'emails', 'enable_email_notification', 'paypal_account', 'paypal_skey', 'paypal_test_mode',
                  'paypal_fee', 'stripe_fee', 'stripe_publisher_key', 'stripe_secret_key',
                  # 'stripe_test_mode',
                  'auto_carrier_notification', 'payment_received_confirmation', 'paypal_charge_type',
                  'stripe_charge_type')


class SystemParameterPaymentSettingSchemeGet(SystemParameterPaymentSettingScheme):
    class Meta:
        model = model.SystemParameter
        fields = SystemParameterPaymentSettingScheme.Meta.fields + ('webhook_stripe_url', 'webhook_paypal_url')


class SystemParameterPaymentSettingPublicSchemeGet(SystemParameterPaymentSettingScheme):
    class Meta:
        model = model.SystemParameter
        fields = ('paypal_account', 'stripe_publisher_key', 'paypal_test_mode')


# ---

# +++ SystemParameterInvoiceSettingScheme
class SystemParameterInvoiceSettingScheme(BaseModelScheme):
    # gen module  invoice-setting
    # --- Invoice PDF Setting ---
    # allow_invoice_overlap = Bool()
    overlap_invoice_protection = Bool(allow_none=True)
    billing_info = Str(allow_none=True)
    billing_info_position = Choice(allow_none=True)
    cdr_fields = List(Str(), allow_none=True)
    company_info = Str(allow_none=True)
    company_info_position = Choice(allow_none=True)
    decimal = Int(allow_none=True)  # Str()
    invoice_number_format = Str(validate=validate.Regexp(DIGITS_REGEXP), allow_none=True)
    # logo = Str()
    logo_url = Url(dump_only=True, attribute='_logo_url', allow_none=True)
    invoices_logo_id = Int(validate=lambda value: _valid('ImportExportLogs', 'id', value), allow_none=True)
    send_rate_as = Email(allow_none=True)
    invoice_send_mode = Choice(allow_none=True)
    invoice_send_mode_cdr = Choice(allow_none=True)

    class Meta:
        model = model.SystemParameter
        fields = ('overlap_invoice_protection', 'billing_info', 'billing_info_position', 'cdr_fields',
                  'company_info', 'company_info_position', 'decimal', 'invoice_number_format',
                  # 'logo',
                  'invoices_logo_id', 'logo_url', 'send_rate_as', 'invoice_send_mode', 'invoice_send_mode_cdr')


class SystemParameterInvoiceSettingSchemeGet(SystemParameterInvoiceSettingScheme):
    pass


# --- SystemParameterInvoiceSettingScheme

# +++ SystemParameterSystemSettingScheme
class SystemParameterSystemSettingScheme(BaseModelScheme):
    default_us_ij_rule = Choice()
    sys_timezone = Str()
    sys_currency = Str()
    pre_load_data = Bool()
    landing_page = Choice()
    code_deck = Str()
    report_count = Choice()
    inactivity_timeout = Int(validate=validate.Range(1, 60 * 24 * 7))
    default_billing_decimal = Int(validate=validate.Range(0, 6))
    default_report_decimal = Int(validate=validate.Range(0, 6))
    is_hide_unauthorized_ip = Int()
    base_url = Str()
    enable_user_registration = Bool(allow_none=True)
    operating_company = Str()
    invoices_tplno = Str()

    class Meta:
        model = model.SystemParameter
        fields = ('is_show_mutual_balance', 'require_comment', 'inactivity_timeout', 'report_count',
                  'switch_alias', 'sys_timezone', 'welcome_message', 'default_us_ij_rule',
                  'code_deck', 'sys_currency', 'pre_load_data', 'landing_page', 'default_billing_decimal',
                  'default_report_decimal', 'is_hide_unauthorized_ip', 'base_url', 'enable_user_registration',
                  'operating_company', 'invoices_tplno')


class SystemParameterSystemSettingSchemeGet(SystemParameterSystemSettingScheme):
    pass


class SystemParameterSystemSettingSchemePublicGet(SystemParameterSystemSettingScheme):
    class Meta:
        model = model.SystemParameter
        fields = ('enable_user_registration',)


# --- SystemParameterSystemSettingScheme

# +++ SystemParameterSystemEmailsScheme
class SystemParameterSystemEmailsScheme(BaseModelScheme):
    noc_email = Emails()
    system_admin_email = Emails()
    finance_email = Emails()
    ftp_email = Emails()

    class Meta:
        model = model.SystemParameter
        fields = ('noc_email', 'system_admin_email', 'finance_email', 'ftp_email')


class SystemParameterSystemEmailsSchemeGet(SystemParameterSystemEmailsScheme):
    pass


# --- SystemParameterSystemEmailsScheme


# +++ SystemParameterAutoRateEmailScheme
class SystemParameterAutoRateEmailScheme(BaseModelScheme):
    host = Str(attribute='auto_rate_smtp')
    port = Int(attribute='auto_rate_smtp_port', validate=validate.OneOf((143, 993)))
    username = Str(attribute='auto_rate_username')
    pwd = Str(attribute='auto_rate_pwd', load_only=True)
    ssl = Str(attribute='auto_rate_mail_ssl', validate=validate.OneOf(('no', 'yes')))

    class Meta:
        model = model.SystemParameter
        fields = ('host', 'port', 'username', 'pwd', 'ssl')


class SystemParameterAutoRateEmailSchemeGet(SystemParameterAutoRateEmailScheme):
    pass


# --- SystemParameterAutoRateEmailScheme

# +++ SystemParameterSystemTimeoutsScheme
class SystemParameterSystemTimeoutsScheme(BaseModelScheme):
    ingress_pdd_timeout = Int(validate=validate.Range(0, 600000))
    egress_pdd_timeout = Int(validate=validate.Range(0, 60000))
    ring_timeout = Int(validate=validate.Range(0, 60000))
    call_timeout = Int(validate=validate.Range(0, 86400), allow_none=True, attribute='_call_timeout')

    class Meta:
        model = model.SystemParameter
        fields = ('ingress_pdd_timeout', 'egress_pdd_timeout',
                  'ring_timeout', 'call_timeout')


class SystemParameterSystemTimeoutsSchemeGet(SystemParameterSystemTimeoutsScheme):
    pass


# --- SystemParameterSystemTimeoutsScheme


# +++ SystemParameterQosRoutingScheme
class SystemParameterQosRoutingScheme(BaseModelScheme):
    low_call_attempt_handling = Choice()
    qos_sample_period = Choice()
    minimal_call_attempt_required = Int(validate=validate.Range(0, 60000))

    class Meta:
        model = model.SystemParameter
        fields = ('low_call_attempt_handling', 'qos_sample_period', 'minimal_call_attempt_required')


class SystemParameterQosRoutingSchemeGet(SystemParameterQosRoutingScheme):
    pass


# --- SystemParameterQosRoutingScheme


class SystemParameterFirstTimeWizardScheme(BaseModelScheme):
    enable_first_time_wizard = Bool()

    class Meta:
        model = model.SystemParameter
        fields = ('enable_first_time_wizard',)


class SystemParameterFirstTimeWizardSchemeGet(SystemParameterFirstTimeWizardScheme):
    pass


# ---

# region +++C4Lrn+++

class C4LrnScheme(BaseModelScheme):
    id = Int()
    srv1_ip = Str(validate=[validate.Length(max=16), validate.Regexp(IP_REGEXP)])
    srv1_port = Int()
    srv2_ip = Str(validate=[validate.Length(max=16), validate.Regexp(IP_REGEXP)])
    srv2_port = Int()
    timeout = Int()
    retry_attempts = Int()

    class Meta:
        model = model.C4Lrn
        fields = ('srv1_ip', 'srv1_port', 'srv2_ip', 'srv2_port', 'retry_attempts', 'timeout',)


class C4LrnSchemeGet(C4LrnScheme):
    class Meta:
        model = model.C4Lrn
        fields = ('id', 'srv1_ip', 'srv1_port', 'srv2_ip', 'srv2_port', 'retry_attempts', 'timeout',)
        search_fields = ('id', 'srv1_ip', 'srv1_port', 'srv2_ip', 'srv2_port', 'retry_attempts', 'timeout',)


class C4LrnSchemeModify(C4LrnScheme):
    pass


# endregion ---C4Lrn---


# +++ Jurisdiction_prefix
class JurisdictionPrefixScheme(BaseModelScheme):
    # --- US Jurisdiction ---
    prefix = Str(validate=validate.Regexp(PREFIX_REGEXP))
    country = Str()
    lata = Str()
    ocn = Str()
    state = Str()
    effective_date = DateTime()

    class Meta:
        model = model.JurisdictionPrefix
        fields = ('prefix', 'country', 'lata', 'ocn', 'state', 'effective_date')


class JurisdictionPrefixGetScheme(JurisdictionPrefixScheme):
    class Meta:
        model = model.JurisdictionPrefix
        fields = ('id', 'prefix', 'country', 'lata', 'ocn', 'state', 'effective_date')
        search_fields = ('prefix', 'country', 'lata', 'ocn', 'state', 'block_id')
        query_fields = ('effective_date_gt', 'effective_date_lt', 'id_in')


# ---JurisdictionPrefix

# +++ LoopDetection
class LoopDetectionDetailInnerScheme(BaseModelScheme):
    trunk_id = Int(validate=lambda value: _valid('Resource', 'resource_id', value))
    trunk_name = Str(dump_only=True)

    class Meta:
        model = model.LoopDetectionDetail
        fields = ('trunk_id',)


class LoopDetectionDetailInnerGetScheme(BaseModelScheme):
    class Meta:
        model = model.LoopDetectionDetail
        fields = ('trunk_id', 'trunk_name')


class LoopDetectionScheme(BaseModelScheme):
    # --- Loop Detection Rule ---
    block_second = Int()
    ingress_trunks = Nested(LoopDetectionDetailInnerScheme, attribute='_ingress_trunks', many=True)
    name = Str(validate=validate.Regexp(NAME_REGEXP))
    occurance = Int()
    period = Int()

    class Meta:
        model = model.LoopDetection
        fields = ('block_second', 'ingress_trunks', 'name', 'occurance', 'period')


class LoopDetectionGetScheme(LoopDetectionScheme):
    ingress_trunks = Nested(LoopDetectionDetailInnerGetScheme, many=True)
    trunk_count = Int()

    class Meta:
        model = model.LoopDetection
        fields = ('id', 'trunk_count') + LoopDetectionScheme.Meta.fields
        search_fields = ('block_second', 'name', 'occurance', 'period')
        query_fields = ('id_in', 'trunk_count_gt', 'trunk_count_lt')


class LoopDetectionActivateScheme(LoopDetectionGetScheme):
    class Meta:
        model = model.LoopDetection
        fields = ('block_second',)
        search_fields = ('block_second', 'name', 'occurance', 'period')


class LoopDetectionDetailGetScheme(LoopDetectionScheme):
    trunk_id = Integer()
    trunk_name = Str()
    counter_time = Int()
    number = Int()
    block_time = Int()
    loop_detection_id = Int()

    class Meta:
        model = model.Resource
        fields = ('loop_detection_id', 'trunk_id', 'trunk_name', 'counter_time', 'number', 'block_time')
        search_fields = ('loop_detection_id', 'trunk_id', 'trunk_name')
        query_fields = ('block_time_gt', 'block_time_lt')


# --- LoopDetection

# +++ FraudDetection
class FraudDetectionScheme(BaseModelScheme):
    # gen module  fraud-detection-rule
    # --- Fraud Detection Rule ---
    hour_24_duration = Int()
    hour_24_revenue = Int()
    enable_block = Bool()
    enable_email = Bool()

    ingress_trunks = List(Int(), allow_none=True)
    # mail_sender_id = Int()
    email_to = Choice()
    name = Str(validate=validate.Regexp(NAME_REGEXP))
    one_hour_duration = Int()
    one_hour_revenue = Int()
    active = Bool(allow_none=True, default=True)
    html_content = Str()
    # send_to_admin = Str()
    # send_to_partner = Str()
    subject = Str()

    class Meta:
        model = model.FraudDetection
        fields = ('hour_24_duration', 'hour_24_revenue', 'enable_block', 'enable_email',
                  'ingress_trunks', 'email_to', 'name', 'one_hour_duration', 'one_hour_revenue',
                  'active',
                  # 'html_content','subject'
                  # 'send_to_admin','send_to_partner',
                  )


class FraudDetectionGetScheme(FraudDetectionScheme):
    class Meta:
        model = model.FraudDetection
        fields = ('id', 'update_on', 'update_by') + FraudDetectionScheme.Meta.fields
        search_fields = ('active', 'name', 'enable_block', 'update_by', 'enable_email', 'email_to',
                         'name', 'one_hour_duration', 'one_hour_revenue',
                         'hour_24_duration', 'hour_24_revenue')
        query_fields = ('update_on_gt', 'update_on_lt')


class FraudDetectionActivateScheme(FraudDetectionScheme):
    class Meta:
        model = model.FraudDetection
        fields = ('active',)
        search_fields = ('active', 'name', 'enable_block', 'update_by',
                         'name', 'one_hour_duration', 'one_hour_revenue',
                         'hour_24_duration', 'hour_24_revenue')
        query_fields = ('update_on_gt', 'update_on_lt', 'id_in')


class FraudDetectionLogDetailGetScheme(BaseModelScheme):
    id = Int()
    fraud_detection_log_id = Int()
    ingress_id = Int()
    trunk_name = Str()
    block_type = Choice()
    limit_value = Int()
    actual_value = Str()
    partner_email_status = Bool()
    system_email_status = Bool()
    is_block = Bool()
    is_send_email = Bool()

    class Meta:
        model = model.FraudDetectionLogDetail
        fields = ('id', 'fraud_detection_log_id', 'ingress_id', 'trunk_name', 'block_type', 'limit_value',
                  'actual_value', 'partner_email_status', 'system_email_status', 'is_block', 'is_send_email')
        search_fields = ('fraud_detection_log_id', 'ingress_id', 'trunk_name')


class FraudDetectionLogGetScheme(FraudDetectionScheme):
    id = Int()
    fraud_detection_id = Int()
    create_on = DateTime()
    create_by_name = Str()
    status = Choice()
    finish_time = DateTime()
    rule_name = Str()
    detail = Nested(FraudDetectionLogDetailGetScheme, many=True)
    duration = Str()

    class Meta:
        model = model.FraudDetectionLog
        fields = (
            'id', 'fraud_detection_id', 'create_on', 'create_by_name', 'status', 'finish_time', 'rule_name', 'detail',
            'duration')
        search_fields = ('fraud_detection_id', 'status', 'rule_name', 'create_by_name', 'duration')
        query_fields = ('create_on_gt', 'create_on_lt', 'finish_time_gt', 'finish_time_lt')


# ---FraudDetection

# +++ AlertRule

valid_op = validate.OneOf(('1', '>', '<', '='))


class AlertRuleScheme(BaseModelScheme):
    # gen module  monitoring_rule
    # --- Monitoring_Rule ---
    acd_operator = Str(validate=valid_op, default='1')
    asr_operator = Str(validate=valid_op, default='1')
    pdd_operator = Str(validate=valid_op, default='1')
    profitability_operator = Str(validate=valid_op, default='1')
    revenue_operator = Str(validate=valid_op, default='1')
    acd_value = Float()
    activate = Bool()
    all_trunks = Bool()
    asr_value = Float()
    auto_unblock = Bool()
    auto_unblock_after = Int()
    block = Bool()
    exclude = Choice()
    exclude_codes = List(Int())
    ex_code_name_id = Int()
    min_attempt = Int()
    monitor_ingress = Bool()
    monitor_by = Choice()
    name = Str(validate=validate.Regexp(NAME_REGEXP))
    pdd_value = Float()
    profitability_value = Float()
    revenue_value = Float()
    run_day_of_week = Choice()
    run_every = Int()
    run_on = Int()
    run_every_minute = Int()
    sample_period = Int()
    schedule = Choice()
    scope = Choice()
    send_email = Bool()
    trunks = List(Int())
    trunk_type = Choice()
    from_mail_id = Int()
    html_content = Str()
    include = Choice()
    include_codes = List(Int())
    in_code_name_id = Int()
    send_to = Choice()
    subject = Str()
    sdp_type = Choice()
    sdp_sign = Choice()
    sdp_value = Int()

    @pre_load
    def check(self, data):
        if 'include' in data:
            if data['include'] == 'Specific Codes':
                if not 'include_codes' in data or len(data['include_codes']) == 0:
                    raise ValidationError({'include_codes': ['must exists when Specific Codes choosen']})
            if data['include'] == 'Specific Code Names':
                if not 'in_code_name_id' in data or not data['in_code_name_id']:
                    raise ValidationError({'in_code_name_id': ['must exists when Specific Code Names choosen']})
        if 'exclude' in data:
            if data['exclude'] == 'Specific Codes':
                if not 'exclude_codes' in data or len(data['exclude_codes']) == 0:
                    raise ValidationError({'exclude_codes': ['must exists when Specific Codes choosen']})
            if data['exclude'] == 'Specific Code Names':
                if not 'ex_code_name_id' in data or not data['ex_code_name_id']:
                    raise ValidationError({'ex_code_name_id': ['must exists when Specific Code Names choosen']})
        return data

    class Meta:
        model = model.AlertRule
        fields = (
            'acd_operator', 'acd_value', 'activate', 'all_trunks', 'asr_operator', 'asr_value', 'auto_unblock',
            'auto_unblock_after', 'block', 'exclude', 'exclude_codes', 'ex_code_name_id', 'include', 'include_codes',
            'in_code_name_id',
            'min_attempt', 'monitor_ingress', 'name', 'pdd_operator', 'pdd_value', 'profitability_operator',
            'profitability_value', 'revenue_operator', 'revenue_value', 'run_day_of_week',
            'run_every', 'run_on', 'run_every_minute',
            'sample_period', 'schedule', 'scope', 'send_email', 'trunks',
            'from_mail_id', 'html_content', 'send_to', 'subject', 'monitor_by',
            'sdp_type', 'sdp_sign', 'sdp_value'
            # 'include_codes',
        )


class AlertRuleGetScheme(AlertRuleScheme):
    name = Str()
    active = Bool()
    update_by = Str()
    update_at = DateTime()
    last_run_time = DateTime()
    next_run_time = DateTime()

    class Meta:
        model = model.AlertRule
        fields = ('id', 'update_by', 'update_at', 'last_run_time', 'next_run_time') + AlertRuleScheme.Meta.fields
        search_fields = ('name', 'activate', 'block', 'update_at', 'update_by')
        query_fields = ('last_run_time_gt', 'last_run_time_lt', 'next_run_time_gt', 'next_run_time_lt', 'id_in')


class AlertRuleActivateScheme(AlertRuleGetScheme):
    active = Bool()
    last_run_time = DateTime()

    class Meta:
        model = model.AlertRule
        fields = ('active',)
        search_fields = ('name', 'activate', 'block', 'update_at', 'update_by')
        query_fields = ('last_run_time_gt', 'last_run_time_lt', 'next_run_time_gt', 'next_run_time_lt', 'id_in')


class AlertRuleLogGetScheme(BaseModelScheme):
    id = Int()
    alert_rules_id = Int()
    create_on = DateTime()
    finish_time = DateTime()
    limit_acd_value = Float()
    limit_asr_value = Float()
    limit_abr_value = Float()
    limit_pdd_value = Float()
    limit_profitability_value = Float()
    limit_revenue_value = Float()
    limit_asr = Str()
    limit_abr = Str()
    limit_acd = Str()
    limit_pdd = Str()
    limit_profitability = Str()
    limit_revenue = Str()
    time = DateTime()
    status = Choice()
    rule_name = Str()

    # detail = Nested('AlertRuleLogDetailGetScheme',many=True)
    class Meta:
        model = model.AlertRuleLog
        fields = ('id', 'alert_rules_id', 'create_on', 'finish_time', 'limit_acd_value', 'limit_asr_value',
                  # 'limit_abr_value','limit_abr',
                  'limit_pdd_value', 'limit_profitability_value', 'limit_revenue_value', 'limit_asr', 'limit_acd',
                  'limit_pdd', 'limit_profitability', 'limit_revenue', 'time', 'status', 'rule_name')
        search_fields = ('status', 'rule_name', 'alert_rules_id')
        query_fields = ('finish_time_gt', 'finish_time_lt', 'alert_rules_id_in')


class AlertRuleLogDetailGetScheme(BaseModelScheme):
    log = Nested('AlertRuleLogGetScheme', many=False)
    status = Choice()
    rule_name = Str()
    rule_id = Str()
    trunk_name = Str()
    id = Int()
    update_at = DateTime()
    trunk_id = Int()
    code = Str()
    asr = Float()
    abr = Float()
    acd = Float()
    pdd = Float()
    profitability = Float()
    revenue = Float()
    system_email_address = Str()
    system_email_status = Bool()
    system_status = Str(validate=validate.OneOf(('No send', 'Success', 'Fail')))
    partner_email_address = Str()
    partner_email_status = Bool()
    partner_status = Str(validate=validate.OneOf(('No send', 'Success', 'Fail')))
    is_email = Bool()
    is_block = Bool()
    resource_block_id = Int()
    email_type = Int()
    time = Int()

    class Meta:
        model = model.AlertRulesLogDetail
        fields = (
            'id', 'log', 'status', 'rule_name', 'rule_id', 'trunk_name', 'update_at', 'trunk_id', 'code', 'asr',
            # 'abr',
            'acd', 'pdd', 'profitability', 'revenue', 'system_email_address', 'system_status',
            'partner_email_address', 'partner_status', 'is_email', 'is_block', 'resource_block_id',
            # 'email_type',
            'time')
        search_fields = (
            'id', 'status', 'rule_name', 'rule_id', 'trunk_name', 'trunk_id', 'system_email_address', 'system_status',
            'partner_email_address', 'partner_status', 'is_email', 'is_block', 'resource_block_id', 'log_id')
        query_fields = ('update_at_gt', 'update_at_lt', 'time_gt', 'time_lt',)


# --- AlertRule


# +++ BalanceDailyResetTask
class BalanceDailyResetTaskScheme(BaseModelScheme):
    start_time = DateTime()
    # client_id = Int()
    reset_actual_balance = Bool()
    reset_mutual_balance = Bool()
    new_balance = Decimal()
    reset_balance = Bool()

    class Meta:
        model = model.BalanceDailyResetTask
        fields = (  # 'client_id',
            'start_time', 'new_balance', 'reset_actual_balance', 'reset_mutual_balance', 'reset_balance')


class BalanceDailyResetTaskGetScheme(BaseModelScheme):
    id = Int()
    client_id = Int()
    reset_actual_balance = Bool()
    reset_mutual_balance = Bool()
    # new_balance = Decimal()
    start_time = DateTime()
    finished_time = DateTime()
    status = Choice()
    create_by = Str()
    create_time = DateTime()
    old_amount = Float()
    new_amount = Float()
    client_name = Str()
    reset_balance = Int()

    class Meta:
        model = model.BalanceDailyResetTask
        fields = ('id', 'client_id', 'new_balance', 'reset_actual_balance', 'reset_mutual_balance', 'create_by',
                  'start_time', 'finished_time', 'status', 'create_time', 'old_amount', 'new_amount', 'client_name',
                  'reset_balance')
        search_fields = (
            'id', 'start_time', 'create_by', 'status', 'client_id', 'create_by', 'create_time', 'old_amount',
            'new_amount')
        query_fields = ('start_time_gt', 'start_time_lt', 'finished_time_gt', 'finished_time_lt', 'create_time_gt',
                        'create_time_lt')


# +++ BalanceDailyResetTask

# +++
class PaymentGatewayHistoryScheme(BaseModelScheme):
    client_id = Int()
    client_name = Str()
    amount = Decimal()
    type = Choice()
    paid_on = DateTime()
    entered_on = DateTime()
    paypal_id = Str()
    paypal_fee = Decimal()
    actual_received = Decimal()
    strip_id = Str()
    strip_transaction_id = Str()
    paypal_transaction_id = Str()
    # ---
    status = Choice()
    return_code = Str()
    error = Str()
    response = Str()
    # ---
    cardnumber = Str()
    cardexpmonth = Str()
    cardexpyear = Str()

    class Meta:
        model = model.PaymentGatewayHistory
        fields = ('client_name', 'amount', 'type',
                  'paid_on', 'entered_on', 'paypal_id', 'paypal_fee',
                  'actual_received', 'strip_id', 'strip_transaction_id', 'paypal_transaction_id',
                  'status', 'return_code', 'error', 'response', 'cardnumber', 'cardexpmonth', 'cardexpyear'
                  )


class PaymentGatewayHistoryGetScheme(PaymentGatewayHistoryScheme):
    id = Int()
    client_id = Int()
    status = Choice()
    date_start = DateTime()
    date_end = DateTime()
    client_name = Str()

    class Meta:
        model = model.PaymentGatewayHistory
        fields = ('id', 'client_id', 'client_name', 'amount', 'type',
                  'paid_on', 'entered_on', 'paypal_id', 'paypal_fee',
                  'actual_received', 'strip_id', 'strip_transaction_id', 'paypal_transaction_id',
                  'status', 'return_code', 'error', 'response', 'cardnumber', 'cardexpmonth', 'cardexpyear')
        search_fields = ('client_id', 'client_name',
                         'type', 'paypal_id', 'paypal_transaction_id', 'cardnumber', 'status')
        query_fields = ('paid_on_gt', 'paid_on_lt')


class PaymentCheckoutScheme(Schema):
    # user_id = Int()
    amount = Float()
    base_url = Str()
    invoice_id = Int()


class PaymentGatewayHistoryClientGetScheme(PaymentGatewayHistoryScheme):
    id = Int()
    client_id = Int()
    status = Choice()

    class Meta:
        model = model.PaymentGatewayHistory
        fields = ('id', 'client_id', 'client_name', 'amount', 'type',
                  'paid_on', 'entered_on', 'paypal_id', 'paypal_fee',
                  'actual_received', 'strip_id', 'strip_transaction_id', 'paypal_transaction_id',
                  'status', 'return_code', 'error', 'response')
        search_fields = ('type', 'paypal_id',
                         'paypal_transaction_id', 'client_id', 'status', 'client_name')
        query_fields = ('paid_on_gt', 'paid_on_lt')


# ---

class CreditLogScheme(BaseModelScheme):
    id = Int()
    modified_by = Str()
    modified_from = Decimal()
    modified_to = Decimal()
    modified_on = DateTime()
    carrier_name = String()

    class Meta:
        model = model.CreditLog
        fields = ('id', 'modified_by', 'modified_from', 'modified_to', 'modified_on', 'carrier_name')
        search_fields = ('id', 'modified_by', 'carrier_name')
        query_fields = ('modified_on_gt', 'modified_on_lt')


# ---BalanceLog
class BalanceLogScheme(BaseModelScheme):
    id = Int()
    client_id = Int()
    balance = Decimal()
    date = Str()
    client_name = String()

    class Meta:
        model = model.BalanceLog
        fields = ('id', 'client_id', 'balance', 'date', 'client_name')
        search_fields = ('id', 'client_id', 'client_name')
        query_fields = ('date_gt', 'date_lt', 'balance_gt', 'balance_lt')


# ---BalanceLog

# --- HasUser
class HasUserGetScheme(BaseModelScheme):
    has_user = Bool()

    class Meta:
        model = model.User
        fields = ()


# --- HasUser

# +++ AuthToken
class AuthTokenScheme(BaseModelScheme):
    id = Int()
    token = Str()
    user_name = Str()
    start_time = DateTime()
    expired_on = DateTime()
    is_disabled = Bool()

    class Meta:
        model = model.AuthToken
        search_fields = ('id', 'start_time', 'expired_on')


class AuthTokenDisableScheme(BaseModelScheme):
    token = Str()

    class Meta:
        model = model.AuthToken


# --- AuthToken

# +++ ImportExportLogsScheme
class ImportExportLogsExportScheme(BaseModelScheme):
    id = Int()
    user_name = Str()
    status = Choice()
    obj = Str()
    download_time = DateTime()
    finished_time = DateTime()
    file_name = Str()

    class Meta:
        model = model.ImportExportLogs
        fields = ('id', 'user_name', 'status', 'obj', 'file_name', 'download_time', 'finished_time', 'rate_table_id',
                  'rate_table_name')
        search_fields = ('id', 'status', 'obj', 'user_id')
        query_fields = ('download_time_gt', 'download_time_lt', 'finished_time_gt', 'finished_time_lt')

    @pre_dump
    def parse_rate_table_id(self, data):
        rate_table_id = None
        rate_table_name = None
        if hasattr(data, "ext_attributes") and data.ext_attributes:
            try:
                ext_attributes = json.loads(data.ext_attributes)
                rate_table_id = next(
                    (item['value'] for item in ext_attributes.get('where', [])
                    if item.get('field') == 'rate_table_id'),
                    None
                )
                if rate_table_id:
                    try:
                        rate_table_id = int(rate_table_id)
                        rate_table = get_db().session.query(model.RateTable.name).filter_by(rate_table_id=rate_table_id).first()
                        if rate_table:
                            rate_table_name = rate_table.name
                    except Exception as e:
                        log.warning(f"Failed to get rate_table_name: {str(e)}")
            except Exception as e:
                log.warning(f"Failed to parse ext_attributes JSON: {str(e)}")

        data.rate_table_id = rate_table_id
        data.rate_table_name = rate_table_name
        return data


class ImportExportLogsImportScheme(BaseModelScheme):
    id = Int()
    user_name = Str()
    status = Choice()
    obj = Str()
    upload_time = DateTime()
    finished_time = DateTime()
    file_name = Str()
    records_succ = Int()
    records_fail = Int()
    records_dup = Int()
    method = Str()
    upload_file = Str()
    error_file_path = Str()
    db_error_file_path = Str()
    code_deck_name = Str()

    class Meta:
        model = model.ImportExportLogs
        fields = ('id', 'user_name', 'status', 'obj', 'file_name', 'records_succ', 'records_fail', 'records_dup',
                  'method', 'upload_time', 'upload_file', 'error_file_path', 'db_error_file_path', 'finished_time',
                  'code_deck_name', 'total')
        search_fields = ('id', 'status', 'obj', 'user_id', 'file_name', 'is_public')
        query_fields = ('upload_time_gt', 'upload_time_lt', 'finished_time_gt', 'finished_time_lt')


class FilterScheme(Schema):
    field = Str()
    op = Str(validate=validate.OneOf(('=', '!=', '>', '<', '>=', '<=', 'like', '@>', 'in', 'between')))
    value = Str()


class ExportQueryScheme(Schema):
    select = List(Str())
    where = Nested(FilterScheme, many=True)
    or_where = Nested(FilterScheme, many=True)
    format = Str(validate=validate.OneOf(('csv', 'xml', 'json', 'xls')))
    header = Str(validate=validate.OneOf(['no', 'yes']))
    header_text = Str()
    footer_text = Str()


class ExportCreateScheme(BaseModelScheme):
    entity = Str(attribute='obj')
    query = Nested(ExportQueryScheme, attribute='_query', many=False)
    is_public = Bool()

    class Meta:
        model = model.ImportExportLogs
        fields = ('entity', 'query', 'is_public')


class RateCreateScheme(BaseModelScheme):
    rate_table_id = Int(validate=[lambda value: _valid('RateTable', 'rate_table_id', value)])
    is_effective_now = Bool()
    headers = Str(validate=[validate.Length(max=2048)])
    header_text = Str()
    footer_text = Str()
    effective_at = Str()

    class Meta:
        model = model.ImportExportLogs
        fields = ('rate_table_id', 'headers', 'is_effective_now', 'header_text', 'footer_text', 'effective_at')


class ExportGetScheme(BaseModelScheme):
    class Meta:
        model = model.ImportExportLogs
        fields = ('status',)


class ImportQueryScheme(Schema):
    field_map = Dict()
    format = Str(validate=validate.OneOf(('csv', 'xml', 'json', 'png', 'jpg')))


import falcon_rest


class ObjectCreatedWithErrorsScheme(falcon_rest.schemes.ObjectCreated):
    errors = Integer(allow_none=True)


class ImportCreateScheme(BaseModelScheme):
    entity = Str(attribute='obj')
    query = Dict()

    class Meta:
        model = model.ImportExportLogs
        fields = ('entity', 'query')


class ImportGetScheme(BaseModelScheme):
    class Meta:
        model = model.ImportExportLogs
        fields = ('status',)


# ---ImportExportLogsScheme

# ---InvoiceCdrLog
class InvoiceCdrLogScheme(BaseModelScheme):
    id = Int()
    invoice_number = Decimal()
    start_time = DateTime()
    end_time = DateTime()
    client_name = String()

    class Meta:
        model = model.InvoiceCdrLog
        fields = ('id', 'invoice_number', 'start_time', 'end_time', 'client_name')
        search_fields = ('id', 'client_name', 'invoice_number')
        query_fields = ('start_time_gt', 'start_time_lt', 'end_time_gt', 'end_time_lt')


# ---InvoiceCdrLog

# ---InvoiceLog
class InvoiceLogScheme(BaseModelScheme):
    id = Int()
    status = Choice()
    start_time = DateTime()
    end_time = DateTime()
    invoice_due_date = Str(attribute='total_due_date')
    cnt = Int()
    amount = Decimal()
    # total_count = Int()
    client_name = String()
    related_total = Decimal()
    related_count = Int(attribute='total_count')
    progress = Str()
    invoices = Nested('InvoiceGetScheme', many=True)

    class Meta:
        model = model.InvoiceLog
        fields = ('id', 'status', 'start_time', 'end_time',
                  'cnt', 'client_name',
                  # 'total', 'total_count',
                  'related_count',
                  'amount', 'progress', 'invoices',
                  'invoice_due_date'
                  )
        search_fields = ('id', 'status')
        query_fields = ('start_time_gt', 'start_time_lt', 'end_time_gt', 'end_time_lt')


# ---InvoiceLog

# +++IpModifLog
class IpModifLogScheme(BaseModelScheme):
    id = Int()
    modify = Int()
    detail = Str()
    update_at = DateTime()
    update_by = Str()
    trunk_id = Str()
    email = Emails()
    old = Str()
    new = Str()

    class Meta:
        model = model.IpModifLog
        fields = ('id', 'modify', 'detail', 'update_at', 'update_by', 'trunk_id', 'carrier_name',
                  'old', 'new')
        search_fields = ('id', 'trunk_id', 'old', 'new', 'update_by')
        query_fields = ('update_at_gt', 'update_at_lt')


# ---IpModifLog

# +++ModifLog
class ModifLogScheme(BaseModelScheme):
    id = Int()
    time = DateTime()
    module = Str()
    type = Choice()
    name = Str()
    detail = Str()
    rollback = Str()
    rollback_msg = Str()
    rollback_flg = Int()
    rollback_extra_info = Str()

    class Meta:
        model = model.ModifLog
        fields = ('id', 'time', 'detail', 'type', 'name', 'rollback', 'rollback_extra_info', 'module')
        search_fields = ('id', 'time', 'detail', 'type', 'name', 'module')
        query_fields = ('time_gt', 'time_lt')


# ---ModifLog

# +++LicenseModificationLog
class LicenseModificationLogScheme(BaseModelScheme):
    id = Int()
    modify_on = DateTime()
    modify_by = Str()
    old_value = Int()
    new_value = Int()
    type = Choice()
    switch_name = Str()

    class Meta:
        model = model.LicenseModificationLog
        fields = ('id', 'modify_on', 'modify_by', 'old_value', 'new_value', 'type', 'switch_name',)
        search_fields = ('id', 'modify_by', 'old_value', 'new_value', 'type', 'switch_name')
        query_fields = ('modify_on_gt', 'modify_on_lt')


# ---LicenseModificationLog

# +++OrigLog
class OrigLogScheme(BaseModelScheme):
    log_id = Int()
    update_on = DateTime()
    update_by = Str()
    module = Str()
    type = Choice()
    detail = Str()

    class Meta:
        model = model.OrigLog
        fields = ('log_id', 'update_on', 'update_by', 'module', 'type', 'detail')
        search_fields = ('log_id', 'update_by', 'module', 'type', 'detail')
        query_fields = ('update_on_gt', 'update_on_lt')


# ---OrigLog


# region +++RateImportTask+++
class RateImportTaskScheme(BaseModelScheme):
    id = Int()
    operator_user = Str(validate=[validate.Length(max=40)])
    import_file_path = Str(validate=[validate.Length(max=256)])
    orig_rate_file = Str(validate=[validate.Length(max=256)])
    format_rate_file = Str(validate=[validate.Length(max=256)])
    import_log_path = Str(validate=[validate.Length(max=256)])
    rate_table_id = Int(validate=[lambda value: _valid('RateTable', 'rate_table_id', value)])
    code_deck_id = Int(validate=[lambda value: _valid('CodeDeck', 'code_deck_id', value)])
    code_name_provider = Choice()
    import_rate_date_format = Str(
        validate=[validate.Length(max=40), validate.OneOf(model.RateImportTask.IMPORT_RATE_DATE_FORMAT.values())])
    # end_date_all_records_time = Str(validate=valid_date)
    # end_date_other_code_time = Str(validate=valid_date)
    # redup_in_rate_table_action = Choice(attribute='redup_in_rate_rable_action')
    redup_in_rate_table_action = Choice()
    redup_in_file_action = Choice()
    status = Choice()
    progress = Str(validate=[validate.Length(max=1024)])
    expense_detail = Str(validate=[validate.Length(max=100)])
    start_time = DateTime()
    end_time = DateTime()
    create_time = DateTime()
    progress_parsed = Dict()
    auto_ij_rate = Str(validate=validate.OneOf(['None', 'Inter Rate', 'Intra Rate', 'Max (Inter, Intra)']))

    # effective_date = Str(validate=valid_date)

    class Meta:
        model = model.RateImportTask

        fields = (
            'operator_user', 'import_file_path', 'orig_rate_file', 'format_rate_file', 'import_log_path',
            'rate_table_id',
            'code_deck_id', 'code_name_provider', 'import_rate_date_format', 'end_date_all_records_time',
            'end_date_other_code_time', 'redup_in_rate_table_action', 'redup_in_file_action', 'status', 'progress',
            'expense_detail', 'start_time', 'end_time', 'create_time')


class RateImportTaskSchemeGet(RateImportTaskScheme):
    orig_download_link = Str()
    fmt_download_link = Str()
    download_link = Str()
    error_download_link = Str()
    rate_table_name = Str()

    class Meta:
        model = model.RateImportTask
        fields = ('id',
                  'operator_user', 'import_file_path', 'orig_rate_file', 'format_rate_file', 'import_log_path',
                  'rate_table_id', 'rate_table_name', 'download_link', 'orig_download_link', 'fmt_download_link',
                  'error_download_link', 'code_deck_id', 'code_name_provider', 'import_rate_date_format',
                  'end_date_all_records_time', 'end_date_other_code_time', 'redup_in_rate_table_action',
                  'redup_in_file_action', 'status', 'progress',
                  'expense_detail', 'start_time', 'end_time', 'create_time', 'progress_parsed')
        search_fields = (
            'id', 'operator_user', 'import_file_path', 'orig_rate_file', 'format_rate_file', 'import_log_path',
            'rate_table_id', 'rate_table_name',
            'code_deck_id', 'code_name_provider', 'import_rate_date_format',
            'redup_in_rate_table_action',
            'redup_in_file_action', 'status', 'progress', 'expense_detail',)
        query_fields = ('end_date_all_records_time_gt', 'end_date_all_records_time_lt', 'end_date_other_code_time_gt',
                        'end_date_other_code_time_lt', 'start_time_gt', 'start_time_lt', 'end_time_gt', 'end_time_lt',
                        'create_time_gt', 'create_time_lt',)


class RateImportTaskSchemeModify(RateImportTaskScheme):
    class Meta:
        model = model.RateImportTask
        fields = (
            'rate_table_id', 'code_deck_id', 'code_name_provider', 'import_rate_date_format',
            'end_date_all_records_time',
            'end_date_other_code_time', 'redup_in_rate_table_action', 'redup_in_file_action')


# endregion ---RateImportTask---

# region ---RateAnalysisTask---


class RateAnalysisTaskScheme(BaseModelScheme):
    id = Int()
    operator_user = Str()
    status = Int()
    progress = Int()
    start_time = DateTime()
    end_time = DateTime()
    create_time = DateTime()
    expense_detail = Str()

    class Meta:
        model = model.RateAnalysisTask
        fields = ('operator_user', 'status', 'progress', 'expense_detail', 'start_time', 'end_time', 'create_time')


class RateAnalysisTaskSchemeGet(RateAnalysisTaskScheme):
    download_link = Str()
    error_download_link = Str()

    class Meta:
        model = model.RateAnalysisTask
        fields = ('task_id', 'operator_user', 'download_link', 'error_download_link', 'status', 'progress',
                 'expense_detail', 'start_time', 'end_time', 'create_time')
        search_fields = ('task_id', 'operator_user', 'status', 'progress', 'expense_detail')
        query_fields = ('start_time_gt', 'start_time_lt', 'end_time_gt', 'end_time_lt',
                       'create_time_gt', 'create_time_lt')


# endregion ---RateAnalysisTask---

# region ---LcrTask---

class LcrTaskScheme(BaseModelScheme):
    task_id = Int()
    target_rate_table_id = Int()
    egress_id = List(Int())
    lcr_rate_table_id = List(Int())
    created_on = DateTime()
    started_on = DateTime()
    finished_on = DateTime()
    result_path = Str()
    effective_date = Str()
    target_rate_table_name = Str()

    class Meta:
        model = model.LcrTask
        fields = ('task_id', 'target_rate_table_id', 'egress_id', 'lcr_rate_table_id',
                 'created_on', 'started_on', 'finished_on', 'result_path', 'effective_date',
                 'target_rate_table_name')
        search_fields = ('task_id', 'target_rate_table_id', 'result_path')
        query_fields = ('created_on_gt', 'created_on_lt', 'started_on_gt', 'started_on_lt',
                       'finished_on_gt', 'finished_on_lt')


class LcrTaskSchemeGet(LcrTaskScheme):
    class Meta:
        model = model.LcrTask
        fields = ('task_id', 'target_rate_table_id', 'egress_id', 'lcr_rate_table_id',
                 'created_on', 'started_on', 'finished_on', 'result_path', 'effective_date',
                 'target_rate_table_name')


class LcrTaskCreateScheme(BaseModelScheme):
    task_id = Int()
    target_rate_table_id = Int(required=True)
    egress_id = List(Int())
    lcr_rate_table_id = List(Int(), required=True)
    effective_date = Str()

    class Meta:
        model = model.LcrTask
        fields = ('target_rate_table_id', 'egress_id', 'lcr_rate_table_id', 'effective_date')

# endregion ---LcrTask---


# +++RateUploadTask

_FIELD_REGEXP = r'^[0-9,A-Z]+$'


class RateUploadTaskInfoFieldNamesScheme(Schema):
    Code = Str(default='Code', validate=validate.Length(0, 32), required=True)
    Rate = Str(default='Rate', validate=validate.Length(0, 32), required=True)
    Inter_rate = Str(validate=validate.Length(0, 32))
    Intra_rate = Str(validate=validate.Length(0, 32))
    Local_rate = Str(validate=validate.Length(0, 32))
    Effective_date = Str(validate=validate.Length(0, 32))
    Min_time = Str(validate=validate.Length(0, 32))
    Interval = Str(validate=validate.Length(0, 32))
    Code_name = Str(validate=validate.Length(0, 32))
    Country = Str(validate=validate.Length(0, 32))

    @pre_load
    def check_numbers(self, data):
        if type(data) != type({}):
            raise ValidationError('wrong "info" type. must be dictionary')
        if data['Code'].isdigit():
            val = data.values()
            for i in val:
                if not i.isdigit():
                    raise ValidationError('Wrong field numeration! {}'.format(val))
        return data


class RateUploadTaskInfoInnerScheme(Schema):
    code = Str(allow_none=True, validate=[validate.Regexp(DIGITS_REGEXP), validate.Length(0, 32)], default=None)
    min_time = Int(validate=validate.Range(1, 60000))
    interval = Int(validate=validate.Range(1, 60000))


class RateUploadTaskInfoScheme(Schema):
    start_from = Int(validate=validate.Range(0, 128))
    effective_date_default = Str(allow_none=False, validate=[validate.Regexp(DATE_REGEXP)])
    prefix_default = Str(allow_none=True, validate=[validate.Regexp(PREFIX_REGEXP), validate.Length(1, 32)])
    fields = Nested(RateUploadTaskInfoFieldNamesScheme, allow_none=True, required=False)
    time_intervals = Nested(RateUploadTaskInfoInnerScheme, many=True)


class RateUploadTaskUploadScheme(BaseModelScheme):
    id = Int()

    class Meta:
        model = model.RateUploadTask
        fields = ('upload_orig_file',)


class RateUploadTaskScheme(BaseModelScheme):
    id = Int()
    operator_user = Str()
    upload_file_path = Str()
    upload_orig_file = Str()
    upload_format_file = Str()
    result_file_path = Str()
    rate_table_id = Int(validate=lambda value: _valid('RateTable', 'rate_table_id', value), allow_none=False,
                        required=True)
    rate_table_code_deck_id = Int(validate=lambda value: _valid('CodeDeck', 'code_deck_id', value))
    rate_date_format = Str(validate=validate.OneOf(model.RateUploadTask.DATE_FORMAT.values()))
    rate_end_date = Str(required=True, allow_none=False, validate=validate.Regexp(DATE_REGEXP))
    method = Choice()
    code_deck_flag = Choice()
    use_ocn_lata_code = Choice()
    status = Choice()
    progress = Str()
    expense_detail = Str()
    start_date_time = DateTime()
    end_date_time = DateTime()
    upload_date_time = DateTime()
    start_time = DateTime()
    rate_table_name = Str()
    # default_info = Str()
    info = Nested(RateUploadTaskInfoScheme, allow_none=False, required=True)
    all_rate_end_date = DateTime()
    code_name_match = Bool()
    type = Str()

    class Meta:
        model = model.RateUploadTask
        fields = ('rate_table_id', 'method', 'rate_table_code_deck_id', 'rate_date_format',
                  'code_deck_flag', 'use_ocn_lata_code', 'rate_end_date', 'all_rate_end_date', 'code_name_match',
                  'info',
                  'code_name_match', 'start_from')


class RateUploadTaskGetScheme(RateUploadTaskScheme):
    id = Int()
    operator_user = Str()
    upload_file_path = Str()
    upload_orig_file = Str()
    upload_format_file = Str()
    result_file_path = Str()
    rate_table_id = Int()
    rate_table_code_deck_id = Int()
    rate_date_format = Str()
    rate_end_date = Str()
    method = Choice()
    code_deck_flag = Int()
    use_ocn_lata_code = Int()
    status = Choice()
    pre_status = Choice()
    progress = Str()
    progress_parsed = Dict()
    expense_detail = Str()
    start_date_time = DateTime()
    end_date_time = DateTime()
    default_info = Str()
    upload_date_time = DateTime()
    rate_table_name = Str()
    type = Str(validate=validate.OneOf(('manual', 'auto')))

    class Meta:
        model = model.RateUploadTask
        fields = ('id', 'rate_table_id', 'rate_table_name', 'upload_orig_file', 'operator_user', 'status', 'method',
                  'start_date_time', 'end_date_time', 'upload_date_time', 'progress', 'progress_parsed',
                  'expense_detail', 'rate_end_date',
                  'all_rate_end_date', 'default_info', 'type', 'pre_status')
        search_fields = (
            'id', 'rate_table_id', 'operator_user', 'status', 'method', 'rate_table_name', 'type', 'pre_status')
        query_fields = ('start_date_time_gt', 'start_date_time_lt', 'upload_date_time_gt', 'upload_date_time_lt',
                        'end_date_time_gt', 'end_date_time_lt',
                        'id_in', 'pre_status')


class RateUploadTaskApproveScheme(RateUploadTaskScheme):
    id = Int()

    class Meta:
        model = model.RateUploadTask


# fields = ()


class RateUploadTaskGetHeadersScheme(RateUploadTaskScheme):
    fields = List(Str())

    class Meta:
        model = model.RateUploadTask
        fields = ('fields', 'start_from')


class RateAutoImportRuleScheme(BaseModelScheme):
    name = Str(validate=[validate.Regexp(NAME_REGEXP), validate.Length(min=2, max=100)])
    active = Bool()
    egress_trunk_id = Int(allow_none=False, validate=lambda value: _valid('Resource', 'resource_id', value))
    from_email = Email()
    mail_server_ip = Str(validate=validate.Regexp(IP_REGEXP))
    subject_keyword = Str(validate=[validate.Length(min=2, max=100)])
    info = Nested(RateUploadTaskInfoScheme, allow_none=False, required=True)
    rate_date_format = Str(validate=validate.OneOf(model.RateUploadTask.DATE_FORMAT.values()))
    rate_table_code_deck_id = Int(validate=lambda value: _valid('CodeDeck', 'code_deck_id', value))
    code_deck_flag = Choice()
    use_ocn_lata_code = Choice()
    reduplicate_rate_action = Choice()
    is_link = Boolean()
    read_effective_date_from = Choice()

    class Meta:
        model = model.RateAutoImportRule
        fields = ('name', 'active', 'egress_trunk_id', 'from_email', 'subject_keyword', 'info', 'rate_date_format',
                  'rate_table_code_deck_id', 'code_deck_flag', 'use_ocn_lata_code', 'reduplicate_rate_action',
                  'is_link', 'mail_server_ip',
                  'read_effective_date_from')


class RateAutoImportRuleGetScheme(RateAutoImportRuleScheme):
    id = Int()
    create_by = Str()
    create_on = DateTime()
    update_by = Str()
    update_on = DateTime()
    egress_trunk_name = Str()
    rate_table_name = Str()

    class Meta:
        model = model.RateAutoImportRule
        fields = ('id', 'create_by', 'create_on', 'update_by', 'update_on',
                  'egress_trunk_name', 'rate_table_name') + RateAutoImportRuleScheme.Meta.fields
        search_fields = (
            'id', 'name', 'active', 'egress_trunk_id', 'from_email', 'subject_keyword', 'create_by', 'update_by',
            'egress_trunk_name', 'rate_table_name')
        query_fields = ('create_on_gt', 'create_on_lt', 'update_by_gt', 'update_by_lt')


class RateAutoImportRuleLogGetScheme(BaseModelScheme):
    id = Int()
    rule_id = Int()
    status = Choice()
    start = DateTime()
    finish = DateTime()
    received_from_ip = Str()
    error = Str()
    task_id = Int()
    task = Nested('RateUploadTaskGetScheme')
    rule = Nested('RateAutoImportRuleGetScheme')

    class Meta:
        model = model.RateAutoImportRuleLog
        fields = ('id', 'rule_id', 'status', 'start', 'finish', 'received_from_ip', 'error', 'task_id', 'task')
        search_fields = (
            'id', 'rule_id', 'status', 'rule_name', 'egress_trunk_id', 'rate_table_id', 'from_email', 'subject',
            'received_from_ip', 'error', 'task_id')
        query_fields = ('start_gt', 'finish_gt', 'start_lt', 'finish_lt',)


class RateUploadValuesGetScheme(BaseModelScheme):
    Code = Str()
    Rate = Decimal()
    Inter_rate = Decimal()
    Intra_rate = Decimal()
    Local_rate = Decimal()
    Effective_date = Str()
    Min_time = Int()
    Interval = Int()
    Code_name = Str()
    Country = Str()

    class Meta:
        model = model.RateUploadValues
        fields = (
            'Code', 'Rate', 'Inter_rate', 'Intra_rate', 'Local_rate', 'Effective_date', 'Min_time', 'Interval',
            'Code_name',
            'Country')
        search_fields = ('Code', 'Effective_date', 'Min_time', 'Interval', 'Code_name', 'Country')
        query_fields = ('Rate_gt', 'Rate_lt', 'Inter_rate_gt', 'Inter_rate_lt',
                        'Intra_rate_gt', 'Intra_rate_lt', 'Local_rate_gt', 'Local_rate_lt')


# ---RateUploadTask


# region +++RateUploadTemplate+++
class RateUploadTemplateScheme(BaseModelScheme):
    id = Int()
    name = Str(validate=[validate.Length(max=100), validate.Regexp(NAME_REGEXP)])
    auto_ij_rate = Choice()
    rate_table_id = Int()
    code_deck_id = Int(allow_none=True)
    code_name_provider = Choice()
    import_rate_date_format = Str(validate=[validate.Length(max=40)])
    effective_date = DateTime(allow_none=True)
    end_date_all_records_time = DateTime(allow_none=True)
    end_date_other_code_time = DateTime(allow_none=True)
    redup_in_rate_table_action = Choice()
    redup_in_file_action = Choice()
    create_on = DateTime()
    create_by = Str(validate=[validate.Length(max=40)])
    update_on = DateTime()
    update_by = Str(validate=[validate.Length(max=100)])
    default_interval = Int(allow_none=True)
    default_min_time = Int(allow_none=True)
    replace_fields = Str()
    replace_values = Str()
    ignore_fields = Str()
    map_header = Str()

    class Meta:
        model = model.RateUploadTemplate
        fields = (
            'name', 'auto_ij_rate', 'rate_table_id', 'code_deck_id', 'code_name_provider', 'import_rate_date_format',
            'effective_date', 'end_date_all_records_time', 'end_date_other_code_time', 'redup_in_rate_table_action',
            'redup_in_file_action', 'default_interval', 'default_min_time', 'replace_fields', 'replace_values',
            'ignore_fields', 'map_header')


class RateUploadTemplateSchemeGet(RateUploadTemplateScheme):
    class Meta:
        model = model.RateUploadTemplate
        fields = (
            'id', 'name', 'auto_ij_rate', 'rate_table_id', 'code_deck_id', 'code_name_provider',
            'import_rate_date_format', 'effective_date', 'end_date_all_records_time', 'end_date_other_code_time',
            'redup_in_rate_table_action', 'redup_in_file_action', 'create_on', 'create_by', 'update_on', 'update_by',
            'default_interval', 'default_min_time', 'replace_fields', 'replace_values', 'ignore_fields', 'map_header')
        search_fields = (
            'id', 'name', 'auto_ij_rate', 'rate_table_id', 'code_deck_id', 'code_name_provider',
            'import_rate_date_format', 'redup_in_rate_table_action', 'redup_in_file_action', 'create_by', 'update_by',
            'default_interval', 'default_min_time', 'replace_fields', 'replace_values', 'ignore_fields')
        query_fields = (
            'effective_date_gt', 'effective_date_lt', 'end_date_all_records_time_gt', 'end_date_all_records_time_lt',
            'end_date_other_code_time_gt', 'end_date_other_code_time_lt', 'create_on_gt', 'create_on_lt',
            'update_on_gt',
            'update_on_lt',)


class RateUploadTemplateSchemeModify(RateUploadTemplateScheme):
    pass


# endregion ---RateUploadTemplate---

class RateDownloadLogScheme(BaseModelScheme):
    __tablename__ = 'rate_download_log'

    id = Int()
    resource_id = Int()
    file_path = Str()
    download_time = DateTime()
    download_ip = Str(validate=validate.Regexp(IP_REGEXP))
    download_username = Str()
    log_detail_id = Str()
    trunk_name = Str()
    client_name = Str()
    client_id = Int()
    sent_on = DateTime()
    send_to = Emails()
    rate_table_name = Str()

    class Meta:
        model = model.RateDownloadLog
        fields = ('id', 'resource_id', 'file_path', 'download_time', 'resource_id', 'log_detail_id',
                  'trunk_name', 'client_name', 'client_id', 'send_to', 'sent_on', 'rate_table_name',
                  'download_ip', 'download_username')
        search_fields = ('id', 'resource_id', 'log_detail_id', 'trunk_name', 'client_name', 'client_id', 'send_to',
                         'rate_table_name', 'download_ip', 'download_username')
        query_fields = ('download_time_gt', 'download_time_lt', 'sent_on_gt', 'sent_on_lt')


# +++RateMassEditLog
class RateMassEditLogScheme(BaseModelScheme):
    id = Int()
    action_time = DateTime()
    client_id = Int()
    action_type = Choice()
    down_file_path = Str()
    rate_table_id = Int()
    action_rate_rows = Int()
    client_name = Str()
    rate_table_name = Str()
    edited_by = Str()

    class Meta:
        model = model.RateMassEditLog
        fields = (
            'id', 'action_time', 'client_id', 'action_type', 'down_file_path', 'rate_table_id', 'action_rate_rows',
            'client_name', 'rate_table_name', 'edited_by')
        search_fields = (
            'id', 'client_id', 'action_type', 'rate_table_id', 'edited_by', 'client_name', 'rate_table_name')
        query_fields = ('action_time_gt', 'action_time_lt')


# ---RateMassEditLog

# ++++RetrievePasswordLog
class RetrievePasswordLogScheme(BaseModelScheme):
    id = Int()
    username = Str()
    operation_time = DateTime()
    email_addresses = Str()
    modify_time = DateTime()
    status = Choice()
    request_ip = Str()
    confirm_ip = Str()

    class Meta:
        model = model.RetrievePasswordLog
        fields = (
            'id', 'username', 'operation_time', 'email_addresses', 'modify_time', 'status', 'request_ip', 'confirm_ip')
        search_fields = ('id', 'username', 'email_addresses', 'status', 'request_ip', 'confirm_ip')
        query_fields = ('modify_time_gt', 'modify_time_lt', 'operation_time_gt', 'operation_time_lt')


# ---RetrievePasswordLog
# +++ScheduledReportLog
class ScheduledReportLogScheme(BaseModelScheme):
    id = Int()
    email_to = Emails()
    report_name = Str(validate=validate.Regexp(NAME_REGEXP))
    attachment_path = Str()
    execute_time = DateTime()

    class Meta:
        model = model.ScheduledReportLog
        fields = ('id', 'email_to', 'report_name', 'attachment_path', 'execute_time')
        search_fields = ('id', 'email_to', 'report_name')
        query_fields = ('execute_time_gt', 'execute_time_lt')


# ---ScheduledReportLog

# +++SchedulerLog
class SchedulerLogScheme(BaseModelScheme):
    id = Int()
    script_name = Str()
    start_time = DateTime()
    end_time = DateTime()

    class Meta:
        model = model.SchedulerLog
        fields = ('id', 'script_name', 'start_time', 'end_time')
        search_fields = ('id', 'script_name')
        query_fields = ('start_time_gt', 'start_time_lt', 'end_time_gt', 'end_time_lt')


# ---SchedulerLog

# +++SipRegistration
class SipRegistrationScheme(BaseModelScheme):
    __tablename__ = 'sip_registrations'

    id = Int()
    username = Str(validate=validate.Regexp(NAME_REGEXP))
    network_ip = Str(validate=validate.Regexp(IP_REGEXP))
    network_port = Int(validate=validate.Range(1000, 65535))
    status = Choice()
    expires = Int()
    contact = Str()
    time = DateTime()
    uptime = DateTime()
    trunk_name = Str(dump_only=True)
    carrier_name = Str(dump_only=True)
    is_spam = Bool(dump_only=True)

    class Meta:
        model = model.SipRegistration
        fields = ('id', 'username', 'network_ip', 'network_port', 'status', 'expires', 'contact', 'time',
                  'trunk_name', 'carrier_name', 'is_spam')
        search_fields = ('id', 'username', 'network_ip', 'network_port', 'status', 'expires', 'contact',
                         'trunk_name', 'carrier_name', 'is_spam')
        query_fields = ('uptime_gt', 'uptime_lt', 'time_gt', 'time_lt')


# ---SipRegistration


# +++InvoiceEmail
class InvoiceEmailScheme(BaseModelScheme):
    id = Int()
    invoice_no = Str()
    mail_content = Str()
    send_time = Str()
    pdf_file = Str()
    send_address = Str()
    mail_sub = Str()

    class Meta:
        model = model.InvoiceEmail
        fields = ('id', 'invoice_no', 'mail_content', 'send_time', 'pdf_file', 'send_address', 'mail_sub')


# ---InvoiceEmail

# +++++CarrierTemplate

class CarrierTemplateLowBalConfigScheme(BaseModelScheme):
    actual_notify_balance = Float(allow_none=True)
    percentage_notify_balance = Float(allow_none=True)
    value_type = Choice(allow_none=True)
    send_time_type = Choice(allow_none=True)

    class Meta:
        model = model.CarrierTemplateLowBalConfig
        # exclude = ('client_id', 'client')
        fields = (
            'is_notify',
            'value_type',
            'actual_notify_balance',
            'percentage_notify_balance',
            'send_time_type',
            'daily_send_time',
            'duplicate_days',
            'send_to',
            'duplicate_send_days',
            'disable_trunks_days')


class CarrierTemplateFromClientScheme(BaseModelScheme):
    template_name = Str(validate=[validate.Length(1, 100), validate.Regexp(NAME_REGEXP)])

    class Meta:
        model = model.CarrierTemplate
        fields = ('template_name',)


class CarrierTemplateScheme(BaseModelScheme):
    auto_daily_balance_recipient = Choice()
    auto_invoice_type = Choice()
    auto_send_invoice = Bool(attribute='auto_invoicing')
    auto_send_zone = Str(allow_none=True)
    auto_summary_group_by = Choice()
    breakdown_by_rate_table = Choice()
    cdr_format = Choice()
    credit_limit = Decimal(validate=validate.Range(-10000000.0, 10000000.0), allow_none=True, default=0.0)
    currency = Str(attribute='currency_name', validate=validate_currency, allow_none=True)
    decimal = Int(attribute='decimal_place')
    daily_balance_summary = Bool()
    daily_cdr_delivery = Bool()
    daily_usage_group_by = Choice()
    daily_usage_summary = Bool()
    format = Choice()
    include_short_call_charge = Bool(attribute='is_short_duration_call_surcharge_detail')
    include_tax = Bool()
    inlcude_cdr_in_email = Bool(attribute='attach_cdrs_list')
    invoice_use_balance_type = Choice()
    invoice_zone = Str(validate=validate.Length(max=10))
    is_short_duration_call_surcharge_detail = Bool()
    is_show_total_trunk = Bool()
    unlimited_credit = Bool(allow_none=True)  # attribute='unlimited_credit')
    low_balance_config = Nested('CarrierTemplateLowBalConfigScheme')
    # low_balance_notification_time_cycle ###
    mode = Choice()
    billing_mode = Choice()
    non_zero_invoice_only = Bool()
    notify_client_balance = Decimal(validate=validate.Range(0, 1000000.0))
    payment_received_notice = Bool()
    payment_term = Str(attribute='pt_name', validate=validate_payment_term)
    profit_margin = Float(allow_none=True, validate=validate.Range(0, 999))
    profit_type = Choice()
    # rate_delivery_email = Email(attribute='rate_email')
    rate_value = Choice()
    scc_below = Decimal(attribute='scc_bellow')
    scc_charge = Decimal(validate=validate.Range(0, 1000000.0))
    tax = Decimal(validate=validate.Range(0, 1000000.0))
    scc_type = Choice()
    send_invoice_as_link = Bool()
    show_account_summary = Bool(attribute='is_invoice_account_summary')
    show_code_name_summary = Bool(attribute='is_show_code_name')
    show_code_summary = Bool(attribute='is_show_code_100')
    show_country_summary = Bool(attribute='is_show_country')
    show_daily_usage = Bool(attribute='is_show_daily_usage')
    show_detail_by_trunk = Bool(attribute='is_show_detail_trunk')
    show_jurisdiction_detail = Bool(attribute='invoice_jurisdictional_detail')
    show_payment_summary = Bool(attribute='invoice_include_payment')
    show_trunk_summary = Bool(attribute='is_show_detail_trunk')
    template_name = Str(validate=[validate.Length(min=1, max=100), validate.Regexp(USERNAME_REGEXP),
                                  lambda value: _valid_unique('CarrierTemplate', 'template_name', value)])
    test_credit = Decimal(validate=validate.Range(0, 10000000.0), allow_none=True)
    time_zone = Str(attribute='daily_balance_send_time_zone', validate=validate.Length(max=10))
    # usage_detail_fields = Str()
    usage_fields = List(Str(validate=lambda value: _valid('DailyCdrField', 'field', value)))

    class Meta:
        model = model.CarrierTemplate
        fields = (
            'auto_daily_balance_recipient',
            'auto_send_invoice',
            'auto_send_zone',
            'auto_summary_group_by',
            'auto_summary_hour',
            'auto_summary_not_zero',
            'auto_summary_period',
            # 'billing_email',
            'call_limit',
            'cdr_format',
            # 'company_name',
            'cps_limit',
            'credit_limit',
            'currency',
            'daily_cdr_generation',
            'daily_cdr_generation_zone',
            'decimal',
            # 'email',
            'format',
            'include_short_call_charge',
            'include_tax',
            'inlcude_cdr_in_email',
            'invoice_zone',
            'is_auto_balance',
            'is_auto_summary',
            'is_auto_summary',
            'is_breakdown_by_rate_table',
            'is_daily_balance_notification',
            # 'is_panel_accountsummary',
            # 'is_panel_balance',
            # 'is_panel_cdrslist',
            # 'is_panel_invoices',
            # 'is_panel_onlinepayment',
            # 'is_panel_paymenthistory',
            # 'is_panel_products',
            # 'is_panel_ratetable',
            # 'is_panel_sippacket',
            # 'is_panel_summaryreport',
            # 'is_panel_trunks',
            'is_send_trunk_update',
            'is_short_duration_call_surcharge_detail',
            'is_show_code_name',
            'is_show_country',
            'is_show_daily_usage',
            'is_show_total_trunk',
            'unlimited_credit',
            'low_balance_config',
            'mode',
            # 'noc_email',
            'non_zero_invoice_only',
            'notify_client_balance_type',
            'payment_received_notice',
            'payment_term',
            'profit_margin',
            'profit_type',
            # 'rate_delivery_email',
            'rate_value',
            'scc_below',
            'scc_charge',
            'scc_percent',
            'scc_type',
            'send_invoice_as_link',
            'show_account_summary',
            'show_code_name_summary',
            'show_code_summary',
            'show_country_summary',
            'show_daily_usage',
            'show_detail_by_trunk',
            'show_jurisdiction_detail',
            'show_payment_summary',
            'show_trunk_summary',
            'tax',
            'template_name',
            'test_credit',
            'time_zone',
            # 'usage_detail_fields',
            'usage_fields',
        )
        exclude = ('id', 'create_by', 'create_on', 'update_on')


class CarrierTemplateModifyScheme(CarrierTemplateScheme):
    template_name = Str(
        validate=[validate.Length(max=100), validate.Regexp(NAME_REGEXP)])


class CarrierTemplateGetScheme(CarrierTemplateScheme):
    used_by = Int()

    class Meta:
        model = model.CarrierTemplate
        fields = ('id', 'create_by', 'create_on', 'update_on', 'used_by') + CarrierTemplateScheme.Meta.fields
        search_fields = ('id', 'template_name', 'create_by', 'used_by')
        query_fields = ('create_on_gte', 'create_on_lt', 'update_on_gte', 'update_on_lt', 'id_in')


# +++++CarrierTemplate

# +++++TrunkTemplate

class TrunkTemplateIpScheme(Schema):
    type = Str(validate=validate.OneOf(('IP Address', 'FQDN')))
    host = Str(validate=lambda value: validate.Regexp(IP_REGEXP)(value))
    port = Int(validate=validate.Range(1000, 65535))


class IngressTrunkTemplateScheme(BaseModelScheme):
    name = Str(
        validate=[lambda value: _valid_unique('IngressTrunkTemplate', 'name', value), validate.Regexp(NAME_REGEXP)])
    billing_method = Choice()
    direction = Choice()
    trunk_type = Choice()
    trunk_type2 = Choice()
    profit_margin = Decimal(allow_none=True, required=False)
    profit_type = Choice()
    res_strategy = Choice()
    media_type = Choice()
    # lnp_dipping = Choice()
    ring_timeout = Int(allow_none=True, required=False, validate=validate.Range(0, 60))
    rpid_screen = Choice()
    rpid_party = Choice()
    rpid_privacy = Choice()
    rpid_id_type = Choice()
    rpid = Choice()
    us_other = Choice()
    us_route = Choice()
    canada_other = Choice()
    canada_route = Choice()
    intl_route = Choice()
    min_duration = Int(allow_none=True, required=False)
    amount_per_port = Decimal(validate=validate.Range(0, 1000.0))
    ip = Nested(TrunkTemplateIpScheme, many=True)
    failover_rule = Nested(TrunkTemplateFailoverScheme, many=True)
    rate_table_id = Int(allow_none=True, validate=lambda value: _valid('RateTable', 'rate_table_id', value))
    rate_rounding = Choice()
    t38 = Bool()
    codecs_str = Str(attribute='_codecs_str')

    div = Choice()
    oli = Choice()
    paid = Choice()
    pci = Choice()
    priv = Choice()

    class Meta:
        model = model.IngressTrunkTemplate
        exclude = ('resource_template_id', 'create_by', 'create_on', 'update_on', 'hosts', 'delay_bye_second',
                   'direction', 'resource', 'failover')


class IngressTrunkTemplateModifyScheme(IngressTrunkTemplateScheme):
    name = Str(
        validate=[validate.Regexp(NAME_REGEXP)])
    rate_profile = Int(allow_none=True)
    oli = Str(allow_none=True)
    priv = Str(allow_none=True)
    div = Str(allow_none=True)
    pci = Str(allow_none=True)
    paid = Str(allow_none=True)


class IngressTrunkTemplateFromResourceScheme(BaseModelScheme):
    name = Str(
        validate=[lambda value: _valid_unique('IngressTrunkTemplate', 'name', value), validate.Regexp(USERNAME_REGEXP),
                  validate.Length(min=1)])

    class Meta:
        model = model.IngressTrunkTemplate
        fields = ('name',)


class IngressTrunkTemplateGetScheme(IngressTrunkTemplateScheme):
    used_by = Int()
    min_duration = Int(allow_none=True, required=False)

    class Meta:
        model = model.IngressTrunkTemplate
        exclude = ('hosts', 'failover', 'delay_bye_second')
        search_fields = ('resource_template_id', 'create_by', 'name')
        query_fields = (
            'create_on_gte', 'create_on_lt', 'update_on_gte', 'update_on_lt', 'used_by_gt', 'resource_template_id_in')


# ==
class EgressTrunkTemplateScheme(BaseModelScheme):
    name = Str(
        validate=[lambda value: _valid_unique('EgressTrunkTemplate', 'name', value), validate.Regexp(NAME_REGEXP)])
    billing_method = Choice()
    direction = Choice()
    trunk_type = Choice()
    trunk_type2 = Choice()
    profit_margin = Decimal(allow_none=True, required=False)
    profit_type = Choice()
    res_strategy = Choice()
    media_type = Choice()
    # lnp_dipping = Choice()
    ring_timeout = Int(allow_none=True, required=False, validate=validate.Range(0, 60))
    rpid_screen = Choice()
    rpid_party = Choice()
    rpid_privacy = Choice()
    rpid_id_type = Choice()
    rpid = Choice()
    us_other = Choice()
    us_route = Choice()
    canada_other = Choice()
    canada_route = Choice()
    intl_route = Choice()
    min_duration = Int(allow_none=True, required=False)
    amount_per_port = Decimal(validate=validate.Range(0, 1000.0))
    ip = Nested(TrunkTemplateIpScheme, many=True)
    failover_rule = Nested(TrunkTemplateFailoverScheme, many=True)
    rate_table_id = Int(allow_none=True, validate=lambda value: _valid('RateTable', 'rate_table_id', value))
    rate_rounding = Choice()
    t38 = Bool()

    div = Choice()
    oli = Choice()
    paid = Choice()
    pci = Choice()
    priv = Choice()

    class Meta:
        model = model.EgressTrunkTemplate
        exclude = ('resource_template_id', 'create_by', 'create_on', 'update_on', 'hosts', 'delay_bye_second',
                   'direction', 'resource', 'failover')


class EgressTrunkTemplateModifyScheme(EgressTrunkTemplateScheme):
    name = Str(validate=[validate.Regexp(NAME_REGEXP)])


class EgressTrunkTemplateFromResourceScheme(BaseModelScheme):
    name = Str(
        validate=[lambda value: _valid_unique('EgressTrunkTemplate', 'name', value), validate.Regexp(USERNAME_REGEXP),
                  validate.Length(min=1)])

    class Meta:
        model = model.EgressTrunkTemplate
        fields = ('name',)


class EgressTrunkTemplateGetScheme(EgressTrunkTemplateScheme):
    min_duration = Int(allow_none=True, required=False)

    class Meta:
        model = model.EgressTrunkTemplate
        exclude = ('hosts', 'failover', 'delay_bye_second')
        search_fields = ('resource_template_id', 'create_by', 'name')
        query_fields = (
            'create_on_gte', 'create_on_lt', 'update_on_gte', 'update_on_lt', 'used_by_gt', 'resource_template_id_in')


# +++++TrunkTemplate

# +++ DnlApiBaseModel
class DidBillingPlanScheme(BaseModelScheme):
    name = Str(validate=[lambda value: _valid_unique('DidBillingPlan', 'name', value), validate.Regexp(NAME_REGEXP),
                         validate.Length(1, 100)], required=True)
    setup_fee = Float(validate=validate.Range(0, 10000.0))
    monthly_fee = Decimal(validate=validate.Range(0, 10000.0))
    rate_per_min = Float(validate=validate.Range(0, 10000.0))
    # fee_per_port = Float(validate=validate.Range(0, 10000.0))
    payphone_subcharge = Str(validate=validate.Length(max=100))
    rate_table_id = Int(allow_none=True, validate=lambda value: _valid('RateTable', 'rate_table_id', value))
    rate_type = Choice()
    pay_type = Choice()
    price_type = Choice()
    min_time = Int(validate=validate.Range(1, 10000))
    interval = Int(validate=validate.Range(1, 10000))
    rate_per_sms_sent = Float(validate=validate.Range(0, 10000.0))
    rate_per_sms_received = Float(validate=validate.Range(0, 10000.0))
    client_id = Int(allow_none=True, validate=lambda value: _valid('Client', 'client_id', value))

    class Meta:
        model = model.DidBillingPlan
        fields = ('name', 'setup_fee', 'monthly_fee', 'rate_per_min',
                  'payphone_subcharge', 'rate_table_id', 'rate_type', 'pay_type', 'price_type', 'min_time', 'interval',
                  'rate_per_sms_sent', 'rate_per_sms_received', 'client_id')


class DidBillingPlanModifyScheme(DidBillingPlanScheme):
    name = Str(validate=[validate.Length(0, 100), validate.Regexp(NAME_REGEXP)])

    class Meta:
        model = model.DidBillingPlan
        fields = ('name', 'setup_fee', 'monthly_fee', 'rate_per_min',
                  'payphone_subcharge', 'rate_type', 'pay_type', 'price_type', 'rate_table_id', 'min_time', 'interval',
                  'rate_per_sms_sent', 'rate_per_sms_received', 'client_id')


class DidBillingPlanModifyRateParamsScheme(BaseModelScheme):
    min_time = Int(allow_none=False, validate=validate.Range(1, 10000))
    interval = Int(allow_none=False, validate=validate.Range(1, 10000))

    class Meta:
        model = model.DidBillingPlan
        fields = ('min_time', 'interval')


class DidBillingPlanGetScheme(DidBillingPlanScheme):
    id = Int()
    rate_table_name = Str()
    min_time = Int()
    interval = Int()

    class Meta:
        model = model.DidBillingPlan
        fields = ('id', 'name', 'setup_fee', 'monthly_fee', 'rate_per_min', 'payphone_subcharge',
                  'rate_type', 'pay_type', 'price_type', 'rate_table_id', 'rate_table_name', 'update_by', 'update_at',
                  'min_time', 'interval', 'rate_per_sms_sent', 'rate_per_sms_received', 'client_id')
        search_fields = (
            'name', 'rate_table_id', 'rate_type', 'pay_type', 'price_type', 'payphone_subcharge', 'update_by',
            'min_time', 'interval', 'id')
        query_fields = ('rate_per_min_gt', 'rate_per_min_lt', 'monthly_fee_gt', 'monthly_fee_lt',
                        'setup_fee_gt', 'setup_fee_lt', 'id_in', 'update_at_gt',
                        'update_at_lt', 'id')


# --- DnlApiBaseModel

# ---CdrDownloadTask
class SendCdrDirectScheme(BaseModelScheme):
    # name = Str(validate=[validate.Length(max=200),lambda value:_valid_unique('SendRateTemplate','name',value)])
    subject = Str(validate=validate.Length(max=200))
    html_content = Str()
    sender_id = Int(validate=lambda value: _valid('MailSender', 'id', value))
    cc_mail = Emails()

    class Meta:
        model = model.SendCdrDirect
        fields = ('subject', 'html_content', 'sender_id', 'cc_mail')


class RtCallbackScheme(BaseModelScheme):
    user_name = Str()

    class Meta:
        model = model.CdrDownloadTask
        fields = ('user_name',)


class CdrDownloadTaskScheme(BaseModelScheme):
    # request_id = Str(validate=[validate.Length(max=16),lambda value:_valid_unique('CdrDownloadTask','request_id',value)])
    request_id = Str(validate=[validate.Length(max=16), lambda value: _valid('CdrAsyncTask', 'request_id', value)])
    email_to = Emails()
    method = Str(validate=validate.OneOf(['link', 'attachment']))
    client_id = Int(allow_none=True, validate=lambda value: _valid('Client', 'client_id', value))
    direct = Nested('SendCdrDirectScheme', allow_none=True)

    class Meta:
        model = model.CdrDownloadTask
        fields = ('request_id', 'email_to', 'method', 'client_id', 'direct')


class CdrDownloadTaskGetScheme(CdrDownloadTaskScheme):
    cached = Bool()
    code = Int()
    count = Int()
    download_link = Str(attribute='fmt_download_link')
    end_time = Int()
    expiration_time = Int()
    job_end_time = Int()
    job_id = Int()
    job_start_time = Int()
    msg = Str()
    progress = Str()
    size = Int()
    start_time = Int()
    status = Str()
    email_to = Str()
    orig_file = Str()
    mail_status = Str()
    method = Str()
    client_id = Int(allow_none=True, validate=lambda value: _valid('Client', 'client_id', value))

    class Meta:
        model = model.CdrDownloadTask
        fields = ('request_id', 'cached', 'code', 'count', 'download_link', 'end_time', 'expiration_time',
                  'job_end_time', 'job_id', 'job_start_time', 'msg', 'progress', 'size', 'start_time', 'status',
                  'mail_status', 'orig_file', 'email_to', 'method', 'client_id', 'direct')
        search_fields = ('request_id', 'cached', 'code', 'count', 'end_time', 'expiration_time',
                         'job_end_time', 'job_id', 'job_start_time', 'msg', 'progress', 'size', 'start_time', 'status',
                         'mail_status', 'orig_file', 'email_to', 'method', 'client_id')


# ---CdrDownloadTask
# --- RerateCdrTask ---
class _RerateCdrTask_TrunksScheme(falcon_rest.schemes.Schema):
    trunk_id = Int()
    update_rate = Bool()
    rate_table_id = Int()
    effective_date = DateTime()

    class Meta:
        model = model.RerateCdrTaskTrunkRec
        fields = ('trunk_id', 'update_rate', 'rate_table_id', 'effective_date')


class RerateCdrTaskScheme(BaseModelScheme):
    from_date = DateTime()
    to_date = DateTime()
    gmt = Int()
    update_us_jurisdiction = Choice()
    # rerate_type = Choice()
    # egress_trunks = List(Dict())
    egress_trunks = Nested('_RerateCdrTask_TrunksScheme', many=True)
    # ingress_trunks = List(Dict())
    ingress_trunks = Nested('_RerateCdrTask_TrunksScheme', many=True)

    class Meta:
        model = model.RerateCdrTask
        fields = ('from_date', 'to_date', 'gmt', 'egress_trunks', 'ingress_trunks', 'update_us_jurisdiction')


class RerateCdrTaskGetScheme(RerateCdrTaskScheme):
    id = Int()
    status = Choice()
    progress = Str()
    download_file = Str()
    err_code = Int()

    class Meta:
        model = model.RerateCdrTask
        fields = ('id', 'status', 'progress', 'download_file', 'err_code') + RerateCdrTaskScheme.Meta.fields
        search_fields = ('id', 'status', 'progress', 'err_code')
        query_fields = ('from_date_gt', 'from_date_lt', 'to_date_gt', 'to_date_lt')


# --- RerateCdrTask ---

# +++ RerateReportExecLog
class RerateReportExecLogScheme(BaseModelScheme):
    status = Choice()

    class Meta:
        model = model.RerateReportExecLog
        # fields = ('status',)
        exclude = ('task',)
        search_fields = ('task_id', 'status', 'exec_type', 'create_by',)
        query_fields = ('finish_time_gt', 'finish_time_lt', 'create_on_gt', 'create_on_lt',)


# --- RerateReportExecLog

# +++ RerateCdrDownloadLog
class RerateCdrDownloadLogScheme(BaseModelScheme):
    status = Choice()

    class Meta:
        model = model.RerateCdrDownloadLog
        # fields = ('status',)
        exclude = ('task',)
        search_fields = ('task_id', 'status', 'create_by')
        query_fields = ('finished_time_gt', 'finished_time_lt', 'create_on_gt', 'create_on_lt')


# --- RerateCdrDownloadLog

# +++ VersionInformation
class VersionInformationGetScheme(BaseModelScheme):
    id = Int()

    class Meta:
        model = model.VersionInformation
        fields = ('id', 'program_name', 'major_ver', 'minor_ver', 'build_date', 'start_time', 'licensed_cps',
                  'licensed_channel', 'expires', 'switch_name', 'serial_number')
        search_fields = ('id', 'program_name', 'switch_name')


class VersionInformationInnerGetScheme(BaseModelScheme):
    id = Int()

    class Meta:
        model = model.VersionInformation
        fields = ('major_ver', 'minor_ver', 'build_date', 'start_time', 'licensed_cps',
                  'licensed_channel', 'expires', 'serial_number')
        search_fields = ('id', 'program_name', 'switch_name')


# --- VersionInformation

# +++PcapQuery
class PcapQueryEmptyScheme(BaseModelScheme):
    class Meta:
        model = model.PcapQuery
        fields = ('foo',)


class PcapQueryScheme(BaseModelScheme):
    query_key = Str()
    callid = Str(validate=validate.Length(max=100))
    switch_ip = Str(validate=validate.Regexp(IP_REGEXP))
    start = DateTime()
    # finish = DateTime()
    duration = Int(validate=validate.Range(0, 86400))  # 60*60*24))
    queued_time = DateTime()
    complete_time = DateTime()
    status = Choice()
    msg = Str(validate=validate.Length(max=100))
    public_url = Str(validate=validate.Length(max=255))

    class Meta:
        model = model.PcapQuery
        fields = ('callid', 'switch_ip', 'start', 'duration')


class PcapQueryGetScheme(PcapQueryScheme):
    requested_by = Str()

    class Meta:
        model = model.PcapQuery
        fields = (
            'query_key', 'callid', 'switch_ip', 'start', 'finish', 'queued_time', 'complete_time', 'status', 'msg',
            'public_url', 'requested_by')
        search_fields = ('query_key', 'callid', 'switch_ip', 'status', 'msg', 'public_url', 'requested_by')
        query_fields = (
            'start_gt', 'start_lt', 'finish_gt', 'finish_lt', 'queued_time_gt', 'queued_time_lt', 'complete_time_gt',
            'complete_time_lt')


# ---PcapQuery

def validate_pcap_query_filter(value):
    tokens = value.split(',')
    for tok in tokens:
        tok = tok.strip()
        if '=' not in tok:
            raise ValidationError('Bad filter syntax not found  "=" in token {}'.format(tok))
        expr = tok.split('=')
        if len(expr) != 2:
            raise ValidationError('Bad filter syntax bad use = in expression {}'.format(expr))
        allowed_fields = ('call-id', 'caller', 'callee', 'ip.src', 'ip.dst')
        if expr[0].strip() not in allowed_fields:
            raise ValidationError(
                'Bad filter syntax field name {} must be one of {}'.format(expr[0], str(allowed_fields)))
        if ' ' in expr[1].strip():
            raise ValidationError('Bad filter syntax field value {} has spaces'.format(expr[1]))
        if expr[0].strip() in ('ip.src', 'ip.dst'):
            import re
            if not re.match(IP_REGEXP, expr[1].strip()):
                raise ValidationError('Bad filter syntax field value {} must be valid ip address'.format(expr[1]))

    return True


# +++PcapQueryTask+++
class PcapQueryTaskScheme(BaseModelScheme):
    id = Int()
    operator_user = Str(validate=[validate.Length(max=40)])
    pcap_start_time = DateTime(allow_none=False)
    pcap_end_time = DateTime(allow_none=False)
    filter = Str(allow_none=False, validate=[validate.Length(max=1024), validate_pcap_query_filter])
    result_file = Str(validate=[validate.Length(max=1024)])
    status = Choice()
    progress = Str(validate=[validate.Length(max=1024)])
    create_time = DateTime()
    proc_start_time = DateTime()
    proc_end_time = DateTime()
    comment_number = Str()

    class Meta:
        model = model.PcapQueryTask
        fields = ('pcap_start_time', 'pcap_end_time', 'filter', 'comment_number')

class ShowStirShakenGetScheme(Schema):
    time = fields.Float(required=True)
    from_ip = fields.String(required=True)
    to_ip = fields.String(required=True)
    ani = fields.String(allow_none=True)
    dnis = fields.String(allow_none=True)
    identity = fields.String(allow_none=True)


class PcapQueryTaskSchemeGet(PcapQueryTaskScheme):
    download_link = Str()
    cloud_shark_open_url = Str()
    sip_shark_open_url = Str()
    is_identity = Bool()

    class Meta:
        model = model.PcapQueryTask
        fields = (
            'id', 'operator_user', 'pcap_start_time', 'pcap_end_time', 'filter', 'result_file', 'status', 'progress',
            'create_time', 'proc_start_time', 'proc_end_time', 'download_link', 'cloud_shark_open_url',
            'comment_number', 'is_identity')
        search_fields = ('id', 'operator_user', 'filter', 'result_file', 'status', 'progress', 'comment_number')
        query_fields = (
            'pcap_start_time_gt', 'pcap_start_time_lt', 'pcap_end_time_gt', 'pcap_end_time_lt', 'create_time_gt',
            'create_time_lt', 'proc_start_time_gt', 'proc_start_time_lt', 'proc_end_time_gt', 'proc_end_time_lt',)


class PcapQueryTaskSchemeModify(PcapQueryTaskScheme):
    pass


class PcapQueryTaskEmailScheme(BaseModelScheme):
    # request_id = Str(validate=[validate.Length(max=16),lambda value:_valid_unique('CdrDownloadTask','request_id',value)])
    # request_id = Str(validate=[validate.Length(max=16), lambda value: _valid('CdrAsyncTask', 'request_id', value)])
    email = Emails()
    client_id = Int(allow_none=True, validate=lambda value: _valid('Client', 'client_id', value))

    # direct = Nested('SendCdrDirectScheme', allow_none=True)

    class Meta:
        model = model.PcapQueryTask
        fields = ('email', 'client_id')


# ---PcapQueryTask---


class StorageConfigScheme(BaseModelScheme):
    conf_type = Str(validate=validate.OneOf(('cdr', 'pcap')))
    storage = Str(validate=validate.OneOf(('local', 'ftp', 'sftp', 'gcs')))
    remote_server_port = Int(validate=validate.Range(1, 65535))
    ftp_port = Int(validate=validate.Range(1, 65535))
    storage_days = Int(validate=validate.Range(1, 65535))
    storage_days_local = Int(validate=validate.Range(1, 65535))
    remote_server_ip = Str(validate=validate.Regexp(IP_REGEXP))
    storage_path = Str(validate=validate.Length(max=1000))
    username = Str(validate=validate.Length(max=100))
    password = Str(validate=validate.Length(max=100))
    ftp_ip = Str(validate=validate.Regexp(IP_REGEXP))
    google_drive_key = Str(validate=validate.Length(max=1000))
    google_drive_name = Str(validate=validate.Length(max=1000))
    active = Bool()

    class Meta:
        model = model.StorageConfig
        fields = ('conf_type', 'storage', 'remote_server_port', 'ftp_port', 'storage_days', 'storage_days_local',
                  'remote_server_ip',
                  'storage_path', 'username', 'password', 'ftp_ip', 'google_drive_key', 'google_drive_name', 'active')


class StorageConfigModifyScheme(StorageConfigScheme):
    class Meta:
        model = model.StorageConfig
        fields = ('storage', 'remote_server_port', 'ftp_port', 'storage_days', 'storage_days_local', 'remote_server_ip',
                  'storage_path', 'username', 'password', 'ftp_ip', 'google_drive_key', 'google_drive_name', 'active')


# +++LcrTest
class LcrTestScheme(BaseModelScheme):
    name = Str()
    egress_id = Int(allow_none=False, validate=lambda value: _valid('EgressTrunk', 'resource_id', value))
    switch_id = Int(validate=lambda value: _valid('VoipGateway', 'id', value))

    class Meta:
        model = model.LcrTest
        fields = ('name', 'egress_id', 'switch_id')


class LcrTestGetScheme(BaseModelScheme):
    name = Str()
    routing_plan_id = Int()
    static_route_id = Int()
    type = Choice()
    switch_id = Int()
    client_id = Int()
    trunk_id = Int()
    egress_id = Int()
    rate_table_id = Int()
    ip_id = Int()
    prefix_id = Int()

    class Meta:
        model = model.LcrTest
        fields = (
            'name', 'routing_plan_id', 'static_route_id', 'type', 'switch_id', 'client_id', 'trunk_id', 'egress_id',
            'rate_table_id', 'ip_id', 'prefix_id')


class LcrTestCallScheme(BaseModelScheme):
    code = Str()

    class Meta:
        model = model.LcrTest
        fields = ('code',)


# ---LcrTest

# +++Ticket
class TicketAttachmentScheme(BaseModelScheme):
    orig_file = Str()

    class Meta:
        model = model.TicketAttachment
        fields = ()


class TicketAttachmentGetScheme(BaseModelScheme):
    uuid = Str()
    orig_file = Str()
    download_link = Str()

    class Meta:
        model = model.TicketAttachment
        fields = ('uuid', 'orig_file', 'download_link')


class TicketCommentScheme(BaseModelScheme):
    url = Str()
    content = Str()
    subject = Str(validate=validate.Length(max=100))
    ticket_status = Choice()

    class Meta:
        model = model.Ticket
        fields = ('url', 'content', 'subject', 'ticket_status')


class TicketSendScheme(TicketCommentScheme):
    class Meta:
        model = model.Ticket
        fields = ()


class TicketCommentGetScheme(TicketCommentScheme):
    id = Int()
    ticket_id = Int()
    status = Choice()
    user_ip = Str(validate=validate.Regexp(IP_REGEXP))
    user_name = Str(validate=validate.Regexp(USERNAME_REGEXP))
    attachments = Nested('TicketAttachmentGetScheme', many=True)
    create_by = Str()
    create_on = DateTime()
    update_by = Str()
    update_on = DateTime()
    url = Str(attribute='_url')

    class Meta:
        model = model.Ticket
        fields = ('id', 'ticket_id', 'status', 'user_ip', 'url', 'content', 'subject',
                  'attachments', 'create_by', 'create_on', 'update_by', 'update_on') + TicketCommentScheme.Meta.fields


class TicketScheme(BaseModelScheme):
    gateway_ip = Str(validate=validate.Regexp(IP_REGEXP))
    url = Str()
    content = Str()
    subject = Str(validate=validate.Length(max=100))
    department = Choice(allow_none=False, required=True)
    ticket_type = Str(allow_none=False, required=True)

    class Meta:
        model = model.Ticket
        fields = ('gateway_ip', 'url', 'content', 'subject', 'department', 'ticket_type')


class TicketTypeScheme(BaseModelScheme):
    ticket_type = Str(validate=validate.Length(max=100))

    class Meta:
        model = model.TicketType
        fields = ('ticket_type',)


class TicketTypeGetScheme(TicketTypeScheme):
    pass


class TicketSendScheme(TicketScheme):
    class Meta:
        model = model.Ticket
        fields = ()


class TicketStatusHistoryGetScheme(BaseModelScheme):
    update_on = DateTime()
    update_by = Str()
    ticket_status = Choice()

    class Meta:
        model = model.TicketHistory
        fields = ('update_on', 'update_by', 'ticket_status')


class TicketGetScheme(TicketScheme):
    id = Int()
    status = Choice()
    ticket_status = Choice()
    ticket_type = Str()
    user_ip = Str(validate=validate.Regexp(IP_REGEXP))
    attachments = Nested('TicketAttachmentGetScheme', many=True)
    comments = Nested('TicketCommentGetScheme', many=True)
    status_history = Nested('TicketStatusHistoryGetScheme', many=True)
    department = Str()
    create_by = Str()
    create_on = DateTime()
    update_by = Str()
    update_on = DateTime()
    client_company = Str()
    client_name = Str()
    company_type = Choice()
    client_email = Str()
    url = Str(attribute='_url')

    class Meta:
        model = model.Ticket
        fields = ('id', 'status', 'ticket_status', 'user_ip', 'gateway_ip', 'user_name', 'url', 'content', 'subject',
                  'attachments', 'comments', 'create_on', 'update_by', 'update_on', 'status_history', 'ticket_type',
                  'client_company', 'client_email', 'company_type', 'client_name') + TicketScheme.Meta.fields
        search_fields = (
            'id', 'status', 'ticket_status', 'user_ip', 'gateway_ip', 'user_name', 'url', 'content', 'subject',
            'update_by', 'client_company', 'client_email', 'company_type', 'client_name')
        query_fields = ('create_on_gt', 'create_on_lt', 'update_on_gt', 'update_on_lt')


# ---Ticket


# +++AsyncFtpExport
class AsyncFtpExportScheme(BaseModelScheme):
    request_id = Str()
    job_id = Int()
    username = Str()
    status = Choice()
    err_code = Int()
    error_str = Str()
    time = DateTime()

    class Meta:
        model = model.AsyncFtpExport
        fields = ('request_id', 'job_id', 'username', 'status', 'err_code', 'error_str', 'time')
        search_fields = ('request_id', 'job_id', 'username', 'status', 'err_code', 'error_str')
        query_fields = ('time_gt', 'time_lt')


# ---AsyncFtpExport

class AllowedSendToIpScheme(BaseModelScheme):
    sip_profile_id = Int(validate=lambda value: _valid('SwitchProfile', 'id', value), allow_none=True)
    voip_gateway_name = Str()

    class Meta:
        model = model.AllowedSendToIp
        fields = ('sip_profile_id', 'voip_gateway_name')


class AllowedSendToIpGetScheme(BaseModelScheme):
    id = Int()
    sip_profile_id = Int(validate=lambda value: _valid('SwitchProfile', 'id', value))
    resource_id = Int()
    sip_profile_ip = Str()
    sip_profile_port = Int()
    voip_gateway_name = Str()
    sip_profile_name = Str()

    class Meta:
        model = model.AllowedSendToIp
        fields = ('id', 'sip_profile_id', 'resource_id', 'sip_profile_ip', 'sip_profile_port', 'voip_gateway_name',
                  'sip_profile_name')
        search_fields = (
            'id', 'sip_profile_id', 'resource_id', 'sip_profile_ip', 'sip_profile_port', 'voip_gateway_name',
            'sip_profile_name')


# region +++DidVendor+++
class DidVendorScheme(BaseModelScheme):
    id = Int()
    type = Choice()
    search_type = Choice()
    username = Str(validate=[validate.Length(max=64)])
    password = Str(validate=[validate.Length(max=64)])
    msg_api_key = Str(validate=[validate.Length(max=64)])
    sms_enabled = Bool()
    default_received_sms_url = Str()
    mrc = Str(validate=[validate.Length(max=10)])
    nrc = Str(validate=[validate.Length(max=10)])
    vendor_billing_rule_id = Integer(allow_none=True, validate=lambda value: _valid('DidBillingPlan', 'id', value))
    client_billing_rule_id = Integer(allow_none=True, validate=lambda value: _valid('DidBillingPlan', 'id', value))
    api_enabled = Bool()
    trunk_group_name = Str(validate=[validate.Length(max=128)])
    toll_free_routing_label = Str(validate=[validate.Length(max=128)])

    class Meta:
        model = model.DidVendor
        fields = (
            'type', 'search_type', 'username', 'password', 'msg_api_key', 'sms_enabled',
            'default_received_sms_url', 'vendor_billing_rule_id', 'client_billing_rule_id',
            'trunk_group_name', 'api_enabled', 'toll_free_routing_label')


class DidVendorSchemeGet(DidVendorScheme):
    vendor_billing_rule_rate_table_id = Int()

    class Meta:
        model = model.DidVendor
        fields = ('id', 'type', 'search_type', 'username', 'sms_enabled', 'vendor_billing_rule_rate_table_id',
                  'default_received_sms_url', 'vendor_billing_rule_id', 'client_billing_rule_id',
                  'trunk_group_name', 'api_enabled', 'toll_free_routing_label')
        search_fields = (
            'id', 'type', 'search_type', 'username', 'password', 'sms_enabled', 'default_received_sms_url',
            'vendor_billing_rule_id', 'client_billing_rule_id', 'vendor_billing_rule_rate_table_id',
            'trunk_group_name', 'api_enabled', 'toll_free_routing_label')
        query_fields = ()


class DidVendorSchemeModify(DidVendorScheme):
    pass


# endregion ---DidVendor---
# region +++IngressDidRepository+++
class IngressDidRepositoryScheme(BaseModelScheme):
    number = Str(validate=[validate.Length(max=255)])
    ingress_id = Int()
    egress_id = Int()
    created_time = DateTime()
    updated_time = DateTime()
    state = Str(validate=[validate.Length(max=200)])
    country = Str(validate=[validate.Length(max=200)])
    city = Str(validate=[validate.Length(max=200)])
    rate_center = Str(validate=[validate.Length(max=200)])
    status = Int()
    type = Int()
    lata = Str(validate=[validate.Length(max=100)])
    egress = Nested('ResourceMinScheme', many=False)
    ingress = Nested('ResourceMinScheme', many=False)

    class Meta:
        model = model.IngressDidRepository
        fields = (
            'ingress_id', 'egress_id', 'created_time', 'updated_time', 'state', 'country', 'city', 'rate_center',
            'status',
            'type', 'lata',)


class IngressDidRepositorySchemeGet(IngressDidRepositoryScheme):
    class Meta:
        model = model.IngressDidRepository
        fields = (
            'number', 'ingress_id', 'egress_id', 'state', 'country', 'city', 'rate_center', 'status', 'type', 'lata',
            'egress', 'ingress',)
        search_fields = (
            'number', 'ingress_id', 'egress_id', 'state', 'country', 'city', 'rate_center', 'status', 'type', 'lata',
            'egress', 'ingress',)
        query_fields = ('created_time_gt', 'created_time_lt', 'updated_time_gt', 'updated_time_lt',)


class IngressDidRepositorySchemeModify(IngressDidRepositoryScheme):
    pass


# endregion ---IngressDidRepository---
# region +++SpamTrafficIp+++
class SpamTrafficIpScheme(BaseModelScheme):
    ip = Str(validate=[validate.Length(max=30), validate.Regexp(IP_REGEXP)])
    brief = Str(validate=[validate.Length(max=100)])
    create_time = DateTime()
    netmask = Int(validate=[validate.Range(0, 32)])
    created_by = Str()
    auto_block = Bool()

    class Meta:
        model = model.SpamTrafficIp
        fields = ('ip', 'brief', 'netmask')


class SpamTrafficIpSchemeGet(SpamTrafficIpScheme):
    class Meta:
        model = model.SpamTrafficIp
        fields = ('ip', 'brief', 'netmask', 'auto_block', 'created_by', 'create_time')
        search_fields = ('ip', 'brief', 'netmask', 'auto_block', 'created_by')
        query_fields = ('create_time_gt', 'create_time_lt',)


class SpamTrafficIpSchemeModify(SpamTrafficIpScheme):
    class Meta:
        model = model.SpamTrafficIp
        fields = ('brief', 'netmask', 'ip')


class SpamTrafficIpActivateScheme(SpamTrafficIpSchemeGet):
    auto_block = Bool()

    class Meta:
        model = model.SpamTrafficIp
        fields = ('auto_block',)
        search_fields = ('ip', 'brief', 'netmask', 'auto_block', 'created_by')
        query_fields = ('create_time_gt', 'create_time_lt',)


# endregion ---SpamTrafficIp---
# region +++CleanDataManagement+++
class CleanDataManagementScheme(BaseModelScheme):
    id = Int()
    tbl_name = Str(validate=[validate.Length(max=100)])
    days_of_save_in_db = Int()
    create_time = DateTime()

    class Meta:
        model = model.CleanDataManagement
        fields = ('tbl_name', 'days_of_save_in_db')


class CleanDataManagementSchemeGet(CleanDataManagementScheme):
    class Meta:
        model = model.CleanDataManagement
        fields = ('id', 'tbl_name', 'days_of_save_in_db',)
        search_fields = ('id', 'tbl_name', 'days_of_save_in_db',)
        query_fields = ('create_time_gt', 'create_time_lt',)


class CleanDataManagementSchemeModify(CleanDataManagementScheme):
    pass


# endregion ---CleanDataManagement---
# region +++DiskManagement+++
class DiskManagementScheme(BaseModelScheme):
    id = Int()
    days_of_cdr_keep = Int()
    days_of_pcap_keep = Int()
    create_time = DateTime()

    class Meta:
        model = model.DiskManagement
        fields = ('days_of_cdr_keep', 'days_of_pcap_keep')


class ReleaseCauseScheme(BaseModelScheme):
    id = Int()
    name = Str()
    err = Str()
    err_string = Str()

    class Meta:
        model = model.ReleaseCause
        fields = ('id', 'name', 'err', 'err_string')


class EgressReleaseCauseScheme(BaseModelScheme):
    id = Int()
    name = Str()

    class Meta:
        model = model.EgressErrorString
        fields = ('id', 'name')


# endregion +++DiskManagement+++
# region +++MonitoredRule+++
class EnfourceCidScheme(BaseModelScheme):
    acd_value = Int()
    asr_value = Float()
    sdp_value = Float()
    pdd_value = Int()
    min_call_attempt = Int()
    sample_size = Int()
    unblock_after_min = Int()

    class Meta:
        model = model.MonitoredRule
        fields = (
            'acd_value', 'asr_value', 'sdp_value', 'pdd_value',
            'min_call_attempt', 'sample_size', 'unblock_after_min')


class MonitoredRuleScheme(BaseModelScheme):
    id = Int()
    rule_name = Str(validate=[validate.Length(max=100)])
    is_active = Bool()
    is_ingress_trunk = Bool()
    is_all_trunk = Bool()
    trunk_id_list = Str(validate=[validate.Length(max=200)])
    include_codes = Str(validate=[validate.Length(max=500)])
    exclude_codes = Str(validate=[validate.Length(max=500)])
    monitored_type = Choice()
    acd_action = Choice()
    acd_value = Int()
    asr_action = Choice()
    asr_value = Float()
    sdp_action = Choice()
    sdp_value = Float()
    sdp_duation = Choice()
    pdd_action = Choice()
    pdd_value = Int()
    profit_action = Choice()
    profit_value = Float()
    revenue_action = Choice()
    revenue_value = Float()
    min_call_attempt = Int()
    exe_schedule_type = Choice()
    action_type = Choice()
    specific_minutes = Int()
    daily_run_time = Str(validate=[validate.Regexp(TIME_REGEXP)])
    weekly_run_time = Str(validate=[validate.Regexp(TIME_REGEXP)])
    run_day_of_week = Choice()
    sample_size = Int()
    is_block = Bool()
    is_auto_unblock = Bool()
    unblock_after_min = Int()
    last_run_time = DateTime()
    next_run_time = DateTime()
    send_email_type = Choice()
    email_from = Int()
    email_ticket_subject = Str(validate=[validate.Length(max=200)])
    email_ticket_content = Str()
    create_time = DateTime()

    class Meta:
        model = model.MonitoredRule
        fields = (
            'rule_name', 'is_active', 'is_ingress_trunk', 'is_all_trunk', 'trunk_id_list', 'include_codes',
            'exclude_codes',
            'monitored_type', 'acd_action', 'acd_value', 'asr_action', 'asr_value', 'sdp_action', 'sdp_value',
            'sdp_duation', 'pdd_action', 'pdd_value', 'profit_action', 'profit_value', 'revenue_action',
            'revenue_value', 'action_type',
            'min_call_attempt', 'exe_schedule_type', 'specific_minutes', 'daily_run_time', 'weekly_run_time',
            'run_day_of_week', 'sample_size', 'is_block', 'is_auto_unblock', 'unblock_after_min',
            'send_email_type', 'email_from', 'email_ticket_subject', 'email_ticket_content',)


class MonitoredRuleSchemeGet(MonitoredRuleScheme):
    updated_by = Str()
    updated_at = DateTime()

    class Meta:
        model = model.MonitoredRule
        fields = ('id',
                  'rule_name', 'is_active', 'is_ingress_trunk', 'is_all_trunk', 'trunk_id_list', 'include_codes',
                  'exclude_codes',
                  'monitored_type', 'acd_action', 'acd_value', 'asr_action', 'asr_value', 'sdp_action', 'sdp_value',
                  'sdp_duation', 'pdd_action', 'pdd_value', 'profit_action', 'profit_value', 'revenue_action',
                  'revenue_value', 'action_type',
                  'min_call_attempt', 'exe_schedule_type', 'specific_minutes', 'daily_run_time', 'weekly_run_time',
                  'run_day_of_week', 'sample_size', 'is_block', 'is_auto_unblock', 'unblock_after_min', 'last_run_time',
                  'next_run_time', 'send_email_type', 'email_from', 'email_ticket_subject', 'email_ticket_content',
                  'create_time', 'updated_by', 'updated_at')
        search_fields = (
            'id', 'rule_name', 'is_active', 'is_ingress_trunk', 'is_all_trunk', 'trunk_id_list', 'include_codes',
            'exclude_codes', 'monitored_type', 'acd_action', 'acd_value', 'asr_action', 'sdp_action', 'sdp_duation',
            'pdd_action', 'pdd_value', 'profit_action', 'revenue_action', 'min_call_attempt', 'exe_schedule_type',
            'specific_minutes', 'daily_run_time', 'weekly_run_time', 'run_day_of_week', 'sample_size', 'is_block',
            'is_auto_unblock', 'unblock_after_min', 'send_email_type', 'email_from', 'email_ticket_subject',
            'email_ticket_content', 'updated_by')
        query_fields = (
            'asr_value_gt', 'asr_value_lt', 'sdp_value_gt', 'sdp_value_lt', 'profit_value_gt', 'profit_value_lt',
            'revenue_value_gt', 'revenue_value_lt', 'last_run_time_gt', 'last_run_time_lt', 'next_run_time_gt',
            'next_run_time_lt', 'create_time_gt', 'create_time_lt', 'updated_at_gt', 'updated_at_lt',)


class MonitoredRuleSchemeModify(MonitoredRuleScheme):
    pass


class MonitoredRuleActivateScheme(MonitoredRuleSchemeGet):
    is_active = Bool()
    is_block = Bool()
    last_run_time = DateTime()

    class Meta:
        model = model.MonitoredRule
        fields = ('is_active',)
        search_fields = ('rule_name', 'is_active', 'is_block')
        query_fields = ('last_run_time_gt', 'last_run_time_lt', 'next_run_time_gt', 'next_run_time_lt', 'id_in')


# endregion ---MonitoredRule---
# region +++MonitoredRuleHistory+++
class MonitoredRuleHistoryScheme(BaseModelScheme):
    id = Int()
    monitored_rule_id = Int()
    start_time = DateTime()
    end_time = DateTime()
    is_ingress_trunk = Bool()
    trunk_id = Int()
    dnis = Str(validate=[validate.Length(max=64)])
    ani = Str(validate=[validate.Length(max=64)])
    code_name = Str(validate=[validate.Length(max=100)])
    country = Str(validate=[validate.Length(max=100)])
    code = Str(validate=[validate.Length(max=64)])
    total_call = Int()
    not_zero_call = Int()
    sdp_call = Int()
    call_duration = Int()
    pdd = Int()
    ingress_cost = Float()
    egress_cost = Float()
    block_id_list = Str()
    unblock_time = DateTime()
    unblock_exe_time = DateTime()
    create_time = DateTime()

    class Meta:
        model = model.MonitoredRuleHistory
        fields = (
            'monitored_rule_id', 'start_time', 'end_time', 'is_ingress_trunk', 'trunk_id', 'dnis', 'ani', 'code_name',
            'country', 'code', 'total_call', 'not_zero_call', 'sdp_call', 'call_duration', 'pdd', 'ingress_cost',
            'egress_cost', 'block_id_list', 'unblock_time', 'unblock_exe_time', 'create_time',)


class MonitoredRuleHistorySchemeGet(MonitoredRuleHistoryScheme):
    class Meta:
        model = model.MonitoredRuleHistory
        fields = (
            'monitored_rule_id', 'start_time', 'end_time', 'is_ingress_trunk', 'trunk_id', 'dnis', 'ani', 'code_name',
            'country', 'code', 'total_call', 'not_zero_call', 'sdp_call', 'call_duration', 'pdd', 'ingress_cost',
            'egress_cost', 'block_id_list', 'unblock_time', 'unblock_exe_time', 'create_time', 'rule_name')
        search_fields = (
            'id', 'monitored_rule_id', 'is_ingress_trunk', 'trunk_id', 'dnis', 'ani', 'code_name', 'country', 'code',
            'total_call', 'not_zero_call', 'sdp_call', 'call_duration', 'pdd', 'block_id_list', 'rule_name')
        query_fields = (
            'start_time_gt', 'start_time_lt', 'end_time_gt', 'end_time_lt', 'ingress_cost_gt', 'ingress_cost_lt',
            'egress_cost_gt', 'egress_cost_lt', 'unblock_time_gt', 'unblock_time_lt', 'unblock_exe_time_gt',
            'unblock_exe_time_lt', 'create_time_gt', 'create_time_lt',)


class MonitoredRuleHistorySchemeModify(MonitoredRuleHistoryScheme):
    pass


# endregion ---MonitoredRuleHistory---
# region +++FrundDetection+++
class FrundDetectionScheme(BaseModelScheme):
    id = Int()
    rule_name = Str(validate=[validate.Length(max=100)])
    trunk_id_list = Str(validate=[validate.Length(max=100)])
    criteria_1hour_min = Float()
    criteria_1hour_revenue = Float()
    criteria_24hour_min = Float()
    criteria_24hour_revenue = Float()
    is_block = Bool()
    send_email_type = Choice()
    email_from = Int()
    email_ticket_subject = Str(validate=[validate.Length(max=200)])
    email_ticket_content = Str()
    trigger_criteria_1hour_time = DateTime()
    trigger_criteria_24hour_time = DateTime()
    create_time = DateTime()
    hourly_asr_of_trunk_is_below = Integer()
    hourly_acd_of_trunk_is_below = Integer()
    hourly_minute_of_trunk_is_above = Integer()
    hourly_cost_of_trunk_is_above = Integer()
    daily_asr_of_trunk_is_below = Integer()
    daily_acd_of_trunk_is_below = Integer()
    daily_minute_of_trunk_is_above = Integer()
    daily_cost_of_trunk_is_above = Integer()
    detection_level = Choice()

    class Meta:
        model = model.FrundDetection
        fields = ('rule_name', 'trunk_id_list', 'criteria_1hour_min', 'criteria_1hour_revenue', 'criteria_24hour_min',
                  'criteria_24hour_revenue', 'is_block', 'send_email_type', 'email_from', 'email_ticket_subject',
                  'email_ticket_content', 'trigger_criteria_1hour_time', 'trigger_criteria_24hour_time',
                  'hourly_asr_of_trunk_is_below', 'hourly_acd_of_trunk_is_below', 'hourly_minute_of_trunk_is_above',
                  'hourly_cost_of_trunk_is_above', 'daily_asr_of_trunk_is_below', 'daily_acd_of_trunk_is_below',
                  'daily_minute_of_trunk_is_above', 'daily_cost_of_trunk_is_above', 'detection_level')


class FrundDetectionSchemeGet(FrundDetectionScheme):
    class Meta:
        model = model.FrundDetection
        fields = (
            'id', 'rule_name', 'trunk_id_list', 'criteria_1hour_min', 'criteria_1hour_revenue', 'criteria_24hour_min',
            'criteria_24hour_revenue', 'is_block', 'send_email_type', 'email_from', 'email_ticket_subject',
            'email_ticket_content', 'trigger_criteria_1hour_time', 'trigger_criteria_24hour_time', 'create_time',
            'hourly_asr_of_trunk_is_below', 'hourly_acd_of_trunk_is_below', 'hourly_minute_of_trunk_is_above',
            'hourly_cost_of_trunk_is_above', 'daily_asr_of_trunk_is_below', 'daily_acd_of_trunk_is_below',
            'daily_minute_of_trunk_is_above', 'daily_cost_of_trunk_is_above', 'detection_level')
        search_fields = (
            'id', 'rule_name', 'trunk_id_list', 'is_block', 'send_email_type', 'email_from', 'email_ticket_subject',
            'email_ticket_content',)
        query_fields = (
            'criteria_1hour_min_gt', 'criteria_1hour_min_lt', 'criteria_1hour_revenue_gt', 'criteria_1hour_revenue_lt',
            'criteria_24hour_min_gt', 'criteria_24hour_min_lt', 'criteria_24hour_revenue_gt',
            'criteria_24hour_revenue_lt',
            'trigger_criteria_1hour_time_gt', 'trigger_criteria_1hour_time_lt', 'trigger_criteria_24hour_time_gt',
            'trigger_criteria_24hour_time_lt', 'create_time_gt', 'create_time_lt',)


class FrundDetectionSchemeModify(FrundDetectionScheme):
    pass


# endregion ---FrundDetection---

# region +++FrundDetectionHistory++
class FrundDetectionHistoryScheme(BaseModelScheme):
    id = Int()
    frund_detection_id = Int()
    ingress_id = Int()
    block_type = Choice()
    criteria_1hour_min = Float()
    criteria_1hour_revenue = Float()
    criteria_24hour_min = Float()
    criteria_24hour_revenue = Float()
    actual_1h_minute = Float()
    actual_24h_minute = Float()
    actual_1h_revenue = Float()
    actual_24h_revenue = Float()
    is_block = Bool()
    send_email_type = Int()
    is_send_email = Bool()
    partner_email_msg = Str()
    partner_email_status = Bool()
    partner_email = Str(validate=[validate.Length(max=200)])
    system_email_msg = Str()
    system_email_status = Bool()
    system_email = Str(validate=[validate.Length(max=200)])
    create_time = DateTime()

    class Meta:
        model = model.FrundDetectionHistory
        fields = ('frund_detection_id', 'ingress_id', 'block_type', 'criteria_1hour_min', 'criteria_1hour_revenue',
                  'criteria_24hour_min', 'criteria_24hour_revenue', 'actual_1h_minute', 'actual_24h_minute',
                  'actual_1h_revenue', 'actual_24h_revenue', 'is_block', 'send_email_type', 'is_send_email',
                  'partner_email_msg', 'partner_email_status', 'partner_email', 'system_email_msg',
                  'system_email_status', 'system_email', 'create_time',)


class FrundDetectionHistorySchemeGet(FrundDetectionHistoryScheme):
    ingress_name = Str()

    class Meta:
        model = model.FrundDetectionHistory
        fields = ('frund_detection_id', 'ingress_id', 'block_type', 'criteria_1hour_min', 'criteria_1hour_revenue',
                  'criteria_24hour_min', 'criteria_24hour_revenue', 'actual_1h_minute', 'actual_24h_minute',
                  'actual_1h_revenue', 'actual_24h_revenue', 'is_block', 'send_email_type', 'is_send_email',
                  'partner_email_msg', 'partner_email_status', 'partner_email', 'system_email_msg',
                  'system_email_status', 'system_email', 'create_time', 'ingress_name')
        search_fields = (
            'id', 'frund_detection_id', 'ingress_id', 'block_type', 'is_block', 'send_email_type', 'is_send_email',
            'partner_email_msg', 'partner_email_status', 'partner_email', 'system_email_msg', 'system_email_status',
            'system_email', 'ingress_name')
        query_fields = (
            'criteria_1hour_min_gt', 'criteria_1hour_min_lt', 'criteria_1hour_revenue_gt', 'criteria_1hour_revenue_lt',
            'criteria_24hour_min_gt', 'criteria_24hour_min_lt', 'criteria_24hour_revenue_gt',
            'criteria_24hour_revenue_lt',
            'actual_1h_minute_gt', 'actual_1h_minute_lt', 'actual_24h_minute_gt', 'actual_24h_minute_lt',
            'actual_1h_revenue_gt', 'actual_1h_revenue_lt', 'actual_24h_revenue_gt', 'actual_24h_revenue_lt',
            'create_time_gt', 'create_time_lt',)


class FrundDetectionHistorySchemeModify(FrundDetectionHistoryScheme):
    pass


# endregion ---FrundDetectionHistory---
# region +++DidChargeDetail+++
class DidChargeDetailScheme(BaseModelScheme):
    did_mrc = Float()
    total = Float()
    id = Int()
    params = Str(validate=[validate.Length(max=4096)])
    created_on = DateTime()
    did_number = Str(validate=[validate.Length(max=64)])
    client_billing_plan_id = Int()
    client_trunk_id = Int()
    vendor_trunk_id = Int()
    nrc = Float()
    mrc = Float()
    minutes = Int()
    attempts = Int()
    call_charge = Float()
    start_date = Str(allow_none=False, validate=valid_date)
    end_date = Str(allow_none=False, validate=valid_date)
    mrc_number = Int()
    pay_type = Int()
    client = Str()
    vendor = Str()
    did = Str()

    class Meta:
        model = model.DidChargeDetail
        fields = (
            'params', 'created_on', 'did', 'client_billing_plan_id', 'client_trunk_id', 'nrc', 'mrc', 'minutes',
            'attempts',
            'call_charge', 'start_date', 'end_date', 'mrc_number', 'pay_type',)


class DidChargeDetailSchemeGet(DidChargeDetailScheme):
    client_name = Str()
    vendor_name = Str()

    class Meta:
        model = model.DidChargeDetail
        fields = (
            'did', 'client_billing_plan_id', 'client_trunk_id', 'vendor_trunk_id', 'nrc', 'did_mrc', 'minutes',
            'attempts',
            'call_charge', 'start_date', 'end_date', 'mrc_number', 'pay_type', 'client_name', 'vendor_name')
        search_fields = ('start_date', 'end_date', 'client', 'vendor', 'did')
        # search_fields = ('id','params','did_number','client_billing_plan_id','client_trunk_id','minutes','attempts','start_date','end_date','mrc_number','pay_type',)
        # query_fields=('did_mrc_gt','did_mrc_lt','total_gt','total_lt','created_on_gt','created_on_lt','nrc_gt','nrc_lt','mrc_gt','mrc_lt','call_charge_gt','call_charge_lt',)


class DidCostDetailSchemeGet(DidChargeDetailScheme):
    client_name = Str()
    vendor_name = Str()

    class Meta:
        model = model.DidCostDetail
        fields = (
            'did', 'client_trunk_id', 'vendor_trunk_id', 'nrc', 'did_mrc', 'minutes',
            'start_date', 'end_date', 'client_name', 'vendor_name',
            'vendor_cost', 'client_cost', 'total')
        search_fields = ('start_date', 'end_date', 'client', 'vendor', 'did')


class DidChargeDetailSchemeModify(DidChargeDetailScheme):
    pass


# endregion ---DidChargeDetail---
# region +++LergImportTask+++
class LergImportTaskScheme(BaseModelScheme):
    status = Choice()

    class Meta:
        model = model.LergImportTask
        fields = ('uuid',)


class LergImportTaskSchemeGet(BaseModelScheme):
    status = Choice()
    result = Str()
    file = Str()
    created_on = DateTime()
    created_by = Str()
    finished_on = DateTime()

    class Meta:
        model = model.LergImportTask
        fields = (
            'uuid', 'status', 'result', 'file', 'created_on', 'finished_on', 'created_by')
        search_fields = ('uuid', 'status', 'result', 'file', 'created_by')
        query_fields = ('created_on_gt', 'created_on_lt', 'finished_on_gt', 'finished_on_lt',)


# endregion +++LergImportTask+++
# region +++LergDownloadSchedule+++
class LergDownloadScheme(BaseModelScheme):
    include_CA = Bool()

    class Meta:
        model = model.LergDownloadSchedule
        fields = ('include_CA',)


class LergDownloadScheduleScheme(BaseModelScheme):
    id = Int()
    include_CA = Bool()
    url = Str(validate=[validate.Length(max=100)])
    schedule_hour = Int()
    created_on = DateTime()
    created_by = Str(validate=[validate.Length(max=255)])
    last_downloaded_count = Int()
    last_downloaded_on = DateTime()

    class Meta:
        model = model.LergDownloadSchedule
        fields = ('include_CA', 'schedule_hour')


class LergDownloadScheduleSchemeGet(LergDownloadScheduleScheme):
    class Meta:
        model = model.LergDownloadSchedule
        fields = (
            'id', 'url', 'schedule_hour', 'created_on', 'created_by', 'last_downloaded_on', 'last_downloaded_count')
        search_fields = ('id', 'url', 'schedule_hour', 'created_by', 'last_downloaded_count')
        query_fields = ('created_on_gt', 'created_on_lt', 'last_downloaded_on_gt', 'last_downloaded_on_lt',)


class LergDownloadScheduleSchemeModify(LergDownloadScheduleScheme):
    pass


# endregion ---LergDownloadSchedule---
# region +++LergDownloadTask+++
class LergDownloadTaskScheme(BaseModelScheme):
    uuid = Str(validate=[validate.Length(max=36)])
    status = Choice()
    result = Str()
    created_on = DateTime()
    finished_on = DateTime()
    lerg_import_task_uuid = Str(validate=[validate.Length(max=36)])
    schedule_id = Int()

    class Meta:
        model = model.LergDownloadTask
        fields = ('status', 'result', 'created_on', 'finished_on', 'lerg_import_task_uuid', 'schedule_id',)


class LergDownloadTaskSchemeGet(LergDownloadTaskScheme):
    import_task = Nested('LergImportTaskSchemeGet', many=False)

    class Meta:
        model = model.LergDownloadTask
        fields = (
            'uuid', 'status', 'result', 'created_on', 'finished_on', 'lerg_import_task_uuid', 'schedule_id',
            'import_task', 'count')
        search_fields = ('uuid', 'status', 'result', 'lerg_import_task_uuid', 'schedule_id',)
        query_fields = ('created_on_gt', 'created_on_lt', 'finished_on_gt', 'finished_on_lt',)


class LergDownloadTaskSchemeModify(LergDownloadTaskScheme):
    pass


# endregion ---LergDownloadTask---
# region +++LawfulIntercept+++

class MediaAsrFilterScheme(BaseModelScheme):
    ip_adress = Str(validate=validate.Regexp(IP_REGEXP))
    number = Str()
    enabled = Bool()
    comment = Str()
    add_dt = DateTime()
    num_filter_type = Int()

    class Meta:
        model = model.MediaAsrFilter
        fields = ('number', 'ip_adress', 'enabled', 'comment', 'add_dt', 'num_filter_type',)


class MediaCaptureFilterScheme(MediaAsrFilterScheme):
    class Meta:
        model = model.MediaCaptureFilter
        fields = ('number', 'ip_adress', 'enabled', 'comment', 'add_dt', 'num_filter_type',)


class SignalCaptureFilterScheme(MediaAsrFilterScheme):
    class Meta:
        model = model.SignalCaptureFilter
        fields = ('number', 'ip_adress', 'enabled', 'comment', 'add_dt', 'num_filter_type',)


# endregion ---LawfulIntercept---
# region +++SmsReportScheme+++
class SmsReportScheme(BaseModelScheme):
    report_time = Str()
    client_name = Str()
    client_id = Int()
    vendor_name = Str()
    vendor_id = Int()
    attempt_count = Int()
    success_count = Int()
    failed_count = Int()

    class Meta:
        model = model.Sms
        fields = (
            'report_time', 'client_name', 'client_id', 'vendor_name',
            'vendor_id', 'attempt_count', 'success_count', 'failed_count',
        )


# endregion SmsReportScheme
# region +++DidDisconnectTask+++

class DidApiAdminDisconnectLocalInnerScheme(Schema):
    number = Str(validate=validate.Regexp(PHONE_REGEXP))


class DidApiAdminDisconnectScheme(Schema):
    items = Nested(DidApiAdminDisconnectLocalInnerScheme, many=True)
    release_on_time = DateTime()


class DidApiAdminDisconnectAllScheme(Schema):
    release_on_time = DateTime()


class DidDisconnectTaskScheme(BaseModelScheme):
    id = Int()
    operator_user = Str(validate=[validate.Length(max=40)])
    data = Dict()
    release_on_time = DateTime()
    started_at = DateTime()
    finished_at = DateTime()
    reply = Dict()
    success = Bool()
    success_count = Int()
    error_count = Int()

    class Meta:
        model = model.DidDisconnectTask
        fields = ('operator_user', 'data', 'release_on_time', 'started_at', 'finished_at', 'reply', 'success')


class DidDisconnectTaskSchemeGet(DidDisconnectTaskScheme):
    class Meta:
        model = model.DidDisconnectTask
        fields = ('id', 'operator_user', 'data', 'release_on_time', 'started_at', 'finished_at', 'reply', 'success',
                  'success_count', 'error_count')
        search_fields = ('id', 'operator_user', 'success', 'success_count', 'error_count')
        query_fields = ('release_on_time_gt', 'release_on_time_lt', 'started_at_gt', 'started_at_lt', 'finished_at_gt',
                        'finished_at_lt',)


class DidDisconnectTaskSchemeModify(DidDisconnectTaskScheme):
    pass


class DidApiAdminAssignScheme(Schema):
    did_client_name = Str()
    client_billing_rule_name = Str()


# endregion ---DidDisconnectTask---
# region +++ClientDidProduct+++
class ClientDidProductScheme(BaseModelScheme):
    client_id = Int()
    did_product_id = Int(allow_none=True, validate=lambda value: _valid('DidProduct', 'id', value))
    rate_table_id = Int(allow_none=True, validate=lambda value: _valid('SmsRateTable', 'id', value))
    enable_sms = Bool()
    created_by = Str(validate=[validate.Length(max=40)])
    created_on = DateTime()

    class Meta:
        model = model.ClientDidProduct
        fields = ('did_product_id', 'rate_table_id', 'enable_sms')


class ClientDidProductSchemeGet(ClientDidProductScheme):
    client_name = Str(validate=[validate.Length(max=500)])
    product_name = Str()
    rate_table_name = Str()
    type = Choice()
    items = Nested('DidProductItemScheme', many=True)

    class Meta:
        model = model.ClientDidProduct
        fields = (
            'did_product_id', 'rate_table_id', 'enable_sms', 'created_by', 'created_on', 'client_name',
            'product_name', 'country', 'type', 'billing_rule_ids',
            'rate_table_name', 'mrc', 'nrc', 'rate_per_min', 'items')
        search_fields = (
            'client_id', 'did_product_id', 'rate_table_id', 'enable_sms', 'created_by', 'client_name',
            'product_name',
            'rate_table_name')
        query_fields = ('created_on_gt', 'created_on_lt',)


class ClientDidProductSchemeModify(ClientDidProductScheme):
    pass


# endregion ---ClientDidProduct---
# region +++DidProduct+++
class DidProductScheme(BaseModelScheme):
    id = Int()
    name = Str(validate=[validate.Length(1, 40), validate.Regexp(USERNAME_REGEXP),
                         lambda value: _valid_unique('DidProduct', 'name', value)])
    items = Nested('DidProductItemScheme', many=True)

    class Meta:
        model = model.DidProduct
        fields = ('name',)


class DidProductSchemeGet(DidProductScheme):
    items = Nested('DidProductItemSchemeGet', many=True)

    class Meta:
        model = model.DidProduct
        fields = ('id', 'name', 'items', 'billing_rule_id', 'used_by', 'created_by', 'created_on')
        search_fields = ('id', 'name', 'billing_rule_id', 'used_by', 'created_by', 'created_on')


class DidProductSchemeModify(DidProductScheme):
    name = Str(validate=[validate.Length(1, 40), validate.Regexp(USERNAME_REGEXP)])


# endregion ---DidProduct---
# region +++DidProductItem+++
class DidProductItemScheme(BaseModelScheme):
    product_name = Str(validate=[validate.Length(max=40)])
    id = Int()
    did_product_id = Int()
    country = Str(allow_none=False, required=True, validate=[validate.Length(max=20)])
    type = Choice(allow_none=False, required=True)
    billing_rule_id = Int()
    created_by = Str(validate=[validate.Length(max=40)])
    created_on = DateTime()

    class Meta:
        model = model.DidProductItem
        fields = ('did_product_id', 'country', 'type', 'billing_rule_id')


class DidProductItemSchemeGet(DidProductItemScheme):
    class Meta:
        model = model.DidProductItem
        fields = (
            'id', 'did_product_id', 'country', 'type', 'billing_rule_id', 'created_by', 'created_on',
            'billing_rule_name', 'nrc', 'mrc', 'client_id', 'usage_count')
        search_fields = (
            'id', 'product_name', 'id', 'did_product_id', 'country', 'type', 'billing_rule_id', 'created_by',
            'client_id', 'usage_count', 'billing_rule_name')
        query_fields = ('created_on_gt', 'created_on_lt',)


class DidProductItemSchemeModify(DidProductItemScheme):
    pass


# endregion ---DidProductItem---
# region +++RandomAniGeneration+++
class RandomAniGenerationScheme(BaseModelScheme):
    id = Int()
    ani_number = Str(validate=validate.Length(max=32))
    random_table_id = Int(validate=[lambda value: _valid('RandomAniGroup', 'id', value)])

    class Meta:
        model = model.RandomAniGeneration
        fields = ('ani_number', 'random_table_id',)


class RandomAniGenerationSchemeGet(RandomAniGenerationScheme):
    id = Str()
    class Meta:
        model = model.RandomAniGeneration
        fields = ('id', 'ani_number', 'random_table_id', 'enabled')
        search_fields = ('id', 'ani_number', 'random_table_id', 'enabled')


class RandomAniGenerationSchemeModify(RandomAniGenerationScheme):
    pass


# endregion ---RandomAniGeneration---
# region +++RandomAniGroup+++
class RandomAniGroupScheme(BaseModelScheme):
    id = Int()
    group_name = Str(validate=[validate.Length(max=100)])

    class Meta:
        model = model.RandomAniGroup
        fields = ('group_name',)


class RandomAniGroupSchemeGet(RandomAniGroupScheme):
    class Meta:
        model = model.RandomAniGroup
        fields = ('id', 'group_name', 'create_time',)
        search_fields = ('id', 'group_name', 'create_time',)


class RandomAniGroupSchemeModify(RandomAniGroupScheme):
    pass


# endregion ---RandomAniGroup---
# region +++CdrExportEmailTask+++
class CdrExportEmailTaskScheme(BaseModelScheme):
    id = Int()
    email_status = Choice()
    started_on = DateTime()
    finished_on = DateTime()
    email_error = Str()
    template = Nested('SendCdrExportTemplateScheme', many=False)
    export = Nested('CdrExportTaskScheme', many=False)

    class Meta:
        model = model.CdrExportEmailTask
        fields = ('template',)


class CdrExportEmailTaskSchemeGet(CdrExportEmailTaskScheme):
    class Meta:
        model = model.CdrExportEmailTask
        fields = ('email_status', 'started_on', 'finished_on', 'email_error',)
        search_fields = ('id', 'email_status', 'email_error', 'template', 'export',)
        query_fields = ('started_on_gt', 'started_on_lt', 'finished_on_gt', 'finished_on_lt',)


class CdrExportEmailTaskSchemeModify(CdrExportEmailTaskScheme):
    pass


# endregion ---CdrExportEmailTask---
# region +++SendCdrExportTemplate+++
class SendCdrExportTemplateScheme(BaseModelScheme):
    id = Int()
    subject = Str(validate=[validate.Length(max=200)])
    html_content = Str()
    sender_id = Int()
    cc_mail = Str(validate=[validate.Length(max=200)])
    to_mail = Str(validate=[validate.Length(max=200)])

    class Meta:
        model = model.SendCdrExportTemplate
        fields = ('subject', 'html_content', 'sender_id', 'cc_mail', 'to_mail',)


class SendCdrExportTemplateSchemeGet(SendCdrExportTemplateScheme):
    class Meta:
        model = model.SendCdrExportTemplate
        fields = ('subject', 'html_content', 'sender_id', 'cc_mail', 'to_mail',)
        search_fields = ('id', 'subject', 'html_content', 'sender_id', 'cc_mail', 'to_mail',)


class SendCdrExportTemplateSchemeModify(SendCdrExportTemplateScheme):
    pass


# endregion ---SendCdrExportTemplate---
# region +++CdrExportTask+++
class CdrExportTaskScheme(BaseModelScheme):
    id = Int()
    operator_user = Str(validate=[validate.Length(max=40)])
    export_cdr_file = Str(validate=[validate.Length(max=1024)])
    cdr_counts = Int()
    cdr_start_time = DateTime()
    cdr_end_time = DateTime()
    csv_file_headers = Str(validate=[validate.Length(max=2048)])
    cdr_headers = Str(validate=[validate.Length(max=2048)])
    cdr_filter = Str(validate=[validate.Length(max=2048)])
    status = Choice()
    progress = Str(validate=[validate.Length(max=1024)])
    start_time = DateTime()
    end_time = DateTime()
    create_time = DateTime()
    email_task = Nested('CdrExportEmailTaskScheme', many=False)

    class Meta:
        model = model.CdrExportTask
        fields = ('cdr_start_time', 'cdr_end_time', 'csv_file_headers', 'cdr_headers', 'cdr_filter', 'email_task')


class CdrExportTaskSchemeGet(CdrExportTaskScheme):
    email_status = Choice()
    email_error = Str()

    class Meta:
        model = model.CdrExportTask
        fields = (
            'operator_user', 'export_cdr_file', 'cdr_counts', 'cdr_start_time', 'cdr_end_time', 'csv_file_headers',
            'cdr_headers', 'cdr_filter', 'status', 'progress', 'start_time', 'end_time', 'create_time', 'email_status',
            'email_error')
        search_fields = (
            'id', 'operator_user', 'export_cdr_file', 'cdr_counts', 'csv_file_headers', 'cdr_headers', 'cdr_filter',
            'status', 'progress', 'email_status', 'email_error')
        query_fields = ('cdr_start_time_gt', 'cdr_start_time_lt', 'cdr_end_time_gt', 'cdr_end_time_lt', 'start_time_gt',
                        'start_time_lt', 'end_time_gt', 'end_time_lt', 'create_time_gt', 'create_time_lt',)


class CdrExportTaskSchemeModify(CdrExportTaskScheme):
    pass


# endregion ---CdrExportTask---
# region +++RateAutoImportMailboxLog+++
class RateAutoImportMailboxLogScheme(BaseModelScheme):
    id = Int()
    start = DateTime()
    finish = DateTime()
    error = Str()

    class Meta:
        model = model.RateAutoImportMailboxLog
        fields = ('start', 'finish', 'error',)


class RateAutoImportMailboxLogSchemeGet(RateAutoImportMailboxLogScheme):
    class Meta:
        model = model.RateAutoImportMailboxLog
        fields = ('start', 'finish', 'error',)
        search_fields = ('id', 'error',)
        query_fields = ('start_gt', 'start_lt', 'finish_gt', 'finish_lt',)


class RateAutoImportMailboxLogSchemeModify(RateAutoImportMailboxLogScheme):
    pass
# endregion ---RateAutoImportMailboxLog---


class SendTroubleTicketScheme(BaseModelScheme):
    mail_sender_id = Int(required=True)
    cc_email = String(required=False)
    dest_email = String(required=True)
    content = String(required=True)
    subject = String(required=True)

    class Meta:
        fields = ('mail_sender_id', 'cc_email', 'dest_email', 'content', 'subject')

