"""Contain Google Cloud Storage functions."""
import os
import hashlib
from google.cloud import storage as gcs
from google.cloud import exceptions as gcs_exception


class GCSWorker():

    """Contain Google Cloud Storage functions."""

    def __init__(self,credentials_file,logger):
        """Using the service account with Application Default Credentials.

        Set the GOOGLE_APPLICATION_CREDENTIALS environment variable to
        point to your service account key file.

        export GOOGLE_APPLICATION_CREDENTIALS=<path_to_account_file>

        """
        os.environ["GOOGLE_APPLICATION_CREDENTIALS"] = credentials_file
        self.logger = logger

    @staticmethod
    def set_gac(credentials_file):
        """Using the service account with Application Default Credentials.

        Set the GOOGLE_APPLICATION_CREDENTIALS environment variable to
        point to your service account key file.

        export GOOGLE_APPLICATION_CREDENTIALS=<path_to_account_file>

        """
        os.environ["GOOGLE_APPLICATION_CREDENTIALS"] = credentials_file

    def get_bucket(self):
        """Get working Google Cloud Storage bucket.

        :return: A current app working bucket.
        :return type: google.cloud.storage.bucket.Bucket.

        """
        storage_client = gcs.Client()
        bucket = list(storage_client.list_buckets())[0]
        return bucket

    def upload_file(self,source,target,bucket):
        """Upload one file to Google Cloud Storage.

        :param source: A local file`s absolute name.
        :param target: A remote file`s absolute name.
        :param bucket: A current app working bucket.
        :return: ``True`` if success, ``False`` otherwise.

        """
        try:
            blob = bucket.blob(target)
            blob.upload_from_filename(source,
                    content_type='application/octet-stream')
            self.logger.debug('File {} has been uploaded to remote {}.'.format(
                        source,target))
            return True
        except gcs_exception.GoogleCloudError as e:
            self.logger.error('File {0} has not been uploaded to GCS target '
                    '{1} at bucket {2} due to GoogleAPICallError: {3}'.format(
                    source,target,bucket,e))
        return False

    def download_file(self,remote,local,bucket):
        """Download one file from Google Cloud Storage.

        :param source: A remote file`s absolute name.
        :param target: A local file`s absolute name.
        :param bucket: A current app working bucket.
        :return: ``True`` if success, ``None`` otherwise.

        """
        try:
            blob = bucket.blob(remote)
            blob.download_to_filename(local)
            return True
        except gcs_exception.NotFound as e:
            self.logger.error('File {0} not found at GCS bucket {1}.'.format(
                    remote,bucket))
        except gcs_exception.GoogleCloudError as e:
            self.logger.error('File {0} has not been downloaded due to '
                    'GoogleAPICallError: {1}'.format(remote, e))
        return False

    def upload_file_check_hash(self,source,target):
        """Upload one file to Google Cloud Storage.

        :param source: A local file`s absolute name.
        :param target: A remote file`s absolute name.
        :return: ``True`` if success, ``False`` otherwise.

        """
        bucket = self.get_bucket()

        if not self.upload_file(source,target,bucket):
            return False
        read_copy = source + ".bak"
        if not self.download_file(target,read_copy,bucket):
            return False
        pcaphash_from=hashlib.md5(open(source,'rb').read()).hexdigest()
        pcaphash_to=hashlib.md5(open(read_copy,'rb').read()).hexdigest()
        if pcaphash_from == pcaphash_to:
            blob = bucket.blob(target + ".hash")
            blob.upload_from_string(pcaphash_from)
            """Check remote hash string contents.

            When get download_as_string() has recive buffer contents,
            it is needed to decode to str/unicode.
            """
            pcaphash_to = blob.download_as_string().decode('utf-8')
            if pcaphash_from == pcaphash_to:
                self.logger.info('File {} with hash has been uploaded to '
                        'remote {}. The local source file is deleted.'.format(
                        source,target))
                os.remove(source)
                os.remove(read_copy)
                return True
            else:
                self.logger.error("The hash string get corrupted during the "
                        "upload. The hash string is {}, the downloaded one is"
                        " {}".format(pcaphash_from, pcaphash_to))
        self.logger.error("The hash check show that the file get corrupted "
                "during the upload. The source file {} is intact, the "
                "remote one is deleted".format(source))
        blob = bucket.blob(target)
        blob.delete()
        os.remove(read_copy)
        return False
