import json
import sys, os
import requests
from urllib.parse import parse_qsl, urlencode
from datetime import date, datetime, timedelta
import traceback
from pytz import UTC
import mimetypes
from time import mktime
from dateutil.parser import parse as parse_datetime
import falcon
from sqlalchemy.orm import aliased, make_transient, foreign
from sqlalchemy import or_, and_, select, inspect, alias
from api_dnl import model
from falcon_rest import conf
from falcon_rest.logger import log
from falcon_rest.responses import responses
from falcon_rest.responses import errors
from falcon_rest.resources.resources import swagger, ResourcesBaseClass, DEFAULT_SECURITY, ATTRIBUTE_ERROR_RE
from falcon_rest.schemes import BaseModelScheme, Schema, fields, validate

from api_dnl.resources import DnlResource, DnlList, DnlCreate, CustomAction, CustomGetAction, CustomPostAction, \
    CustomPatchAction, ValidationError, client_context, check_context
from api_dnl.view import NoResultFound, DEFAULT_SECURITY, OperationErrorResponse, generate_uuid_str
from api_dnl.utils.statisticapi2 import StatisticAPI, cdr_fields, StatisticException
from api_dnl import settings


def StatisticErrorResponce(e):
    if len(e.args):
        res = e.args[0]
        if 'code' in e.args[0]:
            return responses.OperationErrorResponse(
                data=dict(code=res['code'], reason=res['msg'], message=res['data']))
        else:
            return responses.OperationErrorResponse(data=dict(code=406, reason=res))
    else:
        return responses.OperationErrorResponse(data=dict(code=406, reason=str(e)))


class RtSyncGet(CustomGetAction):
    query_parameters = [
        # Required arguments:
        {"name": "start_time", "type": "integer", "description": "search from specified time (allow negative values)"},
        # Optional arguments:
        {"name": "end_time", "type": "integer",
         "description": "search till time (end_time > start_time). Default: 0 do not limit search by time"},
        {"name": "format", "type": "string", "description": "output format:"
                                                            "format=csv", "descripton": "csv file (default)"
                                                                                        "format=json",
         "descripton": "json output"
                       "format=cdr_json", "descripton": "cdr lines, wrapped into json list "},
        {"name": "field", "type": "string",
         "description": "comma separated list of fields to print. Default: print all fields"},
        {"name": "human_readable", "type": "string", "description": "use human-readable values"},
        {"name": "print_only_amount", "type": "string", "description": "print only amount of found results"},
        {"name": "print_only_headers", "type": "string", "description": "print only header"},
        # Search filter:
        {"name": "switch_name", "type": "string", "description": "name of switch instance (string)"},
        {"name": "release_cause", "type": "integer", "description": "release cause (integer)"},
        {"name": "orig_code", "type": "string", "description": "originating code: 'country_name:code_name:code'"},
        {"name": "term_code", "type": "string", "description": "termination code: 'country_name:code_name:code'"},
        {"name": "ingress_id", "type": "integer", "description": "ingress id (integer)"},
        {"name": "egress_id", "type": "integer", "description": "egress id (integer)"},
        {"name": "orig_dest_host", "type": "string", "description": "originating destination host name (string)"},
        {"name": "source_number", "type": "string", "description": "originating source number (string)"},
        {"name": "dest_number", "type": "string", "description": "originating destination numbers (string)"},
        {"name": "term_source_number", "type": "string", "description": "termination source number (string)"},
        {"name": "term_dest_number", "type": "string", "description": "termination destination numbers (string)"},
        {"name": "orig_call_id", "type": "string", "description": "origination call id (string)"},
        {"name": "term_call_id", "type": "string", "description": "termination call id (string)"},
        {"name": "routing_digits", "type": "string", "description": "routing number (string)"},
        {"name": "ingress_rcause", "type": "string", "description": "ingress release cause (string)"},
        {"name": "egress_rcause", "type": "string", "description": "egress release cause (string)"},
        {"name": "is_final_call", "type": "integer", "description": "is final call (0 or 1)"},
        {"name": "non_zero", "type": "integer", "description": "is non-zero call duration (0 or 1)"},
        {"name": "duration_min", "type": "integer", "description": "call duration in seconds (integer)"},
        {"name": "duration_max", "type": "integer", "description": "call duration in seconds (integer)"},
        {"name": "cost_min", "type": "float", "description": "egress cost (floating point)"},
        {"name": "cost_max", "type": "float", "description": "egress cost (floating point)"},
        {"name": "call_tz", "type": "integer", "description": "call timezone in minutes (integer)"},
        {"name": "only_authorized", "type": "integer", "description": "show only authorized calls (0 or 1)"},
        {"name": "start_from", "type": "integer"},
        {"name": "count", "type": "integer"},
    ]

    def cut_body(self, req, resp, **kwargs):
        params = dict(parse_qsl(req.query_string))
        start = 0
        count = 4098
        format = None
        if 'start_from' in params:
            start = int(params['start_from'])
        if 'count' in params:
            count = int(params['count'])
        if 'format' in params:
            format = params['format']
        if format == 'json':
            try:
                d = json.loads(resp.body.decode())
                if 'data' in d and type(d['data'] == type([])):
                    d['data'] = d['data'][start:start + count]
                    d['total'] = len(d['data'])
                    resp.body = json.dumps(d)
                    log.debug('cdr big json data cutted to count={}'.format(count))
            except Exception as e:
                log.debug('cdr big json data not parsed {}: {}...'.format(e, resp.body[0:32]))
        if format == 'csv':
            d = resp.body.split(b'\n')
            resp.body = '\n'.join(d[start:start + count])
            log.debug('cdr big csv data cutted to count=256')

    def on_get(self, req, resp, **kwargs):
        self.init_req(req)
        if not check_context(self, req, resp, **kwargs):
            return False
        user = self.get_user(self.req)
        params = dict(parse_qsl(req.query_string))
        api = StatisticAPI
        try:
            if 'field' in params:
                params['field'] = cdr_fields(params['field'])
            ret = api.send_request(params=params, token=user.cdr_api_token, rt_api='cdr', timeout=300)
            if ret:
                resp.body = ret.content
                # self.cut_body(req, resp, **kwargs)
                resp.status = falcon.HTTP_200
        except Exception as e:
            if 'Read timed out' in str(e):
                try:
                    ret = api.send_request(params=params, token=user.cdr_api_token, rt_api='cdr', timeout=900)
                    if ret:
                        resp.body = ret.content
                        # self.cut_body(req, resp, **kwargs)
                        resp.status = falcon.HTTP_200
                except Exception as e:
                    self.set_response(resp, OperationErrorResponse(e))
                    return None
            self.set_response(resp, OperationErrorResponse(e))
            return None
        pass


class RtAsyncPostScheme(BaseModelScheme):
    # Required arguments:
    start_time = fields.Int()  # search from specified time (allow negative values)
    # Optional arguments:
    end_time = fields.Int()  # search till time (end_time > start_time). Default: 0","descripton":"do not limit search by time
    field = fields.Str()  # comma separated list of fields to print. Default: print all fields
    human_readable = fields.Str()  # use human-readable values
    keep_longer = fields.Int()  # define storage to use. Default: as set in default_async_storage (see 'Configuration')
    # keep_longer=0 -> keep on server
    # keep_longer=1 -> send to gcloud storage
    email = fields.Email()  # email address to send notification on completion
    # Search filter: See 'CDR Engine' section
    switch_name = fields.Str()  # name of switch instance (string)
    release_cause = fields.Str()  # release cause (integer)
    orig_code = fields.Str()  # originating code: 'country_name:code_name:code'
    term_code = fields.Str()  # termination code: 'country_name:code_name:code'
    ingress_id = fields.Str()  # ingress id (integer)
    egress_id = fields.Str()  # egress id (integer)
    orig_dest_host = fields.Str()  # originating destination host name (string)
    source_number = fields.Str()  # originating source number (string)
    dest_number = fields.Str()  # originating destination numbers (string)
    term_source_number = fields.Str()  # termination source number (string)
    term_dest_number = fields.Str()  # termination destination numbers (string)
    orig_call_id = fields.Str()  # origination call id (string)
    term_call_id = fields.Str()  # termination call id (string)
    routing_digits = fields.Str()  # routing number (string)
    ingress_rcause = fields.Str()  # ingress release cause (string)
    egress_rcause = fields.Str()  # egress release cause (string)
    is_final_call = fields.Int()  # is final call (0 or 1)
    non_zero = fields.Int()  # is non-zero call duration (0 or 1)
    duration_min = fields.Int()  #
    duration_max = fields.Int()  # call duration in seconds (integer)
    cost_min = fields.Float()  #
    cost_max = fields.Float()  # egress cost (floating point)
    call_tz = fields.Int()  # call tiselect callmezone in minutes (integer)
    only_authorized = fields.Int()  # show only authorized calls (0 or 1)
    # FTP uploading:
    ftp_url = fields.Str()  # (s)ftp server url to upload (ftp=ftp://myftpserver.com)
    ftp_port = fields.Int()  # server port (default: 21 -> ftp; 22 -> sftp)
    ftp_user = fields.Str()  # server user name (default: Anonymous)
    ftp_password = fields.Str()  # server password (default: None)
    ftp_dir = fields.Str()  # server path to upload: for ftp path from access point; for sftp -> path from root. (default: "/")
    ftp_file_name = fields.Str()  # filename prefix (70 chars max). Engine will generate name of the file using the following template:
    # [prefix]_[yyyy-mm-dd__hh_mm_ss]_[yyyy-mm-dd__hh_mm_ss]_[breakdown].[ext]
    ftp_max_lines = fields.Int()  # max lines per file
    ftp_compress = fields.Str()  # compress file. Allowed formats: gz, tar.gz, tar.bz2
    ftp_include_headers = fields.Int()  # include headers line in csv file: 0 -> do not include; 1 -> include. Default: 1 -> with headers
    ftp_file_breakdown = fields.Int()  # breakdown results: 0","descripton":"as one file; 1 -> as hourly file; 2 -> as daily file. Default: 0 -> single file
    tz = fields.Str()


class RtAsyncPost(CustomPostAction):
    body_parameters = ('Asynchronous cdr files search', RtAsyncPostScheme,)

    def on_post(self, req, resp, **kwargs):
        self.init_req(req)
        if not check_context(self, req, resp, **kwargs):
            return False
        user = self.get_user(self.req)
        params = req.data
        api = StatisticAPI
        try:
            if 'field' in params:
                params['field'] = cdr_fields(params['field'])
            ret = api.send_request(params=params, token=user.cdr_api_token, rt_api='async', method='post')
            if ret:
                resp.body = ret.content
                resp.status = falcon.HTTP_200
        except Exception as e:
            self.set_response(resp, OperationErrorResponse(e))
            return None
        pass


class RtAsyncGet(CustomGetAction):
    description = "Asynchronous cdr request status"
    path_parameters = ({"name": "request_id", "type": "string", "description": "request id"},)
    allow_methods = ["delete", "get"]

    def on_get(self, req, resp, **kwargs):
        self.init_req(req)
        if not check_context(self, req, resp, **kwargs):
            return False
        user = self.get_user(self.req)
        # params = req.data
        params = dict(parse_qsl(req.query_string))
        api = StatisticAPI
        try:
            request_id = kwargs['request_id']
            ret = api.send_request(params=params, cdr_request=request_id, token=user.cdr_api_token, rt_api='async',
                                   method='get')
            if ret:
                resp.body = ret.content
                resp.status = falcon.HTTP_200
        except Exception as e:
            self.set_response(resp, OperationErrorResponse(e))
            return None
        pass

    def on_delete(self, req, resp, **kwargs):
        self.init_req(req)
        if not check_context(self, req, resp, **kwargs):
            return False
        user = self.get_user(self.req)
        params = req.data
        api = StatisticAPI
        try:
            request_id = kwargs['request_id']
            ret = api.send_request(params=params, cdr_request=request_id, token=user.cdr_api_token, rt_api='async',
                                   method='delete')
            if ret:
                resp.body = ret.content
                resp.status = falcon.HTTP_200
        except Exception as e:
            self.set_response(resp, OperationErrorResponse(e))
            return None
        pass


class RtAsyncDelete(RtAsyncGet):
    method = 'delete'


class RtAsyncList(CustomGetAction):
    description = "Asynchronous cdr requests list"
    query_parameters = [
        {"name": "order_by", "type": "string", "descripton": "order field",
         'enum': ['job_id', 'job_start_time', 'status']},
        {"name": "order_dir", "type": "string", "descripton": "order direction", 'enum': ['ask', 'desc']},
    ]

    #     query_parameters = [
    # {"name":"expiration_time","type":"string","descripton":"request expiration time from epoch"},
    # {"name":"job_id","type":"string","descripton":"human-readable request id"},
    # {"name":"job_start_time","type":"string","descripton":"time when request was started"},
    # {"name":"job_end_time","type":"string","descripton":"time when request has finished"},
    # {"name":"status","type":"string","enum":['Processing', 'Uploading', 'Complete', 'Failed to process', 'Failed to upload', 'Expired', 'Interrupted by user', 'Caching']},
    # {"name":"progress","type":"integer","descripton":"percentage done"},
    # {"name":"err_str","type":"string","descripton":"error explanation"},
    # {"name":"start_time","type":"string","descripton":"requested cdr start time"},
    # {"name":"end_time","type":"string","descripton":"requested cdr end time"},
    # {"name":"count","type":"integer","descripton":"amount of lines found"},
    # {"name":"size","type":"integer","descripton":"size of file"},
    # {"name":"cached","type":"string","descripton":"true if file cached and can be opened"},
    # {"name":"download_link","type":"string","descripton":"link to download file"},
    # {"name":"to_email","type":"string","descripton":"requested address for email notifications"} ]
    def on_get(self, req, resp, **kwargs):
        self.init_req(req)
        if not check_context(self, req, resp, **kwargs):
            return False
        user = self.get_user(self.req)
        # params = req.data
        params = dict(parse_qsl(req.query_string))
        ordering = {}
        if 'order_by' in params:
            ordering['by'] = params['order_by']
            del params['order_by']
            if 'order_dir' in params:
                ordering['dir'] = params['order_dir']
                del params['order_dir']
            else:
                ordering['dir'] = 'asc'
        api = StatisticAPI
        try:
            import json
            ret = api.send_request(params=params, token=user.cdr_api_token, rt_api='async', method='get')
            if ret:
                if ordering:
                    js = json.loads(ret.content.decode('UTF-8'))
                    reverse = False
                    if ordering['dir'] == 'desc':
                        reverse = True
                    js['data'] = sorted(js['data'], key=lambda v: v[ordering['by']], reverse=reverse)
                    resp.body = json.dumps(js)
                else:
                    resp.body = ret.content
                resp.status = falcon.HTTP_200
        except Exception as e:
            self.set_response(resp, OperationErrorResponse(e))
            return None
        pass


class RtAsyncOpenFileScheme(BaseModelScheme):
    # Optional arguments:
    sort = fields.Str()  # sort results by column
    order = fields.Str(
        validate=validate.OneOf(("asc", "desc")))  # sorting order: "asc", "desc". Default: ascending order
    start_from = fields.Int()
    count = fields.Int()  # printing range


class RtAsyncOpenFile(CustomGetAction):
    # body_parameters = ('Asynchronous cdr open file',RtAsyncOpenFileScheme,)
    path_parameters = ({"name": "request_id", "type": "string", "description": "request id"},)
    query_parameters = [
        {"name": "start_from", "type": "integer"},
        {"name": "count", "type": "integer"},
        {"name": "sort", "type": "string"},
        {"name": "order", "description": "sorting order: 'asc', 'desc'. Default: ascending order", "type": "string",
         "enum": ['asc', 'desc']},
    ]

    # no_auth_needed = True

    def on_get(self, req, resp, **kwargs):
        self.init_req(req)
        if not check_context(self, req, resp, **kwargs):
            return False
        user = self.get_user(self.req)
        # user = model.User.get(1)
        if datetime.now(UTC) > user.cdr_expire:
            user.cdr_token()
        # params = req.data
        params = dict(parse_qsl(req.query_string))
        if 'start_from' in params:
            params['from'] = params['start_from']
            del params['start_from']
        api = StatisticAPI
        try:
            try:
                # check type
                request_id = kwargs['request_id']
                ret = api.send_request(params=params, cdr_request=request_id, token=user.cdr_api_token, rt_api='async',
                                       method='get')
                type = ret.json()['type']
            except StatisticException as e:
                type = 'list'
            if type == 'download':
                request_id = kwargs['request_id']
                ret = api.send_request(params=params, cdr_request=request_id + '/download', token=user.cdr_api_token,
                                       rt_api='async',
                                       method='get')
                resp.body = ret.content
                resp.append_header('Content-Disposition', 'attachment; filename="request_{}.csv"'.format(request_id))
                resp.content_type = 'text/csv;charset=utf-8'
                resp.status = falcon.HTTP_200
                return
            else:
                request_id = kwargs['request_id']
                ret = api.send_request(params=params, cdr_request=request_id,
                                       token=user.cdr_api_token,
                                       # token=settings.STATISTIC['token'],
                                       rt_api='async',
                                       method='post')
                if ret:
                    resp.body = ret.content
                    resp.status = falcon.HTTP_200
                    return
            raise StatisticException('no response from rt index')
        except Exception as e:
            self.set_response(resp, OperationErrorResponse(e))
            return None
        pass


class RtAsyncRestart(CustomPostAction):
    description = "Asynchronous cdr request restart"
    path_parameters = ({"name": "request_id", "type": "string", "description": "request id"},)

    def on_post(self, req, resp, **kwargs):
        self.init_req(req)
        if not check_context(self, req, resp, **kwargs):
            return False
        user = self.get_user(self.req)
        log.debug('user:{}'.format(user))
        params = req.data
        api = StatisticAPI
        try:
            request_id = kwargs['request_id']
            ret = api.send_request(params=params, cdr_request=request_id + '/restart', token=user.cdr_api_token,
                                   rt_api='async',
                                   method='post')
            if ret:
                resp.body = ret.content
                resp.status = falcon.HTTP_200
        except Exception as e:
            self.set_response(resp, OperationErrorResponse(e))
            return None
        pass


class RtAsyncSendEmailScheme(BaseModelScheme):
    # Optional arguments:
    email = fields.Email()


class RtAsyncSend(CustomPostAction):
    description = "Asynchronous cdr send email notification"
    body_parameters = ('Asynchronous cdr open file', RtAsyncSendEmailScheme,)
    path_parameters = ({"name": "request_id", "type": "string", "description": "request id"},)

    def on_post(self, req, resp, **kwargs):
        self.init_req(req)
        if not check_context(self, req, resp, **kwargs):
            return False
        user = self.get_user(self.req)
        params = req.data
        api = StatisticAPI
        try:
            request_id = kwargs['request_id']
            ret = api.send_request(params=params, cdr_request=request_id + '/sendmail', token=user.cdr_api_token,
                                   rt_api='async',
                                   method='post')
            if ret:
                resp.body = ret.content
                resp.status = falcon.HTTP_200
        except StatisticException as e:
            res = e.args[0]
            self.set_response(resp, responses.OperationErrorResponse(
                data=dict(code=res['code'], reason=res['msg'], message=res['data'])))
            # resp.status = str(code)
            return None
        except Exception as e:
            self.set_response(resp, OperationErrorResponse(e))
            return None
        pass


class RtReportGet(CustomGetAction):
    description = "Reports Engine"
    query_parameters = [
        # Required arguments:
        {"name": "start_time", "description": "search from specified time (allow negative values)", "type": "string"},
        {"name": "method", "description": "aggregation method", "type": "string",
         "enum": ['max', 'min', 'avrg', 'total']},
        {"name": "step",
         "description": "granularity: 'hour', 'day', 'week', 'month', 'year', 'all', or amount of minutes. With custom step first timestamp will be Jan 1 1970 00:00.",
         "type": "string"},
        # Optional arguments:
        {"name": "end_time",
         "description": "search till time (end_time > start_time). Default: 0 -> do not limit search by time",
         "type": "string"},
        {"name": "field", "description": "fields to aggregate (default: aggregate all fields)", "type": "string"},
        # "enum":['ingress_cost','ingress_time','ingress_billed_time','egress_cost','egress_time','egress_billed_time','pdd','ring_time', 'calls','zero_epoch_calls','lrn_calls','npr_calls','non_zero_calls','non_zero_calls_6','non_zero_calls_30','ingress_calls','ingress_success_calls','ingress_busy_calls','ingress_cancel_calls','egress_calls','egress_busy_calls','egress_cancel_calls','ingress_call_cost_intra','ingress_call_cost_inter','ingress_call_cost_ij','ingress_call_cost_local','egress_call_cost_intra','egress_call_cost_inter','egress_call_cost_ij','egress_call_cost_local','ingress_bill_time_intra','ingress_bill_time_inter','ingress_bill_time_ij','ingress_bill_time_local','egress_bill_time_intra','egress_bill_time_inter','egress_bill_time_ij','egress_bill_time_local','inter_ingress_calls','intra_ingress_calls','ij_ingress_calls','local_ingress_calls','inter_egress_calls','intra_egress_calls','ij_egress_calls','local_egress_calls','ij_not_zero_calls','local_not_zero_calls','duration','ingress_cps_limit','ingress_cap_limit','egress_limit','nrf_calls']
        {"name": "group", "description": "keys to group by (order is ignored; default: no grouping)", "type": "string"},
        #   "enum":['switch_name','release_cause','orig_sip_resp','term_sip_resp','ingress_id','egress_id','rate_table_id','route_plan','ingress_host','egress_host','country','code_name','orig_code','source_number','dest_number']},
        {"name": "tz", "description": "timezone (in hours from UTC)", "type": "string"},
        {"name": "sort", "description": "sort results by field", "type": "string"},
        {"name": "order", "description": "sorting order: 'asc', 'desc'. Default: descending order", "type": "string",
         "enum": ['asc', 'desc']},
        {"name": "from", "description": "printing range. Default: print everything", "type": "integer"},
        {"name": "count", "description": "printing range. Default: print everything", "type": "integer"},
        {"name": "human_readable", "description": "none description in readme.md", "type": "integer"},
        # Search filter:
        {"name": "switch_name", "description": "name of switch instance (string)", "type": "string"},
        {"name": "release_cause", "description": "release cause (integer)", "type": "string"},
        {"name": "orig_sip_resp", "description": "SIP response (integer)", "type": "string"},
        {"name": "term_sip_resp", "description": "SIP response (integer)", "type": "string"},
        {"name": "ingress_id", "description": "ingress id (integer)", "type": "string"},
        {"name": "egress_id", "description": "egress id (integer)", "type": "string"},
        {"name": "rate_table_id", "description": "ingress client rate table id (integer)", "type": "string"},
        {"name": "route_plan", "description": "routing plan (integer)", "type": "string"},
        {"name": "ingress_host", "description": "origination source host name (ip addr)", "type": "string"},
        {"name": "egress_host", "description": "termination destination host name (ip addr)", "type": "string"},
        {"name": "orig_code", "description": "originating code: 'country_name:code_name:code'", "type": "string"},
        {"name": "source_number", "description": "originating source number (string)", "type": "string"},
        {"name": "dest_number", "description": "originating destination numbers (string)", "type": "string"}
    ]

    def on_get(self, req, resp, **kwargs):
        self.init_req(req)
        if not check_context(self, req, resp, **kwargs):
            return False
        user = self.get_user(self.req)
        # params = req.data
        params = dict(parse_qsl(req.query_string))
        api = StatisticAPI
        try:
            ret = api.send_request(params=params, token=user.cdr_api_token)
            if ret:
                try:
                    def res_name(res_id):
                        if res_id:
                            res = model.Resource.get(res_id)
                            if res:
                                return res.alias
                        return None

                    d = json.loads(ret.content.decode())
                    if d['data']:
                        d['data'] = [dict(**i, ingress_name=res_name(i['ingress_id'])) if 'ingress_id' in i else i for i
                                     in d['data']]
                        d['data'] = [dict(**i, egress_name=res_name(i['egress_id'])) if 'egress_id' in i else i for i
                                     in d['data']]
                        resp.body = json.dumps(d)
                    else:
                        resp.body = ret.content
                    resp.status = falcon.HTTP_200
                except:
                    resp.body = ret.content
                    resp.status = falcon.HTTP_200
        # except StatisticException as e:
        # if '403' in str(e):
        # 	try:
        # 		user.cdr_token()
        # 		user.save()
        # 		ret = api.send_request(params=params, token=user.cdr_api_token)
        # 		if ret:
        # 			resp.body = ret.content
        # 			resp.status = falcon.HTTP_200
        # 	except StatisticException as e:
        # 		self.set_response(resp, StatisticErrorResponce(e))
        # 		return None
        # else:
        # 	self.set_response(resp, StatisticErrorResponce(e))
        # 	return None
        except Exception as e:
            self.set_response(resp, OperationErrorResponse(e))
            return None
        pass


class RtQosPostScheme(BaseModelScheme):
    # Required arguments:
    start_time = fields.Int()  # search from specified time (allow negative values)
    method = fields.String()  # aggregation method
    step = fields.String()  # granularity: "hour", "day", "week", "month", "year", "all", or amount of minutes. With custom step first timestamp will be Jan 1 1970 00:00.
    # Optional arguments:
    end_time = fields.Int()  # search till time (end_time > start_time). Default: 0 -> do not limit search by time
    group = fields.String()  # group results by value (switch_name, resource). Default: no grouping
    upd_time = fields.String()  # push updates interval in minutes. Default: 0 (do not send updates)
    tz = fields.String()  # timezone (in hours from UTC)
    sort = fields.String()  # sort results by field
    order = fields.String()  # sorting order: "asc", "desc". Default: descending order
    start_from = fields.Int()
    count = fields.Int()  # printing range. Default: print everything
    # Aggregation methods:
    #        max, min, avrg, total
    # Search filters and groups (search by ingress and egress at the same time is not allowed):
    switch_name = fields.String()  # switch instance name (string)
    ingress_id = fields.String()  # ingress id (integer)
    ingress_host = fields.String()  # ingress host (ip address)
    egress_id = fields.String()  # egress id (integer)
    egress_host = fields.String()  # egress host (ip address)


class RtQosPost(CustomPostAction):
    description = "### QOS Engine"
    body_parameters = ('QOS parameters', RtQosPostScheme,)

    def on_post(self, req, resp, **kwargs):
        self.init_req(req)
        if not check_context(self, req, resp, **kwargs):
            return False
        user = self.get_user(self.req)
        params = req.data
        if 'start_from' in params:
            params['from'] = params['start_from']
            del params['start_from']
        api = StatisticAPI
        try:
            ret = api.send_request(params=params, token=user.cdr_api_token, rt_api='qos', method='post')
            # if ret:
            #    resp.body=ret.content
            #    resp.status = falcon.HTTP_200
            if ret:
                from websocket import create_connection
                from time import sleep
                ws = create_connection('ws://localhost:{port}'.format_map(ret.json()))
                all_data = []
                while ws.connected:
                    item = ws.recv()
                    log.debug('qos websocket item received {}'.format(item))
                    if type(item) == type(b''):
                        item = item.decode('utf-8')
                    if item == '':
                        sleep(0.5)
                        continue
                    data = json.loads(item)
                    all_data.append(data)
                    sleep(0.1)
                log.debug('qos websocket all data {}'.format(all_data))
                resp.body = json.dumps({"items": all_data}).encode('utf-8')
                resp.status = falcon.HTTP_200
        except StatisticException as e:
            res = e.args[0]
            self.set_response(resp, responses.OperationErrorResponse(
                data=dict(code=res['code'], reason=res['msg'], message=res['data'])))
            # resp.status = str(code)
            return None
        except Exception as e:
            self.set_response(resp, OperationErrorResponse(e))
            return None
        pass


class RtWsHelper(CustomGetAction):
    description = "### QOS Engine websocket helper"
    path_parameters = [
        # Required arguments:
        {"name": "port", "description": "port number from qos engine", "type": "string"},
    ]

    def on_get(self, req, resp, **kwargs):
        self.init_req(req)
        if not check_context(self, req, resp, **kwargs):
            return False
        user = self.get_user(self.req)
        # params = req.data
        # params = dict(parse_qsl(req.query_string))
        params = kwargs
        try:

            resp.status = falcon.HTTP_200
        except StatisticException as e:
            res = e.args[0]
            self.set_response(resp, responses.OperationErrorResponse(
                data=dict(code=res['code'], reason=res['msg'], message=res['data'])))
            # resp.status = str(code)
            return None
        except Exception as e:
            self.set_response(resp, OperationErrorResponse(e))
            return None


class RtPortStatusScheme(BaseModelScheme):
    PORT_STATUSES = ('ok', 'closed', 'test failed')
    auth_port = fields.Str(validate=validate.OneOf(PORT_STATUSES))
    cdr_port = fields.Str(validate=validate.OneOf(PORT_STATUSES))
    async_port = fields.Str(validate=validate.OneOf(PORT_STATUSES))
    port = fields.Str(validate=validate.OneOf(PORT_STATUSES))
    qos_port = fields.Str(validate=validate.OneOf(PORT_STATUSES))
    ftp_port = fields.Str(validate=validate.OneOf(PORT_STATUSES))


class RtPortStatus(CustomGetAction):
    description = "Configuration status report"
    additional_responses = (responses.SuccessResponseObjectInfo(payload_scheme=RtPortStatusScheme),)

    def on_get(self, req, resp, **kwargs):
        try:
            self.init_req(req)
            if not check_context(self, req, resp, **kwargs):
                return False
            user = self.get_user(self.req)
            # params = req.data
            # params = dict(parse_qsl(req.query_string))
            params = kwargs
            api = StatisticAPI()
            result = {}

            token = None
            exp = None
            try:
                result['auth_port'] = 'ok'
                token, exp = api.authenticate()
            except Exception as e:
                result['auth_port'] = 'closed'

            try:
                # rt_api == 'report':
                result['port'] = 'ok'
                params = dict(method='total',
                              start_time=int(datetime.utcnow().timestamp() - 1000),
                              field='ingress_cost', step='day',
                              end_time=int(datetime.utcnow().timestamp()) - 900
                              )
                ret = api.send_request(params=params, token=token, method='get',
                                       rt_api='report')
                log.debug(ret.content.decode())
                assert ('"data"' in ret.content.decode())

            except Exception as e:
                if 'Connection error' in str(e):
                    result['port'] = 'closed'
                else:
                    result['port'] = 'test failed'

            try:
                # rt_api == 'async':
                result['async_port'] = 'ok'
                params = dict(start_time=int(datetime.utcnow().timestamp() - 1000),
                              field='ingress_id',
                              end_time=int(datetime.utcnow().timestamp()) - 900
                              )
                ret = api.send_request(params=params, token=token, method='post', rt_api='async')
                log.debug(ret.content.decode())
                assert ('Async request successfully created' in ret.content.decode())
            except Exception as e:
                if 'Connection error' in str(e):
                    result['async_port'] = 'closed'
                else:
                    result['async_port'] = 'test failed'

            try:
                # rt_api == 'cdr':
                result['cdr_port'] = 'ok'
                params = dict(start_time=int(datetime.utcnow().timestamp() - 1000),
                              field='0,1,2',
                              end_time=int(datetime.utcnow().timestamp()) - 900,
                              format='csv'
                              )
                ret = api.send_request(params=params, token=token, method='get', rt_api='cdr')
                log.debug(ret.content.decode())
                assert ('start_time_of_date' in ret.content.decode())
            except Exception as e:
                if 'Connection error' in str(e):
                    result['cdr_port'] = 'closed'
                else:
                    result['cdr_port'] = 'test failed'

            try:
                # rt_api == 'ftp':

                params = dict(frequency=1,
                              ftp_url='ftp://localhost'
                              )
                ret = api.send_request(params=params, token=token, method='post', rt_api='ftp')
                rj = json.loads(ret.content.decode())
                job_id = rj['job_id']
                ret1 = api.send_request(params={}, cdr_request=job_id, token=token, method='delete', rt_api='ftp')
                log.debug(ret.content.decode())
                assert ('Scheduled job successfully created' in ret.content.decode())
            except Exception as e:
                if 'Connection error' in str(e):
                    result['ftp_port'] = 'closed'
                else:
                    result['ftp_port'] = 'test failed'
            try:
                # rt_api == 'qos':
                result['qos_port'] = 'ok'

                params = dict(start_time=int(datetime.utcnow().timestamp() - 1000),
                              method='total',
                              count=1,
                              step='day',
                              end_time=int(datetime.utcnow().timestamp()) - 900
                              )
                ret = api.send_request(params=params, token=token, method='post', rt_api='qos')
                log.debug(ret.content.decode())
                assert ('Statistic request successfully created' in ret.content.decode())
            except StatisticException as e:
                if 'Connection error' in str(e):
                    result['qos_port'] = 'closed'
                else:
                    result['qos_port'] = 'test failed'
            log.debug(json.dumps({'payload': result}).encode('utf-8'))
            resp.body = json.dumps({'payload': result}).encode('utf-8')
            resp.status = falcon.HTTP_200

        except Exception as e:
            self.set_response(resp, OperationErrorResponse(e))
            return None


class VersionInfoGetScheme(BaseModelScheme):
    version = fields.Str()
    minor = fields.Str()
    build_date = fields.Str()
    author = fields.Str()
    title = fields.Str()


class VersionInfoGet(CustomGetAction):
    description = "get version"
    additional_responses = (responses.SuccessResponseObjectInfo(payload_scheme=VersionInfoGetScheme),)
    no_auth_needed = True

    def on_get(self, req, resp, **kwargs):
        try:
            from ..__version__ import __version__ as v, __minor__ as m, __date__ as d
            from .. import settings
            result = dict(version=v, minor=m, build_date=d,
                          author='novvvster@gmail.com',
                          title=settings.API_TITLE)
            resp.body = json.dumps({'payload': result}).encode('utf-8')
            resp.status = falcon.HTTP_200
        except Exception as e:
            self.set_response(resp, OperationErrorResponse(e))
            return None
