import json
import os
import sys
import string

import jwt
import requests
from urllib.parse import urlencode, unquote, quote_plus
from api_dnl.__version__ import __version__
from datetime import date, datetime, timedelta
from pytz import UTC
from time import mktime
from dateutil.parser import parse as parse_datetime
import calendar
from traceback import format_exc
import bcrypt
from urllib3.util import parse_url
from marshmallow import ValidationError
from inspect import isclass
from sqlalchemy import (Column, event, distinct, desc, or_, not_, and_, text as text_, PrimaryKeyConstraint, inspect,
                        Index, UniqueConstraint, literal, ForeignKey, CheckConstraint)
from sqlalchemy import (
    Integer, SmallInteger, Float, Text, String, DateTime, Date, Time, Boolean, ForeignKey, BigInteger, Enum,
    Table, Sequence, ARRAY, FetchedValue, CHAR, JSON
)

from sqlalchemy.dialects import postgresql
from sqlalchemy.dialects.postgresql import array, array_agg
from sqlalchemy.ext.hybrid import hybrid_property
from sqlalchemy.ext.mutable import MutableDict
from sqlalchemy.exc import IntegrityError
from psycopg2.errors import UniqueViolation
from sqlalchemy.orm import backref, foreign, relationship, synonym, column_property, aliased, validates
from sqlalchemy.orm import deferred
from sqlalchemy.orm.exc import NoResultFound
from sqlalchemy.sql import func, select, literal_column, case

# from falcon_rest.db import Column
from api_dnl.fields import ChoiceType, PrefixRange, Ip4, Ip4r

from api_dnl.models.user import User, WebSession, UserAuthIp, UserCodedeckAlerts, UserApiKey
from api_dnl.utils.statisticapi2 import IStatistic, StatisticAPI
from api_dnl.utils.statisticapi2 import cdr_fields, cdr_names
from api_dnl.base_model import DnlApiBaseModel, init_db
from falcon_rest.db.fields import (Numeric)
from falcon_rest.db.orm import generate_uuid_str
from api_dnl.objects_history.object_revision.object_revision import ObjectRevisionModel, \
    ObjectRevisionRecordModel
from falcon_rest.logger import log

from api_dnl import settings
from api_dnl.models.role import *
from api_dnl.models.logs import *
from api_dnl.models.mail import *
from api_dnl.models.role import Role
from api_dnl.models.dnl_cloud_model import DnlCloudStorage, DnlCloudDbmanCfg, DnlCloudDownloaderCfg, DnlCloudFtpCfg, \
    DnlCloudGcloudCfg, DnlCloudSftpCfg, DnlCloudLog, DnlCloudSearchCfg, DnlCloudStatus
from api_dnl.utils.dnl_active_calls import DnlActiveCallSession
# from api_dnl.models.invoice_model import *

# from api_dnl.models.logs import
REV = ObjectRevisionModel

ObjectRevisionModel.user_name = column_property(select([User.name]). \
    where(User.user_id == ObjectRevisionModel.user_id).correlate_except(
    User))

XOR_KEY = 0xABCD
HEX_CHARS = string.digits + string.ascii_uppercase + string.ascii_lowercase
perm_map = {char: HEX_CHARS[(i * 7) % len(HEX_CHARS)] for i, char in enumerate(HEX_CHARS)}


def _encode_agent_id(agent_id):
    transformed_id = agent_id ^ XOR_KEY
    hex_str = hex(transformed_id)[2:]
    permuted_hex = ''.join(perm_map[char] for char in hex_str)
    return permuted_hex.zfill(5)

def _decode_agent_id(encoded_str):
    stripped_hex = encoded_str.lstrip('0')
    reverse_perm_map = {v: k for k, v in perm_map.items()}
    unpermuted_hex = ''.join(reverse_perm_map[char] for char in stripped_hex)
    transformed_id = int(unpermuted_hex, 16)
    return transformed_id ^ XOR_KEY

# ObjectRevisionModel.object_name = column_property(select([
#     case([(ObjectRevisionRecordModel.new_value.isnot(None), ObjectRevisionRecordModel.new_value),
#           (ObjectRevisionRecordModel.old_value.isnot(None), ObjectRevisionRecordModel.old_value)], else_=None)]). \
#                                                   where(
#     and_(ObjectRevisionRecordModel.object_revision_id == ObjectRevisionModel.id,
#          ObjectRevisionRecordModel.field_name == 'name')).limit(1).correlate_except(ObjectRevisionRecordModel))


def _init_db(db_label='default', declarative_base=DnlApiBaseModel):
    from falcon_rest.db import initialize_db, _DB, _databases
    from .settings import DB_CONN_STRING
    import sqlalchemy
    # initialize_db(DB_CONN_STRING, DnlApiBaseModel)

    # conn_string = settings.DB_CONN_STRING
    # log.info('DB [{}]: Initializing DB, label {}'.format(db_label, db_label))
    # print('DB [{}]: Initializing DB, label {}'.format(db_label, db_label))
    # _databases[db_label] = _DB(declarative_base, db_label)
    # _databases[db_label].engine = sqlalchemy.create_engine(conn_string, pool_size=int(settings.DB_POOLSIZE))
    # _databases[db_label].create_session()
    return _databases[db_label]


from falcon_rest.db import get_db


def get_metadata():
    return get_db().declarative_base.metadata


def add_months(sourcedate, months):
    month = sourcedate.month - 1 + months
    year = sourcedate.year + month // 12
    month = month % 12 + 1
    day = min(sourcedate.day, calendar.monthrange(year, month)[1])
    return date(year, month, day)


"""
class ChoiceType(types.TypeDecorator):

    impl = Integer

    def __init__(self, choices, **kw):
        self.choices = dict(choices)
        super(ChoiceType, self).__init__(**kw)

    def process_bind_param(self, value, dialect):
        return [k for k, v in self.choices.iteritems() if v == value][0]

    def process_result_value(self, value, dialect):
        return self.choices[value]
"""


def query_to_sting(q):
    try:
        return str(q.statement.compile(dialect=postgresql.dialect(), compile_kwargs={"literal_binds": True}))
    except:
        return str(q)


def rev(d):
    return dict((v, k) for k, v in d.items())


def _ref_id(cls_name, field, value):
    if type(value) == type(''):
        value = "'" + value + "'"
    mod = sys.modules['api_dnl.model']
    cls = mod.__dict__[cls_name]
    pk_field_name = cls.get_model_class_primary_key()
    pk_field = cls.get_field(pk_field_name)
    # val = cls.filter(text_('{}={}'.format(field, value))).one().get_field(pk_field_name)
    r = cls.filter(text_('{}={}'.format(field, value))).first()
    if r:
        return getattr(r, pk_field_name)
    else:
        raise ValidationError({field: '{} is invalid (not exists)'.format(value)})


def dig(l=12):
    import random
    letters = [chr(i) for i in range(48, 58)]
    ret = ''
    for c in range(0, l):
        ret = ret + random.choice(letters)
    return ret


default_billing_decimal = None


def get_default_billing_decimal():
    global default_billing_decimal
    if default_billing_decimal is None:
        from decimal import getcontext
        config = SystemParameter.get(1)
        default_billing_decimal = config.default_billing_decimal
        if default_billing_decimal is None:
            default_billing_decimal = 2
    # getcontext().prec = default_billing_decimal
    return default_billing_decimal


def get_time_zone_hour(time_zone):
    if '(' in time_zone:
        time_zone = time_zone.replace('(', '').replace(')', '')
    sign = time_zone[0]
    hour = int(time_zone[1:3])
    mins = int(time_zone[4:])
    mins = (10 / 6) * mins
    if sign == "+":
        hour = hour
    else:
        hour = -1 * hour
    return hour


def get_time_zone(str_time_zone):
    from datetime import timezone
    if str_time_zone is None:
        str_time_zone = '+00:00'
    h = get_time_zone_hour(str_time_zone)
    return timezone(timedelta(hours=h))


def dec_prn(v, d=None):
    if d is None:
        d = get_default_billing_decimal()
    if d is None or d not in range(0, 16):
        d = 2
    if type(v) == type(float):
        v = Decimal(v)
    if isinstance(v, Decimal):
        if v < 0:
            fmt = '({0:.' + str(d) + 'f})'
            return fmt.format(abs(v))
        fmt = '{0:.' + str(d) + 'f}'
        return fmt.format(v)
    return str(v)


def user_id_filter(cls, user_id):
    return or_(and_(User.user_id == user_id, User.user_type == 'admin'),
               and_(User.user_id == user_id, User.user_type == 'client', cls.client_id == User.client_id),
               cls.client_id.in_(select([UsersLimit.client_id]).where(
                   and_(User.user_id == user_id, User.user_type == 'client',
                        User.user_id == UsersLimit.user_id))),
               cls.client_id.in_(select([AgentClient.client_id]).where(
                   and_(User.user_id == user_id, User.user_type == 'agent', User.user_id == Agent.user_id,
                        Agent.agent_id == AgentClient.agent_id)))
               )


class PgStatActivity(DnlApiBaseModel):
    __tablename__ = 'pg_stat_activity'
    # info = dict(is_view=True)

    usename = Column(String())
    query_ = Column('query', String())
    pid = Column(Integer, primary_key=True)
    query_start = Column(DateTime())

    @hybrid_property
    def query_text(self):
        return self.query_


class Agent(DnlApiBaseModel):
    __tablename__ = 'agent'
    METHOD_TYPE_DICT = {0: 'By Profit', 1: 'By Revenue'}
    FREQUENCY_TYPE_DICT = {0: 'daily', 1: 'weekly', 2: 'monthly'}

    agent_id = Column(Integer, primary_key=True)
    agent_name = Column(String(100), unique=True, nullable=False)
    create_on = Column(DateTime(timezone=True), nullable=False, server_default=func.now())
    email = Column(String, nullable=False)
    commission = Column(Float)
    # method_type = Column(Integer, nullable=False)  # 0 - By Profit, 1 - By Revenue
    method_type = Column(ChoiceType(METHOD_TYPE_DICT), nullable=False)
    frequency_type = Column(ChoiceType(FREQUENCY_TYPE_DICT), nullable=False)  # 0 - Daily, 1 - Weekly, 2 - monthly
    user_id = Column(Integer)
    agent_role_id = Column(Integer)
    status = Column(Boolean, default=True)
    update_on = Column(DateTime(timezone=True), nullable=True, onupdate=func.now())
    update_by = Column(String(100), nullable=True)
    edit_permission = Column(Boolean, default=True)
    invoice_setting_id = Column(ForeignKey('invoice_settings.id', ondelete='CASCADE'), nullable=True)
    mail_sender_id = Column(ForeignKey('mail_sender.id', ondelete='CASCADE'), nullable=True)
    mail_template_title = Column(ForeignKey('mail_template.title', ondelete='CASCADE'), nullable=True)

    carriers = relationship('AgentClient', back_populates='agent', uselist=True, single_parent=True,
                            cascade="all, delete-orphan")
    user = relationship('User', primaryjoin='foreign(Agent.user_id) == User.user_id',
                        back_populates='agent', uselist=False, single_parent=True,
                        cascade="all, delete-orphan"
                        )

    active = synonym('status')

    @property
    def client_id_list(self):
        cl = AgentClient
        carriers = cl.session().query(cl.client_id).filter(cl.agent_id == self.agent_id).all()
        return [c.client_id for c in carriers]

    @property
    def res_id_list(self):
        res = Resource
        resources = res.session().query(res.resource_id).filter(res.client_id.in_(self.client_id_list)).all() 
        return [int(r.resource_id) for r in resources]

    @property
    def ingress_ids(self):
        res = Resource
        return ','.join([str(r.resource_id) for r in res.filter(
            and_(res.client_id.in_([c.client_id for c in self.carriers]), res.ingress == True))])

    @property
    def egress_ids(self):
        res = Resource
        return ','.join([str(r.resource_id) for r in res.filter(
            and_(res.client_id.in_([c.client_id for c in self.carriers]), res.egress == True))])

    # usage_count = column_property
    @hybrid_property
    def user_name(self):
        return self.user.name

    def create_user(self):
        if not self.user:
            self.user = User(name=self.agent_name, passwd='agent')
            self.user.email = self.email
            self.user.user_type = 'agent'
            self.user.role_id = 2
            # self.user.client_id = self.client_id
            return self.user.save()
        else:
            return self.user_id
    
    @property
    def referal_key(self):
        return _encode_agent_id(self.agent_id)

    @property
    def referal_link(self):
        config = SystemParameter.get(1)
        return '{}/#/referral/{}'.format(config.root_url, self.referal_key)

    @classmethod
    def get_from_referral(cls, code):
        try:
            agent_id = _decode_agent_id(code)
            return cls.get(agent_id)
        except Exception as e:
            log.debug('cannot get agent form referal-key {} error is {}'.format(agent_id, str(e)))
            return None

    def after_save(self):
        for ac in AgentClient.filter(AgentClient.agent_id == self.agent_id).all():
            ac.client.save()


class AgentClient(DnlApiBaseModel):
    __tablename__ = 'agent_clients'
    __table_args__ = (UniqueConstraint('agent_id', 'client_id'),)
    METHOD_TYPE_DICT = {0: 'By Profit', 1: 'By Revenue'}

    id = Column(Integer, primary_key=True)
    agent_id = Column(ForeignKey('agent.agent_id', ondelete='CASCADE'), nullable=False)
    client_id = Column(ForeignKey('client.client_id', ondelete='CASCADE'), nullable=False)
    commission = Column(Numeric(5, 2))
    assigned_on = Column(DateTime(timezone=True), server_default=func.now())
    assigned_by = Column(String(100))
    method_type = Column(ChoiceType(METHOD_TYPE_DICT), nullable=False, default=0)

    agent = relationship(Agent, back_populates='carriers', uselist=False)
    client = relationship('Client', back_populates='agents', uselist=False)

    registered_on = synonym('assigned_on')

    agent_name = column_property(
        select([Agent.agent_name]).where(Agent.agent_id == agent_id).correlate_except(Agent))
    # commission = column_property(
    #     select([Agent.commission]).where(Agent.agent_id == agent_id).correlate_except(Agent))

    def _method_type(self):
        return self.client.profit_type


Agent.usage_count = column_property(select([func.coalesce(func.count(AgentClient.client_id), text_("0"))]). \
    where(AgentClient.agent_id == Agent.agent_id).correlate_except(AgentClient).label(
    'usage_count'))


class AgentClientClient(DnlApiBaseModel):
    __tablename__ = 'agent_client_client'

    id = Column(Integer, primary_key=True, server_default=text_("nextval('agent_client_client_id_seq'::regclass)"))
    agent_client_id = Column(Integer, nullable=False)
    client_id = Column(Integer, nullable=False, unique=True)
    commission = Column(Integer)


class AgentCommissionHistory(DnlApiBaseModel):
    __tablename__ = 'agent_commission_history'

    history_id = Column(Integer, primary_key=True,
                        server_default=text_("nextval('agent_commission_history_history_id_seq'::regclass)"))
    start_time = Column(DateTime(True))
    end_time = Column(DateTime(True))
    total_date = Column(Integer)
    amount = Column(Numeric)
    agent_id = Column(ForeignKey('agent.agent_id', ondelete='CASCADE'))
    create_date = Column(Date)
    finished = Column(Boolean)
    commission = synonym('amount')

    agent = relationship('Agent')
    details = relationship('AgentCommissionHistoryDetail', back_populates='parent', uselist=False)


class AgentCommissionHistoryDetail(DnlApiBaseModel):
    __tablename__ = 'agent_commission_history_detail'

    id = Column(Integer, primary_key=True,
                server_default=text_("nextval('agent_commission_history_detail_id_seq'::regclass)"))
    history_id = Column(ForeignKey('agent_commission_history.history_id', ondelete='CASCADE'))
    start_time = Column(DateTime(True))
    end_time = Column(DateTime(True))
    total_date = Column(Integer)
    commission = Column(Numeric)
    client_id = Column(ForeignKey('client.client_id', ondelete='CASCADE'))
    client_cost = Column(Numeric)

    client = relationship('Client')
    parent = relationship('AgentCommissionHistory', back_populates='details', uselist=False)

    def agent_name(self):
        return self.parent.agent_id

    def agent_name(self):
        return self.parent.agent.agent_name

    def client_name(self):
        return self.client.name


AgentCommissionHistory.client_id = column_property(select([AgentCommissionHistoryDetail.client_id]). \
    where(
    AgentCommissionHistoryDetail.history_id == AgentCommissionHistory.history_id).correlate_except(
    AgentCommissionHistoryDetail))


class AgentCommissionPayment(DnlApiBaseModel):
    __tablename__ = 'agent_commission_payment'

    id = Column(Integer, primary_key=True, server_default=text_("nextval('agent_commission_payment_id_seq'::regclass)"))
    agent_id = Column(ForeignKey('agent.agent_id', ondelete='CASCADE'))
    create_on = Column(DateTime(True))
    create_by = Column(String(100))
    amount = Column(Numeric)
    note = Column(Text)
    start_time = Column(DateTime(True))
    end_time = Column(DateTime(True))
    agent_name = column_property(
        select([Agent.agent_name]).where(Agent.agent_id == agent_id).correlate_except(Agent))
    commission = column_property(
        select([Agent.commission]).where(Agent.agent_id == agent_id).correlate_except(Agent))


class Currency(DnlApiBaseModel):
    __tablename__ = 'currency'

    currency_id = Column(Integer, primary_key=True)
    code = Column(String(100), nullable=False, unique=True)
    # rate = Column(Float, nullable=False, default=1)#, server_default=func.text_('1.0'))
    active = Column(Boolean, nullable=False, default=True)
    update_by = Column(String)
    rates = relationship('CurrencyUpdates', back_populates='currency', uselist=True, single_parent=True,
                         cascade="all, delete-orphan")

    @hybrid_property
    def rate(self):
        try:
            rate = CurrencyUpdates.filter(CurrencyUpdates.currency_id == self.currency_id). \
                filter(CurrencyUpdates.currency_updates_id == CurrencyUpdates.query().
                       session.query(func.max(CurrencyUpdates.currency_updates_id)).filter(
                CurrencyUpdates.currency_id == self.currency_id)).first().rate
            return rate
        except:
            return None

    @rate.setter
    def rate(self, value):
        try:
            if not value is None and not self.currency_id is None:
                r = CurrencyUpdates(currency_id=self.currency_id, rate=value)
                self.rates.append(r)
        except:
            pass


# def __repr__(self):
#    return str(self.currency_id) + ':' + self.code


# Currency.names = Currency.query().session.query(Currency.code).all()
def validate_currency(cur):
    if sys.modules['model'].Currency.filter(Currency.code == cur).count() == 0:
        raise ValidationError('currency name is invalid.')


class CurrencyUpdates(DnlApiBaseModel):
    __tablename__ = 'currency_updates'
    currency_updates_id = Column(Integer, primary_key=True)
    currency_id = Column(ForeignKey('currency.currency_id', ondelete='SET NULL'), nullable=False, index=True)
    rate = Column(Float, nullable=False, default=1)
    currency = relationship('Currency', back_populates='rates', uselist=False)


class PaymentTerm(DnlApiBaseModel):
    __tablename__ = 'payment_term'
    DAYS_OF_WEEK = {1: "Monday", 2: "Tuesday", 3: "Wednesday", 4: "Thursday", 5: "Friday", 6: "Saturday", 7: "Sunday"}
    TYPE_DICT = {1: 'Every', 2: 'Day of Month', 3: 'Day of Week', 4: 'Some day of month', 5: 'Twice a month'}
    payment_term_id = Column(Integer, primary_key=True)
    name = Column(String(100), nullable=False, unique=True)
    type = Column(ChoiceType(TYPE_DICT), nullable=False)
    days = Column(Integer, nullable=False, default=1)
    grace_days = Column(Integer)
    notify_days = Column(Integer)
    more_days = Column(String(100))
    finance_rate = Column(Float)

    grace_period = synonym('grace_days')
    type_name = synonym('type')

    clients = relationship('Client', back_populates='payment_term', uselist=True)

    @property
    def cycle(self):
        try:
            ordinal = lambda n: "%d%s" % (n, "tsnrhtdd"[(n / 10 % 10 != 1) * (n % 10 < 4) * n % 10::4])
            if self.type == 'Day of Month':
                return 'Every {} of the month'.format(ordinal(int(self.days)))
            if self.type == 'Every':
                return 'Every {} day(s)'.format(ordinal(int(self.days)))
            if self.type == 'Day of Week':
                return 'Every {} of the week'.format(self.DAYS_OF_WEEK[self.days])
            if self.type == 'Some day of month':
                return 'At {} days of month'.format(','.join([ordinal(int(d)) for d in self.more_days_arr]))
            if self.type == 'Twice a month':
                return 'Twice in a month (1st and the 16th of the month)'
        except Exception as e:
            return self.type + ' (not completed)'
        return None

    def due_date(self, d):
        nx = self.next_date(d)
        return nx + timedelta(days=self.grace_days)

    def next_date(self, d):
        now = datetime.now(UTC).date()
        return self.next_date_for_now(d, now)

    def next_date_for_now(self, d, now):
        from datetime import date

        if isinstance(d, type('')):
            d = parse_datetime(d).date()
        elif isinstance(d, datetime):
            d = date(d.year, d.month, d.day)
        elif isinstance(d, date):
            d = date(d.year, d.month, d.day)
        elif d is None:
            d = date(2001, 9, 11)
        else:
            raise Exception("next_date:wrong input data type {}".format(d))

        if not d or d > now:
            d = now
        if self.type == 'Day of Month':
            next = date(d.year, d.month, 1) + timedelta(days=self.days - 1)
            if next <= d:
                next = add_months(next, 1)
            while next < now:
                next = add_months(next, 1)
            return next
        if self.type == 'Every':
            next = d + timedelta(days=int(self.days))
            while next < now:
                next = next + timedelta(days=self.days)
            return next
        if self.type == 'Day of Week':
            days_ahead = (self.days - 1) - d.weekday()
            if days_ahead <= 0:
                days_ahead += 7
            next = d + timedelta(days=days_ahead)
            while next < now:
                next = next + timedelta(days=7)
            return next
        if self.type == 'Some day of month':
            days = sorted([int(i) for i in self.more_days_arr])
            next = date(d.year, d.month, days[0])
            while next <= d or next <= now:
                for day in days:
                    next = date(next.year, next.month, day)
                    if next > d:# and next > now:
                        return next
                next = date(next.year, next.month, days[0])
                next = add_months(next, 1)
            return next
        if self.type == 'Twice a month':
            days = sorted([1, 16])
            next = date(d.year, d.month, days[0])
            while next <= d or next <= now:
                for day in days:
                    next = date(next.year, next.month, day)
                    if next > d:# and next > now:
                        return next
                next = date(next.year, next.month, days[0])
                next = add_months(next, 1)
            return next

        return None

    @hybrid_property
    def more_days_arr(self):
        try:
            return sorted([int(i) for i in set([str(self.days)] + self.more_days.split(',')) if i != ''])
        except:
            return []

    @more_days_arr.setter
    def more_days_arr(self, value):
        self.days = value[0]
        # raise Exception(value)
        self.more_days = ','.join([str(s) for s in value[1:]])


# @hybrid_property
# def usage_count(self):
#    return len(self.clients)  # self.query().session.query('client').filter('client.payment_term_id=payment_term_id').count()

# def __repr__(self):
#    return self.name


class RouteStrategy(DnlApiBaseModel):
    __tablename__ = 'route_strategy'

    route_strategy_id = Column(Integer, primary_key=True)
    name = Column(String(256), nullable=False, unique=True)
    update_at = Column(DateTime(timezone=True), onupdate=func.now())
    update_by = Column(String)
    is_virtual = Column(Boolean)
    routes = relationship('Route', back_populates='route_strategy', uselist=True,
                          single_parent=True,
                          cascade="all, delete-orphan",
                          )

    route_plan_id = synonym('route_strategy_id')

    @hybrid_property
    def usage_count_(self):
        return self.routes.count()

    @classmethod
    def init(cls):
        q = cls.get(1)
        if not q:
            q = cls(route_strategy_id=1, name='ORIGINATION_ROUTING_PLAN', update_by='system')
            q.save()
        if not q.name == 'ORIGINATION_ROUTING_PLAN':
            q.name = 'ORIGINATION_ROUTING_PLAN'
            q.update_by = 'system'
            q.save()

    @staticmethod
    def get_orig_routing_plan_id():
        RouteStrategy.init()
        return 1


# def __repr__(self):
#    return self.name


class TimeProfile(DnlApiBaseModel):
    __tablename__ = 'time_profile'
    TYPE_DICT = {0: 'all time', 1: 'weekly', 2: 'daily'}
    DAYS_OF_WEEK = {1: "Monday", 2: "Tuesday", 3: "Wednesday", 4: "Thursday", 5: "Friday", 6: "Saturday", 7: "Sunday"}
    time_profile_id = Column(Integer, primary_key=True)
    name = Column(String(100), nullable=False, unique=True)
    start_time = Column(Time)
    end_time = Column(Time)
    start_week = Column(ChoiceType(DAYS_OF_WEEK))
    end_week = Column(ChoiceType(DAYS_OF_WEEK))
    type = Column(ChoiceType(TYPE_DICT), nullable=False)  # 0：all time;1：weekly;2：daily
    time_zone = Column(String(10))

    start_day_of_week = synonym('start_week')
    end_day_of_week = synonym('end_week')
    type_name = synonym('type')

    def before_save(self):
        if self.type == 'all time':
            self.start_time = None
            self.start_week = None
            self.end_time = None
            self.end_week = None
        if self.type == 'daily':
            self.start_week = None
            self.end_week = None

    def __repr__(self):
        return str(self.time_profile_id) + ':' + self.name + ' start ' + str(self.start_time) + ' end ' + \
               str(self.end_time) + ' start week ' + str(self.start_week) + ' end week' + str(
            self.end_week) + ' ' + str(self.type)


class DigitTranslation(DnlApiBaseModel):
    __tablename__ = 'digit_translation'

    translation_id = Column(Integer, primary_key=True)
    translation_name = Column(String(100), nullable=False, unique=True)
    translation = Column(DateTime(timezone=True), nullable=False, default=datetime.now(UTC))
    update_at = synonym('translation')
    digit_map_name = synonym('translation_name')

    items = relationship('TranslationItem', back_populates='translation', uselist=True)
    trunks = relationship('ResourceTranslationRef', back_populates='translation', uselist=True)

    # @hybrid_property
    def digit_map_count(self):
        return TranslationItem.filter(translation_id=self.translation_id).count()

    def __repr__(self):
        return str(self.translation_id) + ':' + self.translation_name


class TranslationItem(DnlApiBaseModel):
    # Master called conversion
    __tablename__ = 'translation_item'
    METHOD_DICT = {0: 'ignore', 1: 'partial replacement', 2: 'all replace'}
    ref_id = Column(Integer, primary_key=True)
    translation_id = Column(ForeignKey('digit_translation.translation_id', ondelete='CASCADE'), nullable=False,
                            index=True)  # number conversion
    ani = Column(PrefixRange, default='')  # prefix_range caller
    dnis = Column(PrefixRange, default='')  # prefix_range called
    action_ani = Column(String(30), nullable=False)  # after conversion of the caller
    action_dnis = Column(String(30), nullable=False)  # converted after the call
    ani_method = Column(ChoiceType(METHOD_DICT), nullable=False)  # 0 - ignore; 1 - partial replacement; 2 - all replace
    dnis_method = Column(ChoiceType(METHOD_DICT),
                         nullable=False)  # 0 - ignore; 1 - partial replacement; 2 - all replace

    ANI_replace_to = synonym('action_ani')
    DNIS_replace_to = synonym('action_dnis')
    ANI_prefix = synonym('ani')
    DNIS_prefix = synonym('dnis')

    translation = relationship('DigitTranslation', back_populates='items', uselist=False)

    # @hybrid_property
    # def digit_map_name(self): return self.translation.translation_name
    digit_map_name = column_property(select([DigitTranslation.translation_name]). \
                                     where(DigitTranslation.translation_id == translation_id). \
                                     correlate_except(DigitTranslation))


class ResourceTranslationRef(DnlApiBaseModel):
    __tablename__ = 'resource_translation_ref'
    # different time periods, different master calls are converted
    ref_id = Column(Integer, primary_key=True)
    resource_id = Column(ForeignKey('resource.resource_id'), index=True)  # gateway group - foreign key; docking gateway
    translation_id = Column(ForeignKey('digit_translation.translation_id'),
                            index=True)  # conversion rules - foreign key
    time_profile_id = Column(ForeignKey('time_profile.time_profile_id'), index=True)  # time period - foreign key

    gateway = relationship('Resource', back_populates='digit_maps', uselist=False)
    translation = relationship('DigitTranslation', back_populates='trunks', uselist=False)
    time_profile = relationship('TimeProfile', uselist=False)

    @hybrid_property
    def digit_map_name(self):
        return self.translation.translation_name


class PartitionGatewayRef(DnlApiBaseModel):
    __tablename__ = 'partition_gateway_ref'
    id = Column(Integer, primary_key=True, server_default=text_("nextval('partition_gateway_ref_id_seq'::regclass))"))
    partition_id = Column(Integer)
    gateway_ip = Column(Ip4)
    gateway_port = Column(Integer)


from api_dnl.utils.dnl_switch import CallApi, DnlSwitchSession, get_dnl_switch_session


class VoipGateway(DnlApiBaseModel):
    __tablename__ = 'voip_gateway'
    __table_args__ = (UniqueConstraint('lan_ip', 'lan_port'),)
    id = Column(Integer, primary_key=True)
    name = Column(String(200), unique=True)
    paid_replace_ip = Column(Integer, default=0)
    lan_ip = Column(String(255))
    lan_port = Column(Integer)
    active_call_ip = Column(String(64))
    active_call_port = Column(Integer)
    sip_capture_ip = Column(String(255))
    sip_capture_port = Column(Integer)
    sip_capture_path = Column(String(255))

    self_cps = Column(Integer)
    self_cap = Column(Integer)
    cps = synonym('self_cps')
    cap = synonym('self_cap')

    telnet_port = synonym('lan_port')
    telnet_ip = synonym('lan_ip')
    switch_id = synonym('id')

    sip_host = relationship('SwitchProfile', uselist=True, back_populates='voip_gateway',
                            single_parent=True,
                            cascade="all, delete-orphan"
                            )

    info = relationship('VoipGatewayInfo', uselist=False, single_parent=True, cascade="all, delete-orphan")
    version_info = relationship('VersionInformation',
                                primaryjoin='VersionInformation.switch_name==foreign(VoipGateway.name)')

    @classmethod
    def expired7(cls):
        ol = cls.filter(cls.expire < int((datetime.now() - timedelta(days=7)).timestamp())).all()
        return [dict(name=i.name, lan_ip=i.lan_ip, expire=i.expire) for i in ol]

    def check_info(self):
        if not self.info:
            log.warning('Check active switch:VoipGatewayInfo {} created'.format(self.id))
            self.info = VoipGatewayInfo(id=self.id)
            try:
                self.info.check_info()
                log.warning('Check active switch:VoipGatewayInfo first created')
            except:
                pass
            self.info.save()

    @hybrid_property
    def _connected(self):
        # if self.info:
        #     self.info.check_connected()
        # if self.connected:
        #     return self.connected
        try:
            api = DnlSwitchSession(self.lan_ip, self.lan_port, timeout=5)
            api.login()
            return True
        except Exception as e:
            log.debug('error when tring to connect {}'.format(e))
            return False

    @property
    def _limits(self):
        s = None
        try:
            s = get_dnl_switch_session(self.lan_ip, self.lan_port)
            ret = s.get_license_limit()
            self._limits_ = ret
            return ret
        except Exception as e:
            return {'result': str(e)}

    # return not self.active_call_ip is None

    @property
    def _sip_profile_show(self):
        s = None
        try:
            s = get_dnl_switch_session(self.lan_ip, self.lan_port)
            ret = s.sip_profile_show()
            self._sip_profile_show_ = ret
            return ret
        except Exception as e:
            if s:
                s.logout()
            return {'result': str(e)}

    # return not self.active_call_ip is None

    @property
    def _get_disk_info(self):
        s = None
        try:
            s = get_dnl_switch_session(self.lan_ip, self.lan_port)
            ret = s.get_disk_info()
            self._get_disk_info_ = ret
            return ret
        except Exception as e:
            return {'result': str(e)}

    def delete(self, **kwargs):
        SwitchProfile.filter(SwitchProfile.voip_gateway_id == self.id).delete(synchronize_session='fetch')
        ret = super(VoipGateway, self).delete(**kwargs)
        return ret

    @classmethod
    def get_objects_list(cls, query=None, filtering=None, paging=None,
                        ordering=None, query_only=False):
        from sqlalchemy import alias, cast, String
        from api_dnl.base_model import LIST_DEBUG, _q_str, get_count

        log.debug("qs = query or cls.query()")
        qs = query or cls.query()
        log.debug("GOT QS")
        non_column_filters = {}

        if filtering:
            log.debug("STARTING FILTERING")
            for k, v in filtering.items():
                if k == "connected":
                    non_column_filters[k] = v
                    log.debug(f"Skipping filter for {k} as it is not a valid column of {cls}")
                    continue
                if type(v) == str and ('*' in v or '_' in v):  # pragma: no cover
                    try:
                        v = v.upper().replace('*', '%')
                        qs = qs.filter(func.upper(cast(getattr(cls, k), String)).like(v))
                    except Exception as e:
                        qs = qs.filter(getattr(cls, k) == v)
                else:
                    qs = qs.filter(getattr(cls, k) == v)
            log.debug("FINISHED FILTERING")

        if ordering:  # pragma: no cover
            log.debug("ORDERING")
            if ',' in ordering['by']:
                ordl = ordering['by'].split(',')
                dirl = ordering['dir'].split(',')
                ob = []
                for i, ord in enumerate(ordl):
                    order_by = getattr(cls, ord)
                    if dirl[i] == 'desc':
                        order_by = order_by.desc().nullslast()
                    ob.append(order_by)
                qs = qs.order_by(*ob)
            else:
                order_by = getattr(cls, ordering['by'])
                if ordering['dir'] == 'desc':
                    order_by = order_by.desc().nullslast()
                qs = qs.order_by(order_by)
            log.debug("FINISHED ORDERING")

        if non_column_filters:
            results = qs.all()
            log.debug("STARTING IN-MEMORY FILTERING")
            filtered_results = []
            for item in results:
                match = True
                for k, v in non_column_filters.items():
                    log.debug(f"in memory filtering of {k}")
                    attr_name = f"_{k}" if hasattr(item, f"_{k}") else k
                    attr_value = getattr(item, attr_name, None)
                    if isinstance(attr_value, bool):
                        v = v.lower() == 'true'
                        if attr_value != v:
                            match = False
                            break
                    elif isinstance(v, str) and ('*' in v or '_' in v):
                        pattern = v.replace('*', '.*').replace('_', '.')
                        if not re.match(pattern, str(attr_value)):
                            match = False
                            break
                    elif attr_value != v:
                        match = False
                        break
                if match:
                    filtered_results.append(item)
            results = filtered_results
            cnt = len(results)
            log.debug("FINISHED IN-MEMORY FILTERING")

            if paging:
                log.debug("APPLYING PAGINATION AFTER FILTERING")
                start = paging.get('from', 0)
                end = paging.get('till', cnt)
                results = results[start:end]
                log.debug("FINISHED PAGING AFTER FILTERING")
        else:
            try:
                log.debug("GETTING COUNT")
                cnt = get_count(qs)
            except Exception as e:
                log.debug(f"ERROR - {e}")
                cls.session().rollback()
                cnt = paging['till'] - paging['from'] if paging and 'from' in paging and 'till' in paging else 10
            if paging:
                log.debug("STARTING PAGING")
                if 'from' in paging:
                    qs = qs.offset(paging['from'])
                if 'till' in paging:
                    qs = qs.limit(paging['till'] - paging.get('from', 0))
                log.debug("FINISHED PAGING")
            results = qs.all()

        if LIST_DEBUG:
            try:
                log.debug(_q_str(qs))
            except:
                log.debug(str(qs))

        if query_only:
            return qs, cnt
        else:
            return results, cnt

    def _get_dnl_switch_session(self):
        """Get a DNL switch session for the given IP and port"""
        try:
            api = DnlSwitchSession(self.lan_ip, self.lan_port)
            api.login()
            return api
        except Exception as e:
            log.debug('error when tring to connect {}'.format(e))
            try:
                api = DnlSwitchSession(self.active_call_ip, self.active_call_port)
                api.login()
                return api
            except Exception as e:
                log.debug('error when tring to connect {}'.format(e))
                return None


class VoipGatewayInfo(DnlApiBaseModel):
    __tablename__ = 'voip_gateway_info'
    id = Column(ForeignKey('voip_gateway.id', ondelete='CASCADE'), primary_key=True)
    time = Column(DateTime(True), server_default=func.now())
    connected = Column(Boolean, default=False)
    limits = Column(JSON)
    sip_profile_show = Column(JSON)
    get_disk_info = Column(JSON)
    system_peak_stats = Column(JSON)
    system_call_stat = Column(JSON)
    switch = relationship('VoipGateway')

    def check_connected(self):
        try:
            s = get_dnl_switch_session(self.switch.lan_ip, self.switch.lan_port)
            self.connected = True
        except Exception as e:
            log.error('voip_gateway_info check_connected {}'.format(e))
            self.connected = False
        log.info('switch connected {}'.format(self.connected))

    def check_info(self):
        try:
            s = get_dnl_switch_session(self.switch.lan_ip, self.switch.lan_port)
            self.connected = True
            self.limits = s.get_license_limit()
            self.sip_profile_show = s.sip_profile_show()
            self.get_disk_info = s.get_disk_info()
            self.system_call_stat = s.get_system_call_stat()
            self.system_peak_stats = s.get_system_peak_stat()
            self.time = datetime.now(UTC)
        except Exception as e:
            self.connected = False


VoipGateway.expire = column_property(
    select([text_("(voip_gateway_info.limits->>'expire')::bigint as expire")]).where(
        VoipGatewayInfo.id == VoipGateway.id).correlate_except(
        VoipGatewayInfo))
VoipGateway.connected = column_property(
    select([VoipGatewayInfo.connected]).where(VoipGatewayInfo.id == VoipGateway.id).correlate_except(VoipGatewayInfo))
VoipGateway.limits = column_property(
    select([VoipGatewayInfo.limits]).where(VoipGatewayInfo.id == VoipGateway.id).correlate_except(VoipGatewayInfo))
VoipGateway.sip_profile_show = column_property(
    select([VoipGatewayInfo.sip_profile_show]).where(VoipGatewayInfo.id == VoipGateway.id).correlate_except(
        VoipGatewayInfo))
VoipGateway.get_disk_info = column_property(
    select([VoipGatewayInfo.get_disk_info]).where(VoipGatewayInfo.id == VoipGateway.id).correlate_except(
        VoipGatewayInfo))
VoipGateway.system_call_stat = column_property(
    select([VoipGatewayInfo.system_call_stat]).where(VoipGatewayInfo.id == VoipGateway.id).correlate_except(
        VoipGatewayInfo))
VoipGateway.system_peak_stats = column_property(
    select([VoipGatewayInfo.system_peak_stats]).where(VoipGatewayInfo.id == VoipGateway.id).correlate_except(
        VoipGatewayInfo))


class C4LivecallUser(DnlApiBaseModel):
    __tablename__ = 'c4_livecall_user'
    USER_TYPE = {0: 'softswitch', 1: 'web user'}
    id = Column(Integer, primary_key=True, server_default=text_("nextval('c4_livecall_user_id_seq'::regclass)"))
    name = Column(String(50), nullable=False, index=True)
    password = Column(String(50), nullable=False)
    user_type = Column(ChoiceType(USER_TYPE), nullable=False, server_default=text_("0"))


class C4Lrn(DnlApiBaseModel):
    __tablename__ = 'c4_lrn'

    id = Column(Integer, primary_key=True, server_default=text_("nextval('c4_lrn_id_seq'::regclass)"))
    srv1_ip = Column(String(16))
    srv1_port = Column(Integer, nullable=False, server_default=text_("5060"))
    srv2_ip = Column(String(16))
    srv2_port = Column(Integer, nullable=False, server_default=text_("5060"))
    timeout = Column(Integer, server_default=text_('3'))
    retry_attempts = Column(Integer, server_default=text_('1'))


class SwitchProfile(DnlApiBaseModel):
    __tablename__ = 'switch_profile'
    __table_args__ = (Index('switch_profile_sip_ip_sip_port_idx',
                            'sip_ip', 'sip_port', unique=True),)
    PROFILE_STATUS = {0: 'disabled', 1: 'connected'}

    id = Column(Integer, primary_key=True)
    voip_gateway_id = Column(ForeignKey('voip_gateway.id', ondelete='CASCADE'), index=True)
    switch_name = Column(String(100))
    profile_name = Column(String(100))
    sip_ip = Column(String(64))
    sip_port = Column(Integer)
    sip_debug = Column(Integer, default=0)
    sip_trace = Column(Boolean, nullable=False, default=False)
    proxy_ip = Column(String(64))
    proxy_port = Column(Integer)
    support_rpid = Column(Integer, nullable=False, default=0)
    support_oli = Column(Integer, nullable=False, default=0)
    support_priv = Column(Integer, nullable=False, default=0)
    support_div = Column(Integer, nullable=False, default=0)
    support_paid = Column(Integer, nullable=False, default=0)
    support_pci = Column(Integer, nullable=False, default=0)
    support_x_lrn = Column(Integer, nullable=False, default=1)
    support_x_header = Column(Integer, nullable=False, default=1)
    sip_capture_ip = Column(String(255))
    sip_capture_port = Column(Integer)
    sip_capture_path = Column(String(255))
    lan_ip = Column(String(255))
    lan_port = Column(Integer)
    profile_status = Column(ChoiceType(PROFILE_STATUS))
    paid_replace_ip = Column(Integer, default=0)
    auth_register = Column(Integer)
    default_register = Column(Integer)
    report_ip = Column(String(100))
    report_port = Column(Integer, default=3300)
    active_call_ip = Column(String(64))
    active_call_port = Column(Integer)
    cps = Column(Integer)
    cap = Column(Integer)
    pcap_token = Column(String(255))
    last_started_on = Column(DateTime())

    status = synonym('profile_status')
    cli_ip = synonym('sip_ip')
    cli_port = synonym('sip_port')
    switch_id = synonym('voip_gateway_id')
    host_name = synonym('profile_name')

    server_name = column_property(select([VoipGateway.name]). \
                                  where(voip_gateway_id == VoipGateway.switch_id).correlate_except(VoipGateway))

    voip_gateway = relationship(VoipGateway, uselist=False, back_populates='sip_host')
    egress = relationship('EgressProfile', uselist=False, back_populates='profile')

    cps = column_property(select([VoipGateway.cps]).where(VoipGateway.id == voip_gateway_id).correlate_except(VoipGateway))
    cap = column_property(select([VoipGateway.cap]).where(VoipGateway.id == voip_gateway_id).correlate_except(VoipGateway))
    self_cps = column_property(select([VoipGateway.self_cps]).where(VoipGateway.id == voip_gateway_id).correlate_except(VoipGateway))
    self_cap = column_property(select([VoipGateway.self_cap]).where(VoipGateway.id == voip_gateway_id).correlate_except(VoipGateway))

    def before_save(self):
        # return
        try:
            api = DnlActiveCallSession(self.voip_gateway.active_call_ip, self.voip_gateway.lan_port)
            api.login()
            api.update_system_limit(self.self_cap, self.self_cps)
        except Exception as e:
            log.debug('Error when saving SwitchProfile {} {}'.format(self.id, str(e)))

    def expiration_date(self):
        try:
            return self.pcap_token
        except:
            return ''

    @property
    def limits(self):
        try:
            # if self.voip_gateway.info:
            #     self.voip_gateway.info.check_info()
            return self.voip_gateway.limits
        except:
            return {}

    @property
    def _status(self):
        if self.voip_gateway and self.switch_name != self.voip_gateway.name:
            return 'disabled'
        return self.status

class EgressProfile(DnlApiBaseModel):
    __tablename__ = 'egress_profile'
    __table_args__ = (Index('egress_profile_egress_ingress_profile_idx',
                            'server_name', 'egress_id', 'ingress_id', unique=True),)
    id = Column(Integer, primary_key=True)
    egress_id = Column(ForeignKey('resource.resource_id', ondelete='CASCADE'))
    profile_id = Column(ForeignKey('switch_profile.id', ondelete='CASCADE'))
    server_name = Column(String(200))
    ingress_id = Column(Integer)

    resource_id = synonym('egress_id')

    egress = relationship('EgressTrunk', uselist=False, back_populates='profiles')
    profile = relationship(SwitchProfile, uselist=False, back_populates='egress')
    switch_name = column_property(select([VoipGateway.name]). \
        where(
        and_(SwitchProfile.id == profile_id, VoipGateway.switch_id == SwitchProfile.switch_id)).correlate_except(
        SwitchProfile))
    # _server_name = column_property(
    #     select([func.concat(SwitchProfile.sip_ip, ':', cast(SwitchProfile.sip_port, String))]). \
    #         where(profile_id == SwitchProfile.id).correlate_except(SwitchProfile))
    _server_name = column_property(
        select([SwitchProfile.switch_name]). \
            where(profile_id == SwitchProfile.id).correlate_except(SwitchProfile))
    profile_sip_ip = column_property(
        select([SwitchProfile.sip_ip]). \
            where(profile_id == SwitchProfile.id).correlate_except(SwitchProfile))
    profile_sip_port = column_property(
        select([SwitchProfile.sip_port]). \
            where(profile_id == SwitchProfile.id).correlate_except(SwitchProfile))

    def after_save(self):
        if self.server_name != self._server_name:
            self.server_name = self._server_name
            self.save()


# EgressProfile._server_name = column_property(select([VoipGateway.name]).where(and_(SwitchProfile.id == EgressProfile.profile_id,VoipGateway.switch_id==SwitchProfile.switch_id)).correlate_except(SwitchProfile))

class ProductRoutRateTable(DnlApiBaseModel):
    __tablename__ = 'product_rout_rate_table'
    # __table_args__ = (UniqueConstraint('rate_table_id', 'rout_id'),)

    STATUS = {0: 'Inactive', 1: 'Active'}
    TYPE = {0: 'public', 1: 'private'}
    MARKETPLACE = {0: 'A-Z', 1: 'US'}
    id = Column(Integer, primary_key=True, server_default=text_("nextval('product_rout_rate_table_id_seq'::regclass)"))
    rout_id = Column(ForeignKey('route_strategy.route_strategy_id', ondelete='CASCADE'), nullable=False)
    rate_table_id = Column(ForeignKey('rate_table.rate_table_id', ondelete='CASCADE'), nullable=False)
    product_name = Column(String(256), nullable=False, unique=True)
    agent_id = Column(Integer)
    status = Column(ChoiceType(STATUS), default=1, server_default=text_("1"))  # 0:Inactive; 1:Active
    type = Column(ChoiceType(TYPE), default=0, server_default=text_("0"))  # 0:Public; 1:Private
    marketplace = Column(ChoiceType(MARKETPLACE), default=0, server_default=text_("0"))  # 0:A-Z; 1:US
    par_id = Column(Integer)
    prefix = Column(String)
    description = Column(Text)
    allowed_client = Column(Text)
    update_by = Column(String)
    update_at = Column(DateTime(timezone=True), default=datetime.now(UTC))

    route_plan_id = synonym('rout_id')
    name = synonym('product_name')
    tech_prefix = synonym('prefix')
    update_on = synonym('update_at')

    route_plan = relationship('RouteStrategy')
    rate_table = relationship('RateTable')

    clients_ref = relationship('ProductClientsRef', uselist=True, back_populates='product',
                               single_parent=True, cascade="all, delete-orphan")
    agents_ref = relationship('ProductAgentsRef', uselist=True, back_populates='product',
                              single_parent=True, cascade="all, delete-orphan")

    trunks = relationship('ResourcePrefix', primaryjoin='foreign(ResourcePrefix.product_id) == ProductRoutRateTable.id',
                          uselist=True, back_populates='product')

    @hybrid_property
    def client_id(self):
        try:
            clients = set([str(tr.client_id) for tr in self.clients_ref])
            return ','.join(list(clients))
        except:
            return ''

    @hybrid_property
    def clients(self):
        try:
            clients = set([tr.client_id for tr in self.clients_ref])
            return list(clients)
        except:
            return []

    @clients.setter
    def clients(self, value):
        exc = []
        for tr in self.clients_ref:
            if not tr.client_id in value:
                tr.delete()
            else:
                exc.append(tr.client_id)
        for client_id in value:
            if client_id in exc:
                continue
            try:
                self.clients_ref.append(ProductClientsRef(client_id=client_id))
            except Exception as e:
                pass

    @hybrid_property
    def agents(self):
        try:
            agents = set([tr.agent_id for tr in self.agents_ref])
            return list(agents)
        except:
            return []

    @agents.setter
    def agents(self, value):
        exc = []
        for tr in self.agents_ref:
            if not tr.agent_id in value:
                tr.delete()
            else:
                exc.append(tr.agent_id)
        for agent_id in value:
            if agent_id in exc:
                continue
            try:
                self.agents_ref.append(ProductAgentsRef(agent_id=agent_id))
            except Exception as e:
                pass

    @hybrid_property
    def allowed_clients(self):
        try:
            return self.allowed_client.split(',')
        except:
            return []

    @allowed_clients.setter
    def allowed_clients(self, value):
        try:
            self.allowed_client = ','.join([str(x) for x in value])
        except:
            pass

    @property
    def agent_rate_download_link(self):
        return self.rate_table.rate_download_link

        import jwt
        exp = datetime.now(UTC) + timedelta(days=settings.JWT_TTL_DAYS)
        token_data = {'agent_id': self.agent_id, 'rate_table_id': self.rate_table_id, 'exp': exp}
        token = jwt.encode(token_data, settings.JWT_SIGNATURE).decode('utf-8')
        return '{}/home/agent/download/{}'.format(settings.API_URL, token)

    @property
    def rate_download_link(self):
        return self.rate_table.rate_download_link

        import jwt
        exp = datetime.now(UTC) + timedelta(days=settings.JWT_TTL_DAYS)
        token_data = {'agent_id': self.agent_id, 'rate_table_id': self.rate_table_id, 'exp': exp}
        token = jwt.encode(token_data, settings.JWT_SIGNATURE).decode('utf-8')
        return '{}/home/agent/download/{}'.format(settings.API_URL, token)

    def delete(self, **kwargs):
        ResourcePrefix.filter(ResourcePrefix.product_id == self.id).update({'product_id': 0},
                                                                         synchronize_session='fetch')
        ret = super(ProductRoutRateTable, self).delete(**kwargs)
        return ret


class ProductRouteAnalysis(DnlApiBaseModel):
    __tablename__ = 'product_route_analysis'
    CHARACTERISTICS = {0: 'All', 1: 'No Active Egress Trunk', 2: 'No profitable route',
                       3: 'Number of profitable routes equals'}
    STATUS = {0: 'initial', 1: 'success', 2: 'fail', 3: 'process'}
    uuid = Column(String(36), primary_key=True, default=generate_uuid_str())
    characteristics = Column(ChoiceType(CHARACTERISTICS))
    count_profiltable_routes = Column(Integer)
    route_plan_id = Column(ForeignKey('route_strategy.route_strategy_id', ondelete='CASCADE'), nullable=False)
    rate_table_id = Column(ForeignKey('rate_table.rate_table_id', ondelete='CASCADE'), nullable=False)
    code_deck_id = Column(ForeignKey('code_deck.code_deck_id', ondelete='SET NULL'), nullable=True)
    date = Column(DateTime, default=datetime.now())
    status = Column(ChoiceType(STATUS), default='initial')
    finish = Column(DateTime)
    process = Column(Text)
    rec_num = Column(Integer)
    create_by = Column(String(100))

    items = relationship('ProductRouteAnalysisLog', uselist=True, back_populates='request', single_parent=True,
                         cascade="all, delete-orphan", lazy='dynamic')
    route_plan = relationship('RouteStrategy', uselist=False)
    rate_table = relationship('RateTable', uselist=False)
    code_deck = relationship('CodeDeck', uselist=False)
    codes = relationship('Code', primaryjoin='Code.code_deck_id == foreign(ProductRouteAnalysis.code_deck_id)',
                         uselist=True)


class ProductRouteAnalysisLog(DnlApiBaseModel):
    __tablename__ = 'product_route_analysis_log'
    id = Column(Integer, primary_key=True)
    uuid = Column(ForeignKey('product_route_analysis.uuid', ondelete='CASCADE'))
    code = Column(PrefixRange, default='')
    code_name = Column(String(100))
    country = Column(String(100))
    rate = Column(Numeric(30, 10))
    trunk_count = Column(Integer, default=0)
    profitable_trunk_count = Column(Integer, default=0)
    request = relationship('ProductRouteAnalysis', uselist=True, back_populates='items')


class ProductAgentsRef(DnlApiBaseModel):
    __tablename__ = 'product_agents_ref'

    id = Column(Integer, primary_key=True)  # , server_default=text("nextval('product_agents_ref_id_seq'::regclass)"))
    product_id = Column(ForeignKey('product_rout_rate_table.id', ondelete='CASCADE'))
    agent_id = Column(ForeignKey('agent.agent_id', ondelete='CASCADE'))
    agent = relationship('Agent', )
    product = relationship('ProductRoutRateTable', uselist=False, back_populates='agents_ref')

    name = column_property(
        select([ProductRoutRateTable.name]).where(ProductRoutRateTable.id == product_id).correlate_except(
            ProductRoutRateTable))
    tech_prefix = column_property(
        select([ProductRoutRateTable.tech_prefix]).where(ProductRoutRateTable.id == product_id).correlate_except(
            ProductRoutRateTable))
    type = column_property(
        select([ProductRoutRateTable.type]).where(ProductRoutRateTable.id == product_id).correlate_except(
            ProductRoutRateTable))
    marketplace = column_property(
        select([ProductRoutRateTable.marketplace]).where(ProductRoutRateTable.id == product_id).correlate_except(
            ProductRoutRateTable))
    description = column_property(
        select([ProductRoutRateTable.description]).where(ProductRoutRateTable.id == product_id).correlate_except(
            ProductRoutRateTable))
    rate_table_id = column_property(
        select([ProductRoutRateTable.rate_table_id]).where(ProductRoutRateTable.id == product_id).correlate_except(
            ProductRoutRateTable))

    route_plan_id = column_property(
        select([ProductRoutRateTable.route_plan_id]).where(ProductRoutRateTable.id == product_id).correlate_except(
            ProductRoutRateTable))

    @property
    def agent_rate_download_link(self):
        return self.product.rate_download_link

        import jwt
        exp = datetime.now(UTC) + timedelta(days=settings.JWT_TTL_DAYS)
        token_data = {'agent_id': self.agent_id, 'rate_table_id': self.rate_table_id, 'exp': exp}
        token = jwt.encode(token_data, settings.JWT_SIGNATURE).decode('utf-8')
        return '{}/home/agent/download/{}'.format(settings.API_URL, token)


ProductAgentsRef.agent_name = column_property(
    select([Agent.agent_name]).where(Agent.agent_id == ProductAgentsRef.agent_id).correlate_except(Agent))


class ProductClientsRef(DnlApiBaseModel):
    __tablename__ = 'product_clients_ref'
    __table_args__ = (
        UniqueConstraint('product_id', 'client_id'),
    )
    id = Column(Integer, primary_key=True)  # , server_default=text_("nextval('product_clients_ref_id_seq'::regclass)"))
    product_id = Column(ForeignKey('product_rout_rate_table.id', ondelete='CASCADE'))
    client_id = Column(ForeignKey('client.client_id', ondelete='CASCADE'))
    product = relationship('ProductRoutRateTable', uselist=False, back_populates='clients_ref')
    client = relationship('Client', uselist=True)
    trunks = relationship('Resource', primaryjoin='and_(Resource.client_id==foreign(ProductClientsRef.client_id),'
                                                  'ResourcePrefix.product_id==foreign(ProductClientsRef.product_id),'
                                                  'foreign(ResourcePrefix.resource_id)==Resource.resource_id)',
                          viewonly=True, uselist=True)


ProductRoutRateTable.clients_count = column_property(
    select([func.count(ProductClientsRef.client_id)]).where(ProductClientsRef.product_id == ProductRoutRateTable.id))

ProductRoutRateTable.agents_count = column_property(
    select([func.count(ProductAgentsRef.agent_id)]).where(ProductAgentsRef.product_id == ProductRoutRateTable.id))


class ProductClientsRefUsed(DnlApiBaseModel):
    __tablename__ = 'product_clients_ref_used'
    __table_args__ = (
        UniqueConstraint('product_id', 'client_id'),
    )
    id = Column(Integer, primary_key=True)  # , server_default=text_("nextval('product_clients_ref_id_seq'::regclass)"))
    product_id = Column(ForeignKey('product_rout_rate_table.id', ondelete='CASCADE'))
    client_id = Column(ForeignKey('client.client_id', ondelete='CASCADE'))
    _rate_last_sent = Column(DateTime(True))
    product = relationship('ProductRoutRateTable', uselist=False)
    client = relationship('Client', uselist=True)
    trunks = relationship('Resource', primaryjoin='and_(Resource.client_id==foreign(ProductClientsRefUsed.client_id),'
                                                  'ResourcePrefix.product_id==foreign(ProductClientsRefUsed.product_id),'
                                                  'foreign(ResourcePrefix.resource_id)==Resource.resource_id)',
                          viewonly=True, uselist=True)

    name = column_property(
        select([ProductRoutRateTable.name]).where(ProductRoutRateTable.id == product_id).correlate_except(
            ProductRoutRateTable))
    tech_prefix = column_property(
        select([ProductRoutRateTable.tech_prefix]).where(ProductRoutRateTable.id == product_id).correlate_except(
            ProductRoutRateTable))
    type = column_property(
        select([ProductRoutRateTable.type]).where(ProductRoutRateTable.id == product_id).correlate_except(
            ProductRoutRateTable))
    marketplace = column_property(
        select([ProductRoutRateTable.marketplace]).where(ProductRoutRateTable.id == product_id).correlate_except(
            ProductRoutRateTable))
    description = column_property(
        select([ProductRoutRateTable.description]).where(ProductRoutRateTable.id == product_id).correlate_except(
            ProductRoutRateTable))
    rate_table_id = column_property(
        select([ProductRoutRateTable.rate_table_id]).where(ProductRoutRateTable.id == product_id).correlate_except(
            ProductRoutRateTable))

    route_plan_id = column_property(
        select([ProductRoutRateTable.route_plan_id]).where(ProductRoutRateTable.id == product_id).correlate_except(
            ProductRoutRateTable))

    @property
    def rate_download_link(self):
        return self.product.rate_download_link

    @classmethod
    def update_used(cls, client_id):
        client = Client.get(client_id)
        products = []
        for r in client.resources:
            for p in r.prefixes:
                if p.product_id:
                    if ProductRoutRateTable.get(p.product_id):
                        products.append(p.product_id)
                    else:
                        log.warning(
                            'product {} set in prefixes (resource_id={}, resource_prefix_id={}), but not exists.'
                            '(was deleted?)'.format(p.product_id, r.resource_id, p.id))
        for r in ProductClientsRef.filter(ProductClientsRef.client_id == client_id).all():
            products.append(r.product_id)
        for r in ResourcePrefix.filter(ResourcePrefix.client_id == client_id).all():
            if ProductClientsRef.filter(ProductClientsRef.product_id == r.product_id).first():
                products.append(r.product_id)
        for r in ProductRoutRateTable.filter(ProductRoutRateTable.type == 'public'):
            products.append(r.id)
        products = set(products)
        cls.filter(and_(cls.client_id == client_id, cls.product_id.notin_(products))).delete(
            synchronize_session='fetch')
        exists = set([i.product_id for i in cls.filter(and_(cls.client_id == client_id,
                                                            cls.product_id.in_(products))).all()])
        for p_id in products - exists:
            obj = cls(client_id=client_id, product_id=p_id)
            old = cls.filter(and_(cls.client_id == client_id, cls.product_id == p_id)).first()
            if not old:
                obj.save()


class Product(DnlApiBaseModel):
    __tablename__ = 'product'

    DEFINED_BY_DICT = {0: 'Code', 1: 'Code Name', 3: 'Country'}
    ROUTED_BY_DICT = {1: 'LRN', 0: 'DNIS'}
    # ROUTED_BY_DICT = {1: 'LRN', 0: 'DNIS', None: 'None'}
    product_id = Column(Integer, primary_key=True)
    static_route_id = synonym('product_id')
    name = Column(String(100))
    modify_time = Column(DateTime(True), nullable=False, onupdate=func.now(), default=datetime.now(UTC))
    introduction = Column(String(80))
    dynamic_route_id = Column(Integer)
    update_by = Column(String)
    code_type = Column(ChoiceType(DEFINED_BY_DICT), default=0)
    defined_by = synonym('code_type')
    code_deck_id = Column(Integer, ForeignKey('code_deck.code_deck_id', ondelete='SET NULL'))
    route_lrn = Column(ChoiceType(ROUTED_BY_DICT))
    routed_by = synonym('route_lrn')

    code_deck = relationship('CodeDeck')
    items = relationship('ProductItems', uselist=True, back_populates='product', lazy='dynamic')
    update_at = synonym('modify_time')

    # @classmethod
    # def get_model_class_primary_key(cls):
    #    return cls.static_route_id #get_db(cls.db_label).get_model_class_primary_key(cls)
    def route_count(self):
        return len({route.route_strategy_id for route in Route.filter(Route.static_route_id == self.product_id).all()})

    def prefix_count(self):
        return ProductItems.filter(ProductItems.product_id == self.product_id).count()

    @hybrid_property
    def code_deck_name(self):
        try:
            return self.code_deck.name
        except:
            return ""

    @classmethod
    def init(cls):
        q = cls.get(1)
        if not q:
            q = Product(product_id=1, name='ORIGINATION_STATIC_ROUTE', update_by='system', code_type='Code',
                        routed_by='DNIS')
            q.save()
        if not q.name == 'ORIGINATION_STATIC_ROUTE' or not q.routed_by == 'DNIS':
            q.name = 'ORIGINATION_STATIC_ROUTE'
            q.update_by = 'system'
            q.code_type = 'Code'
            q.routed_by = 'DNIS'
            q.save()

    @staticmethod
    def get_orig_static_route_id():
        Product.init()
        return 1

    def __repr__(self):
        return self.name


class ProductItems(DnlApiBaseModel):
    __tablename__ = 'product_items'
    STRATEGY_CHOICES = {
        0: 'By Percentage',
        1: 'Top-Down',
        2: 'Round-Robin'
    }
    item_id = Column(Integer, primary_key=True)
    product_id = Column(Integer, ForeignKey('product.product_id', ondelete='CASCADE'))
    alias = Column(String(100))
    digits = Column(PrefixRange(), nullable=False, server_default=text_("''::prefix_range"))
    # strategy = Column( Integer)# 0-;By Percentage;1-Top-Down;2-Round-Robin
    strategy = Column(ChoiceType(STRATEGY_CHOICES), default='Top-Down', nullable=False, server_default=text_("1"))
    time_profile_id = Column(Integer, ForeignKey('time_profile.time_profile_id', ondelete='SET NULL'))
    min_len = Column(Integer)
    max_len = Column(Integer)
    min_asr = Column(Float)
    max_asr = Column(Float)
    min_abr = Column(Float)
    max_abr = Column(Float)
    min_acd = Column(Float)
    max_acd = Column(Float)
    min_pdd = Column(Integer)
    max_pdd = Column(Integer)
    min_aloc = Column(Float)
    max_aloc = Column(Float)
    limit_price = Column(Float)
    update_at = Column(DateTime(timezone=True), default=datetime.now(UTC))
    update_by = Column(String)
    code_name = Column(String(100))
    # time = Column( Float)
    # flag = Column( SmallInteger, autoincrement=False)
    # record_id = Column( Integer, nullable=False)
    static_route_id = synonym('product_id')
    did = synonym('digits')

    product = relationship('Product', uselist=False, back_populates='items')
    trunks = relationship('ProductItemsResource', uselist=True, back_populates='product',
                          # lazy='joined',
                          single_parent=True,
                          cascade="all, delete-orphan", lazy='dynamic')
    time_profile = relationship('TimeProfile', uselist=False)

    time_profile_name = column_property(
        select([TimeProfile.name]).where(TimeProfile.time_profile_id == time_profile_id).correlate_except(TimeProfile))
    static_route_name = column_property(
        select([Product.name]).where(Product.product_id == product_id).correlate_except(Product))


    __table_args__ = (
        Index(
            "product_items_product_id_digits_key",
            "product_id",
            "digits",
            "time_profile_id",
            unique=True,
            postgresql_where=digits.isnot(None)
        ),
        Index(
            "product_items_product_id_digits_min_len_max_len_key",
            "product_id",
            "digits",
            "min_len",
            "max_len",
            unique=True
        ),
    )

    @hybrid_property
    def _static_route_name(self):
        return self.static_route_name

    @_static_route_name.setter
    def _static_route_name(self, value):
        product = Product.filter(Product.product_id == self.product_id).first()
        if product:
            product.name = value
            product.save()

    @hybrid_property
    def product_name(self):
        return self.product.name

    @classmethod
    def assign_client_did(cls, resource, did, user=None):
        Route.init()
        static_route_id = Product.get_orig_static_route_id()
        cls.filter(cls.digits == did).delete()
        item = cls(product_id=static_route_id, digits=did, strategy='Top-Down', update_by=user)
        res_item = ProductItemsResource(order_type=1)
        res_item.resource_id = resource.resource_id
        item.trunks.append(res_item)
        cls.session().add(item)
        return item
 
    def before_save(self):
        if self.product_id == 1:
            old_obj = ProductItems.filter(and_(ProductItems.product_id == 1,
                                               ProductItems.digits == self.digits,
                                               ProductItems.item_id != self.item_id)).first()
            if old_obj:
                self.delete()
                raise Exception('digits are duplicate')


class ProductItemsResource(DnlApiBaseModel):
    __tablename__ = 'product_items_resource'
    __table_args__ = (UniqueConstraint('item_id', 'resource_id'),)

    id = Column(Integer, primary_key=True, server_default=text_("nextval('product_items_resource_id_seq'::regclass)"))
    item_id = Column(ForeignKey('product_items.item_id', ondelete='CASCADE'), nullable=False, index=True)
    resource_id = Column(ForeignKey('resource.resource_id', ondelete='CASCADE'), nullable=False, index=True)
    by_percentage = Column(Integer)
    order_id = Column(Integer)
    order_type = Column(Integer)  # 1-buy order;2-sell order

    trunk_id = synonym('resource_id')

    product = relationship('ProductItems', uselist=False, back_populates='trunks')
    resource = relationship('Resource', uselist=False, back_populates='product_items')

    @hybrid_property
    def trunk_name(self):
        return self.resource.alias

    @hybrid_property
    def carrier_name(self):
        return self.resource.client.name


class DidAssign(DnlApiBaseModel):
    __tablename__ = 'did_assign'

    number = Column(String(100), primary_key=True)
    egress_id = Column(ForeignKey('resource.resource_id'), nullable=False)
    created_time = Column(DateTime(timezone=True), default=datetime.now(UTC))
    assigned_time = Column(DateTime(timezone=True))
    status = Column(SmallInteger, default=0)
    ingress_id = Column(Integer)

    egress = relationship("Resource")


class banned_ip_record(DnlApiBaseModel):
    __tablename__ = 'banned_ip_record'
    ip = Column(String(30))
    brief = Column(String(100))
    create_time = Column(DateTime(timezone=True))
    time = Column(Numeric)
    flag = Column(CHAR(1))
    record_id = Column(Integer, primary_key=True)


class DidBillingBrief(DnlApiBaseModel):
    __tablename__ = 'did_billing_brief'
    id = Column(Integer, primary_key=True)
    did_number = Column(String(50))
    vendor_billing_plan_id = Column(Integer)
    client_billing_plan_id = Column(Integer)
    vendor_trunk_id = Column(Integer)
    client_trunk_id = Column(Integer)
    start_date = Column(DateTime(timezone=True))
    end_date = Column(DateTime(timezone=True))
    enable_for_clients = Column(Boolean, server_default='false')
    last_charged_monthly_fees = Column(Integer, server_default='-1')
    next_mrc_billing_date = Column(Date(), index=True)
    nrc_billing_date = Column(Date(), index=True)
    did = synonym('did_number')
    buy_billing_plan_id = synonym('client_billing_plan_id')
    client_res_id = synonym('client_trunk_id')


# +++ DidBillingPlan
class DidBillingPlan(DnlApiBaseModel):
    __tablename__ = 'did_billing_plan'
    RATE_TYPE = {1: 'fixed', 2: 'variable'}
    PAY_TYPE = {0: 'weekly', 1: 'monthly', 3: 'daily'}
    PRICE_TYPE = {0: 'weekly', 1: 'monthly', 3: 'daily'}

    id = Column(Integer, primary_key=True, server_default=text_("nextval('did_billing_plan_id_seq'::regclass)"))
    did_price = Column(Float, default=0.0, server_default=text_("0"))
    min_price = Column(Float, default=0.0, server_default=text_("0"))
    name = Column(String(255), unique=True)
    payphone_subcharge = Column(String)
    monthly_charge = Column(Numeric, default=0.0)
    rate_table_id = Column(ForeignKey('rate_table.rate_table_id', ondelete='SET NULL'), nullable=True)
    rate_type = Column(ChoiceType(RATE_TYPE), server_default='1')
    pay_type = Column(ChoiceType(PAY_TYPE), server_default='1')
    # fee_per_port = Column(Numeric, default=0.0)
    price_type = Column(ChoiceType(PRICE_TYPE), server_default='1')
    client_id = Column(ForeignKey('client.client_id', ondelete='CASCADE'), nullable=False)

    update_at = Column(DateTime(timezone=True), default=datetime.now(UTC))
    update_by = Column(String)
    min_time = Column(Integer)
    interval = Column(Integer)
    rate_per_sms_sent = Column(Numeric(15, 6))
    rate_per_sms_received = Column(Numeric(15, 6))

    setup_fee = synonym('did_price')
    monthly_fee = synonym('monthly_charge')
    rate_per_min = synonym('min_price')

    rate_table = relationship('RateTable',
                              primaryjoin='foreign(DidBillingPlan.rate_table_id) == RateTable.rate_table_id',
                              uselist=False)

    # rate_table_name = column_property(
    #     select([RateTable.name]).where(RateTable.rate_table_id == rate_table_id).correlate_except(RateTable))

    # sell_dids = relationship('DidBillingPlan',
    #                           primaryjoin='foreign(DidBillingRel.sell_billing_plan_id) == DidBillingPlan.id',
    #                           uselist=True, back_populates='sell_billing_plan')
    # buy_dids = relationship('DidBillingPlan',
    #                         primaryjoin='foreign(DidBillingRel.buy_billing_plan_id) == DidBillingPlan.id',
    #                         uselist=True, back_populates='buy_billing_plan')

    # def rate_table_name(self):
    #     return self.rate_table.name


# --- DidBillingPlan


def did_report(date):
    tablename = 'did_report{}'.format(date)
    try:
        r = get_db().session.execute(text_("select tablename from pg_tables where tablename='{}';".format(tablename)))
        if not r.fetchall()[0][0] == tablename:
            return None
    except:
        return None
    return Table(
        tablename, DnlApiBaseModel.metadata,
        Column('report_time', DateTime(True)),
        Column('duration', Integer),
        Column('ingress_bill_time', Integer),
        Column('ingress_call_cost', Numeric(15, 6)),
        Column('ingress_total_calls', Integer),
        Column('not_zero_calls', Integer),
        Column('ingress_success_calls', Integer),
        Column('ingress_client_id', Integer),
        Column('ingress_id', Integer),
        Column('egress_client_id', Integer),
        Column('egress_id', Integer),
        Column('egress_bill_time', Integer),
        Column('egress_call_cost', Numeric(15, 6)),
        Column('egress_total_calls', Integer),
        Column('egress_success_calls', Integer),
        Column('did', String(100)),
        Column('ingress_ip', Text()),
        Column('egress_ip', Text()),
        Column('cdr_date', String(24)),
        Column('pdd', Integer),
        Column('ingress_busy_calls', Integer),
        Column('egress_busy_calls', Integer),
        Column('ingress_cancel_calls', Integer),
        Column('egress_cancel_calls', Integer),
        Column('cdr_date', String(24)),
        Column('mrc', Numeric(15, 6)),
        Column('nrc', Numeric(15, 6)),
        Column('port_fee', Numeric(15, 6)),
        extend_existing=True,
    )


class DidReport(DnlApiBaseModel):
    __tablename__ = 'did_report'

    report_time = Column(DateTime(True))
    duration = Column(Integer)
    ingress_bill_time = Column(Integer)
    ingress_call_cost = Column(Numeric(15, 6))
    ingress_total_calls = Column(Integer)
    not_zero_calls = Column(Integer)
    ingress_success_calls = Column(Integer)
    ingress_client_id = Column(Integer)
    ingress_id = Column(Integer)
    egress_client_id = Column(Integer)
    egress_id = Column(Integer)
    egress_bill_time = Column(Integer)
    egress_call_cost = Column(Numeric(15, 6))
    egress_total_calls = Column(Integer)
    egress_success_calls = Column(Integer)
    did = Column(String(100))
    ingress_ip = Column(Text())
    egress_ip = Column(Text())
    cdr_date = Column(String(24))
    pdd = Column(Integer)
    ingress_busy_calls = Column(Integer)
    egress_busy_calls = Column(Integer)
    ingress_cancel_calls = Column(Integer)
    egress_cancel_calls = Column(Integer)
    mrc = Column(Numeric(15, 6))
    nrc = Column(Numeric(15, 6))
    port_fee = Column(Numeric(15, 6))
    # surcharge = Column(Numeric(15, 6))
    __table_args__ = (PrimaryKeyConstraint(
        'did', 'ingress_client_id', 'ingress_id', 'egress_client_id', 'egress_id', 'report_time'
    ),)

    @classmethod
    def update_mrc(cls, report_time):
        return
        rel = DidRepository
        pln = DidBillingPlan
        trn = DidTransaction
        res = Resource
        res1 = aliased(Resource)
        rep = DidReport
        # clean old data if exists
        rep.filter(and_(rep.report_time == report_time, rep.cdr_date.is_(None))).delete(synchronize_session='fetch')
        start_time = report_time - timedelta(days=1)
        end_time = datetime(start_time.year, start_time.month, start_time.day, 23, 59, 59)
        q = select([literal(report_time), rel.did, rel.client_trunk_id, rel.vendor_trunk_id,
                    select([res.client_id]).where(rel.client_trunk_id == res.resource_id).as_scalar(),
                    select([res1.client_id]).where(rel.vendor_trunk_id == res1.resource_id).as_scalar(),
                    pln.did_price * pln.did_price, #case([(rel.start_date <= start_time, pln.did_price)], else_=0.0),
                    pln.monthly_charge * case([(and_(pln.pay_type == 0, func.extract('dow', report_time) == 1), 1),
                                               (and_(pln.pay_type == 1, func.extract('day', report_time) == 1), 1),
                                               (pln.pay_type == 3, 1)
                                               ], else_=0),
                    # trn.nrc, trn.mrc,
                    pln.fee_per_port * case([(and_(pln.price_type == 0, func.extract('dow', report_time) == 1), 1),
                                             (and_(pln.price_type == 1, func.extract('day', report_time) == 1), 1),
                                             (pln.price_type == 3, 1)
                                             ], else_=0),
                    ]).where(pln.id == rel.client_billing_plan_id)
        q1 = rep.__table__.insert().from_select(
            ['report_time', 'did', 'egress_id', 'egress_client_id', 'ingress_id', 'ingress_client_id', 'nrc', 'mrc',
             'port_fee'], q)
        r = get_db().session.execute(q1)
        get_db().session.commit()


class CodeReport(DnlApiBaseModel):
    __tablename__ = 'code_report'

    id = Column(BigInteger, primary_key=True)
    cdr_date = Column(String, nullable=False)
    report_time = Column(DateTime(False), primary_key=True)
    code = Column(String, nullable=False)
    ingress_client_id = Column(Integer)
    ingress_id = Column(Integer)
    egress_client_id = Column(Integer)
    egress_id = Column(Integer)
    not_zero_calls = Column(Integer)
    total_calls = Column(Integer)
    ingress_not_zero_calls = Column(Integer)
    ingress_total_calls = Column(Integer)


class CodeReportDaily(DnlApiBaseModel):
    __tablename__ = 'code_report_daily'

    id = Column(BigInteger, primary_key=True)
    cdr_date = Column(String, nullable=False)
    report_time = Column(DateTime(False), primary_key=True)
    code = Column(String, nullable=False)
    ingress_client_id = Column(Integer)
    ingress_id = Column(Integer)
    egress_client_id = Column(Integer)
    egress_id = Column(Integer)
    not_zero_calls = Column(Integer)
    total_calls = Column(Integer)
    ingress_not_zero_calls = Column(Integer)
    ingress_total_calls = Column(Integer)


class ShakenStiSpConf(DnlApiBaseModel):
    __tablename__ = 'shaken_sti_sp_conf'

    __table_args__ = (
        CheckConstraint("shaken_sti_sp_conf_id_check"),
    )
    id = Column(Boolean, default=True, server_default=text_("True"))
    priv_key = Column(String, primary_key=True)
    passphrase = Column(String)
    x5u = Column(String)
    enabled = Column(Boolean, default=True, nullable=False)

class DynamicRoute(DnlApiBaseModel):
    __tablename__ = 'dynamic_route'

    ROUTING_RULE_DICT = {6: 'LCR', 1: 'By LCR', 2: 'By ASR', 3: 'By ACD',
                         4: 'by maximum ASR', 5: 'by maximum ACD'}
    LCR_FLAG_DICT = {None: 'not set', 1: '15 minutes', 2: '30 minutes', 3: '1 hours', 4: '1 days'}

    dynamic_route_id = Column(Integer, primary_key=True)
    name = Column(String(100), nullable=False, unique=True)
    routing_rule = Column(ChoiceType(ROUTING_RULE_DICT), nullable=False,
                          default=6)  # Routing rules: 4 - by maximum ASR; 5 - by maximum ACD; 6 - at the lowest cost
    time_profile_id = Column(ForeignKey('time_profile.time_profile_id', ondelete='SET NULL'), index=True)
    update_at = Column(DateTime(timezone=True), onupdate=func.now())
    update_by = Column(String)
    lcr_flag = Column(ChoiceType(LCR_FLAG_DICT), default=1)  # 1-15 minutes; 2-30 minutes; 3-1 hours; 4-1 days
    is_virtual = Column(Boolean)

    lcr_flag_name = synonym('lcr_flag')
    route_rule_name = synonym('routing_rule')
    # rule = synonym('routing_rule')
    last_modified = synonym('update_at')
    modified_by = synonym('update_by')

    qos_cycle = synonym('lcr_flag')

    time_profile = relationship(TimeProfile)
    routes = relationship('Route', back_populates='dynamic_route', uselist=True)

    egress_trunks = relationship('DynamicRouteItem', uselist=True, back_populates='route',
                                 single_parent=True,
                                 cascade="all, delete-orphan"
                                 )
    qos = relationship('DynamicRouteQos', uselist=True, back_populates='dynamic_route',
                       single_parent=True, cascade="all, delete-orphan"
                       )
    pri = relationship('DynamicRoutePri', uselist=True, back_populates='dynamic_route',
                       single_parent=True, cascade="all, delete-orphan"
                       )
    over = relationship('DynamicRouteOverride', uselist=True, back_populates='dynamic_route',
                        single_parent=True, cascade="all, delete-orphan"
                        )
    # @hybrid_property
    # def usage_count(self):return len(self.routes)
    time_profile_name = column_property(
        select([TimeProfile.name]).where(TimeProfile.time_profile_id == time_profile_id).correlate_except(TimeProfile))


class DynamicRouteItem(DnlApiBaseModel):
    __tablename__ = 'dynamic_route_items'
    __table_args__ = (
        UniqueConstraint('dynamic_route_id', 'resource_id'),
    )
    id = Column(Integer, primary_key=True)
    dynamic_route_id = Column(
        ForeignKey('dynamic_route.dynamic_route_id', ondelete='CASCADE'), nullable=False, index=True
    )
    resource_id = Column(ForeignKey('resource.resource_id', ondelete='CASCADE'), nullable=False, index=True)

    route = relationship(DynamicRoute, uselist=False, back_populates='egress_trunks')
    trunk = relationship('EgressTrunk',
                         primaryjoin='EgressTrunk.resource_id==foreign(DynamicRouteItem.resource_id)',
                         back_populates='dynamic_routes',
                         uselist=False)

    @hybrid_property
    def trunk_name(self):
        try:
            return self.trunk.alias
        except:
            return None

    @hybrid_property
    def is_active(self):
        try:
            return self.trunk.is_active
        except:
            return None

    @hybrid_property
    def client_id(self):
        try:
            return self.trunk.client_id
        except:
            return None

    @hybrid_property
    def client_name(self):
        try:
            return self.trunk.client.name
        except:
            return None


DynamicRoute.trunk_count = column_property(select([func.count(DynamicRouteItem.id).label('trunk_count')]). \
                                           where(DynamicRoute.dynamic_route_id == DynamicRouteItem.dynamic_route_id). \
                                           correlate_except(DynamicRouteItem))


class DynamicRouteQos(DnlApiBaseModel):
    __tablename__ = 'dynamic_route_qos'
    __table_args__ = (
        UniqueConstraint('dynamic_route_id', 'digits'),
    )

    id = Column(Integer, primary_key=True, server_default=text_("nextval('dynamic_route_qos_id_seq'::regclass)"))
    dynamic_route_id = Column(ForeignKey('dynamic_route.dynamic_route_id', ondelete='CASCADE'), nullable=False,
                              index=True)
    digits = Column(PrefixRange, nullable=False, server_default=text_("''::prefix_range"))
    min_asr = Column(Float)
    max_asr = Column(Float)
    min_abr = Column(Float)
    max_abr = Column(Float)
    min_acd = Column(Float)
    max_acd = Column(Float)
    min_pdd = Column(Integer)
    max_pdd = Column(Integer)
    min_aloc = Column(Float)
    max_aloc = Column(Float)
    limit_price = Column(Float)

    dynamic_route = relationship('DynamicRoute', uselist=False, back_populates='qos')


class DynamicRoutePri(DnlApiBaseModel):
    __tablename__ = 'dynamic_route_pri'

    id = Column(Integer, primary_key=True, server_default=text_("nextval('dynamic_route_pri_id_seq'::regclass)"))
    dynamic_route_id = Column(ForeignKey('dynamic_route.dynamic_route_id', ondelete='CASCADE'), nullable=False,
                              index=True)
    digits = Column(PrefixRange, nullable=False, server_default=text_("''::prefix_range"))
    resource_id = Column(ForeignKey('resource.resource_id', ondelete='CASCADE'), nullable=False, index=True)
    resource_pri = Column(Integer, nullable=False, server_default=text_("0"))

    dynamic_route = relationship('DynamicRoute', uselist=False, back_populates='pri')
    resource = relationship('Resource')


class DynamicRouteOverride(DnlApiBaseModel):
    __tablename__ = 'dynamic_route_override'

    id = Column(Integer, primary_key=True, server_default=text_("nextval('dynamic_route_override_id_seq'::regclass)"))
    dynamic_route_id = Column(ForeignKey('dynamic_route.dynamic_route_id', ondelete='CASCADE'), nullable=False,
                              index=True)
    digits = Column(PrefixRange, nullable=False, server_default=text_("''::prefix_range"))
    resource_id = Column(ForeignKey('resource.resource_id', ondelete='CASCADE'), nullable=False, index=True)
    percentage = Column(Integer, nullable=False)

    dynamic_route = relationship('DynamicRoute', uselist=False, back_populates='over')
    resource = relationship('Resource')


class Route(DnlApiBaseModel):
    __tablename__ = 'route'
    __table_args__ = (
        UniqueConstraint('route_strategy_id', 'digits', 'ani_prefix'),
        # UniqueConstraint('route_strategy_id', 'route_type', 'static_route_id', 'dynamic_route_id'),
    )

    ROUTE_TYPE_DICT = {1: 'dynamic routing', 2: 'static routing', 3: 'dynamic routing-static routing',
                       4: 'static routing-dynamic routing'}
    ROUTE_TYPE_FLG_DICT = {
        1: 'Dynamic Routing',
        2: 'Static Routing',
        3: 'Static Routing JD',
        4: 'Static Routing - Dynamic routing',
        6: 'Static Routing JD - Dynamic routing',
        6: 'Dynamic Routing - Static Routing',
        7: 'Dynamic Routing - Static Routing JD'
    }
    route_id = Column(Integer, primary_key=True)
    digits = Column(PrefixRange, default='')
    dynamic_route_id = Column(ForeignKey('dynamic_route.dynamic_route_id', ondelete='SET NULL'), index=True)
    static_route_id = Column(ForeignKey('product.product_id', ondelete='SET NULL'), index=True)
    route_type = Column(ChoiceType(ROUTE_TYPE_DICT),
                        nullable=False)  # Routing: 1 - dynamic routing; 2 - static routing; 3 - dynamic routing and then static routing; 4 - static routing and then dynamic routing
    route_type_name = synonym('route_type_flg')
    route_strategy_id = Column(
        ForeignKey('route_strategy.route_strategy_id', ondelete='CASCADE'), nullable=False, index=True
    )
    lnp = Column(Boolean, default=False)
    lrn_block = Column(Boolean, default=False)
    dnis_only = Column(Boolean, default=True)
    code_deck_type = Column(Integer, default=0)  # 0-icx code deck;1-custom code deck
    update_at = Column(DateTime(True), onupdate=func.now())
    update_by = Column(String)
    intra_static_route_id = Column(Integer)
    inter_static_route_id = Column(Integer)
    jurisdiction_country_id = Column(Integer)
    ani_prefix = Column(PrefixRange)
    ani_min_length = Column(Integer, default=0)
    ani_max_length = Column(Integer, default=32)
    digits_min_length = Column(Integer, default=0)
    digits_max_length = Column(Integer, default=32)
    code_name = Column(String)
    country = Column(String(50))
    route_type_flg = Column(ChoiceType(ROUTE_TYPE_FLG_DICT))

    ANI_prefix = synonym('ani_prefix')
    DNIS_prefix = synonym('digits')
    ANI_min = synonym('ani_min_length')
    ANI_max = synonym('ani_max_length')
    DNIS_min = synonym('digits_min_length')
    DNIS_max = synonym('digits_max_length')
    route_plan_id = synonym('route_strategy_id')

    dynamic_route = relationship(DynamicRoute, back_populates='routes', uselist=False)
    static_route = relationship(Product)
    route_strategy = relationship(RouteStrategy, back_populates='routes', uselist=False)

    @classmethod
    def check_duplicates(cls, ani_prefix, digits, ani_min_length, ani_max_length, digits_min_length, digits_max_length,
                         obj=None):
        ani_min_length = ani_min_length or 0
        ani_max_length = ani_max_length or 32
        digits_min_length = digits_min_length or 0
        digits_max_length = digits_max_length or 32
        q = cls.filter(and_(cls.ani_min_length == ani_min_length, cls.ani_max_length == ani_max_length,
                            cls.digits_min_length == digits_min_length, cls.digits_max_length == digits_max_length))
        if ani_prefix is None or ani_prefix == '':
            q = q.filter(or_(cls.ani_prefix.is_(None), cls.ani_prefix == ''))
        else:
            q = q.filter(cls.ani_prefix == ani_prefix)
        if digits is None or digits == '':
            q = q.filter(or_(cls.digits.is_(None), cls.digits == ''))
        else:
            q = q.filter(cls.digits == digits)
        if obj:
            q = q.filter(cls.route_strategy_id == obj.route_strategy_id)
            if obj.route_id:
                q = q.filter(cls.route_id != obj.route_id)
        item = q.first()
        if item and (ani_prefix or digits):
            return True
        return False

    @hybrid_property
    def route_types(self):
        return self.ROUTE_TYPE_FLG_DICT

    @classmethod
    def init(cls):
        # RouteStrategy.init()
        # Product.init()
        rs = RouteStrategy.get_orig_routing_plan_id()
        rp = Product.get_orig_static_route_id()
        r = cls.filter(and_(cls.route_strategy_id == rs, cls.static_route_id == rp)).first()
        if not r:
            cls(route_type='static routing', route_strategy_id=rs, static_route_id=rp,
                update_by='system', update_at=datetime.now(UTC), code_deck_type=0).save()

    route_plan_name = column_property(
        select([RouteStrategy.name]).where(RouteStrategy.route_strategy_id == route_strategy_id).correlate_except(
            RouteStrategy))

    DynamicRouteX = DynamicRoute.__table__.alias('dynamic_route_x')
    dynamic_route_name = column_property(
        select([DynamicRouteX.c.name]).where(DynamicRouteX.c.dynamic_route_id == dynamic_route_id).correlate_except(
            DynamicRouteX))

    ProductX = Product.__table__.alias('product_x')
    static_route_name = column_property(
        select([Product.name]).where(Product.product_id == static_route_id).correlate_except(Product))

    intra_static_route_name = column_property(
        select([Product.name]).where(Product.product_id == intra_static_route_id).correlate_except(Product))

    inter_static_route_name = column_property(
        select([Product.name]).where(Product.product_id == inter_static_route_id).correlate_except(Product))


DynamicRoute.usage_count = column_property(select([func.count(Route.route_strategy_id.distinct()).label('usage_count')]). \
                                           where(DynamicRoute.dynamic_route_id == Route.dynamic_route_id). \
                                           correlate_except(Route))

RouteX = Route.__table__.alias('route_x')
RouteStrategy.dynamic_usage_count = column_property(select([func.count(RouteX.c.route_strategy_id.distinct())]). \
                                                    where(
    and_(RouteX.c.route_strategy_id == RouteStrategy.route_strategy_id)). \
                                                    group_by(RouteX.c.route_strategy_id).correlate_except(
    RouteX).as_scalar())
RouteStrategy.static_usage_count = column_property(select([func.count(RouteX.c.route_strategy_id.distinct())]). \
                                                    where(
    and_(RouteX.c.route_strategy_id == RouteStrategy.route_strategy_id)). \
                                                    group_by(RouteX.c.route_strategy_id).correlate_except(
    RouteX).as_scalar())
ProductY = Product.__table__.alias('product_y')
RouteStrategy.static_name = column_property(select([func.string_agg(Product.name, ',')]). \
                                            where(and_(Product.product_id == RouteX.c.static_route_id,
                                                       RouteX.c.route_strategy_id == RouteStrategy.route_strategy_id)). \
                                            group_by(RouteX.c.route_strategy_id).correlate_except(RouteX,
                                                                                                  Product).as_scalar())

RouteY = Route.__table__.alias('route_y')
DynamicRouteX = DynamicRoute.__table__.alias('dynamic_route_x')
RouteStrategy.dynamic_name = column_property(select([func.string_agg(DynamicRouteX.c.name, ',')]). \
                                             where(and_(DynamicRouteX.c.dynamic_route_id == RouteY.c.dynamic_route_id,
                                                        RouteY.c.route_strategy_id == RouteStrategy.route_strategy_id)). \
                                             group_by(RouteY.c.route_strategy_id).correlate_except(RouteY,
                                                                                                   DynamicRouteX).as_scalar())


class RouteBlock(DnlApiBaseModel):
    __tablename__ = 'route_block'

    id = Column(Integer, primary_key=True)
    egress_trunk_id = Column(Integer)
    code_name = Column(String(256))
    create_by = Column(String(256))
    create_on = Column(DateTime, default=datetime.now(UTC))


class CarrierGroup(DnlApiBaseModel):
    __tablename__ = 'carrier_group'

    group_id = Column(Integer, primary_key=True)
    group_name = Column(String, unique=True, nullable=False)

    clients_detail = relationship('Client', back_populates='group', uselist=True)

    def clients_count(self):
        return len(self.clients_detail)

    @hybrid_property
    def clients(self):
        try:
            cli = self.clients_detail
            if cli:
                return [q.client_id for q in cli]
            else:
                return []
        except:
            return None

    @clients.setter
    def clients(self, value):
        for t in self.clients_detail:
            t.group_id = None  # self.clients_rel.remove(t)
        for client_id in value:
            try:

                q = Client.get(client_id)
                q.group_id = self.group_id
                self.clients_detail.append(q)
            except Exception as e:
                pass


# def __repr__(self):
#    return +self.group_name


class TrunkGroup(DnlApiBaseModel):
    __tablename__ = 'trunk_group'
    # TRUNK_TYPE_DICT = {1: "class4", 2: "exchange", 3: "product_default", 4: "product_agent"}
    TRUNK_TYPE_DICT = {1: "ingress", 2: "egress"}
    group_id = Column(Integer, primary_key=True)
    group_name = Column(String, nullable=False, unique=True)
    trunk_type = Column(ChoiceType(TRUNK_TYPE_DICT), default=1)

    trunks = relationship('Resource', back_populates='group', uselist=True)

    # --

    def trunk_match(self, q):
        return q and ((q.egress and self.trunk_type == 'egress') or
                      (q.ingress and self.trunk_type == 'ingress'))

    @hybrid_property
    def all_trunks(self):
        try:
            return [q.resource_id for q in self.trunks]
        except:
            return []

    @all_trunks.setter
    def all_trunks(self, value):

        # for t in self.trunks:
        # if self.group_id and t.group_id == self.group_id:
        #    t.group_id = None
        #    t.save()
        # self.trunks.remove(t)
        if self.group_id:
            tr = Resource.filter(Resource.group_id == self.group_id).all()
            for t in tr:
                # if t.resource_id in value:
                t.group_id = None
                t.save()
                if t in self.trunks:
                    self.trunks.remove(t)
        for trunk_id in value:
            try:
                q = Resource.filter(Resource.resource_id == trunk_id).first()
                if self.trunk_match(q):
                    q.group_id = self.group_id
                    self.session().add(q)
                    self.trunks.append(q)
            except:
                pass

    @hybrid_property
    def add_trunks(self):
        try:
            return [q.resource_id for q in self.trunks]
        except:
            return []

    @add_trunks.setter
    def add_trunks(self, value):
        for trunk_id in value:
            try:
                q = Resource.filter(Resource.resource_id == trunk_id).first()
                if self.trunk_match(q):
                    self.trunks.append(q)
            except:
                pass

    def __repr__(self):
        return self.group_name


class Code(DnlApiBaseModel):
    __tablename__ = 'code'
    __table_args__ = (Index('class4_idx_code_code_deck_prefix', 'code_deck_id', 'code'),)

    code_id = Column(Integer, primary_key=True)
    code = Column(PrefixRange, nullable=False, index=True)
    code_deck_id = Column(ForeignKey('code_deck.code_deck_id', ondelete='CASCADE'), index=True)
    city = Column(String(20))
    state = Column(String(20))
    country = Column(String(100))
    name = Column(String(100))
    code_name = synonym('name')

    code_deck = relationship('CodeDeck', back_populates='codes', uselist=False)

    def before_save(self):
        old_obj = Code.filter(and_(Code.code == self.code, 
                                                  Code.code_deck_id == self.code_deck_id, 
                                                  Code.code_id != self.code_id)).first()
        if old_obj:
            raise Exception('code and code_deck_id are duplicate')


class CodeCountry(DnlApiBaseModel):
    __tablename__ = 'code_country'

    country_code = Column(String(20), primary_key=True)
    country = Column(String(80), unique=True)
    phone_code = Column(PrefixRange())

    @classmethod
    def init(cls):
        from api_dnl.const import COUNTRY
        if len(cls.query().all()) != len(COUNTRY):
            cls.query().delete()
            for c in COUNTRY:
                cls.session().add(cls(country_code=c[0], country=c[1], phone_code=c[2]))
            cls.session().commit()
            log.info('CodeCountry reinited...')


class CodeDestination(DnlApiBaseModel):
    __tablename__ = '#code_destination'
    # alias(Code.query().session.query(Code.name.label('destination'),
    #                func.count().label('codes_count')).group_by(Code.name)) #'#code_destination'
    __table_args__ = {'prefixes': ['TEMPORARY']}

    @classmethod
    def query(cls):
        cls = Code
        return cls.query().session.query(cls.name.label('destination'), cls.country, cls.code,
                                         func.count().label('codes_count')).group_by(
            cls.country, cls.name, cls.code)

    destination = Column(String(100), primary_key=True)
    country = Column(String(100))
    code = Column(String(100))
    usage_count = Column(Integer)


class CodeDeck(DnlApiBaseModel):
    __tablename__ = 'code_deck'

    code_deck_id = Column(Integer, primary_key=True)
    name = Column(String(100), nullable=False, unique=True)
    update_at = Column(DateTime(timezone=True), onupdate=func.now())
    update_by = Column(String)
    client_id = Column(Integer)  # ,
    # default=0)  # Equal to zero is the default, 0-a / z, 1-us, other values for the code deck dedicated
    codes = relationship(Code, back_populates='code_deck', uselist=True,
                         single_parent=True,
                         cascade="all, delete-orphan")
    update_on = synonym('update_at')

    @staticmethod
    def init():
        if not CodeDeck.get(0):
            r = CodeDeck(code_deck_id=0, name='A/Z', client_id=1)
            r.save()
        else:
            r = CodeDeck.get(0)
            if not (r.name == 'A/Z' and r.client_id == 0):
                r.name = 'A/Z'
                r.client_id = 0
                r.update_by = 'system'
                r.save()

        if not CodeDeck.get(1):
            r = CodeDeck(code_deck_id=1, name='US', client_id=2)
            r.save()
        else:
            r = CodeDeck.get(1)
            if not (r.name == 'US' and r.client_id == 1):
                r.name = 'US'
                r.client_id = 1
                r.update_by = 'system'
                r.save()

    # def code_count(self):return Code.filter(Code.code_deck_id==self.code_deck_id).count()

    def __repr__(self):
        return self.name


CodeDeck.code_count = column_property(
    select([func.count(Code.code_id).label('code_count')]).where(Code.code_deck_id == CodeDeck.code_deck_id). \
        group_by(Code.code_deck_id).correlate_except(Code))


class ClientDefaultIp(DnlApiBaseModel):
    __tablename__ = 'client_default_ip'

    id = Column(Integer, primary_key=True)
    client_id = Column(ForeignKey('client.client_id', ondelete='CASCADE'), nullable=False)
    ip = Column(String(100))
    port = Column(Integer, default=5060)
    netmark = Column(Integer, default=32)
    product_id = Column(Integer, nullable=True)
    mask = synonym('netmark')
    client = relationship('Client', uselist=False, back_populates='default_ip')


class ClientTaxes(DnlApiBaseModel):
    __tablename__ = 'client_taxes'
    id = Column(Integer, primary_key=True)
    client_id = Column(ForeignKey('client.client_id', ondelete='CASCADE'), nullable=False)
    tax_name = Column(String(255), nullable=False)
    tax_percent = Column(Numeric, nullable=False)
    client = relationship('Client', uselist=False, back_populates='client_taxes', enable_typechecks=False)


CDR_FIELDS = {
    0: "ani_code_id",
    1: "answer_time_of_date",
    2: "binary_value_of_release_cause_from_protocol_stack",
    3: "call_duration",
    4: "call_type",
    5: "connection_type",
    6: "contract_id",
    7: "dnis_code_id",
    8: "dst_lata",
    9: "dst_ocn",
    10: "dst_rc",
    11: "dynamic_route",
    12: "dynamic_route_name",
    13: "egress_bill_minutes",
    14: "egress_bill_result",
    15: "egress_bill_time",
    16: "egress_client_currency",
    17: "egress_client_currency_id",
    18: "egress_client_id",
    19: "egress_client_name",
    20: "egress_cost",
    21: "egress_dnis_type",
    22: "egress_erro_string",
    23: "egress_id",
    24: "egress_name",
    25: "egress_rate",
    26: "egress_rate_effective_date",
    27: "egress_rate_id",
    28: "egress_rate_table_id",
    29: "egress_rate_table_name",
    30: "egress_rate_type",
    31: "egress_six_seconds",
    32: "egress_trunk_trace",
    33: "end_epoch",
    34: "final_route_indication",
    35: "first_release_dialogue",
    36: "ingress_bill_minutes",
    37: "ingress_client_bill_result",
    38: "ingress_client_bill_time",
    39: "ingress_client_cost",
    40: "ingress_client_currency",
    41: "ingress_client_currency_id",
    42: "ingress_client_id",
    43: "ingress_client_name",
    44: "ingress_client_rate",
    45: "ingress_client_rate_table_id",
    46: "ingress_dnis_type",
    47: "ingress_id",
    48: "ingress_name",
    49: "ingress_rate_effective_date",
    50: "ingress_rate_id",
    51: "ingress_rate_type",
    52: "is_final_call",
    53: "is_manual_kill",
    54: "item_id",
    55: "lnp_dipping_cost",
    56: "lrn_dnis",
    57: "lrn_lata",
    58: "lrn_number_vendor",
    59: "lrn_ocn",
    60: "lrn_rc",
    61: "minutes_west_of_greenwich_mean_time",
    62: "o_billing_method",
    63: "order_type",
    64: "orig_call_duration",
    65: "orig_code",
    66: "orig_code_name",
    67: "orig_country",
    68: "orig_delay_second",
    69: "origination_call_id",
    70: "origination_codec_list",
    71: "origination_destination_host_name",
    72: "origination_destination_number",
    73: "origination_egress_octets",
    74: "origination_egress_packets",
    75: "origination_ingress_delay",
    76: "origination_ingress_octets",
    77: "origination_ingress_packet_jitter",
    78: "origination_ingress_packet_loss",
    79: "origination_ingress_packets",
    80: "origination_local_payload_ip_address",
    81: "origination_local_payload_udp_address",
    82: "origination_profile_port",
    83: "origination_remote_payload_ip_address",
    84: "origination_remote_payload_udp_address",
    85: "origination_source_host_name",
    86: "origination_source_number",
    87: "orig_jur_type",
    88: "o_trunk_type2",
    89: "paid_user",
    90: "par_id",
    91: "pdd",
    92: "q850_cause",
    93: "q850_cause_string",
    94: "record_type",
    95: "release_cause",
    96: "release_cause_from_protocol_stack",
    97: "release_cause_name",
    98: "release_tod",
    99: "rerate_time",
    100: "ring_epoch",
    101: "ring_time",
    102: "route_plan",
    103: "route_plan_name",
    104: "route_prefix",
    105: "routing_digits",
    106: "rpid_user",
    107: "session_id",
    108: "src_lata",
    109: "src_ocn",
    110: "src_rc",
    111: "start_time_of_date",
    112: "static_route",
    113: "static_route_name",
    114: "tax",
    115: "t_billing_method",
    116: "term_code",
    117: "term_code_name",
    118: "term_country",
    119: "term_delay_second",
    120: "termination_call_id",
    121: "termination_codec_list",
    122: "termination_destination_host_name",
    123: "termination_destination_number",
    124: "termination_egress_octets",
    125: "termination_egress_packets",
    126: "termination_ingress_delay",
    127: "termination_ingress_octets",
    128: "termination_ingress_packet_jitter",
    129: "termination_ingress_packet_loss",
    130: "termination_ingress_packets",
    131: "termination_local_payload_ip_address",
    132: "termination_local_payload_udp_address",
    133: "termination_profile_port",
    134: "termination_remote_payload_ip_address",
    135: "termination_remote_payload_udp_address",
    136: "termination_source_host_name",
    137: "termination_source_number",
    138: "term_jur_type",
    139: "time",
    140: "timeout_type",
    141: "translation_ani",
    142: "trunk_id_origination",
    143: "trunk_id_termination",
    144: "t_trunk_type2",
    145: "version_number",
    146: "voip_protocol_origination",
    147: "voip_protocol_termination",
}


class InvoiceSetting(DnlApiBaseModel):
    __tablename__ = 'invoice_settings'
    INVOICE_FORMAT = {1: 'PDF', 2: 'Excel', 3: 'HTML'}
    POSITION = {1: 'top', 2: 'bottom', 3: 'middle'}
    INVOICE_SEND_MODE = {0: 'none', 1: 'link', 2: 'attachment'}
    INVOICE_SEND_MODE_CDR = {0: 'none', 1: 'link', 2: 'attachment'}
    id = Column(BigInteger, primary_key=True)
    company_info = Column(String(2500))
    company_info_location = Column(ChoiceType(POSITION))
    created_at = Column(DateTime(True), server_default=func.now())
    invoice_decimal_digits = Column(Integer)
    invoice_name = Column(String(200))
    invoice_send_mode = Column(ChoiceType(INVOICE_SEND_MODE))
    invoice_send_mode_cdr = Column(ChoiceType(INVOICE_SEND_MODE))
    invoice_setting_name = Column(String(200))
    logo_path = Column(String(256))
    mail_sender_id = Column(Integer)
    overlap_invoice_protection = Column(Boolean)
    pdf_tpl = Column(Text)
    send_cdr_fields = Column(Text)
    tpl_number = Column(ChoiceType(POSITION))
    updated_at = Column(DateTime(True))
    updated_by = Column(String(255))
    invoices_logo_id = Column(Integer)

    billing_info = synonym('pdf_tpl')
    billing_info_position = synonym('tpl_number')
    billing_info_location = synonym('tpl_number')
    create_by = synonym('updated_by')
    create_on = synonym('created_at')
    update_on = synonym('updated_at')
    invoices_cdr_fields = synonym('send_cdr_fields')

    @hybrid_property
    def allow_invoice_overlap(self):
        return not bool(self.overlap_invoice_protection)

    @allow_invoice_overlap.setter
    def allow_invoice_overlap(self, value):
        self.overlap_invoice_protection = not bool(value)

    @property
    def _logo_path(self):
        if self.invoices_logo_id and ImportExportLogs.get(self.invoices_logo_id):
            obj = ImportExportLogs.get(self.invoices_logo_id)
            if obj:
                return obj.file
        if self.logo_path:
            return self.logo_path
        else:
            return SystemParameter.get(1)._logo_path

    @hybrid_property
    def cdr_fields(self):
        try:
            return self.send_cdr_fields.split(',')
        except:
            return []

    @cdr_fields.setter
    def cdr_fields(self, value):
        try:
            self.send_cdr_fields = ','.join([str(x) for x in value])
        except:
            pass


class ClientInvoiceSetting(DnlApiBaseModel):
    __tablename__ = 'client_invoice_settings'

    id = Column(Integer, primary_key=True)
    client_id = Column(ForeignKey('client.client_id', ondelete='CASCADE'), nullable=False)
    invoice_setting_id = Column(ForeignKey('invoice_settings.id', ondelete='CASCADE'))

    client = relationship('Client', back_populates='invoice_setting')
    setting = relationship('InvoiceSetting')


class ClientRateGeneration(DnlApiBaseModel):
    __tablename__ = 'client_rate_generation'
    id = Column(Integer, primary_key=True)
    client_id = Column(ForeignKey('client.client_id'), nullable=False)
    rate_generation_template_id = Column(ForeignKey('rate_generation_template.id', ondelete='CASCADE'))


class Client(DnlApiBaseModel, IStatistic):
    __tablename__ = 'client'
    INVOICE_FORMAT = {1: 'PDF', 2: 'Excel', 3: 'HTML'}
    CDR_LIST_FORMAT = {1: 'Excel', 2: 'CSV', 3: 'zip', 4: 'tar.gz'}
    PROFIT_TYPE = {1: 'percentage', 2: 'value'}
    AUTO_INVOICE_TYPE = {0: 'buy', 1: 'sell', 2: 'both'}
    AUTO_DAILY_BALANCE_RECIPIENT = {0: "Partner Billing Contact", 1: "Owner Billing Contact", 2: "Both"}
    SCC_TYPE = {0: 'meeting the short duration defined neighboring', 1: 'that exceed the defined percentage'}
    AUTO_SUMMARY_GROUP_BY = {0: 'By Country', 1: 'By Code Name', 2: 'By Code'}
    CLIENT_TYPE = {0: 'vendor', 1: 'client', 2: 'both client and vendor'}
    COMPANY_TYPE = {0: 'termination', 1: 'origination', 2: 'both termination and origination'}
    INVOICE_USE_BALANCE_TYPE = {0: 'actual', 1: 'mutual'}
    BREAKDOWN_BY_RATE_TABLE = {1: 'Breakdown A-Z Rate Table by Destination',
                               2: 'Breakdown US Rate Table by Jurisdiction'}
    RATE_VALUE = {1: 'Actual Value', 2: 'Average Value', None: 'null', 0: 'undefined'}
    BILLING_MODE = {1: 'prepay', 2: 'postpay'}
    DID_BILLING_BY = {0: 'by minute', 1: 'by port'}

    client_id = Column(Integer, primary_key=True)

    name = Column(String(500))  # name IS 'name';
    address = Column(String(500))  # address
    allowed_credit = Column(Numeric(30, 10), nullable=False, default=0, server_default='0')  # 'Allow arrears'
    attach_cdrs_list = Column(Boolean)  # whether the attachment bill details
    auto_daily_balance_recipient = Column(ChoiceType(AUTO_DAILY_BALANCE_RECIPIENT),
                                          default=0)  # 0: Partner' s Billing Contact, 1: Owner 's Billing Contact, 2: Both'
    auto_invoice_type = Column(ChoiceType(AUTO_INVOICE_TYPE), nullable=False, default=2,
                               server_default='2')  # 0: buy (client), 1: sell (vendor), 2: both
    auto_invoicing = Column(Boolean, nullable=False, default=False)  # Automatically generate invoice
    auto_send_zone = Column(String, server_default=text_("'+00:00'"))
    auto_summary_group_by = Column(ChoiceType(AUTO_SUMMARY_GROUP_BY),
                                   default=0)  # 0: By Country; 1: By Code Name; 2: By Code
    auto_summary_hour = Column(Integer)
    auto_summary_include_cdr = Column(Boolean, nullable=False, default=False)
    auto_summary_not_zero = Column(SmallInteger, default=0)
    auto_summary_period = Column(Integer, default=24)  # 1,2,4,6,8,12,24, -15, -30
    billing_email = Column(String)
    breakdown_by_rate_table = Column(ChoiceType(
        BREAKDOWN_BY_RATE_TABLE))  # 1: Breakdown A-Z Rate Table by Destination, 2: Breakdown US Rate Table by Jurisdiction';
    call_limit = Column(Integer)
    carrier_template_id = Column(Integer, nullable=False,
                                 default=0)  # using the Carrier Template template, defaults to 0, does not use the template
    cdr_format = synonym('cdr_list_format')
    cdr_list_format = Column(ChoiceType(CDR_LIST_FORMAT))  # bill format: 1-Excel, 2-CSV, 3-zip, 4-tar.gz
    client_type = Column(ChoiceType(CLIENT_TYPE))  # did: 0 vendor, 1 client
    company = Column(String)  # company name
    company_type = Column(ChoiceType(COMPANY_TYPE))  # did: 0 vendor, 1 client
    corporate_contact_email = Column(String(200))
    cps_limit = Column(Integer)
    create_time = Column(DateTime(True), nullable=False, server_default=func.now())  # useless - ready to delete
    currency_id = Column(ForeignKey('currency.currency_id', ondelete='SET NULL'), nullable=False, index=True, default=1)
    daily_balance_already_sent = Column(Boolean, default=False)
    daily_balance_notification = Column(Integer)
    daily_balance_recipient = Column(Integer)
    daily_balance_send_time = Column(Time)
    daily_balance_send_time_zone = Column(String(20))
    daily_cdr_generation = Column(Boolean, default=False)
    daily_cdr_generation_type = Column(SmallInteger, default=0)
    daily_cdr_generation_zone = Column(String, server_default=text_("'+00:00'"))
    daily_limit = Column(Integer)
    database_name = Column(String(50))
    decimals_num = Column(Integer)
    details = Column(String(1000))
    email = Column(String(200))  # email

    enable_auto_invoice = Column(Boolean, default=False)
    enable_auto_report = Column(Boolean, default=False)
    enable_client_portal = Column(Boolean, default=False)
    enable_notification = Column(Boolean, default=False)
    enable_short_dur_penalty = Column(Boolean, default=False)
    enable_payment_alert = Column(Boolean, default=False)
    enough_balance = Column(Boolean, default=True)  # if the balance is insufficient
    enable_paypal = Column(Boolean, default=False, server_default=text_('false'))
    enable_strip = Column(Boolean, default=False, server_default=text_('false'))
    enable_sign_in_notification = Column(Boolean, default=False, server_default=text_('true'))
    enable_trunk_view = Column(Boolean, default=False, server_default=text_('false'))
    enable_trunk_edit = Column(Boolean, default=False, server_default=text_('false'))
    finance_email_cc = Column(String)
    format = synonym('invoice_format')
    group_id = Column(ForeignKey('carrier_group.group_id', ondelete='SET NULL'), index=True)
    hourly_limit = Column(Integer)
    include_available_credit = Column(SmallInteger, default=0)
    include_payment_history = Column(SmallInteger, default=0)
    include_payment_history_days = Column(Integer, default=1)
    invoice_include_payment = Column(Boolean)
    invoice_jurisdictional_detail = Column(Boolean, default=False)
    invoice_past_amount = Column(Numeric(30, 10), default=0)
    invoice_show_details = Column(Boolean, default=False)
    invoice_start_from = Column(Date)
    invoice_use_balance_type = Column(ChoiceType(INVOICE_USE_BALANCE_TYPE), default=0)  # 0-actual, 1-mutual'
    is_active_call = Column(Boolean, default=False)
    is_auto_balance = Column(Boolean, default=False)
    is_auto_summary = Column(Boolean, default=False)
    is_bill_mismatch = Column(Boolean, default=False)
    is_breakdown_by_rate_table = Column(Boolean)  # invoice whether to display Break Down by Rate Table
    is_call_simulation = Column(Boolean, default=False)
    is_capture = Column(Boolean, default=False)
    is_daily_balance_notification = Column(Boolean, default=False)
    is_discon = Column(Boolean, default=False)
    is_egress_sim = Column(Boolean, default=False)
    is_ingress_sim = Column(Boolean, default=False)
    is_invoice_account_summary = Column(Boolean, default=False)
    is_link_cdr = Column(Boolean, default=True)  # can link to download cdr
    is_location = Column(Boolean, default=False)
    is_manage = Column(Boolean, default=False)
    is_orig_term = Column(Boolean, default=False)
    is_panel_accountsummary = Column(Boolean, default=True)  # can access Account Summary
    is_panel_balance = Column(Boolean, default=True)  # can access Balance
    is_panel_cdrslist = Column(Boolean, default=True)  # Can access CDRs List
    is_panel_cid_blocking = Column(Boolean, default=True)
    is_panel_didrequest = Column(Boolean, default=True)  # can access DID Request
    is_panel_invoices = Column(Boolean, default=True)  # can access Invoices
    is_panel_mydid = Column(Boolean, default=False)  # can access My DID
    is_panel_onlinepayment = Column(Boolean, default=True)  # can access Online Payment
    is_panel_paymenthistory = Column(Boolean, default=True)  # can access Payment History
    is_panel_products = Column(Boolean, default=True)  # can access products
    is_panel_ratetable = Column(Boolean, default=True)  # can access RateTable
    is_panel_sippacket = Column(Boolean, default=True)  # Can access SIP PACKET Search
    is_panel_summaryreport = Column(Boolean, default=True)  # can access Summary Report
    is_panel_trunks = Column(Boolean, default=True)  # can access Trunks
    is_panel_block_list = Column(Boolean, default=True)  # can access Block List
    is_panelaccess = Column(Boolean)  # can access the client template - no longer use, ready to delete
    is_qos = Column(Boolean, default=False)
    is_rate_anal = Column(Boolean, default=False)
    is_report = Column(Boolean, default=False)
    is_route = Column(Boolean, default=False)
    is_send_trunk_update = Column(Boolean, default=False)  # whether to send mail when changing its trunk ip
    is_show_by_date = Column(Boolean, default=False)
    is_show_detail_trunk = Column(Boolean, default=False)
    is_show_total_trunk = Column(Boolean, default=False)
    is_spam = Column(Boolean, default=False)
    is_termin = Column(Boolean, default=False)
    is_tools = Column(Boolean, default=False)
    is_trunk = Column(Boolean, default=False)
    is_usage = Column(Boolean, default=False)
    is_vendor_invoice = Column(Boolean, default=False)
    last_autobalance_time = Column(DateTime(True))
    last_autoreport_time = Column(DateTime(True))
    last_cdrdown_time = Column(DateTime(True))
    last_invoiced = Column(DateTime(True), server_default=func.now())  # last invoice generation date
    last_lowbalance_time = Column(DateTime(True))
    login = Column(String(60), unique=True)  # customer template access account, customer login account
    logo = Column(String(100))  # logo
    low_balance_notice = Column(Boolean, nullable=False, default=True)  # whether to provide balance insufficient hints
    low_balance_notification_time_cycle = Column(Integer, default=1)
    low_balance_notification_time_type = Column(Integer, default=0)
    low_balance_number = Column(Integer)
    mail_sended = Column(Integer, default=0)  # 0 - no send, 1 - balance insufficient send
    mode = Column(ChoiceType(BILLING_MODE), nullable=False, default=1)  # PAYMENT MODE: 1 - Prepaid, 2 postpaid
    # mode = Column(Integer, nullable=False, default=1)  # PAYMENT MODE: 1 - Prepaid, 2 postpaid
    noc_email = Column(String(100))
    notify_admin_balance = Column(Numeric(30, 10), default=0)  # prompt system balance
    notify_client_balance = Column(Numeric(30, 10), default=0)  # prompt customer balance
    notify_client_balance_type = Column(Integer, default=0)
    numer_of_days_balance = Column(Integer, default=1)
    offset_balance = Column(Boolean, default=True)
    orig_rate_table_id = Column(Integer, index=True)
    par_id = Column(Integer)
    password = Column(String(60))  # client template access password
    payment_received_notice = Column(Boolean)

    paypal = Column(String(100))
    phone = Column(String(20))
    profit_margin = Column(Float(53), nullable=False, default=0)  # minimum profit margin
    profit_type = Column(ChoiceType(PROFIT_TYPE), default=1)  # 1 - Percentage, 2 - value
    rate_delivery_email = Column(String(250))
    rate_email = Column(String(100))
    role_id = Column(Integer, index=True)  #
    scc_bellow = Column(Integer, default=0)  # call_duration Call time is less than this value of a short call
    scc_charge = Column(Numeric(30, 10))  # short call amount
    scc_percent = Column(Integer,
                         default=0)  # short call answered * 100 / all greater than this value in the invoice to add a brief call fine
    scc_type = Column(Integer,
                      default=0)  # 0 meeting the short duration defined neighboring, 1 that exceed the defined percentage
    service_charge_id = Column(Integer)
    status = Column(Boolean, nullable=False, default=True)  ##Status: true-active, false-inactive
    tax = Column(Numeric(5, 2))
    tax_id = Column(String(100))
    term_rate_table_id = Column(Integer, index=True)  # 'cost - useless, ready to delete';
    transaction_fee_id = Column(Integer)
    unlimited_credit = Column(Boolean, default=False)
    update_at = Column(DateTime(True), onupdate=func.now())
    update_by = Column(String)
    usage_detail_fields = Column(Text)
    user_id = Column(Integer)
    vendor_payment_term_id = Column(Integer)
    zero_balance_notice = Column(Boolean)
    zero_balance_notice_first_sended = Column(Boolean, default=False)
    zero_balance_notice_last_sent = Column(DateTime(True))
    zero_balance_notice_time = Column(Integer, default=0)

    # invoice setting fields
    payment_term_id = Column(ForeignKey('payment_term.payment_term_id', ondelete='SET NULL'), index=True)
    invoice_format = Column(ChoiceType(INVOICE_FORMAT))  # Generate Invoice Format: 1-PDF, 2-Excel, 3-HTML
    invoice_hour = Column(Integer)
    invoice_zone = Column(String(10))
    decimal_place = Column(Integer, default=5)  # number of decimal places, default 5 bits
    rate_value = Column(ChoiceType(RATE_VALUE), default=1)  # rate_value 1 - Actual Value 2 - Average Value
    invoice_zero = Column(Boolean, default=True)  # default does not generate

    include_tax = Column(Boolean, default=False)
    email_invoice = Column(Boolean, default=False)
    is_send_as_link = Column(Boolean, default=False)
    is_show_daily_usage = Column(Boolean, default=False)
    is_short_duration_call_surcharge_detail = Column(Boolean, default=False)

    is_show_code_100 = Column(Boolean, default=False)
    is_show_code_name = Column(Boolean, default=False)
    is_show_country = Column(Boolean, default=False)

    mask_did_start_post = Column(Integer)
    mask_did_end_post = Column(Integer)
    
    break_call_on_no_balance = Column(Boolean, nullable=False, default=False)

    # client_invoice_settings_id = Column(ForeignKey('client_invoice_settings.id', ondelete='SET NULL'), index=True)
    _client_invoice_settings_id = column_property(select([ClientInvoiceSetting.invoice_setting_id]).where(
        ClientInvoiceSetting.client_id == client_id).limit(1).correlate_except(ClientInvoiceSetting))
    did_billing_by = Column(ChoiceType(DID_BILLING_BY), server_default='0')
    # fee_per_port = Column(Numeric(30, 10))
    login_ip = column_property(select([User.login_ip]).where(User.user_id == user_id).correlate_except(User))

    is_online = column_property(select([User.is_online == 1]).where(User.user_id == user_id).correlate_except(User))
    # CLIENT_TYPE = { 0: 'vendor', 1: 'client', 2: 'both client and vendor'}
    # COMPANY_TYPE = {0: 'termination', 1: 'origination', 2: 'both termination and origination'}
    is_vendor = column_property(client_type.in_(['vendor', 'both client and vendor']) == True)
    is_client = column_property(client_type.in_(['client', 'both client and vendor']) == True)
    is_orig = column_property(company_type.in_(['origination', 'both termination and origination']) == True)
    is_term = column_property(company_type.in_(['termination', 'both termination and origination']) == True)
    credit_limit = column_property(-allowed_credit)

    client_name = synonym('name')
    auto_send_invoice = synonym('email_invoice')
    daily_usage_group_by = synonym('auto_summary_group_by')
    daily_balance_summary = synonym('is_daily_balance_notification')
    daily_usage_summary = synonym('is_show_daily_usage')
    daily_cdr_delivery = synonym('daily_cdr_generation')
    main_email = synonym('email')
    billing_method = synonym('profit_type')
    username = synonym('login')
    active = synonym('status')
    is_active = synonym('status')
    send_invoice_as_link = synonym('is_send_as_link')
    is_unlimited = synonym('unlimited_credit')
    scc_below = synonym('scc_bellow')
    port_limit = synonym('call_limit')
    channel_limit = synonym('call_limit')
    company_name = synonym('company')

    payment_term_name = column_property(
        select([PaymentTerm.name]).where(PaymentTerm.payment_term_id == payment_term_id).correlate_except(PaymentTerm))

    currency = relationship(Currency)
    group = relationship(CarrierGroup, back_populates='clients_detail', uselist=False)

    payment_term = relationship(PaymentTerm, back_populates='clients', uselist=False)

    agents = relationship(AgentClient, back_populates='client', uselist=True, single_parent=True,
                          cascade="all, delete-orphan")
    invoices = relationship('Invoice', back_populates='client', uselist=True, single_parent=True,
                            cascade="all, delete-orphan")
    last_invoice_history = relationship('InvoiceHistory',
                                        order_by="desc(InvoiceHistory.id)",
                                        primaryjoin="foreign(InvoiceHistory.client_id)==Client.client_id",
                                        back_populates='client', uselist=False, single_parent=True,
                                        cascade="all, delete-orphan")

    payments_history = relationship('PaymentGatewayHistory', uselist=False, back_populates='client')

    default_ip = relationship('ClientDefaultIp', uselist=True, back_populates='client', single_parent=True,
                              cascade="all, delete-orphan")
    client_taxes = relationship('ClientTaxes', uselist=True, back_populates='client', single_parent=True,
                                cascade="all, delete-orphan")

    user = relationship(User, back_populates='client',
                        primaryjoin='foreign(Client.user_id) == User.user_id', uselist=False)
    # foreign(Client.user_id) == User.user_id
    resources = relationship('Resource', uselist=True, back_populates='client')
    resource = relationship('Resource', uselist=False,
                            primaryjoin='and_(Resource.client_id==Client.client_id,foreign(Resource.alias)==Client.name,Resource.purged==False)'
                            , single_parent=True, cascade="all, delete-orphan")

    ip_addresses = relationship('ResourceIp',
                                secondary="join(Resource, ResourceIp, Resource.resource_id == ResourceIp.resource_id)",
                                primaryjoin='Resource.client_id==Client.client_id',
                                secondaryjoin='ResourceIp.resource_id==Resource.resource_id',
                                viewonly=True, uselist=True)
    payment = relationship('ClientPayment', uselist=True, back_populates='client',
                           single_parent=True,
                           cascade="all, delete-orphan")
    c4 = relationship('C4ClientBalance',
                      primaryjoin="Client.client_id == foreign( cast(C4ClientBalance.client_id,Integer) )",
                      viewonly=True, uselist=True)
    low_balance_config = relationship('CarrierLowBalanceConfig', uselist=False, back_populates='carrier',
                                      single_parent=True,
                                      cascade="all, delete-orphan"
                                      )
    c4 = relationship('C4ClientBalance', uselist=False,
                      # back_populates='carrier',
                      primaryjoin="foreign(Client.client_id) == cast(C4ClientBalance.client_id,Integer)"
                      )
    balance_actual = relationship('BalanceHistoryActual', back_populates='carrier', uselist=True, lazy='dynamic',
                                  single_parent=True,
                                  cascade="all, delete-orphan")
    balance_mutual = relationship('BalanceHistory', back_populates='carrier', uselist=True, lazy='dynamic',
                                  single_parent=True,
                                  cascade="all, delete-orphan")
    egress_trunks = relationship('EgressTrunk',
                                 primaryjoin='and_(EgressTrunk.client_id == Client.client_id,EgressTrunk.egress == True,IngressTrunk.purged == False)'
                                 )
    ingress_trunks = relationship('IngressTrunk',
                                  primaryjoin='and_(IngressTrunk.client_id == Client.client_id,IngressTrunk.ingress == True,IngressTrunk.purged == False)'
                                  )
    origination_trunks = relationship('Resource',
                                      primaryjoin='and_(Resource.client_id == Client.client_id,foreign(Resource.alias)==Client.name,Resource.trunk_type2 == 1)'
                                      )
    client_limits = relationship('UsersLimit', uselist=True, back_populates='client', cascade="all, delete-orphan")

    vendor_api = relationship('DidVendor', uselist=False, back_populates='vendor', cascade="all, delete-orphan")

    invoice_setting = relationship(ClientInvoiceSetting, uselist=False, back_populates='client',
                                   cascade="all, delete-orphan")

    did_product = relationship('ClientDidProduct', uselist=False, back_populates='client',
                               cascade="all, delete-orphan")
    __is_new = None

    @property
    def res_id_list(self):
        res = Resource
        return [int(r.resource_id) for r in res.filter(res.client_id == self.client_id)]

    @property
    def _resources(self):
        return [resource for resource in self.resources if not resource.purged]

    @property
    def invoice_conf(self):
        if self.invoice_setting and self.invoice_setting.setting:
            return self.invoice_setting.setting
        else:
            return SystemParameter.query().first()

    @property
    def invoice_send_mode_cdr(self):
        if self.attach_cdrs_list:
            return 'link'
        return None

    # ++IStatistic implementations
    def _ingress(self):
        return ','.join([str(tr.resource_id) for tr in self.ingress_trunks]) if self.ingress_trunks else ''

    def _egress(self):
        return ','.join([str(tr.resource_id) for tr in self.egress_trunks]) if self.egress_trunks else ''

    @validates('is_panel_didrequest')
    def validate_is_panel_didrequest(self, key, value):
        if self.is_term:
            return False
        else:
            return value

    @hybrid_property
    def non_zero_invoice_only(self):
        return not self.invoice_zero

    @non_zero_invoice_only.setter
    def non_zero_invoice_only(self, value):
        self.invoice_zero = not value

    @hybrid_property
    def vendor_tech_prefix(self):
        try:
            return self.resource.vendor_tech_prefix
        except:
            return ""
        return ""

    @hybrid_property
    def auth_ip(self):
        user = User.filter(User.user_id == self.user_id).first()
        return user.auth_ip if user else None

    @auth_ip.setter
    def auth_ip(self, value):
        user = User.filter(User.user_id == self.user_id).first()
        if user:
            user.auth_ip = value
            user.save()

    @hybrid_property
    def _enable_auto_report(self):
        return self.enable_auto_report

    @_enable_auto_report.setter
    def _enable_auto_report(self, value):
        if value:
            self.enable_auto_report = value
        else:
            self.enable_auto_report = False
            self.daily_cdr_generation = False
            self.show_account_summary = False
            self.enable_sign_in_notification = False
            self.is_daily_balance_notification = False
            self.is_show_daily_usage = False

    @property
    def _trunks_term_ids(self):
        return ','.join([str(tr.resource_id) for tr in self.resources if tr.trunk_type2 == 'Termination Traffic'])

    @property
    def _trunks_term_az_ids(self):
        return ','.join(
            [str(tr.resource_id) for tr in Resource.filter(
                and_(Resource.client_id == self.client_id, ResourcePrefix.resource_id == Resource.resource_id,
                     RateTable.rate_table_id == ResourcePrefix.rate_table_id, RateTable.jur_type == 0,
                     Resource.trunk_type2 == 'Termination Traffic'
                     )).all()])

    @property
    def _trunks_term_us_ids(self):
        return ','.join(
            [str(tr.resource_id) for tr in Resource.filter(
                and_(Resource.client_id == self.client_id, ResourcePrefix.resource_id == Resource.resource_id,
                     RateTable.rate_table_id == ResourcePrefix.rate_table_id, RateTable.jur_type.in_([1, 2]),
                     Resource.trunk_type2 == 'Termination Traffic')).all()])

    @property
    def _trunks_orig_ids(self):
        return ','.join([str(tr.resource_id) for tr in self.resources if tr.trunk_type2 == 'DID Traffic'])

    @property
    def ingress_ids(self):
        return self._ingress()

    @property
    def egress_ids(self):
        return self._egress()

    @property
    def dnis_cap_limit(self):
        return self.resource.dnis_cap_limit if self.resource else None

    @dnis_cap_limit.setter
    def dnis_cap_limit(self, value):
        Resource.filter(Resource.client_id == self.client_id).update({'dnis_cap_limit': value}, synchronize_session=False)
        # if self.resource:
        #     self.resource.dnis_cap_limit = value
        #     self.resource.save()

    @property
    def allowed_ports(self):
        return self.resource.allowed_ports if self.resource else None

    @allowed_ports.setter
    def allowed_ports(self, value):
        # Resource.filter(Resource.client_id == self.client_id).update({'allowed_ports': value}, synchronize_session=False)
        if self.resource:
            self.resource.allowed_ports = value
            self.resource.save()

    @property
    def ani_cap_limit(self):
        return self.resource.ani_cap_limit if self.resource else None

    @ani_cap_limit.setter
    def ani_cap_limit(self, value):
        Resource.filter(Resource.client_id == self.client_id).update({'ani_cap_limit': value}, synchronize_session=False)

    # --IStatistic implementations
    def before_save(self):
        log.debug('Client before save')

        if self.company:
            old_obj = Client.filter(
                and_(Client.company == self.company, Client.client_id != self.client_id)).first()
            if old_obj:
                raise ValidationError(
                    'company_name {} is duplicate! (duplicate client_id {})'.format(
                        self.company, self.client_id))

        agent = AgentClient.filter(AgentClient.client_id == self.client_id).first()
        if agent and agent.agent.invoice_setting_id:
            agent = agent.agent
            self._client_invoice_settings_id = agent.invoice_setting_id

        if self.resource:
            if self.resource.cps_limit and self.cps_limit and self.cps_limit < self.resource.cps_limit:
                raise ValidationError('Resource\'s cps_limit shouldn\'t exceed client\'s cps_limit')
            if self.resource.call_limit and self.call_limit and self.call_limit < self.resource.call_limit:
                raise ValidationError('Resource\'s call_limit shouldn\'t exceed client\'s call_limit')

        # super(Client,self).before_save()
        # if self.client_id:
        # Resource.filter(Resource.client_id == self.client_id).update({'enough_balance': self.enough_balance})
        #    pass

        if self.mode == 'prepay' and self.client_type != 'vendor':
            # if hasattr(self,'_test_credit' ) and self._test_credit:
            #    self._test_credit = None
            #    raise ValidationError('Cannot set test_credit when mode is prepay!')
            # if self.allowed_credit:
            #     self.allowed_credit = 0.0
            #    raise ValidationError('Cannot set credit limit when mode is prepay!')
            if self.unlimited_credit:
                self.unlimited_credit = False
            #    raise ValidationError('Cannot set unlimited credit when mode is prepay!')
            pass

        if self.unlimited_credit:
            if self.allowed_credit:
                self.allowed_credit = 0.0
        log.debug('Client before save allowed credit {}'.format(self.allowed_credit))
        # if self.allowed_credit is None or self.mode == 'prepay':
        #     self.allowed_credit = 0.0
        try:
            get_db(self.db_label).session.autoflush = False
            if self.resource:
                log.debug('Client resource before save')
                self.resource.before_save()
        except Exception as e:
            log.error('Client before save error {}'.format(traceback.print_exc()))
            get_db(self.db_label).session.autoflush = True
        # for r in self.resources:
        #    r.before_save()

        if self.enable_notification is False:
            self.is_send_trunk_update = False
            self.payment_received_notice = False
            self.zero_balance_notice = False
        
        if not self.auto_invoicing:
            self.payment_term_id = None
        self.__is_new = not bool(self.client_id)
        log.debug('Client __is_new {}'.format(self.__is_new))

    def after_save(self):
        log.debug('Client resource after save')
        balance = 0.0

        if self.company_type == 'termination' and self.is_panel_didrequest:
            self.is_panel_didrequest = False

        _p_changed = False
        if not self.user and self.login and self.password:
            self.user = User(name=self.login, passwd=self.password)
            _p_changed = True
        if self.user:
            try:
                if self.user.active != self.status:
                    self.user.active = self.status
                    _p_changed = True
                if self.user.client_id != self.client_id:
                    self.user.client_id = self.client_id
                    _p_changed = True
                if self.user.name != self.login:
                    self.user.name = self.login
                    _p_changed = True
                try:
                    if not bcrypt.checkpw(self.password.encode('utf-8'), self.user.password.encode('utf-8')):
                        self.user.passwd = self.password
                        _p_changed = True
                except:
                    self.user.passwd = self.password
                    _p_changed = True
                if not self.user.email and self.email and len(self.email.split(';')):
                    self.user.email = self.email.split(';')[0]
                    _p_changed = True
                conf = SystemParameter.get(1)
                if conf and not self.user.landing_page:
                    self.user.landing_page = conf.base_url
                    _p_changed = True
                if not self.user.role_id:
                    self.user.role_id = 1
                    _p_changed = True
                if self.user.user_type != 'client':
                    self.user.user_type = 'client'
                if _p_changed:
                    log.debug('Client resource after save user changed!')
                    self.user.save()
            except Exception as e:
                log.error('save client {} user error {}'.format(self.client_id, str(e)))

        if self.__is_new:

            ClientBalanceOperationAction.add_create(self.client_id, Decimal(0.0), self.update_by + '_create_balance')
            log.debug('Client after create balance!')

            # if not C4ClientBalance.filter(cast(C4ClientBalance.client_id,Integer) == self.client_id).first():
            #    C4ClientBalance(client_id=str(self.client_id),balance=balance,ingress_balance=balance).save()
            if not CarrierLowBalanceConfig.filter(CarrierLowBalanceConfig.client_id == self.client_id).first():
                CarrierLowBalanceConfig(client_id=self.client_id).save()
            try:
                show_users = User.filter(User.show_carrier_trunk_drop_only.is_(True)).all()
                for usr in show_users:
                    if not UsersLimit.filter(
                            and_(UsersLimit.user_id == usr.user_id, UsersLimit.client_id == self.client_id)
                    ).first():
                        UsersLimit(user_id=usr.user_id, client_id=self.client_id).save()
                log.debug('Added client_id={} to UsersLimit for {} users with show_carrier_trunk_drop_only=True'.format(self.client_id, len(show_users)))
            except Exception as e:
                log.error('Error adding UsersLimit entries for client {}: {}'.format(self.client_id, str(e)))
        if self.mode == 'prepay' and hasattr(self, '_test_credit') and self._test_credit:
            balance = self._test_credit
            # ClientBalanceOperationAction.add_create(self.client_id, Decimal(0.0),self.update_by+'_test_credit')
            if self.test_credit_value:
                ClientPayment(client_id=self.client_id, amount=self.test_credit_value, payment_type='payment sent',
                            description='test credit', update_by=self.update_by,
                            receiving_time=self.create_time).save()
            ClientPayment(client_id=self.client_id, amount=balance, payment_type='Prepayment Received',
                          description='test credit', update_by=self.update_by,
                          receiving_time=self.create_time).save()
            del self._test_credit
        try:
            Resource.filter(Resource.client_id == self.client_id).update({'enough_balance': self.enough_balance},
                                                                         synchronize_session='fetch')

            Resource.filter(and_(Resource.status == 1, Resource.client_id.is_(None))).update({'status': 0},
                                                                                             synchronize_session='fetch')

            Resource.session().commit()

            pass
        except:
            Resource.session().rollback()
            pass

    # def ingress_count(self):
    #    return Resource.filter(Resource.client_id==self.client_id).filter(Resource.ingress==True).count()
    # def egress_count(self):
    #    return Resource.filter(Resource.client_id==self.client_id).filter(Resource.egress==True).count()

    @hybrid_property
    def auto_summary_not_zero_bool(self):
        try:
            return bool(self.auto_summary_not_zero)
        except:
            return True

    @auto_summary_not_zero_bool.setter
    def auto_summary_not_zero_bool(self, value):
        try:
            self.auto_summary_not_zero = int(value)
        except:
            self.auto_summary_not_zero = 1

    def get_product_trunk(self, product_id):
        p = ProductRoutRateTable.get(product_id)
        if p:
            return Resource.filter(Resource.client_id == self.client_id).filter(
                Resource.product_id == p.id).first()

    def add_product_trunk(self, product_id):
        p = ProductRoutRateTable.get(product_id)
        if p:
            tr = self.get_product_trunk(product_id)
            if not tr:
                trunk = Resource(product_id=p.id, route_strategy_id=p.route_plan_id, rate_table_id=p.rate_table_id,
                                 name=self.name + '_' + p.name, alias=self.name)
                pref = ResourcePrefix(tech_prefix=p.tech_prefix)
                trunk.prefixes.append(pref)
                self.resources.append(trunk)

    def del_product_trunk(self, product_id):
        p = ProductRoutRateTable.get(product_id)
        if p:
            tr = self.get_product_trunk(product_id)
            if tr:
                self.resources.remove(tr)
            tr.delete()

    def balance(self):
        return self.actual_balance

    @hybrid_property
    def _actual_balance(self):
        # d = str(datetime.today().date())
        if self.c4:
            return self.c4.balance
        else:
            return 0.0

        d = self.query().session.query(
            func.max(BalanceHistoryActual.date).label('d')).filter(
            BalanceHistoryActual.client_id == self.client_id).first().d
        try:
            return BalanceHistoryActual.filter(BalanceHistoryActual.client_id == self.client_id).filter(
                BalanceHistoryActual.date == d).first().actual_balance
        except:
            return 0.0

    @hybrid_property
    def _credit_limit(self):
        if self.allowed_credit:
            return -self.allowed_credit
        else:
            return None

    @_credit_limit.setter
    def _credit_limit(self, value):
        if self and value:
            self.allowed_credit = -Decimal(value)
        else:
            self.allowed_credit = 0.0

    @hybrid_property
    def ip_address(self):
        return self.ip_addresses
        ret = []
        try:
            for res in self.resources:
                ret.append(res)
            return ret
        except:
            return ''

    @ip_address.setter
    def ip_address(self, value):
        try:
            if len(self.resources):
                res = self.resources[0]  # TODO: what resource to take?
            else:
                res = Resource(name=self.name, alias=self.name, client_id=self.client_id)
                self.resources.append(res)
                res = self.resources[0]
            for v in value:
                dir = 1
                if self.ingress:
                    dir = 0
                ip = ResourceIp(ip=value.ip, port=v.port, direction=dir)
                res.append(ip)

        except:
            return None

    @hybrid_property
    def billing_mode(self):
        try:
            return self.mode
        except:
            return ''

    @billing_mode.setter
    def billing_mode(self, value):
        try:
            self.mode = value
        except:
            self.mode = 1
            pass

    @hybrid_property
    def test_credit(self):
        return self.test_credit_value
        """
        try:
            s=sum([p.amount for p in self.payment if ( p.payment_type == 'prepay payment received' and p.description == 'test credit') ])
            return s
        except Exception as e:
            log.debug(str(e))
            return 0
        """

    @test_credit.setter
    def test_credit(self, value):
        self._test_credit = value

    @hybrid_property
    def template_name(self):
        if self.carrier_template_id != 0:
            templ = CarrierTemplate.filter(CarrierTemplate.id == self.carrier_template_id).first()
            if templ:
                return templ.template_name
        return None

    @template_name.setter
    def template_name(self, value):
        templ = CarrierTemplate.filter(CarrierTemplate.template_name == value).first()
        if templ:
            self.carrier_template_id = templ.id

    def create_user(self):
        if not self.user:
            if not (self.login and self.password):
                return None
            if not self.login:
                self.login = self.name
            if not self.password:
                self.password = self.name
            q = User.filter(User.name == self.login).first()
            if q:
                return None
            self.user = User(name=self.login, passwd=self.password)
            if not self.user.email and self.email and len(self.email.split(';')):
                self.user.email = self.email.split(';')[0]
            self.user.user_type = 'client'
            if Role.get(1):
                self.user.role_id = 1
            else:
                self.user.user_type = 'client'
            self.user.client_id = self.client_id
            return self.user.save()
        else:
            return self.user_id

    def apply_mail(self, template, to=None, att=[]):
        # if not to:
        #    to=self.email
        # class Ret:
        #    pass
        # obj=Ret()
        # for k,v in self.__dict__.items():
        #    obj.__dict__['client_'+k]=v
        #    obj.__dict__[k] = v
        if not to:
            to = self.email
        return MailSender.apply_mail(self, template, to, att=att)

    def _past_due(self):
        state = aliased(Invoice.state, 'inv_state')
        return Invoice.query().session.query(func.sum(func.coalesce(Invoice.total_amount, 0) -
                                                      func.coalesce(Invoice.pay_amount, 0)).label('total')). \
            filter(and_(Invoice.client_id == self.client_id, Invoice.paid.isnot(True), state == 9))

    def past_due(self):
        class Rec:
            pass

        rec = Rec()

        today = datetime.today()
        p7 = today - timedelta(days=7)
        p15 = today - timedelta(days=15)
        p30 = today - timedelta(days=30)
        i_d_d = Invoice.due_date
        qtotal = self._past_due().filter(i_d_d < today)
        rec.total = qtotal.one().total
        rec.past_7 = self._past_due().filter(and_(i_d_d < today, i_d_d >= p7)).one().total
        rec.past_15 = self._past_due().filter(and_(i_d_d < p7, i_d_d >= p15)).one().total
        rec.past_30 = self._past_due().filter(and_(i_d_d < p15, i_d_d >= p30)).one().total
        rec.past_gt_30 = self._past_due().filter(i_d_d < p30).one().total
        rec.balance = self.balance()
        return rec

    def regenerate_balance(self):
        td = datetime.today()
        yd = datetime.today() - timedelta(hours=24)
        d0 = date(yd.year, yd.month, yd.day)
        d1 = date(td.year, td.month, td.day)

        incoming_q = BalanceHistoryActual.filter("client_id = %d" % self.client_id).filter(
            "date='%s'" % d1).first()
        if incoming_q:
            incoming = incoming_q.actual_ingress_balance
            if incoming is None:
                incoming = 0.0
        else:
            incoming = 0.0
        pays = ClientPayment.filter("client_id = %d" % self.client_id).filter("payment_time>'%s'" % d1).all()
        psent = sum([p.total for p in pays if p.payment_type in ('invoice payment sent', 'payment sent')])
        precv = sum(
            [p.total for p in pays if p.payment_type in ('invoice payment received', 'Prepayment Received')])
        drecv = sum([p.total for p in pays if p.payment_type in ('debit note received',)])
        dsent = sum([p.total for p in pays if p.payment_type in ('debit note sent',)])
        crecv = sum([p.total for p in pays if p.payment_type in ('credit note received',)])
        csent = sum([p.total for p in pays if p.payment_type in ('credit note sent',)])

        scc = sum([p.total for p in pays if p.payment_type in ('scc',)])

        sump = sum([p.total for p in pays])
        payment_received = float(sump)
        outcoming = float(incoming) + float(sump)

        q = BalanceHistoryActual.filter("client_id = %d" % self.client_id).filter(
            "date='%s'" % d1).first()
        if not q:
            q = BalanceHistoryActual(client_id=self.client_id, date=d1)
        if q:
            q.actual_balance = outcoming
            q.actual_egress_balance = outcoming
            q.payment_received = precv
            q.payment_sent = psent
            q.debit_received = drecv
            q.debit_sent = dsent
            q.credit_note_received = crecv
            q.credit_note_sent = csent
            q.short_charges = scc
            q.save()

        q = C4ClientBalance.filter(C4ClientBalance.client_id == str(self.client_id)).first()
        if not q:
            q = C4ClientBalance(client_id=self.client_id)
        if q:
            q.balance = outcoming
            q.ingress_balance = incoming
            q.egress_balance = outcoming
            q.save()

        q = BalanceHistory.filter("client_id = %d" % self.client_id).filter(
            "date='%s'" % d1).first()
        if not q:
            q = BalanceHistory(client_id=self.client_id, date=d1)
        if q:
            if q.mutual_ingress_balance:
                outcoming = q.mutual_ingress_balance + sump
            else:
                outcoming = sump
            q.mutual_balance = outcoming
            q.mutual_egress_balance = outcoming
            q.payment_received = precv
            q.payment_sent = psent
            q.debit_received = drecv
            q.debit_sent = dsent
            q.credit_note_received = crecv
            q.credit_note_sent = csent
            q.short_charges = scc
            q.save()

    def payment_terms(self):
        return self.payment_term.name

    @property
    def invoice_cycle(self):
        return self.payment_term.cycle

    @property
    def next_invoice_date(self):
        if self.payment_term:
            if self.last_invoice_history:
                return self.payment_term.next_date(self.last_invoice_history.invoice_generation_time)
            else:
                return self.payment_term.next_date(None)
        return None

    def is_auto_report_time(self, t):

        zone_hours = get_time_zone_hour(self.auto_send_zone)
        t_client_tz = t + timedelta(hours=zone_hours)
        client_hours = t_client_tz.hour
        if self.auto_summary_hour is None:
            return client_hours == 0
        else:
            return self.auto_summary_hour == client_hours

    @property
    def invoice_send_mode_cdr(self):
        if self.invoice_setting and self.invoice_setting.setting:
            return self.invoice_setting.setting.invoice_send_mode_cdr
        else:
            q = SystemParameter.query().first()
            if q:
                return q.invoice_send_mode_cdr
        return 'none'

    @property
    def invoices_cdr_fields(self):
        if self.invoice_setting:
            return self.invoice_setting.setting.send_cdr_fields
        else:
            q = SystemParameter.query().first()
            if q:
                return q.invoices_cdr_fields
        return None

    @property
    def sys_company_name(self):
        return SystemParameter.get(1).company_info

    @property
    def login_url(self):
        # if self.user:
        #     return self.user.login_url
        # else:
        return SystemParameter.get(1).base_url

    # return SystemParameter.get(1).landing_page

    @property
    def logo_url(self):
        return "{}/{}".format(self.login_url, SystemParameter.get(1).public_logo_url)

    @property
    def sys_login_url(self):
        return SystemParameter.get(1).base_url

    @hybrid_property
    def usage_fields(self):
        try:
            return self.usage_detail_fields.split(',')
        except:
            return []

    @usage_fields.setter
    def usage_fields(self, value):
        try:
            self.usage_detail_fields = ','.join([str(x) for x in value])
        except:
            pass

    def create_from_template(self, name, template_id):
        obj = CarrierTemplate.filter(CarrierTemplate.id == template_id).first()
        if not obj:
            raise Exception('Bad template id: {} not found'.format(template_id))
            return
        self.name = name
        self.carrier_template_id = obj.id
        self.allowed_credit = obj.allowed_credit if obj.allowed_credit else 0.0
        self.attach_cdrs_list = obj.attach_cdrs_list
        self.auto_daily_balance_recipient = obj.auto_daily_balance_recipient
        self.auto_invoice_type = obj.auto_invoice_type if obj.auto_invoice_type else 'both'
        self.auto_invoicing = False if not obj.auto_invoicing is None and not obj.auto_invoicing else True
        self.auto_send_zone = obj.auto_send_zone
        self.auto_summary_group_by = obj.auto_summary_group_by
        self.auto_summary_hour = obj.auto_summary_hour
        self.auto_summary_include_cdr = obj.auto_summary_include_cdr if obj.auto_summary_include_cdr else False
        self.auto_summary_not_zero = obj.auto_summary_not_zero
        self.auto_summary_period = obj.auto_summary_period
        self.billing_mode = obj.billing_mode
        self.breakdown_by_rate_table = obj.breakdown_by_rate_table
        self.call_limit = obj.call_limit
        self.cdr_format = obj.cdr_format
        self.cdr_list_format = obj.cdr_list_format
        self.cps_limit = obj.cps_limit
        # self.create_by = obj.create_by
        # self.create_on = obj.create_on
        if obj.currency_id and Currency.get(obj.currency_id):
            self.currency_id = obj.currency_id
        self.daily_balance_notification = obj.daily_balance_notification
        self.daily_balance_recipient = obj.daily_balance_recipient
        self.daily_balance_send_time = obj.daily_balance_send_time
        self.daily_balance_send_time_zone = obj.daily_balance_send_time_zone
        self.daily_balance_summary = obj.daily_balance_summary
        self.daily_cdr_delivery = obj.daily_cdr_delivery
        self.daily_cdr_generation = obj.daily_cdr_generation
        self.daily_cdr_generation_type = obj.daily_cdr_generation_type
        self.daily_cdr_generation_zone = obj.daily_cdr_generation_zone
        self.daily_limit = obj.daily_limit
        self.daily_usage_group_by = obj.daily_usage_group_by
        self.daily_usage_summary = obj.daily_usage_summary
        self.decimal_place = obj.decimal_place
        # self.email_invoice = obj.email_invoice
        self.format = obj.format
        self.hourly_limit = obj.hourly_limit
        self.include_available_credit = obj.include_available_credit
        self.include_tax = obj.include_tax
        self.invoice_format = obj.invoice_format
        self.invoice_include_payment = obj.invoice_include_payment
        self.invoice_jurisdictional_detail = obj.invoice_jurisdictional_detail
        self.invoice_show_details = obj.invoice_show_details
        self.invoice_start_from = obj.invoice_start_from
        self.invoice_use_balance_type = obj.invoice_use_balance_type
        self.invoice_zero = obj.invoice_zero
        self.invoice_zone = obj.invoice_zone
        self.is_auto_balance = obj.is_auto_balance
        self.is_auto_summary = obj.is_auto_summary
        self.is_breakdown_by_rate_table = obj.is_breakdown_by_rate_table
        self.is_daily_balance_notification = obj.is_daily_balance_notification
        self.email_invoice = obj.is_email_invoice
        self.is_invoice_account_summary = obj.is_invoice_account_summary
        self.is_send_as_link = obj.is_send_as_link
        self.is_send_trunk_update = obj.is_send_trunk_update
        self.is_short_duration_call_surcharge_detail = obj.is_short_duration_call_surcharge_detail
        self.is_show_by_date = obj.is_show_by_date
        self.is_show_code_100 = obj.is_show_code_100
        self.is_show_code_name = obj.is_show_code_name
        self.is_show_country = obj.is_show_country
        # self.is_show_daily_trunk = obj.is_show_daily_trunk
        self.is_show_daily_usage = obj.is_show_daily_usage
        self.is_show_detail_trunk = obj.is_show_detail_trunk
        self.is_show_total_trunk = obj.is_show_total_trunk
        self.last_invoiced = obj.last_invoiced
        self.mode = obj.mode
        self.notify_client_balance = obj.notify_client_balance
        self.notify_client_balance_type = obj.notify_client_balance_type
        self.numer_of_days_balance = obj.numer_of_days_balance
        self.payment_received_notice = obj.payment_received_notice
        if obj.payment_term_id and PaymentTerm.get(obj.payment_term_id):
            self.payment_term_id = obj.payment_term_id
        self.profit_margin = obj.profit_margin
        self.profit_type = obj.profit_type
        self.rate_value = obj.rate_value
        self.scc_bellow = obj.scc_bellow
        self.scc_below = obj.scc_below
        self.scc_charge = obj.scc_charge
        self.scc_percent = obj.scc_percent
        self.scc_type = 1 if obj.scc_type == 'that exceed the defined percentage' else 0
        self.send_invoice_as_link = obj.send_invoice_as_link
        self.tax = obj.tax
        if obj.test_credit:
            self._test_credit = obj.test_credit
        self.unlimited_credit = obj.unlimited_credit
        self.update_on = datetime.now(UTC)
        self.usage_detail_fields = obj.usage_detail_fields
        c = CarrierLowBalanceConfig()
        c.create_from_template(obj.low_balance_config)
        # c.carrier=self
        self.low_balance_config = c

    def get_assigned_billing_plan_resource(self, plan, ip):
        # plan=DidBillingPlan.get(billing_plan_id)
        log.debug(
            'get_assigned_billing_plan_resource client={} {} plan={} {} rate_table='.format(self.client_id, self.name,
                                                                                            plan.id, plan.name,
                                                                                            plan.rate_table_id))
        name = '{}_DID_{}_{}'.format(self.name, plan.name, ip or 'ANY')
        ret = Resource.filter(and_(Resource.client_id == self.client_id, Resource.alias == name)).first()
        if not ret:
            r = self.resource
            if not r:
                if self.resources and len(self.resources):
                    r = self.resources[0]
                else:
                    raise ValidationError(
                        {'client_id': ['client {} have not default resource!'.format(self.name)]})
            ret = Resource(alias=name, client_id=self.client_id, media_type=r.media_type, capacity=r.capacity,
                           trunk_type2=r.trunk_type2, egress=r.egress, ingress=r.ingress,
                           # rate_table_id=plan.rate_table_id,
                           enough_balance=r.enough_balance, billing_rule=r.billing_rule,
                           route_strategy_id=r.route_strategy_id, t38=True)
            # for item in r.ip:
            #     i = ResourceIp(ip=item.ip, port=item.port, direction=1)
            #     ret.ip.append(i)
            item = r.ip[0] if r.ip else None
            if item:
                if not ip:
                    ip = item.ip
                ret.ip.append(ResourceIp(ip=ip, port=item.port, direction=1))
        # ret.save()
        ret.ingress = False  # True
        ret.egress = True  # False
        ret.trunk_type2 = 'DID Traffic'
        ret.update_at = datetime.now(UTC)
        # ret.create_time = datetime.now(UTC)
        # ret.rate_table = plan.rate_table ?? works bad
        ret.rate_table_id = plan.rate_table_id
        ret.client_id = self.client_id
        # self.resources.append(ret)
        self.session().add(ret)
        log.debug('get_assigned_billing_plan_resource  {}'.format(ret.resource_id, ret.alias, ret.rate_table_id))
        return ret

    def fmt_balance(self):
        return dec_prn(self.c4.balance, 2)

    @property
    def beginning_balance(self):
        b = BalanceHistoryActual
        q = b.filter(
            and_(b.client_id == self.client_id, b.date == (datetime.now(UTC) - timedelta(days=1)).date())).first()
        if q:
            return q.actual_balance
        return None

    @property
    def fmt_current_date(self):
        return str(datetime.now(UTC).date())

    @property
    def fmt_remaining_credit(self):
        beg = self.actual_balance
        if beg >= 0:
            return dec_prn(self.credit_limit, 2)
        if beg < 0:
            return dec_prn(float(self.credit_limit) + float(beg), 2)
        return '0.00'

    # return dec_prn(self.c4.balance, 2)

    @property
    def notify_balance(self):
        lb = self.low_balance_config
        if not lb:
            return None
        c4 = self.c4
        if not c4:
            return None
        if lb.value_type == 'Actual Balance':
            if self.mode == 'prepay':
                return lb.actual_notify_balance
            if self.mode == 'postpay':
                return self.notify_client_balance

        elif lb.value_type == 'Percentage':
            return self.allowed_credit - lb.percentage_notify_balance * self.allowed_credit / 100.0
        return None

    @property
    def fmt_notify_balance(self):
        if not self.notify_balance is None:
            return dec_prn(self.notify_balance, 2)
        return '-/-'

    @property
    def fmt_payment_terms(self):
        if self.payment_term:
            return self.payment_term.cycle
        return self.mode

    @property
    def fmt_allow_credit(self):
        return dec_prn(self.credit_limit, 2)

    @property
    def fmt_credit_limit(self):
        if self.unlimited_credit:
            return 'unlimited'
        return dec_prn(self.credit_limit, 2)

    def fmt_first_name(self):
        try:
            return self.user.first_name
        except:
            return ''

    @property
    def fmt_last_name(self):
        try:
            return self.user.last_name
        except:
            return ''

    #   egress_trunks = relationship("egress_trunk.EgressTrunkModel",

    #                       backref="client_id",secondaryjoin = "egress_trunk.EgressTrunkModel.egress==True")
    #   igress_trunks = relationship("entities.resource.egress_trunk.IngressTrunkModel",

    def last_invoice(self):
        return Invoice.filter(Invoice.client_id == self.client_id). \
            order_by(Invoice.invoice_time.desc()).first()

    def egress_trunks_count(self):
        return len(self.egress_trunks)

    def ingress_trunks_count(self):
        return len(self.ingress_trunks)

    def unpaid_invoices_count(self):
        return Invoice.filter(Invoice.client_id == self.client_id).filter(Invoice.paid == False).count()

    def unread_message_count(self):
        cls = EmailLog
        return cls.filter(and_(cls.type.in_(cls.MESSAGE_TYPES), cls.status != 'fail',
                               cls.is_view != 1, cls.client_id == self.client_id)).count()

    def unread_alert_count(self):
        cls = EmailLog
        return cls.filter(and_(cls.type.in_(cls.ALERT_TYPES), cls.status != 'fail',
                               cls.is_view != 1, cls.client_id == self.client_id)).count()

    @hybrid_property
    def daily_usage_on_non_zero(self):
        return self.auto_summary_not_zero == 1

    @daily_usage_on_non_zero.setter
    def daily_usage_on_non_zero(self, value):
        if value:
            self.auto_summary_not_zero = 1
        else:
            self.auto_summary_not_zero = 0

    @hybrid_property
    def currency_name(self):
        try:
            return self.currency.code
        except:
            return ''

    @currency_name.setter
    def currency_name(self, value):
        try:
            self.currency_id = Currency.filter(Currency.code == value).one().currency_id
        except:
            return None
        return value

    @hybrid_property
    def pt_name(self):
        try:
            return self.payment_term.name
        except:
            return ''

    @pt_name.setter
    def pt_name(self, value):
        try:
            self.payment_term_id = PaymentTerm.filter(PaymentTerm.name == value).one().payment_term_id
        except:
            raise Exception('Bad payment term:' + str(value) + ' not found')

    # pay.save()

    @hybrid_property
    def is_prepay(self):
        return self.mode == 'prepay'

    @is_prepay.setter
    def is_prepay(self, value):
        if value:
            self.mode = 'prepay'
            self.unlimited_credit = False
            self.allowed_credit = 0
        else:
            self.mode = 'postpay'

    @hybrid_property
    def fake(self):
        return self.low_balance_config.is_notify

    @fake.setter
    def fake(self, value):
        self.low_balance_config.is_notify = value

    @hybrid_property
    def low_balance_alert(self):
        return self.low_balance_config.is_notify

    @low_balance_alert.setter
    def low_balance_alert(self, value):
        self.low_balance_config.is_notify = value

    # --user-prop
    def last_payment(self):
        return ClientPayment.filter(
            and_(ClientPayment.client_id == self.client_id, ClientPayment.payment_type.in_([3, 4, 5, 6]))). \
            order_by(ClientPayment.date.desc()).first()

    def last_payment_invoice(self):
        return ClientPayment.filter(
            and_(ClientPayment.client_id == self.client_id, ClientPayment.payment_type.in_([3, 4]))). \
            order_by(ClientPayment.date.desc()).first()

    @hybrid_property
    def _mutual_balance(self):
        d = str(datetime.today().date())
        d = self.query().session.query(
            func.max(BalanceHistory.date).label('d')).filter(
            BalanceHistory.client_id == self.client_id).first().d
        try:
            return BalanceHistory.filter(BalanceHistory.client_id == self.client_id).filter(
                BalanceHistory.date == d).first().mutual_balance
        except:
            return 0.0

    @classmethod
    def user_id_filter(cls, user_id):
        return user_id_filter(cls, user_id)

    @hybrid_property
    def rate_generation_template_id(self):
        ext = ClientRateGeneration.filter(ClientRateGeneration.client_id == self.client_id).first()
        if ext:
            return ext.rate_generation_template_id
        else:
            return None

    @rate_generation_template_id.setter
    def rate_generation_template_id(self, value):
        ext = ClientRateGeneration.filter(ClientRateGeneration.client_id == self.client_id).first()
        if ext:
            ext.rate_generation_template_id = value
        else:
            ext = ClientRateGeneration(client_id=self.client_id, rate_generation_template_id=value)
        self.session().add(ext)

    @hybrid_property
    def client_invoice_settings_id(self):
        return self._client_invoice_settings_id

    @client_invoice_settings_id.setter
    def client_invoice_settings_id(self, value):

        ext = ClientInvoiceSetting.filter(ClientInvoiceSetting.client_id == self.client_id).first()
        if value is None:
            if ext:
                ext.delete()
                return
            else:
                return
        if not ext:
            ext = ClientInvoiceSetting(client_id=self.client_id)
        ext.invoice_setting_id = value
        self.session().add(ext)
        # self.invoice_setting=ext

    def invoice_check_and_generate(self, now, generate_callback, AUTO_INVOICE_DEBUG):
        try:
            if AUTO_INVOICE_DEBUG:
                log.debug('Started invoice check and generate')
            now_date = int(now.timestamp())
            time_zone_hour = 0
            hour = now.hour
            if not self.invoice_zone is None:
                time_zone_hour = get_time_zone_hour(self.invoice_zone)

            today = now + timedelta(hours=time_zone_hour)
            # skip other time zone
            if AUTO_INVOICE_DEBUG:
                log.debug(f"Getting Client invoice history (id = {self.client_id})")
            client_invoice_history = Invoice.filter(
                and_(Invoice.client_id == self.client_id, Invoice.state != 'void')).order_by(
                Invoice.invoice_time.desc()).first()
            invoice_end_time = None
            if client_invoice_history:
                _l = client_invoice_history.invoice_end + timedelta(seconds=1)
                last_invoiced = datetime(_l.year, _l.month, _l.day)
                _n = self.payment_term.next_date_for_now(last_invoiced, today.date())
                invoice_end_time = datetime(_n.year, _n.month, _n.day, 0, 0, 0) - timedelta(seconds=1)
            if today.hour != (self.invoice_hour or 0):  # time_zone_hour != today.hour:
                if client_invoice_history and client_invoice_history.invoice_time.day == today.day or today.hour < (self.invoice_hour or 0):
                    msg = 'invoice, client {} timezone {} invoice_hour {} does not match current hour {}. continue'.format(
                        self.client_id, self.invoice_zone, self.invoice_hour, today.hour)
                    log.debug(msg)
                    return 0, msg

            if not self.name or not self.payment_term or not self.payment_term.type:
                msg = 'invoice, client {} has not name or payment_term {} or payment_term.type'.format(
                    self.client_id, self.payment_term_id)
                log.debug(msg)
                if AUTO_INVOICE_DEBUG:
                    InvoiceDebug(client_id=self.client_id, text=msg).save()
                return 0, msg

            client_invoice_history = Invoice.filter(
                and_(Invoice.client_id == self.client_id, Invoice.state != 'void')).order_by(
                Invoice.invoice_end.desc()).first()

            if not client_invoice_history:
                msg = 'Invoice Client {} with Id:  {} is type {} is never invoiced before'.format(
                    self.name, self.client_id, self.payment_term.type)
                log.debug(msg)
                if AUTO_INVOICE_DEBUG:
                    InvoiceDebug(client_id=self.client_id, text=msg).save()
                _l = self.create_time
                term_type = self.payment_term.type
                type_dict = {'Every': 1, 'Day of Month': 30, 'Day of Week': 7}
                if term_type in type_dict:
                    _l = datetime.now(UTC) - timedelta(days=type_dict[term_type])
                if term_type == 'Day of Month':
                    _l = datetime.now(UTC).replace(month=(datetime.now(UTC).month-2)%12+1)
                    if _l.month == 12:
                        _l = _l.replace(year=_l.year-1)
                if term_type == 'Twice a month':
                    if datetime.now(UTC).day > 15:
                        _l = datetime.now(UTC).replace(day=1)
                    else:
                        _l = datetime.now(UTC).replace(day=1) - timedelta(days=1)
                        _l = _l.replace(day=16)
                last_invoiced = datetime(_l.year, _l.month, _l.day)
                InvoiceHistory(last_invoice_for=last_invoiced,
                               next_invoice_date=self.payment_term.next_date_for_now(last_invoiced, today.date()),
                               client_id=self.client_id).save()
            else:
                _l = client_invoice_history.invoice_end + timedelta(seconds=1)
                last_invoiced = datetime(_l.year, _l.month, _l.day)
                msg = 'Invoice Client {} Id {} last_invoiced {}'.format(
                    self.name, self.client_id, last_invoiced)
                log.debug(msg)
            if (today.replace(tzinfo=UTC) - last_invoiced.replace(tzinfo=UTC)).days > 90:
                last_invoiced = (today - timedelta(days=90)).replace(day=1)
            invoice_start_time = last_invoiced
            _n = self.payment_term.next_date_for_now(last_invoiced, today.date())
            invoice_end_time = datetime(_n.year, _n.month, _n.day, 0, 0, 0) - timedelta(seconds=1)
            msg = 'Client {} id {} invoice term {} type {} cycle {}, next invoice invoice_end_time {}, today is {}'. \
                format(self.name, self.client_id, self.payment_term_name, self.payment_term.type,
                       self.payment_term.cycle, invoice_end_time, today)
            log.debug(msg)
            if AUTO_INVOICE_DEBUG:
                InvoiceDebug(client_id=self.client_id, text=msg).save()

            if invoice_end_time.timestamp() <= today.timestamp():
                inv, warn = generate_callback(self.client_id, invoice_start_time, invoice_end_time)
                if hasattr(inv, 'invoice_number'):
                    msg = 'Client {} id {} SUCCESS invoice_number {} period {} {} {}'.format(
                        self.name, self.client_id, inv.invoice_number, str(invoice_start_time)[0:10],
                        str(invoice_end_time)[0:10], warn)
                    log.debug(msg)
                    if AUTO_INVOICE_DEBUG:
                        InvoiceDebug(client_id=self.client_id, text=msg).save()

                    InvoiceHistory(last_invoice_for=today.date(),
                                   next_invoice_date=self.payment_term.next_date(today.date()),
                                   client_id=self.client_id,
                                   last_invoice_period='{} {}'.format(str(invoice_start_time)[0:10],
                                                                      str(invoice_end_time)[0:10]),
                                   last_invoice_amount=inv.total_amount,
                                   # invoice_filename=inv.filename,
                                   # next_reminder_timestamp=int(
                                   #     datetime(*inv.payment_dur_date.timetuple()[:-4]).timestamp()),
                                   # reminders_count=1,
                                   # start_time=invoice_start_time,
                                   # end_time=invoice_end_time
                                   ).save()
                    return 1, warn
                else:
                    msg = 'Client {} id {}  period {} {} {}'.format(
                        self.name, self.client_id, str(invoice_start_time)[0:10],
                        str(invoice_end_time)[0:10], warn)
                    return 1, msg

            msg = 'Client {} id {} SKIP period {} {} does not match payment term, or overlapped.last auto invoice was at {}'.format(
                self.name, self.client_id, str(invoice_start_time)[0:10], str(invoice_end_time)[0:10], last_invoiced
            )
            log.debug(msg)
            if AUTO_INVOICE_DEBUG:
                InvoiceDebug(client_id=self.client_id, text=msg).save()
            return 0, msg
        except Exception as e:
            exception_type, exception_object, exception_traceback = sys.exc_info()
            filename = exception_traceback.tb_frame.f_code.co_filename
            line_number = exception_traceback.tb_lineno
            msg = 'Invoice error:{} {} {} {}'.format(e, str(traceback.print_exc(), filename, line_number))
            log.debug(msg)
            if AUTO_INVOICE_DEBUG:
                if self.client_id and self.client_name:
                    msg = 'Client {} id {} {}'.format(self.name, self.client_id, msg)
                InvoiceDebug(client_id=self.client_id, text=msg).save()
            return 0, msg
        return 0, 'no messages'


# return and_(cls.client_type.is_(None),
# 			or_(and_(User.user_id == user_id, User.user_type == 'admin'),
# 				and_(User.user_id == user_id, User.user_type == 'client', cls.client_id == User.client_id),
# 				cls.client_id.in_(select([UsersLimit.client_id]).where(
# 					and_(User.user_id == user_id, User.user_type == 'client',
# 						 User.user_id == UsersLimit.user_id))),
# 				cls.client_id.in_(select([AgentClient.client_id]).where(
# 					and_(User.user_id == user_id, User.user_type == 'agent', User.user_id == Agent.user_id,
# 						 Agent.agent_id == AgentClient.agent_id)))
# 				)
# 			)


#                        backref="client_id",secondaryjoin = "entities.resource.egress_trunk.IngressTrunkModel.ingress==True")
@event.listens_for(Client.enough_balance, 'set')
def enough_balance_set(target, value, oldvalue, initiator):
    if not target.client_type == 'vendor':
        Resource.filter(Resource.client_id == target.client_id).update({'enough_balance': value})


# @event.listens_for(Client,'after_update')
# def _enough_balance_set(mapper,connection,target):
#    connection.execute('update resource set enough_balance={} where client_id={}'.format(target.enough_balance,target.client_id))
#        #$Resource.session().commit()
#    return True
InvoiceSetting.used_by_client = column_property(select([func.count(ClientInvoiceSetting.client_id)]).where(
    ClientInvoiceSetting.invoice_setting_id == InvoiceSetting.id).correlate_except(ClientInvoiceSetting))
InvoiceSetting.used_by_agent = column_property(select([func.count(Agent.agent_id)]).where(
    Agent.invoice_setting_id == InvoiceSetting.id).correlate_except(Agent))
PaymentTerm.clients_count = lambda pt_id: Client.filter(payment_term_id=pt_id).count()
PaymentTerm.usage_count = column_property(select([func.count(Client.client_id).label('usage_count')]).where(
    Client.payment_term_id == PaymentTerm.payment_term_id). \
                                          group_by(Client.payment_term_id).correlate_except(Client))
PaymentTerm.term_usage_count = column_property(select([func.count(Client.client_id).label('term_usage_count')]).where(and_(
    Client.payment_term_id == PaymentTerm.payment_term_id, Client.company_type == 0)). \
                                          group_by(Client.payment_term_id).correlate_except(Client))
PaymentTerm.orig_usage_count = column_property(select([func.count(Client.client_id).label('orig_usage_count')]).where(and_(
    Client.payment_term_id == PaymentTerm.payment_term_id, Client.company_type == 1)). \
                                          group_by(Client.payment_term_id).correlate_except(Client))
# CarrierGroup.clients_count = lambda group_id: Client.filter(group_id=group_id).count()
ProductClientsRef.client_name = column_property(
    select([Client.name]).where(Client.client_id == ProductClientsRef.client_id).correlate_except(Client))

AgentClient.client_name = column_property(
    select([Client.name]).where(Client.client_id == AgentClient.client_id).correlate_except(Client))
AgentClient.company = column_property(
    select([Client.company]).where(Client.client_id == AgentClient.client_id).correlate_except(Client))
AgentClient.status = column_property(
    select([Client.status]).where(Client.client_id == AgentClient.client_id).correlate_except(Client))
AgentClient.mode = column_property(
    select([Client.mode]).where(Client.client_id == AgentClient.client_id).correlate_except(Client))
AgentClient.company_type = column_property(
    select([Client.company_type]).where(Client.client_id == AgentClient.client_id).correlate_except(Client))
AgentClient.unlimited_credit = column_property(
    select([Client.unlimited_credit]).where(Client.client_id == AgentClient.client_id).correlate_except(Client))

AgentClient.credit_limit = column_property(
    select([-Client.allowed_credit]).where(Client.client_id == AgentClient.client_id).correlate_except(Client))
AgentClient.registered_on = column_property(
    select([Client.create_time]).where(Client.client_id == AgentClient.client_id).correlate_except(Client))
AgentClient.created_on = column_property(
    select([Client.create_time]).where(Client.client_id == AgentClient.client_id).correlate_except(Client))

User.carrier_name = column_property(
    select([Client.name]).where(Client.client_id == User.client_id).correlate_except(Client))
User.company_type = column_property(
    select([Client.company_type]).where(Client.client_id == User.client_id).correlate_except(Client))
EmailLog.client_name = column_property(
    select([Client.name]).where(Client.client_id == EmailLog.client_id).correlate_except(Client))
BalanceLog.client_name = column_property(
    select([Client.name]).where(Client.client_id == BalanceLog.client_id).correlate_except(Client))


class File(DnlApiBaseModel):
    __tablename__ = 'file'
    uuid = Column(String(36), primary_key=True, default=generate_uuid_str())
    path = Column(String(1024), nullable=False)
    belongs_to_table = Column(String(255), nullable=False)
    belongs_to_field = Column(String(255), nullable=False)
    belongs_to_pk = Column(String(128))
    uploaded_on = Column(DateTime(True), nullable=False)
    attached_on = Column(DateTime(True), server_default=text_("('now'::text)::timestamp(0) with time zone"))
    public = Column(Boolean, server_default=text_("false"))


# region NEW TABLES
text = text_
metadata = DnlApiBaseModel.metadata

t_banned_ip = Table(
    'banned_ip', metadata,
    Column('ip', String(30), nullable=False, index=True),
    Column('brief', String(100)),
    Column('create_time', DateTime(True), server_default=text("('now'::text)::timestamp(0) with time zone"))
)

create_banned_ip_record = """
ALTER TABLE public.banned_ip OWNER TO postgres;

--
-- Name: banned_ip_record; Type: TABLE; Schema: public; Owner: postgres
--

CREATE TABLE public.banned_ip_record (
    ip character varying(30),
    brief character varying(100),
    create_time timestamp with time zone,
    "time" numeric,
    flag character(1),
    record_id integer NOT NULL
);


ALTER TABLE public.banned_ip_record OWNER TO postgres;

--
-- Name: banned_ip_record_record_id_seq; Type: SEQUENCE; Schema: public; Owner: postgres
--

CREATE SEQUENCE public.banned_ip_record_record_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;


ALTER TABLE public.banned_ip_record_record_id_seq OWNER TO postgres;

--
-- Name: banned_ip_record_record_id_seq; Type: SEQUENCE OWNED BY; Schema: public; Owner: postgres
--

ALTER SEQUENCE public.banned_ip_record_record_id_seq OWNED BY public.banned_ip_record.record_id;

"""


class BlockNumberImportTask(DnlApiBaseModel):
    __tablename__ = 'block_number_import_task'
    # COMMENT ON COLUMN public.block_number_import_task.orig_import_file IS 'Original import block number file name';
    # COMMENT ON COLUMN public.block_number_import_task.format_import_file IS 'CSV format file after WEB check';
    # COMMENT ON COLUMN public.block_number_import_task.import_log_path IS 'Import log file path';
    # COMMENT ON COLUMN public.block_number_import_task.redup_in_block_rable_action IS '0 - reject entire import;
    # 1 - skip the record;
    # 2 - overwrite existing record';
    # COMMENT ON COLUMN public.block_number_import_task.redup_in_file_action IS '0 - reject entire import;
    # 1 - skip the record;
    # 2 - overwrite previous record';
    # COMMENT ON COLUMN public.block_number_import_task.status IS '0 - Initial; 1 - Parse and check import file;
    # 2 - Load block table data; 3 - Compare block number;
    # 4 - Insert new block number; 5 - Update same block number;
    # 6 - Upload successful; 7 - Upload failed';
    REDUP_IN_RATE_TABLE_ACTION = {0: 'reject entire import', 1: 'skip the record', 2: 'overwrite existing record'}
    REDUP_IN_FILE_ACTION = {0: 'reject entire import', 1: 'skip the record', 2: 'overwrite previous record'}
    STATUS = {0: 'Initial', 1: 'Parse and check import file',
              2: 'Load block table data', 3: 'Compare block number',
              4: 'Insert new block number', 5: 'Update same block number',
              6: 'Upload successful', 7: 'Upload failed'}
    OP_METHOD = {0: 'Upload', 1: 'Delete'}
    id = Column(Integer,
                primary_key=True)  # , server_default=text("nextval('block_number_import_task_id_seq'::regclass)"))
    operator_user = Column(String(40))
    import_file_path = Column(String(256))
    orig_import_file = Column(String(256))
    format_import_file = Column(String(256))
    import_log_path = Column(String(256))
    redup_in_block_rable_action = Column(ChoiceType(REDUP_IN_RATE_TABLE_ACTION), nullable=False,
                                         server_default=text("0"))
    redup_in_file_action = Column(ChoiceType(REDUP_IN_FILE_ACTION), nullable=False, server_default=text("0"))
    status = Column(ChoiceType(STATUS), nullable=False, server_default=text("0"))
    progress = Column(String(1024))
    start_time = Column(DateTime(True))
    end_time = Column(DateTime(True))
    create_time = Column(DateTime(True), server_default=text("('now'::text)::timestamp(0) with time zone"))
    op_method = Column(ChoiceType(OP_METHOD), nullable=False, server_default=text("0"))


class C4UsLerg(DnlApiBaseModel):
    __tablename__ = 'c4_us_lerg'
    __table_args__ = (
        Index('c4_us_lerg_npanxx_idx', 'npa', 'nxx'),
        Index('c4_us_lerg_npanxx_block_idx', 'npa', 'nxx', 'block', unique=True)
    )

    id = Column(Integer, primary_key=True)  # , server_default=text("nextval('c4_us_lerg_id_seq'::regclass)"))
    npa = Column(String(4))
    nxx = Column(String(4))
    block = Column(String(4))
    lata = Column(String(8), index=True)
    ocn = Column(String(8), index=True)
    ratecenter = Column(String(24))
    state = Column(String(4), index=True)
    category = Column(String(24))
    company = Column(String(100))
    country = Column(String(2), index=True)


class Lerg(DnlApiBaseModel):
    __tablename__ = 'lerg'
    uuid = Column(String(36), primary_key=True, server_default=func.uuid_generate_v4())
    npa = Column(String(3), index=True)
    nxx = Column(String(3), index=True)
    thousands = Column(String(2), index=True)
    state = Column(String(2))
    company = Column(String(128))
    ocn = Column(String(6), index=True)
    ratecenter = Column(String(16))
    clli = Column(String(16))
    assign_date = Column(String(10))
    prefix_type = Column(String(18))
    switch_name = Column(String(48))
    switch_type = Column(String(64))
    lata = Column(String(3), index=True)
    country = Column(String(2))
    npanxx = column_property(npa + nxx + thousands)


Index('ix_lerg_npanxx', Lerg.npanxx)


class LergImportTask(DnlApiBaseModel):
    __tablename__ = 'lerg_import_task'
    STATUS = {-1: 'fail', 0: 'initial', 1: 'success', 2: 'busy'}
    uuid = Column(String(36), primary_key=True, default=generate_uuid_str(), server_default=func.uuid_generate_v4())
    status = Column(ChoiceType(STATUS), nullable=False, server_default="0")
    result = Column(Text())
    file = Column(String(512))
    created_on = Column(DateTime(True), server_default=func.now())
    created_by = Column(String(255))
    finished_on = Column(DateTime(True))

    @property
    def url(self):
        return '{}/lerg/file/{}.{}'.format(settings.API_URL, self.uuid, self.ext)

    @property
    def file_name(self):
        if self.file:
            return '{}/{}.{}'.format(settings.FILES['upload_to'], self.uuid, self.ext)
        return None

    @property
    def ext(self):
        if self.file and len(self.file.split('.')) > 1:
            return self.file.split('.')[-1]
        return ''

    @property
    def data(self):
        if self.file:
            return open(self.file_name, 'wb').read()


class LergDownloadSchedule(DnlApiBaseModel):
    __tablename__ = 'lerg_download_schedule'
    id = Column(Integer, primary_key=True)
    url = Column(String(100))
    schedule_hour = Column(Integer())
    created_on = Column(DateTime(True), server_default=func.now())
    created_by = Column(String(255))
    last_downloaded_on = Column(DateTime(True))
    last_downloaded_count = Column(Integer())


class LergDownloadTask(DnlApiBaseModel):
    __tablename__ = 'lerg_download_task'
    STATUS = {-1: 'fail', 0: 'initial', 1: 'success', 2: 'busy'}
    uuid = Column(String(36), primary_key=True, default=generate_uuid_str(), server_default=func.uuid_generate_v4())
    url = Column(String(100))
    status = Column(ChoiceType(STATUS), nullable=False, server_default="0")
    result = Column(Text())
    created_on = Column(DateTime(True), server_default=func.now())
    finished_on = Column(DateTime(True))
    lerg_import_task_uuid = Column(ForeignKey('lerg_import_task.uuid', ondelete='CASCADE'), index=True, nullable=True)
    schedule_id = Column(ForeignKey('lerg_download_schedule.id', ondelete='CASCADE'), index=True, nullable=False)
    import_task = relationship(LergImportTask)
    schedule = relationship(LergDownloadSchedule)

    @property
    def count(self):
        if self.import_task:
            result = self.import_task.result
            if result and 'numbers imported ' in result:
                return result.split('numbers imported ')[1].split(' ')[0]
        return 0


class FrundDetection(DnlApiBaseModel):
    __tablename__ = 'frund_detection'
    SEND_EMAIL_TYPE = {0: 'None', 1: 'Your Own NOC', 2: 'Partner NOC', 3: 'Both NOC'}
    DETECTION_LEVEL = {0: 'Trunk', 1: 'ANI'}
    id = Column(Integer, primary_key=True)  # , server_default=text("nextval('frund_detection_id_seq'::regclass)"))
    rule_name = Column(String(100), index=True)
    trunk_id_list = Column(String(100))
    criteria_1hour_min = Column(Numeric(15, 6))
    criteria_1hour_revenue = Column(Numeric(15, 6))
    criteria_24hour_min = Column(Numeric(15, 6))
    criteria_24hour_revenue = Column(Numeric(15, 6))
    is_block = Column(Boolean)
    send_email_type = Column(ChoiceType(SEND_EMAIL_TYPE))
    email_from = Column(Integer)
    email_ticket_subject = Column(String(200))
    email_ticket_content = Column(Text)
    trigger_criteria_1hour_time = Column(DateTime(True))
    trigger_criteria_24hour_time = Column(DateTime(True))
    create_time = Column(DateTime(True), server_default=text("('now'::text)::timestamp(0) with time zone"))
    hourly_asr_of_trunk_is_below = Column(Integer, nullable=True)
    hourly_acd_of_trunk_is_below = Column(Integer, nullable=True)
    hourly_minute_of_trunk_is_above = Column(Integer, nullable=True)
    hourly_cost_of_trunk_is_above = Column(Integer, nullable=True)
    daily_asr_of_trunk_is_below = Column(Integer, nullable=True)
    daily_acd_of_trunk_is_below = Column(Integer, nullable=True)
    daily_minute_of_trunk_is_above = Column(Integer, nullable=True)
    daily_cost_of_trunk_is_above = Column(Integer, nullable=True)
    detection_level = Column(ChoiceType(DETECTION_LEVEL))

    @property
    def subject(self):
        return self.email_ticket_subject
        # return 'Fraud detected'

    @property
    def html_content(self):
        return self.email_ticket_content

    @property
    def cc_mail(self):
        return ''

    @property
    def from_mail_id(self):
        return self.email_from

    @property
    def to_mail(self):
        return ''

    def get_attachment(self, env):
        return []


class FrundDetectionHistory(DnlApiBaseModel):
    __tablename__ = 'frund_detection_history'
    BLOCK_TYPE = {0: 'hourly duration', 1: 'hourly revenue', 2: '24 hour  duration', 3: '24 hour  revenue'}
    id = Column(Integer,
                primary_key=True)  # , server_default=text("nextval('frund_detection_history_id_seq'::regclass)"))
    frund_detection_id = Column(Integer, index=True)
    ingress_id = Column(Integer)
    block_type = Column(ChoiceType(BLOCK_TYPE))
    criteria_1hour_min = Column(Numeric(15, 6))
    criteria_1hour_revenue = Column(Numeric(15, 6))
    criteria_24hour_min = Column(Numeric(15, 6))
    criteria_24hour_revenue = Column(Numeric(15, 6))
    actual_1h_minute = Column(Numeric(15, 6))
    actual_24h_minute = Column(Numeric(15, 6))
    actual_1h_revenue = Column(Numeric(15, 6))
    actual_24h_revenue = Column(Numeric(15, 6))
    is_block = Column(Boolean, server_default=text("false"))
    send_email_type = Column(Integer)
    is_send_email = Column(Boolean, server_default=text("false"))
    partner_email_msg = Column(Text)
    partner_email_status = Column(Boolean)
    partner_email = Column(String(200))
    system_email_msg = Column(Text)
    system_email_status = Column(Boolean)
    system_email = Column(String(200))
    create_time = Column(DateTime(True), server_default=text("('now'::text)::timestamp(0) with time zone"))
    api_email_send_time = Column(DateTime(True), index=True)
    api_email_result = Column(String(1024))

    trunk = relationship('Resource', primaryjoin='foreign(FrundDetectionHistory.ingress_id) == Resource.resource_id')
    rule = relationship(FrundDetection, primaryjoin=foreign(frund_detection_id) == FrundDetection.id)
    rule_name = column_property(
        select([FrundDetection.rule_name]).where(FrundDetection.id == frund_detection_id).correlate_except(
            FrundDetection))

    @property
    def block_value(self):
        if self.block_type == 'hourly duration':
            return self.actual_1h_minute
        elif self.block_type == 'hourly revenue':
            return self.actual_1h_revenue
        elif self.block_type == '24 hour  duration':
            return self.actual_24h_minute
        elif self.block_type == '24 hour  revenue':
            return self.actual_24h_revenue
        else:
            return None

    @property
    def limit_value(self):
        if self.block_type == 'hourly duration':
            return self.criteria_1hour_min
        elif self.block_type == 'hourly revenue':
            return self.criteria_1hour_revenue
        elif self.block_type == '24 hour  duration':
            return self.criteria_24hour_min
        elif self.block_type == '24 hour  revenue':
            return self.criteria_24hour_revenue
        else:
            return None


class MonitoredRule(DnlApiBaseModel):
    __tablename__ = 'monitored_rule'
    """COMMENT ON COLUMN monitored_rule.trunk_id_list is 'Monitored trunk id list, separated by comma';
COMMENT ON COLUMN monitored_rule.include_codes is 'Monitored code list,separated by comma. if empty, monitor all code';
COMMENT ON COLUMN monitored_rule.exclude_codes is 'Unmonitored code list,separated by comma';
COMMENT ON COLUMN monitored_rule.monitored_type is '0: trunk; 1: trunk and dnis; 2: trunk and ani; 3: dnis; 4: ani; 5: trunk and destination; 6: trunk and country; 7: trunk and code';
COMMENT ON COLUMN monitored_rule.acd_action is 'xxx_ation   0: ignore; 1: >; 2: <; 3: =';
COMMENT ON COLUMN monitored_rule.sdp_duation is '0: none; 1: 6sec; 2: 12sec; 3: 18sec; 4: 24sec; 5: 30sec';
COMMENT ON COLUMN monitored_rule.exe_schedule_type is '0: never; 1: every specific minuses; 2: daily; 3: weekly';
COMMENT ON COLUMN monitored_rule.run_day_of_week is '0: Sunday; 1: Monday; 2: Tuesday; 3: Wednesday; 4: Thursday; 5: Friday; 6: Saturday';
COMMENT ON COLUMN monitored_rule.send_email_type is '0 - None; 1 - Your Own NOC; 2 - Partner NOC; 3 - Both NOC';
"""
    MONITORED_TYPE = {0: 'trunk', 1: 'trunk and dnis', 2: 'trunk and ani', 3: 'dnis', 4: 'ani',
                      5: 'trunk and destination', 6: 'trunk and country', 7: 'trunk and code'}
    ACTION = {0: 'ignore', 1: '>', 2: '<', 3: '='}
    SDP_DUATION = {0: 'none', 1: '6sec', 2: '12sec', 3: '18sec', 4: '24sec', 5: '30sec'}
    EXE_SCHEDULE_TYPE = {0: 'never', 1: 'every specific minuses', 2: 'daily', 3: 'weekly'}
    RUN_DAY_OF_WEEK = {0: 'Sunday', 1: 'Monday', 2: 'Tuesday', 3: 'Wednesday', 4: 'Thursday', 5: 'Friday',
                       6: 'Saturday'}
    SEND_EMAIL_TYPE = {0: 'None', 1: 'Your Own NOC', 2: 'Partner NOC', 3: 'Both NOC'}
    ACTION_TYPE = {1: 'Disable Entire Trunk', 2: 'Disable Specific Code',
                   3: 'Disable Specific Code Name'}

    id = Column(Integer, primary_key=True)  # , server_default=text("nextval('monitored_rule_id_seq'::regclass)"))
    rule_name = Column(String(100))
    is_active = Column(Boolean, nullable=False, server_default=text("true"))
    is_ingress_trunk = Column(Boolean, server_default=text("true"))
    is_all_trunk = Column(Boolean, nullable=False, server_default=text("false"))
    trunk_id_list = Column(String(200))
    include_codes = Column(String(500))
    exclude_codes = Column(String(500))
    monitored_type = Column(ChoiceType(MONITORED_TYPE), server_default=text("0"))
    acd_action = Column(ChoiceType(ACTION))
    acd_value = Column(Integer)
    asr_action = Column(ChoiceType(ACTION))
    asr_value = Column(Numeric(15, 6))
    sdp_action = Column(ChoiceType(ACTION))
    sdp_value = Column(Numeric(15, 6))
    sdp_duation = Column(ChoiceType(SDP_DUATION))
    pdd_action = Column(ChoiceType(ACTION))
    pdd_value = Column(Integer)
    profit_action = Column(ChoiceType(ACTION))
    profit_value = Column(Numeric(15, 6))
    revenue_action = Column(ChoiceType(ACTION))
    revenue_value = Column(Numeric(15, 6))
    min_call_attempt = Column(Integer)
    exe_schedule_type = Column(ChoiceType(EXE_SCHEDULE_TYPE))
    action_type = Column(ChoiceType(ACTION_TYPE))
    specific_minutes = Column(Integer)
    daily_run_time = Column(Time)
    weekly_run_time = Column(Time)
    run_day_of_week = Column(ChoiceType(RUN_DAY_OF_WEEK))
    sample_size = Column(Integer, nullable=False, server_default=text("5"))
    is_block = Column(Boolean, server_default=text("false"))
    is_auto_unblock = Column(Boolean, server_default=text("false"))
    unblock_after_min = Column(Integer)
    last_run_time = Column(DateTime(True))
    next_run_time = Column(DateTime(True))
    send_email_type = Column(ChoiceType(SEND_EMAIL_TYPE), server_default=text("3"))
    email_from = Column(Integer)
    email_ticket_subject = Column(String(200))
    email_ticket_content = Column(Text)
    create_time = Column(DateTime(True), server_default=text("('now'::text)::timestamp(0) with time zone"))
    updated_at = Column(DateTime(True), server_default=text("('now'::text)::timestamp(0) with time zone"))
    updated_by = Column(String(200))

    @property
    def subject(self):
        return self.email_ticket_subject

    @property
    def html_content(self):
        return self.email_ticket_content

    @property
    def cc_mail(self):
        return ''

    @property
    def to_mail(self):
        return 'email'

    def get_attachment(self, env):
        return []


class MonitoredRuleHistory(DnlApiBaseModel):
    __tablename__ = 'monitored_rule_history'

    id = Column(Integer,
                primary_key=True)  # , server_default=text("nextval('monitored_rule_history_id_seq'::regclass)"))
    monitored_rule_id = Column(Integer, index=True)
    start_time = Column(DateTime(True))
    end_time = Column(DateTime(True))
    is_ingress_trunk = Column(Boolean)
    trunk_id = Column(Integer)
    dnis = Column(String(64))
    ani = Column(String(64))
    code_name = Column(String(100))
    country = Column(String(100))
    code = Column(String(64))
    total_call = Column(Integer)
    not_zero_call = Column(Integer)
    sdp_call = Column(Integer)
    call_duration = Column(Integer)
    pdd = Column(Integer)
    ingress_cost = Column(Numeric(30, 6))
    egress_cost = Column(Numeric(30, 6))
    block_id_list = Column(Text)
    unblock_time = Column(DateTime(True), index=True)
    unblock_exe_time = Column(DateTime(True), index=True)
    create_time = Column(DateTime(True), server_default=text("('now'::text)::timestamp(0) with time zone"))
    api_email_send_time = Column(DateTime(True), index=True)
    api_email_result = Column(String(1024))
    monitored_rule = relationship(MonitoredRule, primaryjoin=foreign(monitored_rule_id) == MonitoredRule.id)

    rule_name = column_property(
        select([MonitoredRule.rule_name]).where(MonitoredRule.id == monitored_rule_id).correlate_except(MonitoredRule))
    block_list = synonym('block_id_list')
    sample_start_time = synonym('start_time')
    sample_end_time = synonym('end_time')
    total_calls = synonym('total_call')

    @property
    def limit_table(self):
        rule = self.monitored_rule
        return """
    <table class="limit_table" name="limit_table">
    <tr><th></th><th>Criteria</th><th>Limit value</th><th>Actual value</th></tr>
    <tr><th></th><th>acd</th><th>{}{}</th><th>{}</th></tr>
    <tr><th></th><th>asr</th><th>{}{}</th><th>{}</th></tr>
    <tr><th></th><th>sdp</th><th>{}{}</th><th>{}</th></tr>
    <tr><th></th><th>pdd</th><th>{}{}</th><th>{}</th></tr>
    <tr><th></th><th>profitability</th><th>{}{}</th><th></th></tr>
    <tr><th></th><th>revenue</th><th>{}{}</th><th></th></tr>
    </table>
    """.format(rule.acd_value, rule.acd_action, self.acd,
               rule.asr_value, rule.asr_action, self.asr,
               rule.sdp_value, rule.sdp_action, self.sdp_call,
               rule.pdd_value, rule.pdd_action, self.pdd,
               rule.profit_value, rule.profit_action, rule.revenue_value,
               rule.revenue_action)


    @property
    def rule_setup(self):
        rule = self.monitored_rule
        all = 'all' if rule.is_all_trunk else 'selected'
        return """
            <table class="rule_setup" name="rule_setup">
            <tr><th>Name</th><th>Value</th></tr>
            <tr><td>scope</td><td>{} {}</td></tr>
            <tr><td>include</td><td>{}</td></tr>
            <tr><td>exclude</td><td>{}</td></tr>
            <tr><td>monitoring on</td><td>{}</td></tr>
            <tr><td>trunks</td><td>{}</td></tr>
            <tr><td>execution schedule</td><td>{}</td></tr>
            <tr><td>do block</td><td>{}</td></tr>
            </table>
            """.format(all, 'ingress' if rule.is_ingress_trunk else 'egress', rule.include_codes, rule.exclude_codes,
                       rule.monitored_type,
                       rule.trunk_id_list, rule.exe_schedule_type, rule.is_block)

    @property
    def acd(self):
        if self.not_zero_call:
            return self.call_duration / self.not_zero_call
        else:
            return None

    @property
    def asr(self):
        if self.total_call:
            return self.not_zero_call / self.total_call * 100
        else:
            return None

    def monitoring_result(self):
        row = ''
        for key in (
                'total_call', 'not_zero_call', 'sdp_call', 'call_duration', 'pdd', 'ingress_cost', 'egress_cost', 'acd',
                'asr'):
            row = row + '<tr><td>{}</td><td>{}</td></tr>'.format(key.replace('_', ' '),round(getattr(self, key), 2))
        return '<table class="monitoring_result" name="monitoring_result">{}</table>'.format(row)


t_sip_404_number = Table(
    'sip_404_number', metadata,
    Column('number', String(32), unique=True),
    Column('create_time', DateTime(True), server_default=text("('now'::text)::timestamp(0) with time zone"))
)


# endregion


# class HostInfo(DnlApiBaseModel):
#     __tablename__ = 'host_info'
#
#     time = Column(String(100), primary_key=True, nullable=False, index=True)
#     res_id = Column(String(100), primary_key=True, nullable=False)
#     ip_id = Column(String(100), primary_key=True, nullable=False)
#     acd = Column(String(100))
#     asr = Column(String(100))
#     ca = Column(String(100))
#     call_count = Column(String(100))
#     direction = Column(String(100))
#     call_count_asr = Column(String(100))
#     pdd = Column(String(100))
#     ip = Column(String(100))


class RateTable(DnlApiBaseModel):
    __tablename__ = 'rate_table'
    RATE_TYPE_DICT = {0: 'DNIS', 1: 'LRN', 2: 'LRN BLOCK', 3: 'ANI'}
    JUR_TYPE_DICT = {0: 'A-Z', 1: 'US Non Jurisdictional', 2: 'US Jurisdictional', 3: 'OCN-LATA'}
    rate_table_id = Column(Integer, primary_key=True)
    name = Column(String(100), nullable=False, unique=True)
    modify_time = Column(DateTime(True), onupdate=func.now())
    create_time = Column(DateTime(True), nullable=False, server_default=func.now())
    code_deck_id = Column(ForeignKey('code_deck.code_deck_id', ondelete='SET NULL'), index=True, nullable=True)
    currency_id = Column(ForeignKey('currency.currency_id', ondelete='SET NULL'), nullable=True, index=True,
                         server_default='1')
    rate_type = Column(ChoiceType(RATE_TYPE_DICT), default=0)  # 0-dnis;1-lrn;2-lrn block

    jurisdiction_country_id = Column(Integer,
                                     index=True)  # Billing area countries, there are values that the rate is calculated according to intra / inter, no value that dnis billing.
    update_at = Column(DateTime(True), onupdate=func.now())
    update_by = Column(String)
    lnp_dipping_rate = Column(Float)
    jurisdiction_prefix = Column(PrefixRange)
    noprefix_min_length = Column(Integer)
    noprefix_max_length = Column(Integer)
    prefix_min_length = Column(Integer)
    prefix_max_length = Column(Integer)
    jur_type = Column(ChoiceType(JUR_TYPE_DICT), default=0)
    # rate_type_name = synonym('jur_type')

    origination = Column(Boolean, default=False)
    is_virtual = Column(Boolean)

    require_setup_fee = Column(Boolean, server_default='False')
    default_setup_fee = Column(Float, server_default='0')
    require_local_rate = Column(Boolean, server_default='False')

    billing_method = synonym('rate_type')
    rate_type_name = synonym('jur_type')
    code_deck = relationship(CodeDeck)
    currency = relationship(Currency)
    rates = relationship('Rate', back_populates='rate_table', uselist=True,
                         single_parent=True,
                         cascade="all, delete-orphan", )
    resources = relationship('Resource', back_populates='rate_table', uselist=True)
    trunks = relationship('ResourcePrefix', uselist=True,
                          primaryjoin='and_(ResourcePrefix.rate_table_id == RateTable.rate_table_id)')

    # def usage_count(self):return Resource.filter(Resource.rate_table_id==self.rate_table_id).count()
    code_deck_name = column_property(
        select([CodeDeck.name]).where(CodeDeck.code_deck_id == code_deck_id).correlate_except(CodeDeck))
    currency_name = column_property(
        select([Currency.code]).where(Currency.currency_id == currency_id).correlate_except(Currency))

    def before_save(self):
        # if self.rate_type in ('LRN', 'LRN BLOCK') and self.jurisdiction_country_id is None:
        #     self.jurisdiction_country_id = 1
        if self.jur_type == 'US Jurisdictional':
            self.jurisdiction_country_id = 1
        else:
            self.jurisdiction_country_id = None
        if self.jur_type in ('A-Z', 'US Non Jurisdictional') and self.rate_type is None:
            self.rate_type = 'DNIS'

    @property
    def trunks(self):
        if not self.resources:
            return []
        return [r.resource_id for r in self.resources if r.client_id and r.client_type == 'carrier']

    @classmethod
    def init(cls):
        q = cls.filter(cls.name == 'DEFAULT_TESTING').first()
        if q:
            q.delete()
        # if not q:
        #     cls(name='DEFAULT_TESTING').save()

    @property
    def rate_download_link(self):
        import jwt
        exp = datetime.now(UTC) + timedelta(days=settings.JWT_TTL_DAYS)
        token_data = {'rate_table_id': self.rate_table_id, 'exp': exp}
        token = jwt.encode(token_data, settings.JWT_SIGNATURE).decode('utf-8')
        return '{}/rate/download/{}/rate_{}.csv'.format(settings.API_URL, token, self.rate_table_id)

    @property
    def switch_ip(self):
        ips = []
        for r in VoipGateway.query().all():
            ips = ips + ['{}:{}'.format(r.active_call_ip, r.active_call_port)]
        return ','.join(ips)


DidBillingPlan.rate_table_name = column_property(
    select([RateTable.name]).where(RateTable.rate_table_id == DidBillingPlan.rate_table_id).correlate_except(RateTable))
RateMassEditLog.rate_table_name = column_property(
    select([RateTable.name]).where(RateTable.rate_table_id == RateMassEditLog.rate_table_id).correlate_except(
        RateTable))
RateMassEditLog.client_name = column_property(
    select([Client.name]).where(Client.client_id == RateMassEditLog.client_id).correlate_except(Client))

Currency.get_rate_tables_count_using_currency = \
    lambda currency_id: RateTable.filter(currency_id=currency_id).count()


class Rate(DnlApiBaseModel):
    __tablename__ = 'rate'

    rate_id = Column(Integer, primary_key=True, server_default=text_("nextval('rate_rate_id_seq'::regclass)"))
    rate_table_id = Column(ForeignKey('rate_table.rate_table_id', ondelete='CASCADE'), nullable=False, index=True)
    code = Column(PrefixRange, nullable=True, server_default=text_("''::prefix_range"), index=True)
    rate = Column(Numeric(30, 10))
    setup_fee = Column(Numeric(30, 10), nullable=False, server_default=text_("0"))
    effective_date = Column(DateTime(True), default=datetime.now(UTC), index=True)
    end_date = Column(DateTime(True), index=True)
    min_time = Column(Integer, nullable=False, default=1, server_default=text_("0"))
    grace_time = Column(Integer, nullable=False, server_default=text_("0"))
    interval = Column(Integer, nullable=False, default=1, server_default=text_("1"))
    time_profile_id = Column(ForeignKey('time_profile.time_profile_id', ondelete='CASCADE'), index=True)
    seconds = Column(Integer, default=60, nullable=False, server_default=text_("60"))
    code_name = Column(String(100))
    basic_percentages = Column(Float, server_default=text_("0"))
    gift_percentages = Column(Float, default=0, server_default=text_("0"))
    rate_type = Column(Integer, default=3, nullable=False, server_default=text_("3"))
    intra_rate = Column(Numeric(30, 10))
    inter_rate = Column(Numeric(30, 10))
    local_rate = Column(Numeric(30, 10))
    country = Column(String(1000))
    zone = Column(String(10), default='0', server_default=text_("0"))
    ocn = Column(String(10))
    lata = Column(String(10))
    create_time = Column(DateTime(True))
    did_type = Column(Integer, index=True)
    __table_args__ = (
        # UniqueConstraint('rate_table_id', 'code', 'effective_date','end_date'),
        UniqueConstraint('rate_table_id', 'code', 'effective_date', name='rate_rate_table_id_code_effective_date_key'),
        Index('class4_idx_rate_table_id_code', code, rate_table_id, postgresql_using='gist', postgresql_ops={
            'code': 'gist_prefix_range_ops',
        })
    )

    id = synonym('rate_id')
    profile = synonym('time_profile_id')
    gmt = synonym('zone')
    # indeterminte = synonym('rate')
    indeterminate = synonym('rate')
    indet_rate = synonym('rate')
    ij_rate = synonym('rate')

    rate_table = relationship(RateTable, back_populates='rates', uselist=False)
    is_effective_now = column_property(
        and_(effective_date <= func.now(), or_(end_date.is_(None), end_date >= func.now())))
    rate_table_name = column_property(
        select([RateTable.name]).where(RateTable.rate_table_id == rate_table_id).correlate_except(RateTable))

    def before_save(self):
        old_obj = Rate.filter(
            and_(Rate.code == self.code, Rate.rate_table_id == self.rate_table_id, Rate.rate_id != self.rate_id,
                 Rate.end_date != None))
        if old_obj.first() and False:
            raise ValidationError(
                'code {} is duplicate on Rate with same rate_table_id! (duplicate rate_table_id {})'.format(
                    self.code, self.rate_table_id))

    # @classmethod
    # def get_objects_list(cls, query=None, filtering=None, paging=None,
	# 					 ordering=None,query_only = False):
    #     query = get_db().tr_session.query(cls)
    #     return super().get_objects_list(query, filtering, paging,ordering,query_only)

# def __repr__(self):
#    return 'Rate ' + self.code_name + ' %7.4f' % self.rate


Rate_ = Rate.__table__.alias('rate_a')
# Rate.new_rate = column_property(
#     select([Rate_.c.rate]).where(and_(Rate.rate_table_id == Rate_.c.rate_table_id,
#                                       Rate.code == Rate_.c.code,
#                                       Rate.effective_date <= datetime.now(UTC),
#                                       Rate_.c.effective_date > datetime.now(UTC),
#                                       )).order_by(Rate_.c.effective_date).limit(1))

Rate.change_status = column_property(
    select([func.coalesce(case([(Rate_.c.rate > Rate.rate, 'increased'),
                                (Rate_.c.rate < Rate.rate, 'decreased'),
                                (Rate_.c.rate == Rate.rate, 'unchanged'),
                                ], else_='unchanged'), 'none')]).correlate(Rate).where(
        and_(Rate.rate_table_id == Rate_.c.rate_table_id,
             Rate.code == Rate_.c.code,
             Rate.effective_date <= datetime.now(UTC), Rate_.c.effective_date > datetime.now(UTC),
             )).order_by(Rate_.c.effective_date).limit(1).label('change_status'), doc='status of next rate changing',
    deferred=True)

rate1 = Rate.__table__.alias('rate1')
RateTable.code_count = column_property(literal(0)
                                       # select([func.count(rate1.c.rate_id).label('code_count')]).where(rate1.c.rate_table_id == RateTable.rate_table_id). \
                                       #     limit(1).correlate_except(rate1)
                                       )


# DidBillingPlan.min_time = column_property(
#     select([Rate.min_time]).where(DidBillingPlan.rate_table_id == Rate.rate_table_id).limit(1).correlate_except(Rate))
#
# DidBillingPlan.interval = column_property(
#     select([Rate.interval]).where(DidBillingPlan.rate_table_id == Rate.rate_table_id).limit(1).correlate_except(Rate))


# ---RateGeneration
class RateGenerationTemplate(DnlApiBaseModel):
    __tablename__ = 'rate_generation_template'
    RATE_TABLE_TYPE = {0: 'A-Z', 1: 'US Jurisdictional', 2: 'US Non Jurisdictional', 3: 'OCN-LATA', 4: 'IJ - Inter',
                       5: 'IJ - Intra', 6: 'IJ - Max(Inter,Intra)', 7: 'IJ - True Math', 8: 'IJ - True Match'}
    MARGIN_TYPE = {1: 'Percentage', 2: 'Value'}
    id = Column(Integer, primary_key=True, server_default=text_("nextval('rate_generation_template_id_seq'::regclass)"))
    name = Column(String(50))
    include_blocked_route = Column(Boolean, server_default=text_("false"))
    rate_table_type = Column(ChoiceType(RATE_TABLE_TYPE), nullable=False, server_default=text_("1"))
    lcr_digit = Column(Integer)
    default_rate = Column(Numeric(30, 10), default=0.0)
    margin_default_type = Column(ChoiceType(MARGIN_TYPE), default=1)
    margin_default_value = Column(String(30))
    default_interval = Column(Integer)
    default_min_time = Column(Integer)
    create_by = Column(String(100))
    create_on = Column(DateTime(True))
    last_generated = Column(DateTime(True))
    code_deck_id = Column(Integer)
    egress_str = Column(String(500))
    include_local_rate = Column(Boolean, server_default=text_("false"))
    effective_days = Column(Integer, server_default=text_("0"))
    decimal_places = Column(Integer, server_default=text_("0"))
    user_defined_code_deck = Column(Boolean, server_default=text_("false"))
    asr_days = Column(Integer)
    min_asr = Column(Float)
    ignore_zero_rates = Column(Boolean, server_default=text_("false"))

    details = relationship('RateGenerationTemplateDetail', back_populates='template', uselist=True,
                           single_parent=True,
                           cascade="all, delete-orphan")
    margins = relationship('RateGenerationTemplateMargin', back_populates='template', uselist=True,
                           single_parent=True,
                           cascade="all, delete-orphan")
    user_code_deck = relationship('RateGenerationCodeDeck', back_populates='template', uselist=True,
                                  single_parent=True,
                                  cascade="all, delete-orphan")

    @hybrid_property
    def _rate_table_type(self):
        return {v: k for k,v in type(self).RATE_TABLE_TYPE.items()}[self.rate_table_type] if self.rate_table_type else None

    @_rate_table_type.setter
    def _rate_table_type(self, value):
        if value and value in type(self).RATE_TABLE_TYPE:
            self.rate_table_type = type(self).RATE_TABLE_TYPE[value]

    @hybrid_property
    def calculate_ij_based_on(self):
        if self.rate_table_type in ['A-Z', 'US Jurisdictional', 'US Non Jurisdictional', 'OCN-LATA']:
            return None
        return self.rate_table_type

    @calculate_ij_based_on.setter
    def calculate_ij_based_on(self, value):
        if value:
            # self.rate_table_type = value
            self.rate_table_type = {v: k for k, v in self.RATE_TABLE_TYPE.items()}.get(value)

    @validates('name')
    def validate_name(self, field, value):
        old_obj = RateGenerationTemplate.filter(and_(RateGenerationTemplate.name == value,
                                                     RateGenerationTemplate.id != self.id)).first()
        if old_obj:
            raise Exception('name is duplicate')

        return value

    def csv_head(self):
        if self.rate_table_type=='A-Z':
            return ['code', 'rate', 'effective_date', 'interval', 'min_time', 'code_name', 'country']
        else:
            return ['code', 'inter_rate', 'intra_rate', 'rate', 'effective_date', 'interval' , 'min_time', 'code_name', 'country']

    @hybrid_property
    def egress_trunks(self):
        try:
            return [int(res.split(',')[0]) for res in self.egress_str.split(';')]
        except Exception as e:
            return []

    @egress_trunks.setter
    def egress_trunks(self, value):
        try:
            self.egress_str = ';'.join([str(x) + ',' + str(Resource.get(x).rate_table_id) for x in value])
        except Exception as e:
            pass

    @property
    def egress_trunks_names(self):
        ret = []
        for t in self.egress_trunks:
            ret.append(Resource.get(t).alias)
        return ret


class RateGenerationRate(DnlApiBaseModel):
    __tablename__ = 'rate_generation_rate'
    __table_args__ = (
        Index('rate_generation_rate_rate_generation_history_id_code_idx', 'rate_generation_history_id', 'code'),
    )
    generation_rate_id = Column(Integer, primary_key=True, server_default=text_(
        "nextval('rate_generation_rate_generation_rate_id_seq'::regclass)"))
    rate_generation_history_id = Column(ForeignKey('rate_generation_history.id', ondelete='CASCADE'), nullable=False,
                                        index=True)
    code = Column(PrefixRange, nullable=False, server_default=text_("''::prefix_range"))
    rate = Column(Numeric(30, 10))
    setup_fee = Column(Numeric(30, 10), nullable=False, server_default=text_("0"))
    effective_date = Column(DateTime(True))
    end_date = Column(DateTime(True))
    min_time = Column(Integer, nullable=False, server_default=text_("0"))
    grace_time = Column(Integer, nullable=False, server_default=text_("0"))
    interval = Column(Integer, nullable=False, server_default=text_("1"))
    time_profile_id = Column(Integer)
    seconds = Column(Integer, nullable=False, server_default=text_("60"))
    code_name = Column(String(100))
    rate_type = Column(Integer, nullable=False, server_default=text_("3"))
    intra_rate = Column(Numeric(30, 10))
    inter_rate = Column(Numeric(30, 10))
    local_rate = Column(Numeric(30, 10))
    country = Column(String(1000))
    zone = Column(String(10), server_default=text_("0"))
    ocn = Column(String(10))
    lata = Column(String(10))
    lcr_rate = Column(String(100))
    lcr_intra_rate = Column(String(100))
    lcr_inter_rate = Column(String(100))
    lcr_local_rate = Column(String(100))

    rate_generation_history = relationship('RateGenerationHistory', back_populates='rates', uselist=False)

    @classmethod
    def csv_head(cls):
        return ['code', 'rate', 'effective_date',
                'end_date',
                'min_time', 'grace_time', 'interval',
                'time_profile_id', 'seconds', 'code_name', 'rate_type', 'intra_rate', 'inter_rate', 'local_rate',
                'country', 'zone', 'ocn', 'lata']

    def copy_to_rate_table(self, rate_table, action, end_date=None, effective_date=None):
        if action == 'end date existing records':
            Rate.filter(and_(Rate.rate_table_id == rate_table.rate_table_id, Rate.code == self.code,
                             or_(Rate.end_date.is_(None), Rate.end_date > end_date))). \
                update({'end_date': end_date}, synchronize_session='fetch')
            end_date = None
        dup = Rate.filter(and_(Rate.rate_table_id == rate_table.rate_table_id, Rate.code == self.code,
                               Rate.effective_date == effective_date)).first()
        if dup:
            obj = dup
        else:
            obj = Rate(code=self.code, rate=self.rate, rate_table_id=rate_table.rate_table_id)
        obj.setup_fee = self.setup_fee
        if not effective_date is None:
            obj.effective_date = effective_date
        else:
            obj.effective_date = self.effective_date
        obj.end_date = None
        if not end_date is None:
            obj.end_date = end_date
        obj.min_time = self.min_time
        obj.grace_time = self.grace_time
        obj.interval = self.interval
        obj.time_profile_id = self.time_profile_id
        obj.seconds = self.seconds
        obj.code_name = self.code_name
        obj.rate_type = self.rate_type
        obj.intra_rate = self.intra_rate
        obj.inter_rate = self.inter_rate
        obj.local_rate = self.local_rate
        obj.country = self.country
        obj.zone = self.zone
        obj.ocn = self.ocn
        obj.lata = self.lata
        Rate.session().add(obj)

    def copy_to_dict(self, end_date=None, effective_date=None):
        return dict(
            code=self.code,
            rate=float(self.rate),
            effective_date=str(effective_date) if effective_date else '',
            end_date=str(end_date) if end_date else '',
            min_time=self.min_time if self.min_time else '',
            grace_time=self.grace_time if self.grace_time else '',
            interval=self.interval if self.interval else '',
            time_profile_id=self.time_profile_id if self.time_profile_id else '',
            seconds=self.seconds if self.seconds else '',
            code_name=self.code_name if self.code_name else '',
            rate_type=str(self.rate_type) if self.rate_type else '',
            intra_rate=self.intra_rate if self.intra_rate else '',
            inter_rate=self.inter_rate if self.inter_rate else '',
            local_rate=self.local_rate if self.local_rate else '',
            country=str(self.country) if self.country else '',
            zone=str(self.zone) if self.zone else '',
            ocn=str(self.ocn) if self.ocn else '',
            lata=str(self.lata) if self.lata else '',
        )


class RateGenerationHistory(DnlApiBaseModel):
    __tablename__ = 'rate_generation_history'
    STATUS = {0: 'initial', 1: 'in process', 2: 'finished', 3: 'error'}

    id = Column(Integer, primary_key=True, server_default=text_("nextval('rate_generation_history_id_seq'::regclass)"))
    is_applied = Column(Boolean, server_default=text_("false"))
    rate_generation_template_id = Column(ForeignKey('rate_generation_template.id', ondelete='CASCADE'))
    status = Column(ChoiceType(STATUS), nullable=False, index=True, default=0, server_default=text_("0"))
    finished_time = Column(DateTime(True))
    rate_count = Column(Integer, nullable=False, server_default=text_("0"))
    processing_time = Column(Integer, nullable=False, server_default=text_("0"))
    progress = Column(String(200))
    create_on = Column(DateTime(True), server_default=text_("('now'::text)::timestamp(0) with time zone"))
    create_by = Column(String(100))
    rate_table_type = Column(Integer)

    rate_generation_history_id = synonym('id')

    rate_generation_template = relationship('RateGenerationTemplate')
    rates = relationship('RateGenerationRate', back_populates='rate_generation_history', uselist=True,
                         single_parent=True,
                         cascade="all, delete-orphan")

    @hybrid_property
    def _is_applied(self):
        return self.finished_time == None

    @_is_applied.setter
    def _is_applied(self, value):
        self.finished_time = value

class RateGenerationHistoryDetail(DnlApiBaseModel):
    __tablename__ = 'rate_generation_history_detail'
    END_DATE_METHOD = {1: 'delete old rates', 2: 'end date all records', 3: 'end date existing records', 4: 'none'}
    rate_generation_history_detail_id_seq = Sequence('rate_generation_history_detail_id_seq')
    id = Column(Integer, primary_key=True,
                server_default=text_("nextval('rate_generation_history_detail_id_seq'::regclass)"))
    rate_generation_history_id = Column(ForeignKey('rate_generation_history.id', ondelete='CASCADE'))
    rate_table_id = Column(Integer)
    effective_date_new = Column(DateTime(True))
    effective_date_increase = Column(DateTime(True))
    effective_date_decrease = Column(DateTime(True))
    is_send_mail = Column(Boolean)
    end_date = Column(Date)
    code_deck_id = Column(Integer)
    email_template_id = Column(ForeignKey('send_rate_template.id', ondelete='SET NULL'))
    create_on = Column(DateTime(True))
    create_by = Column(String(100))
    finished_time = Column(DateTime(True))
    end_date_method = Column(ChoiceType(END_DATE_METHOD), server_default=text_("1"))
    rand_flg = Column(String(100))
    new_rate_end_date = Column(Date)
    progress = Column(String(200))
    rate_import_task_id = Column(ForeignKey('rate_import_task.id', ondelete='SET NULL'), nullable=True)

    action = synonym('end_date_method')

    override_records = column_property(
        end_date_method.in_(['delete old rates', 'end date all records', 'end date existing records']) == True)
    rate_table_name = column_property(
        select([RateTable.name]).where(RateTable.rate_table_id == rate_table_id).correlate_except(RateTable))
    count = column_property(
        select([RateGenerationHistory.rate_count]).where(RateGenerationHistory.id == rate_generation_history_id). \
            correlate_except(RateGenerationHistory))
    email_template = relationship('SendRateTemplate')
    rate_generation_history = relationship('RateGenerationHistory')


RateGenerationHistory.rate_table_name = column_property(
    select([RateTable.name]).where(RateTable.rate_table_id == RateGenerationHistoryDetail.rate_table_id). \
        where(RateGenerationHistoryDetail.rate_generation_history_id == RateGenerationHistory.id).limit(
        1).correlate_except(RateTable))


class RateGenerationTemplateDetail(DnlApiBaseModel):
    __tablename__ = 'rate_generation_template_detail'

    id = Column(Integer, primary_key=True,
                server_default=text_("nextval('rate_generation_template_detail_id_seq'::regclass)"))
    rate_generation_template_id = Column(ForeignKey('rate_generation_template.id', ondelete='CASCADE'), index=True)
    rate_interval = Column(Integer)
    min_time = Column(Integer)
    code = Column(String(50))

    template = relationship('RateGenerationTemplate', back_populates='details', uselist=False)


class RateGenerationTemplateMargin(DnlApiBaseModel):
    __tablename__ = 'rate_generation_template_margin'
    MARKUP_TYPE = {1: 'Percentage', 2: 'Value'}
    id = Column(Integer, primary_key=True,
                server_default=text_("nextval('rate_generation_template_margin_id_seq'::regclass)"))
    rate_generation_template_id = Column(ForeignKey('rate_generation_template.id', ondelete='CASCADE'), index=True)
    min_rate = Column(Numeric(30, 10))
    max_rate = Column(Numeric(30, 10))
    markup_type = Column(ChoiceType(MARKUP_TYPE))
    markup_value = Column(Numeric(30, 10))

    template = relationship('RateGenerationTemplate', back_populates='margins', uselist=False)


class RateGenerationCodeDeck(DnlApiBaseModel):
    __tablename__ = 'rate_generation_code_deck'
    rate_generation_template_id = Column(ForeignKey('rate_generation_template.id', ondelete='CASCADE'),
                                         primary_key=True, index=True)
    code = Column(String(30))
    code_name = Column(String(100))
    country = Column(String(100))

    template = relationship('RateGenerationTemplate', back_populates='user_code_deck', uselist=False)


class RateTypeOverride(DnlApiBaseModel):
    __tablename__ = 'rate_type_override'
    __table_args__ = (
        UniqueConstraint('resource_id', 'src_country', 'dest_country'),
    )

    id = Column(Integer, primary_key=True)
    src_country = Column(String(4), nullable=False)
    dest_country = Column(String(2), nullable=False, server_default='US')
    rate_type = Column(Integer, nullable=False, default=1,
                         server_default=text_('1'))
    comment = Column(Text)
    resource_id = Column(Integer, nullable=False, server_default=text_('0'))

# ---RateGeneration


class RandomAniGroup(DnlApiBaseModel):
    __tablename__ = 'random_ani_group'
    id = Column(Integer, primary_key=True)
    group_name = Column(String(100), index=True)
    create_time = Column(DateTime(True), server_default=func.now())


class RandomAniGeneration(DnlApiBaseModel):
    __tablename__ = 'random_ani_generation'

    id = Column(Integer, primary_key=True, server_default=text_("nextval('random_ani_generation_id_seq'::regclass)"))
    ani_number = Column(String(32), index=True)
    random_table_id = Column(Integer, index=True)
    enabled = Column(Boolean, nullable=False, default=True)

    def before_save(self):
        old_obj = RandomAniGeneration.filter(and_(RandomAniGeneration.ani_number == self.ani_number, 
                                                  RandomAniGeneration.random_table_id == self.random_table_id, 
                                                  RandomAniGeneration.id != self.id)).first()
        if old_obj:
            raise Exception('ani_number and random_table_id are duplicate')


class ResourceExt(DnlApiBaseModel):
    __tablename__ = 'resource_ext'
    resource_id = Column(ForeignKey('resource.resource_id', ondelete='CASCADE'), primary_key=True)
    vendor_tech_prefix = Column(PrefixRange)


t_resource_record = Table(
    'resource_record', DnlApiBaseModel.metadata,
    Column('record_id', BigInteger),
    Column('resource_id', Integer),
    Column('alias', String(100)),
    Column('time', Numeric()),
    extend_existing=True,
)

t_client_record = Table(
    'client_record', DnlApiBaseModel.metadata,
    Column('record_id', BigInteger),
    Column('client_id', Integer),
    Column('name', String(100)),
    Column('flag', String(1)),
    Column('time', Numeric()),
    extend_existing=True,
)

t_resource_ip_record = Table(
    'resource_ip_record', DnlApiBaseModel.metadata,
    Column('record_id', BigInteger),
    Column('resource_ip_id', Integer),
    Column('resource_id', String(100)),
    Column('time', Numeric()),
    extend_existing=True,
)


class Resource(DnlApiBaseModel):
    __tablename__ = 'resource'

    # __table_args__ = (
    #     UniqueConstraint('alias', 'client_id'),
    # )
    AGENT_TYPE = {1: 'exchange', 2: 'default agent', 3: 'client agent'}
    # AUTH_TYPE={0:'All',1:'Authorized by Host Only', 2: 'Accept Egress Registration', 3:'Register to Egress Trunk'}
    AUTH_TYPE = {1: 'Authorized by Host Only', 2: 'Authorized by SIP Registration', 3: 'Register to gateway'}
    BILLING_METHOD = {0: 'by minutes', 1: 'by port'}
    BILLING_TYPE = {0: 'DNIS', 1: 'LRN', 2: 'LRN Block All', 3: 'LRN Block Higher Rate Codes Only'}
    BILLING_PORT_TYPE = {0: 'Bill by allowed total port', 1: 'Bill By allowed per DID port',
                         2: 'Bill by max total port'}
    CLI_TYPE = {0: 'white', 1: 'white non cli', 2: 'gray'}
    DTMF_TYPE = {0: 'all', 1: 'info', 2: 'rfc2833', 3: 'inband'}
    JURISDICTION_USE_DNIS = {False: 'LRN', True: 'DNIS'}
    LNP_DIPPING = {1: 'Add Header', 0: 'Not Add Header'}
    MEDIA_TYPE = {2: 'Bypass Media', 0: 'Proxy Media + Transcoding', 1: 'Proxy Media'}
    PASS_THROUGH = {1: 'transparent', 2: 'transparent', 3: 'not pass through the ban'}
    PROFIT_TYPE = {1: 'percentage', 2: 'value'}
    PROTO = {1: 'sip', 2: 'h323', 0: 'all'}
    RATE_ROUNDING = {0: 'Up', 1: 'Down'}
    RES_STRATEGY = {1: 'top-down', 2: 'round-robin'}
    ROUTE_TYPE = {0: 'other', 1: 'intra', 2: 'inter', 3: 'highest'}
    RPID = {0: 'Never', 1: 'Pass Through', 3: 'Always'}
    PAID = {0: 'Drop', 1: 'Pass Through', 
            2: 'Always'}
    PAID_PRIVACY = {0: 'none', 1: 'id', 2: 'proxy'}
    OLI = {0: 'No', 1: 'Yes'}
    PCI = {0: 'never', 1: 'pass through', 2: 'always'}
    PANI = {0: 'never', 1: 'pass through', 2: 'always'}
    PRIV = {0: 'No', 1: 'Yes'}
    DIV = {0: 'No', 1: 'Yes'}
    RPID_ID_TYPE = {0: 'None', 1: 'Subscriber', 2: 'User', 3: 'Term', 4: 'Pass Through'}
    RPID_PARTY = {0: 'None', 1: 'Calling', 2: 'Called', 3: 'Pass Through'}
    RPID_PRIVACY = {0: 'None', 1: 'Full', 2: 'Name', 3: 'Url', 4: 'OFF', 5: 'Ipaddr', 6: 'Pass Through'}
    RPID_SCREEN = {0: 'None', 1: 'No', 2: 'Yes', 3: 'Pass Through'}
    SERVICE_TYPE = {0: 'Self Service', 1: 'Standand Deck'}
    STATUS = {1: 'untested', 2: 'tested', 3: 'failed', None: 'None'}
    SHAKEN_SIGN_POLICY = {0: 'Do not sign calls', 1: 'Sign only US numbers', 2: 'Sign any phone number',
                          3: 'Sign only numbers from the SHAKEN ANI group'}
    SHAKEN_VFY_POLICY = {0: 'Do not check identity', 1: 'Block if no identity',
                         2: 'Block if identity is not valid or missing', 3: 'Verify identity, but bypass if invalid.',
                         4: 'Require Valid Signature or Sign Call if no Valid Signature'}
    TRUNK_TYPE = {1: 'class4', 2: 'exchange', 3: 'product_default', 4: 'product_agent'}
    TRUNK_TYPE2 = {0: 'Termination Traffic', 1: 'DID Traffic'}

    resource_id = Column(Integer, primary_key=True)
    name = Column(String(100), unique=True)  # , nullable=False)
    ingress = Column(Boolean)
    egress = Column(Boolean)
    active = Column(Boolean, nullable=False, default=True)

    account_id = Column(String(32))
    agent_type = Column(ChoiceType(AGENT_TYPE), default=2)  # 1-exchange; 2-default agent; 3-client agent
    alias = Column(String(100), nullable=False, default='')
    amount_per_port = Column(Numeric)  # money per port
    ani_cap_limit = Column(Integer)
    ani_cps_limit = Column(Integer)
    auth_type = Column(ChoiceType(AUTH_TYPE),
                       default=1)  # 1 - Authorized by Host Only 2 - Accept Egress Registration 3 - Register to Egress Trunk###permission field: 0 - ALL; 1 - resource_auth;
    bill_by = Column(Integer,
                     default=4)  # 0 - DNIS, DNIS Billing, do not need to query LRN
    # 1 - LRN, LRN billing, ignoring carrier changes
    # 2 - LRN Block, LRN Billing Number Operator has changed, then declined
    # 3 - LRN Block Higher Rate, LRN rates higher than DNIS rates
    # 4 - Follow Rate Deck, using the above schedule for the rate table
    # rate_profile = Column(Integer, default=0)  # 0 --- false; 1 --- true

    billing_method = Column(ChoiceType(BILLING_METHOD), default=0)  # 0 - by minutes; 1- by port
    billing_port_type = Column(ChoiceType(BILLING_PORT_TYPE), default=0)
    billing_rule = Column(Integer)
    billing_type = Column(ChoiceType(BILLING_TYPE),
                          default=0)  # 0: DNIS; 1: LRN; 2: LRN Block All; 3: LRN Block Higher Rate Codes Only
    block_404_number_time = Column(Integer)
    block_time = Column(Integer)  # loop detection used in seconds to meet the block condition after the block how long
    canada_other = Column(ChoiceType(ROUTE_TYPE), default=0)  # 0 --- other; 1 --- intra; 2 --- inter; 3 --- highest
    canada_route = Column(ChoiceType(ROUTE_TYPE), default=0)  # 0 --- other; 1 --- intra; 2 --- inter; 3 --- highest
    capacity = Column(Integer)  # Allows the number of online calls
    cli_type = Column(ChoiceType(CLI_TYPE))  # 0-white; 1-white non cli; 2-gray
    client_id = Column(ForeignKey('client.client_id', ondelete='SET NULL'), index=True)  # wholesaler
    cost_per_port = Column(Numeric(30, 10))
    counter_time = Column(
        Integer)  # loop detection used, the unit seconds, this time period to receive the number of calls are called the same, then block
    cps_limit = Column(Integer)  # Allows the number of calls per second
    create_time = Column(DateTime(timezone=True), default=datetime.now(UTC))
    delay_bye_limit = Column(Integer)
    delay_bye_second = Column(Integer)
    disable_by_alert = Column(Boolean, default=False)
    display_name = Column(Integer)  # 0 --- false; 1 --- true; whether or not to pass through the caller's display name
    div = Column(ChoiceType(DIV))
    dnis_cap_limit = Column(Integer)
    dnis_cps_limit = Column(Integer)
    dnis_only = Column(Boolean, nullable=False, default=True)
    dtmf_detect = Column(Integer,
                         default=0)  # 0 - auto detect, media_type = transcode; 1 - scheduled detection, media_type = transcode and main and called dtmf are different and have the same speech code
    dtmf_type = Column(ChoiceType(DTMF_TYPE), default=3)  # 0 - all; 1 - info; 2 - rfc2833; 3 - inband
    dummy_trunk = Column(Boolean)  # whether the origination client failover trunk
    egress_bill_after_action = Column(Boolean, default=True)
    enfource_cid = Column(Boolean, default=False)
    enough_balance = Column(Boolean, default=True)  # false whether the balance is insufficient
    group_id = Column(ForeignKey('trunk_group.group_id', ondelete='SET NULL'))
    resource_block_group_id = Column(Integer)
    ignore_early_media = Column(Boolean, default=False)
    ignore_early_nosdp = Column(Integer)
    ignore_ring = Column(Boolean, default=False)
    inband = Column(Integer)  # 0 Does Not Support / Does Not Require; 1 Support / Require
    info = Column(Integer)  # 0 Does Not Support / Does Not Require; 1 Support / Require
    intl_route = Column(ChoiceType(ROUTE_TYPE), default=0)  # 0 --- other; 1 --- intra; 2 --- inter; 3 --- highest
    is_del = Column(Integer, default=0)  # 0: not del; 1: is del
    is_virtual = Column(Boolean)
    jurisdiction_use_dnis = Column(ChoiceType(JURISDICTION_USE_DNIS), default=False)
    last_priority = Column(Integer, default=0)
    lnp = Column(Boolean, nullable=False, default=False)
    lnp_dipping = Column(Boolean, default=False)
    lnp_dipping_rate = Column(Float)
    lrn_block = Column(Boolean, nullable=False, default=False)
    lrn_prefix = Column(Integer, nullable=False, default=1)  # 0 --- false; 1 --- true
    marketplace = Column(Integer, default=0)  # 0: International A-Z; 1: US Domestic
    max_duration = Column(Integer)
    media_timeout = Column(Integer)
    media_type = Column(ChoiceType(MEDIA_TYPE), nullable=False,
                        default=2)  # 0 - Proxy Media + Transcoding; 1 - Proxy Media; 2 - Bypass Media
    number = Column(
        Integer)  # loop detection Used to indicate the number of calls received for the same time in the same time, then block
    oli = Column(ChoiceType(OLI))
    paid = Column(ChoiceType(PAID))
    paid_privacy = Column(ChoiceType(PAID_PRIVACY), default=0)
    paid_uri = Column(Text)
    paid_display = Column(Text)
    pass_response_code = Column(Integer,
                                default=0)  # 0 - false, 1 - true, if this option is true, the last sip error returned from the egress is returned to ingress
    pass_through = Column(ChoiceType(
        PASS_THROUGH))  # pass through the rules: 1 - transparent; 2 - not transparent; 3 - not pass through the ban
    pci = Column(ChoiceType(PCI))
    pani = Column(ChoiceType(PANI))
    pani_value = Column(Text)
    price_per_actual_channel = Column(Float, default=0)
    price_per_max_channel = Column(Float, default=0)
    priority = Column(Integer, default=0)
    priv = Column(ChoiceType(PRIV))
    private = Column(Integer, nullable=False, default=0)
    product_id = Column(Integer)
    profit_margin = Column(Float(53), nullable=False, default=0)  # minimum profit margin
    profit_type = Column(ChoiceType(PROFIT_TYPE), nullable=False, default=1,
                         server_default=text_('1'))  # 1 - Percentage; 2 - value
    proto = Column(ChoiceType(PROTO), default=1)  # 1 - sip; 2 - h323; 0 - all
    purged = Column(Boolean, nullable=False, server_default=text("false"))
    random_table_id = Column(ForeignKey('random_ani_group.id', ondelete='SET NULL'))
    rate_decimal = Column(Integer, default=6)  # Determines how many bits are the fractional bits of the rate.
    rate_profile = Column(Integer, default=0)
    rate_rounding = Column(ChoiceType(RATE_ROUNDING),
                           default=0)  # rate decimal place method: 0 --- rounding; 1 --- discarded
    rate_table_id = Column(ForeignKey('rate_table.rate_table_id', ondelete='SET NULL'), index=True)
    rate_use_rpid = Column(Boolean, default=False)
    rating_type = Column(Integer, default=0)  # 0: US JD; 1: US Non-JD; 2: OCN-LATA JD; 3: OCN-LATA Non-JD
    re_invite = Column(Integer)  # 0 - false; 1 - true; send re-INVITE to detect whether the other party is active
    re_invite_interval = Column(Integer)  # seconds unit, it is recommended not less than 60 seconds
    redirect = Column(Integer, default=0)  # 0 --- false; 1 --- true
    res_strategy = Column(ChoiceType(RES_STRATEGY))  # Landing Gateway Selection Strategy: 1 - top-down, 2 - round-robin
    resource_block_group_id = Column(Integer)
    resource_template_id = Column(Integer)
    rfc2833 = Column(Integer)  # 0 Does Not Support / Does Not Require; 1 Support / Require
    rfc_2833 = Column(Boolean, nullable=False, default=False)
    rfc_r833_payload = Column(Integer)
    ring_timeout = Column(Integer, nullable=False, default=60)
    route_strategy_id = Column(ForeignKey('route_strategy.route_strategy_id', ondelete='SET NULL'), index=True)
    rpid = Column(ChoiceType(RPID), default=1)
    rpid_id_type = Column(ChoiceType(RPID_ID_TYPE),
                          default=0)  # 0 --- none call out without id_type: 1 --- subscriber; 2 --- user; 3 --- term; 4 --- proxy use caller IDID_type
    rpid_party = Column(ChoiceType(RPID_PARTY),
                        default=0)  # 0 --- none call out without party; 1 --- calling; 2 --- called; 3 --- proxy use caller RPID party
    rpid_privacy = Column(ChoiceType(RPID_PRIVACY),
                          default=0)  # 0 --- none call out without privacy: 1 --- full; 2 --- name; 3 --- url; 4 --- off; 5 --- ipaddr; 6 --- proxy use Calling RPID privacy
    rpid_screen = Column(ChoiceType(RPID_SCREEN),
                         default=0)  # 0 - none without screen; 1 - no; 2 - yes; 3 - proxy use caller RPID screen
    service_type = Column(ChoiceType(SERVICE_TYPE), nullable=False, default=0)  # 0: Self Service; 1: Standand Deck

    shaken_sign_policy = Column(SmallInteger, server_default='0', nullable=False) #ChoiceType(SHAKEN_SIGN_POLICY), server_default='0', nullable=False)
    shaken_vfy_policy = Column(SmallInteger, server_default='0', nullable=True)
    shaken_allow_resign = Column(Boolean, nullable=False, server_default='false')
    shaken_ani_group_list_id = Column(ForeignKey('shaken_ani_group_list.id', ondelete='SET NULL'), index=True)
    shaken_default_attest_lvl = Column(String(1), nullable=False, server_default="'A'")
    shaken_p_headers = Column(Integer, nullable=True, server_default=text("0")) # Add SHAKEN P-headers to outgoing INVITE: 0 - None; 1 - Regular; 2 - Extended
    status = Column(ChoiceType(STATUS), default=1)  # 1 - untested; 2 - tested; 3 - failed
    switch_profile_id = Column(Integer)
    t38 = Column(Boolean, nullable=False, default=True)
    tdm = Column(Boolean, nullable=False, default=True)
    transaction_fee_id = Column(Integer)
    transnexus = Column(Integer)
    trunk_type = Column(ChoiceType(TRUNK_TYPE))  # 1-class4; 2-exchange; 3-product_default; 4-product_agent
    trunk_type2 = Column(ChoiceType(TRUNK_TYPE2), nullable=False,
                         default=0)  # {0:'Termination Traffic', 1:'DID Traffic'}
    update_at = Column(DateTime(timezone=True), onupdate=func.now())
    update_by = Column(String)
    update_time = Column(DateTime(timezone=True), onupdate=func.now())
    us_other = Column(ChoiceType(ROUTE_TYPE), default=0)  # 0 --- other; 1 --- intra; 2 --- inter; 3 --- highest
    us_route = Column(ChoiceType(ROUTE_TYPE), default=0)  # 0 --- other; 1 --- intra; 2 --- inter; 3 --- highest
    vfy_policy = Column(SmallInteger, server_default='0', nullable=False)
    wait_ringtime180 = Column(Integer, server_default=text('60000'))
    ym_fraud_prob = Column(Float)
    ym_spam_score = Column(Float)
    ym_tcpa_prob = Column(Float)
    # vendor_tech_prefix = Column(PrefixRange)#v5

    rate_profile = Column(Integer)
    priv_value = Column(Integer, server_default=text("0"))
    use_sip404_ani = Column(Boolean, server_default=text("false"))

    old_detail_table = Column(JSON)
    max_failover_route_count = Column(Integer)
    #Add ANI/DNIS blocking by Lerg
    # non_us_ani_action = Column(Integer,nullable=False, server_default='0')
    block_dnc = Column(Boolean, nullable=False, server_default=text("false"))
    block_wireless = Column(Boolean, nullable=False, server_default=text("false"))
    block_wireless_ani = Column(Boolean)
    block_toll_free_ani = Column(Boolean)
    invalid_ani_action = Column(Integer)
    ani_country_whitelist = Column(String)
    block_dno = Column(String(15))
    ftc_days = Column(Integer, nullable=True)

    ani_type_block = Column(Text)
    dnis_type_block = Column(Text)
    shaken_force_validation = Column(Boolean, default=True)

    shaken_sign_tollfree = Column(Boolean)
    shaken_vfy_countries = Column(Text)

    __mapper_args__ = {'polymorphic_on': ingress}

    authorization_type = synonym('auth_type')
    bypass_media = synonym('media_type')
    call_limit = synonym('capacity')
    carrier_id = synonym('client_id')
    host_routing_strategy = synonym('res_strategy')
    is_active = synonym('active')
    min_duration = synonym('delay_bye_second')
    min_profit_type = synonym('profit_type')
    billing_by_port = synonym('billing_method')
    min_profit_value = synonym('profit_margin')
    pass_lrn = synonym('lnp_dipping')
    pdd = synonym('wait_ringtime180')
    route_plan_id = synonym('route_strategy_id')
    trunk_id = synonym('resource_id')
    ingress_trunk_name = synonym('alias')
    allowed_ports = synonym('dnis_cap_limit')
    trunk_name = synonym('alias')
    ani_group_id = synonym('random_table_id')

    client_name = column_property(select([Client.name]).where(Client.client_id == client_id).correlate_except(Client))
    client_type = column_property(
        select([Client.client_type]).where(Client.client_id == client_id).correlate_except(Client))
    client_mode = column_property(select([Client.mode]).where(Client.client_id == client_id).correlate_except(Client))
    client_status = column_property(
        select([Client.status]).where(Client.client_id == client_id).correlate_except(Client))
    client_allowed_credit = column_property(
        select([Client.allowed_credit]).where(Client.client_id == client_id).correlate_except(Client))
    client_call_limit = column_property(
        select([Client.call_limit]).where(Client.client_id == client_id).correlate_except(Client))
    client_cps_limit = column_property(
        select([Client.cps_limit]).where(Client.client_id == client_id).correlate_except(Client))

    rate_email = column_property(
        select([func.coalesce(Client.rate_email, Client.email)]).where(Client.client_id == client_id).correlate_except(
            Client))
    email = column_property(select([Client.email]).where(Client.client_id == client_id).correlate_except(Client))
    carrier = column_property(select([Client.name]).where(Client.client_id == client_id).correlate_except(Client))
    client_is_online = column_property(
        select([Client.is_online]).where(Client.client_id == client_id).correlate_except(Client))
    carrier_is_active = column_property(
        select([Client.status]).where(Client.client_id == client_id).correlate_except(Client))

    rate_table_name = column_property(
        select([RateTable.name]).where(RateTable.rate_table_id == rate_table_id).correlate_except(RateTable))
    dynamic_count = column_property(select([func.count(DynamicRouteItem.id)]). \
                                    where(DynamicRouteItem.resource_id == resource_id). \
                                    correlate_except(DynamicRouteItem)
                                    )
    static_count = column_property(select([func.count(distinct(Product.product_id))]). \
                                   where(and_(ProductItemsResource.resource_id == resource_id, ProductItems.product_id == Product.product_id,
                                              ProductItemsResource.item_id == ProductItems.item_id, ProductItems.product_id.isnot(None),
                                              Product.name != 'ORIGINATION_STATIC_ROUTE')). \
                                   correlate_except(ProductItemsResource)
                                   )
    ani_group_name = column_property(
        select([RandomAniGroup.group_name]).where(RandomAniGroup.id == random_table_id).correlate_except(RandomAniGroup)
    )

    client = relationship(Client, uselist=False, back_populates='resources', enable_typechecks=False)
    group = relationship(TrunkGroup, back_populates='trunks', uselist=False)
    route_strategy = relationship(RouteStrategy)
    product_items = relationship(ProductItemsResource, uselist=True, back_populates='resource',
                                 single_parent=True, cascade="all, delete-orphan")
    ip = relationship('ResourceIp',
                      primaryjoin='and_(ResourceIp.resource_id == Resource.resource_id,ResourceIp.reg_type==0)',
                      uselist=True, back_populates='resource', single_parent=True, cascade="all, delete-orphan")

    reg_user = relationship('ResourceIp',
                            primaryjoin='and_(ResourceIp.resource_id == Resource.resource_id,ResourceIp.reg_type==1)',
                            uselist=True,  # back_populates='resource',
                            single_parent=True, cascade="all, delete-orphan")

    reg_gateway = relationship('ResourceIp',
                               primaryjoin='and_(ResourceIp.resource_id == Resource.resource_id,ResourceIp.reg_type==2)',
                               uselist=True,  # back_populates='resource',
                               single_parent=True, cascade="all, delete-orphan")

    rate_table = relationship('RateTable', back_populates='resources', uselist=False)
    codec_list = relationship('ResourceCodecsRef', uselist=True, back_populates='resource', single_parent=True,
                              cascade="all, delete-orphan", order_by='ResourceCodecsRef.id')
    vendor_dids = relationship('DidBillingRel',
                               primaryjoin='and_(foreign(DidBillingRel.vendor_res_id) == Resource.resource_id)',
                               # ,DidBillingRel.end_date.is_(None))',
                               uselist=True, back_populates='vendor_res',
                               cascade="all, delete-orphan", lazy='noload')
    client_dids = relationship('DidBillingRel',
                               primaryjoin='and_(foreign(DidBillingRel.client_res_id) == Resource.resource_id)',
                               # DidBillingRel.end_date.is_(None))',
                               uselist=True, back_populates='client_res',
                               cascade="all, delete-orphan")
    # client_did_assignments = relationship('DidBillingRel',
    #                             primaryjoin='and_(foreign(DidBillingRel.client_res_id) == Resource.resource_id, DidBillingRel.end_date.is_(None))',
    #                             uselist=True, back_populates='client_res',
    #                             cascade="all, delete-orphan")

    prefixes = relationship('ResourcePrefix', uselist=True, back_populates='resource',
                            single_parent=True, cascade="all, delete-orphan", lazy='dynamic'
                            )
    directions = relationship('ResourceDirection', uselist=True, back_populates='resource',
                              cascade="all, delete-orphan")
    replace_actions = relationship('ResourceReplaceAction', uselist=True, back_populates='resource',
                                   cascade="all, delete-orphan")
    digit_maps = relationship(ResourceTranslationRef, back_populates='gateway', uselist=True, single_parent=True,
                              cascade="all, delete-orphan", enable_typechecks=False)
    ingress_switch_profiles = relationship('EgressProfile',
                                           primaryjoin='and_(foreign(EgressProfile.ingress_id)==Resource.resource_id,Resource.ingress==True)')
    # failover = relationship('ResourceFailover',
    #                         primaryjoin='foreign(ResourceFailover.resource_id)==Resource.resource_id',
                            # cascade="save-update, merge, delete, delete-orphan")

    attestation_settings = relationship('ShakenAniGroupListRel',
                      primaryjoin='and_(foreign(ShakenAniGroupListRel.ani_group_list_id) == Resource.shaken_ani_group_list_id)',
                      uselist=True, single_parent=True, cascade="all, delete-orphan")

    cid_block_config = relationship('ResourceCidBlockConfig',
                            primaryjoin='foreign(ResourceCidBlockConfig.resource_id)==Resource.resource_id')

    ext = relationship('ResourceExt', uselist=False, cascade="all, delete-orphan")

    host = synonym('ip')

    def before_save(self):
        # if self.client_id:
        #     cl_obj = Client.get(self.client_id)
        #     if (not cl_obj) or cl_obj.is_orig == self.ingress:
        #         raise Exception('egress trunk can only have vendor and ingress trunk can only have client')

        # if self.client:
        #     if self.client.cps_limit and self.cps_limit and self.cps_limit > self.client.cps_limit:
        #         raise ValidationError('Resource\'s cps_limit shouldn\'t exceed client\'s cps_limit')
        #     if self.client.call_limit and self.call_limit and self.call_limit > self.client.call_limit:
        #         raise ValidationError('Resource\'s call_limit shouldn\'t exceed client\'s call_limit')

        self.ani_country_whitelist = self.ani_country_whitelist.strip() if self.ani_country_whitelist else None
        for p in self.prefixes:
            p.before_save()
        if self.rate_decimal is None:
            conf = SystemParameter.get(1)
            if conf:
                self.rate_decimal = conf.default_billing_decimal
        if self.client:
            if self.client.cps_limit and self.cps_limit and self.cps_limit > self.client.cps_limit:
                Resource.session().rollback()
                raise ValidationError('Resource\'s cps_limit shouldn\'t exceed client\'s cps_limit')
            if self.client.call_limit and self.call_limit and self.call_limit > self.client.call_limit:
                Resource.session().rollback()
                raise ValidationError('Resource\'s call_limit shouldn\'t exceed client\'s call_limit')
        for i in self.ip + self.reg_gateway + self.reg_user:
            if i.direction is None:
                if self.ingress:
                    i.direction = 0
                else:
                    i.direction = 1

    def sync_adjacent_ips(self):
        if self.trunk_type2 == 'DID Traffic' and self.egress and self.client and self.client.name == self.alias:
            for adj in Resource.filter(and_(Resource.client_id == self.client_id, Resource.egress == self.egress,
                                            Resource.client_id == self.client_id, Resource.alias != self.alias)):
                adj.ip = []
                for i in self.ip:
                    adj.ip.append(ResourceIp(ip=i.ip, port=i.port, fqdn=i.fqdn, reg_type=i.reg_type))
                self.session().add(adj)
            log.debug('sync_adjacent_ips success')

    def apply_mail(self, tpl, to=None, att=[]):
        return MailSender.apply_mail(self, tpl, to, att)

    @hybrid_property
    def _min_profit_value(self):
        if self.min_profit_value != -1:
            return self.min_profit_value
        return None

    @_min_profit_value.setter
    def _min_profit_value(self, value):
        if value is not None:
            self.min_profit_value = value
            return
        self.min_profit_value = -1

    @hybrid_property
    def vendor_tech_prefix(self):
        try:
            return self.ext.vendor_tech_prefix
        except:
            return None
        return None

    @vendor_tech_prefix.setter
    def vendor_tech_prefix(self, value):
        ext = self.ext
        if not ext:
            self.ext = ResourceExt(vendor_tech_prefix=value)
        else:
            self.ext.vendor_tech_prefix = value
        # self.session().add(ext)

    @hybrid_property
    def enable_global_404_blocking(self):
        try:
            return bool(self.block_404_number_time)
        except:
            return False

    @enable_global_404_blocking.setter
    def enable_global_404_blocking(self, value):
        self.block_404_number_time = int(value)

    @hybrid_property
    def ignore_early_no_sdp(self):
        try:
            return bool(self.ignore_early_nosdp)
        except:
            return False

    @ignore_early_no_sdp.setter
    def ignore_early_no_sdp(self, value):
        self.ignore_early_nosdp = int(value)

    @hybrid_property
    def prefix(self):
        try:
            return ','.join([p.tech_prefix for p in self.prefixes])
        except:
            return ''

    @prefix.setter
    def prefix(self, value):
        try:
            if value != None:
                if len(self.prefixes) > 0:
                    self.prefixes[0].tech_prefix = value
                else:
                    pr = ResourcePrefix(tech_prefix=value,
                                        route_strategy_id=self.route_strategy_id, rate_table_id=self.rate_table_id)
                    self.prefixes.append(pr)
        except:
            pass

    @hybrid_property
    def static_route_items(self):
        ret = []

        for it in self.product_items:
            ret.append(dict(prefix=it.product.digits, static_route_id=it.product.product_id))
        return ret

    @static_route_items.setter
    def static_route_items(self, value):
        for it in self.product_items:
            it.delete()
        for it in value:
            prefix = it['prefix']
            id = it['static_route_id']
            p = ProductItems.filter(and_(ProductItems.product_id == id, ProductItems.digits == prefix)).first()
            if not p:
                p = ProductItems(product_id=id, digits=prefix)
            tr = ProductItemsResource(resource_id=self.resource_id)
            p.trunks.append(tr)
            self.product_items.append(tr)

    @hybrid_property
    def _reg_user(self):
        return self.reg_user

    @_reg_user.setter
    def _reg_user(self, value):
        cls = model.ResourceIp
        last = cls.filter(and_(cls.resource_id == self.resource_id, cls.reg_type == 1)).first()
        port = last.port if last else None
        cls.filter(and_(cls.resource_id == self.resource_id, cls.reg_type == 1)).delete(synchronize_session='fetch')
        for user in value:
            if not getattr(user, 'port', None):
                user.port = port
        self.reg_user = value
    # self.session().add(tr)
    # self.session().add(p)

    # def rate_table_name(self):
    #    return self.rate.name

    def route_plan_name(self):
        return self.route_strategy.name

    def group_name(self):
        return self.group.name

    def product_name(self):
        try:
            return self.product_rout.product_name
        except:
            return None

    # def carrier(self):
    #    return self.client.name

    def usage_count(self):
        cnt1 = ProductItemsResource.filter(ProductItemsResource.resource_id == self.resource_id).count()
        cnt2 = DynamicRouteItem.filter(DynamicRouteItem.resource_id == self.resource_id).count()
        return cnt1 + cnt2

    def trunk_count(self):
        return Resource.filter(Resource.client_id == self.client_id).count()

    @hybrid_property
    def direction(self):
        if self.ingress:
            return 'ingress'
        else:
            return 'egress'

    @hybrid_property
    def type(self):
        if self.ingress and self.active:
            return 'orig'
        if self.egress and self.active:
            return 'term'
        return None

    @hybrid_property
    def codes(self):
        try:
            return [t.translation.translation_name for t in self.digit_maps]
        except:
            return []

    @codes.setter
    def codes(self, value):
        for t in self.digit_maps:
            self.digit_maps.remove(t)
        for name in value:
            q = DigitTranslation.filter(DigitTranslation.translation_name == name).first()
            if q:
                ref = ResourceTranslationRef(translation_id=q.translation_id)
                self.digit_maps.append(ref)

    @hybrid_property
    def codecs(self):
        try:
            return [c.codec.name for c in self.codec_list]
        except Exception as e:
            return []

    @codecs.setter
    def codecs(self, value):
        # for t in self.codec_list:
        # 	self.codec_list.remove(t)
        self.codec_list[:] = []
        for name in value:
            q = Codec.filter(Codec.name == name).first()
            if q:
                ref = ResourceCodecsRef(codec_id=q.id)
                self.codec_list.append(ref)

    @hybrid_property
    def actions(self):
        try:
            DIRECTION = {0: 'ALL', 1: 'ingress', 2: 'egress'}
            ACT = {'plus prefix': 'Add Prefix', 'minus prefix': 'Delete Prefix', 'plus suffix': 'Add Suffix',
                   'minus suffix': 'Delete Suffix'}
            AF = {'plus prefix': 'new_digits', 'minus prefix': 'num_digits', 'plus suffix': 'new_digits',
                  'minus suffix': 'num_digits'}
            DIR = {'modify the caller': 'ani', 'modify the called': 'dnis'}
            TYPE = {0: 'modify the caller', 1: 'modify the called'}
            ret = []
            ac = None
            if self.directions:
                ac = {'ani': {}, 'dnis': {}}
                for a in self.directions:
                    if not a.action in ['plus prefix', 'minus prefix', 'plus suffix', 'minus suffix']:
                        continue
                    if not a.direction == self.direction:
                        continue
                    if ac[DIR[a.type]] != {}:
                        ret.append(ac)
                        ac = {'ani': {}, 'dnis': {}}

                    ac[DIR[a.type]]['action'] = ACT[a.action]
                    ac[DIR[a.type]][AF[a.action]] = a.digits
                    ac[DIR[a.type]]['match_prefix'] = a.dnis
                    ac[DIR[a.type]]['number_length'] = a.number_length
                    ac[DIR[a.type]]['number_type'] = a.number_type
                    if ac:
                        ret.append(ac)
                        ac = {'ani': {}, 'dnis': {}}

            if self.replace_actions:
                if not ac:
                    ac = {'ani': {}, 'dnis': {}}
                for a in self.replace_actions:
                    TYPE = {0: 'ani only replace the caller', 1: 'dnis only replace the called',
                            2: 'both replace the calling and called'}
                    if a.type in ['ani only replace the caller', 'both replace the calling and called']:
                        if ac['ani'] != {}:
                            ret.append(ac)
                            ac = {'ani': {}, 'dnis': {}}
                        ac['ani']['action'] = 'Replace'
                        ac['ani']['new_digits'] = a.ani
                        ac['ani']['match_prefix'] = a.ani_prefix
                        ac['ani']['min_length'] = a.ani_min_length
                        ac['ani']['max_length'] = a.ani_max_length

                    if a.type in ['dnis only replace the called', 'both replace the calling and called']:
                        if ac['dnis'] != {}:
                            ret.append(ac)
                            ac = {'ani': {}, 'dnis': {}}
                        ac['dnis']['action'] = 'Replace'
                        ac['dnis']['new_digits'] = a.dnis
                        ac['dnis']['match_prefix'] = a.dnis_prefix
                        ac['dnis']['min_length'] = a.dnis_min_length
                        ac['dnis']['max_length'] = a.dnis_max_length
                    if ac:
                        ret.append(ac)
                        ac = {'ani': {}, 'dnis': {}}
            return ret
        except Exception as e:
            log.warning('Cannot get actions for did repository {}'.format(str(e)))

    @actions.setter
    def actions(self, value):
        try:
            ACT = {'plus prefix': 'Add Prefix', 'minus prefix': 'Delete Prefix', 'plus suffix': 'Add Suffix',
                   'minus suffix': 'Delete Suffix'}
            ACTT = rev(ACT)
            for d in self.directions:
                d.delete()
            for d in self.replace_actions:
                d.delete()
            # if value:
            #    a=value
            for a in value:
                ani = None
                dnis = None
                for t in ['ani', 'dnis']:
                    if not t in a:
                        continue
                    ac = a[t]
                    digits = None
                    num_digits = None
                    if not 'action' in ac:
                        continue
                    actp = ac['action']
                    if actp in ('Add Prefix', 'Add Suffix') and 'new_digits' in ac:
                        digits = ac.get('new_digits', None)
                        number_length = ac.get('number_length', None)
                        number_type = ac.get('number_type', None)
                    if actp in ('Delete Prefix', 'Delete Suffix') and 'num_digits' in ac:
                        digits = ac.get('num_digits', None)
                        number_length = ac.get('number_length', None)
                        number_type = ac.get('number_type', None)

                    if actp == 'Replace' and t == 'ani' and 'new_digits' in ac:
                        ani = ac.get('new_digits', None)

                    if actp == 'Replace' and t == 'dnis' and 'new_digits' in ac:
                        dnis = ac.get('new_digits', None)

                    rd = 'modify the caller' if t == 'ani' else 'modify the called'
                    if digits or num_digits:
                        self.directions.append(
                            ResourceDirection(action=ACTT[actp], type=rd, digits=digits,
                                              direction=self.direction, dnis=ac.get('match_prefix'),
                                              number_length=number_length, number_type=number_type))

                    if ani or dnis:
                        tp, ani_prefix, ani_min_length, ani_max_length, dnis_prefix, dnis_min_length, dnis_max_length = None, None, None, None, None, None, None
                        if ani:
                            tp = 'ani only replace the caller'
                            aac = a['ani']
                            ani_prefix = aac.get('match_prefix', None)
                            ani_min_length = aac.get('min_length', None)
                            ani_max_length = aac.get('max_length', None)
                        if dnis:
                            tp = 'dnis only replace the called'
                            aac = a['dnis']
                            dnis_prefix = aac.get('match_prefix', None)
                            dnis_min_length = aac.get('min_length', None)
                            dnis_max_length = aac.get('max_length', None)
                        if ani and dnis:
                            tp = 'both replace the calling and called'
                        if tp:
                            rep = ResourceReplaceAction(type=tp, ani=ani, dnis=dnis,
                                                        ani_prefix=ani_prefix, ani_min_length=ani_min_length,
                                                        ani_max_length=ani_max_length,
                                                        dnis_prefix=dnis_prefix, dnis_min_length=dnis_min_length,
                                                        dnis_max_length=dnis_max_length)
                            self.replace_actions.append(rep)
        except Exception as e:
            log.warning('Cannot get actions for did repository {}'.format(str(e)))

    def create_from_template(self, name, template_id):
        obj = ResourceTemplate.filter(ResourceTemplate.resource_template_id == template_id).first()
        if not obj:
            return
        self.resource_template_id = template_id
        if obj.direction == 'egress':
            dir = 1
            self.egress = True
            self.ingress = False
        else:
            dir = 0
            self.egress = False
            self.ingress = True
        self.name = name
        self.alias = name
        self.trunk_type = obj.trunk_type
        self.profit_type = obj.profit_type
        self.res_strategy = obj.res_strategy
        self.media_type = obj.media_type
        if obj.rpid_screen:
            self.rpid_screen = obj.rpid_screen
        self.rpid_party = obj.rpid_party
        self.rpid_privacy = obj.rpid_privacy
        self.rpid_id_type = obj.rpid_id_type

        self.amount_per_port = obj.amount_per_port
        self.bill_by = obj.bill_by
        self.billing_method = obj.billing_method
        self.billing_rule = obj.billing_rule
        self.canada_other = obj.canada_other
        self.canada_route = obj.canada_route
        if obj.codecs_str:
            self.codecs = obj._codecs_str.split(',')
        self.delay_bye_second = obj.delay_bye_second
        self.display_name = obj.display_name
        self.div = obj.div
        self.ignore_early_media = obj.ignore_early_media
        self.ignore_early_nosdp = obj.ignore_early_nosdp
        self.ignore_ring = obj.ignore_ring
        self.inband = obj.inband
        self.info = obj.info
        self.intl_route = obj.intl_route
        self.lnp_dipping = obj.lnp_dipping
        self.lnp_dipping_rate = obj.lnp_dipping_rate
        self.max_duration = obj.max_duration
        self.media_timeout = obj.media_timeout
        self.oli = obj.oli
        self.paid = obj.paid
        self.pci = obj.pci
        self.priv = obj.priv
        self.profit_margin = obj.profit_margin
        self.random_table_id = obj.random_table_id
        self.rate_decimal = obj.rate_decimal
        self.rate_profile = obj.rate_profile
        self.rate_rounding = obj.rate_rounding
        self.rate_table_id = obj.rate_table_id
        self.re_invite = obj.re_invite
        self.re_invite_interval = obj.re_invite_interval
        self.rfc2833 = obj.rfc2833
        self.ring_timeout = obj.ring_timeout
        self.trunk_type2 = obj.trunk_type2
        self.us_other = obj.us_other
        self.us_route = obj.us_route
        self.wait_ringtime180 = obj.wait_ringtime180
        for rec in obj.ip:
            if rec['type'] == 'IP Address':
                r = ResourceIp(ip=rec['host'], port=rec['port'], direction=dir)
            else:
                r = ResourceIp(fqdn=rec['host'], port=rec['port'], direction=dir)
            if self.client_id:
                r.client_id = self.client_id
            self.ip.append(r)
        for rec in obj.failover_rule:
            r = ResourceFailover(resource_id=self.resource_id, failover_strategy=rec['failover_method'],
                                 from_sip_code=rec['match_code'],
                                 to_sip_string=rec['return_clause'], to_sip_code=rec['return_code'])
            r.save()

    def assign_vendor_did_prefix(self, did, rate_table):
        log.debug('assign_vendor_did_prefix(resource_id={},alias={}, did={}, rate_table={})'.format(self.resource_id,
                                                                                                    self.alias, did,
                                                                                                    rate_table))
        if self.client and self.client.resource:
            tech_prefix = self.client.resource.vendor_tech_prefix or ''
            ResourcePrefix.filter(
                and_(ResourcePrefix.code == did, ResourcePrefix.resource_id == self.resource_id)).delete()
            did_prefix = ResourcePrefix(code=did, route_strategy_id=RouteStrategy.get_orig_routing_plan_id(),
                                        # tech_prefix=tech_prefix+did
                                        tech_prefix=tech_prefix
                                        )
            did_prefix.rate_table = rate_table
            self.prefixes.append(did_prefix)
            self.enough_balance = True
            if not self.route_strategy_id:
                self.route_strategy_id = RouteStrategy.get_orig_routing_plan_id()
            return did_prefix
        else:
            raise Exception('Bad vendor configuration!')

    # self.save()

    @property
    def fmt_trunk_name(self):
        return self.alias

    @property
    def fmt_switch_ip(self):
        # ret = ','.join(
        #    ['{}:{}'.format(s.profile.sip_ip, s.profile.sip_port) for s in self.ingress_switch_profiles if s.profile])
        if True:  # ret == '':
            ret = ','.join(
                ['{}:{}'.format(s.sip_ip, s.sip_port) for s in
                 SwitchProfile.filter(SwitchProfile.status == 'connected')])
        return ret

    @property
    def fmt_allowed_ip(self):
        return self.fmt_ip_listing

    @property
    def fmt_ip_listing(self):
        ret = "<table class='ip_listing'><tr><th>IP Address</th><th>Port</th></tr>"
        if self.ip and len(list(self.ip)) > 0:
            for i in self.ip:
                ret += "<tr><td>{}</td><td>{}</td></tr>".format(i.ip, i.port)
            ret += "</table>"
        return ret

    @property
    def fmt_IP_listing(self):
        return self.fmt_ip_listing

    @property
    def fmt_route_info(self):
        ret = 'no route info'
        if self.prefixes and len(list(self.prefixes)) > 0:
            ret = "<table class='route_info'><tr><th>Tech Prefix</th><th>Rate Table</th></tr>"
            for i in self.prefixes:
                ret += "<tr><td>{}</td><td>{}</td></tr>".format(i.tech_prefix, i.rate_table_name)
            ret += "</table>"
        return ret

    @property
    def fmt_trunk_type(self):
        return self.direction

    @property
    def is_ip_changed(self):
        if self.fmt_previous_IP != self.fmt_new_IP:
            return True
        if self.fmt_previous_tech_prefix != self.fmt_new_tech_prefix:
            return True
        if self.fmt_previous_rate_table != self.fmt_new_rate_table:
            return True
        return False

    def store_old_detail(self):
        obj = self
        ip = ','.join(['{}:{}'.format(i.ip, i.port) for i in self.ip])
        tech_prefix = ','.join([i.tech_prefix for i in self.prefixes])
        rt = [i.rate_table_name or '' for i in self.prefixes]
        if self.rate_table:
            rt.append(self.rate_table.name)
        rate_table = ','.join(rt)
        obj.old_detail_table = dict(alias=obj.alias, direction=obj.direction, fmt_ip_listing=obj.fmt_ip_listing,
                                    fmt_route_info=obj.fmt_route_info, capacity=obj.capacity,
                                    cps_limit=obj.cps_limit, ip=ip, tech_prefix=tech_prefix, rate_table=rate_table)
        self.session().add(self)

    @property
    def fmt_previous_IP(self):
        if self.old_detail_table and 'ip' in self.old_detail_table:
            return self.old_detail_table.get('ip')
        return ''

    @property
    def fmt_new_IP(self):
        return ','.join(['{}:{}'.format(i.ip, i.port) for i in self.ip])

    @property
    def fmt_IP(self):
        return ','.join(['{}'.format(i.ip) for i in self.ip])

    @property
    def fmt_port(self):
        return ','.join(['{}'.format(i.ip, i.port) for i in self.ip])

    @property
    def fmt_previous_tech_prefix(self):
        if self.old_detail_table and 'tech_prefix' in self.old_detail_table:
            return self.old_detail_table.get('tech_prefix')
        return ''

    @property
    def fmt_routing_plan(self):
        return ','.join([i.routing_plan_name for i in self.prefixes])

    @property
    def fmt_new_tech_prefix(self):
        return ','.join([i.tech_prefix for i in self.prefixes])

    @property
    def fmt_previous_rate_table(self):
        if self.old_detail_table and 'rate_table' in self.old_detail_table:
            return self.old_detail_table.get('rate_table')
        return ''

    @property
    def fmt_new_rate_table(self):
        rt = [i.rate_table_name for i in self.prefixes]
        if self.rate_table:
            rt.append(self.rate_table.name)
        return ','.join(rt)

    @property
    def fmt_detail_table(self):
        class REC:
            pass

        old = REC()

        if hasattr(self, 'old_detail_table') and self.old_detail_table:
            old.__dict__ = dict(self.old_detail_table)
        else:
            old.__dict__ = dict(alias='', direction='', fmt_ip_listing='',
                                fmt_route_info='', capacity='',
                                cps_limit='')
        return """
        <table class="detail_table">
        <tr><td>Previous IP setting</td><td>{}</td></tr>
        <tr><td>Current IP setting</td><td>{}</td></tr>
        </table>
        """.format(old.fmt_ip_listing, self.fmt_ip_listing)

    @property
    def fmt_carrier_name(self):
        return self.client_name


# Resource.trunk_name = column_property(Resource.alias)
Client.ingress_count = column_property(select([func.count(Resource.resource_id)]). \
                                       where(and_(Resource.client_id == Client.client_id, Resource.ingress == True,
                                                  Resource.purged == False)). \
                                       correlate_except(Resource)
                                       )

Client.egress_count = column_property(select([func.count(Resource.resource_id)]). \
                                      where(and_(Resource.client_id == Client.client_id, Resource.egress == True,
                                                  Resource.purged == False)). \
                                      correlate_except(Resource)
                                      )
Client.vendor_res_id = column_property(select([Resource.resource_id]). \
                                       where(
    and_(Resource.client_id == Client.client_id, foreign(Resource.alias) == Client.name)).limit(1). \
                                       correlate_except(Resource)
                                       )

TrunkGroup.trunks_count = column_property(select([func.count(Resource.resource_id)]). \
                                          where(TrunkGroup.group_id == Resource.group_id). \
                                          correlate_except(Resource)
                                          )

EgressProfile.ingress_name = column_property(select([Resource.alias]). \
    where(Resource.resource_id == EgressProfile.ingress_id).correlate_except(
    Resource))
EgressProfile.egress_name = column_property(select([Resource.alias]). \
    where(Resource.resource_id == EgressProfile.egress_id).correlate_except(
    Resource))
FrundDetectionHistory.ingress_name = column_property(
    select([Resource.alias]).where(Resource.resource_id == FrundDetectionHistory.ingress_id).correlate_except(Resource))

DynamicRouteOverride.resource_name = column_property(select([Resource.alias]). \
                                                     where(
    Resource.resource_id == DynamicRouteOverride.resource_id).correlate_except(Resource))

DynamicRoutePri.trunk_name = column_property(select([Resource.alias]). \
                                             where(
    and_(Resource.resource_id == DynamicRoutePri.resource_id)).correlate_except(Resource))

AgentClient.has_ingress = column_property(
    select([Client.ingress_count > 0]).where(Client.client_id == AgentClient.client_id).correlate_except(Client))
AgentClient.has_egress = column_property(
    select([Client.egress_count > 0]).where(Client.client_id == AgentClient.client_id).correlate_except(Client))


class EgressTrunk(Resource):
    # carrier=relationship(Client,backref=backref("egress_trunks"))
    __mapper_args__ = {'polymorphic_identity': False}
    dynamic_routes = relationship('DynamicRouteItem',
                                  primaryjoin='EgressTrunk.resource_id==foreign(DynamicRouteItem.resource_id)',
                                  uselist=True, back_populates='trunk', single_parent=True, cascade="all, delete-orphan"
                                  )
    profiles = relationship(EgressProfile, uselist=True, back_populates='egress', single_parent=True,
                            cascade="all, delete-orphan")

    def _egress(self):
        return str(self.resource_id)

    @classmethod
    def get_object_by_primary_key(cls, **kwargs):
        obj = get_db(cls.db_label).get_object_by_primary_key(
            cls, **{k: v for k, v in kwargs.items() if k in cls.get_fields()}
        )
        if obj and not obj.egress:
            # return None
            raise NoResultFound('no result')
        if cls.use_history:
            obj.store_previous_data()
        return obj

    @classmethod
    def delete_query(cls):
        return Resource.query().filter(Resource.egress == True)


#    @classmethod
#    def query(cls):
##return get_db(cls.db_label).session.query(cls).filter(Resource.egress==True)


class IngressTrunk(Resource, IStatistic):
    # codes = relationship(code.CodeModel,backref='resource_id')
    # carrier=relationship(Client,backref=backref("ingress_trunks"))
    # ingress_trunk_name = synonym('alias')
    __mapper_args__ = {'polymorphic_identity': True}

    allowed_sendto_ips = relationship('AllowedSendToIp', uselist=True, back_populates='trunk')

    def _ingress(self):
        return str(self.resource_id)

    @classmethod
    def get_object_by_primary_key(cls, **kwargs):
        log.debug(f"KWARGS = {kwargs}")
        obj = get_db(cls.db_label).get_object_by_primary_key(
            cls, **{k: v for k, v in kwargs.items() if k in cls.get_fields()}
        )
        log.debug(f"OBJ = {obj}")
        if obj and not obj.ingress:
            # return None
            raise NoResultFound('no result')
        if cls.use_history:
            obj.store_previous_data()
        return obj

    @classmethod
    def delete_query(cls):
        return Resource.query().filter(Resource.ingress == True)
    
    # def before_save(self):
    #     super().before_save()
    #     if not self.allowed_sendto_ips:
    #         allow_send_to_ip = model.AllowedSendToIp(resource_id=self.resource_id)
    #         self.allowed_sendto_ips.append(allow_send_to_ip)


class RegisterOfRecord(DnlApiBaseModel):
    __tablename__ = 'register_of_record'
    STATE = {0: 'no', 1: 'yes'}
    id = Column(Integer, primary_key=True)  # , server_default=text_("nextval('register_of_record_id_seq'::regclass)"))
    time = Column(BigInteger)
    state = Column(ChoiceType(STATE), default=1)
    username = Column(String(100), index=True)
    direction = Column(Integer)
    ip = Column(Ip4)
    port = Column(Integer)
    expires_time = Column(BigInteger)


class ResourceIp(DnlApiBaseModel):
    __tablename__ = 'resource_ip'
    # REG_STATUS = {0: 'not registered', 1: 'success', 2: 'error'}
    ADDR_TYPE = {0: 'ip', 1: 'host'}
    REG_STATUS = {0: 'not registered', 1: 'registered', 2: 'register failed'}
    # __table_args__ = (UniqueConstraint('resource_id', 'ip','port'),)
    resource_ip_id = Column(Integer, primary_key=True)
    resource_id = Column(ForeignKey('resource.resource_id', ondelete='CASCADE'), nullable=False, index=True)
    ip = Column(String(50))
    port = Column(Integer, default=5060)
    fqdn = Column(String(100))  # domain name
    sip_rpid = Column(String(64))  # opensips
    disable_by_alert = Column(Boolean, default=False)
    priority = Column(Integer, default=0)
    last_priority = Column(Integer, default=0)
    addr_type = Column(ChoiceType(ADDR_TYPE), default=0)
    direction = Column(Integer)  # Mark this ip is incoming or outgoing: 0 - incoming; 1 - outgoing
    username = Column(String(50))
    password = Column(String(50))
    reg_type = Column(SmallInteger, nullable=False,
                      default=0)  # register user type, 0 for none; 1 for register user; 2 for register gateway account
    reg_status = Column(ChoiceType(REG_STATUS), nullable=False,
                        default=0)  # 0 for un-register; 1 for registered; 2 for register failed
    reg_srv_ip = Column(String(100))
    reg_srv_port = Column(Integer, nullable=False, default=5060)
    expires = Column(Integer, nullable=False, default=3600)
    profile_id = Column(Integer, nullable=False, default=0)
    need_register = Column(Boolean, default=False)  # 0:Authorized by IP Only;1:Authorized by SIP Registration
    options_ping_inv = Column(Integer, nullable=True)

    trunk_type2 = column_property(
        select([Resource.trunk_type2]).where(Resource.resource_id == resource_id).correlate_except(Resource))
    ingress = column_property(
        select([Resource.ingress]).where(Resource.resource_id == resource_id).correlate_except(Resource))
    client_id = column_property(
        select([Resource.client_id]).where(Resource.resource_id == resource_id).correlate_except(Resource))
    trunk_name = column_property(
        select([Resource.alias]).where(Resource.resource_id == resource_id).correlate_except(Resource))

    resource = relationship(Resource,
                            primaryjoin='and_(ResourceIp.resource_id == Resource.resource_id,ResourceIp.reg_type==0)',
                            uselist=False, back_populates='ip', enable_typechecks=False)
    limits = relationship('ResourceIpLimit', back_populates='ip', uselist=True, single_parent=True,
                          cascade="all, delete-orphan")
    id = synonym('resource_ip_id')

    # registered = column_property(case([(select([RegisterOfRecord.state]). \
    #                                     where(RegisterOfRecord.username == username).correlate_except(
    #     RegisterOfRecord).as_scalar() == 1, 'yes')], else_='no'))
    registered = column_property(case([(reg_status == 1, 'yes')], else_='no'))
    
    __table_args__ = (
        Index(
            "resource_ip_username_key",
            "username",
            unique=True,
            postgresql_where=username.isnot(None)
        ),
    )

    @validates('ip')
    def validate_ip(self, key, value):
        if value == None:
            return None
        ip_regexp = r"^(([0-9]|[1-9][0-9]|1[0-9]{2}|2[0-4][0-9]|25[0-5])\.){3}(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9][0-9]|[0-9])|localhost$"

        if re.search(ip_regexp, value) and re.search(ip_regexp, value).group() == value:
            return re.search(ip_regexp, value).group()
        else:
            raise Exception('ip is invalid')

    def before_save(self):
        if self.resource_id:
            res = Resource.get(self.resource_id)
            if res and res.client.client_type != 'vendor':
                prefs = list(set([p.tech_prefix for p in
                                  ResourcePrefix.filter(ResourcePrefix.resource_id == self.resource_id).all()]))
                if res.ingress:
                    sq = ResourceIp.filter(and_(ResourceIp.ip == self.ip, ResourceIp.port == self.port)). \
                        filter(ResourceIp.resource_ip_id != self.resource_ip_id). \
                        outerjoin(Resource, Resource.resource_id == ResourceIp.resource_id). \
                        filter(Resource.ingress == True)
                    if prefs:
                        sq = sq.outerjoin(ResourcePrefix, Resource.resource_id == ResourcePrefix.resource_id). \
                            filter(ResourcePrefix.tech_prefix.in_(prefs))
                    else:
                        sq = sq.outerjoin(ResourcePrefix, Resource.resource_id == ResourcePrefix.resource_id). \
                            filter(ResourcePrefix.tech_prefix.is_(None))
                    ids = [res_ip.resource_ip_id for res_ip in sq.all()]
                    old = ResourceIp.filter(ResourceIp.resource_ip_id.in_(ids)).delete(synchronize_session='fetch')
                    # sq.delete(synchronize_session='fetch')
                    # if q:
                    #     raise ValidationError(
                    #         'Ip {} port {} is duplicate on ingress trunks with same prefix! (duplicate resource_ip_id {})'.format(
                    #             self.ip, self.port, q.resource_ip_id))

    def after_save(self):
        pass


Resource.ip_count = column_property(select([func.count(ResourceIp.resource_ip_id)]). \
                                    where(
    and_(Resource.resource_id == ResourceIp.resource_id, ResourceIp.reg_type == 0)). \
                                    correlate_except(ResourceIp)
                                    )


class ResourceIpLimit(DnlApiBaseModel):
    __tablename__ = 'resource_ip_limit'

    limit_id = Column(Integer, primary_key=True,
                      server_default=text_("nextval('resource_ip_limit_limit_id_seq'::regclass)"))
    ip_id = Column(ForeignKey('resource_ip.resource_ip_id', ondelete='CASCADE'), nullable=False, index=True)
    cps = Column(Integer)
    capacity = Column(Integer)
    time_profile_id = Column(ForeignKey('time_profile.time_profile_id', ondelete='SET NULL'), index=True)

    ip = relationship('ResourceIp', uselist=False, back_populates='limits')
    time_profile = relationship('TimeProfile')
    resource_ip_id = synonym('ip_id')

    resource_id = column_property(
        select([ResourceIp.resource_id]).where(ResourceIp.resource_ip_id == ip_id).correlate_except(ResourceIp))

    time_profile_name = column_property(
        select([TimeProfile.name]).where(TimeProfile.time_profile_id == time_profile_id).correlate_except(TimeProfile))
    resource_direction = column_property(
        select([ResourceIp.direction]).where(ResourceIp.resource_ip_id == ip_id).correlate_except(ResourceIp))
    resource_ip = column_property(
        select([ResourceIp.ip]).where(ResourceIp.resource_ip_id == ip_id).correlate_except(ResourceIp))
    resource_port = column_property(
        select([ResourceIp.port]).where(ResourceIp.resource_ip_id == ip_id).correlate_except(ResourceIp))
    resource_active = column_property(
        select([Resource.active]).where(
            and_(ResourceIp.resource_ip_id == ip_id, ResourceIp.resource_id == Resource.resource_id)).correlate_except(
            ResourceIp, Resource))
    resource_alias = column_property(
        select([Resource.alias]).where(
            and_(ResourceIp.resource_ip_id == ip_id, ResourceIp.resource_id == Resource.resource_id)).correlate_except(
            ResourceIp, Resource))
    resource_cps_limit = column_property(
        select([Resource.cps_limit]).where(
            and_(ResourceIp.resource_ip_id == ip_id, ResourceIp.resource_id == Resource.resource_id)).correlate_except(
            ResourceIp, Resource))
    resource_capacity = column_property(
        select([Resource.capacity]).where(
            and_(ResourceIp.resource_ip_id == ip_id, ResourceIp.resource_id == Resource.resource_id)).correlate_except(
            ResourceIp, Resource))
    client_name = column_property(
        select([Client.name]).where(
            and_(ResourceIp.resource_ip_id == ip_id, ResourceIp.resource_id == Resource.resource_id,
                 Resource.client_id == Client.client_id)).correlate_except(
            ResourceIp, Resource, Client))
    client_call_limit = column_property(
        select([Client.call_limit]).where(
            and_(ResourceIp.resource_ip_id == ip_id, ResourceIp.resource_id == Resource.resource_id,
                 Resource.client_id == Client.client_id)).correlate_except(
            ResourceIp, Resource, Client))
    client_cps_limit = column_property(
        select([Client.cps_limit]).where(
            and_(ResourceIp.resource_ip_id == ip_id, ResourceIp.resource_id == Resource.resource_id,
                 Resource.client_id == Client.client_id)).correlate_except(
            ResourceIp, Resource, Client))


class ResourceIpLimitQuery(DnlApiBaseModel):
    __tablename__ = '#resource_ip_limit_query'

    # __tablename__ = alias(resource_ip_limit_query, '#resource_ip_limit_query')  # '#resource_ip_limit_query'
    # alias(Code.query().session.query(Code.name.label('destination'),
    #                func.count().label('codes_count')).group_by(Code.name)) #'#code_destination'

    @classmethod
    def query(cls):
        c = Client
        r = Resource
        i = ResourceIp
        l = ResourceIpLimit
        q = get_db(r.db_label).session.query(r.resource_id, r.alias.label('resource_alias'),
                                             c.client_id.label('client_id'),
                                             c.name.label('client_name'), r.active.label('resource_active'),
                                             r.cps_limit.label('resource_cps_limit'),
                                             r.call_limit.label('resource_call_limit'),
                                             r.capacity.label('resource_capacity'),
                                             c.cps_limit.label('client_cps_limit'),
                                             c.call_limit.label('client_call_limit'),
                                             c.client_type.label('client_type'),
                                             i.resource_ip_id, i.ip.label('resource_ip'), i.port.label('resource_port'),
                                             l.cps, l.capacity). \
            outerjoin(c).outerjoin(i, and_(r.resource_id == i.resource_id, i.reg_type == 0)).outerjoin(l).filter(
            )#r.ingress == True)
        # q1 = alias(q,'#resource_ip_limit_query')
        # resource_ip_limit_cte = q.cte("resource_ip_limit")
        # final_query = get_db(r.db_label).session.query(resource_ip_limit_cte)

        return q

    __table_args__ = (PrimaryKeyConstraint('resource_id', 'resource_ip_id', 'limit_id'), {'prefixes': ['TEMPORARY']})

    resource_id = Column(Integer)
    resource_alias = Column(String)
    client_id = Column(Integer)
    client_name = Column(String)
    client_type = Column(ChoiceType(Client.CLIENT_TYPE))
    company_type = Column(ChoiceType(Client.COMPANY_TYPE))
    resource_active = Column(Boolean)
    resource_cps_limit = Column(Integer)
    resource_call_limit = Column(Integer)
    resource_capacity = Column(Integer)
    client_cps_limit = Column(Integer)
    client_call_limit = Column(Integer)
    resource_ip_id = Column(Integer)
    ip = Column(String)
    limit_id = Column(Integer)
    port = Column(Integer)
    cps = Column(Integer)
    capacity = Column(Integer)


class ResourceNextRouteRule(DnlApiBaseModel):
    __tablename__ = 'resource_next_route_rule'
    __table_args__ = (
        UniqueConstraint('reponse_code', 'resource_id'),
    )
    FAILOVER_STRATEGY = {1: 'Fail to Next Host', 2: 'Fail to Next Trunk', 3: 'Stop'}

    id = Column(Integer, primary_key=True, server_default=text_("nextval('resource_next_route_rule_id_seq'::regclass)"))
    route_type = Column(ChoiceType(FAILOVER_STRATEGY))
    reponse_code = Column(String(100))
    resource_id = Column(Integer)
    return_code = Column(String(100))
    return_string = Column(String(100))

    failover_method = synonym('route_type')
    failover_strategy = synonym('route_type')
    match_code = synonym('reponse_code')
    from_sip_code = synonym('reponse_code')
    return_clause = synonym('return_string')
    to_sip_string = synonym('return_string')
    to_sip_code = synonym('return_code')


class Codec(DnlApiBaseModel):
    __tablename__ = 'codecs'

    id = Column(Integer, primary_key=True)
    name = Column(String(40), nullable=False, unique=True)
    detail = Column(String(100))

    trunks = relationship('ResourceCodecsRef', uselist=True, back_populates='codec')

    def __repr__(self):
        return ' '.join(filter(None, [str(self.id), self.name, self.detail]))


class ResourceCodecsRef(DnlApiBaseModel):
    __tablename__ = 'resource_codecs_ref'

    id = Column(Integer, primary_key=True, server_default=text_("nextval('resource_codecs_ref_id_seq'::regclass)"))
    resource_id = Column(ForeignKey('resource.resource_id', ondelete='CASCADE'), nullable=False, index=True)
    codec_id = Column(Integer, ForeignKey('codecs.id'), index=True)

    resource = relationship(Resource, uselist=False, back_populates='codec_list')
    codec = relationship(Codec, uselist=False, back_populates='trunks')


# resource.ResourceModel.codec = relationship('resource_codecs_ref.ResourceCodecsRefModel',backref='ResourceModel.resource_id')

class ResourceDirection(DnlApiBaseModel):
    __tablename__ = 'resource_direction'
    DIRECTION = {0: 'ALL', 1: 'ingress', 2: 'egress'}
    ACTION = {1: 'plus prefix', 2: 'plus suffix', 3: 'minus prefix', 4: 'minus suffix'}
    TYPE = {0: 'modify the caller', 1: 'modify the called'}
    NUMBER_TYPE = {0: 'all', 1: 'greater than', 2: 'equal to', 3: 'less than'}

    direction_id = Column(Integer, primary_key=True,
                          server_default=text_("nextval('resource_direction_direction_id_seq'::regclass)"))
    resource_id = Column(ForeignKey('resource.resource_id', ondelete='CASCADE'), nullable=False, index=True)
    time_profile_id = Column(ForeignKey('time_profile.time_profile_id', ondelete='SET NULL'), index=True)
    action = Column(ChoiceType(ACTION), nullable=False)
    direction = Column(ChoiceType(DIRECTION), nullable=False)
    digits = Column(String(16), nullable=False)
    dnis = Column(PrefixRange, server_default=text_("''::prefix_range"))
    number_length = Column(Integer)
    number_type = Column(ChoiceType(NUMBER_TYPE))
    type = Column(ChoiceType(TYPE), nullable=False)

    resource = relationship('Resource', uselist=False, back_populates='directions')
    time_profile = relationship('TimeProfile')

    tech_prefix = synonym('digits')

    alias = column_property(select([Resource.alias]). \
                            where(Resource.resource_id == resource_id).correlate_except(Resource))


class ResourcePrefix(DnlApiBaseModel):
    __tablename__ = 'resource_prefix'
    # __table_args__ = (UniqueConstraint('resource_id', 'code'),)
    __table_args__ = (UniqueConstraint('resource_id', 'tech_prefix', 'code'),)
    id = Column(Integer, primary_key=True)
    resource_id = Column(ForeignKey('resource.resource_id', ondelete='CASCADE'), index=True)
    tech_prefix = Column(PrefixRange, nullable=False, default='', index=True)  # Префикс номера соответствия
    route_strategy_id = Column(Integer, index=True)
    rate_table_id = Column(Integer, index=True)
    code = Column(PrefixRange)  # Called number prefix, with tech_prefix not the same, do not need to delete the prefix,
    # select routing_palan and rate_table role. The same ingress + tech_prefix code must be unique.
    code_cps = Column(Integer)
    code_cap = Column(Integer)
    product_id = Column(Integer, index=True)
    resource = relationship(Resource, uselist=False, back_populates='prefixes', enable_typechecks=False)
    rate_table = relationship('RateTable',
                              primaryjoin='foreign(ResourcePrefix.rate_table_id) == RateTable.rate_table_id')
    route_strategy = relationship('RouteStrategy',
                                  primaryjoin='foreign(ResourcePrefix.route_strategy_id) == RouteStrategy.route_strategy_id')
    routing_plan_id = synonym('route_strategy_id')
    trunk_id = synonym('resource_id')
    prefix = synonym('tech_prefix')
    # prefix = column_property(tech_prefix)

    # ips = relationship(ResourceIp,uselist=True,primaryjoin=foreign(resource_id)==ResourceIp.resource_id)
    product = relationship('ProductRoutRateTable',
                           primaryjoin='foreign(ResourcePrefix.product_id) == ProductRoutRateTable.id',
                           uselist=False, back_populates='trunks')

    product_name = column_property(select([ProductRoutRateTable.name]). \
                                   where(ProductRoutRateTable.id == product_id).correlate_except(ProductRoutRateTable))
    name = column_property(select([ProductRoutRateTable.name]). \
                           where(ProductRoutRateTable.id == product_id).correlate_except(ProductRoutRateTable))
    rate_table_name = column_property(select([RateTable.name]). \
                                      where(RateTable.rate_table_id == rate_table_id).correlate_except(RateTable))
    code_deck_name = column_property(select([RateTable.code_deck_name]). \
                                     where(RateTable.rate_table_id == rate_table_id).correlate_except(RateTable))
    trunk_name = column_property(select([Resource.alias]). \
                                 where(Resource.resource_id == resource_id).correlate_except(Resource))
    is_active = column_property(select([Resource.active]). \
                                 where(Resource.resource_id == resource_id).correlate_except(Resource))
    routing_plan_name = column_property(select([RouteStrategy.name]). \
        where(RouteStrategy.route_strategy_id == route_strategy_id).correlate_except(
        RouteStrategy))
    carrier_name = column_property(select([Client.name]). \
                                   where(
        and_(Resource.resource_id == resource_id, Client.client_id == Resource.client_id)).correlate_except(Resource,
                                                                                                            Client))

    rate_email = column_property(select([Client.rate_email]). \
                                 where(
        and_(Resource.resource_id == resource_id, Client.client_id == Resource.client_id)).correlate_except(Resource,
                                                                                                            Client))
    alias = column_property(select([Resource.alias]). \
                            where(Resource.resource_id == resource_id).correlate_except(Resource))

    carrier_id = column_property(select([Resource.client_id]). \
                                 where(Resource.resource_id == resource_id).correlate_except(Resource))
    client_id = column_property(select([Resource.client_id]). \
                                where(Resource.resource_id == resource_id).correlate_except(Resource))

    ingress = column_property(select([Resource.ingress]). \
                              where(Resource.resource_id == resource_id).correlate_except(Resource))

    trunk_type2 = column_property(select([Resource.trunk_type2]). \
                              where(Resource.resource_id == resource_id).correlate_except(Resource))

    @hybrid_property
    def ips(self):
        return self.resource.ip

    def before_save(self):
        old = ResourcePrefix.filter(and_(ResourcePrefix.code == self.code, ResourcePrefix.code.isnot(None), ResourcePrefix.rate_table_id == self.rate_table_id, ResourcePrefix.id != self.id,
                                         ResourcePrefix.ingress, ResourcePrefix.trunk_type2 == 'DID Traffic'))
        old.delete(synchronize_session='fetch')
        if self.product_id and self.rate_table_id is None and self.route_strategy_id is None:
            prod = ProductRoutRateTable.get(self.product_id)
            if prod:
                self.rate_table_id = prod.rate_table_id
                self.route_strategy_id = prod.route_plan_id

class ProductCodeName(DnlApiBaseModel):
    __tablename__ = 'product_codename'

    __table_args__ = (
        UniqueConstraint('code_name', 'product_id', name='product_codename_code_name_product_id_key'),
    )

    item_id = Column(Integer, primary_key=True)
    product_id = Column(Integer)
    code_name = Column(String(255), nullable=False)

    @hybrid_property
    def resources(self):
        code_resources = ProductCodeNameResource.filter(ProductCodeNameResource.item_id == self.item_id).all()
        return [resource.resource_id for resource in code_resources]

    @resources.setter
    def resources(self, value):
        current_resources = self.resources
        if not isinstance(value, list):
            value = [value]
        for res_id in current_resources:
            if res_id not in value:
                ProductCodeNameResource.filter(ProductCodeNameResource.resource_id == res_id).delete()

        for res_id in value:
            if res_id not in current_resources:
                ProductCodeNameResource(item_id=self.item_id, resource_id=res_id).save()

class ProductCodeNameResource(DnlApiBaseModel):
    __tablename__ = 'product_codename_resource'

    item_id = Column(Integer, primary_key=True)
    resource_id = Column(Integer)

ResPref = ResourcePrefix.__table__.alias('res_pref')

Resource.routing_plan_id = column_property(
    select([func.string_agg(cast(ResourcePrefix.route_strategy_id, String), ',')])
        .where(ResourcePrefix.resource_id == Resource.resource_id).group_by(ResourcePrefix.resource_id).limit(
        1).correlate_except(ResourcePrefix))
RouteStrategy.usage_count = column_property(
    select([func.count(distinct(Resource.resource_id))]).where(
        and_(ResourcePrefix.route_strategy_id == RouteStrategy.route_strategy_id,
             Resource.resource_id == ResourcePrefix.resource_id, Resource.ingress,
             Resource.purged == False)).limit(1).correlate_except(
        ResourcePrefix, Resource))
RouteStrategy.term_usage_count = column_property(
    select([func.count(distinct(Resource.resource_id))]).where(
        and_(ResourcePrefix.route_strategy_id == RouteStrategy.route_strategy_id,
             Resource.resource_id == ResourcePrefix.resource_id, Resource.ingress,
             Resource.trunk_type2 == 'Termination Traffic')).limit(1).correlate_except(
        ResourcePrefix, Resource))
RouteStrategy.orig_usage_count = column_property(
    select([func.count(distinct(Resource.resource_id))]).where(
        and_(ResourcePrefix.route_strategy_id == RouteStrategy.route_strategy_id,
             Resource.resource_id == ResourcePrefix.resource_id, Resource.ingress,
             Resource.trunk_type2 == 'DID Traffic')).limit(1).correlate_except(
        ResourcePrefix, Resource))
Resource.first_prefix = column_property(select([ResourcePrefix.tech_prefix])
                                        .where(
    and_(ResourcePrefix.resource_id == Resource.resource_id, ResourcePrefix.resource_id.isnot(None))).order_by(
    ResourcePrefix.id).limit(1).correlate_except(ResourcePrefix))

RateTable.ingress_count = column_property(
    # select([func.count(Resource.resource_id.distinct()).label('ingress_count')]).where(
    #     and_(Resource.rate_table_id == RateTable.rate_table_id,Resource.ingress==True)).correlate_except(Resource)

    select([func.count(Resource.resource_id.distinct()).label('ingress_count')]).where(
        and_(or_(Resource.rate_table_id == RateTable.rate_table_id, ResourcePrefix.rate_table_id == RateTable.rate_table_id),
             Resource.resource_id == ResourcePrefix.resource_id, Resource.ingress == True,
             Resource.purged == False, Resource.client_id.isnot(None))).correlate_except(ResourcePrefix, Resource)

    # select([func.count(Resource.resource_id.distinct()).label('ingress_count')]).where(
    # and_(ResourcePrefix.rate_table_id == RateTable.rate_table_id, Resource.resource_id == ResourcePrefix.resource_id,
    # Resource.ingress==True,Resource.client_id.isnot(None))). \
    # correlate_except(ResourcePrefix, Resource)
)
RateTable.egress_count = column_property(
    select([func.count(Resource.resource_id.distinct()).label('egress_count')]).where(
        and_(Resource.rate_table_id == RateTable.rate_table_id, Resource.egress == True,
             Resource.purged == False)).correlate_except(Resource)
    # select([func.count(Resource.resource_id).label('egress_count')]). where(
    # and_(Resource.rate_table_id == RateTable.rate_table_id, Resource.egress == True,
    #      Resource.client_id.isnot(None), Resource.client_type.is_(None))). \
    #                                      group_by(Resource.rate_table_id).correlate_except(Resource)
)

RateTable.orig_count = column_property(
    select([func.count(Resource.resource_id.distinct()).label('orig_count')]).where(
        and_(Resource.rate_table_id == RateTable.rate_table_id,
             Resource.trunk_type2 == 'DID Traffic')).correlate_except(Resource)
    # select([func.count(Resource.resource_id).label('egress_count')]). where(
    # and_(Resource.rate_table_id == RateTable.rate_table_id, Resource.egress == True,
    #      Resource.client_id.isnot(None), Resource.client_type.is_(None))). \
    #                                      group_by(Resource.rate_table_id).correlate_except(Resource)
)

CodeDeck.usage_count = column_property(select([func.count(RateTable.rate_table_id).label('usage_count')]).where(and_(
    RateTable.code_deck_id == CodeDeck.code_deck_id, RateTable.orig_count == 0)). \
                                       group_by(RateTable.code_deck_id).correlate_except(RateTable))

DynamicRoute.route_plan_id = column_property(
    select([func.string_agg(cast(ResourcePrefix.route_strategy_id, String), ',').label('usage_count')]). \
        where(and_(DynamicRoute.dynamic_route_id == DynamicRouteItem.dynamic_route_id,
                   DynamicRouteItem.resource_id == ResourcePrefix.resource_id)). \
        correlate_except(DynamicRouteItem, ResourcePrefix))
ProductRoutRateTable.usage_count = column_property(
    select([func.count(ResourcePrefix.resource_id).label('usage_count')]). \
        where(ResourcePrefix.product_id == ProductRoutRateTable.id). \
        group_by(ResourcePrefix.product_id).correlate_except(ResourcePrefix))
ProductRoutRateTable.route_plan_name = column_property(
    select([RouteStrategy.name]).where(
        RouteStrategy.route_strategy_id == ProductRoutRateTable.route_plan_id).correlate_except(RouteStrategy))
ProductRoutRateTable.rate_table_name = column_property(
    select([RateTable.name]).where(RateTable.rate_table_id == ProductRoutRateTable.rate_table_id).correlate_except(
        RateTable))
ProductRoutRateTable.rate_table_type = column_property(
    select([RateTable.jur_type]).where(RateTable.rate_table_id == ProductRoutRateTable.rate_table_id).correlate_except(
        RateTable))
ProductAgentsRef.usage_count = column_property(select([ProductRoutRateTable.usage_count]).where(
    ProductRoutRateTable.id == ProductAgentsRef.product_id).correlate_except(ProductRoutRateTable))
ProductAgentsRef.rate_table_name = column_property(
    select([ProductRoutRateTable.rate_table_name]).where(
        ProductRoutRateTable.id == ProductAgentsRef.product_id).correlate_except(
        ProductRoutRateTable))
ProductAgentsRef.rate_table_type = column_property(
    select([ProductRoutRateTable.rate_table_type]).where(
        ProductRoutRateTable.id == ProductAgentsRef.product_id).correlate_except(
        ProductRoutRateTable))


# Resource.prefix = relationship(ResourcePrefix,backref='resource.resource_id')


class ResourceReplaceAction(DnlApiBaseModel):
    __tablename__ = 'resource_replace_action'
    TYPE = {0: 'ani only replace the caller', 1: 'dnis only replace the called',
            2: 'both replace the calling and called'}
    id = Column(Integer, primary_key=True, server_default=text_("nextval('resource_replace_action_id_seq'::regclass)"))
    resource_id = Column(ForeignKey('resource.resource_id', ondelete='CASCADE'), index=True)
    ani_prefix = Column(String(50))
    ani = Column(String(30))
    ani_min_length = Column(Integer)
    ani_max_length = Column(Integer)
    type = Column(ChoiceType(TYPE))
    dnis_prefix = Column(String(30))
    dnis = Column(String(30))
    dnis_min_length = Column(Integer)
    dnis_max_length = Column(Integer)

    resource = relationship('Resource', uselist=False, back_populates='replace_actions')


# +++ RateSendLog
class RateSendLog(DnlApiBaseModel):
    __tablename__ = 'rate_send_log'
    STATUS = {
        0: 'waiting',
        2: 'generated',
        3: 'sent',
        4: 'sent failed',
        5: 'sent success'
    }
    FORMAT = {0: 'csv', 1: 'json', 2: 'xls'}
    SEND_TYPE = {0: 'Send to carriers using this rate table', 1: 'Specify my own recipients',
                 2: 'Send to specified trunks', 3: 'Send to specified carriers'}
    DOWNLOAD_METHOD = {0: 'link', 1: 'attachment'}
    ZIP = {0: 'false', 1: 'true'}
    SENT_AREA = {0: 'product', 1: 'rate'}
    DATE_FORMAT = {0: 'Date Only', 1: 'Date and Time'}

    rate_table_id = Column(Integer, nullable=False)
    status = Column(ChoiceType(STATUS), nullable=False)
    zip = Column(SmallInteger)  # Column(ChoiceType(ZIP))
    file = Column(String(500))
    error = Column(String(500))
    format = Column(ChoiceType(FORMAT), nullable=False)
    id = Column(Integer, primary_key=True, server_default=text_("nextval('rate_send_log_id_seq'::regclass)"))
    email_template_id = Column(Integer)
    create_time = Column(DateTime(True))
    effective_date = Column(String(50))
    start_effective_date = Column(String(50))
    download_deadline = Column(Date)
    is_email_alert = Column(Boolean, nullable=False, server_default=text_("false"))
    is_disable = Column(Boolean, nullable=False, server_default=text_("false"))
    is_temp = Column(Boolean, server_default=text_("false"))
    headers = Column(String(500))
    send_type = Column(ChoiceType(SEND_TYPE), default=0, server_default=text_("0"))
    send_specify_email = Column(String)
    resource_ids = Column(String(500))
    download_method = Column(ChoiceType(DOWNLOAD_METHOD), default=1, server_default=text_("1"))
    total_records = Column(Integer)
    completed_records = Column(Integer)

    sent_area = Column(ChoiceType(SENT_AREA), default=1, server_default=text_("1"))
    date_format = Column(ChoiceType(DATE_FORMAT), default=0, server_default=text_("0"))

    indicate_inc_dec = Column(Boolean, server_default=text_("false"))

    rate_table = relationship(RateTable, primaryjoin='foreign(RateSendLog.rate_table_id)==RateTable.rate_table_id')
    email_template = relationship(SendRateTemplate,
                                  primaryjoin='foreign(RateSendLog.email_template_id)==SendRateTemplate.id',
                                  uselist=False)
    email_direct = relationship(SendRateDirect, uselist=False, single_parent=True, cascade="all, delete-orphan")
    details = relationship('RateSendLogDetail', back_populates='parent_job', uselist=True,
                           single_parent=True, cascade="all, delete-orphan")

    job_id = synonym('id')

    process = column_property(case([(total_records > 0, (1.0 * completed_records) / (1.0 * total_records))], else_=0.0))

    rate_table_name = column_property(select([RateTable.name]). \
                                      where(RateTable.rate_table_id == rate_table_id).correlate_except(RateTable))

    @property
    def download_token(self):
        try:
            return self.details[0].download_token
        except:
            return None

    @hybrid_property
    def trunks(self):
        try:
            return self.resource_ids.split(',')
        except:
            return []

    @trunks.setter
    def trunks(self, value):
        try:
            self.resource_ids = ','.join([str(x) for x in value])
        except:
            pass

    @hybrid_property
    def carriers(self):
        try:
            carriers = []
            for i in self.trunks:
                resources = Resource.filter(Resource.resource_id.in_(self.trunks)).all()
                client_ids = [i.client_id for i in resources]
            return list(set(client_ids))
        except:
            return []

    @carriers.setter
    def carriers(self, value):
        try:
            resource_ids = []
            for i in value:
                resources = Resource.filter(Resource.client_id == i).all()
                resource_ids += [i.resource_id for i in resources]
            self.trunks = resource_ids
        except:
            pass

    @hybrid_property
    def err(self):
        return self.error

    @err.setter
    def err(self, value):
        try:
            if len(value) < 500:
                self.error = value
            else:
                self.error = value[0:500]
        except:
            self.error = value


class RateSendLogDetail(DnlApiBaseModel):
    __tablename__ = 'rate_send_log_detail'
    STATUS = {
        None: 'Waiting',
        1: 'In Progress',
        2: 'completed',
        3: 'failed',
        4: 'Downloaded',
        5: 'Not Yet Downloaded'
    }
    id = Column(Integer, primary_key=True, server_default=text_("nextval('rate_send_log_detail_id_seq'::regclass)"))
    log_id = Column(ForeignKey('rate_send_log.id', ondelete='CASCADE'))
    resource_id = Column(Integer)
    send_to = Column(String(100))
    status = Column(ChoiceType(STATUS))
    error = Column(Text)
    salt = Column(String)
    download_date = Column(Date)
    sent_on = Column(DateTime(True))

    parent_job = relationship('RateSendLog', back_populates='details', uselist=False)
    resource = relationship('Resource', primaryjoin='foreign(RateSendLogDetail.resource_id)==Resource.resource_id')
    download_log = relationship('RateDownloadLog',
                                primaryjoin='foreign(RateSendLogDetail.id) == RateDownloadLog.log_detail_id',
                                back_populates='send_log', uselist=True)

    trunk_id = synonym('resource_id')
    job_id = synonym('log_id')

    result = column_property(
        case([(download_date > datetime.now(UTC), 'expired'), (status == 'failed', 'canceled')], else_='normal'))

    process = column_property(select([RateSendLog.process]). \
                              where(RateSendLog.id == log_id).correlate_except(RateSendLog))

    create_time = column_property(select([RateSendLog.create_time]). \
                                  where(RateSendLog.id == log_id).correlate_except(RateSendLog))

    download_deadline = column_property(select([RateSendLog.download_deadline]). \
                                        where(RateSendLog.id == log_id).correlate_except(RateSendLog))

    effective_date = column_property(select([RateSendLog.effective_date]). \
                                     where(RateSendLog.id == log_id).correlate_except(RateSendLog))

    file = column_property(select([RateSendLog.file]). \
                           where(RateSendLog.id == log_id).correlate_except(RateSendLog))

    send_type = column_property(select([RateSendLog.send_type]). \
                                where(RateSendLog.id == log_id).correlate_except(RateSendLog))

    sent_area = column_property(select([RateSendLog.sent_area]). \
                                where(RateSendLog.id == log_id).correlate_except(RateSendLog))

    file = column_property(select([RateSendLog.file]). \
                           where(RateSendLog.id == log_id).correlate_except(RateSendLog))

    rate_table_name = column_property(select([RateSendLog.rate_table_name]). \
                                      where(RateSendLog.id == log_id).correlate_except(RateSendLog))

    rate_table_id = column_property(select([RateSendLog.rate_table_id]). \
                                    where(RateSendLog.id == log_id).correlate_except(RateSendLog))

    trunk_name = column_property(select([Resource.alias]). \
                                 where(Resource.resource_id == resource_id).correlate_except(Resource))
    client_name = column_property(select([Client.name]). \
                                  where(
        and_(Resource.resource_id == resource_id, Client.client_id == Resource.client_id)).correlate_except(Resource,
                                                                                                            Client))

    client_id = column_property(select([Client.client_id]). \
                                where(
        and_(Resource.resource_id == resource_id, Client.client_id == Resource.client_id)).correlate_except(Resource,
                                                                                                            Client))
    rate_download_deadline = synonym(download_deadline)
    #name = synonym(client_name)

    @hybrid_property
    def download_token(self):
        import jwt
        exp = datetime.now(UTC) + timedelta(days=settings.JWT_TTL_DAYS)
        token_data = {'id': self.id, 'exp': exp}
        token = jwt.encode(token_data, settings.JWT_SIGNATURE).decode('utf-8')
        return token

    @hybrid_property
    def download_link(self):
        token = self.download_token
        config = SystemParameter.get(1)
        # return '{}/tool/send_rate/rate_download/{}'.format(settings.API_URL,token)
        return '{}/#/download/rate/{}'.format(config.root_url, token)

    @property
    def fmt_rate_update_file_name(self):
        return self.file


ProductClientsRefUsed.rate_last_sent = column_property(select([func.max(RateSendLogDetail.sent_on)]).where(
    and_(  # RateSendLogDetail.client_id==ProductClientsRefUsed.client_id,
        RateSendLogDetail.rate_table_id == ProductClientsRefUsed.rate_table_id
    )))

Resource.last_rate_sent_time = column_property(
    select([RateSendLogDetail.sent_on]).where(Resource.resource_id == RateSendLogDetail.resource_id).order_by(
        RateSendLogDetail.sent_on.desc()).limit(1).correlate_except(RateSendLogDetail))

RateSendLog.email_count = column_property(
    select([func.sum(func.array_length(func.regexp_split_to_array(RateSendLogDetail.send_to, ';'), 1))]). \
        where(RateSendLog.id == RateSendLogDetail.log_id).correlate_except(RateSendLogDetail))

RateSendLog.recipient_count = column_property(select([func.count(RateSendLogDetail.id)]). \
    where(RateSendLog.id == RateSendLogDetail.log_id).correlate_except(
    RateSendLogDetail))

# RateSendLogDetail.email_count = column_property(select([RateSendLog.email_count]). \
#     where(RateSendLog.id == RateSendLogDetail.log_id).correlate_except(
#     RateSendLog))
# RateSendLogDetail.recipient_count = column_property(select([RateSendLog.recipient_count]). \
#     where(RateSendLog.id == RateSendLogDetail.log_id).correlate_except(
#     RateSendLog))
RateSendLogDetail.email_count = column_property(
    func.array_length(func.regexp_split_to_array(RateSendLogDetail.send_to, ';'), 1))

RateSendLogDetail.recipient_count = column_property(select([func.count(RateSendLog.id)]). \
    where(RateSendLog.id == RateSendLogDetail.log_id).correlate_except(
    RateSendLog))

RateSendLogDetail.process = column_property(select([RateSendLog.process]). \
    where(RateSendLog.id == RateSendLogDetail.log_id).correlate_except(
    RateSendLog))

RateDownloadLog.trunk_name = column_property(select([Resource.alias]). \
                                             where(
    Resource.resource_id == RateDownloadLog.resource_id).correlate_except(Resource))
RateDownloadLog.client_name = column_property(select([Client.name]). \
    where(
    and_(Resource.resource_id == RateDownloadLog.resource_id, Client.client_id == Resource.client_id)).correlate_except(
    Resource, Client))

RateDownloadLog.client_id = column_property(select([Client.client_id]). \
    where(
    and_(Resource.resource_id == RateDownloadLog.resource_id, Client.client_id == Resource.client_id)).correlate_except(
    Resource,
    Client))
RateDownloadLog.sent_on = column_property(select([RateSendLogDetail.sent_on]). \
    where(RateSendLogDetail.id == RateDownloadLog.log_detail_id).correlate_except(
    RateSendLogDetail))

RateDownloadLog.send_to = column_property(select([RateSendLogDetail.send_to]). \
    where(RateSendLogDetail.id == RateDownloadLog.log_detail_id).correlate_except(
    RateSendLogDetail))

RateDownloadLog.rate_table_name = column_property(select([RateSendLogDetail.rate_table_name]). \
                                                  where(
    RateSendLogDetail.id == RateDownloadLog.log_detail_id).correlate_except(RateSendLogDetail))


# --- RateSendLog


class IngressDidRepository(DnlApiBaseModel):
    __tablename__ = 'ingress_did_repository'

    number = Column(String(255), primary_key=True)
    ingress_id = Column(ForeignKey('resource.resource_id', ondelete='CASCADE'))
    egress_id = Column(ForeignKey('resource.resource_id', ondelete='CASCADE'))
    created_time = Column(DateTime(True), default=datetime.now(UTC))
    updated_time = Column(DateTime(True), onupdate=func.now())
    state = Column(String(200))
    country = Column(String(200))
    city = Column(String(200))
    rate_center = Column(String(200))
    status = Column(SmallInteger, default=0)
    type = Column(SmallInteger)
    lata = Column(String(100))

    egress = relationship(
        Resource, primaryjoin='IngressDidRepository.egress_id == Resource.resource_id'
    )

    ingress = relationship(
        Resource, primaryjoin='IngressDidRepository.ingress_id == Resource.resource_id'
    )


class ClientPayment(DnlApiBaseModel):
    __tablename__ = 'client_payment'
    PAYMENT_TYPE_DICT = {1: 'n/a', 2: 'undefined',
                         3: 'invoice payment sent',
                         4: 'invoice payment received',
                         5: 'Prepayment Received',
                         6: 'payment sent',
                         7: 'credit note received',
                         8: 'credit note sent',
                         9: 'reset',
                         10: 'offset',
                         11: 'debit received',
                         12: 'debit sent',
                         13: 'mutual reset',
                         14: 'actual reset',
                         15: 'prepayment',
                         16: 'extra charge'}
    USER_PAYMENT_TYPES = ('invoice payment sent',
                          'invoice payment received',
                          'Prepayment Received',
                          'payment sent',
                          'credit note received',
                          'credit note sent',
                          'debit received',
                          'debit sent',
                          'credit',
                          'invoice Credit'
                          )
    METHOD = {None: 'undefined', 0: 'paypal', 1: 'stripe'}

    @hybrid_property
    def total(self):
        if self.payment_type in (
                'invoice payment sent', 'payment sent', 'credit note sent', 'debit received', 'scc cost'):
            return -self.amount
        return self.amount

    client_payment_id = Column(Integer, primary_key=True)
    payment_time = Column(DateTime(True), nullable=False,
                          server_default=text_("('now'::text)::timestamp(0) with time zone"))
    amount = Column(Numeric(20, 10), nullable=False)
    result = Column(Boolean, nullable=False, default=True)
    client_id = Column(ForeignKey('client.client_id', ondelete='SET NULL'), index=True)
    cause = Column(String(20), server_default=text_('NULL'))
    description = Column(String(500))  # Set the note for reset balance
    approved = Column(Boolean)
    current_balance = Column(Numeric(30, 10), default=0)
    invoice_number = Column(String(100), index=True)
    payment_type = Column(ChoiceType(PAYMENT_TYPE_DICT), default=1)

    payment_method = Column(ChoiceType(METHOD), server_default=text_('NULL'))
    email_sended = Column(Boolean, server_default=text_('false'))
    update_by = Column(String)
    receiving_time = Column(DateTime(True))
    egress_amount = Column(Numeric, server_default=text_('NULL'))
    charge_name = Column(String(50), server_default=text_('NULL'))

    payment_type_name = synonym('payment_type')
    transaction_type = synonym('payment_type')
    date = synonym('payment_time')
    paid_on = synonym('receiving_time')
    note = synonym('description')
    balance = synonym('current_balance')

    # inv_list = relationship('Invoice', primaryjoin='foreign(Invoice.invoice_number)==ClientPayment.invoice_number', uselist=True, back_populates='payments')

    invoices = relationship('PaymentInvoice', uselist=True, back_populates='payment')
    client = relationship(Client, uselist=False, back_populates='payment', enable_typechecks=False)

    invoice_unpaid_amount = column_property(egress_amount - amount)

    carrier_name = column_property(
        select([Client.name]).where(Client.client_id == client_id).correlate_except(Client))

    client_name = column_property(
        select([Client.name]).where(Client.client_id == client_id).correlate_except(Client))

    __is_new = None

    @hybrid_property
    def company_name(self):
        return self.client.company_name

    @hybrid_property
    def payment_terms(self):
        return self.client.payment_terms()

    def before_save(self):
        self.__is_new = not bool(self.client_payment_id)
        if self.invoice_number and self.__is_new:
            i = Invoice.filter(Invoice.invoice_number == self.invoice_number).one()
            if i.paid:
                raise ValidationError({'invoice_number': ['invoice fully paid. Cannot add payment']})
            if float(i.unpaid_amount) - float(self.amount) < -0.01:
                raise ValidationError({'invoice_number': ['Entered amount is higher than unpaid amount.']})
            if i:
                self.egress_amount = i.total_amount - i.pay_amount
        pass

    def after_save(self):
        try:
            if self.__is_new and self.client_id and self.result:
                # self.client.regenerate_balance()
                ClientBalanceOperationAction.check_and_create(self.client_id, user=self.update_by)
                self.current_balance = Client.get(self.client_id).actual_balance

                if self.payment_type in (
                        'invoice payment received', 'Prepayment Received', 'credit note sent', 'debit sent'):
                    ClientBalanceOperationAction(client_id=self.client_id,
                                                 balance=self.amount,
                                                 ingress_balance=self.amount,
                                                 egress_balance=Decimal(0.0),
                                                 action='Increase ingress balance', create_by=self.update_by).save()
                elif self.payment_type in (
                        'invoice payment sent', 'payment sent', 'credit note received', 'debit received', 'extra charge'):
                    ClientBalanceOperationAction(client_id=self.client_id,
                                                 balance=self.amount,
                                                 ingress_balance=Decimal(0.0),
                                                 egress_balance=self.amount,
                                                 action='Decrease egress balance', create_by=self.update_by).save()
                """
                if self.payment_type in ('reset'):
                    BalanceDailyResetTask(start_time=self.date, mutual=1, actual=1,client_id=self.client_id,create_by=self.update_by).save()
                    BalanceDailyResetTask(start_time=self.date,reset_balance=1,mutual=1,actual=1,client_id=self.client_id,create_by=self.update_by).save()
                if self.payment_type in ('mutual reset'):
                    BalanceDailyResetTask(start_time=self.date, mutual=1,client_id=self.client_id,create_by=self.update_by).save()
                    BalanceDailyResetTask(start_time=self.date, reset_balance=1, mutual=1,client_id=self.client_id,create_by=self.update_by).save()
                if self.payment_type in ('actual reset'):
                    BalanceDailyResetTask(start_time=self.date, actual=1,client_id=self.client_id,create_by=self.update_by).save()
                    BalanceDailyResetTask(start_time=self.date, reset_balance=1, actual=1,client_id=self.client_id,create_by=self.update_by).save()
                """

                if self.invoice_number:
                    i = Invoice.filter(Invoice.invoice_number == self.invoice_number).first()
                    if i:
                        i.pay_amount = ClientPayment.session().query(
                            func.sum(ClientPayment.amount).label('amount')).filter(
                            ClientPayment.invoice_number == self.invoice_number).one().amount  # ( i.pay_amount if i.pay_amount else Decimal(0.0) ) + self.amount
                        if i.pay_amount >= i.current_charge:
                            i.paid = True
                        i.save()
                        ip = PaymentInvoice(amount=self.amount, invoice_id=i.invoice_id,
                                            payment_id=self.client_payment_id)
                        ip.save()
                self.auto_apply_mail()
        except Exception as e:
            log.error('Client payment error: {}'.format(e))
            self.result = False
            get_db().session.rollback()
            pass

    def auto_apply_mail(self):
        try:
            from api_dnl.tasks import payment_apply_mail_task
            if self.client.payment_received_notice and not self.email_sended:
                if self.payment_type in ('invoice payment received', 'Prepayment Received', 'credit note received',
                                         'debit received', 'extra charge'):
                    payment_apply_mail_task.delay(self.client_payment_id, 'payment_received')
            if not self.email_sended:
                if self.payment_type in ('invoice payment sent', 'payment sent', 'credit note sent', 'debit sent'):
                    payment_apply_mail_task.delay(self.client_payment_id, 'payment_sent')

        except Exception as e:
            log.error('Client payment send mail error: {}'.format(e))
            pass

    def delete(self, **kwargs):
        user = kwargs['user'].name if 'user' in kwargs else ''
        op = None
        if self.payment_type in ('invoice payment received', 'Prepayment Received',
                                 'credit note received', 'debit sent'):
            op = ClientBalanceOperationAction(client_id=self.client_id,
                                              balance=self.amount,
                                              ingress_balance=self.amount,
                                              egress_balance=Decimal(0.0),
                                              action='Decrease ingress balance', create_by=self.update_by)
        elif self.payment_type in ('invoice payment sent', 'payment sent', 'credit note sent', 'debit received', 'extra charge'):
            op = ClientBalanceOperationAction(client_id=self.client_id,
                                              balance=self.amount,
                                              ingress_balance=Decimal(0.0),
                                              egress_balance=self.amount,
                                              action='Increase egress balance', create_by=self.update_by)

        if self.invoice_number:
            i = Invoice.filter(Invoice.invoice_number == self.invoice_number).first()
            if i:
                amount = ClientPayment.session().query(
                    func.sum(ClientPayment.amount).label('amount')).filter(and_(
                    ClientPayment.invoice_number == self.invoice_number,
                    ClientPayment.client_payment_id != self.client_payment_id)).one()
                i.pay_amount = amount.amount if amount and amount.amount else 0
                if i.pay_amount < i.current_charge:
                    i.paid = False
                i.save()
                ip = PaymentInvoice(amount=-self.amount, invoice_id=i.invoice_id,
                                    payment_id=self.client_payment_id)
                ip.save()

        ret = super(ClientPayment, self).delete(**kwargs)
        if op:
            op.save()
        return ret

    def fmt_name(self):
        return self.client.login

    def fmt_amount(self):
        # if self.payment_type in ('invoice payment received', 'prepay payment received', 'credit note sent',
        #                          'debit received'):
        #     return None
        return dec_prn(self.amount, 2)

    def fmt_pay_amount(self):
        if self.payment_type in ('invoice payment sent', 'payment sent', 'credit note received', 'debit sent'):
            return None
        return dec_prn(self.amount, 2)

    def fmt_current_balance(self):
        return dec_prn(self.current_balance + self.total, 2)

    def fmt_prev_balance(self):
        return dec_prn(self.current_balance, 2)

    def fmt_receiving_time(self):
        return str(self.receiving_time)

    def fmt_note(self):
        return self.note

    # return """<table> <tr><td>note</td><td>{}</td></tr> <tr><td>description</td><td>{}</td></tr>
    # <tr><td>cause</td><td>{}</td></tr> </table>""".format(self.note,self.description,self.cause)

    def apply_mail(self, template, to=None, att=[]):
        return MailSender.apply_mail(self, template, to, att=att)


# Client.test_credit = lambda pt_id: Client.filter(payment_term_id=pt_id).count()

Client.last_payment_date = column_property(select([ClientPayment.date]).where(
    and_(ClientPayment.client_id == Client.client_id, ClientPayment.payment_type.in_([3, 4, 5, 6]))). \
                                           order_by(ClientPayment.date.desc()).limit(1).correlate_except(ClientPayment))

Client.last_payment_amount = column_property(select([ClientPayment.amount]).where(
    and_(ClientPayment.client_id == Client.client_id, ClientPayment.payment_type.in_([3, 4, 5, 6]))). \
    order_by(ClientPayment.date.desc()).limit(1).correlate_except(
    ClientPayment))

Client.test_credit_value = column_property(select([ClientPayment.amount]). \
                                           where(and_(Client.client_id == ClientPayment.client_id,
                                                      ClientPayment.payment_type == 'Prepayment Received',
                                                      ClientPayment.description == 'test credit')). \
                                           order_by(ClientPayment.payment_time.desc()).limit(1). \
                                           correlate_except(ClientPayment).label('test_credit_value'))


class C4ClientBalance(DnlApiBaseModel):
    __tablename__ = 'c4_client_balance'
    id = Column(Integer, primary_key=True, server_default=text_("nextval('c4_client_balance_id_seq'::regclass)"))
    client_id = Column(String(100), nullable=False, index=True, unique=True)
    balance = Column(Numeric(30, 6), nullable=False, server_default=text_("0.00"))
    ingress_balance = Column(Numeric(30, 6), nullable=False, server_default=text_("0.00"))
    egress_balance = Column(Numeric(30, 6), nullable=False, server_default=text_("0.00"))
    create_time = Column(DateTime(True), nullable=False,
                         server_default=text_("('now'::text)::timestamp(0) with time zone"))

    # carrier = relationship(Client,   primaryjoin="Client.client_id == foreign( cast(C4ClientBalance.client_id,Integer) )")

    @classmethod
    def client_balance(cls, client_id):
        q = cls().filter(text_("client_id::integer=%s" % client_id)).first()
        if q:
            return q.balance
        else:
            return 0


# Client.actual_balance = column_property(select([func.coalesce(func.sum(C4ClientBalance.balance),text_("0.0"))]).
Client.actual_balance = column_property(select([C4ClientBalance.balance]). \
                                        where(Client.client_id == cast(C4ClientBalance.client_id, Integer)). \
                                        correlate_except(C4ClientBalance))
AgentClient.actual_balance = column_property(
    select([Client.actual_balance]).where(Client.client_id == AgentClient.client_id).correlate_except(Client))
AgentClient.credit_limit = column_property(
    select([-Client.allowed_credit]).where(Client.client_id == AgentClient.client_id).correlate_except(Client))


class DidBillingOperationAction(DnlApiBaseModel):
    """'Notify dnl_tools on new item in did_billing_brief table (API version 5) OR did_billing_rel table (API version >=6)'"""
    __tablename__ = 'did_billing_operation_action'
    CODE = {0: 'Success',
            1: 'Failure'}

    id = Column(Integer, primary_key=True)
    did_billing_id = Column(ForeignKey('did_billing_rel.id', ondelete='CASCADE'), index=True,
                            doc='Reference to  did_billing_rel(id) (API version >=6)')
    created_by = Column(String(100))
    created_on = Column(DateTime(True), nullable=False, server_default=func.now())
    code = Column(ChoiceType(CODE), doc="0: 'Success', 1: 'Failure'")
    error = Column(Text)
    finished_by = Column(String(100))
    finished_on = Column(DateTime(True))


class DidTransaction(DnlApiBaseModel):
    __tablename__ = 'did_transaction'

    id = Column(Integer, primary_key=True)
    # did_billing_id = Column(ForeignKey('did_billing_rel.id', ondelete='CASCADE'), index=True,
    #                         doc='Reference to  did_billing_rel(id) (API version >=6)')
    date = Column(DateTime(True), nullable=False, server_default=func.now())
    did_number = Column(String(50), nullable=False)
    client_id = Column(nullable=False)
    mrc = Column(Numeric(20, 5), nullable=True)
    nrc = Column(Numeric(20, 5), nullable=True)
    # port_fee = Column(Numeric(20, 5), nullable=True)
    created_on = Column(DateTime(True), nullable=False, server_default=func.now())


class ClientBalance(DnlApiBaseModel):  # Not used??
    __tablename__ = 'client_balance'

    client_id = Column(String(100), primary_key=True)
    balance = Column(String(100), nullable=False)
    create_time = Column(DateTime(True), nullable=False)
    ingress_balance = Column(String(100))
    egress_balance = Column(String(100))
    bod_balance = Column(String(100), nullable=False)


class ClientBalanceOperationAction(DnlApiBaseModel):
    __tablename__ = 'client_balance_operation_action'
    ACTION = {0: 'Added a new client balance',
              1: 'Reset client balance, ingress balance, egress balance',
              2: 'Increase ingress balance', 3: 'Decrease ingress balance',
              4: 'Increase egress balance', 5: 'Decrease egress balance'}
    UPDATE_RESULT = {0: 'Initial', 1: 'Update completed', 2: 'Update failed'}

    id = Column(Integer, primary_key=True,
                server_default=text_("nextval('client_balance_operation_action_id_seq'::regclass)"))
    client_id = Column(Integer, index=True)
    balance = Column(Numeric(12, 6), nullable=False, server_default=text_("0.00"))
    ingress_balance = Column(Numeric(12, 6), nullable=False, server_default=text_("0.00"))
    egress_balance = Column(Numeric(12, 6), nullable=False, server_default=text_("0.00"))
    action = Column(ChoiceType(ACTION))
    update_result = Column(ChoiceType(UPDATE_RESULT), nullable=False, index=True, default=0, server_default=text_("0"))
    update_by = Column(String(100))
    update_time = Column(DateTime(True))
    create_by = Column(String(100))
    create_time = Column(DateTime(True), nullable=False,
                         server_default=text_("('now'::text)::timestamp(0) with time zone"))

    def after_save(self):
        try:
            from api_dnl.tasks import balance_log_task
            balance_log_task.delay(self.client_id)
        except:
            pass

    @staticmethod
    def check_and_create(client_id, user, summ=0.0):
        c4 = C4ClientBalance.filter(C4ClientBalance.client_id == str(client_id)).first()
        if not c4:
            # c4 = C4ClientBalance(client_id = client_id)
            # c4.save()
            id = ClientBalanceOperationAction(client_id=client_id, balance=summ, ingress_balance=summ,
                                              egress_balance=summ,
                                              action='Added a new client balance', create_by=user).save()
            log.debug('ClientBalanceOperationAction Added a new client balance {}'.format(id))
        return c4

    @staticmethod
    def add_create(client_id, summ, user):
        balance = Decimal(summ)
        ingress_balance = Decimal(summ)
        egress_balance = Decimal(0.0)
        id = ClientBalanceOperationAction(client_id=client_id, balance=balance, ingress_balance=ingress_balance,
                                          egress_balance=egress_balance,
                                          action='Added a new client balance', create_by=user).save()
        log.debug('ClientBalanceOperationAction Added a new client balance {}'.format(id))

    @staticmethod
    def add_payment(client_id, summ, user):
        balance = Decimal(summ)
        ingress_balance = Decimal(summ)
        egress_balance = Decimal(0.0)
        id = ClientBalanceOperationAction(client_id=client_id,
                                          balance=balance,
                                          ingress_balance=ingress_balance,
                                          egress_balance=egress_balance,
                                          action='Increase ingress balance', create_by=user).save()
        log.debug('ClientBalanceOperationAction Increase ingress balance {}'.format(id))

    @staticmethod
    def del_payment(client_id, summ, user):
        ingress_balance = Decimal(summ)
        egress_balance = Decimal(summ)
        balance = Decimal(summ)
        id = ClientBalanceOperationAction(client_id=client_id,
                                          balance=balance,
                                          ingress_balance=ingress_balance,
                                          egress_balance=egress_balance,
                                          action='Decrease ingress balance', create_by=user).save()
        log.debug('ClientBalanceOperationAction Decrease ingress balance {}'.format(id))

    @staticmethod
    def add_invoice(client_id, summ, user):

        ingress_balance = Decimal(summ)
        egress_balance = Decimal(summ)
        balance = Decimal(summ)
        id = ClientBalanceOperationAction(client_id=client_id,
                                          balance=balance,
                                          ingress_balance=ingress_balance,
                                          egress_balance=egress_balance,
                                          action='Decrease egress balance', create_by=user).save()
        log.debug('ClientBalanceOperationAction Decrease egress balance {}'.format(id))

    @staticmethod
    def del_invoice(client_id, summ, user):
        ingress_balance = Decimal(summ)
        egress_balance = Decimal(summ)
        balance = Decimal(summ)
        id = ClientBalanceOperationAction(client_id=client_id,
                                          balance=balance,
                                          ingress_balance=ingress_balance,
                                          egress_balance=egress_balance,
                                          action='Increase egress balance', create_by=user).save()
        log.debug('ClientBalanceOperationAction Increase egress balance {}'.format(id))

    @staticmethod
    def add_reset(client_id, summ, user):
        ingress_balance = Decimal(summ)
        egress_balance = Decimal(summ)
        balance = Decimal(summ)
        id = ClientBalanceOperationAction(client_id=client_id,
                                          balance=balance,
                                          ingress_balance=ingress_balance,
                                          egress_balance=egress_balance,
                                          action='Reset client balance, ingress balance, egress balance',
                                          create_by=user).save()
        log.debug('ClientBalanceOperationAction Reset balance {}'.format(id))


class BalanceHistoryActual(DnlApiBaseModel):
    __tablename__ = 'balance_history_actual'
    __table_args__ = (UniqueConstraint('date', 'client_id'),)

    id = Column(Integer, primary_key=True, server_default=text_("nextval('balance_history_actual_id_seq'::regclass)"))
    date = Column(Date, index=True, nullable=False)
    payment_received = Column(Numeric(20, 5), nullable=False, server_default=text_("0"))
    credit_note_sent = Column(Numeric(20, 5), nullable=False, server_default=text_("0"))
    debit_note_sent = Column(Numeric(20, 5), nullable=False, server_default=text_("0"))
    unbilled_incoming_traffic = Column(Numeric(20, 5), nullable=False, server_default=text_("0"))
    short_charges = Column(Numeric(20, 5), nullable=False, server_default=text_("0"))
    payment_sent = Column(Numeric(20, 5), nullable=False, server_default=text_("0"))
    credit_note_received = Column(Numeric(20, 5), nullable=False, server_default=text_("0"))
    debit_note_received = Column(Numeric(20, 5), nullable=False, server_default=text_("0"))
    unbilled_outgoing_traffic = Column(Numeric(20, 5), nullable=False, server_default=text_("0"))
    actual_balance = Column(Numeric(20, 5), nullable=False)
    client_id = Column(ForeignKey('client.client_id', ondelete='CASCADE'), nullable=False)
    actual_ingress_balance = Column(Numeric(20, 5), nullable=False, server_default=text_("0"))
    actual_egress_balance = Column(Numeric(20, 5), nullable=False, server_default=text_("0"))
    mrc = Column(Numeric(20, 5), nullable=False, server_default='0.0')
    nrc = Column(Numeric(20, 5), nullable=False, server_default='0.0')
    port_fee = Column(Numeric(20, 5), nullable=False, server_default='0.0')
    sms = Column(Numeric(20, 5), nullable=False, server_default='0.0')

    incoming_traffic = synonym('unbilled_incoming_traffic')
    outgoing_traffic = synonym('unbilled_outgoing_traffic')
    credit_sent = synonym('credit_note_sent')
    credit_received = synonym('credit_note_received')
    carrier_id = synonym('client_id')

    carrier = relationship(Client, back_populates='balance_actual', uselist=False)

    client_name = column_property(select([Client.name]).where(Client.client_id == client_id).correlate_except(Client))

    carrier_name = synonym('client_name')


class BalanceHistory(DnlApiBaseModel):
    __tablename__ = 'balance_history'

    id = Column(Integer, primary_key=True)
    date = Column(Date)
    payment_received = Column(Numeric(20, 5), nullable=False, default=0)
    credit_note_sent = Column(Numeric(20, 5), nullable=False, default=0)
    debit_note_sent = Column(Numeric(20, 5), nullable=False, default=0)
    payment_sent = Column(Numeric(20, 5), nullable=False, default=0)
    credit_note_received = Column(Numeric(20, 5), nullable=False, default=0)
    debit_note_received = Column(Numeric(20, 5), nullable=False, default=0)
    invoice_set = Column(Numeric(20, 5), nullable=False, default=0)
    invoice_received = Column(Numeric(20, 5), nullable=False, default=0)
    mutual_balance = Column(Numeric(20, 5), nullable=False)
    client_id = Column(ForeignKey('client.client_id', ondelete='CASCADE'))
    mutual_ingress_balance = Column(Numeric(20, 5), nullable=False, default=0)
    mutual_egress_balance = Column(Numeric(20, 5), nullable=False, default=0)

    invoice_sent = synonym('invoice_set')
    credit_sent = synonym('credit_note_sent')
    credit_received = synonym('credit_note_received')

    carrier = relationship(Client, back_populates='balance_mutual', uselist=False)

    client_name = column_property(select([Client.name]).where(Client.client_id == client_id).correlate_except(Client))
    carrier_name = synonym('client_name')

    @staticmethod
    def today_mutual_balance(client_id):
        def z(d):
            return d if not d is None else Decimal(0.0)

        dt = datetime.now(UTC)
        d = datetime(year=dt.year, month=dt.month, day=dt.day)
        dp = (d - timedelta(days=1))
        # dt0 = datetime(year=dt.year, month=dt.month, day=1)
        dt0 = d
        p = ClientPayment
        payment_received_total = z(p.session().query(func.sum(p.amount).label('payment_received_total')).filter(
            and_(p.client_id == client_id, p.payment_type.in_([4, 5]),
                 p.payment_time.between(dt0, dt))).first().payment_received_total)
        print('payment_received_total {}'.format(payment_received_total))

        payment_sent_total = z(p.session().query(func.sum(p.amount).label('payment_sent_total')).filter(
            and_(p.client_id == client_id, p.payment_type.in_([3, 6]),
                 p.payment_time.between(dt0, dt))).first().payment_sent_total)
        print('payment_sent_total {}'.format(payment_sent_total))

        debit_note_received = z(p.session().query(func.sum(p.amount).label('debit_note_received')).filter(
            and_(p.client_id == client_id, p.payment_type.in_([11]),
                 p.payment_time.between(dt0, dt))).first().debit_note_received)
        print('debit_note_received {}'.format(debit_note_received))

        debit_note_sent = z(p.session().query(func.sum(p.amount).label('debit_note_sent')).filter(
            and_(p.client_id == client_id, p.payment_type.in_([12]),
                 p.payment_time.between(dt0, dt))).first().debit_note_sent)
        print('debit_note_sent {}'.format(debit_note_sent))

        credit_note_received = z(p.session().query(func.sum(p.amount).label('credit_note_received')).filter(
            and_(p.client_id == client_id, p.payment_type.in_([7]),
                 p.payment_time.between(dt0, dt))).first().credit_note_received)
        print('credit_note_received {}'.format(credit_note_received))

        credit_note_sent = z(p.session().query(func.sum(p.amount).label('credit_note_sent')).filter(
            and_(p.client_id == client_id, p.payment_type.in_([8]),
                 p.payment_time.between(dt0, dt))).first().credit_note_sent)
        print('credit_note_sent {}'.format(credit_note_sent))

        p = Invoice
        invoice_received = z(p.session().query(func.sum(p.total_amount).label('invoice_received')).filter(
            and_(p.client_id == client_id, p.state.in_([1, 9]), type == 0,
                 p.invoice_time == d)).first().invoice_received)
        print('invoice_received {}'.format(invoice_received))
        invoice_sent = z(p.session().query(func.sum(p.total_amount).label('invoice_sent')).filter(
            and_(p.client_id == client_id, p.state.in_([1, 9]), type == 1,
                 p.invoice_time == d)).first().invoice_sent)
        print('invoice_sent {}'.format(invoice_sent))
        p = BalanceHistory
        begin_balance = z(p.session().query(func.sum(p.mutual_balance).label('begin_balance')).filter(
            and_(p.client_id == client_id,
                 p.date == dp)).first().begin_balance)
        print('begin_balance {}'.format(begin_balance))
        mutual_balance = begin_balance + payment_received_total + debit_note_received - debit_note_sent - \
                         payment_sent_total - credit_note_sent - invoice_received + invoice_sent

        client_name = Client.get(client_id).name

        return dict(client_id=client_id, client_name=client_name, date=str(d.date()),
                    payment_received=payment_received_total, payment_sent=payment_sent_total,
                    debit_received=debit_note_received, debit_sent=debit_note_sent,
                    credit_received=credit_note_received, credit_sent=credit_note_sent,
                    invoice_received=invoice_received, invoice_sent=invoice_sent,
                    begin_balance=begin_balance, mutual_balance=mutual_balance)


class TransactionFeeItems(DnlApiBaseModel):
    __tablename__ = 'transaction_fee_items'
    TRANS_TYPE = {1: 'payment', 2: 'service'}

    id = Column(Integer, primary_key=True)
    trans_type = Column(ChoiceType(TRANS_TYPE))  # integer   	 1- payment;2 - service charge
    transaction_fee_id = Column(Integer, default=0)
    trans_id = Column(Integer)  # payment_term_id, server_charge_items_id
    use_fee = Column(Float)  # real

    amount = synonym('use_fee')
    transaction_type = synonym('trans_type')

    payment = relationship(ClientPayment, uselist=False,
                           primaryjoin=trans_id == foreign(ClientPayment.client_payment_id))

    @hybrid_property
    def date(self):
        return self.payment.date

    @hybrid_property
    def note(self):
        return self.payment.description


class Invoice(DnlApiBaseModel):
    __tablename__ = 'invoice'
    STATE = {-1: 'void', 0: 'normal', 1: 'verify', 9: 'send'}
    TYPE = {0: 'sent(out--buy)', 1: 'received(in--sell)', 2: 'sent(all)', 3: 'incoming'}
    DISPUTED = {0: 'Non-Disputed', 1: 'Disputed', 2: 'Dispute Resolved'}
    STATUS = {0: 'creating', 1: 'zero cdr', 2: 'done', 3: 'only support buy/sell'}
    OUTPUT_TYPE = {0: 'pdf', 1: 'excel', 2: 'html'}
    CREATE_TYPE = {0: 'auto', 1: 'manual'}

    invoice_id = Column(Integer, primary_key=True)
    invoice_number = Column(String(20), nullable=False, unique=True)
    buy_minutes = Column(Float)
    buy_service_charge = Column(Numeric(30, 10))
    buy_total = Column(Numeric(30, 10))
    cdr_path = Column(String(300))  # Cdr storage path
    client_id = Column(ForeignKey('client.client_id', ondelete='CASCADE'), index=True)
    create_type = Column(ChoiceType(CREATE_TYPE), default=0)  # 0-Auto-generated Invoice; 1-Manual Invoice
    credit_amount = Column(Numeric(30, 5), default=0)
    credit_note = Column(Numeric(30, 10))
    credit_remaining = Column(Numeric(20, 5))
    current_balance = Column(Numeric(30, 10), nullable=False)
    current_charge = Column(Numeric(30, 10))
    decimal_place = Column(Integer, default=5)
    decimals_num = Column(Integer)
    decurring_charge = Column(Numeric(30, 10))
    disputed = Column(ChoiceType(DISPUTED), default=0)  # 0-Non-Disputed; 1-Disputed; 2-Dispute Resolved
    disputed_amount = Column(Numeric(30, 10))
    due_date = Column(Date)  # Expiry date
    egress_cdr_file = Column(String(255))
    finance_charge = Column(Numeric(30, 10))
    generate_copy_time = Column(DateTime(True))
    generate_end_time = Column(DateTime(True))
    generate_start_time = Column(DateTime(True))
    generate_stats_time = Column(DateTime(True))
    include_detail = Column(Boolean, default=True)
    ingress_cdr_file = Column(String(255))
    invoice_balance_time = Column(DateTime(True))
    invoice_end = Column(DateTime(True), nullable=False)  # expiration date
    invoice_include_payment = Column(Boolean)
    invoice_jurisdictional_detail = Column(Boolean, default=True)
    invoice_log_id = Column(ForeignKey('invoice_log.id', ondelete='SET NULL'), index=True)  # Column(Integer)
    invoice_start = Column(DateTime(True), nullable=False)  # start date
    invoice_time = Column(DateTime(True), nullable=False, default=datetime.now(UTC))  # generation time
    invoice_use_balance_type = Column(Integer, default=0)
    invoice_zone = Column(String(10))
    is_invoice_account_summary = Column(Boolean, default=False)
    is_send_as_link = Column(Boolean, default=False)
    is_short_duration_call_surcharge_detail = Column(Boolean, default=False)
    is_show_by_date = Column(Boolean, default=False)
    is_show_code_100 = Column(Boolean, default=False)
    is_show_code_name = Column(Boolean, default=False)
    is_show_country = Column(Boolean, default=False)
    is_show_daily_usage = Column(Boolean, default=False)
    is_show_detail_trunk = Column(Boolean, default=False)
    is_show_total_trunk = Column(Boolean, default=False)
    link_cdr = Column(String(50))
    lrn_cost = Column(Numeric(30, 10))
    lrn_numbers = Column(Integer)
    lrn_rate = Column(Numeric(30, 10))
    new_balance = Column(Numeric(30, 10))
    non_recurring_charge = Column(Numeric(30, 10))
    output_type = Column(ChoiceType(OUTPUT_TYPE), default=0)
    paid = Column(Boolean, nullable=False, default=False)  # whether to pay
    pay_amount = Column(Numeric(30, 5), nullable=False, default=0)  # The payment amount of the routing partner
    payment_credit = Column(Numeric(30, 10))
    payment_term = Column(String)
    pdf_path = Column(String(300))  # pdf storage path
    please_pay = Column(Numeric(20, 5))
    previous_balance = Column(Numeric(30, 10))
    rate_value = Column(Integer, default=0)
    reconcile_file_path = Column(String)
    reconcile_state = Column(Integer, default=0)  # 0: there is a comparable file; 1: in progress; 2: has been completed
    scc_calls = Column(Integer)
    scc_cost = Column(Numeric(30, 5))
    scc_per = Column(Float)
    scc_sec = Column(Integer)
    sell_minutes = Column(Float)
    sell_service_charge = Column(Numeric(30, 10))
    sell_total = Column(Numeric(30, 10))
    send_time = Column(DateTime(True))  # invoice time
    state = Column(ChoiceType(STATE), nullable=False, default=0)  # 1:void;0:normal;1:verify;9:send
    status = Column(ChoiceType(STATUS))
    tax = Column(Numeric(30, 10))
    total_amount = Column(Numeric(30, 5), nullable=False, default=0)  # invoice amount
    total_calls = Column(Integer, default=0)
    total_minutes = Column(Float)
    type = Column(ChoiceType(TYPE))  # 0:sent(out--buy);1:received(in--sell);2:sent(all);3:incoming
    unlimited_credit_unlimited = Column(Boolean, default=False)
    usage_detail_fields = Column(Text)
    company_type = Column(ChoiceType(Client.COMPANY_TYPE), server_default='1')

    is_orig = column_property(company_type.in_(['origination', 'both termination and origination']) == True)
    is_term = column_property(company_type.in_(['termination', 'both termination and origination']) == True)

    amount = synonym('total_amount')
    invoice_date = synonym('invoice_time')
    invoice_period_from = synonym('invoice_start')
    invoice_period_to = synonym('invoice_end')
    start_date = synonym('invoice_start')
    end_date = synonym('invoice_end')
    invoice_amount = synonym('total_amount')
    # url=synonym('link_cdr')

    log = relationship('InvoiceLog', back_populates='invoices', uselist=False)
    client = relationship(Client, back_populates='invoices', uselist=False)
    payments = relationship('PaymentInvoice', uselist=True, back_populates='invoice')

    invoice_credit_amount = column_property(
        select([func.sum(ClientPayment.amount)]).where(and_(ClientPayment.invoice_number == invoice_number,
                                                            ClientPayment.payment_type == 8)).correlate_except(
            ClientPayment))

    invoice_period = column_property(
        cast(invoice_start, String) + ' ' + \
        cast(invoice_end, String))

    # invoice_period = column_property(
    #     cast(cast(invoice_start, Date), String) + ' ' + \
    #     cast(cast(case([(create_type == 'auto', invoice_end),], else_ = invoice_end), Date), String))

    client_name = column_property(select([Client.name]).where(Client.client_id == client_id).correlate_except(Client))
    carrier_name = synonym('client_name')
    company_name = column_property(
        select([Client.company]).where(Client.client_id == client_id).correlate_except(Client))
    auto_send_invoice = column_property(
        select([Client.auto_send_invoice]).where(Client.client_id == client_id).correlate_except(Client))
    payment_term_name = column_property(
        select([PaymentTerm.name]).where(and_(Client.payment_term_id == PaymentTerm.payment_term_id,
                                              Client.client_id == client_id)).correlate_except(PaymentTerm, Client)
    )
    payment_term_id = column_property(
        select([PaymentTerm.payment_term_id]).where(and_(Client.payment_term_id == PaymentTerm.payment_term_id,
                                              Client.client_id == client_id)).correlate_except(PaymentTerm, Client)
    )
    unpaid_amount = column_property(
        case([(current_charge > pay_amount, current_charge - pay_amount),
              ], else_=0))

    def delete(self, **kwargs):
        PaymentInvoice.filter(PaymentInvoice.invoice_id == self.invoice_id).delete(
            synchronize_session=False)
        ret = super(Invoice, self).delete(**kwargs)
        return ret

    @property
    def cdr_url(self):
        if self.ingress_cdr_file:  # self.status == 'done':
            return '{}/files/{}'.format(settings.FILE_URL,
                                        self.ingress_cdr_file)
        else:
            return 'cdr file was not created'

    @property
    def invoice_num(self):
        import hashlib
        return hashlib.md5(str(self.invoice_number).encode()).hexdigest()

    @property
    def cdr_data(self):
        if self.ingress_cdr_file:  # self.status == 'done':
            # return open(self.cdr_path+'/'+self.ingress_cdr_file,'rb').read()
            return open(settings.FILES['upload_to'] + '/' + self.ingress_cdr_file, 'rb').read()
        else:
            return None

    @property
    def url(self):
        if self.pdf_path:  # self.status == 'done':
            return '{}/files/invoices/{}'.format(settings.FILE_URL, self.pdf_path)
        else:
            return None

    @property
    def invoice_link(self):
        return self.url

    @property
    def download_link(self):
        return self.url

    @property
    def file_data(self):
        try:
            f = open(settings.FILES['invoices'] + '/' + self.pdf_path, 'rb')
            return f.read()
        except:
            return None

    @classmethod
    def _max_number(cls):
        ret = cls.query().session.query(func.max(func.cast(cls.invoice_number, Integer)).label("max_number")).first()
        if ret and ret.max_number:
            return ret.max_number
        else:
            return 0

    @classmethod
    def max_number(cls):
        sql = "SELECT nextval('class4_seq_invoice_no') AS last_invoice_number"
        last_invoice_number = cls.session.execute(sql).fetchall()[0][0]
        while cls.filter(cls.invoice_number == str(last_invoice_number)).first():
            last_invoice_number = cls.session.execute(sql).fetchall()[0][0]
        return last_invoice_number

    @classmethod
    def create_invoice(cls, client_id, start_date, end_date, gmt, show_cdr, show_trunk_breakdown):
        # start_date=start_date+gmt;
        # end_date=end_date+gmt;
        obj = cls(client_id=client_id, invoice_start=start_date, invoice_end=end_date, invoice_zone=gmt,
                  is_short_duration_call_surcharge_detail=show_cdr, is_show_detail_trunk=show_trunk_breakdown)
        obj.state = 0

        return obj

    @property
    def fmt_start_date(self):
        return str(self.invoice_start)

    @property
    def fmt_end_date(self):
        return str(self.invoice_end)

    @property
    def fmt_invoice_amount(self):
        return dec_prn(self.current_charge, 2)

    @property
    def fmt_current_charge(self):
        return dec_prn(self.current_charge, 2)

    @property
    def fmt_cdr_url(self):
        return self.cdr_url
    
    @hybrid_property
    def void(self):
        return self.state == -1


INVOICE_SEND_STATE = 'send'
# Invoice.company_type = column_property(
#     select([Client.company_type]).where(Client.client_id == Invoice.client_id).correlate_except(Client))
Client.over_due = column_property(select([func.sum(Invoice.total_amount - Invoice.pay_amount).label('over_due')]). \
                                  where(
    and_(Invoice.client_id == Client.client_id, Invoice.paid.isnot(True), Invoice.state == literal_column("9"),
         Invoice.due_date < datetime.now(UTC))). \
                                  correlate_except(Invoice))
ClientPayment.invoice_total_amount = column_property(
    select([Invoice.total_amount]).where(ClientPayment.invoice_number == Invoice.invoice_number).correlate_except(
        Invoice))
ClientPayment.invoice_paid_amount = column_property(
    select([ClientPayment.amount.op('+')(Invoice.total_amount).op('-')(ClientPayment.egress_amount)]).where(
        ClientPayment.invoice_number == Invoice.invoice_number).correlate_except(Invoice))
# ClientPayment.invoice_unpaid_amount = column_property(select([Invoice.total_amount-Invoice.pay_amount]).where(ClientPayment.invoice_number==Invoice.invoice_number).correlate_except(Invoice))
ClientPayment.invoice_due_date = column_property(
    select([Invoice.due_date]).where(ClientPayment.invoice_number == Invoice.invoice_number).correlate_except(Invoice))
ClientPayment.invoice_period = column_property(
    select([Invoice.invoice_period]).where(ClientPayment.invoice_number == Invoice.invoice_number).correlate_except(
        Invoice))
ClientPayment.invoice_id = column_property(
    select([Invoice.invoice_id]).where(Invoice.invoice_number == ClientPayment.invoice_number).correlate_except(
        Invoice))
Client.mutual_balance_old = column_property(select([BalanceHistory.mutual_balance]). \
                                            where(Client.client_id == BalanceHistory.client_id). \
                                            order_by(BalanceHistory.date.desc()).limit(1). \
                                            correlate_except(BalanceHistory).label('mutual_balance'))


def now_date():
    dt = datetime.now(UTC)
    return str(datetime(year=dt.year, month=dt.month, day=dt.day))


def prev_date():
    dt = datetime.now(UTC) - timedelta(days=1)
    return str(datetime(year=dt.year, month=dt.month, day=dt.day))


Client.cur_payment_total = column_property(
    func.coalesce(select([func.sum(case([(ClientPayment.payment_type.in_([4, 5, 7, 11]),
                                          ClientPayment.amount),
                                         (ClientPayment.payment_type.in_([3, 6, 12, 8]), -ClientPayment.amount)],
                                        else_=0.0))]). \
                  where(
        and_(Client.client_id == ClientPayment.client_id, ClientPayment.payment_time >= now_date())).limit(
        1).correlate_except(ClientPayment).as_scalar(), 0.0))

Client.cur_invoices_total = column_property(
    func.coalesce(select([func.sum(case([(Invoice.type == 1, Invoice.total_amount),
                                         (Invoice.type == 0, -Invoice.total_amount)], else_=0.0))]). \
                  where(and_(Client.client_id == Invoice.client_id, Invoice.invoice_time >= now_date())).limit(
        1).correlate_except(Invoice).as_scalar(), 0.0))

Client.cur_mutual_balance = column_property(func.coalesce(select([BalanceHistory.mutual_balance]). \
                                                          where(
    and_(Client.client_id == BalanceHistory.client_id, BalanceHistory.date == now_date())).limit(1).correlate_except(
    BalanceHistory).as_scalar(), 0.0))

Client.mutual_balance = column_property(
    Client.cur_mutual_balance.op('+')(Client.cur_payment_total).op('+')(Client.cur_invoices_total))

Client.prev_payment_total = column_property(
    func.coalesce(select([func.sum(case([(ClientPayment.payment_type.in_([4, 5, 7, 11]),
                                          ClientPayment.amount),
                                         (ClientPayment.payment_type.in_([3, 6, 12, 8]), -ClientPayment.amount)],
                                        else_=0.0))]). \
                  where(
        and_(Client.client_id == ClientPayment.client_id, ClientPayment.payment_time >= prev_date())).limit(
        1).correlate_except(ClientPayment).as_scalar(), 0.0))

Client.prev_invoices_total = column_property(
    func.coalesce(select([func.sum(case([(Invoice.type == 1, Invoice.total_amount),
                                         (Invoice.type == 0, -Invoice.total_amount)], else_=0.0))]). \
                  where(and_(Client.client_id == Invoice.client_id, Invoice.invoice_time >= prev_date())).limit(
        1).correlate_except(Invoice).as_scalar(), 0.0))


class InvoiceLog(DnlApiBaseModel):
    __tablename__ = 'invoice_log'
    STATUS = {1: 'progress', 2: 'done', 3: 'error'}
    id = Column(Integer, primary_key=True, server_default=text_("nextval('invoice_log_id_seq'::regclass)"))
    start_time = Column(DateTime(True))
    end_time = Column(DateTime(True))
    status = Column(ChoiceType(STATUS), server_default=text_("0"))
    cnt = Column(Integer, server_default=text_("1"))
    invoices = relationship('Invoice', back_populates='log', uselist=True)

    def progress(self):
        cnt = self.cnt if self.cnt else 0
        count = len(self.invoices)
        return str(cnt) + '/' + str(count)

    # @hybrid_property
    def related_count(self):
        return self.query().session. \
            query(func.count(Invoice.invoice_id)). \
            with_parent(self).scalar()

    # @hybrid_property
    def amount(self):
        return self.query().session. \
            query(func.sum(Invoice.total_amount)). \
            with_parent(self).scalar()

    def invoice_due_date(self):
        return self.query().session. \
            query(func.max(Invoice.due_date)). \
            with_parent(self).scalar()

    # @hybrid_property
    def total(self):
        try:
            return sum([i.total_amount for i in self.invoices])
        except:
            return 0

    # @hybrid_property
    def total_count(self):
        try:
            return len(self.invoices)
        except:
            return 0

    def total_due_date(self):
        try:
            return max([i.due_date for i in self.invoices])
        except:
            return None

    # @hybrid_property
    def client_name(self):
        try:
            return self.invoices[0].client.name
        except:
            return ''


# ---PaymentInvoice---
class PaymentInvoice(DnlApiBaseModel):
    __tablename__ = 'payment_invoice'

    id = Column(Integer, primary_key=True, server_default=text_("nextval('payment_invoice_id_seq'::regclass)"))
    payment_id = Column(ForeignKey('client_payment.client_payment_id'))
    invoice_id = Column(ForeignKey('invoice.invoice_id'))
    amount = Column(Numeric(20, 5))
    client_id = column_property(
        select([Invoice.client_id.label('client_client_id')]).where(Invoice.invoice_id == invoice_id).correlate_except(
            Invoice))

    invoice = relationship('Invoice', uselist=False, back_populates='payments')
    payment = relationship('ClientPayment', uselist=False, back_populates='invoices')


# +++ PaymentInvoice
class InvoicePayment(DnlApiBaseModel):
    __tablename__ = 'invoice_payment'

    id = Column(Integer, primary_key=True, server_default=text_("nextval('invoice_payment_id_seq'::regclass)"))
    invoice_no = Column(String(50))
    payment_time = Column(DateTime(True))
    payment_amount = Column(Numeric)


class CarrierLowBalanceConfig(DnlApiBaseModel):
    __tablename__ = 'client_low_balance_config'
    SEND_TIME_TYPE = {0: 'daily', 1: 'hourly'}
    VALUE_TYPE = {0: 'Actual Balance', 1: 'Percentage'}

    client_id = Column(Integer, ForeignKey('client.client_id', ondelete='CASCADE'), primary_key=True)
    is_notify = Column(Boolean)
    value_type = Column(ChoiceType(VALUE_TYPE))
    send_time_type = Column(ChoiceType(SEND_TIME_TYPE))
    actual_notify_balance = Column(Numeric(30, 2))
    percentage_notify_balance = Column(Numeric(30, 2))
    daily_send_time = Column(Integer)
    duplicate_days = Column(Integer)
    send_to = Column(Integer)
    duplicate_send_days = Column(Integer, nullable=False, default=0)
    last_alert_time = Column(DateTime(timezone=True))
    disable_trunks_days = Column(Integer, default=5)

    carrier = relationship(Client, uselist=False, back_populates='low_balance_config', enable_typechecks=False)
    client = synonym('carrier')

    @hybrid_property
    def carrier_name(self):
        return self.carrier.name

    def create_from_template(self, obj):
        if obj:
            self.is_notify = obj.is_notify
            self.value_type = obj.value_type
            self.send_time_type = obj.send_time_type
            self.actual_notify_balance = obj.actual_notify_balance
            self.percentage_notify_balance = obj.percentage_notify_balance
            self.daily_send_time = obj.daily_send_time
            self.duplicate_days = obj.duplicate_days
            self.send_to = obj.send_to
            self.duplicate_send_days = obj.duplicate_send_days
            self.last_alert_time = obj.last_alert_time
            self.disable_trunks_days = obj.disable_trunks_days


class QosTotal(DnlApiBaseModel):
    __tablename__ = 'qos_total'

    report_time = Column(
        DateTime(timezone=True),
        primary_key=True,
        default=datetime.now(UTC), index=True, server_default=text_("('now'::text)::timestamp(0) with time zone")
    )

    call = Column(Integer)
    cps = Column(Integer)
    server_ip = Column(String(50))
    server_port = Column(Integer)
    channels = Column(Integer)
    ingress_channels = Column(Integer)
    ingress_cps = Column(Integer)
    egress_cps = Column(Integer)
    egress_channels = Column(Integer)

    @hybrid_property
    def start_date(self):
        try:
            return self._start_date
        except:
            return None

    @start_date.setter
    def start_date(self, value):
        self._start_date = value
        self.filter('report_time>%s' % value)

    @hybrid_property
    def end_date(self):
        try:
            return self._end_date
        except:
            return None

    @end_date.setter
    def end_date(self, value):
        self._end_date = value
        self.filter('report_time<%s' % value)

    @classmethod
    def filter(cls, *args, **kwargs):
        fields = cls.get_fields()
        remove_keys = []
        if not args and kwargs:
            for k, v in kwargs.items():
                if k not in fields:  # pragma: no cover
                    continue
                args = args + (getattr(cls, k) == v,)
                remove_keys.append(k)

            for k in remove_keys:
                del kwargs[k]
        raise Exception(str(args) + '---' + str(kwargs))
        result = cls.query().filter(*args, **kwargs)
        return result

    @staticmethod
    def get_max_ingress_by_last_hours(hours=24):
        now = datetime.now()
        last = now - timedelta(hours=hours)

        query = QosTotal.query().session \
            .query(func.max(QosTotal.ingress_channels)
                   .label('max_ingress')) \
            .filter(QosTotal.report_time.between(last, now)) \
            .group_by(func.date(QosTotal.report_time)) \
            .order_by(desc('max_ingress')) \
            .limit(1)

        return query.first()

    @staticmethod
    def get_max_ingress_by_period(start, end, ip):

        query = QosTotal.query().session \
            .query(func.date(QosTotal.report_time).label('report_date'), func.max(QosTotal.ingress_channels)
                   .label('max_ingress')) \
            .filter(QosTotal.report_time.between(start, end)).filter(QosTotal.server_ip == ip) \
            .group_by(func.date(QosTotal.report_time)) \
            .order_by(desc('max_ingress')) \
            .limit(1)
        # raise Exception(_q_str(query))
        return query.all(), 1, 0, 51, None


class QosResource(DnlApiBaseModel):
    __tablename__ = 'qos_resource'
    DIRECTION = {0: 'ingress', 1: 'egress'}
    report_time = Column(DateTime(True), nullable=False, index=True,
                         server_default=text_("('now'::text)::timestamp(0) with time zone"), primary_key=True)
    res_id = Column(Integer, primary_key=True)
    call = Column(Integer)
    cps = Column(Integer)
    direction = Column(ChoiceType(DIRECTION))
    server_ip = Column(String(50), index=True)
    server_port = Column(Integer, index=True)
    channels = Column(Integer)
    ingress_channels = synonym('channels')
    ingress_cps = synonym('cps')
    egress_cps = synonym('cps')
    egress_channels = synonym('channels')
    ingress_id = synonym('res_id')
    egress_id = synonym('res_id')
    res_name = column_property(
        select([Resource.alias]).where(Resource.resource_id == res_id).correlate_except(Resource))
    ingress_client_id = column_property(
        select([Resource.client_id]).where(Resource.resource_id == res_id).correlate_except(Resource))
    egress_client_id = column_property(
        select([Resource.client_id]).where(Resource.resource_id == res_id).correlate_except(Resource))
    trunk_type2 = column_property(
        select([Resource.trunk_type2]).where(Resource.resource_id == res_id).correlate_except(Resource))
    qos_resource_report_time_res_id_idx = Index('qos_resource_report_time_res_id_idx', 'report_time', 'res_id')

QosTotal.ingress_id = column_property(select([QosResource.res_id]). \
                    where(and_(QosResource.server_ip == QosTotal.server_ip,
                               QosResource.direction == 'ingress'))
                                    .limit(1).correlate(QosResource).label('ingress_id'))
QosTotal.egress_id = column_property(select([QosResource.res_id]). \
                    where(and_(QosResource.server_ip == QosTotal.server_ip,
                               QosResource.direction == 'ingress'))
                                    .limit(1).correlate(QosResource).label('egress_id'))
QosTotal.ingress_client_id = column_property(select([Resource.client_id]). \
                    where(Resource.resource_id == QosTotal.ingress_id).correlate(Resource).label('ingress_client_id'))
QosTotal.egress_client_id = column_property(select([Resource.client_id]). \
                    where(Resource.resource_id == QosTotal.egress_id).correlate(Resource).label('egress_client_id'))
QosTotal.server_name = column_property(select([VoipGateway.name]).where(and_(VoipGateway.lan_ip == QosTotal.server_ip,
                                       VoipGateway.lan_port == QosTotal.server_port)).correlate_except(VoipGateway).limit(1).label('server_name'))
QosResource.server_name = column_property(select([VoipGateway.name]).where(and_(VoipGateway.lan_ip == QosResource.server_ip,
                                       VoipGateway.lan_port == QosResource.server_port)).correlate_except(VoipGateway).limit(1).label('server_name'))
# QosResource.ingress_client_id = column_property(select([Resource.client_id]). \
#                     where(Resource.resource_id == QosResource.ingress_id).correlate(Resource).limit(1).label('ingress_client_id'))
# QosResource.egress_client_id = column_property(select([Resource.client_id]). \
#                     where(Resource.resource_id == QosResource.egress_id).correlate(Resource).limit(1).label('egress_client_id'))
# t_qos_ip = Table(
#     'qos_ip', DnlApiBaseModel.metadata,
#     Column('report_time', DateTime(True), index=True,
#            server_default=text_("('now'::text)::timestamp(0) with time zone")),
#     Column('ip_id', Integer),
#     Column('call', Integer),
#     Column('cps', Integer),
#     Column('direction', Integer),
#     Column('server_ip', String(50), index=True),
#     Column('server_port', Integer, index=True),
#     Column('channels', Integer),
#     Index('qos_ip_report_time_ip_id_idx', 'report_time', 'ip_id')
# )

class QosClient(DnlApiBaseModel):
    __tablename__ = 'qos_client'
    DIRECTION = {0: 'ingress', 1: 'egress'}
    report_time = Column(DateTime(True), nullable=False, index=True,
                         server_default=text_("('now'::text)::timestamp(0) with time zone"), primary_key=True)
    client_id = Column(Integer, primary_key=True)
    call = Column(Integer)
    inbound_cps = Column(Integer)
    inbound_chan = Column(Integer)
    outbound_cps = Column(Integer)
    outbound_chan = Column(Integer)
    server_ip = Column(String(50), index=True)
    server_port = Column(Integer, index=True)

    client_name = column_property(
        select([Client.name]).where(Client.client_id == client_id).correlate_except(Client))
    is_orig = column_property(
        select([Client.is_orig]).where(Client.client_id == client_id).correlate_except(Client))
    is_term = column_property(
        select([Client.is_term]).where(Client.client_id == client_id).correlate_except(Client))



class QosIp(DnlApiBaseModel):
    __tablename__ = 'qos_ip'
    DIRECTION = {0: 'ingress', 1: 'egress'}
    report_time = Column(DateTime(True), nullable=False, index=True,
                         server_default=text_("('now'::text)::timestamp(0) with time zone"), primary_key=True)
    ip_id = Column(Integer, primary_key=True)
    call = Column(Integer)
    cps = Column(Integer)
    direction = Column(ChoiceType(DIRECTION))
    server_ip = Column(String(50), index=True)
    server_port = Column(Integer, index=True)
    channels = Column(Integer)
    qos_resource_report_time_res_id_idx = Index('qos_resource_report_time_res_id_idx', 'report_time', 'ip_id')
    res_id = column_property(
        select([ResourceIp.resource_id]).where(ResourceIp.resource_ip_id == ip_id).correlate_except(ResourceIp))
    client_id = column_property(
        select([ResourceIp.client_id]).where(ResourceIp.resource_ip_id == ip_id).correlate_except(ResourceIp))
    res_ip = column_property(
        select([ResourceIp.ip]).where(ResourceIp.resource_ip_id == ip_id).correlate_except(ResourceIp))
    res_name = column_property(
        select([Resource.alias]).where(and_(Resource.resource_id == ResourceIp.resource_id,ResourceIp.resource_ip_id == ip_id)).correlate_except(Resource))
    trunk_type2 = column_property(
        select([Resource.trunk_type2]).where(and_(Resource.resource_id == ResourceIp.resource_id,ResourceIp.resource_ip_id == ip_id)).correlate_except(Resource))
    ingress = column_property(
        select([Resource.ingress]).where(and_(Resource.resource_id == ResourceIp.resource_id,ResourceIp.resource_ip_id == ip_id)).correlate_except(Resource))

    @hybrid_property
    def _res_id(self):
        resource_id = aliased(t_resource_ip_record)
        return select([resource_id.c.resource_id]).where(resource_id.c.resource_ip_id == self.ip_id).limit(1).as_scalar()

    @hybrid_property
    def _trunk_type2(self):
        resource_ip = ResourceIp.filter(ResourceIp.resource_ip_id == self.ip_id).first()
        if resource_ip:
            return self.trunk_type2
        return select([Resource.trunk_type2]).where(Resource.resource_id == self._res_id).as_scalar()

class ResourceBlock(DnlApiBaseModel):
    __tablename__ = 'resource_block'
    ACTION_TYPE = {0: 'manual', 1: 'alert rule', 2: 'dialer rule', 3: 'fraud rule', 4: 'loop rule',
                   5: 'invalid number detecton', 6: 'bad ANI/DNIS detection'}
    BLOCK_TYPE = {1: 'specific trunk', 2: 'specific carrier', 3: 'specific group', 4: 'all'}
    # reject number management
    res_block_id = Column(Integer, primary_key=True)  # serial primary key
    ingress_res_id = Column(ForeignKey('resource.resource_id', ondelete='CASCADE'))  # access gateway
    engress_res_id = Column(ForeignKey('resource.resource_id', ondelete='CASCADE'))  # landing gateway
    digit = Column(PrefixRange, default='')  # prefix_range #matches the prefix
    time_profile_id = Column(ForeignKey('time_profile.time_profile_id', ondelete='SET NULL'), index=True)  # time period
    ingress_client_id = Column(ForeignKey('client.client_id', ondelete='CASCADE'))
    egress_client_id = Column(ForeignKey('client.client_id', ondelete='CASCADE'))
    disable_by_alert = Column(Boolean, default=False)
    ani_prefix = Column(PrefixRange, default='')  # prefix_range
    ani_length = Column(Integer)
    dnis_length = Column(Integer)
    ani_method = Column(Integer)  # 0: <; 1: =; 2:>
    dnis_method = Column(Integer)  # 0: <; 1: =; 2:>
    ani_empty = Column(Boolean, default=False)
    action_type = Column(ChoiceType(ACTION_TYPE), default=0)  # 0 - manual; 1 - automatic
    block_log_id = Column(Integer)
    loop_block_id = Column(Integer)
    ticket_log_id = Column(Integer)
    code_name = Column(String)
    country = Column(String)
    create_by = Column(String)
    create_time = Column(DateTime(True),# default=datetime.now(UTC),
                         server_default=text_("('now'::text)::timestamp(0) with time zone"))
    update_by = Column(String)
    dialer_detection_id = Column(Integer)
    ani_max_length = Column(Integer,
                            default=32)  # used with ani_prefix, ani_length (minimum length), opening and closing.
    dnis_max_length = Column(Integer, default=32)  # used with digit, dnis_length (minimum length), opening and closing.
    ingress_group_id = Column(Integer)
    egress_group_id = Column(Integer)
    full_match = Column(Integer, default=0)
    unblock_at = Column(Numeric)
    block_at = Column(Numeric)
    unblock_after = Column(Numeric)
    block_type = Column(ChoiceType(BLOCK_TYPE), default=4)
    dnis_block_type = Column(ChoiceType(BLOCK_TYPE), default=0)

    block_by = synonym('create_by')
    ingress_trunk_id = synonym('ingress_res_id')
    egress_trunk_id = synonym('engress_res_id')

    ANI_empty = synonym('ani_empty')
    ANI_prefix = synonym('ani_prefix')
    DNIS_prefix = synonym('digit')
    ANI_min = synonym('ani_length')
    ANI_max = synonym('ani_max_length')
    DNIS_min = synonym('dnis_length')
    DNIS_max = synonym('dnis_max_length')

    block_type__ = column_property(
        case([(or_(ingress_res_id.isnot(None), engress_res_id.isnot(None)), 'specific trunk'),
              (or_(ingress_client_id.isnot(None), egress_client_id.isnot(None)), 'specific carrier'),
              (or_(ingress_group_id.isnot(None), egress_group_id.isnot(None)), 'specific group'),
              ], else_='all'))

    time_profile_name = column_property(
        select([TimeProfile.name]).where(TimeProfile.time_profile_id == time_profile_id).correlate_except(TimeProfile))
    ingress_trunk_name = column_property(
        select([Resource.alias]).where(Resource.resource_id == ingress_res_id).correlate_except(Resource))
    egress_trunk_name = column_property(
        select([Resource.alias]).where(Resource.resource_id == engress_res_id).correlate_except(Resource))

    ingress_client_name = column_property(
        select([Client.name]).where(or_(and_(Client.client_id == ingress_client_id, ingress_res_id.is_(None)),
                                        and_(Client.client_id == Resource.client_id,
                                             Resource.resource_id == ingress_res_id))).limit(1).correlate_except(Client,
                                                                                                                 Resource))
    egress_client_name = column_property(
        select([Client.name]).where(or_(and_(Client.client_id == egress_client_id, engress_res_id.is_(None)),
                                        and_(Client.client_id == Resource.client_id,
                                             Resource.resource_id == engress_res_id))).limit(1).correlate_except(Client,
                                                                                                                 Resource))
    fake_ingress_client_id = column_property(
        select([Client.client_id]).where(or_(and_(Client.client_id == ingress_client_id, ingress_res_id.is_(None)),
                                             and_(Client.client_id == Resource.client_id,
                                                  Resource.resource_id == ingress_res_id))).limit(1).correlate_except(
            Client,
            Resource))
    fake_egress_client_id = column_property(
        select([Client.client_id]).where(or_(and_(Client.client_id == egress_client_id, engress_res_id.is_(None)),
                                             and_(Client.client_id == Resource.client_id,
                                                  Resource.resource_id == engress_res_id))).limit(1).correlate_except(
            Client,
            Resource))

    ingress_group_name = column_property(
        select([TrunkGroup.group_name]).where(
            or_(and_(TrunkGroup.group_id == ingress_group_id, ingress_res_id.is_(None)),
                and_(TrunkGroup.group_id == Resource.group_id,
                     Resource.resource_id == ingress_res_id))).limit(1).correlate_except(TrunkGroup, Resource))

    egress_group_name = column_property(
        select([TrunkGroup.group_name]).where(
            or_(and_(TrunkGroup.group_id == egress_group_id, engress_res_id.is_(None)),
                and_(TrunkGroup.group_id == Resource.group_id,
                     Resource.resource_id == engress_res_id))).limit(1).correlate_except(TrunkGroup, Resource))

    ingress_trunk = relationship(IngressTrunk, uselist=False,
                                 primaryjoin='ResourceBlock.ingress_res_id == IngressTrunk.resource_id')
    egress_trunk = relationship(EgressTrunk, uselist=False,
                                primaryjoin='ResourceBlock.engress_res_id == EgressTrunk.resource_id')
    ingress_carrier = relationship(Client, uselist=False,
                                   primaryjoin='ResourceBlock.ingress_client_id == Client.client_id')
    egress_carrier = relationship(Client, uselist=False,
                                  primaryjoin='ResourceBlock.egress_client_id == Client.client_id')

    ingress_group = relationship(TrunkGroup, uselist=False,
                                 primaryjoin='foreign(ResourceBlock.ingress_group_id) == TrunkGroup.group_id')
    egress_group = relationship(TrunkGroup, uselist=False,
                                primaryjoin='foreign(ResourceBlock.egress_group_id) == TrunkGroup.group_id')

    # time_profile = relationship(TimeProfile)
    """
    def egress_trunk_name(self):
        try:
            return self.egress_trunk.alias
        except:
            return None
    def ingress_trunk_name(self):
        try:
            return self.ingress_trunk.alias
        except:
            return None

    #def time_profile_name(self):
    #    return self.time_profile.name

    def egress_group_name(self):
        try:
            return self.egress_group.group_name
        except:
            return None
    def ingress_group_name(self):
        try:
            return self.ingress_group.group_name
        except:
            return None
    def egress_client_name(self):
        try:
            return self.egress_carrier.name
        except:
            return None
    def ingress_client_name(self):
        try:
            return self.ingress_carrier.name
        except:
            return None
    """

    @hybrid_property
    def block_type_(self):
        if self.egress_group_id or self.ingress_group_id:
            return 'specific group'
        if self.engress_res_id or self.ingress_res_id:
            return 'specific trunk'
        if self.egress_client_id or self.ingress_client_id:
            return 'specific carrier'
        return 'all'

    @block_type_.setter
    def block_type_(self, value):
        if value == 'specific group':
            self.engress_res_id = None
            self.ingress_res_id = None
            self.egress_client_id = None
            self.ingress_client_id = None
        if value == 'specific trunk':
            self.egress_group_id = None
            self.ingress_group_id = None
            self.egress_client_id = None
            self.ingress_client_id = None
        if value == 'specific carrier':
            self.engress_res_id = None
            self.ingress_res_id = None
            self.egress_group_id = None
            self.ingress_group_id = None
        if value == 'all':
            self.engress_res_id = None
            self.ingress_res_id = None
            self.egress_group_id = None
            self.ingress_group_id = None
            self.egress_client_id = None
            self.ingress_client_id = None

    def before_save(self):
        self.block_type = self.block_type_

    @classmethod
    def on_import_delete(cls, rec, mode):
        # if mode == 'exact':
        if rec.ani_prefix is None:
            rec.ani_prefix = ''
        if rec.digit is None:
            rec.digit = ''
        """ AND resource_block. = 834 AND resource_block. = 815 
        AND resource_block. = 650 AND resource_block. = 668 AND 
        resource_block. IS NULL AND resource_block. IS NULL;"""
        ret = cls.filter(and_(cls.ani_prefix == rec.ani_prefix, cls.digit == rec.digit,
                              cls.egress_client_id == rec.egress_client_id,
                              cls.ingress_client_id == rec.ingress_client_id,
                              cls.ingress_res_id == rec.ingress_res_id, cls.engress_res_id == rec.engress_res_id,
                              cls.ingress_group_id == rec.ingress_group_id,
                              cls.egress_group_id == rec.egress_group_id)).delete(synchronize_session='fetch')
        if ret:
            cls.session().commit()


class CdrDownloadTask(DnlApiBaseModel):
    __tablename__ = 'cdr_download_task'
    TYPE = {0: 'manual', 1: 'auto'}
    request_id = Column(String(16), primary_key=True)
    cached = Column(Boolean)
    code = Column(Integer)
    count = Column(Integer)
    download_link = Column(String(32))
    end_time = Column(Integer)
    expiration_time = Column(Integer)
    job_end_time = Column(Integer)
    job_id = Column(Integer)
    job_start_time = Column(Integer)
    msg = Column(String(512))
    progress = Column(String(4))
    size = Column(Integer)
    start_time = Column(Integer)
    status = Column(String(32))
    email_to = Column(String(512))
    orig_file = Column(String(512))
    mail_status = Column(String(512))
    method = Column(String(32))
    client_id = Column(ForeignKey('client.client_id', ondelete='CASCADE'))
    sender_id = Column(ForeignKey('mail_sender.id', ondelete='CASCADE'))
    subject = Column(String(512))
    content = Column(Text)
    mail_cc = Column(String(512))
    type = Column(ChoiceType(TYPE), default=0)

    client = relationship(Client, uselist=False,  # back_populates='low_balance_config',
                          enable_typechecks=False)

    ##direct = relationship('SendCdrDirect', uselist=False, single_parent=True, cascade="all, delete-orphan")

    # @property
    # def email(self):
    #    return self.email_to

    @property
    def file_name(self):
        if self.orig_file:
            return '{}/{}'.format(settings.FILES['upload_to'], self.orig_file)
        return None

    @property
    def file_data(self):
        if self.orig_file:
            return open(self.file_name, 'rb').read()
        return None

    @property
    def fmt_start_time(self):
        return datetime.fromtimestamp(self.start_time, UTC)

    @property
    def fmt_end_time(self):
        return datetime.fromtimestamp(self.end_time, UTC)

    @property
    def fmt_download_link(self):
        return '{}/tool/cdr_task/{}/cdr_{}.xls'.format(settings.API_URL, self.request_id,
                                                       self.fmt_start_time.date())


def generate_request_id():
    def gen():
        import random
        letters = [chr(i) for i in range(97, 123)] + [chr(i) for i in range(65, 91)]
        digits = [chr(i) for i in range(48, 58)]
        ret = ''
        for c in range(0, 16):
            ret = ret + random.choice(letters + digits)
        return ret.capitalize()

    return gen


class CdrAsyncTask(DnlApiBaseModel):
    __tablename__ = 'cdr_async_task'
    TYPE = {0: 'manual', 1: 'auto', 2: 'download'}
    STATUS = {0: 'initial', 1: 'finished', 2: 'running', 3: 'success', 4: 'fail'}
    METHOD = {0: 'create', 1: 'upload', 2: 'email'}
    request_id = Column(String(16), primary_key=True, default=generate_request_id())
    created_on = Column(DateTime(True), server_default=func.now())
    request_client_id = Column(ForeignKey('client.client_id', ondelete='CASCADE'))
    user_id = Column(ForeignKey('users.user_id', ondelete='CASCADE'))
    fields = Column(String())
    filter = Column(JSON)
    download_link = Column(String(512))
    expiration_time = Column(Integer)
    job_start_time = Column(Integer)
    job_end_time = Column(Integer)
    job_id_seq = Sequence('cdr_async_task_job_id_seq')
    # job_id = Column(Integer,server_default=text_("nextval('cdr_async_task_job_id_seq'::regclass)"))
    job_id = Column(Integer, server_default=job_id_seq.next_value())
    msg = Column(String(512))
    progress = Column(String(4))
    count = Column(Integer)
    size = Column(BigInteger)
    status = Column(String(32))
    email = Column(String(512))
    orig_file = Column(String(512))
    mail_status = Column(String(512))
    method = Column(String(32))
    do_zip = Column(Boolean, default = False, server_default='false')
    sender_id = Column(ForeignKey('mail_sender.id', ondelete='CASCADE'))
    subject = Column(String(512))
    content = Column(Text)
    mail_cc = Column(String(512))
    type = Column(ChoiceType(TYPE), server_default='1')

    ftp_url = Column(String(512))  # (s)ftp server url to upload (ftp=ftp://myftpserver.com)
    ftp_port = Column(SmallInteger)  # server port (default: 21 -> ftp; 22 -> sftp)
    ftp_user = Column(String(512))  # server user name (default: Anonymous)
    ftp_password = Column(String(512))  # server password (default: None)
    ftp_dir = Column(String(
        512))  # server path to upload: for ftp path from access point; for sftp -> path from root. (default: "/")
    ftp_file_name = Column(String(
        512))  # filename prefix (70 chars max). Engine will generate name of the file using the following template:
    # [prefix]_[yyyy-mm-dd__hh_mm_ss]_[yyyy-mm-dd__hh_mm_ss]_[breakdown].[ext]
    ftp_max_lines = Column(Integer)  # max lines per file
    ftp_compress = Column(String(512))  # compress file. Allowed formats: gz, tar.gz, tar.bz2
    ftp_include_headers = Column(
        SmallInteger)  # include headers line in csv file: 0 -> do not include; 1 -> include. Default: 1 -> with headers
    ftp_file_breakdown = Column(
        SmallInteger)  # breakdown results: 0","descripton":"as one file; 1 -> as hourly file; 2 -> as daily file. Default: 0 -> single file

    client = relationship(Client, uselist=False,  # back_populates='low_balance_config',
                          enable_typechecks=False)
    user = relationship(User, uselist=False)
    client_id = synonym('request_client_id')
    cdr_count = synonym('count')

    direct = relationship('SendCdrDirect', uselist=False, single_parent=True, cascade="all, delete-orphan")
    # customer_gmt = column_property(
    #     select([Client.auto_send_zone]).where(Client.client_id == request_client_id).correlate_except(Client))
    client_name = column_property(
        select([Client.client_name]).where(Client.client_id == request_client_id).correlate_except(Client))
    company = column_property(
        select([Client.company]).where(Client.client_id == request_client_id).correlate_except(Client))

    # @property
    # def email(self):
    #    return self.email_to
    @hybrid_property
    def start_time(self):
        try:
            return cast(self.filter['start_time'], String)
        except:
            return cast('', String)



    @hybrid_property
    def end_time(self):
        try:
            return cast(self.filter['end_time'], String)
        except:
            return cast('', String)



    @property
    def _download_link(self):
        if self.orig_file:
            conf = model.SystemParameter.get(1)
            if conf.base_url:
                return '{}/files/{}'.format(conf.base_url,
                                              self.orig_file)
            return '{}{}/files/{}'.format(settings.API_SCHEME, settings.API_HOST,
                                          self.orig_file)
        return None

    @property
    def file_name(self):
        if self.orig_file:
            return '{}/{}'.format(settings.FILES['upload_to'], self.orig_file)
        return None

    @property
    def file_name_xls(self):
        return '{}/cdr_{}.xls'.format(settings.FILES['upload_to'], self.request_id)

    @property
    def file_data(self):
        if self.orig_file:
            return open(self.file_name, 'rb').read()
        return None

    @property
    def fmt_start_time(self):
        if 'start_time' in self.filter:
            return datetime.fromtimestamp(int(self.filter['start_time']), UTC)
        return None

    @property
    def fmt_end_time(self):
        if 'end_time' in self.filter:
            return datetime.fromtimestamp(int(self.filter['end_time']), UTC)
        return None

    @property
    def fmt_start_date(self):
        return self.fmt_start_time

    @property
    def fmt_end_date(self):
        return self.fmt_end_time

    @property
    def fmt_current_date(self):
        return str(datetime.utcnow().date())

    @property
    def fmt_download_link(self):
        return self._download_link
        # return '{}/files/cdr_{}.xls'.format(settings.FILE_URL,
        #                                     self.request_id)

    @property
    def fmt_cdr_url(self):
        return self.fmt_download_link


    @property
    def fmt_current_day(self):
        return str(datetime.now(UTC))

    @property
    def fmt_company(self):
        if self.company:
            return self.company
        if self.user_id:
            client_id = model.User.get(self.user_id).client_id
            if client_id:
                client = model.Client.get(client_id)
                return client.company
        return ''

    @property
    def fmt_client_name(self):
        if self.client_name:
            return self.client_name
        if self.user_id:
            client_id = model.User.get(self.user_id).client_id
            if client_id:
                client = model.Client.get(client_id)
                return client.client_name
        return ''


def cdr_reports_start_date():
    tablename = 'cdr_report_detail2%'
    try:
        r = get_db().session.execute(
            text_("select min(relname) as tablename from pg_class where relname like '{}';".format(
                tablename)))
        ret = r.fetchone()
        if not ret or not ret.tablename:
            return None
        tn = ret.tablename
        return datetime(int(tn[17:21]), int(tn[21:23]), int(tn[23:25])).date()
    except:
        return None


def cdr_reports_daily_start_date():
    tablename = 'cdr_report_daily2%'
    try:
        r = get_db().session.execute(
            text_("select min(relname) as tablename from pg_class where relname like '{}';".format(
                tablename)))
        ret = r.fetchone()
        if not ret or not ret.tablename:
            return None
        tn = ret.tablename
        return datetime(int(tn[17:21]), int(tn[21:23]), int(tn[23:25])).date()
    except:
        return None


def cdr_reports_iterator(start_date_time, end_date_time):
    start_day = start_date_time.date()
    database_start_day = cdr_reports_start_date()
    if database_start_day is None:
        raise ValidationError('database has no cdr report data!')
    if database_start_day > start_day:
        start_day = database_start_day
    end_day = end_date_time.date()
    if end_day < start_day:
        end_day = start_day
    days = (end_day - start_day).days
    for day_idx in range(days + 1):
        day_cdr = start_day + timedelta(days=day_idx)
        cdr_date = "%s%02d%02d" % (day_cdr.year, day_cdr.month, day_cdr.day)
        cdr = cdr_report_detail(cdr_date)
        yield cdr


def client_cdr_iterator(start_date_time, end_date_time):
    start_day = start_date_time.date()
    database_start_day = client_cdr_start_date()
    if database_start_day is None:
        raise ValidationError('database has no cdr report data!')
    if database_start_day > start_day:
        start_day = database_start_day
    end_day = end_date_time.date()
    if end_day < start_day:
        end_day = start_day
    days = (end_day - start_day).days
    for day_idx in range(days + 1):
        day_cdr = start_day + timedelta(days=day_idx)
        cdr_date = "%s%02d%02d" % (day_cdr.year, day_cdr.month, day_cdr.day)
        cdr = client_cdr(cdr_date)
        yield cdr


class HostBasedReport(DnlApiBaseModel):
    __tablename__ = 'host_based_report'
    report_time = Column(DateTime(True))
    duration = Column(Integer)
    ingress_bill_time = Column(Integer)
    ingress_call_cost = Column(Numeric(15, 6))
    ingress_total_calls = Column(Integer)
    not_zero_calls = Column(Integer)
    ingress_success_calls = Column(Integer)
    ingress_busy_calls = Column(Integer)
    pdd = Column(BigInteger)
    ingress_cancel_calls = Column(Integer)
    ingress_client_id = Column(Integer)
    egress_client_id = Column(Integer)
    egress_bill_time = Column(Integer)
    egress_call_cost = Column(Numeric(15, 6))
    egress_total_calls = Column(Integer)
    egress_success_calls = Column(Integer)
    egress_busy_calls = Column(Integer)
    egress_cancel_calls = Column(Integer)
    ingress_ip = Column(String(50))
    egress_ip = Column(String(50))
    ingress_avg_rate = Column(Float(53))
    egress_avg_rate = Column(Float(53))
    cdr_date = Column(String(24))
    ingress_duration = Column(Integer)
    ingress_not_zero_calls = Column(Integer)
    __table_args__ = (PrimaryKeyConstraint(
        'ingress_client_id', 'ingress_ip', 'egress_client_id', 'egress_ip', 'report_time'
    ),)


def host_based_report(date):
    tablename = 'host_based_report{}'.format(date)
    try:
        r = get_db().session.execute(text_("select tablename from pg_tables where tablename='{}';".format(tablename)))
        if not r.fetchall()[0][0] == tablename:
            return None
    except:
        return None
    return Table(
        tablename, DnlApiBaseModel.metadata,
        Column('report_time', DateTime(True)),
        Column('duration', Integer),
        Column('ingress_bill_time', Integer),
        Column('ingress_call_cost', Numeric(15, 6)),
        Column('ingress_total_calls', Integer),
        Column('not_zero_calls', Integer),
        Column('ingress_success_calls', Integer),
        Column('ingress_busy_calls', Integer),
        Column('pdd', BigInteger),
        Column('ingress_cancel_calls', Integer),
        Column('ingress_client_id', Integer),
        Column('egress_client_id', Integer),
        Column('egress_bill_time', Integer),
        Column('egress_call_cost', Numeric(15, 6)),
        Column('egress_total_calls', Integer),
        Column('egress_success_calls', Integer),
        Column('egress_busy_calls', Integer),
        Column('egress_cancel_calls', Integer),
        Column('ingress_ip', String(50)),
        Column('egress_ip', String(50)),
        Column('ingress_avg_rate', Float(53)),
        Column('egress_avg_rate', Float(53)),
        Column('cdr_date', String(24)),
        Column('ingress_duration', Integer),
        Column('ingress_not_zero_calls', Integer),
        extend_existing=True,
    )


def cdr_report_detail(date):
    tablename = 'cdr_report_detail{}'.format(date)
    try:
        r = get_db().session.execute(
            text_("select relname from pg_class where relname='{}';".format(tablename)))
        if not r.fetchall()[0][0] == tablename:
            return None
    except Exception as e:
        return None
    return Table(
        tablename, DnlApiBaseModel.metadata,
        Column('report_time', DateTime(True)),
        Column('duration', Integer),
        Column('ingress_bill_time', Integer),
        Column('ingress_call_cost', Numeric(15, 6)),
        Column('lnp_cost', Numeric(15, 8)),
        Column('ingress_total_calls', Integer),
        Column('not_zero_calls', Integer),
        Column('ingress_success_calls', Integer),
        Column('ingress_busy_calls', Integer),
        Column('lrn_calls', Integer),
        Column('pdd', BigInteger),
        Column('ingress_cancel_calls', Integer),
        Column('ingress_client_id', Integer),
        Column('ingress_id', Integer),
        Column('ingress_country', String(100)),
        Column('ingress_code_name', String(100)),
        Column('egress_client_id', Integer),
        Column('egress_id', Integer),
        Column('egress_country', String(100)),
        Column('egress_code_name', String(100)),
        Column('egress_code', String(100)),
        Column('ingress_prefix', String(100)),
        Column('egress_bill_time', Integer),
        Column('egress_call_cost', Numeric(15, 6)),
        Column('egress_total_calls', Integer),
        Column('egress_success_calls', Integer),
        Column('egress_busy_calls', Integer),
        Column('egress_cancel_calls', Integer),
        Column('not_zero_calls_30', Integer),
        Column('duration_30', Integer),
        Column('not_zero_calls_6', Integer),
        Column('duration_6', Integer),
        Column('call_18s', Integer),
        Column('call_24s', Integer),
        Column('call_2h', Integer),
        Column('call_3h', Integer),
        Column('call_4h', Integer),
        Column('call_12s', Integer),
        Column('ingress_rate', Float(53)),
        Column('egress_rate', Float(53)),
        Column('product_rout_id', Integer),
        Column('ingress_rate_date', BigInteger),
        Column('egress_rate_date', BigInteger),
        Column('incoming_bandwidth', Integer),
        Column('outgoing_bandwidth', Integer),
        Column('ingress_call_cost_intra', Numeric(15, 6)),
        Column('ingress_call_cost_inter', Numeric(15, 6)),
        Column('egress_call_cost_intra', Numeric(15, 6)),
        Column('egress_call_cost_inter', Numeric(15, 6)),
        Column('ingress_bill_time_intra', Integer),
        Column('ingress_bill_time_inter', Integer),
        Column('egress_bill_time_intra', Integer),
        Column('egress_bill_time_inter', Integer),
        Column('agent_id', Integer),
        Column('agent_rate', Float(53)),
        Column('agent_cost', Numeric(12, 6)),
        Column('ingress_rate_table_id', Integer),
        Column('route_plan_id', Integer),
        Column('orig_jur_type', Integer),
        Column('term_jur_type', Integer),
        Column('par_id', Integer),
        Column('origination_destination_host_name', String(255)),
        Column('termination_source_host_name', String(255)),
        Column('binary_value_of_release_cause_from_protocol_stack', String(100)),
        Column('release_cause', SmallInteger),
        Column('release_cause_from_protocol_stack', String(100)),
        Column('inter_ingress_total_calls', Integer),
        Column('intra_ingress_total_calls', Integer),
        Column('inter_duration', Integer),
        Column('intra_duration', Integer),
        Column('inter_not_zero_calls', Integer),
        Column('intra_not_zero_calls', Integer),
        Column('q850_cause_count', Integer),
        Column('npr_count', Integer),
        Column('ingress_call_cost_local', Numeric(15, 6)),
        Column('ingress_call_cost_ij', Numeric(15, 6)),
        Column('egress_call_cost_local', Numeric(15, 6)),
        Column('egress_call_cost_ij', Numeric(15, 6)),
        Column('cdr_date', String(24)),
        Column('ingress_code', String(100)),
        Column('ring_pdd', Integer),
        Column('egress_no_ring', Integer),
        Column('ingress_duration', Integer),
        Column('ingress_inter_duration', Integer),
        Column('ingress_intra_duration', Integer),
        Column('ingress_inter_not_zero_calls', Integer),
        Column('ingress_intra_not_zero_calls', Integer),
        Column('ingress_not_zero_calls', Integer),
        Column('ingress_not_zero_calls_30', Integer),
        Column('ingress_duration_30', Integer),
        Column('ingress_not_zero_calls_6', Integer),
        Column('ingress_duration_6', Integer),
        Column('ingress_call_18s', Integer),
        Column('ingress_call_24s', Integer),
        Column('ingress_call_2h', Integer),
        Column('ingress_call_3h', Integer),
        Column('ingress_call_4h', Integer),
        Column('ingress_call_12s', Integer),
        Column('src_ocn', String(100)),
        Column('src_lata', Integer),
        Column('dst_ocn', String(100)),
        Column('dst_lata', Integer),
        Column('lrn_ocn', String(100)),
        Column('lrn_lata', Integer),
        Column('origination_source_host_name', String(255)),
        Column('termination_destination_host_name', String(255)),
        Column('orig_shaken_ocn', Text),
        Column('orig_shaken_lvl', String(1)),
        Column('term_shaken_ocn', Text),
        Column('term_shaken_lvl', String(1)),
        # Column('ani_in_dno', Integer),
        # Column('ani_in_ftc', Integer),
        # Column('ani_in_ftc_week', Integer),
        # Column('ani_in_ftc_month', Integer),
        # Column('ani_in_spam', Integer),
        # Column('ani_in_fraud', Integer),
        # Column('ani_in_tcpa', Integer),
        # Column('dnis_in_dnc', Integer),
        Column('ani_in_dno_count', Integer),
        Column('ani_in_ftc_count', Integer),
        Column('ani_in_ftc_week_count', Integer),
        Column('ani_in_ftc_month_count', Integer),
        Column('ani_in_spam_count', Integer),
        Column('ani_in_fraud_count', Integer),
        Column('ani_in_tcpa_count', Integer),
        Column('dnis_in_dnc_count', Integer),
        Column('origination_remote_payload_ip_address', Text),
        Column('termination_remote_payload_ip_address', Text),
        extend_existing=True,
    )


def cdr_report_daily(date):
    tablename = 'cdr_report_daily{}'.format(date)
    try:
        r = get_db().session.execute(
            text_("select relname from pg_class where relname='{}';".format(tablename)))
        if not r.fetchall()[0][0] == tablename:
            return None
    except Exception as e:
        return None
    return Table(
        tablename, DnlApiBaseModel.metadata,
        Column('id', BigInteger),
        Column('cdr_date', String),
        Column('report_time', DateTime(True)),
        Column('ingress_client_id', Integer),
        Column('ingress_id', Integer),
        Column('egress_client_id', Integer),
        Column('egress_id', Integer),
        Column('ingress_code_name', String),
        Column('egress_code_name', String),
        Column('ingress_total_calls', Integer),
        Column('ingress_success_calls', Integer),
        Column('ingress_busy_calls', Integer),
        Column('ingress_cancel_calls', Integer),
        Column('ingress_bill_time', Integer),
        Column('ingress_bill_time_intra', Integer),
        Column('ingress_bill_time_inter', Integer),
        Column('ingress_call_cost', Float),
        Column('ingress_call_cost_intra', Float),
        Column('ingress_call_cost_inter', Float),
        Column('ingress_call_cost_local', Float),
        Column('ingress_call_cost_ij', Float),
        Column('ingress_duration', Integer),
        Column('ingress_duration_6', Integer),
        Column('ingress_duration_30', Integer),
        Column('ingress_inter_duration', Integer),
        Column('ingress_intra_duration', Integer),
        Column('ingress_not_zero_calls', Integer),
        Column('ingress_not_zero_calls_6', Integer),
        Column('ingress_not_zero_calls_30', Integer),
        Column('ingress_inter_not_zero_calls', Integer),
        Column('ingress_intra_not_zero_calls', Integer),
        Column('ingress_call_12s', Integer),
        Column('ingress_call_18s', Integer),
        Column('ingress_call_24s', Integer),
        Column('ingress_call_2h', Integer),
        Column('ingress_call_3h', Integer),
        Column('ingress_call_4h', Integer),
        Column('egress_total_calls', Integer),
        Column('egress_success_calls', Integer),
        Column('egress_busy_calls', Integer),
        Column('egress_cancel_calls', Integer),
        Column('egress_bill_time', Integer),
        Column('egress_bill_time_intra', Integer),
        Column('egress_bill_time_inter', Integer),
        Column('egress_call_cost', Float),
        Column('egress_call_cost_intra', Float),
        Column('egress_call_cost_inter', Float),
        Column('egress_call_cost_local', Float),
        Column('egress_call_cost_ij', Float),
        Column('duration', Integer),
        Column('duration_6', Integer),
        Column('duration_30', Integer),
        Column('inter_duration', Integer),
        Column('intra_duration', Integer),
        Column('not_zero_calls', Integer),
        Column('not_zero_calls_6', Integer),
        Column('not_zero_calls_30', Integer),
        Column('inter_not_zero_calls', Integer),
        Column('intra_not_zero_calls', Integer),
        Column('call_12s', Integer),
        Column('call_18s', Integer),
        Column('call_24s', Integer),
        Column('call_2h', Integer),
        Column('call_3h', Integer),
        Column('call_4h', Integer),
        Column('agent_cost', Float),
        Column('inter_ingress_total_calls', Integer),
        Column('intra_ingress_total_calls', Integer),
        Column('npr_count', Integer),
        Column('egress_no_ring', Integer),
        Column('pdd', Integer),
        Column('q850_cause_count', Integer),
        Column('orig_shaken_ocn', Text),
        Column('orig_shaken_lvl', String(1)),
        Column('term_shaken_ocn', Text),
        Column('term_shaken_lvl', String(1)),
        Column('ani_in_dno', Integer),
        Column('ani_in_ftc', Integer),
        Column('ani_in_ftc_week', Integer),
        Column('ani_in_ftc_month', Integer),
        Column('ani_in_spam', Integer),
        Column('ani_in_fraud', Integer),
        Column('ani_in_tcpa', Integer),
        Column('dnis_in_dnc', Integer),
        Column('ani_in_dno_count', Integer),
        Column('ani_in_ftc_count', Integer),
        Column('ani_in_ftc_week_count', Integer),
        Column('ani_in_ftc_month_count', Integer),
        Column('ani_in_spam_count', Integer),
        Column('ani_in_fraud_count', Integer),
        Column('ani_in_tcpa_count', Integer),
        Column('dnis_in_dnc_count', Integer),
        Column('pdd_count', BigInteger),
        Column('origination_remote_payload_ip_address', Text),
        Column('termination_remote_payload_ip_address', Text),
        extend_existing=True,
    )


def code_report_daily(date):
    tablename = 'code_report_daily{}'.format(date)
    try:
        r = get_db().session.execute(
            text_("select relname from pg_class where relname='{}';".format(tablename)))
        if not r.fetchall()[0][0] == tablename:
            return None
    except Exception as e:
        return None
    return Table(
        tablename, DnlApiBaseModel.metadata,
        Column('id', BigInteger),
        Column('cdr_date', String),
        Column('report_time', DateTime(True)),
        Column('code', String),
        Column('ingress_client_id', Integer),
        Column('ingress_id', Integer),
        Column('egress_client_id', Integer),
        Column('egress_id', Integer),
        Column('not_zero_calls', Integer),
        Column('total_calls', Integer),
        Column('ingress_not_zero_calls', Integer),
        Column('ingress_total_calls', Integer),
        extend_existing=True,
    )


def code_report(date):
    tablename = 'code_report{}'.format(date)
    try:
        r = get_db().session.execute(
            text_("select relname from pg_class where relname='{}';".format(tablename)))
        if not r.fetchall()[0][0] == tablename:
            return None
    except Exception as e:
        return None
    return Table(
        tablename, DnlApiBaseModel.metadata,
        Column('id', BigInteger),
        Column('cdr_date', String),
        Column('report_time', DateTime(True)),
        Column('code', String),
        Column('ingress_client_id', Integer),
        Column('ingress_id', Integer),
        Column('egress_client_id', Integer),
        Column('egress_id', Integer),
        Column('not_zero_calls', Integer),
        Column('total_calls', Integer),
        Column('ingress_not_zero_calls', Integer),
        Column('ingress_total_calls', Integer),
        extend_existing=True,
    )


class CrdReportDetailTable(object):
    report_time = Column(DateTime(timezone=True))
    duration = Column(Integer)
    ingress_bill_time = Column(Integer)
    ingress_call_cost = Column(Numeric(15, 6))
    lnp_cost = Column(Numeric(15, 8))
    ingress_total_calls = Column(Integer)
    not_zero_calls = Column(Integer)
    ingress_success_calls = Column(Integer)
    ingress_busy_calls = Column(Integer)
    lrn_calls = Column(Integer)
    pdd = Column(BigInteger)
    ingress_cancel_calls = Column(Integer)
    ingress_client_id = Column(Integer)
    ingress_id = Column(Integer)
    ingress_country = Column(String(100))
    ingress_code_name = Column(String(100))
    egress_client_id = Column(Integer)
    egress_id = Column(Integer)
    egress_country = Column(String(100))
    egress_code_name = Column(String(100))
    egress_code = Column(String(100))
    ingress_prefix = Column(String(100))
    egress_bill_time = Column(Integer)
    egress_call_cost = Column(Numeric(15, 6))
    egress_total_calls = Column(Integer)
    egress_success_calls = Column(Integer)
    egress_busy_calls = Column(Integer)
    egress_cancel_calls = Column(Integer)
    not_zero_calls_30 = Column(Integer)
    duration_30 = Column(Integer)
    not_zero_calls_6 = Column(Integer)
    duration_6 = Column(Integer)
    call_18s = Column(Integer)
    call_24s = Column(Integer)
    call_2h = Column(Integer)
    call_3h = Column(Integer)
    call_4h = Column(Integer)
    call_12s = Column(Integer)
    ingress_rate = Column(Float(53))
    egress_rate = Column(Float(53))
    product_rout_id = Column(Integer)
    ingress_rate_date = Column(BigInteger)
    egress_rate_date = Column(BigInteger)
    incoming_bandwidth = Column(Integer)
    outgoing_bandwidth = Column(Integer)
    ingress_call_cost_intra = Column(Numeric(15, 6))
    ingress_call_cost_inter = Column(Numeric(15, 6))
    egress_call_cost_intra = Column(Numeric(15, 6))
    egress_call_cost_inter = Column(Numeric(15, 6))
    ingress_bill_time_intra = Column(Integer)
    ingress_bill_time_inter = Column(Integer)
    egress_bill_time_intra = Column(Integer)
    egress_bill_time_inter = Column(Integer)
    agent_id = Column(Integer)
    agent_rate = Column(Float(53))
    agent_cost = Column(Numeric(12, 6))
    ingress_rate_table_id = Column(Integer)
    route_plan_id = Column(Integer)
    orig_jur_type = Column(Integer)
    term_jur_type = Column(Integer)
    par_id = Column(Integer)
    origination_destination_host_name = Column(String(255))
    termination_source_host_name = Column(String(255))
    binary_value_of_release_cause_from_protocol_stack = Column(String(100))
    release_cause = Column(SmallInteger)
    release_cause_from_protocol_stack = Column(String(100))
    inter_ingress_total_calls = Column(Integer)
    intra_ingress_total_calls = Column(Integer)
    inter_duration = Column(Integer)
    intra_duration = Column(Integer)
    inter_not_zero_calls = Column(Integer)
    intra_not_zero_calls = Column(Integer)
    q850_cause_count = Column(Integer)
    npr_count = Column(Integer)
    ingress_call_cost_local = Column(Numeric(15, 6))
    ingress_call_cost_ij = Column(Numeric(15, 6))
    egress_call_cost_local = Column(Numeric(15, 6))
    egress_call_cost_ij = Column(Numeric(15, 6))
    cdr_date = Column(String(24))
    ingress_code = Column(String(100))



class CdrReportDaily(DnlApiBaseModel):
    __tablename__ = 'cdr_report_daily'

    id = Column(BigInteger)
    cdr_date = Column(String())
    report_time = Column(DateTime(True), primary_key=True)
    ingress_client_id = Column(Integer)
    ingress_id = Column(Integer)
    egress_client_id = Column(Integer)
    egress_id = Column(Integer)
    ingress_code_name = Column(String())
    egress_code_name = Column(String())
    ingress_total_calls = Column(Integer)
    ingress_success_calls = Column(Integer)
    ingress_busy_calls = Column(Integer)
    ingress_cancel_calls = Column(Integer)
    ingress_bill_time = Column(Integer)
    ingress_bill_time_intra = Column(Integer)
    ingress_bill_time_inter = Column(Integer)
    ingress_call_cost = Column(Float)
    ingress_call_cost_intra = Column(Float)
    ingress_call_cost_inter = Column(Float)
    ingress_call_cost_local = Column(Float)
    ingress_call_cost_ij = Column(Float)
    ingress_duration = Column(Integer)
    ingress_duration_6 = Column(Integer)
    ingress_duration_30 = Column(Integer)
    ingress_inter_duration = Column(Integer)
    ingress_intra_duration = Column(Integer)
    ingress_not_zero_calls = Column(Integer)
    ingress_not_zero_calls_6 = Column(Integer)
    ingress_not_zero_calls_30 = Column(Integer)
    ingress_inter_not_zero_calls = Column(Integer)
    ingress_intra_not_zero_calls = Column(Integer)
    ingress_call_12s = Column(Integer)
    ingress_call_18s = Column(Integer)
    ingress_call_24s = Column(Integer)
    ingress_call_2h = Column(Integer)
    ingress_call_3h = Column(Integer)
    ingress_call_4h = Column(Integer)
    egress_total_calls = Column(Integer)
    egress_success_calls = Column(Integer)
    egress_busy_calls = Column(Integer)
    egress_cancel_calls = Column(Integer)
    egress_bill_time = Column(Integer)
    egress_bill_time_intra = Column(Integer)
    egress_bill_time_inter = Column(Integer)
    egress_call_cost = Column(Float)
    egress_call_cost_intra = Column(Float)
    egress_call_cost_inter = Column(Float)
    egress_call_cost_local = Column(Float)
    egress_call_cost_ij = Column(Float)
    duration = Column(Integer)
    duration_6 = Column(Integer)
    duration_30 = Column(Integer)
    inter_duration = Column(Integer)
    intra_duration = Column(Integer)
    not_zero_calls = Column(Integer)
    not_zero_calls_6 = Column(Integer)
    not_zero_calls_30 = Column(Integer)
    inter_not_zero_calls = Column(Integer)
    intra_not_zero_calls = Column(Integer)
    call_12s = Column(Integer)
    call_18s = Column(Integer)
    call_24s = Column(Integer)
    call_2h = Column(Integer)
    call_3h = Column(Integer)
    call_4h = Column(Integer)
    agent_cost = Column(Float)
    inter_ingress_total_calls = Column(Integer)
    intra_ingress_total_calls = Column(Integer)
    npr_count = Column(Integer)
    egress_no_ring = Column(Integer)
    q850_cause_count = Column(Integer)
    orig_shaken_ocn = Column(Text)
    orig_shaken_lvl = Column(String(1))
    term_shaken_ocn = Column(Text)
    term_shaken_lvl = Column(String(1))
    ani_in_dno_count = Column(Integer)
    ani_in_ftc_count = Column(Integer)
    ani_in_ftc_week_count = Column(Integer)
    ani_in_ftc_month_count = Column(Integer)
    ani_in_spam_count = Column(Integer)
    ani_in_fraud_count = Column(Integer)
    ani_in_tcpa_count = Column(Integer)
    dnis_in_dnc_count = Column(Integer)
    pdd_count = Column(BigInteger)
    pdd = synonym('pdd_count')
    origination_remote_payload_ip_address = Column(Text)
    termination_remote_payload_ip_address = Column(Text)
    SDP30_calls = column_property(
        call_12s + call_18s + call_24s + not_zero_calls_30
    )
    SDP6_calls = synonym('not_zero_calls_6')

class CdrReportDetail(DnlApiBaseModel):
    __tablename__ = 'cdr_report_detail'

    report_time = Column(DateTime(timezone=True))
    duration = Column(Integer)
    ingress_bill_time = Column(Integer)
    ingress_call_cost = Column(Numeric(15, 6))
    lnp_cost = Column(Numeric(15, 8))
    ingress_total_calls = Column(Integer)
    not_zero_calls = Column(Integer)
    ingress_success_calls = Column(Integer)
    ingress_busy_calls = Column(Integer)
    lrn_calls = Column(Integer)
    pdd = Column(BigInteger)
    ingress_cancel_calls = Column(Integer)
    ingress_client_id = Column(Integer)
    ingress_id = Column(Integer)
    ingress_country = Column(String(100))
    ingress_code_name = Column(String(100))
    egress_client_id = Column(Integer)
    egress_id = Column(Integer)
    egress_country = Column(String(100))
    egress_code_name = Column(String(100))
    egress_code = Column(String(100))
    ingress_prefix = Column(String(100))
    egress_bill_time = Column(Integer)
    egress_call_cost = Column(Numeric(15, 6))
    egress_total_calls = Column(Integer)
    egress_success_calls = Column(Integer)
    egress_busy_calls = Column(Integer)
    egress_cancel_calls = Column(Integer)
    not_zero_calls_30 = Column(Integer)
    duration_30 = Column(Integer)
    not_zero_calls_6 = Column(Integer)
    duration_6 = Column(Integer)
    call_18s = Column(Integer)
    call_24s = Column(Integer)
    call_2h = Column(Integer)
    call_3h = Column(Integer)
    call_4h = Column(Integer)
    call_12s = Column(Integer)
    ingress_rate = Column(Float(53))
    egress_rate = Column(Float(53))
    product_rout_id = Column(Integer)
    ingress_rate_date = Column(BigInteger)
    egress_rate_date = Column(BigInteger)
    incoming_bandwidth = Column(Integer)
    outgoing_bandwidth = Column(Integer)
    ingress_call_cost_intra = Column(Numeric(15, 6))
    ingress_call_cost_inter = Column(Numeric(15, 6))
    egress_call_cost_intra = Column(Numeric(15, 6))
    egress_call_cost_inter = Column(Numeric(15, 6))
    ingress_bill_time_intra = Column(Integer)
    ingress_bill_time_inter = Column(Integer)
    egress_bill_time_intra = Column(Integer)
    egress_bill_time_inter = Column(Integer)
    agent_id = Column(Integer)
    agent_rate = Column(Float(53))
    agent_cost = Column(Numeric(12, 6))
    ingress_rate_table_id = Column(Integer)
    route_plan_id = Column(Integer)
    orig_jur_type = Column(Integer)
    term_jur_type = Column(Integer)
    par_id = Column(Integer)
    origination_destination_host_name = Column(String(255))
    termination_source_host_name = Column(String(255))
    binary_value_of_release_cause_from_protocol_stack = Column(String(100))
    release_cause = Column(SmallInteger)
    release_cause_from_protocol_stack = Column(String(100))
    inter_ingress_total_calls = Column(Integer)
    intra_ingress_total_calls = Column(Integer)
    inter_duration = Column(Integer)
    intra_duration = Column(Integer)
    inter_not_zero_calls = Column(Integer)
    intra_not_zero_calls = Column(Integer)
    q850_cause_count = Column(Integer)
    npr_count = Column(Integer)
    ingress_call_cost_local = Column(Numeric(15, 6))
    ingress_call_cost_ij = Column(Numeric(15, 6))
    egress_call_cost_local = Column(Numeric(15, 6))
    egress_call_cost_ij = Column(Numeric(15, 6))
    cdr_date = Column(String(24))
    ingress_code = Column(String(100))
    ring_pdd = Column(Integer)
    egress_no_ring = Column(Integer)
    ingress_duration = Column(Integer)
    ingress_inter_duration = Column(Integer)
    ingress_intra_duration = Column(Integer)
    ingress_inter_not_zero_calls = Column(Integer)
    ingress_intra_not_zero_calls = Column(Integer)
    ingress_not_zero_calls = Column(Integer)
    ingress_not_zero_calls_30 = Column(Integer)
    ingress_duration_30 = Column(Integer)
    ingress_not_zero_calls_6 = Column(Integer)
    ingress_duration_6 = Column(Integer)
    ingress_call_18s = Column(Integer)
    ingress_call_24s = Column(Integer)
    ingress_call_2h = Column(Integer)
    ingress_call_3h = Column(Integer)
    ingress_call_4h = Column(Integer)
    ingress_call_12s = Column(Integer)
    src_ocn = Column(String(100))
    src_lata = Column(Integer)
    dst_ocn = Column(String(100))
    dst_lata = Column(Integer)
    lrn_ocn = Column(String(100))
    lrn_lata = Column(Integer)
    origination_source_host_name = Column(String(255))
    termination_destination_host_name = Column(String(255))

    orig_shaken_ocn = Column(Text)
    orig_shaken_lvl = Column(String(1))
    term_shaken_ocn = Column(Text)
    term_shaken_lvl = Column(String(1))

    ani_in_dno_count = Column(Integer)
    ani_in_ftc_count = Column(Integer)
    ani_in_ftc_week_count = Column(Integer)
    ani_in_ftc_month_count = Column(Integer)
    ani_in_spam_count = Column(Integer)
    ani_in_fraud_count = Column(Integer)
    ani_in_tcpa_count = Column(Integer)
    dnis_in_dnc_count = Column(Integer)
    origination_remote_payload_ip_address = Column(Text)
    termination_remote_payload_ip_address = Column(Text)
    o_trunk_type2 = Column(SmallInteger)

    __table_args__ = (PrimaryKeyConstraint(
        'ingress_client_id', 'ingress_id', 'egress_client_id', 'egress_id', 'product_rout_id', 'agent_id',
        'ingress_rate_table_id', 'route_plan_id', 'par_id', 'report_time'
    ),)


# +++
def client_cdr_start_date():
    tablename = 'client_cdr2%'
    try:
        r = get_db().session.execute(
            text_("select min(relname) as tablename from pg_class where relname like '{}';".format(
                tablename)))
        ret = r.fetchone()
        if not ret or not ret.tablename:
            return None
        tn = ret.tablename
        return datetime(int(tn[10:14]), int(tn[14:16]), int(tn[16:18])).date()
    except:
        return None


def client_cdr(date):
    log.debug("CLIENT_CDR")
    tablename = ('client_cdr{}'.format(date)).replace('-', '')
    log.debug(f"TABLENAME = {tablename}")
    try:
        log.debug("r = get_db().session.execute(text_(select tablename from pg_tables where tablename='{}';.format(tablename)))")
        r = get_db().session.execute(text_("select tablename from pg_tables where tablename='{}';".format(tablename)))
        log.debug("r.fetchall()[0][0] == tablename")
        if not r.fetchall()[0][0] == tablename:
            return None
    except:
        log.debug("returning None")
        return None
    log.debug("RETURNING TABLE")
    return Table(
        tablename, DnlApiBaseModel.metadata,
        Column('id', BigInteger),
        Column('record_sequence_number', String(100)),
        Column('version_number', String(100)),
        Column('record_type', String(100)),
        Column('connection_type', String(100)),
        Column('session_id', String(100)),
        Column('release_cause', SmallInteger),
        Column('start_time_of_date', BigInteger),
        Column('answer_time_of_date', BigInteger),
        Column('release_tod', BigInteger),
        Column('minutes_west_of_greenwich_mean_time', SmallInteger),
        Column('release_cause_from_protocol_stack', String(100)),
        Column('binary_value_of_release_cause_from_protocol_stack', String(100)),
        Column('first_release_dialogue', String(100)),
        Column('trunk_id_origination', String(100)),
        Column('voip_protocol_origination', String(100)),
        Column('origination_source_number', String(100), index=True),
        Column('origination_source_host_name', String(100)),
        Column('origination_destination_number', String(100), index=True),
        Column('origination_destination_host_name', String(100)),
        Column('origination_call_id', String(150)),
        Column('origination_remote_payload_ip_address', String(100)),
        Column('origination_remote_payload_udp_address', Integer),
        Column('origination_local_payload_ip_address', String(100)),
        Column('origination_local_payload_udp_address', Integer),
        Column('origination_codec_list', String(100)),
        Column('origination_ingress_packets', Integer),
        Column('origination_egress_packets', Integer),
        Column('origination_ingress_octets', Integer),
        Column('origination_egress_octets', Integer),
        Column('origination_ingress_packet_loss', Integer),
        Column('origination_ingress_delay', Integer),
        Column('origination_ingress_packet_jitter', Integer),
        Column('trunk_id_termination', String(100)),
        Column('voip_protocol_termination', String(100)),
        Column('termination_source_number', String(100)),
        Column('termination_source_host_name', String(100)),
        Column('termination_destination_number', String(100)),
        Column('termination_destination_host_name', String(100)),
        Column('termination_call_id', String(150)),
        Column('termination_remote_payload_ip_address', String(100)),
        Column('termination_remote_payload_udp_address', Integer),
        Column('termination_local_payload_ip_address', String(100)),
        Column('termination_local_payload_udp_address', Integer),
        Column('termination_codec_list', String(100)),
        Column('termination_ingress_packets', Integer),
        Column('termination_egress_packets', Integer),
        Column('termination_ingress_octets', Integer),
        Column('termination_egress_octets', Integer),
        Column('termination_ingress_packet_loss', Integer),
        Column('termination_ingress_delay', Integer),
        Column('termination_ingress_packet_jitter', Integer),
        Column('final_route_indication', String(100)),
        Column('routing_digits', String(100), index=True),
        Column('call_duration', Integer, index=True),
        Column('pdd', Integer),
        Column('ring_time', BigInteger),
        Column('callduration_in_ms', Integer),
        Column('conf_id', String(100)),
        Column('call_type', SmallInteger),
        Column('ingress_id', Integer),
        Column('ingress_client_id', Integer),
        Column('ingress_client_rate_table_id', Integer),
        Column('ingress_client_currency_id', Integer),
        Column('ingress_client_rate', Float(53)),
        Column('ingress_client_currency', String(100)),
        Column('ingress_client_bill_time', Integer),
        Column('ingress_client_bill_result', SmallInteger),
        Column('ingress_client_cost', Numeric(12, 6)),
        Column('time', DateTime(True), index=True, server_default=text_("('now'::text)::timestamp(0) with time zone")),
        Column('egress_id', Integer),
        Column('egress_rate_table_id', Integer),
        Column('egress_rate', Float(53)),
        Column('egress_cost', Numeric(12, 6)),
        Column('egress_bill_time', Integer),
        Column('egress_client_id', Integer),
        Column('egress_client_currency_id', Integer),
        Column('egress_client_currency', String(100)),
        Column('egress_six_seconds', Integer),
        Column('egress_bill_minutes', Float),
        Column('egress_bill_result', SmallInteger),
        Column('ingress_bill_minutes', Float),
        Column('ingress_dnis_type', SmallInteger),
        Column('ingress_rate_type', SmallInteger),
        Column('lrn_dnis', String(100)),
        Column('egress_dnis_type', SmallInteger),
        Column('egress_rate_type', SmallInteger),
        Column('translation_ani', String(100)),
        Column('ani_code_id', Integer),
        Column('dnis_code_id', Integer),
        Column('item_id', Integer),
        Column('ingress_rate_id', Integer),
        Column('egress_rate_id', Integer),
        Column('rerate_time', DateTime(True)),
        Column('orig_code', String(100)),
        Column('orig_code_name', String(100)),
        Column('orig_country', String(100)),
        Column('term_code', String(100)),
        Column('term_code_name', String(100)),
        Column('term_country', String(100)),
        Column('ingress_rate_effective_date', BigInteger),
        Column('egress_rate_effective_date', BigInteger),
        Column('egress_erro_string', Text),
        Column('route_plan', Integer),
        Column('dynamic_route', Integer),
        Column('static_route', Integer),
        Column('contract_id', String(100)),
        Column('order_id', String(100)),
        Column('order_type', String(100)),
        Column('lrn_number_vendor', SmallInteger),
        Column('lnp_dipping_cost', Numeric(10, 8)),
        Column('is_final_call', SmallInteger, index=True),
        Column('egress_code_asr', Float),
        Column('egress_code_acd', Float),
        Column('route_prefix', String(100)),
        Column('is_manual_kill', Boolean, server_default=text_("false")),
        Column('orig_call_duration', Integer),
        Column('orig_delay_second', Integer),
        Column('term_delay_second', Integer),
        Column('trunk_type', SmallInteger),
        Column('origination_profile_port', Integer),
        Column('termination_profile_port', Integer),
        Column('o_trunk_type2', SmallInteger),
        Column('o_billing_method', SmallInteger),
        Column('t_trunk_type2', SmallInteger),
        Column('t_billing_method', SmallInteger),
        Column('campaign_id', Integer),
        Column('tax', Integer),
        Column('agent_id', Integer),
        Column('agent_rate', Float(53)),
        Column('agent_cost', Numeric(12, 6)),
        Column('orig_jur_type', Integer),
        Column('term_jur_type', Integer),
        Column('ring_epoch', BigInteger),
        Column('end_epoch', BigInteger),
        Column('par_id', Integer),
        Column('paid_user', String(100)),
        Column('rpid_user', String(100)),
        Column('timeout_type', SmallInteger),
        Column('q850_cause', Integer),
        Column('q850_cause_string', String(100)),
        Column('src_lata', Integer),
        Column('src_ocn', String(100)),
        Column('src_rc', String(100)),
        Column('dst_lata', Integer),
        Column('dst_ocn', String(100)),
        Column('dst_rc', String(100)),
        Column('lrn_lata', Integer),
        Column('lrn_ocn', String(100)),
        Column('lrn_rc', String(100)),
        Column('cdr_date', String(24)),
        Column('orig_shaken_status', Integer),
        Column('orig_shaken_lvl', String(1)),
        Column('orig_shaken_ocn', String(100)),
        Column('orig_shaken_subject', String(512)),
        Column('term_shaken_status', Integer),
        Column('term_shaken_lvl', String(1)),
        Column('term_shaken_ocn', String(512)),
        Column('term_shaken_subject', String(512)),
        Column('ani_in_dno', Integer),
        Column('ani_in_ftc', DateTime(True)),
        Column('ani_in_spam', Boolean),
        Column('ani_in_fraud', Boolean),
        Column('ani_in_tcpa', Boolean),
        Column('dnis_in_dnc', Boolean),
        Column('src_type', String(512)),
        Column('dst_type', String(512)),
        extend_existing=True,
    )


class ClientCdr(DnlApiBaseModel):
    __tablename__ = 'client_cdr'

    id = Column(Integer, primary_key=True, server_default=text_("nextval('client_cdr_id_seq'::regclass)"))
    record_sequence_number = Column(String(100))
    version_number = Column(String(100))
    record_type = Column(String(100))
    connection_type = Column(String(100))
    session_id = Column(String(100))
    release_cause = Column(SmallInteger)
    start_time_of_date = Column(BigInteger)
    answer_time_of_date = Column(BigInteger)
    release_tod = Column(BigInteger)
    minutes_west_of_greenwich_mean_time = Column(SmallInteger)
    release_cause_from_protocol_stack = Column(String(100))
    binary_value_of_release_cause_from_protocol_stack = Column(String(100))
    first_release_dialogue = Column(String(100))
    trunk_id_origination = Column(String(100))
    voip_protocol_origination = Column(String(100))
    origination_source_number = Column(String(100))
    origination_source_host_name = Column(String(100))
    origination_destination_number = Column(String(100))
    origination_destination_host_name = Column(String(100))
    origination_call_id = Column(String(150))
    origination_remote_payload_ip_address = Column(String(100))
    origination_remote_payload_udp_address = Column(Integer)
    origination_local_payload_ip_address = Column(String(100))
    origination_local_payload_udp_address = Column(Integer)
    origination_codec_list = Column(String(100))
    origination_ingress_packets = Column(Integer)
    origination_egress_packets = Column(Integer)
    origination_ingress_octets = Column(Integer)
    origination_egress_octets = Column(Integer)
    origination_ingress_packet_loss = Column(Integer)
    origination_ingress_delay = Column(Integer)
    origination_ingress_packet_jitter = Column(Integer)
    trunk_id_termination = Column(String(100))
    voip_protocol_termination = Column(String(100))
    termination_source_number = Column(String(100))
    termination_source_host_name = Column(String(100))
    termination_destination_number = Column(String(100))
    termination_destination_host_name = Column(String(100))
    termination_call_id = Column(String(150))
    termination_remote_payload_ip_address = Column(String(100))
    termination_remote_payload_udp_address = Column(Integer)
    termination_local_payload_ip_address = Column(String(100))
    termination_local_payload_udp_address = Column(Integer)
    termination_codec_list = Column(String(100))
    termination_ingress_packets = Column(Integer)
    termination_egress_packets = Column(Integer)
    termination_ingress_octets = Column(Integer)
    termination_egress_octets = Column(Integer)
    termination_ingress_packet_loss = Column(Integer)
    termination_ingress_delay = Column(Integer)
    termination_ingress_packet_jitter = Column(Integer)
    final_route_indication = Column(String(100))
    routing_digits = Column(String(100))
    call_duration = Column(Integer)
    pdd = Column(Integer)
    ring_time = Column(BigInteger)
    callduration_in_ms = Column(Integer)
    conf_id = Column(String(100))
    call_type = Column(SmallInteger)
    ingress_id = Column(Integer)
    ingress_client_id = Column(Integer)
    ingress_client_rate_table_id = Column(Integer)
    ingress_client_currency_id = Column(Integer)
    ingress_client_rate = Column(Float(53))
    ingress_client_currency = Column(String(100))
    ingress_client_bill_time = Column(Integer)
    ingress_client_bill_result = Column(SmallInteger)
    ingress_client_cost = Column(Numeric(12, 6))
    time = Column(DateTime(timezone=True), index=True, default=datetime.now(UTC))
    egress_id = Column(Integer)
    egress_rate_table_id = Column(Integer)
    egress_rate = Column(Float(53))
    egress_cost = Column(Numeric(12, 6))
    egress_bill_time = Column(Integer)
    egress_client_id = Column(Integer)
    egress_client_currency_id = Column(Integer)
    egress_client_currency = Column(String(100))
    egress_six_seconds = Column(Integer)
    egress_bill_minutes = Column(Float)
    egress_bill_result = Column(SmallInteger)
    ingress_bill_minutes = Column(Float)
    ingress_dnis_type = Column(SmallInteger)
    ingress_rate_type = Column(SmallInteger)
    lrn_dnis = Column(String(100))
    egress_dnis_type = Column(SmallInteger)
    egress_rate_type = Column(SmallInteger)
    translation_ani = Column(String(100))
    ani_code_id = Column(Integer)
    dnis_code_id = Column(Integer)
    item_id = Column(Integer)
    ingress_rate_id = Column(Integer)
    egress_rate_id = Column(Integer)
    rerate_time = Column(DateTime(timezone=True))
    orig_code = Column(String(100))
    orig_code_name = Column(String(100))
    orig_country = Column(String(100))
    term_code = Column(String(100))
    term_code_name = Column(String(100))
    term_country = Column(String(100))
    ingress_rate_effective_date = Column(BigInteger)
    egress_rate_effective_date = Column(BigInteger)
    egress_erro_string = Column(Text)
    route_plan = Column(Integer)
    dynamic_route = Column(Integer)
    static_route = Column(Integer)
    contract_id = Column(String(100))
    order_id = Column(String(100))
    order_type = Column(String(100))
    lrn_number_vendor = Column(SmallInteger)
    lnp_dipping_cost = Column(Numeric(10, 8))
    is_final_call = Column(SmallInteger)
    egress_code_asr = Column(Float)
    egress_code_acd = Column(Float)
    route_prefix = Column(String(100))
    is_manual_kill = Column(Boolean, default=False)
    orig_call_duration = Column(Integer)
    orig_delay_second = Column(Integer)
    term_delay_second = Column(Integer)
    trunk_type = Column(SmallInteger)
    origination_profile_port = Column(Integer)
    termination_profile_port = Column(Integer)
    o_trunk_type2 = Column(SmallInteger)
    o_billing_method = Column(SmallInteger)
    t_trunk_type2 = Column(SmallInteger)
    t_billing_method = Column(SmallInteger)
    campaign_id = Column(Integer)
    tax = Column(Integer)
    agent_id = Column(Integer)
    agent_rate = Column(Float(53))
    agent_cost = Column(Numeric(12, 6))
    orig_jur_type = Column(Integer)
    term_jur_type = Column(Integer)
    ring_epoch = Column(BigInteger)
    end_epoch = Column(BigInteger)
    par_id = Column(Integer)
    paid_user = Column(String(100))
    rpid_user = Column(String(100))
    timeout_type = Column(SmallInteger)
    q850_cause = Column(Integer)
    q850_cause_string = Column(String(100))
    src_lata = Column(Integer)
    src_ocn = Column(String(100))
    src_rc = Column(String(100))
    dst_lata = Column(Integer)
    dst_ocn = Column(String(100))
    dst_rc = Column(String(100))
    lrn_lata = Column(Integer)
    lrn_ocn = Column(String(100))
    lrn_rc = Column(String(100))
    cdr_date = Column(String(24))
    orig_shaken_status = Column(Integer)
    orig_shaken_lvl = Column(String(1))
    orig_shaken_ocn = Column(String(100))
    orig_shaken_subject = Column(String(512))
    term_shaken_status = Column(Integer)
    term_shaken_lvl = Column(String(1))
    term_shaken_ocn = Column(String(512))
    term_shaken_subject = Column(String(512))
    ani_in_dno = Column(Integer)
    ani_in_spam = Column(Boolean)
    ani_in_fraud = Column(Boolean)
    ani_in_tcpa = Column(Boolean)
    ani_in_dnc = Column(Boolean)
    dnis_in_dnc = Column(Boolean)
    ani_in_ftc = Column(Boolean)
    src_type = Column(Text)
    dst_type = Column(Text)
    ingress_client_rate_table_name = column_property(
        select([RateTable.name]).where(RateTable.rate_table_id == ingress_client_rate_table_id).correlate_except(RateTable))
    """
    server = synonym('egress_id')
    ingress_trunk = synonym('ingress_id')
    ingress_host = synonym('origination_destination_host_name')
    egress_host = synonym('termination_source_host_name')
    ani = synonym('translation_ani')
    dnis = synonym('lrn_dnis')
    time = synonym('call_duration')
    """


class CdrRerate(DnlApiBaseModel):
    __tablename__ = 'cdr_rerate'

    id = Column(Integer, primary_key=True, server_default=text_("nextval('cdr_rerate_id_seq'::regclass)"))
    create_time = Column(DateTime(True))
    finish_time = Column(DateTime(True))
    status = Column(Integer, nullable=False)
    rerate_type = Column(Integer, nullable=False)
    rerate_rate_time = Column(DateTime(True))
    rate_table_id = Column(Integer)
    rate_table_name = Column(String(255))
    cdr_backup_file = Column(String(255))
    where_condition = Column(Text)
    start_time = Column(DateTime(True), nullable=False)
    end_time = Column(DateTime(True), nullable=False)
    pid = Column(Integer)


# --- SignupSignup
class Signup(DnlApiBaseModel):
    __tablename__ = 'signup'
    STATUS = {-1: 'initial', 0: 'pending', 1: 'approved', 2: 'rejected'}
    id = Column(Integer, primary_key=True)
    company = Column(String(100))
    login = Column(String(40))
    password = Column(String(50))
    address = Column(String(500))
    email = Column(String(100))
    noc_email = Column(String(100))
    billing_email = Column(String(100))
    rate_email = Column(String(100))
    rate_delivery_email = Column(String(100))
    tax_id = Column(String(100))
    details = Column(String(1000))
    phone = Column(String(32))
    status = Column(ChoiceType(STATUS), default=-1)
    modify_time = Column(DateTime(timezone=True), onupdate=func.now())
    signup_time = Column(DateTime(timezone=True), default=datetime.now(UTC))
    send_email = Column(SmallInteger, default=0)
    contact_name = Column(String(100))
    city = Column(String(40))
    state = Column(String(40))
    zip = Column(String(40))
    country = Column(String(40))
    billing_address = Column(String(40))
    billing_city = Column(String(40))
    billing_state = Column(String(40))
    billing_zip = Column(String(40))
    billing_country = Column(String(40))
    billing_contact_name = Column(String(40))
    billing_phone = Column(String(40))
    product_id = Column(String(100))
    referral = Column(String(500))
    agent_id = Column(Integer)
    client_id = Column(Integer)
    company_type = Column(ChoiceType(Client.COMPANY_TYPE), server_default='1')
    client_type = Column(ChoiceType(Client.CLIENT_TYPE), server_default='1')

    confirmed_at = Column(DateTime(timezone=True))
    confirmed_ip = Column(String(100))

    company_detail = synonym('details')
    company_name = synonym('company')
    main_email = synonym('email')
    modified_on = synonym('modify_time')
    signed_up_on = synonym('signup_time')
    username = synonym('login')
    rate_send_from_email = synonym('rate_email')
    rate_send_to_email = synonym('rate_delivery_email')
    send_signup_notification = synonym('send_email')
    update_at = synonym('modify_time')
    client_name = synonym('contact_name')
    name = synonym('login')
    # referral = synonym('product_id')

    ip_address = relationship('SignupIp', uselist=True, back_populates='signup',
                              single_parent=True,
                              cascade="all, delete-orphan")
    is_finished = column_property(status.in_(['approved', 'rejected']) == True)

    @property
    def token(self):
        import jwt
        exp = datetime.now(UTC) + timedelta(days=settings.JWT_TTL_DAYS)
        token_data = {'signup_id': self.id, 'exp': exp}
        token = jwt.encode(token_data, settings.JWT_SIGNATURE).decode('utf-8')
        return token

    @property
    def confirm_url(self):
        conf = SystemParameter.get(1)
        return '{}/confirmation/{}'.format(conf.base_url, self.token)

    @property
    def first_name(self):
        try:
            return self.contact_name.split(' ')[0]
        except:
            return self.contact_name

    @property
    def last_name(self):
        try:
            return self.contact_name.split(' ')[1]
        except:
            return None

    @hybrid_property
    def products(self):
        try:
            return [int(i) for i in self.product_id.split(',') if i != '']
        except:
            return []

    @products.setter
    def products(self, value):
        try:
            self.product_id = ','.join([str(x) for x in value])
        except:
            pass

    @property
    def login_url(self):
        # return SystemParameter.get(1).landing_page
        # SystemParameter.base_url
        return SystemParameter.get(1).base_url

    @property
    def referral_name(self):
        if self.agent_id:
            a = Agent.get(self.agent_id)
            if a:
                return a.agent_name

        return None

    def update_client_record(self, client=None):
        obj = self
        if not client:
            client = Client(name=obj.contact_name)
        client.status = True
        client.enough_balance = True
        client.enable_client_portal = True
        client.update_by = obj.login
        client.update_at = datetime.now(UTC)
        client.company = obj.company
        client.details = obj.company_detail
        client.email = obj.email
        client.billing_email = obj.billing_email
        client.login = obj.login
        client.password = obj.password
        # client.user=model.User(name=obj.login,passwd=obj.password)
        # client.user.email=obj.email
        # client.user.user_type='client'
        client.profit_type = 'value'
        client.auto_invoice_type = 'buy'
        client.allowed_credit = 0.0
        client.company_type = obj.company_type
        client.client_type = obj.client_type
        addr = '{contact_name}\n{company}\n{address}\n{city}\n{state} {zip}\n{country}\n'.format_map(obj.as_dict())
        baddr = 'bill to: {billing_contact_name}\n{company}\n{billing_address}\n{billing_city}\n{billing_state} {billing_zip}\n{billing_country}\nphone:{billing_phone}'.format_map(
            obj.as_dict())
        client.address = addr + baddr
        client.phone = obj.phone
        client.rate_email = obj.rate_email
        client.rate_delivery_email = obj.rate_send_from_email
        client.noc_email = obj.noc_email

        # client.tax_id = obj.tax_id
        # client.tax = float(obj.tax_id)

        client.default_ip = []
        for i in obj.ip_address:
            client.default_ip.append(i.client_ip())
        cl_id = client.save()
        obj.client_id = cl_id
        user = client.user
        if user:
            user.active = True
            fl = obj.contact_name.split(' ')
            if fl:
                user.first_name = fl[0]
                if len(fl) > 1:
                    user.last_name = fl[1]
            user.save()
        return cl_id


class SignupIp(DnlApiBaseModel):
    __tablename__ = 'signup_ip'

    id = Column(Integer, primary_key=True)
    signup_id = Column(ForeignKey('signup.id', ondelete='CASCADE'), nullable=False)
    ip = Column(String(100))
    port = Column(Integer, default=5060)
    netmark = Column(Integer, default=32)
    mask = synonym('netmark')

    signup = relationship(Signup, uselist=False, back_populates='ip_address')

    def client_ip(self):
        return ClientDefaultIp(ip=self.ip, port=self.port, netmark=self.netmark)


class DailyCdrField(DnlApiBaseModel):
    __tablename__ = 'daily_cdr_fields'

    NEEDED_FIELDS = [
        'time',
        'release_cause',
        'release_cause_name',
        'start_time_of_date',
        'answer_time_of_date',
        'release_tod',
        'minutes_west_of_greenwich_mean_time',
        'release_cause_from_protocol_stack',
        'binary_value_of_release_cause_from_protocol_stack',
        'first_release_dialogue',
        'origination_source_number',
        'origination_source_host_name',
        'origination_destination_number',
        'origination_destination_host_name',
        'origination_call_id',
        'origination_remote_payload_ip_address',
        'origination_remote_payload_udp_address',
        'origination_local_payload_udp_address',
        'origination_codec_list',
        'termination_source_number',
        'termination_source_host_name',
        'termination_destination_number',
        'termination_destination_host_name',
        'termination_call_id',
        'termination_remote_payload_ip_address',
        'termination_remote_payload_udp_address',
        'termination_local_payload_ip_address',
        'termination_local_payload_udp_address',
        'termination_codec_list',
        'final_route_indication',
        'routing_digits',
        'call_duration',
        'pdd',
        'ring_time',
        'callduration_in_ms',
        'call_type',
        'ingress_id',
        'ingress_client_id',
        'ingress_client_rate_table_id',
        'ingress_client_currency_id',
        'ingress_client_currency',
        'ingress_client_rate',
        'ingress_client_bill_time',
        'ingress_client_bill_result',
        'ingress_client_cost',
        'egress_id',
        'egress_rate_table_id',
        'egress_rate',
        'egress_cost',
        'egress_bill_time',
        'egress_client_id',
        'egress_client_currency_id',
        'egress_client_currency',
        'egress_six_seconds',
        'egress_bill_minutes',
        'egress_bill_result',
        'ingress_bill_minutes',
        'ingress_dnis_type',
        'ingress_rate_type',
        'lrn_dnis',
        'egress_dnis_type',
        'egress_rate_type',
        'translation_ani',
        'ingress_rate_id',
        'egress_rate_id',
        'orig_code',
        'orig_code_name',
        'orig_country',
        'term_code',
        'term_code_name',
        'term_country',
        'ingress_rate_effective_date',
        'egress_rate_effective_date',
        'egress_erro_string',
        'egress_code_asr',
        'egress_code_acd',
        'static_route',
        'dynamic_route',
        'route_plan',
        'route_prefix',
        'orig_call_duration',
        'origination_profile_port',
        'termination_profile_port',
        'orig_jur_type',
        'term_jur_type',
        'ring_epoch',
        'end_epoch',
        'paid_user',
        'rpid_user',
        'timeout_type'
        'q850_cause',
        'q850_cause_string',
        'orig_shaken_lvl',
        'orig_shaken_ocn',
        'orig_shaken_status',
        'orig_shaken_subject',
        'term_shaken_lvl',
        'term_shaken_ocn',
        'term_shaken_status',
        'term_shaken_subject'
    ]
    id = Column(Integer, primary_key=True)
    type = Column(SmallInteger, default=0)
    field = Column(String, nullable=False)
    label = Column(String)

    db_name = synonym('field')
    display_name = synonym('label')

    @hybrid_property
    def admin_default_origination(self):
        try:
            return ((int(self.type) & (1 << 5)) != 0)
        except:
            return False

    @admin_default_origination.setter
    def admin_default_origination(self, value):
        if value:
            self.type = (self.type if self.type else 0) | (1 << 5)
        else:
            self.type = (self.type if self.type else 0) & ~(1 << 5)

    @hybrid_property
    def admin_default(self):
        try:
            return ((int(self.type) & (1 << 4)) != 0)
        except:
            return False

    @admin_default.setter
    def admin_default(self, value):
        if value:
            self.type = (self.type if self.type else 0) | (1 << 4)
        else:
            self.type = (self.type if self.type else 0) & ~(1 << 4)

    @hybrid_property
    def client_viewable(self):
        try:
            return ((int(self.type) & (1 << 3)) != 0)
        except:
            return False

    @client_viewable.setter
    def client_viewable(self, value):
        if value:
            self.type = (self.type if self.type else 0) | (1 << 3)
        else:
            self.type = (self.type if self.type else 0) & ~(1 << 3)

    @hybrid_property
    def vendor_viewable(self):
        try:
            return ((int(self.type) & (1 << 2)) != 0)
        except:
            return False

    @vendor_viewable.setter
    def vendor_viewable(self, value):
        if value:
            self.type = (self.type if self.type else 0) | (1 << 2)
        else:
            self.type = (self.type if self.type else 0) & ~(1 << 2)

    @hybrid_property
    def client_cdr_delivery(self):
        try:
            return ((int(self.type) & (1 << 1)) != 0)
        except:
            return False

    @client_cdr_delivery.setter
    def client_cdr_delivery(self, value):
        if value:
            self.type = (self.type if self.type else 0) | (1 << 1)
        else:
            self.type = (self.type if self.type else 0) & ~(1 << 1)

    @hybrid_property
    def vendor_cdr_delivery(self):
        try:
            return ((int(self.type) & (1 << 0)) != 0)
        except:
            return False

    @vendor_cdr_delivery.setter
    def vendor_cdr_delivery(self, value):
        if value:
            self.type = (self.type if self.type else 0) | (1 << 0)
        else:
            self.type = (self.type if self.type else 0) & ~(1 << 0)

    @hybrid_property
    def client_viewable_origination(self):
        try:
            return ((int(self.type) & (1 << 6)) != 0)
        except:
            return False

    @client_viewable_origination.setter
    def client_viewable_origination(self, value):
        if value:
            self.type = (self.type if self.type else 0) | (1 << 6)
        else:
            self.type = (self.type if self.type else 0) & ~(1 << 6)

    @classmethod
    def convert_fields(cls, fields):
        d = {r.field: r.label for r in cls.query().all()}
        ret = [d[f] if f in d else f for f in fields]
        # for f in fields:
        #     if f in d:
        #         ret.append(d[f])
        #     else:
        #         ret.append(f)
        return ret

    @staticmethod
    def convert_fields_to_user_friendly(fields):
        user_friendly_names = [
            { "origination_destination_host_name": "Inbound Loc IP" },
            { "origination_local_payload_udp_address": "Inbound Loc Media Port" },
            { "ring_time": "Ring Time" },
            { "origination_remote_payload_udp_address": "Inbound Media Port" },
            { "termination_local_payload_ip_address": "Outbound Loc IP" },
            { "pdd": "PDD" },
            { "binary_value_of_release_cause_from_protocol_stack": "Response to Ingress" },
            { "release_tod": "End Time" },
            { "termination_local_payload_udp_address": "Outbound Media Port" },
            { "call_type": "Call Type" },
            { "conf_id": "Conf id" },
            { "egress_bill_minutes": "Outbound Bill Min" },
            { "egress_cost": "Outbound Cost" },
            { "ingress_client_id": "Ingress Client id" },
            { "ingress_client_rate_table_id": "Inbound Rate Table" },
            { "tax": "Tax" },
            { "termination_destination_host_name": "Outbound Loc IP" },
            { "egress_id": "Egress id" },
            { "call_duration": "Outbound Dur" },
            { "egress_client_id": "Egress Carrier id" },
            { "termination_remote_payload_udp_address": "Outbound Media Port" },
            { "termination_source_host_name": "Outbound IP" },
            { "egress_bill_time": "Egress Bill Time" },
            { "callduration_in_ms": "Call Duration in MS" },
            { "termination_destination_number": "Outbound DNIS" },
            { "minutes_west_of_greenwich_mean_time": "Timezone" },
            { "release_cause_from_protocol_stack": "Response From Egress" },
            { "time": "Time" },
            { "origination_source_host_name": "Inbound IP" },
            { "origination_remote_payload_ip_address": "Inbound Media IP" },
            { "egress_rate": "Outbound Rate" },
            { "voip_protocol_origination": "voip_protocol_origination" },
            { "origination_codec_list": "Inbound Codecs" },
            { "ingress_client_bill_result": "Ingress Bill Result" },
            { "termination_remote_payload_ip_address": "Outbound Media IP" },
            { "ingress_client_rate": "Inbound Rate" },
            { "termination_codec_list": "Term Codecs" },
            { "final_route_indication": "Final Route" },
            { "routing_digits": "Translation DNIS" },
            { "ingress_dnis_type": "Ingress DNIS type" },
            { "origination_local_payload_ip_address": "Inbound Loc Media IP" },
            { "ingress_id": "Ingress id" },
            { "egress_rate_table_id": "Outbound Rate table" },
            { "agent_id": "Agent id" },
            { "agent_rate": "Agent Rate" },
            { "dst_rc": "Dst Number Rate Center" },
            { "dynamic_route": "Dynamic Route Name" },
            { "egress_code_acd": "Egress Code ACD" },
            { "egress_code_asr": "Egress Code ASR" },
            { "order_id": "Order id" },
            { "trunk_type": "Trunk Type" },
            { "egress_erro_string": "Egress Trunk Trace" },
            { "rpid_user": "RPID" },
            { "static_route": "Static Route Name" },
            { "route_plan": "Routing Plan" },
            { "egress_client_currency": "Egress Currency" },
            { "campaign_id": "Campaign id" },
            { "ingress_rate_effective_date": "Ingress Rate Effective Date" },
            { "term_code": "Outbound Code" },
            { "term_code_name": "Outbound Code Name" },
            { "term_country": "Outbound Country" },
            { "o_trunk_type2": "O Trunk Type2" },
            { "t_trunk_type2": "T Trunk Type2" },
            { "termination_profile_port": "Term Profile Port" },
            { "timeout_type": "Timeout type" },
            { "egress_bill_result": "Egress Bill Result" },
            { "translation_ani": "Translation ANI" },
            { "lrn_lata": "LRN Number LATA" },
            { "ingress_bill_minutes": "Inbound Bill Min" },
            { "ingress_rate_type": "Inbound Rate Type" },
            { "lrn_ocn": "LRN Number OCN" },
            { "lrn_rc": "LRN Number Rate Center" },
            { "egress_dnis_type": "Egress DNIS Type" },
            { "egress_rate_effective_date": "Egress Rate Effective Date" },
            { "egress_rate_type": "Outbound Rate Type" },
            { "is_final_call": "Final" },
            { "lrn_dnis": "LRN Number" },
            { "orig_call_duration": "Inbound Dur" },
            { "orig_code": "Inbound Code" },
            { "orig_code_name": "Inbound Code Name" },
            { "orig_country": "Inbound Country" },
            { "origination_profile_port": "Orig Profile Port" },
            { "paid_user": "PAid user" },
            { "q850_cause": "Q850" },
            { "q850_cause_string": "Q850 Media CAUSE" },
            { "route_prefix": "Prefix" },
            { "egress_client_currency_id": "Egress Currency id" },
            { "end_epoch": "End Time epoch" },
            { "agent_cost": "Agent Cost" },
            { "connection_type": "Connection Type" },
            { "dst_lata": "Dst Number LATA" },
            { "dst_ocn": "Dst Number OCN" },
            { "dynamic_route_name": "Dynamic Route" },
            { "egress_name": "Outbound Trunk" },
            { "ingress_client_name": "Inbound Carrier" },
            { "answer_time_of_date": "Answer Time" },
            { "egress_client_name": "Outbound Carrier" },
            { "first_release_dialogue": "Orig/Term Release" },
            { "origination_call_id": "Inbound Call-ID" },
            { "ingress_client_bill_time": "Ingress Bill Time" },
            { "par_id": "par_id" },
            { "origination_destination_number": "Inbound DNIS" },
            { "egress_six_seconds": "egress_six_seconds" },
            { "lnp_dipping_cost": "lnp_dipping_cost" },
            { "lrn_number_vendor": "lrn_number_vendor" },
            { "ring_epoch": "Ring Start time" },
            { "term_jur_type": "Term Jurisdiction Type" },
            { "termination_call_id": "Outbound Call-ID" },
            { "origination_source_number": "Inbound ANI" },
            { "start_time_of_date": "Start Time" },
            { "termination_source_number": "Outbound ANI" },
            { "ingress_client_cost": "Inbound Cost" },
            { "orig_jur_type": "Orig Jurisdiction type" },
            { "egress_rate_table_name": "Outbound Rate Table" },
            { "egress_trunk_trace": "Egress Trunk Trace" },
            { "ingress_name": "Inbound Trunk" },
            { "release_cause_name": "Release Cause" },
            { "rerate_time": "Rerate Time" },
            { "route_plan_name": "Route Plan" },
            { "src_lata": "Src Number LATA" },
            { "src_ocn": "Src Number OCN" },
            { "src_rc": "Src Number Rate Center" },
            { "static_route_name": "Static Route Name" },
            { "egress_rate_id": "Egress Rate id" },
            { "release_cause": "Release Cause" },
            { "dnis_in_dnc": "DNC DNIS" },
            { "ani_in_dno": "DNO ANI" },
            { "ani_in_ftc": "FTC ANI" },
            { "ani_in_fraud": "Fraud ANI" },
            { "ingress_client_rate_table_name": "Inbound Rate Table" },
            { "ingress_rate_type_name": "Inbound Rate Type" },
            { "orig_shaken_lvl": "Inbound Stir/Shaken Attestation" },
            { "orig_shaken_ocn": "Inbound Stir/Shaken OCN" },
            { "orig_shaken_status": "Inbound Stir/Shaken Status" },
            { "orig_shaken_subject": "Inbound Stir/Shaken Subject" },
            { "egress_rate_type_name": "Outbound Rate Type" },
            { "term_shaken_lvl": "Outbound Stir/Shaken Attestation" },
            { "term_shaken_ocn": "Outbound Stir/Shaken OCN" },
            { "term_shaken_status": "Outbound Stir/Shaken Status" },
            { "term_shaken_subject": "Outbound Stir/Shaken Subject" },
            { "ani_in_spam": "SPAM ANI" },
            { "ani_in_tcpa": "TCPA ANI" },
        ]

        m = {list(n.keys())[0]: list(n.values())[0] for n in user_friendly_names}
        return [m.get(f, f) for f in fields]

    @staticmethod
    def init():
        def to_label(old):
            def cap(s):
                if s in ('id', 'Id'):
                    return 'ID'
                else:
                    return s.capitalize()

            return ' '.join([cap(s) for s in old.split('_')])

        # cols=[c.name for c in inspect(CdrReportDetail).columns]
        from api_dnl.utils.statisticapi2 import FIELDS, UIFIELDS
        for k, v in FIELDS.items():
            q = DailyCdrField.get(k)
            if q:
                if q.field != v:
                    q.field = v
                    if UIFIELDS[k] != '':
                        q.label = UIFIELDS[k]
                    else:
                        q.label = to_label(FIELDS[k])
                    try:
                        q.save()
                    except:
                        pass
            else:
                if UIFIELDS[k] != '':
                    label = UIFIELDS[k]
                else:
                    label = to_label(FIELDS[k])
                rec = DailyCdrField(id=k, field=v, label=label)
                try:
                    rec.save()
                except:
                    pass
        for rec in DailyCdrField.query().all():
            if rec.id not in FIELDS:
                rec.delete()


class GlobalRouteError(DnlApiBaseModel):
    __tablename__ = 'global_route_error'

    id = Column(Integer, primary_key=True)
    error_code = Column(Integer, unique=True)
    error_description = Column(String(100))
    to_sip_code = Column(Integer)
    to_sip_string = Column(String(100))
    default_to_sip_code = Column(Integer)
    default_to_sip_string = Column(String(100))

    def before_save(self):
        self.id = self.error_code


# class GlobalFailover(DnlApiBaseModel):
#     __tablename__ = 'global_failover'
#     CODES = {101: 'Invalid Argument',
#              200: 'OK',
#              300: 'Multiple Choices',
#              302: 'Moved Temporarily',
#              305: 'Use Proxy',
#              380: 'Alternative Service',
#              400: 'Bad Request',
#              401: 'Unauthorized',
#              402: 'Payment Required',
#              403: 'Forbidden',
#              404: 'Not Found',
#              405: 'Method Not Allowed',
#              406: 'Not Acceptable',
#              407: 'Proxy Authentication Required',
#              408: 'Request Timeout',
#              409: 'Conflict',
#              410: 'Gone',
#              411: 'Length Required',
#              412: 'Precondition Failed',
#              413: 'Request Entity Too Large',
#              414: 'Request-URI Too Long',
#              415: 'Unsupported Media Type',
#              416: 'Unsupported URI Scheme',
#              417: 'Unknown Resource-Priority',
#              420: 'Bad Extension',
#              421: 'Extension Required',
#              422: 'Session Interval Too Small',
#              423: 'Interval Too Brief',
#              480: 'Temporarily Unavailable',
#              481: 'Transaction Does Not Exist',
#              482: 'Loop Detected',
#              483: 'Too Many Hops',
#              484: 'Address Incomplete',
#              485: 'Ambiguous',
#              486: 'Busy Here',
#              487: 'Request Terminated',
#              488: 'Not Acceptable Here',
#              489: 'Bad Event',
#              490: 'Request Updated',
#              491: 'Request Pending',
#              493: 'Undecipherable',
#              494: 'Security Agreement Required',
#              500: 'Internal Server Error',
#              501: 'Not Implemented',
#              502: 'Bad Gateway',
#              503: 'Service Unavailable',
#              504: 'Gateway Time-out',
#              505: 'Version Not Supported',
#              513: 'Message Too Large',
#              580: 'Precondition Failure',
#              600: 'Busy Everywhere',
#              603: 'Decline',
#              604: 'Does Not Exist Anywhere',
#              606: 'Not Acceptable',
#              608: 'Unwanted',
#              687: 'Dialog Terminated'}

#     FAILOVER_STRATEGY = {1: 'Fail to Next Host', 2: 'Fail to Next Trunk', 3: 'Stop'}
#     id = Column(Integer, primary_key=True)
#     failover_strategy = Column(ChoiceType(FAILOVER_STRATEGY), default=1)
#     from_sip_code = Column(Integer, unique=True)
#     to_sip_code = Column(Integer)
#     to_sip_string = Column(String(100))
#     failover_method = synonym('failover_strategy')
#     match_code = synonym('from_sip_code')
#     return_code = synonym('to_sip_code')
#     return_clause = synonym('to_sip_string')
#     termination = relationship('TerminationFailover', uselist=False, back_populates='defaults', single_parent=True,
#                                cascade="all, delete-orphan")
#     origination = relationship('OriginationFailover', uselist=False, back_populates='defaults', single_parent=True,
#                                cascade="all, delete-orphan")

#     def before_save(self):
#         self.id = self.from_sip_code

#     def after_save(self):
#         if not TerminationFailover.get(self.id):
#             TerminationFailover(id=self.id, from_sip_code=self.from_sip_code, to_sip_code=self.to_sip_code,
#                                 to_sip_string=self.to_sip_string, failover_strategy=self.failover_strategy).save()
#         if not OriginationFailover.get(self.id):
#             OriginationFailover(id=self.id, from_sip_code=self.from_sip_code, to_sip_code=self.to_sip_code,
#                                 to_sip_string=self.to_sip_string, failover_strategy=self.failover_strategy).save()
#         self.session().refresh(self)


class TerminationFailover(DnlApiBaseModel):
    __tablename__ = 'termination_global_failover'
    FAILOVER_STRATEGY = {1: 'Fail to Next Host', 2: 'Fail to Next Trunk', 3: 'Stop'}
    id = Column(Integer, primary_key=True)
    failover_strategy = Column(ChoiceType(FAILOVER_STRATEGY), default=1)
    from_sip_code = Column(Integer)
    to_sip_code = Column(Integer)
    to_sip_string = Column(String(100))
    failover_method = synonym('failover_strategy')
    match_code = synonym('from_sip_code')
    return_code = synonym('to_sip_code')
    return_clause = synonym('to_sip_string')
    # defaults = relationship('GlobalFailover', uselist=False, back_populates='termination')


#    def before_save(self):
#        self.id=self.from_sip_code


class OriginationFailover(DnlApiBaseModel):
    __tablename__ = 'origination_global_failover'
    FAILOVER_STRATEGY = {1: 'Fail to Next Host', 2: 'Fail to Next Trunk', 3: 'Stop'}
    id = Column(Integer, primary_key=True)
    failover_strategy = Column(ChoiceType(FAILOVER_STRATEGY), default=1)
    from_sip_code = Column(Integer)
    to_sip_code = Column(Integer)
    to_sip_string = Column(String(100))
    failover_method = synonym('failover_strategy')
    match_code = synonym('from_sip_code')
    return_code = synonym('to_sip_code')
    return_clause = synonym('to_sip_string')
    # defaults = relationship('GlobalFailover', uselist=False, back_populates='origination')


#    def before_save(self):
#        self.id=self.from_sip_code

class ResourceFailover(DnlApiBaseModel):
    __tablename__ = 'resource_failover'
    FAILOVER_STRATEGY = {1: 'Fail to Next Host', 2: 'Fail to Next Trunk', 3: 'Stop'}

    id = Column(Integer, primary_key=True)
    failover_strategy = Column(ChoiceType(FAILOVER_STRATEGY), default=1)
    from_sip_code = Column(Integer)
    to_sip_code = Column(Integer)
    to_sip_string = Column(String(100))
    failover_method = synonym('failover_strategy')
    match_code = synonym('from_sip_code')
    return_code = synonym('to_sip_code')
    return_clause = synonym('to_sip_string')
    # defaults = relationship('GlobalFailover', uselist=False, back_populates='origination')

    def before_save(self):
        self.id=self.from_sip_code

# class ResourceFailover(DnlApiBaseModel):
#     __tablename__ = 'resource_failover'
#     FAILOVER_STRATEGY = {1: 'Fail to Next Host', 2: 'Fail to Next Trunk', 3: 'Stop'}

#     id = Column(Integer, primary_key=True)
#     failover_strategy = Column(ChoiceType(FAILOVER_STRATEGY), default=1)
#     from_sip_code = Column(Integer)
#     to_sip_code = Column(Integer)
#     to_sip_string = Column(String(100))

#     failover_method = synonym('failover_strategy')
#     match_code = synonym('from_sip_code')
#     return_code = synonym('to_sip_code')
#     return_clause = synonym('to_sip_string')

#     resource_id = Column(ForeignKey('resource.resource_id', ondelete='CASCADE'))
#     # resource = relationship(Resource, primaryjoin=foreign(Resource.resource_id) == resource_id,
#     #                         uselist=True)


# +++ FtpConf


class FtpConf(DnlApiBaseModel):
    __tablename__ = 'ftp_conf'
    DAYS_OF_WEEK = {0: "Sunday", 1: "Monday", 2: "Tuesday", 3: "Wednesday", 4: "Thursday", 5: "Friday", 6: "Saturday"}
    FREQUENCY = {1: 'daily', 2: 'weekly', 3: 'hourly'}
    FILE_TYPE = {0: 'no compression', 1: 'gz', 2: 'tar.gz', 3: 'tar.bz2'}  # no zip in cdr api,3:'zip'}
    DURATION = {0: 'all', 1: 'non-zero', 2: 'zero'}
    FILE_BREAKDOWN = {0: 'as one big file', 1: 'as hourly file', 2: 'as daily file'}
    STATUS = {0: 'disconnected', 1: 'connected'}

    id = Column(Integer, primary_key=True)
    server_ip = Column(String(100))
    server_port = Column(String(100))
    username = Column(String(100))
    password = Column(String(100))
    frequency = Column(ChoiceType(FREQUENCY), default=1)
    # frequency = Column(Integer, default=0)
    fields = Column(Text)
    headers = Column(Text)
    contain_headers = Column(Boolean)
    file_type = Column(ChoiceType(FILE_TYPE))
    ingress_carriers = Column(Text)
    egress_carriers = Column(Text)
    ingress_carriers_all = Column(Boolean, server_default=text_("True"))
    egress_carriers_all = Column(Boolean, server_default=text_("True"))
    duration = Column(ChoiceType(DURATION))
    ingress_release_cause = Column(String(255), default=0)
    egress_release_cause = Column(String(255), default=0)
    conditions = Column(Text)
    ingresses = Column(Text)
    egresses = Column(Text)
    ingresses_all = Column(Boolean, server_default=text_("True"))
    egresses_all = Column(Boolean, server_default=text_("True"))
    time = Column(String(255))
    alias = Column(String(255))
    server_dir = Column(String(255))
    max_lines = Column(Integer, nullable=False, default=10000)
    active = Column(Boolean, default=True)
    every_hours = Column(Integer, default=1)
    file_breakdown = Column(ChoiceType(FILE_BREAKDOWN), default=0)
    every_minutes = Column(Integer, default=15)
    every_day = Column(ChoiceType(DAYS_OF_WEEK), default=1)
    status = Column(ChoiceType(STATUS), server_default='0')
    # job_id = Column(Integer)

    compression = synonym('file_type')
    ftp_directory = synonym('server_dir')
    ftp_password = synonym('password')

    ftp_server_port = synonym('server_port')
    ftp_username = synonym('username')
    include_all_egress = synonym('egresses_all')
    include_all_ingress = synonym('ingresses_all')
    include_header = synonym('contain_headers')
    max_line_per_file = synonym('max_lines')
    name = synonym('alias')
    day_of_week = synonym('every_day')

    ftp_server_ip = column_property(func.substring(server_ip, 7))

    last_ftp_time = column_property(
        select([func.max(FtpCdrLog.ftp_end_time)]).where(FtpCdrLog.ftp_conf_id == id).correlate_except(FtpCdrLog))

    @hybrid_property
    def ftp_server_ip_1(self):
        try:
            return self.ftp_server_ip[6:]
        except:
            return None

    @ftp_server_ip_1.setter
    def ftp_server_ip_1(self, value):
        try:
            self.server_ip = 'ftp://' + value
        except:
            pass

    @hybrid_property
    def egress_trunks(self):
        try:
            return [int(i) for i in self.egresses.split(',') if int(i) != -1]
        except:
            return []

    @egress_trunks.setter
    def egress_trunks(self, value):
        try:
            self.egresses = ','.join([str(x) for x in value])
        except:
            pass

    @hybrid_property
    def ingress_trunks(self):
        try:
            return [int(i) for i in self.ingresses.split(',') if int(i) != -1]
        except:
            return []

    @ingress_trunks.setter
    def ingress_trunks(self, value):
        try:
            self.ingresses = ','.join([str(x) for x in value])
        except:
            pass

    @hybrid_property
    def include_fields(self):
        try:
            return cdr_names(self.fields).split(',')
        except:
            return None

    @include_fields.setter
    def include_fields(self, value):
        try:
            self.fields = cdr_fields(','.join(value))
        except:
            pass

    @hybrid_property
    def include_headers(self):
        try:
            return self.headers.split(',')
        except:
            return []

    @include_headers.setter
    def include_headers(self, value):
        try:
            self.headers = ','.join(value)
        except:
            pass

    @hybrid_property
    def non_zero(self):
        try:
            return self.duration == 'non-zero'
        except:
            return False

    @non_zero.setter
    def non_zero(self, value):
        try:
            if value:
                self.duration = 1
            else:
                self.duration = 0
        except:
            pass

    @hybrid_property
    def ingress_release_cause_x(self):
        try:
            return ','.join([i[0:3] + '*' for i in self.ingress_release_cause.split(',') if i])
        except:
            return ''

    @hybrid_property
    def egress_release_cause_x(self):
        try:
            return ','.join([i[0:3] + '*' for i in self.egress_release_cause.split(',') if i])
        except:
            return ''

    @hybrid_property
    def orig_return_code(self):
        try:
            return [int(i[0:3]) for i in self.ingress_release_cause.split(',')]
        except:
            return []

    @orig_return_code.setter
    def orig_return_code(self, value):
        try:
            self.ingress_release_cause = ','.join([str(x) + '*' for x in value])
        except:
            pass

    @hybrid_property
    def term_return_code(self):
        try:
            return [int(i[0:3]) for i in self.egress_release_cause.split(',')]
        except:
            return []

    @term_return_code.setter
    def term_return_code(self, value):
        try:
            self.egress_release_cause = ','.join([str(x) + '*' for x in value])
        except:
            pass

    def test_ftp_server(self):
        from ftplib import FTP
        self.status = 'disconnected'
        host = self.ftp_server_ip
        # if 'ftp://' == host[:6]:
        #    host = host[6:]
        ftp = FTP(host=host, user=self.ftp_username, passwd=self.ftp_password, timeout=1)
        if self.ftp_server_port:
            ftp.connect(port=int(self.ftp_server_port))
        ftp.login(user=self.ftp_username, passwd=self.ftp_password)
        ftp.cwd("/")
        self.status = 'connected'

    def _freq(self):
        if self.frequency == 'daily':
            return 24
        elif self.frequency == 'hourly':
            if self.every_hours:
                return self.every_hours
            return 1
        elif self.frequency == 'minutely':
            return 2
        elif self.frequency == 'weekly':
            return 168
        return 1

    def _file_type(self):
        return rev(self.FILE_TYPE)[self.file_type]

    def _ftp_url(self):
        # return 'ftp://{}'.format(self.server_ip)
        return self.server_ip

    def _fields(self):
        ret = []

    def trigger_manual(self, start_time, end_time):
        from api_dnl.task.ftp_upload import ftp_upload
        try:
            ftp_upload.delay(self.id, start_time, end_time)
        except:
            ftp_upload(self.id, start_time, end_time)


# --- FtpConf
# +++ FtpServerLog
# ---FtpServerLog

# +++ FtpCdr
class FtpCdr(DnlApiBaseModel):
    __tablename__ = 'ftp_cdr'

    ftp_user = Column(String(100), primary_key=True)
    ftp_pw = Column(String(100))
    run_freq = Column(Integer)
    cdr_fields = Column(Text)
    cdr_start_time = Column(DateTime(True))
    cdr_end_time = Column(DateTime(True))
    last_run_time = Column(DateTime(True))
    cdr_alias = Column(Text)


# --- FtpCdr


# +++SystemParameter
class SystemParameter(DnlApiBaseModel):
    __tablename__ = 'system_parameter'
    AUTO_RATE_MAIL_SSL = {0: 'no', 1: 'yes'}
    # DEFAULT_US_IJ_RULE = {0: 'A-Z', 1: 'US Non-JD', 2: 'US JD', 3: 'OCN-LATA-JD', 4: 'OCN-LATA-NON-JD'}
    DEFAULT_US_IJ_RULE = {0: 'Inter', 1: 'Intra', 2: 'Max of Inter/Intra'}
    LANDING_PAGE = {0: 'QoS Report', 1: 'Summary Report', 2: 'Orig - Term Report', 3: 'Carrier Management'}
    POSITION = {1: 'top', 2: 'bottom'}
    CHARGE_TYPE = {1: 'credit total amount', 2: 'create actual received amount'}
    INVOICE_SEND_MODE = {0: 'link', 1: 'attachment'}
    INVOICE_SEND_MODE_CDR = {0: 'none', 1: 'link', 2: 'attachment'}
    QOS_SAMPLE_PERIOD = {0: '15 min', 2: '1 hour', 3: '1 day'}
    LOW_CALL_ATTEMPT_HANDLING = {0: 'none', 1: 'use latest value'}
    REPORT_COUNT = {0: '1 Hour', 1: '24 Hour'}
    sys_id = Column(Integer, primary_key=True)
    default_avatar_id = Column(Integer)
    allow_cdr_fields = Column(Text)
    auto_carrier_notification = Column(Boolean)
    auto_delivery_address = Column(String(200))
    auto_delivery_group_by = Column(SmallInteger, default=0)
    auto_delivery_timezone = Column(String(20), default=text_("'+00'::character varying"))
    auto_rate_mail_ssl = Column(ChoiceType(AUTO_RATE_MAIL_SSL), default=1)
    auto_rate_pwd = Column(String(100))
    auto_rate_smtp = Column(String(50))
    auto_rate_smtp_port = Column(Integer)
    auto_rate_username = Column(String(100))
    backup_leave_last = Column(String(10))
    backup_period = Column(String(10))
    bar_color = Column(String(20), default=text_("'6B9B20'::character varying"))
    base_url = Column(String(512))
    billing_info = Column(Text)
    billing_info_location = Column(ChoiceType(POSITION), default=0)
    call_timeout = Column(Integer, default=3600)
    cdrs_deleteafterdays = Column(String(20))
    charge_type = Column(ChoiceType(CHARGE_TYPE))
    stripe_charge_type = Column(ChoiceType(CHARGE_TYPE))
    codedeck_last_az_update = Column(DateTime(timezone=True))
    codedeck_last_us_update = Column(DateTime(timezone=True))
    codedeck_prompt_az_update = Column(DateTime(timezone=True))
    codedeck_prompt_us_update = Column(DateTime(timezone=True))
    company_info = Column(String(2500))
    company_info_location = Column(ChoiceType(POSITION), default=0)
    company_name = Column(String(200))
    conf_max_duration = Column(Integer)
    conf_number = Column(String(10))
    csv_delimiter = Column(String(5))
    daily_payment_confirmation = Column(Boolean)
    daily_payment_email = Column(Text)
    date_format = Column(String(20))
    datetime_format = Column(String(20))
    de_pin_len = Column(Integer)
    default_billing_decimal = Column(Integer, nullable=False, default=6)
    default_report_decimal = Column(Integer, nullable=False, server_default='2')
    default_code_deck = Column(Integer)
    default_us_ij_rule = Column(ChoiceType(DEFAULT_US_IJ_RULE))
    doc_url = Column(String(512))
    dr_period = Column(String(10))
    egress_pdd_timeout = Column(Integer, nullable=False, default=6000)
    emailname = Column(String(50))
    emailpassword = Column(String(50))
    emailusername = Column(String(50))
    enable_first_time_wizard = Column(Boolean)
    enable_origination_traffic = Column(Boolean)
    enable_termination_traffic = Column(Boolean)
    enable_user_registration = Column(Boolean)
    events_alertszerotime = Column(Boolean)
    events_deleteafterdays = Column(String(10))
    events_notfoundaccount = Column(Boolean)
    events_notfoundtariff = Column(Boolean)
    events_unprofitable = Column(Boolean)
    fail_calls = Column(Integer)
    favicon_id = Column(Integer)
    finance_email = Column(String(1000))
    forbidden_times = Column(Integer)
    fromemail = Column(String(50))
    ftp_email = Column(String(50))
    ftp_pass = Column(String(20))
    ftp_username = Column(String(50))
    full_cdr_save_days = Column(Integer, default=30)
    inactivity_timeout = Column(Integer, default=30)
    ingress_pdd_timeout = Column(Integer, nullable=False, default=60000)
    invoice_decimal_digits = Column(Integer, default=2)
    invoice_name = Column(String(200))
    invoice_send_mode = Column(ChoiceType(INVOICE_SEND_MODE), default=0)
    invoice_send_mode_cdr = Column(ChoiceType(INVOICE_SEND_MODE_CDR), default=0)
    invoices_cdr_fields = Column(String(80))
    invoices_delay = Column(String(10))
    invoices_fields = Column(String(50))
    invoices_lastno = Column(String(20))
    invoices_logo_id = Column(Integer)
    invoices_separate = Column(String(10))
    invoices_tplno = Column(String(20))
    is_hide_unauthorized_ip = Column(SmallInteger, default=0)
    is_preload = Column(Boolean, default=True)
    is_show_mutual_balance = Column(SmallInteger, default=1)
    landing_page = Column(ChoiceType(LANDING_PAGE), default=0)
    login_captcha = Column(Boolean)
    login_fit_image = Column(Boolean)
    login_footer_color = Column(String(12))
    login_image = Column(String)
    login_page_content = Column(Text)
    loginemail = Column(String(12))
    logo_icon_id = Column(Integer)
    logo_image_id = Column(Integer)
    logs_deleteafterdays = Column(String(10))
    lowBalance_period = Column(Integer)
    low_call_attempt_handling = Column(ChoiceType(LOW_CALL_ATTEMPT_HANDLING))
    mail_server_from = Column(String(50))
    mailserver_host = Column(String(20))
    minimal_call_attempt_required = Column(Integer)
    msgmonthlyfee = Column(Numeric(20, 3))
    noc_email = Column(String(1000))
    non_zero_cdr_save_days = Column(Integer, default=60)
    notify_carrier = Column(Boolean)
    notify_carrier_cc = Column(Text)
    overlap_invoice_protection = Column(Boolean, default=True)
    payment_content = Column(Text)
    payment_received_confirmation = Column(Boolean)
    payment_setting_subject = Column(String(100))
    paypal_account = Column(String(255))
    paypal_skey = Column(String(255))
    paypal_test_mode = Column(Boolean)
    paypal_service_charge = Column(Integer, default=0)
    pdf_tpl = Column(Text)
    portal_display_sip_ip = Column(Boolean, default=True)
    portal_enable_egress_trunks = Column(Boolean, default=True)
    portal_enable_ingress_trunks = Column(Boolean, default=True)
    portal_enable_ip_change = Column(Boolean, default=True)
    portal_enable_public_products = Column(Boolean, default=True)
    portal_offset = Column(Boolean, default=True)
    public_logo_url = Column(String(20), server_default='/logo')
    qos_sample_period = Column(ChoiceType(QOS_SAMPLE_PERIOD))
    radius_log_routes = Column(String(10))
    rate_clean_days = Column(Integer)
    rates_deleteafterdays = Column(String(10))
    realm = Column(String)
    report_code_save_days = Column(Integer, default=60)
    report_count = Column(ChoiceType(REPORT_COUNT), default=0)
    report_daily_save_days = Column(Integer, default=180)
    report_hourly_save_days = Column(Integer, default=30)
    require_comment = Column(Integer)
    ring_timeout = Column(Integer, nullable=False, default=60)
    send_cdr_fields = Column(Text)
    signup_content = Column(String(200))
    simple_cdr_save_days = Column(Integer, default=180)
    smtp_secure = Column(Integer, default=0)
    smtphost = Column(String(50))
    smtpport = Column(String(50))
    stats_rotate_delay = Column(String(10))
    stripe_account = Column(String(255))
    stripe_public_account = Column(String(255))
    stripe_service_charge = Column(Integer, default=0)
    switch_alias = Column(String(255))
    switch_ip = Column(String(20))
    switch_port = Column(String(20))
    sys_ani = Column(PrefixRange)
    sys_area = Column(String(50))
    sys_currency = Column(String(20))
    sys_timezone = Column(String(20))
    system_admin_email = Column(String(1000))
    system_rate_mail = Column(String(255))
    themer = Column(Integer, default=0)
    tpl_number = Column(Integer, default=0)
    welcome_message = Column(String)
    withdraw_email = Column(String(1000))
    workstation = Column(String)
    yourpay_store_number = Column(String(255))

    rel_code_deck = relationship(CodeDeck, primaryjoin=foreign(default_code_deck) == CodeDeck.code_deck_id)
    # cmd_debug = Column(String(200))
    # payment_from = Column(Text)
    # payment_subject = Column(Text)
    # payment_from_cc = Column(Text)
    # login_fit_image = Column(Boolean)
    # cdr_token = Column(String(255))
    # cdr_token_alias = Column(String(255))
    # payment section
    cc_email = synonym('notify_carrier_cc')
    # credit_full_payment = synonym(credit_full_payment)
    emails = synonym('finance_email')
    enable_email_notification = synonym('notify_carrier')

    operating_company = synonym('company_name')
    # paypal_account
    paypal_fee = synonym('paypal_service_charge')
    stripe_fee = synonym('stripe_service_charge')
    stripe_publisher_key = synonym('stripe_public_account')
    stripe_secret_key = synonym('stripe_account')
    # --- login page synonyms ---
    image_url = synonym('login_image')
    fit_to_screen = synonym('login_fit_image')
    use_captcha = synonym('login_captcha')
    content = synonym('login_page_content')
    # mail sender
    smtp_host = synonym('mailserver_host')
    smtp_port = synonym('smtpport')
    username = synonym('emailusername')
    password = synonym('emailpassword')
    email = synonym('fromemail')
    # ---
    # ---Invoice settings synonyms---#
    # allow_invoice_overlap = synonym('overlap_invoice_protection')
    # billing_info = synonym('billing_info')#??
    billing_info_position = synonym('billing_info_location')  # ??
    # cdr_fields = synonym('send_cdr_fields')
    # +company_info = synonym()
    company_info_position = synonym('company_info_location')
    decimal = synonym('invoice_decimal_digits')
    invoice_number_format = synonym('invoices_lastno')
    # logo = synonym()
    logo_url = synonym('login_image')
    send_rate_as = synonym('system_rate_mail')
    pre_load_data = synonym('is_preload')

    paypal_charge_type = synonym('charge_type')

    @classmethod
    def init(cls):
        q = cls.get(1)
        if not q:
            q = cls(sys_id=1)
            q.save()

    @hybrid_property
    def _call_timeout(self):
        return self.call_timeout

    @_call_timeout.setter
    def _call_timeout(self, value):
        self.call_timeout = value if value is not None else 86400

    @hybrid_property
    def allow_invoice_overlap(self):
        return not bool(self.overlap_invoice_protection)

    @allow_invoice_overlap.setter
    def allow_invoice_overlap(self, value):
        self.overlap_invoice_protection = not bool(value)

    @hybrid_property
    def cdr_fields(self):
        try:
            return self.send_cdr_fields.split(',')
        except:
            return []

    @cdr_fields.setter
    def cdr_fields(self, value):
        try:
            self.send_cdr_fields = ','.join([str(x) for x in value])
        except:
            pass

    @hybrid_property
    def code_deck(self):
        try:
            return self.rel_code_deck.name
        except:
            return ''

    @code_deck.setter
    def code_deck(self, value):
        try:
            self.default_code_deck = CodeDeck.filter(CodeDeck.name == value).one().code_deck_id
        except:
            # self.default_code_deck = 1
            raise ValidationError('bad code deck:{}'.format(value))

    @hybrid_property
    def currency_name(self):
        try:
            return self.currency.code
        except:
            return ''

    @currency_name.setter
    def currency_name(self, value):
        try:
            self.currency_id = Currency.filter(Currency.code == value).one().currency_id
        except:
            pass

    @hybrid_property
    def _logo_url(self):
        try:
            return '{}/config/export/public/{}'.format(settings.API_URL, self.invoices_logo_id)
        except:
            return None

    @hybrid_property
    def _logo_path(self):
        try:
            return ImportExportLogs.get(self.invoices_logo_id).file
        except:
            return None

    @hybrid_property
    def _image_url(self):
        try:
            return '{}/config/export/public/{}'.format(settings.API_URL, self.image_url) if self.image_url else ''
        except:
            return None

    @_image_url.setter
    def _image_url(self, value):
        try:
            self.image_url = int(value.split('/')[-1]) if value else ''
        except:
            pass

    @property
    def root_url(self):
        url = parse_url(self.base_url)
        return '{}://{}'.format(url.scheme, url.netloc)

    @property
    def webhook_stripe_url(self):
        return settings.API_URL + '/stripe/webhook'

    @property
    def webhook_paypal_url(self):
        return settings.API_URL + '/paypal/webhook'


# ---SystemParameter

# +++JurisdictionPrefix
class JurisdictionPrefix(DnlApiBaseModel):
    __tablename__ = 'jurisdiction_prefix'

    id = Column(Integer, primary_key=True)
    alias = Column(String(40))
    prefix = Column(PrefixRange, nullable=False, index=True, unique=True)
    jurisdiction_id = Column(Integer, index=True,server_default='1')  # The charging area
    jurisdiction_country_id = Column(Integer, index=True, server_default='1')  # country
    jurisdiction_name = Column(String(100))  # state
    jurisdiction_country_name = Column(String(100))  # , unique=True???)
    ocn = Column(String(10))
    lata = Column(String(10))
    block_id = Column(String(1))
    effective_date = Column(DateTime(True))

    # code = synonym('prefix')
    country = synonym('jurisdiction_country_name')
    state = synonym('jurisdiction_name')

    def before_save(self):
        if self.jurisdiction_id is None:
            if self.jurisdiction_country_name in ('US','CA'):
                self.jurisdiction_id=1
            else:
                self.jurisdiction_id = 0


# ---JurisdictionPrefix

# +++ LoopDetection
class LoopDetectionDetail(DnlApiBaseModel):
    __tablename__ = 'loop_detection_detail'

    id = Column(Integer, primary_key=True)
    loop_detection_id = Column(ForeignKey('loop_detection.id', ondelete='CASCADE'), nullable=False)
    resource_id = Column(ForeignKey('resource.resource_id', ondelete='CASCADE'), unique=True)
    trunk_id = synonym('resource_id')
    trunk_name = column_property(
        select([Resource.alias]).where(Resource.resource_id == resource_id).correlate_except(Resource))
    counter_time = column_property(
        select([Resource.counter_time]).where(Resource.resource_id == resource_id).correlate_except(Resource))
    block_time = column_property(
        select([Resource.block_time]).where(Resource.resource_id == resource_id).correlate_except(Resource))
    number = column_property(
        select([Resource.number]).where(Resource.resource_id == resource_id).correlate_except(Resource))

    parent = relationship('LoopDetection', uselist=False, back_populates='ingress_trunks')
    trunk = relationship('Resource', primaryjoin='foreign(LoopDetectionDetail.resource_id)==Resource.resource_id',
                         uselist=False)


class LoopDetection(DnlApiBaseModel):
    __tablename__ = 'loop_detection'

    id = Column(Integer, primary_key=True)
    rule_name = Column(String)
    number = Column(Integer)
    counter_time = Column(Integer)
    block_time = Column(Integer)

    block_second = synonym('block_time')
    name = synonym('rule_name')
    occurance = synonym('number')
    period = synonym('counter_time')

    ingress_trunks = relationship(LoopDetectionDetail, uselist=True, back_populates='parent',
                                  single_parent=True,
                                  cascade="all, delete-orphan")
    trunk_count = column_property(select([func.count(LoopDetectionDetail.id)]).where(
        LoopDetectionDetail.loop_detection_id == id).correlate_except(LoopDetectionDetail))

    @hybrid_property
    def _ingress_trunks(self):
        return self.ingress_trunks

    @_ingress_trunks.setter
    def _ingress_trunks(self, value):
        try:
            self.reset_trunks()
            for resource in value:
                trunk = LoopDetectionDetail.filter(LoopDetectionDetail.resource_id == getattr(resource, 'trunk_id', -1)).first()
                if trunk:
                    self.ingress_trunks.append(trunk)
                elif getattr(resource, 'trunk_id', None):
                    self.ingress_trunks.append(resource)
            self.update_trunks()
        except Exception as e:
            log.warning(str(e))
            pass

    @validates('rule_name')
    def validate_name(self, key, value):
        old_obj = LoopDetection.filter(and_(LoopDetection.rule_name == value, LoopDetection.id != self.id)).first()
        if old_obj:
            raise Exception('rule_name is duplicate')

        rule_name_re = r'[A-Za-z0-9]+[\w@_\s]+[A-Za-z0-9]'

        if re.search(rule_name_re, value) and re.search(rule_name_re, value).group() == value:
            return value
        else:
            raise Exception('rule_name is invalid')

    def update_trunks(self):
        for tr in self.ingress_trunks:
            if tr.trunk:
                tr.trunk.counter_time = self.counter_time
                tr.trunk.block_time = self.block_time
                tr.trunk.number = self.number
                self.session().add(tr.trunk)

    def reset_trunks(self):
        for tr in self.ingress_trunks:
            if tr.trunk:
                tr.trunk.counter_time = None
                tr.trunk.block_time = None
                tr.trunk.number = None
                self.session().add(tr.trunk)


"""
LoopDetection.trunk_count = column_property(select([func.coalesce(func.count(LoopDetectionDetail.trunk_id), text_("0"))]). \
                    where(LoopDetectionDetail.loop_detection_id == LoopDetection.id)
                                            .correlate(LoopDetectionDetail).label('trunk_count'))
"""


# ---LoopDetection


# +++ FraudDetection
class FraudDetection(DnlApiBaseModel):
    __tablename__ = 'fraud_detection'
    SEND_TO = {1: 'Own NOC Email', 2: 'Partner NOC Email', 3: 'Both'}
    id = Column(Integer, primary_key=True)
    rule_name = Column(String(100), unique=True)
    active = Column(Boolean, nullable=False, server_default=text_("true"))
    hourly_minute = Column(Integer)
    hourly_revenue = Column(Integer)
    daily_minute = Column(Integer)
    daily_revenue = Column(Integer)
    is_block = Column(Boolean)
    is_send_mail = Column(Boolean)
    email_to = Column(ChoiceType(SEND_TO))
    ingress_ids = Column(String(500))
    update_on = Column(DateTime(timezone=True), onupdate=func.now(), server_default=func.now())
    update_by = Column(String(100))

    # ---synonyms---#
    hour_24_duration = synonym('daily_minute')
    hour_24_revenue = synonym('daily_revenue')
    enable_block = synonym('is_block')
    enable_email = synonym('is_send_mail')

    mail_sender_id = synonym('email_to')
    name = synonym('rule_name')
    one_hour_duration = synonym('hourly_minute')
    one_hour_revenue = synonym('hourly_revenue')

    log = relationship('FraudDetectionLog', uselist=True, back_populates='fraud_detection', single_parent=True,
                       cascade="all, delete-orphan")

    @hybrid_property
    def ingress_trunks(self):
        try:
            return self.ingress_ids.split(',')
        except:
            return []

    @ingress_trunks.setter
    def ingress_trunks(self, value):
        try:
            self.ingress_ids = ','.join([str(x) for x in value])
        except:
            pass


class FraudDetectionLog(DnlApiBaseModel):
    __tablename__ = 'fraud_detection_log'
    STATUS = {0: 'normal', 1: 'over limit'}
    id = Column(Integer, primary_key=True,
                server_default=text_("nextval('fraud_detection_log_id_seq'::regclass)"))
    fraud_detection_id = Column(ForeignKey('fraud_detection.id', ondelete='CASCADE'))
    create_on = Column(DateTime(True), server_default=func.now())
    create_by = Column(Integer)
    status = Column(ChoiceType(STATUS))
    finish_time = Column(DateTime(True))
    time = Column(Numeric, index=True, server_default=text_('EXTRACT(EPOCH from current_timestamp(0))'))

    fraud_detection = relationship('FraudDetection', uselist=False, back_populates='log')
    detail = relationship('FraudDetectionLogDetail', uselist=True, back_populates='fraud_detection_log',
                          single_parent=True,
                          cascade="all, delete-orphan")

    rule_name = column_property(
        select([FraudDetection.rule_name]).where(FraudDetection.id == fraud_detection_id).correlate_except(
            FraudDetection))
    create_by_name = column_property(
        select([case([(create_by == 1, 'auto')], else_=User.name)]).where(User.user_id == create_by).correlate_except(
            User))
    duration = column_property(finish_time - create_on)


class FraudDetectionLogDetail(DnlApiBaseModel):
    __tablename__ = 'fraud_detection_log_detail'
    BLOCK_TYPE = {0: '1 hour minute', 1: '1 hour revenue', 2: '24 hour minute', 3: '24 hour revenue'}
    id = Column(Integer, primary_key=True,
                server_default=text_("nextval('fraud_detection_log_detail_id_seq'::regclass)"))
    fraud_detection_log_id = Column(ForeignKey('fraud_detection_log.id', ondelete='CASCADE'))
    ingress_id = Column(Integer)
    block_type = Column(ChoiceType(BLOCK_TYPE))
    limit_value = Column(Integer)
    actual_value = Column(String(50))
    partner_email_msg = Column(Text)
    partner_email_status = Column(Boolean)
    partner_email = Column(String(200))
    system_email_msg = Column(Text)
    system_email_status = Column(Boolean)
    system_email = Column(String(200))
    is_block = Column(Boolean, server_default=text_("false"))
    is_send_email = Column(Boolean, server_default=text_("false"))
    fraud_detection_id = column_property(select([FraudDetectionLog.fraud_detection_id]). \
        where(FraudDetectionLog.id == fraud_detection_log_id).correlate_except(
        FraudDetectionLog))
    finish_time = column_property(select([FraudDetectionLog.finish_time]). \
                                  where(
        FraudDetectionLog.id == fraud_detection_log_id).correlate_except(FraudDetectionLog))

    fraud_detection_log = relationship('FraudDetectionLog', uselist=False, back_populates='detail')
    trunk_name = column_property(
        select([Resource.alias]).where(Resource.resource_id == ingress_id).correlate_except(
            Resource))


# --- FraudDetection

# region  +++ FaultRouteAlertRule
class FaultRouteAlertRule(DnlApiBaseModel):
    __tablename__ = 'fault_route_alert_rule'
    id = Column(Integer, primary_key=True)
    rule_name = Column(String(100), unique=True)
    # schedule
    weekly_days = Column(String(7))
    daily_time = Column(Time())
    # criteria
    # ProductRoutRateTable.id
    product_id = Column(ForeignKey('product_rout_rate_table.id', ondelete='CASCADE'))
    routing_plan_id = Column(Integer)
    # conditions
    look_days = Column(Integer)
    min_profitable_trunks = Column(Integer)
    min_profitability = Column(Float)
    # action
    from_mail_id = Column(Integer)
    email_to = Column(String(100))
    subject = Column(String(200))
    html_content = Column(Text)
    # results
    is_active = Column(Boolean, server_default='true')
    last_run_on = Column(DateTime(timezone=True))
    next_run_on = Column(DateTime(timezone=True))

    @property
    def cc_mail(self):
        return ''

    @property
    def to_mail(self):
        return ''

    def get_attachment(self, env):
        return []

    @property
    def wdays(self):
        if self.weekly_days:
            wd = self.weekly_days.split(',')
            return [int(d) - 1 for d in wd]
        return []

    @property
    def frequency(self):
        if self.weekly_days:
            return 'Weekly'
        return 'Daily'

    @property
    def based_on(self):
        if self.routing_plan_id:
            return 'Product'
        return 'Routing Plan'

    def next_run(self, d):
        if self.weekly_days:
            for i in range(1, 7):
                nd = d.replace(hour=0, minute=0, second=0, microsecond=0) + timedelta(days=i)
                ndwd = nd.weekday()
                if ndwd in self.wdays:
                    return nd
        else:
            h = self.daily_time.hour
            m = self.daily_time.minute
            s = self.daily_time.second
            nd = d.replace(hour=h, minute=m, second=s, microsecond=0)
            if nd < d:
                nd = nd + timedelta(days=1)
            return nd


class FaultRouteAlertRuleLog(DnlApiBaseModel):
    __tablename__ = 'fault_route_alert_rule_log'
    STATUS = {0: 'ok', 1: 'alert', 2: 'error'}
    id = Column(Integer, primary_key=True)
    rule_id = Column(ForeignKey('fault_route_alert_rule.id', ondelete='CASCADE'))
    started_on = Column(DateTime(timezone=True))
    finished_on = Column(DateTime(timezone=True))
    result = Column(Text)
    status = Column(ChoiceType(STATUS))
    mail_sent = Column(Boolean, server_default='false')
    profitable_trunks = Column(Integer)
    profitability = Column(Float)
    rule_name = column_property(
        select([FaultRouteAlertRule.rule_name]).where(FaultRouteAlertRule.id == rule_id).correlate_except(
            FaultRouteAlertRule))
    min_profitable_trunks = column_property(
        select([FaultRouteAlertRule.min_profitable_trunks]).where(FaultRouteAlertRule.id == rule_id).correlate_except(
            FaultRouteAlertRule))
    min_profitability = column_property(
        select([FaultRouteAlertRule.min_profitability]).where(FaultRouteAlertRule.id == rule_id).correlate_except(
            FaultRouteAlertRule))


# endregion  +++ FaultRouteAlertRule
# +++ AlertRule
class AlertRule(DnlApiBaseModel):
    __tablename__ = 'alert_rules'
    DAYS_OF_WEEK = {1: "Monday", 2: "Tuesday", 3: "Wednesday", 4: "Thursday", 5: "Friday", 6: "Saturday", 7: "Sunday"}
    FREQ_TYPE = {1: 'every xx minutes', 2: 'every week', 0: 'never'}
    TRUNK_TYPE = {None: 'all', 1: 'ingress', 2: 'egress'}
    INCLUDE = {0: 'All Codes', 1: 'Specific Codes', 2: 'Specific Code Names'}
    EXCLUDE = {0: 'None', 1: 'Specific Codes', 2: 'Specific Code Names'}
    STEP3_TYPE = {1: 'Disable Entire Trunk', 2: 'Disable Specific Code',
                  3: 'Disable Specific Code Name'}
    DISABLE_SCOPE = {1: 'Disable Entire Trunk', 2: 'Disable Specific Code',
                     3: 'Disable Specific Code Name'}
    SEND_TO = {1: 'Own NOC Email', 2: 'Partner NOC Email', 3: 'Both'}
    EXECUTION_SCHEDULE = {0: 'never', 1: 'Every Specific Minutes', 2: 'Daily', 3: 'Weekly'}
    MONITOR_BY = {0: 'Trunk', 1: 'Trunk And DNIS', 2: 'Trunk And ANI',
                  3: 'DNIS', 4: 'ANI', 5: 'Trunk And Destination', 6: 'Trunk And Country', 7: 'Trunk And Code', }
    AUTO_ENABLE_TYPE = {1: 'yes', 0: 'no'}
    SDP_SIGN = {0: 'ignore', 1: '=', 2: '<', 3: '>'}
    SDP_TYPE = {0: 'none', 1: '6sec', 2: '12sec', 3: '18sec', 4: '24sec', 5: '30sec'}

    id = Column(Integer, primary_key=True)
    rule_name = Column(String(100), unique=True)
    abr = Column(String(1))
    abr_value = Column(Float)
    acd = Column(String(1))
    acd_value = Column(Float)
    active = Column(Boolean, default=True)
    all_trunk = Column(Boolean, nullable=False, default=False)
    asr = Column(String(1))
    asr_value = Column(Float)
    auto_enable = Column(Integer)
    auto_enable_type = Column(ChoiceType(AUTO_ENABLE_TYPE))
    daily_time = Column(SmallInteger)
    disable_scope = Column(ChoiceType(DISABLE_SCOPE))
    ex_code_deck = Column(Integer)
    ex_code_name_id = Column(String(100))
    ex_codes = Column(String(500))
    exclude = Column(ChoiceType(EXCLUDE))
    execution_schedule = Column(ChoiceType(EXECUTION_SCHEDULE))
    in_code_deck = Column(Integer)
    in_code_name_id = Column(String(100))
    in_codes = Column(String(500))
    include = Column(ChoiceType(INCLUDE), nullable=False, default=0)
    is_block = Column(Boolean, default=True)
    is_email = Column(Boolean, default=True)
    last_run_time = Column(DateTime(timezone=True))
    min_call_attempt = Column(Integer)
    monitor_by = Column(ChoiceType(MONITOR_BY), default=0)
    pdd = Column(String(1))
    pdd_value = Column(Float(53))
    profitability = Column(String(1))
    profitability_value = Column(Float)
    res_id = Column(Text)
    revenue = Column(String(1))
    revenue_value = Column(Float)
    sample_size = Column(Integer)
    sdp_sign = Column(ChoiceType(SDP_SIGN), server_default=text_('0'))
    sdp_type = Column(ChoiceType(SDP_TYPE), server_default=text_('0'))
    sdp_value = Column(Integer, server_default=text_('0'))
    specific_minutes = Column(Integer)
    status = Column(Boolean, default=False)
    step3_type = Column(ChoiceType(STEP3_TYPE), nullable=False, default=2)
    trouble_ticket_content = Column(Text)
    trouble_ticket_sent_from = Column(Integer)
    trouble_ticket_sent_to = Column(ChoiceType(SEND_TO))
    trouble_ticket_subject = Column(String(100))
    trunk_type = Column(ChoiceType(TRUNK_TYPE), nullable=False, default=1)
    update_at = Column(DateTime(timezone=True), onupdate=func.now())
    update_by = Column(String(100))
    weekly_time = Column(SmallInteger)
    weekly_value = Column(ChoiceType(DAYS_OF_WEEK))
    next_run_time = Column(DateTime(timezone=True))

    # ---synonyms---#
    name = synonym('rule_name')
    acd_operator = synonym('acd')
    # acd_value = synonym()
    activate = synonym('active')
    all_trunks = synonym('all_trunk')
    asr_operator = synonym('asr')
    # asr_value = synonym()
    # auto_unblock = synonym('auto_enable')
    auto_unblock_after = synonym('auto_enable_type')
    block = synonym('is_block')
    # exclude_codes = synonym()
    min_attempt = synonym('min_call_attempt')
    monitor_ingress = synonym('status')
    name = synonym('rule_name')
    pdd_operator = synonym('pdd')
    # pdd_value = synonym()
    profitability_operator = synonym('profitability')
    # profitability_value = synonym()
    revenue_operator = synonym('revenue')
    # revenue_value = synonym()
    run_day_of_week = synonym('weekly_value')
    run_every = synonym('weekly_time')
    run_on = synonym('daily_time')
    run_every_minute = synonym('specific_minutes')
    sample_period = synonym('sample_size')
    schedule = synonym('execution_schedule')
    scope = synonym('disable_scope')
    send_email = synonym('is_email')
    from_mail_id = synonym('trouble_ticket_sent_from')
    html_content = synonym('trouble_ticket_content')
    ##include_codes = synonym()
    send_to = synonym('trouble_ticket_sent_to')
    subject = synonym('trouble_ticket_subject')

    # trunks = synonym()

    @property
    def _next_run_time(self):
        # 0: 'never', 1: 'Every Specific Minutes', 2: 'Daily', 3: 'Weekly'
        now = datetime.now(UTC)
        now_timestamp = int(now.timestamp())  # mktime(now.timetuple())
        now_day = now.day
        now_hour = now.hour
        now_wday = now.weekday()

        last_run_time = None
        last_run_day = None
        last_run_hour = None
        last_run_wday = None
        if self.last_run_time:
            last_run_time = int(self.last_run_time.timestamp())  # mktime(self.last_run_time.timetuple())
            lr_utc = datetime.fromtimestamp(last_run_time, tz=UTC)
            last_run_day = lr_utc.day
            last_run_hour = lr_utc.hour
            last_run_wday = lr_utc.weekday()

        if self.execution_schedule == 'Every Specific Minutes':
            if not self.last_run_time:
                return now
            every_min = self.specific_minutes if self.specific_minutes else 1
            next_plan_run_time = self.last_run_time + timedelta(minutes=every_min)
            while next_plan_run_time < now:
                next_plan_run_time = next_plan_run_time + timedelta(minutes=every_min)
            return next_plan_run_time
        elif self.execution_schedule == 'Daily':
            if not self.last_run_time:
                return now
            next_plan_run_time = self.last_run_time + timedelta(hours=24)
            if next_plan_run_time.hour != self.daily_time:
                next_plan_run_time.hour = self.daily_time
            while next_plan_run_time < now:
                next_plan_run_time = next_plan_run_time + timedelta(hours=24)
            return next_plan_run_time
        elif self.execution_schedule == 'Weekly':
            week_time = self.weekly_time
            week_day = self.weekly_value
            if now_wday != last_run_wday:
                if int(now_hour) == int(week_time) and int(now_wday) == int(week_day):
                    return now
            else:
                next_plan_run_time = self.last_run_time + timedelta(days=7)
                if next_plan_run_time.weekday() != self.weekly_value - 1:
                    next_plan_run_time = next_plan_run_time + timedelta(
                        self.weekly_value - 1 - next_plan_run_time.weekday())
                while next_plan_run_time < now:
                    next_plan_run_time = next_plan_run_time + timedelta(days=7)
                return next_plan_run_time
        else:
            return None

    @hybrid_property
    def auto_unblock(self):
        try:
            return bool(self.auto_enable)
        except:
            return None

    @auto_unblock.setter
    def auto_unblock(self, value):
        try:
            self.auto_enable = int(value)
        except:
            pass

    @hybrid_property
    def exclude_codes(self):
        try:
            return self.ex_codes.split(',')
        except:
            return []

    @exclude_codes.setter
    def exclude_codes(self, value):
        try:
            self.ex_codes = ','.join([str(x) for x in value])
        except:
            pass

    @hybrid_property
    def include_codes(self):
        try:
            return self.in_codes.split(',')
        except:
            return []

    @include_codes.setter
    def include_codes(self, value):
        try:
            self.in_codes = ','.join([str(x) for x in value])
        except:
            pass

    @hybrid_property
    def trunks(self):
        try:
            return self.res_id.split(',')
        except:
            return []

    @trunks.setter
    def trunks(self, value):
        try:
            self.res_id = ','.join([str(x) for x in value])
        except:
            pass


class AlertRuleLog(DnlApiBaseModel):
    __tablename__ = 'alert_rules_log'
    STATUS = {0: 'normal', 1: 'over limit'}
    id = Column(Integer, primary_key=True, server_default=text_("nextval('alert_rules_log_id_seq'::regclass)"))
    alert_rules_id = Column(ForeignKey('alert_rules.id', ondelete='CASCADE'))
    create_on = Column(DateTime(True))
    status = Column(ChoiceType(STATUS))
    finish_time = Column(DateTime(True))
    limit_acd_value = Column(Float)
    limit_asr_value = Column(Float)
    limit_abr_value = Column(Float)
    limit_pdd_value = Column(Float(53))
    limit_profitability_value = Column(Float)
    limit_revenue_value = Column(Float)
    limit_asr = Column(String(1))
    limit_abr = Column(String(1))
    limit_acd = Column(String(1))
    limit_pdd = Column(String(1))
    limit_profitability = Column(String(1))
    limit_revenue = Column(String(1))
    time = Column(Numeric, index=True, server_default=text_('EXTRACT(EPOCH from current_timestamp(0))'))

    detail = relationship('AlertRulesLogDetail', uselist=True, back_populates='log')
    rule_name = column_property(
        select([AlertRule.rule_name]).where(AlertRule.id == alert_rules_id).correlate_except(AlertRule))


class AlertRulesLogDetail(DnlApiBaseModel):
    __tablename__ = 'alert_rules_log_detail'

    id = Column(Integer, primary_key=True, server_default=text_("nextval('alert_rules_log_detail_id_seq'::regclass)"))
    update_at = Column(DateTime(True), server_default=text_("('now'::text)::timestamp(0) with time zone"))
    alert_rules_log_id = Column(ForeignKey('alert_rules_log.id', ondelete='CASCADE'))
    resource_id = Column(Integer)
    code = Column(String(50))
    asr = Column(Float)
    abr = Column(Float)
    acd = Column(Float)
    pdd = Column(Float)
    profitability = Column(Float)
    revenue = Column(Float)
    system_email_address = Column(String(100))
    system_email_status = Column(Boolean)
    partner_email_address = Column(String(100))
    partner_email_status = Column(Boolean)
    is_email = Column(Boolean)
    is_block = Column(Boolean)
    resource_block_id = Column(Integer)
    email_type = Column(Integer)
    time = Column(Numeric, index=True, server_default=text_('EXTRACT(EPOCH from current_timestamp(0))'))
    trunk_id = synonym('resource_id')
    log_id = synonym('alert_rules_log_id')

    log = relationship('AlertRuleLog', uselist=False, back_populates='detail')

    partner_status = column_property(case([(partner_email_status.is_(None), literal_column("'No sent'")),
                                           (partner_email_status == True, literal_column("'Success'"))],
                                          else_=literal_column("'Fail'")))
    system_status = column_property(case([(partner_email_status.is_(None), literal_column("'No sent'")),
                                          (partner_email_status == True, literal_column("'Success'"))],
                                         else_=literal_column("'Fail'")))
    status = column_property(
        select([AlertRuleLog.status]).where(AlertRuleLog.id == alert_rules_log_id).correlate_except(AlertRuleLog))
    start_time = column_property(
        select([AlertRuleLog.create_on]).where(AlertRuleLog.id == alert_rules_log_id).correlate_except(AlertRuleLog))
    finish_time = column_property(
        select([AlertRuleLog.finish_time]).where(AlertRuleLog.id == alert_rules_log_id).correlate_except(AlertRuleLog))
    rule_name = column_property(
        select([AlertRuleLog.rule_name]).where(AlertRuleLog.id == alert_rules_log_id).correlate_except(AlertRuleLog))
    rule_id = column_property(
        select([AlertRuleLog.alert_rules_id]).where(AlertRuleLog.id == alert_rules_log_id).correlate_except(
            AlertRuleLog))
    trunk_name = column_property(
        select([Resource.alias]).where(Resource.resource_id == resource_id).correlate_except(Resource))


# ---AlertRule

# t_allowed_sendto_ip = Table(
#    'allowed_sendto_ip', DnlApiBaseModel.metadata,
#    Column('id', Integer, nullable=False, server_default=text_("nextval('allowed_sendto_ip_id_seq'::regclass)")),
#    Column('resource_id', Integer),
#    Column('direction', Integer),
#    Column('sip_profile_ip', Ip4r),
#    Column('sip_profile_port', Integer)
# )
# +++ AuthorizationLog

# ---AuthorizationLog

# +++ BalanceDailyResetTask
class BalanceDailyResetTask(DnlApiBaseModel):
    __tablename__ = 'balance_daily_reset_task'

    STATUS = {0: 'initial', 1: 'in process', 2: 'finished', 3: 'error'}
    id = Column(Integer, primary_key=True)
    start_time = Column(DateTime(True))  # The start time of balance daily resetting
    reset_balance = Column(SmallInteger, server_default=text_('NULL'))  # 1 for reset client balance, other for ignore
    mutual = Column(SmallInteger, server_default=text_('NULL'))  # 1 for reset mutual balance, other for ignore
    actual = Column(SmallInteger, server_default=text_('NULL'))  # 1 for reset actual balance, other for ignore
    client_id = Column(Integer)  # 0 for all client, other is the specified client ID
    status = Column(ChoiceType(STATUS), FetchedValue())  # 0 for initial; 1 for in process, 2 for finished; 3 for error
    finished_time = Column(DateTime(True), FetchedValue())
    create_by = Column(String(100))
    create_time = Column(DateTime(True), nullable=False, server_default=func.now())
    did_billing = Column(SmallInteger, server_default=text_('NULL'), doc='Reset did_transaction table')
    new_amount = Column(Numeric)
    old_amount = Column(Numeric)

    create_at = synonym('create_time')
    client = relationship(Client, primaryjoin=foreign(client_id) == Client.client_id, uselist=False)

    client_name = column_property(
        select([Client.name]).where(Client.client_id == client_id).correlate_except(Client))

    @hybrid_property
    def reset_mutual_balance(self):
        try:
            return bool(self.mutual)
        except:
            return False

    @reset_mutual_balance.setter
    def reset_mutual_balance(self, value):
        self.mutual = int(value)

    @hybrid_property
    def reset_actual_balance(self):
        try:
            return bool(self.actual)
        except:
            return False

    @reset_actual_balance.setter
    def reset_actual_balance(self, value):
        self.actual = int(value)

    @hybrid_property
    def new_balance(self):
        try:
            return self.client.c4.balance
        except:
            return None

    @new_balance.setter
    def new_balance(self, value):
        if self.client_id:
            a = ClientBalanceOperationAction(client_id=self.client_id, balance=value, action='')
            # ClientBalanceOperationAction.action=''
            self.session().add(a)

    def do_reset(self):
        if self.start_time:
            d = self.start_time.date()  # -timedelta(days=1)
        else:
            d = datetime.today().date()  # -timedelta(days=1)
        try:
            cl = Client.get(self.client_id)
            value = cl.c4.balance
            if self.mutual and self.client_id:
                q = BalanceHistory.filter(BalanceHistory.client_id == self.client_id).filter(
                    BalanceHistory.date == d).first()
                if not q:
                    q = BalanceHistory(date=d, client_id=self.client_id, mutual_balance=value,
                                       mutual_ingress_balance=value)
                if q:
                    q.mutual_balance = value
                    q.mutual_ingress_balance = value
                    q.save()
            if self.actual and self.client_id:
                q = BalanceHistoryActual.filter(BalanceHistoryActual.client_id == self.client_id).filter(
                    BalanceHistoryActual.date == d).first()
                if not q:
                    q = BalanceHistoryActual(date=d, client_id=self.client_id, actual_balance=value,
                                             actual_ingress_balance=value)
                if q:
                    q.actual_balance = value
                    q.actual_ingress_balance = value
                    q.payment_received = 0;
                    q.credit_note_sent = 0;
                    q.debit_note_sent = 0;
                    q.unbilled_incoming_traffic = 0; \
                            q.short_charges = 0;
                    q.payment_sent = 0;
                    q.unbilled_outgoing_traffic = 0; \
                            q.actual_egress_balance = 0
                    q.save()
            self.reset_balance = 1
            try:
                if self.client_id:
                    cl.regenerate_balance()
            # from api_dnl.tasks import carrier_regenerate_balance
            # carrier_regenerate_balance.delay(self.client_id)

            # Client.get(self.client_id).regenerate_balance()
            except:
                pass
        except Exception as e:
            log.error(format_exc())
            return None


# +++ BalanceDailyResetTask

# +++ PaymentGatewayHistory
class PaymentGatewayHistory(DnlApiBaseModel):
    __tablename__ = 'payment_gateway_history'
    METHOD = {None: 'undefined', 0: 'paypal', 1: 'stripe'}
    STATUS = {0: 'initial', 1: 'error', 2: 'success'}
    id = Column(Integer, primary_key=True)
    chargetotal = Column(Numeric())
    method = Column(ChoiceType(METHOD))
    cardnumber = Column(String())
    cardexpmonth = Column(String())
    cardexpyear = Column(String())
    created_time = Column(DateTime(True), server_default=func.now())
    modified_time = Column(DateTime())
    error = Column(String())
    confirmed = Column(Boolean(), default=False)
    client_id = Column(ForeignKey('client.client_id', ondelete='SET NULL'), index=True)
    fee = Column(Numeric())
    status = Column(ChoiceType(STATUS), default=0, index=True)
    invoice_id = Column(String(255))
    par_id = Column(Integer())
    address1 = Column(String(255))
    address2 = Column(String(255))
    city = Column(String(255))
    state_province = Column(String(255))
    zip_code = Column(String(255))
    country = Column(String(255))
    credit_card_type = Column(Integer())
    paypal_id = Column(String(100))
    transaction_id = Column(String(100), index=True)
    return_code = Column(String(100))
    charge_type = Column(Numeric(), default=0)
    charge_amount = Column(Numeric(), default=0)
    response = Column(String())
    from_ip = Column(String(36))
    transaction_src = Column(JSON())
    # synonyms
    type = synonym('method')
    paid_on = synonym('created_time')
    entered_on = synonym('modified_time')
    paypal_fee = synonym('fee')
    amount = synonym('charge_amount')
    actual_received = synonym('chargetotal')
    strip_id = synonym('paypal_id')
    strip_transaction_id = synonym('transaction_id')
    paypal_transaction_id = synonym('transaction_id')

    client = relationship(Client, uselist=False, back_populates='payments_history')

    client_name = column_property(
        select([Client.name]).where(Client.client_id == client_id).correlate_except(Client))


ClientPayment.transaction_id = column_property(func.coalesce(select([PaymentGatewayHistory.transaction_id]).where(
    ClientPayment.client_payment_id == PaymentGatewayHistory.par_id).as_scalar(),
                                                             'local_' + cast(ClientPayment.client_payment_id, String)))


# ---PaymentGatewayHistory
# +++BalanceLog
# ---BalanceLog
# +++CreditLog
# ---CreditLog

# +++ AuthToken

class AuthToken(DnlApiBaseModel):
    __tablename__ = 'auth_token'

    id = Column(Integer, primary_key=True, nullable=False)
    token = Column(String(150))
    user_name = Column(String(30))
    start_time = Column(DateTime(True))
    expired_on = Column(DateTime(True))
    is_disabled = Column(Boolean, server_default=text_("false"))
    last_used = Column(DateTime(True))


# --- AuthToken

# +++CdrExportLog
# ---CdrExportLog

class ImportExportLogs(DnlApiBaseModel):
    __tablename__ = 'import_export_logs'
    STATUS = {0: 'success', 1: 'fail', 2: 'initial', 3: 'in progress'}
    LOG_TYPE = {0: 'export', 1: 'import'}
    UPLOAD_TYPE = {0: 'http', 1: 'ftp', 2: 'mail'}
    id = Column(Integer, primary_key=True, nullable=False,
                server_default=text_("nextval('import_export_logs_id_seq'::regclass)"))
    file_path = Column(String(300), nullable=False, server_default=text_("''::character varying"))
    error_file_path = Column(String(300), nullable=False, server_default=text_("''::character varying"))
    status = Column(ChoiceType(STATUS), nullable=False, server_default=text_("2"))
    user_id = Column(Integer, nullable=False, server_default=text_("0"))
    obj = Column(String(80))
    log_type = Column(ChoiceType(LOG_TYPE), nullable=False, server_default=text_("0"))
    time = Column(DateTime(True))
    finished_time = Column(DateTime(True))
    duplicate_type = Column(String(16))
    ext_attributes = Column(Text)
    server_id = Column(Integer, nullable=False, server_default=text_("0"))
    upload_table = Column(String(100))
    upload_type = Column(ChoiceType(UPLOAD_TYPE))
    db_error_file_path = Column(String(100))
    duplicate_numbers = Column(Integer, server_default=text_("0"))
    foreign_id = Column(Integer)
    success_numbers = Column(Integer, server_default=text_("0"))
    error_row = Column(Integer, server_default=text_("0"))
    error_rollback = Column(Boolean, server_default=text_("false"))
    php_process_number = Column(BigInteger, server_default=text_("0"))
    db_process_number = Column(BigInteger, server_default=text_("0"))
    foreign_name = Column(String(80))
    auto_enddate = Column(Integer, server_default=text_("0"))
    custom_end_date = Column(DateTime(True))
    pid = Column(Integer)
    myfile_filename = Column(String(100))
    is_public = Column(Boolean)
    task_uuid = Column(String(36))

    download_time = synonym('time')

    records_succ = synonym('success_numbers')
    records_fail = synonym('error_row')
    records_dup = synonym('duplicate_numbers')
    method = synonym('duplicate_type')
    upload_time = synonym('time')
    upload_file = synonym('myfile_filename')
    file_name = synonym('myfile_filename')

    user = relationship(User, primaryjoin=foreign(user_id) == User.user_id, uselist=False)

    @hybrid_property
    def user_name(self):
        try:
            return self.user.name
        except:
            return ''

    @hybrid_property
    def _query(self):
        try:
            return json.JSONDecoder().decode(self.ext_attributes)
        except:
            return []

    @_query.setter
    def _query(self, value):
        try:
            self.ext_attributes = json.JSONEncoder().encode(value)
        except:
            pass

    @property
    def file(self):
        return self.file_path + '/' + self.myfile_filename

    @property
    def code_deck_name(self):
        q = self._query
        if q:
            if 'replace_fields' in q and q['replace_fields'] == 'code_deck_id':
                if 'replace_values' in q:
                    try:
                        cd = CodeDeck.get(int(q['replace_values']))
                        if cd:
                            return cd.name
                        else:
                            return '<code_deck_id {} not found>'.format(q['replace_values'])
                    except:
                        pass
        return None

    @property
    def total(self):
        if self.status == 'success':
            return self.success_numbers + self.error_row + self.duplicate_numbers
        if self.file:
            import os
            filename = self.file
            i = -1
            if os.path.exists(filename):
                with open(filename) as f:
                    for i, l in enumerate(f):
                        pass
                return i-1
        return None


DnlCloudGcloudCfg.key_file_id = column_property(
    select([ImportExportLogs.id]).where(DnlCloudGcloudCfg.key_file_path == ImportExportLogs.file_path.op('||')(
        '/').op('||')(ImportExportLogs.file_name)))

DnlCloudFtpCfg.netrc_id = column_property(
    select([ImportExportLogs.id]).where(DnlCloudFtpCfg.netrc_path == ImportExportLogs.file_path.op('||')('/').op(
        '||')(ImportExportLogs.file_name)))

DnlCloudSftpCfg.pubkey_id = column_property(
    select([ImportExportLogs.id]).where(DnlCloudSftpCfg.pubkey_path == ImportExportLogs.file_path.op('||')('/').op(
        '||')(ImportExportLogs.file_name)))

DnlCloudSftpCfg.privkey_id = column_property(
    select([ImportExportLogs.id]).where(DnlCloudSftpCfg.privkey_path == ImportExportLogs.file_path.op('||')('/').op(
        '||')(ImportExportLogs.file_name)))


class DidExportAsyncTask(DnlApiBaseModel):
    __tablename__ = 'did_export_async_task'
    STATUS = {0: 'initial', 1: 'finished', 2: 'running', 3: 'success', 4: 'fail'}
    ACTION = {0: 'download', 1: 'create file'}
    id = Column(Integer, primary_key=True)
    request_client_id = Column(ForeignKey('client.client_id', ondelete='CASCADE'))
    user_id = Column(ForeignKey('users.user_id', ondelete='CASCADE'))

    did_client_id = Column(Integer)
    did_vendor_id = Column(Integer)
    did_prefix = Column(String(30))
    did_is_active = Column(Boolean)
    did_is_assigned = Column(Boolean)
    did_end_dated = Column(Boolean)

    job_start_time = Column(Integer)
    job_end_time = Column(Integer)
    msg = Column(String(512))
    progress = Column(String(4))
    count = Column(Integer)
    size = Column(Integer)

    status = Column(ChoiceType(STATUS))
    action = Column(ChoiceType(ACTION))

    orig_file = Column(String(512))
    client = relationship(Client, uselist=False,  # back_populates='low_balance_config',
                          enable_typechecks=False)
    user = relationship(User, uselist=False)
    client_id = synonym('request_client_id')

    def get_did_query(self):
        cls = DidRepository
        q = get_db().tr_session.query(cls)
        if self.did_prefix:
            pat = self.did_prefix.replace('*', '%').replace('x', '_') + '%'
            pat1 = '_' + pat
            filt = or_(cls.did.like(pat), cls.did.like(pat1))
            q = q.filter(filt)
        if self.did_client_id:
            q = q.filter(cls.client_id == self.did_client_id)
        if self.did_vendor_id:
            q = q.filter(cls.vendor_id == self.did_vendor_id)
        # q = q.filter(cls.is_available == True)
        # if not self.did_is_active is None:
        #     q = q.filter(cls.is_effective == self.did_is_active)
        # if not self.did_is_assigned is None:
        #     q = q.filter(cls.is_assigned == self.did_is_assigned)
        # if not self.did_end_dated is None:
        #     q = q.filter(cls.end_date.isnot(None) == self.did_end_dated)

        return q

    @property
    def download_link(self):
        if self.orig_file:
            return '{}{}/files/did_export/{}'.format(settings.API_SCHEME, settings.API_HOST,
                                                     self.orig_file)
        return None

    @property
    def file_name(self):
        import os
        if self.orig_file:
            dir = '{}/did_export'.format(settings.FILES['upload_to'])
            if not os.path.exists(dir):
                os.mkdir(dir)
            return '{}/did_export/{}'.format(settings.FILES['upload_to'], self.orig_file)
        return None

    @property
    def file_data(self):
        if self.orig_file:
            return open(self.file_name, 'rb').read()
        return None


class DidNumberUploadTask(DnlApiBaseModel):
    __tablename__ = 'did_number_upload_task'
    STATUS = {0: 'Initial', 1: 'In Process', 2: 'Copy To DB', 3: 'Finished', 4: 'Error'}
    OP_METHOD = {0: 'Upload', 1: 'Release', 2: 'Delete', 3: 'Assign'}
    REPEATED_ACTION = {0: 'Ignore', 1: 'Overwrite'}
    ENABLE_FOR_CLIENTS = {0: 'No', 1: 'Yes'}
    id = Column(Integer, primary_key=True)
    operator_user = Column(String(40))
    upload_file_path = Column(String(256))
    upload_orig_file = Column(String(100))
    upload_format_file = Column(String(100))
    result_file_path = Column(String(256))
    repeated_action = Column(ChoiceType(REPEATED_ACTION), server_default='0')
    status = Column(ChoiceType(STATUS), nullable=True, server_default='0')
    progress = Column(String(200))
    create_time = Column(DateTime(True), server_default=func.now())
    start_time = Column(DateTime(True))
    end_time = Column(DateTime(True))
    import_export_logs_id = Column(Integer)
    op_method = Column(ChoiceType(OP_METHOD), server_default='0')
    did_vendor_name = Column(String(100))
    vendor_billing_rule_name = Column(String(100))
    enable_for_clients = Column(ChoiceType(ENABLE_FOR_CLIENTS), server_default='0')
    did_client_name = Column(String(100))
    client_billing_rule_name = Column(String(100))

    error_cause = column_property(case([(status=='Error',progress)],else_=''))

    @property
    def error_file_url(self):
        if self.result_file_path:
            logfile=self.result_file_path+'/did_upload.log'
            if os.path.exists(logfile):
                if os.stat(logfile).st_size>0:
                    return '{}/tool/did_number_upload_task/{}/log'.format(settings.API_URL,self.id)
        return None


    @property
    def total(self):
        if self.progress and 'Total: ' in self.progress:
            return self.progress.split('Total: ')[-1].split(',')[0]
        if self.upload_orig_file:
            import os
            filename = self.upload_file_path + '/' + self.upload_orig_file
            i = -1
            try:
                if os.path.exists(filename):
                    with open(filename) as f:
                        for i, l in enumerate(f):
                            pass
                    return i
            except Exception as e:
                return 0
        if not self.progress or not self.result_file_path:
            return 0
        return 0

    @property
    def success(self):
        if not self.progress:
            return 0
        new = 0
        update = 0
        release = 0
        delete = 0
        assign = 0
        if 'New: ' in self.progress:
            new = self.progress.split('New: ')[-1].split(',')[0]
        if 'Update: ' in self.progress:
            update = self.progress.split('Update: ')[-1].split(',')[0]
        if 'Release: ' in self.progress:
            release = self.progress.split('Release: ')[-1].split(',')[0]
        if 'Delete: ' in self.progress:
            delete = self.progress.split('Delete: ')[-1].split(',')[0]
        if 'Assign: ' in self.progress:
            assign = self.progress.split('Assign: ')[-1].split(',')[0]
        try:
            success = int(new) + int(update) + int(release) + int(delete) + int(assign)
            success = min(success, int(self.total))
        except:
            success = 0

        return success

    @property
    def fail(self):
        if not self.progress or not self.result_file_path:
            return 0
        if self.total:
            return int(self.total) - self.success
        elif 'Error: ' in self.progress:
            return self.progress.split('Error: ')[-1].split(',')[0]
        return 0


class DidNumberDeleteTask(DnlApiBaseModel):
    __tablename__ = 'did_number_delete_task'
    STATUS = {0: 'Initial', 1: 'In Process', 2: 'Copy To DB', 3: 'Finished', 4: 'Error'}
    OP_METHOD = {1: 'Remove DIDs from a specific client', 2: 'Remove DIDs from any client',
                 3: 'Remove DIDs from a specific vendor', 4: 'Remove DIDs from any vendor'}

    REPEATED_ACTION = {0: 'End date', 1: 'Delete'}
    id = Column(Integer, primary_key=True)
    operator_user = Column(String(40))
    upload_file_path = Column(String(256))
    upload_orig_file = Column(String(100))
    upload_format_file = Column(String(100))
    orig_name = Column(String(256))
    repeated_action = Column(ChoiceType(REPEATED_ACTION), server_default='0')
    status = Column(ChoiceType(STATUS), nullable=False, server_default='0')
    progress = Column(Text)
    create_time = Column(DateTime(True), server_default=func.now())
    start_time = Column(DateTime(True))
    end_time = Column(DateTime(True))
    import_export_logs_id = Column(Integer)
    op_method = Column(ChoiceType(OP_METHOD), server_default='0')
    client_id = Column(Integer)
    vendor_id = Column(Integer)

    error_cause = column_property(case([(status == 'Error', progress)], else_=''))

    @property
    def error_file_url(self):
        if self.result_file_path:
            logfile = self.result_file_path + '/did_upload.log'
            if os.path.exists(logfile):
                if os.stat(logfile).st_size > 0:
                    return '{}/tool/did_number_delete_task/{}/log'.format(settings.API_URL, self.id)
        return None

    @property
    def url(self):
        return '{}//home/client/did/upload/{}.{}'.format(settings.API_URL, self.uuid, self.ext)

    @property
    def file_name(self):
        if self.upload_format_file:
            return '{}/{}'.format(settings.FILES['upload_to'], self.upload_format_file)
        return None

    @property
    def ext(self):
        if self.upload_format_file and len(self.upload_format_file.split('.')) > 1:
            return self.upload_format_file.split('.')[-1]
        return ''

    @property
    def data(self):
        if self.file:
            return open(self.file_name, 'wb').read()

    @property
    def total(self):
        if self.upload_orig_file:
            import os
            filename = self.upload_file_path + '/' + self.upload_orig_file
            i = 0
            if os.path.exists(filename):
                with open(filename) as f:
                    for i, l in enumerate(f):
                        pass
                return i
        try:
            if 'numbers deleted' in self.progress and ' of ' in self.progress:
                return self.progress.split('numbers deleted')[1].split(' ')[3][:-1]
            return self.success
        except:
            return 0

    @property
    def success(self):
        try:
            if 'numbers deleted' in self.progress:
                return self.progress.split('numbers deleted')[1].split(' ')[1]
            return 0
        except:
            return 0

    @property
    def fail(self):
        try:
            if 'numbers skiped' in self.progress:
                return self.progress.split('numbers skiped')[1].split('\n')[0].split(',')[0][1:]
            return 0
        except:
            return 0


class DidNumberAssignTask(DnlApiBaseModel):
    __tablename__ = 'did_number_assign_task'
    STATUS = {0: 'Initial', 1: 'In Process', 2: 'Copy To DB', 3: 'Finished', 4: 'Error'}
    OP_METHOD = {1: 'Assign existing only'}

    REPEATED_ACTION = {0: 'End date', 1: 'Replace'}
    id = Column(Integer, primary_key=True)
    operator_user = Column(String(40))
    upload_file_path = Column(String(256))
    upload_orig_file = Column(String(100))
    upload_format_file = Column(String(100))
    orig_name = Column(String(256))
    repeated_action = Column(ChoiceType(REPEATED_ACTION), server_default='0')
    status = Column(ChoiceType(STATUS), nullable=False, server_default='0')
    progress = Column(Text)
    create_time = Column(DateTime(True), server_default=func.now())
    start_time = Column(DateTime(True))
    end_time = Column(DateTime(True))
    import_export_logs_id = Column(Integer)
    op_method = Column(ChoiceType(OP_METHOD), server_default='0')
    client_id = Column(Integer)
    client_billing_rule_id = Column(Integer)

    @property
    def url(self):
        return '{}//home/client/did/assign/{}.{}'.format(settings.API_URL, self.uuid, self.ext)

    @property
    def file_name(self):
        if self.upload_format_file:
            return '{}/{}'.format(settings.FILES['upload_to'], self.upload_format_file)
        return None

    @property
    def ext(self):
        if self.upload_format_file and len(self.upload_format_file.split('.')) > 1:
            return self.upload_format_file.split('.')[-1]
        return ''

    @property
    def data(self):
        if self.file:
            return open(self.file_name, 'wb').read()


class DidNumberUpload(DnlApiBaseModel):
    __tablename__ = 'did_number_upload'
    STATUS = {-1: 'fail', 0: 'initial', 1: 'success', 2: 'busy'}
    DUPLICATES_ACTION = {0: 'skip', 1: 'override'}
    uuid = Column(String(36), primary_key=True, default=generate_uuid_str())
    file = Column(String(512))
    created_on = Column(DateTime(True), server_default=func.now())
    created_by = Column(String(255))
    finished_on = Column(DateTime(True))
    status = Column(ChoiceType(STATUS), nullable=False, server_default='0')
    result = Column(Text)
    num_records = Column(Integer)
    num_imported = Column(Integer)
    num_skiped = Column(Integer)
    duplicates_action = Column(ChoiceType(DUPLICATES_ACTION), nullable=False, server_default="0")
    client_id = Column(Integer)
    vendor_id = Column(Integer)
    client_billing_rule_id = Column(Integer)
    vendor_billing_rule_id = Column(Integer)

    @property
    def url(self):
        return '{}//home/client/did/upload/{}.{}'.format(settings.API_URL, self.uuid, self.ext)

    @property
    def file_name(self):
        if self.file:
            return '{}/{}.{}'.format(settings.FILES['upload_to'], self.uuid, self.ext)
        return None

    @property
    def ext(self):
        if self.file and len(self.file.split('.')) > 1:
            return self.file.split('.')[-1]
        return ''

    @property
    def data(self):
        if self.file:
            return open(self.file_name, 'wb').read()


# region +++RateImportTask

class RateImportTask(DnlApiBaseModel):
    __tablename__ = 'rate_import_task'

    CODE_NAME_PROVIDER = {0: 'get code name from import rate file', 1: 'get code name from code deck table'}
    IMPORT_RATE_DATE_FORMAT = {0: 'MM/DD/YYYY', 1: 'YYYY-MM-DD', 2: 'DD-MM-YYYY', 3: 'MM-DD-YYYY', 4: 'DD/MM/YYYY',
                               5: 'YYYY/MM/DD'}
    REDUP_IN_RATE_TABLE_ACTION = {0: 'reject entire import', 1: 'skip the record', 2: 'overwrite existing record'}
    REDUP_IN_FILE_ACTION = {0: 'reject entire import', 1: 'skip the record', 2: 'overwrite previous record'}
    STATUS = {0: 'Initial', 1: 'Parse and check the rate file', 2: 'Load rate table data',
              3: 'Compare rate with rate table',
              4: 'Insert new rate', 5: 'end_date_all_records', 6: 'Update same code rate', 7: 'Upload successful',
              8: 'Upload failed', 9: 'API Processing', 10: 'API Failed'}

    id = Column(Integer, primary_key=True, server_default=text_("nextval('rate_import_task_id_seq'::regclass)"))
    operator_user = Column(String(40))
    import_file_path = Column(String(256))
    orig_rate_file = Column(String(256), doc='Original import rate file name')
    format_rate_file = Column(String(256), doc='CSV format file after WEB check')
    import_log_path = Column(String(256), doc='Import log file path')
    rate_table_id = Column(Integer, nullable=False)
    code_deck_id = Column(Integer, doc='code dec table ID, used to get code name')
    code_name_provider = Column(ChoiceType(CODE_NAME_PROVIDER), nullable=False, server_default='0')
    import_rate_date_format = Column(String(40))
    end_date_all_records_time = Column(DateTime(True))
    end_date_other_code_time = Column(DateTime(True))
    redup_in_rate_table_action = Column(ChoiceType(REDUP_IN_RATE_TABLE_ACTION), nullable=False, server_default='0')
    # redup_in_rate_rable_action = Column(ChoiceType(REDUP_IN_RATE_TABLE_ACTION), nullable=False, server_default='0')
    # redup_in_rate_table_action = synonym('redup_in_rate_rable_action')

    redup_in_file_action = Column(ChoiceType(REDUP_IN_FILE_ACTION), nullable=False, server_default='0')
    status = Column(ChoiceType(STATUS), nullable=True, server_default=text_('0'))
    progress = Column(String(1024))
    expense_detail = Column(String(100))
    start_time = Column(DateTime(True))
    end_time = Column(DateTime(True))
    create_time = Column(DateTime(True))

    rate_table_name = column_property(
        select([RateTable.name]).where(RateTable.rate_table_id == rate_table_id).correlate_except(RateTable))

    @property
    def file_name(self):
        if self.format_rate_file:
            return self.import_file_path + '/' + self.format_rate_file
        return None

    @property
    def file_data(self):
        try:
            f = open(self.file_name, 'rb')
            return f.read()
        except:
            return None

    @property
    def download_link(self):
        return '{}/tool/rate/import/{}/orig/rate.csv'.format(settings.API_URL, self.id)

    @property
    def fmt_download_link(self):
        return '{}/files/{}'.format(settings.FILE_URL, self.format_rate_file)

    @property
    def orig_download_link(self):
        return '{}/files/{}'.format(settings.FILE_URL, self.orig_rate_file)

    @property
    def error_file_name(self):
        if self.import_log_path:
            return self.import_log_path + '/rate_import.log'
        return None

    @property
    def error_file_data(self):
        try:
            f = open(self.error_file_name, 'rb')
            return f.read()
        except:
            return None

    @property
    def error_download_link(self):
        return '{}/tool/rate/import/{}/error/rate_import.log'.format(settings.API_URL, self.id)

    @property
    def progress_parsed(self):
        m = None
        if type(self.progress) == type(''):
            m = re.match(
                r'Insert: (?P<insert>[0-9]*), Delete: (?P<delete>[0-9]*), Update: (?P<update>[0-9]*), Error: (?P<error>[0-9]*)',
                self.progress)
        if m:
            return {'insert': m.group('insert'),
                    'delete': m.group('delete'),
                    'update': m.group('update'),
                    'error': m.group('error')
                    }
        else:
            return {'error': self.progress}

    def before_save(self):
        if not self.create_time:
            self.create_time = datetime.now(UTC)


# endregion ---RateImportTask

# region ---RateAnalysisTask
class RateAnalysisTask(DnlApiBaseModel):
    __tablename__ = 'rate_analysis_task'
    STATUS = {
        0: 'Initial',
        1: 'Processing', 
        2: 'Completed',
        3: 'Failed'
    }

    task_id = Column(Integer, primary_key=True)
    status = Column(ChoiceType(STATUS), nullable=False, default='Initial', server_default=text_("0"))
    progress = Column(Integer)
    operator_user = Column(String(40))
    start_time = Column(DateTime(True))
    end_time = Column(DateTime(True))
    create_time = Column(DateTime(True))
    expense_detail = Column(String(100))
    source_rate_table_id = Column(Integer)
    source_effective_date = Column(DateTime(True))
    target_rate_table_id = Column(Integer)
    target_effective_date = Column(DateTime(True))
    target_file_name = Column(String(512))
    rate_generation_history_id = Column(Integer)
    created_on = Column(DateTime(True), nullable=False, server_default=func.now())
    started_on = Column(DateTime(True))
    finished_on = Column(DateTime(True))
    result_json = Column(String(512))

    def before_save(self):
        if not self.created_on:
            self.created_on = datetime.now(UTC)

    @property
    def increases(self):
        """Number of rates that increased"""
        if not self.result_json:
            return 0
        try:
            result = eval(self.result_json)
            return result.get('increase', 0)
        except:
            return 0

    @property 
    def decreases(self):
        """Number of rates that decreased"""
        if not self.result_json:
            return 0
        try:
            result = eval(self.result_json)
            return result.get('decrease', 0)
        except:
            return 0

    @property
    def no_changes(self):
        """Number of rates that stayed the same"""
        if not self.result_json:
            return 0
        try:
            result = eval(self.result_json)
            return result.get('no_change', 0)
        except:
            return 0

    @property
    def new_rates(self):
        """Number of new rates added"""
        if not self.result_json:
            return 0
        try:
            result = eval(self.result_json)
            return result.get('new', 0)
        except:
            return 0

    @property
    def deleted_rates(self):
        """Number of rates deleted"""
        if not self.result_json:
            return 0
        try:
            result = eval(self.result_json)
            return result.get('delete', 0)
        except:
            return 0


# endregion ---RateAnalysisTask

# region ---LcrTask

class LcrTask(DnlApiBaseModel):
    __tablename__ = 'lcr_task'

    task_id = Column(Integer, primary_key=True)
    target_rate_table_id = Column(ForeignKey('rate_table.rate_table_id'), nullable=False)
    egress_id = Column(ARRAY(Integer))
    lcr_rate_table_id = Column(ARRAY(Integer), nullable=False)
    created_on = Column(DateTime(True), default=datetime.now(UTC))
    started_on = Column(DateTime(True))
    finished_on = Column(DateTime(True))
    result_path = Column(String(256))
    effective_date = Column(String(40), default=datetime.now(UTC))

    target_rate_table = relationship('RateTable', uselist=False)
    target_rate_table_name = column_property(
        select([RateTable.name]).where(RateTable.rate_table_id == target_rate_table_id).correlate_except(RateTable))

# endregion ---LcrTask


# +++RateUploadTask

class RateUploadTask(DnlApiBaseModel):
    __tablename__ = 'rate_upload_task'

    STATUS = {0: 'initial', 1: 'download rate', 2: 'process rate', 3: 'commit rate to db', 4: 'finished', 5: 'error'}
    PRE_STATUS = {5: 'error', 6: 'uploaded', 7: 'preprocess', 8: 'busy', 9: 'converted'}
    REDUPLICATE_RATE_ACTION = {1: 'delete', 2: 'update', 0: 'ignore'}
    CODE_DECK_FLAG = {0: 'use rate code', 1: 'use system global code deck'}
    USE_OCN_LATA_CODE = {1: 'Yes', 0: 'No'}
    DATE_FORMAT = {0: 'mm/dd/yyyy', 1: 'yyyy-mm-dd', 2: 'dd-mm-yyyy', 3: 'dd/mm/yyyy', 4: 'yyyy/mm/dd'}
    TYPE = {0: 'manual', 1: 'auto'}
    id = Column(Integer, primary_key=True, server_default=text_("nextval('rate_upload_task_id_seq'::regclass)"))
    operator_user = Column(String(40))

    upload_file_path = Column(String(256))
    upload_orig_file = Column(String(256))
    upload_format_file = Column(String(256))
    result_file_path = Column(String(256))
    rate_table_id = Column(Integer, nullable=False, default=0)
    rate_table_code_deck_id = Column(Integer)
    rate_date_format = Column(String(40))
    rate_end_date = Column(String(40))
    reduplicate_rate_action = Column(ChoiceType(REDUPLICATE_RATE_ACTION), nullable=False, server_default=text_("0"))
    code_deck_flag = Column(ChoiceType(CODE_DECK_FLAG), nullable=False, server_default=text_(
        "0"))  # Use system code deck, 0 for use rate code, 1 for use system global code deck
    use_ocn_lata_code = Column(ChoiceType(USE_OCN_LATA_CODE), nullable=False, server_default=text_("0"))
    status = Column(ChoiceType(STATUS), nullable=False, default='initial', server_default=text_("0"))
    progress = Column(String(200))
    expense_detail = Column(String(100))
    create_time = Column(BigInteger)
    start_time = Column(BigInteger)
    end_time = Column(BigInteger)
    default_info = Column(String(200))
    all_rate_end_date = Column(DateTime(True))
    code_name_match = Column(Boolean)
    start_from = Column(Integer)
    pre_info = Column(Text)
    pre_status = Column(ChoiceType(PRE_STATUS), nullable=False, default='preprocess')
    task_id = Column(String(36))
    rule_log_id = Column(ForeignKey('rate_auto_import_rule_log.id', ondelete='SET NULL'), index=True)
    import_task_id = Column(ForeignKey('rate_import_task.id'))

    method = synonym('reduplicate_rate_action')
    template_name = synonym('default_info')

    rate_table = relationship(RateTable, primaryjoin=foreign(rate_table_id) == RateTable.rate_table_id, uselist=False)
    rate_table_name = column_property(
        select([RateTable.name]).where(RateTable.rate_table_id == rate_table_id).correlate_except(RateTable))

    # create_date_time = column_property(func.to_timestamp(create_time))
    upload_date_time = column_property(func.to_timestamp(create_time))
    start_date_time = column_property(func.to_timestamp(start_time))
    end_date_time = column_property(func.to_timestamp(end_time))
    type = column_property(case([(rule_log_id.is_(None), literal_column("'manual'"))], else_=literal_column("'auto'")))

    values = relationship('RateUploadValues', uselist=True, back_populates='task', lazy='dynamic')
    rule_log = relationship('RateAutoImportRuleLog', uselist=False, back_populates='task')

    @property
    def orig_file(self):
        return '{}/{}'.format(self.upload_file_path, self.upload_orig_file)

    @hybrid_property
    def info(self):
        try:
            return json.loads(self.pre_info)
        except:
            return {}

    @info.setter
    def info(self, value):
        self.pre_info = json.dumps(value)

    @property
    def progress_parsed(self):
        m = re.match(
            r'Insert: (?P<insert>[0-9]*), Delete: (?P<delete>[0-9]*), Update: (?P<update>[0-9]*), Error: (?P<error>[0-9]*)',
            self.progress)
        if m:
            return {'insert': m.group('insert'),
                    'delete': m.group('delete'),
                    'update': m.group('update'),
                    'error': m.group('error')
                    }
        else:
            return {'error': self.progress}

    def create_import_task(self):
        obj = RateImportTask(
            operator_user=self.operator_user,
            import_file_path=self.upload_file_path,
            orig_rate_file=self.upload_orig_file,
            format_rate_file=self.upload_format_file,
            rate_table_id=self.rate_table_id,
            code_deck_id=self.rate_table_code_deck_id,
            code_name_provider=0,
            import_rate_date_format=self.rate_date_format,
            end_date_all_records_time=self.all_rate_end_date,
            end_date_other_code_time=self.end_date_time,
            redup_in_rate_table_action='overwrite existing record',
            redup_in_file_action='overwrite previous record',
            create_time = datetime.now(UTC)
        )
        return obj


class RateAutoImportRule(DnlApiBaseModel):
    __tablename__ = 'rate_auto_import_rule'
    READ_EFFECTIVE_DATE_FROM = {0: 'file', 1: 'subject', 2: 'content', 3: 'today'}
    id = Column(Integer, primary_key=True)
    name = Column(String(100), unique=True)
    active = Column(Boolean, default=True)
    create_by = Column(String(100))
    create_on = Column(DateTime(True), server_default=text_("('now'::text)::date"))
    update_by = Column(String(100))
    update_on = Column(DateTime(True), server_default=text_("('now'::text)::date"))
    egress_trunk_id = Column(ForeignKey('resource.resource_id', ondelete='CASCADE'), index=True)
    from_email = Column(String(100))
    subject_keyword = Column(String(100))
    pre_info = Column(Text)
    rate_date_format = Column(String(40))
    rate_table_code_deck_id = Column(Integer)
    reduplicate_rate_action = Column(ChoiceType(RateUploadTask.REDUPLICATE_RATE_ACTION), nullable=False,
                                     server_default=text_("0"))
    code_deck_flag = Column(ChoiceType(RateUploadTask.CODE_DECK_FLAG), nullable=False, server_default=text_(
        "0"))  # Use system code deck, 0 for use rate code, 1 for use system global code deck
    use_ocn_lata_code = Column(ChoiceType(RateUploadTask.USE_OCN_LATA_CODE), nullable=False, server_default=text_("0"))
    is_link = Column(Boolean, default=False)
    read_effective_date_from = Column(ChoiceType(READ_EFFECTIVE_DATE_FROM))
    mail_server_ip = Column(String(100))

    rule_name = synonym('name')
    date_pattern = synonym('rate_date_format')
    log = relationship('RateAutoImportRuleLog', uselist=True, back_populates='rule')
    egress_trunk = relationship('Resource', uselist=False)

    egress_trunk_name = column_property(
        select([Resource.alias]).where(Resource.resource_id == egress_trunk_id).correlate_except(Resource))

    rate_table_name = column_property(
        select([RateTable.name]).where(and_(Resource.rate_table_id == RateTable.rate_table_id,
                                            Resource.resource_id == egress_trunk_id)).correlate_except(Resource,
                                                                                                       RateTable))

    @hybrid_property
    def info(self):
        try:
            return json.loads(self.pre_info)
        except:
            return {}

    @info.setter
    def info(self, value):
        self.pre_info = json.dumps(value)


class RateAutoImportMailboxLog(DnlApiBaseModel):
    __tablename__ = 'rate_auto_import_mailbox_log'
    id = Column(Integer, primary_key=True)
    start = Column(DateTime(True), server_default=func.now())
    finish = Column(DateTime(True))
    received_letters = Integer()
    matched_to_rules = Integer()
    error = Column(Text)


class RateAutoImportRuleLog(DnlApiBaseModel):
    __tablename__ = 'rate_auto_import_rule_log'
    STATUS = {0: 'email upload', 1: 'email error', 2: 'task waiting', 3: 'upload success', 4: 'upload error'}
    id = Column(Integer, primary_key=True)
    rule_id = Column(ForeignKey('rate_auto_import_rule.id', ondelete='CASCADE'), index=True)
    status = Column(ChoiceType(STATUS), nullable=False, default='preprocess')
    start = Column(DateTime(True), server_default=func.now())
    finish = Column(DateTime(True))
    received_from_ip = Column(String(16))
    error = Column(Text)
    task = relationship('RateUploadTask', uselist=False, back_populates='rule_log')
    rule = relationship('RateAutoImportRule', uselist=False, back_populates='log')

    rule_name = column_property(
        select([RateAutoImportRule.name]).where(RateAutoImportRule.id == rule_id).correlate_except(RateAutoImportRule))
    egress_trunk_id = column_property(
        select([RateAutoImportRule.egress_trunk_id]).where(RateAutoImportRule.id == rule_id).correlate_except(
            RateAutoImportRule))
    from_email = column_property(
        select([RateAutoImportRule.from_email]).where(RateAutoImportRule.id == rule_id).correlate_except(
            RateAutoImportRule))
    subject_keyword = column_property(
        select([RateAutoImportRule.subject_keyword]).where(RateAutoImportRule.id == rule_id).correlate_except(
            RateAutoImportRule))
    task_id = column_property(
        select([RateUploadTask.id]).where(RateUploadTask.rule_log_id == id).correlate_except(
            RateUploadTask))

    rate_table_id = column_property(
        select([RateUploadTask.rate_table_id]).where(RateUploadTask.rule_log_id == id).correlate_except(
            RateUploadTask))


class RateUploadValues(DnlApiBaseModel):
    __tablename__ = 'rate_upload_values'
    __table_args__ = (UniqueConstraint('task_id', 'Code', 'Effective_date'),)
    id = Column(Integer, primary_key=True)
    task_id = Column(ForeignKey('rate_upload_task.id', ondelete='CASCADE'), index=True)
    Code = Column(PrefixRange, nullable=True, server_default=text_("''::prefix_range"))
    Rate = Column(Numeric(30, 10))
    Inter_rate = Column(Numeric(30, 10))
    Intra_rate = Column(Numeric(30, 10))
    Local_rate = Column(Numeric(30, 10))
    Effective_date = Column(String(10))  # Column(DateTime(True))
    Min_time = Column(Integer, nullable=False, server_default=text_("1"))
    Interval = Column(Integer, nullable=False, server_default=text_("0"))
    Code_name = Column(String(100))
    Country = Column(String(1000))
    task = relationship(RateUploadTask, uselist=False, back_populates='values')


# ---RateUploadTask

# +++RateUploadTemplate
class RateUploadTemplate(DnlApiBaseModel):
    __tablename__ = 'rate_upload_template'
    AUTO_IJ_RATE = {0: 'None', 1: 'Inter Rate', 2: 'Intra Rate', 3: 'Max (Inter, Intra)'}
    CODE_NAME_PROVIDER = {0: 'get code name from import rate file', 1: 'get code name from code deck table'}
    IMPORT_RATE_DATE_FORMAT = {0: 'MM/DD/YYYY', 1: 'YYYY-MM-DD', 2: 'DD-MM-YYYY', 3: 'DD/MM/YYYY', 4: 'YYYY/MM/DD'}
    REDUP_IN_RATE_TABLE_ACTION = {0: 'reject entire import', 1: 'skip the record', 2: 'overwrite existing record'}
    REDUP_IN_FILE_ACTION = {0: 'reject entire import', 1: 'skip the record', 2: 'overwrite previous record'}

    id = Column(Integer, primary_key=True)
    name = Column(String(100), unique=True)
    auto_ij_rate = Column(ChoiceType(AUTO_IJ_RATE))
    rate_table_id = Column(Integer)
    code_deck_id = Column(Integer, doc='code dec table ID, used to get code name')
    code_name_provider = Column(ChoiceType(CODE_NAME_PROVIDER), nullable=False, server_default='0')
    import_rate_date_format = Column(String(40))
    effective_date = Column(DateTime(True))
    default_interval = Column(Integer, default=None, server_default=text_("null"))
    default_min_time = Column(Integer, default=None, server_default=text_("null"))
    end_date_all_records_time = Column(DateTime(True))
    end_date_other_code_time = Column(DateTime(True))
    redup_in_rate_table_action = Column(ChoiceType(REDUP_IN_RATE_TABLE_ACTION), nullable=False, server_default='0')
    redup_in_file_action = Column(ChoiceType(REDUP_IN_FILE_ACTION), nullable=False, server_default='0')

    create_on = Column(DateTime(True))
    create_by = Column(String(40))
    update_on = Column(DateTime(True), server_default=text_("('now'::text)::timestamp(0) with time zone"))
    update_by = Column(String(100))
    replace_fields = Column(Text)
    replace_values = Column(Text)
    ignore_fields = Column(Text)
    map_header = Column(Text)


# ---RateUploadTemplate
# +++SipRegistration
class SipRegistration(DnlApiBaseModel):
    __tablename__ = 'sip_registrations'
    # STATUS = {0: 'un-register', 1: 'registered', 2: 'register failed'}
    STATUS = {0: 'un-register', 1: 'success', 2: 'error'}
    id = Column(Integer, primary_key=True, server_default=text_("nextval('sip_registrations_id_seq'::regclass)"))
    username = Column(String(40), nullable=False, index=True)
    network_ip = Column(Ip4)
    network_port = Column(Integer)
    status = Column(ChoiceType(STATUS), nullable=False, server_default=text_("0"))
    expires = Column(Integer)
    contact = Column(String(256))
    uptime = Column(BigInteger)

    ip = relationship(ResourceIp, primaryjoin=foreign(ResourceIp.username) == username, uselist=False)
    time = column_property(func.to_timestamp(uptime))

    trunk_name = column_property(select([Resource.alias]). \
                                 where(and_(Resource.resource_id == ResourceIp.resource_id,
                                            ResourceIp.username == username)).limit(1).correlate_except(Resource,
                                                                                                        ResourceIp))

    carrier_name = column_property(select([Client.name]). \
                                   where(and_(Client.client_id == Resource.client_id,
                                              Resource.resource_id == ResourceIp.resource_id,
                                              ResourceIp.username == username)).limit(1).correlate_except(Resource,
                                                                                                          ResourceIp))
    is_spam = column_property(
        select([func.count(ResourceIp.username) == 0]).where(ResourceIp.username == username).correlate_except(
            ResourceIp))

    @hybrid_property
    def trunk_name_(self):
        if self.ip and self.ip.resource:
            return self.ip.resource.alias
        else:
            return None

    @hybrid_property
    def carrier_name_(self):
        if self.ip and self.ip.resource:
            return self.ip.resource.client_name
        else:
            return None

    @hybrid_property
    def _time(self):
        try:
            return datetime.fromtimestamp(self.uptime)
        except:
            return None


# ---SipRegistration

class SpamTrafficIp(DnlApiBaseModel):
    __tablename__ = 'spam_traffic_ip'
    ip = Column(String(30), nullable=False, index=True, primary_key=True)
    brief = Column(String(100))
    create_time = Column(DateTime(True), server_default=func.now())
    netmask = Column(Integer)
    created_by = Column(String(100))
    auto_block = Column(Boolean, server_default='true')


class InvoiceEmail(DnlApiBaseModel):
    __tablename__ = 'invoice_email'

    id = Column(Integer, primary_key=True, server_default=text_("nextval('invoice_email_id_seq'::regclass)"))
    invoice_no = Column(String(20))
    mail_content = Column(String(1000))
    send_time = Column(DateTime(True))
    pdf_file = Column(String(100))
    send_address = Column(String(100))
    mail_sub = Column(String(100))


# ---

# +++++CarrierTemplate

class CarrierTemplateLowBalConfig(DnlApiBaseModel):
    __tablename__ = 'carrier_template_low_bal_config'
    SEND_TIME_TYPE = {0: 'daily', 1: 'hourly'}
    VALUE_TYPE = {0: 'Actual Balance', 1: 'Percentage'}

    carrier_template_id = Column(ForeignKey('carrier_template.id', ondelete='CASCADE'), primary_key=True)
    is_notify = Column(Boolean)
    value_type = Column(ChoiceType(VALUE_TYPE))
    actual_notify_balance = Column(Numeric(30, 2))
    percentage_notify_balance = Column(Numeric(30, 2))
    send_time_type = Column(ChoiceType(SEND_TIME_TYPE))
    daily_send_time = Column(Integer)
    duplicate_days = Column(Integer)
    send_to = Column(Integer)
    duplicate_send_days = Column(Integer, nullable=False, server_default=text_("0"))
    last_alert_time = Column(DateTime(True))
    disable_trunks_days = Column(Integer, server_default=text_("5"))
    carrier = relationship('CarrierTemplate', uselist=False, back_populates='low_balance_config')

    def create_from_client(self, obj):
        self.is_notify = obj.is_notify
        self.value_type = obj.value_type
        self.actual_notify_balance = obj.actual_notify_balance
        self.percentage_notify_balance = obj.percentage_notify_balance
        self.send_time_type = obj.send_time_type
        self.daily_send_time = obj.daily_send_time
        self.duplicate_days = obj.duplicate_days
        self.send_to = obj.send_to
        self.duplicate_send_days = obj.duplicate_send_days
        self.last_alert_time = obj.last_alert_time
        self.disable_trunks_days = obj.disable_trunks_days


class CarrierTemplate(DnlApiBaseModel):
    __tablename__ = 'carrier_template'
    id = Column(Integer, primary_key=True, server_default=text_("nextval('carrier_template_id_seq'::regclass)"))
    allowed_credit = Column(Numeric, nullable=False, default=0.0, server_default=text_("0"))
    attach_cdrs_list = Column(Boolean)
    auto_daily_balance_recipient = Column(ChoiceType(Client.AUTO_DAILY_BALANCE_RECIPIENT), default=0)
    auto_invoice_type = Column(ChoiceType(Client.AUTO_INVOICE_TYPE))
    auto_invoicing = Column(Boolean)
    auto_send_zone = Column(String)
    auto_summary_group_by = Column(ChoiceType(Client.AUTO_SUMMARY_GROUP_BY))
    auto_summary_hour = Column(Integer)
    auto_summary_include_cdr = Column(Boolean)
    auto_summary_not_zero = Column(SmallInteger)
    auto_summary_period = Column(Integer)
    billing_mode = synonym('mode')
    breakdown_by_rate_table = Column(ChoiceType(Client.BREAKDOWN_BY_RATE_TABLE))
    call_limit = Column(Integer)
    cdr_format = synonym('cdr_list_format')
    cdr_list_format = Column(ChoiceType(Client.CDR_LIST_FORMAT))
    cps_limit = Column(Integer)
    create_by = Column(String(100))
    create_on = Column(DateTime(True))
    currency_id = Column(Integer, nullable=True, server_default=text_("1"))
    daily_balance_notification = Column(Integer)
    daily_balance_recipient = Column(Integer)
    daily_balance_send_time = Column(Time)
    daily_balance_send_time_zone = Column(String(20))
    daily_balance_summary = synonym('is_daily_balance_notification')
    daily_cdr_delivery = synonym('daily_cdr_generation')
    daily_cdr_generation = Column(Boolean)
    daily_cdr_generation_type = Column(SmallInteger)
    daily_cdr_generation_zone = Column(String)
    daily_limit = Column(Integer)
    daily_usage_group_by = synonym('auto_summary_group_by')
    daily_usage_summary = synonym('is_show_daily_usage')
    decimal_place = Column(Integer)
    email_invoice = Column(Boolean)
    format = synonym('invoice_format')
    hourly_limit = Column(Integer)
    include_available_credit = Column(SmallInteger)
    include_tax = Column(Boolean)
    invoice_format = Column(ChoiceType(Client.INVOICE_FORMAT))
    invoice_include_payment = Column(Boolean)
    invoice_jurisdictional_detail = Column(Boolean)
    invoice_show_details = Column(Boolean)
    invoice_start_from = Column(Date)
    invoice_use_balance_type = Column(ChoiceType(Client.INVOICE_USE_BALANCE_TYPE))
    invoice_zero = Column(Boolean)
    invoice_zone = Column(String(10))
    is_auto_balance = Column(Boolean)
    is_auto_summary = Column(Boolean)
    is_breakdown_by_rate_table = Column(Boolean)
    is_daily_balance_notification = Column(Boolean)
    is_email_invoice = Column(Boolean)
    is_invoice_account_summary = Column(Boolean)
    is_send_as_link = Column(Boolean)
    is_send_trunk_update = Column(Boolean)
    is_short_duration_call_surcharge_detail = Column(Boolean)
    is_show_by_date = Column(Boolean)
    is_show_code_100 = Column(Boolean)
    is_show_code_name = Column(Boolean)
    is_show_code_100 = Column(Boolean)
    is_show_country = Column(Boolean)
    is_show_daily_usage = Column(Boolean)
    is_show_detail_trunk = Column(Boolean)
    is_show_total_trunk = Column(Boolean)
    last_invoiced = Column(DateTime(True))
    mode = Column(ChoiceType(Client.BILLING_MODE), nullable=False, default=1, server_default=text_("1"))
    notify_client_balance = Column(Numeric)
    notify_client_balance_type = Column(Integer)
    numer_of_days_balance = Column(Integer)
    payment_received_notice = Column(Boolean)
    payment_term_id = Column(Integer)
    profit_margin = Column(Float(53), nullable=False, default=0.0, server_default=text_("0"))
    profit_type = Column(ChoiceType(Client.PROFIT_TYPE), nullable=False, default=1, server_default=text_("1"))
    rate_value = Column(ChoiceType(Client.RATE_VALUE))
    scc_bellow = Column(Integer)
    scc_below = synonym('scc_bellow')
    scc_charge = Column(Numeric)
    scc_percent = Column(Integer)
    scc_type = Column(ChoiceType(Client.SCC_TYPE))
    send_invoice_as_link = synonym('is_send_as_link')
    tax = Column(Float)
    template_name = Column(String(100), unique=True)
    test_credit = Column(Numeric(15, 6))
    unlimited_credit = Column(Integer)
    update_on = Column(DateTime(True))
    usage_detail_fields = Column(Text)

    # show_trunk_summary = synonym('is_show_total_trunk')

    currency = relationship(Currency, primaryjoin=foreign(currency_id) == Currency.currency_id)
    payment_term = relationship(PaymentTerm, primaryjoin=foreign(payment_term_id) == PaymentTerm.payment_term_id,
                                uselist=False)
    low_balance_config = relationship(CarrierTemplateLowBalConfig, uselist=False,
                                      single_parent=True,
                                      cascade="all, delete-orphan")

    @hybrid_property
    def currency_name(self):
        try:
            return self.currency.code
        except Exception as e:
            return ''

    @currency_name.setter
    def currency_name(self, value):
        try:
            self.currency_id = Currency.filter(Currency.code == value).one().currency_id
        except Exception as e:
            pass

    @hybrid_property
    def pt_name(self):
        try:
            return self.payment_term.name
        except Exception as e:
            return ''

    @pt_name.setter
    def pt_name(self, value):
        try:
            self.payment_term_id = PaymentTerm.filter(PaymentTerm.name == value).one().payment_term_id
        except:
            raise Exception('Bad payment term:' + str(value) + ' not found')

    # pay.save()

    @hybrid_property
    def credit_limit(self):
        if self.allowed_credit:
            return -self.allowed_credit
        else:
            return 0.0

    @credit_limit.setter
    def credit_limit(self, value):
        if self and value:
            self.allowed_credit = -Decimal(value)

    @hybrid_property
    def non_zero_invoice_only(self):
        return not self.invoice_zero

    @non_zero_invoice_only.setter
    def non_zero_invoice_only(self, value):
        self.invoice_zero = not value

    @hybrid_property
    def usage_fields(self):
        try:
            return self.usage_detail_fields.split(',') if self.usage_detail_fields else []
        except:
            return []

    @usage_fields.setter
    def usage_fields(self, value):
        try:
            self.usage_detail_fields = ','.join([str(x) for x in value]) if value else ''
        except:
            pass

    # def used_by(self):        return Client.filter(Client.carrier_template_id==self.id).count()
    used_by = column_property(select([func.count(Client.client_id)]).where(Client.carrier_template_id == id).
                              group_by(Client.carrier_template_id).correlate_except(Client))

    def template_from_client(self, obj):
        # self.include_short_call_charge=obj.include_short_call_charge
        self.allowed_credit = obj.allowed_credit
        self.attach_cdrs_list = obj.attach_cdrs_list
        self.auto_invoice_type = obj.auto_invoice_type
        self.auto_invoicing = obj.auto_invoicing
        self.auto_send_zone = obj.auto_send_zone
        self.auto_summary_group_by = obj.auto_summary_group_by
        self.auto_summary_hour = obj.auto_summary_hour
        self.auto_summary_include_cdr = obj.auto_summary_include_cdr
        self.auto_summary_not_zero = obj.auto_summary_not_zero
        self.auto_summary_period = obj.auto_summary_period
        self.breakdown_by_rate_table = obj.breakdown_by_rate_table
        self.call_limit = obj.call_limit
        self.cdr_format = obj.cdr_format
        self.cps_limit = obj.cps_limit
        # self.credit_limit=obj.credit_limit
        self.currency_id = obj.currency_id
        self.daily_balance_notification = obj.daily_balance_notification
        self.daily_balance_recipient = obj.daily_balance_recipient
        self.daily_balance_send_time = obj.daily_balance_send_time
        self.daily_balance_send_time_zone = obj.daily_balance_notification

        self.daily_cdr_generation = obj.daily_cdr_generation
        self.daily_cdr_generation_type = obj.daily_cdr_generation_type
        self.daily_cdr_generation_zone = obj.daily_cdr_generation_zone
        self.decimal_place = obj.decimal_place
        self.email_invoice = obj.email_invoice

        self.hourly_limit = obj.hourly_limit
        self.include_available_credit = obj.include_available_credit

        self.include_tax = obj.include_tax
        self.invoice_format = obj.invoice_format
        self.invoice_include_payment = obj.invoice_include_payment
        self.invoice_jurisdictional_detail = obj.invoice_jurisdictional_detail
        self.invoice_show_details = obj.invoice_show_details
        self.invoice_start_from = obj.invoice_start_from
        self.invoice_use_balance_type = obj.invoice_use_balance_type
        self.invoice_zero = obj.invoice_zero
        self.invoice_zone = obj.invoice_zone

        self.is_auto_summary = obj.is_auto_summary
        self.is_breakdown_by_rate_table = obj.is_breakdown_by_rate_table
        self.is_daily_balance_notification = obj.is_daily_balance_notification
        self.is_email_invoice = obj.email_invoice
        self.is_invoice_account_summary = obj.is_invoice_account_summary
        self.is_send_as_link = obj.is_send_as_link
        self.is_send_trunk_update = obj.is_send_trunk_update
        self.is_short_duration_call_surcharge_detail = obj.is_short_duration_call_surcharge_detail
        self.is_show_by_date = obj.is_show_by_date
        self.is_show_code_100 = obj.is_show_code_100
        self.is_show_code_name = obj.is_show_code_name
        self.is_show_code_100 = obj.is_show_code_100
        self.is_show_country = obj.is_show_country
        # self.is_show_daily_trunk = obj.is_show_daily_trunk
        self.is_show_daily_usage = obj.is_show_daily_usage
        self.is_show_detail_trunk = obj.is_show_detail_trunk
        self.is_show_total_trunk = obj.is_show_total_trunk
        self.is_unlimited = obj.is_unlimited
        self.mode = obj.mode
        self.notify_client_balance = obj.notify_client_balance
        self.notify_client_balance_type = obj.notify_client_balance_type
        self.payment_received_notice = obj.payment_received_notice
        self.payment_term = obj.payment_term
        self.profit_margin = obj.profit_margin
        self.profit_type = obj.profit_type
        self.rate_value = obj.rate_value
        self.scc_bellow = obj.scc_bellow
        self.scc_charge = obj.scc_charge
        self.scc_percent = obj.scc_percent
        self.scc_type = obj.scc_type
        self.tax = obj.tax
        self.test_credit = obj.test_credit
        self.unlimited_credit = obj.unlimited_credit
        self.update_on = datetime.now(UTC)
        self.usage_detail_fields = obj.usage_detail_fields

        obj_cfg = CarrierLowBalanceConfig.filter(CarrierLowBalanceConfig.client_id == obj.client_id).first()
        if obj_cfg:
            cfg = CarrierTemplateLowBalConfig()
            cfg.create_from_client(obj_cfg)
            self.low_balance_config = cfg


# +++++CarrierTemplate


class ResourceTemplate(DnlApiBaseModel):
    __tablename__ = 'resource_template'

    ROUTE_TYPE = {0: 'other', 1: 'intra', 2: 'inter', 3: 'highest'}
    LNP_DIPPING = {1: 'Add Header', 0: 'Not Add Header'}
    RPID_SCREEN = {0: 'None', 1: 'No', 2: 'Yes', 3: 'Proxy'}
    RPID_PARTY = {0: 'None', 1: 'Calling', 2: 'Called', 3: 'Proxy'}
    RPID_PRIVACY = {0: 'None', 1: 'Full', 2: 'Name', 3: 'Url', 4: 'OFF', 5: 'Ipaddr', 6: 'Proxy'}
    RPID_ID_TYPE = {0: 'None', 1: 'Subscriber', 2: 'User', 3: 'Term', 4: 'Proxy'}
    RPID = {0: 'Never', 1: 'Pass Through', 3: 'Always'}
    DIRECTION = {0: 'ingress', 1: 'egress'}

    name = Column(String(100))
    resource_template_id = Column(Integer, primary_key=True, server_default=text_(
        "nextval('resource_template_resource_template_id_seq'::regclass)"))

    direction = Column(ChoiceType(DIRECTION))

    trunk_type = Column(ChoiceType(Resource.TRUNK_TYPE), nullable=True)
    profit_type = Column(ChoiceType(Resource.PROFIT_TYPE), nullable=False)
    res_strategy = Column(ChoiceType(Resource.RES_STRATEGY))
    media_type = Column(ChoiceType(Resource.MEDIA_TYPE))
    rpid_screen = Column(ChoiceType(Resource.RPID_SCREEN))
    rpid_party = Column(ChoiceType(Resource.RPID_PARTY))
    rpid_privacy = Column(ChoiceType(Resource.RPID_PRIVACY))
    rpid_id_type = Column(ChoiceType(Resource.RPID_ID_TYPE))
    rpid = Column(ChoiceType(Resource.RPID), default=1)
    us_other = Column(ChoiceType(Resource.ROUTE_TYPE))
    us_route = Column(ChoiceType(Resource.ROUTE_TYPE))

    lnp_dipping = Column(Boolean)  # Column(ChoiceType(Resource.LNP_DIPPING))
    amount_per_port = Column(Numeric)
    bill_by = Column(Integer)
    billing_method = Column(ChoiceType(Resource.BILLING_METHOD), default=0)
    billing_rule = Column(Integer)
    canada_other = Column(ChoiceType(Resource.ROUTE_TYPE))
    canada_route = Column(ChoiceType(Resource.ROUTE_TYPE))
    codecs_str = Column(String(200))
    create_by = Column(String(100))
    create_on = Column(DateTime(True))
    delay_bye_second = Column(Integer)
    display_name = Column(SmallInteger)

    ignore_early_media = Column(Boolean)
    ignore_early_nosdp = Column(Integer)
    ignore_ring = Column(Boolean)
    inband = Column(Integer)
    info = Column(Integer)
    intl_route = Column(ChoiceType(Resource.ROUTE_TYPE))

    div = Column(ChoiceType(Resource.DIV))
    oli = Column(ChoiceType(Resource.OLI))
    paid = Column(ChoiceType(Resource.PAID))
    pci = Column(ChoiceType(Resource.PCI))
    priv = Column(ChoiceType(Resource.PRIV))

    lnp_dipping_rate = Column(Float(53))
    max_duration = Column(Integer)
    media_timeout = Column(Integer)

    profit_margin = Column(Float(53), nullable=False, default=0.0)
    random_table_id = Column(Integer)
    rate_decimal = Column(Integer)
    rate_profile = Column(Integer)
    rate_rounding = Column(ChoiceType(Resource.RATE_ROUNDING), default=0)
    rate_table_id = Column(Integer)
    re_invite = Column(Integer)
    re_invite_interval = Column(Integer)
    rfc2833 = Column(Integer)
    ring_timeout = Column(Integer, nullable=False, default=0)
    t38 = Column(Boolean, default=False, server_default=text_('false'))
    trunk_type2 = Column(ChoiceType(Resource.TRUNK_TYPE2), nullable=False, default=0)
    update_on = Column(DateTime(True))
    wait_ringtime180 = Column(Integer)
    hosts = Column(String(4096))
    failover = Column(String(4096))

    min_duration = synonym('delay_bye_second')
    resource = relationship(Resource, primaryjoin=foreign(Resource.resource_template_id) == resource_template_id,
                            uselist=True)

    @hybrid_property
    def used_by(self):
        return Resource.filter(and_(Resource.resource_template_id == self.resource_template_id,Resource.purged == False)).count()

    @hybrid_property
    def failover_rule(self):
        try:
            ret = []
            recs = self.failover.split(',')
            for r in recs:
                rec = r.split(':')
                try:
                    return_code = int(rec[0])
                    match_code = int(rec[1])
                except:
                    return_code = None
                    match_code = None
                ret.append({'return_code': return_code, 'match_code': match_code, 'return_clause': rec[2],
                            'failover_method': rec[3]})
            return ret
        except:
            return []

    @failover_rule.setter
    def failover_rule(self, value):
        try:
            self.failover = ','.join([str(x['return_code']) + ':' + str(x['match_code']) + ':' + x[
                'return_clause'] + ':' + x['failover_method'] for x in value])
        except:
            pass

    @hybrid_property
    def ip(self):
        try:
            ret = []
            recs = self.hosts.split(',')
            for r in recs:
                rec = r.split(':')
                try:
                    port = int(rec[2])
                    if 1000 > port or port > 65535:
                        port = None
                except:
                    port = None
                ret.append({'type': rec[0], 'host': rec[1], 'port': port})
            return ret
        except:
            return []

    @ip.setter
    def ip(self, value):
        try:
            self.hosts = ','.join([x['type'] + ':' + x['host'] + ':' + str(x['port']) for x in value])
        except:
            pass

    def create_from_resource(self, name, resource_id):
        obj = Resource.filter(Resource.resource_id == resource_id).first()
        if not obj:
            return
        self.name = name
        if obj.egress:
            self.direction = 'egress'
        else:
            self.direction = 'ingress'

        self.trunk_type = obj.trunk_type
        self.profit_type = obj.profit_type
        self.res_strategy = obj.res_strategy
        self.media_type = obj.media_type
        self.rpid_screen = obj.rpid_screen
        self.rpid_party = obj.rpid_party
        self.rpid_privacy = obj.rpid_privacy
        self.rpid_id_type = obj.rpid_id_type
        self.rpid = obj.rpid

        self.amount_per_port = obj.amount_per_port
        self.bill_by = obj.bill_by
        self.billing_method = obj.billing_method
        self.billing_rule = obj.billing_rule
        self.canada_other = obj.canada_other
        self.canada_route = obj.canada_route
        self._codecs_str = ','.join(obj.codecs)
        self.delay_bye_second = obj.delay_bye_second
        self.display_name = obj.display_name
        self.div = obj.div
        self.ignore_early_media = obj.ignore_early_media
        self.ignore_early_nosdp = obj.ignore_early_nosdp
        self.ignore_ring = obj.ignore_ring
        self.inband = obj.inband
        self.info = obj.info
        self.intl_route = obj.intl_route
        self.lnp_dipping = obj.lnp_dipping
        self.lnp_dipping_rate = obj.lnp_dipping_rate
        self.max_duration = obj.max_duration
        self.media_timeout = obj.media_timeout
        self.oli = obj.oli
        self.paid = obj.paid
        self.pci = obj.pci
        self.priv = obj.priv
        self.profit_margin = obj.profit_margin
        self.random_table_id = obj.random_table_id
        self.rate_decimal = obj.rate_decimal
        # todo ?? self.rate_profile = obj.rate_profile
        self.rate_rounding = obj.rate_rounding
        self.rate_table_id = obj.rate_table_id
        self.re_invite = obj.re_invite
        self.re_invite_interval = obj.re_invite_interval
        self.rfc2833 = obj.rfc2833
        self.ring_timeout = obj.ring_timeout
        self.trunk_type2 = obj.trunk_type2
        self.us_other = obj.us_other
        self.us_route = obj.us_route
        self.wait_ringtime180 = obj.wait_ringtime180
        ip = []
        for r in obj.ip:
            rec = {}
            if r.ip:
                rec['type'] = 'IP Address'
                rec['host'] = r.ip
            else:
                rec['type'] = 'FQDN'
                rec['host'] = r.fqdn
            if r.port:
                rec['port'] = r.port
            else:
                rec['port'] = 0
            if obj.egress:
                rec['direction'] = 1
            else:
                rec['direction'] = 0
            ip.append(rec)
        self.ip = ip
        self.resource.append(obj)

    @property
    def _codecs_str(self):
        result = []
        if self.codecs_str and self.codecs_str.startswith('['):
            codecs = eval(self.codecs_str)
        elif ',' in self.codecs_str:
            codecs = self.codecs_str.split(',')
        else:
            return '[]'
        for codec in codecs:
            if isinstance(codec, str):
                result.append(codec)
                continue
            codec_o = Codec.filter(Codec.id == codec).first()
            result.append(codec_o.name if codec_o else codec)
        return str(result)

    @_codecs_str.setter
    def _codecs_str(self, value):
        result = []

        if value and value.startswith('['):
            codecs = eval(value)
        elif ',' in value:
            codecs = value.split(',')
        else:
            codecs = [value]
        if codecs:
            for codec in codecs:
                codec_o = Codec.filter(Codec.name == codec).first()
                result.append(codec_o.id if codec_o else codec)
        self.codecs_str = str(result)

class IngressTrunkTemplate(ResourceTemplate):
    @classmethod
    def delete_query(cls):
        return ResourceTemplate.query().filter(ResourceTemplate.direction == 'ingress')


class EgressTrunkTemplate(ResourceTemplate):
    @classmethod
    def delete_query(cls):
        return ResourceTemplate.query().filter(ResourceTemplate.direction == 'egress')


# --- CDR --
### _cdr = select([func.sum(ClientCdr.egress_code_asr)])

# --- RerateCdrTask ---
class RerateCdrTaskTrunkRec(object):
    def __init__(self, str):
        if isinstance(str, type('')):
            d = str.split(',')
            self.trunk_id = int(d[0])
            self.update_rate = bool(d[1])
            self.rate_table_id = int(d[2])
            self.effective_date = datetime.utcfromtimestamp(int(d[3]))
        else:
            self.__dict__.update(str)

    def __repr__(self):
        upd = '1' if self.update_rate else '0'
        dat = str(int(mktime(self.effective_date.timetuple())))
        return '{},{},{},{}'.format(self.trunk_id, upd, self.rate_table_id, dat)


class RerateCdrTask(DnlApiBaseModel):
    __tablename__ = 'rerate_cdr_task'
    STATUS = {0: 'initial', 1: 'download cdr', 2: 'load rate', 3: 'rerate cdr', 4: 'finished', 5: 'error'}
    UPDATE_US_JURISDICTION = {0: 'no', 1: 'yes'}

    id = Column(Integer, primary_key=True, server_default=text_("nextval('rerate_cdr_task_id_seq'::regclass)"))
    cdr_file = Column(String(4096))
    cdr_file_path = Column(String(100))
    client_ids = Column(String(1024))
    create_time = Column(BigInteger, server_default=text_("EXTRACT(EPOCH from current_timestamp(0))"))
    download_file = Column(String(200))
    egress_trunk = Column(String(1024))
    end_time = Column(BigInteger)
    err_code = Column(Integer, nullable=False, server_default=text_("0"))
    from_time = Column(BigInteger)
    include_q850 = Column(SmallInteger, nullable=False, server_default=text_("0"))
    ingress_trunk = Column(String(1024))
    load_cdr_elapsed_sec = Column(Integer, nullable=False, server_default=text_("0"))
    load_rate_elapsed_sec = Column(Integer, nullable=False, server_default=text_("0"))
    lrn_server_ip = Column(String(16))
    lrn_server_port = Column(Integer, nullable=False, server_default=text_("5060"))
    progress = Column(String(200))
    rerate_elapsed_sec = Column(Integer, nullable=False, server_default=text_("0"))
    start_time = Column(BigInteger)
    status = Column(ChoiceType(STATUS), nullable=False, server_default=text_("0"))
    timezone = Column(Integer, nullable=False, server_default=text_("0"))
    to_time = Column(BigInteger)
    update_local_rate = Column(SmallInteger, nullable=False, server_default=text_("1"))
    update_lrn = Column(SmallInteger, nullable=False, server_default=text_("0"))
    update_us_jurisdiction = Column(ChoiceType(UPDATE_US_JURISDICTION), nullable=False, server_default=text_("0"))

    gmt = synonym('timezone')

    @hybrid_property
    def from_date(self):
        try:
            return datetime.fromtimestamp(self.from_time)
        except:
            return ''

    @from_date.setter
    def from_date(self, value):
        self.from_time = mktime(value.timetuple())

    @hybrid_property
    def to_date(self):
        try:
            return datetime.fromtimestamp(self.to_time)
        except:
            return ''

    @to_date.setter
    def to_date(self, value):
        self.to_time = mktime(value.timetuple())

    @hybrid_property
    def egress_trunks(self):
        ret = []
        try:
            recs = self.egress_trunk.split(';')
            for r in recs:
                ret.append(RerateCdrTaskTrunkRec(r))
        except Exception as e:
            log.warning(str(e))
            pass
        return ret

    @egress_trunks.setter
    def egress_trunks(self, value):
        try:
            ret = []
            for d in value:
                ret.append(str(RerateCdrTaskTrunkRec(d)))
            self.egress_trunk = ';'.join(ret)
        except Exception as e:
            log.warning(str(e))
            pass

    @hybrid_property
    def ingress_trunks(self):
        ret = []
        try:
            recs = self.ingress_trunk.split(';')
            for r in recs:
                ret.append(RerateCdrTaskTrunkRec(r))
        except Exception as e:
            log.warning(str(e))
            pass
        return ret

    @ingress_trunks.setter
    def ingress_trunks(self, value):
        try:
            ret = []
            for d in value:
                ret.append(str(RerateCdrTaskTrunkRec(d)))
            self.ingress_trunk = ';'.join(ret)
        except Exception as e:
            log.warning(str(e))
            pass


# --- RerateCdrTask ---

# +++ RerateReportExecLog
# --- RerateReportExecLog

# +++ RerateCdrDownloadLog
# --- RerateCdrDownloadLog
# +++ VersionInformation
class VersionInformation(DnlApiBaseModel):
    __tablename__ = 'version_information'
    __table_args__ = (
        # UniqueConstraint('program_name', 'switch_name', name='version_information_program_switch_name_idx'),
    )
    id = Column(Integer, primary_key=True, server_default=text_("nextval('version_information_id_seq'::regclass)"))
    program_name = Column(String(50))
    major_ver = Column(String(100))
    minor_ver = Column(String(100))
    build_date = Column(String(11))
    start_time = Column(DateTime(True))
    cli_ip = Column(String(30))
    cli_port = Column(Integer())
    last_status_time = Column(DateTime(True))
    status = Column(Integer())
    licensed_cps = Column(Integer())
    licensed_channel = Column(Integer())
    expires = Column(DateTime(True))
    switch_name = Column(String(100))
    serial_number = Column(String(100))


VoipGateway.uuid = column_property(select([VersionInformation.serial_number]).where(
    VersionInformation.switch_name == VoipGateway.name).limit(1).correlate_except(
    VersionInformation))


# --- VersionInformation


# -----  Not needed?
class DashboardTimeOption(DnlApiBaseModel):
    __tablename__ = 'dashboard_time_option'

    id = Column(Integer, primary_key=True, server_default=text_("nextval('dashboard_time_option_id_seq'::regclass)"))
    admin_point_time = Column(DateTime(True), index=True)
    client_point_time = Column(DateTime(True), index=True)
    iden = Column(String(30), index=True)


class ImportRateStatus(DnlApiBaseModel):
    __tablename__ = 'import_rate_status'

    id = Column(Integer, primary_key=True, server_default=text_("nextval('import_rate_status_id_seq'::regclass)"))
    rate_table_id = Column(Integer)
    status = Column(Integer)
    delete_queue = Column(Integer)
    update_queue = Column(Integer)
    insert_queue = Column(Integer)
    error_counter = Column(Integer)
    reimport_counter = Column(Integer)
    error_log_file = Column(String(256))
    reimport_log_file = Column(String(256))
    time = Column(DateTime)
    upload_file_name = Column(String(256))
    local_file = Column(String(256))
    method = Column(Integer)
    user_id = Column(Integer)
    start_epoch = Column(BigInteger)
    end_epoch = Column(BigInteger)
    sql_record = Column(Text)
    default_info = Column(String(200))


# +++
t_qos_route_report = Table(
    'qos_route_report', DnlApiBaseModel.metadata,
    Column('report_time', DateTime(True), nullable=False, index=True),
    Column('resource_id', Integer),
    Column('code', String(32)),
    Column('not_zero_calls', Integer),
    Column('total_calls', Integer),
    Column('bill_time', Integer),
    Column('cancel_calls', Integer),
    Column('busy_calls', Integer),
    Column('direction', Integer),
    Column('server_ip', String(36)),
    Column('pdd', Integer),
    Column('cost', Numeric(12, 6)),
    Column('call_duration', Integer),
    Column('total_pdd', Integer),
    Column('egress_channels', Integer),
    Index('qos_route_report_report_time_resource_id_idx', 'report_time', 'resource_id')
)


class ResourceBlockItem(DnlApiBaseModel):
    __tablename__ = 'resource_block_items'

    id = Column(Integer, primary_key=True, server_default=text_("nextval('resource_block_items_id_seq'::regclass)"))
    resource_id = Column(Integer)
    block_resource_id = Column(Integer)
    active = Column(SmallInteger, server_default=text_("1"))


class ResourceCapacity(DnlApiBaseModel):
    __tablename__ = 'resource_capacity'
    egress_id = Column(Integer, index=True, primary_key=True)
    ingress_id = Column(Integer, index=True, primary_key=True)
    max_cps = Column(Integer)
    max_cap = Column(Integer)

    @property
    def id(self):
        return str((self.egress_id, self.ingress_id))


ResourceCapacity.ingress_name = column_property(select([Resource.alias]). \
                                                where(
    Resource.resource_id == ResourceCapacity.ingress_id).correlate_except(Resource))
ResourceCapacity.egress_name = column_property(select([Resource.alias]). \
                                               where(
    Resource.resource_id == ResourceCapacity.egress_id).correlate_except(Resource))


class ResourceLrnAction(DnlApiBaseModel):
    __tablename__ = 'resource_lrn_action'

    id = Column(Integer, primary_key=True, server_default=text_("nextval('resource_lrn_action_id_seq'::regclass)"))
    direction = Column(SmallInteger, nullable=False)
    action = Column(SmallInteger, nullable=False)
    digits = Column(String(10), nullable=False)
    dnis = Column(PrefixRange, server_default=text_("''::prefix_range"))
    resource_id = Column(Integer, nullable=False)


class ServiceChargeItem(DnlApiBaseModel):
    __tablename__ = 'service_charge_items'

    id = Column(Integer, primary_key=True, server_default=text_("nextval('service_charge_items_id_seq'::regclass)"))
    min_rate = Column(Numeric(30, 10))
    max_rate = Column(Numeric(30, 10))
    charge_value = Column(Float)
    service_charge_id = Column(Integer)


class SipErrorCode(DnlApiBaseModel):
    __tablename__ = 'sip_error_code'

    sip_error_code_id = Column(Integer, primary_key=True,
                               server_default=text_("nextval('sip_error_code_sip_error_code_id_seq'::regclass)"))
    return_code = Column(Integer, nullable=False)
    return_code_str = Column(String(100))
    switch_error_code = Column(Integer)
    switch_error_code_str = Column(String(100))
    resource_id = Column(Integer)


t_system_configure = Table(
    'system_configure', DnlApiBaseModel.metadata,
    Column('id', Integer, primary_key=True, server_default=text_("nextval('system_configure_id_seq'::regclass)")),
    Column('limit_cps', Integer, nullable=False, server_default=text_("0")),
    Column('limit_cap', Integer, nullable=False, server_default=text_("0")),
    Column('switch_name', String(100), nullable=False, index=True),
    Column('create_time', DateTime(True), index=True,
           server_default=text_("('now'::text)::timestamp(0) with time zone"))
)

t_system_limit = Table(
    'system_limit', DnlApiBaseModel.metadata,
    Column('ingress_call_limit', Integer, nullable=False, server_default=text_("1")),
    Column('ingress_cps_limit', Integer, nullable=False, server_default=text_("1")),
    Column('egress_call_limit', Integer, nullable=False, server_default=text_("1")),
    Column('egress_cps_limit', Integer, nullable=False, server_default=text_("1"))
)


class TroubleTicketsTemplate(DnlApiBaseModel):
    __tablename__ = 'trouble_tickets_template'

    id = Column(Integer, primary_key=True, server_default=text_("nextval('trouble_tickets_template_id_seq'::regclass)"))
    name = Column(String(255))
    title = Column(String(255))
    content = Column(Text)
    created_at = Column(DateTime(True))
    updated_at = Column(DateTime(True))
    updated_by = Column(String(255))


class UsersLimit(DnlApiBaseModel):
    __tablename__ = 'users_limit'

    id = Column(Integer, primary_key=True, server_default=text_("nextval('users_limit_id_seq'::regclass)"))
    user_id = Column(ForeignKey('users.user_id', ondelete='CASCADE'))
    client_id = Column(ForeignKey('client.client_id', ondelete='CASCADE'))

    client = relationship('Client', uselist=False, back_populates='client_limits')
    user = relationship('User', uselist=False, back_populates='client_limits')

    @hybrid_property
    def client_name(self):
        return self.client.name


class AutoRateImportRule(DnlApiBaseModel):
    __tablename__ = 'auto_rate_import_rule'
    id = Column(Integer, primary_key=True)
    name = Column(String(100), unique=True)
    from_email = Column(String(100))
    mail_server_ip = Column(String(100))
    mail_server_password = Column(String(100))
    create_by = Column(String(100))
    create_on = Column(DateTime(True))
    egress_trunk_id = Column(ForeignKey('resource.resource_id', ondelete='SET NULL'), index=True)
    digits = Column(PrefixRange)


#  pcap query
class PcapQuery(DnlApiBaseModel):
    __tablename__ = 'pcap_query'
    STATUS = {
        9: 'progress',
        10: 'wrong parameters',
        8: 'not found',
        7: 'file error',
        6: 'no call id',
        5: 'can not copy',
        4: 'unknown error',
        1: 'initial',
        0: 'success'}
    STATUS_MESSAGE = {
        'progress': 'Work in progress.',
        'wrong parameters': 'Wrong query parameters, query aborted.',
        'not found': 'Have not found any pcap.lzma for the period, query aborted.',
        'file error': 'Сan not unpack downloaded from storage *.pcap.lzma and merge to '
                      'pcap staging, query aborted.',
        'no call id': 'No call id in PCAP but all parts in storage, abort query.',
        'can not copy': 'Can not copy pcap to web server via some reason (but have all for '
                        'that). Abort query.',
        'unknown error': 'Failure on pcap query',
        'success': 'Valid pcap query result file generated.'}
    query_key = Column(String(100), primary_key=True, default=generate_uuid_str())
    callid = Column(String(100), nullable=False)
    switch_ip = Column(String(15), nullable=False)
    start = Column(DateTime(True), nullable=False)
    finish = Column(DateTime(True), nullable=False)
    queued_time = Column(DateTime(True), default=datetime.now(UTC))
    complete_time = Column(DateTime(True))
    status = Column(ChoiceType(STATUS), default=1)
    msg = Column(String(100))
    public_url = Column(String(255))
    requested_by = Column(String(100))

    @property
    def file_name(self):
        if self.public_url:
            return '{}/{}.pcap'.format(settings.PCAP_WEB_PUBLIC_DIR, self.query_key)
        return None

    @hybrid_property
    def duration(self):
        try:
            return (self.finish - self.start).seconds
        except:
            return 0

    @duration.setter
    def duration(self, value):
        if self.start:
            self.finish = self.start + timedelta(seconds=value)

    def set_query_status(self, status, msg='', public_url=''):
        self.status = status
        if status == 'progress':  # 9:
            self.queued_time = datetime.now(UTC)
        else:
            self.complete_time = datetime.now(UTC)
        if status == 'success':
            self.public_url = public_url
        if not msg:
            msg = self.STATUS_MESSAGE[status]
        self.msg = msg[:99]
        self.save()


class PcapQueryTask(DnlApiBaseModel):
    __tablename__ = 'pcap_query_task'
    STATUS = {0: 'Initial', 1: 'In Process', 2: 'Partially Successful', 3: 'All Successful', 4: 'Failed'}
    id = Column(Integer, primary_key=True)
    operator_user = Column(String(40))
    pcap_start_time = Column(DateTime(True))
    pcap_end_time = Column(DateTime(True))
    filter = Column(String(1024))
    result_file = Column(String(1024))
    status = Column(ChoiceType(STATUS), nullable=False, server_default='0')
    progress = Column(String(1024))
    create_time = Column(DateTime(True), server_default=func.now())
    proc_start_time = Column(DateTime(True))
    proc_end_time = Column(DateTime(True))
    # capture_id = Column(String(16))
    comment_number = Column(String(32))
    capture_uuid = Column(String(36))
    mail = relationship('PcapQueryTaskMail', uselist=False, single_parent=True, cascade="all,delete")

    @property
    def download_link(self):
        return '{}/tool/pcap/download_result/{}'.format(settings.API_URL, self.id)

    @property
    def download_link_old(self):
        return '{}/tool/pcap/query_task/{}'.format(settings.API_URL, self.id)

    @property
    def cloud_shark_open_url(self):
        return self.sip_shark_open_url

    @property
    def sip_shark_open_url(self):
        if self.capture_uuid == 'no file found':
            return None
        if not self.capture_uuid or self.capture_uuid == 'error':
            try:
                import requests
                url = '{}/pcap/public/upload'.format(settings.SIPSHARK['base'])
                if not self.result_file:
                    if self.status == 'All Successful':
                        self.capture_uuid = 'no file found'
                        return None
                    self.capture_uuid = 'error'
                else:
                    captures_uuid = []
                    for result_file in self.result_file.split(','):
                        multipart_form_data = {
                            'file': open(result_file, 'rb')
                        }
                        log.debug('sip_shark_open_url {}'.format(url))
                        resp = requests.post(url=url, files=multipart_form_data, timeout=0.5)
                        log.debug('sip_shark_open_url {} responce {} {}'.format(self.id, resp.status_code, resp.text))
                        if resp.status_code in (200, 201, 202):
                            try:
                                captures_uuid.append(resp.json()['id'])
                            except Exception as e:
                                pass
                    self.capture_uuid = ','.join(captures_uuid) if len(captures_uuid) > 0 else 'error'
            except Exception as e:
                log.debug('sip_shark_open_url {} error {}'.format(self.id, e))
                self.capture_uuid = 'error'
        try:
            if self.capture_uuid != 'error':
                return '{}/{}'.format(settings.SIPSHARK['baseui'], self.capture_uuid)
        except Exception as e:
            log.debug('sip_shark_open_url SIPSHARK baseui not configured {}'.format(e))
            return 'SIPSHARK baseui not configured! capture_uuid is {}'.format(self.capture_uuid)
        return None
    @property
    def is_identity(self):
        import subprocess
        """Check if pcap file contains SIP Identity header"""
        if not self.result_file:
            return False
            
        try:
            for result_file in self.result_file.split(','):
                # Use ngrep to search for Identity header in pcap
                cmd = f"ngrep -I {result_file} -pq -W byline 'Identity:'"
                process = subprocess.Popen(cmd, shell=True, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
                stdout, stderr = process.communicate()
                
                # If ngrep found any matches, file contains Identity
                if process.returncode == 0 and stdout:
                    return True
                    
        except Exception as e:
            log.debug(f'Error checking Identity header in pcap {self.id}: {e}')
            
        return False


class PcapQueryTaskMail(DnlApiBaseModel):
    __tablename__ = 'pcap_query_task_mail'
    STATUS = {0: 'initial', 1: 'finished', 2: 'running', 3: 'success', 4: 'fail'}
    id = Column(ForeignKey('pcap_query_task.id', ondelete='CASCADE'), primary_key=True)
    direct = Column(JSON)
    email = Column(String(255))
    client_id = Column(Integer)
    job_start_time = Column(DateTime(True))
    job_end_time = Column(DateTime(True))
    status = Column(ChoiceType(STATUS), nullable=False, server_default='0')
    progress = Column(String(1024))

    start_time = column_property(
        select([PcapQueryTask.pcap_start_time]).where(PcapQueryTask.id == id).correlate_except(PcapQueryTask))
    end_time = column_property(
        select([PcapQueryTask.pcap_end_time]).where(PcapQueryTask.id == id).correlate_except(PcapQueryTask))

    pcap = relationship('PcapQueryTask', uselist=False, )
    client = relationship(Client, primaryjoin=foreign(client_id) == Client.client_id)

    @property
    def download_link(self):
        return self.pcap.cloud_shark_open_url


class StorageConfig(DnlApiBaseModel):
    # todo old seems deprecated
    __tablename__ = 'storage_config'
    storage = Column(Enum('local', 'ftp', 'sftp', 'gcs', name='storage_type'), nullable=False)
    conf_type = Column(Enum('cdr', 'pcap', name='parser_type'), nullable=False, primary_key=True, unique=True)
    remote_server_port = Column(Integer)
    ftp_port = Column(Integer)
    storage_days = Column(Integer, server_default=text_("60"))
    storage_days_local = Column(Integer, server_default=text_("60"))
    remote_server_ip = Column(Text)
    storage_path = Column(Text)
    username = Column(Text)
    password = Column(Text)
    ftp_ip = Column(Text)
    google_drive_key = Column(Text)
    google_drive_name = Column(Text)
    active = Column(Boolean, default=True)

    def update_settings(self):
        settings.PCAP_VOICE_PORT = str(self.remote_server_port)
        settings.PCAP_VOICE_IPS = self.remote_server_ip
        if self.remote_server_ip:
            settings.PCAP_VOICE_IPS_LIST = list(set(settings.PCAP_VOICE_IPS.split(',')))
        settings.PCAP_STORAGE_DIR = self.storage_path + '/storage/'
        settings.PCAP_TCPDUMP_DIR = self.storage_path + '/tcpdumpdir/'
        settings.PCAP_PCAP_DIR = self.storage_path + '/hash/'
        settings.PCAP_STAGING_DIR = self.storage_path + '/tmp/'
        settings.PCAP_WEB_PUBLIC_DIR = self.storage_path + '/www/'

        settings.PCAP_FTP_HOST = self.ftp_ip
        settings.PCAP_FTP_PORT = self.ftp_port

        settings.PCAP_FTP_USER = self.username
        settings.PCAP_FTP_PASS = self.password
        settings.PCAP_STORAGE_TYPE = self.storage
        settings.PCAP_CREDENTIALS_FILE = self.google_drive_key
        settings.PCAP_PROJECT_NAME = self.google_drive_name
        settings.PCAP_ACTIVE = self.active


class LcrTestLog(DnlApiBaseModel):
    __tablename__ = 'lcr_test_log'
    lcr_test_log_id_seq = Sequence('lcr_test_log_id_seq')
    id = Column(Integer, primary_key=True, server_default=lcr_test_log_id_seq.next_value())
    test_time = Column(DateTime(True), default=datetime.now(UTC))
    routing_plan_id = ForeignKey('route_strategy.route_strategy_id', ondelete='CASCADE')
    ani = Column(String(64))
    dnis = Column(String(64))
    test_result = Column(Text)


class LcrTest(DnlApiBaseModel):
    __tablename__ = 'lcr_test'
    JUR_TYPE_DICT = {0: 'A-Z', 1: 'US Jurisdictional', 2: 'US Non Jurisdictional', 3: 'OCN-LATA', 4: 'IJ - Inter',
                     5: 'IJ - Intra', 6: 'IJ - Max(Inter,Intra)', 7: 'IJ - True Math'}
    # {0: 'A-Z', 1: 'US Jurisdictional', 2: 'US Non Jurisdictional', 3: 'OCN-LATA'}
    lcr_test_id_seq = Sequence('lcr_test_id_seq')
    id = Column(Integer, primary_key=True, server_default=lcr_test_id_seq.next_value())
    name = Column(String(100), unique=True)
    type = Column(ChoiceType(JUR_TYPE_DICT))
    rate_table_id = Column(ForeignKey('rate_table.rate_table_id', ondelete='CASCADE'))
    static_route_id = Column(ForeignKey('product.product_id', ondelete='CASCADE'))
    product_item_id = Column(ForeignKey('product_items.item_id', ondelete='CASCADE'))
    product_item_resource_id = Column(ForeignKey('product_items_resource.id', ondelete='CASCADE'))
    routing_plan_id = Column(ForeignKey('route_strategy.route_strategy_id', ondelete='CASCADE'))
    route_id = Column(ForeignKey('route.route_id', ondelete='CASCADE'))
    client_id = Column(ForeignKey('client.client_id', ondelete='CASCADE'))
    trunk_id = Column(ForeignKey('resource.resource_id', ondelete='CASCADE'))
    ip_id = Column(ForeignKey('resource_ip.resource_ip_id', ondelete='CASCADE'))
    prefix_id = Column(ForeignKey('resource_prefix.id', ondelete='CASCADE'))

    egress_id = Column(ForeignKey('resource.resource_id', ondelete='CASCADE'))
    switch_id = Column(ForeignKey('voip_gateway.id', ondelete='CASCADE'))

    egress = relationship('Resource', primaryjoin=Resource.resource_id == egress_id, uselist=False)
    client = relationship('Client', uselist=False, single_parent=True, cascade="all,delete")
    trunk = relationship('Resource', primaryjoin=Resource.resource_id == trunk_id, uselist=False, single_parent=True,
                         cascade="all,delete")
    ip = relationship('ResourceIp', uselist=False, single_parent=True, cascade="all,delete")
    prefix = relationship('ResourcePrefix', uselist=False, single_parent=True, cascade="all,delete")
    product_item = relationship(ProductItems, uselist=False, single_parent=True, cascade="all,delete")
    product_item_resource = relationship(ProductItemsResource, uselist=False, single_parent=True, cascade="all,delete")
    static_route = relationship(Product, uselist=False, single_parent=True, cascade="all,delete")
    rate_table = relationship('RateTable', uselist=False, single_parent=True, cascade="all,delete")
    route_strategy = relationship('RouteStrategy', uselist=False, single_parent=True, cascade="all,delete")
    route = relationship('Route', uselist=False, single_parent=True, cascade="all,delete")

    switch = relationship(VoipGateway, uselist=False)

    def create_data(self):
        def random_ip():
            import random
            l = []
            while True:
                for i in range(0, 4):
                    l.append(str(random.randint(0, 255)))
                ip = '.'.join(l)
                return ip

        name = '#' + self.name
        self.egress = Resource.get(self.egress_id)
        jur_type = self.egress.rate_table.jur_type
        RATE = 1.0
        NOW = datetime.now(UTC)

        ####1. rate_table
        RateTable.filter(RateTable.name == name).delete()
        self.rate_table = RateTable(name=name, jur_type=jur_type)
        for c in [chr(a) for a in range(ord('1'), ord('9') + 1)]:
            self.rate_table.rates.append(Rate(code=c, rate=RATE, effective_date=datetime(NOW.year, 1, 1),
                                              min_time=6, interval=6
                                              ))
        ####2. product (static_route)
        Product.filter(Product.name == name).delete()
        self.static_route = Product(name=name)
        # id=self.static_route.save()
        self.product_item = ProductItems()  # product_id=id)
        self.product_item.product = self.static_route
        self.product_item_resource = ProductItemsResource()
        self.product_item_resource.product = self.product_item
        self.product_item_resource.resource = self.egress
        # self.product_item_resource.resource
        # self.product_item_resource.save()

        ####3.route
        RouteStrategy.filter(RouteStrategy.name == name).delete()
        self.route_strategy = RouteStrategy(name=name)
        self.route = Route(route_type=2)
        self.route.route_strategy = self.route_strategy
        self.route.static_route = self.static_route

        ####4. ingress resource
        Resource.filter(Resource.alias == name).delete()
        Client.filter(Client.name == name).delete()
        self.client = Client(name=name, enough_balance=True, mode='postpay',
                             # unlimited_credit=True,
                             allowed_credit=-1000,
                             cps_limit=10, update_by='system')

        self.trunk = Resource(alias=name, ingress=True, egress=False, enough_balance=True, res_strategy=1,
                              max_duration=3600, rpid=1)

        self.client.resource = self.trunk
        ####5. resource ip
        my_ip = VoipGateway.get(self.switch_id).sip_capture_ip
        # self.ip = ResourceIp(ip=my_ip, port=5065,direction=0)
        # self.ip = ResourceIp(ip=random_ip(), port = 5061)
        # self.ip = ResourceIp(ip='192.168.0.1', port=5060)
        self.ip = ResourceIp(ip='127.0.0.1', port=5080, direction=0)
        self.trunk.ip.append(self.ip)

        ####6. resource prefix
        self.prefix0 = ResourcePrefix()
        self.prefix0.rate_table = self.rate_table
        self.prefix0.route_strategy = self.route_strategy
        self.trunk.prefixes.append(self.prefix0)

        self.prefix = ResourcePrefix(tech_prefix='9999')
        self.prefix.rate_table = self.rate_table
        self.prefix.route_strategy = self.route_strategy
        self.trunk.prefixes.append(self.prefix)

        self.client.resources.append(self.trunk)

    # self.client.session().add(self.product_item_resource)
    # self.client.session().add(self.route)
    # self.client.save()

    def login(self):
        from api_dnl.utils.dnl_switch import DnlSwitchSession
        self.sess = DnlSwitchSession(self.switch.lan_ip, self.switch.lan_port)
        self.sess.login()

    def logout(self):
        self.sess.logout()
        self.sess = None

    def test_code(self, code):
        logged_in = False
        if not hasattr(self, 'sess') or not self.sess:
            self.login()
            logged_in = True
        dig = code  # + '777'
        it = None
        """
        if self.static_route:
            it = self.static_route.items.filter(ProductItems.digits==dig).first()
        else:
            raise ValidationError('No static route!')
        if not it:
            it=ProductItems(digits=dig,min_len=0,max_len=32,update_by='system')
            self.static_route.items.append(it)
        res=it.trunks.filter(ProductItemsResource.resource_id==self.trunk_id).first()
        if not res:
            res=ProductItemsResource(resource_id=self.trunk_id)
            it.trunks.append(res)
        self.static_route.save()
        pref=self.trunk.prefixes.filter(ResourcePrefix.code==code).first()
        if not pref:
            pref=ResourcePrefix(resource_id=self.trunk_id,rate_table_id=self.rate_table_id,
                                tech_prefix='9999',route_strategy_id=self.routing_plan_id,code=code,product_id=self.static_route_id)
            pref.save()
        """
        code_rate = None
        q = Rate.filter(and_(Rate.rate_table_id == self.egress.rate_table_id, Rate.code == code[0])).first()
        if not q:
            code_rate = Rate(code=code[0], rate=0.1, effective_date=str(datetime.now(UTC).date()),
                             rate_table_id=self.egress.rate_table_id)
            code_rate.save()
            sleep(3)

        ret = self.sess.call_simulation(caller_ip=self.ip.ip, caller_port=self.ip.port, ani='#' + str(self.trunk_id),
                                        dnis='9999' + code, include_blocked_route=1)
        count = 0
        while count < 32 and ret['payload']['Global_Route_State']['Origination_State'] == 'No Ingress Resource Found':
            sleep(0.5)
            ret = self.sess.call_simulation(caller_ip=self.ip.ip, caller_port=self.ip.port,
                                            ani='#' + str(self.trunk_id),
                                            dnis='9999' + code, include_blocked_route=1)
            count += 1
        if code_rate:
            code_rate.delete()

        if logged_in:
            self.logout()
        return ret

    def test_all_codes(self):
        pass


class TicketType(DnlApiBaseModel):
    __tablename__ = 'ticket_type'
    ticket_type = Column(String(50), primary_key=True)


class Ticket(DnlApiBaseModel):
    __tablename__ = 'ticket'
    STATUS = {0: 'none', 1: 'initial', 2: 'sent'}
    TICKET_STATUS = {0: 'open', 1: 'in progress', 2: 'closed', 4: 'waiting', 5: 'reopen'}
    DEPARTMENT = {0: 'Admin', 1: 'Finance', 2: 'NOC'}
    TICKET_TYPE = {1: 'AUTHCODE', 2: 'BUSYSIG', 3: 'CALLLOOP', 4: 'CALLTREAT', 5: 'CALLFWRD', 6: 'CALLERID',
                   7: 'CALLDROP',
                   8: 'WRONGROUTE', 9: 'DTMF', 10: 'EQUIPALRM', 11: 'FACDEGR', 12: 'FACILOOS', 13: 'FAS', 14: 'FAXFAIL',
                   15: 'FIBERCUT', 16: 'FRAUD8XX', 17: 'FRAUDAS', 18: 'FRAUDAC', 19: 'FRAUDEC', 20: 'FRAUDTRACE',
                   21: 'FRAUDTP',
                   22: 'FRAUDDRSF', 23: 'FRAUDO', 24: 'FRAUDNTWK', 25: 'FRAUDRC', 26: 'FRAUDSCAM', 27: 'FRAUDSPAM',
                   28: 'FRAUDSPAM',
                   29: 'WRNGTGSET', 30: 'CAPACITY', 31: 'INTLIBCALF', 32: 'INTLIBQOS', 33: 'INTLCALLF', 34: 'BLACKLIST',
                   35: 'NORINGBACK', 36: 'PKTLOSS', 37: 'PDD', 38: 'QUALITY', 39: 'RNRVPCLNTREGF', 40: 'VPINCALLF'}
    ticket_id_seq = Sequence('ticket_id_seq')
    id = Column(Integer, primary_key=True, server_default=ticket_id_seq.next_value())
    ticket_id = Column(ForeignKey('ticket.id', ondelete='CASCADE'))
    user_ip = Column(String(32))
    gateway_ip = Column(String(32))
    user_name = Column(ForeignKey('users.name', ondelete='CASCADE', onupdate='CASCADE'))
    url = Column(String(500))
    content = Column(Text)
    subject = Column(String(100))
    create_on = Column(DateTime(True), default=datetime.now(UTC),
                       server_default=text_("('now'::text)::timestamp(0) with time zone"))
    update_on = Column(DateTime(True), default=datetime.now(UTC),
                       server_default=text_("('now'::text)::timestamp(0) with time zone"))
    update_by = Column(String(100))
    attachments = relationship('TicketAttachment', uselist=True, back_populates='ticket',
                               order_by="TicketAttachment.create_on")
    ticket = relationship('Ticket', remote_side=[id], back_populates='comments', uselist=False)
    comments = relationship('Ticket', remote_side=[ticket_id], back_populates='ticket', uselist=True,
                            order_by="Ticket.update_on")
    status = Column(ChoiceType(STATUS))
    status_history = relationship('TicketHistory', uselist=True, back_populates='ticket',
                                  order_by="TicketHistory.update_on")
    ticket_status = Column(ChoiceType(TICKET_STATUS))
    department = Column(ChoiceType(DEPARTMENT))
    ticket_type = Column(String(50))

    name = synonym('user_name')
    title = synonym('subject')
    create_by = synonym('user_name')

    client_company = column_property(select([Client.company]). \
                                     where(user_name == Client.login).correlate_except(Client))
    client_name = column_property(select([Client.name]). \
                                  where(user_name == Client.login).correlate_except(Client))
    company_type = column_property(select([Client.company_type]). \
                                   where(user_name == Client.login).correlate_except(Client))
    # client_email = column_property(select(
    #     [case([(Client.email.isnot(None), Client.email), (Agent.email.isnot(None), Agent.email)], else_=User.email)]). \
    #                                where(
    #     or_(user_name == Client.login, and_(Agent.user_id == User.user_id, User.name == user_name),
    #         User.name == user_name)).limit(1).correlate_except(Client, Agent, User))
    client_email = column_property(select(
        [User.email]).where(User.name == user_name).limit(1).correlate_except(User))

    @property
    def _url(self):
        user = User.filter(User.name == self.user_name).first()
        if user:
            return '{}/config/export/public/{}'.format(settings.API_URL, user.avatar_id)

    @validates('ticket_type')
    def validate_ticket_type(self, key, value):
        if value in list(self.TICKET_TYPE.values()):
            return value
        else:
            ticket_type = TicketType.get(value)
            if ticket_type:
                return value
        raise Exception('incorrect ticket_type')

    @property
    def root(self):
        if self.ticket:
            return self.ticket
        else:
            return self

    @property
    def ticket_user(self):
        if self.ticket:
            return User.filter(User.name == self.ticket.user_name).first()
        else:
            return User.filter(User.name == self.user_name).first()

    def check_user(self, user):
        if user.user_type != 'admin' and self.ticket_user.name != user.name:
            return False
        return True

    def set_ticket_status(self, ticket_status, user_name):
        hist = TicketHistory(ticket_id=self.root.ticket_id, ticket_status=ticket_status, update_by=user_name)
        self.root.status_history.append(hist)
        self.root.ticket_status = ticket_status
        self.ticket_status = ticket_status

    def set_ticket_update(self, date, user_name):
        self.root.update_on = date
        self.root.update_by = user_name
        self.update_on = date
        self.update_by = user_name

    @property
    def mail_list(self):
        def user_mail(name):
            u = User.filter(User.name == name).first()
            if u and u.email:
                return u.email
            else:
                return None

        lst = []
        if self.ticket:
            return self.ticket.mail_list
        else:
            lst.append(user_mail(self.user_name))
            lst.append(user_mail(self.update_by))
            for t in self.comments:
                lst.append(user_mail(t.user_name))
                lst.append(user_mail(t.update_by))
        ret = [i for i in set(lst) if i]
        return ';'.join(ret)


class TicketHistory(DnlApiBaseModel):
    __tablename__ = 'ticket_history'
    id = Column(Integer, primary_key=True)
    ticket_id = Column(ForeignKey('ticket.id', ondelete='CASCADE'))
    update_by = Column(String(100))
    update_on = Column(DateTime(True), default=datetime.now(UTC),
                       server_default=text_("('now'::text)::timestamp(0) with time zone"))
    ticket_status = Column(ChoiceType(Ticket.TICKET_STATUS))
    ticket = relationship('Ticket', uselist=False, back_populates='status_history')


class TicketAttachment(DnlApiBaseModel):
    __tablename__ = 'ticket_attachment'
    # uuid = Column(Integer, primary_key=True)
    uuid = Column(String(36), primary_key=True, default=generate_uuid_str())
    ticket_id = Column(ForeignKey('ticket.id', ondelete='CASCADE'))
    orig_file = Column(String(500))
    create_on = Column(DateTime(True), default=datetime.now(UTC))
    ticket = relationship('Ticket', uselist=False, back_populates='attachments')

    @property
    def file_name(self):
        if self.orig_file:
            return '{}/att_{}.{}'.format(settings.FILES['upload_to'], self.uuid, self.orig_file.split('.')[-1])
        return None

    @property
    def file_data(self):
        try:
            f = open(self.file_name, 'rb')
            return f.read()
        except:
            return None

    @property
    def download_link(self):
        return '{}/ticket/file/{}'.format(settings.API_URL, self.uuid)


class PaymentAndGateway(DnlApiBaseModel):
    __tablename__ = '#payment_and_gateway'
    # alias(Code.query().session.query(Code.name.label('destination'),
    #                func.count().label('codes_count')).group_by(Code.name)) #'#code_destination'
    __table_args__ = {'prefixes': ['TEMPORARY']}

    amount = Column(Numeric(20, 10), nullable=False)
    charge_amount = Column(Numeric(20, 10), nullable=False)
    client_id = Column(Integer())
    client_name = Column(String(100))
    fee = Column(Numeric(20, 10), nullable=False)
    paid_on = Column(DateTime(True))
    transaction_id = Column(String(100), primary_key=True)
    method = Column(String(100))
    invoice_id = Column(String(255))
    status = Column(ChoiceType(PaymentGatewayHistory.STATUS))

    @classmethod
    def query(cls):
        p = ClientPayment
        q = select([p.amount.label('amount'), p.client_id.label('client_id'), p.client_name.label('client_name'),
                    func.cast(p.client_payment_id, String).label('transaction_id'), p.note.label('note'),
                    p.receiving_time.label('paid_on'), literal_column('0').label('fee'),
                    literal_column("''").label('invoice_id'), literal_column('0').label('charge_amount'),
                    literal_column("'manual'").label('method'), literal_column("2").label('status')]).where(p.payment_method.is_(None))

        p = PaymentGatewayHistory
        q1 = select([
            p.amount.label('amount'), p.client_id.label('client_id'), p.client_name.label('client_name'),
            p.transaction_id.label('transaction_id'), literal_column("''").label('note'),
            p.created_time.label('paid_on'), p.fee.label('fee'), p.invoice_id.label('invoice_id'),
            p.charge_amount.label('charge_amount'), func.cast(p.method, String).label('method'), p.status.label('status')]
        )
        xu = q.union(q1).alias(cls.__tablename__)
        cls._tab = xu
        return get_db().session.query(xu.c.amount.label('amount'), xu.c.client_id.label('client_id'),
                                      xu.c.client_name.label('client_name'), xu.c.note.label('note'),
                                      xu.c.transaction_id.label('transaction_id'),
                                      xu.c.paid_on.label('paid_on'), xu.c.invoice_id.label('invoice_id'),
                                      xu.c.fee.label('fee'), xu.c.status.label('status'),
                                      xu.c.charge_amount.label('charge_amount'), xu.c.method.label('method'))

    @property
    def paid_on_fmt(self):
        return self.paid_on.strftime('%Y-%m-%d %T')


class AsyncFtpExport(DnlApiBaseModel):
    __tablename__ = 'async_ftp_export'
    STATUS = {0: 'Processing', 1: 'Uploading', 2: 'Complete', 3: 'Failed to process', 4: 'Failed to upload',
              6: ' Request interrupted by user'}
    time = Column(DateTime(True), default=datetime.now(UTC), nullable=False)
    request_id = Column(String(20), nullable=False, primary_key=True)
    job_id = Column(BigInteger(), nullable=False)
    username = Column(String(100))
    status = Column(ChoiceType(STATUS), nullable=False, server_default='0')
    err_code = Column(Integer(), nullable=False, server_default='0')
    error_str = Column(Text())


class AllowedSendToIp(DnlApiBaseModel):
    __tablename__ = 'allowed_sendto_ip'
    id = Column(Integer, primary_key=True)
    resource_id = Column(ForeignKey('resource.resource_id', ondelete='CASCADE'))
    direction = Column(Integer)
    sip_profile_ip = Column(Ip4r)
    sip_profile_port = Column(Integer)

    trunk_id = synonym('resource_id')

    trunk = relationship(IngressTrunk, uselist=False, back_populates='allowed_sendto_ips')
    sip_profile = relationship(SwitchProfile,
                               primaryjoin=and_(SwitchProfile.sip_ip == foreign(cast(sip_profile_ip, String)),
                                                SwitchProfile.sip_port == foreign(sip_profile_port)))

    voip_gateway_name = column_property(select([SwitchProfile.server_name]). \
        where(and_(SwitchProfile.sip_ip == cast(sip_profile_ip, String),
                   SwitchProfile.sip_port == sip_profile_port)).correlate_except(
        SwitchProfile))
    sip_profile_name = column_property(select([SwitchProfile.profile_name]). \
        where(and_(SwitchProfile.sip_ip == cast(sip_profile_ip, String),
                   SwitchProfile.sip_port == sip_profile_port)).correlate_except(
        SwitchProfile))

    sip_profile_id = column_property(select([SwitchProfile.id]). \
        where(and_(SwitchProfile.sip_ip == cast(sip_profile_ip, String),
                   SwitchProfile.sip_port == sip_profile_port)).correlate_except(
        SwitchProfile))


class CallingQueue(DnlApiBaseModel):
    __tablename__ = 'calling_queue'
    __table_args__ = (
        UniqueConstraint('req_uuid', 'call_id'),
    )

    qid = Column(BigInteger, primary_key=True)
    req_uuid = Column(String(36), nullable=False)
    call_id = Column(BigInteger, nullable=False)

    @classmethod
    def get_next(cls):
        q = cls.query().order_by(cls.req_uuid, cls.call_id).first()
        if q:
            req_uuid, call_id = q.req_uuid, q.call_id
            q.delete()
        else:
            req_uuid, call_id = None, None
        return req_uuid, call_id

    @classmethod
    def put(cls, batch_uuid, phone_numbers):
        if type(phone_numbers) == list:
            for phone_number in phone_numbers:
                cls.session().add(cls(req_uuid=batch_uuid, call_id=phone_number))
        else:
            cls.session().add(cls(req_uuid=batch_uuid, call_id=phone_numbers))
        try:
            cls.session().commit()
        except Exception as e:
            log.debug('queue put:{}'.format(str(e)))
            pass


class CallingTrack(DnlApiBaseModel):
    __tablename__ = 'calling_track'
    uuid = Column(String(36), primary_key=True, default=generate_uuid_str())
    orig_file = Column(String(256))
    create_by = Column(String(100))
    create_on = Column(DateTime(True), nullable=False,
                       server_default=text_("('now'::text)::timestamp(0) with time zone"))

    @property
    def file_name(self):
        if self.orig_file:
            return '{}/{}.{}'.format(settings.FILES['call_tracks'], self.uuid, self.orig_file.split('.')[-1])
        return None

    @property
    def file_data(self):
        try:
            f = open(self.file_name, 'rb')
            return f.read()
        except:
            return None

    @property
    def download_link(self):
        return '{}/tool/call/track/{}'.format(settings.API_URL, self.uuid)


class CallingCall(DnlApiBaseModel):
    __tablename__ = 'calling_call'
    __table_args__ = (
        UniqueConstraint('req_uuid', 'dst_number'),
    )
    STATE = {0: 'WAITING', 1: 'RINGING', 2: 'CALLING', 3: 'COMPLETE'}
    call_id = Column(BigInteger, primary_key=True)
    call_uuid = Column(String(36))
    req_uuid = Column(ForeignKey('calling_request.req_uuid', ondelete='CASCADE'))
    start_time = Column(DateTime(True))
    finish_time = Column(DateTime(True))
    dst_number = Column(String(128), nullable=False)
    track_uuid = Column(ForeignKey('calling_track.uuid', ondelete='CASCADE'))
    state = Column(ChoiceType(STATE), nullable=False, default=0)
    pdd = Column(Integer)
    ring_duration = Column(Integer)
    call_duration = Column(Integer)
    rbt_packet = Column(Integer)
    call_packet = Column(Integer)
    response_code = Column(Integer)
    result = Column(String(512))
    rbt_audio_path = Column(String(256))
    call_audio_path = Column(String(256))
    sip_pcap_path = Column(String(256))
    caller_id = Column(String(128))
    pcap_stat = Column(String(1024))
    create_by = Column(String(100))

    calling_request = relationship('CallingRequest', uselist=False, back_populates='calls')
    queue = relationship('CallingQueue', uselist=False, primaryjoin=and_(foreign(CallingQueue.req_uuid) == req_uuid,
                                                                         foreign(CallingQueue.call_id) == call_id))
    track = relationship('CallingTrack', uselist=False)

    @hybrid_property
    def pcap(self):
        try:
            return json.loads(self.pcap_stat)
        except:
            return None

    @pcap.setter
    def pcap(self, value):
        self.pcap_stat = json.dumps(value)

    @property
    def file_name(self):
        if self.orig_file:
            return '{}/{}/{}'.format(settings.FILES['call_tracks'], self.call_uuid, self.call_audio_path)
        return None

    @property
    def file_data(self):
        try:
            f = open(self.file_name, 'rb')
            return f.read()
        except:
            return None

    @property
    def download_link(self):
        return '{}/tool/call/{}'.format(settings.API_URL, self.uuid)


class CallingRequest(DnlApiBaseModel):
    __tablename__ = 'calling_request'

    req_uuid = Column(String(36), primary_key=True, default=generate_uuid_str())
    req_from_ip = Column(String(64))
    req_from_port = Column(Integer)
    start_time = Column(DateTime(True), nullable=False,
                        server_default=text_("('now'::text)::timestamp(0) with time zone"))
    finish_time = Column(DateTime(True))
    total_call_cnt = Column(Integer, nullable=False, default=0)
    complete_call_cnt = Column(Integer, server_default=text_("0"))
    status = Column(String(64))

    calls = relationship('CallingCall', uselist=True, back_populates='calling_request')

    def on_complete_call(self):
        cnt = len([c for c in self.calls if c.state == 'COMPLETE'])
        self.complete_call_cnt = cnt  # self.complete_call_cnt+1
        if self.complete_call_cnt >= self.total_call_cnt:
            self.status = 'COMPLETE'
            self.finish_time = datetime.now(UTC)
        else:
            self.status = 'IN_PROGRESS'
            self.finish_time = None

    @classmethod
    def new(cls, phone_numbers, track_uuid=None, caller_id=None, ip=None, port=None):
        req = cls(req_from_ip=ip, req_from_port=port)
        for num_elem in phone_numbers:
            if isinstance(num_elem, dict):
                call = CallingCall(dst_number=num_elem.get('phone_number'),
                                   track_uuid=num_elem.get('track_uuid', track_uuid),
                                   state='WAITING', caller_id=caller_id)
            else:
                call = CallingCall(dst_number=num_elem,
                                   track_uuid=track_uuid,
                                   state='WAITING', caller_id=caller_id)
            call.queue = CallingQueue()
            req.calls.append(call)
        req_uuid = req.save()

        return req_uuid


class DidVendor(DnlApiBaseModel):
    __tablename__ = 'did_vendor_api'
    TYPE = {0: 'Manual dids', 1: 'Inteliquent API', 2: 'Teli API', 3: 'SMSC API'}
    id = Column(Integer, ForeignKey('client.client_id', ondelete='CASCADE'), primary_key=True)
    type = Column(ChoiceType(TYPE), server_default='0')
    SEARCH_TYPE = {None: 'all', 1: 'local', 2: 'toll_free', 3: 'international'}
    search_type = Column(ChoiceType(SEARCH_TYPE))
    username = Column(String(64))
    password = Column(String(64))
    msg_api_key = Column(String(64))
    sms_enabled = Column(Boolean(), default=False)
    default_received_sms_url = Column(Text)
    vendor_billing_rule_id = Column(Integer, ForeignKey('did_billing_plan.id', ondelete='CASCADE'))
    client_billing_rule_id = Column(Integer, ForeignKey('did_billing_plan.id', ondelete='CASCADE'))
    vendor = relationship('Client', uselist=False, back_populates='vendor_api')
    api_enabled = Column(Boolean, server_default='True')
    trunk_group_name = Column(String(128))
    toll_free_routing_label = Column(String(128))
    vendor_billing_rule_rate_table_id = column_property(
        select([DidBillingPlan.rate_table_id]).where(DidBillingPlan.id == vendor_billing_rule_id).correlate_except(
            DidBillingPlan))


    @classmethod
    def get_client_rule(cls, vendor_id, did):
        if vendor_id:
            plan = None
            # if vendor and vendor.client_billing_rule_id:
            #     plan = DidBillingPlan.get(vendor.client_billing_rule_id)
            #     if plan:
            #         ret.update(plan.as_dict())

            # obj = DidBillingRel.filter(
            #         and_(DidBillingRel.did == did,
            #             DidBillingRel.end_date.is_(None), DidBillingRel.egress_res_id.is_(None))).first()
            obj = DidRepository.filter(DidRepository.did == did).first()
            cli = Client.get(int(vendor_id))
            if cli and cli.did_product and obj:
                pcls = DidProductItem
                product = pcls.filter(and_(pcls.did_product_id == cli.did_product.did_product_id,
                                            (pcls.type == 'Toll-Free') == obj.is_toll_free,
                                            pcls.country == obj.country)).first()
                plan = DidBillingPlan.get(product.billing_rule_id) if product else None
                # plan = DidBillingPlan.get(obj.vendor_billing_plan_id) if obj else None
                if not plan:
                    pcls = DidProductItem
                    product = pcls.filter(pcls.did_product_id == cli.did_product.did_product_id).first()
                    plan = DidBillingPlan.get(product.billing_rule_id) if product else None
            # if cli and cli.resource:
            #     if obj and obj.buy_billing_plan:
            #         plan = obj.buy_billing_plan
            #         ret.update(plan.as_dict())
            return plan
        return None

    @classmethod
    def fees(cls, vendor_id=None, did=None):
        ret = {}
        rule = cls.get_client_rule(vendor_id, did)
        if rule:
            ret.update(rule.as_dict())
        return ret

    def get_api(self):
        from api_dnl.utils.did_api import DidAPIInteliquent, DidAPITeli, DidAPISmsc, DidAPIManual
        api = None
        if self.type == 'Inteliquent API':
            api = DidAPIInteliquent(self.username, self.password, self)
            api.authenticate()
        if self.type == 'Teli API':
            api = DidAPITeli(self.username, self.password, self)
            api.authenticate()
        if self.type == 'SMSC API':
            api = DidAPISmsc(self.username, self.password, self)
            api.authenticate()
        if self.type == 'Manual dids':
            api = DidAPIManual(self.username, self.password, self)
        return api


Client.has_api = column_property(
    select([func.count(DidVendor.id) > 0]).where(DidVendor.id == Client.client_id).correlate_except(Client))

DidVendorLog.created_by_name = column_property(
    select([User.name]).where(User.user_id == DidVendorLog.created_by).correlate_except(User))
DidVendorLog.deleted_by_name = column_property(
    select([User.name]).where(User.user_id == DidVendorLog.deleted_by).correlate_except(User))
DidClientLog.assigned_by_name = column_property(
    select([User.name]).where(User.user_id == DidClientLog.assigned_by).correlate_except(User))
DidClientLog.released_by_name = column_property(
    select([User.name]).where(User.user_id == DidClientLog.released_by).correlate_except(User))
DidNumberDeleteTask.operator_user_id = column_property(
    select([User.user_id]).where(User.name == DidNumberDeleteTask.operator_user).correlate_except(User))


# +++ DidParam
class DidParam(DnlApiBaseModel):
    __tablename__ = 'did_param'
    TOLL_FREE_DID_PREFIXES = ['1800', '1888', '1877', '1866', '1855', '1844', '1833']
    TYPE = {0: 'long code', 2: 'toll free', 3: 'short code'}
    did = Column(String(11), primary_key=True)
    country_iso = Column(String(3), index=True)
    state = Column(String(3), index=True)
    ocn = Column(String(8), index=True)
    lata = Column(String(8), index=True)
    rate_type = Column(String(), index=True)
    clec = Column(String(), index=True)
    type = Column(ChoiceType(TYPE), server_default='0')
    company = Column(String(), index=True)
    switch = Column(String(), index=True)
    country = synonym('country_iso')

    def update(self):
        did = self.did
        npa, nxx, prefix = '', '', ''
        if len(did) == 10:
            did = '1' + did
            self.type = 3
        if len(did) == 11:
            if did[0] == '1':
                npa = did[1:4]
                nxx = did[4:7]
                prefix = did[0:7]
            else:
                log.debug('international prefix did_param {}'.format(self.did))
                return
        else:
            log.debug('wrong did_param {}'.format(self.did))
            return
        if self.type != 3:
            if '1' + npa in self.TOLL_FREE_DID_PREFIXES:
                self.type = 2
            else:
                self.type = 0
        # lerg = C4UsLerg.filter(and_(C4UsLerg.npa == npa, C4UsLerg.nxx == nxx)).first()
        # if lerg:
        #     self.country_iso = lerg.country
        #     self.state = lerg.state
        #     self.lata = lerg.lata
        #     self.ocn = lerg.ocn
        # else:
        #     pref = JurisdictionPrefix.filter(JurisdictionPrefix.prefix == prefix).first()
        #     if pref:
        #         self.country_iso = pref.jurisdiction_country_name
        #         self.state = pref.jurisdiction_name
        #         self.lata = pref.lata
        #         self.ocn = pref.ocn
        try:

            url = f'http://lerg.denovolab.com/number/{did}'
            resp = requests.get(url)
            if resp.status_code in (200, 201, 202):
                data = resp.json()
                self.country_iso = data.get('country', None)
                self.state = data.get('state', None)
                self.clec = data.get('clec_type', None)
                self.lata = data.get('lata', None)
                self.ocn = data.get('ocn', None)
                self.rate_type = data.get('rate_center', None)
                self.company = data.get('company', None)
                self.switch = data.get('switch', None)
            else:
                log.error('did_param.update status {} message {}'.format(resp.status_code, str(resp.content)))
        except Exception as e:
            log.error('did_param.update error {}'.format(e))


# +++ DidBillingRel
class DidBillingRel(DnlApiBaseModel):
    __tablename__ = 'did_billing_rel'
    TOLL_FREE_DID_PREFIXES = ['1800', '1888', '1877', '1866', '1855', '1844', '1833']
    STATUS = {0: 'PNDNG', 1: 'INSVC', 2: 'PDISC'}
    id = Column(Integer, primary_key=True, server_default=text_("nextval('did_billing_rel_id_seq'::regclass)"))
    did = Column(String(50), index=True, nullable=False, unique=True)
    sell_billing_plan_id = Column(Integer)
    buy_billing_plan_id = Column(Integer)
    did_billing_id = Column(Integer)
    ingress_res_id = Column(Integer, index=True)
    egress_res_id = Column(Integer, index=True)
    assigned_client_id = Column(Integer, index=True)
    start_date = Column(DateTime(True), index=True)
    end_date = Column(DateTime(True), index=True)
    is_available = Column(Boolean, default=False)
    status = Column(ChoiceType(STATUS), server_default='0')
    is_sms = Column(Boolean, default=False)
    ip = Column(String(50), index=True)
    note = Column(String(16), index=True)

    did_plus = column_property(case([(func.length(did) == 10, func.concat('+1', did))], else_=func.concat('+', did)))

    # state = column_property(select([JurisdictionPrefix.jurisdiction_name]).where(
    #     JurisdictionPrefix.prefix == func.substr(did, 1, 7)).correlate_except(JurisdictionPrefix))
    # ocn = column_property(select([JurisdictionPrefix.ocn]).where(
    #     JurisdictionPrefix.prefix == func.substr(did, 1, 7)).correlate_except(JurisdictionPrefix))
    # lata = column_property(select([JurisdictionPrefix.lata]).where(
    #     JurisdictionPrefix.prefix == func.substr(did, 1, 7)).correlate_except(JurisdictionPrefix))
    is_toll_free = column_property(func.substr(did, 1, 4).in_(TOLL_FREE_DID_PREFIXES))
    npa = column_property(func.substr(did, 1, 3))
    client_id = column_property(
        select([Resource.client_id]).where(Resource.resource_id == egress_res_id).correlate_except(Resource))
    vendor_id = column_property(
        select([Resource.client_id]).where(Resource.resource_id == ingress_res_id).correlate_except(Resource))
    # did_type = column_property(case([(func.substr(did, 1, 4).in_(TOLL_FREE_DID_PREFIXES), 'toll_free')], else_='local'))

    country = column_property(select([DidParam.country_iso]).where(DidParam.did == did).correlate_except(DidParam))
    state = column_property(select([DidParam.state]).where(DidParam.did == did).correlate_except(DidParam))
    ocn = column_property(select([DidParam.ocn]).where(DidParam.did == did).correlate_except(DidParam))
    lata = column_property(select([DidParam.ocn]).where(DidParam.did == did).correlate_except(DidParam))
    rate_type = column_property(select([DidParam.rate_type]).where(DidParam.did == did).correlate_except(DidParam))
    company = column_property(select([DidParam.company]).where(DidParam.did == did).correlate_except(DidParam))
    switch = column_property(select([DidParam.switch]).where(DidParam.did == did).correlate_except(DidParam))
    # is_toll_free = column_property(select([DidParam.type == 2]).where(DidParam.did == did).correlate_except(DidParam))
    did_type = column_property(select([DidParam.type]).where(DidParam.did == did).correlate_except(DidParam))

    mrc = column_property(
        select([DidBillingPlan.monthly_charge]).where(DidBillingPlan.id == buy_billing_plan_id).correlate_except(
            DidBillingPlan))
    nrc = column_property(
        select([DidBillingPlan.did_price]).where(DidBillingPlan.id == buy_billing_plan_id).correlate_except(
            DidBillingPlan))
    rate_table_id = column_property(
        select([DidBillingPlan.rate_table_id]).where(DidBillingPlan.id == buy_billing_plan_id).correlate_except(
            DidBillingPlan))
    rate_per_min = column_property(
        select([DidBillingPlan.rate_per_min]).where(DidBillingPlan.id == buy_billing_plan_id).correlate_except(
            DidBillingPlan))

    # created_time = synonym('start_date')???
    assigned_date = column_property(case([(egress_res_id.is_(None), None)], else_=start_date))

    # client_res_id = synonym('ingress_res_id')
    # vendor_res_id = synonym('egress_res_id')
    client_res_id = synonym('egress_res_id')
    vendor_res_id = synonym('ingress_res_id')

    client_billing_rule_id = synonym('sell_billing_plan_id')

    vendor_billing_rule_id = synonym('buy_billing_plan_id')

    sell_billing_plan = relationship('DidBillingPlan',
                                     primaryjoin='foreign(DidBillingRel.sell_billing_plan_id) == DidBillingPlan.id',
                                     uselist=False)  # , back_populates='sell_dids')

    client_billing_rule = synonym('sell_billing_plan')

    buy_billing_plan = relationship('DidBillingPlan',
                                    primaryjoin='foreign(DidBillingRel.buy_billing_plan_id) == DidBillingPlan.id',
                                    uselist=False)  # , back_populates='buy_dids')
    vendor_billing_rule = synonym('buy_billing_plan')

    vendor_res = relationship('Resource',
                              primaryjoin='foreign(DidBillingRel.vendor_res_id) == Resource.resource_id',
                              uselist=False, back_populates='vendor_dids')

    client_res = relationship('Resource',
                              primaryjoin='foreign(DidBillingRel.client_res_id) == Resource.resource_id',
                              uselist=False, back_populates='client_dids')
    rate = relationship('Rate', primaryjoin='Rate.code==foreign(DidBillingRel.did)')
    item = relationship('ProductItems', primaryjoin='ProductItems.digits==foreign(DidBillingRel.did)')
    did_param = relationship('DidParam', primaryjoin='DidParam.did==foreign(DidBillingRel.did)')

    vendor_billing_rule_name = column_property(
        select([DidBillingPlan.name]).where(DidBillingPlan.id == buy_billing_plan_id).correlate_except(DidBillingPlan))
    client_billing_rule_name = column_property(
        select([DidBillingPlan.name]).where(DidBillingPlan.id == sell_billing_plan_id).correlate_except(DidBillingPlan))
    assigned_client_name = column_property(
        select([Client.name]).where(Client.client_id == assigned_client_id).correlate_except(Client))

    _is_available = column_property(
        and_(or_(end_date.is_(None), end_date > datetime.now(UTC).date()), egress_res_id.is_(None)))
    is_effective = column_property(or_(end_date.is_(None), end_date > datetime.now(UTC).date()))
    is_assigned = column_property(is_available==False)

    # is_unassigned = column_property(and_(ingress_res_id.is_(None), or_(end_date.is_(None), end_date > datetime.now(UTC))))

    @property
    def npa(self):
        return self.did[1:4] if len(self.did) == 11 else self.did[0:3]

    def before_save(self):
        # if self.is_available and self.client_res_id:
        #    self.is_available=False
        if not self.end_date is None:
            self.on_end_date()
        did_param = DidParam.get(self.did)
        if not did_param and len(self.did) in (10, 11):
            d = DidParam(did=self.did)
            d.update()
            self.session().add(d)
        # if not self.client_id:
        #     self.is_available = True
        # else:
        #     self.is_available = False


    @validates('did')
    def validate_did(self, key, value):
        try:
            int(value)
            return value
        except Exception as e:
            raise Exception('did can\'t contain non numeric characters')

    def on_end_date(self):
        "remove assotiated records"
        ResourcePrefix.filter(
            and_(ResourcePrefix.code == self.did, ResourcePrefix.resource_id == self.ingress_res_id)).delete(
            synchronize_session='fetch')
        piq = ProductItems.filter(ProductItems.digits == self.did)
        pil = [i.item_id for i in piq.all()]
        if pil:
            ProductItemsResource.filter(
                ProductItemsResource.item_id.in_(pil)).delete(
                synchronize_session='fetch')
        piq.delete(synchronize_session='fetch')
        log.debug('did.on_end_date success {} '.format(self.did))

    @classmethod
    def on_import_delete(cls, rec, mode):
        if mode == 'exact':
            res_ids = [r.resource_id for r in Resource.get(rec.egress_res_id).client.egress_trunks]
            old = cls.filter(and_(cls.did == rec.did, cls.client_res_id.in_(res_ids))).filter(
                cls.end_date.is_(None)).all()
        else:
            old = cls.filter(and_(cls.did == rec.did)).filter(
                cls.end_date.is_(None)).all()
        for o in old:
            o.on_end_date()
            o.end_date = datetime.utcnow().replace(microsecond=0)
            o.is_available = False
            o.save()

    @classmethod
    def did_billing_rel_disconnect(cls, did, client_id, vendor_id, buy_billing_plan_id=None, user_name='unknown'):
        old = cls.filter(or_(cls.did == did, cls.did == did[1:], cls.did == '+{}'.format(did))).filter(cls.end_date.is_(None)).all()
        user = User.filter(User.name == user_name).first()
        user_id = user.user_id if user else None
        for o in old:
            # o.delete()
            # o.on_end_date()
            # o.end_date = datetime.utcnow().replace(microsecond=0)
            client_id = client_id or getattr(o, 'client_id', None)
            o.is_available = True
            o.client_billing_rule_id = None
            o.client_res_id = None
            o.ip = None
            o.assigned_client_id = None
            DidClientLog.on_release_did(o.did, user_id, 'Import', client_id=client_id)
            o.save()
        product_items = ProductItems.filter(ProductItems.digits == did[1:])
        if product_items.first():
            ProductItems.filter(ProductItems.digits == did[1:]).delete(synchronize_session='fetch')

    @classmethod
    def did_billing_rel_disconnect_client(cls, did, client_id, vendor_id, buy_billing_plan_id=None,
                                          user_name='unknown'):
        old = cls.filter(or_(cls.did == did, cls.did == did[1:])).filter(cls.end_date.is_(None)).all()
        for o in old:
            now = datetime.utcnow().replace(microsecond=0)
            n = cls(did=o.did,
                    sell_billing_plan_id=o.sell_billing_plan_id,
                    # buy_billing_plan_id = o.buy_billing_plan_id,
                    did_billing_id=o.did_billing_id,
                    ingress_res_id=o.ingress_res_id,
                    assigned_client_id=o.assigned_client_id,
                    start_date=now,
                    is_available=True
                    )
            cls.session().add(n)
            o.on_end_date()
            # o.end_date = now
            # o.is_available = False
            o.save()
        product_items = ProductItems.filter(ProductItems.digits == did[1:])
        if product_items.first():
            ProductItems.filter(ProductItems.digits == did[1:]).delete(synchronize_session='fetch')

    @classmethod
    def did_billing_rel_create(cls, num, client_id, vendor_id, buy_billing_plan_id=None, user_name='unknown'):
        # obj.created_time = datetime.now(UTC)
        if not len(num) in (10, 11):
            raise Exception('wrong did length {}, must be 10 or 11 numbers'.format(len(num)))
        if len(num) == 11:
            num1 = num
        if len(num) == 10:
            num1 = '1' + num
        log.debug(
            'did_billing_rel_create(did={},client_id={},'
            'vendor_id={},buy_billing_plan_id={},user_name={})'.format(num1, client_id, vendor_id, buy_billing_plan_id,
                                                                       user_name))
        vendor = Client.get(vendor_id)
        if not vendor:
            raise ValidationError('Bad did vendor {} '.format(client_id))
        vendor_api = DidVendor.get(vendor_id)
        log.debug('vendor api {}'.format(vendor_api))
        client = Client.get(client_id)
        if not client:
            raise ValidationError('Bad did client {} '.format(client_id))
        did = cls.filter(and_(cls.did == num, cls.end_date.is_(None),  # cls.start_date.isnot(None),
                              or_(cls.assigned_client_id.is_(None), cls.assigned_client_id == client_id),
                              cls.vendor_res_id == vendor.resource.resource_id)).first()
        if not did and num1 != num:
            did = cls.filter(and_(cls.did == num1, cls.end_date.is_(None),  # cls.start_date.isnot(None),
                                  or_(cls.assigned_client_id.is_(None), cls.assigned_client_id == client_id),
                                  cls.vendor_res_id == vendor.resource.resource_id)).first()

        ip = client.resource.host[0].ip if client and client.resource and client.resource.host else did.ip
        if did:
            obj = cls(did=did.did, vendor_res_id=did.vendor_res_id, client_res_id=did.client_res_id,
                      sell_billing_plan_id=did.sell_billing_plan_id, buy_billing_plan_id=did.buy_billing_plan_id,
                      is_sms=did.is_sms, ip=ip, assigned_client_id=client.client_id)
        else:
            obj = cls(did=num, vendor_res_id=vendor.resource.resource_id, client_res_id=client.resource.resource_id,
                      ip=ip)

        if not obj.sell_billing_plan_id:
            if vendor_api:
                obj.sell_billing_plan_id = vendor_api.client_billing_rule_id
            elif did:
                obj.sell_billing_plan_id = did.sell_billing_plan_id
                obj.did_billing_id = did.sell_billing_plan_id
        if not obj.buy_billing_plan_id:
            if vendor_api:
                obj.buy_billing_plan_id = vendor_api.vendor_billing_rule_id
            elif did:
                obj.buy_billing_plan_id = did.buy_billing_plan_id
        if not obj.sell_billing_plan_id:
            res = client.resource
        else:
            rule = DidBillingPlan.get(obj.sell_billing_plan_id)
            res = client.get_assigned_billing_plan_resource(rule, None)

        old = cls.filter(cls.did == obj.did).filter(cls.end_date.is_(None)).all()

        for o in old:
            if o.id == obj.id:
                continue
            o.on_end_date()
            o.end_date = datetime.utcnow().replace(microsecond=0)
            o.is_available = False
            obj.ingress_res_id = o.ingress_res_id
            obj.buy_billing_plan_id = o.buy_billing_plan_id
            obj.sell_billing_plan_id = o.sell_billing_plan_id
            #obj.assigned_client_id = o.assigned_client_id
            obj.ip = o.ip
            o.save()
            if o.egress_res_id is None:
                o.delete()
        if buy_billing_plan_id:
            obj.sell_billing_plan_id = buy_billing_plan_id
        client_rule = DidVendor.get_client_rule(client_id, num)
        obj.sell_billing_plan_id = client_rule.id if client_rule else obj.sell_billing_plan_id

        res.update_by = user_name
        ProductItems.assign_client_did(res, obj.did, user_name)
        obj.client_res = res
        obj.did_billing_id = obj.buy_billing_plan_id
        if obj.vendor_res_id:
            if obj.vendor_billing_rule_id:
                vendor_billing_rule = DidBillingPlan.get(obj.vendor_billing_rule_id)
                rate_table = vendor_billing_rule.rate_table
                vendor_res = Resource.get(obj.vendor_res_id)
                prefix = vendor_res.assign_vendor_did_prefix(obj.did, rate_table)
                prefix.save()
        obj.start_date = datetime.utcnow().date()
        obj.save()
        try:
            if obj.buy_billing_plan_id and obj.client_res_id:
                plan = DidBillingPlan.get(obj.buy_billing_plan_id)
                client_id = Resource.get(obj.client_res_id).client_id
                DidBillingOperationAction(did_billing_id=obj.id, created_by=user_name).save()
                # if plan.setup_fee:
                #     ClientBalanceOperationAction(client_id=client_id, balance=plan.setup_fee, action=3,
                #                                  create_by='did_setup_fee').save()
        except Exception as e:
            log.debug('did_billing_rel created, but deducting setupo fee errors: {}'.format(e))
        log.debug('did_billing_rel created {}'.format(obj))
        return obj

    def create_time(self):
        if self.rate:
            return self.rate.create_time
        else:
            return None

    def assign_time(self):
        if self.item:
            return self.item.update_at
        else:
            return None

    @hybrid_property
    def active(self):
        product_items = ProductItems.filter(or_(ProductItems.digits == self.did[1:],
                                                ProductItems.digits == self.did)).all()
        return bool(product_items)

    @active.setter
    def active(self, value):
        if not value:
            ProductItems.filter(or_(ProductItems.digits == self.did[1:],
                                    ProductItems.digits == self.did)).delete(synchronize_session='fetch')
        else:
            if not self.active:
                res = self.client_res if self.client_res else type('new_dict', (object,), {'resource_id': self.ingress_res_id})
                ProductItems.assign_client_did(res, self.did, None)

    @hybrid_property
    def _client_id(self):
        return self._client_id

    @_client_id.setter
    def _client_id(self, value):
        if self.client_id == value:
            return
        if value is None:
            self.egress_res_id = None
            return
        client = model.Client.get(value)
        if self.sell_billing_plan_id or self.client_billing_rule_id:
            rule = model.DidBillingPlan.get(self.sell_billing_plan_id or self.client_billing_rule_id)
        else:
            rule = None
        res = client.get_assigned_billing_plan_resource(rule, self.ip)
        self.client_res = res
        ProductItems.assign_client_did(res, self.did)

    @hybrid_property
    def actions(self):
        try:
            DIRECTION = {0: 'ALL', 1: 'ingress', 2: 'egress'}
            ACT = {'plus prefix': 'Add Prefix', 'minus prefix': 'Remove Prefix'}
            AF = {'plus prefix': 'digits', 'minus prefix': 'num_digits'}
            DIR = {'modify the caller': 'ani', 'modify the called': 'dnis'}
            TYPE = {0: 'modify the caller', 1: 'modify the called'}
            ret = []
            ac = None
            if self.client_res and self.client_res.directions:
                ac = {'ani': {}, 'dnis': {}}
                for a in self.client_res.directions:
                    if not a.action in ['plus prefix', 'minus prefix']:
                        continue
                    if not a.direction == 'egress':
                        continue
                    ac[DIR[a.type]]['type'] = ACT[a.action]
                    ac[DIR[a.type]][AF[a.action]] = a.digits

            if self.client_res and self.client_res.replace_actions:
                if not ac:
                    ac = {'ani': {}, 'dnis': {}}
                for a in self.client_res.replace_actions:
                    TYPE = {0: 'ani only replace the caller', 1: 'dnis only replace the called',
                            2: 'both replace the calling and called'}
                    if a.type in ['ani only replace the caller', 'both replace the calling and called']:
                        ac['ani']['type'] = 'Replace'
                        ac['ani']['new_number'] = a.ani
                        ac['ani']['digits'] = a.ani_prefix
                    if a.type in ['dnis only replace the called', 'both replace the calling and called']:
                        ac['dnis']['type'] = 'Replace'
                        ac['dnis']['new_number'] = a.dnis
                        ac['dnis']['digits'] = a.dnis_prefix
            return ac
        except Exception as e:
            log.warning('Cannot get actions for did repository {}'.format(str(e)))

    @actions.setter
    def actions(self, value):
        try:
            ACT = {'Add Prefix': 'plus prefix', 'Remove Prefix': 'minus prefix',
                   'Replace': 'both replace the calling and called'}
            for d in self.client_res.directions:
                d.delete()
            for d in self.client_res.replace_actions:
                d.delete()
            if value:

                a = value
                ani = ani_pref = ani_len = None
                dnis = dnis_pref = dnis_len = None
                digits = None
                for t in ['ani', 'dnis']:
                    if t in a:
                        ac = a[t]
                        if not 'num_digits' in ac:
                            ac['num_digits'] = 0
                        if not 'digits' in ac:
                            ac['digits'] = ''
                        num_digits = None
                        if 'type' in ac:
                            actp = ac['type']
                            if actp == 'Add Prefix' and 'digits' in ac:
                                digits = ac['digits']
                            if actp == 'Remove Prefix' and 'num_digits' in ac:
                                digits = ac['num_digits']
                            if actp == 'Replace' and t == 'ani' and 'new_number' in ac:
                                ani = ac['new_number']
                                ani_pref = ac['digits']
                                ani_len = ac['num_digits']
                            if actp == 'Replace' and t == 'dnis' and 'new_number' in ac:
                                dnis = ac['new_number']
                                dnis_pref = ac['digits']
                                dnis_len = ac['num_digits']
                        rd = 'modify the caller' if t == 'ani' else 'modify the called'
                        if digits:
                            self.client_res.directions.append(
                                ResourceDirection(action=ACT[actp], type=rd, digits=digits, direction='egress'))

                if ani or dnis:
                    tp = None
                    if ani:
                        tp = 'ani only replace the caller'
                    if dnis:
                        tp = 'dnis only replace the called'
                    if ani and dnis:
                        tp = 'both replace the calling and called'
                    if tp:
                        rep = ResourceReplaceAction(type=tp, ani_prefix=ani_pref, ani=ani, dnis_prefix=dnis_pref,
                                                    dnis=dnis,
                                                    ani_min_length=ani_len, ani_max_length=ani_len,
                                                    dnis_min_length=dnis_len, dnis_max_length=dnis_len)
                        self.client_res.replace_actions.append(rep)
        except Exception as e:
            log.warning('Cannot get actions for did repository {}'.format(str(e)))


DidBillingRel.client_id = column_property(select([Resource.client_id]). \
    where(Resource.resource_id == DidBillingRel.client_res_id).correlate_except(
    Resource))
DidBillingRel.did_client_res_name = column_property(select([Resource.alias]). \
                                                    where(
    Resource.resource_id == DidBillingRel.client_res_id).correlate_except(Resource))
DidBillingRel.did_client_name = column_property(select([Resource.client_name]). \
                                                where(
    Resource.resource_id == DidBillingRel.client_res_id).correlate_except(Resource))
DidBillingRel.did_client_is_online = column_property(select([Resource.client_is_online]). \
                                                     where(
    Resource.resource_id == DidBillingRel.client_res_id).correlate_except(Resource))
DidBillingRel.vendor_id = column_property(select([Resource.client_id]). \
    where(Resource.resource_id == DidBillingRel.vendor_res_id).correlate_except(
    Resource))
DidBillingRel.did_vendor_res_name = column_property(select([Resource.alias]). \
                                                    where(
    Resource.resource_id == DidBillingRel.vendor_res_id).correlate_except(Resource))
DidBillingRel.did_vendor_name = column_property(select([Resource.client_name]). \
                                                where(
    Resource.resource_id == DidBillingRel.vendor_res_id).correlate_except(Resource))
DidClientLog.did_vendor_name = column_property(select([Resource.client_name]). \
                                                where(
    Resource.resource_id == DidClientLog.ingress_res_id).correlate_except(Resource))


class DidRepository(DnlApiBaseModel):
    __tablename__ = 'did_repository'
    TOLL_FREE_DID_PREFIXES = ['1800', '1888', '1877', '1866', '1855', '1844', '1833']
    id = Column(BigInteger, nullable=False, primary_key=True)
    did = Column(String(50), index=True, nullable=False, unique=True)
    vendor_trunk_id = Column(Integer, nullable=False)
    vendor_billing_plan_id = Column(Integer, nullable=False)
    created_at = Column(DateTime(True), server_default=func.now())
    created_by = Column(Integer) # 0 - UI; 1 - API; 2 - Mass Import
    is_available = Column(Boolean, default=False)
    vendor_id = Column(ForeignKey('client.client_id', ondelete='RESTRICT'), nullable=False)

    status = synonym('is_available')
    active = synonym('is_available')
    # vendor_id = synonym('vendor_trunk_id')

    did_assignment = relationship('DidAssignments',
                            primaryjoin='foreign(DidAssignments.did) == DidRepository.did',
                            uselist=False)
    vendor_res = relationship('Resource',
                              primaryjoin='foreign(DidRepository.vendor_trunk_id) == Resource.resource_id',
                              uselist=False)


    vendor_billing_rule_name = column_property(
        select([DidBillingPlan.name]).where(DidBillingPlan.id == vendor_billing_plan_id).correlate_except(DidBillingPlan))
    did_vendor_name = column_property(select([Resource.alias]). \
        where(Resource.resource_id == vendor_trunk_id).correlate_except(Resource))
    vendor_trunk_name  = column_property(select([Resource.name]). \
        where(Resource.resource_id == vendor_trunk_id).correlate_except(Resource))

    mrc = column_property(
        select([DidBillingPlan.monthly_charge]).where(DidBillingPlan.id == vendor_billing_plan_id).correlate_except(
            DidBillingPlan))
    nrc = column_property(
        select([DidBillingPlan.did_price]).where(DidBillingPlan.id == vendor_billing_plan_id).correlate_except(
            DidBillingPlan))
    rate_per_min = column_property(
        select([DidBillingPlan.rate_per_min]).where(DidBillingPlan.id == vendor_billing_plan_id).correlate_except(DidBillingPlan))

    rate_table_id = column_property(
        select([DidBillingPlan.rate_table_id]).where(DidBillingPlan.id == vendor_billing_plan_id).correlate_except(
            DidBillingPlan))

    is_toll_free = column_property(func.substr(did, 1, 4).in_(TOLL_FREE_DID_PREFIXES))
    npa = column_property(case([(func.length(did) == 11, func.substr(did, 2, 3))], else_=func.substr(did, 1, 3)))
    is_assigned = column_property(is_available==False)

    country = column_property(select([DidParam.country_iso]).where(DidParam.did == did).correlate_except(DidParam))
    state = column_property(select([DidParam.state]).where(DidParam.did == did).correlate_except(DidParam))
    # ocn = column_property(select([DidParam.ocn]).where(DidParam.did == did).correlate_except(DidParam))
    lata = column_property(select([DidParam.lata]).where(DidParam.did == did).correlate_except(DidParam))
    # rate_type = column_property(select([DidParam.rate_type]).where(DidParam.did == did).correlate_except(DidParam))
    # company = column_property(select([DidParam.company]).where(DidParam.did == did).correlate_except(DidParam))
    # switch = column_property(select([DidParam.switch]).where(DidParam.did == did).correlate_except(DidParam))
    # did_type = column_property(select([DidParam.type]).where(DidParam.did == did).correlate_except(DidParam))

    # @hybrid_property
    # def active(self):
    #     product_items = ProductItems.filter(or_(ProductItems.digits == self.did[1:],
    #                                             ProductItems.digits == self.did)).all()
    #     return bool(product_items)

    # @active.setter
    # def active(self, value):
    #     if not value:
    #         ProductItems.filter(or_(ProductItems.digits == self.did[1:],
    #                                 ProductItems.digits == self.did)).delete(synchronize_session='fetch')
        # else:
        #     res = self.client_res if self.client_res else type('new_dict', (object,), {'resource_id': self.ingress_res_id})
        #     ProductItems.assign_client_did(res, self.did, None)

    @hybrid_property
    def actions(self):
        try:
            did_assignments = DidAssignments.filter(DidAssignments.did == self.did).first()
            client_res = Resource.get(did_assignments.client_trunk_id) if did_assignments else None
            DIRECTION = {0: 'ALL', 1: 'ingress', 2: 'egress'}
            ACT = {'plus prefix': 'Add Prefix', 'minus prefix': 'Remove Prefix'}
            AF = {'plus prefix': 'digits', 'minus prefix': 'num_digits'}
            DIR = {'modify the caller': 'ani', 'modify the called': 'dnis'}
            TYPE = {0: 'modify the caller', 1: 'modify the called'}
            ret = []
            ac = None
            if client_res and client_res.directions:
                ac = {'ani': {}, 'dnis': {}}
                for a in client_res.directions:
                    if not a.action in ['plus prefix', 'minus prefix']:
                        continue
                    if not a.direction == 'egress':
                        continue
                    ac[DIR[a.type]]['type'] = ACT[a.action]
                    ac[DIR[a.type]][AF[a.action]] = a.digits

            if client_res and client_res.replace_actions:
                if not ac:
                    ac = {'ani': {}, 'dnis': {}}
                for a in client_res.replace_actions:
                    TYPE = {0: 'ani only replace the caller', 1: 'dnis only replace the called',
                            2: 'both replace the calling and called'}
                    if a.type in ['ani only replace the caller', 'both replace the calling and called']:
                        ac['ani']['type'] = 'Replace'
                        ac['ani']['new_number'] = a.ani
                        ac['ani']['digits'] = a.ani_prefix
                    if a.type in ['dnis only replace the called', 'both replace the calling and called']:
                        ac['dnis']['type'] = 'Replace'
                        ac['dnis']['new_number'] = a.dnis
                        ac['dnis']['digits'] = a.dnis_prefix
            return ac
        except Exception as e:
            log.warning('Cannot get actions for did repository {}'.format(str(e)))

    @actions.setter
    def actions(self, value):
        try:
            ACT = {'Add Prefix': 'plus prefix', 'Remove Prefix': 'minus prefix',
                   'Replace': 'both replace the calling and called'}
            for d in self.did_assignment.client_res.directions:
                d.delete()
            for d in self.did_assignment.client_res.replace_actions:
                d.delete()
            if value:

                a = value
                ani = ani_pref = ani_len = None
                dnis = dnis_pref = dnis_len = None
                digits = None
                for t in ['ani', 'dnis']:
                    if t in a:
                        ac = a[t]
                        if not 'num_digits' in ac:
                            ac['num_digits'] = 0
                        if not 'digits' in ac:
                            ac['digits'] = ''
                        num_digits = None
                        if 'type' in ac:
                            actp = ac['type']
                            if actp == 'Add Prefix' and 'digits' in ac:
                                digits = ac['digits']
                            if actp == 'Remove Prefix' and 'num_digits' in ac:
                                digits = ac['num_digits']
                            if actp == 'Replace' and t == 'ani' and 'new_number' in ac:
                                ani = ac['new_number']
                                ani_pref = ac['digits']
                                ani_len = ac['num_digits']
                            if actp == 'Replace' and t == 'dnis' and 'new_number' in ac:
                                dnis = ac['new_number']
                                dnis_pref = ac['digits']
                                dnis_len = ac['num_digits']
                        rd = 'modify the caller' if t == 'ani' else 'modify the called'
                        if digits:
                            self.did_assignment.client_res.directions.append(
                                ResourceDirection(action=ACT[actp], type=rd, digits=digits, direction='egress'))

                if ani or dnis:
                    tp = None
                    if ani:
                        tp = 'ani only replace the caller'
                    if dnis:
                        tp = 'dnis only replace the called'
                    if ani and dnis:
                        tp = 'both replace the calling and called'
                    if tp:
                        rep = ResourceReplaceAction(type=tp, ani_prefix=ani_pref, ani=ani, dnis_prefix=dnis_pref,
                                                    dnis=dnis,
                                                    ani_min_length=ani_len, ani_max_length=ani_len,
                                                    dnis_min_length=dnis_len, dnis_max_length=dnis_len)
                        self.did_assignment.client_res.replace_actions.append(rep)
        except Exception as e:
            log.warning('Cannot get actions for did repository {}'.format(str(e)))


    @hybrid_property
    def actions(self):
        try:
            DIRECTION = {0: 'ALL', 1: 'ingress', 2: 'egress'}
            ACT = {'plus prefix': 'Add Prefix', 'minus prefix': 'Remove Prefix'}
            AF = {'plus prefix': 'digits', 'minus prefix': 'num_digits'}
            DIR = {'modify the caller': 'ani', 'modify the called': 'dnis'}
            TYPE = {0: 'modify the caller', 1: 'modify the called'}
            ret = []
            ac = None
            if self.did_assignment.client_res and self.did_assignment.client_res.directions:
                ac = {'ani': {}, 'dnis': {}}
                for a in self.client_res.directions:
                    if not a.action in ['plus prefix', 'minus prefix']:
                        continue
                    if not a.direction == 'egress':
                        continue
                    ac[DIR[a.type]]['type'] = ACT[a.action]
                    ac[DIR[a.type]][AF[a.action]] = a.digits

            if self.did_assignment.client_res and self.did_assignment.client_res.replace_actions:
                if not ac:
                    ac = {'ani': {}, 'dnis': {}}
                for a in self.did_assignment.client_res.replace_actions:
                    TYPE = {0: 'ani only replace the caller', 1: 'dnis only replace the called',
                            2: 'both replace the calling and called'}
                    if a.type in ['ani only replace the caller', 'both replace the calling and called']:
                        ac['ani']['type'] = 'Replace'
                        ac['ani']['new_number'] = a.ani
                        ac['ani']['digits'] = a.ani_prefix
                    if a.type in ['dnis only replace the called', 'both replace the calling and called']:
                        ac['dnis']['type'] = 'Replace'
                        ac['dnis']['new_number'] = a.dnis
                        ac['dnis']['digits'] = a.dnis_prefix
            return ac
        except Exception as e:
            log.warning('Cannot get actions for did repository {}'.format(str(e)))

    @actions.setter
    def actions(self, value):
        try:
            ACT = {'Add Prefix': 'plus prefix', 'Remove Prefix': 'minus prefix',
                   'Replace': 'both replace the calling and called'}
            for d in self.did_assignment.client_res.directions:
                d.delete()
            for d in self.did_assignment.client_res.replace_actions:
                d.delete()
            if value:

                a = value
                ani = ani_pref = ani_len = None
                dnis = dnis_pref = dnis_len = None
                digits = None
                for t in ['ani', 'dnis']:
                    if t in a:
                        ac = a[t]
                        if not 'num_digits' in ac:
                            ac['num_digits'] = 0
                        if not 'digits' in ac:
                            ac['digits'] = ''
                        num_digits = None
                        if 'type' in ac:
                            actp = ac['type']
                            if actp == 'Add Prefix' and 'digits' in ac:
                                digits = ac['digits']
                            if actp == 'Remove Prefix' and 'num_digits' in ac:
                                digits = ac['num_digits']
                            if actp == 'Replace' and t == 'ani' and 'new_number' in ac:
                                ani = ac['new_number']
                                ani_pref = ac['digits']
                                ani_len = ac['num_digits']
                            if actp == 'Replace' and t == 'dnis' and 'new_number' in ac:
                                dnis = ac['new_number']
                                dnis_pref = ac['digits']
                                dnis_len = ac['num_digits']
                        rd = 'modify the caller' if t == 'ani' else 'modify the called'
                        if digits:
                            self.did_assignment.client_res.directions.append(
                                ResourceDirection(action=ACT[actp], type=rd, digits=digits, direction='egress'))

                if ani or dnis:
                    tp = None
                    if ani:
                        tp = 'ani only replace the caller'
                    if dnis:
                        tp = 'dnis only replace the called'
                    if ani and dnis:
                        tp = 'both replace the calling and called'
                    if tp:
                        rep = ResourceReplaceAction(type=tp, ani_prefix=ani_pref, ani=ani, dnis_prefix=dnis_pref,
                                                    dnis=dnis,
                                                    ani_min_length=ani_len, ani_max_length=ani_len,
                                                    dnis_min_length=dnis_len, dnis_max_length=dnis_len)
                        self.did_assignment.client_res.replace_actions.append(rep)
        except Exception as e:
            log.warning('Cannot get actions for did repository {}'.format(str(e)))


class DidAssignments(DnlApiBaseModel):
    __tablename__ = 'did_assignments'
    id = Column(BigInteger, nullable=False, primary_key=True)
    did = Column(String(50), index=True, nullable=False, unique=True)
    vendor_trunk_id = Column(Integer, nullable=False)
    vendor_billing_plan_id = Column(Integer, nullable=False)
    client_trunk_id = Column(Integer, nullable=False)
    client_billing_plan_id = Column(Integer, nullable=False)
    fallback_id = Column(Integer)
    created_at = Column(DateTime(True), server_default=func.now())
    created_by = Column(Integer) # 0 - UI; 1 - API; 2 - Mass Import
    vendor_id = Column(ForeignKey('client.client_id', ondelete='RESTRICT'), nullable=False)
    vendor_mrc_cycle = Column(Integer)
    vendor_mrc = Column(Numeric(20, 5))
    vendor_nrc = Column(Numeric(20, 5))
    client_mrc_cycle = Column(Integer)
    client_mrc = Column(Numeric(20, 5))
    client_nrc = Column(Numeric(20, 5))
    # is_toll_free = column_property(func.substr(did, 1, 4).in_(TOLL_FREE_DID_PREFIXES))

    did_repository = relationship('DidRepository',
                            primaryjoin='DidAssignments.did == foreign(DidRepository.did)',
                            uselist=False)
    client_res = relationship('Resource',
                            primaryjoin='foreign(DidAssignments.client_trunk_id) == Resource.resource_id',
                            uselist=False)

    client_id = column_property(
        select([Resource.client_id]).where(Resource.resource_id == client_trunk_id).correlate_except(Resource))
    client_name = column_property(
        select([Resource.alias]).where(Resource.resource_id == client_trunk_id).correlate_except(Resource))
    client_billing_rule_name = column_property(
        select([DidBillingPlan.name]).where(DidBillingPlan.id == client_billing_plan_id).correlate_except(DidBillingPlan))
    vendor_trunk_name  = column_property(select([Resource.name]). \
        where(Resource.resource_id == vendor_trunk_id).correlate_except(Resource))
    # did_vendor_name = column_property(select([Resource.alias]). \
    #     where(Resource.resource_id == vendor_trunk_id).correlate_except(Resource))
    # vendor_billing_rule_name = column_property(select([DidBillingPlan.name]). \
    #     where(DidBillingPlan.id == vendor_billing_plan_id).correlate_except(DidBillingPlan))

DidRepository.client_billing_rule_name = column_property(select([DidAssignments.client_billing_rule_name])
                                             .where(DidAssignments.did == DidRepository.did)
                                             .correlate_except(DidAssignments))
DidRepository.client_id = column_property(select([DidAssignments.client_id])
                                             .where(DidAssignments.did == DidRepository.did)
                                             .correlate_except(DidAssignments))
# DidRepository.client_name = column_property(select([DidAssignments.client_name])
#                                              .where(DidAssignments.did == DidRepository.did)
#                                              .correlate_except(DidAssignments))
# DidRepository.did_client_name_is_null = column_property(DidRepository.client_name.is_(None))
DidRepository.did_client_name = column_property(select([DidAssignments.client_name])
                                             .where(DidAssignments.did == DidRepository.did)
                                             .correlate_except(DidAssignments))
DidRepository.client_trunk_id = column_property(select([DidAssignments.client_trunk_id]). \
        where(DidAssignments.did == DidRepository.did).correlate_except(DidAssignments))
DidRepository.client_billing_plan_id = column_property(select([DidAssignments.client_billing_plan_id]). \
        where(DidAssignments.did == DidRepository.did).correlate_except(DidAssignments))
# DidRepository.fallback_id = column_property(select([DidAssignments.fallback_id]). \
#         where(DidAssignments.did == DidRepository.did).correlate_except(DidAssignments))
# --- DidBillingRel


class DidCostDetail(DnlApiBaseModel):
    __tablename__ = 'did_cost_detail'
    id = Column(Integer, primary_key=True)
    params = Column(String(4096), index=True)
    created_on = Column(DateTime(True), server_default=func.now())
    did_number = Column(String(64), nullable=False, index=True)
    client_billing_plan_id = Column(ForeignKey('did_billing_plan.id', ondelete='CASCADE'))
    client_trunk_id = Column(ForeignKey('resource.resource_id', ondelete='CASCADE'))
    vendor_trunk_id = Column(ForeignKey('resource.resource_id', ondelete='CASCADE'))
    nrc = Column(Numeric(32, 6), nullable=False, server_default=text_('0.00'))
    mrc = Column(Numeric(32, 6), nullable=False, server_default=text_('0.00'))
    minutes = Column(Integer, server_default='0')
    attempts = Column(Integer, server_default='0')
    vendor_cost = Column(Numeric(15, 6), nullable=False, server_default=text_('0.00'))
    client_cost = Column(Numeric(15, 6), nullable=False, server_default=text_('0.00'))
    start_date = Column(Date)
    end_date = Column(Date)
    mrc_number = Column(Integer, server_default='0')
    pay_type = Column(Integer, server_default='0')
    did_mrc = column_property(mrc * mrc_number)

    did = synonym('did_number')
    egress_id = synonym('client_trunk_id')
    ingress_id = synonym('vendor_trunk_id')

    total = column_property((mrc * mrc_number).op('+')(nrc).op('+')(client_cost))
    egress_client_id = column_property(
        select([Resource.client_id]).where(
            and_(Resource.resource_id == client_trunk_id)).correlate_except(Resource))
    ingress_client_id = column_property(
        select([Resource.client_id]).where(
            and_(Resource.resource_id == vendor_trunk_id)).correlate_except(Resource))

    client_name = column_property(
        select([Client.name]).where(
            and_(Client.client_id == Resource.client_id, Resource.resource_id == client_trunk_id)).correlate_except(
            Client, Resource))
    vendor_name = column_property(
        select([Client.name]).where(
            and_(Client.client_id == Resource.client_id, Resource.resource_id == vendor_trunk_id)).correlate_except(
            Client, Resource))


class DidChargeDetail(DnlApiBaseModel):
    __tablename__ = 'did_charge_detail'
    id = Column(Integer, primary_key=True)
    params = Column(String(4096), index=True)
    created_on = Column(DateTime(True), server_default=func.now())
    did_number = Column(String(64), nullable=False, index=True)
    client_billing_plan_id = Column(ForeignKey('did_billing_plan.id', ondelete='CASCADE'))
    client_trunk_id = Column(ForeignKey('resource.resource_id', ondelete='CASCADE'))
    vendor_trunk_id = Column(ForeignKey('resource.resource_id', ondelete='CASCADE'))
    nrc = Column(Numeric(32, 6), nullable=False, server_default=text_('0.00'))
    mrc = Column(Numeric(32, 6), nullable=False, server_default=text_('0.00'))
    port_fee = Column(Numeric(32, 6), nullable=False, server_default=text_('0.00'))
    minutes = Column(Integer, server_default='0')
    attempts = Column(Integer, server_default='0')
    call_charge = Column(Numeric(32, 6), nullable=False, server_default=text_('0.00'))
    start_date = Column(Date)
    end_date = Column(Date)
    mrc_number = Column(Integer, server_default='0')
    pay_type = Column(Integer, server_default='0')
    did_mrc = column_property(mrc * mrc_number)
    did_port_fee = column_property(port_fee * mrc_number)

    did = synonym('did_number')
    egress_id = synonym('client_trunk_id')
    ingress_id = synonym('vendor_trunk_id')

    total = column_property((mrc * mrc_number).op('+')(nrc).op('+')(call_charge))
    egress_client_id = column_property(
        select([Resource.client_id]).where(
            and_(Resource.resource_id == client_trunk_id)).correlate_except(Resource))
    ingress_client_id = column_property(
        select([Resource.client_id]).where(
            and_(Resource.resource_id == vendor_trunk_id)).correlate_except(Resource))

    client_name = column_property(
        select([Client.name]).where(
            and_(Client.client_id == Resource.client_id, Resource.resource_id == client_trunk_id)).correlate_except(
            Client, Resource))
    vendor_name = column_property(
        select([Client.name]).where(
            and_(Client.client_id == Resource.client_id, Resource.resource_id == vendor_trunk_id)).correlate_except(
            Client, Resource))


class DidInvoiceChargeDetail(DnlApiBaseModel):
    __tablename__ = 'did_invoice_charge_detail'
    id = Column(Integer, primary_key=True)
    invoice_number = Column(ForeignKey('invoice_summary.invoice_number', ondelete='CASCADE'))
    did_number = Column(String(64), nullable=False, index=True)
    client_billing_plan_id = Column(ForeignKey('did_billing_plan.id', ondelete='CASCADE'))
    client_trunk_id = Column(ForeignKey('resource.resource_id', ondelete='CASCADE'))
    nrc = Column(Numeric(32, 6), nullable=False, server_default=text_('0.00'))
    mrc = Column(Numeric(32, 6), nullable=False, server_default=text_('0.00'))
    minutes = Column(Integer, server_default='0')
    attempts = Column(Integer, server_default='0')
    call_charge = Column(Numeric(32, 6), nullable=False, server_default=text_('0.00'))
    start_date = Column(Date)
    end_date = Column(Date)
    mrc_number = Column(Integer, server_default='0')
    pay_type = Column(Integer, server_default='0')
    did_mrc = column_property(mrc * mrc_number)
    total = column_property((mrc * mrc_number).op('+')(nrc).op('+')(call_charge))
    billing_plan_name = column_property(select([DidBillingPlan.name]).where(
        DidBillingPlan.id == client_billing_plan_id).correlate_except(DidBillingPlan))
    billing_plan_mrc_rate = column_property(select([DidBillingPlan.monthly_charge]).where(
        DidBillingPlan.id == client_billing_plan_id).correlate_except(DidBillingPlan))
    billing_plan_nrc_rate = column_property(select([DidBillingPlan.did_price]).where(
        DidBillingPlan.id == client_billing_plan_id).correlate_except(DidBillingPlan))


class DidInvoiceChargeBreakdown(DnlApiBaseModel):
    __tablename__ = 'did_invoice_charge_breakdown'
    id = Column(Integer, primary_key=True)
    invoice_number = Column(ForeignKey('invoice_summary.invoice_number', ondelete='CASCADE'))
    client_billing_plan_id = Column(ForeignKey('did_billing_plan.id', ondelete='CASCADE'))
    client_trunk_id = Column(ForeignKey('resource.resource_id', ondelete='CASCADE'))
    nrc = Column(Numeric(32, 6), nullable=False, server_default=text_('0.00'))
    mrc = Column(Numeric(32, 6), nullable=False, server_default=text_('0.00'))
    minutes = Column(Integer, server_default='0')
    attempts = Column(Integer, server_default='0')
    call_charge = Column(Numeric(32, 6), nullable=False, server_default=text_('0.00'))
    did_count = Column(Integer, server_default='0')


class DidInvoicePortCharge(DnlApiBaseModel):
    __tablename__ = 'did_invoice_port_charge'
    id = Column(Integer, primary_key=True)
    invoice_number = Column(ForeignKey('invoice_summary.invoice_number', ondelete='CASCADE'))
    client_trunk_id = Column(ForeignKey('resource.resource_id', ondelete='CASCADE'))
    billing_port_type = Column(ChoiceType(Resource.BILLING_PORT_TYPE), server_default='0')
    price_per_max_channel = Column(Numeric(32, 6), server_default=text_('0.00'))
    max_channels = Column(Integer, server_default='0')
    price_per_actual_channel = Column(Numeric(32, 6), server_default=text_('0.00'))
    did_count = Column(Integer, server_default='0')
    cost_per_port = Column(Numeric(32, 6), server_default=text_('0.00'))
    capacity = Column(Integer, server_default='0')
    dnis_cap_limit = Column(Integer, server_default='0')
    did_count = Column(Integer, server_default='0')
    channels = Column(Integer, server_default='0')
    port_charge = Column(Numeric(32, 6), server_default=text_('0.00'))


# +++ InvoiceCdrLog
# ---
class InvoiceHistory(DnlApiBaseModel):
    __tablename__ = 'invoice_history'

    id = Column(Integer, primary_key=True, server_default=text_("nextval('invoice_history_id_seq'::regclass)"))
    client_id = Column(Integer, nullable=False, index=True)
    last_invoice_for = Column(Date, index=True, server_default=text_("('now'::text)::date"))
    next_invoice_date = Column(Date)
    last_invoice_amount = Column(Numeric(20, 5))
    last_invoice_period = Column(String(32))
    client = relationship(Client, uselist=False, primaryjoin="foreign(InvoiceHistory.client_id)==Client.client_id",
                          back_populates='last_invoice_history')


Client._next_invoice_date = column_property(select([InvoiceHistory.next_invoice_date]). \
                                            where(InvoiceHistory.client_id == Client.client_id). \
                                            order_by(InvoiceHistory.next_invoice_date.desc()). \
                                            limit(1).correlate_except(InvoiceHistory))


class InvoiceDebug(DnlApiBaseModel):
    __tablename__ = 'invoice_debug'
    id = Column(Integer, primary_key=True)
    time = Column(DateTime(True), index=True, nullable=False, server_default=func.now())
    client_id = Column(Integer, index=True, nullable=True)
    text = Column(Text)


Invoice.last_invoice_for = column_property(select([InvoiceHistory.last_invoice_for]). \
                                           where(Invoice.client_id == InvoiceHistory.client_id).order_by(
    InvoiceHistory.last_invoice_for.desc()).limit(1).correlate_except(InvoiceHistory))
Client.last_invoice_date = column_property(select([Invoice.invoice_time]). \
                                             where(and_(Client.client_id == Invoice.client_id,
                                                        Invoice.create_type == 'auto')).order_by(
    Invoice.invoice_time.desc()).limit(1).correlate_except(Invoice))

Client.last_invoice_amount = column_property(select([Invoice.amount]). \
                                             where(Client.client_id == Invoice.client_id).order_by(
    Invoice.invoice_time.desc()).limit(1).correlate_except(Invoice))

Client.last_invoice_total_charge = column_property(select([Invoice.current_charge]). \
                                             where(Client.client_id == Invoice.client_id).order_by(
    Invoice.invoice_time.desc()).limit(1).correlate_except(Invoice))

Client.last_invoice_number = column_property(select([Invoice.invoice_number]). \
                                             where(Client.client_id == Invoice.client_id).order_by(
    Invoice.invoice_id.desc()).limit(1).correlate_except(Invoice))

Client.last_invoice_period = column_property(select([InvoiceHistory.last_invoice_period]). \
                                             where(Client.client_id == InvoiceHistory.client_id).order_by(
    InvoiceHistory.last_invoice_for.desc()).limit(1).correlate_except(InvoiceHistory))

Client.next_invoice_date = column_property(select([InvoiceHistory.next_invoice_date]). \
                                           where(Client.client_id == InvoiceHistory.client_id).order_by(
    InvoiceHistory.last_invoice_for.desc()).limit(1).correlate_except(InvoiceHistory))
# Client.assigned_did_count = column_property(select([func.count(DidRepository.id)]). \
#                                        where(DidRepository.client_id == Client.client_id). \
#                                        correlate_except(DidRepository)
#                                        )

class InvoiceTask(DnlApiBaseModel):
    __tablename__ = 'invoice_task'
    STATUS = {0: 'Initial', 1: 'In Process', 2: 'Successful', 3: 'Failed'}
    id = Column(Integer, primary_key=True)
    client_id = Column(Integer)
    is_auto = Column(Boolean, nullable=False, server_default=text("false"))
    created_by_user_id = Column(Integer)
    created_on = Column(DateTime(True), server_default=func.now())
    bill_period_start = Column(Date)
    bill_period_end = Column(Date)
    show_acc_summary = Column(Boolean, nullable=False, server_default=text("true"))
    show_summary_of_charges = Column(Boolean, nullable=True, default=False)
    show_payment_applied = Column(Boolean, nullable=False, server_default=text("false"))
    show_rec_charge = Column(Boolean, nullable=False, server_default=text("false"))
    show_non_rec_charge = Column(Boolean, nullable=False, server_default=text("false"))
    show_usage_charge = Column(Boolean, nullable=False, server_default=text("false"))
    show_trunk_detail = Column(Boolean, nullable=False, server_default=text("false"))
    show_daily_summary = Column(Boolean, nullable=False, server_default=text("false"))
    show_jd_daily_summary = Column(Boolean, nullable=False, server_default=text("false"))
    show_jd_trunk_detail = Column(Boolean, nullable=False, server_default=text("false"))
    show_prefix_summary = Column(Boolean, nullable=False, server_default=text("false"))
    show_code_name_summary = Column(Boolean, nullable=False, server_default=text("false"))
    show_country_summary = Column(Boolean, nullable=False, server_default=text("false"))
    show_inbound_summary = Column(Boolean, nullable=False, server_default=text("false"))
    show_top_100tn_summary = Column(Boolean, nullable=False, server_default=text("false"))

    show_did_charge_breakdown = Column(Boolean, nullable=False, server_default=text("false"))
    show_did_charge_summary = Column(Boolean, nullable=False, server_default=text("false"))

    enable_email = Column(Boolean, nullable=False, server_default=text("false"))
    enable_email_with_cdr = Column(Boolean, nullable=False, server_default=text("false"))
    # addons
    include_tax = Column(Boolean, nullable=False, server_default=text_("false"))
    include_scc_charges = Column(Boolean, nullable=False, server_default=text_("false"))
    payment_dur_date = Column(Date)
    invoice_zone = Column(String(20))
    # processing
    status = Column(ChoiceType(STATUS), nullable=False, server_default=text_("0"))
    progress = Column(String(1024))
    proc_start_time = Column(DateTime(True))
    proc_end_time = Column(DateTime(True))

    summary = relationship('InvoiceSummary', uselist=False, back_populates='task')

    client_name = column_property(
        select([Client.name]).where(Client.client_id == foreign(client_id)).correlate_except(Client))
    client_type = column_property(
        select([Client.client_type]).where(Client.client_id == foreign(client_id)).correlate_except(Client))

    show_scc_charges = synonym('include_scc_charges')

    @property
    def show_client_data(self):
        return True

    @classmethod
    def create_auto_invoice_task(cls, client_id, start_time, end_time, user_id=None):
        client = Client.get(client_id)
        if client:
            orig = client.is_orig
            obj = cls(client_id=client_id, bill_period_start=start_time, bill_period_end=end_time,
                      show_acc_summary=client.is_invoice_account_summary or orig,
                      show_payment_applied=client.invoice_include_payment or orig,
                      show_rec_charge=orig,
                      show_non_rec_charge=orig,
                      show_usage_charge=client.invoice_show_details,
                      show_trunk_detail=client.is_show_detail_trunk,
                      show_daily_summary=client.is_show_daily_usage,
                      show_jd_daily_summary=client.is_show_daily_usage and client.invoice_jurisdictional_detail,
                      show_jd_trunk_detail=client.invoice_jurisdictional_detail,
                      show_prefix_summary=client.is_show_total_trunk,
                      show_code_name_summary=client.is_show_code_name,
                      show_country_summary=client.is_show_country,
                      show_inbound_summary=client.is_show_by_date,
                      show_top_100tn_summary=client.is_show_code_100 or orig,
                      show_did_charge_breakdown=orig,
                      show_did_charge_summary=orig,
                      enable_email=client.email_invoice,
                      enable_email_with_cdr=client.is_link_cdr,
                      include_tax=client.include_tax,
                      include_scc_charges=client.is_short_duration_call_surcharge_detail,
                      payment_dur_date=client.payment_term.next_date(end_time),
                      invoice_zone=client.invoice_zone,

                      is_auto=True,
                      created_by_user_id=user_id
                      )
            obj.save()
            return obj
        else:
            return None

    def create_summary(self, invoice_number):
        from api_dnl.invoice_summary import InvoiceSummaryGenerator
        import hashlib
        s = self.bill_period_start
        e = self.bill_period_end
        tz = get_time_zone(self.invoice_zone)
        create_time = datetime.now(tz)
        obj = InvoiceSummaryGenerator(
            invoice_number=invoice_number,
            invoice_num=hashlib.md5(str(invoice_number).encode()).hexdigest(),
            client_id=self.client_id,
            invoice_start_time=datetime(s.year, s.month, s.day, 0, 0, 0),
            invoice_end_time=datetime(e.year, e.month, e.day, 23, 59, 59),
            task_id=self.id,
            invoice_type=0 if self.is_auto else 1,
            invoice_zone=self.invoice_zone,
            invoice_generation_time=create_time,
            create_time=create_time
        )

        obj.save()
        return obj

    def get_new_invoice_number(self):
        invoice_number = model.InvoiceSummary.query().session.query(
            func.max(cast(InvoiceSummary.invoice_number, Integer)).label('invoice_number')
        ).one().invoice_number
        invoice_number_2 = model.Invoice.query().session.query(
            func.max(cast(Invoice.invoice_number, Integer)).label('invoice_number')
        ).one().invoice_number
        if invoice_number and invoice_number_2 and int(invoice_number) < int(invoice_number_2):
            invoice_number = invoice_number_2
        return int(invoice_number) + 1 if invoice_number else 1

    def run(self):
        log.debug('start run InvocieTask. id: {}, client_id: {}'.format(self.id, self.client_id))
        self.status = 'In Process'
        self.proc_start_time = datetime.now(UTC)
        try:
            self.save()
            iter = 0
            while iter < 100:
                iter += 1
                try:
                    invoice_number = self.get_new_invoice_number()
                    obj = self.create_summary(invoice_number)
                    log.debug('summary created')
                    break
                except IntegrityError as e:
                    InvoiceDebug(client_id=self.client_id, text='invoice_number {} is already \
                                 taken generating a new one'.format(invoice_number)).save()
                    if isinstance(e.orig, UniqueViolation):
                        if iter < 100:
                            continue
                    raise e
            obj.update()
            obj.save()
            if self.is_auto and not obj.client.invoice_zero and (obj.current_charges == 0.0 or obj.total_amount == 0.0):
                self.progress = 'Summary Successful,Zero invoice not to generate'
                self.save()
                obj.delete()
                if settings.AUTO_INVOICE_DEBUG:
                    InvoiceDebug(client_id=self.client_id, text='Zero invoice not to generate').save()
                raise Exception('Zero invoice not to generate')
            else:
                self.progress = 'Summary Successful'
                self.save()
                obj.create_invoice()
                self.status = 'Successful'
                self.progress = 'Summary Successful,File successful'
                self.save()
        except Exception as e:
            InvoiceDebug(client_id=self.client_id, text='Error while generating invoice' + str(e)[:1000]).save()
            self.progress = 'error:' + str(e)[:1000]
            self.status = 'Failed'
            self.save()
            raise e
        finally:
            self.proc_end_time = datetime.now(UTC)
            self.save()
        log.debug('end run InvocieTask. id: {}, client_id: {}'.format(self.id, self.client_id))



class InvoiceSummary(DnlApiBaseModel):
    __tablename__ = 'invoice_summary'
    TYPE = {0: 'auto invoice', 1: 'manual invoice', 2: 'DID auto invoice', 3: 'DID manual invoice'}
    invoice_number = Column(Integer, primary_key=True,
                            server_default=text_("nextval('invoice_summary_invoice_number_seq'::regclass)"))
    invoice_num = Column(String(256), index=True)

    client_id = Column(Integer, nullable=False, index=True)
    invoice_generation_time = Column(DateTime, index=True)
    invoice_start_time = Column(DateTime)
    invoice_end_time = Column(DateTime)
    payment_dur_date = Column(Date)
    invoice_zone = Column(String(20))
    payment_id = Column(Integer)
    current_balance = Column(Numeric(32, 6), nullable=False, server_default=text_('0.00'))
    previous_balance = Column(Numeric(32, 6), nullable=False, server_default=text_('0.00'))
    payments_and_credits = Column(Numeric(32, 6), nullable=False, server_default=text_('0.00'))
    balance_forward = Column(Numeric(32, 6), nullable=False, server_default=text_('0.00'))
    long_distance_charges = Column(Numeric(32, 6), nullable=False, server_default=text_('0.00'))

    taxes = Column(Numeric(32, 6), nullable=False, server_default=text_('0.00'))
    scc_calls = Column(Numeric(32, 6), nullable=False, server_default=text_('0.00'))  # v5
    scc_amount = Column(Numeric(32, 6), nullable=False, server_default=text_('0.00'))

    current_charges = Column(Numeric(32, 6), nullable=False, server_default=text_('0.00'))
    total_calls = Column(Integer, nullable=False, server_default=text_("0"))
    total_minutes = Column(Numeric(32, 6), nullable=False, server_default=text_('0.00'))
    total_amount = Column(Numeric(32, 6), nullable=False, server_default=text_('0.00'))
    unlimited_credit = Column(Boolean, nullable=False, server_default=text_("false"))
    invoice_type = Column(ChoiceType(TYPE), nullable=False, server_default=text_("0"))
    did_mrc = Column(Numeric(32, 6), nullable=False, server_default=text_('0.00'))
    did_nrc = Column(Numeric(32, 6), nullable=False, server_default=text_('0.00'))
    create_time = Column(DateTime(True), server_default=text_("('now'::text)::timestamp(0) with time zone"))
    json_content = Column(MutableDict.as_mutable(JSON))  # Column(JSON)
    task_id = Column(ForeignKey('invoice_task.id', ondelete='CASCADE'))

    port_charges = Column(Numeric(32, 6), nullable=False, server_default=text_('0.00'))

    task = relationship(InvoiceTask, uselist=False, back_populates='summary')

    did_invoice_charge_detail = relationship(DidInvoiceChargeDetail,
                                             uselist=True, single_parent=True, cascade="all, delete-orphan")

    did_invoice_charge_breakdown = relationship(DidInvoiceChargeBreakdown,
                                                uselist=True, single_parent=True, cascade="all, delete-orphan")

    client = relationship(Client, primaryjoin=Client.client_id == foreign(client_id),
                          uselist=False)
    client_name = column_property(
        select([Client.name]).where(Client.client_id == foreign(client_id)).correlate_except(Client))
    company_name = column_property(
        select([Client.company]).where(Client.client_id == foreign(client_id)).correlate_except(Client))
    client_address = column_property(
        select([Client.address]).where(Client.client_id == foreign(client_id)).correlate_except(Client))
    client_invoice_zone = column_property(
        select([Client.invoice_zone]).where(Client.client_id == foreign(client_id)).correlate_except(Client))

    @property
    def include_tax(self):
        return self.task.include_tax

    @property
    def include_scc_charges(self):
        return self.task.include_scc_charges

    @property
    def include_mrc(self):
        return bool(self.did_mrc) or bool(self.did_nrc)

    @property
    def invoice_start_time_tz(self):
        return self.invoice_start_time.replace(tzinfo=get_time_zone(self.invoice_zone))

    @property
    def invoice_end_time_tz(self):
        return self.invoice_end_time.replace(tzinfo=get_time_zone(self.invoice_zone))

    @property
    def start_time(self):
        return int(self.invoice_start_time.timestamp())

    @property
    def end_time(self):
        return int(self.invoice_end_time.timestamp())

    @property
    def billing_period(self):
        return self.invoice_start_time.strftime("%Y-%m-%d") + ' ' + self.invoice_end_time.strftime("%Y-%m-%d")

    @property
    def config(self):
        if not hasattr(self, '_config_') or self._config_ is None:
            if self.client.invoice_setting and self.client.invoice_setting.setting:
                self._config_ = self.client.invoice_setting.setting
            else:
                self._config_ = SystemParameter.get(1)
        return self._config_

    @property
    def pdf_tpl(self):
        return self.config.pdf_tpl

    @property
    def tpl_number(self):
        return self.config.tpl_number

    @property
    def billing_info_position(self):
        return self.config.billing_info_position

    @property
    def logo(self):
        import os, base64
        path = self.config._logo_path or SystemParameter.get(1)._logo_path
        if path:
            if os.path.exists(path):
                return path
        path = settings.FILES['upload_to'] + '/logo.png'
        if not os.path.exists(path):
            s = b'iVBORw0KGgoAAAANSUhEUgAAAAEAAAABCAYAAAAfFcSJAAAADUlEQVR42mP8z8BQDwAEhQGAhKmMIQAAAABJRU5ErkJggg=='
            with open(path, 'wb') as f:
                f.write(base64.b64decode(s))
        return path

    @property
    def logo_url(self):
        return self.logo.replace(settings.FILES['upload_to'], settings.API_URL)

    @property
    def company_info(self):
        return self.config.company_info or ''

    @property
    def company_info_location(self):
        return self.config.company_info_location

    @property
    def billing_info(self):
        return self.config.billing_info or ''

    @property
    def billing_info_location(self):
        return self.config.billing_info_location

    @property
    def filename(self):
        extension = "pdf"
        if self.client.invoice_format == 'HTML':
            extension = "htm"
        elif self.client.invoice_format == 'Excel':
            extension = "pdf"
        return '{}_{}_{}.{}'.format(self.invoice_number, str(self.client_name),
                                     datetime.now().strftime("%Y%m%d%H%M"), extension)

    def create_invoice(self):
        from api_dnl.invoice_pdf import InvoicePdf
        from api_dnl.invoice_summary import InvoiceSummaryGenerator
        filename = self.filename
        log.warning('payment_dur_date set to {}'.format(str(self.payment_dur_date)))
        try:
            # if self.invoice_type in ('auto invoice', 'manual invoice'):
            old = Invoice.filter(Invoice.invoice_number == str(self.invoice_number)).all()
            if len(old):
                # log.warning('Delete old invoice {}'.format(str(self.invoice_number)))
                # old.delete()
                log.warning('Void old invoices {}'.format(str(self.invoice_number)))
                for o in old:
                    # o.state='void'
                    # o.save()
                    o.delete()
            invoice_type = 0 if self.invoice_type == 'auto invoice' else 1
            gen = InvoiceSummaryGenerator.get(self.invoice_number)
            log.debug('create invoice test')
            pdf = InvoicePdf(self, settings.FILES['invoices'] + '/' + filename)
            log.debug('create invoice test')
            pdf.generate()
            obj = Invoice(
                invoice_number=str(self.invoice_number),
                client_id=self.client_id,
                create_type=invoice_type,
                invoice_start=self.invoice_start_time,
                invoice_end=self.invoice_end_time,
                invoice_time=self.create_time
            )
            obj.due_date = self.payment_dur_date,
            obj.current_balance = self.current_balance,
            obj.previous_balance = self.previous_balance,
            obj.payment_credit = self.payments_and_credits,
            obj.current_charge = self.current_charges,
            obj.total_amount = self.total_amount,
            obj.decimals_num = self.config.invoice_decimal_digits,
            obj.pdf_path = filename,
            obj.company_type = 0 if self.client.company_type == 'termination' else 1,  # todo this is hack why obj.company_type not converted by ChoiceType?
            obj.tax = self.taxes,
            obj.scc_calls = self.scc_calls,
            obj.scc_cost = self.scc_amount,
            obj.invoice_zone = self.client.invoice_zone,
            obj.is_invoice_account_summary = self.task.show_acc_summary,
            obj.is_send_as_link = self.config.invoice_send_mode == 'link',
            obj.is_short_duration_call_surcharge_detail = self.task.include_scc_charges,
            obj.is_show_by_date = self.task.show_daily_summary,
            obj.is_show_code_100 = self.task.show_top_100tn_summary,
            obj.is_show_code_name = self.task.show_code_name_summary,
            obj.is_show_country = self.task.show_country_summary,
            obj.is_show_daily_usage = self.task.show_daily_summary,
            obj.is_show_detail_trunk = self.task.show_jd_trunk_detail,
            obj.is_show_total_trunk = self.task.show_trunk_detail,
            obj.non_recurring_charge = self.did_nrc,
            # obj.pay_amount = self.json_content['acc_summary']['usage_charges'] if 'acc_summary' in self.json_content and 'usage_charges' in self.json_content['acc_summary'] else 0,
            # obj.payment_term = self.client.payment_term_name,
            obj.status = 2,  # 'done',
            # obj.type = 0,  # 'sent(out--buy)'
            # if obj.pay_amount >= obj.current_charge:
                # obj.paid = True
            invoice_id = obj.save()
            log.info('create invoice success {}'.format(invoice_id))
        except Exception as e:
            log.error('create invoice error {}'.format(e))


InvoiceTask.invoice_number = column_property(
    select([InvoiceSummary.invoice_number]).where(InvoiceTask.id == InvoiceSummary.task_id).correlate_except(
        InvoiceSummary))

InvoiceTask.invoice_num = column_property(
    select([InvoiceSummary.invoice_num]).where(InvoiceTask.id == InvoiceSummary.task_id).correlate_except(
        InvoiceSummary))

InvoiceTask.invoice_id = column_property(
    select([Invoice.invoice_id]).where(cast(InvoiceTask.invoice_number, String) == Invoice.invoice_number). \
        correlate_except(Invoice))

InvoiceTask.invoice_amount = column_property(
    select([Invoice.amount]).where(cast(InvoiceTask.invoice_number, String) == Invoice.invoice_number). \
        correlate_except(Invoice))


# endregion ---InvoiceSummary---

class Sms(DnlApiBaseModel):
    __tablename__ = 'sms'
    TYPE = {0: 'sent', 1: 'received'}
    id = Column(Integer, primary_key=True)
    client_id = Column(Integer)
    created_on = Column(DateTime(True), server_default=func.now())
    sender = Column(String(128), index=True)
    receiver = Column(String(128), index=True)
    message = Column(String(256), index=True)
    cost = Column(Numeric(32, 6), server_default='0.0')
    direction = Column(ChoiceType(TYPE), server_default='1')
    reference_id = Column(String(128), index=True)
    stat = Column(String(128), index=True)
    delivery_error = Column(String(128))
    vendor_id = Column(Integer)
    vendor_cost = Column(Numeric(32, 6), server_default='0.0')
    delivered_on = Column(DateTime(True))
    client_name = column_property(
        select([Client.name]).where(Client.client_id == foreign(client_id)).correlate_except(Client))
    vendor_name = column_property(
        select([Client.name]).where(Client.client_id == foreign(vendor_id)).correlate_except(Client))
    sender_plus = column_property(
        case([(func.length(sender) == 10, func.concat('+1', sender))], else_=func.concat('+', sender)))
    receiver_plus = column_property(
        case([(func.length(receiver) == 10, func.concat('+1', receiver))], else_=func.concat('+', receiver)))
    dlr_hook = relationship('SmsDlrHook', uselist=False, single_parent=True, cascade="all, delete-orphan")

    def send_dlr(self, **params):
        if self.dlr_hook and self.dlr_hook.dlr_url:
            self.dlr_hook.send_dlr(**params)

    def send_fail_dlr(self, **params):
        if self.dlr_hook and self.dlr_hook.dlr_fail_url:
            self.dlr_hook.send_fail_dlr(**params)


class SmsDlrHook(DnlApiBaseModel):
    __tablename__ = 'sms_dlr_hook'
    id = Column(Integer, ForeignKey('sms.id', ondelete='CASCADE'), primary_key=True)
    dlr_url = Column(String(1024))
    dlr_fail_url = Column(String(1024))

    def send_dlr(self, **params):
        try:
            headers = {'CONTENT-TYPE': 'application/json',
                       'USER-AGENT': '{}/{}'.format(settings.API_TITLE, __version__)}
            res = requests.post(self.dlr_url, data=json.dumps(params), headers=headers)
            log.debug('SmsDlrHook.send_dlr status:{} result:{}'.format(res.status_code, res.text))
        except Exception as e:
            log.debug('SmsDlrHook.send_dlr:{}'.format(e))

    def send_fail_dlr(self, **params):
        try:
            headers = {'CONTENT-TYPE': 'application/json',
                       'USER-AGENT': '{}/{}'.format(settings.API_TITLE, __version__)}
            res = requests.post(self.dlr_fail_url, data=json.dumps(params), headers=headers)
            log.debug('SmsDlrHook.send_fail_dlr status:{} result:{}'.format(res.status_code, res.text))
        except Exception as e:
            log.debug('SmsDlrHook.send_fail_dlr:{}'.format(e))


class WebHook(DnlApiBaseModel):
    __tablename__ = 'webhook'
    __table_args__ = (
        UniqueConstraint('client_id', 'number'),
    )
    id = Column(Integer, primary_key=True)
    client_id = Column(Integer, ForeignKey('client.client_id', ondelete='CASCADE'))
    number = Column(String(128), nullable=False)
    url = Column(String(1024), nullable=False)
    header_vars = Column(JSON)
    created_on = Column(DateTime(True), server_default=func.now())


class EgressErrorString(DnlApiBaseModel):
    __tablename__ = 'egress_error_string'
    id = Column(Integer, primary_key=True)
    name = Column(String(64))


class ReleaseCause(DnlApiBaseModel):
    __tablename__ = 'release_cause_string'
    id = Column(Integer, primary_key=True)
    name = Column(String(64))
    err = Column(Integer)
    err_string = Column(String(64))


class CleanDataManagement(DnlApiBaseModel):
    __tablename__ = 'data_clean_management'
    id = Column(Integer, primary_key=True)
    tbl_name = Column(String(100), nullable=False, unique=True)
    days_of_save_in_db = Column(Integer, nullable=False, server_default='0')
    create_time = Column(DateTime(True), server_default=func.now())


class DiskManagement(DnlApiBaseModel):
    __tablename__ = 'disk_management'
    id = Column(Integer, primary_key=True)
    days_of_cdr_keep = Column(Integer, nullable=False, server_default='365')
    days_of_pcap_keep = Column(Integer, nullable=False, server_default='365')
    create_time = Column(DateTime(True), server_default=func.now())


class RateTypeName(DnlApiBaseModel):
    __tablename__ = 'rate_type_name'
    id = Column(Integer, primary_key=True)
    name = Column(String(100), nullable=False, unique=True)

    @classmethod
    def init(cls):
        if not cls.query().first():
            sess = cls.session()
            sess.add(cls(id=1, name='interstate'))
            sess.add(cls(id=2, name='intrastate'))
            sess.add(cls(id=3, name='others'))
            sess.add(cls(id=4, name='error'))
            sess.add(cls(id=5, name='local rate'))
            sess.commit()


class DnisTypeName(DnlApiBaseModel):
    __tablename__ = 'dnis_type_name'
    id = Column(Integer, primary_key=True)
    name = Column(String(100), nullable=False, unique=True)

    @classmethod
    def init(cls):
        if not cls.query().first():
            sess = cls.session()
            sess.add(cls(id=0, name='dnis'))
            sess.add(cls(id=1, name='lrn'))
            sess.add(cls(id=2, name='lrn block'))
            sess.commit()


class JurTypeName(DnlApiBaseModel):
    __tablename__ = 'jur_type_name'
    id = Column(Integer, primary_key=True)
    name = Column(String(100), nullable=False, unique=True)

    @classmethod
    def init(cls):
        if not cls.query().first():
            sess = cls.session()
            sess.add(cls(id=0, name='A-Z'))
            sess.add(cls(id=1, name='US Non-JD'))
            sess.add(cls(id=2, name='US JD'))
            sess.add(cls(id=3, name='OCN-LATA-JD'))
            sess.add(cls(id=4, name='OCN-LATA-NON-JD'))
            sess.commit()


class CdrExportEmailTask(DnlApiBaseModel):
    __tablename__ = 'cdr_export_email_task'
    EMAIL_STATUS = {0: 'Initial', 1: 'In Process', 2: 'Successful', 3: 'Failed'}
    id = Column(Integer, ForeignKey('cdr_export_task.id', ondelete='CASCADE'), primary_key=True)
    email_status = Column(ChoiceType(EMAIL_STATUS), server_default='0')
    started_on = Column(DateTime(True))
    finished_on = Column(DateTime(True))
    email_error = Column(Text)
    template = relationship('SendCdrExportTemplate', uselist=False, single_parent=True, cascade="all, delete-orphan")
    export = relationship('CdrExportTask', uselist=False, back_populates='email_task')


MonitoredRuleHistory.client = relationship(Client,
                                           secondary=Resource.__table__,
                                           primaryjoin=foreign(MonitoredRuleHistory.trunk_id) == Resource.resource_id,
                                           secondaryjoin=Client.client_id == foreign(Resource.client_id)
                                           )


class CdrExportTask(DnlApiBaseModel):
    __tablename__ = 'cdr_export_task'
    STATUS = {0: 'Initial', 1: 'In Process', 2: 'Successful', 3: 'Failed'}
    id = Column(Integer, primary_key=True)
    operator_user = Column(String(40))
    export_cdr_file = Column(String(1024))
    cdr_counts = Column(Integer, server_default='0')
    cdr_start_time = Column(DateTime(True))
    cdr_end_time = Column(DateTime(True))
    csv_file_headers = Column(String(2048))
    cdr_headers = Column(String(2048))
    cdr_filter = Column(String(2048))
    status = Column(ChoiceType(STATUS), server_default='0')
    progress = Column(String(1024))
    start_time = Column(DateTime(True))
    end_time = Column(DateTime(True))
    create_time = Column(DateTime(True), server_default=func.now())
    email_status = column_property(
        select([CdrExportEmailTask.email_status]).where(CdrExportEmailTask.id == id).correlate_except(
            CdrExportEmailTask))
    email_finished_on = column_property(
        select([CdrExportEmailTask.finished_on]).where(CdrExportEmailTask.id == id).correlate_except(
            CdrExportEmailTask))
    email_error = column_property(
        select([CdrExportEmailTask.email_error]).where(CdrExportEmailTask.id == id).correlate_except(
            CdrExportEmailTask))

    email_task = relationship('CdrExportEmailTask', uselist=False, single_parent=True, cascade="all, delete-orphan",
                              back_populates='export')

    @property
    def file_name(self):
        return self.export_cdr_file

    @property
    def file_name_xls(self):
        return '{}/cdr_{}.xls'.format(settings.FILES['upload_to'], self.id)


class ReportTemplate(DnlApiBaseModel):
    __tablename__ = 'report_template'
    PERIOD = {0: '15 min', 2: 'last hour', 3: 'last day', 4: 'yesterday', 5: 'last week', 6: 'prev week',
              7: 'last month', 8: 'prev month', 9: 'last year'}
    id = Column(Integer, primary_key=True)
    created_by = Column(String(40))
    created_on = Column(DateTime(True), server_default=func.now())
    template_name = Column(String(100))
    period = Column(ChoiceType(PERIOD), server_default='0')
    params = Column(JSON())

    def get_period_times(self, period):
        now = datetime.now(UTC)
        now0 = now.replace(hour=0, minute=0, second=0, microsecond=0)
        now_week = now0 - timedelta(days=now0.weekday())
        now_month = now0.replace(day=1)
        now_year = now0.replace(month=1, day=1)
        prev_month = add_months(now_month, -1)
        d = {
            '15 min': (now - timedelta(minutes=15), now),
            'last hour': (now - timedelta(minutes=15), now),
            'last day': (now - timedelta(hours=24), now),
            'yesterday': (now0 - timedelta(hours=24), now0),
            'last week': (now_week, now0),
            'prev week': (now_week - timedelta(days=7), now0 - timedelta(days=7)),
            'last month': (now_month, now0),
            'prev month': (prev_month, now_month - timedelta(seconds=1)),
            'last year': (now_year, now0),
        }
        d0, d1 = d[period]
        return int(d0.timestamp()), int(d1.timestamp())

    @property
    def report_url(self):
        params = {}
        # t0, t1 = self.get_period_times(self.period)
        # params['start_time'] = t0
        # params['end_time'] = t1
        params.update(self.params)
        par = urlencode(params)
        return '{}/report?{}'.format(settings.API_URL, par)


class DidDisconnectTask(DnlApiBaseModel):
    __tablename__ = 'did_disconnect_task'
    id = Column(Integer, primary_key=True)
    operator_user = Column(String(40))
    data = Column(JSON)
    release_on_time = Column(DateTime(True))
    started_at = Column(DateTime(True))
    finished_at = Column(DateTime(True))
    reply = Column(JSON)
    success = Column(Boolean)
    success_count = Column(Integer)
    error_count = Column(Integer)


class SmsRateTable(DnlApiBaseModel):
    __tablename__ = 'sms_rate_table'
    id = Column(Integer, primary_key=True)
    name = Column(String(100), nullable=False, unique=True)
    created_on = Column(DateTime(True), onupdate=func.now())
    created_by = Column(String)


class SmsRate(DnlApiBaseModel):
    __tablename__ = 'sms_rate'
    id = Column(Integer, primary_key=True)
    sms_rate_table_id = Column(ForeignKey('sms_rate_table.id', ondelete='CASCADE'), index=True)
    code = Column(PrefixRange, nullable=True, server_default=text_("''::prefix_range"), index=True)
    country = Column(String(1000), index=True)
    effective_date = Column(DateTime(True), default=datetime.now(UTC), index=True)
    end_date = Column(DateTime(True), index=True)
    rate = Column(Numeric(32, 6), server_default='0.0')


class DidProduct(DnlApiBaseModel):
    __tablename__ = 'did_product'
    id = Column(Integer, primary_key=True)
    name = Column(String(40), nullable=False, unique=True)
    created_by = Column(String(40))
    created_on = Column(DateTime(True), server_default=func.now())
    items = relationship('DidProductItem', uselist=True, single_parent=True, cascade="all, delete-orphan")

    @validates('name')
    def validate_name(self, field, value):
        old_obj = DidProduct.filter(and_(DidProduct.name == value,
                                                     DidProduct.id != self.id)).first()
        if old_obj:
            raise Exception('name is duplicate')

        return value


class DidProductItem(DnlApiBaseModel):
    __tablename__ = 'did_product_item'
    TYPE = {0: 'Local', 1: 'Short Co', 2: 'Toll-Free'}
    id = Column(Integer, primary_key=True)
    did_product_id = Column(ForeignKey('did_product.id', ondelete='CASCADE'))
    country = Column(ForeignKey('code_country.country_code', ondelete='CASCADE'))
    type = Column(ChoiceType(TYPE), server_default='0')
    billing_rule_id = Column(ForeignKey('did_billing_plan.id', ondelete='CASCADE'))
    created_by = Column(String(40))
    created_on = Column(DateTime(True), server_default=func.now())
    product_name = column_property(
        select([DidProduct.name]).where(DidProduct.id == did_product_id).correlate_except(DidProduct))
    billing_rule_name = column_property(
        select([DidBillingPlan.name]).where(DidBillingPlan.id == billing_rule_id).correlate_except(DidBillingPlan))
    mrc = column_property(
        select([DidBillingPlan.monthly_charge]).where(DidBillingPlan.id == billing_rule_id).correlate_except(
            DidBillingPlan))
    nrc = column_property(
        select([DidBillingPlan.did_price]).where(DidBillingPlan.id == billing_rule_id).correlate_except(DidBillingPlan))
    rate_per_min = column_property(
        select([DidBillingPlan.rate_per_min]).where(DidBillingPlan.id == billing_rule_id).correlate_except(DidBillingPlan))

DidProduct.billing_rule_id = column_property(
        select([DidProductItem.billing_rule_id]).where(DidProductItem.did_product_id == DidProduct.id). \
        limit(1).correlate_except(DidProductItem))


class ClientDidProduct(DnlApiBaseModel):
    __tablename__ = 'client_did_product'
    client_id = Column(ForeignKey('client.client_id', ondelete='CASCADE'), primary_key=True)
    did_product_id = Column(ForeignKey('did_product.id', ondelete='CASCADE'))
    rate_table_id = Column(ForeignKey('sms_rate_table.id', ondelete='CASCADE'))
    enable_sms = Column(Boolean, server_default='false')
    created_by = Column(String(40))
    created_on = Column(DateTime(True), server_default=func.now())
    client_name = column_property(
        select([Client.name]).where(Client.client_id == foreign(client_id)).correlate_except(Client))
    product_name = column_property(
        select([DidProduct.name]).where(DidProduct.id == did_product_id).correlate_except(DidProduct))
    rate_table_name = column_property(
        select([SmsRateTable.name]).where(SmsRateTable.id == rate_table_id).correlate_except(SmsRateTable))
    country = column_property(
        select([DidProductItem.country]).where(DidProductItem.did_product_id == did_product_id). \
        limit(1).correlate_except(DidProductItem))
    type = column_property(
        select([DidProductItem.type]).where(DidProductItem.did_product_id == did_product_id). \
        limit(1).correlate_except(DidProductItem))
    billing_rule_id = column_property(
        select([DidProductItem.billing_rule_id]).where(DidProductItem.did_product_id == did_product_id). \
        limit(1).correlate_except(DidProductItem))
    billing_rule_name = column_property(
        select([DidProductItem.billing_rule_name]).where(DidProductItem.did_product_id == did_product_id). \
        limit(1).correlate_except(DidProductItem))
    mrc = column_property(
        select([DidProductItem.mrc]).where(DidProductItem.did_product_id == did_product_id). \
        limit(1).correlate_except(DidProductItem))
    nrc = column_property(
        select([DidProductItem.nrc]).where(DidProductItem.did_product_id == did_product_id). \
        limit(1).correlate_except(DidProductItem))
    rate_per_min = column_property(
        select([DidProductItem.rate_per_min]).where(DidProductItem.did_product_id == did_product_id). \
        limit(1).correlate_except(DidProductItem))
    client = relationship(Client, uselist=False, back_populates='did_product', )

    @hybrid_property
    def items(self):
        return DidProductItem.filter(DidProductItem.did_product_id == self.did_product_id).all()

    @hybrid_property
    def rate_table_ids(self):
        product_items = DidProductItem.filter(DidProductItem.did_product_id == self.did_product_id).all()
        billig_plan_ids = [product_item.billing_rule_id for product_item in product_items]
        return [billing_plan.rate_table_id for billing_plan in DidBillingPlan.filter(DidBillingPlan.id.in_(billig_plan_ids)).all()]

    @hybrid_property
    def billing_rule_ids(self):
        product_items = DidProductItem.filter(DidProductItem.did_product_id == self.did_product_id).all()
        return [product_item.billing_rule_id for product_item in product_items]


DidProduct.client_id = column_property(select([ClientDidProduct.client_id]). \
                                       where(ClientDidProduct.did_product_id == DidProduct.id).limit(
    1).correlate_except(ClientDidProduct))
DidProductItem.client_id = column_property(select([ClientDidProduct.client_id]). \
                                           where(
    ClientDidProduct.did_product_id == DidProductItem.did_product_id).limit(
    1).correlate_except(ClientDidProduct))
DidProductItem.usage_count = column_property(
    select([func.coalesce(func.count(ClientDidProduct.client_id), text_("0"))]). \
        where(ClientDidProduct.did_product_id == DidProductItem.did_product_id).correlate_except(
        ClientDidProduct).label(
        'usage_count'))
DidProduct.used_by = column_property(
    select([func.coalesce(func.count(ClientDidProduct.client_id), text_("0"))]). \
        where(ClientDidProduct.did_product_id == DidProduct.id).correlate_except(
        ClientDidProduct).label('used_by'))
Client.did_product_id = column_property(select([ClientDidProduct.did_product_id]). \
                                           where(
    ClientDidProduct.client_id == Client.client_id).limit(
    1).correlate_except(ClientDidProduct))

# t_qos_client = Table(
#     'qos_client', DnlApiBaseModel.metadata,
#     Column('report_time', DateTime(True), nullable=False,
#            server_default=text("('now'::text)::timestamp(0) with time zone")),
#     Column('client_id', Integer),
#     Column('call', Integer),
#     Column('inbound_cps', Integer),
#     Column('inbound_chan', Integer),
#     Column('outbound_cps', Integer),
#     Column('outbound_chan', Integer),
#     Column('server_ip', String(50)),
#     Column('server_port', Integer)
# )


class ShakenAniGroup(DnlApiBaseModel):
    __tablename__ = 'shaken_ani_group'
    id = Column(Integer, primary_key=True)
    name = Column(Text)
    created_on = Column(DateTime(True), server_default=func.now())
    created_by = Column(String(100))


class ShakenAniGroupRel(DnlApiBaseModel):
    __tablename__ = 'shaken_ani_group_rel'
    group_id = Column(ForeignKey('shaken_ani_group.id', ondelete='CASCADE'), primary_key=True, nullable=False)
    did = Column(String(16), primary_key=True, nullable=False)
    created_on = Column(DateTime(True), server_default=func.now())
    created_by = Column(String(100), index=True)


ShakenAniGroup.count = column_property(select([func.coalesce(func.count(ShakenAniGroupRel.did), text_("0"))]). \
    where(ShakenAniGroupRel.group_id == ShakenAniGroup.id).correlate_except(ShakenAniGroupRel).label(
    'count'))


class ShakenAniGroupList(DnlApiBaseModel):
    __tablename__ = 'shaken_ani_group_list'
    id = Column(Integer, primary_key=True)
    name = Column(Text)
    created_on = Column(DateTime(True), server_default=func.now())
    created_by = Column(String(100), index=True)


class ShakenAniGroupListRel(DnlApiBaseModel):
    __tablename__ = 'shaken_ani_group_list_rel'
    __table_args__ = (
        CheckConstraint("(attest_lvl = 'A' OR attest_lvl = 'B' OR attest_lvl = 'C')"),
    )
    id = Column(Integer, primary_key=True)
    ani_group_list_id = Column(ForeignKey('shaken_ani_group_list.id', ondelete='CASCADE'),
                               nullable=False)
    ani_group_id = Column(ForeignKey('shaken_ani_group.id', ondelete='CASCADE'), nullable=False)
    attest_lvl = Column(String(1), nullable=False)
    created_on = Column(DateTime(True), server_default=func.now())
    created_by = Column(String(100), index=True)

    ani_group_name = column_property(
        select([ShakenAniGroup.name]).where(ShakenAniGroup.id == ani_group_id).correlate_except(ShakenAniGroup))
    count = column_property(
        select([ShakenAniGroup.count]).where(ShakenAniGroup.id == ani_group_id).correlate_except(ShakenAniGroup))


ShakenAniGroupList.count = column_property(select([
    func.count(ShakenAniGroupListRel.ani_group_id)]).where(
    ShakenAniGroupList.id == ShakenAniGroupListRel.ani_group_list_id).correlate_except(ShakenAniGroupListRel))


class C4ShakenConf(DnlApiBaseModel):
    __tablename__ = 'c4_shaken_conf'
    TYPE = {0: 'Call signing', 1: 'Call verification'}
    id = Column(Integer, primary_key=True)
    name = Column(Text)
    is_primary = Column(Boolean, nullable=False, server_default='false')
    conn_type = Column(String(3))
    hostaddr = Column(Text, nullable=False)
    port = Column(Integer, nullable=False)
    heartbeat = Column(Boolean, nullable=False, server_default='false')
    heartbeat_interval = Column(Integer)
    type = Column(ChoiceType(TYPE), nullable = False, server_default='0')


class C4ShakenStatus(DnlApiBaseModel):
    __tablename__ = 'c4_shaken_status'
    id = Column(BigInteger, nullable=False, primary_key=True)
    name = Column(Text)
    upd_time = Column(DateTime(True), server_default=func.now())
    is_primary = Column(Boolean, nullable=False, server_default='false')
    is_online = Column(Boolean, nullable=False, server_default='false')
    error = Column(Text)


class ShakenAniGroupRelImportTask(DnlApiBaseModel):
    __tablename__ = 'shaken_ani_group_rel_import_task'
    STATUS = {0: 'Initial', 1: 'In Process', 2: 'Copy To DB', 3: 'Finished', 4: 'Error'}
    OP_METHOD = {1: 'Import', 2: 'Number Range', 3: 'Paste'}
    REPEATED_ACTION = {0: 'Add', 1: 'Delete'}

    id = Column(Integer, primary_key=True)
    operator_user = Column(String(40))
    upload_file_path = Column(String(256))
    upload_orig_file = Column(String(100))
    upload_format_file = Column(String(100))
    orig_name = Column(String(256))
    repeated_action = Column(ChoiceType(REPEATED_ACTION), server_default='0')
    status = Column(ChoiceType(STATUS), nullable=False, server_default='0')
    progress = Column(Text)
    create_time = Column(DateTime(True), server_default=func.now())
    start_time = Column(DateTime(True))
    end_time = Column(DateTime(True))
    op_method = Column(ChoiceType(OP_METHOD), server_default='1')
    group_id = Column(Integer)
    first_number = Column(BigInteger)
    last_number = Column(BigInteger)

    @property
    def warnings(self):
        if 'IMPORT WARNINGS:\n' in self.progress:
            return self.progress.split('IMPORT WARNINGS:\n')[1]
        return ''

    @property
    def total(self):
        try:
            if 'numbers imported' in self.progress and ' of ' in self.progress:
                return self.progress.split('numbers imported')[1].split(' ')[3][:-1]
            return self.success
        except:
            return 0

    @property
    def success(self):
        try:
            if 'numbers imported' in self.progress:
                return self.progress.split('numbers imported')[1].split(' ')[1]
            return 0
        except:
            return 0

    @property
    def fail(self):
        try:
            if 'numbers skipped' in self.progress:
                return self.progress.split('numbers skipped')[1].split('\n')[0].split(',')[0][1:]
            return 0
        except:
            return 0

    @property
    def duplicate(self):
        try:
            if self.progress:
                return self.progress.count(' exists')
            return 0
        except:
            return 0

    @property
    def url(self):
        return '{}/home/client/did/assign/{}.{}'.format(settings.API_URL, self.uuid, self.ext)

    @property
    def file_name(self):
        if self.upload_format_file:
            return '{}/{}'.format(settings.FILES['upload_to'], self.upload_format_file)
        return None

    @property
    def ext(self):
        if self.upload_format_file and len(self.upload_format_file.split('.')) > 1:
            return self.upload_format_file.split('.')[-1]
        return ''

    @property
    def data(self):
        if self.file:
            return open(self.file_name, 'wb').read()


# region +++CodeDeckImportTask+++
class CodeDeckImportTask(DnlApiBaseModel):
    __tablename__ = 'code_deck_import_task'

    REDUP_IN_TABLE_ACTION = {0: 'reject entire import', 1: 'skip the record', 2: 'overwrite existing record'}
    REDUP_IN_FILE_ACTION = {0: 'reject entire import', 1: 'skip the record', 2: 'overwrite previous record'}
    STATUS = {0: 'Initial', 1: 'In Process', 2: 'Copy To DB', 7: 'Finished', 8: 'Error'}
    OP_METHOD = {0: 'Upload', 1: 'Delete', 2: 'Release'}
    id = Column(Integer, nullable=False, primary_key=True)
    operator_user = Column(String(40))
    orig_import_filename = Column(String(256))
    format_import_filename = Column(String(256))
    import_log_filename = Column(String(256))
    redup_in_code_deck_table_action = Column(ChoiceType(REDUP_IN_TABLE_ACTION), nullable=False, server_default='0')
    redup_in_code_deck_table_action = Column(ChoiceType(REDUP_IN_TABLE_ACTION), nullable=False, server_default='0')
    redup_in_file_action = Column(ChoiceType(REDUP_IN_FILE_ACTION), nullable=False, server_default='0')
    delete_all = Column(Boolean, nullable=False, server_default='false')
    code_deck_id = Column(Integer, nullable=False)
    status = Column(ChoiceType(STATUS), nullable=False, server_default='0')
    progress = Column(String(1024))
    start_time = Column(DateTime(True))
    end_time = Column(DateTime(True))
    create_time = Column(DateTime(True), server_default=func.now())
    d_all = column_property(case([(delete_all, 'Yes')], else_='No'))


# endregion

class ResourceCidBlockConfig(DnlApiBaseModel):
    __tablename__ = 'resource_cid_block_config'
    resource_id = Column(ForeignKey('resource.resource_id', ondelete='CASCADE'), primary_key=True)
    min_asr = Column(Integer)
    min_acd = Column(Integer)
    max_sdp = Column(Integer)
    max_cpm = Column(Integer)

    resource = relationship(Resource, uselist=False)
    trunk_name = column_property(
        select([Resource.alias]).where(Resource.resource_id == resource_id).correlate_except(Resource))


class ResourceCidBlockLog(DnlApiBaseModel):
    __tablename__ = 'resource_cid_block_log'
    id = Column(Integer, nullable=False, primary_key=True)
    time = Column(DateTime(True), server_default=func.now())
    result = Column(Text())
    data = Column(JSON)
    resource_id = Column(Integer)
    trunk_name = column_property(
        select([Resource.alias]).where(Resource.resource_id == foreign(resource_id)).correlate_except(Resource))


class DnlDailyCdrCloudCfg(DnlApiBaseModel):
    __tablename__ = 'dnl_daily_cdr_cloud_cfg'
    param_name = Column(Text, nullable=False, primary_key=True)
    param_value = Column(Text, nullable=False)
    json_key = Column(Text)
    bucket = Column(Text)
    workdir = Column(Text)
    non_zero_cdr_only = Column(Boolean)


class MediaAsrFilter(DnlApiBaseModel):
    __tablename__ = 'media_asr_filter'
    id = Column(Integer, nullable=False, primary_key=True)
    ip_address = Column(String(32))
    number = Column(String(32))
    enabled = Column(Boolean, nullable=False, server_default='true')
    comment = Column(String(128))
    add_dt = Column(DateTime(True), server_default=func.now())
    num_filter_type = Column(Integer, nullable=False, server_default='1')


class MediaCaptureFilter(DnlApiBaseModel):
    __tablename__ = 'media_capture_filter'
    id = Column(Integer, nullable=False, primary_key=True)
    ip_address = Column(String(32))
    number = Column(String(32))
    enabled = Column(Boolean, nullable=False, server_default='true')
    comment = Column(String(128))
    add_dt = Column(DateTime(True), server_default=func.now())
    num_filter_type = Column(Integer, nullable=False, server_default='1')


class SignalCaptureFilter(DnlApiBaseModel):
    __tablename__ = 'signal_capture_filter'
    id = Column(Integer, nullable=False, primary_key=True)
    ip_address = Column(String(32))
    number = Column(String(32))
    enabled = Column(Boolean, nullable=False, server_default='true')
    comment = Column(String(128))
    add_dt = Column(DateTime(True), server_default=func.now())
    num_filter_type = Column(Integer, nullable=False, server_default='1')


class RtpProxyLog(DnlApiBaseModel):
    __tablename__ = 'rtp_proxy_log'
    id = Column(Integer, nullable=False, primary_key=True)
    session_uuid = Column(Text, nullable=False)
    source_addr = Column(String(16), nullable=False)
    source_port = Column(Integer, nullable=False)
    dest_addr = Column(String(16), nullable=False)
    dest_port = Column(Integer, nullable=False)
    timeout = Column(Integer, nullable=False)
    start_time = Column(DateTime(True), nullable=False)
    end_time = Column(DateTime(True), nullable=False)
    is_killed = Column(Boolean, nullable=False)
    last_caller_packet = Column(DateTime(True))
    caller_packets = Column(BigInteger, nullable=False)
    caller_bytes = Column(BigInteger, nullable=False)
    last_callee_packet = Column(DateTime(True))
    callee_packets = Column(BigInteger, nullable=False)
    callee_bytes = Column(BigInteger, nullable=False)


class OcnBlocklist(DnlApiBaseModel):
    __tablename__ = 'ocn_blocklist'
    id = Column(BigInteger, nullable=False, primary_key=True)
    res_id = Column(Integer)
    ocn = Column(String(100), nullable=False)
    created_at = Column(DateTime(True), default=datetime.now(UTC))
    created_by = Column(String(100))
