from urllib.parse import urlencode
import json
import requests
import jwt
from falcon_rest.logger import log

from api_dnl.settings import SMSC,JWT_SIGNATURE
from .did_api import DidAPI, TIMEOUT, DidApiException, fmt_content

SMSC_BASE = SMSC['base']

class DidAPISmsc(DidAPI):
    is_cache_token = True
    _token = None
    _token_expiration_time = None

    def __init__(self, user, password=None, vendor=None):
        self.user = user
        self.password = password
        self.vendor = vendor

    def authenticate(self, **kwargs):
        headers = {'Content-type': 'application/json'}
        url = '{}/{}'.format(SMSC_BASE, 'auth')
        data = json.dumps(dict(user_name=self.user, password=self.password))
        response = requests.post(url, data=data, headers=headers,timeout=TIMEOUT, verify=False)
        if not response.status_code in (200, 201, 202, 204):
            log.debug('DidAPISmsc response {} {}'.format(response.status_code, response.content))
            raise DidApiException(
                fmt_content(response.content, url))
        self._token = response.json()['payload']['token']

    def post(self, name, **kwargs):
        url = '{}/{}'.format(SMSC_BASE, name)
        # kwargs['token'] = self._token
        # data = json.dumps(kwargs)
        data = json.dumps(kwargs)
        # url='{}?{}'.format(url,data)
        try:
            headers = {'Content-type': 'application/json','X-Auth-Token': self._token}
            log.debug('DidAPISmsc POST {} -H {} -d {}'.format(url, headers, kwargs))
            response = requests.post(url, data=data, timeout=TIMEOUT, headers=headers, verify=False)
            if not response.status_code in (200, 201, 202, 204):
                log.debug('DidAPISmsc response {} {}'.format(response.status_code, response.content))
                raise DidApiException(
                    fmt_content(response.content, url))

        except requests.ConnectionError as e:
            raise DidApiException(dict(code=410, msg='ConnectionError',
                                       data='configured auth host {}'.format(url)))
        else:
            if response.status_code not in (200, 201, 202, 204):
                raise DidApiException(fmt_content(response.content, url))
        result = response.json()
        return result['payload']

    def get(self, name, **kwargs):
        url = '{}/{}'.format(SMSC_BASE, name)
        #kwargs['token'] = self._token
        # data = json.dumps(kwargs)
        data = urlencode(kwargs)
        url = '{}?{}'.format(url, data)
        try:
            headers = {'Content-type': 'application/json','X-Auth-Token': self._token}
            log.debug('DidAPISmsc GET {}  -H {} -d {}'.format(url, headers, kwargs))
            response = requests.get(url, timeout=TIMEOUT, headers=headers, verify=False)
            if not response.status_code in (200, 201, 202, 204):
                log.debug('DidAPISmsc response {} {}'.format(response.status_code, response.content))
                raise DidApiException(
                    fmt_content(response.content, url))

        except requests.ConnectionError as e:
            raise DidApiException(dict(code=410, msg='ConnectionError',
                                       data='configured auth host {}'.format(url)))
        else:
            if response.status_code not in (200, 201, 202, 204):
                raise DidApiException(fmt_content(response.content, url))
            result = response.json()
            if result['success'] == 'false':
                raise DidApiException(fmt_content(response.content, url))
            return result['payload']

    def lata(self, number):
        # item=cls.filter(cls.prefix=='1'+number[0:6]).first()
        # if item:
        #     return str(item.lata)
        return None

    def search_local(self, state=None, npa=None, lata=None, pattern=None, count=100, client_id=None, nxx=None):
        import re
        params = {}
        if state:
            params['state'] = state
        if npa:
            params['npa'] = npa
        if pattern:
            pattern = pattern.replace('x', '[0-9]').replace('*', '[0-9]')
            params['pattern'] = pattern
        if lata:
            params['lata'] = lata
        result = self.get('did/search', **params)
        log.debug('search local:{}'.format(str(result)[0:4096]))
        items = [dict()]
        return result['items']


    def coverage_local(self, country=None, state=None, npa=None, lata=None,group=None):
        params = {}
        if state:
            params['state'] = state
        if npa:
            params['npa'] = npa
        result = self.get('dids/list', **params)
        log.debug('search local:{}'.format(str(result)[0:4096]))
        ds = {}
        if 'data' in result and 'dids' in result['data']:
            dids = result['data']['dids']
            i = 0
            ret = []
            for item in dids:
                xi = {'number': item['number'], 'npa': item['npa'], 'lata': self.lata(item['number']),
                      'state': item['state']}
                number = item['number']
                if lata and not str(lata) == xi['lata']:
                    continue
                key = (xi['state'], xi['npa'], xi['lata'])
                if not key in ds:
                    ds[key] = 0
                ds[key] += 1
            return [{'state': k[0], 'npa': k[1], 'lata': k[2], 'count': v} for k, v in ds.items()]
        return []

    def search_order_local(self, country=None, state=None, npa=None, lata=None, count=100, client_id=None):
        ret = self.search_local(country,state, npa, lata, None, count, None)
        err = []
        succ = []
        for item in ret:
            try:
                res = self.get('dids/order', number=item['number'], cnam=str(client_id) if client_id else None)
                item.update(res['data'])
                item['status'] = 'success'
            except DidApiException as e:
                item['status'] = 'error'
                item['error'] = str(e)
            succ.append(item)
        return succ

    def order_local(self, items, client_id=None):
        succ = []
        for item in items:
            try:
                res = self.post('did/order', items=items)
                item.update(res['data'])
                item['status'] = 'success'
            except DidApiException as e:
                item['status'] = 'error'
                item['error'] = str(e)
            succ = res['items']
        return succ

    def assigned_local(self):
        params = {'number_type': 'local'}
        ret = self.get('did/list', **params)
        return ret['data']

    def disconnect_local(self, items):
        ret = []
        res = self.post('/did/disconnect', items=items)
        return res['items']

    def pending(self):
        return []

    def send_sms(self, sender, receiver, msg):
        if len(sender) == 10:
            sender = '1' + sender
        if len(receiver) == 10:
            receiver = '1' + receiver
        params = {'ani': sender, 'dnis': receiver, 'text': msg}
        res = self.post('sms', **params)
        return res

    def set_sms_receive_hook(self, number, url, client_id):

        token = jwt.encode(dict(client_id=client_id), JWT_SIGNATURE).decode('utf-8')
        params = {'authorizations': [
            {'number': number, 'header_vars': {'X-Auth-Token',  token} , 'url': url}
        ]}
        res = self.post('did/webhook', **params)
        return res
