import time
from pytz import UTC
import requests
from urllib.parse import urlencode,unquote, quote_plus
from falcon_rest.logger import log
#from api_dnl.settings import STATISTIC
import json


class StatisticException(Exception):
    pass

def fmt_content(cont,data=None):
    if type(cont)==type(b''):
        cont = cont.decode('utf-8')
    if cont and cont[0] == '{':
        ret=json.loads(cont)
        if not ret.get('data',None) and data:
            ret['data']=data
        return ret
    return dict(code=411,msg='Unknown error',data=cont)

TIMEOUT = (60,120)
FIELDS={0:'time',1:'release_cause',2:'start_time_of_date',3:'answer_time_of_date',4:'release_tod',5:'minutes_west_of_greenwich_mean_time',6:'release_cause_from_protocol_stack',7:'binary_value_of_release_cause_from_protocol_stack',8:'first_release_dialogue',9:'origination_source_number',10:'origination_source_host_name',11:'origination_destination_number',12:'origination_destination_host_name',13:'origination_call_id',14:'origination_remote_payload_ip_address',15:'origination_remote_payload_udp_address',16:'origination_local_payload_ip_address',17:'origination_local_payload_udp_address',18:'origination_codec_list',19:'termination_source_number',20:'termination_source_host_name',21:'termination_destination_number',22:'termination_destination_host_name',23:'termination_call_id',24:'termination_remote_payload_ip_address',25:'termination_remote_payload_udp_address',26:'termination_local_payload_ip_address',27:'termination_local_payload_udp_address',28:'termination_codec_list',29:'final_route_indication',30:'routing_digits',31:'call_duration',32:'pdd',33:'ring_time',34:'callduration_in_ms',35:'conf_id',36:'call_type',37:'ingress_id',38:'ingress_client_id',39:'ingress_client_rate_table_id',40:'ingress_client_currency_id',41:'ingress_client_rate',42:'ingress_client_currency',43:'ingress_client_bill_time',44:'ingress_client_bill_result',45:'ingress_client_cost',46:'egress_id',47:'egress_rate_table_id',48:'egress_rate',49:'egress_cost',50:'egress_bill_time',51:'egress_client_id',52:'egress_client_currency_id',53:'egress_client_currency',
        #54:'egress_six_seconds',
        55:'egress_bill_minutes',56:'egress_bill_result',57:'ingress_bill_minutes',58:'ingress_dnis_type',59:'ingress_rate_type',60:'lrn_dnis',61:'egress_dnis_type',62:'egress_rate_type',63:'item_id',64:'translation_ani',65:'ingress_rate_id',66:'egress_rate_id',67:'orig_code',68:'orig_code_name',69:'orig_country',70:'term_code',71:'term_code_name',72:'term_country',73:'ingress_rate_effective_date',74:'egress_rate_effective_date',75:'egress_erro_string',76:'order_id',77:'order_type',78:'is_final_call',79:'egress_code_asr',80:'egress_code_acd',81:'static_route',82:'dynamic_route',83:'route_plan',84:'route_prefix',85:'orig_delay_second',86:'term_delay_second',87:'orig_call_duration',88:'trunk_type',89:'origination_profile_port',90:'termination_profile_port',91:'o_trunk_type2',92:'o_billing_method',93:'t_trunk_type2',94:'t_billing_method',95:'campaign_id',96:'agent_id',97:'agent_rate',98:'agent_cost',99:'orig_jur_type',100:'term_jur_type',101:'ring_epoch',102:'end_epoch',103:'paid_user',104:'rpid_user',105:'timeout_type',106:'q850_cause'
    ,107:'q850_cause_string'
        }
FIELD_MAP=dict((v, k) for k, v in FIELDS.items())
UIFIELDS={0:'Time', 1:'Release Cause', 2:'Start Time',3:'Answer Time',4:'End Time',5:'Timezone',6:'Response to Egress',7:'Response from Ingress',8:'Orig/Term Release',9:'Orig Src Number',10:'Orig IP',11:'Orig DST Number',12:'orig_switch_ip',13:'Orig Call ID',14:'Orig Media IP',15:'Orig Media Port',16:'',17:'Origination Local Payload Port',18:'Orig Codecs',19:'Term Src Number',20:'Class4 IP (Egress)',21:'Term DST Number',22:'Term IP',23:'Term Call ID',24:'Term Media IP',25:'Term Media Port',26:'Term Media IP',27:'Term Media Port',28:'Term Codecs',29:'final_route',30:'Translational DNIS',31:'Call Duration',32:'PDD',33:'Ring Time',34:'Call Duration in MS',35:'',36:'Call Type',37:'Ingress ID',38:'Ingress Client ID',39:'Ingress Rate Table ID',40:'Ingress Client Currency ID',41:'Ingress Client Rate',42:'Ingress Client Currency',43:'Ingress Bill Time',44:'Ingress Bill Result',45:'Ingress Client Cost',46:'Egress ID',47:'Egress Rate table ID',48:'Egress Rate',49:'Egress Cost',50:'Egress Bill Time',51:'Egress Carrier ID',52:'Egress Currency ID',53:'Egress Currency',
          #54:'Egress Six Second',
          55:'Egress Bill Minutes',56:'Egress Bill Result',57:'Ingress Bill Minutes',58:'Ingress DNIS type',59:'Egress Rate type',60:'LRN Number',61:'Egress DNIS Type',62:'Egress Rate Type',63:'',64:'Translational ANI',65:'Ingress Rate ID',66:'Egress Rate ID',67:'Orig Code',68:'Orig Code Name',69:'Orig Country',70:'Term Code',71:'Term Code Name',72:'Term Country',73:'Ingress Rate Effective Date',74:'Egress Rate Effective Date',75:'Egress Trunk Trace',76:'',77:'',78:'',79:'Egress Code ASR',80:'Egress Code ACD',81:'Start Route Name',82:'Dynamic Route Name',83:'Routing Plan',84:'Routing Prefix',85:'',86:'',87:'Orig Call Duration',88:'Trunk Type',89:'Orig Profile Port',90:'Term Profile Port',91:'',92:'',93:'',94:'T Billing method',95:'Campaign ID',96:'Agent ID',97:'Agent Rate',98:'Agent Cost',99:'Orig Jurisdiction type',100:'Term Jurisdiction Type',101:'Ring Start time',102:'End Time',103:'PAID user',104:'Rpid user',105:'Timeout type',106:'Q850 Media Q850'
    ,107:'Q850 Media CAUSE'
          }

def cdr_fields(fields):
    if type(fields)== type(''):
        fields=fields.split(',')
    ret=[]
    for f in fields:
        if f in FIELD_MAP:
            ret.append(str(FIELD_MAP[f]))
        else:
            try:
                if int(f) in FIELDS:
                    ret.append(f)
            except:
                log.error('bad field {}'.format(f))
                pass
    return ','.join(ret)

def cdr_names(fields):
    if type(fields)== type(''):
        fields=fields.split(',')
    ret=[]
    for f in fields:
        try:
          if int(f) in FIELDS:
            ret.append(FIELDS[int(f)])
        except:
            log.error('bad field {}'.format(f))
            pass
    return ','.join(ret)

class StatisticAPI(object):
    is_cache_token = True

    __token = None
    __token_expiration_time = None

    def __init__(self, is_cache_token=True):
        self.is_cache_token = is_cache_token

    @classmethod
    def set_token(cls,token,exp):
        cls.__token = token
        cls.__token_expiration_time = exp

    @staticmethod
    def __get_base_url(rt_api=None):
        url = 'http://{host}:{port}/'
        if rt_api is None or rt_api=='report':
            pass
        elif rt_api=='cdr':
            if not 'cdr_port' in STATISTIC:
                raise StatisticException(dict(code=410, msg='ConfigurationError',
                                              data='configure cdr sync port'))
            url = 'http://{cdr_host}:{cdr_port}/'
        elif rt_api=='ftp':
            if not 'ftp_port' in STATISTIC:
                raise StatisticException(dict(code=410, msg='ConfigurationError',
                                              data='configure ftp  port'))
            url = 'http://{ftp_host}:{ftp_port}/'
        elif rt_api == 'async':
            if not 'async_port' in STATISTIC:
                raise StatisticException(dict(code=410, msg='ConfigurationError',
                                              data='configure async port'))
            url = 'http://{async_host}:{async_port}/'
        elif rt_api == 'qos':
            if not 'qos_port' in STATISTIC:
                raise StatisticException(dict(code=410, msg='ConfigurationError',
                                              data='configure qos port'))
            url = 'http://{qos_host}:{qos_port}/'
        else:
            raise StatisticException(dict(code=411,msg='Bad use of rt api',data='wrong rtapi type {}'.format(rt_api)))
        return url.format(**STATISTIC)

    @staticmethod
    def check_token(token):
        STATISTIC['token'] = token
        try:
            response = requests.post('http://{auth_host}:{auth_port}/{token}'.format(
                **STATISTIC), timeout=(2, 4))
            if response.status_code != 200:
                return False
        except requests.ConnectionError as e:
            raise StatisticException(dict(code=410,msg='ConnectionError',data='configured auth host {} port {}'.format(STATISTIC['auth_host'],STATISTIC['auth_port'])))
        return True

    @staticmethod
    def authenticate(**kwargs):
        return None,None

        #if user:
        #    q_user='user={}'.format(user)
        #    q_user=q_user.encode()
        #else:
        #    q_user = None
        for k in list(kwargs.keys()):
            if kwargs[k] is None:
                del kwargs[k]
        q_user = urlencode(kwargs)
        if not 'auth_host' in  STATISTIC:
            raise StatisticException(dict(code=410, msg='ConfigurationError',
                                      data='configure api auth host'))
        if not 'auth_port' in  STATISTIC:
            raise StatisticException(dict(code=410, msg='ConfigurationError',
                                      data='configure auth port'))
        try:
            headers = {'Content-Type': 'application/x-www-form-urlencoded'}
            response = requests.post('http://{auth_host}:{auth_port}'.format(
                **STATISTIC),data=q_user,timeout=(2,4),headers=headers)
            if not response.status_code in (200,201,202,204):
                raise StatisticException(fmt_content(response.content,data='http://{auth_host}:{auth_port}'.format(**STATISTIC)))
    
        except requests.ConnectionError as e:
            raise StatisticException(dict(code=410, msg='ConnectionError',
                                          data='configured auth host {} port {}'.format(STATISTIC['auth_host'],
                                                                                   STATISTIC['auth_port'])))
        else:
            result = response.json()
            STATISTIC['token']=result['token']
            try:
                response = requests.post('http://{auth_host}:{auth_port}/{token}'.format(
                    **STATISTIC),timeout=TIMEOUT)
                if response.status_code != 200:
                    raise StatisticException(fmt_content(response.content))
            except requests.ConnectionError as e:
                raise StatisticException(dict(code=410, msg='ConnectionError',
                                              data='configured auth host {} port {}'.format(STATISTIC['auth_host'],
                                                                                            STATISTIC['auth_port'])))
            else:
                result = response.json()
                log.debug('STATISTIC auth success {} {} {}'.format('http://{auth_host}:{auth_port}'.format(
                    **STATISTIC), q_user, result))
                return result['token'], result['expiration_time']

    @classmethod
    def get_auth_token(cls):
        def _is_token_expire():
            return not cls.__token_expiration_time or \
                   time.time() >= cls.__token_expiration_time

        if not cls.is_cache_token or \
                not cls.__token or \
                _is_token_expire():
            cls.__token, cls.__token_expiration_time = cls.authenticate()
        return cls.__token

    @classmethod
    def get_auth_token_expiration(cls):
        return cls.__token_expiration_time


    @classmethod
    def send_request(cls, params, token=None, is_second_time=False,
                     rt_api=None,cdr_request='',method='get',timeout=30):
        from api_dnl.views.report import Report
        return Report().send_request(params)
        ## old code
        headers = {}
        for k in list(params.keys()):
            if params[k] is None:
                del params[k]
        tok = None
        if not token:
            tok = cls.get_auth_token()
        else:
            tok = token
        headers['Authorization'] = 'Token {}'.format(tok)
        url = cls.__get_base_url(rt_api=rt_api)

        try:
            if cdr_request!='':
                url=url+cdr_request
            if method=='delete':
                response = requests.delete(url=url,headers=headers, timeout=TIMEOUT)
            elif method=='post':
                if 'field' in params:
                    params['field']=cdr_fields(params['field'])
                response = requests.post(
                    url=url,  # cls.__get_base_url(rt_api=rt_api),
                    data=params,
                    headers=headers, timeout=TIMEOUT
                )

            else:
                response = requests.get(
                url=url,#cls.__get_base_url(rt_api=rt_api),
                params=params,
                headers=headers,timeout=TIMEOUT

            )
            log.debug(
                    'STATISTIC: curl -H "Authorization: Token {}" -X {} {} --data "{}" '.format(token, method.upper(),
                                                                                    url, unquote(urlencode(params))))


        except requests.ConnectionError as e:
            raise StatisticException(dict(code=410, msg='Connection error', data=url+' '+str(e)))
        except Exception as e:
            log.error(e)
            raise StatisticException(dict(code=411,msg='Unknown error',data=str(e)))
        else:
            # if not token and \
            #         token is not False:
            #     if response.status_code in (401, 403) and \
            #             not is_second_time:
            #         cls.__token = None
            #         return cls.send_request(
            #             params=params,
            #             token=token,
            #             is_second_time=True,
            #             rt_api=rt_api
            #         )

            if response.status_code not in (200, 201, 204):
                raise StatisticException(fmt_content(response.content))
            return response


class StatisticParams(object):

    def get_fields(self):
        raise NotImplementedError

    def _get_params(self):
        params = {}
        for field in self.get_fields():
            if isinstance(field, list):
                params[field[0]] = field[1]
                continue

            value = getattr(self, field, None)
            if value is None:
                raise AttributeError('`{}` is required.'.format(field))

            params[field] = value

        return  params


class TrafficStatistics(StatisticParams):
    """
    Formulas:
        ASR = non_zero/total calls *100
        ACD = Total Duration / Total Non_zero
    """
    def _get_totals(self, field):
        params = dict(field=field, **self._get_params())
        return StatisticAPI.send_request(params=params).json()

    def total_calls(self):
        """
        {'data': [{'time': 1496127600, 'value': 0.0}, {}],
         'code': 200, 'msg': None}
        """
        return self._get_totals('calls')

    def total_duration(self):
        """
        {'data': [{'time': 1496127600, 'value': 0.0}, {}],
         'code': 200, 'msg': None}
        """
        return self._get_totals('egress_billed_time')

    def total_non_zero(self):
        """
        {'data': [{'time': 1496127600, 'value': 0.0}, {}],
         'code': 200, 'msg': None}
        """
        return self._get_totals('non_zero_calls')

    def total_costs(self):
        """
        {'data': [{'time': 1496127600, 'value': 0.0}, {}],
         'code': 200, 'msg': None}
        """
        return self._get_totals('egress_cost')

    @staticmethod
    def asr(total_non_zero, total_calls):
        if not total_calls:
            return 0
        return total_non_zero / total_calls * 100

    @staticmethod
    def acd(total_duration, total_non_zero):
        if not total_non_zero:
            return 0
        return total_duration / total_non_zero

    @staticmethod
    def get_val(data, index, is_time=False):
        try:
            return data['data'][index]['value' if not is_time else 'time']
        except (ValueError, IndexError, KeyError):
            return 0


class StepMixin(object):
    STEP_HOURLY = 60
    STEP_DAILY = 1440
    STEP_WEAKLY = 10080
    STEP_YEARLY = 525600

    def _get_step(self, step):
        return {
            'd': self.STEP_DAILY,
            'h': self.STEP_HOURLY,
            'w': self.STEP_WEAKLY,
            'y': self.STEP_YEARLY
        }.get(step, self.STEP_HOURLY)


class TrafficSellerRoute(StepMixin, TrafficStatistics):
    """
    Get statistic for the traffic to one seller route

    Get AGG data of
    Call_attempt, ASr, ACD, egress_cost, duration, non_zero

    For a specific egress_resource_id and code_name

    Between start_time, end_time
    """
    start_time = None
    end_time = None
    egress_id = None
    orig_code = None
    step = None

    def __init__(self, start_time, end_time, egress_id, orig_code, step):
        self.start_time = start_time
        self.end_time = end_time
        self.egress_id = egress_id
        self.orig_code = orig_code
        self.step = self._get_step(step)

    def get_fields(self):
        return ['start_time', 'end_time', 'egress_id',
                'orig_code', 'step', ['method', 'total']]


class TrafficBuyerRoute(StepMixin, TrafficStatistics):
    """
    Get the hourly statistic for the traffic to one buyer route

    Get AGG data of
    Call_attempt, ASr, ACD, egress_cost, duration, non_zero

    For a specific egress_resource_id and code_name

    Between start_time, end_time
    """
    start_time = None
    end_time = None
    ingress_id = None
    orig_code = None
    step = None

    def __init__(self, start_time, end_time, ingress_id, orig_code, step):
        self.start_time = start_time
        self.end_time = end_time
        self.ingress_id = ingress_id
        self.orig_code = orig_code
        self.step = self._get_step(step)

    def get_fields(self):
        return ['start_time', 'end_time', 'ingress_id',
                'orig_code', 'step', ['method', 'total']]


class TrafficFromBuyerToSellerRoute(StepMixin, TrafficStatistics):
    """
    Get the hourly statistic for the traffic from one buyer
        to one seller route

    Get AGG data of
    Call_attempt, ASr, ACD, ingress_cost, duration, non_zero

    For a specific ingress_resource_id and code_name

    Between start_time, end_time
    """
    start_time = None
    end_time = None
    ingress_id = None
    egress_id = None
    orig_code = None
    step = None

    def __init__(self, start_time, end_time, ingress_id, egress_id,
                 orig_code, step):
        self.start_time = start_time
        self.end_time = end_time
        self.ingress_id = ingress_id
        self.egress_id = egress_id
        self.orig_code = orig_code
        self.step = self._get_step(step)

    def get_fields(self):
        return ['start_time', 'end_time', 'ingress_id', 'egress_id',
                'orig_code', 'step', ['method', 'total']]


class TrafficSellerRouteCodeGroup(StepMixin, TrafficStatistics):
    """
    Get the hourly statistic for the traffic to one seller route

    Get AGG data of
    Call_attempt, ASr, ACD, egress_cost, duration, non_zero

    For a specific egress_resource_id and code_name

    Between start_time, end_time
    """
    start_time = None
    end_time = None
    egress_id = None
    step = None

    def __init__(self, start_time, end_time, egress_id, step):
        self.start_time = start_time
        self.end_time = end_time
        self.egress_id = egress_id
        self.step = self._get_step(step)

    def get_fields(self):
        return ['start_time', 'end_time', 'egress_id', ['group', 'orig_code'],
                'step', ['method', 'total']]


# ------------------------------ CDR ------------------------------


class CdrStatistic(StatisticParams):
    """
    3 - time,
    10 - buyer,
    12 - caller_id from buyer
    14 - calling_number from buyer
    29 - seller_route
    31 - caller_id for seller
    33 - calling_number for seller
    50 - duration
    51 - pdd
    60 - buy_rate
    62 - buyer_bill_time
    64 - buy_cost
    67 - sell_rate
    68 - sell_cost
    69 - seller_bill_time
    87 - buyer_code
    88 - buyer_code_name
    89 - buying_country
    91 - seller_code
    92 - selling_code_name
    93 - selling_country

    buy_cost - ingress_client_cost
    sell_cost - egress_cost
    buy_rate - ingress_client_rate
    Sell_rate - egress_rate
    ingress_ip - origination_ingress_octets,
    egress_ip - termination_egress_octets
    """
    def get_statistic(self, is_process_data=True):
        response = StatisticAPI.send_request(
            params=self._get_params(), rt_api='cdr')
        if not is_process_data:
            return response

        return self.process_response(response)

    def process_response(self, response):
        data = []
        lines = response.content.decode('utf-8').split()
        if 'answer_time_of_date' in lines[0]:
            lines=lines[1:]
        for line in lines:
            data.append(dict(zip(self.get_field_names(), line.split('?'))))
        return data

    def get_field_names(self):
        raise NotImplementedError


class CdrNonZeroFromBuyerToSellerRoute(CdrStatistic):
    """
    Purpose: Get the Non Zero CDR for the traffic from one
        buyer to one seller route

    Get CDR fields Time, duration, caller_id, calling_number,
        buy_rate, sell_rate, buy_cost, sell_cost, code,
        code_name, country

    For specific ingress_resource_id, egress_resource_id and code_name

    Between start_time, end_time

    Filter by Non Zero
    """
    start_time = None
    end_time = None
    ingress_id = None
    egress_id = None
    orig_code = None
    step = None

    is_for_ingress_only = False

    def __init__(self, start_time, end_time, ingress_id, egress_id,
                 orig_code, step):
        """
        If total of an ingress needed then remove egress_id and orig_code
           and same like that for others
        """
        self.start_time = start_time
        self.end_time = end_time
        self.ingress_id = ingress_id
        self.egress_id = egress_id
        self.orig_code = orig_code
        self.step = step

        self.is_for_ingress_only = not self.ingress_id and not self.orig_code

    def get_fields(self):
        egress = ['ingress_id', 'orig_code'] if not self.is_for_ingress_only \
            else []
        return ['start_time', 'end_time', 'ingress_id', 'egress_id',
                'orig_code', ['non_zero', 1], 'step',
                ['field', '3,64,60,31,48,41,49,45,67,68,69,16,24']] + egress

    def get_field_names(self):
        return ['time', 'ani', 'dnis', 'duration', 'sell_rate', 'buy_rate',
                'sell_cost', 'buy_cost', 'code', 'code_name', 'country',
                'ingress_ip', 'egress_ip']


class CdrNonZeroTrafficToSeller(CdrStatistic):
    """
    Purpose: Get the Non Zero CDR for the traffic
        to a specific seller route

    Get CDR fields Time, duration, caller_id, calling_number,
     buy_rate, sell_rate, buy_cost, sell_cost, code, code_name, country

    For specific egress_resource_id and code_name

    Between start_time, end_time

    Filter by Non Zero
    """
    start_time = None
    end_time = None
    egress_id = None
    orig_code = None
    step = None

    def __init__(self, start_time, end_time, egress_id, orig_code, step):
        self.start_time = start_time
        self.end_time = end_time
        self.egress_id = egress_id
        self.orig_code = orig_code
        self.step = step

    def get_fields(self):
        return ['start_time', 'end_time', 'egress_id', 'orig_code',
                ['non_zero', 1], 'step',
                ['field', '3,64,60,31,48,41,49,45,67,68,69,16,24']]

    def get_field_names(self):
        return ['time', 'ani', 'dnis', 'duration', 'sell_rate', 'buy_rate',
                'sell_cost', 'buy_cost', 'code', 'code_name', 'country',
                'ingress_ip', 'egress_ip']


class CdrTrafficFromBuyerToSeller(CdrStatistic):
    """
    Purpose: Get all CDR for the traffic from one buyer
        to one seller route

    Get CDR fields Time, duration, caller_id, calling_number,
        buy_rate, sell_rate, buy_cost, sell_cost, code, code_name, country

    For specific ingress_resource_id, egress_resource_id and code_name

    Between start_time, end_time
    """
    start_time = None
    end_time = None
    ingress_id = None
    egress_id = None
    orig_code = None
    step = None

    is_for_ingress_only = False

    def __init__(self, start_time, end_time, ingress_id,
                 egress_id=None, orig_code=None, step=None):
        """
        If total of an ingress needed then remove egress_id and orig_code
           and same like that for others
        """
        self.start_time = start_time
        self.end_time = end_time
        self.egress_id = egress_id
        self.ingress_id = ingress_id
        self.orig_code = orig_code
        self.step = step

        self.is_for_ingress_only = not self.ingress_id and not self.orig_code

    def get_fields(self):
        egress = ['ingress_id', 'orig_code'] if not self.is_for_ingress_only \
            else []
        return ['start_time', 'end_time', 'egress_id',# 'step',
                ['field', '3,12,14,50,67,60,68,64,87,88,89,24,44']] + egress

    def get_field_names(self):
        return ['time', 'ani', 'dnis', 'duration', 'sell_rate', 'buy_rate',
                'sell_cost', 'buy_cost', 'code', 'code_name', 'country',
                'ingress_ip', 'egress_ip']


class CdrTrafficToSellerRoute(CdrStatistic):
    """
    Purpose: Get all CDR for the traffic  to a specific seller route

    Get CDR fields Time, duration, caller_id, calling_number, buy_rate,
        sell_rate, buy_cost, sell_cost, code, code_name, country


    For specific egress_resource_id and code_name

    Between start_time, end_time
    """
    start_time = None
    end_time = None
    egress_id = None
    orig_code = None
    step = None

    def __init__(self, start_time, end_time, egress_id, orig_code, step):
        self.start_time = start_time
        self.end_time = end_time
        self.egress_id = egress_id
        self.orig_code = orig_code
        self.step = step

    def get_fields(self):
        return ['start_time', 'end_time', 'egress_id', 'orig_code', #'step',
                ['field', '3,12,14,50,67,60,68,64,87,88,89,24,44']]

    def get_field_names(self):
        return ['time', 'ani', 'dnis', 'duration', 'sell_rate', 'buy_rate',
                'sell_cost', 'buy_cost', 'code', 'code_name', 'country',
                'ingress_ip', 'egress_ip']

from time import mktime
from datetime import datetime,timedelta


def _join(d1, d2):
    return [d1[k].update(d2[k]) for k in d1 if k in d2]

def _calc(d, callback, field):
    for k in d:
        d[k][field] = callback(**d[k])

def _asr(**kwargs):
    if not kwargs['calls']:
        return 0
    return kwargs['non_zero_calls'] / kwargs['calls'] * 100

def _acd(**kwargs):
    if not kwargs['non_zero_calls']:
        return 0
    return kwargs['egress_billed_time'] / kwargs['non_zero_calls']

def _htime(**kwargs):
    if not kwargs['time']:
        return 0
    return datetime.fromtimestamp(kwargs['time'])

def _ingress_billed_min(**kwargs):
    return kwargs['ingress_billed_time']/60

def _ingress_avg_rate(**kwargs):
    if not kwargs['ingress_billed_min']:
        return 0
    return kwargs['ingress_cost']/kwargs['ingress_billed_min']


class IStatistic:
    def _ingress(self):
        return None
    def _egress(self):
        return None
    def _code(self):
        return None

    @staticmethod
    def _list_to_str(lst):
        if isinstance(lst,list):
            return ','.join([str(i) for i in lst])
        else:
            return lst


    def get_total(self,filtering,field,delta=timedelta(hours=1)):
        mp= {'ingress_id':self._ingress,'egress_id':self._egress,'orig_code':self._code}

        end = int(datetime.now(UTC).timestamp())
        start = int((datetime.now(UTC) - delta).timestamp())
        step = (end - start)/60
        if step >= 525600:step = 525600
        elif step >= 10080:step >= 10080
        elif step >= 1440:step = 1440
        else:step = 60

        params=dict(start_time=start, end_time=end, step=step,method='total',field=field)
        params[filtering]=self._list_to_str(mp[filtering]())

        total=0
        if params[filtering]:
            ret=StatisticAPI.send_request(params=params).json()
            total=sum([v[field] for v in ret['data'] if field in v] )
        log.debug('Statictic API {}={} {} Total value: {}'.format(filtering,params[filtering],field,total))
        return total

    def get_detail(self,filtering,field,delta=timedelta(hours=1),end=None,group_by=None):
        mp= {'ingress_id':self._ingress,'egress_id':self._egress,'orig_code':self._code}
        if not end:
            end = int(datetime.now(UTC).timestamp())
        else:
            end = int(end.timestamp())
        start = int((datetime.now(UTC) - delta).timestamp())
        step = (end - start)/60
        if step >= 525600*3:step = 525600
        elif step >= 10080*3:step > 10080
        elif step >= 1440*3:step = 1440
        elif step >=60*3:step = 60
        else:step = 1

        params=dict(start_time=start, end_time=end, step=step,method='total',field=field)
        params[filtering]=self._list_to_str(mp[filtering]())
        if group_by:
            params['group']=group_by
        r = {}
        if params[filtering]:
            ret=StatisticAPI.send_request(params=params).json()
            for v in ret['data']:
                if 'time' in v:
                    r[v['time']] = v
        return r

    def ingress_non_zero_calls(self,delta=timedelta(hours=1)):
        return self.get_total('ingress_id','non_zero_calls',delta)

    def egress_calls(self,delta=timedelta(hours=1)):
        return self.get_total('egress_id','calls',delta)

    def ingress_calls(self,delta=timedelta(hours=1)):
        return self.get_total('ingress_id','calls',delta)

    def egress_non_zero_calls(self,delta=timedelta(hours=1)):
        return self.get_total('egress_id','non_zero_calls',delta)

    def egress_cost(self,delta=timedelta(hours=1)):
        return self.get_total('egress_id','egress_cost',delta)

    def ingress_cost(self,delta=timedelta(hours=1)):
        return self.get_total('ingress_id','ingress_cost',delta)

    def egress_billed_time(self,delta=timedelta(hours=1)):
        return self.get_total('egress_id','egress_billed_time',delta)

    def ingress_billed_time(self,delta=timedelta(hours=1)):
        return self.get_total('ingress_id','ingress_billed_time',delta)

    def egress_asr(self,delta=timedelta(hours=1)):
        calls=self.egress_calls(delta)
        if not calls:
            return 0
        return self.egress_non_zero_calls(delta)/calls

    def egress_acd(self,delta=timedelta(hours=1)):
        calls=self.egress_non_zero_calls(delta)
        if not calls:
            return 0
        return self.egress_billed_time(delta)/calls


    def common_egress_stat(self,delta=timedelta(hours=1),end=datetime.now(),group_by=None):

        total_calls = self.get_detail('egress_id','calls',delta,end,group_by)
        total_duration = self.get_detail('egress_id','egress_billed_time',delta,end,group_by)
        total_non_zero = self.get_detail('egress_id','non_zero_calls',delta,end,group_by)
        total_costs = self.get_detail('egress_id','egress_cost',delta,end,group_by)

        d=total_calls
        _join(d,total_duration)
        _join(d, total_non_zero)
        _join(d, total_costs)

        _calc(d,_asr,'asr')
        _calc(d, _acd, 'acd')
        _calc(d, _htime, 'time')
        return list(d.values())

    def common_ingress_stat(self,delta=timedelta(hours=1),end=datetime.now(),group_by=None):

        total_calls = self.get_detail('ingress_id','calls',delta,end,group_by)
        total_duration = self.get_detail('ingress_id','ingress_billed_time',delta,end,group_by)
        total_non_zero = self.get_detail('ingress_id','non_zero_calls',delta,end,group_by)
        total_costs = self.get_detail('ingress_id','ingress_cost',delta,end,group_by)

        d=total_calls
        _join(d,total_duration)
        _join(d, total_non_zero)
        _join(d, total_costs)

        _calc(d,_asr,'asr')
        _calc(d, _acd, 'acd')

        return list(d.values())

    def common_ingress_usage_report (self,delta=timedelta(hours=1),end=datetime.now(),group_by=None):

        total_calls = self.get_detail('ingress_id','calls',delta,end,group_by)
        total_duration = self.get_detail('ingress_id','ingress_billed_time',delta,end,group_by)
        total_non_zero = self.get_detail('ingress_id','non_zero_calls',delta,end,group_by)
        total_costs = self.get_detail('ingress_id','ingress_cost',delta,end,group_by)

        d=total_calls
        _join(d,total_duration)
        _join(d, total_non_zero)
        _join(d, total_costs)

        _calc(d, _asr,'asr')
        _calc(d, _acd, 'acd')
        _calc(d, _ingress_billed_min, 'ingress_billed_min')
        _calc(d, _ingress_avg_rate, 'ingress_avg_rate')

        return list(d.values())

def test_fields():
    api = StatisticAPI()
    for f in FIELDS.keys():
        print(api.authenticate(allowed_fields=str(f)))
