import traceback
import json
from datetime import datetime, timedelta, timezone
from urllib.parse import urlencode, parse_qsl

import falcon
from falcon_rest import swagger
from falcon_rest.logger import log
from falcon_rest.responses import responses, errors
from falcon_rest.resources.resources import swagger, ResourcesBaseClass, DEFAULT_SECURITY, ATTRIBUTE_ERROR_RE
from marshmallow import ValidationError
from pytz import UTC
from sqlalchemy.exc import InternalError, IntegrityError, DataError
from sqlalchemy import cast, case, literal, inspect, and_, or_, alias, func, cast, Integer, DateTime, select, text, union_all
from sqlalchemy.orm import aliased

from api_dnl import model
from api_dnl.resources import OperationErrorResponse, DnlList, DnlCreate, DnlResource, CustomGetAction

from api_dnl.schemes.cdr import CdrReportGroppedRequestScheme, CdrReportDetailInnerScheme, \
    CdrReportGroppedResponseScheme
from api_dnl.schemes.cdr import DidReportGroppedRequestScheme, DidReportInnerScheme, \
    DidReportGroppedResponseScheme, ClientCdrReportGroppedResponseScheme, CdrReportGetScheme, \
    ClientCdrReportDailyGroppedResponseScheme, CdrReportDailyGetScheme, CdrReportDetailGroupableGetScheme, \
    ClientCodeReportDailyGroppedResponseScheme, CodeReportDailyGetScheme
from api_dnl.schemes.cdr import ReportTemplateScheme, ReportTemplateSchemeModify, ReportTemplateSchemeGet

from api_dnl.views.cdr import CdrReportDetailList

REPORT_DEBUG = False


# _model_class = model.CdrReportDetail
# _columns = _model_class.get_fields()
# _groupable = [fn for fn in _columns if
#                  getattr(_model_class, fn).type.__class__.__name__ in ('String', 'DateTime') or '_id' in fn or fn in (
#                      'orig_jur_type', 'release_cause', 'term_jur_type')]
# _aggregable = [fn for fn in _columns if fn not in _groupable]

class SuccessReportResponse(responses.BaseResponse):
    description = 'Successfully executed'
    scheme = CdrReportGroppedResponseScheme

    def get_response_body(self):
        return self.data


_grp = ('orig_jur_type', 'release_cause', 'term_jur_type', 'dst_lata', 'dst_ocn', 'src_lata', 'dst_ocn', 'lrn_lata',
        'lrn_ocn')


class Report(DnlList):
    entity_plural = 'Report items'
    scheme_class = CdrReportGroppedRequestScheme
    scheme_class_get = CdrReportGroppedResponseScheme
    scheme_class_get_inner = CdrReportDetailInnerScheme
    model_class = model.CdrReportDetail
    columns = model_class.get_fields()

    groupable = list(set([fn for fn in columns if
                 getattr(model.CdrReportDetail, fn).type.__class__.__name__ in (
                     'String', 'DateTime', 'Text', 'Boolean') or '_id' in fn or fn in _grp] +
                        ['origination_remote_payload_ip_address', 'termination_remote_payload_ip_address',
                         'ingress_name', 'egress_name', 'ani_in_dno', 'ani_in_dno_count', 'ani_in_ftc_count',
                         'ani_in_ftc_week_count', 'ani_in_ftc_month_count', 'ani_in_spam_count',
                         'ani_in_fraud_count', 'ani_in_tcpa_count', 'dnis_in_dnc_count']))
    group_mapping = {
        'ingress_name': 'ingress_id',
        'egress_name': 'egress_id',
    }
    aggregable = list(set([fn for fn in columns if
                           getattr(model.CdrReportDetail, fn).type.__class__.__name__ not in (
                               'String', 'DateTime', 'Text', 'Boolean') and '_id' not in fn and fn not in _grp] +
                          ['nrf_calls', 'asr', 'qsr', 'ingress_npr', 'egress_npr', 'SDP30_calls', 'SDP6_calls',
                           'ingress_call_dur', 'egress_call_dur', 'ingress_bill_dur', 'egress_bill_dur',
                           'not_zero_calls_12', 'not_zero_calls_18', 'not_zero_calls_24', 'commission', 'limited']))

    def get_spec(self):
        sfields_names, search_fields_list = self.get_fields_from_scheme(self.scheme_class)
        qfields_names, query_fields_list = self.get_query_fields_from_scheme(self.scheme_class)
        fields_names = [f.name for f in inspect(self.model_class).columns]
        spec = swagger.specify.get_spec(
            method='get', description='Gets {}'.format(self.entity_plural.lower()),
            path_parameters=self.path_parameters,
            query_parameters=search_fields_list + [
                {"name": "per_page", "description": "printing range. Default: print 1000 items",
                 "type": "integer"},
                {"name": "page", "description": "printing range. Default: print 1st page",
                 "type": "integer"},
                {"name": "fields",
                 "description": "aggregated fields, available values:{}".format(self.aggregable)},
                {"name": "group",
                 "description": "groupping fields, available values:{}".format(self.groupable)},

            ] +
                             [dict(name=i, type='string') for i in self.groupable],
            responses=(
                          SuccessReportResponse(),
                          responses.AttributeNotExitsErrorResponse()
                      ) + self.get_additional_responses(method='get'),
            security=self.get_security(method='get')
        )
        spec['get']['produces'] = ['application/json', 'text/csv', 'text/xml']
        return spec

    def run(self, **kwargs):
        class Req:
            def __init__(self, req_user):
                self.query_string = None
                self.context = {'user': req_user}

            pass

        if 'user' in kwargs:
            user = kwargs['user']
            del kwargs['user']
        else:
            user = model.User.get(1)
        req = Req(user)
        req.query_string = urlencode(kwargs)
        self.req = req
        objects_list, total, page, per_page, fields = self.get_objects_list(req, self.model_class, **kwargs)
        # items = [i for i in objects_list]
        items = self.scheme_class_get_inner().dump(objects_list, many=True).data
        return items

    def send_request(self, params):
        m = dict(calls='ingress_total_calls',
                 ingress_billed_time='ingress_bill_time',
                 ingress_cost='ingress_call_cost',
                 ingress_billed_min='ingress_bill_time',
                 code='ingress_code',
                 non_zero_calls='not_zero_calls',
                 egress_cost='egress_total_cost',
                 egress_billed_time='egress_bill_time',
                 )
        args = {k: v for k, v in params.items() if k not in m}
        upd = {m[k]: v for k, v in params.items() if k in m}
        args.update(upd)

        class Responce:
            def json(self):
                return self.data

        items = self.run(**args)
        resp = Responce()
        resp.data = dict(data=items)
        return resp

    def _on_get(self, req, resp, **kwargs):
        from falcon_rest import schemes
        from falcon_rest.helpers import check_permission

        if (not hasattr(self, 'no_auth_needed') or self.no_auth_needed == False) and not check_permission(self, req,
                                                                                                          'list'):
            self.set_response(
                resp, responses.ForbiddenErrorResponse(data=errors.AuthErrors.Forbidden)
            )
            return
        try:
            import time
            _start = time.time()
            objects_list, total, _from, count, fields = self.get_objects_list(req, self.model_class, **kwargs)
            # items = [i for i in objects_list]
            items = self.scheme_class_get_inner().dump(objects_list, many=True).data
            data = {
                'success': True,
                'data': items,
                'total': total, 'from': _from, 'count': len(items)
            }
            self.set_response(resp, SuccessReportResponse(data=data, scheme=self.scheme_class_get))

            _end = time.time()
            log.debug('report execution time {}'.format(_end - _start))
            # resp.body = json.dumps(data)
            # resp.status = falcon.HTTP_200

        # except AttributeError as e:
        #     self.set_response(
        #         resp, responses.AttributeNotExitsErrorResponse(
        #             data=ATTRIBUTE_ERROR_RE.search(str(e)).group(1)
        #         )
        #     )
        except DataError as e:
            import traceback
            log.error(traceback.format_exc())
            self.set_response(resp, SuccessReportResponse(data={'message':e}))
        except (InternalError, IntegrityError, ) as e:
            import traceback
            model.get_db().session.rollback()
            log.error(traceback.format_exc())
            self.set_response(resp, OperationErrorResponse(e))
        except Exception as e:
            import traceback
            log.error(traceback.format_exc())
            self.set_response(resp, OperationErrorResponse(e))

    def time_field(self, cdr):
        return cdr.c.report_time.label('report_time')

    def time_field_str(self):
        return 'report_time'

    def get_daily_cdr(self, cdr_date):
        cdr = model.cdr_report_detail(cdr_date)
        return cdr

    def get_fields_for_list(self, parsed_qs, **kwargs):
        fields = parsed_qs.get('fields', None)
        if fields:
            fields = [field.strip() for field in fields.split(',')]
            for f in fields:
                # if f not in self.get_all_fields_from_scheme(self.scheme_class_get)[0]:
                if f not in self.aggregable and f not in self.columns and f not in (self.time_field_str(), 'time'):
                    raise Exception('fields {} not valid!'.format(f))
        return fields

    def get_objects_list(self, req, model_class, **kwargs):
        parsed_qs = dict(parse_qsl(req.query_string))
        if REPORT_DEBUG:
            log.debug('report parsed parameters:{}'.format(parsed_qs))
        try:
            per_page = int(parsed_qs.get('per_page', 1000000))
        except ValueError:
            per_page = 1000000

        try:
            page = int(parsed_qs.get('page'))
        except (ValueError, TypeError):
            page = 0


        ordering = {}
        log.debug("GETTING FIELDS")
        fields = self.get_fields_for_list(parsed_qs, **kwargs)
        user = self.get_user(req) 
        if user.agent and user.agent.method_type == "By Revenue":
            fields = [field for field in fields if field != "egress_call_cost"]
        log.debug("GETTING FILTERING")
        filtering = self.get_filtering_for_list(parsed_qs, **kwargs)
        if fields:
            filtering['fields'] = ','.join(fields)
        log.debug("GETTING FILTERING AND QUERY")
        filtering, query = \
            self.modify_query_from_filtering_for_list(filtering, **kwargs)

        # for c in inspect(self.model_class).columns:
        #     if c.name in ['alias','name','template_name','group_name']:
        #         query = query.filter(or_(c.is_(None),c.notlike('#%')))
        #         break

        if ordering:
            log.debug("Modifying query")
            ordering, query = \
                self.modify_query_from_ordering_for_list(ordering, query, **kwargs)
        if query is None:
            return [], 0, page, per_page, fields
        log.debug(f"QUERY = {query}")
        log.debug(f"FILTERING = {filtering}")
        log.debug(f"ORDERING = {ordering}")
        return model_class.get_objects_list(
            query=query,
            filtering=filtering,
            ordering=ordering,
            paging={'from': page * per_page, 'till': (page + 1) * per_page}
        ) + (page, per_page, fields)

    def on_filter_agent_id(self, v, cdr):
        cls = model.AgentClient
        clients = cls.session().query(cls.client_id).filter(cls.agent_id == v).all()
        return cdr.c.ingress_client_id.in_([int(i.client_id) for i in clients])

    def on_filter_o_trunk_type2(self, v, cdr):
        cls = model.Client
        clients = cls.session().query(cls.client_id).filter(cls.company_type==v).all()
        return cdr.c.ingress_client_id.in_([int(i.client_id) for i in clients])

    def on_filter_trunk_type2(self, v, cdr):
        cls = model.Resource
        ingress = v == 0
        resources = cls.session().query(cls.resource_id).filter(and_(cls.trunk_type2==v, cls.ingress==ingress)).all()
        return cdr.c.ingress_id.in_([int(i.resource_id) for i in resources])

    def on_filter_ingress_id(self, v, cdr):
        # return text("ingress_id in ({})".format(v))
        return cdr.c.ingress_id.in_([int(i) for i in v.split(',')])

    def on_filter_egress_id(self, v, cdr):
        # return text("egress_id in ({})".format(v))
        return cdr.c.egress_id.in_([int(i) for i in v.split(',')])

    def on_filter_ingress_client_id(self, v, cdr):
        # return text("egress_id in ({})".format(v))
        return cdr.c.ingress_client_id.in_([int(i) for i in v.split(',')])

    def on_filter_egress_client_id(self, v, cdr):
        # return text("egress_id in ({})".format(v))
        return cdr.c.egress_client_id.in_([int(i) for i in v.split(',')])

    def on_filter_non_zero_calls(self, v, cdr):
        # return text("egress_id in ({})".format(v))
        if v == '1':
            return cdr.c.ingress_not_zero_calls > 0

    def on_filter_scc_below_sec(self, v, cdr):
        return cdr.c.duration < v

    def on_filter_is_hide_unauthorized(self, v, cdr):
        if v is None:
            return None
        if v:
            return cdr.c.release_cause != 3
        else:
            return cdr.c.release_cause == 3

    def on_select_ingress_call_dur(self, cdr):
        return 1.0 * cdr.c.ingress_duration / 60

    def on_select_egress_call_dur(self, cdr):
        return 1.0 * cdr.c.duration / 60

    def on_select_ingress_bill_dur(self, cdr):
        return cdr.c.ingress_bill_time / 60

    def on_select_egress_bill_dur(self, cdr):
        return cdr.c.egress_bill_time / 60

    def on_select_nrf_calls(self, cdr):
        return cdr.c.lrn_calls

    def on_select_asr(self, cdr):
        return 100.00 * ((1.0 * cdr.c.not_zero_calls) / (1.0 * cdr.c.ingress_total_calls))

    def on_select_qsr(self, cdr):
        return 100.00 * ((cdr.c.not_zero_calls + cdr.c.ingress_busy_calls + cdr.c.ingress_cancel_calls) /
                         cdr.c.ingress_total_calls)

    def on_select_ingress_npr(self, cdr):
        return 100.00 * ((1.0 * cdr.c.npr_count) / (1.0 * cdr.c.ingress_total_calls))

    def on_select_egress_npr(self, cdr):
        return 100.00 * ((1.0 * cdr.c.npr_count) / (1.0 * cdr.c.egress_total_calls))

    def on_select_SDP30_calls(self, cdr):
        return cdr.c.not_zero_calls_6 + cdr.c.call_12s + cdr.c.call_18s + cdr.c.call_24s + cdr.c.not_zero_calls_30

    def on_select_not_zero_calls_12(self, cdr):
        return cdr.c.call_12s

    def on_select_not_zero_calls_18(self, cdr):
        return cdr.c.call_18s

    def on_select_not_zero_calls_24(self, cdr):
        return cdr.c.call_24s

    def on_select_SDP6_calls(self, cdr):
        return cdr.c.not_zero_calls_6

    def on_select_ACD(self, cdr):
        return ((1.0 * cdr.c.egress_bill_time) / (1.0 * cdr.c.not_zero_calls))

    def on_select_agent_id(self, cdr):
        cls = model.AgentClient
        return select([cls.agent_id]).where(cls.client_id == cdr.c.ingress_client_id)
    
    def on_select_ftc_ani(self, cdr):
        return 1 if cdr.c.ani_in_ftc else 0
    
    def on_select_non_ftc_ani(self, cdr):
        return 0 if cdr.c.ani_in_ftc else 1
    
    def on_select_week_ftc_ani(self, cdr):
        return 1 if cdr.c.ani_in_week_ftc else 0
    
    def on_select_non_week_ftc_ani(self, cdr):
        return 0 if cdr.c.ani_in_week_ftc else 1
    
    def on_select_month_ftc_ani(self, cdr):
        return 1 if cdr.c.ani_in_month_ftc else 0
    
    def on_select_non_month_ftc_ani(self, cdr):
        return 0 if cdr.c.ani_in_month_ftc else 1
    
    def on_select_dno_ani(self, cdr):
        return 1 if cdr.c.ani_in_dno else 0
    
    def on_select_non_dno_ani(self, cdr):
        return 0 if cdr.c.ani_in_dno else 1
    
    def on_select_spam_ani(self, cdr):
        return 1 if cdr.c.spam_ani else 0
    
    def on_select_non_spam_ani(self, cdr):
        return 0 if cdr.c.spam_ani else 1
    
    def on_select_fraud_ani(self, cdr):
        return 1 if cdr.c.fraud_ani else 0
    
    def on_select_non_fraud_ani(self, cdr):
        return 0 if cdr.c.fraud_ani else 1
    
    def on_select_tcpa_ani(self, cdr):
        return 1 if cdr.c.tcpa_ani else 0
    
    def on_select_non_tcpa_ani(self, cdr):
        return 0 if cdr.c.tcpa_ani else 1

    def on_filter_show_non_zero_only(self, v, cdr):
        if v == 'true':
            return cdr.c.duration > 0
        else:
            return cdr.c.duration == 0

    def on_select_commission(self, cdr):
        cls = model.AgentClient
        return select([cls.commission * (cdr.c.ingress_call_cost - ((cls.method_type+1)%2)*cdr.c.egress_call_cost)/100]). \
                    where(cls.client_id == cdr.c.ingress_client_id).as_scalar()

    def on_select_limited(self, cdr):
        return func.cast(cdr.c.release_cause.in_([1, 2, 6, 7, 8, 9, 29, 30, 45, 46, 50, 51, 52, 53, 56, 65, 66, 67, 68, 69]), Integer)

    def modify_query_from_filtering_for_list(self, filtering, **kwargs):

        def similar_fields(field, field_list):
            sf = field.split('_')
            sim = []
            for f in sf:
                for ef in field_list:
                    if f in ef:
                        sim.append(ef)
            if sim:
                return 'similar fields:{}'.format(set(sim))
            return ''

        import sys
        log.debug("SETTING RECURSION LIMIT")
        sys.setrecursionlimit(10000)
        cls = self.model_class
        log.debug(f"MODEL - {cls}")
        columns = cls.get_fields()
        log.debug("GOT COLUMNS")
        # columns = [f.name for f in inspect(self.model_class).columns]
        req_query = dict(parse_qsl(self.req.query_string))
        _from = filtering.pop('from', None)
        count = filtering.pop('count', None)

        sort = filtering.pop('sort', None)
        order = filtering.pop('order', '')
        groups = filtering.pop('group', None)

        sort_order = None
        if sort:
            log.debug("SORT")
            if sort in ('report_time', 'time'):
                sort = self.time_field_str()
            _sort = sort.split(',')
            for s in _sort:
                log.debug(f"SORT - {s}")
                if s not in columns and not s.replace('name', 'id') in columns and not hasattr(self, 'on_select_' + s):
                    raise ValidationError('sort field missing {}'.format(s))
                if s.replace('name', 'id') in self.groupable and s != self.time_field_str():
                    if groups:
                        groups += ',' + s.replace('name', 'id')
                    else:
                        groups = s.replace('name', 'id')
            _order = order.split(',')
            for o in _order:
                log.debug(f"ORDER - {o}")
                if o not in ('', 'asc', 'desc'):
                    raise ValidationError('sort order invalid value {}, must be asc or desc (can be empty)'.format(o))
            if len(_sort) != len(_order):
                if sort != '':
                    raise ValidationError('sort length must be equal order length')
                else:
                    _order = ['' for s in _sort]
            sort_order = list(zip(_sort, _order))
        log.debug("GETTING START TIME AND END TIME")
        start_time = filtering.pop('start_time', None)
        end_time = filtering.pop('end_time', str(int(datetime.now(UTC).timestamp())))
        try:
            start_time = int(start_time)
            if start_time < 0:
                start_time = int(datetime.now(UTC).timestamp()) + start_time
            start_date_time = datetime.fromtimestamp(start_time, tz=timezone.utc)
        except (ValueError, TypeError):
            raise ValidationError('wrong start time {}'.format(start_time))
        try:
            end_time = int(end_time)
            end_date_time = datetime.fromtimestamp(end_time, tz=timezone.utc)
        except (ValueError, TypeError):
            raise ValidationError('wrong end time')
        if start_time >= end_time:
            raise ValidationError('start time greater than end_time!')
        # aggregable = ['agent_cost', 'agent_rate', 'call_12s', 'call_18s', 'call_24s', 'call_2h', 'call_3h', 'call_4h',
        #               'duration', 'duration_30', 'duration_6', 'egress_bill_time', 'egress_bill_time_inter',
        #               'egress_bill_time_intra', 'egress_busy_calls', 'egress_call_cost', 'egress_call_cost_ij',
        #               'egress_call_cost_inter', 'egress_call_cost_intra', 'egress_call_cost_local',
        #               'egress_cancel_calls', 'egress_no_ring', 'egress_rate', 'egress_rate_date',
        #               'egress_success_calls', 'egress_total_calls', 'incoming_bandwidth', 'ingress_bill_time',
        #               'ingress_bill_time_inter', 'ingress_bill_time_intra', 'ingress_busy_calls', 'ingress_call_12s',
        #               'ingress_call_18s', 'ingress_call_24s', 'ingress_call_2h', 'ingress_call_3h', 'ingress_call_4h',
        #               'ingress_call_cost', 'ingress_call_cost_ij', 'ingress_call_cost_inter', 'ingress_call_cost_intra',
        #               'ingress_call_cost_local', 'ingress_cancel_calls', 'ingress_duration', 'ingress_duration_30',
        #               'ingress_duration_6', 'ingress_inter_duration', 'ingress_inter_not_zero_calls',
        #               'ingress_intra_duration', 'ingress_intra_not_zero_calls', 'ingress_not_zero_calls',
        #               'ingress_not_zero_calls_30', 'ingress_not_zero_calls_6', 'ingress_rate', 'ingress_rate_date',
        #               'ingress_success_calls', 'ingress_total_calls', 'inter_duration', 'inter_ingress_total_calls',
        #               'inter_not_zero_calls', 'intra_duration', 'intra_ingress_total_calls', 'intra_not_zero_calls',
        #               'lnp_cost', 'lrn_calls', 'not_zero_calls', 'not_zero_calls_30', 'not_zero_calls_6', 'npr_count',
        #               # 'orig_jur_type',
        #               'outgoing_bandwidth', 'pdd', 'q850_cause_count',
        #               # 'release_cause',
        #               'ring_pdd',
        #               # 'term_jur_type'
        #               ]

        # #groupable = [fn for fn in columns if
        #              getattr(cls, fn).type.__class__.__name__ in ('String', 'DateTime') or '_id' in fn or fn in (
        #                  'orig_jur_type', 'release_cause', 'term_jur_type')]
        # #aggregable = [fn for fn in columns if fn not in groupable]
        groupable = self.groupable
        aggregable = self.aggregable
        group_mapping = self.group_mapping
        log.debug(f"groupable = {groupable} | aggregable = {aggregable}")
        tz = filtering.pop('tz', None)
        if REPORT_DEBUG:
            log.debug('tz={}'.format(tz))
        non_zero_calls = filtering.pop('non_zero_calls', None)
        if REPORT_DEBUG:
            log.debug('non_zero_calls={}'.format(non_zero_calls))
        human_readable = filtering.pop('human_readable', None)
        if REPORT_DEBUG:
            log.debug('human_readable={}'.format(human_readable))
        log.debug("getting data")
        method = filtering.pop('method', 'total')
        step = filtering.pop('step', 'all')
        fields = filtering.pop('fields', None)
        field = filtering.pop('field', None)
        fields = fields or field
        log.debug(f"FIELDS = {fields}")
        if fields:
            fields = fields.split(',')
        if not fields:
            fields = aggregable.copy()
        for fk in filtering.keys():
            log.debug(f"FK = {fk}")
            if fk in groupable and not hasattr(self, 'on_filter_{}'.format(fk)):
                if groups:
                    groups += ',{}'.format(fk)
                else:
                    groups = fk
            if fk not in columns:
                if not hasattr(self, 'on_filter_{}'.format(fk)):
                    raise ValidationError('{}:wrong filter field. {}'.format(fk, similar_fields(fk, columns)))
            if fk not in fields and not hasattr(self, 'on_filter_{}'.format(fk)):
                fields.append(fk)

        if groups is not None:
            # groups = groups.replace('_name', '_id') #.replace('egress_name', 'egress_id')
            for field_id, field_name in group_mapping.items():
                groups = groups.replace(field_id, field_name)
            groups = [i.strip() for i in groups.split(',')]
        else:
            groups = []
        if self.time_field_str() in groups:
            step = 60
            groups.remove(self.time_field_str())
        for fk in groups:
            if fk not in groupable:
                raise ValidationError('{}:wrong group field. {}'.format(fk, similar_fields(fk, groupable)))
            if fk not in fields:
                fields.append(fk)
        if sort:
            for fk in sort.split(','):
                if fk not in fields and fk not in ('ingress_name', 'egress_name'):
                    fields.append(fk)

        fields.append(self.time_field_str())
        fields = list(set(fields))

        user = self.get_user(self.req)

        if user and user.user_type in ('client', 'vendor'):
            # if user.user_type == 'client' and user.client.company_type == 'termination' and ('egress_id' in filtering):
            #     raise ValidationError('Term client can\'t query the Reports using the egress_id!')
            # if user.user_type == 'client' and user.client.company_type == 'origination' and ('ingress_id' in filtering):
            #     raise ValidationError('DID client can\'t query the Reports using the ingress_id!')

            if 'ingress_id' in filtering and filtering['ingress_id']:
                ids = set(filtering['ingress_id'].split(','))
                alw = set([str(i) for i in user.allowed_ingress_id])
                filtering['ingress_id'] = ','.join(list(ids.intersection(alw)))
            else:
                filtering['ingress_id'] = ','.join([str(i) for i in user.allowed_ingress_id])
            if filtering['ingress_id'] == '' or (user.user_type == 'client' and user.client.company_type == 'origination'):
                filtering['ingress_id'] = ''
                if user.allowed_egress_id:
                    alw = set([str(i) for i in user.allowed_egress_id])
                    if 'egress_id' in filtering and filtering['egress_id']:
                        log.debug("EGRESS_ID IN FILTERING")
                        ids = set(filtering['egress_id'].split(','))
                        filtering['egress_id'] = ','.join(list(ids.intersection(alw)))
                        log.debug("SET FILTERING")
                    else:
                        filtering['egress_id'] = ','.join(list(alw))
                        log.debug("SET FILTERING")
                    del filtering['ingress_id']
                else:
                    filtering['ingress_id'] = '-1'
            if REPORT_DEBUG:
                log.debug('result {} filtering={}'.format(user.user_type, filtering))
        elif user and user.user_type == 'agent':
            filtering['agent_id'] = user.agent.agent_id

        start_day = start_date_time.date()
        database_start_day = model.cdr_reports_start_date()
        if database_start_day is None:
            raise ValidationError('database has no cdr report data!')
        if database_start_day > start_day:
            start_day = database_start_day
        end_day = end_date_time.date()
        if end_day < start_day:
            end_day = start_day

        days = (end_day - start_day).days
        union_query = None
        union_list = []
        for day_idx in range(days + 1):

            day_cdr = start_day + timedelta(days=day_idx)
            # if hasattr(self, 'ya_query'):
            #     q = self.ya_query(fields, day_cdr, day_cdr, filtering)
            #     if q:
            #         union_list.append(q)
            cdr_date = "%s%02d%02d" % (day_cdr.year, day_cdr.month, day_cdr.day)
            cdr = self.get_daily_cdr(cdr_date)
            if cdr is None:
                log.warning('missing table {}'.format(cdr_date))
                continue
            elif hasattr(self, 'ya_query'):
                self.ya_query(fields, day_cdr, day_cdr, filtering)

            _select = []
            for k in fields:
                # if hasattr(self, '_on_select_from_did_charge_' + k):
                #     _f = getattr(self, '_on_select_from_did_charge_' + k)(cdr, day_cdr, filtering)
                #     if _f is not None:
                #         _select.append(_f.label(k))
                #     continue
                keys = [x.key for x in _select]
                if k in keys:  # already here
                    continue
                # k=k.replace('non_zero_calls','not_zero_calls')
                if k in ('asr', 'qsr', 'ingress_npr', 'egress_npr'):
                    keys = [x.key for x in _select]
                    for df in ['ingress_busy_calls', 'ingress_cancel_calls', 'not_zero_calls',
                               'ingress_total_calls', 'egress_total_calls', 'npr_count']:
                        if not df in keys:
                            _select.append(getattr(cdr.c, df).label(df))
                    continue
                if k == self.time_field_str():
                    _select.append(self.time_field(cdr))
                    continue
                if hasattr(self, 'on_select_' + k):
                    _f = getattr(self, 'on_select_' + k)(cdr)
                    if _f is not None:
                        _select.append(_f.label(k))
                    continue
                elif hasattr(cdr.c, k):
                    _select.append(getattr(cdr.c, k).label(k))
                # elif k in ('ingress_name', 'egress_name'):
                #     _select.append(_aggr(cdr, k, method))
                else:
                    raise ValidationError('{}:no such field to select.{}'.format(k, similar_fields(k, columns)))
                    # _select.append(getattr(cdr.c, k))
            # _select = tuple(_select)
            _filter = []
            if day_idx == 0:
                _filter = [self.time_field(cdr) >= start_date_time]
                if day_idx == days:
                    _filter = [self.time_field(cdr) >= start_date_time, self.time_field(cdr) <= end_date_time]
            else:
                if day_idx == days:
                    _filter = [self.time_field(cdr) <= end_date_time]
            for k, v in filtering.items():
                if hasattr(self, 'on_filter_' + k):
                    _f = getattr(self, 'on_filter_' + k)(v, cdr)
                    if _f is not None:
                        _filter.append(_f)
                    continue
                if not k in columns:
                    raise ValidationError('{}:no such field to filter.{}'.format(k, similar_fields(k, columns)))
                if hasattr(cdr.c, k):
                    if '*' in v:
                        _filter.append(getattr(cdr.c, k).like(v.replace('*', '%')))
                    else:
                        _filter.append(getattr(cdr.c, k) == v)
            if 'agent_id' in groups:
                cls = model.AgentClient
                agent_clients = [agent_client.client_id for agent_client in cls.filter().all()]
                _filter.append(or_(cdr.c.ingress_client_id.in_(agent_clients), cdr.c.egress_client_id.in_(agent_clients)))
            if user and user.outbound_report:
                cls = model.UsersLimit
                limits = [client_id for client_id, in cls.session().query(cls.client_id).filter(cls.user_id == user.user_id).all()]
                if limits:
                    _filter.append(cdr.c.ingress_client_id.in_(limits))

            q = model.get_db().session.query(*_select)
            if _filter:
                q = q.filter(and_(*_filter))
            union_list.append(q)
        # if hasattr(self,'ya_query'):
        #     q=self.ya_query(fields,start_day,end_day,filtering)
        #     if q:
        #         union_list.append(q)
        import traceback
        log.debug("Aggregate Function called. Stack trace:\n%s", traceback.format_stack())
        if union_list:
            union_query = union_list[0].union_all(*(union_list[1:]))
        if union_query:
            query_tab = alias(union_query.subquery(), '_all_cdr_')
            self.model_class._tab = query_tab
            _select = []
            q_tab = query_tab

            def _aggr(cdr_table, field_name, agg_method=None, need_label=True):
                fd = {'total': func.sum, 'min': func.min, 'max': func.max, 'avrg': func.avg, }
                if field_name in ('ingress_name', 'egress_name', 'ingress_client_name', 'egress_client_name', 'release_cause_name', 'agent_name', 'egress_client_id', 'ingress_client_id', 'start_time_of_date'):
                    if field_name == 'ingress_name':
                        ing = aliased(model.Resource)
                        return select([ing.alias]).where(ing.resource_id == cdr_table.c.ingress_id).as_scalar().label('ingress_name')
                    if field_name == 'egress_name':
                        ing = aliased(model.Resource)
                        log.debug("return select([ing.alias]).where(ing.resource_id == cdr_table.c.egress_id).as_scalar().label('egress_name')")
                        return select([ing.alias]).where(ing.resource_id == cdr_table.c.egress_id).as_scalar().label('egress_name')
                    if field_name == 'ingress_client_name':
                        ing = aliased(model.Client)
                        return select([ing.name]).where(ing.client_id == cdr_table.c.ingress_client_id).as_scalar().label(
                                'ingress_client_name')
                    if field_name == 'ingress_client_id':
                        ing = aliased(model.Client)
                        return select([ing.client_id]).where(ing.client_id == cdr_table.c.ingress_client_id).as_scalar().label(
                                'ingress_client_id'
                        )
                    if field_name == 'egress_client_name':
                        ing = aliased(model.Client)
                        return select([ing.name]).where(ing.client_id == cdr_table.c.egress_client_id).as_scalar().label(
                                'egress_client_name')
                    if field_name == 'egress_client_id':
                        ing = aliased(model.Client)
                        return select([ing.client_id]).where(
                            ing.client_id == cdr_table.c.egress_client_id).as_scalar().label(
                            'egress_client_id'
                        )
                    if field_name == 'release_cause_name':
                        cls = aliased(model.GlobalRouteError)
                        return select([cls.error_description]).where(cls.error_code == cdr_table.c.release_cause).as_scalar().label(
                                'release_cause_name')
                    if field_name == 'agent_name':
                        cls = aliased(model.Agent)
                        return select([cls.agent_name]).where(cls.agent_id == cdr_table.c.agent_id).as_scalar().label(
                                'agent_name')
                    if field_name == 'start_time_of_date':
                        subquery = select([cdr_table.c.start_time_of_date]).order_by(cdr_table.c.start_time_of_date).limit(1)
                        return subquery.as_scalar().label('start_time_of_date') if need_label else subquery.as_scalar()                 
                if agg_method and (field_name in aggregable or hasattr(self,
                                                                        'on_select_from_did_charge_' + field_name)):
                    if agg_method not in fd:
                        raise ValidationError(
                            '{}:no such method.  Allowed values :{}'.format(k, similar_fields(agg_method, fd.keys())))
                    f = fd[agg_method]
                    tf = f(func.coalesce(getattr(cdr_table.c, field_name), 0.0))
                    # tf = f(getattr(cdr_table.c, field_name))
                    if need_label:
                        return tf.label(field_name)
                    else:
                        return tf
                else:
                    tf = getattr(cdr_table.c, field_name)
                    if need_label:
                        return tf.label(field_name)
                    else:
                        return tf

            for k in fields:
                if k == 'asr':
                    _select.append(func.round(
                        100.0 * _aggr(q_tab, 'not_zero_calls', 'total') / _aggr(q_tab, 'ingress_total_calls', 'total'),
                        2).label('asr'))
                    continue
                if k == 'qsr':
                    _select.append(func.round(100.0 * (
                            _aggr(q_tab, 'not_zero_calls', 'total') + _aggr(q_tab, 'ingress_busy_calls',
                                                                            'total') + _aggr(q_tab,
                                                                                             'ingress_cancel_calls',
                                                                                             'total')) / _aggr(
                        q_tab, 'ingress_total_calls', 'total'), 2).label('qsr'))
                    continue
                if k == 'ingress_npr':
                    _select.append(func.round(100.0 * (_aggr(q_tab, 'npr_count', 'total')) / _aggr(
                        q_tab, 'ingress_total_calls', 'total'), 2).label('ingress_npr'))
                    continue
                if k == 'egress_npr':
                    _select.append(func.round(100.0 * (_aggr(q_tab, 'npr_count', 'total')) / _aggr(
                        q_tab, 'egress_total_calls', 'total'), 2).label('egress_npr'))
                    continue
                if k in ['ani_in_spam', 'ani_in_fraud', 'ani_in_tcpa', 'dnis_in_dnc']:
                    col = getattr(q_tab.c, k)
                    _select.append(
                        func.max(
                            case(
                                [(col == 1, 1), (col == 0, 0)],
                                else_=None
                            )
                        ).label(k)
                    )
                if hasattr(q_tab.c, k):
                    if not (hasattr(self, 'on_select_' + k) or hasattr(self,
                                                                       'on_select_from_did_charge_' + k)) and k not in aggregable:
                        continue
                    _select.append(_aggr(q_tab, k, method))
            # _select = tuple(_select)
            _group = []
            new_time_field = None
            my_step = 0
            if step:
                step_dict = {'hour': 60 * 60,
                             'day': 60 * 60 * 24,
                             'week': 60 * 60 * 24 * 7,
                             'year': 60 * 60 * 24 * 365,
                             'month': 60 * 60 * 24 * 30,
                             'all': 0}
                if step in step_dict:
                    my_step = step_dict[step]
                else:
                    my_step = int(step) * 60
            if my_step:
                # .op('AT TIME ZONE')(tz)
                # new_time_field = (my_step * (cast(func.extract('epoch', self.time_field(q_tab)), Integer) / my_step))
                if step in step_dict:
                    new_time_field = case(
                        [
                            (cast(func.extract('epoch', func.date_trunc(step, self.time_field(q_tab))), Integer) < start_time,
                             start_time)
                        ],
                        else_=cast(func.extract('epoch', func.date_trunc(step, self.time_field(q_tab))), Integer)
                    )
                else:
                    new_time_field = (
                            my_step * (cast(func.extract('epoch', self.time_field(q_tab)), Integer) / my_step))
                _select.append(new_time_field.label(self.time_field_str()))
                _group.append(new_time_field)
            else:
                _group.append(cast(func.extract('millennium', self.time_field(q_tab)), Integer))

            if groups:
                _group = _group + list(map(lambda k: getattr(q_tab.c, k), groups))
                _select = _select + list(map(lambda k: getattr(q_tab.c, k), groups))

            self.modify_select_from_groupping(_select, groups, q_tab)

            query = model.get_db().session.query(*_select)

            if _group:
                query = query.group_by(*tuple(_group))
            if sort_order is None:
                if new_time_field is not None:
                    query = query.order_by(new_time_field)
            else:
                order_by_expr = tuple(
                    map(lambda i:
                        (_aggr(q_tab, i[0], method, False) if i[0] != self.time_field_str() else new_time_field).desc() 
                        if i[1] == 'desc' else 
                        _aggr(q_tab, i[0],method, False) if i[0] != self.time_field_str() else new_time_field,
                        sort_order))
                query = query.order_by(*order_by_expr)
            # return filtering, query
            if REPORT_DEBUG:
                log.debug(model.query_to_sting(query))

            if 'release_cause_from_protocol_stack' in groups:
                query = query.filter(text('release_cause_from_protocol_stack IS NOT null'))

            # if self.model_class == model.DidReport:
            #     for i in fields:
            #         if not i in ['mrc', 'nrc', 'did', 'subcharge']:
            #             query = query.filter(text('{field_name} IS NOT null'.format(field_name=i)))
            #             break

            return {}, query
        else:
            return None, None
        pass

    def modify_select_from_groupping(self, _select, groups, q_tab):
        if 'ingress_id' in groups:
            ing = aliased(model.Resource)
            _select.append(
                select([ing.alias]).where(ing.resource_id == q_tab.c.ingress_id).as_scalar().label('ingress_name'))
        if 'egress_id' in groups:
            log.debug("egress_id on groups")
            ing = aliased(model.Resource)
            log.debug("SELECT APPEND")
            _select.append(
                select([ing.alias]).where(ing.resource_id == q_tab.c.egress_id).as_scalar().label('egress_name'))
            log.debug("APPENDED")
        if 'ingress_client_id' in groups:
            cl = model.Client
            ing = aliased(model.t_client_record)
            _select.append(func.coalesce(
                select([ing.c.name]).where(and_(ing.c.client_id == q_tab.c.ingress_client_id,
                                                ing.c.flag == 'A')).order_by(ing.c.time.desc()).limit(1).as_scalar(),
                select([cl.name]).where(cl.client_id == q_tab.c.ingress_client_id).as_scalar()).label('ingress_client_name'))
        if 'egress_client_id' in groups:
            cl = model.Client
            ing = aliased(model.t_client_record)
            _select.append(func.coalesce(
                select([ing.c.name]).where(and_(ing.c.client_id == q_tab.c.egress_client_id,
                                                ing.c.flag == 'A')).order_by(ing.c.time.desc()).limit(1).as_scalar(),
                select([cl.name]).where(cl.client_id == q_tab.c.egress_client_id).as_scalar()).label('egress_client_name'))
        if 'agent_id' in groups:
            ag = model.Agent
            _select.append(
                select([ag.agent_name]).where(ag.agent_id == q_tab.c.agent_id).as_scalar().label('agent_name'))
            # if 'ingress_client_id' in groups:
            #     ag = model.AgentClient
            #     _select.append(
            #         select([ag.commission * (func.sum(q_tab.c.ingress_call_cost) - ((ag.method_type+1)%2)*func.sum(q_tab.c.egress_call_cost))/100]). \
            #             where(and_(ag.agent_id == q_tab.c.agent_id, 
            #             ag.client_id == q_tab.c.ingress_client_id)).as_scalar().label('commission'))
            # else:
            #     ag = model.Agent
            #     _select.append(
            #         select([ag.commission * (func.sum(q_tab.c.ingress_call_cost) - ((ag.method_type+1)%2)*func.sum(q_tab.c.egress_call_cost))/100]). \
            #             where(and_(ag.agent_id == q_tab.c.agent_id)).as_scalar().label('commission'))
        if 'release_cause' in groups:
            cls = aliased(model.GlobalRouteError)
            _select.append(
                select([cls.error_description]).where(cls.error_code == q_tab.c.release_cause).as_scalar().label(
                    'release_cause_name'))


class ReportSimple(DnlList):
    entity_plural = 'Report items'
    scheme_class = CdrReportGroppedRequestScheme
    scheme_class_get = CdrReportGroppedResponseScheme
    scheme_class_get_inner = CdrReportDetailInnerScheme
    model_class = model.CdrReportDetail
    columns = model_class.get_fields()

    groupable = list(set([fn for fn in columns if
                 getattr(model.CdrReportDetail, fn).type.__class__.__name__ in (
                     'String', 'DateTime', 'Text', 'Boolean') or '_id' in fn or fn in _grp] +
                        ['origination_remote_payload_ip_address', 'termination_remote_payload_ip_address',
                         'ingress_name', 'egress_name', 'ani_in_dno', 'ani_in_dno_count', 'ani_in_ftc_count',
                         'ani_in_ftc_week_count', 'ani_in_ftc_month_count', 'ani_in_spam_count',
                         'ani_in_fraud_count', 'ani_in_tcpa_count', 'dnis_in_dnc_count']))
    group_mapping = {
        'ingress_name': 'ingress_id',
        'egress_name': 'egress_id',
    }
    aggregable = list(set([fn for fn in columns if
                           getattr(model.CdrReportDetail, fn).type.__class__.__name__ not in (
                               'String', 'DateTime', 'Text', 'Boolean') and '_id' not in fn and fn not in _grp] +
                          ['nrf_calls', 'asr', 'qsr', 'ingress_npr', 'egress_npr', 'SDP30_calls', 'SDP6_calls',
                           'ingress_call_dur', 'egress_call_dur', 'ingress_bill_dur', 'egress_bill_dur',
                           'not_zero_calls_12', 'not_zero_calls_18', 'not_zero_calls_24', 'commission', 'limited']))

    def get_spec(self):
        sfields_names, search_fields_list = self.get_fields_from_scheme(self.scheme_class)
        qfields_names, query_fields_list = self.get_query_fields_from_scheme(self.scheme_class)
        fields_names = [f.name for f in inspect(self.model_class).columns]
        spec = swagger.specify.get_spec(
            method='get', description='Gets {}'.format(self.entity_plural.lower()),
            path_parameters=self.path_parameters,
            query_parameters=search_fields_list + [
                {"name": "per_page", "description": "printing range. Default: print 1000 items",
                 "type": "integer"},
                {"name": "page", "description": "printing range. Default: print 1st page",
                 "type": "integer"},
                {"name": "fields",
                 "description": "aggregated fields, available values:{}".format(self.aggregable)},
                {"name": "group",
                 "description": "groupping fields, available values:{}".format(self.groupable)},

            ] +
                             [dict(name=i, type='string') for i in self.groupable],
            responses=(
                          SuccessReportResponse(),
                          responses.AttributeNotExitsErrorResponse()
                      ) + self.get_additional_responses(method='get'),
            security=self.get_security(method='get')
        )
        spec['get']['produces'] = ['application/json', 'text/csv', 'text/xml']
        return spec

    def run(self, **kwargs):
        class Req:
            def __init__(self, req_user):
                self.query_string = None
                self.context = {'user': req_user}

            pass

        if 'user' in kwargs:
            user = kwargs['user']
            del kwargs['user']
        else:
            user = model.User.get(1)
        req = Req(user)
        req.query_string = urlencode(kwargs)
        self.req = req
        objects_list, total, page, per_page, fields = self.get_objects_list(req, self.model_class, **kwargs)
        # items = [i for i in objects_list]
        items = self.scheme_class_get_inner().dump(objects_list, many=True).data
        return items

    def send_request(self, params):
        m = dict(calls='ingress_total_calls',
                 ingress_billed_time='ingress_bill_time',
                 ingress_cost='ingress_call_cost',
                 ingress_billed_min='ingress_bill_time',
                 code='ingress_code',
                 non_zero_calls='not_zero_calls',
                 egress_cost='egress_total_cost',
                 egress_billed_time='egress_bill_time',
                 )
        args = {k: v for k, v in params.items() if k not in m}
        upd = {m[k]: v for k, v in params.items() if k in m}
        args.update(upd)

        class Responce:
            def json(self):
                return self.data

        items = self.run(**args)
        resp = Responce()
        resp.data = dict(data=items)
        return resp

    def _on_get(self, req, resp, **kwargs):
        from falcon_rest import schemes
        from falcon_rest.helpers import check_permission

        if (not hasattr(self, 'no_auth_needed') or self.no_auth_needed == False) and not check_permission(self, req,
                                                                                                          'list'):
            self.set_response(
                resp, responses.ForbiddenErrorResponse(data=errors.AuthErrors.Forbidden)
            )
            return
        try:
            import time
            _start = time.time()
            objects_list, total, _from, count, fields = self.get_objects_list(req, self.model_class, **kwargs)
            # items = [i for i in objects_list]
            items = self.scheme_class_get_inner().dump(objects_list, many=True).data
            data = {
                'success': True,
                'data': items,
                'total': total, 'from': _from, 'count': len(items)
            }
            self.set_response(resp, SuccessReportResponse(data=data, scheme=self.scheme_class_get))

            _end = time.time()
            log.debug('report execution time {}'.format(_end - _start))
            # resp.body = json.dumps(data)
            # resp.status = falcon.HTTP_200

        # except AttributeError as e:
        #     self.set_response(
        #         resp, responses.AttributeNotExitsErrorResponse(
        #             data=ATTRIBUTE_ERROR_RE.search(str(e)).group(1)
        #         )
        #     )
        except DataError as e:
            import traceback
            log.error(traceback.format_exc())
            self.set_response(resp, SuccessReportResponse(data={'message':e}))
        except (InternalError, IntegrityError, ) as e:
            import traceback
            model.get_db().session.rollback()
            log.error(traceback.format_exc())
            self.set_response(resp, OperationErrorResponse(e))
        except Exception as e:
            import traceback
            log.error(traceback.format_exc())
            self.set_response(resp, OperationErrorResponse(e))

    def time_field(self, cdr):
        return cdr.c.report_time.label('report_time')

    def time_field_str(self):
        return 'report_time'

    def get_daily_cdr(self, cdr_date):
        cdr = model.cdr_report_detail(cdr_date)
        return cdr

    def get_fields_for_list(self, parsed_qs, **kwargs):
        fields = parsed_qs.get('fields', None)
        if fields:
            fields = [field.strip() for field in fields.split(',')]
            for f in fields:
                # if f not in self.get_all_fields_from_scheme(self.scheme_class_get)[0]:
                if f not in self.aggregable and f not in self.columns and f not in (self.time_field_str(), 'time'):
                    raise Exception('fields {} not valid!'.format(f))
        return fields

    def get_objects_list(self, req, model_class, **kwargs):
        parsed_qs = dict(parse_qsl(req.query_string))
        if REPORT_DEBUG:
            log.debug('report parsed parameters:{}'.format(parsed_qs))
        try:
            per_page = int(parsed_qs.get('per_page', 1000000))
        except ValueError:
            per_page = 1000000

        try:
            page = int(parsed_qs.get('page'))
        except (ValueError, TypeError):
            page = 0


        ordering = {}
        log.debug("GETTING FIELDS")
        fields = self.get_fields_for_list(parsed_qs, **kwargs)
        user = self.get_user(req) 
        if user.agent and user.agent.method_type == "By Revenue":
            fields = [field for field in fields if field != "egress_call_cost"]
        log.debug("GETTING FILTERING")
        filtering = self.get_filtering_for_list(parsed_qs, **kwargs)
        if fields:
            filtering['fields'] = ','.join(fields)
        log.debug("GETTING FILTERING AND QUERY")
        filtering, query = \
            self.modify_query_from_filtering_for_list(filtering, **kwargs)

        # for c in inspect(self.model_class).columns:
        #     if c.name in ['alias','name','template_name','group_name']:
        #         query = query.filter(or_(c.is_(None),c.notlike('#%')))
        #         break

        if ordering:
            log.debug("Modifying query")
            ordering, query = \
                self.modify_query_from_ordering_for_list(ordering, query, **kwargs)
        if query is None:
            return [], 0, page, per_page, fields
        log.debug(f"QUERY = {query}")
        log.debug(f"FILTERING = {filtering}")
        log.debug(f"ORDERING = {ordering}")
        return model_class.get_objects_list(
            query=query,
            filtering=filtering,
            ordering=ordering,
            paging={'from': page * per_page, 'till': (page + 1) * per_page}
        ) + (page, per_page, fields)

    def on_filter_agent_id(self, v, cdr):
        cls = model.AgentClient
        clients = cls.session().query(cls.client_id).filter(cls.agent_id == v).all()
        return cdr.c.ingress_client_id.in_([int(i.client_id) for i in clients])

    def on_filter_o_trunk_type2(self, v, cdr):
        cls = model.Client
        clients = cls.session().query(cls.client_id).filter(cls.company_type==v).all()
        return cdr.c.ingress_client_id.in_([int(i.client_id) for i in clients])

    def on_filter_trunk_type2(self, v, cdr):
        cls = model.Resource
        ingress = v == 0
        resources = cls.session().query(cls.resource_id).filter(and_(cls.trunk_type2==v, cls.ingress==ingress)).all()
        return cdr.c.ingress_id.in_([int(i.resource_id) for i in resources])

    def on_filter_ingress_id(self, v, cdr):
        # return text("ingress_id in ({})".format(v))
        return cdr.c.ingress_id.in_([int(i) for i in v.split(',')])

    def on_filter_egress_id(self, v, cdr):
        # return text("egress_id in ({})".format(v))
        return cdr.c.egress_id.in_([int(i) for i in v.split(',')])

    def on_filter_ingress_client_id(self, v, cdr):
        # return text("egress_id in ({})".format(v))
        return cdr.c.ingress_client_id.in_([int(i) for i in v.split(',')])

    def on_filter_egress_client_id(self, v, cdr):
        # return text("egress_id in ({})".format(v))
        return cdr.c.egress_client_id.in_([int(i) for i in v.split(',')])

    def on_filter_non_zero_calls(self, v, cdr):
        # return text("egress_id in ({})".format(v))
        if v == '1':
            return cdr.c.ingress_not_zero_calls > 0

    def on_filter_scc_below_sec(self, v, cdr):
        return cdr.c.duration < v

    def on_filter_is_hide_unauthorized(self, v, cdr):
        if v is None:
            return None
        if v:
            return cdr.c.release_cause != 3
        else:
            return cdr.c.release_cause == 3

    def on_select_ingress_call_dur(self, cdr):
        return 1.0 * cdr.c.ingress_duration / 60

    def on_select_egress_call_dur(self, cdr):
        return 1.0 * cdr.c.duration / 60

    def on_select_ingress_bill_dur(self, cdr):
        return cdr.c.ingress_bill_time / 60

    def on_select_egress_bill_dur(self, cdr):
        return cdr.c.egress_bill_time / 60

    def on_select_nrf_calls(self, cdr):
        return cdr.c.lrn_calls

    def on_select_asr(self, cdr):
        return 100.00 * ((1.0 * cdr.c.not_zero_calls) / (1.0 * cdr.c.ingress_total_calls))

    def on_select_qsr(self, cdr):
        return 100.00 * ((cdr.c.not_zero_calls + cdr.c.ingress_busy_calls + cdr.c.ingress_cancel_calls) /
                         cdr.c.ingress_total_calls)

    def on_select_ingress_npr(self, cdr):
        return 100.00 * ((1.0 * cdr.c.npr_count) / (1.0 * cdr.c.ingress_total_calls))

    def on_select_egress_npr(self, cdr):
        return 100.00 * ((1.0 * cdr.c.npr_count) / (1.0 * cdr.c.egress_total_calls))

    def on_select_SDP30_calls(self, cdr):
        return cdr.c.not_zero_calls_6 + cdr.c.call_12s + cdr.c.call_18s + cdr.c.call_24s + cdr.c.not_zero_calls_30

    def on_select_not_zero_calls_12(self, cdr):
        return cdr.c.call_12s

    def on_select_not_zero_calls_18(self, cdr):
        return cdr.c.call_18s

    def on_select_not_zero_calls_24(self, cdr):
        return cdr.c.call_24s

    def on_select_SDP6_calls(self, cdr):
        return cdr.c.not_zero_calls_6

    def on_select_ACD(self, cdr):
        return ((1.0 * cdr.c.egress_bill_time) / (1.0 * cdr.c.not_zero_calls))

    def on_select_agent_id(self, cdr):
        cls = model.AgentClient
        return select([cls.agent_id]).where(cls.client_id == cdr.c.ingress_client_id)
    
    def on_select_ftc_ani(self, cdr):
        return 1 if cdr.c.ani_in_ftc else 0
    
    def on_select_non_ftc_ani(self, cdr):
        return 0 if cdr.c.ani_in_ftc else 1
    
    def on_select_week_ftc_ani(self, cdr):
        return 1 if cdr.c.ani_in_week_ftc else 0
    
    def on_select_non_week_ftc_ani(self, cdr):
        return 0 if cdr.c.ani_in_week_ftc else 1
    
    def on_select_month_ftc_ani(self, cdr):
        return 1 if cdr.c.ani_in_month_ftc else 0
    
    def on_select_non_month_ftc_ani(self, cdr):
        return 0 if cdr.c.ani_in_month_ftc else 1
    
    def on_select_dno_ani(self, cdr):
        return 1 if cdr.c.ani_in_dno else 0
    
    def on_select_non_dno_ani(self, cdr):
        return 0 if cdr.c.ani_in_dno else 1
    
    def on_select_spam_ani(self, cdr):
        return 1 if cdr.c.spam_ani else 0
    
    def on_select_non_spam_ani(self, cdr):
        return 0 if cdr.c.spam_ani else 1
    
    def on_select_fraud_ani(self, cdr):
        return 1 if cdr.c.fraud_ani else 0
    
    def on_select_non_fraud_ani(self, cdr):
        return 0 if cdr.c.fraud_ani else 1
    
    def on_select_tcpa_ani(self, cdr):
        return 1 if cdr.c.tcpa_ani else 0
    
    def on_select_non_tcpa_ani(self, cdr):
        return 0 if cdr.c.tcpa_ani else 1

    def on_filter_show_non_zero_only(self, v, cdr):
        if v == 'true':
            return cdr.c.duration > 0
        else:
            return cdr.c.duration == 0

    def on_select_commission(self, cdr):
        cls = model.AgentClient
        return select([cls.commission * (cdr.c.ingress_call_cost - ((cls.method_type+1)%2)*cdr.c.egress_call_cost)/100]). \
                    where(cls.client_id == cdr.c.ingress_client_id).as_scalar()

    def on_select_limited(self, cdr):
        return func.cast(cdr.c.release_cause.in_([1, 2, 6, 7, 8, 9, 29, 30, 45, 46, 50, 51, 52, 53, 56, 65, 66, 67, 68, 69]), Integer)

    def modify_query_from_filtering_for_list(self, filtering, **kwargs):

        def similar_fields(field, field_list):
            sf = field.split('_')
            sim = []
            for f in sf:
                for ef in field_list:
                    if f in ef:
                        sim.append(ef)
            if sim:
                return 'similar fields:{}'.format(set(sim))
            return ''

        import sys
        log.debug("SETTING RECURSION LIMIT")
        sys.setrecursionlimit(10000)
        cls = self.model_class
        log.debug(f"MODEL - {cls}")
        columns = cls.get_fields()
        log.debug("GOT COLUMNS")
        # columns = [f.name for f in inspect(self.model_class).columns]
        req_query = dict(parse_qsl(self.req.query_string))
        _from = filtering.pop('from', None)
        count = filtering.pop('count', None)

        sort = filtering.pop('sort', None)
        order = filtering.pop('order', '')
        groups = filtering.pop('group', None)

        sort_order = None
        if sort:
            log.debug("SORT")
            if sort in ('report_time', 'time'):
                sort = self.time_field_str()
            _sort = sort.split(',')
            for s in _sort:
                log.debug(f"SORT - {s}")
                if s not in columns and not s.replace('name', 'id') in columns and not hasattr(self, 'on_select_' + s):
                    raise ValidationError('sort field missing {}'.format(s))
                if s.replace('name', 'id') in self.groupable and s != self.time_field_str():
                    if groups:
                        groups += ',' + s.replace('name', 'id')
                    else:
                        groups = s.replace('name', 'id')
            _order = order.split(',')
            for o in _order:
                log.debug(f"ORDER - {o}")
                if o not in ('', 'asc', 'desc'):
                    raise ValidationError('sort order invalid value {}, must be asc or desc (can be empty)'.format(o))
            if len(_sort) != len(_order):
                if sort != '':
                    raise ValidationError('sort length must be equal order length')
                else:
                    _order = ['' for s in _sort]
            sort_order = list(zip(_sort, _order))
        log.debug("GETTING START TIME AND END TIME")
        start_time = filtering.pop('start_time', None)
        end_time = filtering.pop('end_time', str(int(datetime.now(UTC).timestamp())))
        try:
            start_time = int(start_time)
            if start_time < 0:
                start_time = int(datetime.now(UTC).timestamp()) + start_time
            start_date_time = datetime.fromtimestamp(start_time, tz=timezone.utc)
        except (ValueError, TypeError):
            raise ValidationError('wrong start time {}'.format(start_time))
        try:
            end_time = int(end_time)
            end_date_time = datetime.fromtimestamp(end_time, tz=timezone.utc)
        except (ValueError, TypeError):
            raise ValidationError('wrong end time')
        if start_time >= end_time:
            raise ValidationError('start time greater than end_time!')
        # aggregable = ['agent_cost', 'agent_rate', 'call_12s', 'call_18s', 'call_24s', 'call_2h', 'call_3h', 'call_4h',
        #               'duration', 'duration_30', 'duration_6', 'egress_bill_time', 'egress_bill_time_inter',
        #               'egress_bill_time_intra', 'egress_busy_calls', 'egress_call_cost', 'egress_call_cost_ij',
        #               'egress_call_cost_inter', 'egress_call_cost_intra', 'egress_call_cost_local',
        #               'egress_cancel_calls', 'egress_no_ring', 'egress_rate', 'egress_rate_date',
        #               'egress_success_calls', 'egress_total_calls', 'incoming_bandwidth', 'ingress_bill_time',
        #               'ingress_bill_time_inter', 'ingress_bill_time_intra', 'ingress_busy_calls', 'ingress_call_12s',
        #               'ingress_call_18s', 'ingress_call_24s', 'ingress_call_2h', 'ingress_call_3h', 'ingress_call_4h',
        #               'ingress_call_cost', 'ingress_call_cost_ij', 'ingress_call_cost_inter', 'ingress_call_cost_intra',
        #               'ingress_call_cost_local', 'ingress_cancel_calls', 'ingress_duration', 'ingress_duration_30',
        #               'ingress_duration_6', 'ingress_inter_duration', 'ingress_inter_not_zero_calls',
        #               'ingress_intra_duration', 'ingress_intra_not_zero_calls', 'ingress_not_zero_calls',
        #               'ingress_not_zero_calls_30', 'ingress_not_zero_calls_6', 'ingress_rate', 'ingress_rate_date',
        #               'ingress_success_calls', 'ingress_total_calls', 'inter_duration', 'inter_ingress_total_calls',
        #               'inter_not_zero_calls', 'intra_duration', 'intra_ingress_total_calls', 'intra_not_zero_calls',
        #               'lnp_cost', 'lrn_calls', 'not_zero_calls', 'not_zero_calls_30', 'not_zero_calls_6', 'npr_count',
        #               # 'orig_jur_type',
        #               'outgoing_bandwidth', 'pdd', 'q850_cause_count',
        #               # 'release_cause',
        #               'ring_pdd',
        #               # 'term_jur_type'
        #               ]

        # #groupable = [fn for fn in columns if
        #              getattr(cls, fn).type.__class__.__name__ in ('String', 'DateTime') or '_id' in fn or fn in (
        #                  'orig_jur_type', 'release_cause', 'term_jur_type')]
        # #aggregable = [fn for fn in columns if fn not in groupable]
        groupable = self.groupable
        aggregable = self.aggregable
        group_mapping = self.group_mapping
        log.debug(f"groupable = {groupable} | aggregable = {aggregable}")
        tz = filtering.pop('tz', None)
        if REPORT_DEBUG:
            log.debug('tz={}'.format(tz))
        non_zero_calls = filtering.pop('non_zero_calls', None)
        if REPORT_DEBUG:
            log.debug('non_zero_calls={}'.format(non_zero_calls))
        human_readable = filtering.pop('human_readable', None)
        if REPORT_DEBUG:
            log.debug('human_readable={}'.format(human_readable))
        log.debug("getting data")
        method = filtering.pop('method', 'total')
        step = filtering.pop('step', 'all')
        fields = filtering.pop('fields', None)
        field = filtering.pop('field', None)
        fields = fields or field
        log.debug(f"FIELDS = {fields}")
        if fields:
            fields = fields.split(',')
        if not fields:
            fields = aggregable.copy()
        for fk in filtering.keys():
            log.debug(f"FK = {fk}")
            if fk in groupable and not hasattr(self, 'on_filter_{}'.format(fk)):
                if groups:
                    groups += ',{}'.format(fk)
                else:
                    groups = fk
            if fk not in columns:
                if not hasattr(self, 'on_filter_{}'.format(fk)):
                    raise ValidationError('{}:wrong filter field. {}'.format(fk, similar_fields(fk, columns)))
            if fk not in fields and not hasattr(self, 'on_filter_{}'.format(fk)):
                fields.append(fk)

        if groups is not None:
            # groups = groups.replace('_name', '_id') #.replace('egress_name', 'egress_id')
            for field_id, field_name in group_mapping.items():
                groups = groups.replace(field_id, field_name)
            groups = [i.strip() for i in groups.split(',')]
        else:
            groups = []
        if self.time_field_str() in groups:
            step = 60
            groups.remove(self.time_field_str())
        for fk in groups:
            if fk not in groupable:
                raise ValidationError('{}:wrong group field. {}'.format(fk, similar_fields(fk, groupable)))
            if fk not in fields:
                fields.append(fk)
        if sort:
            for fk in sort.split(','):
                if fk not in fields and fk not in ('ingress_name', 'egress_name'):
                    fields.append(fk)

        fields.append(self.time_field_str())
        fields = list(set(fields))

        user = self.get_user(self.req)

        if user and user.user_type in ('client', 'vendor'):
            # if user.user_type == 'client' and user.client.company_type == 'termination' and ('egress_id' in filtering):
            #     raise ValidationError('Term client can\'t query the Reports using the egress_id!')
            # if user.user_type == 'client' and user.client.company_type == 'origination' and ('ingress_id' in filtering):
            #     raise ValidationError('DID client can\'t query the Reports using the ingress_id!')

            if 'ingress_id' in filtering and filtering['ingress_id']:
                ids = set(filtering['ingress_id'].split(','))
                alw = set([str(i) for i in user.allowed_ingress_id])
                filtering['ingress_id'] = ','.join(list(ids.intersection(alw)))
            else:
                filtering['ingress_id'] = ','.join([str(i) for i in user.allowed_ingress_id])
            if filtering['ingress_id'] == '' or (user.user_type == 'client' and user.client.company_type == 'origination'):
                filtering['ingress_id'] = ''
                if user.allowed_egress_id:
                    alw = set([str(i) for i in user.allowed_egress_id])
                    if 'egress_id' in filtering and filtering['egress_id']:
                        log.debug("EGRESS_ID IN FILTERING")
                        ids = set(filtering['egress_id'].split(','))
                        filtering['egress_id'] = ','.join(list(ids.intersection(alw)))
                        log.debug("SET FILTERING")
                    else:
                        filtering['egress_id'] = ','.join(list(alw))
                        log.debug("SET FILTERING")
                    del filtering['ingress_id']
                else:
                    filtering['ingress_id'] = '-1'
            if REPORT_DEBUG:
                log.debug('result {} filtering={}'.format(user.user_type, filtering))
        elif user and user.user_type == 'agent':
            filtering['agent_id'] = user.agent.agent_id

        start_day = start_date_time.date()
        database_start_day = model.cdr_reports_start_date()
        if database_start_day is None:
            raise ValidationError('database has no cdr report data!')
        if database_start_day > start_day:
            start_day = database_start_day
        end_day = end_date_time.date()
        if end_day < start_day:
            end_day = start_day

        days = (end_day - start_day).days
        union_query = None
        union_list = []
        for day_idx in range(days + 1):

            day_cdr = start_day + timedelta(days=day_idx)
            # if hasattr(self, 'ya_query'):
            #     q = self.ya_query(fields, day_cdr, day_cdr, filtering)
            #     if q:
            #         union_list.append(q)
            cdr_date = "%s%02d%02d" % (day_cdr.year, day_cdr.month, day_cdr.day)
            cdr = self.get_daily_cdr(cdr_date)
            if cdr is None:
                log.warning('missing table {}'.format(cdr_date))
                continue
            elif hasattr(self, 'ya_query'):
                self.ya_query(fields, day_cdr, day_cdr, filtering)

            _select = []
            for k in fields:
                # if hasattr(self, '_on_select_from_did_charge_' + k):
                #     _f = getattr(self, '_on_select_from_did_charge_' + k)(cdr, day_cdr, filtering)
                #     if _f is not None:
                #         _select.append(_f.label(k))
                #     continue
                keys = [x.key for x in _select]
                if k in keys:  # already here
                    continue
                # k=k.replace('non_zero_calls','not_zero_calls')
                if k in ('asr', 'qsr', 'ingress_npr', 'egress_npr'):
                    keys = [x.key for x in _select]
                    for df in ['ingress_busy_calls', 'ingress_cancel_calls', 'not_zero_calls',
                               'ingress_total_calls', 'egress_total_calls', 'npr_count']:
                        if not df in keys:
                            _select.append(getattr(cdr.c, df).label(df))
                    continue
                if k == self.time_field_str():
                    _select.append(self.time_field(cdr))
                    continue
                if hasattr(self, 'on_select_' + k):
                    _f = getattr(self, 'on_select_' + k)(cdr)
                    if _f is not None:
                        _select.append(_f.label(k))
                    continue
                elif hasattr(cdr.c, k):
                    _select.append(getattr(cdr.c, k).label(k))
                # elif k in ('ingress_name', 'egress_name'):
                #     _select.append(_aggr(cdr, k, method))
                else:
                    raise ValidationError('{}:no such field to select.{}'.format(k, similar_fields(k, columns)))
                    # _select.append(getattr(cdr.c, k))
            # _select = tuple(_select)
            _filter = []
            if day_idx == 0:
                _filter = [self.time_field(cdr) >= start_date_time]
                if day_idx == days:
                    _filter = [self.time_field(cdr) >= start_date_time, self.time_field(cdr) <= end_date_time]
            else:
                if day_idx == days:
                    _filter = [self.time_field(cdr) <= end_date_time]
            for k, v in filtering.items():
                if hasattr(self, 'on_filter_' + k):
                    _f = getattr(self, 'on_filter_' + k)(v, cdr)
                    if _f is not None:
                        _filter.append(_f)
                    continue
                if not k in columns:
                    raise ValidationError('{}:no such field to filter.{}'.format(k, similar_fields(k, columns)))
                if hasattr(cdr.c, k):
                    if '*' in v:
                        _filter.append(getattr(cdr.c, k).like(v.replace('*', '%')))
                    else:
                        _filter.append(getattr(cdr.c, k) == v)
            if 'agent_id' in groups:
                cls = model.AgentClient
                agent_clients = [agent_client.client_id for agent_client in cls.filter().all()]
                _filter.append(or_(cdr.c.ingress_client_id.in_(agent_clients), cdr.c.egress_client_id.in_(agent_clients)))
            if user and user.outbound_report:
                cls = model.UsersLimit
                limits = [client_id for client_id, in cls.session().query(cls.client_id).filter(cls.user_id == user.user_id).all()]
                if limits:
                    _filter.append(cdr.c.ingress_client_id.in_(limits))

            q = model.get_db().session.query(*_select)
            if _filter:
                q = q.filter(and_(*_filter))
            union_list.append(q)
        # if hasattr(self,'ya_query'):
        #     q=self.ya_query(fields,start_day,end_day,filtering)
        #     if q:
        #         union_list.append(q)
        import traceback
        log.debug("Aggregate Function called. Stack trace:\n%s", traceback.format_stack())
        if union_list:
            union_query = union_all(*union_list)
        if union_list:
            query_tab = alias(union_query, '_all_cdr_')
            self.model_class._tab = query_tab
            _select = []
            q_tab = query_tab

            def _aggr(cdr_table, field_name, agg_method=None, need_label=True):
                nonlocal q_tab, _select
                fd = {'total': func.sum, 'min': func.min, 'max': func.max, 'avrg': func.avg, }

                def col_safe(name):
                    if hasattr(cdr_table.c, name):
                        return getattr(cdr_table.c, name)
                    prefixed_name = f"_all_cdr__{name}"
                    if hasattr(cdr_table.c, prefixed_name):
                        return getattr(cdr_table.c, prefixed_name)
                    return None  

                if field_name in ('ingress_name', 'egress_name', 'ingress_client_name', 'egress_client_name', 'release_cause_name', 'agent_name', 'egress_client_id', 'ingress_client_id', 'start_time_of_date'):
                    if field_name == 'ingress_name':          
                        ing = aliased(model.Resource)
                        q_tab = q_tab.outerjoin(ing, ing.resource_id == col_safe("ingress_id"))
                        _select.append(func.max(ing.alias).label('ingress_name'))
                        return func.max(ing.alias).label('ingress_name')
                    if field_name == 'egress_name':
                        egr = aliased(model.Resource)
                        q_tab = q_tab.outerjoin(egr, egr.resource_id == col_safe("egress_id"))
                        _select.append(func.max(egr.alias).label('egress_name'))
                        return func.max(egr.alias).label('egress_name')
                    if field_name == 'ingress_client_name':
                        cli = aliased(model.Client)
                        q_tab = q_tab.outerjoin(cli, cli.client_id == col_safe("ingress_client_id"))
                        _select.append(func.max(cli.name).label('ingress_client_name'))
                        return func.max(cli.name).label('ingress_client_name')
                    if field_name == 'ingress_client_id':
                        cli = aliased(model.Client)
                        q_tab = q_tab.outerjoin(cli, cli.client_id == col_safe("ingress_client_id"))
                        _select.append(func.max(cli.client_id).label('ingress_client_id'))
                        return func.max(cli.client_id).label('ingress_client_id')
                    if field_name == 'egress_client_name':
                        cli = aliased(model.Client)
                        q_tab = q_tab.outerjoin(cli, cli.client_id == col_safe("egress_client_id"))
                        _select.append(func.max(cli.name).label('egress_client_name'))
                        return func.max(cli.name).label('egress_client_name')
                    if field_name == 'egress_client_id':
                        cli = aliased(model.Client)
                        q_tab = q_tab.outerjoin(cli, cli.client_id == col_safe("egress_client_id"))
                        _select.append(func.max(cli.client_id).label('egress_client_id'))
                        return func.max(cli.client_id).label('egress_client_id')
                    if field_name == 'release_cause_name':
                        cls = aliased(model.GlobalRouteError)
                        q_tab = q_tab.outerjoin(cls, cls.error_code == col_safe("release_cause"))
                        _select.append(func.max(cls.error_description).label('release_cause_name'))
                        return func.max(cls.error_description).label('release_cause_name')
                    if field_name == 'agent_name':
                        cls = aliased(model.Agent)
                        q_tab = q_tab.outerjoin(cls, cls.agent_id == col_safe("agent_id"))
                        _select.append(func.max(cls.agent_name).label('agent_name'))
                        return func.max(cls.agent_name).label('agent_name')
                    if field_name == 'start_time_of_date':
                        subquery = select([cdr_table.c.start_time_of_date]).order_by(cdr_table.c.start_time_of_date).limit(1)
                        return subquery.as_scalar().label('start_time_of_date') if need_label else subquery.as_scalar()
                if agg_method and (field_name in aggregable or hasattr(self,
                                                                        'on_select_from_did_charge_' + field_name)):
                    if agg_method not in fd:
                        raise ValidationError(
                            '{}:no such method.  Allowed values :{}'.format(k, similar_fields(agg_method, fd.keys())))
                    f = fd[agg_method]
                    tf = f(func.coalesce(getattr(cdr_table.c, field_name), 0.0))
                    # tf = f(getattr(cdr_table.c, field_name))
                    if need_label:
                        return tf.label(field_name)
                    else:
                        return tf
                else:
                    tf = getattr(cdr_table.c, field_name)
                    if need_label:
                        return tf.label(field_name)
                    else:
                        return tf

            label_map = {}
            for k in fields:
                if k == 'asr':
                    expr = func.round(
                        100.0 * _aggr(q_tab, 'not_zero_calls', 'total') / _aggr(q_tab, 'ingress_total_calls', 'total'),
                        2).label('asr')
                    _select.append(expr)
                    label_map['asr'] = expr
                    continue

                if k == 'qsr':
                    expr = func.round(100.0 * (
                            _aggr(q_tab, 'not_zero_calls', 'total') + _aggr(q_tab, 'ingress_busy_calls',
                                                                            'total') + _aggr(q_tab,
                                                                                             'ingress_cancel_calls',
                                                                                             'total')) / _aggr(
                        q_tab, 'ingress_total_calls', 'total'), 2).label('qsr')
                    _select.append(expr)
                    label_map['qsr'] = expr
                    continue
                if k == 'ingress_npr':
                    expr = func.round(100.0 * (_aggr(q_tab, 'npr_count', 'total')) / _aggr(
                        q_tab, 'ingress_total_calls', 'total'), 2).label('ingress_npr')
                    _select.append(expr)
                    label_map['ingress_npr'] = expr
                    continue
                if k == 'egress_npr':
                    expr = func.round(100.0 * (_aggr(q_tab, 'npr_count', 'total')) / _aggr(
                        q_tab, 'egress_total_calls', 'total'), 2).label('egress_npr')
                    _select.append(expr)
                    label_map['egress_npr'] = expr
                    continue
                if k in ['ani_in_spam', 'ani_in_fraud', 'ani_in_tcpa', 'dnis_in_dnc']:
                    col = getattr(q_tab.c, k)
                    expr = func.max(
                        case(
                            [(col == 1, 1), (col == 0, 0)],
                            else_=None
                        )
                    ).label(k)
                    _select.append(expr)
                    label_map[k] = expr
                    continue
                if hasattr(q_tab.c, k):
                    if not (hasattr(self, 'on_select_' + k) or hasattr(self,
                                                                       'on_select_from_did_charge_' + k)) and k not in aggregable:
                        continue
                    expr = _aggr(q_tab, k, method)
                    _select.append(expr)
                    label_map[k] = expr
            # _select = tuple(_select)
            _group = []
            new_time_field = None
            my_step = 0
            step_dict = {'hour': 60 * 60,
                         'day': 60 * 60 * 24,
                         'week': 60 * 60 * 24 * 7,
                         'year': 60 * 60 * 24 * 365,
                         'month': 60 * 60 * 24 * 30,
                         'all': 0}
            if step:
                if step in step_dict:
                    my_step = step_dict[step]
                else:
                    my_step = int(step) * 60
            if my_step:
                # .op('AT TIME ZONE')(tz)
                # new_time_field = (my_step * (cast(func.extract('epoch', self.time_field(q_tab)), Integer) / my_step))
                if step in step_dict:
                    # new_time_field = case(
                    #     [
                    #         (cast(func.extract('epoch', func.date_trunc(step, self.time_field(q_tab))), Integer) < start_time,
                    #          start_time)
                    #     ],
                    #     else_=cast(func.extract('epoch', func.date_trunc(step, self.time_field(q_tab))), Integer)
                    # )
                    new_time_field = cast(
                        func.extract('epoch', func.date_trunc(step, self.time_field(q_tab))), Integer
                    )
                else:
                    new_time_field = cast(
                        func.floor(func.extract('epoch', self.time_field(q_tab)) / my_step) * my_step,
                        Integer
                    )
                _select.append(new_time_field.label(self.time_field_str()))
                _group.append(new_time_field)

            if groups:
                _group = _group + list(map(lambda k: getattr(q_tab.c, k), groups))
                _select = _select + list(map(lambda k: getattr(q_tab.c, k), groups))

            q_tab = self.modify_select_from_groupping(_select, groups, q_tab)
            query = model.get_db().session.query(*_select).select_from(q_tab)
            if _group:
                query = query.group_by(*tuple(_group))
            if sort_order is None:
                if new_time_field is not None:
                    query = query.order_by(new_time_field)
            else:
                order_by_expr = []
                virtual_fields = {
                    'ingress_name', 'egress_name', 'ingress_client_name', 'egress_client_name',
                    'release_cause_name', 'agent_name', 'egress_client_id', 'ingress_client_id',
                    'start_time_of_date'
                }
                # build a mapping of virtual labels from _select
                label_map = {col.name: col for col in _select if hasattr(col, 'name')}
                for fld, direction in sort_order:
                    if fld == self.time_field_str():
                        col = new_time_field
                    elif fld in virtual_fields:
                        # pick the labeled column from _select
                        if fld not in label_map:
                            raise AttributeError(f"Virtual column {fld} not in _select")
                        col = label_map[fld]
                    else:
                        if fld in label_map:
                            col = label_map[fld]
                        else:
                            col = _aggr(q_tab, fld, method, False)                    
                    order_by_expr.append(col.desc() if direction == 'desc' else col)


                query = query.order_by(*order_by_expr)
            # return filtering, query
            if REPORT_DEBUG:
                log.debug(model.query_to_sting(query))

            if 'release_cause_from_protocol_stack' in groups:
                query = query.filter(text('release_cause_from_protocol_stack IS NOT null'))

            # if self.model_class == model.DidReport:
            #     for i in fields:
            #         if not i in ['mrc', 'nrc', 'did', 'subcharge']:
            #             query = query.filter(text('{field_name} IS NOT null'.format(field_name=i)))
            #             break

            return {}, query
        else:
            return None, None
        pass

    def modify_select_from_groupping(self, _select, groups, q_tab):
        def get_column_name_safe(q_tab, field_name):
            if hasattr(q_tab.c, field_name):
                return getattr(q_tab.c, field_name)
            prefixed_name = f"_all_cdr__{field_name}"
            if hasattr(q_tab.c, prefixed_name):
                return getattr(q_tab.c, prefixed_name)
            raise AttributeError(f"Column {field_name} not found in q_tab")

        if 'ingress_id' in groups:
            ing = aliased(model.Resource)
            q_tab = q_tab.outerjoin(ing, ing.resource_id == get_column_name_safe(q_tab, 'ingress_id'))
            _select.append(func.max(ing.alias).label('ingress_name'))
        if 'egress_id' in groups:
            egr = aliased(model.Resource)
            q_tab = q_tab.outerjoin(egr, egr.resource_id == get_column_name_safe(q_tab, 'egress_id'))
            _select.append(func.max(egr.alias).label('egress_name'))
        if 'ingress_client_id' in groups:
            cl = aliased(model.Client)
            tcr_in = aliased(model.t_client_record)

            # Subquery to get latest active record per client
            latest_tcr_subq = (
                model.get_db().session.query(
                    tcr_in.c.client_id.label('client_id'),
                    func.max(tcr_in.c.time).label('latest_time')
                )
                .filter(tcr_in.c.flag == 'A')
                .group_by(tcr_in.c.client_id)
                .subquery()
            )

            # Alias for the t_client_record table to join with the subquery
            tcr_join = aliased(model.t_client_record)

            # First, join the subquery to q_tab
            q_tab = q_tab.outerjoin(
                latest_tcr_subq,
                latest_tcr_subq.c.client_id == get_column_name_safe(q_tab, 'ingress_client_id')
            )

            # Then join t_client_record on client_id + latest_time
            q_tab = q_tab.outerjoin(
                tcr_join,
                and_(
                    tcr_join.c.client_id == latest_tcr_subq.c.client_id,
                    tcr_join.c.time == latest_tcr_subq.c.latest_time
                )
            )

            # Finally, outer join client as fallback
            q_tab = q_tab.outerjoin(
                cl,
                cl.client_id == get_column_name_safe(q_tab, 'ingress_client_id')
            )

            # Aggregate: take tcr_join.name if exists, else client.name
            _select.append(
                func.coalesce(
                    func.max(tcr_join.c.name),
                    func.max(cl.name)
                ).label('ingress_client_name')
            )

        if 'egress_client_id' in groups:
            cl = aliased(model.Client)
            tcr_out = aliased(model.t_client_record)

            # Subquery to get latest active record per client
            latest_tcr_subq = (
                model.get_db().session.query(
                    tcr_out.c.client_id.label('client_id'),
                    func.max(tcr_out.c.time).label('latest_time')
                )
                .filter(tcr_out.c.flag == 'A')
                .group_by(tcr_out.c.client_id)
                .subquery()
            )

            # Alias for the t_client_record table to join with the subquery
            tcr_join = aliased(model.t_client_record)

            # First, join the subquery to q_tab
            q_tab = q_tab.outerjoin(
                latest_tcr_subq,
                latest_tcr_subq.c.client_id == get_column_name_safe(q_tab, 'egress_client_id')
            )

            # Then join t_client_record on client_id + latest_time
            q_tab = q_tab.outerjoin(
                tcr_join,
                and_(
                    tcr_join.c.client_id == latest_tcr_subq.c.client_id,
                    tcr_join.c.time == latest_tcr_subq.c.latest_time
                )
            )

            # Finally, outer join client as fallback
            q_tab = q_tab.outerjoin(
                cl,
                cl.client_id == get_column_name_safe(q_tab, 'egress_client_id')
            )

            # Aggregate: take tcr_join.name if exists, else client.name
            _select.append(
                func.coalesce(
                    func.max(tcr_join.c.name),
                    func.max(cl.name)
                ).label('egress_client_name')
            )
        if 'agent_id' in groups:
            ag = model.Agent
            q_tab = q_tab.outerjoin(ag, ag.agent_id == get_column_name_safe(q_tab, 'agent_id'))
            _select.append(func.max(ag.agent_name).label('agent_name'))
        if 'release_cause' in groups:
            cls = aliased(model.GlobalRouteError)
            q_tab = q_tab.outerjoin(cls, cls.error_code == get_column_name_safe(q_tab, 'release_cause'))
            _select.append(func.max(cls.error_description).label('release_cause_name'))

        return q_tab

class ReportM(DnlList):
    entity_plural = 'Report items'
    scheme_class = CdrReportGroppedRequestScheme
    scheme_class_get = CdrReportGroppedResponseScheme
    scheme_class_get_inner = CdrReportDetailInnerScheme
    model_class = model.CdrReportDetail
    columns = model_class.get_fields()

    groupable = [fn for fn in columns if
                 getattr(model.CdrReportDetail, fn).type.__class__.__name__ in (
                     'String', 'DateTime', 'Text') or '_id' in fn or fn in _grp]
    aggregable = list(set([fn for fn in columns if
                           getattr(model.CdrReportDetail, fn).type.__class__.__name__ not in (
                               'String', 'DateTime', 'Text') and '_id' not in fn and fn not in _grp] +
                          ['nrf_calls', 'asr', 'qsr', 'ingress_npr', 'egress_npr']))

    def get_spec(self):
        sfields_names, search_fields_list = self.get_fields_from_scheme(self.scheme_class)
        qfields_names, query_fields_list = self.get_query_fields_from_scheme(self.scheme_class)
        fields_names = [f.name for f in inspect(self.model_class).columns]
        spec = swagger.specify.get_spec(
            method='get', description='Gets {}'.format(self.entity_plural.lower()),
            path_parameters=self.path_parameters,
            query_parameters=search_fields_list + [
                {"name": "from", "description": "printing range. Default: print from start 0 position",
                 "type": "integer"},
                {"name": "count", "description": "printing range. Default: print first 1000 items",
                 "type": "integer"},
                {"name": "fields",
                 "description": "aggregated fields, available values:{}".format(self.aggregable)},
                {"name": "group",
                 "description": "groupping fields, available values:{}".format(self.groupable)},

            ] +
                             [dict(name=i, type='string') for i in self.groupable],
            responses=(
                          SuccessReportResponse(),
                          responses.AttributeNotExitsErrorResponse()
                      ) + self.get_additional_responses(method='get'),
            security=self.get_security(method='get')
        )
        spec['get']['produces'] = ['application/json', 'text/csv', 'text/xml']
        return spec

    def run(self, **kwargs):
        class Req:
            def __init__(self, req_user):
                self.query_string = None
                self.context = {'user': req_user}

            pass

        if 'user' in kwargs:
            user = kwargs['user']
            del kwargs['user']
        else:
            user = model.User.get(1)
        req = Req(user)
        req.query_string = urlencode(kwargs)
        self.req = req
        objects_list, total, page, per_page, fields = self.get_objects_list(req, self.model_class, **kwargs)
        # items = [i for i in objects_list]
        items = self.scheme_class_get_inner().dump(objects_list, many=True).data
        return items

    def send_request(self, params):
        m = dict(calls='ingress_total_calls',
                 ingress_billed_time='ingress_bill_time',
                 ingress_cost='ingress_call_cost',
                 ingress_billed_min='ingress_bill_time',
                 code='ingress_code',
                 non_zero_calls='not_zero_calls',
                 egress_cost='egress_total_cost',
                 egress_billed_time='egress_bill_time',
                 )
        args = {k: v for k, v in params.items() if k not in m}
        upd = {m[k]: v for k, v in params.items() if k in m}
        args.update(upd)

        class Responce:
            def json(self):
                return self.data

        items = self.run(**args)
        resp = Responce()
        resp.data = dict(data=items)
        return resp

    def _on_get(self, req, resp, **kwargs):

        from .helpers import check_permission

        if (not hasattr(self, 'no_auth_needed') or self.no_auth_needed == False) and not check_permission(self, req,
                                                                                                          'list'):
            self.set_response(
                resp, responses.ForbiddenErrorResponse(data=errors.AuthErrors.Forbidden)
            )
            return
        try:
            import time
            _start = time.time()

            objects_list, total, _from, count, fields = self.get_objects_list(req, self.model_class, **kwargs)
            # items = [i for i in objects_list]
            items = objects_list #self.scheme_class_get_inner().dump(objects_list, many=True).data
            data = {
                'success': True,
                'data': items,
                'total': total, 'from': _from, 'count': len(items)
            }
            self.set_response(resp, SuccessReportResponse(data=data, scheme=self.scheme_class_get))

            _end = time.time()
            log.debug('report execution time {}'.format(_end - _start))
            # resp.body = json.dumps(data)
            # resp.status = falcon.HTTP_200

        # except AttributeError as e:
        #     self.set_response(
        #         resp, responses.AttributeNotExitsErrorResponse(
        #             data=ATTRIBUTE_ERROR_RE.search(str(e)).group(1)
        #         )
        #     )
        except (InternalError, IntegrityError, DataError) as e:
            import traceback
            model.get_db().session.rollback()
            log.error(traceback.format_exc())
            self.set_response(resp, OperationErrorResponse(e))
        except Exception as e:
            import traceback
            log.error(traceback.format_exc())
            self.set_response(resp, OperationErrorResponse(e))

    def time_field(self, cdr):
        return cdr.c.report_time.label('report_time')

    def get_daily_cdr(self, cdr_date):
        cdr = model.cdr_report_detail(cdr_date)
        return cdr

    def get_fields_for_list(self, parsed_qs, **kwargs):
        fields = parsed_qs.get('fields', None)
        if fields:
            fields = fields.split(',')
            for f in fields:
                # if f not in self.get_all_fields_from_scheme(self.scheme_class_get)[0]:
                if f not in self.aggregable:
                    raise Exception('fields {} not valid!'.format(f))
        return fields

    def get_objects_list(self, req, model_class, **kwargs):
        parsed_qs = dict(parse_qsl(req.query_string))
        if REPORT_DEBUG:
            log.debug('report parsed parameters:{}'.format(parsed_qs))

        try:
            _from = int(parsed_qs.get('from'))
        except (ValueError, TypeError):
            _from = 0

        try:
            count = int(parsed_qs.get('count', 1000))
        except ValueError:
            count = 1000

        ordering = {}

        fields = self.get_fields_for_list(parsed_qs, **kwargs)

        filtering = self.get_filtering_for_list(parsed_qs, **kwargs)
        if fields:
            filtering['fields'] = ','.join(fields)

        filtering, query = \
            self.modify_query_from_filtering_for_list(filtering, **kwargs)

        if ordering:
            ordering, query = \
                self.modify_query_from_ordering_for_list(ordering, query, **kwargs)
        if query is None:
            return [], 0, _from, count, fields

        return (query, len(query), _from, count, fields)

    def on_filter_ingress_id(self, v, cdr):
        # return text("ingress_id in ({})".format(v))
        return cdr.c.ingress_id.in_([int(i) for i in v.split(',')])

    def on_filter_egress_id(self, v, cdr):
        # return text("egress_id in ({})".format(v))
        return cdr.c.egress_id.in_([int(i) for i in v.split(',')])

    def on_filter_non_zero_calls(self, v, cdr):
        # return text("egress_id in ({})".format(v))
        if v == '1':
            return cdr.c.ingress_not_zero_calls > 0

    def on_filter_scc_below_sec(self, v, cdr):
        return cdr.c.duration < v

    def on_filter_is_hide_unauthorized(self, v, cdr):
        if v is None:
            return None
        if v:
            return cdr.c.release_cause != 3
        else:
            return cdr.c.release_cause == 3

    def on_select_nrf_calls(self, cdr):
        return cdr.c.lrn_calls

    def on_select_asr(self, cdr):
        return 100.00 * ((1.0 * cdr.c.not_zero_calls) / (1.0 * cdr.c.ingress_total_calls))

    def on_select_qsr(self, cdr):
        return 100.00 * ((cdr.c.not_zero_calls + cdr.c.ingress_busy_calls + cdr.c.ingress_cancel_calls) /
                         cdr.c.ingress_total_calls)

    def on_select_ingress_npr(self, cdr):
        return 100.00 * ((1.0 * cdr.c.npr_count) / (1.0 * cdr.c.ingress_total_calls))

    def on_select_egress_npr(self, cdr):
        return 100.00 * ((1.0 * cdr.c.npr_count) / (1.0 * cdr.c.egress_total_calls))

    def on_filter_show_non_zero_only(self, v, cdr):
        if v == 'true':
            return cdr.c.duration > 0
        else:
            return cdr.c.duration == 0

    def modify_query_from_filtering_for_list(self, filtering, **kwargs):

        def similar_fields(field, field_list):
            sf = field.split('_')
            sim = []
            for f in sf:
                for ef in field_list:
                    if f in ef:
                        sim.append(ef)
            if sim:
                return 'similar fields:{}'.format(set(sim))
            return ''

        import sys
        sys.setrecursionlimit(10000)
        cls = self.model_class
        columns = cls.get_fields()
        # columns = [f.name for f in inspect(self.model_class).columns]
        req_query = dict(parse_qsl(self.req.query_string))
        _from = filtering.pop('from', None)
        count = filtering.pop('count', None)

        sort = filtering.pop('sort', None)
        order = filtering.pop('order', '')

        sort_order = None
        if sort:
            _sort = sort.split(',')
            for s in _sort:
                if s not in columns:
                    raise ValidationError('sort field missing {}'.format(s))
            _order = order.split(',')
            for o in _order:
                if o not in ('', 'asc', 'desc'):
                    raise ValidationError('sort order invalid value {}, must be asc or desc (can be empty)'.format(o))
            if len(_sort) != len(_order):
                if sort != '':
                    raise ValidationError('sort length must be equal order length')
                else:
                    _order = ['' for s in _sort]
            sort_order = list(zip(_sort, _order))

        start_time = filtering.pop('start_time', None)
        end_time = filtering.pop('end_time', str(int(datetime.now(UTC).timestamp())))
        try:
            start_time = int(start_time)
            if start_time < 0:
                start_time = int(datetime.now(UTC).timestamp()) + start_time
            start_date_time = datetime.utcfromtimestamp(start_time).replace(tzinfo=UTC)
        except (ValueError, TypeError):
            raise ValidationError('wrong start time {}'.format(start_time))
        try:
            end_time = int(end_time)
            end_date_time = datetime.utcfromtimestamp(end_time).replace(tzinfo=UTC)
        except (ValueError, TypeError):
            raise ValidationError('wrong end time')
        if start_time >= end_time:
            raise ValidationError('start time greater than end_time!')
        groupable = self.groupable
        aggregable = self.aggregable
        tz = str(filtering.pop('tz', '0'))
        if REPORT_DEBUG:
            log.debug('tz={}'.format(tz))
        non_zero_calls = filtering.pop('non_zero_calls', None)
        if REPORT_DEBUG:
            log.debug('non_zero_calls={}'.format(non_zero_calls))
        human_readable = filtering.pop('human_readable', None)
        if REPORT_DEBUG:
            log.debug('human_readable={}'.format(human_readable))

        method = filtering.pop('method', 'total')
        step = filtering.pop('step', 'all')
        groups = filtering.pop('group', None)
        fields = filtering.pop('fields', None)
        field = filtering.pop('field', None)
        fields = fields or field
        if fields:
            fields = fields.split(',')
        if not fields:
            fields = aggregable.copy()
        for fk in filtering.keys():
            if fk not in columns:
                if not hasattr(self, 'on_filter_{}'.format(fk)):
                    raise ValidationError('{}:wrong filter field. {}'.format(fk, similar_fields(fk, columns)))
            if fk not in fields and not hasattr(self, 'on_filter_{}'.format(fk)):
                fields.append(fk)

        if groups is not None:
            groups = groups.split(',')
        else:
            groups = []
        if 'report_time' in groups:
            step = 60
            groups.remove('report_time')
        for fk in groups:
            if fk not in groupable:
                raise ValidationError('{}:wrong group field. {}'.format(fk, similar_fields(fk, groupable)))
            if fk not in fields:
                fields.append(fk)

        fields.append('report_time')
        fields = list(set(fields))

        user = self.get_user(self.req)

        if user and user.user_type in ('client', 'agent', 'vendor'):
            if 'ingress_id' in filtering and filtering['ingress_id']:
                ids = set(filtering['ingress_id'].split(','))
                alw = set([str(i) for i in user.allowed_ingress_id])
                filtering['ingress_id'] = ','.join(list(ids.intersection(alw)))
            else:
                filtering['ingress_id'] = ','.join([str(i) for i in user.allowed_ingress_id])
            if filtering['ingress_id'] == '':
                if user.allowed_egress_id:
                    alw = set([str(i) for i in user.allowed_egress_id])
                    if 'egress_id' in filtering and filtering['egress_id']:
                        ids = set(filtering['egress_id'].split(','))
                        filtering['egress_id'] = ','.join(list(ids.intersection(alw)))
                    else:
                        filtering['egress_id'] = ','.join(list(alw))
                    del filtering['ingress_id']
                else:
                    filtering['ingress_id'] = '-1'
            if REPORT_DEBUG:
                log.debug('result {} filtering={}'.format(user.user_type, filtering))

        start_day = start_date_time.date()
        database_start_day = model.cdr_reports_start_date()
        if database_start_day is None:
            raise ValidationError('database has no cdr report data!')
        if database_start_day > start_day:
            start_day = database_start_day
        end_day = end_date_time.date()
        if end_day < start_day:
            end_day = start_day

        days = (end_day - start_day).days
        union_query = None
        union_list = []
        union_dict = {}

        def _aggr(cdr_table, field_name, agg_method=None, need_label=True):
            fd = {'total': func.sum, 'min': func.min, 'max': func.max, 'avrg': func.avg, }

            if agg_method and (field_name in aggregable or hasattr(self, 'on_select_' + field_name)):
                if agg_method not in fd:
                    raise ValidationError(
                        '{}:no such method.  Allowed values :{}'.format(k, similar_fields(agg_method, fd.keys())))
                f = fd[agg_method]
                if hasattr(self, 'on_select_' + field_name):
                    tf = f(getattr(self, 'on_select_' + k)(cdr))
                elif hasattr(cdr_table.c, field_name):
                    tf = f(getattr(cdr_table.c, field_name))
                else:
                    tf = literal(0.0)
                if need_label:
                    return tf.label(field_name)
                else:
                    return tf
            else:
                tf = getattr(cdr_table.c, field_name)
                if need_label:
                    return tf.label(field_name)
                else:
                    return tf

        _gkeys = []
        _agr = []
        for day_idx in range(days + 1):
            day_cdr = start_day + timedelta(days=day_idx)
            cdr_date = "%s%02d%02d" % (day_cdr.year, day_cdr.month, day_cdr.day)
            cdr = self.get_daily_cdr(cdr_date)
            if cdr is None:
                log.warning('missing table {}'.format(cdr_date))
                continue
            _select = []
            for k in fields:
                keys = [x.key for x in _select]
                if k in keys:  # already here
                    continue
                # k=k.replace('non_zero_calls','not_zero_calls')
                if k in ('asr', 'qsr', 'ingress_npr', 'egress_npr'):
                    keys = [x.key for x in _select]
                    for df in ['ingress_busy_calls', 'ingress_cancel_calls', 'not_zero_calls',
                               'ingress_total_calls', 'egress_total_calls', 'npr_count']:
                        if not df in keys:
                            _select.append(_aggr(cdr.c, df, method))
                    continue
                if hasattr(self, 'on_select_' + k):
                    _f = getattr(self, 'on_select_' + k)(cdr)
                    if _f is not None:
                        _select.append(_aggr(cdr, k, method))
                    continue
                elif hasattr(cdr.c, k) or k in ('ingress_name', 'egress_name'):
                    if not hasattr(self, 'on_select_' + k) and k not in aggregable:
                        continue
                    _select.append(_aggr(cdr, k, method))
                else:
                    raise ValidationError('{}:no such field to select.{}'.format(k, similar_fields(k, columns)))
                    # _select.append(getattr(cdr.c, k))
            # _select = tuple(_select)
            _filter = []
            if day_idx == 0:
                _filter = [self.time_field(cdr) >= start_date_time]
                if day_idx == days:
                    _filter = [self.time_field(cdr) >= start_date_time, self.time_field(cdr) <= end_date_time]
            else:
                if day_idx == days:
                    _filter = [self.time_field(cdr) <= end_date_time]
            for k, v in filtering.items():
                if hasattr(self, 'on_filter_' + k):
                    _f = getattr(self, 'on_filter_' + k)(v, cdr)
                    if _f is not None:
                        _filter.append(_f)
                    continue
                if not k in columns:
                    raise ValidationError('{}:no such field to filter.{}'.format(k, similar_fields(k, columns)))
                if hasattr(cdr.c, k):
                    if '*' in v:
                        _filter.append(getattr(cdr.c, k).like(v.replace('*', '%')))
                    else:
                        _filter.append(getattr(cdr.c, k) == v)
            _group = []
            new_time_field = None
            my_step = 0
            if step:
                step_dict = {'hour': 60 * 60,
                             'day': 60 * 60 * 24,
                             'week': 60 * 60 * 24 * 7,
                             'year': 60 * 60 * 24 * 365,
                             'month': 60 * 60 * 24 * 30,
                             'all': 0}
                if step in step_dict:
                    my_step = step_dict[step]
                else:
                    my_step = int(step)
            if my_step:
                if step in step_dict:
                    new_time_field = cast(func.extract('epoch', func.date_trunc(step, self.time_field(cdr))),
                                          Integer).label('report_time')
                else:
                    new_time_field = (
                            my_step * (cast(func.extract('epoch', self.time_field(cdr)), Integer) / my_step)).label(
                        'report_time')
                _select.append(new_time_field.label('report_time'))
                _group.append(new_time_field)
            # else:
            #     _group.append(cast(func.extract('millennium', self.time_field(cdr)), Integer))

            if groups:
                _group = _group + list(map(lambda k: getattr(cdr.c, k), groups))
                _select = _select + list(map(lambda k: getattr(cdr.c, k), groups))

            self.modify_select_from_groupping(_select, groups, cdr)

            q = model.get_db().session.query(*_select)
            if _filter:
                q = q.filter(and_(*_filter))
            if _group:
                q = q.group_by(*_group)
                _gkeys = [i.key for i in _group]
                if 'ingress_id' in _gkeys:
                    _gkeys.append('ingress_name')
                if 'egress_id' in _gkeys:
                    _gkeys.append('egress_name')
                if 'ingress_client_id' in _gkeys:
                    _gkeys.append('ingress_client_name')
                if 'egress_client_id' in _gkeys:
                    _gkeys.append('egress_client_name')
                if 'release_cause' in _gkeys:
                    _gkeys.append('release_cause_name')
            _agr = [x.key for x in _select if x.key in aggregable]
            union_list = union_list + q.all()

        # post groupping and filtering
        res = []
        if union_list:
            print(union_list)
            def get0(k):
                if getattr(cls, k, None) and getattr(cls, k).type.python_type.__name__ == 'int':
                    return 0
                return ''

            def gkey(x):
                return tuple(get0(k) if getattr(x, k) is None else getattr(x, k) for k in _gkeys)

            def keydict(x):
                return dict(zip(_gkeys, x))

            def agrdict(x):
                return dict(zip(_agr, x))

            def agrdict0():
                return {k: 0 for k in _agr}

            union_list.sort(key=gkey)
            for key, grp in groupby(union_list, gkey):
                row = {**keydict(key), **agrdict0()}
                for thing in grp:
                    for k in _agr:
                        if not getattr(thing, k, 0) is None:
                            row[k] += getattr(thing, k, 0)
                    #print(key, thing)
                #print()
                from decimal import Decimal
                for k in fields:
                    if k == 'asr':
                        row['asr'] = round(
                            Decimal(100.0) * row['not_zero_calls'] / row['ingress_total_calls'])
                    if k == 'qsr':
                        row['qsr'] = round(
                            Decimal(100.0) * (row['not_zero_calls'] + row['ingress_busy_calls'] + row[
                                'ingress_cancel_calls']) / row['ingress_total_calls'])
                    if k == 'ingress_npr':
                        row['ingress_npr'] = round(
                            Decimal(100.0) * row['npr_count'] / row['ingress_total_calls'])
                    if k == 'egress_npr':
                        row['egress_npr'] = round(
                            Decimal(100.0) * row['npr_count'] / row['ingress_total_calls'])
                for k in list(row.keys()):
                    if k not in fields and k not in _gkeys:
                        del row[k]
                res.append(row)
            #print(res)

            return {}, res
        else:
            return None, None
        pass

    def modify_select_from_groupping(self, _select, groups, q_tab):
        if 'ingress_id' in groups:
            ing = aliased(model.Resource)
            _select.append(
                select([ing.alias]).where(ing.resource_id == q_tab.c.ingress_id).as_scalar().label('ingress_name'))
        if 'egress_id' in groups:
            ing = aliased(model.Resource)
            _select.append(
                select([ing.alias]).where(ing.resource_id == q_tab.c.egress_id).as_scalar().label('egress_name'))
        if 'ingress_client_id' in groups:
            ing = aliased(model.Client)
            _select.append(
                select([ing.name]).where(ing.client_id == q_tab.c.ingress_client_id).as_scalar().label(
                    'ingress_client_name'))
        if 'egress_client_id' in groups:
            ing = aliased(model.Client)
            _select.append(
                select([ing.name]).where(ing.client_id == q_tab.c.egress_client_id).as_scalar().label(
                    'egress_client_name'))
        if 'release_cause' in groups:
            cls = aliased(model.GlobalRouteError)
            _select.append(
                select([cls.error_description]).where(cls.error_code == q_tab.c.release_cause).as_scalar().label(
                    'release_cause_name'))



class ReportDailySimple(ReportSimple):
    entity_plural = 'Client Cdr Report items'
    scheme_class = DidReportGroppedRequestScheme
    scheme_class_get = ClientCdrReportDailyGroppedResponseScheme
    scheme_class_get_inner = CdrReportDailyGetScheme
    model_class = model.CdrReportDaily
    columns = model_class.get_fields()

    groupable = [fn for fn in columns if
                 getattr(model.CdrReportDaily, fn).type.__class__.__name__ in (
                     'String', 'DateTime', 'Text', 'Boolean') or '_id' in fn or fn in _grp] + \
                ['ingress_name', 'egress_name', 'ani_in_dno_count', 'ani_in_ftc_count', 
                 'ani_in_ftc_week_count', 'ani_in_ftc_month_count', 'ani_in_spam_count',
                 'ani_in_fraud_count', 'ani_in_tcpa_count', 'dnis_in_dnc_count']
    aggregable = [fn for fn in columns if not (
                  getattr(model.CdrReportDaily, fn).type.__class__.__name__ in (
                      'String', 'DateTime', 'Text', 'Boolean') or '_id' in fn or fn in _grp)] + \
                      ['SDP30_calls', 'SDP6_calls']

    def get_daily_cdr(self, cdr_date):
        cdr = model.cdr_report_daily(cdr_date)
        return cdr

    def time_field(self, cdr):
        return cdr.c.report_time.label('report_time')

    def on_filter_o_trunk_type2(self, v, cdr):
        cls = model.Client
        clients = cls.session().query(cls.client_id).filter(cls.company_type==v).all()
        return cdr.c.ingress_client_id.in_([int(i.client_id) for i in clients])


class ReportDaily(Report):
    entity_plural = 'Client Cdr Report items'
    scheme_class = DidReportGroppedRequestScheme
    scheme_class_get = ClientCdrReportDailyGroppedResponseScheme
    scheme_class_get_inner = CdrReportDailyGetScheme
    model_class = model.CdrReportDaily
    columns = model_class.get_fields()

    groupable = [fn for fn in columns if
                 getattr(model.CdrReportDaily, fn).type.__class__.__name__ in (
                     'String', 'DateTime', 'Text', 'Boolean') or '_id' in fn or fn in _grp] + \
                ['ingress_name', 'egress_name', 'ani_in_dno_count', 'ani_in_ftc_count', 
                 'ani_in_ftc_week_count', 'ani_in_ftc_month_count', 'ani_in_spam_count',
                 'ani_in_fraud_count', 'ani_in_tcpa_count', 'dnis_in_dnc_count']
    aggregable = [fn for fn in columns if not (
                  getattr(model.CdrReportDaily, fn).type.__class__.__name__ in (
                      'String', 'DateTime', 'Text', 'Boolean') or '_id' in fn or fn in _grp)] + \
                      ['SDP30_calls', 'SDP6_calls']

    def get_daily_cdr(self, cdr_date):
        cdr = model.cdr_report_daily(cdr_date)
        return cdr

    def time_field(self, cdr):
        return cdr.c.report_time.label('report_time')

    def on_filter_o_trunk_type2(self, v, cdr):
        cls = model.Client
        clients = cls.session().query(cls.client_id).filter(cls.company_type==v).all()
        return cdr.c.ingress_client_id.in_([int(i.client_id) for i in clients])


class CodeReportDaily(Report):
    entity_plural = 'Client Cdr Report items'
    scheme_class = DidReportGroppedRequestScheme
    scheme_class_get = ClientCodeReportDailyGroppedResponseScheme
    scheme_class_get_inner = CodeReportDailyGetScheme
    model_class = model.CodeReportDaily
    columns = model_class.get_fields()

    groupable = [fn for fn in columns if
                 getattr(model.CodeReportDaily, fn).type.__class__.__name__ in (
                     'String', 'DateTime', 'Text') or '_id' in fn or fn in _grp] + \
                ['ingress_name', 'egress_name', 'ingress_id', 'egress_id']
    aggregable = [fn for fn in columns if not (
                  getattr(model.CodeReportDaily, fn).type.__class__.__name__ in (
                      'String', 'DateTime', 'Text', 'Boolean') or '_id' in fn or fn in _grp)]

    def get_daily_cdr(self, cdr_date):
        cdr = model.code_report_daily(cdr_date)
        return cdr

    def time_field(self, cdr):
        return cdr.c.report_time.label('report_time')


class ReportCodeReport(Report):
    entity_plural = 'Client Cdr Report items'
    scheme_class = DidReportGroppedRequestScheme
    scheme_class_get = ClientCodeReportDailyGroppedResponseScheme
    scheme_class_get_inner = CodeReportDailyGetScheme
    model_class = model.CodeReport
    columns = model_class.get_fields()

    groupable = [fn for fn in columns if
                 getattr(model.CodeReport, fn).type.__class__.__name__ in (
                     'String', 'DateTime', 'Text') or '_id' in fn or fn in _grp] + \
                ['ingress_name', 'egress_name', 'ingress_id', 'egress_id']
    aggregable = [fn for fn in columns if not (
                  getattr(model.CodeReport, fn).type.__class__.__name__ in (
                      'String', 'DateTime', 'Text', 'Boolean') or '_id' in fn or fn in _grp)]

    def get_daily_cdr(self, cdr_date):
        cdr = model.code_report(cdr_date)
        return cdr

    def time_field(self, cdr):
        return cdr.c.report_time.label('report_time')


class CdrReportDetailGroupableList(Report):
    scheme_class = DidReportGroppedRequestScheme
    scheme_class_get = ClientCdrReportDailyGroppedResponseScheme
    scheme_class = CdrReportDetailGroupableGetScheme
    model_class = model.CdrReportDetail
    entity_plural = 'CdrReportDetail'

    def get_daily_cdr(self, cdr_date):
        cdr = model.cdr_report_detail(cdr_date)
        return cdr

    def time_field(self, cdr):
        return cdr.c.report_time.label('report_time')

    def time_field_str(self):
        return 'report_time'

    def get_cdr_start(self):
        date = model.cdr_reports_start_date()
        return date

class DidReport(Report):
    entity_plural = 'Did Report items'
    scheme_class = DidReportGroppedRequestScheme
    scheme_class_get = DidReportGroppedResponseScheme
    scheme_class_get_inner = DidReportInnerScheme
    model_class = model.DidReport
    columns = model_class.get_fields()

    groupable = [fn for fn in columns if
                 getattr(model.DidReport, fn).type.__class__.__name__ in (
                     'String', 'DateTime', 'Text') or '_id' in fn or fn in _grp] + ['country', 'state', 'agent_id']
    aggregable = [fn for fn in columns if
                  getattr(model.DidReport, fn).type.__class__.__name__ not in (
                      'String', 'DateTime', 'Text') and '_id' not in fn and fn not in _grp]

    def modify_query_from_filtering_for_list(self, filtering, **kwargs):
        groups = filtering.get('group', '').split(',')
        if 'country' in groups:
            groups.remove('country')
            if not 'did' in groups:
                groups += ['did']
            filtering['group'] = ','.join(groups)
        if 'state' in groups:
            groups.remove('state')
            if not 'did' in groups:
                groups += ['did']
            filtering['group'] = ','.join(groups)
        # if not 'show_non_zero_only' in filtering:
        #     filtering['show_non_zero_only'] = 'true'
        return super(DidReport, self).modify_query_from_filtering_for_list(filtering, **kwargs)

    def on_filter_show_non_zero_only(self, v, cdr):
        if v == 'true':
            return cdr.c.duration > 0
        else:
            return cdr.c.duration == 0

    def on_filter_is_hide_unauthorized(self, v, cdr):
        rep = aliased(model.DidBillingRel)
        if v is None:
            return None
        if v == 'true':
            q = select([func.count(rep.did) > 0]).where(rep.did == cdr.c.did).as_scalar()
            return q
        else:
            q = select([func.count(rep.did) == 0]).where(rep.did == cdr.c.did).as_scalar()
            return q

    def on_filter_agent_id(self, v, cdr):
        cls = model.AgentClient
        clients = cls.session().query(cls.client_id).filter(cls.agent_id == v).all()
        return or_(cdr.c.ingress_client_id.in_([int(i.client_id) for i in clients]),
                   cdr.c.egress_client_id.in_([int(i.client_id) for i in clients]))

    def on_select_agent_id(self, cdr):
        cls = model.AgentClient
        return select([cls.agent_id]).where(cls.client_id == cdr.c.egress_client_id)

    def get_daily_cdr(self, cdr_date):
        cdr = model.did_report(cdr_date)
        return cdr

    def _ya_query(self, fields, start_date, end_date, filtering):
        # def query_did_charge(start_date, end_date, client, vendor, did):
        vendor = filtering.get('ingress_client_id', '')
        vendor_res = filtering.get('egress_id', '')
        client = filtering.get('egress_client_id', '')
        client_res = filtering.get('egress_id', '')
        did = filtering.get('did', '')

        params = ';'.join(
            [str(start_date), str(end_date), str(client), str(vendor), str(client_res), str(vendor_res), str(did)])
        clsd = model.DidChargeDetail
        rel = model.DidBillingRel  # v6
        # rel = DidBillingBrief
        pln = model.DidBillingPlan
        rep = model.DidReport
        Resource = model.Resource
        dids = None
        if did:
            dids = [i for i in did.split(',')]

        trunks = None
        client_id = None
        if client:
            client_id = [int(i) for i in client.split(',')]
            trunks = [r.resource_id for r in Resource.filter(Resource.client_id.in_(client_id)).all()]
        if client_res:
            client_res_id = [int(i) for i in client_res.split(',')]
            trunks = [r for r in client_res_id]
        vendor_trunks = None
        if vendor:
            vendor_id = [int(i) for i in vendor.split(',')]
            vendor_trunks = [r.resource_id for r in Resource.filter(Resource.client_id.in_(vendor_id)).all()]
        if vendor_res:
            vendor_res_id = [int(i) for i in vendor_res.split(',')]
            trunks = [r for r in vendor_res_id]

        exists = clsd.filter(clsd.params == params).first()
        if exists:
            clsd.filter(clsd.params == params).delete(synchronize_session='fetch')
        if True:  # exists is None:
            ret = clsd.session().execute(clsd.__table__.insert().from_select(
                ['params', 'did_number', 'client_billing_plan_id', 'client_trunk_id', 'vendor_trunk_id', 'nrc', 'mrc',
                 'port_fee', 'pay_type',
                 'start_date', 'end_date'],
                select([literal(params), rel.did, rel.buy_billing_plan_id, rel.client_res_id, rel.vendor_res_id,
                        case([(rel.start_date >= start_date, pln.did_price)], else_=0.0),
                        pln.monthly_charge, pln.fee_per_port,
                        pln.pay_type,
                        func.greatest(cast(rel.start_date, DateTime), start_date),
                        func.least(cast(rel.end_date, DateTime), end_date),
                        ]
                       ).where(
                            and_(
                                or_(rel.end_date.is_(None), rel.end_date > start_date),
                                rel.start_date <= end_date,
                                rel.client_res_id.in_(trunks) if trunks else True,
                                rel.vendor_res_id.in_(vendor_trunks) if vendor_trunks else True,
                                rel.did.in_(dids) if dids else True,
                                pln.id == rel.buy_billing_plan_id
                            )
                )
            )
            )
            # print(ret)
            q = clsd.__table__.update().values(mrc_number=
                                               case([(clsd.pay_type == 0,
                                                      func.extract('isoyear', clsd.end_date) * 52 +
                                                      func.extract('week', clsd.end_date) - func.extract('week',
                                                                                                         clsd.start_date)
                                                      - func.extract('isoyear', clsd.start_date) * 52
                                                      ),
                                                     (clsd.pay_type == 1, func.extract('month',
                                                                                       clsd.end_date + timedelta(
                                                                                           days=1)) - func.extract(
                                                         'month', clsd.start_date)
                                                      + func.extract('year', clsd.end_date + timedelta(
                                                         days=1)) * 12 - func.extract('year', clsd.start_date) * 12)
                                                     ],
                                                    else_=clsd.end_date - clsd.start_date
                                                    )
                                               ).where(clsd.params == params)
            # print(str(q))
            ret = clsd.session().execute(q)
            # print(ret)
            clsd.session().commit()

    def _on_select_from_did_charge_mrc(self, cdr, cdr_date, filtering):
        vendor = filtering.get('ingress_client_id', '')
        vendor_res = filtering.get('egress_id', '')
        client = filtering.get('egress_client_id', '')
        client_res = filtering.get('egress_id', '')
        did = filtering.get('did', '')
        params = ';'.join(
            [str(cdr_date), str(cdr_date), str(client), str(vendor), str(client_res), str(vendor_res), str(did)])

        cls = model.DidChargeDetail
        return func.coalesce(
            select([func.coalesce(cls.did_mrc, 0)]).where(and_(cls.params == params, cls.did == cdr.c.did)).order_by(
                cls.id.desc()).limit(1).label('mrc'), 0)

    def _on_select_from_did_charge_nrc(self, cdr, cdr_date, filtering):
        vendor = filtering.get('ingress_client_id', '')
        vendor_res = filtering.get('egress_id', '')
        client = filtering.get('egress_client_id', '')
        client_res = filtering.get('egress_id', '')
        did = filtering.get('did', '')
        params = ';'.join(
            [str(cdr_date), str(cdr_date), str(client), str(vendor), str(client_res), str(vendor_res), str(did)])

        cls = model.DidChargeDetail
        return func.coalesce(
            select([func.coalesce(cls.nrc, 0)]).where(and_(cls.params == params, cls.did == cdr.c.did)).order_by(
                cls.id.desc()).limit(1).label('nrc'), 0)

    def _on_select_from_did_charge_port_fee(self, cdr, cdr_date, filtering):
        vendor = filtering.get('ingress_client_id', '')
        vendor_res = filtering.get('egress_id', '')
        client = filtering.get('egress_client_id', '')
        client_res = filtering.get('egress_id', '')
        did = filtering.get('did', '')
        params = ';'.join(
            [str(cdr_date), str(cdr_date), str(client), str(vendor), str(client_res), str(vendor_res), str(did)])

        cls = model.DidChargeDetail
        return func.coalesce(select([func.coalesce(cls.did_port_fee, 0)]).where(
            and_(cls.params == params, cls.did == cdr.c.did)).order_by(cls.id.desc()).limit(1).label('port_fee'), 0)

    def modify_select_from_groupping(self, _select, groups, q_tab):
        if 'agent_id' in groups:
            ag = model.Agent
            _select.append(
                select([ag.agent_name]).where(ag.agent_id == q_tab.c.agent_id).as_scalar().label('agent_name'))
        if 'ingress_id' in groups:
            ing = aliased(model.Resource)
            _select.append(
                select([ing.alias]).where(ing.resource_id == q_tab.c.ingress_id).as_scalar().label('ingress_name'))
        if 'egress_id' in groups:
            ing = aliased(model.Resource)
            _select.append(
                select([ing.alias]).where(ing.resource_id == q_tab.c.egress_id).as_scalar().label('egress_name'))
        if 'ingress_client_id' in groups:
            cl = model.Client
            ing = aliased(model.t_client_record)
            _select.append(func.coalesce(
                select([ing.c.name]).where(and_(ing.c.client_id == q_tab.c.ingress_client_id,
                                                ing.c.flag == 'A')).order_by(ing.c.time.desc()).limit(1).as_scalar(),
                select([cl.name]).where(cl.client_id == q_tab.c.ingress_client_id).as_scalar()).label('ingress_client_name'))
        if 'egress_client_id' in groups:
            cl = model.Client
            ing = aliased(model.t_client_record)
            _select.append(func.coalesce(
                select([ing.c.name]).where(and_(ing.c.client_id == q_tab.c.egress_client_id,
                                                ing.c.flag == 'A')).order_by(ing.c.time.desc()).limit(1).as_scalar(),
                select([cl.name]).where(cl.client_id == q_tab.c.egress_client_id).as_scalar()).label('egress_client_name'))
        if 'did' in groups:
            param = aliased(model.DidParam)
            _select.append(
                select([param.country_iso]).where(param.did == q_tab.c.did).as_scalar().label('country'))
            _select.append(
                select([param.state]).where(param.did == q_tab.c.did).as_scalar().label('state'))


class HostBasedReport(Report):
    entity_plural = 'Host Based Report items'
    scheme_class = DidReportGroppedRequestScheme
    scheme_class_get = DidReportGroppedResponseScheme
    scheme_class_get_inner = DidReportInnerScheme
    model_class = model.HostBasedReport
    columns = model_class.get_fields()

    groupable = [fn for fn in columns if
                 getattr(model.HostBasedReport, fn).type.__class__.__name__ in (
                     'String', 'DateTime', 'Text') or '_id' in fn or fn in _grp] + \
                ['ingress_name', 'egress_name', 'ingress_id', 'egress_id']
    aggregable = [fn for fn in columns if not (
                  getattr(model.HostBasedReport, fn).type.__class__.__name__ in (
                      'String', 'DateTime', 'Text') or '_id' in fn or fn in _grp)]

    def get_daily_cdr(self, cdr_date):
        cdr = model.host_based_report(cdr_date)
        return cdr

    def on_filter_show_non_zero_only(self, v, cdr):
        if v == 'true':
            return and_(cdr.c.duration > 0, cdr.c.ingress_call_cost > 0)
        else:
            return cdr.c.duration == 0

    def on_filter_trunk_type2(self, v, cdr):
        clsr = aliased(model.Resource)
        clsip = aliased(model.ResourceIp)
        resources = select([clsr.resource_id]).where(and_(clsr.trunk_type2 == v, clsr.client_id == cdr.c.egress_client_id))#clsr.trunk_type2 == v)
        #ips = select([model.ResourceIp.ip]).where(model.ResourceIp.resource_id.in_(resources
        return cdr.c.egress_ip.in_(select([clsip.ip]).where(clsip.resource_id.in_(select([clsr.resource_id]). \
                               where(and_(clsr.trunk_type2 == v, clsr.client_id == cdr.c.egress_client_id)). \
                               correlate_except(clsr))))

    def on_filter_egress_id(self, v, cdr):
        egress = model.Resource.filter(model.Resource.resource_id == v).first()
        egress_ip = model.ResourceIp.filter(model.ResourceIp.resource_id == v).first()
        if egress and egress_ip:
            return and_(cdr.c.egress_ip == egress_ip.ip,cdr.c.egress_client_id == egress.client_id)
        return False

    def on_filter_ingress_id(self, v, cdr):
        ingress = model.Resource.filter(model.Resource.resource_id == v).first()
        ingress_ips = model.ResourceIp.session().query(model.ResourceIp.ip).filter(model.ResourceIp.resource_id == ingress.resource_id).all()
        ips = [ip for ip in ingress_ips]
        if ingress and ips:
            return and_(cdr.c.ingress_ip.in_(ips), cdr.c.ingress_client_id == ingress.client_id)
        return False

    def on_filter_ingress_name(self, v, cdr):
        ingress = model.Resource.filter(model.Resource.alias == v).first()
        if ingress:
            ingress_ip = model.ResourceIp.filter(model.ResourceIp.resource_id == ingress.resource_id).first()
            return and_(cdr.c.ingress_ip == ingress_ip.ip, cdr.c.ingress_client_id == ingress.client_id)
        return False

    def on_filter_egress_name(self, v, cdr):
        egress = model.Resource.filter(model.Resource.alias == v).first()
        if egress:
            egress_ip = model.ResourceIp.filter(model.ResourceIp.resource_id == egress.resource_id).first()
            return and_(cdr.c.egress_ip == egress_ip.ip,cdr.c.egress_client_id == egress.client_id)
        return False

    def modify_select_from_groupping(self, _select, groups, q_tab):
        super().modify_select_from_groupping(_select, groups, q_tab)
        ipc = aliased(model.ResourceIp)
        ing = aliased(model.Resource)
        if 'ingress_client_id' in groups and 'ingress_ip' in groups:
            if self.filtering.get("ingress_id", None):
                ingress_id = self.filtering.get("ingress_id")
                _select.append(
                    select([func.max(ing.resource_id)]).where(
                            ing.resource_id == ingress_id
                    ).as_scalar().label('ingress_id')
                )
                _select.append(
                    select([func.max(ing.alias)]).where(
                        ing.resource_id == ingress_id
                    ).as_scalar().label('ingress_name')
                )
            else:
                _select.append(
                    select([func.max(ing.alias)]).where(and_(
                        ing.resource_id == ipc.resource_id,
                        ipc.ip == q_tab.c.ingress_ip,
                        ing.client_id == q_tab.c.ingress_client_id,
                    )
                    ).as_scalar().label('ingress_name')
                )
                _select.append(
                    select([func.max(ing.resource_id)]).where(and_(
                        ing.resource_id == ipc.resource_id,
                        ipc.ip == q_tab.c.ingress_ip,
                        ing.client_id == q_tab.c.ingress_client_id,
                    )
                    ).as_scalar().label('ingress_id')
                )
        if 'egress_client_id' in groups and 'egress_ip' in groups:
            if self.filtering.get("egress_id", None):
                egress_id = self.filtering.get("egress_id")
                _select.append(
                    select([func.max(ing.resource_id)]).where(
                        ing.resource_id == egress_id
                    ).as_scalar().label('egress_id')
                )
                _select.append(
                    select([func.max(ing.alias)]).where(
                        ing.resource_id == egress_id
                    ).as_scalar().label('egress_name')
                )
            else:
                _select.append(
                    select([func.max(ing.alias)]).where(and_(
                        ing.resource_id == ipc.resource_id,
                        ipc.ip == q_tab.c.egress_ip,
                        ing.client_id == q_tab.c.egress_client_id,
                    )
                    ).as_scalar().label('egress_name')
                )
                _select.append(
                    select([func.max(ing.resource_id)]).where(and_(
                        ing.resource_id == ipc.resource_id,
                        ipc.ip == q_tab.c.egress_ip,
                        ing.client_id == q_tab.c.egress_client_id,
                    )
                    ).as_scalar().label('egress_id')
                )

    def modify_query_from_filtering_for_list(self, filtering, **kwargs):
        groups = filtering.get('group', '').split(',')
        self.filtering = filtering
        if 'ingress_name' in groups or 'ingress_id' in groups:
            if 'ingress_name' in groups:
                groups.remove('ingress_name')
            if 'ingress_id' in groups:
                groups.remove('ingress_id')
            if not ('ingress_client_id' in groups and 'ingress_ip' in groups):
                groups += ['ingress_ip,ingress_client_id']
            filtering['group'] = ','.join(groups)
        if 'egress_name' in groups or 'egress_id' in groups:
            if 'egress_name' in groups:
                groups.remove('egress_name')
            if 'egress_id' in groups:
                groups.remove('egress_id')
            if not ('egress_client_id' in groups and 'egress_ip' in groups):
                groups += ['egress_ip,egress_client_id']
            filtering['group'] = ','.join(groups)
        return super().modify_query_from_filtering_for_list(filtering, **kwargs)


class ClientCdrReport(Report):
    entity_plural = 'Client Cdr Report items'
    scheme_class = DidReportGroppedRequestScheme
    scheme_class_get = ClientCdrReportGroppedResponseScheme
    scheme_class_get_inner = CdrReportGetScheme
    model_class = model.ClientCdr
    columns = model_class.get_fields()

    groupable = [fn for fn in columns if
                 getattr(model.ClientCdr, fn).type.__class__.__name__ in (
                     'String', 'DateTime', 'Text', 'SmallInteger') or '_id' in fn or fn in _grp] + \
                ['ingress_name', 'egress_name', 'ingress_id', 'egress_id']
    aggregable = [fn for fn in columns if not (
                  getattr(model.ClientCdr, fn).type.__class__.__name__ in (
                      'String', 'DateTime', 'Text', 'Boolean') or '_id' in fn or fn in _grp)] + \
                      ['ingress_total_calls', 'not_zero_calls', 'ingress_duration', 'egress_total_calls',
                       'start_time_of_date']

    def get_daily_cdr(self, cdr_date):
        cdr = model.client_cdr(cdr_date)
        return cdr

    def time_field(self, cdr):
        return cdr.c.time.label('time')

    def time_field_str(self):
        return 'time'

    def on_filter_o_trunk_type2(self, v, cdr):
        return cdr.c.o_trunk_type2 == v
    
    def on_select_ingress_total_calls(self, cdr):
        return cdr.c.is_final_call#case([(cdr.c.is_final_call == 1, 1)], else_=0)

    def on_select_egress_total_calls(self, cdr):
        return case([(cdr.c.is_final_call == cdr.c.is_final_call, 1)], else_=0)#cdr.c.is_final_call#case([(cdr.c.is_final_call == 1, 1)], else_=0)
    
    def on_select_ingress_duration(self, cdr):
        return case([(cdr.c.is_final_call == 1, cdr.c.call_duration)], else_=0)
    
    def on_select_not_zero_calls(self, cdr):
        return case([(cdr.c.call_duration > 0, 1)], else_=0)


# region +++ReportTemplate+++
class ReportTemplateCreate(DnlCreate):
    scheme_class = ReportTemplateScheme
    model_class = model.ReportTemplate
    entity = 'ReportTemplate'
    path_parameters = ()
    security = (DEFAULT_SECURITY)
    restrict = ()

    def before_create(self, obj, **kwargs):
        user = self.get_user(self.req)
        obj.created_by = user.name
        obj.created_on = datetime.now(UTC)
        if obj.period is not None:
            obj.params['start_time'], obj.params['end_time'] = obj.get_period_times(obj.period)
        report = Report()
        # t1 = obj.params.pop("end_time", int(datetime.now(UTC).timestamp()))
        # t0 = obj.params.pop("start_time", int((datetime.now(UTC) - timedelta(seconds=1)).timestamp()))
        report.run(**obj.params)
        return obj


class ReportTemplateResource(DnlResource):
    model_class = model.ReportTemplate
    scheme_class = ReportTemplateScheme
    scheme_class_get = ReportTemplateSchemeGet
    scheme_class_modify = ReportTemplateSchemeModify
    entity = 'ReportTemplate'
    id_field = 'id'
    security = (DEFAULT_SECURITY)
    path_parameters = ()
    restrict = ()

    # def get_object(self, resp, model_class, **kwargs):
    #     return model_class.filter(model_class.template_name == kwargs['template_name']).first()

    def before_update(self, obj, req):
        report = Report()
        t1 = int(datetime.now(UTC).timestamp())
        t0 = int((datetime.now(UTC) - timedelta(seconds=1)).timestamp())
        report.run(start_time=t0, end_time=t1, **req.data['params'])
        return obj


class ReportTemplateList(DnlList):
    scheme_class = ReportTemplateSchemeGet
    model_class = model.ReportTemplate
    entity_plural = 'ReportTemplates'
    path_parameters = ()
    security = (DEFAULT_SECURITY)
    restrict = ()

    def modify_query_from_filtering_for_list(self, filtering, **kwargs):
        filt, ret = super().modify_query_from_filtering_for_list(filtering, **kwargs)
        user = self.get_user(self.req)
        if not user.is_admin:
            cls = self.model_class
            # ret = ret.filter(cls.pool_id != 0)#TODO:filter for user
        return filt, ret


class ReportTemplateApply(CustomGetAction):
    model_class = model.ReportTemplate
    scheme_class_get = CdrReportGroppedResponseScheme
    path_parameters = ({'name': 'id'},)

    def apply(self, obj, req, resp, **kwargs):
        if obj:
            t0, t1 = obj.get_period_times(obj.period)
            if not 'start_time' in obj.params:
                items = Report().run(start_time=t0, end_time=t1, **obj.params)
            else:
                items = Report().run(**obj.params)
            total = len(items)
            _from = 0
            data = {
                'success': True,
                'data': items,
                'total': total, 'from': _from, 'count': len(items)
            }
            self.set_response(resp, SuccessReportResponse(data=data, scheme=self.scheme_class_get))
            return False
        self.set_response(resp, responses.ObjectNotFoundErrorResponse())
        return False
# endregion ---ReportTemplate---

class StatisticAPI(object):
    @classmethod
    def run(cls, **params):
        return Report().run(**params)


class StatisticAPI2(object):
    @classmethod
    def run(cls, **params):
        return ReportM().run(**params)
