from urllib.parse import urlencode, parse_qsl
from datetime import datetime, timedelta
from pytz import UTC
from dateutil.parser import parse as parse_datetime
from sqlalchemy.exc import InternalError, IntegrityError, DataError
from sqlalchemy import and_, or_, inspect, alias, text, func, select, case, cast
from sqlalchemy.orm import aliased
from api_dnl import model
import falcon
from falcon_rest import conf
from falcon_rest.logger import log
from falcon_rest import schemes
from falcon_rest.responses import responses
from falcon_rest.responses import errors
from falcon_rest.resources.resources import swagger, ATTRIBUTE_ERROR_RE
from falcon_rest.helpers import check_permission

from api_dnl.resources import check_context, DnlList, ValidationError, DnlCreate, CustomGetAction, CustomPostAction, \
    CustomDeleteAction
from api_dnl.schemes.stat import SystemCallStatScheme, SystemCallStatGetScheme
from api_dnl.scheme import SendCdrDirectScheme
from api_dnl.view import DEFAULT_SECURITY, OperationErrorResponse
import json
from api_dnl.views.report import Report


class SystemCallStats(CustomGetAction):
    model_class = model.User
    scheme_class = SystemCallStatScheme
    entity = 'VoipGateway system statistic report'
    id_field = 'switch_id'
    # body_parameters = ('Query for statistics', VoipGatewaySystemCallStatScheme)
    body_parameters = ()
    path_parameters = ({"name": "type", 'description': 'orig/term'},
                       {'name': 'period', 'type': 'string', 'description': "'hour', 'hours24', 'today', 'week', 'month', 'year', 'all'",
                        "default": "hours24"},)

    additional_responses = (responses.SuccessResponseObjectInfo(payload_scheme=SystemCallStatGetScheme,
                                                                description='System Call statistics result'),)

    def apply(self, obj, req, resp, **kwargs):
        try:
            user = self.get_user(self.req)
            params = dict(parse_qsl(req.query_string))
            orig_type = kwargs.get('type', 'term')
            types = ['term', 'orig']
            if orig_type not in types:
                raise Exception('wrong type {} must be in {}'.format(orig_type, ','.join(types)))
            period = kwargs.get('period', 'hours24')
            periods = ['hour', 'hours24', 'today', 'week', 'month', 'year', 'all']
            if period not in periods:
                raise Exception('wrong period {} must be in {}'.format(period, ','.join(periods)))

            t1 = datetime.now(UTC)
            if period == 'hour':
                t0 = t1 - timedelta(hours=1)
            elif period == 'hours24':
                t0 = t1 - timedelta(hours=24)
            elif period == 'today':
                t0 = t1.replace(hour=0, minute=0, second=0, microsecond=0)
            elif period == 'week':
                t0 = (t1 - timedelta(days=t1.weekday())).replace(hour=0, minute=0, second=0, microsecond=0)
            elif period == 'month':
                t0 = t1.replace(day=1, hour=0, minute=0, second=0, microsecond=0)
            elif period == 'year':
                t0 = t1.replace(month=1, day=1, hour=0, minute=0, second=0, microsecond=0)
            elif period == 'all':
                t0 = t1.replace(year=2000, month=1, day=1, hour=0, minute=0, second=0, microsecond=0)

            d0 = int(t0.timestamp())
            d1 = int(t1.timestamp())
            res = model.Resource
            qos = model.QosResource
            res_ids = []
            if orig_type == 'term':
                trunk_type2 = 0
            else:
                trunk_type2 = 1

            in_res_ids = [r.resource_id for r in res.session().query(res.resource_id).filter(and_(res.trunk_type2 == trunk_type2, res.ingress))]
            out_res_ids = [r.resource_id for r in res.session().query(res.resource_id).filter(and_(res.trunk_type2 == trunk_type2, res.egress))]
            res_ids = [r.resource_id for r in res.session().query(res.resource_id).filter(and_(res.trunk_type2 == trunk_type2, res.ingress))]
            cl_res_ids = set([])
            if user.user_type == 'client':
                cl_res_ids = set([r.resource_id for r in user.client.resources])
            if user.user_type == 'agent':
                cl_res_ids = set(user.agent.res_id_list)

            if cl_res_ids:
                in_res_ids = list(cl_res_ids.intersection(set(in_res_ids)))
                out_res_ids = list(cl_res_ids.intersection(set(out_res_ids)))
                res_ids = list(cl_res_ids.intersection(set(res_ids)))

            qos_query = qos.session().query(func.sum(qos.call).label('call'),
                                            func.sum(qos.channels).label('channel'),
                                            func.sum(qos.cps).label('cps'),
                                            ).filter(and_(qos.report_time >= t0, qos.report_time < t1))

            qos_in = qos_query.filter(qos.res_id.in_(in_res_ids)).first()
            qos_out = qos_query.filter(qos.res_id.in_(out_res_ids)).first()

            inbound_channel = 0
            inbound_cps = 0
            outbound_channel = 0
            outbound_cps = 0
            connected_calls = 0
            if qos_in:
                inbound_channel = qos_in.channel or 0
                inbound_cps = qos_in.cps or 0
                connected_calls = qos_in.call or 0
            if qos_out:
                outbound_channel = qos_out.channel or 0
                outbound_cps = qos_out.cps or 0
                connected_calls = qos_out.call or 0

            data = dict(
                inbound_channel=inbound_channel, inbound_cps=inbound_cps,
                outbound_channel=outbound_channel, outbound_cps=outbound_cps,
                connected_calls=connected_calls
            )

            log.debug('SystemCallStats returned %s' % str(data))
            self.set_response(resp,
                              responses.SuccessResponseObjectInfo(data=data, payload_scheme=SystemCallStatGetScheme))
            return False
        except Exception as e:
            self.set_response(resp, OperationErrorResponse(e)
                              # responses.OperationErrorResponse(data=errors.Error(code=406, message=str(e), reason=e.__class__.__name__))
                              )
            return False

    def x(self):
        pass
