from datetime import datetime, timedelta

from pytz import UTC
from sqlalchemy import (and_)
from sqlalchemy.orm import foreign
from sqlalchemy.sql import func

from api_dnl import model, settings
from api_dnl.model import Client, Resource, SystemParameter, FaultRouteAlertRule, FaultRouteAlertRuleLog, \
    ResourcePrefix, MailSender
from api_dnl.views.report import Report
from api_dnl.tasks import app, log, db, SqlAlchemyTask
from api_dnl.utils.imp_exp import (csv2xls, dict_to_csv)

TEST_TO = settings.TEST_TO


@app.task(base=SqlAlchemyTask)
def do_fault_route_alert_rules():
    log.debug('Start do_fault_route_alert_rules!')
    query = FaultRouteAlertRule.filter(
        and_(FaultRouteAlertRule.is_active, FaultRouteAlertRule.next_run_on <= datetime.now(UTC)))
    for obj in query.all():
        do_fault_route_alert_rule(obj.id)
    pass


@app.task(base=SqlAlchemyTask)
def do_fault_route_alert_rule(rule_id):
    log.debug('Start do_fault_route_alert_rule {}'.format(rule_id))
    rule = FaultRouteAlertRule.get(rule_id)
    rule_log = FaultRouteAlertRuleLog(rule_id=rule_id, started_on=datetime.now(UTC))
    if rule:
        try:
            rule_log.profitability = 0.0
            rule_log.profitable_trunks = 0

            trunks = []
            q = ResourcePrefix.session().query(ResourcePrefix.resource_id)
            if rule.product_id:
                q = q.filter(ResourcePrefix.product_id == rule.product_id)
            if rule.routing_plan_id:
                q = q.filter(ResourcePrefix.routing_plan_id == rule.routing_plan_id)
            q = q.group_by(ResourcePrefix.resource_id)
            trunks = [str(r.resource_id) for r in q.all()]
            if trunks:
                end_time = datetime.now(UTC)
                start_time = end_time - timedelta(days=1)
                if rule.look_days:
                    start_time = end_time - timedelta(days=rule.look_days)
                d0 = int(start_time.timestamp())
                d1 = int(end_time.timestamp())
                ingress_id = ','.join(trunks)
                all = Report().run(start_time=d0, end_time=d1, field='ingress_call_cost,egress_call_cost',
                                   ingress_id=ingress_id)
                if len(all):
                    a0 = all[0]
                    rule_log.profitability = (
                                                     a0.ingress_call_cost - a0.egress_call_cost) / a0.ingress_call_cost * 100 if a0.ingress_call_cost != 0 else 0
                all = Report().run(start_time=d0, end_time=d1, field='ingress_call_cost,egress_call_cost',
                                   ingress_id=ingress_id,
                                   group='ingress_id')

                for a0 in all:
                    if a0.ingress_call_cost > a0.egress_call_cost:
                        rule_log.profitable_trunks = rule_log.profitable_trunks + 1
            if rule_log.profitable_trunks < rule.min_profitable_trunks or rule_log.profitability < rule._min_profitability:
                rule_log.status = 'alert'
                rule_log.result = 'condition true'
                rule_log.save()
                if rule.email_to:
                    res = MailSender.apply_mail(rule_log, 'faultroutealertrule_{}'.format(rule_id), to=rule.email_to)
                    log.info('fault_route_alert_rule email {}'.format(res))
                    if res:
                        rule_log.result = 'condition true,email failed {}'.format(res)
                        rule_log.mail_sent = False
                    else:
                        rule_log.mail_sent = True
            else:
                rule_log.status = 'ok'
                rule_log.result = ''
            rule_log.finished_on = datetime.now(UTC)
            rule_log.save()
            rule.last_run_on = rule_log.finished_on
            rule.next_run_on = rule.next_run(datetime.now(UTC))
            rule.save()
        except Exception as e:
            db.session.rollback()
            log.error('do_fault_route_alert_rule {} error {}'.format(rule_id, str(e)))
            rule_log.result = str(e)
            rule_log.status = 'error'
            rule_log.finished_on = datetime.now(UTC)
            rule_log.save()
            rule.last_run_on = rule_log.finished_on
            rule.next_run_on = rule.next_run(datetime.now(UTC))
            rule.save()
    else:
        log.error('do_fault_route_alert_rule {} rule not found'.format(rule_id))
        rule_log.result = 'rule {} not found'.format(rule_id)
        rule_log.status = 'error'
        rule_log.finished_on = datetime.now(UTC)
        rule_log.save()
        rule.last_run_on = rule_log.finished_on
        rule.next_run_on = rule.next_run(datetime.now(UTC))
        rule.save()
