from urllib.parse import urlencode, parse_qsl
from datetime import datetime, timedelta
from pytz import UTC
import falcon
import mimetypes
from dateutil.parser import parse as parse_datetime
from sqlalchemy.exc import InternalError, IntegrityError, DataError
from sqlalchemy import and_, or_, inspect, alias, text, func, select, case, cast
from sqlalchemy import Integer
from sqlalchemy.orm import aliased
from api_dnl import model, settings
from falcon_rest import conf
from falcon_rest.logger import log
from falcon_rest import schemes
from falcon_rest.responses import responses
from falcon_rest.responses import errors, ObjectCreatedResponse
from falcon_rest.resources.resources import swagger, ATTRIBUTE_ERROR_RE
from falcon_rest.helpers import check_permission
from falcon_rest.db.errors import IntegrityError, FkConstraintViolation, NoResultFound

from api_dnl.resources import check_context, DnlCreate, DnlResource, DnlList, ValidationError, CustomGetAction, \
    CustomPostAction, CustomDeleteAction, CustomPatchAction,DnlResourceAll
from api_dnl.scheme import ObjectCreatedMultipleScheme,ObjectUpdatedScheme
from api_dnl.scheme import DATE_REGEXP
from api_dnl.schemes.invoice import InvoiceScheme, InvoiceGenerationScheme, InvoiceSendScheme, InvoiceFileGetScheme, \
    InvoiceReCreateScheme, InvoiceGetScheme, InvoiceGenerationManyScheme, InvoiceVoidScheme, InvoiceModifyScheme, \
    InvoiceSummaryScheme, InvoiceSummaryDataSchemeGet, InvoiceSummarySchemeModify, InvoiceSummarySchemeGet, \
    InvoiceTaskScheme, InvoiceTaskSchemeGet, InvoiceTaskSchemeModify,InvoiceAllStatusScheme, InvoiceTaskManyScheme
from api_dnl.view import DEFAULT_SECURITY, OperationErrorResponse, client_context
import json


# --------------Invoice

class InvoiceManualCreate(DnlCreate):
    scheme_class = InvoiceScheme
    entity = 'Invoice'
    unique_field = 'invoice_id'
    additional_responses = ()
    path_parameters = ()
    security = (DEFAULT_SECURITY)
    restrict = ()
    path_parameters = ({'name': 'client_id', 'description': 'Parent carrier'},)
    body_params = ()

    def get_spec(self):
        additional_responses = (
                                   responses.OperationErrorResponse(data=errors.CommonErrors.PermissionError,
                                                                    description='Can\'t create file'),
                               ) + self.additional_responses

        ext_body_parameters = self.body_params or (
            {
                'name': 'pdf_file',
                'in': 'formData',
                'description': 'Invoice File to upload',
                'required': True,
                'type': 'file'
            },
            {
                'name': 'start_date',
                'in': 'formData',
                'description': 'Invoice start date',
                'required': True,
                'type': 'datetime'
            },
            {
                'name': 'end_date',
                'in': 'formData',
                'description': 'Invoice end date',
                'required': True,
                'type': 'datetime'
            },
        )

        return swagger.specify.get_spec(
            method='post', description='Creates new {}'.format(self.entity.lower()),
            consumes=['multipart/form-data'],
            path_parameters=self.path_parameters,
            responses=(
                          self.scheme_class.get_object_created_response(entity=self.entity),
                          # ,scheme=ObjectCreatedWithErrorsScheme),
                      ) + additional_responses + self.get_additional_responses(method='post'),
            security=self.get_security(method='post'),
            # body_parameters=('{} to create'.format(self.entity), self.scheme_class),
            ext_body_parameters=ext_body_parameters
        )

    def on_post(self, req, resp, **kwargs):
        kwargs = client_context(self, req, resp, **kwargs)
        return self._on_post(req, resp, **kwargs)

    def before_create(self, obj, **kwargs):
        obj.client_id = kwargs['client_id']
        self.req_data['state'] = 'void'
        obj.state = 'verify'
        obj.update_by = self.get_user().name
        obj.invoice_number = str(int(model.Invoice.max_number()))
        try:
            obj.current_balance = model.C4ClientBalance.client_balance(obj.client_id)
        except:
            obj.current_balance = 0
            pass
        obj.create_type = 'manual'
        obj.pdf_path = settings.FILES['invoiced'] + '/'
        file_path = obj.pdf_path + obj.invoice_number + '.pdf'
        file = self.req.files['pdf_file']
        byte_str = file.read()
        local_file = open(file_path, 'wb')
        local_file.write(byte_str)
        return obj


class InvoiceCreate(DnlCreate):
    scheme_class = InvoiceGenerationScheme
    entity = 'Invoice'
    unique_field = 'invoice_id'
    additional_responses = ()
    path_parameters = ()
    security = (DEFAULT_SECURITY)
    restrict = ()
    path_parameters = ({'name': 'client_id', 'description': 'Parent carrier'},)

    def before_create(self, obj, **kwargs):
        obj.client_id = kwargs['client_id']
        self.req_data['state'] = 'void'
        obj.state = 'normal'
        obj.update_by = self.get_user().name
        obj.invoice_number = str(int(model.Invoice.max_number()) + 1)
        try:
            obj.current_balance = model.C4ClientBalance.client_balance(obj.client_id)
        except:
            obj.current_balance = 0
            pass
        obj.create_type = 'manual'
        return obj

    def create_object(self, req, scheme, **kwargs):
        from api_dnl.task.invoice import render_invoice
        data = req.data
        data['client_id'] = int(kwargs['client_id'])
        obj = render_invoice(data)
        # obj=resources.BaseResource.create_object(self, req, scheme, **kwargs)
        obj.update_by = self.get_user(req).name
        # obj.client_id = kwargs['client_id']
        return obj.invoice_id


class InvoiceReCreate(CustomPostAction):
    scheme_class = InvoiceSendScheme
    model_class = model.Invoice
    entity_plural = 'Invoice'
    security = (DEFAULT_SECURITY)
    restrict = ()
    path_parameters = ({'name': 'invoice_id', 'description': 'regenerate'},)

    # body_parameters = ('Emails to send', InvoiceReCreateScheme)

    def apply(self, obj, req, resp, **kwargs):
        obj = self.model_class.get(int(kwargs['invoice_id']))
        user = self.get_user(req)
        if obj:
            obj.state = 'void'
            obj.save()
            old = model.InvoiceSummary.get(int(obj.invoice_number))
            if old:
                t = old.task
                task = model.InvoiceTask(client_id=t.client_id,
                                         bill_period_start=t.bill_period_start,
                                         bill_period_end=t.bill_period_end,
                                         show_acc_summary=t.show_acc_summary,
                                         show_payment_applied=t.show_payment_applied,
                                         show_rec_charge=t.show_rec_charge,
                                         show_non_rec_charge=t.show_non_rec_charge,
                                         show_usage_charge=t.show_usage_charge,
                                         show_trunk_detail=t.show_trunk_detail,
                                         show_daily_summary=t.show_daily_summary,
                                         show_jd_daily_summary=t.show_jd_daily_summary,
                                         show_jd_trunk_detail=t.show_jd_trunk_detail,
                                         show_prefix_summary=t.show_prefix_summary,
                                         show_code_name_summary=t.show_code_name_summary,
                                         show_country_summary=t.show_country_summary,
                                         show_inbound_summary=t.show_inbound_summary,
                                         show_top_100tn_summary=t.show_top_100tn_summary,
                                         show_did_charge_breakdown=t.show_did_charge_breakdown,
                                         show_did_charge_summary=t.show_did_charge_summary,
                                         enable_email=t.enable_email,
                                         enable_email_with_cdr=t.enable_email_with_cdr,
                                         include_tax=t.include_tax,
                                         include_scc_charges=t.include_scc_charges,
                                         payment_dur_date=t.payment_dur_date,
                                         invoice_zone=t.invoice_zone,
                                         is_auto=t.is_auto,
                                         created_by_user_id=user.user_id
                                         )
                task_id = task.save()
                from api_dnl.task.invoice import do_invoice_task
                from kombu.exceptions import OperationalError
                try:
                    do_invoice_task.delay(task_id)
                except OperationalError as e:
                    do_invoice_task(task_id)
                    pass
                self.set_response(resp, ObjectCreatedResponse(data=task_id))
                return False
            else:
                self.set_response(resp, responses.ObjectNotFoundErrorResponse())
                return False
        else:
            self.set_response(resp, responses.ObjectNotFoundErrorResponse())
            return False


class InvoiceReCreateOld(DnlCreate):
    scheme_class = InvoiceReCreateScheme
    entity = 'Invoice'
    unique_field = 'invoice_id'
    additional_responses = ()
    path_parameters = ()
    security = (DEFAULT_SECURITY)
    restrict = ()
    path_parameters = ({'name': 'invoice_id', 'description': 'Regenerate this invoice id'},)

    def before_create(self, obj, **kwargs):
        cls = model.Invoice
        inv = cls.get(int(kwargs['invoice_id']))
        if not inv:
            raise NoResultFound
        inv.state = 'void'
        inv.save()

        obj.client_id = inv.client_id

        obj.state = 'normal'
        obj.update_by = self.get_user().name
        obj.invoice_number = str(int(model.Invoice.max_number()) + 1)
        try:
            obj.current_balance = model.C4ClientBalance.client_balance(obj.client_id)
        except:
            obj.current_balance = 0
            pass
        obj.create_type = 'manual'
        return obj

    def create_object(self, req, scheme, **kwargs):
        from api_dnl.task.invoice import render_invoice
        cls = model.Invoice
        inv = cls.get(int(kwargs['invoice_id']))
        if not inv:
            raise NoResultFound
        inv.state = 'void'
        inv.save()
        data = dict(client_id=inv.client_id,
                    start_date=inv.invoice_start,
                    end_date=inv.invoice_start,
                    gmt=inv.invoice_zone,
                    show_cdr=inv.is_show_code_name,
                    show_trunk_breakdown=inv.is_show_total_trunk,
                    show_rate_table_breakdown=inv.is_show_daily_usage,
                    show_daily_breakdown=inv.is_show_by_date,
                    send_invoice_as_link=inv.is_send_as_link,
                    show_jurisdiction_detail=inv.invoice_jurisdictional_detail,
                    include_account_detail=inv.is_invoice_account_summary,
                    show_payment_summary=inv.invoice_include_payment,
                    show_detail_by_trunk=inv.is_show_detail_trunk,
                    include_cdr_in_email=False,
                    show_detail_by_code_name=True,
                    rate_decimal=inv.decimals_num,
                    show_code_summary=inv.is_show_code_100,
                    auto_send_invoice=False,
                    invoice_time=datetime.now(UTC),
                    due_date=inv.due_date,
                    show_detail_by_country=inv.is_show_country
                    )
        obj = render_invoice(data)
        # obj=resources.BaseResource.create_object(self, req, scheme, **kwargs)
        obj.update_by = self.get_user(req).name
        # obj.client_id = kwargs['client_id']
        return obj.invoice_id


class InvoiceVoid(CustomPostAction):
    scheme_class = InvoiceVoidScheme
    model_class = model.Invoice
    entity = 'Invoice'
    unique_field = 'invoice_id'
    additional_responses = ()
    path_parameters = ()
    security = (DEFAULT_SECURITY)
    restrict = ()
    path_parameters = ({'name': 'invoice_id', 'description': 'Void this invoice id'},)

    def apply(self, obj, req, resp, **kwargs):
        if not obj:
            self.set_response(resp, responses.ObjectNotFoundErrorResponse(
                data={'invoice_id not provided in path!'}))
            return False

        obj.state = -1
        obj.save()

        return True


class InvoiceManyCreate(DnlCreate):
    scheme_class = InvoiceGenerationManyScheme
    entity = 'Invoice'
    unique_field = 'invoice_id'
    additional_responses = ()
    path_parameters = ()
    security = (DEFAULT_SECURITY)
    restrict = ()
    path_parameters = ()
    additional_responses = (ObjectCreatedResponse(scheme=ObjectCreatedMultipleScheme),)

    def _on_post(self, req, resp, **kwargs):
        scheme = self.scheme_class()
        if not check_permission(self, req, 'create'):
            self.set_response(
                resp, responses.ForbiddenErrorResponse(data=errors.AuthErrors.Forbidden)
            )
            return
        try:
            if self.check_request_data(req, resp, scheme):
                ret = []
                for client_id in req.data['clients']:
                    create = InvoiceCreate()
                    kwargs['client_id'] = client_id
                    create.on_post(req, resp, **kwargs)
                    if resp.status != falcon.HTTP_200:
                        return
                    object_id = json.loads(resp.body)['object_id']
                    ret.append(object_id)
                resp.data = {'success': True, 'object_id': ret}
                self.set_response(resp, ObjectCreatedResponse(scheme=ObjectCreatedMultipleScheme, data_key='object_ids',
                                                              data=ret))
                return True
        except Exception as e:
            self.set_response(resp, responses.OperationErrorResponse(e))
            return True


class InvoiceResource(DnlResource):
    model_class = model.Invoice
    scheme_class = InvoiceModifyScheme
    scheme_class_get = InvoiceGetScheme
    scheme_class_modify = InvoiceModifyScheme
    entity = 'Invoice'
    id_field = 'invoice_id'
    has_update_by = True
    # has_update_operation = False
    # has_delete_operation =  False
    unique_field = 'invoice_number'
    security = (DEFAULT_SECURITY)
    restrict = ()

class InvoiceDeleteResource(InvoiceResource):
    has_modify_operation = False
    has_delete_operation =  True
    has_info_operation = False

class InvoiceClientList(DnlList):
    scheme_class = InvoiceGetScheme  # InvoiceClientGetScheme
    model_class = model.Invoice
    entity_plural = 'Invoices'
    path_parameters = ()
    security = (DEFAULT_SECURITY)
    restrict = ()
    path_parameters = ({'name': 'client_id', 'description': 'Parent carrier'},)

    def get_filtering_for_list(self, parsed_qs, **kwargs):
        ret = super(InvoiceClientList, self).get_filtering_for_list(parsed_qs, **kwargs)
        ret['client_id'] = kwargs['client_id']
        return ret


class InvoiceList(DnlList):
    scheme_class = InvoiceGetScheme
    model_class = model.Invoice
    entity_plural = 'Invoices'
    path_parameters = ()
    security = (DEFAULT_SECURITY)
    restrict = ()

    # path_parameters=( {'name': 'client_id', 'description': 'Parent carrier'}, )

    # def get_filtering_for_list(self, parsed_qs, **kwargs):
    #    ret = super(DnlList, self).get_filtering_for_list(parsed_qs, **kwargs)
    #    ret['client_id'] = kwargs['client_id']
    #    return ret

    def on_filter_state(self, query, value, kwargs):
        if 'all' not in value:
            query = query.filter(self.model_class.state.in_(value.split(',')))
        return query

    def on_filter_invoice_time_lt(self, query, value, kwargs):
        import re
        cls = self.model_class
        if re.match(DATE_REGEXP, value):
            value = value + ' 23:59:59'
        query = query.filter(cls.invoice_time <= value)
        return query

    def modify_query_from_filtering_for_list(self, filtering, **kwargs):
        cls = self.model_class
        filt, ret = super(InvoiceList, self).modify_query_from_filtering_for_list(filtering, **kwargs)
        ret = ret.filter(model.user_id_filter(cls, self.get_user(self.req).user_id))
        if 'did' in self.req.url:
            filt['company_type'] = 'origination'
            # if not 'create_type' in self.req.url:
            #     filt['create_type']='manual'
        else:
            filt['company_type'] = 'termination'
        return filt, ret

    def modify_query_from_ordering_for_list(self, ordering, query, **kwargs):
        model_class = self.model_class

        def ToInt(numb):
            try:
                return cast(func.nullif(func.regexp_replace(numb, "\\D", "", "g"), ""), Integer)
            except:
                return text("")

        if 'by' in ordering and ordering['by'] == 'invoice_number':
            model_class = self.model_class
            if ordering['dir'] == 'desc':
                query = query.order_by(ToInt(model.Invoice.invoice_number).desc())
            else:
                query = query.order_by(ToInt(model.Invoice.invoice_number))

            ordering = {}

        return ordering, query


class InvoiceFileGet(DnlResource):
    no_auth_needed = True  # todo what todo

    model_class = model.Invoice
    scheme_class = InvoiceScheme
    scheme_class_get = InvoiceFileGetScheme
    entity = 'Invoice pdf file'
    id_field = 'invoice_id'
    has_modify_operation = False
    has_delete_operation = False
    security = (DEFAULT_SECURITY)
    restrict = ()

    def get_spec_info(self):
        spec = super(InvoiceFileGet, self).get_spec_info()
        spec['get']['produces'] = ['application/pdf', 'text/html']
        return spec

    def on_get(self, req, resp, **kwargs):
        if not self.has_info_operation:  # pragma: no cover
            return self.method_not_allowed_response(resp, 'GET')
        if not hasattr(self, 'no_auth_needed') or (hasattr(self, 'no_auth_needed ') and not self.no_auth_needed):
            if not check_permission(self, req, 'show', self.get_object(resp, self.model_class, **kwargs)):
                self.set_response(
                    resp, responses.ForbiddenErrorResponse(data=errors.AuthErrors.Forbidden)
                )
                return
        if 'invoice_id' not in kwargs:
            self.set_response(resp, responses.ObjectNotFoundErrorResponse(
                data={'invoice_id not provided in path!'}))
        try:
            obj = self.get_object(resp, self.model_class, **kwargs)
            if obj:
                if obj.pdf_path:
                    file_path = '{}/{}'.format(settings.FILES['invoices'], obj.pdf_path)
                    file = open(file_path, 'rb')
                    resp.data = file.read()
                    resp.content_type = mimetypes.guess_type(obj.pdf_path)[0]
                    resp.append_header('Content-Disposition', 'attachment; filename="{}"'.format(obj.pdf_path))
                    resp.status = falcon.HTTP_200
                else:
                    self.set_response(resp, responses.ObjectNotFoundErrorResponse(
                        data={'result': 'Invoice {} pdf file not generated'.format(kwargs['invoice_id'])}))
            else:
                self.set_response(resp, responses.ObjectNotFoundErrorResponse(
                    data={'result': 'Invoice {} not found'.format(kwargs['invoice_id'])}))
        except Exception as e:
            self.set_response(resp, OperationErrorResponse(e))


class InvoiceSend(CustomPatchAction):
    scheme_class = InvoiceSendScheme
    model_class = model.Invoice
    entity_plural = 'Invoice'
    security = (DEFAULT_SECURITY)
    restrict = ()
    path_parameters = ({'name': 'invoice_id', 'description': 'mark as sent'},)
    body_parameters = ('Emails to send', InvoiceSendScheme)

    def apply(self, obj, req, resp, **kwargs):
        if obj:
            objs = model.InvoiceSummary.get(obj.invoice_number)
            pdf = obj.file_data
            if pdf:
                if objs.config.invoice_send_mode == 'attachment':
                    att = [(obj.pdf_path.split('.')[-1], pdf)]
                else:
                    att = []
            else:
                log.error('Invoice has no pdf file to send {}'.format(obj.invoice_id))
                self.set_response(resp, responses.OperationErrorResponse(
                    data=errors.Error(code=44, message='Invoice has no pdf file to send', reason='invoice send error')))
                return False
                att = []
            invoice_mail = model.InvoiceEmail()
            invoice_mail.invoice_no = obj.invoice_number
            if obj.ingress_cdr_file and objs.config.invoice_send_mode_cdr == 'attachment':
                att.append(('csv', obj.cdr_data))
            if 'to_emails' in req.data:
                if not req.data['to_emails']:
                    raise ValidationError('invallid to_emails')
                invoice_mail.send_address = req.data['to_emails']
                ret = model.MailSender.apply_mail(obj, 'invoice', to=req.data['to_emails'], att=att)
            else:
                ret = model.MailSender.apply_mail(obj, 'invoice', att=att)
            invoice_mail.save()
            if ret:
                self.set_response(resp, responses.OperationErrorResponse(
                    data=errors.Error(code=43, message=ret[1], reason='mail error')))
                return False
            if obj.client.payment_term and obj.due_date is None:
                obj.due_date = (datetime.now(UTC) + timedelta(days=(obj.client.payment_term.grace_days or 1))).date()
            obj.state = 'send'
            obj.save()
            return True
        self.set_response(resp, responses.ObjectNotFoundErrorResponse(
            data={'result': 'Invoice {} not found'.format(kwargs['invoice_id'])}))
        return False


class InvoiceAllDelete(DnlResourceAll):
    #has_modify_operation = False
    scheme_class = InvoiceAllStatusScheme
    model_class = model.Invoice
    scheme_class_get = ObjectUpdatedScheme
    entity = 'Invoice'

# ------ Invoice ------


# region +++InvoiceSummary+++
class InvoiceSummaryCreate(DnlCreate):
    scheme_class = InvoiceSummaryScheme
    model_class = model.InvoiceSummary
    entity = 'InvoiceSummary'
    path_parameters = ()
    security = (DEFAULT_SECURITY)
    restrict = ()

    def before_create(self, obj, **kwargs):
        user = self.get_user(self.req)
        # obj.created_by=user.name
        # obj.created_on=datetime.now(UTC)
        return obj


class InvoiceSummaryResource(DnlResource):
    model_class = model.InvoiceSummary
    scheme_class = InvoiceSummaryScheme
    scheme_class_get = InvoiceSummaryDataSchemeGet
    scheme_class_modify = InvoiceSummarySchemeModify
    entity = 'InvoiceSummary'
    id_field = 'invoice_num'
    security = (DEFAULT_SECURITY)
    path_parameters = ()
    restrict = ()
    no_auth_needed = True
    has_delete_operation = False
    has_modify_operation = False


class InvoiceSummaryList(DnlList):
    scheme_class = InvoiceSummarySchemeGet
    model_class = model.InvoiceSummary
    entity_plural = 'InvoiceSummarys'
    path_parameters = ()
    security = (DEFAULT_SECURITY)
    restrict = ()

    def modify_query_from_filtering_for_list(self, filtering, **kwargs):
        filt, ret = super().modify_query_from_filtering_for_list(filtering, **kwargs)
        user = self.get_user(self.req)
        if not user.is_admin:
            cls = self.model_class
        # ret = ret.filter(cls.pool_id != 0)#TODO:filter for user
        return filt, ret


class InvoiceData(CustomGetAction):
    model_class = model.InvoiceSummary
    scheme_class = InvoiceSummaryDataSchemeGet
    path_parameters = ({'name': 'invoice_num', 'description': 'invoice md5 number'},)
    restrict = ()
    no_auth_needed = True
    has_delete_operation = False
    has_modify_operation = False

    def on_get(self, req, resp, **kwargs):
        cls = self.model_class
        invoice_num = kwargs.get('invoice_num', '')
        obj = cls.filter(cls.invoice_num == invoice_num).first()
        if obj:
            resp.data = json.dumps(obj.json_content).encode()
            resp.content_type = 'application/json'
            resp.status = falcon.HTTP_200
        else:
            resp.data = str.encode(json.dumps({'success': False, 'message': 'Not found'}))
            resp.content_type = 'application/json'
            resp.status = falcon.HTTP_404


# endregion ---InvoiceSummary---

# region +++ManualInvoiceTask+++
class ManualInvoiceTaskCreate(DnlCreate):
    scheme_class = InvoiceTaskScheme
    model_class = model.InvoiceTask
    entity = 'ManualInvoiceTask'
    path_parameters = ()
    security = (DEFAULT_SECURITY)
    restrict = ()

    def before_create(self, obj, **kwargs):
        user = self.get_user(self.req)
        conf = model.SystemParameter.get(1)
        if obj.client_id:
            cls = model.ClientInvoiceSetting
            invoice_setting = cls.filter(cls.client_id == obj.client_id).first()
            if invoice_setting and invoice_setting.setting:
                conf = invoice_setting.setting
        if conf and conf.overlap_invoice_protection:
            cls = model.Invoice
            q = cls.filter(and_(cls.client_id==obj.client_id,cls.invoice_start>=obj.bill_period_start,
                              cls.invoice_start<=obj.bill_period_start,cls.state!=-1)).first()
            if q:
                raise Exception('Overlap invoice number {} start {} end {}'.format(
                    q.invoice_number, q.invoice_start, q.invoice_end)
                )
        obj.created_by_user_id = user.user_id
        obj.create_time = datetime.now(UTC)
        obj.status = 'Initial'
        obj.is_auto = False
        return obj

    def after_create(self, object_id, req, resp, **kwargs):
        from kombu.exceptions import OperationalError
        from api_dnl.task.invoice import do_invoice_task
        try:
            do_invoice_task.delay(object_id)
            log.debug('ManualInvoiceTaskCreate scheduled success')
        except OperationalError as e:
            log.debug('ManualInvoiceTaskCreate error:{}'.format(e))
            do_invoice_task(object_id)
            log.debug('ManualInvoiceTaskCreate finished success')

class ManualInvoiceTaskManyCreate(DnlCreate):
    scheme_class = InvoiceTaskManyScheme
    model_class = model.InvoiceTask
    entity = 'ManualInvoiceTask'
    path_parameters = ()
    security = (DEFAULT_SECURITY)
    restrict = ()

    def _on_post(self, req, resp, **kwargs):
        scheme = self.scheme_class()
        if not check_permission(self, req, 'create'):
            self.set_response(
                resp, responses.ForbiddenErrorResponse(data=errors.AuthErrors.Forbidden)
            )
            return
        try:
            if self.check_request_data(req, resp, scheme):
                ret = []
                for client_id in req.data['client_id']:
                    create = ManualInvoiceTaskCreate()
                    new_req = req
                    new_req.data['client_id'] = client_id
                    create.on_post(new_req, resp, **kwargs)
                    if resp.status != falcon.HTTP_200:
                        return
                    object_id = json.loads(resp.body)['object_id']
                    ret.append(object_id)
                resp.data = {'success': True, 'object_id': ret}
                self.set_response(resp, ObjectCreatedResponse(scheme=ObjectCreatedMultipleScheme, data_key='object_ids',
                                                              data=ret))
                return True
        except Exception as e:
            self.set_response(resp, responses.OperationErrorResponse(e))
            return True

class ManualInvoiceTaskResource(DnlResource):
    model_class = model.InvoiceTask
    scheme_class = InvoiceTaskScheme
    scheme_class_get = InvoiceTaskSchemeGet
    scheme_class_modify = InvoiceTaskSchemeModify
    entity = 'ManualInvoiceTask'
    id_field = 'id'
    security = (DEFAULT_SECURITY)
    path_parameters = ()
    restrict = ()


class ManualInvoiceTaskList(DnlList):
    scheme_class = InvoiceTaskSchemeGet
    model_class = model.InvoiceTask
    entity_plural = 'ManualInvoiceTasks'
    path_parameters = ()
    security = (DEFAULT_SECURITY)
    restrict = ()

    def on_filter_client_type(self, q, val, kwargs):
        cls = self.model_class
        if val == 'carrier':
            return q.filter(cls.client_type.is_(None))
        else:
            return q.filter(cls.client_type == val)

    def modify_query_from_filtering_for_list(self, filtering, **kwargs):
        filt, ret = super().modify_query_from_filtering_for_list(filtering, **kwargs)
        user = self.get_user(self.req)
        if not user.is_admin:
            cls = self.model_class
        # ret = ret.filter(cls.pool_id != 0)#TODO:filter for user
        return filt, ret

# endregion ---ManualInvoiceTask---
