from marshmallow import validate, pre_load
from marshmallow.fields import Bool, Str, Decimal, Int, Nested, DateTime, Float, List, UUID, Dict

from api_dnl import model
from api_dnl.fields import Choice
from api_dnl.scheme import Schema, BaseModelScheme, _valid, _valid_unique
from api_dnl.scheme import IP_REGEXP, NAME_REGEXP, PREFIX_REGEXP, TIME_REGEXP, ValidationError, SuccessScheme


# region +++DnlCloudDbmanCfg+++
class DnlCloudDbmanCfgScheme(BaseModelScheme):
    server_name = Str()
    server_use_ssl = Bool()
    cdr_storage_cfg_uuid = Str(validate=lambda value: _valid('DnlCloudStorage', 'uuid', value))
    pcap_storage_cfg_uuid = Str(validate=lambda value: _valid('DnlCloudStorage', 'uuid', value))
    keep_cdr = Int(allow_none=True)
    keep_pcap = Int(allow_none=True)
    keep_cdr_archives = Int(allow_none=True)
    keep_pcap_archives = Int(allow_none=True)
    keep_cdr_remote = Int(allow_none=True)
    keep_pcap_remote = Int(allow_none=True)
    hourly_cdr_backup = Bool()
    server_ip = Str(validate=[validate.Length(max=16), validate.Regexp(IP_REGEXP)], defailt='127.0.0.1')
    server_port = Int(validate=[validate.Range(1, 65535)], default=33500)
    server_ip_whitelist = Str()
    server_ip_blacklist = Str()
    server_con_limit = Int()
    server_per_ip_con_limit = Int()
    server_con_timeout = Int()
    log_level = Int(validate=[validate.Range(-1, 2)])
    log_dir_path = Str()
    disable_logfile = Bool()
    enable_postgresql_log = Bool()
    tmp_dir_path = Str()
    disable_tmp_dir = Bool()
    startup_delay = Int()
    run_on_start = Bool()
    dnl_cloud_storage = Nested('DnlCloudStorageScheme', many=False)
    dnl_cloud_storage1 = Nested('DnlCloudStorageScheme', many=False)

    class Meta:
        model = model.DnlCloudDbmanCfg
        fields = (
            'server_name', 'server_use_ssl', 'cdr_storage_cfg_uuid', 'pcap_storage_cfg_uuid', 'keep_cdr', 'keep_pcap',
            'keep_cdr_archives', 'keep_pcap_archives', 'keep_cdr_remote', 'keep_pcap_remote', 'hourly_cdr_backup',
            'server_ip', 'server_port', 'server_ip_whitelist', 'server_ip_blacklist', 'server_con_limit',
            'server_per_ip_con_limit', 'server_con_timeout', 'log_level', 'log_dir_path', 'disable_logfile',
            'enable_postgresql_log', 'tmp_dir_path', 'disable_tmp_dir', 'startup_delay', 'run_on_start',)


class DnlCloudDbmanCfgSchemeGet(DnlCloudDbmanCfgScheme):
    class Meta:
        model = model.DnlCloudDbmanCfg
        fields = (
            'server_name', 'server_use_ssl', 'cdr_storage_cfg_uuid', 'pcap_storage_cfg_uuid', 'keep_cdr', 'keep_pcap',
            'keep_cdr_archives', 'keep_pcap_archives', 'keep_cdr_remote', 'keep_pcap_remote', 'hourly_cdr_backup',
            'server_ip', 'server_port', 'server_ip_whitelist', 'server_ip_blacklist', 'server_con_limit',
            'server_per_ip_con_limit', 'server_con_timeout', 'log_level', 'log_dir_path', 'disable_logfile',
            'enable_postgresql_log', 'tmp_dir_path', 'disable_tmp_dir', 'startup_delay', 'run_on_start',)
        search_fields = (
            'server_name', 'server_use_ssl', 'cdr_storage_cfg_uuid', 'pcap_storage_cfg_uuid', 'keep_cdr', 'keep_pcap',
            'keep_cdr_archives', 'keep_pcap_archives', 'keep_cdr_remote', 'keep_pcap_remote', 'hourly_cdr_backup',
            'server_ip', 'server_port', 'server_ip_whitelist', 'server_ip_blacklist', 'server_con_limit',
            'server_per_ip_con_limit', 'server_con_timeout', 'log_level', 'log_dir_path', 'disable_logfile',
            'enable_postgresql_log', 'tmp_dir_path', 'disable_tmp_dir', 'startup_delay', 'run_on_start',
            'dnl_cloud_storage', 'dnl_cloud_storage1',)


class DnlCloudDbmanCfgSchemeModify(DnlCloudDbmanCfgScheme):
    pass


# endregion ---DnlCloudDbmanCfg---
# region +++DnlCloudDownloaderCfg+++
class DnlCloudDownloaderCfgScheme(BaseModelScheme):
    server_name = Str()
    server_use_ssl = Bool()
    server_ip = Str(validate=[validate.Length(max=16)])
    server_port = Int(validate=[validate.Range(1, 65535)], default=33501)
    server_ip_whitelist = Str()
    server_ip_blacklist = Str()
    server_con_limit = Int(validate=[validate.Range(1)])
    server_per_ip_con_limit = Int(validate=[validate.Range(1)])
    server_con_timeout = Int(validate=[validate.Range(1)])
    log_level = Int(validate=[validate.Range(-1, 2)])
    log_dir_path = Str()
    disable_logfile = Bool()
    enable_postgresql_log = Bool()
    tmp_dir_path = Str()
    disable_tmp_dir = Bool()

    class Meta:
        model = model.DnlCloudDownloaderCfg
        fields = (
            'server_name', 'server_use_ssl', 'server_ip', 'server_port', 'server_ip_whitelist', 'server_ip_blacklist',
            'server_con_limit', 'server_per_ip_con_limit', 'server_con_timeout', 'log_level', 'log_dir_path',
            'disable_logfile', 'enable_postgresql_log', 'tmp_dir_path', 'disable_tmp_dir',)

    @pre_load
    def check_downloader(self, data):
        if 'server_ip_blacklist' in data and 'server_ip_whitelist' in data:
            raise ValidationError(
                {'server_ip_whitelist': ["must be set only one of:  'server_ip_blacklist' or 'server_ip_whitelist'"]})
        return data


class DnlCloudDownloaderCfgSchemeGet(DnlCloudDownloaderCfgScheme):
    class Meta:
        model = model.DnlCloudDownloaderCfg
        fields = (
            'server_name', 'server_use_ssl', 'server_ip', 'server_port', 'server_ip_whitelist', 'server_ip_blacklist',
            'server_con_limit', 'server_per_ip_con_limit', 'server_con_timeout', 'log_level', 'log_dir_path',
            'disable_logfile', 'enable_postgresql_log', 'tmp_dir_path', 'disable_tmp_dir',)
        search_fields = (
            'server_name', 'server_use_ssl', 'server_ip', 'server_port', 'server_ip_whitelist', 'server_ip_blacklist',
            'server_con_limit', 'server_per_ip_con_limit', 'server_con_timeout', 'log_level', 'log_dir_path',
            'disable_logfile', 'enable_postgresql_log', 'tmp_dir_path', 'disable_tmp_dir',)


class DnlCloudDownloaderCfgSchemeModify(DnlCloudDownloaderCfgScheme):
    pass


# endregion ---DnlCloudDownloaderCfg---
# region +++DnlCloudFtpCfg+++
class DnlCloudFtpCfgScheme(BaseModelScheme):
    uuid = Str(validate=lambda value: _valid('DnlCloudStorage', 'uuid', value))
    ip = Str(validate=[validate.Length(max=16)])
    port = Int(validate=[validate.Range(1, 65535)], default=21)
    username = Str()
    password = Str()
    workdir = Str()
    use_relative_paths = Bool()
    netrc_path = Str()
    netrc_id = Int(validate=lambda value: _valid('ImportExportLogs', 'id', value))
    debug = Int(validate=[validate.Range(0, 2)])

    class Meta:
        model = model.DnlCloudFtpCfg
        fields = (
            'ip', 'port', 'username', 'password', 'workdir', 'use_relative_paths', 'netrc_path', 'debug', 'netrc_id')


class DnlCloudFtpCfgSchemeGet(DnlCloudFtpCfgScheme):
    class Meta:
        model = model.DnlCloudFtpCfg
        fields = (
            'ip', 'port', 'username', 'password', 'workdir', 'use_relative_paths', 'netrc_path', 'debug', 'netrc_id')
        search_fields = (
            'uuid', 'ip', 'port', 'username', 'password', 'workdir', 'use_relative_paths', 'netrc_path', 'debug',)


class DnlCloudFtpCfgSchemeModify(DnlCloudFtpCfgScheme):
    pass


class DnlCloudFtpCfgSchemeList(DnlCloudFtpCfgScheme):
    class Meta:
        model = model.DnlCloudFtpCfg
        fields = (
            'uuid', 'ip', 'port', 'username', 'password', 'workdir', 'use_relative_paths', 'netrc_path', 'debug', 'netrc_id')
        search_fields = (
            'uuid', 'ip', 'port', 'username', 'password', 'workdir', 'use_relative_paths', 'netrc_path', 'debug',)


# endregion ---DnlCloudFtpCfg---
# region +++DnlCloudGcloudCfg+++
class DnlCloudGcloudCfgScheme(BaseModelScheme):
    uuid = Str(validate=lambda value: _valid('DnlCloudStorage', 'uuid', value))
    type = Str(validate=validate.OneOf(['oauth2', 'service']))
    bucket = Str()
    workdir = Str()
    key_file_path = Str()
    key_file_id = Int(validate=lambda value: _valid('ImportExportLogs', 'id', value))
    json_key = Dict()
    email = Str()
    secret = Str()
    token = Str()
    debug = Int(validate=[validate.Range(0, 2)])
    json_download_url = Str()


    class Meta:
        model = model.DnlCloudGcloudCfg
        fields = (
            'bucket', 'workdir', 'key_file_path', 'email', 'secret', 'token', 'debug', 'type', 'key_file_id',
            'json_key', 'json_download_url')


class DnlCloudGcloudCfgSchemeGet(DnlCloudGcloudCfgScheme):
    class Meta:
        model = model.DnlCloudGcloudCfg
        fields = (
            'bucket', 'workdir', 'key_file_path', 'email', 'secret', 'token', 'debug', 'type', 'key_file_id',
            'json_key')
        search_fields = (
            'uuid', 'bucket', 'workdir', 'key_file_path', 'email', 'secret', 'token', 'debug', 'type', 'key_file_id')


class DnlCloudGcloudCfgSchemeModify(DnlCloudGcloudCfgScheme):
    pass


class DnlCloudGcloudCfgSchemeList(DnlCloudGcloudCfgScheme):
    class Meta:
        model = model.DnlCloudGcloudCfg
        fields = (
            'uuid', 'bucket', 'workdir', 'key_file_path', 'email', 'secret', 'token', 'debug', 'type', 'key_file_id',
            'json_key')
        search_fields = (
            'uuid', 'bucket', 'workdir', 'key_file_path', 'email', 'secret', 'token', 'debug', 'type', 'key_file_id')


# endregion ---DnlCloudGcloudCfg---
# region +++DnlCloudLog+++
class DnlCloudLogScheme(BaseModelScheme):
    id = Int()
    time = DateTime()
    server_name = Str(validate=[validate.Length(max=256)])
    instance = Str(validate=[validate.Length(max=128)])
    level = Str(validate=[validate.Length(max=16)])
    msg = Str()

    class Meta:
        model = model.DnlCloudLog
        fields = ('time', 'instance', 'level', 'msg', 'server_name')


class DnlCloudLogSchemeGet(DnlCloudLogScheme):
    class Meta:
        model = model.DnlCloudLog
        fields = ('time', 'instance', 'level', 'msg', 'server_name')
        search_fields = ('id', 'instance', 'level', 'msg', 'server_name')
        query_fields = ('time_gt', 'time_lt',)


class DnlCloudLogSchemeModify(DnlCloudLogScheme):
    pass


# endregion ---DnlCloudLog---
# region +++DnlCloudSftpCfg+++
class DnlCloudSftpCfgScheme(BaseModelScheme):
    uuid = Str(validate=lambda value: _valid('DnlCloudStorage', 'uuid', value))
    ip = Str(validate=[validate.Length(max=16)])
    port = Int(validate=[validate.Range(1, 65535)], default=22)
    username = Str()
    password = Str()
    workdir = Str()
    pubkey_path = Str()
    privkey_path = Str()
    pubkey_id = Int(validate=lambda value: _valid('ImportExportLogs', 'id', value))
    privkey_id = Int(validate=lambda value: _valid('ImportExportLogs', 'id', value))
    passphrase = Str()
    debug = Int(validate=[validate.Range(0, 2)])

    class Meta:
        model = model.DnlCloudSftpCfg
        fields = ('ip', 'port', 'username', 'password', 'workdir', 'pubkey_path', 'privkey_path', 'passphrase', 'debug',
                  'pubkey_id', 'privkey_id')


class DnlCloudSftpCfgSchemeGet(DnlCloudSftpCfgScheme):
    class Meta:
        model = model.DnlCloudSftpCfg
        fields = (
            'ip', 'port', 'username', 'password', 'workdir', 'pubkey_path', 'privkey_path', 'passphrase', 'debug',
            'pubkey_id', 'privkey_id')
        search_fields = (
            'uuid', 'ip', 'port', 'username', 'password', 'workdir', 'pubkey_path', 'privkey_path', 'passphrase',
            'debug',)


class DnlCloudSftpCfgSchemeModify(DnlCloudSftpCfgScheme):
    pass


# endregion ---DnlCloudSftpCfg---
# region +++DnlCloudStorage+++
class DnlCloudStorageScheme(BaseModelScheme):
    uuid = Str()#validate=lambda value:_valid('DnlCloudStorage','uuid',value))
    type = Str(validate=[validate.Length(max=16), validate.OneOf(['ftp', 'sftp', 'gcloud'])])
    gcloud = Nested('DnlCloudGcloudCfgScheme', many=False)
    sftp = Nested('DnlCloudSftpCfgScheme', many=False)
    ftp = Nested('DnlCloudFtpCfgScheme', many=False, attribute='_ftp')

    class Meta:
        model = model.DnlCloudStorage
        fields = ('type', 'gcloud', 'sftp', 'ftp',)

    @pre_load
    def check_storage(self, data):
        types = ['gcloud', 'sftp', 'ftp']
        if 'type' in data and data['type'] in types:
            if data['type'] not in data:
                raise ValidationError({data['type']: ['must be defined when type={}'.format(data['type'])]})
            rest = list(set(types) - set([data['type']]))
            for r in rest:
                if r in data:
                    raise ValidationError({r: ['cannot set {} when type={}'.format(r, data['type'])]})
        else:
            raise ValidationError({'type': ['mandatory field']})
        return data


class DnlCloudStorageSchemeGet(DnlCloudStorageScheme):
    class Meta:
        model = model.DnlCloudStorage
        fields = ('uuid', 'type', 'gcloud', 'sftp', 'ftp',)
        search_fields = ('uuid', 'type',)


class DnlCloudStorageSchemeModify(DnlCloudStorageScheme):
    class Meta:
        model = model.DnlCloudStorage
        fields = ('type', 'gcloud', 'sftp', 'ftp',)


# endregion ---DnlCloudStorage---

# region +++DnlCloudSearchCfg+++
class DnlCloudSearchCfgScheme(BaseModelScheme):
    server_name = Str(validate=[validate.Length(max=256)])
    server_use_ssl = Bool()
    pcap_time_margin = Int()
    req_exp_time = Int()
    threads_max = Int()
    server_ip = Str(validate=[validate.Length(max=16)])
    server_port = Int()
    server_ip_whitelist = Str()
    server_ip_blacklist = Str()
    server_con_limit = Int()
    server_per_ip_con_limit = Int()
    server_con_timeout = Int()
    log_level = Int()
    log_dir_path = Str()
    disable_logfile = Bool()
    enable_postgresql_log = Bool()
    tmp_dir_path = Str()

    class Meta:
        model = model.DnlCloudSearchCfg
        fields = ('server_use_ssl', 'pcap_time_margin', 'req_exp_time', 'threads_max', 'server_ip', 'server_port',
                  'server_ip_whitelist', 'server_ip_blacklist', 'server_con_limit', 'server_per_ip_con_limit',
                  'server_con_timeout', 'log_level', 'log_dir_path', 'disable_logfile', 'enable_postgresql_log',
                  'tmp_dir_path',)


class DnlCloudSearchCfgSchemeGet(DnlCloudSearchCfgScheme):
    class Meta:
        model = model.DnlCloudSearchCfg
        fields = ('server_use_ssl', 'pcap_time_margin', 'req_exp_time', 'threads_max', 'server_ip', 'server_port',
                  'server_ip_whitelist', 'server_ip_blacklist', 'server_con_limit', 'server_per_ip_con_limit',
                  'server_con_timeout', 'log_level', 'log_dir_path', 'disable_logfile', 'enable_postgresql_log',
                  'tmp_dir_path',)
        search_fields = (
            'server_name', 'server_use_ssl', 'pcap_time_margin', 'req_exp_time', 'threads_max', 'server_ip',
            'server_port',
            'server_ip_whitelist', 'server_ip_blacklist', 'server_con_limit', 'server_per_ip_con_limit',
            'server_con_timeout', 'log_level', 'log_dir_path', 'disable_logfile', 'enable_postgresql_log',
            'tmp_dir_path',)


class DnlCloudSearchCfgSchemeModify(DnlCloudSearchCfgScheme):
    pass


# endregion ---DnlCloudSearchCfg---
# region +++DnlCloudStatus+++
class DnlCloudStatusScheme(BaseModelScheme):
    server_name = Str(validate=[validate.Length(max=256)])
    instance = Str(validate=[validate.Length(max=128)])
    operation = Str()
    is_online = Bool()
    upd_time = DateTime()
    server_ip = Str(validate=[validate.Length(max=16)])
    server_port = Int()
    server_use_ssl = Bool()
    switch_list = Str()

    class Meta:
        model = model.DnlCloudStatus
        fields = ('operation', 'is_online', 'upd_time', 'server_ip', 'server_port', 'server_use_ssl', 'switch_list',
                  'instance')


class DnlCloudStatusSchemeGet(DnlCloudStatusScheme):
    class Meta:
        model = model.DnlCloudStatus
        fields = ('server_name', 'operation', 'is_online', 'upd_time', 'server_ip', 'server_port', 'server_use_ssl',
                  'switch_list', 'instance')
        search_fields = (
            'server_name', 'instance', 'operation', 'is_online', 'server_ip', 'server_port', 'server_use_ssl',
            'switch_list',)
        query_fields = ('upd_time_gt', 'upd_time_lt',)


class DnlCloudStatusSchemeModify(DnlCloudStatusScheme):
    pass


# endregion ---DnlCloudStatus---


class CloudCdrScheme(BaseModelScheme):
    switch_name = Str()  # - `switch_name` - switch to filter (mandatory)
    start_time = Int()  # call start time since Unix Epoch (mandatory)
    end_time = Int()  # call end time since Unix Epoch (mandatory)
    fields = Str()  # CDR fields to printout (comma-separated list of digits [0;107])
    ani = Str()
    dnis = Str()  # comma-separated list of phone numbers
    ingress_trunk_id = Str()  # comma-separated list of trunk IDs
    egress_trunk_id = Str()  # comma-separated list of trunk IDs
    ingress_carrier_id = Str(validate=lambda value: _valid('Client', 'client_id', value))  # comma-separated list of carrier IDs
    egress_carrier_id = Str(validate=lambda value: _valid('Client', 'client_id', value))  # comma-separated list of carrier IDs
    ingress_call_id = Str()  # comma-separated list of call IDs
    egress_call_id = Str()  # comma-separated list of call IDs
    tz = Int()  # time zone
    is_final_call = Int()
    non_zero = Int()


class CloudPcapScheme(BaseModelScheme):
    switch_name = Str()  # switch to filter (mandatory)
    start_time = Int()  # call start time since Unix Epoch (mandatory)
    end_time = Int()  # call end time since Unix Epoch (mandatory)
    call_id = Str()  # comma-separated list of Call IDs (mandatory)
    ani = Str()
    dnis = Str()  # comma-separated list of phone numbers


class CloudSwitchScheme(BaseModelScheme):
    switch_name = Str()
    cdr_dir = Str()
    pcap_dir = Str()


class CloudSwitchDeleteScheme(BaseModelScheme):
    switch_name = Str()
