from api_dnl import settings
from api_dnl.__version__ import __version__, __minor__, __date__
from falcon_rest.logger import log
import celery
from celery_singleton import Singleton
from functools import wraps
from datetime import datetime, timedelta
import time
from pytz import UTC
import csv
from io import StringIO
import traceback
from api_dnl.utils.imp_exp import (csv2xls, dict_to_csv)
from api_dnl.utils.dnl_active_calls import get_dnl_active_calls_session
from api_dnl.utils.dnl_switch import get_dnl_switch_session
from celery import Celery
from celery.signals import task_prerun, task_postrun
from celery.schedules import crontab
from falcon_rest.db import initialize_db, get_db
from shutil import copyfile
from api_dnl.base_model import DnlApiBaseModel
from api_dnl.model import Client, Resource, BalanceHistoryActual, C4ClientBalance, \
    RateSendLogDetail, MailSender, ClientPayment, BalanceLog, SchedulerLog, \
    SwitchProfile, VoipGateway, VoipGatewayInfo, \
    CdrAsyncTask, PcapQueryTaskMail, DidExportAsyncTask
from api_dnl.model import InvoiceTask, InvoiceSummary, InvoiceHistory, InvoiceDebug
# from api_dnl.task.invoice2 import InvoicePdf
from api_dnl.model import CodeDeck, Role, User, MailTemplate, SendRateTemplate, \
    VersionInformation, RateTable, Route, SystemParameter, SystemFunction, CodeCountry, CdrExportTask
from api_dnl.model import func, and_, select
from sqlalchemy.orm import make_transient, foreign
from sqlalchemy import exists
from api_dnl.settings import DB_CONN_STRING, SENTRY_URL

settings.LOG_FILE = settings.LOG_FILE + '.' + __name__

# +++ sentry integration
if SENTRY_URL:
    import raven
    import logging

    sentry = raven.Client(SENTRY_URL)
    from raven.handlers.logging import SentryHandler

    # Manually specify a client
    handler = SentryHandler(sentry)
    handler.setLevel(logging.ERROR)
    from raven.conf import setup_logging

    setup_logging(handler)
# --- sentry integration
log.debug(DB_CONN_STRING)
db = None
from api_dnl import model

if '_databases' in globals() and 'default' in globals()['_databases']:
    db = get_db()
if not db:
    # db = initialize_db(DB_CONN_STRING, DnlApiBaseModel)
    db = model.init_db(DB_CONN_STRING, 'default', DnlApiBaseModel)


def custom_logger(name):
    import os, logging
    logger = logging.getLogger(name)
    logger.setLevel(logging.DEBUG)
    handler = logging.FileHandler(os.path.join('./', name + '.log'), 'a+')
    formatter = logging.Formatter("%(asctime)s:%(levelname)s:%(name)s:%(message)s")
    handler.setFormatter(formatter)
    logger.addHandler(handler)
    return logger


class SqlAlchemyTask(celery.Task):
    """An abstract Celery Task that ensures that the connection the the
    database is closed on task completion"""
    abstract = True

    def xx_after_return(self, status, retval, task_id, args, kwargs, einfo):
        try:
            db.session.commit()
        except:
            try:
                db.session.rollback()
            except:
                pass
        finally:
            db.session.remove()


from kombu import Exchange, Queue

app = Celery('task', broker=settings.CELERY['CELERY_BROKER_URL'])

settings.CELERY['CELERY_QUEUES'] = (
    Queue('high', Exchange('high'), routing_key='high'),
    Queue('normal', Exchange('normal'), routing_key='normal'),
    Queue('low', Exchange('low'), routing_key='low'),
)
settings.CELERY['CELERY_DEFAULT_QUEUE'] = 'normal'
settings.CELERY['CELERY_DEFAULT_EXCHANGE'] = 'normal'
settings.CELERY['CELERY_DEFAULT_ROUTING_KEY'] = 'normal'
settings.CELERY['CELERY_ROUTES'] = {
    # -- HIGH PRIORITY QUEUE -- #
    'api_dnl.task.alerts.do_clear_last_lowbalance_send_time': {'queue': 'high'},
    'api_dnl.task.alerts.do_notify_client_balance': {'queue': 'high'},
    'api_dnl.task.alerts.do_notify_zero_balance': {'queue': 'high'},
    'api_dnl.task.alerts.do_daily_usage_summary': {'queue': 'high'},
    'api_dnl.task.alerts.do_daily_balance_summary': {'queue': 'high'},
    'api_dnl.task.alerts.do_trunk_pending_suspension_notice': {'queue': 'high'},
    'api_dnl.task.alerts.do_monitored_rule_history': {'queue': 'high'},
    'api_dnl.task.alerts.do_frund_detection': {'queue': 'high'},
    'api_dnl.task.invoice.do_invoice_task': {'queue': 'high'},
    'api_dnl.tasks.do_save_history': {'queue': 'high'},
    'api_dnl.tasks.do_rate_import_preprocess': {'queue': 'high'},
    'api_dnl.tasks.do_rate_analysis': {'queue': 'high'},
    'api_dnl.tasks.do_lcr_task': {'queue': 'high'},
    'api_dnl.tasks.do_block_number_import_preprocess': {'queue': 'high'},
    'api_dnl.tasks.do_did_number_upload_task_preprocess': {'queue': 'high'},
    'api_dnl.tasks.daily_balance_carry_task': {'queue': 'high'},
    # -- LOW PRIORITY QUEUE -- #
    'api_dnl.task.cid_block.do_cid_block': {'queue': 'low'},
    'api_dnl.task.export_file.export_file': {'queue': 'low'},
    'api_dnl.task.apply_rates.apply_rates_task': {'queue': 'low'},
    'api_dnl.task.cdr_download.cdr_download': {'queue': 'low'},
    'api_dnl.task.send_rate.do_send_rate': {'queue': 'low'},
    'api_dnl.task.import_file.do_import_file': {'queue': 'low'},
    'api_dnl.tasks.do_cdr_async_task': {'queue': 'low'},
    'api_dnl.task.did_import_file.shaken_ani_group_rel_import': {'queue': 'low'},
    'api_dnl.task.import_file.import_file': {'queue': 'low'},
    'api_dnl.tasks.do_did_update_mrc': {'queue': 'low'},
    'api_dnl.task.lerg.do_lerg_download_schedules': {'queue': 'low'},
    'api_dnl.task.lerg.do_lerg_download_task': {'queue': 'low'},
    'api_dnl.task.lerg.do_lerg_import_file': {'queue': 'low'},
    'api_dnl.task.lerg.do_c4uslerg_import_file': {'queue': 'low'},

}
app.conf.update(settings.CELERY)
app.conf.update(CELERY_ENABLE_UTC=True, CELERYD_HIJACK_ROOT_LOGGER=False)
app.conf.BROKER_TRANSPORT_OPTIONS = {'visibility_timeout': 6*3600}


@task_prerun.connect
def task_prerun(*args, **kwargs):
    db.engine.dispose()


@task_postrun.connect
def close_session(*args, **kwargs):
    # Flask SQLAlchemy will automatically create new sessions for you from
    # a scoped session factory, given that we are maintaining the same app
    # context, this ensures tasks have a fresh session (e.g. session errors
    # won't propagate across tasks)
    db.session.remove()


@app.on_after_configure.connect
def setup_periodic_tasks(sender, **kwargs):
    try:
        from api_dnl.task.did_disconnect import do_did_disconnect
        sender.add_periodic_task(64.0, do_did_disconnect)
    except Exception as e:
        log.error('cannot import do_did_disconnect! {}'.format(str(e)))
    try:
        from api_dnl.task.did_import_file import did_import_file_delete
    except Exception as e:
        log.error('cannot import did_import_file_delete! {}'.format(str(e)))
    try:
        from api_dnl.task.apply_rates import apply_rates_task
    except Exception as e:
        log.error('cannot import apply_rates_task! {}'.format(str(e)))
    try:
        from api_dnl.task.cdr_download import cdr_download
    except Exception as e:
        log.error('cannot import cdr_download! {}'.format(str(e)))
    try:
        from api_dnl.task.export_file import export_file
    except Exception as e:
        log.error('cannot import export_file! {}'.format(str(e)))
    try:
        from api_dnl.task.invoice import do_auto_invoice_task
        sender.add_periodic_task(
            crontab(minute=0), 
            do_auto_invoice_task,
            options={'queue': 'high'}
        )
    except Exception as e:
        log.error('cannot start do_auto_invoice!{}'.format(str(e)))

    try:
        from api_dnl.task.ftp_upload import do_check_ftp_status, do_ftp_upload
        sender.add_periodic_task(crontab(minute=30), do_check_ftp_status)
        sender.add_periodic_task(crontab(minute=0), do_ftp_upload)
        # sender.add_periodic_task(crontab(minute='*/1'), do_upload_rate)
    except Exception as e:
        log.error('cannot start ftp_upload!{}'.format(str(e)))
    # try:
    #     from api_dnl.task.upload_rate import do_upload_rate
    #     sender.add_periodic_task(65.0, do_upload_rate)
    #     # sender.add_periodic_task(crontab(minute='*/1'), do_upload_rate)
    # except Exception as e:
    #     log.error('cannot start upload rate!{}'.format(str(e)))
    try:
        from api_dnl.task.auto_rate import do_auto_rate
        sender.add_periodic_task(crontab(minute='*/10'), do_auto_rate)
    except Exception as e:
        log.error('cannot start upload rate!{}'.format(str(e)))
    try:
        from api_dnl.task.import_file import do_import_file
        # pcap parser daemon
        # sender.add_periodic_task(crontab(minute='*/3'), do_import_file)
        sender.add_periodic_task(56.0, do_import_file)
    except Exception as e:
        log.error('cannot start import file!{}'.format(str(e)))

    # try:
    #     from api_dnl.task.pcapparser import do_pcap_parser
    #     sender.add_periodic_task(61.0, do_pcap_parser)
    # except Exception as e:
    #     log.error('cannot start pcap parser!{}'.format(str(e)))

    # alert detection daemon
    try:
        from api_dnl.task.alert import do_alert_rule
        sender.add_periodic_task(crontab(minute='*/3'), do_alert_rule)
    except Exception as e:
        log.error('cannot start alert rules!{}'.format(str(e)))
    try:
        from api_dnl.task.send_rate import do_send_rate

        sender.add_periodic_task(60.0, do_send_rate)
    except Exception as e:
        log.error('cannot start send rate!{}'.format(str(e)))

    # try:
    #     from api_dnl.task.code_deck_update import do_code_deck_update_check
    #
    #     sender.add_periodic_task(crontab(minute='*/15'), do_code_deck_update_check)
    # except Exception as e:
    #     log.error('cannot start code_deck_update_check!{}'.format(str(e)))

    # product_route_analysis
    try:
        from api_dnl.task.product_route_analysis import product_route_analysis
        # sender.add_periodic_task(crontab(minute='*/3'), ???)
    except Exception as e:
        log.error('cannot start product_route_analysis!{}'.format(str(e)))

    try:
        from api_dnl.task.cdr_download import do_cdr_download
        ###sender.add_periodic_task(40.0, do_cdr_download)
    except Exception as e:
        log.error('cannot do_cdr_download!{}'.format(str(e)))

    try:
        from api_dnl.task.lerg import do_lerg_download_schedules
        sender.add_periodic_task(crontab(minute=0), do_lerg_download_schedules)

    except Exception as e:
        log.error('cannot do_lerg_download_schedules!{}'.format(str(e)))

    try:
        from api_dnl.task.fault_route_alert_rule import do_fault_route_alert_rules
        sender.add_periodic_task(crontab(minute=0), do_fault_route_alert_rules)

    except Exception as e:
        log.error('cannot do_fault_route_alert_rules!{}'.format(str(e)))

    try:
        from api_dnl.task.cid_block import do_cid_block
        sender.add_periodic_task(crontab(minute='*/5'), do_cid_block)

    except Exception as e:
        log.error('cannot do_cid_block!{}'.format(str(e)))

    try:
        from api_dnl.task.resource_block import do_clear_resource_block
        sender.add_periodic_task(crontab(minute='*/30'), do_clear_resource_block)

    except Exception as e:
        log.error('cannot do_clear_resource_block!{}'.format(str(e)))

    sender.add_periodic_task(crontab(minute=0, hour='12,18,19,0'), daily_balance_carry_task)

    # sender.add_periodic_task(crontab(minute='*/5'), daily_balance_task)
    sender.add_periodic_task(crontab(minute='*/5'), do_check_active_switch)

    # alert daemon
    sender.add_periodic_task(crontab(minute=0, hour=0, day_of_month='1'), do_did_deducting_task)
    sender.add_periodic_task(crontab(minute=0), do_did_pending_task)
    sender.add_periodic_task(crontab(minute=0, hour=0), do_did_update_mrc)
    sender.add_periodic_task(crontab(minute=0), do_resource_records_missing_task)
    sender.add_periodic_task(crontab(minute='*/30'), do_did_param_missing_task)
    sender.add_periodic_task(crontab(minute='*/30'), do_did_param_sync_task)

    try:
        from api_dnl.task.alerts import do_trunk_pending_suspension_notice, do_daily_cdr_delivery, \
            do_daily_usage_summary, \
            do_daily_balance_summary, do_notify_zero_balance, do_notify_client_balance, \
            do_clear_last_lowbalance_send_time, \
            do_frund_detection, do_monitored_rule_history
        sender.add_periodic_task(crontab(minute='*/10'), do_clear_last_lowbalance_send_time)
        sender.add_periodic_task(crontab(minute='*/5'), do_notify_client_balance)
        sender.add_periodic_task(crontab(minute='*/5'), do_notify_zero_balance)
        sender.add_periodic_task(crontab(minute=0), do_daily_usage_summary)
        sender.add_periodic_task(crontab(minute=35), do_daily_balance_summary)
        sender.add_periodic_task(crontab(minute=0), do_daily_cdr_delivery)
        sender.add_periodic_task(crontab(minute=0), do_trunk_pending_suspension_notice)
        sender.add_periodic_task(crontab(minute='*/5'), do_frund_detection)
        sender.add_periodic_task(crontab(minute='*/5'), do_monitored_rule_history)
        sender.add_periodic_task(crontab(hour=0,minute=10), do_client_cdr_dnis_index)
    except Exception as e:
        log.error('cannot start do_auto_invoice_task!{}'.format(str(e)))


class SchLog():
    def __init__(self, name):
        self.name = name
        try:
            self.sl = SchedulerLog(script_name=name, start_time=datetime.now(UTC))
            self.sl.save()
        except:
            log.debug('Cannot save scheduler log {}')

    # def __del__(self):
    #    self.close()

    def close(self):
        try:
            self.sl.end_time = datetime.now(UTC)
            self.sl.save()
        except:
            log.debug('Cannot save scheduler log {}')


def pg_locked(key):
    def decorator(f):
        @wraps(f)
        def wrapped(*args, **kw):
            session = db.session
            acquired = False
            try:
                acquired, = session.execute(select([func.pg_try_advisory_lock(key)])).fetchone()
                if acquired:
                    return f(*args, **kw)
            finally:
                if acquired:
                    session.execute(select([func.pg_advisory_unlock(key)]))

        return wrapped

    return decorator


@app.task(base=SqlAlchemyTask)
def do_client_cdr_dnis_index(d=None):

    if d is None:
        now = datetime.now()
        d = str(now.date()).replace('-','')
    log.info('start do_client_cdr_dnis_index for date {}'.format(d))
    try:
        # db.session.execute(""" create index IF NOT EXISTS client_cdr_origination_destination_number_prefix_idx_{} on client_cdr{} using gist(prefix_range(origination_destination_number)); """.format(d,d))
        # db.session.execute(""" create index IF NOT EXISTS client_cdr_routing_digits_prefix_idx_{} on client_cdr{} using gist(prefix_range(routing_digits)); """.format(d, d))
        # db.session.execute(""" create index IF NOT EXISTS client_cdr_origination_source_number_prefix_idx_{} on client_cdr{} using gist(prefix_range(origination_source_number)); """.format(d, d))
        # db.session.execute(""" create index IF NOT EXISTS client_cdr_translation_ani_prefix_idx_{} on client_cdr{} using gist(prefix_range(translation_ani)); """.format(d, d))

        db.session.execute(
            """ drop index IF EXISTS client_cdr_origination_destination_number_prefix_idx_{} ;""".format(
                d, d))
        db.session.execute(
            """ drop index IF EXISTS client_cdr_routing_digits_prefix_idx_{} ; """.format(
                d, d))
        db.session.execute(
            """ drop index IF EXISTS client_cdr_origination_source_number_prefix_idx_{} ; """.format(
                d, d))
        db.session.execute(
            """ drop index IF EXISTS client_cdr_translation_ani_prefix_idx_{} ; """.format(
                d, d))
        db.session.execute(
            """ create index IF NOT EXISTS client_cdr_translation_ani_idx_{} on client_cdr{}(translation_ani); """.format(
                d, d))
        db.session.execute(
            """ create index IF NOT EXISTS client_cdr_termination_source_number_idx_{} on client_cdr{}(termination_source_number); """.format(
                d, d))
        db.session.execute(
            """ create index IF NOT EXISTS client_cdr_termination_destination_number_idx_{} on client_cdr{}(termination_destination_number); """.format(
                d, d))
        db.session.execute(
            """ create index IF NOT EXISTS client_cdr_translation_ani_idx on client_cdr(translation_ani); """)
    except Exception as e:
        db.session.rollback()
        log.error('do_client_cdr_dnis_index error {}'.format(e))

    log.info('finish do_client_cdr_dnis_index')

@app.task(base=SqlAlchemyTask)
def balance_log_task(client_id):
    b = C4ClientBalance.filter(C4ClientBalance.client_id == str(client_id)).first()
    if b:
        balance = b.balance
        d = datetime.now(UTC).date()
        l = BalanceLog.filter(and_(BalanceLog.client_id == client_id, BalanceLog.date == d)).first()
        if not l:
            l = BalanceLog(client_id=client_id, date=d)
        l.balance = balance
        l.save()


@app.task(base=SqlAlchemyTask)
def do_flush_session():
    log.warning('Recreate db session')
    del db.session
    db.create_session()


@app.task(base=SqlAlchemyTask, time_limit=300, soft_time_limit=300)
def do_check_active_switch():
    from api_dnl.utils.dnl_switch import get_dnl_switch_session
    log.warning('Check active switch')
    for s in SwitchProfile.query().all():
        try:
            if s.voip_gateway:
                ip = s.voip_gateway.lan_ip
                port = s.voip_gateway.lan_port
                old_status = s.profile_status
                try:
                    sw = get_dnl_switch_session(ip, port)
                    s.profile_status = 'connected'
                except Exception as e:
                    s.profile_status = 'disabled'
                if old_status != s.profile_status:
                    if old_status == 'disabled':
                        s.last_started_on = datetime.now(UTC)
                    s.save()
            for s in VoipGateway.query().all():
                if s.info is None:
                    log.warning('Check active switch:VoipGatewayInfo {} created'.format(s.id))
                    s.info = VoipGatewayInfo(id=s.id)
                    try:
                        s.info.check_info()
                        log.warning('Check active switch:VoipGatewayInfo first created')
                    except:
                        pass
                    s.info.save()
                if s.info.connected and s.info.time + timedelta(minutes=30) < datetime.now(UTC):
                    log.warning('Check active switch:VoipGatewayInfo refresh')
                    s.info.check_info()
                    s.info.save()
                else:
                    s.info.check_connected()
                    s.info.save()
                log.warning('Check active switch:VoipGatewayInfo {}'.format(s.info))
        except Exception as e:
            log.error('Check active swsitch {} failed {}'.format(s,e))

    # import os
    # from  . import dbconfig
    # db=dbconfig.DbConfig()
    # db.from_conn_string(settings.DB_CONN_STRING_EXT)
    # cmd = "PGPASSWORD={} psql -h {} -p {} -d {} -U postgres -c 'grant ALL ON ALL TABLES IN SCHEMA public to webbackend; grant ALL ON ALL SEQUENCES IN SCHEMA public to webbackend;'".format(db.password,db.host,db.port,db.dbname)
    # os.system(cmd)


@app.task(base=SqlAlchemyTask)
def copy_rates_task(src_id, dest_id):
    log.info('Copy rates task called src={} dst={}'.format(src_id, dest_id))
    old_obj = RateTable.get(src_id)
    obj = RateTable.get(dest_id)
    t = datetime.now(UTC)
    if (obj and old_obj):
        for rate in old_obj.rates:
            make_transient(rate)
            rate.rate_id = None
            rate.create_time = t
            obj.rates.append(rate)
        obj.update_at = t
        obj.save()
    else:
        log.info('Copy rates task rate tables not found src={} dst={}'.format(src_id, dest_id))


@app.task(base=SqlAlchemyTask)
def payment_apply_mail_task(client_payment_id, template_name, att=[]):
    payment = ClientPayment.get(client_payment_id)
    payment.apply_mail(template_name, att=att)
    payment = ClientPayment.get(client_payment_id)
    payment.email_sended = True
    payment.save()


@app.task(base=SqlAlchemyTask)
def client_apply_mail_task(client_id, template_name, att=[]):
    client = Client.get(client_id)
    client.apply_mail(template_name, att=att)


@app.task(base=SqlAlchemyTask)
def apply_mail_task(resource_id, template_name, att=[]):
    trunk = Resource.get(resource_id)
    if trunk:
        trunk.apply_mail(template_name, att=att)
        trunk = Resource.get(resource_id)
        f = open('task.txt', 'at')
        f.write(str(datetime.now()) + ' ' + ' ' + str(resource_id) + ' ' + str(trunk.alias) + '\n')
        f.close()


@app.task(base=SqlAlchemyTask)
def carrier_regenerate_balance(client_id):
    cl = Client.get(client_id)
    cl.regenerate_balance()


@app.task(base=SqlAlchemyTask)
def daily_balance_carry_task():
    # global daily_balance_task_lastday
    d1 = datetime.now(UTC).date()
    d0 = datetime.now(UTC).date() - timedelta(days=1)
    query = Client.filter(Client.status == True).all()
    for cl in query:
        try:
            c4 = cl.c4
            b0 = cl.balance_actual.filter(BalanceHistoryActual.date == d1).first()
            b1 = cl.balance_actual.filter(BalanceHistoryActual.date == d1).first()
            log.debug('daily_balance_carry_task', c4, b0, b1)
            if not c4:
                c4 = C4ClientBalance(client_id=cl.client_id)
            if not b1:
                b1 = BalanceHistoryActual(date=d1)

            # BalanceHistoryActual.filter()
        except Exception as e:
            log.warning('daily_balance_carry_task:{}'.format(e))
            db.session.rollback()


@app.task(base=SqlAlchemyTask)
def do_resource_records_missing_task():
    db.session.execute(
        "update resource set name=name where resource_id not in (select resource_id from resource_record group by resource_id);")


@app.task(base=SqlAlchemyTask)
def daily_balance_task():
    # global daily_balance_task_lastday
    now = datetime.now()
    # if now.hour == 0 and now.day != daily_balance_task_lastday:
    #    daily_balance_task_lastday = now.day
    log.debug('---Daily_balance_task for {}'.format(now))
    for cl in Client.filter(and_(Client.status == True, Client.client_type != 'vendor')).all():
        # carrier_regenerate_balance.delay(client_id=cl.client_id)
        if cl.c4:
            c4 = cl.c4
            if (cl.mode == 'prepay' and c4.balance > 0) or (
                    cl.mode == 'postpay' and (cl.unlimited_credit or c4.balance > cl.allowed_credit)):
                if not cl.enough_balance:
                    cl.enough_balance = True
                    cl.save()
                continue
            if (cl.mode == 'prepay' and c4.balance <= 0) or (
                    cl.mode == 'postpay' and not (cl.unlimited_credit or c4.balance > cl.allowed_credit)):
                if cl.enough_balance:
                    cl.enough_balance = False
                    cl.save()
                continue
    log.debug('---Daily_balance_task for {}'.format(now))


@app.task(base=SqlAlchemyTask, time_limit=288000, soft_time_limit=287400)
def do_cdr_async_task(request_id):
    log.debug('cdr_async_task entered {}'.format(request_id))
    from api_dnl.views.cdr1 import CdrReportList
    db1 = initialize_db(conn_string=settings.DB_CONN_STRING, declarative_base=DnlApiBaseModel, db_label='support')
    CdrAsyncTask.db_label = 'support'
    obj = CdrAsyncTask.get(request_id)
    import os
    try:
        if not obj:
            raise Exception('CdrAsyncTask {} not found'.format(request_id))
        # get request_client_id

        trunks = []
        if 'ingress_id' in obj.filter and obj.filter['ingress_id']:
            trunks = trunks + obj.filter['ingress_id'].split(',')
        if 'egress_id' in obj.filter and obj.filter['egress_id']:
            trunks = trunks + obj.filter['egress_id'].split(',')
        try:
            client_id_set = set([model.Resource.get(int(r)).client_id for r in trunks])
            if 'ingress_client_id' in obj.filter and obj.filter['ingress_client_id']:
                client_id_set([int(i) for i in obj.filter['ingress_id'].split(',')])
            if len(client_id_set) == 1 or True:
                obj.request_client_id = list(client_id_set)[0]
        except Exception as e:
            pass

        obj.job_start_time = datetime.now(UTC).timestamp()
        obj.status = 'running'
        obj.progress = '0'
        obj.save()
        size = 0
        count = 0
        progress = 0
        user = model.User.get(obj.user_id) if obj.user_id else None
        resp_stream = CdrReportList().run(file=None, user=user, **obj.filter)
        #if not obj.orig_file:
        old_orig_file = obj.orig_file
        obj.orig_file = 'CDR_export_{}_{}.{}'.format(time.strftime("%Y%m%d%H%M%S", time.localtime(obj.job_start_time)),
                                                     time.strftime("%Y%m%d%H%M%S", time.localtime(obj.job_end_time)),
                                                     obj.filter['format'])
        obj.download_link = None
        obj.save()
        ouput = open(obj.file_name, 'wt')
        for data in resp_stream:
            s = data.decode('utf-8')
            if not isinstance(s, str):
                continue
            ouput.write(s)
            size += len(s)
            count += s.count('\n')
            progress += 1
            obj.progress = '{}'.format(int(progress))
            log.debug('cdr_async_task {} rows stored:{}'.format(request_id, count))
            if progress == 100:
                progress = 0
            obj.count = count
            obj.size = size
            obj.save()
            obj = CdrAsyncTask.get(request_id)
            if obj.status == 'finished':
                break
        obj.count = obj.count - (3 if obj.count > 2 else 2)  # !! 2 trailing /n and headers
        obj.progress = '100'
        if obj.status == 'finished':
            obj.msg = 'job was stopped by user request'
            obj.job_end_time = datetime.now(UTC).timestamp()
        else:
            obj.status = 'success'
        obj.save()
    except Exception as e:
        obj.msg = str(e)[:512]
        obj.progress = '99'
        obj.status = 'fail'
        obj.save()
        log.debug('cdr_async_task {} error:{}'.format(request_id, e))
        return
    #zip file
    if obj.do_zip:
        import zipfile
        name = obj.file_name.replace(obj.filter['format'],'zip')

        with zipfile.ZipFile(name, "w", zipfile.ZIP_DEFLATED, allowZip64=True) as f:
            #f.write(obj.file_name, 'cdr.csv')
            f.write(obj.file_name, obj.orig_file)
            f.close()
        obj.orig_file=obj.orig_file.replace(obj.filter['format'],'zip')
    obj.progress = '100'
    # obj.type = 'download'
    if old_orig_file:
        copyfile(obj.file_name, '{}/{}'.format(settings.FILES['upload_to'], old_orig_file))
    obj.job_end_time = datetime.now(UTC).timestamp()

    # if os.path.exists(obj.file_name):
    #     os.rename(obj.file_name,
    #               '{}/CDR_export_{}_{}.{}'.format(settings.FILES['upload_to'], time.strftime("%Y%m%d%H%M%S",
    #                                                                                          time.localtime(
    #                                                                                              obj.job_start_time)),
    #                                               time.strftime("%Y%m%d%H%M%S",
    #                                                             time.localtime(obj.job_end_time)),
    #                                               fmt))
    #
    #     obj.orig_file = 'CDR_export_{}_{}.{}'.format(time.strftime("%Y%m%d%H%M%S", time.localtime(obj.job_start_time)),
    #                                                  time.strftime("%Y%m%d%H%M%S", time.localtime(obj.job_end_time)),
    #                                                  fmt)

    obj.download_link = obj._download_link
    obj.save()
    CdrAsyncTask.db_label = 'default'
    log.debug('cdr_async_task finished {}'.format(request_id))


@app.task(base=SqlAlchemyTask, time_limit=288000, soft_time_limit=287400)
def do_cdr_email_async_task(request_id):  # , time_limit=288000, soft_time_limit=287400):
    log.debug('do_cdr_email_async_task entered {}'.format(request_id))
    db1 = initialize_db(conn_string=settings.DB_CONN_STRING, declarative_base=DnlApiBaseModel, db_label='support_2')
    CdrAsyncTask.db_label = 'support_2'
    obj = CdrAsyncTask.get(request_id)
    count = 0

    try:
        if not obj:
            log.debug('do_cdr_email_async_task running {}'.format(request_id))
            raise Exception('CdrAsyncTask {} not found'.format(request_id))
        while obj.status == 'running':
            time.sleep(300)
        obj.status = 'running'
        att = []
        import os
        if os.path.exists(obj.file_name):
            try:
                if not obj.do_zip:
                    with open(obj.file_name, 'rt') as inf:
                        csv = inf.read()
                        # xls = csv2xls(csv, 'CDR')
                        # with open(obj.file_name_xls, 'wb') as f:
                        #     f.write(xls)
                        att = [('cdr_{}.csv'.format(obj.request_id), csv)]
                else:
                    import zipfile
                    with zipfile.ZipFile(obj.file_name, 'r') as zip:
                        csv = zip.read(obj.orig_file.replace('zip', obj.filter['format'])).decode('utf-8')
                        # xls = csv2xls(csv, 'CDR')
                        # with open(obj.file_name_xls, 'wb') as f:
                        #     f.write(xls)
                        att = [('cdr_{}.csv'.format(obj.request_id), csv)]
            except Exception as e:
                obj.msg = 'cannot do excel:' + str(e)[:500]
        if obj.client_id:
            if not obj.email:
                obj.email = obj.client.email
            else:
                obj.email = (obj.email or '') + ';' + (obj.client.email or '')
        if obj and obj.email and (obj.count or 1) > 0:  # and not sr.email_status=='sent success':
            if obj.direct:
                dir = model.SendCdrDirect.get(obj.request_id)
                setattr(obj, dir.to_mail, ';'.join([obj.email, dir.cc_mail]))
                res = MailSender.apply_mail(obj, 'sendcdrdirect_{}'.format(obj.request_id), att=[])
            else:
                if obj.type == 'auto':
                    tpl_name = 'auto_cdr'
                else:
                    tpl_name = 'send_cdr'
                dir = MailTemplate.get(tpl_name)
                setattr(obj, dir.to_mail, ';'.join([obj.email, dir.cc_mail]))
                res = MailSender.apply_mail(obj, tpl_name, att=att)
            if res:
                if hasattr(MailSender, 'last_error'):
                    obj.mail_status = MailSender.last_error
                else:
                    obj.mail_status = 'unknown mail send error'
        else:
            obj.mail_status = 'no mail address given'
        obj.status = 'success'
        obj.save()

    except Exception as e:
        obj.msg = str(e)[:512]
        obj.status = 'fail'
        log.debug('do_cdr_email_async_task {} error:{} {}'.format(request_id, e, traceback.format_exc()))
    obj.job_end_time = datetime.now(UTC).timestamp()
    obj.download_link = obj._download_link
    obj.save()
    CdrAsyncTask.db_label = 'default'
    log.debug('cdr_async_task finished {}'.format(request_id))


@app.task(base=SqlAlchemyTask, time_limit=288000, soft_time_limit=287400)
def do_cdr_export_email_task(request_id):  # , time_limit=288000, soft_time_limit=287400):
    log.debug('do_cdr_export_email_task entered {}'.format(request_id))
    db1 = initialize_db(conn_string=settings.DB_CONN_STRING, declarative_base=DnlApiBaseModel, db_label='support_2')
    CdrExportTask.db_label = 'support_2'
    obj = CdrExportTask.get(request_id)
    try:
        if not obj:
            log.debug('do_cdr_export_email_task running {}'.format(request_id))
            raise Exception('CdrExportTask {} not found'.format(request_id))
        if not obj.email_task:
            raise Exception('CdrExportTask {} not to be emailed'.format(request_id))
        while obj.status in ('Initial', 'In Process'):
            time.sleep(300)
        if obj.status == 'Failed':
            raise Exception('CdrExportTask {} failed, not to be emailed'.format(request_id))
        email_task = obj.email_task
        email_task.status = 'In Process'
        email_task.started_on = datetime.now(UTC)
        email_task.save()
        att = []
        import os
        if os.path.exists(obj.file_name):
            try:
                with open(obj.file_name, 'rt') as inf:
                    csv = inf.read()
                    xls = csv2xls(csv, 'CDR')
                    with open(obj.file_name_xls, 'wb') as f:
                        f.write(xls)
                    att = [('cdr_{}.xls'.format(obj.request_id), xls)]
            except Exception as e:
                obj.msg = 'cannot do excel:' + str(e)[:500]
        # if obj.client_id:
        #     if not obj.email:
        #         obj.email = obj.client.email
        #     else:
        #         obj.email = obj.email + ';' + obj.client.email
        if obj and email_task:  # and not sr.email_status=='sent success':
            if email_task.template:
                res = MailSender.apply_mail(obj, 'sendcdrexport_{}'.format(obj.request_id), att=att)
            if res:
                if hasattr(MailSender, 'last_error'):
                    email_task.error = MailSender.last_error
                else:
                    email_task.error = 'unknown mail send error'
        else:
            email_task.error = 'no mail address given'
        if email_task.error:
            email_task.email_status = 'Failed'
        else:
            email_task.email_status = 'Successful'
        email_task.save()

    except Exception as e:
        email_task.status = 'Failed'
        email_task.error = str(e)
        email_task.finished_on = datetime.now(UTC)
        email_task.save()
        log.debug('do_cdr_export_email_task {} error:{}'.format(request_id, e))
    email_task.finished_on = datetime.now(UTC)
    email_task.save()
    CdrExportTask.db_label = 'default'
    log.debug('cdr_export_email_task finished {}'.format(request_id))


@app.task(base=SqlAlchemyTask, time_limit=288000, soft_time_limit=287400)
def do_pcap_email_async_task(request_id):  # , time_limit=288000, soft_time_limit=287400):
    log.debug('do_pcap_email_async_task entered {}'.format(request_id))
    obj = PcapQueryTaskMail.get(request_id)
    try:
        if not obj:
            log.debug('do_pcap_email_async_task running {}'.format(request_id))
            raise Exception('PcapAsyncTaskMail {} not found'.format(request_id))
        obj.status = 'running'
        obj.job_start_time = datetime.now(UTC)
        obj.save()
        att = []
        import os
        if os.path.exists(obj.pcap.result_file):
            try:
                with open(obj.file_name, 'rt') as inf:
                    data = inf.read()
                    att = [('pcap_{}.pcap'.format(obj.request_id), data)]
            except Exception as e:
                obj.msg = 'cannot open pcap file:' + str(e)[:500]
        if obj.client_id:
            if not obj.email:
                obj.email = obj.client.email
            else:
                obj.email = (obj.email or '') + ';' + (obj.client.email or '')
        if obj and obj.email:  # and not sr.email_status=='sent success':
            objToSend = type('new_dict', (object,), {**obj.__dict__, **obj.pcap.__dict__})
            setattr(objToSend, 'download_link', obj.pcap.download_link)

            if obj.direct:
                dir = model.SendCdrDirect.get(obj.request_id)
                setattr(obj, dir.to_mail, ';'.join([obj.email, dir.cc_mail]))
                res = MailSender.apply_mail(objToSend, 'sendpcapdirect_{}'.format(obj.request_id), att=att)
            else:
                dir = MailTemplate.get('send_pcap')
                setattr(obj, dir.to_mail, ';'.join([obj.email, dir.cc_mail]))
                res = MailSender.apply_mail(objToSend, 'send_pcap', att=att)
            if res:
                if hasattr(MailSender, 'last_error'):
                    obj.progress = MailSender.last_error
                else:
                    obj.progress = 'unknown mail send error'
        else:
            obj.progress = 'no mail address given'

        obj.status = 'success'
        obj.progress = 'send ok'
        obj.save()

    except Exception as e:
        obj.progress = str(e)[:1024]
        obj.status = 'fail'
        log.debug('do_pcap_email_async_task {} error:{}'.format(request_id, e))
    obj.job_end_time = datetime.now(UTC)

    obj.save()
    log.debug('do_pcap_email_async_task {}'.format(request_id))


@app.task(base=SqlAlchemyTask, time_limit=288000, soft_time_limit=287400)
def do_did_pending_task():
    log.debug('pending update started')
    cls = model.DidVendor
    vendors = cls.filter(cls.api_enabled).all()
    date = datetime.now(UTC)
    cls = model.DidBillingRel
    for v in vendors:
        try:
            api = v.get_api()
            pending = [i['number'] for i in api.pending()]
            # pending = ['12543544168']
            log.debug('pending found {}'.format(pending))
            vendor_res_id = v.vendor.resource.resource_id
            q = cls.filter(and_(cls.start_date <= date, cls.end_date.is_(None), cls.ingress_res_id == vendor_res_id,
                                cls.status == 0))
            q.update(dict(status=1), synchronize_session='fetch')
            if pending:
                q1 = cls.filter(
                    and_(cls.start_date <= date, cls.end_date.is_(None), cls.ingress_res_id == vendor_res_id,
                         cls.did.in_(pending)))
                q1.update(dict(status=0), synchronize_session='fetch')
            cls.session().commit()
            log.debug('pending updated for vendor {}'.format(v.id))
        except Exception as e:
            log.warning('pending update for vendor {} error:{}'.format(v.id, e))
    log.debug('pending update finished')


@app.task(base=SqlAlchemyTask)
def do_did_deducting_task():
    log.debug('do_did_deducting_task started ')
    date = datetime.now(UTC)
    cls = model.DidBillingRel
    q = cls.session().query(
        cls.buy_billing_plan_id, cls.client_res_id, func.count(cls.id).label('cnt')).where(
        and_(cls.start_date <= date, cls.end_date.is_(None))).group_by(
        cls.buy_billing_plan_id, cls.client_res_id)
    is_week_start = date.weekday() == 1
    is_month_start = date.day == 1
    for item in q:
        if item.buy_billing_plan_id and item.client_res_id:
            plan = model.DidBillingPlan.get(item.buy_billing_plan_id)
            client_id = model.Resource.get(item.client_res_id).client_id
            if is_month_start and plan.monthly_fee:
                balance = round(plan.monthly_fee * item.cnt, 2)
                model.ClientBalanceOperationAction(client_id=client_id, balance=balance, action=3,
                                                   create_by='did_monthly_fee').save()
    log.debug('do_did_deducting_task finished ')

@app.task(base=SqlAlchemyTask)
def do_did_param_missing_task():
    try:
        log.debug('do_did_param_missing_task started')
        cls = model.DidRepository
        param = model.DidParam

        dids = cls.session().query(cls.did.distinct().label('did')).filter(
            ~exists(select([1]).where(cls.did == param.did))
        ).all()
        for did in dids:
            if len(did.did) not in (10,11):
                continue
            log.debug('do_did_param_missing_task started for did {}'.format(did.did))
            d = param(did=did.did)
            d.update()
            d.save()
        log.debug('do_did_param_missing_task finished')
    except Exception as e:
        log.error('error during do_did_param_missing_task {}'.format(str(e)))

@app.task(base=SqlAlchemyTask)
def do_did_param_sync_task():
    try:
        log.debug('do_did_param_sync_task started')

        current_dids = set(row.did for row in model.DidRepository.filter().all())
        did_param_dids = set(row.did for row in model.DidParam.filter().all())

        dids = []
        for did in current_dids:
            if len(did) == 10 or (len(did) == 11 and did[0] == "1"):
                dids.append(did)
        current_dids = set(dids)

        new_dids = current_dids - did_param_dids
        log.debug('do_did_param_sync_task number of new dids is {}'.format(len(new_dids)))

        if new_dids:
            obj = model.VoipGateway.filter().first()
            api = get_dnl_active_calls_session(obj.lan_ip, obj.lan_port)
            log.debug('VoipGatewayCallStats with {} logged in'.format(obj.id))

            for did in new_dids:
                try:
                    sw_data = api._api_call('lerg_db_search {}'.format(did)).decode('utf-8')
                    reader = csv.reader(StringIO(sw_data))
                    values = next(reader)
                    if len(values) >= 11:
                        did_param = model.DidParam(
                            did=did,
                            country_iso=values[3],
                            ocn=values[8],
                            lata=values[5],
                            rate_type=None,
                            clec=values[10],
                            type=0 if values[6] == 'false' else 1,
                            state=values[4],
                            company=values[7],
                            switch=values[9]
                        )
                        did_param.save()
                        log.debug('do_did_param_sync_task inserted new DID parameter for did {}'.format(did))
                except Exception as e:
                    log.error('error during do_did_param_sync_task {}'.format(str(e)))

    except Exception as e:
        log.error('error during do_did_param_sync_task {}'.format(str(e)))

    log.debug('do_did_param_sync_task finished')

@app.task
def init_version_task(base=Singleton):
    log.debug('init_version_task started')
    from api_dnl.routes import ROUTES
    from api_dnl.auth import auth
    # icx = User.filter(User.name == settings.AUTH_ICX['name']).first()
    # if icx:
    #     if not auth.check_password(settings.AUTH_ICX['password'], icx.passwd):
    #         icx.passwd = settings.AUTH_ICX['password']
    #         icx.save()
    #         log.debug('icx user added')
    # else:
    #     icx = User(name=settings.AUTH_ICX['name'], passwd=settings.AUTH_ICX['password'],
    #                user_type='admin', role_id=0)
    #     icx.save()
    #     log.debug('icx user checked')

    VersionInformation.filter(VersionInformation.program_name == 'api_dnl').delete(synchronize_session='fetch')
    vi = VersionInformation(program_name='api_dnl')

    log.debug('version information {}'.format(vi))
    # if  vi.major_ver != __version__:
    if not (vi.minor_ver == __minor__ and vi.major_ver == __version__ and vi.build_date == __date__[0:10]):
        log.debug(
            'Database version is old {} {} {} app version {} {} {}'.format(vi.major_ver, vi.build_date, vi.minor_ver,
                                                                           __version__, __date__[0:10], __minor__))
        vi.major_ver = __version__
        vi.minor_ver = __minor__
        vi.build_date = __date__[0:10]
        vi.start_time = datetime.now(UTC)
        vi.save()
        if True:  # not len(SystemFunction.query().all()):
            SystemParameter.init()
            SystemFunction.init(ROUTES)
            CodeDeck.init()
            CodeCountry().init()
            # DailyCdrField.init()
            # Product.init()
            # RouteStrategy.init()
            Route.init()
            Role.init()
            MailTemplate.init()
            SendRateTemplate.init()
            RateTable.init()
            model.RateTypeName.init()
            model.DnisTypeName.init()
            model.JurTypeName.init()
    else:
        log.debug('version information is actual')
    pass
    log.debug('init_version_task finished')


@app.task(base=SqlAlchemyTask, time_limit=1801, soft_time_limit=1800)
def do_rate_import_preprocess(obj_id, auto_ij_rate, replace, ignore, map_header, interval, min_time, data, min_time_interval_per_prefixes):
    from api_dnl.model import RateImportTask
    log.debug('do_rate_import_preprocess started {} {} {}'.format(obj_id, auto_ij_rate, data))
    try:
        obj = RateImportTask.get(obj_id)
        file_path = obj.import_file_path + '/' + obj.orig_rate_file
        format_file_path = obj.import_file_path + '/' + obj.format_rate_file
        effective_date = data.get('effective_date', None)
        with open(file_path, 'rt') as inf:
            reader = csv.DictReader(inf)
            fn = [fn.lower() for fn in reader.fieldnames]
            if map_header:
                fn = map_header
            fmap = dict(zip(reader.fieldnames, fn))
            out_header = list(fmap.values())
            out_header = [x for x in out_header if x != 'ignore']
            if effective_date and 'effective_date' not in out_header:
                out_header.append('effective_date')
            if interval and 'interval' not in out_header:
                out_header.append('interval')
            if min_time and 'min_time' not in out_header:
                out_header.append('min_time')
            if min_time_interval_per_prefixes and not ('min_time' in out_header or 'interval' in out_header):
                out_header.append('min_time')
                out_header.append('interval')
            if auto_ij_rate and 'rate' not in out_header:
                out_header.append('rate')
            if 'ij_rate' in out_header and not 'rate' in out_header:
                out_header.append('rate')
                out_header.remove('ij_rate')

            for key, new_key in replace.items():
                if key.lower() in out_header:
                    out_header.remove(key.lower())
                    out_header.append(new_key.lower())

            npanxx = []
            if 'code' not in out_header:
                npa = [f for f in reader.fieldnames if 'npa' in f.lower()]
                nxx = [f for f in reader.fieldnames if 'nxx' in f.lower()]
                if npa and nxx and npa != nxx:
                    npanxx = [npa[0]] + [nxx[0]]
                    out_header.append('code')

            for field in ignore:
                if field.lower() in out_header:
                    out_header.remove(field.lower())

            # added = set()
            with open(format_file_path, 'wt') as outf:
                writer = csv.DictWriter(outf, fieldnames=out_header)
                writer.writeheader()
                for read_row in reader:
                    row = {fmap[k]: v for k, v in read_row.items() if fmap[k] != 'ignore'}
                    for key, new_key in replace.items():
                        if key.lower() in row and key.lower() != new_key.lower():
                            row[new_key.lower()] = row[key.lower()]
                            del row[key.lower()]
                    if  ('inter_rate' in row and 'intra_rate' in row) and auto_ij_rate is not None and not (row['inter_rate'] and row['intra_rate']):
                        continue
                    if len(set(npanxx)) == 2 and 'code' not in row:
                        row['code'] = read_row[npanxx[0]] + read_row[npanxx[1]]
                    if auto_ij_rate == 'Inter Rate':
                        row['rate'] = row['inter_rate'] if 'inter_rate' in row else 0.0
                    elif auto_ij_rate == 'Intra Rate':
                        row['rate'] = row['intra_rate'] if 'intra_rate' in row else 0.0
                    elif auto_ij_rate:
                        row['rate'] = max(row['inter_rate'] if 'inter_rate' in row else 0.0,
                                          row['intra_rate'] if 'intra_rate' in row else 0.0)
                    if 'ij_rate' in row and not 'rate' in row:
                        row['ij_rate'] = row.pop('rate')
                    if min_time:
                        row['min_time'] = min_time
                    if interval:
                        row['interval'] = interval
                    if effective_date:
                        row['effective_date'] = effective_date[:19]
                    if min_time_interval_per_prefixes and row['code'] in min_time_interval_per_prefixes:
                        row['min_time'] = min_time_interval_per_prefixes[row['code']]['min_time']
                        row['interval'] = min_time_interval_per_prefixes[row['code']]['interval']
                    # for key in replace:
                    #     row[key] = replace[key]
                    # if (row['code'], row.get('effective_date', None)) in added:
                    #     continue
                    # added.add((row['code'], row.get('effective_date', None)))
                    for key in ignore:
                        row.pop(key.lower(), None)
                    writer.writerow(row)
        obj.status = 'Initial'
        obj.save()
    except Exception as e:
        obj.status = 'API Failed'
        obj.save()
        log.debug('do_rate_import_preprocess error %s' % str(e))
    log.debug('do_rate_import_preprocess finished')

@app.task(base=SqlAlchemyTask)
def do_rate_analysis(obj_id):
    """Compare rates between existing rate table and new rates"""
    from api_dnl.model import RateAnalysisTask, RateTable, RateGenerationHistory
    log.debug('do_rate_analysis started {}'.format(obj_id))
    
    try:
        obj = RateAnalysisTask.get(obj_id)
        
        # Get source rate table
        source_table = RateTable.get(obj.source_rate_table_id)
        if not source_table:
            raise Exception("Source rate table not found")

        # Build dict of existing rates
        existing_rates = {}
        for rate in source_table.rates:
            if obj.source_effective_date and rate.effective_date != obj.source_effective_date:
                continue
            existing_rates[rate.code] = {
                'rate': rate.rate,
                'inter_rate': rate.inter_rate,
                'intra_rate': rate.intra_rate
            }

        # Get target rates based on input type
        target_rates = []
        if obj.target_rate_table_id:
            # Compare against existing rate table
            target_table = RateTable.get(obj.target_rate_table_id)
            if not target_table:
                raise Exception("Target rate table not found")
            target_rates = [
                {
                    'code': r.code,
                    'rate': r.rate,
                    'inter_rate': r.inter_rate, 
                    'intra_rate': r.intra_rate
                }
                for r in target_table.rates
                if not obj.target_effective_date or r.effective_date == obj.target_effective_date
            ]
        elif obj.rate_generation_history_id:
            # Compare against rate generation result
            history = RateGenerationHistory.get(obj.rate_generation_history_id)
            if not history:
                raise Exception("Rate generation history not found")
            target_rates = history.get_rates()
        elif obj.target_file_name:
            # Compare against import file
            import csv
            with open(obj.target_file_name, 'r') as f:
                reader = csv.DictReader(f)
                target_rates = list(reader)

        # Initialize counters
        result = {
            'increase': 0,
            'decrease': 0,
            'no_change': 0,
            'new': 0,
            'delete': 0
        }

        # Compare rates
        for new_rate in target_rates:
            code = new_rate.get('code')
            if code in existing_rates:
                old = existing_rates[code]
                
                if source_table.is_us_js:
                    # Compare inter/intra rates for US JS table
                    new_inter = float(new_rate.get('inter_rate', 0))
                    new_intra = float(new_rate.get('intra_rate', 0))
                    
                    if new_inter > old['inter_rate'] or new_intra > old['intra_rate']:
                        result['increase'] += 1
                    elif new_inter < old['inter_rate'] or new_intra < old['intra_rate']:
                        result['decrease'] += 1
                    else:
                        result['no_change'] += 1
                else:
                    # Compare single rate for non-US JS table
                    new_rate_val = float(new_rate.get('rate', 0))
                    if new_rate_val > old['rate']:
                        result['increase'] += 1
                    elif new_rate_val < old['rate']:
                        result['decrease'] += 1
                    else:
                        result['no_change'] += 1
                        
                existing_rates.pop(code)
            else:
                result['new'] += 1

        # Any remaining codes in existing_rates were not in new data
        result['delete'] = len(existing_rates)

        obj.status = 'Completed'
        obj.progress = 100
        obj.result_json = str(result)
        obj.finished_on = datetime.now(UTC)
        obj.save()

    except Exception as e:
        obj.status = 'Failed' 
        obj.expense_detail = str(e)
        obj.save()
        log.error('do_rate_analysis error: %s', str(e))

    log.debug('do_rate_analysis finished')


@app.task(base=SqlAlchemyTask)
def do_lcr_task(task_id):
    """
    Process LCR task by getting sorted rates for each code from rate tables
    """
    from api_dnl.model import LcrTask, RateTable, VoipGateway
    import csv
    
    log.debug('do_lcr_task started %s', task_id)
    
    try:
        task = LcrTask.get(task_id)
        if not task:
            raise Exception(f'LCR task {task_id} not found')
            
        task.started_on = datetime.now(UTC)
        task.save()

        # Get target rate table
        target_table = RateTable.get(task.target_rate_table_id)
        if not target_table:
            raise Exception(f'Target rate table {task.target_rate_table_id} not found')

        # Get codes from target rate table
        codes = [r.code for r in target_table.rates]
        
        # Get VoIP gateway session
        gateway = model.VoipGateway.query().first()
        if not gateway:
            raise Exception('No VoIP gateway found')

        s = get_dnl_switch_session(gateway.lan_ip, gateway.lan_port)
        if not s:
            raise Exception('Could not connect to VoIP gateway')

        # Create result file
        result_path = f'lcr_task_{task_id}_lcr.csv'
        task.result_path = result_path
        
        with open(result_path, 'w', newline='') as f:
            writer = csv.writer(f)
            headers_written = False
            
            # Process each code
            for code in codes:
                # Build rate_sorting command
                cmd = f'rate_sorting {task.effective_date},{code}'
                for rt_id in task.lcr_rate_table_id:
                    cmd += f',{rt_id}'
                    
                # Execute command via gateway session
                response = s.api_call(cmd)
                result = response.decode('utf-8')
                
                # Parse results
                lines = result.split('\n')
                
                # Get headers from first response
                if not headers_written:
                    for line in lines:
                        if line.startswith('['):
                            # Parse header line and write headers
                            headers = line.strip('[]').split(',')
                            writer.writerow(headers)
                            headers_written = True
                            break
                
                # Write data rows
                for line in lines:
                    if line and not line.startswith('['):
                        writer.writerow(line.strip().split(','))

        task.finished_on = datetime.now(UTC)
        task.save()
        
        log.debug('do_lcr_task completed successfully')
        
    except Exception as e:
        log.error('do_lcr_task error: %s', str(e))
        if task:
            task.finished_on = datetime.now(UTC)
            task.save()



@app.task(base=SqlAlchemyTask)
def do_block_number_import_preprocess(obj_id, egress_client_id, egress_trunk_id, egress_trunk_group_id,
                                      ingress_client_id, ingress_trunk_id, ingress_trunk_group_id, block_type, data):
    from api_dnl.model import BlockNumberImportTask
    log.debug('do_block_number_import_preprocess started {} {} {} {} {} {} {}'.format(obj_id, egress_client_id, egress_trunk_id, egress_trunk_group_id,
                                      ingress_client_id, ingress_trunk_id, ingress_trunk_group_id, data))
    try:
        obj = BlockNumberImportTask.get(obj_id)

        file_path = obj.import_file_path + '/' + obj.orig_import_file
        format_file_path = obj.import_file_path + '/'
        format_file_path += obj.format_import_file if obj.format_import_file else obj.orig_import_file
        time_profile_name, ANI_min, ANI_max, DNIS_min, DNIS_max, dnis, ani = (True,) * 7
        with open(file_path, 'rt') as inf:
            reader = csv.DictReader(inf)
            fn = [fn.lower() for fn in reader.fieldnames]
            fmap = dict(zip(reader.fieldnames, fn))
            out_header = list(fmap.values())
            out_header = [x for x in out_header if x != 'ignore']

            if 'ani' not in out_header:
                out_header.append('ani')
                ani = False
            # if 'egress_client_id' not in out_header:
            #     out_header.append('egress_client_id')
            if 'egress_trunk_id' not in out_header:
                out_header.append('egress_trunk_id')
            if 'egress_trunk_group_id' not in out_header:
                out_header.append('egress_group_id')
            # if 'ingress_client_id' not in out_header:
            #     out_header.append('ingress_client_id')
            if 'ingress_trunk_id' not in out_header:
                out_header.append('ingress_trunk_id')
            if 'ingress_trunk_group_id' not in out_header:
                out_header.append('ingress_group_id')
            # if 'block_type' not in out_header:
            #     out_header.append('block_type')
            if 'time_profile_id' not in out_header:
                out_header.append('time_profile_id')
                time_profile_name = False
            if 'ani_min_len' not in out_header:
                out_header.append('ani_min_len')
                ANI_min = False
            if 'ani_max_len' not in out_header:
                out_header.append('ani_max_len')
                ANI_max = False
            if 'dnis_min_len' not in out_header:
                out_header.append('dnis_min_len')
                DNIS_min = False
            if 'dnis_max_len' not in out_header:
                out_header.append('dnis_max_len')
                DNIS_max = False
            if 'dnis' not in out_header:
                out_header.append('dnis')
                dnis = False

            with open(format_file_path, 'wt') as outf:
                writer = csv.DictWriter(outf, fieldnames=out_header)
                writer.writeheader()
                for read_row in reader:
                    row = {fmap[k]: v for k, v in read_row.items() if fmap[k] != 'ignore'}
                    for field in ('egress_trunk_id', 'ingress_trunk_id'):
                        if field not in row:
                            row[field] = str(eval(field)) if eval(field) else ''

                    if not time_profile_name:
                        row['time_profile_id'] = ''
                    if not ANI_min:
                        row['ani_min_len'] = ''
                    if not ANI_max:
                        row['ani_max_len'] = ''
                    if not DNIS_min:
                        row['dnis_min_len'] = ''
                    if not DNIS_max:
                        row['dnis_max_len'] = ''
                    if not dnis:
                        row['dnis'] = ''
                    if not ani:
                        row['ani'] = ''

                    if ingress_trunk_group_id is not None:
                        row['ingress_group_id'] = ingress_trunk_group_id
                    if egress_trunk_group_id is not None:
                        row['egress_group_id'] = egress_trunk_group_id

                    # if not (row['ani'] or row['dnis']):
                    #     continue
                    writer.writerow(row)
        obj.status = 'Initial'
        obj.save()
    except Exception as e:
        obj.status = 'Upload failed'
        obj.save()
        log.debug('do_rate_import_preprocess error %s' % str(e))
    log.debug('do_rate_import_preprocess finished')

@app.task(base=SqlAlchemyTask, time_limit=288000, soft_time_limit=287400)
def do_did_export_async_task(export_id):
    from api_dnl.views.did_export import _gen_csv
    task = DidExportAsyncTask.get(export_id)
    if task:
        for d in _gen_csv(task):
            log.debug('do_did_export_async_task running..')
    log.info('do_did_export_async_task finished success')


@app.task(base=SqlAlchemyTask)
def do_did_number_upload_task_preprocess(task_id, failover_id):
    from api_dnl.model import DidNumberUploadTask
    log.debug('do_did_number_upload_task_preprocess started {}'.format(task_id))
    try:
        obj = DidNumberUploadTask.get(task_id)
        obj.upload_format_file = '_' + obj.upload_orig_file
        file_name = obj.upload_file_path + '/' + obj.upload_orig_file
        format_file_name = obj.upload_file_path + '/' + obj.upload_format_file

        vendor = model.Resource.filter(model.Resource.alias == obj.did_vendor_name)
        vendor = vendor.first().resource_id if vendor.first() else ''
        vendor_billing_plan = model.DidBillingPlan.filter(model.DidBillingPlan.name == obj.vendor_billing_rule_name)
        vendor_billing_plan = vendor_billing_plan.first().id if vendor_billing_plan.first() else ''
        client = model.Resource.filter(model.Resource.alias == obj.did_client_name)
        client = client.first().resource_id if client.first() else ''
        client_billing_plan = model.DidBillingPlan.filter(model.DidBillingPlan.name == obj.client_billing_rule_name)
        client_billing_plan = client_billing_plan.first().id if client_billing_plan.first() else ''

        with open(file_name, 'rt') as inf:
            with open(format_file_name, 'wt') as outf:
                reader = csv.DictReader(inf)
                writer = csv.DictWriter(outf,
                                        fieldnames=list(
                                            set(reader.fieldnames + ['vendor_trunk_id', 'vendor_billing_plan_id',
                                                                     'client_trunk_id', 'client_billing_plan_id',
                                                                     'fallback_trunk_id'])))
                writer.writeheader()
                for row in reader:
                    row['vendor_trunk_id'] = vendor
                    row['vendor_billing_plan_id'] = vendor_billing_plan
                    row['client_trunk_id'] = client
                    row['client_billing_plan_id'] = client_billing_plan
                    row['fallback_trunk_id'] = failover_id or ''
                    writer.writerow(row)
        obj.status = 'Initial'
        obj.save()
    except Exception as e:
        log.debug('do_did_number_upload_task_preprocess error %s' % str(e))
    log.debug('do_did_number_upload_task_preprocess finished')


@app.task(base=SqlAlchemyTask)
def do_save_history(data):
    log.debug('do_save_history start')
    from api_dnl.model import ObjectRevisionRecordModel
    sess = db.session
    for item in data:
        log.debug(item)
        sess.add(ObjectRevisionRecordModel(**item))
    sess.commit()
    log.debug('do_save_history finish')


if __name__ == '__main__':
    # do_notify_zero_balance()
    # do_clear_last_lowbalance_send_time()
    # do_daily_usage_summary()
    # do_daily_balance_summary()
    # daily_balance_carry_task()
    from api_dnl.task.alert import do_frund_detection

    do_frund_detection()


@app.task(base=SqlAlchemyTask, time_limit=288000, soft_time_limit=287400)
def do_did_update_mrc():
    report_date = datetime.now(UTC).date()
    model.DidReport.update_mrc(report_date)
