from telnetlib import Telnet
import re
from xmljson import BadgerFish, Parker as X2J
from xml.etree.ElementTree import fromstring
import json
from time import sleep
from datetime import datetime, timedelta
from falcon_rest.logger import log
import threading

CONNECT_TIMEOUT = 65000
READ_TIMEOUT = 10
CACHE_DELAY_MINUTES = 2
_call_stat = {60: {}, 1440: {}, 0: {}}
_sessions = {}


class DnlActiveCallException(Exception):
    pass


class DnlActiveCallSession(object):
    FIELDS = ['start_time',
              'answer_time',
              'orig_ani',
              'orig_dnis',
              'orig_ip',
              'orig_profile_ip',
              'orig_code',
              'orig_rtp_ip',
              'orig_rtp_port',
              'orig_sw_rtp_ip',
              'orig_sw_rtp_port',
              'ingress_carrier',
              'ingress_trunk',
              'lrn_number',
              'orig_rate_type',
              'orig_rate',
              'orig_rate_table',
              'term_ani',
              'term_dnis',
              'term_ip',
              'term_profile_ip',
              'term_code',
              'term_rtp_ip',
              'term_rtp_port',
              'term_sw_rtp_ip',
              'term_sw_rtp_port',
              'egress_carrier',
              'egress_trunk',
              'term_rate_type',
              'term_rate',
              'egress_rate_table',
              'routing_plan',
              'dynamic_route',
              'static_route',
              'ingress_call_id',
              'egress_call_id']

    def connect(self):
        if hasattr(self, 'conn'):
            try:
                self.conn.close()
            except:
                pass
        self.conn = Telnet(self.ip, self.port, timeout=CONNECT_TIMEOUT)
        # CLAS6-17106 - API for call simulation is very slow
        # Fix: first read didn't return expected response
        # self.conn.read_until(b"Escape character is '^]'.", self.timeout)

    def __init__(self, ip, port, timeout=READ_TIMEOUT):
        self.conn = None
        log.debug('DnlActiveCall try connect host {} port {} '.format(ip, port))
        self.ip = ip
        self.port = port
        _sessions[self.addr()] = self
        self.timeout = timeout
        self.lock = threading.RLock()
        self.connect()

    def __del__(self):
        try:
            self.conn.close()
        except Exception as e:
            log.error('DnlActiveCall: {}'.format(e))

    def addr(self):
        return self.ip, self.port

    def _api_write(self, command):
        log.debug('DnlActiveCall host: {} port: {} command: {}'.format(self.conn.host, self.conn.port, command))
        try:
            self.conn.write(bytes(command, encoding='ascii') + b"\r\n")
            log.debug("SUCCESS")
        except:
            log.debug(
                'DnlActiveCall NOT CONNECTED host: {} port: {} command: {}'.format(self.conn.host, self.conn.port,
                                                                                   command))
            self.connect()
            sleep(0.1)
            self.login()
            sleep(0.1)
            self.conn.write(bytes(command, encoding='ascii') + b"\r\n")

    def _api_read(self):
        response = self.conn.read_very_eager()
        log.debug('DnlActiveCall response: {}'.format(response))
        try:
            result = str(response).strip('\n')
        except Exception as e:
            raise DnlActiveCallException('Bad response. {}'.format(e))

        return result

    def _api_call(self, command, expect=[b'']):
        self.lock.acquire()
        try:
            self._api_write(command)
            sleep(5)
            result = self.conn.read_very_eager()
        except EOFError:
            self.conn = Telnet(self.ip, self.port, timeout=CONNECT_TIMEOUT)
            # self.connect()
            self.conn.read_until(b"Escape character is '^]'.", self.timeout)
            self.login()
            sleep(0.1)
            self._api_write(command)
            sleep(5)
            result = self.conn.read_very_eager()
        finally:
            self.lock.release()
        return result

    def login(self):
        for i in range(0, 10):
            if self._login():
                log.debug('DnlActiveCall login SUCCESS')
                return True
            sleep(0.01)
            self.connect()
            log.debug('DnlActiveCall reconnect retry {} after delay'.format(i))
        log.error('DnlActiveCall Can\'t login. after 5 retries')
        raise DnlActiveCallException('DnlActiveCall Can\'t login. after 5 retries')

    def _login(self):
        try:
            log.debug("_login")
            self._api_write('login')
            r = self.conn.read_until(b'Welcome', timeout=READ_TIMEOUT)
            result = r + self.conn.read_eager()
            if b'Welcome' not in result:
                log.debug('DnlActiveCall not logged in: {}'.format(result))
                return False
            self.logged_in = True
            return True
        except EOFError as e:
            return False

    def get_active_call(self, switch, fields=None, filters=None, first_record_number=1, last_record_number=65535):
        if fields == None:
            fld = ','.join(self.FIELDS)
        else:
            ofields = []
            for f in self.FIELDS:
                if f in fields:
                    ofields.append(f)
            fld = ','.join(ofields)
        if filters == None or filters == {}:
            filt = 'all'
        else:
            filt = ','.join(['{}={}'.format(k, v) for k, v in filters.items()])

        command = 'get_active_call {} {} {} {},{}'.format(switch, fld, filt, first_record_number, last_record_number)

        response = self._api_call(command)
        response = response.decode('utf-8')
        display_count = 0
        total = 0
        a = re.findall(r'^.*display count: (\d+); total: (\d+)', response, re.MULTILINE)
        if a:
            display_count = a[0][0]
            total = a[0][1]

        _resp = 's0-64011fac-0003bf05-1cb2e5e1-0a4f16da;1513134004905568;1513134005011124;86;86;192.99.10.113;172.31' \
                '.1.100;0;192.99.10.113;6004;;0;1937;1740;;3;0.000300;1372;86;86;192.99.10.113;172.31.1.100;0;192.99' \
                '.10.113;6000;;0;1938;1739;3;0.000200;1371;257;303;?\r\n' \
                's0-64011fac-0003bf07-024bc041-70140e78;1513134005404026;1513134005508843;86;86;192.99.10.113;172.31' \
                '.1.100;0;192.99.10.113;6004;;0;1937;1740;;3;0.000300;1372;86;86;192.99.10.113;172.31.1.100;0;192.99' \
                '.10.113;6000;;0;1938;1739;3;0.000200;1371;257;303;?\r\n' \
                'display count'
        resp = response.replace('\r', '').split('\n')
        f = ('uuid,' + fld).split(',')
        ret = []
        for rec in resp:
            if not 'display count' in rec and rec:
                ret.append(dict(zip(f, rec.split(';'))))

        print(ret, display_count, total)
        # x2j = X2J(xml_fromstring=False,dict_type=dict)
        # data=x2j.data(fromstring(resp),preserve_root=True)
        return ret, display_count, total

    def sip_profile_start(self):
        response = self._api_call('sip_profile_start').decode('utf-8')
        resp = '<payload>' + response + '</payload>'
        # resp = resp.replace('sip_profile_start', 'Simulation-progress')
        print(resp)
        x2j = X2J(xml_fromstring=False, dict_type=dict)
        data = x2j.data(fromstring(resp), preserve_root=True)
        return data

    @classmethod
    def get_call_stat_cache(cls, ip, minutes):
        m = minutes if minutes in (60, 1440) else 0
        if _call_stat and ip in _call_stat[m] and _call_stat[m][ip]['time'] > datetime.now():
            log.debug('return _call_stat from cache')
            return _call_stat[m][ip]['value']
        return None

    def get_call_stat(self, minutes):
        m = minutes if minutes in (60, 1440) else 0
        ret = self._api_call('get_call_statistics {}'.format(minutes)).decode('utf-8')
        head = 'daily_minutes,daily_margin,daily_margin_percentage,last_asr_percentage,last_acd,last_revenue,' \
               'last_profitability_percentage,last_non_zero_calls,last_total_calls'.split(
            ',')
        ret = dict(zip(head, list(map(lambda x: float(x), ret.split(',')))))
        return ret

    def update_system_limit(self, self_cap, self_cps):
        # ret = self._api_call('set_system_limit {} {}'.format(cap, cps)).decode('utf-8')
        ret = self._api_call('set_system_limit {} {}'.format(self_cps, self_cap)).decode('utf-8')
        return ret

    def get_rate_sorting(self, effective_date, code, rate_tables):
        ret = self._api_call('rate_sorting {},{},{}'.format(effective_date, code, ','.join([str(i) for i in rate_tables]))).decode('utf-8')
        # head = 'type,rate'.split(',')
        # ret = dict(zip(head, list(map(lambda x: x, ret.split(' ')))))
        return ret

    def sign_call(self, ani, dnis, attestation_lvl):
        ret = self._api_call('shaken_sign {},{},{}'.format(ani,dnis,attestation_lvl)).decode('utf-8')
        return ret.replace('\r\n', '')

    def verify_token_header(self, token_header):
        ret = self._api_call('shaken_vfy {}'.format(token_header)).decode('utf-8')
        return ret

    def lerg_db_search(self, number):
        result = self._api_call('lerg_db_search {}'.format(number)).decode('utf-8').strip()
        return result

    def spamdb_dnc_search(self, number):
        result = self._api_call('spamdb_dnc_search {}'.format(number)).decode('utf-8').strip()
        return result

    def spamdb_dno_search(self, number):
        result = self._api_call('spamdb_dno_search {}'.format(number)).decode('utf-8').strip()
        return result

    def spamdb_youmail_search(self, number):
        result = self._api_call('spamdb_youmail_search {}'.format(number)).decode('utf-8').strip()
        return result

    def spamdb_ftc_search(self, number):
        result = self._api_call('spamdb_ftc_search {}'.format(number)).decode('utf-8').strip()
        return result

    def lrn_search(self, number):
        result = self._api_call('lrn_search {}'.format(number)).decode('utf-8').strip()
        return result


def get_dnl_active_calls_session(ip, port):
    try:
        if (ip, port) in _sessions:
            log.debug('DnlActiveCall session {}:{} found'.format(ip, port))
            sess = _sessions[(ip, port)]
            if hasattr(sess, 'conn') and sess.conn:
                if hasattr(sess, 'logged_in') and sess.logged_in:
                    log.debug('DnlActiveCall session logged in')
                    try:
                        sess._api_read()
                        return sess
                    except Exception as e:
                        log.debug('DnlActiveCall session cannot read: {}'.format(e))
            log.debug('DnlActiveCall session reconnect and login')
            sess.connect()
            sess.login()
            return sess
        else:
            log.debug('DnlActiveCall session new {}:{}'.format(ip, port))
            sess = DnlActiveCallSession(ip, port)
            sess.login()
            return sess
    except Exception as e:
        raise DnlActiveCallException(
            'Active calls telnet interface error. Server IP {} Port {}. Detailed error {}'.format(ip, port, e))


class DnlActiveCall(object):
    """
    """

    def __init__(self, ip='127.0.0.1', port='4320', timeout=5):
        self.ip = ip
        self.port = port
        self.timeout = timeout

    def login(self):
        try:
            fs = DnlActiveCallSession(self.ip, self.port, self.timeout)
            fs.login()
            # data = fs.sip_profile_start()
            # data=fs.logout()
        except Exception as e:
            raise DnlActiveCallException(e)


if __name__ == '__main__':
    # CallApi.test_call('127.0.0.1','6235','1.2.3.4',5555,1111,2222,0)
    api = DnlActiveCallSession('127.0.0.1', '4320')
    ret = api.login()
    fields = 'start_time,answer_time,orig_ani,orig_dnis,orig_ip,orig_profile_ip,orig_code,orig_rtp_ip,orig_rtp_port,' \
             'orig_sw_rtp_ip,orig_sw_rtp_port,ingress_carrier,ingress_trunk,lrn_number,orig_rate_type,orig_rate,' \
             'orig_rate_table,term_ani,term_dnis,term_ip,term_profile_ip,term_code,term_rtp_ip,term_rtp_port,' \
             'term_sw_rtp_ip,term_sw_rtp_port,egress_carrier,egress_trunk,term_rate_type,term_rate,egress_rate_table,' \
             'routing_plan,dynamic_route,static_route '

    print(api.get_active_call('class4'))
    print(api.get_active_call('class4', fields=['start_time'], filters={'ingress_carrier': 123}, first_record_number=3,
                              last_record_number=20))
