import requests
from urllib.parse import urlencode, parse_qsl
from datetime import datetime, timedelta
from pytz import UTC
from time import sleep
from dateutil.parser import parse as parse_datetime
from sqlalchemy.exc import InternalError, IntegrityError, DataError
from sqlalchemy import and_, or_, inspect, alias, text, func, select, case, cast
from sqlalchemy.orm import aliased
from api_dnl import model
from api_dnl.utils.statisticapi2 import cdr_fields

import falcon
from falcon_rest import conf
from falcon_rest.logger import log
from falcon_rest import schemes
from falcon_rest.responses import responses
from falcon_rest.responses import errors
from falcon_rest.resources.resources import swagger, ATTRIBUTE_ERROR_RE
from falcon_rest.helpers import check_permission
from api_dnl.model import ImportExportLogs, VoipGateway

from api_dnl.resources import check_context, DnlList, ValidationError, DnlCreate, DnlResource, CustomGetAction, \
    CustomPostAction, \
    CustomDeleteAction, CustomPatchAction
from api_dnl.schemes.cloud import *
from api_dnl.view import DEFAULT_SECURITY, OperationErrorResponse
import json
import gzip
import io
import tarfile


def db_man_base():
    conf = model.DnlCloudDbmanCfg.filter(model.DnlCloudDbmanCfg.server_ip == '127.0.0.1').first()
    if not conf:
        raise Exception("Cloud Database Manager not configured. Call a tech personnel")
    return '{}://{}:{}'.format('https' if conf.server_use_ssl else 'http', conf.server_ip, conf.server_port)


def fd_base():
    conf = model.DnlCloudDownloaderCfg.filter(model.DnlCloudDownloaderCfg.server_ip == '127.0.0.1').first()
    if not conf:
        raise Exception("Cloud Downloader not configured. Call a tech personnel")
    return '{}://{}:{}'.format('https' if conf.server_use_ssl else 'http', conf.server_ip, conf.server_port)


def ds_base(multiple=False):
    if multiple:
        get_url = lambda conf: '{}://{}:{}'.format('https' if conf.server_use_ssl else 'http', conf.server_ip, conf.server_port)
        result = [get_url(conf) for conf in model.DnlCloudSearchCfg.filter().all()]
        if not result:
            raise Exception("Cloud Search not configured. Call a tech personnel")
        return result
    conf = model.DnlCloudSearchCfg.filter(model.DnlCloudSearchCfg.server_ip == '127.0.0.1').first()
    if not conf:
        conf = model.DnlCloudSearchCfg.filter().first()
        if not conf:
            raise Exception("Cloud Search not configured. Call a tech personnel")
    return '{}://{}:{}'.format('https' if conf.server_use_ssl else 'http', conf.server_ip, conf.server_port)

FIELDS = {
    1:	'time',
    2:	'release_cause',
    3:	'start_time_of_date',
    4:	'answer_time_of_date',
    5:	'release_tod',
    6:	'minutes_west_of_greenwich_mean_time',
    7:	'release_cause_from_protocol_stack',
    8:	'binary_value_of_release_cause_from_protocol_stack',
    9:	'first_release_dialogue',
    10:	'origination_source_number',
    11:	'origination_source_host_name',
    12:	'origination_destination_number',
    13:	'origination_destination_host_name',
    14:	'origination_call_id',
    15:	'origination_remote_payload_ip_address',
    16:	'origination_remote_payload_udp_address',
    17:	'origination_local_payload_ip_address',
    18:	'origination_local_payload_udp_address',
    19:	'origination_codec_list',
    20:	'termination_source_number',
    21:	'termination_source_host_name',
    22:	'termination_destination_number',
    23:	'termination_destination_host_name',
    24:	'termination_call_id',
    25:	'termination_remote_payload_ip_address',
    26:	'termination_remote_payload_udp_address',
    27:	'termination_local_payload_ip_address',
    28:	'termination_local_payload_udp_address',
    29:	'termination_codec_list',
    30:	'final_route_indication',
    31:	'routing_digits',
    32:	'call_duration',
    33:	'pdd',
    34:	'ring_time',
    35:	'callduration_in_ms',
    36:	'conf_id',
    37:	'call_type',
    38:	'ingress_id',
    39:	'ingress_client_id',
    40:	'ingress_client_rate_table_id',
    41:	'ingress_client_currency_id',
    42:	'ingress_client_rate',
    43:	'ingress_client_currency',
    44:	'ingress_client_bill_time',
    45:	'ingress_client_bill_result',
    46:	'ingress_client_cost',
    47:	'egress_id',
    48:	'egress_rate_table_id',
    49:	'egress_rate',
    50:	'egress_cost',
    51:	'egress_bill_time',
    52:	'egress_client_id',
    53:	'egress_client_currency_id',
    54:	'egress_client_currency',
    55:	'egress_six_seconds',
    56:	'egress_bill_minutes',
    57:	'egress_bill_result',
    58:	'ingress_bill_minutes',
    59:	'ingress_dnis_type',
    60:	'ingress_rate_type',
    61:	'lrn_dnis',
    62:	'egress_dnis_type',
    63:	'egress_rate_type',
    64:	'item_id',
    65:	'translation_ani',
    66:	'ingress_rate_id',
    67:	'egress_rate_id',
    68:	'orig_code',
    69:	'orig_code_name',
    70:	'orig_country',
    71:	'term_code',
    72:	'term_code_name',
    73:	'term_country',
    74:	'ingress_rate_effective_date',
    75:	'egress_rate_effective_date',
    76:	'egress_erro_string',
    77:	'order_id',
    78:	'order_type',
    79:	'is_final_call',
    80:	'egress_code_asr',
    81:	'egress_code_acd',
    82:	'static_route',
    83:	'dynamic_route',
    84:	'route_plan',
    85:	'route_prefix',
    86:	'orig_delay_second',
    87:	'term_delay_second',
    88:	'orig_call_duration',
    89:	'trunk_type',
    90:	'origination_profile_port',
    91:	'termination_profile_port',
    92:	'o_trunk_type2',
    93:	'o_billing_method',
    94:	't_trunk_type2',
    95:	't_billing_method',
    96:	'campaign_id',
    97:	'agent_id',
    98:	'agent_rate',
    99:	'agent_cost',
    100: 'orig_jur_type',
    101: 'term_jur_type',
    102: 'ring_epoch',
    103: 'end_epoch',
    104: 'paid_user',
    105: 'rpid_user',
    106: 'timeout_type',
    107: 'q850_cause',
    108: 'q850_cause_string'
}

# region +++DnlCloudDbmanCfg+++
class DnlCloudDbmanCfgCreate(DnlCreate):
    scheme_class = DnlCloudDbmanCfgScheme
    model_class = model.DnlCloudDbmanCfg
    entity = 'DnlCloudDbmanCfg'
    path_parameters = ()
    security = (DEFAULT_SECURITY)
    restrict = ()

    def on_post(self, req, resp, **kwargs):
        voip_gateway_count = model.VoipGateway.filter(True).count()
        dbman_cfg_count = model.DnlCloudDbmanCfg.filter(True).count()
        if voip_gateway_count * 2 < dbman_cfg_count + 1:
            raise Exception('Each VoipGateway should have only 2 DnlCloudDbmanCfg')
        return super().on_post(req, resp, **kwargs)

    def before_create(self, obj, **kwargs):
        server = VoipGateway.session().query(VoipGateway).first()
        obj.server_name = server.name
        # obj.created_by=user.name
        # obj.created_on=datetime.now(UTC)
        return obj


class DnlCloudDbmanCfgResource(DnlResource):
    model_class = model.DnlCloudDbmanCfg
    scheme_class = DnlCloudDbmanCfgScheme
    scheme_class_get = DnlCloudDbmanCfgSchemeGet
    scheme_class_modify = DnlCloudDbmanCfgSchemeModify
    entity = 'DnlCloudDbmanCfg'
    id_field = 'server_name'
    security = (DEFAULT_SECURITY)
    path_parameters = ()
    restrict = ()


class DnlCloudDbmanCfgList(DnlList):
    scheme_class = DnlCloudDbmanCfgSchemeGet
    model_class = model.DnlCloudDbmanCfg
    entity_plural = 'DnlCloudDbmanCfgs'
    path_parameters = ()
    security = (DEFAULT_SECURITY)
    restrict = ()

    def modify_query_from_filtering_for_list(self, filtering, **kwargs):
        filt, ret = super().modify_query_from_filtering_for_list(filtering, **kwargs)
        user = self.get_user(self.req)
        if not user.is_admin:
            cls = self.model_class
            # ret = ret.filter(cls.pool_id != 0)#TODO:filter for user
        return filt, ret


# endregion ---DnlCloudDbmanCfg---
# region +++DnlCloudDownloaderCfg+++
class DnlCloudDownloaderCfgCreate(DnlCreate):
    scheme_class = DnlCloudDownloaderCfgScheme
    model_class = model.DnlCloudDownloaderCfg
    entity = 'DnlCloudDownloaderCfg'
    path_parameters = ()
    security = (DEFAULT_SECURITY)
    restrict = ()

    def before_create(self, obj, **kwargs):
        user = self.get_user(self.req)
        # obj.created_by=user.name
        # obj.created_on=datetime.now(UTC)
        return obj


class DnlCloudDownloaderCfgResource(DnlResource):
    model_class = model.DnlCloudDownloaderCfg
    scheme_class = DnlCloudDownloaderCfgScheme
    scheme_class_get = DnlCloudDownloaderCfgSchemeGet
    scheme_class_modify = DnlCloudDownloaderCfgSchemeModify
    entity = 'DnlCloudDownloaderCfg'
    id_field = 'server_name'
    security = (DEFAULT_SECURITY)
    path_parameters = ()
    restrict = ()


class DnlCloudDownloaderCfgList(DnlList):
    scheme_class = DnlCloudDownloaderCfgSchemeGet
    model_class = model.DnlCloudDownloaderCfg
    entity_plural = 'DnlCloudDownloaderCfgs'
    path_parameters = ()
    security = (DEFAULT_SECURITY)
    restrict = ()

    def modify_query_from_filtering_for_list(self, filtering, **kwargs):
        filt, ret = super().modify_query_from_filtering_for_list(filtering, **kwargs)
        user = self.get_user(self.req)
        if not user.is_admin:
            cls = self.model_class
            # ret = ret.filter(cls.pool_id != 0)#TODO:filter for user
        return filt, ret


# endregion ---DnlCloudDownloaderCfg---
# region +++DnlCloudSearchCfg+++
class DnlCloudSearchCfgCreate(DnlCreate):
    scheme_class = DnlCloudSearchCfgScheme
    model_class = model.DnlCloudSearchCfg
    entity = 'DnlCloudSearchCfg'
    path_parameters = ()
    security = (DEFAULT_SECURITY)
    restrict = ()

    def before_create(self, obj, **kwargs):
        user = self.get_user(self.req)
        # obj.created_by=user.name
        # obj.created_on=datetime.now(UTC)
        return obj


class DnlCloudSearchCfgResource(DnlResource):
    model_class = model.DnlCloudSearchCfg
    scheme_class = DnlCloudSearchCfgScheme
    scheme_class_get = DnlCloudSearchCfgSchemeGet
    scheme_class_modify = DnlCloudSearchCfgSchemeModify
    entity = 'DnlCloudSearchCfg'
    id_field = 'server_name'
    security = (DEFAULT_SECURITY)
    path_parameters = ()
    restrict = ()


class DnlCloudSearchCfgList(DnlList):
    scheme_class = DnlCloudSearchCfgSchemeGet
    model_class = model.DnlCloudSearchCfg
    entity_plural = 'DnlCloudSearchCfgs'
    path_parameters = ()
    security = (DEFAULT_SECURITY)
    restrict = ()

    def modify_query_from_filtering_for_list(self, filtering, **kwargs):
        filt, ret = super().modify_query_from_filtering_for_list(filtering, **kwargs)
        user = self.get_user(self.req)
        if not user.is_admin:
            cls = self.model_class
            # ret = ret.filter(cls.pool_id != 0)#TODO:filter for user
        return filt, ret


# endregion ---DnlCloudSearchCfg---+
# region +++DnlCloudLog+++

class DnlCloudLogList(DnlList):
    scheme_class = DnlCloudLogSchemeGet
    model_class = model.DnlCloudLog
    entity_plural = 'DnlCloudLogs'
    path_parameters = ()
    security = (DEFAULT_SECURITY)
    restrict = ()

    def modify_query_from_filtering_for_list(self, filtering, **kwargs):
        filt, ret = super().modify_query_from_filtering_for_list(filtering, **kwargs)
        user = self.get_user(self.req)
        if not user.is_admin:
            cls = self.model_class
            # ret = ret.filter(cls.pool_id != 0)#TODO:filter for user
        return filt, ret


# endregion ---DnlCloudLog---
# region +++DnlCloudStorage+++
class DnlCloudStorageCreate(DnlCreate):
    scheme_class = DnlCloudStorageScheme
    model_class = model.DnlCloudStorage
    entity = 'DnlCloudStorage'
    path_parameters = ()
    security = (DEFAULT_SECURITY)
    restrict = ()

    def before_create(self, obj, **kwargs):

        user = self.get_user(self.req)
        # obj.created_by=user.name
        # obj.created_on=datetime.now(UTC)
        if obj.gcloud and not obj.gcloud.key_file_path:
            obj.gcloud.key_file_path = 'dummy_path'
        # if not obj.key_file_path:
        #     obj.key_file_path = 'dummy_path'
        if obj.type == 'ftp' and obj.ftp:
            if obj.ftp.netrc_id:
                f = ImportExportLogs.get(obj.ftp.netrc_id)
                obj.ftp.netrc_path = f.file_path + '/' + f.myfile_filename
                del obj.ftp.netrc_id
        if obj.type == 'sftp' and obj.sftp:
            if obj.sftp.pubkey_id:
                f = ImportExportLogs.get(obj.sftp.pubkey_id)
                obj.sftp.pubkey_path = f.file_path + '/' + f.myfile_filename
                del obj.sftp.pubkey_id
            if obj.sftp.privkey_id:
                f = ImportExportLogs.get(obj.sftp.privkey_id)
                obj.sftp.priv_path = f.file_path + '/' + f.myfile_filename
                del obj.sftp.privkey_id
        if obj.type == 'gcloud' and obj.gcloud:
            if obj.type == 'oauth2':
                if not obj.email or not obj.secret or not obj.token:
                    raise ValidationError('When type=oauth2 then email,secret,token are mandatory!')
            if obj.type == 'service':
                if not obj.key_file_id:
                    raise ValidationError('When type=service then json key_file should be uploaded!')
            if obj.gcloud.key_file_id:
                f = ImportExportLogs.get(obj.gcloud.key_file_id)
                obj.gcloud.key_file_path = f.file_path + '/' + f.myfile_filename
                del obj.gcloud.key_file_id

        return obj


class DnlCloudStorageResource(DnlResource):
    model_class = model.DnlCloudStorage
    scheme_class = DnlCloudStorageScheme
    scheme_class_get = DnlCloudStorageSchemeGet
    scheme_class_modify = DnlCloudStorageSchemeModify
    entity = 'DnlCloudStorage'
    id_field = 'uuid'
    security = (DEFAULT_SECURITY)
    path_parameters = ()
    restrict = ()

    def before_update(self, obj, req):
        if obj.ftp and 'ftp' in req.data:
            req.data['ftp']['uuid'] = obj.uuid
        if obj.sftp and 'sftp' in req.data:
            req.data['sftp']['uuid'] = obj.uuid
            if obj.sftp.pubkey_id:
                f = ImportExportLogs.get(obj.sftp.pubkey_id)
                obj.sftp.pubkey_path = f.file_path + '/' + f.myfile_filename
                del obj.sftp.pubkey_id
            if obj.sftp.privkey_id:
                f = ImportExportLogs.get(obj.sftp.privkey_id)
                obj.sftp.priv_path = f.file_path + '/' + f.myfile_filename
                del obj.sftp.privkey_id
        if obj.gcloud and 'gcloud' in req.data:
            req.data['gcloud']['uuid'] = obj.uuid
            ob = req.data['gcloud']

            class _R():
                pass

            obj2 = _R()
            obj2.__dict__ = ob
            # if obj2.type == 'oauth2':
            #     if not getattr(obj2, 'email', None) or not getattr(obj2, 'secret', None) or not getattr(obj2, 'token', None):
            #         raise ValidationError('When type=oauth2 then email,secret,token are mandatory!')
            if obj2.type == 'service':
                if not getattr(obj2, 'key_file_id', None):
                    raise ValidationError('When type=service then json key_file should be uploaded!')

            if obj.gcloud.key_file_id:
                f = ImportExportLogs.get(obj.gcloud.key_file_id)
                obj.gcloud.key_file_path = f.file_path + '/' + f.myfile_filename
                del obj.gcloud.key_file_id
        return obj


class DnlCloudStorageList(DnlList):
    scheme_class = DnlCloudStorageSchemeGet
    model_class = model.DnlCloudStorage
    entity_plural = 'DnlCloudStorages'
    path_parameters = ()
    security = (DEFAULT_SECURITY)
    restrict = ()

    def modify_query_from_filtering_for_list(self, filtering, **kwargs):
        filt, ret = super().modify_query_from_filtering_for_list(filtering, **kwargs)
        user = self.get_user(self.req)
        if not user.is_admin:
            cls = self.model_class
            # ret = ret.filter(cls.pool_id != 0)#TODO:filter for user
        return filt, ret


class DnlCloudGCloudCfgResource(DnlResource):
    model_class = model.DnlCloudGcloudCfg
    scheme_class = DnlCloudGcloudCfgScheme
    scheme_class_get = DnlCloudGcloudCfgSchemeGet
    scheme_class_modify = DnlCloudGcloudCfgSchemeModify
    entity = 'DnlCloudGcloudCfg'
    id_field = 'uuid'
    security = (DEFAULT_SECURITY)
    path_parameters = ()
    restrict = ()


class DnlCloudGCloudCfgList(DnlList):
    scheme_class = DnlCloudGcloudCfgSchemeList
    model_class = model.DnlCloudGcloudCfg
    entity_plural = 'DnlCloudGcloudCfg'
    path_parameters = ()
    security = (DEFAULT_SECURITY)
    restrict = ()


class DnlCloudFtpCfgResource(DnlResource):
    model_class = model.DnlCloudFtpCfg
    scheme_class = DnlCloudFtpCfgScheme
    scheme_class_get = DnlCloudFtpCfgSchemeGet
    scheme_class_modify = DnlCloudFtpCfgSchemeModify
    entity = 'DnlCloudFtpCfg'
    id_field = 'uuid'
    security = (DEFAULT_SECURITY)
    path_parameters = ()
    restrict = ()


class DnlCloudFtpCfgList(DnlList):
    scheme_class = DnlCloudFtpCfgSchemeList
    model_class = model.DnlCloudFtpCfg
    entity_plural = 'DnlCloudFtpCfg'
    path_parameters = ()
    security = (DEFAULT_SECURITY)
    restrict = ()

# endregion ---DnlCloudStorage---
# region +++DnlCloudStatus+++
class DnlCloudStatusCreate(DnlCreate):
    scheme_class = DnlCloudStatusScheme
    model_class = model.DnlCloudStatus
    entity = 'DnlCloudStatus'
    path_parameters = ()
    security = (DEFAULT_SECURITY)
    restrict = ()

    def before_create(self, obj, **kwargs):
        user = self.get_user(self.req)
        # obj.created_by=user.name
        # obj.created_on=datetime.now(UTC)
        return obj


class DnlCloudStatusResource(DnlResource):
    model_class = model.DnlCloudStatus
    scheme_class = DnlCloudStatusScheme
    scheme_class_get = DnlCloudStatusSchemeGet
    scheme_class_modify = DnlCloudStatusSchemeModify
    entity = 'DnlCloudStatus'
    id_field = 'server_name'
    security = (DEFAULT_SECURITY)
    path_parameters = ()
    restrict = ()


class DnlCloudStatusList(DnlList):
    scheme_class = DnlCloudStatusSchemeGet
    model_class = model.DnlCloudStatus
    entity_plural = 'DnlCloudStatuss'
    path_parameters = ()
    security = (DEFAULT_SECURITY)
    restrict = ()

    def modify_query_from_filtering_for_list(self, filtering, **kwargs):
        filt, ret = super().modify_query_from_filtering_for_list(filtering, **kwargs)
        user = self.get_user(self.req)
        if not user.is_admin:
            cls = self.model_class
            # ret = ret.filter(cls.pool_id != 0)#TODO:filter for user
        return filt, ret


# endregion ---DnlCloudStatus---

# region DB manager

class CloudGet(CustomGetAction):
    description = "Custom get"
    path_parameters = ()
    allow_methods = ["get"]
    cloud_method = '/{5}/switch/list'

    @property
    def cloud_base(self):
        return db_man_base()

    @property
    def cloud_url(self):
        if '{' in self.cloud_method:
            return '{}{}'.format(self.cloud_base, self.cloud_method.format(*self.req.path.split('/')))
        return '{}{}'.format(self.cloud_base, self.cloud_method)

    def on_get(self, req, resp, **kwargs):
        self.init_req(req)
        if not check_context(self, req, resp, **kwargs):
            return False
        self.req_uuid = None
        if 'req_uuid' in kwargs:
            self.req_uuid = kwargs['req_uuid']
        user = self.get_user(self.req)
        # params = req.data
        params = dict(parse_qsl(req.query_string))
        # cloud_base = 'http://localhost:33500'
        # url = '{}/switch/list'.format(cloud_base)
        try:
            url = self.cloud_url + '?' + req.query_string if req.query_string else self.cloud_url
            log.debug('CLOUD GET url: {} params:{}'.format(url, params))
            ret = requests.get(url, verify=False)
            log.debug('CLOUD GET status:{} response:{}'.format(ret.status_code, resp.body))
            # if ret.status_code in (200, 201, 204):
            #     resp.content_type = 'application/json'
            #     resp.body = ret.content
            #     resp.status = falcon.HTTP_200
            # Check if ret.content is tar.gz
            try:
                gzip_buf = io.BytesIO(ret.content)
                with gzip.GzipFile(fileobj=gzip_buf) as gz:
                    decompressed_data = gz.read()
                try:
                    with tarfile.open(fileobj=io.BytesIO(decompressed_data), mode='r:*') as tar:
                        log.debug("File is a tar archive")
                        resp.content_type = 'application/x-tgz'
                except tarfile.ReadError:
                    log.debug("File is gzip compressed")
                    resp.content_type = 'application/gzip'
                # with gzip.GzipFile(fileobj=io.BytesIO(ret.content)) as f:
                #     f.read(1)  # Attempt to read to confirm it's a valid gzip file
                # resp.content_type = 'application/gzip'
            except (OSError, EOFError):
                # If not a valid gzip file, keep the original content type
                if 'Content-Type' in ret.headers:
                    resp.content_type = ret.headers['Content-type']
            if 'Content-Disposition' in ret.headers:
                resp.set_header('Content-Disposition', ret.headers['Content-Disposition'])
            if 'Content-Length' in ret.headers:
                resp.set_header('Content-Length', ret.headers['Content-Length'])
            if 'Date' in ret.headers:
                resp.set_header('Date', ret.headers['Date'])
            resp.data = ret.content
            resp.status = str(ret.status_code)
        except Exception as e:
            self.set_response(resp, OperationErrorResponse(e))
            return None
        pass


class CloudSwitchListGet(CloudGet):
    description = "Switch list"
    cloud_method = '/{5}/switch/list'


class CloudStatusGet(CloudGet):
    description = "Status"
    cloud_method = '/{5}/status'


class CloudPost(CustomPostAction):
    description = "Custom post"

    @property
    def cloud_base(self):
        return db_man_base()

    @property
    def cloud_url(self):
        cloud_bases = [self.cloud_base] if isinstance(self.cloud_base, str) else self.cloud_base
        if '{' in self.cloud_method:
            base = lambda base: '{}{}'.format(base, self.cloud_method.format_map(self.req.patch_parameters))
        base = lambda base: '{}{}'.format(base, self.cloud_method)
        return [base(cloud_base) for cloud_base in cloud_bases]

    def on_post(self, req, resp, **kwargs):
        self.init_req(req)
        if not check_context(self, req, resp, **kwargs):
            return False
        user = self.get_user(self.req)
        log.debug('user:{}'.format(user))
        params = req.data
        try:
            log.debug('CLOUD POST url: {} params:{}'.format(self.cloud_url, params))
            responses = []
            for url in self.cloud_url:
                try:
                    sleep(5)
                    log.debug('CLOUD POST url: {} params:{}'.format(url, params))
                    ret = requests.post(url, data=params, verify=False)
                    log.debug('CLOUD POST status:{} response:{}'.format(ret.status_code, ret.content))
                    responses.append(ret.json())
                except Exception as e:
                    log.debug('CLOUD POST url: {} params:{} exception: {}'.format(url, params, e))
            # if ret:
            resp.content_type = 'application/json'
            resp.body = json.dumps({'responses': responses})
            resp.status = falcon.HTTP_200
        except Exception as e:
            self.set_response(resp, OperationErrorResponse(e))
            return None
        pass


class CloudPost(CloudPost):
    description = "Reload configuration"
    cloud_method = '/{5}/reload'

    # path_parameters = ({"name": "request_id", "type": "string", "description": "request id"},)
    @property
    def cloud_base(self):
        return db_man_base()


class CloudRestart(CloudPost):
    description = "Restart daemons"
    cloud_method = '/{5}/restart'


class CloudShutdown(CloudPost):
    description = "Shutdown daemons"
    cloud_method = '/{5}/shutdown'


class CloudReindex(CloudPost):
    description = "Reindex db"
    cloud_method = '/{5}/reindex'


class CloudRunBackup(CloudPost):
    description = "RunBackup"
    cloud_method = '/{5}/run-backup'


class CloudSwitchAdd(CloudPost):
    body_parameters = ('Switch params', CloudSwitchScheme,)
    description = "Add switch"
    cloud_method = '/{5}/switch/add'


class CloudSwitchDelete(CloudPost):
    body_parameters = ('Asynchronous pcap files search', CloudSwitchDeleteScheme,)
    description = "Delete switch"
    cloud_method = '/{5}/switch/delete'


# endregion
# region File downloader
class CloudServerFileAction(CustomPostAction):
    description = "File  action"
    cloud_method = '/{6}/{7}'

    @property
    def cloud_base(self):
        return ds_base()


class CloudServerFileStatus(CloudGet):
    description = "Cdr files list"
    cloud_method = '/{6}/status'

    @property
    def cloud_base(self):
        return fd_base()


class CloudFileListCdr(CloudGet):
    description = "Cdr files list"
    cloud_method = '/list/cdr'
    query_parameters = ({"name": "switch_name", "type": "string", 'required': True, "description": "name of a switch"},)

    @property
    def cloud_base(self):
        return fd_base()
    #     conf = model.DnlCloudDownloaderCfg.filter(model.DnlCloudDownloaderCfg.server_ip == '127.0.0.1').first()
    #     return fd_base() +'{}'.format(conf.server_name)

class CloudFileListPcap(CloudFileListCdr):
    description = "pcap files list"
    cloud_method = '/list/pcap'
    query_parameters = ({"name": "switch_name", "type": "string", 'required': True, "description": "name of a switch"},)


class CloudServerFileListCdr(CloudGet):
    description = "Cdr files list"
    cloud_method = '/{6}/list/cdr'
    path_parameters = ({"name": "server_name", "type": "string", "description": "name of an instance"},)

    @property
    def cloud_base(self):
        return fd_base()


class CloudServerFileListPcap(CloudFileListCdr):
    description = "pcap files list"
    cloud_method = '/{6}/list/pcap'
    path_parameters = ({"name": "server_name", "type": "string", "description": "name of an instance"},)


class CloudFileGetCdr(CloudFileListCdr):
    description = "cdr file"
    cloud_method = '/get/cdr'
    query_parameters = ({"name": "switch_name", "type": "string", 'required': True, "description": "name of a switch"},
                        {"name": "timestamp", "type": "integer", 'required': True, "description": "timestamp of file"},
                        )


class CloudFileGetPcap(CloudFileListCdr):
    description = "pcap file"
    cloud_method = '/get/pcap'
    query_parameters = ({"name": "switch_name", "type": "string", 'required': True, "description": "name of a switch"},
                        {"name": "timestamp", "type": "integer", 'required': True, "description": "timestamp of file"},
                        )


# endregion File downloader
# region Database search

# endregion Database search


class CloudSearchCdr(CloudPost):
    body_parameters = ('Asynchronous cdr files search', CloudCdrScheme,)
    description = "cdr"
    cloud_method = '/cdr'

    @property
    def cloud_base(self):
        return ds_base(multiple=True)

    def on_post(self, req, resp, **kwargs):
        try:
            if 'tz' in req.data:
                tz = int(req.data.pop('tz'))
                req.data['start_time'] += tz * 86400
                req.data['end_time'] += tz * 86400
            if 'ingress_carrier_id' in req.data:
                ingress_carrier_id = int(req.data.pop('ingress_carrier_id'))
                resources = model.Resource.filter(and_(model.Resource.client_id == ingress_carrier_id, model.Resource.ingress))
                ingress_trunk_ids = [str(i.resource_id) for i in resources]
                ingress_trunk_id = req.data.pop('ingress_trunk_id', '').split(',')
                if ingress_trunk_id != ['']:
                    ingress_trunk_ids = list(set(ingress_trunk_ids).intersection(set(ingress_trunk_id)))
                req.data['ingress_trunk_id'] = ','.join(ingress_trunk_ids)
            if 'egress_carrier_id' in req.data:
                egress_carrier_id = int(req.data.pop('egress_carrier_id'))
                resources = model.Resource.filter(and_(model.Resource.client_id == egress_carrier_id, model.Resource.egress))
                egress_trunk_ids = [str(i.resource_id) for i in resources]
                egress_trunk_id = req.data.pop('egress_trunk_id', '').split(',')
                if egress_trunk_id != ['']:
                    egress_trunk_ids = list(set(egress_trunk_ids).intersection(set(egress_trunk_id)))
                req.data['egress_trunk_id'] = ','.join(egress_trunk_ids)
            if 'fields' in req.data:
                fields = req.data['fields'].split(',')
                int_fields = []
                for field in fields:
                    field_to_id = {v: str(k-1) for k, v in FIELDS.items()}
                    if field in field_to_id:
                        int_fields += [field_to_id[field]]
                    if field.isdigit():# and 0 <= int(field) <= 127:
                        int_fields += [field]
                        req.data['fields'] = ','.join(int_fields)
            if 'non_zero' in req.data:
                req.data['non_zero'] = 'true' if req.data['non_zero'] in (1, '1', True, 'true') else 'false'
            return super().on_post(req, resp, **kwargs)
        except Exception as e:
            self.set_response(resp, OperationErrorResponse(e))


class CloudSearchPcap(CloudPost):
    body_parameters = ('Asynchronous pcap files search', CloudPcapScheme,)
    description = "pcap"
    cloud_method = '/pcap'

    @property
    def cloud_base(self):
        return ds_base()

    def on_post(self, req, resp, **kwargs):
        self.init_req(req)
        if not check_context(self, req, resp, **kwargs):
            return False
        user = self.get_user(self.req)
        log.debug('user:{}'.format(user))
        params = req.data
        method = type(self).cloud_method
        try:
            get_url = lambda conf: '{}://{}:{}'.format('https' if conf.server_use_ssl else 'http', conf.server_ip, conf.server_port)
            # log.debug('CLOUD POST url: {} params:{}'.format(self.cloud_url, params))
            responses = []
            configs = model.DnlCloudSearchCfg.filter().all()
            for conf in configs:
                sleep(5)
                server = model.VoipGateway.filter(model.VoipGateway.lan_ip == conf.server_ip).first()
                switch_name = server.name if server else None
                if switch_name:
                    params.update({'switch_name': switch_name})
                for field in ('call_id', 'ani', 'dnis'):
                    if field in params and isinstance(params[field], str):
                        params[field] = params[field].split(',')
                url = get_url(conf) + method
                log.debug('CLOUD POST url: {} params:{}'.format(url, params))
                ret = requests.post(url, data=params, verify=False)
                log.debug('CLOUD POST status:{} response:{}'.format(ret.status_code, ret.content))
                responses.append(ret.json())
            # if ret:
            resp.content_type = 'application/json'
            resp.body = json.dumps({'responses': responses})
            resp.status = falcon.HTTP_200
        except Exception as e:
            self.set_response(resp, OperationErrorResponse(e))
            return None
        pass


class CloudSearchGet(CloudGet):
    description = "get search query results"
    cloud_method = ''
    path_parameters = ({"name": "req_uuid", "type": "string", "description": "request id"},)

    def on_post(self, req, resp, **kwargs):
        if 'tz' in req.data:
            tz = int(req.data.pop('tz'))
            req.data['start_time'] += tz * 86400
            req.data['end_time'] += tz * 86400
        if 'ingress_carrier_id' in req.data:
            ingress_carrier_id = int(req.data.pop('ingress_carrier_id'))
            resources = model.Resource.filter(and_(model.Resource.client_id == ingress_carrier_id, model.Resource.ingress))
            ingress_trunk_ids = [str(i.resource_id) for i in resources]
            ingress_trunk_id = req.data.pop('ingress_trunk_id', '').split(',')
            if ingress_trunk_id:
                ingress_trunk_ids = list(set(ingress_trunk_ids).intersection(set(ingress_trunk_id)))
            req.data['ingress_trunk_id'] = ','.join(ingress_trunk_ids)
        if 'egress_carrier_id' in req.data:
            egress_carrier_id = int(req.data.pop('egress_carrier_id'))
            resources = model.Resource.filter(and_(model.Resource.client_id == egress_carrier_id, model.Resource.egress))
            egress_trunk_ids = [str(i.resource_id) for i in resources]
            egress_trunk_id = req.data.pop('egress_trunk_id', '').split(',')
            if egress_trunk_id:
                egress_trunk_ids = list(set(egress_trunk_ids).intersection(set(egress_trunk_id)))
            req.data['egress_trunk_id'] = ','.join(egress_trunk_ids)
        return super().on_post(req, resp, **kwargs)

    @property
    def cloud_base(self):
        return ds_base(multiple=True)

    @property
    def cloud_url(self):
        return ['{}/{}{}'.format(cloud_base, self.req_uuid, self.cloud_method) for cloud_base in self.cloud_base]

    def on_get(self, req, resp, **kwargs):
        self.init_req(req)
        if not check_context(self, req, resp, **kwargs):
            return False
        self.req_uuid = None
        if 'req_uuid' in kwargs:
            self.req_uuid = kwargs['req_uuid']
        user = self.get_user(self.req)
        # params = req.data
        params = dict(parse_qsl(req.query_string))
        # cloud_base = 'http://localhost:33500'
        # url = '{}/switch/list'.format(cloud_base)
        try:
            cloud_urls = self.cloud_url
            response = {}
            for cloud_url in cloud_urls:
                url = cloud_url + '?' + req.query_string if req.query_string else cloud_url
                log.debug('CLOUD GET url: {} params:{}'.format(url, params))
                ret = requests.get(url, verify=False)
                log.debug('CLOUD GET status:{} response:{}'.format(ret.status_code, resp.body))
                if ret.status_code in (200, 201, 204):
                    response = ret.content
                    break
            resp.content_type = ret.headers.get('Content-Type')
            if not response:
                response = { "code": 404, "error": "Page Not Found" }
                resp.content_type = 'application/json'
            if resp.content_type == 'application/json':
                resp.body = json.dumps(response)
            else:
                resp.body = response
            resp.status = str(ret.status_code)
        except Exception as e:
            self.set_response(resp, OperationErrorResponse(e))
            return None


class CloudSearchGetList(CloudGet):
    description = "get search all cloud query requests"
    cloud_method = '/list'
    query_parameters = ({"name": "start_time", "type": "integer", "description": "time filter"},
                        {"name": "end_time", "type": "integer", "description": "time filter"},
                        )

    @property
    def cloud_base(self):
        return ds_base()


class CloudSearchGetListCdr(CloudGet):
    description = "get search CDR query requests"
    cloud_method = '/list/cdr'
    query_parameters = ({"name": "start_time", "type": "integer", "description": "time filter"},
                        {"name": "end_time", "type": "integer", "description": "time filter"},
                        )

    @property
    def cloud_base(self):
        return ds_base()

    def on_get(self, req, resp, **kwargs):
        super().on_get(req, resp, **kwargs)
        try:
            data = json.loads(resp.data.decode())
            data['data'] = list(map(self.map_response, data.get('data', [])))
            resp.data = json.dumps(data).encode()
        except Exception as e:
            log.debug('cloud search get list cdr error {}'.format(str(e)))

    @staticmethod
    def map_response(data):
        if data['ingress_trunk_id']:
            ingress_trunk_ids = data.pop('ingress_trunk_id').split(',')
            ingress_trunks = model.Resource.session().query(model.Resource.alias, model.Resource.client_name, model.Resource.trunk_type2
                            ).filter(model.Resource.resource_id.in_(ingress_trunk_ids)).all()
            ingress_trunk_names = [trunk.alias for trunk in ingress_trunks]
            ingress_client_names = [trunk.client_name for trunk in ingress_trunks]
            ingress_trunk_types = [trunk.trunk_type2 for trunk in ingress_trunks]
            data['ingress_trunk_name'] = ingress_trunk_names
            data['ingress_client_name'] = ingress_client_names
            data['ingress_trunk_type'] = ingress_trunk_types
        if data['egress_trunk_id']:
            egress_trunk_ids = data.pop('egress_trunk_id').split(',')
            egress_trunks = model.Resource.session().query(model.Resource.alias, model.Resource.client_name).filter(model.Resource.resource_id.in_(egress_trunk_ids)).all()
            egress_trunk_names = [trunk.alias for trunk in egress_trunks]
            egress_client_names = [trunk.client_name for trunk in egress_trunks]
            data['egress_trunk_name'] = egress_trunk_names
            data['egress_client_name'] = egress_client_names
        return data


class CloudSearchGetListPcap(CloudGet):
    description = "get search PCAP query requests"
    cloud_method = '/list/pcap'
    query_parameters = ({"name": "start_time", "type": "integer", "description": "time filter"},
                        {"name": "end_time", "type": "integer", "description": "time filter"},
                        )

    @property
    def cloud_base(self):
        return ds_base(multiple=True)

    @property
    def cloud_url(self):
        return ['{}{}'.format(cloud_base, self.cloud_method) for cloud_base in self.cloud_base]

    def on_get(self, req, resp, **kwargs):
        self.init_req(req)
        if not check_context(self, req, resp, **kwargs):
            return False
        self.req_uuid = None
        if 'req_uuid' in kwargs:
            self.req_uuid = kwargs['req_uuid']
        user = self.get_user(self.req)
        # params = req.data
        params = dict(parse_qsl(req.query_string))
        # cloud_base = 'http://localhost:33500'
        # url = '{}/switch/list'.format(cloud_base)
        try:
            cloud_urls = self.cloud_url
            response = {}
            for cloud_url in cloud_urls:
                url = cloud_url + '?' + req.query_string if req.query_string else cloud_url
                log.debug('CLOUD GET url: {} params:{}'.format(url, params))
                ret = requests.get(url, verify=False)
                log.debug('CLOUD GET status:{} response:{}'.format(ret.status_code, resp.body))
                if ret.status_code in (200, 201, 204):
                    if not response:
                        response = ret.json()
                    else:
                        response['data'] += ret.json()['data']

            resp.content_type = 'application/json'
            resp.body = json.dumps(response)
            resp.status = str(ret.status_code)
        except Exception as e:
            self.set_response(resp, OperationErrorResponse(e))
            return None


class CloudSearchDownload(CloudSearchGet):
    description = "get search query results"
    cloud_method = '/download'

    def on_get(self, req, resp, **kwargs):
        super().on_get(req, resp, **kwargs)
        try:
            if resp.content_type != 'application/zip':
                resp.data = resp.data.replace(b'?', b',')
        except Exception as e:
            log.debug('cloud search download error {}'.format(str(e)))


class CloudReqDelete(CustomDeleteAction):
    description = "Switch list"
    path_parameters = ({"name": "req_uuid", "type": "string", "description": "request id"},)
    cloud_method = ''

    @property
    def cloud_base(self):
        return ds_base()

    @property
    def cloud_url(self):
        return '{}/{}{}'.format(self.cloud_base, self.req_uuid, self.cloud_method)

    def on_delete(self, req, resp, **kwargs):
        self.init_req(req)
        if not check_context(self, req, resp, **kwargs):
            return False
        self.req_uuid = None
        if 'req_uuid' in kwargs:
            self.req_uuid = kwargs['req_uuid']
        user = self.get_user(self.req)
        # params = req.data
        params = dict(parse_qsl(req.query_string))
        try:
            url = self.cloud_url + '?' + req.query_string if req.query_string else self.cloud_url
            ret = requests.delete(url)
            resp.body = ret.content
            resp.status = str(ret.status_code)
        except Exception as e:
            self.set_response(resp, OperationErrorResponse(e))
            return None
        pass


class CloudServerSearchAction(CustomPostAction):
    description = "File  action"
    cloud_method = '/{6}/{7}'

    @property
    def cloud_base(self):
        return ds_base()


class CloudServerSearchStatus(CloudGet):
    description = "Cdr files list"
    cloud_method = '/{6}/status'

    @property
    def cloud_base(self):
        return ds_base()
