import json
from datetime import datetime, timedelta
from pytz import UTC
import jwt
import requests
from falcon_rest.logger import log

from api_dnl.settings import JWT_SIGNATURE
from .did_api import DidAPI, TIMEOUT, DidApiException, fmt_content
from bandwidth.bandwidth_client import BandwidthClient


class DidAPIBandwidth(DidAPI):

    def __init__(self, user, password=None, vendor=None):
        self.user = user
        self.password = password
        self.vendor = vendor

    def authenticate(self, **kwargs):
        self.client = BandwidthClient(voice_basic_auth_user_name=self.user,
            voice_basic_auth_password=self.password,
            messaging_basic_auth_user_name=self.user,
            messaging_basic_auth_password=self.password)


    @property
    def trunk_group(self):
        if self.vendor and hasattr(self.vendor, 'trunk_group_name'):
            return self.vendor.trunk_group_name
        return ''

    @property
    def routing_label(self):
        if self.vendor and hasattr(self.vendor, 'toll_free_routing_label'):
            return self.vendor.toll_free_routing_label
        return ''

    @property
    def sms_enabled(self):
        if self.vendor and hasattr(self.vendor, 'sms_enabled'):
            return self.vendor.sms_enabled
        return False

    def search_local(self, state=None, npa=None, lata=None, pattern=None, count=100, client_id=None, nxx=None):
        params = {}
        if state:
            params['province'] = state
        if npa:
            params['tnMask'] = str(npa) + 'xxxxxxx'
        else:
            if pattern:
                params['tnMask'] = pattern.replace('*', 'x')
        if not 'tnMask' in params:
            params['tnMask'] = 'xxxxxxxxxx'
        if lata:
            params['lata'] = lata
        params['pageSort'] = dict(page=1, size=count)
        ret = self.post('tnInventory', **params)
        log.debug('search local:{}'.format(ret))
        if 'tnResult' in ret:
            return [{'number': i['telephoneNumber'], 'npa': i['telephoneNumber'][0:3], 'lata': i['lata'],
                     'state': i['province'], 'msg': True} for i in ret['tnResult']]
        return []

    def coverage_local(self, state=None, npa=None, lata=None):
        def _m(d):
            ret = {}
            for k, v in d.items():
                if k == 'locState':
                    k = 'state'
                ret[k] = v
            return ret

        params = {}

        def filt(d):
            f = True
            if state:
                f = f and d['state'] == state
            if npa:
                f = f and d['npa'] == npa
            if lata:
                f = f and d['lata'] == lata
            return f

        params['pageSort'] = dict(page=1, size=100)
        ret = self.post('tnInventoryCoverage', **params)
        log.debug('search coverage:{}'.format(ret))
        if 'tnInventoryCoverageList' in ret:
            # return ret['tnInventoryCoverageList']
            ret = [_m(i) for i in ret['tnInventoryCoverageList']]
            return [i for i in ret if filt(i)]
        return []

    def search_order_local(self, state=None, npa=None, lata=None, count=100, client_id=None):
        params = {}
        if state:
            params['province'] = state
        if npa:
            params['tnMask'] = npa + 'xxxxxxx'
        if lata:
            params['lata'] = lata
        if not 'tnMask' in params:
            params['tnMask'] = 'xxxxxxxxxx'
        params['pageSort'] = dict(page=1, size=count)
        ret = self.post('tnInventory', **params)
        log.debug('order local found :{}'.format(ret))
        tnItem = []
        ret1 = []
        if 'tnResult' in ret:
            ret1 = [{'number': i['telephoneNumber'], 'npa': i['telephoneNumber'][0:3], 'lata': i['lata'],
                     'state': i['province']} for i in ret['tnResult']]
            if self.sms_enabled:
                tnItem = [{'tn': i['number'], 'tnNote': str(client_id), 'trunkGroup': self.trunk_group,
                           'tnFeature': {
                               'messaging': {
                                   'messageClass': MSG_CLASS,
                                   'messageType': 'SMSMMS_ALT'
                               }
                           }} for i in ret['tnResult']]
            else:
                tnItem = [{'tn': i['number'], 'tnNote': str(client_id), 'trunkGroup': self.trunk_group}
                          for i in ret['tnResult']]
        tnOrder = {'tnList': {'tnItem': tnItem}}
        res = self.post('tnOrder', tnOrder=tnOrder)
        log.debug('order local result :{}'.format(res))

        st = {}
        try:
            if res['status'] != 'Success':
                li = res['status'].split('[')[1].split(']')[0].split(',')
                for l in li:
                    l = l.strip()
                    st[l.split(' ')[0]] = 'Invalid tn'
        except:
            pass
        log.debug('order local parsed status :{}'.format(st))
        ret = []
        for item in ret1:
            num = item['number']
            if res['status'] != 'Success':
                ret.append(dict(number=num, status='error', error=st[num] if num in st else res['status']))
            else:
                ret.append(dict(number=num, status='success'))

        return ret

    def order_local(self, items, client_id=None):
        params = {}
        tnItem = []
        ret1 = []

        if items:
            if self.sms_enabled:
                tnItem = [{'tn': i['number'], 'tnNote': str(client_id), 'trunkGroup': self.trunk_group,
                           'tnFeature': {
                               'messaging': {
                                   'messageClass': MSG_CLASS,
                                   'messageType': 'SMSMMS_ALT'
                               }
                           }} for i in items]
            else:
                tnItem = [{'tn': i['number'], 'tnNote': str(client_id), 'trunkGroup': self.trunk_group} for i in items]
        tnOrder = {'tnList': {'tnItem': tnItem}}
        res = self.post('tnOrder', tnOrder=tnOrder)
        log.debug('order local result :{}'.format(res))
        # parse status
        st = {}
        try:
            if res['status'] != 'Success':
                li = res['status'].split('[')[1].split(']')[0].split(',')
                for l in li:
                    l = l.strip()
                    st[l.split(' ')[0]] = 'Invalid tn'
        except:
            pass
        log.debug('order local parsed status :{}'.format(st))
        ret = []
        for item in items:
            num = item['number']
            if res['status'] != 'Success':
                ret.append(dict(number=num, status='error', error=st[num] if num in st else res['status']))
            else:
                ret.append(dict(number=num, status='success'))
        return ret

    def assigned_local(self):
        params = {}
        ret = self.post('tnAssignedList', **params)
        return ret['tnBasicList']['tnBasicItem']

    def disconnect_local(self, items):
        params = {'tnList': {'tnItem': [{'tn': i['number']} for i in items]}}
        res = self.post('tnDisconnect', **params)
        # parse status
        st = {}
        if res['status'] != 'Success':
            li = res['status'].split('[')[1].split(']')[0].split(',')
            for l in li:
                l = l.strip()
                st[l.split()[0]] = ' '.join(l.split(' ')[1:])
        ret = []
        for item in items:
            num = item['number']
            if num in res['status']:
                ret.append(dict(number=num, status='error', error=st[num] if num in st else res['status']))
            else:
                ret.append(dict(number=num, status='success'))
        return ret

    def send_sms(self, sender, receiver, msg):
        if len(sender) == 10:
            sender = '1' + sender
        if len(receiver) == 10:
            receiver = '1' + receiver
        params = {'from': sender, 'to': [receiver], 'text': msg}
        res = self.post('publishMessages', **params)
        return res

    def set_sms_receive_hook(self, number, url, client_id):

        token = jwt.encode(dict(client_id=client_id), JWT_SIGNATURE).decode('utf-8')
        params = {'authorizations': [
            {'tn': number, 'headerName': 'X-Auth-Token', 'headerValue': token, 'inboundAuth': True, 'webhookUrl': url}
        ]}
        res = self.post('configureAuthorization', **params)
        return res

    def search_toll_free(self, pattern=None, count=100, start=0):

        # toll_free: {'status': 'Success', 'statusCode': '200', 'tfList': {
        #     'tfItem': [{'tn': 8882453977, 'cic': '0913,0555,5102,2121', 'respOrgPrefix': 'LQX01'},
        #                {'tn': 8882454449, 'cic': '0913,0555,5102,2121', 'respOrgPrefix': 'LQX01'},
        #                {'tn': 8882518732, 'cic': '0913,0555,5102,2121', 'respOrgPrefix': 'LQX01'},
        #                {'tn': 8882540673, 'cic': '0913,0555,5102,2121', 'respOrgPrefix': 'LQX01'},
        #                {'tn': 8882543575, 'cic': '0913,0555,5102,2121', 'respOrgPrefix': 'LQX01'},
        #                {'tn': 8882964833, 'cic': '0913,0555,5102,2121', 'respOrgPrefix': 'LQX01'},
        #                {'tn': 8883035366, 'cic': '0913,0555,5102,2121', 'respOrgPrefix': 'LQX01'},
        #                {'tn': 8883064835, 'cic': '0913,0555,5102,2121', 'respOrgPrefix': 'LQX01'}, {'tn':

        params = {}
        if pattern:
            params['tnMask'] = pattern.replace('*', 'x')
        else:
            params['tnMask'] = 'x' * 10
        # params['pageSort'] = dict(page=1, size=count)
        params['quantity'] = count
        ret = self.post('tfInventory', **params)
        log.debug('search toll_free:{}'.format(ret))
        if 'tfList' in ret and 'tfItem' in ret['tfList']:
            return [{'number': str(i['tn']), 'cic': i['cic'], 'org_prefix': i['respOrgPrefix']} for i in
                    ret['tfList']['tfItem']]
        return []

    def search_order_toll_free(self, pattern=None, routing_label=None, count=100):
        params = {}

        params['tnMask'] = pattern.replace('*', 'x')
        params['pageSort'] = dict(page=1, size=int(count))
        ret = self.post('tfInventory', **params)
        log.debug('order toll_free found :{}'.format(ret))
        tnItem = []
        ret1 = []
        if 'tnResult' in ret:
            ret1 = [{'number': i['telephoneNumber'], 'npa': i['telephoneNumber'][0:3], 'lata': i['lata'],
                     'state': i['province']} for i in ret['tnResult']]
            if self.sms_enabled:
                tnItem = [{'tn': i['number'], 'tnNote': str(routing_label), 'trunkGroup': self.trunk_group,
                           'tnFeature': {
                               'messaging': {
                                   'messageClass': MSG_CLASS,
                                   'messageType': 'SMSMMS_ALT'
                               }
                           }} for i in ret['tnResult']]
            else:
                tnItem = [{'tn': i['number'], 'resporgId': i[''],
                           'routingLabel': str(routing_label or self.vendor.toll_free_routing_label)}
                          for i in ret['tnResult']]
        tnOrder = {'tnList': {'tnItem': tnItem}}
        res = self.post('tnOrder', tnOrder=tnOrder)
        log.debug('order toll_free result :{}'.format(res))

        st = {}
        try:
            if res['status'] != 'Success':
                li = res['status'].split('[')[1].split(']')[0].split(',')
                for l in li:
                    l = l.strip()
                    st[l.split(' ')[0]] = 'Invalid tn'
        except:
            pass
        log.debug('order local parsed status :{}'.format(st))
        ret = []
        for item in ret1:
            num = item['number']
            if res['status'] != 'Success':
                ret.append(dict(number=num, status='error', error=st[num] if num in st else res['status']))
            else:
                ret.append(dict(number=num, status='success'))

        return ret

    def order_toll_free(self, items, routing_label=None, client_id=None):
        params = {}
        ret1 = []
        customer_order = 'DNLOrder{}'.format(datetime.now(UTC).timestamp())
        if items:
            tn_item = [{'tn': i['number'], 'routingLabel': str(routing_label or self.vendor.toll_free_routing_label),
                        } for i in items]
            tn_item2 = [
                {'tn': i['number'], "tnFeature": {"messaging": {"messageClass": "A2P8XX", "messageType": "SMSMMS_ALT"}}
                 } for i in items]

        tf_order = {'tnList': {'tnItem': tn_item},
                    'customerOrderReference': customer_order}
        tf_feature_order = {'tnList': {'tnItem': tn_item2},
                            'customerOrderReference': customer_order}
        res_order = self.post('tfOrder', tfOrder=tf_order, productAbbr='8XXAVAIL')
        log.debug('order toll_free result :{}'.format(res_order))
        if res_order['status'] == 'Success' and self.vendor.sms_enabled:
            res = self.post('tfFeatureOrder', tfFeatureOrder=tf_feature_order)
            log.debug('order features toll_free result :{}'.format(res))
        else:
            res = res_order
        # parse status
        st = {}
        try:
            if res['status'] != 'Success':
                li = res['status'].split('[')[1].split(']')[0].split(',')
                for l in li:
                    l = l.strip()
                    st[l.split(' ')[0]] = l#res['status'].split('[')[0]
        except:
            pass
        log.debug('order toll_free parsed status :{}'.format(st))
        ret = []
        for item in items:
            num = item['number']
            if res['status'] != 'Success':
                ret.append(dict(number=num, status='error', error=st[num] if num in st else res['status']))
            else:
                ret.append(dict(number=num, status='success'))
        return ret

    def assigned_toll_free(self):
        params = {}
        ret = self.post('tfAssignedList', **params)
        return ret['tnBasicList']['tnBasicItem']

    def disconnect_toll_free(self, items):
        params = {'tnList': {'tnItem': [{'tn': i['number']} for i in items]}}
        res = self.post('tfDisconnect', **params)
        log.debug('disconnect toll_free result :{}'.format(res))
        # parse status
        st = {}
        if res['status'] != 'Success':
            li = res['status'].split('[')[1].split(']')[0].split(',')
            for l in li:
                l = l.strip()
                st[l.split()[0]] = ' '.join(l.split(' ')[1:]) or res['status'].split(':')[0]
        ret = []
        for item in items:
            num = item['number']
            if num in res['status']:
                ret.append(dict(number=num, status='error', error=st[num] if num in st else res['status']))
            else:
                ret.append(dict(number=num, status='success'))
        return ret

    def pending(self):
        items = []
        tnSearchList = {
            "tnSearchItem": [{"tnMask": "xxxxxxxxxx", "tnStatus": "PNDNG", "trunkGroupName": self.trunk_group}]}
        res = self.post('tnDetail', tnSearchList=tnSearchList)
        if 'status' in res and res['status'] == 'Success':
            if 'tnList' in res and 'tnItem' in res['tnList']:
                items = items + [{'number': i['tn'], 'status': i['tnStatusRef']} for i in res['tnList']['tnItem']]
        tfSearchList = {
            "tfSearchItem": [
                {"tfMask": "xxxxxxxxxx", "typeTfStatusReference": "PNDNG"}]}
        if self.routing_label:
             tfSearchList["tfSearchItem"][0]["routingLabel"] = self.routing_label

        res = self.post('tfDetail', tfSearchList=tfSearchList)
        if 'status' in res and res['status'] == 'Success':
            if 'tfList' in res and 'tfItem' in res['tfList']:
                items = items + [{'number': i['tn'], 'status': i['tnStatusRef']} for i in res['tfList']['tfItem']]
        return items
