import json
import json
import sys
from urllib.parse import parse_qsl, urlencode
from datetime import date, datetime, timedelta
from pytz import UTC
from time import mktime
from dateutil.parser import parse as parse_datetime
from sqlalchemy.orm import aliased, make_transient, foreign
from sqlalchemy import or_, and_, select, inspect, func, cast, text
from sqlalchemy import Integer
from api_dnl import model
from falcon_rest.logger import log
from falcon_rest.responses import responses
from falcon_rest.responses import errors
from falcon_rest import swagger, schemes
from api_dnl.scheme import DATE_REGEXP, ResourceIpPortScheme, ResourceIpClientGetScheme, IngressTrunkScheme
from api_dnl.utils.did_api import DidApiException
from api_dnl.utils.did_api import DidAPIMuliVendor, VendorManual, unprefix, unprefix_2, set_prefix
from api_dnl.resources import DnlResource, DnlList, DnlCreate, CustomAction, CustomGetAction, CustomPostAction, \
    CustomPatchAction, CustomDeleteAction, ValidationError, client_context, check_context, SuccessResponseJustOk, \
    generate_uuid_str

from api_dnl.schemes.client import ClientPortalScheme, ClientPortalPaymentGatewayHistoryGetScheme, \
    ClientPortalProductRoutRateTableScheme, ClientPortalProductRoutRateTableGetScheme, \
    ClientPortalProductClientsRefGetScheme, ClientPortalIngressTrunkScheme, ClientPortalEgressTrunkScheme, \
    ClientPortalIngressTrunkModifyScheme, ClientPortalEgressTrunkModifyScheme, \
    ClientPortalRateTableGetScheme, ClientPortalRateGetScheme, ClientPortalDidRepositoryGetScheme, \
    ClientPortalFreeDidBillingRelGetScheme, ClientPortalInvoiceGetScheme, \
    ClientPortalPaymentGatewayHistoryScheme, ClientPortalDidVendorSchemeGet, \
    ClientPortalPaymentGetScheme, ClientPortalDidBillingRelDisableScheme, ClientPortalDidBillingRelAssignScheme, \
    ClientPortalEmailLogAlertsScheme, ClientPortalActualBalanceHistoryGetScheme, ClientDefaultIpScheme, \
    ClientDefaultIpGetScheme, ClientPortalGatewayAndPaymentsGetScheme, ClientPortalProductSendToEmailsScheme, \
    DidNumberUploadScheme, DidNumberUploadSchemeGet, DidApiSearchScheme, DidApiSearchTollFreeScheme, \
    DidApiSearchGetScheme, DidApiSearchTollFreeGetScheme, \
    DidApiSearchCoverageScheme, DidApiSearchCoverageGetScheme, DidApiOrderLocalScheme, DidApiOrderTollFreeScheme, DidApiOrderLocalGetScheme, \
    DidApiDisconnectScheme, DidApiSearchOrderLocalScheme, SmsScheme, SmsSchemeGet, SmsSchemeModify, \
    NpaSearchScheme, LataSearchScheme, StateSearchScheme, WebHookScheme, WebHookSchemeGet, WebHookSchemeModify, \
    ApiKeyScheme, ClientApiKeyGetScheme, ClientPortalProductGetScheme, ClientPortalResourceIpPortScheme, \
    ClientPortalDidRepositoryGetScheme
from api_dnl.view import NoResultFound, DEFAULT_SECURITY, RateList, DidBillingRelCreate, IngressTrunkCreate, \
    IngressTrunkList, IngressTrunkResource, EgressTrunkCreate, EgressTrunkResource, EgressTrunkList,\
    TrunkAssignProduct, ResourcePrefixCreate, ResourcePrefixResource, ResourcePrefixList, OperationErrorResponse, \
    DidBillingPlanList, ResourceIpList, ResourceIpResource, DidNumberUploadTaskCreate

from api_dnl import settings


class ClientPortalResource(DnlResource):
    model_class = model.Client
    scheme_class = ClientPortalScheme
    scheme_class_get = ClientPortalScheme
    # scheme_class_modify = CarrierContactsScheme
    entity = 'Carrier'
    id_field = 'client_id'
    has_delete_operation = False
    # has_modify_operation = False
    has_update_by = False
    is_client = True
    unique_field = 'name'
    security = (DEFAULT_SECURITY)
    path_parameters = ()
    restrict = ()

    def get_path_parameters(self):
        return ()


class _ClientPortalPaymentGatewayHistoryList(DnlList):
    scheme_class = ClientPortalPaymentGatewayHistoryGetScheme
    model_class = model.PaymentGatewayHistory
    entity_plural = 'PaymentGatewayHistory'
    path_parameters = ()
    security = (DEFAULT_SECURITY)
    restrict = ()


class ClientPortalProductRoutRateTableCreate(DnlCreate):
    scheme_class = ClientPortalProductRoutRateTableScheme
    model_class = model.ProductRoutRateTable
    entity = 'ClientPortalProduct'
    path_parameters = ()
    security = (DEFAULT_SECURITY)
    restrict = ()
    is_client = True

    def before_create(self, obj, **kwargs):
        ref = model.ProductClientsRef(client_id=kwargs['client_id'])
        obj.clients_ref.append(ref)
        return obj


class ClientPortalProductRoutRateTableResource(DnlResource):
    scheme_class = ClientPortalProductRoutRateTableScheme
    scheme_class_modify = ClientPortalProductRoutRateTableScheme
    scheme_class_get = ClientPortalProductRoutRateTableGetScheme
    model_class = model.ProductRoutRateTable
    entity = 'ClientPortalProduct'
    id_field = 'id'
    path_parameters = ()
    security = (DEFAULT_SECURITY)
    restrict = ()
    is_client = True

    def get_object_data(self, resp, model_class, scheme_class_get, **kwargs):
        ref = model.ProductClientsRef
        q = ref.filter(and_(ref.client_id == kwargs['client_id'], ref.product_id == kwargs['id'])).first()
        if not q:
            return None
        return super(ClientPortalProductRoutRateTableResource, self).get_object_data(resp, model_class,
                                                                                     scheme_class_get, **kwargs)


class ClientPortalProductRoutRateTableList(DnlList):
    scheme_class = ClientPortalProductClientsRefGetScheme
    model_class = model.ProductClientsRefUsed
    entity_plural = 'Available Products'
    path_parameters = ()
    security = (DEFAULT_SECURITY)
    restrict = ()
    is_client = True

    def on_filter_ingress_trunk_id(self, query, value, kwargs):
        import re
        res = model.Resource
        q = res.filter(and_(res.resource_id == int(value), res.ingress == True))
        if not q:
            raise ValidationError('invalid ingress trunk {}'.format(value))
        cls = self.model_class
        pfx = model.ResourcePrefix
        query = query.filter(cls.product_id.in_(select([pfx.product_id]).where(pfx.resource_id == int(value))))
        return query

    def modify_query_from_filtering_for_list(self, filtering, **kwargs):
        ret, q = super(ClientPortalProductRoutRateTableList, self).modify_query_from_filtering_for_list(filtering,
                                                                                                        **kwargs)
        client_id = int(kwargs['client_id'])
        ref = model.ProductClientsRefUsed
        ref.update_used(client_id)
        ref.session().commit()
        q = q.filter(ref.client_id == client_id)
        log.debug(model.query_to_sting(q))
        return ret, q


class ClientPortalProductList(DnlList):
    scheme_class = ClientPortalProductGetScheme
    model_class = model.ProductClientsRefUsed
    entity_plural = 'Products'
    path_parameters = ()
    security = (DEFAULT_SECURITY)
    restrict = ()
    is_client = True

    def modify_query_from_filtering_for_list(self, filtering, **kwargs):
        ret, q = super().modify_query_from_filtering_for_list(filtering, **kwargs)
        client_id = int(kwargs['client_id'])
        cls = model.ProductRoutRateTable
        ref = model.ProductClientsRef
        # products = cls.filter(
        #     or_(cls.type == 'public', cls.id.in_(select([ref.product_id]).where(ref.client_id == client_id))))
        # products_ids = [i.id for i in products]
        self.model_class.update_used(client_id)
        q = q.filter(self.model_class.client_id == client_id)
        return ret, q


class ClientPortalPaymentGatewayHistoryList(DnlList):
    scheme_class = ClientPortalPaymentGatewayHistoryGetScheme
    model_class = model.PaymentGatewayHistory
    entity_plural = 'PaymentGatewayHistory'
    path_parameters = ()
    security = (DEFAULT_SECURITY)
    restrict = ()
    is_client = True

    def modify_query_from_filtering_for_list(self, filtering, **kwargs):
        ret, q = super(ClientPortalPaymentGatewayHistoryList, self).modify_query_from_filtering_for_list(filtering,
                                                                                                         **kwargs)
        cls = model.PaymentGatewayHistory
        q = q.filter(cls.client_id == int(kwargs['client_id']))
        log.debug(model.query_to_sting(q))
        return ret, q


class ClientPortalRateTableList(DnlList):
    scheme_class = ClientPortalRateTableGetScheme
    model_class = model.RateTable
    entity_plural = 'RateTable'
    path_parameters = ()
    security = (DEFAULT_SECURITY)
    restrict = ()
    is_client = True

    def on_filter_trunk_id(self, query, value, kwargs):
        filt = []
        res = model.Resource.get(int(value))
        if res.rate_table_id:
            filt.append(res.rate_table_id)
        # for p in res.prefixes:
        #     if p.rate_table_id:
        #         filt.append(p.rate_table_id)
        filt = list(set(filt))
        cls = self.model_class
        if filt:
            query = query.filter(cls.rate_table_id.in_(filt))
        else:
            query = query.filter(cls.rate_table_id.is_(None))
        return query

    def modify_query_from_filtering_for_list(self, filtering, **kwargs):
        ret, q = super(ClientPortalRateTableList, self).modify_query_from_filtering_for_list(filtering, **kwargs)
        q = q.outerjoin(model.ProductRoutRateTable,
                        model.RateTable.rate_table_id == model.ProductRoutRateTable.rate_table_id). \
            outerjoin(model.ProductClientsRef).filter(or_(model.ProductRoutRateTable.type == 'public',
                                                          model.ProductClientsRef.client_id == int(
                                                              kwargs['client_id']))).group_by(
            model.RateTable.rate_table_id)
        # cls = self.model_class
        # cls_br = model.DidBillingPlan
        # q = q.filter(cls.rate_table_id.notin_(select([cls_br.rate_table_id]).where(cls_br.rate_table_id.isnot(None))))
        log.debug(model.query_to_sting(q))
        return ret, q


class ClientPortalRateList(RateList):
    scheme_class = ClientPortalRateGetScheme
    model_class = model.Rate
    entity_plural = 'Rate'
    path_parameters = ()
    security = (DEFAULT_SECURITY)
    restrict = ()
    is_client = True

    def on_filter_product_id(self, query, value, kwargs):
        prd = model.ProductRoutRateTable.get(int(value))
        if prd:
            cls = self.model_class
            query = query.filter(cls.rate_table_id == prd.rate_table_id)
        return query

    def get_filtering_for_list(self, parsed_qs, **kwargs):
        ret = super(RateList, self).get_filtering_for_list(parsed_qs, **kwargs)
        return ret

    def modify_query_from_filtering_for_list(self, filtering, **kwargs):
        ret, q = super(ClientPortalRateList, self).modify_query_from_filtering_for_list(filtering, **kwargs)

        q = q.outerjoin(model.ProductRoutRateTable,
                        model.Rate.rate_table_id == model.ProductRoutRateTable.rate_table_id). \
            outerjoin(model.ProductClientsRef).filter(or_(model.ProductRoutRateTable.type == 'public',
                                                          model.ProductClientsRef.client_id == int(
                                                              kwargs['client_id']))).group_by(model.Rate.id)
        log.debug(model.query_to_sting(q))
        return ret, q


class ClientPortalDidBillingPlanList(DidBillingPlanList):
    is_client = True

    def modify_query_from_filtering_for_list(self, filtering, **kwargs):
        ret, q = super(ClientPortalDidBillingPlanList, self).modify_query_from_filtering_for_list(filtering, **kwargs)
        # client_id = int(kwargs['client_id'])
        # q = q.filter(model.DidBillingPlan.client_id == client_id)
        # log.debug(str(q))
        # ret['is_effective'] = 'True'
        return ret, q



class ClientPortalOrigDidBillingRelCreate(DidBillingRelCreate):
    is_client = True


class ClientPortalDisableDidBillingRelResource(DnlResource):
    model_class = model.DidBillingRel
    scheme_class = ClientPortalDidBillingRelDisableScheme
    scheme_class_get = ClientPortalDidRepositoryGetScheme
    scheme_class_modify = ClientPortalDidBillingRelDisableScheme
    entity = 'DidRepository item disable'
    id_field = 'id'
    has_update_by = False
    unique_field = 'did'
    security = (DEFAULT_SECURITY)
    restrict = ()
    has_info_operation = False
    has_delete_operation = False
    is_client = True

    def get_object_data(self, resp, model_class, scheme_class_get, **kwargs):
        if self.req_data and '_obj_id' in self.req_data:
            kwargs['id'] = self.req_data['_obj_id']
        return super(ClientPortalDisableDidBillingRelResource, self).get_object_data(resp, model_class,
                                                                                     scheme_class_get, **kwargs)

    def before_update(self, obj, req):
        if obj.end_date:
            raise ValidationError(
                'Did {} id {} was finished at {} and cannot be changed !'.format(obj.did, obj.id, obj.end_date))
        data = req.data
        if obj.client_res:
            p = model.ProductItemsResource
            p.filter(p.resource_id == obj.client_res_id).delete()
            obj.end_date = datetime.utcnow().date()
            obj.save()
            # make_transient(obj)
            obj = model.DidBillingRel(did=obj.did, buy_billing_plan_id=obj.buy_billing_plan_id,
                                      sell_billing_plan_id=obj.sell_billing_plan_id, vendor_res_id=obj.vendor_res_id)
            obj.id = None
            obj.end_date = None
            obj.start_date = datetime.utcnow().date()
        if 'client_billing_rule_id' in req.data:
            obj.buy_billing_plan_id = req.data['client_billing_rule_id']
        if 'vendor_billing_rule_id' in req.data:
            obj.sell_billing_plan_id = req.data['vendor_billing_rule_id']
        if 'vendor_res_id' in req.data:
            obj.vendor_res_id = req.data['vendor_res_id']
        obj.client_res_id = None
        cr = DidBillingRelCreate()
        cr.req = self.req
        obj = cr.before_create(obj, client_id=None)  # data['client_id'])
        self.req_data['_obj_id'] = obj.save()
        obj = self.model_class.get(self.req_data['_obj_id'])
        return obj


class ClientPortalAssignDidBillingRelResource(DnlResource):
    model_class = model.DidBillingRel
    scheme_class = ClientPortalDidBillingRelAssignScheme
    scheme_class_get = ClientPortalDidRepositoryGetScheme
    scheme_class_modify = ClientPortalDidBillingRelAssignScheme
    entity = 'DidRepository item assign'
    id_field = 'id'
    has_update_by = False
    unique_field = 'did'
    security = (DEFAULT_SECURITY)
    restrict = ()
    has_info_operation = False
    has_delete_operation = False
    is_client = True

    def get_object_data(self, resp, model_class, scheme_class_get, **kwargs):
        if self.req_data and '_obj_id' in self.req_data:
            kwargs['id'] = self.req_data['_obj_id']
        return super(ClientPortalAssignDidBillingRelResource, self).get_object_data(resp, model_class, scheme_class_get,
                                                                                    **kwargs)

    def before_update(self, obj, req):
        cl = model.Client.get(req.context['client_id'])
        assign = req.data.get('assign', None)
        if assign == True:
            if not obj.client_billing_rule_id and not (
                    'client_billing_rule_id' in req.data and req.data['client_billing_rule_id']):
                raise ValidationError('no client_billing_rule id !')
            if not obj.is_available:
                raise ValidationError('Did {} id {} not available for clients !'.format(obj.did, obj.id))
            if obj.client_res_id and obj.client_id != req.context['client_id']:
                raise ValidationError('Did {} id {} already assigned! {}'.format(obj.did, obj.id, obj.client_res_id))
        if obj.end_date:
            raise ValidationError(
                'Did {} id {} was finished at {} and cannot be changed !'.format(obj.did, obj.id, obj.end_date))
        data = req.data

        if obj.client_res:
            p = model.ProductItemsResource
            p.filter(p.resource_id == obj.client_res_id).delete()
            obj.end_date = datetime.utcnow().date()
            obj.save()
            # make_transient(obj)
            obj = model.DidBillingRel(did=obj.did, buy_billing_plan_id=obj.buy_billing_plan_id,
                                      sell_billing_plan_id=obj.sell_billing_plan_id, vendor_res_id=obj.vendor_res_id)
            obj.id = None
            obj.end_date = None
            obj.start_date = datetime.utcnow().date()
            obj.is_available = not assign

        if assign:
            if 'client_billing_rule_id' in req.data:
                obj.buy_billing_plan_id = req.data['client_billing_rule_id']
            if 'vendor_billing_rule_id' in req.data:
                obj.sell_billing_plan_id = req.data['vendor_billing_rule_id']
            if 'vendor_res_id' in req.data:
                obj.vendor_res_id = req.data['vendor_res_id']
            obj.client_res_id = None
            cr = DidBillingRelCreate()
            cr.req = self.req
            obj = cr.before_create(obj, client_id=req.context['client_id'])
            self.req_data['_obj_id'] = obj.save()
            obj = self.model_class.get(self.req_data['_obj_id'])
        return obj


class ClientPortalOrigDidBillingRelList(DnlList):
    model_class = model.DidBillingRel
    scheme_class = ClientPortalDidRepositoryGetScheme
    entity_plural = 'My DidRepository items'
    is_client = True

    def modify_query_from_filtering_for_list(self, filtering, **kwargs):
        ret, q = super(ClientPortalOrigDidBillingRelList, self).modify_query_from_filtering_for_list(filtering,
                                                                                                     **kwargs)

        q = q.outerjoin(model.Resource, model.Resource.resource_id == model.DidBillingRel.vendor_res_id). \
            filter(model.Resource.client_id == int(kwargs['client_id'])).group_by(model.DidBillingRel.id)
        log.debug(str(q))
        ret['is_effective'] = 'True'
        return ret, q


class ClientPortalDidBillingRelList(DnlList):
    model_class = model.DidAssignments
    scheme_class = ClientPortalDidRepositoryGetScheme
    entity_plural = 'My DidRepository items'
    is_client = True

    # def on_filter_did(self, q, val, kwargs):
    #     cls = self.model_class
    #     q = q.filter(cls.did.like('{}%'.format(val)))
    #     return q

    def modify_query_from_filtering_for_list(self, filtering, **kwargs):
        ret, q = super(ClientPortalDidBillingRelList, self).modify_query_from_filtering_for_list(filtering, **kwargs)
        client_id = int(kwargs['client_id'])
        q = q.filter(model.DidAssignments.client_id == client_id)
        log.debug(str(q))
        # ret['is_effective'] = 'True'
        return ret, q


class ClientPortalFreeDidBillingRelList(DnlList):
    model_class = model.DidBillingRel
    scheme_class = ClientPortalFreeDidBillingRelGetScheme
    entity_plural = 'Free DidRepository items'
    is_client = True

    def modify_query_from_filtering_for_list(self, filtering, **kwargs):
        ret, q = super(ClientPortalFreeDidBillingRelList, self).modify_query_from_filtering_for_list(filtering,
                                                                                                     **kwargs)
        client_id = int(kwargs['client_id'])
        q = q.filter(
            or_(model.DidBillingRel.assigned_client_id.is_(None), model.DidBillingRel.assigned_client_id == client_id))
        q = q.filter(and_(model.DidBillingRel.client_res_id.is_(None), model.DidBillingRel.end_date.is_(None),
                          model.DidBillingRel.is_available == True))
        log.debug(model.query_to_sting(q))
        ret['is_effective'] = 'True'
        return ret, q


class ClientPortalInvoiceList(DnlList):
    scheme_class = ClientPortalInvoiceGetScheme  # InvoiceClientGetScheme
    model_class = model.Invoice
    entity_plural = 'Invoices'
    path_parameters = ()
    security = (DEFAULT_SECURITY)
    restrict = ()
    is_client = True

    def on_filter_invoice_time_lt(self, query, value, kwargs):
        import re
        cls = self.model_class
        if re.match(DATE_REGEXP, value):
            value = value + ' 23:59:59'
        query = query.filter(cls.invoice_time <= value)
        return query

    def modify_query_from_filtering_for_list(self, filtering, **kwargs):
        ret, q = super(ClientPortalInvoiceList, self).modify_query_from_filtering_for_list(filtering, **kwargs)
        # q=q.filter(and_( #model.Invoice.type.in_(['sent(out--buy)','sent(all)']), model.Invoice.state == 'send', model.Invoice.amount > 0))
        log.debug(model.query_to_sting(q))
        return ret, q

    def get_filtering_for_list(self, parsed_qs, **kwargs):
        ret = super(ClientPortalInvoiceList, self).get_filtering_for_list(parsed_qs, **kwargs)
        ret['client_id'] = kwargs['client_id']
        ret['is_term'] = 'true'
        return ret

    def modify_query_from_ordering_for_list(self, ordering, query, **kwargs):
        model_class = self.model_class

        def ToInt(numb):
            try:
                return cast(func.nullif(func.regexp_replace(numb, "\\D", "", "g"), ""), Integer)
            except:
                return text("")

        if 'by' in ordering and ordering['by'] == 'invoice_number':
            model_class = self.model_class
            if ordering['dir'] == 'desc':
                query = query.order_by(ToInt(model.Invoice.invoice_number).desc())
            else:
                query = query.order_by(ToInt(model.Invoice.invoice_number))

            ordering = {}

        return ordering, query


class ClientPortalOrigInvoiceList(DnlList):
    scheme_class = ClientPortalInvoiceGetScheme  # InvoiceClientGetScheme
    model_class = model.Invoice
    entity_plural = 'OrigInvoices'
    path_parameters = ()
    security = (DEFAULT_SECURITY)
    restrict = ()
    is_client = True

    def modify_query_from_ordering_for_list(self, ordering, query, **kwargs):
        model_class = self.model_class

        def ToInt(numb):
            try:
                return cast(func.nullif(func.regexp_replace(numb, "\\D", "", "g"), ""), Integer)
            except:
                return text("")

        if 'by' in ordering and ordering['by'] == 'invoice_number':
            model_class = self.model_class
            if ordering['dir'] == 'desc':
                query = query.order_by(ToInt(model.Invoice.invoice_number).desc())
            else:
                query = query.order_by(ToInt(model.Invoice.invoice_number))

            ordering = {}

        return ordering, query

    def modify_query_from_filtering_for_list(self, filtering, **kwargs):
        ret, q = super(ClientPortalOrigInvoiceList, self).modify_query_from_filtering_for_list(filtering, **kwargs)
        # q=q.filter(and_( #model.Invoice.type.in_(['sent(out--buy)','sent(all)']), model.Invoice.state == 'send', model.Invoice.amount > 0))
        log.debug(model.query_to_sting(q))
        return ret, q

    def on_filter_invoice_time_lt(self, query, value, kwargs):
        import re
        cls = self.model_class
        if re.match(DATE_REGEXP, value):
            value = value + ' 23:59:59'
        query = query.filter(cls.invoice_time <= value)
        return query

    def get_filtering_for_list(self, parsed_qs, **kwargs):
        ret = super(ClientPortalOrigInvoiceList, self).get_filtering_for_list(parsed_qs, **kwargs)
        ret['client_id'] = kwargs['client_id']
        ret['is_orig'] = 'true'
        return ret


class ClientPortalPaymentGatewayHistoryCreate(DnlCreate):
    scheme_class = ClientPortalPaymentGatewayHistoryScheme
    entity = 'Gateway Payment'
    unique_field = 'id'
    additional_responses = ()
    security = (DEFAULT_SECURITY)
    restrict = ()
    is_client = True

    def before_create(self, obj, **kwargs):
        obj.client_id = kwargs['client_id']
        return obj


class ClientPortalPaymentList(DnlList):
    scheme_class = ClientPortalPaymentGetScheme
    model_class = model.ClientPayment
    entity_plural = 'ClientPortalPayments'
    path_parameters = ()
    security = (DEFAULT_SECURITY)
    restrict = ()
    is_client = True

    # path_parameters = ({'name': 'client_id', 'description': 'Parent carrier'},)

    def get_filtering_for_list(self, parsed_qs, **kwargs):
        ret = super(ClientPortalPaymentList, self).get_filtering_for_list(parsed_qs, **kwargs)
        ret['client_id'] = kwargs['client_id']
        return ret


class ClientPortalGatewayAndPaymentsList(DnlList):
    scheme_class = ClientPortalGatewayAndPaymentsGetScheme
    model_class = model.PaymentAndGateway
    entity_plural = 'Client portal GatewayPayments and ordinal payments'
    path_parameters = ()
    security = DEFAULT_SECURITY
    restrict = ()
    is_client = True

    def get_objects_list(self, req, model_class, **kwargs):
        parsed_qs = dict(parse_qsl(req.query_string))
        try:
            per_page = int(parsed_qs.get('per_page'))
        except (ValueError, TypeError):
            from falcon_rest import conf
            per_page = conf.get_default_items_per_page()

        try:
            page = int(parsed_qs.get('page', 0))
        except ValueError:
            page = 0

        if 'order_by' in parsed_qs:
            ordering = {
                'by': parsed_qs['order_by'],
                'dir': parsed_qs.get('order_dir', 'asc')
            }
        else:
            ordering = {}

        fields = self.get_fields_for_list(parsed_qs, **kwargs)

        filtering = self.get_filtering_for_list(parsed_qs, **kwargs)

        filtering, query = \
            self.modify_query_from_filtering_for_list(filtering, **kwargs)

        if ordering:
            ordering, query = \
                self.modify_query_from_ordering_for_list(ordering, query, **kwargs)

        return model_class.get_objects_query_list(
            query=query,
            filtering=filtering,
            ordering=ordering,
            paging={'from': page * per_page, 'till': (page + 1) * per_page}
        ) + (page, per_page, fields)

    def modify_query_from_filtering_for_list(self, filtering, **kwargs):
        filt, ret = super(ClientPortalGatewayAndPaymentsList, self).modify_query_from_filtering_for_list(filtering,
                                                                                                         **kwargs)
        cls = self.model_class._tab  # self.model_class

        client_id = kwargs['client_id']
        ret = ret.filter(and_(cls.c.client_id == client_id, cls.c.method.isnot(None)))
        return filt, ret

    def get_filtering_for_list(self, parsed_qs, **kwargs):
        ret = super(ClientPortalGatewayAndPaymentsList, self).get_filtering_for_list(parsed_qs, **kwargs)
        # ret['agent_id'] = kwargs['agent_id']
        return ret


class ClientPortalAlertList(DnlList):
    scheme_class = ClientPortalEmailLogAlertsScheme
    model_class = model.EmailLog
    entity_plural = 'Alerts'
    path_parameters = ()
    security = (DEFAULT_SECURITY)
    restrict = ()
    is_client = True

    def modify_query_from_filtering_for_list(self, filtering, **kwargs):
        ret, q = super(ClientPortalAlertList, self).modify_query_from_filtering_for_list(filtering, **kwargs)
        cls = model.EmailLog
        filt = and_(cls.type.in_(cls.ALERT_TYPES), cls.status != 'fail',
                    cls.is_view != 1, cls.client_id == kwargs['client_id'])
        q1 = q.filter(filt)

        parsed_qs = dict(parse_qsl(self.req.query_string))
        try:
            per_page = int(parsed_qs.get('per_page'))
        except (ValueError, TypeError):
            per_page = 10
        try:
            page = int(parsed_qs.get('page', 0))
        except ValueError:
            page = 0
        paging = {'from': page * per_page, 'till': (page + 1) * per_page}
        if paging:
            if 'from' in paging:
                q1 = q1.offset(paging['from'])
            if 'till' in paging:
                q1 = q1.limit(paging['till'] - paging.get('from', 0))

        ids = [l.id for l in q1.all()]
        if ids:
            q2 = q.filter(cls.id.in_(ids))
            q2.update(dict(is_view=1), synchronize_session='fetch')
            q2.session.commit()
            q = q.filter(or_(cls.id.in_(ids), filt))
        else:
            q = q.filter(filt)
        log.debug(model.query_to_sting(q))
        return ret, q


class ClientPortalMessageList(DnlList):
    scheme_class = ClientPortalEmailLogAlertsScheme
    model_class = model.EmailLog
    entity_plural = 'Alerts'
    path_parameters = ()
    security = (DEFAULT_SECURITY)
    restrict = ()
    is_client = True

    def modify_query_from_filtering_for_list(self, filtering, **kwargs):
        ret, q = super(ClientPortalMessageList, self).modify_query_from_filtering_for_list(filtering, **kwargs)
        cls = model.EmailLog
        filt = and_(cls.type.in_(cls.MESSAGE_TYPES), cls.status != 'fail',
                    cls.is_view != 1, cls.client_id == kwargs['client_id'])
        q1 = q.filter(filt)

        parsed_qs = dict(parse_qsl(self.req.query_string))
        try:
            per_page = int(parsed_qs.get('per_page'))
        except (ValueError, TypeError):
            per_page = 10
        try:
            page = int(parsed_qs.get('page', 0))
        except ValueError:
            page = 0
        paging = {'from': page * per_page, 'till': (page + 1) * per_page}
        if paging:
            if 'from' in paging:
                q1 = q1.offset(paging['from'])
            if 'till' in paging:
                q1 = q1.limit(paging['till'] - paging.get('from', 0))

        ids = [l.id for l in q1.all()]
        q2 = q.filter(cls.id.in_(ids))
        q2.update(dict(is_view=1), synchronize_session='fetch')
        q2.session.commit()
        q = q.filter(or_(cls.id.in_(ids), filt))
        log.debug(model.query_to_sting(q))
        return ret, q


class ClientPortalActualBalanceHistoryList(DnlList):
    scheme_class = ClientPortalActualBalanceHistoryGetScheme
    model_class = model.BalanceHistoryActual
    entity_plural = 'Balanses'
    path_parameters = ()
    security = (DEFAULT_SECURITY)
    restrict = ()
    is_client = True

    def modify_query_from_filtering_for_list(self, filtering, **kwargs):
        ret, q = super(ClientPortalActualBalanceHistoryList, self).modify_query_from_filtering_for_list(filtering,
                                                                                                        **kwargs)
        cls = self.model_class
        q = q.filter(and_(cls.client_id == kwargs['client_id']))
        log.debug(model.query_to_sting(q))
        return ret, q


class ClientDefaultIpCreate(DnlCreate):
    scheme_class = ClientDefaultIpScheme
    entity = 'Client default ip'
    unique_field = 'id'
    additional_responses = ()
    security = (DEFAULT_SECURITY)
    restrict = ()
    is_client = True

    def before_create(self, obj, **kwargs):
        obj.client_id = kwargs['client_id']
        return obj


class ClientDefaultIpResource(DnlResource):
    model_class = model.ClientDefaultIp
    scheme_class = ClientDefaultIpScheme
    scheme_class_get = ClientDefaultIpGetScheme
    entity = 'Client default ip'
    id_field = 'id'
    has_update_by = False
    security = (DEFAULT_SECURITY)
    restrict = ()
    is_client = True

    def get_object_data(self, resp, model_class, scheme_class_get, **kwargs):
        cls = self.model_class
        q = cls.get(int(kwargs['id']))
        if not q or q.client_id != int(kwargs['client_id']):
            return None
        return super(ClientDefaultIpResource, self).get_object_data(resp, model_class, scheme_class_get, **kwargs)


class ClientDefaultIpList(DnlList):
    scheme_class = ClientDefaultIpGetScheme
    model_class = model.ClientDefaultIp
    entity_plural = 'Client default ips'
    path_parameters = ()
    security = (DEFAULT_SECURITY)
    restrict = ()
    is_client = True

    def modify_query_from_filtering_for_list(self, filtering, **kwargs):
        ret, q = super(ClientDefaultIpList, self).modify_query_from_filtering_for_list(filtering, **kwargs)
        cls = self.model_class
        q = q.filter(and_(cls.client_id == kwargs['client_id']))
        log.debug(model.query_to_sting(q))
        return ret, q


class ClientPortalIngressTrunkCreate(IngressTrunkCreate):
    scheme_class = ClientPortalIngressTrunkScheme
    is_client = True
    path_parameters = ()

    def before_create(self, obj, **kwargs):
        obj.client_id = kwargs['client_id']
        return super(ClientPortalIngressTrunkCreate, self).before_create(obj, **kwargs)


class ClientPortalIngressTrunkResource(IngressTrunkResource):
    scheme_class = IngressTrunkScheme
    scheme_class_modify = ClientPortalIngressTrunkModifyScheme
    is_client = True

    def on_patch(self, req, resp, **kwargs):
        log.debug("on_patch")
        return super(ClientPortalIngressTrunkResource, self).on_patch(req, resp, **kwargs)

    def get_object_data(self, resp, model_class, scheme_class_get, **kwargs):
        cls = self.model_class
        q = cls.get(int(kwargs['resource_id']))
        log.debug(f"Q = {q}")
        if not q or q.client_id != int(kwargs['client_id']):
            return None
        return super(ClientPortalIngressTrunkResource, self).get_object_data(resp, model_class, scheme_class_get,
                                                                             **kwargs)

    def before_update(self, obj, req):
        log.debug("BEFORE UPDATE")
        if obj:
            client = model.Client.get(obj.client_id)
            if not client.enable_trunk_edit:
                raise ValidationError({'message': 'permission revoked'})
        return obj


class ClientPortalIngressTrunkList(IngressTrunkList):
    path_parameters = ()
    is_client = True

    def get_filtering_for_list(self, parsed_qs, **kwargs):
        kwargs['client_id'] = self.get_user(self.req).client_id
        return super().get_filtering_for_list(parsed_qs, **kwargs)


class ClientPortalEgressTrunkCreate(EgressTrunkCreate):
    scheme_class = ClientPortalEgressTrunkScheme
    is_client = True
    path_parameters = ()

    def before_create(self, obj, **kwargs):
        obj.client_id = kwargs['client_id']
        return super(ClientPortalEgressTrunkCreate, self).before_create(obj, **kwargs)


class ClientPortalEgressTrunkResource(EgressTrunkResource):
    scheme_class = ClientPortalEgressTrunkScheme
    scheme_class_modify = ClientPortalEgressTrunkModifyScheme
    is_client = True

    # def get_object_data(self, resp, model_class, scheme_class_get, **kwargs):
    def get_object(self, resp, model_class, **kwargs):
        cls = self.model_class
        q = cls.get(int(kwargs['resource_id']))
        if not q or q.client_id != int(kwargs['client_id']):
            return None
        # return super(ClientPortalEgressTrunkResource,self).get_object(resp, model_class, **kwargs)
        return super(ClientPortalEgressTrunkResource, self).get_object(resp, model_class, **kwargs)

    def before_update(self, obj, req):
        if obj:
            client = model.Client.get(obj.client_id)
            if not client.enable_trunk_edit:
                raise ValidationError({'message': 'permission revoked'})
        return obj


class ClientPortalEgressTrunkList(EgressTrunkList):
    path_parameters = ()
    is_client = True

    def get_filtering_for_list(self, parsed_qs, **kwargs):
        kwargs['client_id'] = self.get_user(self.req).client_id
        return super().get_filtering_for_list(parsed_qs, **kwargs)


class ClientPortalResourceIpResource(ResourceIpResource):
    entity = 'Client resource ip'
    is_client = True

    def get_object_data(self, resp, model_class, scheme_class_get, **kwargs):
        cls = self.model_class
        q = cls.get(int(kwargs['resource_ip_id']))
        if not q or q.client_id != int(kwargs['client_id']):
            return None
        client = model.Client.get(q.client_id)
        if not client.enable_trunk_edit:
            self.set_response(resp, responses.UnAuthorizedErrorResponse(data={'message': 'permission revoked'}))
            return None
        return super(ClientPortalResourceIpResource, self).get_object_data(resp, model_class, scheme_class_get, **kwargs)

    def before_update(self, obj, req):
        if obj:
            client = model.Client.get(obj.client_id)
            if not client.enable_trunk_edit:
                raise ValidationError({'message': 'permission revoked'})
        return obj


class ClientPortalResourceIpList(ResourceIpList):
    scheme_class = ResourceIpClientGetScheme
    path_parameters = ()
    is_client = True

    def get_objects_list(self, req, model_class, **kwargs):
        if 'client_id' in kwargs:
            client = model.Client.get(int(kwargs['client_id']))
            if not client.enable_trunk_view:
                raise ValidationError({'message': 'permission revoked'})
        return super().get_objects_list(req, model_class, **kwargs)

    def modify_query_from_filtering_for_list(self, filtering, **kwargs):
        ret, q = super().modify_query_from_filtering_for_list(filtering, **kwargs)
        client = model.Client.get(int(kwargs['client_id']))
        cls = self.model_class
        q = q.filter(cls.resource_id == client.resource.resource_id)
        return ret, q


class ClientPortalResourceIpUniqueList(ResourceIpList):
    scheme_class = ResourceIpPortScheme
    path_parameters = ()
    is_client = True

    def modify_query_from_filtering_for_list(self, filtering, **kwargs):
        ret, q = super().modify_query_from_filtering_for_list(filtering, **kwargs)
        client = model.Client.get(int(kwargs['client_id']))
        cls = self.model_class
        q = q.filter(cls.client_id == client.client_id)
        return ret, q

    def on_get(self, req, resp, **kwargs):
        super().on_get(req, resp, **kwargs)
        data = json.loads(resp.body)
        if "payload" in data and "items" in data["payload"]:
            items = data["payload"]["items"]
            unique_items = list({v['ip']: v for v in items}.values())
            data["payload"]["items"] = unique_items
            resp.body = json.dumps(data)


class ClientPortalTrunkAssignProduct(TrunkAssignProduct):
    is_client = True

    def get_object_data(self, resp, model_class, scheme_class_get, **kwargs):
        cls = self.model_class
        q = cls.get(int(kwargs['trunk_id']))
        if not q or q.client_id != int(kwargs['client_id']):
            return None
        return super(ClientPortalTrunkAssignProduct, self).get_object_data(resp, model_class, scheme_class_get,
                                                                           **kwargs)


class ClientPortalResourcePrefixCreate(ResourcePrefixCreate):
    is_client = True

    def before_create(self, obj, **kwargs):
        obj.resource_id = kwargs['trunk_id']
        q = model.Resource.get(int(kwargs['trunk_id']))
        if not q or q.client_id != int(kwargs['client_id']):
            raise NoResultFound()
        return obj


class ClientPortalResourcePrefixResource(ResourcePrefixResource):
    scheme_class = ClientPortalResourceIpPortScheme
    is_client = True

    def get_object_data(self, resp, model_class, scheme_class_get, **kwargs):
        q = model.ResourcePrefix.get(int(kwargs['id']))
        if not q or q.client_id != int(kwargs['client_id']):
            return None
        return super(ClientPortalResourcePrefixResource, self).get_object_data(resp, model_class, scheme_class_get,
                                                                               **kwargs)


class ClientPortalResourcePrefixList(ResourcePrefixList):
    is_client = True

    def modify_query_from_filtering_for_list(self, filtering, **kwargs):
        ret, q = super(ClientPortalResourcePrefixList, self).modify_query_from_filtering_for_list(filtering, **kwargs)
        cls = self.model_class
        res = model.Resource
        q = q.filter(cls.resource_id.in_(select([res.resource_id]).where(res.client_id == kwargs['client_id'])))
        log.debug(model.query_to_sting(q))
        return ret, q

class ClientPortalDidProductSendToEmails(CustomAction):
    scheme_class = ClientPortalProductSendToEmailsScheme
    model_class = model.ClientDidProduct
    description = 'Send product rates to emails'
    path_parameters = ({'name': 'client_id', 'description': 'client_did_product id which rate table will send'},)
    body_parameters = ('Emails to send', ClientPortalProductSendToEmailsScheme)
    method = 'post'
    is_client = True

    def on_post(self, req, resp, **kwargs):
        self.init_req(req)
        if not check_context(self, req, resp, **kwargs):
            return False
        kwargs = client_context(self, req, resp, **kwargs)
        return self.proceed(req, resp, **kwargs)

    def apply(self, obj, req, resp, **kwargs):
        client_id = kwargs['client_id']
        if not obj:
            self.set_response(resp, responses.ObjectNotFoundErrorResponse())
            return False
        errors = ClientPortalProductSendToEmailsScheme().validate(req.data)
        if errors:
            self.set_response(resp, ValidationError(data=errors))
            return False
        try:
            from api_dnl.task.send_rate import send_client_product_to_emails_task
            from api_dnl.model import RateSendLog, MailTemplate, SendRateDirect
            emails = req.data.get('emails', None)
            product = model.DidProductItem.filter(model.DidProductItem.did_product_id == obj.did_product_id).first()
            if not product or not product.billing_rule_id:
                raise ValidationError({'rate_table_id': 'client_did_product don\'t have associated did_product_item, nothing to send'})
            billing_plan = model.DidBillingPlan.filter(model.DidBillingPlan.id == product.billing_rule_id).first()
            if not billing_plan or not billing_plan.rate_table_id:
                raise ValidationError({'rate_table_id': 'client_did_product don\'t have associated rate_table_id, nothing to send'})


            sr = RateSendLog(rate_table_id=billing_plan.rate_table_id, status='waiting')

            sr.error = 'temporary busy'
            tpl = MailTemplate.get('send_product')
            email_direct = SendRateDirect(sender_id=tpl.from_mail_id, subject=tpl.subject, content=tpl.html_content,
                                          # header=SendRateTemplate.HEADERS,
                                          download_method='send as link',
                                          mail_cc=tpl.cc_mail)
            sr.email_direct = email_direct

            sr.send_type = 'Specify my own recipients'
            sr.send_specify_email = emails
            sr.format = 'csv'
            job_id = sr.save()
            # if job_id:  # send_rate(job_id):
            #     q = ProductClientsRefUsed.filter(
            #         and_(ProductClientsRefUsed.product_id == product_id,
            #              ProductClientsRefUsed.client_id == client_id)).first()
            #     if not q:
            #         q = ProductClientsRefUsed(product_id=product_id, client_id=client_id)
            #     q._rate_last_sent = datetime.now(UTC)
            #     q.save()
            self.set_response(resp, responses.ObjectCreatedResponse(data=job_id))
            return False
        except Exception as e:
            self.set_response(resp, OperationErrorResponse(e))
            return False
        return True

class ClientPortalProductSendToEmails(CustomAction):
    scheme_class = ClientPortalProductSendToEmailsScheme
    model_class = model.ProductRoutRateTable
    description = 'Send product rates to emails'
    path_parameters = ({'name': 'id', 'description': 'Product id which rate table will send'},)
    body_parameters = ('Emails to send', ClientPortalProductSendToEmailsScheme)
    method = 'post'
    is_client = True

    def on_post(self, req, resp, **kwargs):
        self.init_req(req)
        if not check_context(self, req, resp, **kwargs):
            return False
        kwargs = client_context(self, req, resp, **kwargs)
        return self.proceed(req, resp, **kwargs)

    def apply(self, obj, req, resp, **kwargs):
        client_id = kwargs['client_id']
        if not obj:
            self.set_response(resp, responses.ObjectNotFoundErrorResponse())
            return False
        errors = ClientPortalProductSendToEmailsScheme().validate(req.data)
        if errors:
            self.set_response(resp, ValidationError(data=errors))
            return False
        try:
            from api_dnl.task.send_rate import send_client_product_to_emails_task
            from api_dnl.model import RateSendLog, MailTemplate, SendRateDirect, ProductClientsRef, \
                ProductClientsRefUsed
            emails = req.data.get('emails', None)
            product_id = obj.id

            sr = RateSendLog(rate_table_id=obj.rate_table_id, status='waiting')

            sr.error = 'temporary busy'
            tpl = MailTemplate.get('send_product')
            email_direct = SendRateDirect(sender_id=tpl.from_mail_id, subject=tpl.subject, content=tpl.html_content,
                                          # header=SendRateTemplate.HEADERS,
                                          download_method='send as link',
                                          mail_cc=tpl.cc_mail)
            sr.email_direct = email_direct

            sr.send_type = 'Send to specified trunks'
            resources = model.Resource.filter(model.Resource.client_id == client_id).all()
            resource_ids = [i.resource_id for i in resources]
            sr.send_specify_email = emails
            sr.trunks = resource_ids
            sr.format = 'csv'
            job_id = sr.save()
            if job_id:  # send_rate(job_id):
                q = ProductClientsRefUsed.filter(
                    and_(ProductClientsRefUsed.product_id == product_id,
                         ProductClientsRefUsed.client_id == client_id)).first()
                if not q:
                    q = ProductClientsRefUsed(product_id=product_id, client_id=client_id)
                q._rate_last_sent = datetime.now(UTC)
                q.save()
            self.set_response(resp, responses.ObjectCreatedResponse(data=job_id))
            return False
        except Exception as e:
            self.set_response(resp, OperationErrorResponse(e))
            return False
        return True


class DidImportCreate(DnlCreate):
    model_class = model.DidNumberUpload
    scheme_class = DidNumberUploadScheme
    scheme_class_get = DidNumberUploadSchemeGet
    entity = 'import of did file '
    id_field = 'uuid'
    path_parameters = ()
    security = (DEFAULT_SECURITY)
    restrict = ()
    body_params = ()

    # is_client = True

    def get_spec(self):
        additional_responses = (
                                   responses.OperationErrorResponse(data=errors.CommonErrors.PermissionError,
                                                                    description='Can\'t create file'),
                               ) + self.additional_responses

        ext_body_parameters = self.body_params or (
            {'name': 'file', 'in': 'formData', 'description': 'File to upload', 'required': True, 'type': 'file'},
            {'name': 'duplicates_action', 'in': 'formData',
             'description': 'skip duplicates (default) or replace existing numbres', 'required': True,
             "enum": ["skip", "override"], "default": "skip", "type": "string"},
            {'name': 'vendor_id', 'in': 'formData', 'description': 'vendor', 'required': False, "type": "number"},
            {'name': 'vendor_billing_rule_id', 'in': 'formData', 'description': 'vendor', 'required': False,
             "type": "number"},
            {'name': 'client_billing_rule_id', 'in': 'formData', 'description': 'vendor', 'required': False,
             "type": "number"},
            {'name': 'client_id', 'in': 'formData', 'description': 'vendor', 'required': False,
             "type": "number"},
        )
        return swagger.specify.get_spec(
            method='post', description='Creates new {}'.format(self.entity.lower()),
            consumes=['multipart/form-data'],
            path_parameters=self.path_parameters,
            responses=(
                          # self.scheme_class.get_object_created_response(entity=self.entity),
                          responses.ObjectCreatedResponse(data_key='object_uuid', scheme=schemes.ObjectCreatedUuidAsPk),
                          SuccessResponseJustOk()

                      ) + additional_responses + self.get_additional_responses(method='post'),
            security=self.get_security(method='post'),
            # body_parameters=('{} to create'.format(self.entity), self.scheme_class),
            ext_body_parameters=ext_body_parameters
        )

    def before_create(self, obj, **kwargs):
        obj.uuid = generate_uuid_str()()
        if 'client_id' in kwargs:
            obj.client_id = kwargs['client_id']

        file = self.req.files['file']
        user = self.get_user(self.req)
        obj.created_by = user.name
        if not obj.client_id:
            obj.client_id = self.req_data.get('client_id', None)

        obj.vendor_id = self.req_data.get('vendor_id', None)

        if obj.vendor_id and (
                not model.Client.get(obj.vendor_id) or model.Client.get(obj.vendor_id).client_type != 'vendor'):
            raise ValidationError({'vendor_id': ['no such vendor {}'.format(obj.vendor_id)]})

        obj.vendor_billing_rule_id = self.req_data.get('vendor_billing_rule_id', None)
        if obj.vendor_billing_rule_id and not model.DidBillingPlan.get(obj.vendor_billing_rule_id):
            raise ValidationError(
                {'vendor_billing_rule_id': ['no such vendor_billing_rule_id {}'.format(obj.vendor_billing_rule_id)]})

        obj.client_billing_rule_id = self.req_data.get('client_billing_rule_id', None)
        if obj.client_billing_rule_id and not model.DidBillingPlan.get(obj.client_billing_rule_id):
            raise ValidationError(
                {'client_billing_rule_id': ['no such client_billing_rule_id {}'.format(obj.client_billing_rule_id)]})

        obj.duplicates_action = self.req_data.get('duplicates_action', 'skip')
        obj.file = file.filename
        b = file.readlines()
        obj.num_records = len(b)
        log.info('Uploaded file rows:{}'.format(obj.num_records))
        local_file = open(obj.file_name, 'wb')
        local_file.writelines(b)
        local_file.close()
        return obj

    def after_create(self, result, req, resp, **kwargs):
        try:
            from api_dnl.task.did_import_file import did_import_file
            try:
                did_import_file.delay(result)
            except Exception as e:
                log.warning('import_file task not started: {}'.format(str(e)))
                did_import_file(result)
        except Exception as e:
            log.error('import_file task error: {}'.format(str(e)))
            pass


class DidImportList(DnlList):
    scheme_class = DidNumberUploadSchemeGet
    model_class = model.DidNumberUpload
    entity_plural = 'DidImportTask'
    path_parameters = ()
    security = (DEFAULT_SECURITY)
    restrict = ()
    is_client = True

    def modify_query_from_filtering_for_list(self, filtering, **kwargs):
        ret, q = super(DidImportList, self).modify_query_from_filtering_for_list(filtering, **kwargs)
        cls = self.model_class
        q = q.filter(and_(cls.client_id == kwargs['client_id']))
        log.debug(model.query_to_sting(q))
        return ret, q


# region DidApi
class DidApiVendorList(DnlList):
    scheme_class = ClientPortalDidVendorSchemeGet
    model_class = model.DidVendor
    entity_plural = 'DidApiVendors'
    path_parameters = ()
    security = (DEFAULT_SECURITY)
    restrict = ()
    is_client = True


class DidApiSetWebhook(CustomPostAction):
    scheme_class = DidApiOrderLocalScheme
    path_parameters = ()
    additional_responses = (responses.SuccessResponseObjectsList(payload_scheme_items=DidApiOrderLocalGetScheme),)
    body_parameters = ('Items', scheme_class,)
    is_client = True

    def on_post(self, req, resp, **kwargs):

        self.init_req(req)
        if not check_context(self, req, resp, **kwargs):
            self.set_response(resp, responses.ForbiddenErrorResponse())
            return
        user = self.get_user(self.req)
        client_id = str(user.client.client_id)
        params = req.data
        # params['client_id'] = 'CLI_{}'.format(client_id)
        # params = dict(parse_qsl(req.query_string))
        errors = self.scheme_class().validate(params)
        if errors:
            self.set_response(resp, responses.ValidationErrorResponse(data=errors))
            return
        try:
            ret = []
            items = unprefix(params['items'])
            for item in items:
                try:

                    cls = model.DidBillingRel
                    q = cls.filter(and_(cls.end_date.is_(None),
                                        or_(cls.did == item['number'], cls.did == item['number'][1:]))).first()
                    if not q:
                        raise ValidationError({'sender': ['number {} not in assigned dids'.format(item['number'])]})
                    vendor_api = q.vendor_res.client.vendor_api
                    if not vendor_api:
                        raise ValidationError(
                            {'sender': ['sender {} assigned did has no vendor api'.format(item['number'])]})
                    api = vendor_api.get_api()
                    url = '{}/smshook'.format(settings.API_URL)
                    number = item['number']
                    if len(number) == 10:
                        number = '1' + number
                    res = api.set_sms_receive_hook(number, url, client_id)
                    if res['success']:
                        item['status'] = 'success'
                    else:
                        item['status'] = 'error'
                        item['error'] = ret
                except Exception as e:
                    item['status'] = 'error'
                    item['error'] = str(e)
                ret.append(item)

            self.set_response(resp, responses.SuccessResponse(data={
                'items': ret,
                'total': len(ret) if ret else 0, 'page': 0,
                'per_page': len(ret) if ret else 0
            }, scheme=schemes.ObjectScheme))

        except Exception as e:
            self.set_response(resp, OperationErrorResponse(e))
            return None
        pass


def _get_all_vendors(manual_vendor=False, search_type='local', is_sms=True):
    # return [VendorManual()]
    cls = model.DidVendor
    vq = cls.filter(and_(cls.api_enabled, cls.sms_enabled == is_sms, or_(cls.search_type == search_type,
                                                                         cls.search_type.is_(None))))
    vl = []
    for v in vq:
        try:
            api = v.get_api()
            if api:
                vl.append(v)
            else:
                log.error('get_all_vendors no api on vendor {}'.format(v.username))
        except Exception as e:
            log.error('get_all_vendors api error on vendor {}: {}'.format(v.username, e))
    if manual_vendor:
        return [VendorManual()] + vl
    else:
        return v


# region CientPortalDidApiLocal

class DidApiSearchLocal(CustomGetAction):
    scheme_class = DidApiSearchScheme
    path_parameters = ()
    additional_responses = (responses.SuccessResponseObjectsList(payload_scheme_items=DidApiSearchGetScheme),)
    query_parameters = [{'name': n, 'type': swagger.specify.get_field_type(f)} for n, f in
                        scheme_class._declared_fields.items()]
    is_client = True

    def on_get(self, req, resp, **kwargs):

        self.init_req(req)
        if not check_context(self, req, resp, **kwargs):
            self.set_response(resp, responses.ForbiddenErrorResponse())
            return
        user = self.get_user(self.req)
        # params = req.data
        params = dict(parse_qsl(req.query_string))
        if 'state' in params and params['state']:
            params['state'] = params['state'].upper()
        params['client_id'] = user.client_id
        page = int(params.pop('page', 0))
        per_page = int(params.pop('per_page', 10))
        params['count'] = 100
        if 'order_by' in params:
            order = [params.pop('order_by'), params.pop('order_dir', 'asc')]
            params['order'] = order
        errors = self.scheme_class().validate(params)
        if errors:
            self.set_response(resp, responses.ValidationErrorResponse(data=errors))
            return

        is_sms = params.pop('is_sms', False)
        vendors = _get_all_vendors(True, 'local', is_sms)
        if not vendors:
            self.set_response(resp, responses.ValidationErrorResponse(data={'none': 'no one available vendor'}))
            return
        err = []
        for v in vendors:
            try:
                v.get_api()
            except Exception as e:
                msg = 'Vendor {} has  wrong api settings {}'.format(v.vendor.name, e)
                err.append(msg)
        if err:
            self.set_response(resp, responses.ValidationErrorResponse(data={'none': str(err)}))
            return False
        try:
            api = DidAPIMuliVendor(vendors)
            ret = api.search_local(**params)
            log.debug('multi vendor search local:{}'.format(ret))
            ret = list(
                map(lambda i: i if 'fees' in i else (
                    dict(**i, fees=model.DidVendor.fees(user.client_id, i['number']))), ret))
            # ret = set_prefix(ret)

            didProduct = model.ClientDidProduct
            clientDidProducts = (
                didProduct
                .filter(didProduct.client_id == user.client_id, didProduct.did_product_id != None)
                .with_entities(didProduct.did_product_id)
                .all()
            )
            did_product_ids = [row.did_product_id for row in clientDidProducts]

            didProductItem = model.DidProductItem
            did_product_items = (
                didProductItem
                .filter(
                    didProductItem.did_product_id.in_(did_product_ids),
                    didProductItem.type == 0  # Only "Local"
                )
                .with_entities(didProductItem.country, didProductItem.type)
                .all()
            )
            allowed_countries = [row.country for row in did_product_items]

            def _is_us_number(number):
                prefixes = ['1800', '1877', '1866', '1855', '1844', '1833', '1888']
                if len(number) == 11 and number[:4] in prefixes:
                    return True
                return False

            if 'US' in allowed_countries:
                ret = [item for item in ret if item['country'] == 'US' or _is_us_number(item['number'])]
            else:
                ret = [item for item in ret if item['country'] in allowed_countries]

            numbers = [item["number"] for item in ret]
            did_repo_entries = (
                model.get_db().session.query(model.DidRepository.did, model.DidRepository.id)
                .filter(model.DidRepository.did.in_(numbers))
                .all()
            )
            did_repo_map = {row.did: row.id for row in did_repo_entries}
            for item in ret:
                item.pop("vendor_id", None)
                item["did_repository_id"] = did_repo_map.get(item["number"])
            start, end = user.client.mask_did_start_post, user.client.mask_did_end_post
            if not start or not end:
                start = 6
                end = 8
            if start and end and end >= start:
                for did in ret:
                    did['number'] = did['number'][:start - 1] + (end - start + 1) * '*' + did['number'][end:]
            log.debug('search local + fees :{}'.format(ret))
            self.set_response(resp, responses.SuccessResponse(data={
                'items': ret[page*per_page:(page+1)*per_page],
                'total': len(ret), 'page': page,
                'per_page': per_page
            }, scheme=schemes.ObjectScheme))

        except Exception as e:
            import traceback
            trace = traceback.format_exc()
            log.error(trace)
            log.debug('search did error {}'.format(e))

            self.set_response(resp, OperationErrorResponse(e))
            return None
        pass


class DidApiCoverageLocal(CustomGetAction):
    scheme_class = DidApiSearchCoverageScheme
    path_parameters = ()
    additional_responses = (responses.SuccessResponseObjectsList(payload_scheme_items=DidApiSearchCoverageGetScheme),)
    query_parameters = [{'name': n, 'type': swagger.specify.get_field_type(f)} for n, f in
                        scheme_class._declared_fields.items()]
    is_client = True

    def on_get(self, req, resp, **kwargs):

        vendors = _get_all_vendors(True, 'local')
        if not vendors:
            self.set_response(resp, responses.ObjectNotFoundErrorResponse())
            return
        self.init_req(req)
        if not check_context(self, req, resp, **kwargs):
            self.set_response(resp, responses.ForbiddenErrorResponse())
            return
        user = self.get_user(self.req)
        # params = req.data
        params = dict(parse_qsl(req.query_string))
        errors = self.scheme_class().validate(params)
        if errors:
            self.set_response(resp, responses.ValidationErrorResponse(data=errors))
            return
        try:
            api = DidAPIMuliVendor(vendors)
            ret = api.coverage_local(**params)
            log.debug('ret:{}'.format(str(ret)[0:4096]))
            self.set_response(resp, responses.SuccessResponse(data={
                'items': ret,
                'total': len(ret), 'page': 0,
                'per_page': len(ret)
            }, scheme=schemes.ObjectScheme))

        except Exception as e:
            self.set_response(resp, OperationErrorResponse(e))
            return None
        pass


class DidApiSearchOrderLocal(CustomPostAction):
    scheme_class = DidApiSearchOrderLocalScheme
    path_parameters = ()
    additional_responses = (responses.SuccessResponseObjectsList(payload_scheme_items=DidApiOrderLocalGetScheme),)
    body_parameters = ('Order parameters', scheme_class,)
    is_client = True

    def on_post(self, req, resp, **kwargs):
        vendors = _get_all_vendors(True, 'local')
        if not vendors:
            self.set_response(resp, responses.ObjectNotFoundErrorResponse())
            return
        self.init_req(req)
        if not check_context(self, req, resp, **kwargs):
            self.set_response(resp, responses.ForbiddenErrorResponse())
            return
        user = self.get_user(self.req)
        client_id = str(user.client.client_id)
        params = req.data
        params['client_id'] = 'CLI_{}'.format(client_id)
        # params = dict(parse_qsl(req.query_string))
        errors = self.scheme_class().validate(params)
        if errors:
            self.set_response(resp, responses.ValidationErrorResponse(data=errors))
            return
        try:
            api = DidAPIMuliVendor(vendors)
            ret = api.search_order_local(**params)
            log.debug('ret:{}'.format(str(ret)[0:4096]))
            for item in ret:
                try:
                    if item['status'] == 'success':
                        obj = model.DidBillingRel.did_billing_rel_create(item['number'], client_id, item['vendor_id'],
                                                                         buy_billing_plan_id=None, user_name=user.name)
                        rule = model.DidVendor.get_client_rule(user.client_id, item['number'])
                        if rule:
                            obj.client_billing_rule_id = rule.id
                        obj.save()
                        item['dnl_status'] = 'success'
                        # assign webhook
                        vendor_api = model.DidVendor.get(int(item['vendor_id']))
                        api = vendor_api.get_api()
                        url = '{}/smshook'.format(settings.API_URL)
                        number = item['number']
                        if len(number) == 10:
                            number = '1' + number
                        res = api.set_sms_receive_hook(number, url, client_id)
                        if not res['success']:
                            raise DidApiException(res)


                except Exception as e:
                    item['dnl_status'] = str(e)

            self.set_response(resp, responses.SuccessResponse(data={
                'items': ret,
                'total': len(ret), 'page': 0,
                'per_page': len(ret)
            }, scheme=schemes.ObjectScheme))
        except Exception as e:
            self.set_response(resp, OperationErrorResponse(e))
            return None
        pass


class DidApiOrderLocal(CustomPostAction):
    scheme_class = DidApiOrderLocalScheme
    path_parameters = ()
    additional_responses = (responses.SuccessResponseObjectsList(payload_scheme_items=DidApiOrderLocalGetScheme),)
    body_parameters = ('Order parameters', scheme_class,)
    is_client = True

    def on_post(self, req, resp, **kwargs):

        vendors = _get_all_vendors(True, 'local')
        # vendors = [vendors[0]]
        self.init_req(req)
        if not check_context(self, req, resp, **kwargs):
            self.set_response(resp, responses.ForbiddenErrorResponse())
            return
        user = self.get_user(self.req)
        if not user.client.resource:
            self.set_response(resp, responses.ForbiddenErrorResponse())
            return
        client_id = str(user.client.client_id)

        params = req.data
        params['client_id'] = 'CLI_{}'.format(client_id)
        # params = dict(parse_qsl(req.query_string))
        errors = self.scheme_class().validate(params)
        if errors:
            self.set_response(resp, responses.ValidationErrorResponse(data=errors))
            return
        try:
            api = DidAPIMuliVendor(vendors)
            for item in params['items']:
                if 'number' not in item:
                    repo = model.get_db().session.query(model.DidRepository).get(item['did_repository_id'])
                    if repo:
                        item['number'] = repo.did
            items = unprefix_2(params['items'])
            # check balance
            cost = 0.0
            for item in items:
                ret = api.search_local(pattern=item['number'], count=1)
                if len(ret) == 0 or 'number' not in ret[0]:
                    continue
                ret = list(
                    map(lambda i: dict(**i, fees=model.DidVendor.fees(i['vendor_id'], i['number']) if (
                            'vendor_id' in i and i['vendor_id']) else None), ret))
                if ret[0]['fees'] and ret[0]['fees']['did_price']:
                    cost += ret[0]['fees']['did_price']
            if user.client.mode == 'postpay' and \
                    user.client.balance() - user.client.allowed_credit < cost and not user.client.unlimited_credit:
                raise Exception('Due to insufficient balance you can not order this DID.')
            if user.client.mode == 'prepay' and user.client.balance() < cost:
                raise Exception('Due to insufficient balance you can not order this DID.')
            # check balance
            ret = api.order_local(items=items, client_id=user.client_id)
            log.debug('ret:{}'.format(str(ret)[0:4096]))

            task_api = DidNumberUploadTaskCreate()
            dids = [item['number'] for item in ret if item['status'] == 'success']
            if client_id:
                client = model.Client.get(client_id)
                if client:
                    req.data['did_client_name'] = client.name
                if dids:
                    client.apply_mail('did_order_letter')
            # vendor_api = model.DidVendor.get(client_id)
            # if client_id:
            #     plan = model.DidBillingPlan
            #     billing_plan = plan.filter(plan.client_id == client_id).first()
            #     if billing_plan:
            #         req.data['client_billing_rule_name'] = billing_plan.name
            if client_id:
                didProduct = model.ClientDidProduct
                clientDidProduct = didProduct.filter(didProduct.client_id == client_id).first()
                if clientDidProduct:
                    item = model.DidProductItem
                    productItem = item.filter(item.did_product_id == clientDidProduct.did_product_id).first()
                    if productItem:
                        req.data['client_billing_rule_name'] = productItem.billing_rule_name
            req.data['paste'] = ','.join(dids)
            req.files = {}
            # req.data['vendor_billing_rule_name'] = obj.vendor_billing_rule_name
            # req.data['did_vendor_name'] = obj.did_vendor_name
            req.data['op_method'] = 'Assign'
            admin = model.User.get(1)
            req.context['user'] = admin
            task_api.on_post(req, resp, **kwargs)
            resp.body = task_api.resp.body
            resp.status = task_api.resp.status
            return

            # for item in ret:
            #     try:

            #         if item['status'] == 'success':
            #             obj = model.DidBillingRel.did_billing_rel_create(item['number'], client_id,
            #                                                              int(item['vendor_id']) if item[
            #                                                                  'vendor_id'] else None, user_name=user.name)
            #             for i in items:
            #                 if i['number'] == item['number'] and 'buy_billing_plan_id' in i:
            #                     obj.client_billing_rule_id = i.get('buy_billing_plan_id')
            #             obj.save()
            #             item['dnl_status'] = 'success'
            #             # assign webhook
            #             vendor_api = model.DidVendor.get(int(item['vendor_id']))
            #             if vendor_api and vendor_api.api_enabled and vendor_api.sms_enabled:
            #                 obj.is_sms = True
            #                 obj.save()
            #                 api = vendor_api.get_api()
            #                 url = '{}/smshook'.format(settings.API_URL)
            #                 number = item['number']
            #                 if len(number) == 10:
            #                     number = '1' + number
            #                 res = api.set_sms_receive_hook(number, url, client_id)
            #                 if not res['success']:
            #                     raise DidApiException(res)
            #                 item['sms_enable'] = 'success'
            #     except Exception as e:
            #         item['dnl_status'] = str(e)
            #     try:
            #         from api_dnl.tasks import do_did_pending_task
            #         do_did_pending_task.delay()
            #     except:
            #         pass
            ret = set_prefix(ret)
            self.set_response(resp, responses.SuccessResponse(data={
                'items': ret,
                'total': len(ret) if ret else 0, 'page': 0,
                'per_page': len(ret) if ret else 0
            }, scheme=schemes.ObjectScheme))

        except Exception as e:
            self.set_response(resp, OperationErrorResponse(e))
            return None
        pass


class DidApiAssigned(CustomGetAction):
    scheme_class = DidApiSearchCoverageScheme
    path_parameters = ()
    additional_responses = (responses.SuccessResponseObjectsList(payload_scheme_items=DidApiSearchCoverageGetScheme),)
    # query_parameters = [{'name': n, 'type': swagger.specify.get_field_type(f)} for n, f in
    # 					scheme_class._declared_fields.items()]
    is_client = True

    def on_get(self, req, resp, **kwargs):
        vendors = _get_all_vendors(True, 'all')
        if not vendors:
            self.set_response(resp, responses.ObjectNotFoundErrorResponse())
            return
        self.init_req(req)
        if not check_context(self, req, resp, **kwargs):
            self.set_response(resp, responses.ForbiddenErrorResponse())
            return
        user = self.get_user(self.req)
        # params = req.data
        params = dict(parse_qsl(req.query_string))
        # errors = self.scheme_class().validate(params)
        # if errors:
        # 	self.set_response(resp, responses.ValidationErrorResponse(data=errors))
        # 	return
        try:
            client_res_id = None
            client = user.client
            if client and client.egress_trunks:
                client_res_ids = [r.resource_id for r in client.egress_trunks]
            ret = []
            for vendor in vendors:
                vendor_id = vendor.id
                if not vendor_id:
                    continue
                vendor_res_id = None
                vendor_client = model.Client.get(vendor_id)
                if vendor_client and vendor_client.resource:
                    vendor_res_id = vendor_client.resource.resource_id
                    cls = model.DidBillingRel
                    q = cls.filter(and_(cls.end_date.is_(None), cls.egress_res_id.in_(client_res_ids),
                                        cls.ingress_res_id == vendor_res_id)).all()
                    ret = ret + [{'number': i.did, 'status': i.status, 'did_type': i.did_type, 'vendor_id': vendor_id}
                                 for i in q]
                    log.debug('search did assigned:{}'.format(str(ret)[0:4096]))
            self.set_response(resp, responses.SuccessResponse(data={
                'items': ret,
                'total': len(ret), 'page': 0,
                'per_page': len(ret)
            }, scheme=schemes.ObjectScheme))
        except Exception as e:
            self.set_response(resp, OperationErrorResponse(e))
            return None
        pass


class DidApiDisconnect(CustomPostAction):
    scheme_class = DidApiDisconnectScheme
    path_parameters = ()
    # additional_responses = (responses.SuccessResponseObjectsList(payload_scheme_items=DidApiOrderLocalGetScheme),)
    body_parameters = ('Release list', scheme_class,)
    is_client = False

    def on_post(self, req, resp, **kwargs):
        # vendors = _get_all_vendors(True, 'local')
        # if not vendors:
        #     self.set_response(resp, responses.ObjectNotFoundErrorResponse())
        #     return
        self.init_req(req)
        if not check_context(self, req, resp, **kwargs):
            self.set_response(resp, responses.ForbiddenErrorResponse())
            return
        user = self.get_user(self.req)
        client_id = str(user.client.client_id) if user.client else 0
        params = req.data
        # params['client_id'] = 'CLI_{}'.format(client_id)
        # params = dict(parse_qsl(req.query_string))
        errors_ = self.scheme_class().validate(params)
        if errors_:
            self.set_response(resp, responses.ValidationErrorResponse(data=errors_))
            return
        try:
            res_ids = [r.resource_id for r in user.client.egress_trunks] if user.client else None
            ret = []
            for item in params['items']:
                number = unprefix_2(item['number'])
                did_number = unprefix(item['number'])
                cls = model.DidRepository

                if res_ids:
                    did = model.DidRepository.filter(
                        and_(or_(cls.did == number, cls.did == did_number)
                             # ,cls.client_trunk_id.in_(res_ids)
                             # , cls.end_date.is_(None)
                             )).first()
                else:
                    did = model.DidRepository.filter(
                        and_(or_(cls.did == number, cls.did == did_number)
                             )).first()

                if did is None:
                    ret.append(dict(number=number, status='error', error='not own did', vendor_id=None))
                    continue
                else:
                    ret.append(dict(number=number, status='success', error='', vendor_id=None))
                # vendor_id = did.vendor_res.client_id
                # vendor_api = model.DidVendor.get(vendor_id)
                # if vendor_api and vendor_api.api_enabled:
                #     try:
                #         api = vendor_api.get_api()
                #         if did.is_toll_free:
                #             r = api.disconnect_toll_free(items=[dict(number=number)])[0]
                #         else:
                #             r = api.disconnect_local(items=[dict(number=number)])[0]
                #     except Exception as e:
                #         r = dict(number=number, status='error', error=str(e), vendor_id=vendor_id)
                #     # if did.end_date:
                #     #     r['warning'] = "did was already ended {}".format(did.end_date)
                #     ret.append(r)
                # else:
                #     r = dict(number=number, status='error', error='number not managed by any vendor', vendor_id=None)
                #     # if did.end_date:
                #     #     r['warning'] = "did was already ended {}".format(did.end_date)
                #     ret.append(r)
            # ret = set_prefix(ret)

            try:
                dids = ','.join([unprefix(item['number']) for item in ret])
                task_api = DidNumberUploadTaskCreate()
                admin = model.User.get(1)
                req.context['user'] = admin

                req.data['paste'] = dids
                req.files = {}
                req.data['op_method'] = 'Release'
                task_api.on_post(req, resp, **kwargs)
                resp.body = task_api.resp.body
                resp.status = task_api.resp.status 
            except Exception as e:
                self.set_response(resp, OperationErrorResponse(e))
                return None
            log.debug('ret:{}'.format(str(ret)[0:4096]))
            self.set_response(resp, responses.SuccessResponse(data={
                'items': ret,
                'total': len(ret), 'page': 0,
                'per_page': len(ret)
            }, scheme=schemes.ObjectScheme))

        except Exception as e:
            self.set_response(resp, OperationErrorResponse(e))
            return None
        pass


# endregion

# region CientPortalDidApiTollFree

class DidApiSearchTollFree(CustomGetAction):
    scheme_class = DidApiSearchTollFreeScheme
    path_parameters = ()
    additional_responses = (responses.SuccessResponseObjectsList(payload_scheme_items=DidApiSearchTollFreeGetScheme),)
    query_parameters = [{'name': n, 'type': swagger.specify.get_field_type(f)} for n, f in
                        scheme_class._declared_fields.items()]
    is_client = True

    def on_get(self, req, resp, **kwargs):

        self.init_req(req)
        if not check_context(self, req, resp, **kwargs):
            self.set_response(resp, responses.ForbiddenErrorResponse())
            return
        user = self.get_user(self.req)
        # params = req.data
        params = dict(parse_qsl(req.query_string))
        params['client_id'] = user.client_id
        errors = self.scheme_class().validate(params)
        if errors:
            self.set_response(resp, responses.ValidationErrorResponse(data=errors))
            return
        is_sms = params.pop('is_sms', False)
        vendors = _get_all_vendors(True, 'toll_free', is_sms)
        if not vendors:
            self.set_response(resp, responses.ValidationErrorResponse(data={'none': 'no one available vendor'}))
            return
        err = []
        for v in vendors:
            try:
                v.get_api()
            except Exception as e:
                msg = 'Vendor {} has  wrong api settings {}'.format(v.vendor.name, e)
                err.append(msg)
        if err:
            self.set_response(resp, responses.ValidationErrorResponse(data={'none': str(err)}))
            return False
        try:
            api = DidAPIMuliVendor(vendors)
            page = int(params.pop('page', '0'))
            per_page = int(params.pop('per_page', '10'))
            params['count'] = 9999
            ret = api.search_toll_free(**params)
            log.debug('multi vendor search toll_free:{}'.format(ret))
            ret = list(
                map(lambda i: i if 'fees' in i else (
                    dict(**i, fees=model.DidVendor.fees(user.client_id, i['number']))), ret))

            didProduct = model.ClientDidProduct
            clientDidProducts = (
                didProduct
                .filter(didProduct.client_id == user.client_id, didProduct.did_product_id != None)
                .with_entities(didProduct.did_product_id)
                .all()
            )
            did_product_ids = [row.did_product_id for row in clientDidProducts]

            didProductItem = model.DidProductItem
            did_product_items = (
                didProductItem
                .filter(
                    didProductItem.did_product_id.in_(did_product_ids),
                    didProductItem.type == 2  # Only "Toll-Free"
                )
                .with_entities(didProductItem.country, didProductItem.type)
                .all()
            )
            allowed_countries = [row.country for row in did_product_items]

            def _is_us_number(number):
                prefixes = ['1800', '1877', '1866', '1855', '1844', '1833', '1888']
                if len(number) == 11 and number[:4] in prefixes:
                    return True
                return False

            if 'US' in allowed_countries:
                ret = [item for item in ret if item['country'] == 'US' or _is_us_number(item['number'])]
            else:
                ret = [item for item in ret if item['country'] in allowed_countries]
            
            numbers = [item["number"] for item in ret]
            did_repo_entries = (
                model.get_db().session.query(model.DidRepository.did, model.DidRepository.id)
                .filter(model.DidRepository.did.in_(numbers))
                .all()
            )
            did_repo_map = {row.did: row.id for row in did_repo_entries}
            for item in ret:
                item.pop("vendor_id", None)
                item["did_repository_id"] = did_repo_map.get(item["number"])

            ret = set_prefix(ret)
            ret = [{k: v for k, v in d.items() if k != 'rate_per_min'} for d in ret]
            log.debug('search toll_free + fees :{}'.format(ret))
            self.set_response(resp, responses.SuccessResponse(data={
                'items': ret[page*per_page:(page+1)*per_page],
                'total': len(ret), 'page': page,
                'per_page': per_page
            }, scheme=schemes.ObjectScheme))

        except Exception as e:
            import traceback
            trace = traceback.format_exc()
            log.error(trace)
            log.debug('search did error {}'.format(e))

            self.set_response(resp, OperationErrorResponse(e))
            return None
        pass


class DidApiCoverageTollFree(CustomGetAction):
    scheme_class = DidApiSearchCoverageScheme
    path_parameters = ()
    additional_responses = (responses.SuccessResponseObjectsList(payload_scheme_items=DidApiSearchCoverageGetScheme),)
    query_parameters = [{'name': n, 'type': swagger.specify.get_field_type(f)} for n, f in
                        scheme_class._declared_fields.items()]
    is_client = True

    def on_get(self, req, resp, **kwargs):

        vendors = _get_all_vendors(True, 'toll_free')
        if not vendors:
            self.set_response(resp, responses.ObjectNotFoundErrorResponse())
            return
        self.init_req(req)
        if not check_context(self, req, resp, **kwargs):
            self.set_response(resp, responses.ForbiddenErrorResponse())
            return
        user = self.get_user(self.req)
        # params = req.data
        params = dict(parse_qsl(req.query_string))
        errors = self.scheme_class().validate(params)
        if errors:
            self.set_response(resp, responses.ValidationErrorResponse(data=errors))
            return
        try:
            api = DidAPIMuliVendor(vendors)
            ret = api.coverage_toll_free(**params)
            log.debug('ret:{}'.format(str(ret)[0:4096]))
            self.set_response(resp, responses.SuccessResponse(data={
                'items': ret,
                'total': len(ret), 'page': 0,
                'per_page': len(ret)
            }, scheme=schemes.ObjectScheme))

        except Exception as e:
            self.set_response(resp, OperationErrorResponse(e))
            return None
        pass


class DidApiSearchOrderTollFree(CustomPostAction):
    scheme_class = DidApiSearchOrderLocalScheme
    path_parameters = ()
    additional_responses = (responses.SuccessResponseObjectsList(payload_scheme_items=DidApiOrderLocalGetScheme),)
    body_parameters = ('Order parameters', scheme_class,)
    is_client = True

    def on_post(self, req, resp, **kwargs):
        vendors = _get_all_vendors(True, 'toll_free')
        if not vendors:
            self.set_response(resp, responses.ObjectNotFoundErrorResponse())
            return
        self.init_req(req)
        if not check_context(self, req, resp, **kwargs):
            self.set_response(resp, responses.ForbiddenErrorResponse())
            return
        user = self.get_user(self.req)
        client_id = str(user.client.client_id)
        params = req.data
        params['client_id'] = 'CLI_{}'.format(client_id)
        # params = dict(parse_qsl(req.query_string))
        errors = self.scheme_class().validate(params)
        if errors:
            self.set_response(resp, responses.ValidationErrorResponse(data=errors))
            return
        try:
            api = DidAPIMuliVendor(vendors)
            ret = api.search_order_toll_free(**params)
            log.debug('ret:{}'.format(str(ret)[0:4096]))
            for item in ret:
                try:
                    if item['status'] == 'success':
                        obj = model.DidBillingRel.did_billing_rel_create(item['number'], client_id, item['vendor_id'],
                                                                         buy_billing_plan_id=None, user_name=user.name)
                        obj.save()
                        item['dnl_status'] = 'success'
                        # assign webhook
                        vendor_api = model.DidVendor.get(int(item['vendor_id']))
                        api = vendor_api.get_api()
                        url = '{}/smshook'.format(settings.API_URL)
                        number = item['number']
                        if len(number) == 10:
                            number = '1' + number
                        res = api.set_sms_receive_hook(number, url, client_id)
                        if not res['success']:
                            raise DidApiException(res)
                except Exception as e:
                    item['dnl_status'] = str(e)
            ret = set_prefix(ret)
            self.set_response(resp, responses.SuccessResponse(data={
                'items': ret,
                'total': len(ret), 'page': 0,
                'per_page': len(ret)
            }, scheme=schemes.ObjectScheme))
        except Exception as e:
            self.set_response(resp, OperationErrorResponse(e))
            return None
        pass


class DidApiOrderTollFree(CustomPostAction):
    scheme_class = DidApiOrderTollFreeScheme
    path_parameters = ()
    additional_responses = (responses.SuccessResponseObjectsList(payload_scheme_items=DidApiOrderLocalGetScheme),)
    body_parameters = ('Order parameters', scheme_class,)
    is_client = True

    def on_post(self, req, resp, **kwargs):

        # vendors = [VendorManual()]
        vendors = _get_all_vendors(True, 'toll_free')
        self.init_req(req)
        if not check_context(self, req, resp, **kwargs):
            self.set_response(resp, responses.ForbiddenErrorResponse())
            return
        user = self.get_user(self.req)
        client_id = str(user.client.client_id)
        params = req.data
        # params['client_id'] = 'CLI_{}'.format(client_id)
        # params = dict(parse_qsl(req.query_string))
        errors = self.scheme_class().validate(params)
        if errors:
            self.set_response(resp, responses.ValidationErrorResponse(data=errors))
            return
        try:
            api = DidAPIMuliVendor(vendors)
            # api = DidAPIMuliVendor([vendors[0]])
            for item in params['items']:
                if 'number' not in item:
                    repo = model.get_db().session.query(model.DidRepository).get(item['did_repository_id'])
                    if repo:
                        item['number'] = repo.did
            items = unprefix_2(params['items'])
            # check balance
            cost = 0.0
            for item in items:
                ret = api.search_toll_free(pattern=item['number'], count=1)
                if len(ret) == 0 or 'number' not in ret[0]:
                    continue
                ret = list(
                    map(lambda i: dict(**i, fees=model.DidVendor.fees(i['vendor_id'], i['number']) if (
                            'vendor_id' in i and i['vendor_id']) else None), ret))
                if ret[0]['fees'] and ret[0]['fees']['did_price']:
                    cost += ret[0]['fees']['did_price']
            if user.client.mode == 'postpay' and \
                    user.client.balance() - user.client.allowed_credit < cost and not user.client.unlimited_credit:
                raise Exception('Due to insufficient balance you can not order this DID.')
            if user.client.mode == 'prepay' and user.client.balance() < cost:
                raise Exception('Due to insufficient balance you can not order this DID.')
                # raise Exception('Due to insufficient balance you can not order this DID.'.format(user.client.balance(), cost))
            # check balance
            ret = api.order_toll_free(items=items, client_id=user.client_id)
            log.debug('ret:{}'.format(str(ret)[0:4096]))

            if any(item['status'] == 'error' for item in ret):
                self.set_response(resp, responses.ValidationErrorResponse(data="There is an error in the input data: {}".format(
                    next(item for item in ret if item['status'] == 'error')['error']
                )))
                return

            task_api = DidNumberUploadTaskCreate()
            dids = [item['number'] for item in ret if item['status'] == 'success']
            if client_id:
                client = model.Client.get(client_id)
                if client:
                    req.data['did_client_name'] = client.name
            # vendor_api = model.DidVendor.get(client_id)
            # if client_id:
            #     plan = model.DidBillingPlan
            #     billing_plan = plan.filter(plan.client_id == client_id).first()
            #     if billing_plan:
            #         req.data['client_billing_rule_name'] = billing_plan.name
            if client_id:
                didProduct = model.ClientDidProduct
                clientDidProduct = didProduct.filter(didProduct.client_id == client_id).first()
                if clientDidProduct:
                    item = model.DidProductItem
                    productItem = item.filter(item.did_product_id == clientDidProduct.did_product_id).first()
                    if productItem:
                        req.data['client_billing_rule_name'] = productItem.billing_rule_name
            req.data['paste'] = ','.join(dids)
            req.files = {}
            # req.data['vendor_billing_rule_name'] = obj.vendor_billing_rule_name
            # req.data['did_vendor_name'] = obj.did_vendor_name
            req.data['op_method'] = 'Assign'
            admin = model.User.get(1)
            req.context['user'] = admin
            task_api.on_post(req, resp, **kwargs)
            resp.body = task_api.resp.body
            resp.status = task_api.resp.status
            return

            # for item in ret:
            #     try:
            #         if item['status'] == 'success':
            #             obj = model.DidBillingRel.did_billing_rel_create(item['number'], client_id,
            #                                                              int(item['vendor_id']),
            #                                                              buy_billing_plan_id=None, user_name=user.name)
            #             obj.save()
            #             item['dnl_status'] = 'success'
            #             # assign webhook
            #             vendor_api = model.DidVendor.get(int(item['vendor_id']))
            #             if vendor_api and vendor_api.api_enabled and vendor_api.sms_enabled:
            #                 obj.is_sms = True
            #                 obj.save()
            #                 api = vendor_api.get_api()
            #                 url = '{}/smshook'.format(settings.API_URL)
            #                 number = item['number']
            #                 if len(number) == 10:
            #                     number = '1' + number
            #                 res = api.set_sms_receive_hook(number, url, client_id)
            #                 if not res['success']:
            #                     raise DidApiException(res)
            #                 item['sms_enable'] = 'success'
            #     except Exception as e:
            #         item['dnl_status'] = str(e)
            ret = set_prefix(ret)
            self.set_response(resp, responses.SuccessResponse(data={
                'items': ret,
                'total': len(ret) if ret else 0, 'page': 0,
                'per_page': len(ret) if ret else 0
            }, scheme=schemes.ObjectScheme))

        except Exception as e:
            self.set_response(resp, OperationErrorResponse(e))
            return None
        pass


class DidApiAssignedTollFree(CustomGetAction):
    scheme_class = DidApiSearchCoverageScheme
    path_parameters = ()
    additional_responses = (responses.SuccessResponseObjectsList(payload_scheme_items=DidApiSearchCoverageGetScheme),)
    # query_parameters = [{'name': n, 'type': swagger.specify.get_field_type(f)} for n, f in
    # 					scheme_class._declared_fields.items()]
    is_client = True

    def on_get(self, req, resp, **kwargs):
        vendors = _get_all_vendors(True, 'toll_free')
        if not vendors:
            self.set_response(resp, responses.ObjectNotFoundErrorResponse())
            return
        self.init_req(req)
        if not check_context(self, req, resp, **kwargs):
            self.set_response(resp, responses.ForbiddenErrorResponse())
            return
        user = self.get_user(self.req)
        # params = req.data
        params = dict(parse_qsl(req.query_string))
        # errors = self.scheme_class().validate(params)
        # if errors:
        # 	self.set_response(resp, responses.ValidationErrorResponse(data=errors))
        # 	return
        try:
            client_res_id = None
            client = user.client
            if client and client.egress_trunks:
                client_res_ids = [r.resource_id for r in client.egress_trunks]
            ret = []
            for vendor in vendors:
                vendor_id = vendor.id
                if not vendor_id:
                    continue
                vendor_res_id = None
                vendor_client = model.Client.get(vendor_id)
                if vendor_client and vendor_client.resource:
                    vendor_res_id = vendor_client.resource.resource_id

                    cls = model.DidBillingRel
                    q = cls.filter(and_(cls.is_toll_free, cls.egress_res_id.in_(client_res_ids),
                                        cls.ingress_res_id == vendor_res_id)).all()
                    ret = ret + [{'number': i.did, 'vendor_id': vendor_id} for i in q]
                    log.debug('search did assigned:{}'.format(str(ret)[0:4096]))
            self.set_response(resp, responses.SuccessResponse(data={
                'items': ret,
                'total': len(ret), 'page': 0,
                'per_page': len(ret)
            }, scheme=schemes.ObjectScheme))

        except Exception as e:
            self.set_response(resp, OperationErrorResponse(e))
            return None
        pass


# endregion
# endregion
# region +++Sms+++


class SmsCreate(DnlCreate):
    scheme_class = SmsScheme
    model_class = model.Sms
    entity = 'Sms'
    path_parameters = ()
    security = (DEFAULT_SECURITY)
    restrict = ()

    def before_create(self, obj, **kwargs):
        obj.direction = 'sent'
        user = self.get_user(self.req)
        if not user.is_admin:
            obj.client_id = user.client_id
        obj.created_on = datetime.now(UTC)
        if len(obj.receiver) != 12 and obj.receiver[0] != '+':
            raise ValidationError({'receiver': 'wrong format!'})
        if len(obj.sender) == 12 and obj.sender[0] != '+':
            raise ValidationError({'sender': 'wrong format!'})
        cls = model.DidBillingRel
        obj.receiver = unprefix(obj.receiver)
        obj.sender = unprefix(obj.sender)
        q = cls.filter(and_(cls.end_date.is_(None), or_(cls.did == obj.sender, cls.did == obj.sender[1:]))).first()
        if not q:
            raise ValidationError({'sender': ['sender {} not in assigned dids'.format(obj.sender)]})
        if not q.vendor_res.client.status:
            raise ValidationError(
                {'sender': [
                    'sender assigned vendor {} not active (check vendor status)'.format(q.vendor_res.client.name)]})
        vendor_api = q.vendor_res.client.vendor_api
        if not vendor_api:
            raise ValidationError({'sender': ['sender {} assigned did has no vendor api'.format(obj.sender)]})
        obj.vendor_id = vendor_api.id
        api = vendor_api.get_api()
        # try:
        #     ret = api.send_sms(obj.sender, obj.receiver, obj.message)
        #     log.debug('send_sms result:{}'.format(ret))
        # except Exception as e:
        #     log.debug('send_sms failed:{}'.format(e))
        # ##billing
        # if vendor_api.rate_per_sms_sent:
        #     model.ClientBalanceOperationAction(client_id=user.client_id, action=3, balance=vendor_api.rate_per_sms_sent,
        #                                        create_by='sms_sent', create_time=datetime.now(UTC)).save()
        # rid = ret['result']['referenceId']
        # obj.reference_id = rid
        # if not ret['success']:
        #     if obj.dlr_hook and obj.dlr_hook.dlr_fail_url:
        #     raise DidApiException(ret)
        return obj

    def after_create(self, object_id, req, resp, **kwargs):
        obj = self.model_class.get(object_id)
        obj.send_dlr(sms_id=obj.id, sender=obj.sender, receiver=obj.receiver, message=obj.message,
                     status='initial')
        cls = model.DidBillingRel
        did = cls.filter(and_(cls.end_date.is_(None), or_(cls.did == obj.sender, cls.did == obj.sender[1:]))).first()
        if not did:
            raise ValidationError({'sender': ['sender {} not in assigned dids'.format(obj.sender)]})
        vendor_api = did.vendor_res.client.vendor_api
        if not vendor_api:
            raise ValidationError({'sender': ['sender {} assigned did has no vendor api'.format(obj.sender)]})
        if not vendor_api.sms_enabled:
            raise ValidationError({'sender': ['sender {} vendor api not sms enabled'.format(obj.sender)]})
        api = vendor_api.get_api()
        try:
            ret = api.sendhi_sms(obj.sender, obj.receiver, obj.message)
            log.debug('send_sms result:{}'.format(ret))
        except Exception as e:
            obj.send_fail_dlr(sms_id=obj.id, sender=obj.sender, receiver=obj.receiver, message=obj.message,
                              status='fail', error=str(e))
            log.debug('send_sms failed:{}'.format(e))
            obj.stat = 'FAILED'
            obj.save()
            # obj.delete()
            # raise

        rid = ret['result']['referenceId']
        # try:
        # 	obj.reference_id = str(int(rid, 0))
        # except ValueError:
        # 	try:
        # 		obj.reference_id = str(int('0x' + rid, 0))
        # 	except:
        # 		obj.reference_id = rid
        if not ret['success']:
            obj.send_fail_dlr(sms_id=obj.id, sender=obj.sender, receiver=obj.receiver, message=obj.message,
                              status='fail', error=str(ret))
            obj.stat = 'FAILED'
            obj.save()
            # obj.delete()
            raise Exception(str(ret))

        else:
            obj.reference_id = rid
            obj.stat = 'ACCEPTD'
            obj.save()
            ##billing
            if did.buy_billing_plan and did.sell_billing_plan:
                user = self.get_user(self.req)
                obj.cost = did.buy_billing_plan.rate_per_sms_sent or 0.0
                obj.vendor_cost = did.sell_billing_plan.rate_per_sms_sent or 0.0
                obj.save()
                model.ClientBalanceOperationAction(client_id=user.client_id, action=3,
                                                   balance=obj.cost,
                                                   create_by='sms_sent', create_time=datetime.now(UTC)).save()
            obj.send_dlr(sms_id=obj.id, sender=obj.sender, receiver=obj.receiver, message=obj.message,
                         status='accepted')


class SmsResource(DnlResource):
    model_class = model.Sms
    scheme_class = SmsScheme
    scheme_class_get = SmsSchemeGet
    scheme_class_modify = SmsSchemeModify
    entity = 'Sms'
    id_field = 'id'
    security = (DEFAULT_SECURITY)
    path_parameters = ()
    restrict = ()


class SmsList(DnlList):
    scheme_class = SmsSchemeGet
    model_class = model.Sms
    entity_plural = 'Smses'
    path_parameters = ()
    security = (DEFAULT_SECURITY)
    restrict = ()

    def modify_query_from_filtering_for_list(self, filtering, **kwargs):
        filt, ret = super().modify_query_from_filtering_for_list(filtering, **kwargs)
        user = self.get_user(self.req)
        if not user.is_admin:
            cls = self.model_class
            ret = ret.filter(cls.client_id == user.client_id)
        return filt, ret


# endregion ---Sms---

# region +++NpaList+++
class NpaList(CustomGetAction):
    scheme_class = NpaSearchScheme
    path_parameters = ()
    additional_responses = (responses.SuccessResponseObjectsList(payload_scheme_items=NpaSearchScheme),)
    query_parameters = [{'name': n, 'type': swagger.specify.get_field_type(f)} for n, f in
                        scheme_class._declared_fields.items()]
    is_client = True

    def on_get(self, req, resp, **kwargs):
        self.init_req(req)
        if not check_context(self, req, resp, **kwargs):
            self.set_response(resp, responses.ForbiddenErrorResponse())
            return
        user = self.get_user(self.req)
        # params = req.data
        params = dict(parse_qsl(req.query_string))
        npa = params.pop('npa', None)
        try:
            cls = model.JurisdictionPrefix
            fld = func.substring(cast(cls.prefix, model.String), 2, 3)
            q = model.get_db().session.query(fld.label('npa'))
            if npa:
                q = q.filter(fld.like(npa.replace('*', '%')))
            q = q.group_by(fld)
            ret = [{'npa': i.npa} for i in q]
            self.set_response(resp, responses.SuccessResponse(data={
                'items': ret,
                'total': len(ret), 'page': 0,
                'per_page': len(ret)
            }, scheme=schemes.ObjectScheme))

        except Exception as e:
            self.set_response(resp, OperationErrorResponse(e))
            return None
        pass


class LataList(CustomGetAction):
    scheme_class = LataSearchScheme
    path_parameters = ()
    additional_responses = (responses.SuccessResponseObjectsList(payload_scheme_items=LataSearchScheme),)
    query_parameters = [{'name': n, 'type': swagger.specify.get_field_type(f)} for n, f in
                        scheme_class._declared_fields.items()]
    is_client = True

    def on_get(self, req, resp, **kwargs):
        self.init_req(req)
        if not check_context(self, req, resp, **kwargs):
            self.set_response(resp, responses.ForbiddenErrorResponse())
            return
        user = self.get_user(self.req)
        # params = req.data
        params = dict(parse_qsl(req.query_string))
        lata = params.pop('lata', None)
        try:
            cls = model.JurisdictionPrefix
            fld = cls.lata
            q = model.get_db().session.query(fld.label('lata'))
            if lata:
                q = q.filter(fld.like(lata.replace('*', '%')))
            q = q.group_by(fld)
            ret = [{'lata': i.lata} for i in q]
            self.set_response(resp, responses.SuccessResponse(data={
                'items': ret,
                'total': len(ret), 'page': 0,
                'per_page': len(ret)
            }, scheme=schemes.ObjectScheme))

        except Exception as e:
            self.set_response(resp, OperationErrorResponse(e))
            return None
        pass


class StateList(CustomGetAction):
    scheme_class = StateSearchScheme
    path_parameters = ()
    additional_responses = (responses.SuccessResponseObjectsList(payload_scheme_items=StateSearchScheme),)
    query_parameters = [{'name': n, 'type': swagger.specify.get_field_type(f)} for n, f in
                        scheme_class._declared_fields.items()]
    is_client = True

    def on_get(self, req, resp, **kwargs):
        self.init_req(req)
        if not check_context(self, req, resp, **kwargs):
            self.set_response(resp, responses.ForbiddenErrorResponse())
            return
        user = self.get_user(self.req)
        # params = req.data
        params = dict(parse_qsl(req.query_string))
        state = params.pop('state', None)
        try:
            cls = model.JurisdictionPrefix
            fld = cls.jurisdiction_name
            q = model.get_db().session.query(fld.label('state'))
            if state:
                q = q.filter(fld.like(state.replace('*', '%')))
            q = q.group_by(fld)
            ret = [{'state': i.state} for i in q]
            self.set_response(resp, responses.SuccessResponse(data={
                'items': ret,
                'total': len(ret), 'page': 0,
                'per_page': len(ret)
            }, scheme=schemes.ObjectScheme))

        except Exception as e:
            self.set_response(resp, OperationErrorResponse(e))
            return None
        pass


# endregion ---NpaList--

# region +++WebHook+++
class WebHookCreate(DnlCreate):
    scheme_class = WebHookScheme
    model_class = model.WebHook
    entity = 'WebHook'
    path_parameters = ()
    security = (DEFAULT_SECURITY)
    restrict = ()
    is_client = True

    def before_create(self, obj, **kwargs):
        user = self.get_user(self.req)
        # obj.created_by=user.name
        obj.created_on = datetime.now(UTC)
        obj.client_id = user.client_id
        res_ids = [r.resource_id for r in user.client.egress_trunks]
        if obj.number != '':
            cls = model.DidBillingRel
            did = model.DidBillingRel.filter(
                and_(cls.did == unprefix(obj.number), cls.egress_res_id.in_(res_ids))).first()
            if did is None:
                raise ValidationError({'number': ['you not own did {}'.format(obj.number)]})
            vendor_id = did.vendor_res.client_id
            vendor_api = model.DidVendor.get(vendor_id)
            if vendor_api and vendor_api.api_enabled and vendor_api.sms_enabled:
                api = vendor_api.get_api()
                url = '{}/smshook'.format(settings.API_URL)
                number = obj.number
                if len(number) == 10:
                    number = '1' + number
                res = api.set_sms_receive_hook(number, url, user.client_id)
                if not res['success']:
                    raise DidApiException(res)
                pass
            else:
                raise ValidationError({'number': ['not sms vendor api enabled for number {}'.format(obj.number)]})
        return obj


class WebHookResource(DnlResource):
    model_class = model.WebHook
    scheme_class = WebHookScheme
    scheme_class_get = WebHookSchemeGet
    scheme_class_modify = WebHookSchemeModify
    entity = 'WebHook'
    id_field = 'id'
    security = (DEFAULT_SECURITY)
    path_parameters = ()
    restrict = ()
    is_client = True


class WebHookList(DnlList):
    scheme_class = WebHookSchemeGet
    model_class = model.WebHook
    entity_plural = 'WebHooks'
    path_parameters = ()
    security = (DEFAULT_SECURITY)
    restrict = ()
    is_client = True

    def modify_query_from_filtering_for_list(self, filtering, **kwargs):
        filt, ret = super().modify_query_from_filtering_for_list(filtering, **kwargs)
        user = self.get_user(self.req)
        if not user.is_admin:
            cls = self.model_class
            ret = ret.filter(cls.client_id == user.client_id)  # TODO:filter for user
        return filt, ret


# endregion ---WebHook---
# region +++ApiKey+++
class ApiKeyCreate(CustomPostAction):
    model_class = model.Client
    scheme_class = ApiKeyScheme
    path_parameters = ()
    additional_responses = (responses.SuccessResponseObjectInfo(payload_scheme=ClientApiKeyGetScheme),)
    # query_parameters = [{'name': n, 'type': swagger.specify.get_field_type(f)} for n, f in scheme_class._declared_fields.items()]
    is_client = True

    def apply(self, obj, req, resp, **kwargs):
        if obj:
            token = obj.user.create_api_key()
            obj.user.api_key.save()
            self.set_response(resp, responses.SuccessResponse(data={
                'token': token,
                'expired_on': str(obj.user.api_key.expired_on.date())
            }, scheme=schemes.ObjectScheme))
            return False


class ApiKeyDelete(CustomDeleteAction):
    model_class = model.Client
    scheme_class = ApiKeyScheme
    path_parameters = ()
    # additional_responses = (responses.SuccessResponseObjectInfo(payload_scheme=ApiKeyGetScheme),)
    # query_parameters = [{'name': n, 'type': swagger.specify.get_field_type(f)} for n, f in scheme_class._declared_fields.items()]
    is_client = True

    def apply(self, obj, req, resp, **kwargs):
        if obj:
            token = obj.user.api_key.delete()
            self.set_response(resp, responses.SuccessResponseJustOk())
            return False
        self.set_response(resp, responses.ObjectNotFoundErrorResponse())
        return False

# endregion ---ApiKey--
