from urllib.parse import urlencode, parse_qsl
from datetime import datetime, timedelta
import time
from pytz import UTC
from dateutil.parser import parse as parse_datetime
from sqlalchemy.exc import InternalError, IntegrityError, DataError
from sqlalchemy import and_, or_, inspect, alias, text, func, select, case, cast
from sqlalchemy.orm import aliased
from api_dnl import model
import falcon
from falcon_rest import conf
from falcon_rest.logger import log
from falcon_rest import schemes
from falcon_rest.responses import responses
from falcon_rest.responses import errors
from falcon_rest.resources.resources import swagger, ATTRIBUTE_ERROR_RE
from falcon_rest.helpers import check_permission

from api_dnl.resources import check_context, DnlList, ValidationError, DnlCreate, CustomGetAction, CustomPostAction, \
    CustomDeleteAction, CustomPatchAction, DnlResourceAll, DnlResource
from api_dnl.schemes.cdr import CdrReportDetailGetScheme, CdrReportGetScheme, CdrAsyncTaskScheme, \
    CdrAsyncTaskSchemeGet, CdrAsyncTaskEmailScheme, CdrReportDetailEarliestGetScheme, DailyCdrCloudUpdateScheme, \
    DailyCdrCloudUpdateSchemeModify
from api_dnl.schemes.common import ObjectUpdatedScheme
from api_dnl.scheme import SendCdrDirectScheme
from api_dnl.view import DEFAULT_SECURITY, OperationErrorResponse
import json
import requests

CDR_DEBUG = False


class SuccessReportResponse(responses.BaseResponse):
    description = 'Successfully executed'

    # scheme = CdrReportGroppedResponseScheme
    def get_response_body(self):
        return self.data


def gen(q, res, _from, total):
    window_size = 10  # or whatever limit you like
    window_idx = 0
    _count = 0
    d = []

    yield b'{"success":true, "data":['
    if type(q) == type([]):
        yield ('],"total":{},"from":{},"count":{}}}\r\n\r\n'.format(total, _from, _count)).encode()
        return
    # all_things = q.all()
    try:
        for thing in q.yield_per(500):
            _count += 1
            window_idx += 1
            d.append(thing)
            if window_idx == window_size:

                data = res.scheme_class().dump(d, many=True).data
                # data[0]["_comment"] = "\r\n\r\n--page--{}".format(int(total / window_size))
                if data:
                    dump = "\r\n\r\n" + json.dumps(data)[1:-1]
                    if _count > window_size:
                        dump = ',' + dump
                    yield dump.encode()
                d = []
                window_idx = 0
        if d:
            data = res.scheme_class().dump(d, many=True).data
            dump = json.dumps(data)[1:-1]
            if _count > window_size:
                dump = ',' + dump
            yield dump.encode()
        if CDR_DEBUG:
            log.debug('cdr_get ended ] "total":{},"from":{},"count":{}}}'.format(total, _from, _count))
        yield ('],"total":{},"from":{},"count":{}}}\r\n\r\n'.format(total, _from, _count)).encode()
    except Exception as e:
        if CDR_DEBUG:
            log.debug('cdr_get UNEXPECTED ERROR: {}'.format(e))
        yield str.encode('\r\n\r\ncdr_get UNEXPECTED ERROR: {}\r\n\r\n'.format(e))
    finally:
        model.get_db().session.commit()


def gen_csv(q, res, resp, _from, total):
    try:
        import csv
        import io
        window_size = 10  # or whatever limit you like
        window_idx = 0
        _count = 0

        if type(q) == type([]):
            yield ('\r\n\r\n').encode()
            return
        d = []
        # yield b'\r\n'
        # all_things = q.all()
        for thing in q.yield_per(500):
            _count += 1
            window_idx += 1
            d.append(thing)
            if window_idx == window_size:
                data = res.scheme_class().dump(d, many=True).data
                csvfile = io.StringIO()
                if len(data):
                    fieldnames = data[0].keys()
                    fn = model.DailyCdrField.convert_fields_to_user_friendly(fieldnames)
                    writer = csv.DictWriter(csvfile, fieldnames=fieldnames)
                    if _count <= window_size:
                        writer.writerow(dict(zip(fieldnames, fn)))
                        # writer.writeheader()
                    writer.writerows(data)
                    yield str.encode(csvfile.getvalue())
                d = []
                window_idx = 0
        if d:  # rest piece
            data = res.scheme_class().dump(d, many=True).data
            csvfile = io.StringIO()
            if len(data):
                fieldnames = data[0].keys()
                fn = model.DailyCdrField.convert_fields_to_user_friendly(fieldnames)
                writer = csv.DictWriter(csvfile, fieldnames=fieldnames)
                if _count <= window_size:
                    writer.writerow(dict(zip(fieldnames, fn)))
                    # writer.writeheader()
                writer.writerows(data)
                yield str.encode(csvfile.getvalue())

        # yield ('],"total":{},"from":{},"count":{}}}\r\n\r\n'.format(count, _from, total)).encode()
        # finish
        yield ('\r\n\r\n').encode()
    except Exception as e:
        # model.get_db().session.rollback()
        if CDR_DEBUG:
            log.debug('cdr_get UNEXPECTED ERROR: {}'.format(e))
        yield str.encode('\r\n\r\ncdr_get UNEXPECTED ERROR: {}\r\n\r\n'.format(e))
    finally:
        model.get_db().session.commit()


class CdrReportList(DnlList):
    scheme_class = CdrReportGetScheme
    model_class = model.ClientCdr
    entity_plural = 'CdrReport'
    path_parameters = ()
    security = (DEFAULT_SECURITY)
    restrict = ()

    def get_spec(self):
        sfields_names, search_fields_list = self.get_fields_from_scheme(self.scheme_class)
        qfields_names, query_fields_list = self.get_query_fields_from_scheme(self.scheme_class)
        fields_names = [f.name for f in inspect(self.model_class).columns]
        fields_list = [{'name': f.name} for f in inspect(self.model_class).columns]
        spec = swagger.specify.get_spec(
            method='get', description='Gets {}'.format(self.entity_plural.lower()),
            path_parameters=self.path_parameters,
            query_parameters=[
                                 # {'name': 'field', 'type': 'string', 'description': 'fields set to select delimited by comma'},
                                 {'name': 'from', 'type': 'integer', 'description': 'start position'},
                                 {'name': 'count', 'type': 'integer', 'description': 'output limit'},
                                 {'name': 'sort', 'enum': fields_names, 'description': 'sort by fields'},
                                 {'name': 'order', 'enum': ['asc', 'desc']}
                             ] + search_fields_list + query_fields_list + fields_list,
            responses=(
                          responses.SuccessResponseObjectsList(payload_scheme_items=self.scheme_class),
                          responses.AttributeNotExitsErrorResponse()
                      ) + self.get_additional_responses(method='get'),
            security=self.get_security(method='get')
        )
        spec['get']['produces'] = ['application/json', 'text/csv', 'text/xml']
        return spec

    def _on_get(self, req, resp, **kwargs):
        from falcon_rest import schemes
        from falcon_rest.helpers import check_permission

        if (not hasattr(self, 'no_auth_needed') or self.no_auth_needed == False) and not check_permission(self, req,
                                                                                                          'list'):
            self.set_response(
                resp, responses.ForbiddenErrorResponse(data=errors.AuthErrors.Forbidden)
            )
            return
        try:
            parsed_qs = dict(parse_qsl(req.query_string))
            fmt = parsed_qs.pop('format', 'json')
            print_only_amount = parsed_qs.pop('print_only_amount', False)
            objects_list, _count, _from, total, fields = self.get_objects_list(req, self.model_class, **kwargs)
            if print_only_amount:
                if fmt == 'csv':
                    resp.body = '''total\n{}'''.format(total).encode()
                else:
                    resp.body = '''{{"success":true,"total":{}}}'''.format(total).encode()

                resp.content_type = 'text/csv' if fmt == 'csv' else 'application/json'
                resp.status = falcon.HTTP_200
                return

            if fmt != 'csv':
                resp_stream = gen(objects_list, self, resp, _from, total)
            else:
                resp_stream = gen_csv(objects_list, self, resp, _from, total)
            resp.content_type = 'text/csv' if fmt == 'csv' else 'application/json'
            resp.stream = resp_stream

            resp.status = falcon.HTTP_PARTIAL_CONTENT
            # self.set_response(
            #     resp, responses.SuccessResponse(
            #         data={
            #             'items': self.scheme_class(only=fields).dump(objects_list, many=True).data,
            #             # 'items': [i for i in objects_list],
            #             'total': total, 'page': page, 'per_page': per_page
            #         },
            #         scheme=schemes.ObjectScheme
            #     )
            # )
        except AttributeError as e:
            log.debug(f'ERROR - {e}')
            model.get_db().session.rollback()
            match = ATTRIBUTE_ERROR_RE.search(str(e))
            self.set_response(
                resp, responses.AttributeNotExitsErrorResponse(
                    data=match.group(1) if match is not None else None
                )
            )
        except (InternalError, IntegrityError, DataError) as e:
            import traceback
            model.get_db().session.rollback()
            log.error(traceback.format_exc())
            self.set_response(resp, OperationErrorResponse(e))
        except Exception as e:
            import traceback
            model.get_db().session.rollback()
            log.error(traceback.format_exc())
            self.set_response(resp, OperationErrorResponse(e))

    def get_objects_list(self, req, model_class, **kwargs):
        parsed_qs = dict(parse_qsl(req.query_string))
        if CDR_DEBUG:
            log.debug('report parsed parameters:{}'.format(parsed_qs))
        try:
            _from = int(parsed_qs.get('from'))
        except (ValueError, TypeError):
            _from = 0

        try:
            count = int(parsed_qs.get('count', -1))
        except ValueError:
            count = -1

        if 'order_by' in parsed_qs:
            ordering = {
                'by': parsed_qs['order_by'],
                'dir': parsed_qs.get('order_dir', 'asc')
            }
        elif hasattr(self, 'ordering'):
            ordering = self.ordering
        else:
            ordering = {}

        fields = self.get_fields_for_list(parsed_qs, **kwargs)

        filtering = self.get_filtering_for_list(parsed_qs, **kwargs)

        filtering, query = \
            self.modify_query_from_filtering_for_list(filtering, **kwargs)

        if ordering:
            ordering, query = \
                self.modify_query_from_ordering_for_list(ordering, query, **kwargs)
        # return query,None,page, per_page, fields
        if filtering['total'] == 0:
            return [], 0, _from, filtering['total'], fields
            # objects_list, _count, _from, total, fields
        else:
            _till = _from + count if count > 0 else filtering['total']
            return model_class.get_objects_query(
                query=query,
                filtering={},
                ordering=ordering,
                paging={'from': _from, 'till': _till}
            ) + (_from, filtering['total'], fields)

    def on_filter_id(self, v, cdr):
        # return text("ingress_id in ({})".format(v))
        if ',' in v:
            return cdr.c.id.in_([int(i) for i in v.split(',')])
        else:
            return cdr.c.id == v

    def on_filter_ingress_id(self, v, cdr):
        # return text("ingress_id in ({})".format(v))
        return cdr.c.ingress_id.in_([int(i) for i in v.split(',')])

    def on_filter_egress_id(self, v, cdr):
        # return text("egress_id in ({})".format(v))
        return cdr.c.egress_id.in_([int(i) for i in v.split(',')])

    def on_filter_non_zero(self, v, cdr):
        # return text("egress_id in ({})".format(v))
        log.debug("ON_FILTER_NON_ZERO")
        log.debug(f"v = {v} | cdr = {cdr}")
        if v == '1':
            return cdr.c.call_duration > 0
        elif v == '0':
            return cdr.c.call_duration == 0
        else:
            return None

    def on_filter_release_cause_failed(self, v, cdr):
        if v == '1':
            return and_(cdr.c.release_cause != 19, cdr.c.release_cause != 3)
        elif v == '0':
            return or_(cdr.c.release_cause == 19, cdr.c.release_cause == 3)
        else:
            return None

    def on_filter_trunk_type2(self, v, cdr):
        return or_(cdr.c.o_trunk_type2 == v, cdr.c.t_trunk_type2 == v)

    def on_select_ingress_name(self, cdr):
        # cls = aliased(model.Resource)
        # return select([cls.alias]).where(cls.resource_id == cdr.c.ingress_id).as_scalar().label('ingress_name')
        cls = aliased(model.t_resource_record)
        return select([cls.c.alias]).where(cls.c.resource_id == cdr.c.ingress_id).order_by(cls.c.time.desc()).limit(
            1).as_scalar().label('ingress_name')

    def on_select_egress_name(self, cdr):
        # cls = aliased(model.Resource)
        # return select([cls.alias]).where(cls.resource_id == cdr.c.egress_id).as_scalar().label('egress_name')
        cls = aliased(model.t_resource_record)
        return select([cls.c.alias]).where(cls.c.resource_id == cdr.c.egress_id).order_by(cls.c.time.desc()).limit(
            1).as_scalar().label('egress_name')

    def on_select_ingress_client_name(self, cdr):
        # cls = aliased(model.Client)
        # return select([cls.name]).where(cls.client_id == cdr.c.ingress_client_id).as_scalar().label(
        #     'ingress_client_name')
        cls = aliased(model.t_client_record)
        return select([cls.c.name]).where(cls.c.client_id == cdr.c.ingress_client_id).order_by(cls.c.time.desc()).limit(
            1).as_scalar().label('ingress_client_name')

    def on_select_egress_client_name(self, cdr):
        # cls = aliased(model.Client)
        # return select([cls.name]).where(cls.client_id == cdr.c.egress_client_id).as_scalar().label('egress_client_name')
        cls = aliased(model.t_client_record)
        return select([cls.c.name]).where(cls.c.client_id == cdr.c.egress_client_id).order_by(cls.c.time.desc()).limit(
            1).as_scalar().label('egress_client_name')

    def on_select_route_plan_name(self, cdr):
        cls = aliased(model.RouteStrategy)
        return select([cls.name]).where(cls.route_plan_id == cdr.c.route_plan).as_scalar().label('route_plan_name')

    def on_select_egress_rate_table_name(self, cdr):
        cls = aliased(model.RateTable)
        return select([cls.name]).where(cls.rate_table_id == cdr.c.egress_rate_table_id).as_scalar().label(
            'egress_rate_table_name')

    def on_select_dynamic_route_name(self, cdr):
        cls = aliased(model.DynamicRoute)
        return select([cls.name]).where(cls.dynamic_route_id == cdr.c.dynamic_route).as_scalar().label(
            'dynamic_route_name')

    def on_select_static_route_name(self, cdr):
        cls = aliased(model.Product)
        return select([cls.name]).where(cls.product_id == cdr.c.static_route).as_scalar().label('static_route_name')

    def on_select_release_cause_name(self, cdr):
        cls = aliased(model.ReleaseCause)
        return select([cls.name]).where(cls.id == cdr.c.release_cause).as_scalar().label('release_cause_name')

    def on_select_egress_rate_type_name(self, cdr):
        cls = aliased(model.RateTypeName)
        return select([cls.name]).where(cls.id == cdr.c.egress_rate_type).as_scalar().label('egress_rate_type_name')

    def on_select_ingress_rate_type_name(self, cdr):
        cls = aliased(model.RateTypeName)
        return select([cls.name]).where(cls.id == cdr.c.ingress_rate_type).as_scalar().label('ingress_rate_type_name')

    def on_select_ingress_dnis_type_name(self, cdr):
        cls = aliased(model.DnisTypeName)
        return select([cls.name]).where(cls.id == cdr.c.ingress_dnis_type).as_scalar().label('ingress_dnis_type_name')

    def on_select_orig_jur_type_name(self, cdr):
        cls = aliased(model.JurTypeName)
        return select([cls.name]).where(cls.id == cdr.c.orig_jur_type).as_scalar().label('orig_jur_type_name')

    def on_select_term_jur_type_name(self, cdr):
        cls = aliased(model.JurTypeName)
        return select([cls.name]).where(cls.id == cdr.c.term_jur_type).as_scalar().label('term_jur_type_name')

    def on_select_timezone_name(self, cdr):
        m = cdr.c.minutes_west_of_greenwich_mean_time
        return case([(m >= 0, func.concat('GMT+', func.to_char(m / 60, 'FM00')))],
                    else_=func.concat('GMT-', func.to_char(m / 60, 'FM00'))).label('timezone_name')

    def on_select_egress_trunk_trace(self, cdr):
        er1 = aliased(model.EgressErrorString)
        er2 = aliased(model.EgressErrorString)
        er3 = aliased(model.EgressErrorString)
        er4 = aliased(model.EgressErrorString)
        er5 = aliased(model.EgressErrorString)
        egr = aliased(model.Resource)
        egr1 = aliased(model.Resource)
        egr2 = aliased(model.Resource)

        def dsplit(i, j):
            return func.split_part(func.split_part(cdr.c.egress_erro_string, ';', i), ':', j)

        def cint(field):
            return cast(field, model.String)

        return case([(cdr.c.egress_erro_string.isnot(None), func.replace(func.concat(
            select([er1.name]).where(cint(er1.id) == dsplit(1, 1)).as_scalar(),
            ':',
            select([er2.name]).where(cint(er2.id) == dsplit(1, 2)).as_scalar(),
            ';',
            select([egr.alias]).where(cint(egr.resource_id) == dsplit(2, 1)).as_scalar(),
            ':',
            select([er3.name]).where(cint(er3.id) == dsplit(2, 2)).as_scalar(),

            ';',
            select([egr1.alias]).where(cint(egr1.resource_id) == dsplit(3, 1)).as_scalar(),
            ':',
            select([er4.name]).where(cint(er4.id) == dsplit(3, 2)).as_scalar(),

            ';',
            select([egr2.alias]).where(cint(egr2.resource_id) == dsplit(4, 1)).as_scalar(),
            ':',
            select([er5.name]).where(cint(er5.id) == dsplit(4, 2)).as_scalar(),

        ), ';:', ''))], else_=None
                    ).label('egress_trunk_trace')

    def get_fields_for_list(self, parsed_qs, **kwargs):
        fields = parsed_qs.get('field', None)
        if fields:
            fields = fields.split(',')
            for f in fields:
                if hasattr(self, 'on_select_{}'.format(f)):
                    continue
                if not hasattr(self.model_class, f):
                    raise Exception('field {} not valid!'.format(f))
        return fields

    def modify_query_from_filtering_for_list(self, filtering, **kwargs):
        import sys
        _token = filtering.pop('auth_token', None)
        sys.setrecursionlimit(10000)
        columns = [f.name for f in inspect(self.model_class).columns]
        bigtimestamp_fields = ('start_time_of_date', 'end_epoch', 'answer_time_of_date', 'release_tod')
        timestamp_fields = ('ingress_rate_effective_date', 'egress_rate_effective_date')
        named_fields = {'egress_erro_string': 'egress_trunk_trace',
                        'egress_rate_type': 'egress_rate_type_name',
                        'ingress_rate_type': 'ingress_rate_type_name',
                        'ingress_dnis_type': 'ingress_dnis_type_name',
                        'orig_jur_type': 'orig_jur_type_name',
                        'route_plan': 'route_plan_name',
                        'term_jur_type': 'term_jur_type_name',
                        'minutes_west_of_greenwich_mean_time': 'timezone_name',
                        }

        for f in columns:
            if '_id' in f and hasattr(self, 'on_select_' + f.replace('_id', '_name')):
                named_fields[f] = f.replace('_id', '_name')
        _from = filtering.pop('from', None)
        count = filtering.pop('count', None)
        sort = filtering.pop('sort', None)
        order = filtering.pop('order', '')
        print_only_amount = filtering.pop('print_only_amount', False)
        sort_order = None
        if sort:
            _sort = sort.split(',')
            for s in _sort:
                if s not in columns:
                    raise ValidationError('sort field missing {}'.format(s))
            _order = order.split(',')
            for o in _order:
                if o not in ('', 'asc', 'desc'):
                    raise ValidationError('sort order invalid value {}, must be asc or desc (can be empty)'.format(o))
            if len(_sort) != len(_order):
                if sort != '':
                    raise ValidationError('sort length must be equal order length')
                else:
                    _order = ['' for s in _sort]
            sort_order = list(zip(_sort, _order))
        req_query = dict(parse_qsl(self.req.query_string))

        format = filtering.pop('format', None)
        tz = str(filtering.pop('tz', '0'))
        non_zero_calls = filtering.pop('non_zero_calls', 1)
        human_readable = filtering.pop('human_readable', None)
        if CDR_DEBUG:
            log.debug('tz={}'.format(tz))
            log.debug('non_zero_calls={}'.format(non_zero_calls))
            log.debug('human_readable={}'.format(human_readable))
        ccls = model.SystemParameter
        is_hide_unauthorized_ip = ccls.session().query(ccls.is_hide_unauthorized_ip).first().is_hide_unauthorized_ip

        start_time = filtering.pop('start_time', None)
        end_time = filtering.pop('end_time', '0')
        if not start_time:
            raise ValidationError('missing start time')
        try:
            st = int(start_time)
            if st < 0:
                raise ValidationError('wrong start time (unsupported date)')
        except ValueError:
            pass
        try:
            start_time = str(datetime.utcfromtimestamp(int(start_time)).replace(tzinfo=UTC))
            if end_time == '0':
                end_time = str(datetime.now(UTC))
            else:
                end_time = str(datetime.utcfromtimestamp(int(end_time)).replace(tzinfo=UTC))
        except ValueError:
            pass
        field = filtering.pop('field', None)
        fields = filtering.pop('fields', None)
        fields = fields or field
        if fields:
            fields = fields.split(',')
        if not fields:
            fields = columns

        # fields.append('time')

        if human_readable == '1':
            for k in fields.copy():
                if k in named_fields and named_fields[k] not in fields:
                    fields.append(named_fields[k])
                    fields.remove(k)

        for fk in filtering.keys():
            if not fk in fields:
                fields.append(fk)
        if sort:
            for fk in sort.split(','):
                if not fk in fields:
                    fields.append(fk)
        fields = list(set(fields))

        user = self.get_user(self.req)

        if user and user.user_type in ('client', 'agent', 'vendor'):
            if 'ingress_id' in filtering and filtering['ingress_id']:
                ids = set(filtering['ingress_id'].split(','))
                alw = set([str(i) for i in user.allowed_ingress_id])
                filtering['ingress_id'] = ','.join(list(ids.intersection(alw)))
            else:
                filtering['ingress_id'] = ','.join([str(i) for i in user.allowed_ingress_id])
            if filtering['ingress_id'] == '':
                if user.allowed_egress_id:
                    alw = set([str(i) for i in user.allowed_egress_id])
                    if 'egress_id' in filtering and filtering['egress_id']:
                        ids = set(filtering['egress_id'].split(','))
                        filtering['egress_id'] = ','.join(list(ids.intersection(alw)))
                    else:
                        filtering['egress_id'] = ','.join(list(alw))
                    del filtering['ingress_id']
                else:
                    filtering['ingress_id'] = '-1'
            if CDR_DEBUG:
                log.debug('result {} filtering={}'.format(user.user_type, filtering))

        if not end_time and start_time:
            end_time = start_time[0:10] + ' 23:59:59'
        if not end_time and not start_time:
            raise ValidationError('time not defined!')
        if start_time >= end_time:
            raise ValidationError('start time greater than end_time!')

        start_day = parse_datetime(start_time).date()
        end_day = parse_datetime(end_time).date()
        cdr_start = self.get_cdr_start()
        if cdr_start and cdr_start > start_day:
            start_day = cdr_start

        days = (end_day - start_day).days
        union_list = []
        union = None
        _total = 0

        for day_idx in range(days + 1):
            day_cdr = start_day + timedelta(days=day_idx)
            cdr_date = "%s%02d%02d" % (day_cdr.year, day_cdr.month, day_cdr.day)
            cdr = self.get_daily_cdr(cdr_date)
            if cdr is None:
                log.warning('missing table {}'.format(cdr_date))
                continue
            _select = []
            for k in set(fields + ['time']):
                if hasattr(self, 'on_select_' + k):
                    _f = getattr(self, 'on_select_' + k)(cdr)
                    if _f is not None:
                        _select.append(_f)
                    continue
                elif hasattr(cdr.c, k):
                    if human_readable is not None and human_readable == '1':
                        if tz is None:
                            tz = '0'
                        if k in timestamp_fields:
                            fmt = 'YYYY-MM-DD HH24:MI:SS'
                            # _select.append(
                            #     func.to_char(func.to_timestamp(getattr(cdr.c, k)).op('AT TIME ZONE')(tz),
                            #                 fmt).label(k))
                            _select.append(case([(getattr(cdr.c, k) == 0, None)], else_=func.to_char(
                                (func.to_timestamp(getattr(cdr.c, k))).op('AT TIME ZONE')(tz),
                                fmt)).label(k))
                        elif k in bigtimestamp_fields:
                            fmt = 'YYYY-MM-DD HH24:MI:SS.US'
                            _select.append(case([(getattr(cdr.c, k) == 0, None)], else_=func.to_char(
                                (func.to_timestamp(getattr(cdr.c, k).op('/')(1000000.0))).op('AT TIME ZONE')(tz),
                                fmt)).label(k))
                        else:
                            _select.append(getattr(cdr.c, k).label(k))
                    else:
                        _select.append(getattr(cdr.c, k).label(k))
                    # _select.append(getattr(cdr.c, k))
            # _select = tuple(_select)
            _filter = []

            if day_idx == 0:
                _filter = [self.time_field(cdr) >= start_time]
                if day_idx == days:
                    _filter = [self.time_field(cdr) >= start_time, self.time_field(cdr) <= end_time]
            else:
                if day_idx == days:
                    _filter = [self.time_field(cdr) <= end_time]
            if not non_zero_calls is None:
                _filter.append(self.on_filter_non_zero(non_zero_calls, cdr))
            if is_hide_unauthorized_ip:
                _filter.append(cdr.c.release_cause != 3)
            for k, v in filtering.items():
                if hasattr(self, 'on_filter_' + k):
                    _f = getattr(self, 'on_filter_' + k)(v, cdr)
                    if _f is not None:
                        _filter.append(getattr(self, 'on_filter_' + k)(v, cdr))
                    continue
                if k not in columns:
                    raise ValidationError('{}:no such field'.format(k))
                if hasattr(cdr.c, k):
                    if type(v) == type('') and getattr(cdr.c, k).type.python_type == type(''):
                        if '*' in v:
                            if ',' in v:
                                _ff = []
                                for vv in v.split(','):
                                    _ff.append(getattr(cdr.c, k).like(vv.replace('*', '%')))
                                _filter.append(or_(*_ff))
                            else:
                                _filter.append(getattr(cdr.c, k).like(v.replace('*', '%')))
                        else:
                            if ',' in v:
                                if ',' in v:
                                    _ff = []
                                    for vv in v.split(','):
                                        _ff.append(getattr(cdr.c, k) == vv)
                                    _filter.append(or_(*_ff))
                            else:
                                _filter.append(getattr(cdr.c, k) == v)
                    else:
                        _filter.append(getattr(cdr.c, k) == v)
            q = model.get_db().session.query(*_select)
            if _filter:
                q = q.filter(and_(*_filter))
                try:
                    tq = model.get_db().session.query(func.count(cdr.c.time).label('cnt')).filter(and_(*_filter))
                    if CDR_DEBUG:
                        log.debug('total for day {} query :{}'.format(day_idx, model.query_to_sting(tq)))
                    _total += tq.first().cnt
                    if CDR_DEBUG:
                        log.debug('total for day {} :{}'.format(day_idx, _total))
                except Exception as e:
                    if CDR_DEBUG:
                        log.debug('total error:{}'.format(e))

            # pre stort
            if sort_order is None:
                q = q.order_by(self.time_field(cdr))
            else:
                order_by_expr = tuple(
                    map(lambda i: getattr(cdr.c, i[0]).desc() if i[1] == 'desc' else getattr(cdr.c, i[0]),
                        sort_order))
                q = q.order_by(*order_by_expr)
            # pre sort

            union_list.append(q)
        if union_list:
            union = union_list[0].union_all(*(union_list[1:]))
        if union:
            query_tab = alias(union.subquery(), '_all_cdr_')
            self.model_class._tab = query_tab
            _select = []
            q_tab = query_tab
            total = _total

            # try:
            #     # total = model.get_db().session.query(query_tab.count().label('cnt')).all()[0].cnt
            #     total = model.get_db().session.query(func.count(query_tab.c.time).label('cnt')).first().cnt
            #     log.debug('total :{}'.format(total))
            # except Exception as e:
            #     total = 0
            #     log.debug('total error:{}'.format(e))
            #     model.get_db().session.rollback()

            for k in fields:
                if k == 'time' and tz is not None:
                    fmt = 'YYYY-MM-DD HH24:MI:SS'
                    _select.append(func.to_char(q_tab.c.time.op('AT TIME ZONE')(tz), fmt).label('time'))
                elif hasattr(q_tab.c, k):
                    _select.append(getattr(q_tab.c, k).label(k))

            query = model.get_db().session.query(*_select)
            if sort_order is None:
                query = query.order_by(self.time_field(q_tab))
            else:
                order_by_expr = tuple(
                    map(lambda i: getattr(q_tab.c, i[0]).desc() if i[1] == 'desc' else getattr(q_tab.c, i[0]),
                        sort_order))
                query = query.order_by(*order_by_expr)
            return {"total": total}, query
        else:
            return {"total": 0}, None
        pass

    def get_daily_cdr(self, cdr_date):
        cdr = model.client_cdr(cdr_date)
        return cdr

    def get_cdr_start(self):
        date = model.client_cdr_start_date()
        return date

    def time_field(self, cdr):
        return cdr.c.time

    def run(self, file=None, **kwargs):
        class Req:
            def __init__(self, req_user):
                self.query_string = None
                self.context = {'user': req_user}

            pass

        user = kwargs.pop('user', None)
        if user is None:
            user = model.User.get(1)
        req = Req(user)
        fmt = kwargs.pop('format', None)
        req.query_string = urlencode(kwargs)
        self.req = req
        # objects_list, total, page, per_page, fields = self.get_objects_list(req, self.model_class, **kwargs)
        objects_list, _count, _from, total, fields = self.get_objects_list(req, self.model_class, **kwargs)
        if fmt != 'csv':
            resp_stream = gen(objects_list, self, None, _from, total)
        else:
            resp_stream = gen_csv(objects_list, self, None, _from, total)
        if file:
            ouput = open(file, 'wt')
            for data in resp_stream:
                ouput.write(data.decode('utf-8'))
        else:
            return resp_stream


class CdrReportDetailList(CdrReportList):
    scheme_class = CdrReportDetailGetScheme
    model_class = model.CdrReportDetail
    entity_plural = 'CdrReportDetail'

    def get_daily_cdr(self, cdr_date):
        cdr = model.cdr_report_detail(cdr_date)
        return cdr

    def get_cdr_start(self):
        date = model.cdr_reports_start_date()
        return date

    def time_field(self, cdr):
        return cdr.c.report_time


class CdrReportDetailEarliestGet(CustomGetAction):
    model_class = model.CdrReportDetail
    scheme_class = CdrReportDetailEarliestGetScheme
    path_parameters = ()

    def proceed(self, req, resp, **kwargs):
        if (not hasattr(self, 'no_auth_needed') or self.no_auth_needed == False) and not check_permission(self, req,
                                                                                                            'list'):
            self.set_response(
                resp, responses.ForbiddenErrorResponse(data=errors.AuthErrors.Forbidden)
            )
            return False
        try:
            user = self.get_user(self.req)

            cls = self.model_class
            obj = model.get_db().session.query(cls.report_time).order_by(cls.report_time).limit(1).first()
            log.debug(f"GOT OBJECT - {obj}")
            data = {
                'success': True,
                'earliest_report_date': str(obj.report_time),
            }
            log.debug("GOT DATA")
            self.set_response(resp, SuccessReportResponse(data=data))

            return True

        except AttributeError as e:
            self.set_response(
                resp, responses.AttributeNotExitsErrorResponse(
                    data=ATTRIBUTE_ERROR_RE.search(str(e)).group(1)
                )
            )
            return False
        except Exception as e:
            import traceback
            log.error(traceback.format_exc())
            self.set_response(resp, OperationErrorResponse(e))
            return False


class CdrReportDetailLatestGet(CustomGetAction):
    model_class = model.CdrReportDetail
    scheme_class = CdrReportDetailEarliestGetScheme
    path_parameters = ()

    def proceed(self, req, resp, **kwargs):
        if (not hasattr(self, 'no_auth_needed') or self.no_auth_needed == False) and not check_permission(self, req,
                                                                                                            'list'):
            self.set_response(
                resp, responses.ForbiddenErrorResponse(data=errors.AuthErrors.Forbidden)
            )
            return False
        try:
            user = self.get_user(self.req)
            day_cdr = datetime.now(UTC)
            cdr_date = "%s%02d%02d" % (day_cdr.year, day_cdr.month, day_cdr.day)
            cls = model.cdr_report_detail(cdr_date)
            i = 60
            while cls is None:
                day_cdr = day_cdr - timedelta(days=1)
                cdr_date = "%s%02d%02d" % (day_cdr.year, day_cdr.month, day_cdr.day)
                cls = model.cdr_report_detail(cdr_date)
                i -= 1
                if i == 0:
                    break
            if cls is None:
                raise Exception('no cdr table found')
            obj = model.get_db().session.query(cls.c.report_time).order_by(cls.c.report_time.desc()).limit(1).first()
            log.debug(f"GOT OBJECT - {obj}")
            if obj:
                data = {
                    'success': True,
                    'latest_report_date': str(obj.report_time),
                }
                log.debug("GOT DATA")
            else:
                data = {
                    'success': True,
                    'latest_report_date': None,
                }
            self.set_response(resp, SuccessReportResponse(data=data))

            return True

        except AttributeError as e:
            self.set_response(
                resp, responses.AttributeNotExitsErrorResponse(
                    data=ATTRIBUTE_ERROR_RE.search(str(e)).group(1)
                )
            )
            return False
        except Exception as e:
            import traceback
            log.error(traceback.format_exc())
            self.set_response(resp, OperationErrorResponse(e))
            return False


class CdrAsyncTaskCreate(DnlCreate):
    model_class = model.CdrAsyncTask
    scheme_class = CdrAsyncTaskScheme
    scheme_class_cdr = CdrReportGetScheme
    entity = 'CdrAsyncTask'
    cdr_fields = ['count', 'field'] + [field_name for (field_name, _) in scheme_class_cdr._declared_fields.items()]

    def get_validation_errors(self, scheme, data=None):
        d = data or self.req_data
        filter = {k: v for k, v in self.req_data.items() if k in self.cdr_fields}
        # self.scheme_class_cdr().validate(filter)
        data_filtered = {k: v for k, v in d.items() if k not in self.cdr_fields}
        return scheme.validate(data_filtered)

    def create_object(self, req, scheme, **kwargs):
        if 'CONTENT-TYPE' not in req.headers:
            raise ValidationError({'CONTENT-TYPE': 'missing'})
        if req.headers['CONTENT-TYPE'] != 'application/json':
            raise ValidationError({'CONTENT-TYPE': 'invalid'})
        cdr_fields = self.cdr_fields
        filter = {k: v for k, v in self.req_data.items() if k in self.cdr_fields}
        field = filter.get('field', '')
        user = self.get_user(self.req)
        if user.user_type == 'client' and False:
            all_fields = model.DailyCdrField.query().filter(model.DailyCdrField.id.notin_([96, 108])).all()
            allowed_fields = set([str(f.field) for f in all_fields if f.client_viewable])
            fields = set(field.split(','))
            filter['field'] = ','.join(list(fields.intersection(allowed_fields)))
            field = filter['field']
        if 'start_time' not in filter:
            raise ValidationError({'start_time': 'missing'})

        data_filtered = {k: v for k, v in self.req_data.items() if k not in self.cdr_fields}
        inst = scheme.load(data_filtered).data
        inst.filter = filter
        if 'format' not in filter:
            filter['format'] = 'csv'
        obj = self.before_create(inst, **kwargs)
        obj.fields = field
        return obj.save()

    def before_create(self, obj, **kwargs):
        obj.request_id = model.generate_request_id()()
        obj.status = 'initial'
        obj.type = 'manual'
        obj.orig_file = 'cdr_{}.{}'.format(obj.request_id, obj.filter['format'])
        user = self.get_user(self.req)
        obj.user_id = user.user_id
        if user.user_type == 'agent':
            ingress_ids = user.agent.ingress_ids
            if 'ingress_id' not in obj.filter:
                filter = obj.filter
                filter['ingress_id'] = ingress_ids
                obj.filter = filter
        return obj

    def after_create(self, object_id, req, resp, **kwargs):
        from api_dnl.tasks import do_cdr_async_task
        try:
            if CDR_DEBUG:
                log.debug('cdr_async_task start {}'.format(object_id))
            obj = self.model_class.get(object_id)
            if obj:
                obj.orig_file = 'cdr_{}.{}'.format(obj.job_id, obj.filter['format'])
                obj.save()
            do_cdr_async_task.delay(object_id)
            if CDR_DEBUG:
                log.debug('cdr_async_task start delayed {}'.format(object_id))
        except Exception as e:
            if CDR_DEBUG:
                log.debug('cdr_async_task after create {}'.format(e))
            do_cdr_async_task(object_id)


class CdrAsyncTaskGet(CustomGetAction):
    model_class = model.CdrAsyncTask
    scheme_class = CdrAsyncTaskSchemeGet
    path_parameters = ({"name": "request_id", "type": "string", "description": "request id"},)

    def apply(self, obj, req, resp, **kwargs):
        if obj:
            user = self.get_user(self.req)
            if not user.is_admin:
                if user.user_id != obj.user_id:
                    self.set_response(resp, responses.ObjectNotFoundErrorResponse())
                    return False
            item = CdrAsyncTaskSchemeGet().dump(obj).data
            data = {
                'success': True,
                'data': item
            }
            self.set_response(resp, SuccessReportResponse(data=item))
            from time import sleep
            sleep(2)
            return False
        self.set_response(resp, responses.ObjectNotFoundErrorResponse())
        return False


class CdrAsyncTaskDelete(CustomDeleteAction):
    model_class = model.CdrAsyncTask
    scheme_class = CdrAsyncTaskSchemeGet
    path_parameters = ({"name": "request_id", "type": "string", "description": "request id"},)

    def apply(self, obj, req, resp, **kwargs):
        import os
        if obj:
            if obj.file_name and os.path.exists(obj.file_name):
                os.remove(obj.file_name)
            obj.delete()
            return True
        self.set_response(resp, responses.ObjectNotFoundErrorResponse())
        return False


class CdrAsyncTaskList(DnlList):
    model_class = model.CdrAsyncTask
    scheme_class = CdrAsyncTaskSchemeGet
    scheme_class_cdr = CdrReportGetScheme
    entity_plural = 'CdrAsyncTasks'

    def on_filter_o_trunk_type2(self, query, value, kwargs):
        cls = model.CdrAsyncTask
        return query.filter(cast(cls.filter, model.String).like('%"o_trunk_type2": {}%'.format(value)))

    def modify_query_from_filtering_for_list(self, filtering, **kwargs):
        filt, ret = super().modify_query_from_filtering_for_list(filtering, **kwargs)
        user = self.get_user(self.req)
        if not user.is_admin:
            cls = self.model_class
            ret = ret.filter(cls.user_id == user.user_id)
        return filt, ret



class CdrAsyncTaskDownload(CustomGetAction):
    # model_class = model.CdrAsyncTask
    # scheme_class = CdrAsyncTaskSchemeGet
    # scheme_class_cdr = CdrReportGetScheme
    # entity_plural = 'CdrAsyncTasks'
    path_parameters = ({"name": "request_id", "type": "string", "description": "request id"},)

    def on_get(self, req, resp, **kwargs):
        try:
            self.init_req(req)
            if not check_context(self, req, resp, **kwargs):
                return False
            user = self.get_user(self.req)
            params = dict(parse_qsl(req.query_string))
            links = []
            text = ""
            obj = model.CdrAsyncTask.get(kwargs['request_id'])
            if not obj or not obj.download_link or (not user.is_admin and user.user_id != obj.user_id):
                self.set_response(resp, responses.ObjectNotFoundErrorResponse())
                return
            links.append(obj._download_link)
            ret = requests.get(obj._download_link)
            if ret.status_code == 404:
                new_name = obj._download_link.split(".")
                new_name[-1] = "zip"
                download_link = ".".join(new_name)
                links.append(download_link)
                ret = requests.get(download_link)
            resp.status = str(ret.status_code)
            if ret.status_code == 404:
                text = "{}\n".format("\n".join(links))
                resp.body = text + str(ret.content)
            else:
                resp.body = ret.content
            resp.append_header('Content-Disposition',
                               'attachment; filename="{}"'.format(obj.orig_file))
            # log.debug('redirect to:{}'.format(resp.headers))
        except Exception as e:
            self.set_response(resp, OperationErrorResponse(e))
            return False


class CdrAsyncTaskRestart(CustomPostAction):
    model_class = model.CdrAsyncTask

    path_parameters = ({"name": "request_id", "type": "string", "description": "request id"},)

    def apply(self, obj, req, resp, **kwargs):
        if obj:
            from api_dnl.tasks import do_cdr_async_task
            obj.orig_file = 'cdr_{}.{}'.format(obj.job_id, obj.filter['format'])
            try:
                import os
                if os.path.exists(obj.file_name):
                    os.unlink(obj.file_name)
            except:
                pass
            obj.status = 'initial'
            obj.job_start_time = None
            obj.job_end_time = None
            obj.save()
            try:
                do_cdr_async_task.delay(obj.request_id)
                return True
            except Exception as e0:
                if CDR_DEBUG:
                    log.debug('cdr_async_task after create {}'.format(e0))
                try:
                    do_cdr_async_task(obj.request_id)
                    return True
                except Exception as e:
                    self.set_response(resp, OperationErrorResponse(e))
                    return False


class CdrAsyncTaskStop(CustomPostAction):
    model_class = model.CdrAsyncTask

    path_parameters = ({"name": "request_id", "type": "string", "description": "request id"},)

    def apply(self, obj, req, resp, **kwargs):
        if obj:
            if obj.status == 'running':
                obj.status = 'finished'
                obj.save()
                return True
            else:
                raise ValidationError("cannot stop, task is not running.")
        else:
            self.set_response(resp, responses.ObjectNotFoundErrorResponse())
            return False


class CdrAsyncTaskSendMail(CustomPostAction):
    model_class = model.CdrAsyncTask
    scheme_class = CdrAsyncTaskEmailScheme
    body_parameters = ('Mail parameters', CdrAsyncTaskEmailScheme)
    path_parameters = ({"name": "request_id", "type": "string", "description": "request id"},)

    def apply(self, obj, req, resp, **kwargs):
        log.debug('send cdr: {} {} {}'.format(req.query_string, req.data, req.headers))
        if obj:
            errors = self.scheme_class().validate(req.data)
            if errors:
                raise ValidationError(errors)
                # self.set_response(resp, responses.ValidationErrorResponse(errors))
                # return False
            if 'email' in req.data:
                obj.email = req.data['email']
            if 'direct' in req.data:
                obj.direct = SendCdrDirectScheme().load(req.data['direct']).data
            if 'client_id' in req.data:
                obj.request_client_id = req.data['client_id']
            obj.save()
            count = 0
            # while obj.status in ('initial', 'running') and count <= 5:
            #     time.sleep(30)
            #     count += 1
            #     obj = model.CdrAsyncTask.get(obj.request_id)
            from api_dnl.tasks import do_cdr_email_async_task
            try:
                do_cdr_email_async_task.delay(obj.request_id)
                return True
            except Exception as e0:
                log.debug('do_cdr_email_async_task after create {}'.format(e0))
                try:
                    do_cdr_email_async_task(obj.request_id)
                    return True
                except Exception as e:
                    self.set_response(resp, OperationErrorResponse(e))
                    return False
        else:
            self.set_response(resp, responses.ObjectNotFoundErrorResponse())
            return False


class DailyCdrCloudUpdate(CustomPatchAction):
    model_class = model.DnlDailyCdrCloudCfg
    scheme_class = DailyCdrCloudUpdateScheme
    scheme_class_get = ObjectUpdatedScheme
    entity = 'DailyCdrCloudUpdate'
    path_parameters = ()
    has_update_by = True
    security = (DEFAULT_SECURITY)
    restrict = ()

    def on_patch(self, req, resp, **kwargs):
        user = self.get_user(req)
        if not check_permission(self, req, 'show', self.get_object(resp, self.model_class, **kwargs)) or not user.is_admin:
            self.set_response(
                resp, responses.ForbiddenErrorResponse(data=errors.AuthErrors.Forbidden)
            )
            return
        try:
            cls = self.model_class
            try:
                file = req.files['json_key']
                json_key = json.load(file)
            except Exception as e:
                self.set_response(
                    resp, responses.AttributeNotExitsErrorResponse(
                        data='json_key'
                    )
                )
                return False
            all_fields = self.scheme_class.Meta.fields
            for field in all_fields:
                if field == 'json_key':
                    continue
                if field not in req.data:
                    self.set_response(
                        resp, responses.AttributeNotExitsErrorResponse(
                            data=field
                        )
                    )
                    return False
            for key in all_fields:
                if key == 'json_key':
                    q = cls.__table__.update().values(param_value=json.dumps(json_key)).where(
                        cls.param_name == 'json_key'
                    )
                    cls.session().execute(q)
                else:
                    q = cls.__table__.update().values(param_value=req.data[key]).where(cls.param_name == key)
                    cls.session().execute(q)
            data = {
                'success': True
            }
            self.set_response(resp, SuccessReportResponse(data=data))

            return True

        except AttributeError as e:
            self.set_response(
                resp, responses.AttributeNotExitsErrorResponse(
                    data=ATTRIBUTE_ERROR_RE.search(str(e)).group(1)
                )
            )
            return False
        except Exception as e:
            import traceback
            log.error(traceback.format_exc())
            self.set_response(resp, OperationErrorResponse(e))
            return False

    def get_spec(self):
        ext_body_parameters = (
            {
                'name': 'bucket',
                'in': 'formData',
                'description': 'bucket',
                'required': True,
                'type': 'string'
            },
            {
                'name': 'workdir',
                'in': 'formData',
                'description': 'workdir',
                'required': True,
                'type': 'string'
            },
            {
                'name': 'non_zero_cdr_only',
                'in': 'formData',
                'description': 'non_zero_cdr_only',
                'required': True,
                'type': 'boolean'
            },
            {
                'name': 'json_key',
                'in': 'formData',
                'description': 'File to upload',
                'required': True,
                'type': 'file'
            }
        )

        return swagger.specify.get_spec(
            method='patch', description='Update {}'.format(self.entity.lower()),
            consumes=['multipart/form-data'],
            path_parameters=self.path_parameters,
            responses=(
                responses.OperationErrorResponse(data=errors.CommonErrors.PermissionError),
                responses.SuccessResponseObjectInfo(payload_scheme=schemes.SuccessScheme),
                responses.ValidationErrorResponse(),

            ),
            security=self.get_security(method='patch'),
            ext_body_parameters=ext_body_parameters
        )


class DailyCdrCloudDelete(CustomDeleteAction):
    model_class = model.DnlDailyCdrCloudCfg
    scheme_class = DailyCdrCloudUpdateScheme
    scheme_class_get = ObjectUpdatedScheme
    entity = 'DailyCdrCloudDelete'
    has_update_by = True
    security = (DEFAULT_SECURITY)
    restrict = ()

    def on_delete(self, req, resp, **kwargs):
        user = self.get_user(req)
        if not check_permission(self, req, 'show', self.get_object(resp, self.model_class, **kwargs)) or not user.is_admin:
            self.set_response(
                resp, responses.ForbiddenErrorResponse(data=errors.AuthErrors.Forbidden)
            )
            return
        try:
            cls = self.model_class
            all_fields = self.scheme_class.Meta.fields
            for key in all_fields:
                q = cls.__table__.update().values(param_value='').where(cls.param_name == key)
                cls.session().execute(q)
            data = {
                'success': True
            }
            self.set_response(resp, SuccessReportResponse(data=data))

            return True

        except AttributeError as e:
            self.set_response(
                resp, responses.AttributeNotExitsErrorResponse(
                    data=ATTRIBUTE_ERROR_RE.search(str(e)).group(1)
                )
            )
            return False
        except Exception as e:
            import traceback
            log.error(traceback.format_exc())
            self.set_response(resp, OperationErrorResponse(e))
            return False

