from urllib.parse import parse_qsl,urlencode
import falcon
from sqlalchemy import select,and_,or_,not_,func
from sqlalchemy.orm import aliased

from api_dnl import settings
from api_dnl import model
from api_dnl.resources import *
from api_dnl.scheme import *
from api_dnl.schemes.agent import *
from api_dnl.view import DEFAULT_SECURITY,RateList,CarrierLongCreate,CarrierLongResource,ActualBalanceHistoryList,AgentSendLinkAction


class AgentPortalResource(DnlResource):
    model_class = model.Agent
    scheme_class = AgentPortalGetScheme
    scheme_class_get = AgentPortalGetScheme
    # scheme_class_modify = CarrierContactsScheme
    entity = 'Agent'
    id_field = 'agent_id'
    has_delete_operation = False
    has_modify_operation = False
    has_update_by = False
    is_agent = True
    unique_field = 'name'
    security = (DEFAULT_SECURITY)
    path_parameters = ()
    restrict = ()

    def get_path_parameters(self):
        return ()


class AgentPortalClientList(DnlList):
    scheme_class = AgentPortalClientGetScheme
    model_class = model.AgentClient
    entity_plural = 'AgentPortalClients'
    path_parameters = ()
    security = DEFAULT_SECURITY
    restrict = ()
    is_agent = True

    def get_objects_list(self, req, model_class, **kwargs):
        parsed_qs = dict(parse_qsl(req.query_string))
        try:
            per_page = int(parsed_qs.get('per_page'))
        except (ValueError, TypeError):
            per_page = conf.get_default_items_per_page()

        try:
            page = int(parsed_qs.get('page', 0))
        except ValueError:
            page = 0

        if hasattr(self, 'per_page_limit') and per_page > self.per_page_limit:
            raise ValidationError('per_page  over limit {}'.format(self.per_page_limit))

        if 'order_by' in parsed_qs:
            ordering = {
                'by': parsed_qs['order_by'],
                'dir': parsed_qs.get('order_dir', 'asc')
            }
        elif hasattr(self, 'ordering'):
            ordering = self.ordering
        else:
            ordering = {}
        fields = self.get_fields_for_list(parsed_qs, **kwargs)
        filtering = self.get_filtering_for_list(parsed_qs, **kwargs)
        result = self.modify_query_from_filtering_for_list(filtering, **kwargs)
        if result is not None:
            filtering, query = result
        else:
            filtering, query = {}, None
        for c in inspect(self.model_class).columns:
            if c.name in ['alias', 'name', 'template_name', 'group_name']:
                query = query.filter(or_(c.is_(None), c.notlike('#%')))
                break

        if ordering:
            ordering, query = \
                self.modify_query_from_ordering_for_list(ordering, query, **kwargs)
        agent = model.Agent.filter(model.Agent.user_id == self.get_user(req).user_id).first()
        query = query.filter(model.AgentClient.agent_id == agent.agent_id)
        if ('ACCEPT' in req.headers and req.headers['ACCEPT'] == 'text/csv'):
            return model_class.get_objects_query(
                query=query,
                filtering=filtering,
                ordering=ordering,
                paging=None
            ) + (0, None, fields)
        else:
            if 'by' in ordering.keys() and ordering['by'] == 'invoice_number':
                per_page = None
            return model_class.get_objects_list(
                query=query,
                filtering=filtering,
                ordering=ordering,
                paging={'from': page * per_page, 'till': (page + 1) * per_page} if per_page else None,
                query_only = 'query_only' in kwargs
            ) + (page, per_page, fields)


class AgentPortalVendorList(DnlList):
    scheme_class = AgentPortalVendorGetScheme
    model_class = model.AgentClient
    entity_plural = 'AgentPortalClients'
    path_parameters = ()
    security = DEFAULT_SECURITY
    restrict = ()
    is_agent = True

    def get_filtering_for_list(self, parsed_qs, **kwargs):
        ret = super(AgentPortalVendorList, self).get_filtering_for_list(parsed_qs, **kwargs)
        if 'agent_id' in kwargs:
            ret['agent_id'] = kwargs['agent_id']
        ret['has_egress'] = 'true'
        return ret


class AgentPortalClientPaymentsList(DnlList):
    scheme_class = AgentPortalClientPaymentGetScheme
    model_class = model.ClientPayment
    entity_plural = 'AgentPortalClientPayments'
    path_parameters = ()
    security = DEFAULT_SECURITY
    restrict = ()
    is_agent = True

    def modify_query_from_filtering_for_list(self, filtering, **kwargs):
        filt, ret = super(AgentPortalClientPaymentsList, self).modify_query_from_filtering_for_list(filtering, **kwargs)
        cls = self.model_class
        a = aliased(model.AgentClient)
        if 'agent_id' in kwargs:
            agent_id = kwargs['agent_id']
            ret = ret.filter(cls.client_id.in_(select([a.client_id]).where(a.agent_id == agent_id)))
        return filt, ret

    def get_filtering_for_list(self, parsed_qs, **kwargs):
        ret = super(AgentPortalClientPaymentsList, self).get_filtering_for_list(parsed_qs, **kwargs)
        # ret['agent_id'] = kwargs['agent_id']
        return ret


class AgentPortalClientGatewayPaymentsList(DnlList):
    scheme_class = AgentPortalClientGatewayPaymentGetScheme
    model_class = model.PaymentGatewayHistory
    entity_plural = 'AgentPortal Clients GatewayPayments'
    path_parameters = ()
    security = DEFAULT_SECURITY
    restrict = ()
    is_agent = True

    def modify_query_from_filtering_for_list(self, filtering, **kwargs):
        filt, ret = super(AgentPortalClientGatewayPaymentsList, self).modify_query_from_filtering_for_list(filtering, **kwargs)
        cls = self.model_class
        a = aliased(model.AgentClient)
        if 'agent_id' in kwargs:
            agent_id = kwargs['agent_id']
            ret = ret.filter(cls.client_id.in_(select([a.client_id]).where(a.agent_id == agent_id)))
        return filt, ret

    def get_filtering_for_list(self, parsed_qs, **kwargs):
        ret = super(AgentPortalClientGatewayPaymentsList, self).get_filtering_for_list(parsed_qs, **kwargs)
        # ret['agent_id'] = kwargs['agent_id']
        return ret


class AgentPortalClientGatewayAndPaymentsList(DnlList):
    scheme_class = AgentPortalClientGatewayAndPaymentsGetScheme
    model_class = model.PaymentAndGateway
    entity_plural = 'AgentPortal Clients GatewayPayments and ordinal payments'
    path_parameters = ()
    security = DEFAULT_SECURITY
    restrict = ()
    is_agent = True

    def get_objects_list(self, req, model_class, **kwargs):
        parsed_qs = dict(parse_qsl(req.query_string))
        try:
            per_page = int(parsed_qs.get('per_page'))
        except (ValueError, TypeError):
            per_page = conf.get_default_items_per_page()

        try:
            page = int(parsed_qs.get('page', 0))
        except ValueError:
            page = 0

        if 'order_by' in parsed_qs:
            ordering = {
                'by': parsed_qs['order_by'],
                'dir': parsed_qs.get('order_dir', 'asc')
            }
        else:
            ordering = {}

        fields = self.get_fields_for_list(parsed_qs, **kwargs)

        filtering = self.get_filtering_for_list(parsed_qs, **kwargs)

        filtering, query = \
            self.modify_query_from_filtering_for_list(filtering, **kwargs)

        for c in inspect(self.model_class).columns:
            if c.name in ['alias','name','template_name','group_name']:
                query = query.filter(or_(c.is_(None),c.notlike('#%')))
                break

        if ordering:
            ordering, query = \
                self.modify_query_from_ordering_for_list(ordering, query, **kwargs)

        return model_class.get_objects_query_list(
            query=query,
            filtering=filtering,
            ordering=ordering,
            paging={'from': page * per_page, 'till': (page + 1) * per_page}
        ) + (page, per_page, fields)

    def modify_query_from_filtering_for_list(self, filtering, **kwargs):
        filt, ret = super(AgentPortalClientGatewayAndPaymentsList, self).modify_query_from_filtering_for_list(filtering, **kwargs)
        cls = self.model_class._tab #self.model_class
        a = aliased(model.AgentClient)
        if 'agent_id' in kwargs:
            agent_id = kwargs['agent_id']
            ret = ret.filter(cls.c.client_id.in_(select([a.client_id]).where(a.agent_id == agent_id)))
        return filt, ret

    def get_filtering_for_list(self, parsed_qs, **kwargs):
        ret = super(AgentPortalClientGatewayAndPaymentsList, self).get_filtering_for_list(parsed_qs, **kwargs)
        # ret['agent_id'] = kwargs['agent_id']
        return ret


class AgentPortalProductList(DnlList):
    scheme_class = AgentPortalProductsGetScheme
    model_class = model.ProductAgentsRef
    entity_plural = 'AgentPortalProducts'
    path_parameters = ()
    security = DEFAULT_SECURITY
    restrict = ()
    is_agent = True

    def modify_query_from_filtering_for_list(self, filtering, **kwargs):
        filt, ret = super(AgentPortalProductList, self).modify_query_from_filtering_for_list(filtering, **kwargs)
        cls = self.model_class
        #ref = aliased(model.ProductAgentsRef)
        if 'agent_id' in kwargs:
            agent_id = kwargs['agent_id']
            ret = ret.filter(cls.agent_id == agent_id)
        return filt, ret


class AgentPortalRateList(RateList):
    scheme_class = AgentPortalRateGetScheme
    model_class = model.Rate
    entity_plural = 'AgentPortalRates'
    is_agent = True

    def get_filtering_for_list(self, parsed_qs, **kwargs):
        log.debug("getting filtering for list")
        agent_id = None
        if 'agent_id' in kwargs:
            agent_id = kwargs['agent_id']
        rate_table_id = kwargs['rate_table_id']
        prod = aliased(model.ProductRoutRateTable)
        ref = aliased(model.ProductAgentsRef)
        q = prod.filter(or_(prod.agent_id == agent_id, prod.id.in_(select([ref.product_id])
                                                                   .where(ref.agent_id == agent_id)))).\
            filter(prod.rate_table_id == rate_table_id).first()
        if not q:
            raise NoResultFound()

        ret = super(AgentPortalRateList, self).get_filtering_for_list(parsed_qs, **kwargs)
        ret['rate_table_id'] = kwargs['rate_table_id']
        return ret

    def get_objects_list(self, req, model_class, **kwargs):
        log.debug("getting objects list")
        parsed_qs = dict(parse_qsl(req.query_string))

        try:
            per_page = int(parsed_qs.get('per_page'))
        except (ValueError, TypeError):
            per_page = -1

        try:
            page = int(parsed_qs.get('page', 0))
        except ValueError:
            page = 0

        if 'order_by' in parsed_qs:
            ordering = {
                'by': parsed_qs['order_by'],
                'dir': parsed_qs.get('order_dir', 'asc')
            }
        elif hasattr(self, 'ordering'):
            ordering = self.ordering
        else:
            ordering = {}

        fields = self.get_fields_for_list(parsed_qs, **kwargs)
        filtering = self.get_filtering_for_list(parsed_qs, **kwargs)
        filtering, query = self.modify_query_from_filtering_for_list(filtering, **kwargs)

        if ordering:
            ordering, query = self.modify_query_from_ordering_for_list(ordering, query, **kwargs)

        if per_page == -1:
            return model_class.get_objects_query(
                query=query,
                filtering=filtering,
                ordering=ordering,
                paging=None
            ) + (0, None, fields)
        else:
            return model_class.get_objects_list(
                query=query,
                filtering=filtering,
                ordering=ordering,
                paging={'from': page * per_page, 'till': (page + 1) * per_page} if per_page else None,
                query_only='query_only' in kwargs
            ) + (page, per_page, fields)

    def _on_get(self, req, resp, **kwargs):
        if (not hasattr(self, 'no_auth_needed') or self.no_auth_needed == False) and not check_permission(self, req,
                                                                                                          'list'):
            self.set_response(
                resp, responses.ForbiddenErrorResponse(data=errors.AuthErrors.Forbidden)
            )
            return
        objects_list, total, page, per_page, fields = self.get_objects_list(req, self.model_class, **kwargs)
        items = self.scheme_class(only=fields).dump(objects_list, many=True).data

        self.set_response(
            resp, responses.SuccessResponse(
                data={
                    'items': items,
                    'total': total,
                    'page': page,
                    'per_page': len(items)
                },
                scheme=schemes.ObjectScheme
            )
        )


class AgentPortalAllRateslList(RateList):
    scheme_class = AgentPortalRateGetScheme
    model_class = model.Rate
    entity_plural = 'AgentPortalRates'
    is_agent = True

    path_parameters = ()

    def modify_query_from_filtering_for_list(self, filtering, **kwargs):
        filt, ret = super(AgentPortalAllRateslList, self).modify_query_from_filtering_for_list(filtering, **kwargs)
        cls = self.model_class
        prod = aliased(model.ProductRoutRateTable)
        ref = aliased(model.ProductAgentsRef)
        agent_id = kwargs['agent_id']
        if agent_id != 'None':
            ret = ret.filter(cls.rate_table_id.in_(select([prod.rate_table_id]).where(
                and_(prod.id == ref.product_id, ref.agent_id == agent_id)
            )))
        return filt, ret


class AgentPortalRateDownload(AgentPortalRateList):
    path_parameters = ({'name': 'token', 'description': 'token to download rates'},)
    no_auth_needed = True
    is_agent = False

    def on_get(self, req, resp, **kwargs):  # pragma: no cover
        import jwt
        try:
            data=jwt.decode(kwargs['token'],settings.JWT_SIGNATURE)
        except jwt.ExpiredSignatureError:
            resp.status = falcon.HTTP_406
            resp.body = '{"error":"link_expired"}'
            return False
        except jwt.InvalidTokenError:
            resp.status = falcon.HTTP_404
            return None
        agent_id = data.get('agent_id',None)
        rate_table_id = data.get('rate_table_id',None)
        #if not agent_id  or \
        if not rate_table_id:
            resp.status = falcon.HTTP_406
            resp.body = '{"error":"invalid download token provided"}'
            return False
        kwargs['agent_id']=str(agent_id)
        kwargs['rate_table_id']=str(rate_table_id)
        del kwargs['token']

        parsed_qs = dict(parse_qsl(req.query_string))
        try:
            per_page = int(parsed_qs.get('per_page'))
        except (ValueError, TypeError):
            parsed_qs['per_page'] = 64000
        req.query_string = urlencode(parsed_qs)

        ret = super(AgentPortalRateDownload,self).on_get(req, resp, **kwargs)
        if not resp.status == falcon.HTTP_200:
            return
        data = json.loads(resp.body)
        csvfile = io.StringIO()
        items = data['payload']['items']
        if len(items):
            fieldnames = data['payload']['items'][0].keys()
            writer = DictWriter(csvfile, fieldnames=fieldnames)
            writer.writeheader()
            writer.writerows(items)
            resp.body = str.encode(csvfile.getvalue())
        else:
            resp.body = ''
        resp.content_type = 'text/csv'
        return ret


class AgentPortalClientCreate(CarrierLongCreate):
    scheme_class =  AgentPortalClientScheme
    is_agent = True

    def after_create(self,object_id, req, resp, **kwargs):
        agent_id = kwargs['agent_id']
        model.AgentClient(agent_id=agent_id,client_id=object_id).save()


class AgentPortalClientResource(CarrierLongResource):
    scheme_class = AgentPortalClientScheme
    scheme_class_modify = AgentPortalClientModifyScheme
    is_agent = True

    def get_object(self, resp, model_class, **kwargs):
        if 'agent_id' in kwargs and 'client_id' in kwargs:
            agent_id = kwargs['agent_id']
            client_id = kwargs['client_id']
        else:
            return None
        acl = model.AgentClient.filter(and_(agent_id == agent_id, client_id == client_id)).first()
        if acl:
            obj = super(AgentPortalClientResource, self).get_object(resp, model_class, **kwargs)
        else:
            obj = None
        return obj


class AgentPortalClientsActualBalanceList(DnlList):#ActualBalanceHistoryList):
    model_class = model.BalanceHistoryActual
    scheme_class = AgentPortalClientsActualBalanceScheme
    entity_plural = 'AgentClientsActualBalances'
    security = DEFAULT_SECURITY
    is_agent = True
    path_parameters = ()

    def modify_query_from_filtering_for_list(self, filtering, **kwargs):
        filt, ret = super(AgentPortalClientsActualBalanceList, self).modify_query_from_filtering_for_list(filtering, **kwargs)
        cls = self.model_class
        a = aliased(model.AgentClient)
        agent_id = kwargs['agent_id']
        ret = ret.filter(cls.client_id.in_(select([a.client_id]).where(a.agent_id == agent_id)))
        return filt, ret


class AgentPortalInvoiceList(DnlList):
    model_class = model.Invoice
    scheme_class = AgentPortalInvoiceGetScheme
    entity_plural = 'AgentPortalInvoices'
    security = DEFAULT_SECURITY
    is_agent = True
    path_parameters = ()

    def modify_query_from_filtering_for_list(self, filtering, **kwargs):
        filt, ret = super(AgentPortalInvoiceList, self).modify_query_from_filtering_for_list(filtering, **kwargs)
        cls = self.model_class
        # a = aliased(model.AgentClient)
        agent_client = model.AgentClient
        if 'agent_id' in kwargs:
            agent_id = kwargs['agent_id']
            client_ids = [client.client_id for client in agent_client.filter(agent_client.agent_id == agent_id).all()]
            ret = ret.filter(cls.client_id.in_(client_ids))
        return filt, ret

    def modify_query_from_ordering_for_list(self, ordering, query, **kwargs):
        ordering, query = super(AgentPortalInvoiceList, self).modify_query_from_ordering_for_list(ordering, query, **kwargs)
        # page = int(self.req.data.pop('page', 0))
        # per_page = int(self.req.data.pop('per_page', 10))
        # query = query.offset(page*per_page).limit(10)
        return ordering, query
    
    def on_get(self, req, resp, **kwargs):
        super().on_get(req, resp, **kwargs)
        parsed_qs = dict(parse_qsl(req.query_string))
        data = json.loads(resp.body)
        if 'payload' in data and 'items' in data['payload']:
            items = data['payload']['items']

            try:
                per_page = int(parsed_qs.get('per_page'))
            except (ValueError, TypeError):
                per_page = 10

            try:
                page = int(parsed_qs.get('page', 0))
            except ValueError:
                page = 0

            data['payload']['items'] = items[page*per_page:page*per_page+per_page]
            resp.body = json.dumps(data)


class AgentPortalPaymentGatewayHistoryCreate(DnlCreate):
    scheme_class = AgentPortalPaymentGatewayHistoryScheme
    entity = 'PaymentGatewayHistoryRecord'
    unique_field = 'id'
    additional_responses = ()
    security = (DEFAULT_SECURITY)
    restrict = ()
    path_parameters = ({'name': 'client_id', 'description': 'Parent carrier'},)
    is_agent = True

    def before_create(self, obj, **kwargs):
        obj.client_id=kwargs['client_id']
        agent_id = kwargs['agent_id']
        a = aliased(model.AgentClient)
        q = a.filter(and_(a.agent_id == agent_id,a.client_id==obj.client_id)).first()
        if not q:
            raise ValidationError('not allowed client {}'.format(obj.client_id))
        return obj


class AgentPortalAgentSendLinkAction(AgentSendLinkAction):
    path_parameters = ()

    is_agent = True

    def on_post(self, req, resp, **kwargs):
        self.init_req(req)
        if not check_context(self, req, resp, **kwargs):
            return False
        kwargs = client_context(self, req, resp, **kwargs)
        return self.proceed(req, resp, **kwargs)
