from datetime import datetime

from pytz import UTC
from sqlalchemy import (and_)

from api_dnl import model
from api_dnl import settings
from falcon_rest.logger import log
from api_dnl.model import MailTemplate, MailSender
from api_dnl.model import InvoiceTask, Client, InvoiceDebug, SystemParameter, Invoice, DailyCdrField, CdrExportTask
from api_dnl.tasks import app, SqlAlchemyTask, db
import redis

@app.task(base=SqlAlchemyTask, time_limit=37000, soft_time_limit=36000)
def do_invoice_task(task_id):
    log.debug('invoice_task %s started' % task_id)
    task = InvoiceTask.get(task_id)
    if task:
        while (task.status == 'Initial' or task.status == 'In Process'):
            log.debug('invoice_task %s waiting....' % task_id)
            task.run()
    else:
        log.debug('invoice_task %s not found' % task_id)
    log.debug('invoice_task %s finished' % task_id)


def invoice_generate(client_id, invoice_start_time, invoice_end_time):
    log.debug('start invoice_generate. client_id: {}'.format(client_id))
    # invoice.delay(client_id, invoice_start_time, invoice_end_time)
    obj = None
    warn = []
    try:
        task = InvoiceTask.create_auto_invoice_task(client_id, invoice_start_time, invoice_end_time)
        if not task:
            log.debug('error:auto_invoice_task for client_id %d file NOT created' % (client_id))
        task.run()
        log.debug('invoice_generate. task run finished. client_id: {}'.format(client_id))
        obj = model.InvoiceSummary.filter(model.InvoiceSummary.task_id==task.id).first()
        try:
            if task.enable_email_with_cdr:
                log.debug('auto_invoice_task %d cdr start' % obj.invoice_number)
                do_invoice_create_cdr_file(obj.invoice_number)
                log.debug('auto_invoice_task %d cdr finished' % obj.invoice_number)
            else:
                log.debug('auto_invoice_task %d cdr skip' % obj.invoice_number)
        except Exception as e:
            db.session.rollback()
            log.debug('error:auto_invoice_task %d CDR NOT created %s' % (obj.invoice_number, str(e)))
            warn.append('warning:auto_invoice_task %d CDR NOT created %s' % (obj.invoice_number, str(e)))
        try:
            # if obj.client.auto_send_invoice:
            if task.enable_email:
                do_invoice_send(obj.invoice_number)
            else:
                log.debug('auto_invoice_task send %d not enabled' % obj.invoice_number)
                warn.append('warning: auto_invoice_task email send %d not enabled' % obj.invoice_number)
        except Exception as e:
            log.debug('error:auto_invoice_task %d email failed %s' % (obj.invoice_number, str(e)))
            warn.append('warning:auto_invoice_task %d email failed %s' % (obj.invoice_number, str(e)))
    except Exception as e:
        log.debug('error:auto_invoice_task for client %d file NOT created %s' % (client_id, str(e)))
        return None, 'error:auto_invoice_task for client %d file NOT created %s' % (client_id, str(e))
    return obj, ''.join(warn)


@app.task(base=SqlAlchemyTask, bind=True, time_limit=24*60*60, soft_time_limit=23*60*60)
def do_auto_invoice_task(self):
    try:
        task_id = self.request.id
        auto_invoice_debug = settings.AUTO_INVOICE_DEBUG
        if auto_invoice_debug:
            log.debug('Started auto invoice function. Connecting to Redis to check the lock.')
        r = redis.Redis(host='localhost', port=6379, db=0)
        if auto_invoice_debug:
            log.debug('Connected to Redis, checking if lock exists.')
        is_new_task = r.setnx('do_auto_invoice_task', 1)
        if auto_invoice_debug:
            log.debug('Got the lock from Redis, checking if it set to proceed')
        if not is_new_task:
            log.warning('do_auto_invoice_task: another task is running, quit')
            return False
        if r.get(task_id):
            log.debug(f"{task_id} is a duplicate job")
            raise False
        r.set(task_id, 1, ex=60*59)

        r.expire('do_auto_invoice_task', 25*60*60)
        if auto_invoice_debug:
            log.debug('Locked successfully, starting invoice task')

        log.debug('Invoice started')

        total_invoices = 0
        now = datetime.now(UTC)

        if auto_invoice_debug:
            log.debug('Getting clients to generate invoice for')
        cl = Client.filter(and_(Client.auto_invoicing == True, #Client.status == True,
                                Client.payment_term_id.isnot(None), Client.name.isnot(None))).all()
        if auto_invoice_debug:
            log.debug('Clients are selected properly, autoinvoice started. clients: {}'.format([c.client_id for c in cl]))
            InvoiceDebug(text='Auto invoice started selected clients {}'.format([c.client_id for c in cl])).save()
        total_msg = ''
        for client in cl:
            log.debug('Start invoice_check_and_generate for client {}'.format(client.client_id))
            cnt, msg = client.invoice_check_and_generate(now, invoice_generate, auto_invoice_debug)
            total_invoices += cnt
            total_msg += msg
            log.debug('End invoice_check_and_generate for client {}. msg: {}'.format(client.client_id, msg))
        msg = 'Invoice finished, done total {} invoices'.format(total_invoices)
        log.debug(msg)
        if auto_invoice_debug:
            log.debug('Auto invoice task finished, total messages: ')
            log.debug(total_msg)
            InvoiceDebug(text=msg).save()
        r.delete('do_auto_invoice_task')
        return True
    except Exception as e:
        log.debug(f"Auto invoice failed: error was {e}")
        return False

@app.task(base=SqlAlchemyTask, time_limit=37000, soft_time_limit=36000)
def do_invoice_send(invoice_number,email=None):
    log.debug('do_invoice_send started {} {}'.format(invoice_number,email))
    try:
        obj = Invoice.filter(Invoice.invoice_number == str(invoice_number)).first()
        if obj:
            client =obj.client
            att = []
            if client.invoice_conf.invoice_send_mode == 'attachment':
                pdf = obj.file_data
                if pdf:
                    filename=obj.pdf_path.replace('invoice/','')
                    att = [(filename, pdf)]
                    log.debug('do_invoice_send added attachment  {}'.format(filename))
                else:
                    log.error('Invoice has no pdf file to send {}'.format(obj.invoice_id))
            tpl = MailTemplate._get('invoice')
            to = getattr(obj.client, tpl.to_mail, '').split(';')
            to_addr = ';'.join(to)
            if email:
                to_addr = email
            log.debug('do_invoice_send must send to {}'.format(to_addr))
            ret = MailSender.apply_mail(obj, 'invoice', to=to_addr, att=att)
            if ret:
                log.error('invoice send mail system error %s', str(invoice_number))
            else:
                obj.state = 'send'
                obj.save()
        else:
            raise Exception('Invoice not exist while sending.')
    except Exception as e:
        log.debug('error:do_invoice_send %d error was %s' % (invoice_number, str(e)))
        raise e
    log.debug('do_invoice_send finished {}'.format(invoice_number))


@app.task(base=SqlAlchemyTask, time_limit=37000, soft_time_limit=36000)
def do_invoice_create_cdr_file(invoice_number):
    # from api_dnl.tasks import do_cdr_async_task,log
    log.debug('do_invoice_create_cdr_file {} started'.format(invoice_number))
    try:
        conf = SystemParameter.get(1)
        obj = Invoice.filter(Invoice.invoice_number == str(invoice_number)).one()
        if not obj.ingress_cdr_file:
            client = obj.client
            fields = None
            try:
                fields = client.invoice_setting.setting.invoices_cdr_fields
            except:
                fields = conf.invoices_cdr_fields

            if not fields:
                fields = ""

            # if not fields:
            all_fields = {'start_time_of_date','origination_source_number','ingress_client_cost',
                            'origination_source_host_name','origination_destination_number',
                            'answer_time_of_date','ingress_rate_type','final_route_indication',
                            'lrn_dnis','call_duration','ingress_client_rate'}
            if isinstance(fields, str):
                fields = fields.split(',')
            fields = list(all_fields.intersection(set(fields)))
            labels = DailyCdrField.convert_fields(fields)
            filt = 'and ingress_id in ({},-2)'.format(client._ingress())
            task = CdrExportTask(cdr_start_time=obj.invoice_start,
                                 cdr_end_time=obj.invoice_end,
                                 csv_file_headers=','.join(labels),
                                 cdr_headers=','.join(fields),
                                 cdr_filter=filt
                                 )
            task_id = task.save()
            counter, limit = 0, 512
            while task.status not in ('Successful', 'Failed'):
                counter += 1
                model.get_db().session.refresh(task)
                if counter >= limit:
                    raise Exception('CdrExportTask {} timeout, attempts count={}'.format(task_id, counter))
            obj.cdr_link = task.cdr_link
            obj.ingress_cdr_file = task.export_cdr_file
            obj.save()
            log.debug('do_invoice_create_cdr_file file created {}'.format(obj.ingress_cdr_file))
    except Exception as e:
        log.error('do_invoice_create_cdr_file invoice {} error:{}'.format(invoice_number, str(e)))
    log.debug('do_invoice_create_cdr_file {} finished'.format(invoice_number))
