from marshmallow import validate,Schema, pre_load
from marshmallow.fields import Bool, Str, Decimal, Int, Nested, DateTime, Float, List, UUID, Dict

from api_dnl import model
from api_dnl.fields import Choice
from api_dnl.scheme import BaseModelScheme, Emails

from api_dnl.scheme import IP_REGEXP, NAME_REGEXP, PREFIX_REGEXP, TIME_REGEXP, ValidationError, SuccessScheme, _valid_unique
from api_dnl.utils.statisticapi2 import FIELD_MAP

# region +++C4ShakenConf+++
class C4ShakenConfScheme(BaseModelScheme):
    id = Int()
    name = Str()
    is_primary = Bool()
    conn_type = Str(validate=[validate.Length(max=3)])
    hostaddr = Str(validate=[validate.Regexp(IP_REGEXP)])
    port = Int()
    heartbeat = Bool()
    heartbeat_interval = Int()
    type = Choice()

    class Meta:
        model = model.C4ShakenConf
        fields = ('name', 'is_primary', 'conn_type', 'hostaddr', 'port', 'heartbeat', 'heartbeat_interval', 'type')

class C4ShakenConfManyScheme(BaseModelScheme):
    servers = Nested(C4ShakenConfScheme, many=True)

    @pre_load
    def check_servers(self, data):
        if 'servers' in data:
            if len(data['servers']) > 4:
                raise ValidationError({'servers': ['you should specify 4 servers']})
            if len(data['servers']) == 4:
                cond = [0]*4
                for server in data['servers']:
                    index = int('{}{}'.format(int(server['is_primary']),int(server['type'] == 'Call signing')), 2)
                    cond[index] = 1
                if sum(cond) != 4:
                    raise ValidationError({'servers': ['you should specify all 4 types of servers']})
        return data

    class Meta:
        model = model.C4ShakenConf
        fields = ('servers',)


class C4ShakenConfSchemeGet(C4ShakenConfScheme):
    class Meta:
        model = model.C4ShakenConf
        fields = ('id', 'name', 'is_primary', 'conn_type', 'hostaddr', 'port', 'heartbeat', 'heartbeat_interval', 'type')
        search_fields = (
            'id', 'name', 'is_primary', 'conn_type', 'hostaddr', 'port', 'heartbeat', 'heartbeat_interval', 'type')


class C4ShakenConfSchemeModify(C4ShakenConfScheme):
    pass


# endregion ---C4ShakenConf---
#region +++ShakenAniGroup+++
class ShakenAniGroupScheme(BaseModelScheme):
    id = Int()
    name = Str()
    class Meta:
        model = model.ShakenAniGroup
        fields = ('name',)
class ShakenAniGroupSchemeGet(ShakenAniGroupScheme):
    class Meta:
        model = model.ShakenAniGroup
        fields = ('id', 'name', 'created_on', 'created_by', 'count')
        search_fields = ('id', 'name', 'created_by', 'count')
        query_fields = ('created_on_gt', 'created_on_lt',)
class ShakenAniGroupSchemeModify(ShakenAniGroupScheme):
    pass
#endregion ---ShakenAniGroup---
#region +++ShakenAniGroupList+++
class ShakenAniGroupListScheme(BaseModelScheme):
    id = Int()
    name = Str()
    class Meta:
        model = model.ShakenAniGroupList
        fields = ('name',)
class ShakenAniGroupListSchemeGet(ShakenAniGroupListScheme):
    class Meta:
        model = model.ShakenAniGroupList
        fields = ('id','name','created_on','created_by','count')
        search_fields = ('id','name','created_on','count')
        query_fields = ('created_on_gt', 'created_on_lt',)
class ShakenAniGroupListSchemeModify(ShakenAniGroupListScheme):
    pass
#endregion ---ShakenAniGroupList---
#region +++ShakenAniGroupListRel+++
class ShakenAniGroupListRelScheme(BaseModelScheme):
    id = Int()
    ani_group_list_id = Int()
    ani_group_id = Int()
    attest_lvl = Str(validate=[validate.Length(max=1)])
    class Meta:
        model = model.ShakenAniGroupListRel
        fields = ('ani_group_list_id','ani_group_id','attest_lvl',)
class ShakenAniGroupListRelAddScheme(BaseModelScheme):
    id = Int()
    ani_group_list_id = Int()
    ani_group_id = Int()
    attest_lvl = Str(validate=[validate.Length(max=1), validate.OneOf(['A', 'B', 'C'])])
    class Meta:
        model = model.ShakenAniGroupListRel
        fields = ('ani_group_id','attest_lvl',)
class ShakenAniGroupListRelAddManyScheme(BaseModelScheme):
    attestation_settings = Nested(ShakenAniGroupListRelAddScheme, many=True)
    class Meta:
        model = model.ShakenAniGroupListRel
        fields = ('attestation_settings',)
class ShakenAniGroupListRelSchemeGet(ShakenAniGroupListRelScheme):
    class Meta:
        model = model.ShakenAniGroupListRel
        fields = ('ani_group_list_id','ani_group_id','attest_lvl','created_by','created_on','ani_group_name','count','id')
        search_fields = ('ani_group_list_id','ani_group_id','attest_lvl','created_by','created_on','ani_group_name','count','id')
class ShakenAniGroupListRelSchemeModify(ShakenAniGroupListRelScheme):
    pass
class ResourceShakenAniGroupListRelSchemeModify(ShakenAniGroupListRelScheme):
    class Meta:
        model = model.ShakenAniGroupListRel
        fields = ('ani_group_id','attest_lvl',)
#endregion ---ShakenAniGroupListRel---

# region +++ShakenAniGroupRel+++


class ShakenAniGroupRelScheme(BaseModelScheme):
    group_id = Int(required=True)
    did = Str(validate=[validate.Length(min=1, max=16), lambda value: _valid_unique('ShakenAniGroupRel', 'did', value)], required=True)
    created_on = DateTime()

    class Meta:
        model = model.ShakenAniGroupRel
        fields = ('group_id', 'did')


class ShakenAniGroupRelSchemeGet(ShakenAniGroupRelScheme):
    class Meta:
        model = model.ShakenAniGroupRel
        fields = ('group_id', 'did', 'created_on', 'created_by',)
        search_fields = ('group_id', 'did',)
        query_fields = ('created_on_gt', 'created_on_lt',)


class ShakenAniGroupRelSchemeModify(ShakenAniGroupRelScheme):
    pass

# endregion ---ShakenAniGroupRel---

#region +++ShakenAniGroupRelImportTask+++
class ShakenAniGroupRelImportTaskScheme(BaseModelScheme):
    id = Int()
    operator_user = Str(validate=[validate.Length(max=40)])
    upload_file_path = Str(validate=[validate.Length(max=256)])
    upload_orig_file = Str(validate=[validate.Length(max=100)])
    upload_format_file = Str(validate=[validate.Length(max=100)])
    orig_name = Str(validate=[validate.Length(max=256)])
    repeated_action = Choice()
    status = Choice()
    progress = Str()
    create_time = DateTime()
    start_time = DateTime()
    end_time = DateTime()
    op_method = Choice()
    group_id = Int()
    first_number = Int()
    last_number = Int()
    class Meta:
        model = model.ShakenAniGroupRelImportTask
        fields = ('operator_user','upload_file_path','upload_orig_file','upload_format_file','orig_name','repeated_action','status','progress','create_time','start_time','end_time','op_method','group_id','first_number','last_number',)
class ShakenAniGroupRelImportTaskSchemeGet(ShakenAniGroupRelImportTaskScheme):
    class Meta:
        model = model.ShakenAniGroupRelImportTask
        fields = ('id','total','fail','success','duplicate','operator_user','upload_file_path','upload_orig_file','upload_format_file','orig_name','repeated_action','status','progress','create_time','start_time','end_time','op_method','group_id','first_number','last_number',)
        search_fields = ('id','total','fail','success','duplicate','operator_user','upload_file_path','upload_orig_file','upload_format_file','orig_name','repeated_action','status','progress','op_method','group_id','first_number','last_number',)
        query_fields=('create_time_gt','create_time_lt','start_time_gt','start_time_lt','end_time_gt','end_time_lt',)
class ShakenAniGroupRelImportTaskWarningsSchemeGet(ShakenAniGroupRelImportTaskScheme):
    class Meta:
        model = model.ShakenAniGroupRelImportTask
        fields = ('warnings',)
        search_fields = ()
        query_fields=()
class ShakenAniGroupRelImportTaskSchemeModify(ShakenAniGroupRelImportTaskScheme):
    pass
#endregion ---ShakenAniGroupRelImportTask---