from marshmallow import validate, Schema
from marshmallow.fields import Bool, Str, Decimal, Int, Nested, DateTime, Float, List, Function, Dict, URL

from api_dnl import model
from api_dnl.fields import Choice
from api_dnl.scheme import BaseModelScheme, Emails, ClientPaymentScheme, PaymentGatewayHistoryGetScheme, \
    RateTableGetScheme, RateGetScheme, DidBillingRelGetScheme, DidBillingPlanGetScheme, PaymentGatewayHistoryScheme, \
    _valid, ResourcePrefixGetScheme, ResourceIpGetScheme, EmailLogGetScheme, ActualBalanceHistoryGetScheme, \
    IngressTrunkScheme, IngressTrunkModifyScheme, EgressTrunkScheme, DidVendorScheme, ResourceWithIpsScheme, \
    ResourcePrefixScheme, ResourcePrefixGetScheme, ResourcePrefixModifyScheme, PaymentCheckoutScheme, \
    ResourcePrefixWithIPScheme, ResourceWithIpAndAliasGetScheme, DidNumberUploadScheme, DidNumberUploadSchemeGet, \
    ResourceIpPortScheme, DidRepositoryGetScheme

from api_dnl.schemes.invoice import InvoiceGetScheme
from api_dnl.scheme import IP_REGEXP, NAME_REGEXP, PREFIX_REGEXP


class ClientPortalScheme(BaseModelScheme):
    # gen   get /client/account_summary
    # Get Account Summary
    # status - active , inactive
    # billing_mode - prepay, postpay
    status = Bool()
    billing_mode = Choice(dump_only=True)
    address = Str()
    company_name = Str()
    main_email = Emails()
    phone = Str()
    noc_email = Str()
    rate_email = Emails()
    rate_delivery_email = Emails()
    billing_email = Emails()
    credit_limit = Decimal(dump_only=True)
    profit_margin = Decimal(allow_none=True, validate=validate.Range(min=0.0, max=1000000.0))
    balance = Decimal(dump_only=True)
    ingress_count = Int()
    ingress_trunks = Nested('ClientPortalIngressTrunkInnerScheme', many=True, dump_only=True)
    egress_count = Int(dump_only=True)
    egress_trunks = Nested('ClientPortalEgressTrunkInnerScheme', many=True, dump_only=True)
    origination_trunks = Nested('ClientPortalOriginationTrunkInnerScheme', many=True, dump_only=True)
    unpaid_invoice_count = Int(dump_only=True)
    unread_alert_count = Int(dump_only=True)
    unread_message_count = Int(dump_only=True)
    # alert settings
    is_show_daily_usage = Bool()
    is_daily_balance_notification = Bool()
    zero_balance_notice = Bool()
    ingress_ids = Str(dump_only=True)
    egress_ids = Str(dump_only=True)
    resource = Nested('ClientPortalEgressTrunkInnerScheme', many=False, dump_only=True)

    class Meta:
        model = model.Client
        fields = ('billing_mode', 'egress_count', 'status', 'unpaid_invoice_count', 'phone', 'egress_trunks', 'address',
                  'credit_limit', 'ingress_trunks', 'main_email', 'unread_alert_count', 'company_name', 'noc_email',
                  'ingress_count', 'unread_message_count', 'balance', 'billing_email', 'rate_email',
                  'rate_delivery_email',
                  'origination_trunks', 'ingress_ids', 'egress_ids', 'resource',
                  'is_show_daily_usage', 'is_daily_balance_notification', 'zero_balance_notice')


class ClientPortalPaymentGetScheme(ClientPaymentScheme):
    client_payment_id = Int()
    carrier_name = Str()
    client_id = Int()
    payment_type = Choice()
    company_type = Choice()

    class Meta:
        model = model.ClientPayment
        fields = ('client_payment_id', 'client_id', 'carrier_name', 'amount', 'balance',
                  'note', 'paid_on', 'update_by', 'payment_time', 'payment_type', 'company_type')
        search_fields = ('client_payment_id', 'payment_type')
        query_fields = ('paid_on_gt', 'paid_on_lt', 'amount_gt', 'amount_lt')


class ClientPortalPaymentGatewayHistoryGetScheme(PaymentGatewayHistoryGetScheme):
    class Meta:
        model = model.PaymentGatewayHistory
        fields = ('id', 'client_id', 'client_name', 'amount', 'type',
                  'paid_on', 'entered_on', 'paypal_id', 'paypal_fee',
                  'actual_received', 'strip_id', 'strip_transaction_id', 'paypal_transaction_id',
                  'status', 'return_code', 'error', 'response', 'cardnumber', 'cardexpmonth', 'cardexpyear')
        search_fields = ('client_id',
                         'type', 'paypal_id', 'paypal_transaction_id', 'cardnumber', 'status')
        query_fields = ('paid_on_gt', 'paid_on_lt')


class ClientPortalGatewayPaymentGetScheme(PaymentGatewayHistoryGetScheme):
    charge_amount = Decimal()
    fee = Decimal()

    class Meta:
        model = model.PaymentGatewayHistory
        fields = ('transaction_id', 'client_id', 'client_name', 'amount',
                  'fee', 'paid_on', 'charge_amount', 'method')
        search_fields = ('transaction_id', 'client_id', 'client_name', 'method')
        query_fields = ('paid_on_gt', 'paid_on_lt', 'amount_gt', 'amount_lt',
                        'charge_amount_gt', 'charge_amount_lt', 'fee_gt', 'fee_lt',)


class ClientPortalGatewayAndPaymentsGetScheme(BaseModelScheme):
    METHOD = {
        '1': 'stripe',
        '0': 'paypal',
        'manual': 'manual'
    }
    STATUS = {
        0: 'initial',
        '0': 'initial', 
        1: 'error',
        '1': 'error',
        2: 'success',
        '2': 'success'
    }
    transaction_id = Str()
    client_id = Int()
    client_name = Str()
    amount = Decimal()
    paid_on = DateTime()
    method = Function(lambda obj: ClientPortalGatewayAndPaymentsGetScheme.METHOD.get(obj.method, ''))
    charge_amount = Decimal()
    fee = Decimal()
    note = Str()
    status = Function(lambda obj: ClientPortalGatewayAndPaymentsGetScheme.STATUS.get(obj.status, ''))
    invoice_id = Str()

    class Meta:
        model = model.PaymentAndGateway
        fields = ('transaction_id', 'client_id', 'client_name', 'amount',
                  'fee', 'paid_on', 'charge_amount', 'method', 'note', 'status', 'invoice_id')
        search_fields = ('transaction_id', 'client_id', 'client_name', 'method', 'note')
        query_fields = ('paid_on_gt', 'paid_on_lt', 'amount_gt', 'amount_lt',
                        'charge_amount_gt', 'charge_amount_lt', 'fee_gt', 'fee_lt',)


class ClientPortalProductRoutRateTableScheme(BaseModelScheme):
    name = Str(allow_none=False, required=True, validate=validate.Regexp(NAME_REGEXP))
    route_plan_id = Int(allow_none=False, required=True,
                        validate=lambda value: _valid('RouteStrategy', 'route_strategy_id', value))
    rate_table_id = Int(allow_none=False, required=True,
                        validate=lambda value: _valid('RateTable', 'rate_table_id', value))
    tech_prefix = Str(validate=validate.Regexp(PREFIX_REGEXP))
    agent_id = Int(validate=lambda value: _valid('Agent', 'agent_id', value))
    type = Choice()
    agents = List(Int(validate=lambda value: _valid('Agent', 'agent_id', value)))
    clients = List(Int(validate=lambda value: _valid('Client', 'client_id', value)))
    description = Str()

    class Meta:
        model = model.ProductRoutRateTable
        fields = ('name', 'route_plan_id', 'rate_table_id',
                  'tech_prefix', 'type', 'description')


class ClientPortalProductRoutRateTableGetScheme(BaseModelScheme):
    name = Str()
    route_plan_id = Int()
    rate_table_id = Int()
    route_plan_name = Str()
    rate_table_name = Str()
    tech_prefix = Int()
    type = Choice()
    description = Str()
    ingress_trunk_id = Str()
    rate_download_link = Str()

    # clients = Nested('CarrierGetMinScheme',many=True)
    class Meta:
        model = model.ProductRoutRateTable
        fields = ('id', 'name', 'route_plan_id', 'rate_table_id', 'route_plan_name',
                  'rate_table_name', 'tech_prefix', 'type', 'description', 'rate_download_link')
        search_fields = ('name', 'route_plan_id', 'rate_table_id', 'ingress_trunk_id')



class ClientPortalProductGetScheme(BaseModelScheme):
    product_name = Str(attribute='name')
    product_id = Int()
    route_plan_id = Int()
    rate_table_id = Int()
    route_plan_name = Str()
    rate_table_name = Str()
    tech_prefix = Int()
    type = Choice()
    description = Str()
    resource_id = Str()
    rate_download_link = Str()
    trunks = Nested('ResourceWithIpAndAliasGetScheme', many=True)
    rate_last_sent = DateTime()

    # clients = Nested('CarrierGetMinScheme',many=True)
    class Meta:
        model = model.ProductClientsRefUsed
        fields = ('product_id', 'product_name', 'route_plan_id', 'rate_table_id', 'route_plan_name',
                  'rate_table_name', 'tech_prefix', 'type', 'description', 'rate_download_link', 'trunks',
                  'rate_last_sent')
        search_fields = ('product_id', 'name', 'route_plan_id', 'rate_table_id', 'resource_id')
        query_fields = ('rate_last_sent_gt', 'rate_last_sent_lt')


class ClientPortalProductClientsRefGetScheme(BaseModelScheme):
    name = Str()
    product_id = Int()
    route_plan_id = Int()
    rate_table_id = Int()
    route_plan_name = Str()
    rate_table_name = Str()
    tech_prefix = Int()
    type = Choice()
    description = Str()
    ingress_trunk_id = Str()
    rate_download_link = Str()
    trunks = Nested('ResourceWithIpAndAliasGetScheme', many=True)
    rate_last_sent = Str()

    # clients = Nested('CarrierGetMinScheme',many=True)
    class Meta:
        model = model.ProductClientsRefUsed
        fields = ('product_id', 'name', 'route_plan_id', 'rate_table_id', 'route_plan_name',
                  'rate_table_name', 'tech_prefix', 'type', 'description', 'rate_download_link', 'trunks',
                  'rate_last_sent')
        search_fields = ('product_id', 'name', 'route_plan_id', 'rate_table_id', 'ingress_trunk_id')
        query_fields = ('rate_last_sent_gt', 'rate_last_sent_lt')


class ClientPortalRateTableGetScheme(RateTableGetScheme):
    rate_download_link = Str()
    trunk_id = Int(validate=lambda value: _valid('Resource', 'resource_id', value))

    class Meta:
        model = model.RateTable
        fields = ('rate_download_link',) + RateTableGetScheme.Meta.fields
        search_fields = ('trunk_id',) + RateTableGetScheme.Meta.search_fields
        query_fields = RateTableGetScheme.Meta.query_fields


class ClientPortalRateGetScheme(RateGetScheme):
    product_id = Int()

    class Meta:
        model = model.Rate
        fields = RateGetScheme.Meta.fields
        search_fields = ('product_id',) + RateGetScheme.Meta.search_fields
        query_fields = RateGetScheme.Meta.query_fields


class ClientPortalDidBillingRelAssignScheme(BaseModelScheme):
    client_billing_rule_id = Int(validate=lambda value: _valid('DidBillingPlan', 'id', value))
    assign = Bool()

    class Meta:
        model = model.DidBillingRel
        fields = ('client_billing_rule_id', 'assign')


class ClientPortalDidBillingRelDisableScheme(BaseModelScheme):
    is_available = Bool()

    class Meta:
        model = model.DidBillingRel
        fields = ('is_available',)


class ClientPortalDidRepositoryGetScheme(BaseModelScheme):
    did_vendor_id = Int(validate=lambda value: _valid('Resource', 'resource_id', value))

    class Meta:
        model = model.DidRepository
        fields = ('id', 'did', 'vendor_trunk_name', 'vendor_billing_plan_id', 'created_at',
                  'created_by', 'did_vendor_id', 'did_vendor_name', 'vendor_billing_rule_name',
                  'client_id', 'did_client_name', 'client_billing_rule_name', 'client_trunk_id')
        search_fields = ('id', 'did', 'vendor_trunk_name', 'vendor_billing_plan_id', 'created_at',
                  'created_by', 'did_vendor_name', 'vendor_billing_rule_name',
                  'client_id', 'client_billing_plan_id', 'fallback_id', 'client_trunk_id',
                  'client_billing_rule_name', 'did_client_name', 'country', 'did_client_name_is_null',
                  'state', 'is_available', 'is_assigned', 'active', 'status', 'vendor_id',
                  'client_id', 'did_client_name', 'client_billing_rule_name'
                  )
    # did_client_is_online = Bool()
    # did_type = Str()
    # status = Choice()
    # is_sms = Bool()
    # did = Str()

    # class Meta:
    #     model = model.DidBillingRel
    #     fields = DidBillingRelGetScheme.Meta.fields #+ ('did_client_is_online', 'did_type', 'status', 'is_sms')
    #     search_fields = DidBillingRelGetScheme.Meta.search_fields + (
    #     'did_client_is_online', 'did_type', 'status', 'is_sms')
    #     query_fields = DidBillingRelGetScheme.Meta.query_fields


class ClientPortalFreeDidBillingRelGetScheme(DidBillingRelGetScheme):
    client_billing_rule = Nested(DidBillingPlanGetScheme)

    class Meta:
        model = model.DidBillingRel
        fields = DidBillingRelGetScheme.Meta.fields + ('client_billing_rule',)
        search_fields = DidBillingRelGetScheme.Meta.search_fields + ('client_billing_rule',)
        query_fields = DidBillingRelGetScheme.Meta.query_fields


class ClientPortalInvoiceGetScheme(InvoiceGetScheme):
    pay_amount = Float()

    class Meta:
        model = model.Invoice
        fields = InvoiceGetScheme.Meta.fields + ('pay_amount',)
        search_fields = InvoiceGetScheme.Meta.search_fields
        query_fields = InvoiceGetScheme.Meta.query_fields + ('pay_amount_gt', 'pay_amount_lt')



class ClientPortalPaymentGatewayHistoryScheme(PaymentGatewayHistoryScheme):
    pass


class ClientPortalIngressTrunkScheme(IngressTrunkScheme):
    pass


class ClientPortalIngressTrunkModifyScheme(IngressTrunkModifyScheme):
    class Meta:
        model = model.IngressTrunk
        fields = tuple(set(IngressTrunkModifyScheme.Meta.fields).difference(('is_active',)))


class ClientPortalIngressTrunkInnerScheme(BaseModelScheme):
    ingress_name = Str(attribute='alias', validate=validate.Length(max=100))
    ingress_id = Str(attribute='resource_id', validate=lambda value: _valid('IngressTrunk', 'resource_id', value))
    product_name = Str(validate=validate.Length(max=100))
    prefixes = Nested(ResourcePrefixGetScheme, many=True)  # Str()
    call_limit = Int(attribute='capacity', validate=validate.Range(0, 100000))
    cps_limit = Int(validate=validate.Range(0, 100000))
    host = Nested(ResourceIpGetScheme, many=True)
    is_active = Bool()
    status = Choice()
    price_per_max_channel = Float(validate=validate.Range(0, 100000))

    class Meta:
        model = model.IngressTrunk
        fields = (
            'ingress_id', 'ingress_name', 'product_name', 'prefixes', 'host', 'call_limit', 'cps_limit',
            'price_per_max_channel', 'is_active', 'status')


class ClientPortalEgressTrunkScheme(EgressTrunkScheme):
    pass


class ClientPortalEgressTrunkModifyScheme(EgressTrunkScheme):  # (EgressTrunkModifyScheme):
    pass


class ClientPortalResourceIpPortScheme(ResourceIpPortScheme):
    pass

class ClientPortalEgressTrunkInnerScheme(BaseModelScheme):
    egress_name = Str(attribute='alias', validate=validate.Length(max=100))
    egress_id = Str(attribute='resource_id', validate=lambda value: _valid('EgressTrunk', 'resource_id', value))
    rate_table_name = Str(validate=validate.Length(max=100))
    rate_table_id = Int()
    # prefixes = Nested(ResourcePrefixGetScheme, many=True)  # Str()
    call_limit = Int(attribute='capacity')
    cps_limit = Int(validate=validate.Range(0, 100000))
    host = Nested(ResourceIpGetScheme, many=True)
    is_active = Bool()
    status = Choice()
    price_per_max_channel = Float(validate=validate.Range(0, 100000))

    class Meta:
        model = model.EgressTrunk
        fields = (
            'egress_id', 'egress_name', 'rate_table_name', 'host', 'call_limit', 'cps_limit', 'is_active', 'status',
            'price_per_max_channel', 'rate_table_id')


class ClientPortalOriginationTrunkInnerScheme(BaseModelScheme):
    trunk_name = Str(attribute='alias', validate=validate.Length(max=100))
    resource_id = Str(validate=lambda value: _valid('Resource', 'resource_id', value))
    ingress = Bool()
    egress = Bool()
    # prefixes = Nested(ResourcePrefixGetScheme, many=True)  # Str()
    call_limit = Int(attribute='capacity')
    cps_limit = Int(validate=validate.Range(0, 100000))
    host = Nested(ResourceIpGetScheme, many=True)
    is_active = Bool()
    status = Choice()

    class Meta:
        model = model.EgressTrunk
        fields = (
            'resource_id', 'trunk_name', 'host', 'call_limit', 'cps_limit', 'is_active', 'status', 'ingress', 'egress')


class ClientPortalEmailLogAlertsScheme(EmailLogGetScheme):
    class Meta:
        model = model.EmailLog
        fields = EmailLogGetScheme.Meta.fields
        search_fields = EmailLogGetScheme.Meta.search_fields
        query_fields = EmailLogGetScheme.Meta.query_fields


class ClientPortalActualBalanceHistoryGetScheme(ActualBalanceHistoryGetScheme):
    invoice_received = Decimal(allow_none=True, allow_nan=True)
    invoice_sent = Decimal()

    class Meta:
        model = model.BalanceHistoryActual
        fields = ('invoice_received', 'invoice_sent') + ActualBalanceHistoryGetScheme.Meta.fields
        search_fields = ActualBalanceHistoryGetScheme.Meta.search_fields
        query_fields = ActualBalanceHistoryGetScheme.Meta.query_fields


class ClientDefaultIpScheme(BaseModelScheme):
    ip = Str(validate=validate.Regexp(IP_REGEXP))
    port = Int(default=5060, validate=validate.Range(1000, 65535))
    mask = Int(default=32, validate=validate.OneOf((1, 2, 8, 16, 32)))
    product_id = Int(validate=lambda value: _valid('ProductRoutRateTable', 'id', value, 'invalid product'))

    class Meta:
        model = model.ClientDefaultIp
        fields = ('ip', 'port', 'mask', 'product_id')


class ClientDefaultIpGetScheme(ClientDefaultIpScheme):
    id = Int()

    class Meta:
        model = model.ClientDefaultIp
        fields = ('id', 'ip', 'port', 'mask', 'product_id')
        search_fields = ('id', 'ip', 'port', 'mask', 'product_id')


class ClientPortalResourceWithIpsScheme(ResourceWithIpsScheme):
    pass


class ClientPortalResourcePrefixWithIPScheme(ResourcePrefixWithIPScheme):
    pass


class ClientPortalResourcePrefixScheme(ResourcePrefixScheme):
    pass


class ClientPortalResourcePrefixGetScheme(ResourcePrefixGetScheme):
    pass


class ClientPortalResourcePrefixModifyScheme(ResourcePrefixModifyScheme):
    pass


class ClientPortalProductSendToEmailsScheme(BaseModelScheme):
    emails = Emails()

    class Meta:
        model = model.ProductRoutRateTable
        fields = ('emails',)


# region DidApi
class ClientPortalDidVendorSchemeGet(DidVendorScheme):
    class Meta:
        model = model.DidVendor
        fields = ('id', 'type', 'username', 'sms_enabled', 'rate_per_sms_sent', 'rate_per_sms_received',
                  'default_received_sms_url', 'mrc', 'nrc',)
        search_fields = ('id', 'type', 'username', 'password', 'sms_enabled', 'default_received_sms_url', 'mrc', 'nrc',)
        query_fields = (
            'rate_per_sms_sent_gt', 'rate_per_sms_sent_lt', 'rate_per_sms_received_gt', 'rate_per_sms_received_lt',)


class DidApiSearchScheme(Schema):
    state = Str(validate=validate.Length(min=2, max=2))
    npa = Str(validate=[validate.Length(min=3, max=7)])
    nxx = Str(validate=[validate.Length(min=3, max=4)])
    lata = Str(validate=[validate.Length(min=3, max=8)])
    country = Str(validate=[validate.Length(min=0, max=3)])
    pattern = Str(validate=[validate.Length(equal=10)])
    per_page = Int(validate=validate.Range(min=1, max=100))
    page = Int(validate=validate.Range(min=0, max=100))
    is_sms = Bool()
    order_by = Str()
    order_dir = Str()


class DidApiSearchTollFreeScheme(Schema):
    pattern = Str(validate=[validate.Length(max=13)])
    per_page = Int(validate=validate.Range(min=1, max=100))
    page = Int(validate=validate.Range(min=0))
    is_sms = Bool()


class DidApiSearchGetScheme(Schema):
    number = Str()
    state = Str()
    npa = Str()
    lata = Str()
    mrc = Str()
    nrc = Str()


class DidApiSearchTollFreeGetScheme(Schema):
    number = Str()
    cic = Str()
    org_prefix = Str()


class DidApiDisconnectLocalInnerScheme(Schema):
    number = Str()
    buy_billing_plan_id = Int()


class DidApiOrderScheme(Schema):
    number = Str()
    did_repository_id = Int()
    buy_billing_plan_id = Int()


class DidApiDisconnectScheme(Schema):
    items = Nested(DidApiDisconnectLocalInnerScheme, many=True)


class DidApiSearchCoverageScheme(Schema):
    country = Str()
    state = Str(validate=[validate.Length(min=2, max=2),
                          lambda value: _valid('JurisdictionPrefix', 'jurisdiction_name', value)])
    npa = Str(validate=[validate.Length(min=3, max=7)])
    lata = Str(validate=[validate.Length(min=3, max=8)])
    group = Str()

class DidApiSearchCoverageGetScheme(Schema):
    country = Str()
    state = Str()
    npa = Str()
    lata = Str()
    vendor_id = Int()
    tn_count = Int()


class DidApiSearchOrderLocalScheme(Schema):
    state = Str(validate=[validate.Length(min=2, max=2),
                          lambda value: _valid('JurisdictionPrefix', 'jurisdiction_name', value)])
    npa = Str(validate=[validate.Length(min=3, max=7)])
    lata = Str(validate=[validate.Length(min=3, max=8)])
    count = Int(validate=validate.Range(min=1, max=100))


class DidApiOrderLocalGetScheme(Schema):
    number = Str()
    state = Str()
    npa = Str()
    lata = Str()
    mrc = Str()
    nrc = Str()


class DidApiOrderLocalScheme(Schema):
    items = Nested(DidApiOrderScheme, many=True)


class DidApiOrderTollFreeLocalInnerScheme(Schema):
    number = Str()
    did_repository_id = Int()


class DidApiOrderTollFreeScheme(Schema):
    items = Nested(DidApiOrderTollFreeLocalInnerScheme, many=True)


# endregion
# region +++Sms+++
class SmsDlrHookScheme(BaseModelScheme):
    dlr_url = URL()
    dlr_fail_url = URL()

    class Meta:
        model = model.SmsDlrHook
        fields = ('dlr_url', 'dlr_fail_url')


class SmsScheme(BaseModelScheme):
    id = Int()
    client_id = Int()
    created_on = DateTime()
    sender = Str(validate=[validate.Length(max=128)])
    receiver = Str(validate=[validate.Length(max=128)])
    message = Str(validate=[validate.Length(max=256)])
    cost = Float()
    dlr_hook = Nested(SmsDlrHookScheme)

    class Meta:
        model = model.Sms
        fields = ('sender', 'receiver', 'message', 'dlr_hook')


class SmsSchemeGet(SmsScheme):
    sender = Str(attribute='sender_plus')
    receiver = Str(attribute='receiver_plus')
    direction = Choice()
    stat = Str()
    delivered_on = DateTime()
    delivery_error = Str()
    cost = Float()
    vendor_cost = Float()
    client_name = Str()
    vendor_name = Str()

    class Meta:
        model = model.Sms
        fields = ('id', 'created_on', 'client_id', 'sender', 'receiver', 'message', 'direction', 'reference_id', 'stat',
                  'delivered_on', 'delivery_error', 'cost', 'vendor_cost', 'client_name', 'vendor_name')
        search_fields = (
            'id', 'client_id', 'sender', 'receiver', 'message', 'direction', 'reference_id', 'stat', 'delivery_error',
            'client_name', 'vendor_name')
        query_fields = ('created_on_gt', 'created_on_lt', 'cost_gt', 'cost_lt', 'delivered_on_gt', 'delivered_on_lt',)


class SmsSchemeModify(SmsScheme):
    pass


# endregion ---Sms---
# region +++NpaLata+++
class NpaSearchScheme(Schema):
    npa = Str(validate=[validate.Length(min=3, max=3)])


class LataSearchScheme(Schema):
    lata = Str(validate=[validate.Length(min=3, max=3)])


class StateSearchScheme(Schema):
    state = Str(validate=[validate.Length(min=2, max=2)])


# endregion ---NpaLata---

# region +++WebHook+++
class WebHookScheme(BaseModelScheme):
    id = Int()
    client_id = Int()
    number = Str(validate=[validate.Length(max=128)])
    url = Str(validate=[validate.Length(max=1024)])
    header_vars = Dict()
    created_on = DateTime()

    class Meta:
        model = model.WebHook
        fields = ('number', 'url', 'header_vars',)


class WebHookSchemeGet(WebHookScheme):
    class Meta:
        model = model.WebHook
        fields = ('id', 'url', 'header_vars', 'number')
        search_fields = ('id', 'number', 'url',)
        query_fields = ('created_on_gt', 'created_on_lt',)


class WebHookSchemeModify(WebHookScheme):
    pass


# endregion ---WebHook---

# region +++ApiKey+++

class ApiKeyScheme(Schema):
    pass


class ClientApiKeyGetScheme(BaseModelScheme):
    token = Str()
    expired_on = DateTime()

    class Meta:
        model = model.UserApiKey
        fields = ('token', 'expired_on')

# endregion ---ApiKey--
