from sqlalchemy import Column
from urllib.parse import urlencode,unquote, quote_plus
from falcon_rest.db import fields, orm
from falcon_rest.db import orm, Column, fields, errors
from falcon_rest import schemes, helpers
from falcon_rest.conf import settings
from falcon_rest.logger import log
from falcon_rest.resources import resources
from falcon_rest.contrib.objects_history.auth_user import AuthUser

# from falcon_rest.contrib.auth.auth import get_hashed_password
from sqlalchemy import Column, ForeignKey
from sqlalchemy.exc import InternalError
from sqlalchemy.ext.hybrid import hybrid_property, hybrid_method
from sqlalchemy.orm import backref, foreign, relationship, synonym, column_property
from sqlalchemy.sql import and_, or_, not_, outerjoin
from sqlalchemy.exc import OperationalError
# from . user_auth_ip import UserAuthIp
from api_dnl.base_model import DnlApiBaseModel
from api_dnl.fields.choice import ChoiceType

# from . role import Role
import bcrypt
import jwt
from datetime import datetime, timedelta
from pytz import UTC
from api_dnl.auth import get_token
from sqlalchemy.sql import func, select, literal_column, case


def get_hashed_password(plain_text_password):
    if not plain_text_password:
        import random
        plain_text_password = str(round(random.random() * 1000000000, 0))
    if isinstance(plain_text_password, type('')):
        plain_text_password = plain_text_password.encode()
    hash = bcrypt.hashpw(
        plain_text_password,
        bcrypt.gensalt()
    )
    if isinstance(hash, type(b'')):
        hash = hash.decode()
    return hash


class UserAuthIp(DnlApiBaseModel):
    __tablename__ = 'user_auth_ip'
    id = Column(fields.Integer, primary_key=True)
    user_id = Column(fields.ForeignKey('users.user_id', ondelete='CASCADE'))
    ip = Column(fields.String(100))
    user = relationship('User', uselist=False, back_populates='auth_ip')


class UserCodedeckAlerts(DnlApiBaseModel):
    __tablename__ = 'user_codedeck_alerts'
    id = Column(fields.Integer, primary_key=True)
    user_id = Column(fields.ForeignKey('users.user_id', ondelete='CASCADE'))
    show_az_codedeck = Column(fields.Boolean, nullable=False, default=True)
    show_us_codedeck = Column(fields.Boolean, nullable=False, default=True)
    user = relationship('User', uselist=False, back_populates='codedeck_alerts')

    @property
    def has_updated_az_codedeck(self):
        from api_dnl.model import SystemParameter
        conf = SystemParameter.get(1)
        if self.user.user_type == 'admin' and self.show_az_codedeck:
            try:
                if conf.codedeck_prompt_az_update > conf.codedeck_last_az_update:
                    return True
            except:
                return False
        return False

    @property
    def has_updated_us_codedeck(self):
        from api_dnl.model import SystemParameter
        conf = SystemParameter.get(1)
        if self.user.user_type == 'admin' and self.show_us_codedeck:
            try:
                if conf.codedeck_prompt_us_update > conf.codedeck_last_us_update:
                    return True
            except:
                return False
        return False


class User(DnlApiBaseModel, AuthUser):
    __tablename__ = 'users'
    USER_TYPE_CHOICES = [
        (0, 'admin'),
        (1, 'client'),
        (2, 'agent'),
        (None, 'viewer')
    ]
    user_id = Column(fields.Integer, primary_key=True)
    name = Column(fields.String(60), nullable=False, unique=True)
    password = Column(fields.String(60), nullable=False)
    role_id = Column(fields.ForeignKey('role.role_id', ondelete='SET NULL'), index=True)
    reseller_id = Column(fields.Integer)
    create_time = Column(fields.DateTime(timezone=True), nullable=False, server_default=orm.func.now())
    active = Column(fields.Boolean, nullable=False, default=True)
    client_id = Column(fields.ForeignKey('client.client_id', ondelete='CASCADE'), nullable=True, index=True)
    email = Column(fields.String(100))
    first_name = Column(fields.String(40))
    last_name = Column(fields.String(40))
    # user_type = Column(fields.Integer, nullable=False)
    user_type = Column(ChoiceType(USER_TYPE_CHOICES), nullable=False)
    create_user_id = Column(fields.Integer)
    last_login_time = Column(fields.DateTime(timezone=True))
    card_id = Column(fields.Integer)
    is_online = Column(fields.Integer, nullable=False, default=2)
    login_ip = Column(fields.String(40))
    default_mod = Column(fields.String(100), default='')
    default_mod2 = Column(fields.String(100), default='')
    last_seen = Column(fields.DateTime(timezone=True))
    report_group = Column(fields.Boolean, nullable=False, default=True)
    outbound_report = Column(fields.Boolean, nullable=False, default=True)
    all_termination = Column(fields.Boolean, nullable=False, default=True)
    show_carrier_trunk_drop_only = Column(fields.Boolean, nullable=False, default=False)
    report_fields = Column(fields.Text)
    landing_page = Column(fields.Text)
    cdr_api_token = Column(fields.String(64))
    cdr_expire = Column(fields.DateTime(timezone=True))
    avatar_id = Column(fields.Integer)
    # last_notification_time = Column(fields.DateTime(timezone=True))

    fullname = column_property(first_name + ' ' + last_name)

    login_url = synonym('landing_page')
    username = column_property(case([(first_name.isnot(None),first_name + ' ' + last_name)],else_=name))

    role = orm.relationship('Role', back_populates='users', uselist=False)
    agent = relationship('Agent', primaryjoin='foreign(Agent.user_id) == User.user_id', back_populates='user',
                         uselist=False)
    client = relationship('Client', primaryjoin='foreign(Client.user_id) == User.user_id', back_populates='user',
                          uselist=False, enable_typechecks=False)
    client_limits = relationship('UsersLimit', uselist=True, back_populates='user', single_parent=True,
                                 cascade="all, delete-orphan")
    auth_ip = relationship('UserAuthIp', uselist=True, back_populates='user', single_parent=True,
                           cascade="all, delete-orphan", lazy="dynamic")

    codedeck_alerts = relationship('UserCodedeckAlerts', uselist=False, back_populates='user',
                                   cascade="all, delete-orphan")

    api_key = relationship('UserApiKey', uselist=False, cascade="all, delete-orphan")

    def before_save(self):
        # if not self.user_id:
        if self.user_type == 'admin':
            if not self.codedeck_alerts:
                ar = UserCodedeckAlerts(show_az_codedeck=True, show_us_codedeck=True)
                self.codedeck_alerts = ar
        else:
            self.codedeck_alerts = None

    @property
    def is_admin(self):
        return self.user_type == 'admin'

    @property
    def soon_expired_gateways(self):
        from api_dnl.model import VoipGateway
        if self.is_admin:
            return VoipGateway.expired7()
        else:
            return None

    @property
    def is_active(self):
        if self.user_type == 'client' and self.client:
            return self.active and self.client.status
        if self.user_type == 'agent' and self.agent:
            return self.active and self.agent.status
        return self.active

    @property
    def referal_key(self):
        if self.user_type == 'agent' and self.agent:
            return self.agent.referal_key
        return None

    @hybrid_property
    def passwd(self):
        return self.password

    @passwd.setter
    def passwd(self, value):
        self.password = get_hashed_password(value)

    @classmethod
    def get_user_from_credentials(cls, credentials):
        try:
            user = cls.filter(
                orm.or_(User.email == credentials['email_or_name'], User.name == credentials['email_or_name'])
            ).first()
            if user:  # .is_active:
                return user
            return None
        except OperationalError as e:  # errors.NoResultFound:
            log.error('database error {}'.format(e))
            # cls.query().session.rollback()
            raise e
        except InternalError as e:  # errors.NoResultFound:
            log.error('database error {}'.format(e))
            cls.query().session.rollback()
            raise e

    def get_token_data(self):
        return dict(user_id=self.user_id)

    @classmethod
    def get_user_from_token_data(cls, token_data):
        try:
            user = cls.filter(User.user_id == token_data['user_id']).filter(User.active == True).one()
            if user.is_active:
                return user
            return None
        except:
            cls.query().session.rollback()
            return None

    @classmethod
    def get_user_by_id(cls, user_id):
        try:
            user = cls.filter(User.user_id == user_id).filter(User.active == True).one()
            if user.is_active:
                return user
            return None
        except errors.NoResultFound:  # pragma: no cover
            return None

    def can_login(self, req, resp):
        from api_dnl.model import Signup
        if not self.can_login_ip(req, resp):
            return False
        if self.is_active:
            return True
        if self.client and not self.client.enable_client_portal:
            resources.BaseResource.set_response(
                resp, resources.responses.UnAuthorizedErrorResponse(
                    data=resources.errors.Error(
                        code,
                        msg,
                        'user\'s client portal is disabled'
                    )
                )
            )
            return False
        msg = 'username disabled by the administrator'
        code = 1000
        sig = Signup.filter(Signup.login == self.name).first()
        if sig:
            if sig.status == 'rejected':
                msg = 'username rejected by the administrator'
                code = 1001
            elif sig.status == 'pending':
                msg = 'username not yet approved by the administrator'
                code = 1002

        resources.BaseResource.set_response(
            resp, resources.responses.UnAuthorizedErrorResponse(
                data=resources.errors.Error(
                    code,
                    msg,
                    'user not active'
                )
            )
        )
        return False

    def can_login_ip(self, req, resp):
        request_ip = helpers.get_request_ip(req)
        if request_ip in settings.ALWAYS_ALLOWED_AUTH_IPS:
            return True
        if self.auth_ip.count():
            match = self.auth_ip.filter(UserAuthIp.ip == request_ip).first()
            if match:
                return True
        else:
            return True

        resources.BaseResource.set_response(
            resp, resources.responses.UnAuthorizedErrorResponse(
                data=resources.errors.Error(
                    1003,
                    'You are not allowed to login from the IP {}'.format(request_ip),
                    'ip_not_allowed'
                )
            )
        )
        return False

    # noinspection PyMethodMayBeStatic,PyUnusedLocal
    def can_restore(self, object_revision):
        return True

    def get_id(self):
        return self.user_id

    # @hybrid_property
    # def role_name(self):
    #     try:
    #         return self.role.role_name
    #     except:
    #         return ''

    def cdr_token(self):
        return None, None, None

        from api_dnl.model import DailyCdrField
        # from api_dnl.utils.statisticapi2 import StatisticAPI
        api = StatisticAPI()
        ingress_id = None
        egress_id = None

        if self.user_type in ['client', 'agent']:
            if self.allowed_ingress_id:
                ingress_id = ','.join([str(i) for i in self.allowed_ingress_id])
            else:
                if self.allowed_egress_id:
                    egress_id = ','.join([str(i) for i in self.allowed_egress_id])
                else:
                    ingress_id = '-1'
        else:
            if self.allowed_egress_id:
                egress_id = ','.join([str(i) for i in self.allowed_egress_id])

        all_fields = DailyCdrField.query().filter(DailyCdrField.id.notin_([96, 108])).all()
        allowed_fields = None
        if self.user_type in ['client', 'agent']:
            allowed_fields = ','.join([str(f.id) for f in all_fields if f.client_viewable])
        elif self.user_type in ['vendor']:
            allowed_fields = ','.join([str(f.id) for f in all_fields if f.vendor_viewable])
        cdr, exp = None, 0
        try:
            if egress_id:
                if ingress_id:
                    cdr, exp = api.authenticate(user=self.name, allowed_ingress_id=ingress_id,
                                                allowed_egress_id=egress_id, allowed_fields=allowed_fields)
                else:
                    cdr, exp = api.authenticate(user=self.name,
                                                allowed_egress_id=egress_id, allowed_fields=allowed_fields)
            else:
                if ingress_id:
                    cdr, exp = api.authenticate(user=self.name, allowed_ingress_id=ingress_id,
                                                allowed_fields=allowed_fields)
                else:
                    cdr, exp = api.authenticate(user=self.name, allowed_fields=allowed_fields)
        except Exception as e:
            log.error('user get cdr token error {}'.format(str(e)))
            cdr_term, exp = None, None
        if cdr:
            self.cdr_api_token = cdr
            self.cdr_expire = datetime.utcfromtimestamp(exp)
            User.session().add(self)
        return cdr, exp, cdr

    @hybrid_property
    def clients_id(self):
        from api_dnl.model import Client
        if self.name in ['admin', '#icx_api']:
            return [l.client_id for l in Client.query().all()]
        else:
            if self.user_type == 'admin':
                return [l.client_id for l in self.client_limits]
            elif self.user_type == 'agent':
                if self.agent and self.agent.carriers:
                    return [l.client_id for l in self.agent.carriers]
            else:  # self.user_type=='client':
                if self.client:
                    return list(set([self.client.client_id] + [l.client_id for l in self.client_limits]))
        return []

    @property
    def allowed_ingress_id(self):
        from api_dnl.model import Resource as res
        if self.user_type == 'admin':
            return None
        if self.clients_id:
            return [r.resource_id for r in res.filter(and_(res.client_id.in_(self.clients_id), res.ingress == True))]
        return [0]

    @property
    def allowed_egress_id(self):
        from api_dnl.model import Resource as res
        if self.user_type in ('admin'):
            return None
        if self.clients_id:
            return [r.resource_id for r in res.filter(and_(res.client_id.in_(self.clients_id), res.egress == True))]
        return None

    @property
    def _client_id(self):
        from api_dnl.model import Client
        q = Client.filter(Client.user_id == self.user_id).first()
        if q:
            return q.client_id
        return None

    @property
    def default_billing_decimal(self):
        from api_dnl.model import SystemParameter
        config = SystemParameter.get(1)
        return config.default_billing_decimal

    @property
    def is_preload(self):
        from api_dnl.model import SystemParameter
        config = SystemParameter.get(1)
        return config.is_preload

    @property
    def report_count(self):
        from api_dnl.model import SystemParameter
        config = SystemParameter.get(1)
        return config.report_count

    def check_api_key(self, token):
        q = UserApiKey.filter(and_(UserApiKey.user_id==self.user_id,UserApiKey.token==token)).first()
        return q is not None

    def create_api_key(self):
        exp = (datetime.now(UTC) + timedelta(days=365 * 3))
        data = dict(user_id=self.user_id, api_key=True, expired=int(exp.timestamp()))
        token = jwt.encode(data, settings.JWT_SIGNATURE).decode('UTF-8')
        key = UserApiKey(user_id=self.user_id, token=token, expired_on=exp)
        self.api_key = key
        return token

    @property
    def fmt_name(self):
        return self.username

    @property
    def fmt_username(self):
        return self.name

    @property
    def fmt_company(self):
        return self.name


class WebSession(DnlApiBaseModel):
    __tablename__ = 'web_session'

    id = Column(fields.Integer, primary_key=True)
    create_time = Column(fields.DateTime(timezone=True), nullable=False,default=orm.func.now())
    user_id = Column(fields.Integer)
    host = Column(fields.String(1000), index=True)
    agent = Column(fields.String(1000))
    msg = Column(fields.String)
    exp = Column(fields.DateTime(timezone=True), default=orm.func.now())
    token = Column(fields.String(1000), index=True)
    user = relationship(User, primaryjoin=foreign(user_id) == User.user_id, uselist=False)

    user_name = column_property(select([User.name]).where(user_id == User.user_id).correlate_except(User))



    @classmethod
    def get_session_from_req(cls, req, create_sess=True):
        from api_dnl.model import SystemParameter
        now = datetime.now(UTC)
        token = req.headers.get('X-AUTH-TOKEN')
        if not token:
            from urllib.parse import parse_qsl
            qs = dict(parse_qsl(req.query_string))
            token = qs.pop('auth_token',None)
            if token:
                req.query_string = urlencode(qs)
        if not token:
            return None
        if hasattr(req, 'access_route') and len(req.access_route):
            host = req.access_route[0]
        else:
            return None
        try:
            token_data = jwt.decode(token, settings.JWT_SIGNATURE)
        except jwt.ExpiredSignatureError:
            token_data = jwt.decode(token, settings.JWT_SIGNATURE,verify=False)
            if not 'check-token' in req.url:
                return False
        except jwt.InvalidTokenError:
            return None
        user_id = token_data['user_id']
        try:
            user = User.get(user_id)  # error on disconnect database???
        except InternalError as e:
            log.critical('INTERNAL DATABASE error {}'.format(e))
            User.session().rollback()
            raise e
        except Exception as e:
            log.critical('DATABASE error {}'.format(e))
            User.session().rollback()
            raise e
        if not user:
            return None
        if not user.is_active:
            return None
        if 'api_key' in token_data and not user.check_api_key(token):
            log.warning('not allowed api key')
            return None
        try:
            config = SystemParameter.session().query(SystemParameter.inactivity_timeout).first()
        except Exception as e:
            SystemParameter.session().rollback()
            config = SystemParameter.session().query(SystemParameter.inactivity_timeout).first()
        sess_exp = now + timedelta(minutes=config.inactivity_timeout)
        sess = cls.filter(orm.and_(cls.token == token, cls.host == host)).first()
        if not sess:
            if create_sess:
                if not 'expired' in token_data:
                    User.filter(and_(User.user_id != user.user_id, User.is_online == 1,
                                     User.user_id.notin_(select([cls.user_id])))).update({'is_online': 0},
                                                                                         synchronize_session='fetch')
                    User.session().commit()
                    # user.is_online = 0
                    # user.save()
                    return False
                expired = token_data['expired']
                if not 'check-token' in req.url:
                    if expired < now.timestamp():
                        User.filter(and_(User.user_id != user.user_id, User.is_online == 1,
                                         User.user_id.notin_(select([cls.user_id])))).update({'is_online': 0},
                                                                                             synchronize_session='fetch')
                        User.session().commit()
                        # user.is_online = 0
                        # user.save()
                        return False
                sess = cls(token=token, host=host, user_id=user_id, exp=sess_exp, agent=req.user_agent)
            else:
                return None
        if sess.exp < now:
            # user.is_online = 0
            # user.save()
            cls.filter(cls.exp < now).delete(synchronize_session='fetch')
            cls.session().commit()
            return None
        if sess.exp != sess_exp:
            sess.exp = sess_exp
            sess.save()
        #User.session().add(sess)
        # User.filter(
        #     and_(User.user_id != user.user_id, User.is_online == 1, User.user_id.notin_(select([cls.user_id])))).update(
        #     {'is_online': 1}, synchronize_session='fetch')


        q = User.filter(User.user_id == user.user_id)
        q.update(dict(last_seen = now,login_ip = host,is_online = 1), synchronize_session='fetch')
        sess.user = User.get(user_id)

        #User.session().add(user)
        return sess


class UserApiKey(DnlApiBaseModel):
    __tablename__ = 'user_api_key'
    user_id = Column(fields.Integer, ForeignKey('users.user_id', ondelete='CASCADE'), primary_key=True)
    token = Column(fields.String(1024),index=True)
    expired_on = Column(fields.DateTime(True))
