from datetime import datetime, timedelta
from dateutil.parser import parse as parse_datetime
from pytz import UTC
from marshmallow import (
    Schema, pre_load, pre_dump, post_dump, post_load, validates_schema,
    validate, validates, fields, ValidationError
)
from marshmallow.fields import (
    Field, Raw, Nested, Dict, List, String, UUID, Number, Integer, Decimal, Boolean,
    FormattedString, Float, DateTime, LocalDateTime, Time, Date, TimeDelta, Url, URL,
    Email, Method, Function, Str, Bool, Int, Constant)
from api_dnl import model
from api_dnl.fields import Choice
from api_dnl.scheme import BaseModelScheme
from api_dnl.scheme import DATE_REGEXP, IP_REGEXP, NAME_REGEXP, PREFIX_REGEXP, TIME_REGEXP, ValidationError, \
    TIMEZONE_REGEXP, SuccessScheme
from api_dnl.scheme import _valid, valid_date, Emails


def valid_ingress_count(value):
    if value:
        if not model.Client.get(int(value)):
            raise ValidationError('Client {} not exists!'.format(value))
        if model.Client.get(int(value)).ingress_count == 0:
            raise ValidationError('Client {} has no ingress trunks!'.format(value))
    return True


# --- Invoice---

class InvoiceScheme(BaseModelScheme):
    # gen   post /carrier/{carrier_id}/invoice
    # Create Invoice
    # start_date = Str()
    start_date = Str(attribute='invoice_start', required=True, allow_none=False, validate=validate.Regexp(DATE_REGEXP))
    end_date = Str(attribute='invoice_end', required=True, allow_none=False, validate=validate.Regexp(DATE_REGEXP))
    # billing_start_time = DateTime(attribute='generate_start_time')
    # billing_end_time = DateTime(attribute='generate_end_time')
    due_date = Str(validate=validate.Regexp(DATE_REGEXP))
    invoice_time = DateTime()
    # gmt = Number()
    gmt = Str(attribute='invoice_zone', default='+00:00',
              validate=validate.OneOf(['{:+03d}:00'.format(i) for i in range(-12, 13)]))
    # show_cdr = Bool()
    show_cdr = Bool(attribute='is_short_duration_call_surcharge_detail')
    # show_trunk_breakdown = Bool()
    show_trunk_breakdown = Bool(attribute='is_show_detail_trunk')

    @pre_load
    def before_create(self, data):  # not needed

        if 'start_date' in data:
            try:
                valid_date(data['start_date'])
            except:
                raise ValidationError({'start_date': ['invalid date {}'.format(data['start_date'])]})
        else:
            raise ValidationError({'start_date': ['missing']})

        if 'end_date' in data:
            try:
                valid_date(data['end_date'])
            except:
                raise ValidationError({'end_date': ['invalid date {}'.format(data['end_date'])]})
        else:
            raise ValidationError({'end_date': ['missing']})

        start = parse_datetime(data['start_date'], ignoretz=True).date()
        if start > datetime.now(UTC).date():
            raise ValidationError({'start_date': ['cannot create invoice in future!']})
        end = parse_datetime(data['end_date'], ignoretz=True).date()
        if end > datetime.now(UTC).date():
            raise ValidationError({'end_date': ['invalid date in future {}'.format(data['end_date'])]})
        if end < start:
            raise ValidationError({'end_date': ['must be greater than start_date!']})
        if 'due_date' in data:
            try:
                valid_date(data['due_date'])
            except:
                raise ValidationError({'due_date': ['invalid date {}'.format(data['due_date'])]})
            due = parse_datetime(data['due_date'], ignoretz=True).date()
            if due < end:
                raise ValidationError({'due_date': ['must be greater than end_date!']})
        return data

    class Meta:
        model = model.Invoice
        fields = ('start_date', 'end_date', 'gmt', 'show_cdr', 'show_trunk_breakdown')


class InvoiceModifyScheme(BaseModelScheme):
    state = Str(validate=validate.OneOf(('verify', 'void', 'send')))  # Choice()

    class Meta:
        model = model.Invoice
        fields = ('state',)


class InvoiceUploadScheme(InvoiceScheme):
    class Meta:
        model = model.Invoice
        fields = ('start_date', 'end_date', 'gmt', 'show_cdr', 'show_trunk_breakdown', 'pdf_file')


class InvoiceSendScheme(InvoiceScheme):
    to_emails = Emails(allow_none=True, validate=validate.Length(min=9, max=500))

    class Meta:
        model = model.Invoice
        fields = ('to_emails',)


class InvoiceReCreateScheme(BaseModelScheme):
    class Meta:
        model = model.Invoice
        fields = ('invoice_id',)
        exclude = ('invoice_id', 'invoice_number', 'invoice_end', 'invoice_start', 'current_balance')


class InvoiceVoidScheme(BaseModelScheme):
    class Meta:
        model = model.Invoice


class InvoiceGenerationScheme(InvoiceScheme):
    # gen module  invoice-generation
    # --- Invoice Generation ---

    show_rate_table_breakdown = Bool(attribute='is_show_code_name')  # TODO ???
    show_daily_breakdown = Bool(attribute='is_show_daily_usage')
    send_invoice_as_link = Bool(attribute='is_send_as_link')
    show_jurisdiction_detail = Bool(attribute='invoice_jurisdictional_detail')
    include_account_detail = Bool(attribute='include_detail')
    show_payment_summary = Bool(attribute='is_invoice_account_summary')
    show_detail_by_trunk = Bool(attribute='is_show_detail_trunk')
    include_cdr_in_email = Bool(attribute='is_show_by_date')
    show_detail_by_code_name = Bool(attribute='is_show_code_name')
    rate_decimal = Int(attribute='decimal_place')
    show_code_summary = Bool(attribute='is_show_code_100')
    auto_send_invoice = Bool()

    show_detail_by_country = Bool(attribute='is_show_country')
    state = Choice()

    class Meta:
        model = model.Invoice
        fields = ('start_date', 'end_date', 'gmt', 'show_cdr', 'show_trunk_breakdown',
                  'show_rate_table_breakdown', 'show_daily_breakdown',
                  'send_invoice_as_link', 'show_jurisdiction_detail', 'include_account_detail',
                  'show_payment_summary', 'show_detail_by_trunk', 'include_cdr_in_email',
                  'show_detail_by_code_name',
                  'rate_decimal', 'show_code_summary', 'auto_send_invoice',
                  'invoice_time',
                  'due_date',
                  'show_detail_by_country',
                  'usage_detail_fields')


class InvoiceGenerationManyScheme(InvoiceGenerationScheme):
    # clients=List(Int(validate=[lambda value:_valid('Client','client_id',value), lambda value:model.Client.get(int(value)).ingress_count>0]))
    clients = List(Int(validate=[lambda value: _valid('Client', 'client_id', value)]))

    class Meta:
        model = model.Invoice
        fields = ('clients', 'start_date', 'end_date', 'gmt', 'show_cdr', 'show_trunk_breakdown',
                  'due_date',
                  'show_rate_table_breakdown', 'show_daily_breakdown',
                  'send_invoice_as_link', 'show_jurisdiction_detail', 'include_account_detail',
                  'show_payment_summary', 'show_detail_by_trunk', 'include_cdr_in_email',
                  'show_detail_by_code_name',
                  'rate_decimal', 'show_code_summary', 'auto_send_invoice',
                  'invoice_time',
                  'show_detail_by_country',
                  'usage_detail_fields')


class InvoiceClientGetScheme(BaseModelScheme):
    # carrier_name = Str()
    # client_id = Int()
    class Meta:
        model = model.Invoice
        fields = ('invoice_id',
                  # 'client_id','carrier_name','amount',
                  # 'note','paid_on','update_by','payment_time','payment_type_name'
                  )


class InvoiceGetScheme(BaseModelScheme):
    invoice_id = Int()
    invoice_number = Str()
    bill_period_start = DateTime()
    bill_period_end = DateTime()
    invoice_time = DateTime()
    state = Choice()
    client_id = Int()
    client_name = Str()
    company_name = Str()
    amount = Decimal()
    paid = Bool()
    due_date = Str()
    url = Str()
    status = Choice()
    create_type = Choice()
    last_invoice_for = Str()
    payment_term_name = Str()
    invoice_period = Str()
    invoice_credit_amount = Float()
    unpaid_amount = Float()
    invoice_num = Str()
    company_type = Choice()
    current_charge = Float()
    void=Bool()

    class Meta:
        model = model.Invoice
        fields = (
            'invoice_id', 'bill_period_start', 'bill_period_end', 'invoice_number', 'invoice_time', 'state',
            'client_id', 'client_name', 'company_name', 'amount', 'paid', 'due_date', 'url', 'status', 'create_type',
            'last_invoice_for', 'payment_term_name', 'invoice_period', 'usage_detail_fields', 'invoice_credit_amount',
            'unpaid_amount', 'invoice_num', 'company_type', 'pay_amount', 'current_charge'
        )
        search_fields = ('invoice_id', 'invoice_number', 'state', 'client_id', 'client_name', 'company_name',
                         'paid', 'status', 'create_type', 'payment_term_name', 'payment_term_id', 'invoice_period', 'company_type',
                         'void')
        query_fields = (
            'start_date_gt', 'start_date_lt', 'end_date_gt', 'end_date_lt', 'invoice_time_gt', 'invoice_time_lt',
            'due_date_gt', 'due_date_lt', 'invoice_id_in', 'last_invoice_for_gt', 'last_invoice_for_lt', 'amount_gt',
            'invoice_start_gt', 'invoice_start_lt', 'invoice_end_gt', 'invoice_end_lt',)


# 'invoice_period_from_gt','invoice_period_from_lt','invoice_period_to_gt','invoice_period_to_lt')


class InvoiceFileGetScheme(InvoiceGetScheme):
    class Meta:
        model = model.ImportExportLogs
        fields = ('invoice_id',)


class InvoiceAllStatusScheme(InvoiceGetScheme):
    class Meta:
        model = model.Invoice
        fields = ('state',)
        search_fields = (
            'invoice_id', 'invoice_number', 'state', 'client_id', 'client_name', 'paid', 'status', 'create_type',
            'payment_term_name', 'invoice_period')
        query_fields = (
        'start_date_gt', 'start_date_lt', 'end_date_gt', 'end_date_lt', 'invoice_time_gt', 'invoice_time_lt',
        'due_date_gt', 'due_date_lt', 'invoice_id_in', 'last_invoice_for_gt', 'last_invoice_for_lt', 'amount_gt')



# --- Invoice---
# region +++InvoiceSummary+++
class InvoiceSummaryScheme(BaseModelScheme):
    client_name = Str(validate=[validate.Length(max=500)])
    invoice_number = Int()
    invoice_num = Str(validate=[validate.Length(max=256)])
    json_content = Dict()
    client_id = Int()
    invoice_generation_time = DateTime()
    invoice_start_time = DateTime()
    invoice_end_time = DateTime()
    task_id = Int()
    payment_id = Int()
    current_balance = Float()
    previous_balance = Float()
    payments_and_credits = Float()
    balance_forward = Float()
    long_distance_charges = Float()
    taxes = Float()
    scc_calls = Float()
    port_charges = Float()
    scc_amount = Float()
    current_charges = Float()
    total_calls = Int()
    total_minutes = Float()
    total_amount = Float()
    unlimited_credit = Bool()
    invoice_type = Choice()
    did_mrc = Float()
    did_nrc = Float()
    create_time = DateTime()
    task = Nested('InvoiceTaskScheme', many=False)
    client = Nested('ClientScheme', many=False)

    class Meta:
        model = model.InvoiceSummary
        fields = (
            'invoice_num', 'json_content', 'client_id', 'invoice_generation_time', 'invoice_start_time',
            'invoice_end_time',
            'task_id', 'payment_id', 'current_balance', 'previous_balance', 'payments_and_credits', 'balance_forward',
            'long_distance_charges', 'taxes', 'scc_calls', 'port_charges', 'scc_amount', 'current_charges',
            'total_calls',
            'total_minutes', 'total_amount', 'unlimited_credit', 'invoice_type', 'did_mrc', 'did_nrc', 'create_time',)


class InvoiceSummarySchemeGet(InvoiceSummaryScheme):
    class Meta:
        model = model.InvoiceSummary
        fields = (
            'invoice_num', 'json_content', 'client_id', 'invoice_generation_time', 'invoice_start_time',
            'invoice_end_time',
            'task_id', 'payment_id', 'current_balance', 'previous_balance', 'payments_and_credits', 'balance_forward',
            'long_distance_charges', 'taxes', 'scc_calls', 'port_charges', 'scc_amount', 'current_charges',
            'total_calls',
            'total_minutes', 'total_amount', 'unlimited_credit', 'invoice_type', 'did_mrc', 'did_nrc', 'create_time',)
        search_fields = (
            'client_name', 'invoice_number', 'invoice_num', 'json_content', 'client_id', 'task_id', 'payment_id',
            'total_calls', 'unlimited_credit', 'invoice_type', 'task', 'did_invoice_charge_detail',
            'did_invoice_charge_breakdown', 'did_invoice_number_fee_detail', 'did_mrc_info', 'did_setup_fee',
            'did_port_usage_info', 'invoice_code_name_detail', 'invoice_country_detail', 'invoice_trunk_detail',
            'invoice_trunk_prefix_detail', 'invoice_daily_detail', 'client',)
        query_fields = (
            'invoice_generation_time_gt', 'invoice_generation_time_lt', 'invoice_start_time_gt',
            'invoice_start_time_lt',
            'invoice_end_time_gt', 'invoice_end_time_lt', 'current_balance_gt', 'current_balance_lt',
            'previous_balance_gt',
            'previous_balance_lt', 'payments_and_credits_gt', 'payments_and_credits_lt', 'balance_forward_gt',
            'balance_forward_lt', 'long_distance_charges_gt', 'long_distance_charges_lt', 'taxes_gt', 'taxes_lt',
            'scc_calls_gt', 'scc_calls_lt', 'port_charges_gt', 'port_charges_lt', 'scc_amount_gt', 'scc_amount_lt',
            'current_charges_gt', 'current_charges_lt', 'total_minutes_gt', 'total_minutes_lt', 'total_amount_gt',
            'total_amount_lt', 'did_mrc_gt', 'did_mrc_lt', 'did_nrc_gt', 'did_nrc_lt', 'create_time_gt',
            'create_time_lt',)


class InvoiceSummarySchemeGet(InvoiceSummaryScheme):
    class InvoiceSummarySchemeModify(InvoiceSummaryScheme):
        pass


class InvoiceSummarySchemeModify(InvoiceSummaryScheme):
    pass


class InvoiceSummaryDataSchemeGet(InvoiceSummaryScheme):
    data = Dict(attribute='json_content')

    class Meta:
        model = model.InvoiceSummary
        fields = ('data',)
        search_fields = (
            'client_name', 'invoice_number', 'invoice_num', 'json_content', 'client_id', 'task_id', 'payment_id',
            'total_calls', 'unlimited_credit', 'invoice_type', 'task', 'did_invoice_charge_detail',
            'did_invoice_charge_breakdown', 'did_invoice_number_fee_detail', 'did_mrc_info', 'did_setup_fee',
            'did_port_usage_info', 'invoice_code_name_detail', 'invoice_country_detail', 'invoice_trunk_detail',
            'invoice_trunk_prefix_detail', 'invoice_daily_detail', 'client',)
        query_fields = (
            'invoice_generation_time_gt', 'invoice_generation_time_lt', 'invoice_start_time_gt',
            'invoice_start_time_lt',
            'invoice_end_time_gt', 'invoice_end_time_lt', 'current_balance_gt', 'current_balance_lt',
            'previous_balance_gt',
            'previous_balance_lt', 'payments_and_credits_gt', 'payments_and_credits_lt', 'balance_forward_gt',
            'balance_forward_lt', 'long_distance_charges_gt', 'long_distance_charges_lt', 'taxes_gt', 'taxes_lt',
            'scc_calls_gt', 'scc_calls_lt', 'port_charges_gt', 'port_charges_lt', 'scc_amount_gt', 'scc_amount_lt',
            'current_charges_gt', 'current_charges_lt', 'total_minutes_gt', 'total_minutes_lt', 'total_amount_gt',
            'total_amount_lt', 'did_mrc_gt', 'did_mrc_lt', 'did_nrc_gt', 'did_nrc_lt', 'create_time_gt',
            'create_time_lt',)


# endregion ---InvoiceSummary---
# region +++InvoiceTask+++
class InvoiceTaskScheme(BaseModelScheme):
    client_name = Str(validate=[validate.Length(max=500)])
    client_type = Choice()
    id = Int()
    client_id = Int(validate=lambda value: _valid('Client', 'client_id', value))
    is_auto = Bool()
    created_by_user_id = Int()
    created_on = DateTime()
    bill_period_start = Str(validate=valid_date)
    bill_period_end = Str(validate=valid_date)
    show_acc_summary = Bool()
    show_payment_applied = Bool()
    show_rec_charge = Bool()
    show_non_rec_charge = Bool()
    show_usage_charge = Bool()
    show_trunk_detail = Bool()
    show_daily_summary = Bool()
    show_jd_daily_summary = Bool()
    show_jd_trunk_detail = Bool()
    show_prefix_summary = Bool()
    show_code_name_summary = Bool()
    show_country_summary = Bool()
    show_inbound_summary = Bool()
    show_top_100tn_summary = Bool()
    show_did_charge_breakdown = Bool()
    show_did_charge_summary = Bool()
    enable_email = Bool()
    enable_email_with_cdr = Bool()
    include_tax = Bool()
    include_scc_charges = Bool()
    payment_dur_date = Str(validate=valid_date)
    invoice_zone = Str(validate=validate.Regexp(TIMEZONE_REGEXP))
    status = Choice()
    progress = Str(validate=[validate.Length(max=1024)])
    proc_start_time = DateTime()
    proc_end_time = DateTime()
    summary = Nested('InvoiceSummaryScheme', many=False)

    class Meta:
        model = model.InvoiceTask
        fields = ('client_id', 'bill_period_start', 'bill_period_end', 'show_acc_summary', 'show_payment_applied',
                  'show_rec_charge', 'show_non_rec_charge', 'show_usage_charge', 'show_trunk_detail',
                  'show_daily_summary', 'show_jd_daily_summary', 'show_jd_trunk_detail', 'show_prefix_summary',
                  'show_code_name_summary', 'show_country_summary', 'show_inbound_summary', 'show_top_100tn_summary',
                  'enable_email', 'enable_email_with_cdr', 'include_tax', 'include_scc_charges', 'payment_dur_date',
                  'invoice_zone', 'show_did_charge_breakdown', 'show_did_charge_summary')

class InvoiceTaskManyScheme(InvoiceTaskScheme):
    client_id = List(Int(validate=[lambda value: _valid('Client', 'client_id', value)]))#, valid_ingress_count]))


class InvoiceTaskSchemeGet(InvoiceTaskScheme):
    invoice_id = Int()
    invoice_amount = Int()
    invoice_number = Int()
    invoice_num = Str()

    class Meta:
        model = model.InvoiceTask
        fields = ('client_id', 'is_auto', 'created_by_user_id', 'created_on', 'bill_period_start', 'bill_period_end',
                  'show_acc_summary', 'show_payment_applied', 'show_rec_charge', 'show_non_rec_charge',
                  'show_usage_charge', 'show_trunk_detail', 'show_daily_summary', 'show_jd_daily_summary',
                  'show_jd_trunk_detail', 'show_prefix_summary', 'show_code_name_summary', 'show_country_summary',
                  'show_inbound_summary', 'show_top_100tn_summary', 'enable_email', 'enable_email_with_cdr',
                  'include_tax', 'include_scc_charges', 'payment_dur_date', 'invoice_zone', 'status', 'progress',
                  'proc_start_time', 'proc_end_time', 'invoice_number', 'invoice_num', 'show_did_charge_breakdown',
                  'invoice_id', 'invoice_amount', 'show_did_charge_summary')
        search_fields = (
            'client_name', 'client_type', 'id', 'client_id', 'is_auto', 'created_by_user_id', 'bill_period_start',
            'bill_period_end', 'show_acc_summary', 'show_payment_applied', 'show_rec_charge', 'show_non_rec_charge',
            'show_usage_charge', 'show_trunk_detail', 'show_daily_summary', 'show_jd_daily_summary',
            'show_jd_trunk_detail',
            'show_prefix_summary', 'show_code_name_summary', 'show_country_summary', 'show_inbound_summary',
            'show_top_100tn_summary', 'enable_email', 'enable_email_with_cdr', 'include_tax', 'include_scc_charges',
            'payment_dur_date', 'invoice_zone', 'status', 'progress', 'summary', 'invoice_number', 'invoice_num',
            'show_did_charge_breakdown', 'show_did_charge_summary')
        query_fields = (
            'created_on_gt', 'created_on_lt', 'proc_start_time_gt', 'proc_start_time_lt', 'proc_end_time_gt',
            'proc_end_time_lt',)


class InvoiceTaskSchemeModify(InvoiceTaskScheme):
    pass

# endregion ---InvoiceTask---
