import json
from xmljson import BadgerFish as bf, Parker as X2J
from xml.etree.ElementTree import Element, fromstring, tostring
import xml.etree.ElementTree as ET
from xml.dom import minidom
from api_dnl.__version__ import __version__ as VERSION
from api_dnl.schemes.common import *
from datetime import datetime, timedelta
from pytz import UTC
from dateutil.parser import parse as parse_datetime
from csv import DictWriter
from api_dnl import model
import io
import falcon
from falcon_rest.db.errors import IntegrityError, FkConstraintViolation, DataError, NoResultFound
from falcon_rest import schemes, swagger, responses
from falcon_rest.responses import errors, SuccessResponse, SuccessResponseJustOk
from falcon_rest.helpers import check_permission
from falcon_rest.resources.base_resource import BaseResource
from falcon_rest.logger import log
from falcon_rest.resources.resources import Create, Resource, List, CustomPatchAction, CustomAction as _CustomAction, \
    ResourcesBaseClass, PreConditionFailedException, ATTRIBUTE_ERROR_RE
from urllib.parse import parse_qsl, urlencode
from falcon_rest import conf
from sqlalchemy.exc import OperationalError
from sqlalchemy import (Column, desc, and_, or_, text as text_, PrimaryKeyConstraint, inspect)
from sqlalchemy import (
    Integer, SmallInteger, Float, Text, String, DateTime, Date, Time, Boolean, ForeignKey, BigInteger, Numeric
)
from marshmallow.fields import validate
from marshmallow import (
    Schema, pre_load, pre_dump, post_dump, post_load, validates_schema,
    validate, validates, fields, ValidationError
)
import traceback
import uuid

from api_dnl.fields import Choice

from functools import wraps
import cProfile


def profile(f):
    dump_file = 'profile.pstat'

    @wraps(f)
    def wrapped(*args, **kw):
        pr = cProfile.Profile()
        pr.enable()
        ret = f(*args, **kw)
        pr.disable()
        pr.dump_stats(dump_file)
        return ret

    return wrapped


def generate_uuid_str():
    return lambda: str(uuid.uuid4())


def OperationErrorResponse(e):
    # msg = str(e).split('\n')
    # if len(msg) > 3:
    #     msg = '\n'.join(msg[0:2])
    msg = '\n'.join(str(e).split('\n')[0:2])
    return responses.OperationErrorResponse(data=responses.errors.Error(406, msg, e.__class__.__name__))


def OperationalErrorResponse(e):
    from api_dnl import settings
    log.debug("RESOURCE ERROR")
    msg = 'Connect to database {} \n'.format(settings.DB_CONN_STRING.split('@')[1].split('/')[0])
    msg = msg + '\n'.join(str(e).split('\n')[0:2])
    return responses.UnauthenticatedErrorResponse(data=responses.errors.Error(401, msg, e.__class__.__name__))


class AlreadyExists(Exception):
    def __init(self, msg):
        self.msg = msg

    def __repr__(self):
        return self.msg


class AlreadyExistsResponse(responses.OperationErrorResponse):
    tpl = '{} error: {}'

    def __init__(self, entity, msg):
        self.data = errors.Error(code=1000, reason='already_exists',
                                 message='{} error {}'.format(entity, msg))


def client_context(self, req, resp, **kwargs):
    user = self.get_user(req)
    # from api_dnl.utils.statisticapi2 import StatisticAPI
    # api = StatisticAPI()
    # if user:
    #     if user.cdr_api_token and user.cdr_expire:
    #         api.set_token(user.cdr_api_token, user.cdr_expire.timestamp())
    #     else:
    #         user.cdr_api_token, cdr_expire, cdr_term = user.cdr_token()
    #         if cdr_expire:
    #             user.cdr_expire = datetime.fromtimestamp(cdr_expire, UTC)
    if user and user.user_type == 'client':
        if user.client:
            kwargs['client_id'] = str(user.client.client_id)
            req.context['client_id'] = kwargs['client_id']
        else:
            self.set_response(resp, responses.UnAuthorizedErrorResponse(data={'message': 'permission revoked'}))
            return {}
    if user and user.user_type == 'agent' and user.agent:
        if user.agent:
            kwargs['agent_id'] = str(user.agent.agent_id)
            req.context['agent_id'] = kwargs['agent_id']
        else:
            self.set_response(resp, responses.UnAuthorizedErrorResponse(data={'message': 'permission revoked'}))
            return {}
    return kwargs


def check_context(self, req, resp, **kwargs):
    user = self.get_user(req)
    if 'X-AUTH-TOKEN' in req.headers:
        token = req.headers.get('X-AUTH-TOKEN')
        auth_token = model.AuthToken.filter(model.AuthToken.token == token).first()
        sp = model.SystemParameter.get(1)
        if sp:
            ttl = sp.inactivity_timeout or 30
        inactive = auth_token.last_used + timedelta(minutes=ttl) < datetime.now(UTC) if auth_token and auth_token.last_used else False
        if auth_token and auth_token.is_disabled or inactive:
            self.set_response(resp, responses.UnAuthorizedErrorResponse(data={'message': 'token is disabled'}))
            return False
        if auth_token:
            auth_token.last_used = datetime.now(UTC)
    if not user:
        if hasattr(self, 'no_auth_needed') and self.no_auth_needed:
            return True
        self.set_response(resp, responses.UnAuthorizedErrorResponse(data={'message': 'permitted for users only'}))
        return False
    if user.user_type == 'agent' and (not user.agent or not user.agent.status):
        self.set_response(resp, responses.UnAuthorizedErrorResponse(data={'message': 'permission revoked'}))
        return False

    if user.user_type == 'client' and (not user.client or not user.client.status):
        self.set_response(resp, responses.UnAuthorizedErrorResponse(data={'message': 'permission revoked'}))
        return False

    if hasattr(self, 'is_client') and self.is_client:
        if not user or not user.user_type == 'client':
            self.set_response(resp, responses.UnAuthorizedErrorResponse(data={'message': 'permitted for clients only'}))
            return False
        if not user.client:
            self.set_response(resp, responses.UnAuthorizedErrorResponse(
                data={'message': 'bad user configuration -  no client'}))
            return False
    if hasattr(self, 'is_agent') and self.is_agent:
        if not user or not user.user_type == 'agent':
            self.set_response(resp, responses.UnAuthorizedErrorResponse(data={'message': 'permitted for agents only'}))
            return False
    return True


class DnlCreate(Create):
    def on_post(self, req, resp, **kwargs):
        self.init_req(req)
        if not check_context(self, req, resp, **kwargs):
            return False
        kwargs = client_context(self, req, resp, **kwargs)
        log.debug("PARENT ON_POST\nKWARGS - {}".format(kwargs))
        return self._on_post(req, resp, **kwargs)

    def _on_post(self, req, resp, **kwargs):
        self.apply_restrict_rules(req, kwargs)
        scheme = self.scheme_class()
        if not check_permission(self, req, 'create'):
            self.set_response(
                resp, responses.ForbiddenErrorResponse(data=errors.AuthErrors.Forbidden)
            )
            return

        if self.check_request_data(req, resp, scheme):
            try:
                log.debug("\nREQ - {}\nRESP - {}\nKWARGS - {}".format(req, resp, kwargs))
                log.debug(f"SELF.SCHEME_CLASS = {self.scheme_class}")
                log.debug(f"SCHEME = {scheme}")
                data = self.create_object(req, scheme, **kwargs)
                log.debug("DATA - {}".format(data))
                self.set_response(
                    resp, self.scheme_class.get_object_created_response(data=data)
                )
                log.debug("AFTER_CREATE() starting")
                self.after_create(data, req, resp, **kwargs)
            except IntegrityError as e:
                try:
                    msg = str(e).split('\n')[1].split(':')[1]
                except:
                    msg = str(e)
                try:
                    model.get_db().session.rollback()
                except Exception as e2:
                    log.error('on create error {} rollback also failed {}'.format(e,e2))

                self.set_response(resp, AlreadyExistsResponse(self.entity, msg))
            except AlreadyExists as e:
                r = AlreadyExistsResponse(self.entity, str(e))
                r.status_code = 1000
                self.set_response(resp, r)
            except FkConstraintViolation as e:
                self.set_response(resp, responses.FkConstraintViolationResponse(
                    data=errors.CommonErrors.get_fk_violation_error(
                        e.table_name, e.fk_column_name, e.column_name, e.value)
                )
                                  )
            except schemes.validate.ValidationError as e:
                self.set_response(resp, responses.ValidationErrorResponse(data=e.messages))

            except NoResultFound as e:
                self.set_response(resp, responses.ObjectNotFoundErrorResponse(data=str(e)))
                return None

            except OperationalError as e:
                self.set_response(resp, OperationalErrorResponse(e))

            except DataError as e:
                try:
                    msg = str(e).split('\n')[1].split(':')[1]
                except:
                    msg = str(e)
                log.error(e)
                # self.set_response(resp, responses.OperationErrorResponse(data=errors.CommonErrors.DataError))
                self.set_response(resp, OperationErrorResponse(e))

            except Exception as e:
                log.error(e)
                # self.set_response(resp, responses.OperationErrorResponse(data=errors.CommonErrors.DataError))
                self.set_response(resp, OperationErrorResponse(e))
        return None

    def create_object(self, req, scheme, **kwargs):
        log.debug("CHECKING FOR AUTH")
        if hasattr(self, 'no_auth_needed') and self.no_auth_needed:
            log.debug("NO AUTH NEEDED, STARTING BEFORE CREATE")
            return self.before_create(
                self.get_loaded_data(scheme), **kwargs
            ).save()
        else:
            log.debug("AUTH NEEDED")
            save_history=not req.context['service']
            log.debug(f"SAVE_HISTORY - {save_history}")
            log.debug("STARTING BEFORE CREATE")
            return self.before_create(
                self.get_loaded_data(scheme), **kwargs
            ).save(save_history=save_history, user=self.get_user(req), module=getattr(self.__class__, 'api_module', 'N/A'))

    def after_create(self, object_id, req, resp, **kwargs):
        pass


class DnlResource(Resource):
    def get_spec_info(self):
        return swagger.specify.get_spec(
            method='get', description='Gets {}'.format(self.entity.lower()),
            path_parameters=self.get_path_parameters(),
            responses=self.get_common_on_get_responses() + self.get_additional_responses(method='get'),
            security=self.get_security(method='get', action='info')
        )

    def get_spec_modify(self):
        if self.scheme_class or self.scheme_class_modify:
            body_parameters = (
                '{} to modify'.format(self.entity),
                self.scheme_class_modify if self.scheme_class_modify else self.scheme_class

            )
        else:
            body_parameters = ()

        return swagger.specify.get_spec(
            method='patch', description='Modifies {}'.format(self.entity.lower()),
            path_parameters=self.get_path_parameters(),
            body_parameters=body_parameters,
            responses=
            # responses.SuccessResponseJustOk(),
            self.get_common_on_get_responses()
            # responses.ObjectNotFoundErrorResponse()
            + self.get_additional_responses(method='patch'),
            security=self.get_security(method='patch', action='modify')
        )

    def on_get(self, req, resp, **kwargs):  # pragma: no cover
        log.debug("ON GET")
        self.init_req(req)
        if not check_context(self, req, resp, **kwargs):
            return False
        kwargs = client_context(self, req, resp, **kwargs)
        if not self.has_info_operation:  # pragma: no cover
            return self.method_not_allowed_response(resp, 'GET')
        user = self.get_user(req)
        if user and user.user_type == 'client':
            pass
        elif not check_permission(self, req, 'show', self.get_object(resp, self.model_class, **kwargs)):
            self.set_response(
                resp, responses.ForbiddenErrorResponse(data=errors.AuthErrors.Forbidden)
            )
            return
        data = self.get_object_data(resp, self.model_class, self.scheme_class_get, **kwargs)
        if data:
            self.set_response(resp, responses.SuccessResponseObjectInfo(data=data))
        else:
            self.set_response(resp, responses.ObjectNotFoundErrorResponse())

    def get_object(self, resp, model_class, **kwargs):
        if hasattr(self, 'id_field') and self.id_field in kwargs:
            id = kwargs[self.id_field]
            key = getattr(self.model_class, self.id_field)
            if key.type.python_type == type(0):
                try:
                    _id = int(id)
                except:
                    return None
        obj = super(DnlResource, self).get_object(resp, model_class, **kwargs)
        if obj and hasattr(obj, 'name') and obj.name and obj.name[0] == '#':
            return None
        return obj

    def before_delete(self, req, resp, model_class, **kwargs):
        pass

    def delete_object(self, req, resp, model_class, **kwargs):
        primary_key = model_class.get_model_class_primary_key()
        if primary_key in kwargs:
            try:
                self.before_delete(req, resp, model_class, **kwargs)
            except Exception as e:
                log.error(e)
                self.set_response(resp, OperationErrorResponse(e))
                return False
        if primary_key in kwargs and kwargs[primary_key] == '-1':
            if hasattr(model_class, 'delete_query'):
                cnt = model_class.delete_query().delete()
            else:
                cnt = model_class.query().delete()
            try:
                cnt = model_class.query().session.commit()
                self.set_response(resp, responses.SuccessResponseJustOk(data={'deleted': cnt}))
            except OperationalError as e:
                self.set_response(resp, OperationalErrorResponse(e))
            except Exception as e:
                model_class.query().session.rollback()
                log.error('Mass delete error:{}'.format(str(e)))
            return False
        else:
            obj = self.get_object(resp, model_class, **kwargs)
            if not obj:
                return None
            return obj.delete(save_history=True, user=self.get_user(req), module=getattr(self.__class__, 'api_module', 'N/A'))
                
    def update_object(self, req, resp, model_class, scheme_class, **kwargs):
        self.init_req(req)
        log.debug("GETTING OBJECT")
        obj = self.get_object(resp, model_class, **kwargs)
        log.debug(f"OBJ = {obj}")
        if not obj:
            return None

        scheme = scheme_class().load(self.req_data, instance=self.before_update(obj, req), partial=True)
        if scheme.errors:
            self.set_response(resp, responses.ValidationErrorResponse(data=scheme.errors))
            return False
        user = self.get_user(req)
        if user:
            save_history = not req.context['service']
            result = scheme.data.save(save_history=save_history, user=self.get_user(req),
                                      module=getattr(self.__class__, 'api_module', 'N/A'))
            log.debug(f"SAVED RESULT - {result}")
        else:
            result = scheme.data.save()
        try:
            log.debug(f"RESULT = {result}")
            self.after_update(result, obj, scheme.data)
            model_class.session().refresh(obj)
        except Exception as e:
            log.warning('after_update something wrong {}'.format(e))

        return result

    def on_delete(self, req, resp, **kwargs):  # pragma: no cover
        try:
            self.init_req(req)
            log.debug("\nREQ - {}\nRESP - {}")
            kw = client_context(self, req, resp, **kwargs)
            log.debug("KW - {}".format(kw))
            new_kw = {}
            for key, value in kw.items():
                new_kw[str(key)] = value
                log.debug("KEY - {}\tKEY.TYPE - {}\nVALUE - {}\tVALUE.TYPE - {}".format(key, type(key), value, type(value)))
            return super(DnlResource, self).on_delete(req, resp, **new_kw)

        except IntegrityError as e:
            try:
                msg = str(e).split('\n')[1].split(':')[1]
            except:
                msg = str(e)
            try:
                model.get_db().session.rollback()
            except Exception as e2:
                log.error('on delete error {} rollback also failed {}'.format(e, e2))
            self.set_response(resp, AlreadyExistsResponse(self.entity, msg))

        except FkConstraintViolation as e:
            self.model_class.session().rollback()

            self.set_response(resp, responses.FkConstraintViolationResponse(
                data=errors.CommonErrors.get_fk_violation_error(e.table_name, e.column_name, e.value))
                              )

        except NoResultFound as e:
            self.set_response(resp, responses.ObjectNotFoundErrorResponse(data=str(e)))
            return None
        except OperationalError as e:
            self.set_response(resp, OperationalErrorResponse(e))
        except Exception as e:
            log.error(e)
            log.debug("FAILED!\nREQ - {}\nRESP - {}".format(req, resp))
            # self.set_response(resp, responses.OperationErrorResponse(data=errors.CommonErrors.DataError))
            self.set_response(resp, OperationErrorResponse(e))

    # @profile
    def on_patch(self, req, resp, **kwargs):
        self.init_req(req)
        if not check_context(self, req, resp, **kwargs):
            return False
        kwargs = client_context(self, req, resp, **kwargs)
        if not self.has_modify_operation:  # pragma: no cover
            return self.method_not_allowed_response(resp, 'PATCH')
        if not check_permission(self, req, 'modify', self.get_object(resp, self.model_class, **kwargs)):
            self.set_response(
                resp, responses.ForbiddenErrorResponse(data=errors.AuthErrors.Forbidden)
            )
            return
        try:
            log.debug("UPDATING OBJECT")
            a = self.update_object(
                    req, resp, self.model_class,
                    self.scheme_class_modify if self.scheme_class_modify else self.scheme_class, **kwargs
            )
            log.debug(a)
            if a not in (None, False):
                data = self.get_object_data(resp, self.model_class, self.scheme_class_get, **kwargs)
                if data:
                    self.set_response(resp, responses.SuccessResponseObjectInfo(data=data))
                else:
                    self.set_response(resp, responses.SuccessResponseJustOk())
        except IntegrityError as e:
            try:
                msg = str(e).split('\n')[1].split(':')[1]
            except:
                msg = str(e)
            try:
                model.get_db().session.rollback()
            except Exception as e2:
                log.error('on patch error {} rollback also failed {}'.format(e, e2))
            self.set_response(resp, AlreadyExistsResponse(self.entity, msg))

        except AlreadyExists as e:
            r = AlreadyExistsResponse(self.entity, str(e))
            r.status_code = 1000
            self.set_response(resp, r)

        except FkConstraintViolation as e:
            self.model_class.session().rollback()
            self.set_response(resp, responses.FkConstraintViolationResponse(
                data=errors.CommonErrors.get_fk_violation_error(e.table_name, e.fk_column_name, e.column_name, e.value))
                              )
        except ValidationError as e:
            self.set_response(resp, responses.ValidationErrorResponse(data=e.messages))

        except NoResultFound as e:
            self.set_response(resp, responses.ObjectNotFoundErrorResponse(data=str(e)))
            return None
        except OperationalError as e:
            self.set_response(resp, OperationalErrorResponse(e))
        except Exception as e:
            log.debug(f"Other ERROR - {e}")
            log.error(traceback.format_exc())
            self.set_response(resp, OperationErrorResponse(e))

    def before_update(self, obj, req):
        if hasattr(self, 'unique_field') and self.unique_field in req.data:
            new = req.data[self.unique_field]
            old = getattr(obj, self.unique_field)
            log.debug(f"OLD - {old}")
            log.debug(f"NEW - {new}")
            cls = self.model_class
            field = getattr(cls, self.unique_field)
            if new != old:
                if cls.filter(field == new).first():
                    raise ValidationError({self.unique_field: ['{} {} is invalid (duplicate)'.format(field.name, new)]})
        return super(DnlResource, self).before_update(obj, req)


def xml_tostring(x):
    if x == None:
        return ''
    else:
        return str(x)


def gen_csv(q, res, resp, _from, total, fields):
    try:
        import csv
        import io
        window_size = 256  # or whatever limit you like
        window_idx = 0
        _count = 0

        if type(q) == type([]):
            yield ('\r\n\r\n').encode()
            return
        d = []
        # yield b'\r\n'
        # all_things = q.all()
        for thing in q.yield_per(1024):
            _count += 1
            window_idx += 1
            d.append(thing)
            if window_idx == window_size:
                data = res.scheme_class(only=fields).dump(d, many=True).data
                csvfile = io.StringIO()
                if len(data):
                    fieldnames = data[0].keys()
                    writer = csv.DictWriter(csvfile, fieldnames=fieldnames)
                    if _count <= window_size:
                        writer.writeheader()
                    writer.writerows(data)
                    yield str.encode(csvfile.getvalue())
                d = []
                window_idx = 0
        if d:  # rest piece
            data = res.scheme_class(only=fields).dump(d, many=True).data
            csvfile = io.StringIO()
            if len(data):
                fieldnames = data[0].keys()
                writer = csv.DictWriter(csvfile, fieldnames=fieldnames)
                if _count <= window_size:
                    writer.writeheader()
                writer.writerows(data)
                yield str.encode(csvfile.getvalue())

        yield ('\r\n\r\n').encode()
    except Exception as e:
        # model.get_db().session.rollback()
        log.debug('gen_csv UNEXPECTED ERROR: {}'.format(e))
        yield str.encode('\r\n\r\ngen_csv UNEXPECTED ERROR: {}\r\n\r\n'.format(e))


class DnlList(List):
    allow_methods = ['get']
    entity_plural = None
    model_class = None

    def _on_get(self, req, resp, **kwargs):
        if (not hasattr(self, 'no_auth_needed') or self.no_auth_needed == False) and not check_permission(self, req,
                                                                                                          'list'):
            self.set_response(
                resp, responses.ForbiddenErrorResponse(data=errors.AuthErrors.Forbidden)
            )
            return
        try:
            log.debug("req.headers['ACCEPT']={}".format(req.headers['ACCEPT']))
            if 'ACCEPT' in req.headers and '*/*' in req.headers['ACCEPT'] and 'application/json' not in req.headers[
                'ACCEPT']:
                req.headers['ACCEPT'] = 'text/csv'
            log.debug("GETTING OBJECTS LIST")
            objects_list, total, page, per_page, fields = self.get_objects_list(req, self.model_class, **kwargs)
            log.debug("GOT OBJECTS LIST")
            try:
                per_page = int(dict(parse_qsl(req.query_string)).get('per_page'))
                log.debug("PER_PAGE TRIES SUCCESSFULLY - {}".format(per_page))
            except (ValueError, TypeError):
                per_page = conf.get_default_items_per_page()
                log.debug("PER_PAGE TRIES, but got an error - {}".format(per_page))
            if 'ACCEPT' in req.headers and req.headers['ACCEPT'] == 'text/csv':
                log.debug("'ACCEPT' in req.headers and req.headers['ACCEPT'] == 'text/csv'")
                resp.content_type = 'text/csv'
                resp.append_header('Transfer-Encoding', 'chunked')
                resp.append_header('Content-Disposition', 'attachment; filename="{}.csv"'.format(self.entity_plural))
                resp_stream = gen_csv(objects_list, self, resp, 0, total, fields)
                resp.stream = resp_stream
                # resp.status = falcon.HTTP_PARTIAL_CONTENT
                resp.status = falcon.HTTP_200
            else:
                try:
                    self.set_response(
                        resp, responses.SuccessResponse(
                            data={
                                'items': self.scheme_class(only=fields).dump(objects_list, many=True).data,
                                'total': total, 'page': page, 'per_page': per_page
                            },
                            scheme=schemes.ObjectScheme
                        )
                    )
                except Exception as e:
                    log.debug(f"ERROR - {e}")
        except AttributeError as e:
            self.set_response(
                resp, responses.AttributeNotExitsErrorResponse(
                    data=ATTRIBUTE_ERROR_RE.search(str(e)).group(1)
                )
            )
        except Exception as e:
            import traceback
            log.error(traceback.format_exc())
            self.set_response(resp, OperationErrorResponse(e))

    def on_get(self, req, resp, **kwargs):  # pragma: no cover

        self.init_req(req)
        if not check_context(self, req, resp, **kwargs):
            return False
        kwargs = client_context(self, req, resp, **kwargs)
        try:
            # ret = super(DnlList,self).on_get(req, resp, **kwargs)
            ret = self._on_get(req, resp, **kwargs)
        except NoResultFound:
            self.set_response(resp, responses.ObjectNotFoundErrorResponse())
            return None
        except OperationalError as e:
            self.set_response(resp, OperationalErrorResponse(e))
        if 'ACCEPT' in req.headers and (
                req.headers['ACCEPT'] == 'application/xml' or req.headers['ACCEPT'] == 'text/xml'):
            try:
                data = json.loads(resp.body)
                result = bf(xml_tostring=xml_tostring).etree(data=data, root=Element('xml', attrib={
                    'entity': self.model_class.__name__, 'api_version': VERSION, 'timestamp': str(datetime.utcnow())}))
                rough_string = tostring(result, 'utf-8')
                reparsed = minidom.parseString(rough_string)
                resp.body = reparsed.toprettyxml(indent='\t')
            except:
                resp.body = '<xml>error</xml>'
            resp.content_type = 'text/xml'
        # if 'ACCEPT' in req.headers and req.headers['ACCEPT'] == 'text/csv':
        #     data = json.loads(resp.body)
        #     try:
        #         csvfile = io.StringIO()
        #         items = data['payload']['items']
        #         if len(items):
        #             fieldnames = data['payload']['items'][0].keys()
        #             writer = DictWriter(csvfile, fieldnames=fieldnames)
        #             writer.writeheader()
        #             writer.writerows(items)
        #             resp.body = str.encode(csvfile.getvalue())
        #         else:
        #             resp.body = ''
        #     except:
        #         # resp.body = 'success,error\n{},{}'.format(data['sucess'],data['error']['message'])
        #         resp.body = 'error'
        #     resp.content_type = 'text/csv'
        log.debug(f"FINAL RET = {ret}")
        return ret

    def csv_resp_stream(res, q,  resp, _from, total):
        try:
            import csv
            import io
            window_size = 10  # or whatever limit you like
            window_idx = 0
            _count = 0

            if type(q) == type([]):
                yield ('\r\n\r\n').encode()
                return
            d = []
            # yield b'\r\n'
            # all_things = q.all()
            for thing in q.yield_per(500):
                _count += 1
                window_idx += 1
                d.append(thing)
                if window_idx == window_size:
                    data = res.scheme_class().dump(d, many=True).data
                    csvfile = io.StringIO()
                    if len(data):
                        fieldnames = data[0].keys()
                        writer = csv.DictWriter(csvfile, fieldnames=fieldnames)
                        if _count <= window_size:
                            writer.writeheader()
                        writer.writerows(data)
                        yield str.encode(csvfile.getvalue())
                    d = []
                    window_idx = 0
            if d:  # rest piece
                data = res.scheme_class().dump(d, many=True).data
                csvfile = io.StringIO()
                if len(data):
                    fieldnames = data[0].keys()
                    writer = csv.DictWriter(csvfile, fieldnames=fieldnames)
                    if _count <= window_size:
                        writer.writeheader()
                    writer.writerows(data)
                    yield str.encode(csvfile.getvalue())

            # yield ('],"total":{},"from":{},"count":{}}}\r\n\r\n'.format(count, _from, total)).encode()
            # finish
            yield ('\r\n\r\n').encode()
        except Exception as e:
            # model.get_db().session.rollback()
            #if CDR_DEBUG:
            log.debug('csv file get UNEXPECTED ERROR: {}'.format(e))
            yield str.encode('\r\n\r\ncsv file get UNEXPECTED ERROR: {}\r\n\r\n'.format(e))

    def get_filtering_for_list(self, parsed_qs, **kwargs):
        filtering = {}
        for k, v in parsed_qs.items():
            if k in ['fields', 'per_page', 'page', 'order_by', 'order_dir']:
                continue
            filtering[k] = v
        return filtering

    def _modify_query_from_ordering_for_list(self, ordering, query, **kwargs):
        if 'by' in ordering:
            if ordering['by'] not in self.get_all_fields_from_scheme(self.scheme_class)[0]:
                raise Exception('Order by {} not valid!'.format(ordering['by']))
        return ordering, query

    def modify_query_from_ordering_for_list(self, ordering, query, **kwargs):
        all_fields = self.get_all_fields_from_scheme(self.scheme_class)[0]
        if ordering:
            if ',' in ordering['by']:
                ordl = ordering['by'].split(',')
                dirl = ordering['dir'].split(',')
                if len(ordl) > 1 and len(ordl) != len(dirl):
                    raise Exception('Multi order_by must contain also all order_dir asc/desc specifications!')
                for ord in ordl:
                    if ord not in all_fields:
                        raise Exception('Order by {} not valid!'.format(ord))
                for dir in dirl:
                    if dir not in ['asc', 'desc']:
                        raise Exception('Wrong order dir {} ! Valid are ["ask","desk"]'.format(ord))
        return ordering, query

    def get_fields_for_list(self, parsed_qs, **kwargs):
        fields = parsed_qs.get('fields', None)
        if fields:
            fields = fields.split(',')
            for f in fields:
                if not f in self.get_all_fields_from_scheme(self.scheme_class)[0]:
                    raise Exception('fields {} not valid!'.format(f))
        elif hasattr(self, 'large_list_fields'):
            log.debug('large_list query {} headers {}'.format(parsed_qs, self.req.headers))
            if 'per_page' in parsed_qs and int(parsed_qs['per_page']) >= 1000:
                fields = self.large_list_fields
        return fields

    def get_objects_list(self, req, model_class, **kwargs):
        parsed_qs = dict(parse_qsl(req.query_string))
        try:
            per_page = int(parsed_qs.get('per_page', 10))
        except (ValueError, TypeError):
            per_page = conf.get_default_items_per_page()

        try:
            page = int(parsed_qs.get('page', 0))
        except ValueError:
            page = 0

        if hasattr(self, 'per_page_limit') and per_page > self.per_page_limit:
            raise ValidationError('per_page  over limit {}'.format(self.per_page_limit))

        if 'order_by' in parsed_qs:
            ordering = {
                'by': parsed_qs['order_by'],
                'dir': parsed_qs.get('order_dir', 'asc')
            }
        elif hasattr(self, 'ordering'):
            ordering = self.ordering
        else:
            ordering = {}
        fields = self.get_fields_for_list(parsed_qs, **kwargs)
        filtering = self.get_filtering_for_list(parsed_qs, **kwargs)
        result = self.modify_query_from_filtering_for_list(filtering, **kwargs)
        log.debug(f"FILTERING = {filtering}")
        log.debug(f"RESULT = {result}")
        if result is not None:
            filtering, query = result
        else:
            filtering, query = {}, None
        log.debug(f"NEW FILTERING = {filtering}")
        log.debug(f"QUERY = {query}")
        for c in inspect(self.model_class).columns:
            if c.name in ['alias', 'name', 'template_name', 'group_name']:
                query = query.filter(or_(c.is_(None), c.notlike('#%')))
                break
        log.debug("if ordering:")
        if ordering:
            ordering, query = \
                self.modify_query_from_ordering_for_list(ordering, query, **kwargs)

        if ('ACCEPT' in req.headers and req.headers['ACCEPT'] == 'text/csv'):
            log.debug("return model_class.get_objects_query")
            return model_class.get_objects_query(
                query=query,
                filtering=filtering,
                ordering=ordering,
                paging=None
            ) + (0, None, fields)
        else:
            if 'by' in ordering.keys() and ordering['by'] == 'invoice_number':
                per_page = None
            log.debug("return model_class.get_objects_list")
            log.debug(f"QUERY = {query}")
            log.debug(f"FILTERING = {filtering}")
            log.debug(f"ORDERING = {ordering}")
            return model_class.get_objects_list(
                query=query,
                filtering=filtering,
                ordering=ordering,
                paging={'from': page * per_page, 'till': (page + 1) * per_page} if per_page else None,
                query_only = 'query_only' in kwargs
            ) + (page, per_page, fields)

    @staticmethod
    def parse_query_field(k, model):
        trans = {'gt': ' >= ', 'gte': ' >= ', 'lt': ' <= ', 'lte': ' <= ', 'in': ' in ({})', 'eq': ' = ',
                 'isnull': ' is null = {}'}
        suffix = k.split('_')[-1]
        if not suffix in trans.keys():
            suffix = 'eq'
            qfield = k
        else:
            qfield = '_'.join(k.split('_')[0:-1])
        insp = inspect(model)
        cols = insp.columns
        syns = insp.synonyms
        field = qfield
        if qfield in syns:
            field = syns[qfield].name
        postfix = "'{}'"
        try:
            typ = cols[field].type
            if isinstance(typ, DateTime) or isinstance(typ, String) or isinstance(typ, Date):
                postfix = "'{}'"
            if suffix in ['in', 'isnull']:
                postfix = ''
        except:
            postfix = "'{}'"

        if field in cols and hasattr(cols[field], '_element'):
            dbfield = str(cols[field].expression)  # _element)
        else:
            dbfield = field
        if suffix in trans.keys():
            return dbfield + trans[suffix] + postfix, qfield
        return k, qfield

    def modify_query_from_filtering_for_list(self, filtering, **kwargs):
        from sqlalchemy.orm import defer, undefer, load_only

        if 'start_query' in kwargs:
            ret = kwargs['start_query']
        else:
            ret = self.model_class.query()
        fields = self.get_fields_for_list(dict(parse_qsl(self.req.query_string)) if self.req else {})
        if fields:
            # ret.options(defer('*'))
            # for f in fields:
            #     ret.options(undefer(f))
            ret = ret.options(load_only(*fields))
        filt = {}
        query_fields = self.scheme_class.Meta.query_fields if hasattr(self.scheme_class.Meta, 'query_fields') else []
        cols = inspect(self.model_class).columns
        syns = inspect(self.model_class).synonyms
        if filtering:
            for k, v in filtering.items():
                if hasattr(self, 'on_filtering'):
                    ret = getattr(self, 'on_filtering')(ret, k, v, kwargs)
                    continue
                if hasattr(self, 'on_filter_' + k):
                    ret = getattr(self, 'on_filter_' + k)(ret, v, kwargs)
                    continue
                if query_fields and k in query_fields:
                    fop, dbfield = DnlList.parse_query_field(k, self.model_class)
                    if dbfield in cols:
                        col = cols[dbfield]
                    elif dbfield in syns:
                        col = cols[syns[dbfield].name]
                    else:
                        raise Exception('Bad value for filtering! field "{}" not found!'.format(dbfield))
                    if ' in ({})' in fop:
                        lst = v.split(',')
                        v1 = []
                        log.debug("LST ??? - {}".format(lst))
                        for i in lst:
                            t = 'unknown'
                            try:
                                if isinstance(col.type, Integer):
                                    t = 'int'
                                    x = int(i)
                                elif isinstance(col.type, Numeric):
                                    t = 'numeric'
                                    x = float(i)
                                    x = "'" + i + "'"
                                elif isinstance(col.type, DateTime) or isinstance(cols[dbfield].type, Date):
                                    t = 'datetime'
                                    x = parse_datetime(i)
                                    x = "'" + i + "'"
                                else:
                                    t = 'string'
                                    x = "'" + i + "'"
                                v1.append(i)
                            except Exception as e:
                                raise Exception('Bad value for filtering list! "{}" "{}" {}'.format(v, i, str(e)))
                        v = ','.join(v1)
                    ret = ret.filter(text_(fop.format(v)))
                    continue
                if k in cols and hasattr(cols[k].type, 'choices'):
                    if not v in tuple(cols[k].type.choices.values()):
                        raise Exception('Bad value for filtering! "{}"="{}" must be one of [{}]'.format(k, v, ','.join(
                            list(cols[k].type.choices.values()))))
                        continue
                if v == 'null':
                    filt[k] = None
                else:
                    filt[k] = v
        log.debug("FILT - {}\nRET - {}".format(filt, ret))
        return filt, ret

    @staticmethod
    def get_fields_from_scheme(scheme_class):  # pragma: no cover
        fields_list = []
        fields_names = []

        search_fields = scheme_class.Meta.search_fields if hasattr(scheme_class.Meta, 'search_fields') else []

        # noinspection PyProtectedMember
        for field_name, field in scheme_class._declared_fields.items():
            field_type = swagger.specify.get_field_type(field)
            if field_type is None and isinstance(field, fields.Decimal):
                field_type = 'float'

            # if isinstance(field,Numeric):
            #    field_type = 'string'
            if not field_type:
                continue

            if field_name not in search_fields:
                continue

            fields_names.append(field_name)

            if field_type == 'datetime' or field_name not in search_fields:
                continue

            f = {
                'name': field_name,
                'type': field_type
            }
            if 'description' in field.metadata:
                f['description'] = field.metadata['description']

            if isinstance(field.validate, validate.OneOf):
                f['enum'] = field.validate.choices
            if isinstance(field, Choice):
                f['enum'] = field._get_choices(field_name, scheme_class)

            fields_list.append(f)

        return sorted(fields_names), sorted(fields_list, key=lambda item: item['name'])

    @staticmethod
    def get_query_fields_from_scheme(scheme_class):
        fields_list = []
        fields_names = []
        query_fields = scheme_class.Meta.query_fields if hasattr(scheme_class.Meta, 'query_fields') else []
        for query_field in query_fields:
            fop, field_name = DnlList.parse_query_field(query_field, scheme_class.Meta.model)
            decl = scheme_class._declared_fields
            if field_name in decl.keys():
                field = decl[field_name]
                field_type = swagger.specify.get_field_type(field)
                if not field_type and isinstance(field, fields.Decimal):
                    field_type = 'number'
                if ' in ({})' in fop:
                    field_type = 'string'
                if ' is null' in fop:
                    field_type = 'boolean'
                if not field_type or field_name in []:
                    continue
                fields_names.append(field_name)
                f = {
                    'name': query_field,
                    'type': field_type
                }

                fields_list.append(f)

        return sorted(fields_names), sorted(fields_list, key=lambda item: item['name'])

    @staticmethod
    def get_all_fields_from_scheme(scheme_class):
        sfields_names, search_fields_list = DnlList.get_fields_from_scheme(scheme_class)
        qfields_names, query_fields_list = DnlList.get_query_fields_from_scheme(scheme_class)
        return sorted(sfields_names + qfields_names), \
               sorted(search_fields_list + query_fields_list, key=lambda item: item['name'])

    def _get_spec_info(self):
        spec = super(DnlList, self).get_spec_info()
        spec['get']['produces'] = ['application/json', 'text/csv', 'text/xml']
        return spec

    def get_spec(self):
        sfields_names, search_fields_list = self.get_fields_from_scheme(self.scheme_class)
        qfields_names, query_fields_list = self.get_query_fields_from_scheme(self.scheme_class)
        fields_names = list(set(sfields_names + qfields_names))
        spec = swagger.specify.get_spec(
            method='get', description='Gets {}'.format(self.entity_plural.lower()),
            path_parameters=self.path_parameters,
            query_parameters=[{'name': 'fields', 'type': 'string'},
                              {'name': 'page', 'type': 'integer'}, {'name': 'per_page', 'type': 'integer'},
                              {'name': 'order_by', 'enum': fields_names}, {'name': 'order_dir', 'enum': ['asc', 'desc']}
                              ] + search_fields_list + query_fields_list,
            responses=(
                          responses.SuccessResponseObjectsList(payload_scheme_items=self.scheme_class),
                          responses.AttributeNotExitsErrorResponse()
                      ) + self.get_additional_responses(method='get'),
            security=self.get_security(method='get')
        )
        spec['get']['produces'] = ['application/json', 'text/csv', 'text/xml']
        return spec


class SuccessResponseObjectsListWithSummary(SuccessResponse):
    # many = True
    scheme = ObjectsListWithSummaryScheme


class DnlSummaryList(DnlList):

    def get_additional_responses(self, method):
        return (SuccessResponseObjectsListWithSummary(payload_scheme_items=self.scheme_class),)

    def get_total_data(self, req, resp, **kwargs):
        raise NotImplemented

    def _on_get(self, req, resp, **kwargs):
        if (not hasattr(self, 'no_auth_needed') or self.no_auth_needed == False) and not check_permission(self, req,
                                                                                                          'list'):
            self.set_response(
                resp, responses.ForbiddenErrorResponse(data=errors.AuthErrors.Forbidden)
            )
            return
        try:
            objects_list, total, page, per_page, fields = self.get_objects_list(req, self.model_class, **kwargs)
            self.set_response(
                resp, responses.SuccessResponse(
                    data={
                        'items': self.scheme_class(only=fields).dump(objects_list, many=True).data,
                        'summary': self.get_total_data(req, resp, **kwargs),
                        'total': total, 'page': page, 'per_page': per_page
                    },
                    scheme=ObjectsListWithSummaryScheme
                )
            )
        except AttributeError as e:
            self.set_response(
                resp, responses.AttributeNotExitsErrorResponse(
                    data=ATTRIBUTE_ERROR_RE.search(str(e)).group(1)
                )
            )
        except OperationalError as e:
            self.set_response(resp, OperationalErrorResponse(e))
        except Exception as e:
            log.error(e)
            # self.set_response(resp, responses.OperationErrorResponse(data=errors.CommonErrors.DataError))
            self.set_response(resp, OperationErrorResponse(e))


class DnlResourceAll(Resource):
    has_info_operation = False

    def modify_query(self, filt, qs, req, resp, model_class, **kwargs):
        return filt, qs

    def get_query_filter(self, req, resp, model_class, **kwargs):
        filtering = dict(parse_qsl(req.query_string))
        qs = model_class.query()
        filtering, qs = self.modify_query(filtering, qs, req, resp, model_class, **kwargs)
        query_fields = self.scheme_class.Meta.query_fields if hasattr(self.scheme_class.Meta, 'query_fields') else []
        cols = inspect(self.model_class).columns
        if filtering:
            for k, v in filtering.items():
                if query_fields and k in query_fields:
                    fop = DnlList.parse_query_field(k, self.model_class)[0]
                    qs = qs.filter(text_(fop.format(v)))
                    continue
                if k in cols and hasattr(cols[k].type, 'choices'):
                    if not v in tuple(cols[k].type.choices.values()):
                        log.error('Bad value for filtering! {}={}'.format(k, v))
                        raise Exception('Bad value for filtering! "{}"="{}" must be one of [{}]'.format(k, v, ','.join(
                            list(cols[k].type.choices.values()))))
                        continue
                if '*' in v:  # pragma: no cover
                    qs = qs.filter(getattr(model_class, k).like(v.replace('*', '%')))
                else:
                    qs = qs.filter(getattr(model_class, k).in_(v.split(',')))
        return qs

    def update_object(self, req, resp, model_class, scheme_class, **kwargs):
        self.init_req(req)

        if hasattr(self.model_class, 'update_by'):
            self.req_data['update_by'] = self.get_user().name
        if hasattr(self.model_class, 'update_on'):
            self.req_data['update_by'] = datetime.now(UTC)
        if hasattr(self.model_class, 'update_at'):
            self.req_data['update_at'] = datetime.now(UTC)
        scheme = scheme_class().load(self.req_data)
        if scheme.errors:
            self.set_response(resp, responses.ValidationErrorResponse(data=scheme.errors))
            return False
        # result = self.model_class.query().update(self.req_data)
        query = self.get_query_filter(req, resp, model_class, **kwargs)
        if hasattr(self, 'on_all_update'):
            self.on_all_update(query, self.req_data)
        result = query.update(self.req_data, synchronize_session='fetch')
        self.result = result
        self.model_class.session().commit()
        self.set_response(resp, responses.SuccessResponseObjectInfo(data={'updated': result}))
        return False

    def delete_object(self, req, resp, model_class, **kwargs):
        try:
            query = self.get_query_filter(req, resp, model_class, **kwargs)
            if hasattr(self, 'on_all_delete'):
                if self.on_all_delete(query):
                    self.set_response(resp, responses.SuccessResponseJustOk())
                    return False
            result = query.delete(synchronize_session='fetch')
            model_class.session().commit()
            self.set_response(resp, responses.SuccessResponseObjectInfo(data={'deleted': result}))
        except Exception as e:
            model_class.query().session.rollback()
            log.error('Mass delete error:{}'.format(str(e)))
        return False

    def get_spec_modify(self):
        if self.scheme_class or self.scheme_class_modify:
            body_parameters = (
                '{} to modify'.format(self.entity),
                self.scheme_class_modify if self.scheme_class_modify else self.scheme_class

            )
        else:
            body_parameters = ()

        return swagger.specify.get_spec(
            method='patch', description='Modifies multiple found {}'.format(self.entity.lower()),
            path_parameters=self.get_path_parameters(),
            body_parameters=body_parameters,
            query_parameters=DnlList.get_all_fields_from_scheme(self.scheme_class)[1],
            responses=
            # responses.SuccessResponseJustOk(),
            self.get_common_on_get_responses()
            # responses.ObjectNotFoundErrorResponse()
            + self.get_additional_responses(method='patch'),
            security=self.get_security(method='patch', action='modify')
        )

    def get_spec_delete(self):
        return swagger.specify.get_spec(
            method='delete', description='Deletes multiple found {}'.format(self.entity.lower()),
            path_parameters=self.get_path_parameters(),
            query_parameters=DnlList.get_all_fields_from_scheme(self.scheme_class)[1],
            responses=(
                          responses.SuccessResponseObjectInfo(payload_scheme=self.scheme_class_get),
                          responses.ObjectNotFoundErrorResponse()
                      ) + self.get_additional_responses(method='delete'),
            security=self.get_security(method='delete', action='delete')
        )


class CustomAction(_CustomAction):
    def proceed(self, req, resp, **kwargs):

        if (not hasattr(self, 'no_auth_needed') or self.no_auth_needed == False) and \
                not check_permission(self, req, self.action, self.get_object(resp, self.model_class, **kwargs)):
            self.set_response(
                resp, responses.ForbiddenErrorResponse(data=errors.AuthErrors.Forbidden)
            )
            return

        try:
            obj = self.get_object(resp, self.model_class, **kwargs)
            if self.check_conditions(obj, self.pre_conditions, lambda a, b: a == b):
                if self.check_conditions(obj, self.pre_conditions_inverted, lambda a, b: a != b):
                    if self.apply(obj, req, resp, **kwargs):
                        self.set_response(resp, responses.SuccessResponseJustOk())

        except FkConstraintViolation as e:
            self.set_response(resp, responses.FkConstraintViolationResponse(
                data=errors.CommonErrors.get_fk_violation_error(e.table_name, e.column_name, e.value))
                              )

        except OperationalError as e:
            self.set_response(resp, OperationalErrorResponse(e))

        except IntegrityError as e:
            try:
                msg = str(e).split('\n')[1].split(':')[1]
            except:
                msg = str(e)
            try:
                model.get_db().session.rollback()
            except Exception as e2:
                log.error('on custom action error {} rollback also failed {}'.format(e, e2))
            self.set_response(resp, AlreadyExistsResponse(self.entity, msg))
            return False
        except PreConditionFailedException as e:
            self.set_response(resp, responses.OperationErrorResponse(
                data=errors.CommonErrors.get_pre_condition_failed_error(e.message))
                              )
        except Exception as e:
            self.set_response(resp, OperationErrorResponse(e))


class CustomPatchAction(CustomAction):
    method = 'patch'

    def on_patch(self, req, resp, **kwargs):
        self.init_req(req)
        kwargs = client_context(self, req, resp, **kwargs)
        return self.proceed(req, resp, **kwargs)


class CustomPostAction(CustomAction):
    method = 'post'

    def on_post(self, req, resp, **kwargs):
        self.init_req(req)
        kwargs = client_context(self, req, resp, **kwargs)
        return self.proceed(req, resp, **kwargs)


class CustomGetAction(CustomAction):
    method = 'get'
    query_parameters = []

    def on_get(self, req, resp, **kwargs):
        self.init_req(req)
        kwargs = client_context(self, req, resp, **kwargs)
        return self.proceed(req, resp, **kwargs)

    def get_spec(self):
        return swagger.specify.get_spec(
            method=self.method, description=self.description,
            path_parameters=self.path_parameters,
            query_parameters=self.query_parameters,
            body_parameters=self.body_parameters,
            responses=(
                          responses.SuccessResponseJustOk(),
                          responses.ObjectNotFoundErrorResponse()
                      ) + self.additional_responses,
            security=self.get_security(method=self.method)
        )


class CustomDeleteAction(CustomAction):
    method = 'delete'
    query_parameters = []

    def on_delete(self, req, resp, **kwargs):
        self.init_req(req)
        kwargs = client_context(self, req, resp, **kwargs)
        return self.proceed(req, resp, **kwargs)


class DnlPatchManyResource(CustomPatchAction):
    modify_only = False

    def on_patch(self, req, resp, **kwargs):
        self.init_req(req)
        if not check_context(self, req, resp, **kwargs):
            return False
        q = self.model_class.query().first()
        if q:
            kwargs[self.id_field] = q.get_object_primary_key_value()
        else:
            self.model_class(**{self.id_field: 1}).save()
            kwargs[self.id_field] = '1'

        return self.proceed(req, resp, **kwargs)

    def apply(self, obj, req, resp, **kwargs):
        result = []
        updated = 0
        added = 0
        deleted = 0
        try:
            ids = [it[self.id_field] for it in req.data['items'] if self.id_field in it]
            if not self.modify_only:
                for obj in self.model_class.query().all():
                    if not obj.get_object_primary_key_value() in ids:
                        kwargs[self.id_field] = str(obj.get_object_primary_key_value())
                        if self.delete_object(req, resp, self.model_class, **kwargs):
                            deleted += 1
            for data in req.data['items']:
                req.data = data
                if self.id_field in data and not data[self.id_field] == 'null':
                    kwargs[self.id_field] = data[self.id_field]
                    if self.update_object(
                            req, resp, self.model_class, self.scheme_class, **kwargs):
                        updated += 1
                else:
                    if 'id' in kwargs:
                        del kwargs[self.id_field]
                    scheme = self.scheme_class(exclude=(self.id_field,))
                    if not self.modify_only:
                        if self.check_request_data(req, resp, scheme):
                            # resp = self.scheme_class.get_object_created_response(data=self.create_object(req, scheme, **kwargs) )
                            obj = self.create_object(req, scheme, **kwargs)
                            added += 1

            if (updated + deleted + added) > 0:
                self.set_response(
                    resp, responses.SuccessResponse(
                        data={
                            'updated': updated,
                            'deleted': deleted,
                            'added': added,

                        },
                        scheme=schemes.ObjectScheme
                    )
                )
                return True
            else:
                self.set_response(
                    resp, responses.SuccessResponseJustOk())
                return True

        except IntegrityError as e:
            try:
                msg = str(e).split('\n')[1].split(':')[1]
            except:
                msg = str(e)
            self.set_response(resp, AlreadyExistsResponse(self.entity, msg))
            try:
                model.get_db().session.rollback()
            except Exception as e2:
                log.error('on patch many error {} rollback also failed {}'.format(e, e2))
            return False

        except OperationalError as e:
            self.set_response(resp, OperationalErrorResponse(e))
            return False

        except AlreadyExists as e:
            r = AlreadyExistsResponse(self.entity, str(e))
            r.status_code = 1000
            self.set_response(resp, r)

        except FkConstraintViolation as e:
            self.set_response(resp, responses.FkConstraintViolationResponse(
                data=errors.CommonErrors.get_fk_violation_error(e.table_name, e.column_name, e.value))
                              )
            self.model_class.session().rollback()
            return False

        except NoResultFound as e:
            self.set_response(resp, responses.ObjectNotFoundErrorResponse(data=str(e)))
            return None

        except Exception as e:
            log.error(e)
            self.set_response(resp, OperationErrorResponse(e))
            return False
