import json
import sys
import falcon
from urllib.parse import parse_qsl, urlencode
from datetime import date, datetime, timedelta
from pytz import UTC
from time import mktime
from dateutil.parser import parse as parse_datetime
from decimal import Decimal
from sqlalchemy.orm import aliased, make_transient, foreign
from sqlalchemy import or_, and_, select, inspect
from api_dnl import model
from api_dnl import settings
from falcon_rest.logger import log
from falcon_rest.responses import responses
from falcon_rest.responses import errors
from falcon_rest import swagger, schemes
from falcon_rest.helpers import get_request_ip
from api_dnl.resources import DnlResource, DnlList, DnlCreate, CustomAction, CustomPostAction, CustomPatchAction, \
	ValidationError, \
	client_context, check_context, SuccessResponseJustOk, generate_uuid_str
from api_dnl.schemes.client import PaymentGatewayHistoryScheme, PaymentGatewayHistoryGetScheme, PaymentCheckoutScheme
from api_dnl.view import NoResultFound, DEFAULT_SECURITY, RateList, DidBillingRelCreate, IngressTrunkCreate, \
	IngressTrunkResource, EgressTrunkCreate, EgressTrunkResource, \
	TrunkAssignProduct, ResourcePrefixCreate, ResourcePrefixResource, ResourcePrefixList, OperationErrorResponse, \
	DidBillingPlanList

import paypalrestsdk
import stripe


class PaypalWebhook(CustomPostAction):
	scheme_class = PaymentGatewayHistoryScheme
	model_class = model.PaymentGatewayHistory
	entity = 'PaymentGateway'
	path_parameters = ()
	security = (DEFAULT_SECURITY)
	restrict = ()

	no_auth_needed = True

	def on_post(self, req, resp, **kwargs):
		return self.proceed(req, resp, **kwargs)

	def proceed(self, req, resp, **kwargs):
		req_ip = get_request_ip(req)
		pgh = model.PaymentGatewayHistory(transaction_src=req.data, from_ip=req_ip, type='paypal')
		try:
			conf = model.SystemParameter.get(1)
			settings_paypal = dict(
				client_id=conf.paypal_account,
				client_secret=conf.paypal_skey,
				mode='sandbox' if conf.paypal_test_mode else 'live'
			)
			log.debug('client_id {} skey {} mode {}'.format(
				settings_paypal['client_id'][0:8],
				settings_paypal['client_secret'][0:8],
				settings_paypal['mode']
			))
			paypalrestsdk.configure(settings_paypal)
		except Exception as e:
			pgh.status = 'error'
			pgh.response = 'bad paypal configuration:{}'.format(e)
			pgh.save()
			self.set_response(resp, OperationErrorResponse(e))
			return False
		try:
			log.debug('paypal webhook called request data {} kwargs {}'.format(req.data, kwargs))
			data = req.data
			if 'event_type' in data:
				pgh.transaction_type = data['event_type']
				if data["event_type"] == 'PAYMENT.SALE.COMPLETED':
					create_time = data['create_time']
					pay_id = data['resource']['parent_payment']
					old = model.PaymentGatewayHistory.filter(and_(
						model.PaymentGatewayHistory.transaction_id == pay_id,
						model.PaymentGatewayHistory.status == 'success'
					)).first()
					if old:
						raise Exception('Duplicate transaction {pay_id}. Already processed')
					if not pay_id:
						raise Exception('Transaction id not found')
					pgh.transaction_id = pay_id
					pay = paypalrestsdk.Payment.find(pay_id)
					log.debug('pay {} retrieved'.format(pay_id))
					log.debug('content of pay {}'.format(pay))
					# pgh.transaction_src = pay
					if 'transactions' in pay and pay['transactions']:
						tr = pay['transactions'][0]
						try:
							pgh.fee = Decimal(data['resource']['transaction_fee']['value'])
						except:
							pass
						if 'amount' in tr and 'total' in tr['amount']:
							pgh.chargetotal = tr['amount']['total']
						items = tr['item_list']['items']
						invoice_number = None
						invoice = None
						client = None
						payment_type = 'Prepayment Received'
						if len(items) != 1:
							raise Exception('allowed only one payment item')
						for item in items:
							if 'invoice' in item['name'].lower():
								pgh.charge_amount = Decimal(item['price'])
								pgh.invoice_id = item['sku']
								invoice_number = str(item['sku'])
								invoice = model.Invoice.filter(
									model.Invoice.invoice_number == str(pgh.invoice_id)).first()
								if not invoice:
									raise Exception('wrong invoice number {}'.format(pgh.invoice_id))
								if item['quantity'] != 1:
									raise Exception('wrong invoice quantity - must be 1')
								pgh.client_id = invoice.client_id
								client = model.Client.get(pgh.client_id)
								payment_type = 'invoice payment received'
							elif 'prepay' in item['name'].lower():
								pgh.charge_amount = Decimal(item['price'])
								client_id = None
								try:
									client_id = int(item['sku'])
								except:
									raise Exception('wrong client_id - must be in "sku" field')
								client = model.Client.get(client_id)
								if not client:
									raise Exception('wrong client_id - must be in "sku" field')
								else:
									pgh.client_id = client_id
							else:
								raise Exception('wrong item name. must be "invoice" or "prepay"')
							if not client:
								raise Exception('wrong client id {}'.format(pgh.client_id))
							if conf.charge_type == 'create actual received amount':
								#payment_type = 'prepay payment received'
								pass
							if conf.charge_type == 'credit total amount':
								pgh.charge_amount = Decimal(pgh.charge_amount) - Decimal(pgh.fee)
							pay = model.ClientPayment(
								client_id=client.client_id, payment_type=payment_type,
								payment_time=create_time,
								receiving_time=datetime.now(UTC), invoice_number=invoice_number,
								amount=pgh.charge_amount, payment_method='paypal',
								description='paypal transaction {}'.format(pay_id)
							)
							payment_id = pay.save()
							pgh.par_id = payment_id
							pgh.response = 'ok'
							pgh.status = 'success'
							pgh.response = str(datetime.now(UTC))
							pgh.return_code = '200'
							pgh.save()
							ret = None
							to = None
							if conf.notify_carrier and not client.payment_received_notice:
								if client.billing_email:
									to = client.billing_email
								else:
									to = client.email
								if conf.notify_carrier_cc:
									to = '{};{}'.format(to, conf.notify_carrier_cc)
							if conf.daily_payment_confirmation and conf.daily_payment_email:
								if to:
									to = '{};{}'.format(to, conf.daily_payment_email)
								else:
									to = conf.daily_payment_email
							if to:
								ret = pay.apply_mail('payment_received', to)
							if ret:
								pgh.error = 'ok, but email notification not sent: {}'.format(str(ret))
								pgh.response = 'ok'
							pgh.response = str(datetime.now(UTC))
							pgh.return_code = '200'
							pgh.save()
					else:
						pgh.error = 'paypal transaction error: empty transaction'
						pgh.status = 'error'
						log.debug('pay {}'.format(pay))
						pgh.response = str(datetime.now(UTC))
						pgh.return_code = '406'
						pgh.save()
						self.set_response(resp, OperationErrorResponse(pgh.error))
						return False
				else:
					log.debug('---event {}'.format(data["event_type"]))
			else:
				pgh.error = 'paypal transaction error: wrong paypal event'
				pgh.status = 'error'
				pgh.response = str(datetime.now(UTC))
				pgh.return_code = '404'
				pgh.save()
				self.set_response(resp, responses.ObjectNotFoundErrorResponse())
				return False
		except Exception as e:
			import traceback
			log.debug(traceback.format_exc())
			try:
				pgh.error = 'paypal transaction error:{}'.format(str(e))
				pgh.status = 'error'
				pgh.response = str(datetime.now(UTC))
				pgh.return_code = '406'
				pgh.save()
				self.set_response(resp, OperationErrorResponse(e))
				return False
			except Exception as e1:
				from traceback import format_exc
				log.debug('paypal accept failure:{}'.format(format_exc()))
				pgh.session().rollback()
				self.set_response(resp, OperationErrorResponse(e1))
				return False
		return True


class StripeWebhook(CustomPostAction):
	scheme_class = PaymentGatewayHistoryScheme
	model_class = model.PaymentGatewayHistory
	entity = 'PaymentGateway'
	path_parameters = ()
	security = (DEFAULT_SECURITY)
	restrict = ()

	no_auth_needed = True

	def on_post(self, req, resp, **kwargs):
		return self.proceed(req, resp, **kwargs)

	def proceed(self, req, resp, **kwargs):
		req_ip = get_request_ip(req)
		pgh = model.PaymentGatewayHistory(transaction_src=req.data, from_ip=req_ip, type='stripe')
		# user = self.get_user(req)
		# pgh.client_id = user.client_id
		try:
			conf = model.SystemParameter.get(1)
			stripe.api_key = conf.stripe_account
			log.debug('stripe webhook called request data {} kwargs {}'.format(req.data, kwargs))
		except Exception as e:
			pgh.error = 'bad stripe configuration:{}'.format(e)
			pgh.save()
			self.set_response(resp, OperationErrorResponse(e))
			return False
		data = req.data
		if "type" in data:
			pgh.transaction_type = data["type"]
			if data["type"] in ('charge.succeeded', 'payment_intent.created'):
				try:
					charge_id = data['data']['object']['id']
					if 'charges' in data['data']['object']:
						if data['data']['object']['charges']['data']:
							charge_id = data['data']['object']['charges']['data'][0]['id']
					old = model.PaymentGatewayHistory.filter(and_(
						model.PaymentGatewayHistory.transaction_id == charge_id,
						model.PaymentGatewayHistory.status == 'success'
					)).first()
					if old:
						raise Exception('Duplicate transaction {charge_id}. Already processed')
					if not charge_id:
						raise Exception('Transaction id not found')
					charge = stripe.Charge.retrieve(charge_id)
					log.debug('charge {} retrieved'.format(charge_id))
					pgh.transaction_id = charge_id
					pgh.transaction_src = charge
					create_time = datetime.utcfromtimestamp(charge['created'])
					description = charge['statement_descriptor'] or ''
					li = description.split(':')
					invoice_number = None
					payment_type = 'Prepayment Received'
					if len(li) == 2:
						name = li[0]
						_id = int(li[1])
					else:
						raise Exception(
							'wrong payment statement_descriptor must be "invoice:<invoice_number>" or "Top up credit:<client_id>"')
					if name not in ('invoice', 'prepay', 'Top up credit'):
						raise Exception(
							'wrong payment statement_descriptor must be "invoice:<invoice_number>" or "Top up credit:<client_id>"')
					log.debug('stripe  {} {}'.format(name, _id))
					if 'invoice' in name.lower():
						invoice = model.Invoice.filter(model.Invoice.invoice_number == str(_id)).first()
						if not invoice:
							raise Exception(
								'wrong invoice number {}'.format(_id))
						invoice_number = str(_id)
						client_id = invoice.client_id
						payment_type = 'invoice payment received'
					else:
						client_id = _id
					client = model.Client.get(client_id)
					if not client:
						raise Exception('wrong client_id {}'.format(_id))
					pgh.client_id = client_id
					amount = charge['amount'] / 100
					pgh.charge_amount = Decimal(amount)
					pgh.fee = 0
					pgh.paypal_id = charge['source']['customer'] if charge['source'] else None
					# if conf.stripe_charge_type == 'create actual received amount':
					# 	payment_type = 'Prepayment Received'
					if conf.stripe_charge_type == 'credit total amount':
						pgh.fee = pgh.charge_amount * Decimal(0.029) + Decimal(0.3)
						pgh.charge_amount = pgh.charge_amount - pgh.fee
					pay = model.ClientPayment(
						client_id=client_id, payment_type=payment_type,
						payment_time=create_time,
						receiving_time=datetime.now(UTC), invoice_number=invoice_number,
						amount=pgh.charge_amount, payment_method='stripe',
						description='stripe transaction {}'.format(charge_id)
					)
					payment_id = pay.save()
					pgh.par_id = payment_id
					pgh.response = 'ok'
					pgh.status = 'success'
					pgh.save()
					ret = None
					to = None
					if conf.notify_carrier and not client.payment_received_notice:
						if client.billing_email:
							to = client.billing_email
						else:
							to = client.email
						if conf.notify_carrier_cc:
							to = '{};{}'.format(to, conf.notify_carrier_cc)
					if conf.daily_payment_confirmation and conf.daily_payment_email:
						if to:
							to = '{};{}'.format(to, conf.daily_payment_email)
						else:
							to = conf.daily_payment_email
					if to:
						ret = pay.apply_mail('payment_received', to)
					if ret:
						pgh.error = 'ok, but email notification not sent: {}'.format(str(ret))
					pgh.response = str(datetime.now(UTC))
					pgh.return_code = '200'
					pgh.save()
				except Exception as e:
					pgh.error = 'stripe transaction error:{}'.format(str(e))
					pgh.response = str(datetime.now(UTC))
					pgh.return_code = '406'
					pgh.save()
					self.set_response(resp, OperationErrorResponse(e))
					return False
			else:
				log.debug('---event {}'.format(data["type"]))
				pgh.error = 'stripe transaction error: wrong stripe event'
				pgh.response = str(datetime.now(UTC))
				pgh.return_code = '404'
				pgh.save()
				self.set_response(resp, responses.ObjectNotFoundErrorResponse())
		else:
			pgh.error = 'stripe transaction error: wrong stripe data'
			pgh.response = str(datetime.now(UTC))
			pgh.return_code = '404'
			pgh.save()
			self.set_response(resp, responses.ObjectNotFoundErrorResponse())
			return False
		return True

class StripeCheckout(CustomPostAction):
	scheme_class = PaymentCheckoutScheme
	body_parameters = ('User data', scheme_class)

	def proceed(self, req, resp, **kwargs):
		req_ip = get_request_ip(req)
		# pgh = model.PaymentGatewayHistory(transaction_src=req.data, from_ip=req_ip, type='stripe')

		user = self.get_user(self.req)
		# pgh.client_id = user.client_id
		try:
			conf = model.SystemParameter.get(1)
			stripe.api_key = conf.stripe_account
			log.debug('stripe checkout called request data {} kwargs {}'.format(req.data, kwargs))
			data = req.data
			#pgh.charge_amount = data['amount']
			base_url = data.pop('base_url', None)
			base_url = base_url or conf.base_url
			invoice_id = data.pop('invoice_id', None)
			statement_descriptor = 'Top up credit:{}'.format(user.client_id)
			if invoice_id:
				invoice = model.Invoice.get(invoice_id)
				if not invoice or invoice.client_id != user.client_id:
					raise ValidationError({'invoice_id':
							['invoice {} not found for client {}'.format(invoice_id, user.client_id)]})
				statement_descriptor = 'invoice:{}'.format(invoice.invoice_number)
			session = stripe.checkout.Session.create(
				# customer_email=user.email.split(';')[0],
				# customer='cus_{}'.format(user.user_id),
				payment_method_types=['card'],
				line_items=[{
					'price_data': {
						'currency': 'usd',
						'product_data': {
							'name': 'Voice Services',
						},
						'unit_amount': int(round(data['amount'] * 100, 0)),
					},
					'quantity': 1,
				}],
				metadata={'user_id': user.user_id},
				payment_intent_data={'statement_descriptor':statement_descriptor},
				mode='payment',
				success_url='{}/#/clients/billing/payment'.format(base_url),
				cancel_url='{}/#/clients/billing/online_payment'.format(base_url),
			)

			# pgh.save()
			resp.status = falcon.HTTP_200
			resp.body = json.dumps({"sessionId": session.id, "success": True})
			return False
		except Exception as e:
			# pgh.save()
			resp.status = falcon.HTTP_400
			resp.body = json.dumps({"success": False, "reply": str(e)})
			return False