import json
import sys,os
from urllib.parse import parse_qsl,urlencode
from datetime import date, datetime, timedelta
import traceback
from pytz import UTC
import mimetypes
from time import mktime
from dateutil.parser import parse as parse_datetime
import falcon
from sqlalchemy.orm import aliased,make_transient,foreign
from sqlalchemy import or_, and_,select,inspect
from api_dnl import model,settings
from falcon_rest.logger import log
from falcon_rest.responses import responses
from falcon_rest.responses import errors
from falcon_rest.resources.resources import swagger, ResourcesBaseClass, DEFAULT_SECURITY, ATTRIBUTE_ERROR_RE

from api_dnl.resources import DnlResource, DnlList, DnlCreate,CustomAction,CustomPatchAction,ValidationError,client_context,check_context
from api_dnl.schemes.testcall import CallingCallGetScheme,CallingRequestGetScheme,CallingRequestModifyScheme,\
    CallingRequestScheme,CallingTrackScheme,CallingTrackGetScheme
from api_dnl.view import NoResultFound,DEFAULT_SECURITY,OperationErrorResponse,generate_uuid_str


class CallingTrackCreate(DnlCreate):
    model_class = model.CallingTrack
    scheme_class = CallingTrackScheme
    scheme_class_get = CallingTrackGetScheme
    entity = 'Media track file'
    id_field = 'uuid'
    path_parameters = ()
    security = (DEFAULT_SECURITY)
    restrict = ()
    body_params = ()

    def get_spec(self):
        additional_responses = (
                                   responses.OperationErrorResponse(data=errors.CommonErrors.PermissionError,
                                                                    description='Can\'t create file'),
                               ) + self.additional_responses

        ext_body_parameters = self.body_params or (

            {
                'name': 'file',
                'in': 'formData',
                'description': 'File to upload',
                'required': True,
                'type': 'file'
            },
        )

        return swagger.specify.get_spec(
            method='post', description='Creates new {}'.format(self.entity.lower()),
            consumes=['multipart/form-data'],
            path_parameters=self.path_parameters,
            responses=(
                          self.scheme_class.get_object_created_response(entity=self.entity),#,scheme=ObjectCreatedWithErrorsScheme),
                      ) + additional_responses + self.get_additional_responses(method='post'),
            security=self.get_security(method='post'),
            #body_parameters=('{} to create'.format(self.entity), self.scheme_class),
            ext_body_parameters=ext_body_parameters
        )

    def before_create(self, obj, **kwargs):
        try:

            obj.uuid = generate_uuid_str()()
            obj.create_by = self.get_user().name
            obj.create_on = datetime.now(UTC)
            file = self.req.files['file']
            obj.orig_file=file.filename
            b=file.read()
            fsz=len(b)
            log.info('Uploaded file size:{}'.format(fsz))
            flimit = 200*1024*1024
            if ( fsz > flimit):
                raise ValidationError('File too big :{} , limit is {}'.format(fsz,flimit))
            local_file = open(obj.file_name, 'wb')
            local_file.write(b)
            #shutil.copyfileobj(file, local_file)
            local_file.close()
        except ValidationError as e:
            raise e
        except Exception as e:
            log.error('Import failure: {}'.format(str(e)))
            raise ValidationError('Import failure: {}'.format(traceback.format_exc()))
        return obj
    


class CallingTrackResource(DnlResource):
    model_class = model.CallingTrack
    scheme_class = CallingTrackScheme
    scheme_class_get = CallingTrackScheme
    entity = 'Export file'
    id_field = 'uuid'
    has_modify_operation = False
    has_delete_operation = True
    security = (DEFAULT_SECURITY)
    restrict = ()

    def on_delete(self, req, resp, **kwargs):
        obj=self.get_object(resp,self.model_class,**kwargs)
        file=None
        if obj and obj.file_name:
            file=obj.file_name
        ret=super(CallingTrackResource,self).on_delete(req, resp, **kwargs)
        if file:
            try:
                os.unlink(file)
            except:
                pass
        return ret


    def get_spec_info(self):
        spec=super(CallingTrackResource,self).get_spec_info()
        spec['get']['produces']=['audio/mpeg','audio/x-wav']
        return spec
    def on_get(self, req, resp, **kwargs):
        if not self.has_info_operation:  # pragma: no cover
            return self.method_not_allowed_response(resp, 'GET')
        try:
            obj = self.get_object(resp, self.model_class, **kwargs)
            if not obj:
                self.set_response(
                    resp, responses.ObjectNotFoundErrorResponse()
                )
                return
            if not True: #not obj.is_public:
                if not check_permission(self, req, 'show', obj):
                    self.set_response(
                        resp, responses.ForbiddenErrorResponse(data=errors.AuthErrors.Forbidden)
                    )
                    return
            if obj and obj.orig_file:
                resp.data = obj.file_data
                resp.content_type = mimetypes.guess_type(obj.orig_file)[0]
                resp.status = falcon.HTTP_200
                return
        except Exception as e:
            self.set_response(resp, OperationErrorResponse(e))
            #self.set_response(resp, responses.SuccessResponseObjectInfo(data=data))

class CallingRequestCreate(DnlCreate):
    scheme_class = CallingRequestScheme
    entity = 'CallingRequest'
    unique_field = 'req_uuid'

    additional_responses = ()
    path_parameters = ()
    security = (DEFAULT_SECURITY)
    restrict = ()

    def get_loaded_data(self, scheme):
        #return scheme.load(self.req_data).data
        return scheme.load(dict(status='NEW')).data

    def before_create(self, obj, **kwargs):
        obj.update_by = self.get_user().name
        obj.update_at = datetime.now(UTC)
        if len(self.req.access_route):
            obj.req_from_ip = self.req.access_route[0]
        calling_list = self.req_data.pop('calling_list',[])
        track_uuid = self.req_data.pop('track_uuid',None)
        caller_id = self.req_data.pop('caller_id',None)
        for dst in calling_list:
            call = model.CallingCall(dst_number=dst,
                                   track_uuid=track_uuid,
                                   state='WAITING', caller_id=caller_id)
            call.queue = model.CallingQueue()
            obj.calls.append(call)
        obj.total_call_cnt = len(calling_list)
        return obj


class CallingRequestResource(DnlResource):
    model_class = model.CallingRequest
    scheme_class = CallingRequestModifyScheme
    scheme_class_get = CallingRequestGetScheme
    scheme_class_modify = CallingRequestModifyScheme
    body_params = ()
    entity = 'CallingRequest'
    id_field = 'req_uuid'
    #has_delete_operation = False
    #has_modify_operation = False
    #has_update_by = False
    security = (DEFAULT_SECURITY)
    #path_parameters = ()
    restrict = ()
    #def get_path_parameters(self):
    #    return ()

    def before_update(self,obj,req):
        for call in obj.calls:
            call.state = 'WAITING'
            call.responce_code = None
            call.result = None
            call.rbt_audio_path = None
            call.call_audio_path = None
        obj.total_call_cnt = len(obj.calls)
        obj.complete_call_cnt = 0
        obj.status = 'NEW'
        for call in obj.calls:
            model.CallingQueue.put(obj.req_uuid, call.call_id)
        return obj

class CallingRequestList(DnlList):
    scheme_class = CallingRequestGetScheme
    model_class = model.CallingRequest
    entity_plural = 'CallingRequests'
    security = (DEFAULT_SECURITY)
    restrict = ()
    path_parameters = ()
    
class CallingCallResource(DnlResource):
    model_class = model.CallingCall
    scheme_class = CallingCallGetScheme
    scheme_class_get = CallingCallGetScheme
    entity = 'Export call file'
    id_field = 'call_id'
    has_modify_operation = False
    has_delete_operation = False
    security = (DEFAULT_SECURITY)
    restrict = ()
    path_parameters = ( {'name':'call_id','description':'Test id to call'},{'name':'file','description':'file to download'},)

    def on_delete(self, req, resp, **kwargs):
        obj=self.get_object(resp,self.model_class,**kwargs)
        file=None
        if obj and obj.file_name:
            file=obj.file_name
        ret=super(CallingCallResource,self).on_delete(req, resp, **kwargs)
        if file:
            try:
                os.unlink(file)
            except:
                pass
        return ret


    def get_spec_info(self):
        spec=super(CallingCallResource,self).get_spec_info()
        spec['get']['produces']=['audio/mpeg','audio/x-wav']
        return spec
    def on_get(self, req, resp, **kwargs):
        if not self.has_info_operation:  # pragma: no cover
            return self.method_not_allowed_response(resp, 'GET')
        try:
            obj = self.get_object(resp, self.model_class, **kwargs)
            if not obj:
                self.set_response(
                    resp, responses.ObjectNotFoundErrorResponse()
                )
                return
            if not True: #not obj.is_public:
                if not check_permission(self, req, 'show', obj):
                    self.set_response(
                        resp, responses.ForbiddenErrorResponse(data=errors.AuthErrors.Forbidden)
                    )
                    return
            file_name = kwargs.get('file',None)
            log.debug('call download {}'.format(file_name))
            if obj and file_name:
                if 'pcap' in file_name:
                    name = '{}.pcap'.format(obj.call_id)
                elif 'call' in file_name:
                    name = '{}_call.wav'.format(obj.call_id)
                elif 'rbt' in file_name:
                    name = '{}_rbt.wav'.format(obj.call_id)
                else:
                    self.set_response(resp, responses.ObjectNotFoundErrorResponse())
                    return

                full_name = "{}/{}/{}".format(settings.FILES['call_uploads'], obj.req_uuid, name)
                log.debug('call download {}'.format(full_name))
                if not os.path.exists(full_name):
                    self.set_response(resp, responses.ObjectNotFoundErrorResponse())
                    return
                f = open(full_name,'rb')
                resp.data = f.read()
                resp.content_type = mimetypes.guess_type(name)[0]
                resp.status = falcon.HTTP_200
                return
            else:
                self.set_response(resp, responses.ObjectNotFoundErrorResponse())
                return
        except Exception as e:
            self.set_response(resp, OperationErrorResponse(e))
            #self.set_response(resp, responses.SuccessResponseObjectInfo(data=data))

class CallingTrackList(DnlList):
    scheme_class = CallingTrackGetScheme
    model_class = model.CallingTrack
    entity_plural = 'CallingTracks'
    security = (DEFAULT_SECURITY)
    restrict = ()
    path_parameters = ()
