from api_dnl.__version__ import __version__, __date__
from api_dnl import settings
from reportlab.lib.pagesizes import letter, A4, landscape
from reportlab.lib import colors
from reportlab.lib.styles import getSampleStyleSheet, ParagraphStyle
from reportlab.lib.units import inch, mm
from reportlab.platypus.flowables import KeepTogether
from reportlab.platypus import SimpleDocTemplate, Paragraph, Spacer, Image, Table, KeepTogether, \
    Frame, BaseDocTemplate, PageTemplate, FrameBreak, PageBreak, TableStyle, NextPageTemplate
from reportlab.lib.enums import TA_LEFT, TA_CENTER, TA_RIGHT, TA_JUSTIFY
import os
import html, re, base64
from datetime import datetime, date
from html.parser import HTMLParser
from logging import getLogger

log = getLogger()

TABLEBACK = '#eaeaea'
HEADERBACK = '#b5b5b5'
HEADERFORE = '#ffffff'
ROWBACK = '#c5c5c5'


def KTable(data, **kwargs):
    return KeepTogether(Table(data, **kwargs))


def cleanhtml(raw_html):
    cleanr = re.compile('<.*?>')
    raw_html = raw_html.replace('\r', '').replace('\n', '').replace('\t', '')
    cleantext = re.sub(cleanr, '', raw_html)
    return cleantext


def clean_quoted_html(raw_html):
    cleanr = re.compile('&lt;.*?&gt;')
    raw_html = raw_html.replace('&amp;', '').replace('&nbsp;', '').replace('nbsp;', '')
    cleantext = re.sub(cleanr, '', raw_html)
    return cleantext


def html_to_rl(html_src, styleSheet):
    elements = list()

    class Handler(HTMLParser):
        mode = ""
        buffer = ""
        listcounter = 0
        listtype = ""
        color = ""

        def handle_starttag(self, name, attrs):
            if name in ["strong", "em", "i", "b"]:
                self.mode = name
            elif name == "ol":
                self.listcounter = 1
                self.listtype = "ol"
            elif name == "ul":
                self.listtype = "ul"
            elif name == "hr":
                elements.append(PageBreak())
            # elif name == "span":
            #     self.mode = name
            #     attrs = dict(attrs)
            #     if 'style' in attrs:
            #         style = attrs['style'].split(':')
            #         if len(style) == 2:
            #             if style[1]=='color':
            #                 self.color = style[1]

        def handle_endtag(self, name):
            if name.startswith("h") and name[-1] in ["1", "2", "3", "4", "5", "6"]:
                elements.append(Paragraph(self.buffer, styleSheet["Heading%s" % name[-1]]))
            elif name in ["strong", "em", "i", "b", "span"]:
                self.mode = ""
            elif name == "p":
                # elements.append(Paragraph(self.buffer+"<br/>", styleSheet["BodyText"]))
                elements.append(Paragraph(self.buffer, styleSheet["InfoText"]))
            elif name == "br":
                elements.append(Paragraph(self.buffer, styleSheet["InfoText"]))
            elif name == "li":
                if self.listtype == "ul":
                    elements.append(Paragraph(self.buffer, styleSheet["InfoText"], bulletText=u"•"))
                else:
                    elements.append(Paragraph(self.buffer, styleSheet["InfoText"], bulletText="%s." % self.listcounter))
                    self.listcounter += 1
            elif name in ["ol", "ul"]:
                self.listcounter = 0
                self.listtype = ""

            if name in ["h1", "h2", "h3", "h4", "h5", "h6", "p", "li", "br"]:
                self.buffer = ""

        def handle_data(self, chars):
            surrounding = None
            if self.mode in ["span"]:
                if self.color:
                    surrounding = "font color={}".format(self.color)
            elif self.mode in ["strong", "em", "i", "b"]:
                if self.mode in ["strong", "b"]:
                    surrounding = "b"
                else:
                    surrounding = "i"

            if surrounding:
                chars = "<%s>%s</%s>" % (surrounding, chars, surrounding)

            self.buffer += chars

    if '&lt;' in html_src:
        html_src = html.unescape(html_src)
        html_src = html_src.replace('<br>', '<br/>').replace('<BR>', '<br/>').replace('</BR>', '<br/>').replace('</br>',
                                                                                                                '<br/>')

    try:
        parseString = Handler()
        parseString.feed('<html>{}</html>'.format(html_src))
        parseString.close()
    except:
        elements = [Paragraph(html_src, styleSheet["InfoText"])]

    return elements


DECIMAL = 2


def zn(x):
    if x is None:
        return 0.0
    else:
        try:
            return float(x)
        except:
            return '(' + str(x) + ')'


def znr(x):
    fmt = '{:.' + str(DECIMAL) + 'f}'
    if x is None:
        return fmt.format(0.0)
    else:
        try:
            result = fmt.format(round(float(x), DECIMAL))
            sresult = result.split('.')
            if len(sresult) == 2 and len(sresult[1]) > 2:
                sresult[1] = sresult[1][:2]
                result = '.'.join(sresult)
            return result
        except:
            return '(' + str(x) + ')'


def zn2(x):
    if x is None:
        return '{:.2f}'.format(0.0)
    else:
        try:
            return '{:.2f}'.format(round(float(x), 2))
        except:
            return '(' + str(x) + ')'


def zni(x):
    if x is None:
        return 0
    else:
        try:
            return int(x)
        except:
            return '(' + str(x) + ')'


class BaseHtmlTemplate(BaseDocTemplate):
    def renderCell(self, tbl, cell, i, j):
        _content = ''
        _span = ''
        if tbl._spanCmds:
            for s, (si, sj), (ei, ej) in tbl._spanCmds:
                if si < 0: si = tbl._ncols + si
                if ei < 0: ei = tbl._ncols + ei
                if sj < 0: sj = tbl._nrows + sj
                if ej < 0: ej = tbl._nrows + ej
                if si == i and sj == j:
                    if si != ei:
                        _span = 'COLSPAN="{}"'.format(ei - si + 1)
                    if sj != ej:
                        _span = 'ROWSPAN="{}"'.format(ej - sj + 1)
                if si != ei and si < i and i <= ei and sj <= j and j <= ej:
                    return '<!-- colspan td -->'
                if sj != ej and si <= i and i <= ei and sj < j and j <= ej:
                    return '<!-- rowspan td -->'
        _bck = ''
        if hasattr(tbl, '_bkgrndcmds') and tbl._bkgrndcmds:
            for cmd, (si, sj), (ei, ej), arg in tbl._bkgrndcmds:
                if si < 0: si = tbl._ncols + si
                if ei < 0: ei = tbl._ncols + ei
                if sj < 0: sj = tbl._nrows + sj
                if ej < 0: ej = tbl._nrows + ej
                if si <= i and i <= ei and sj <= j and j <= ej:
                    if cmd == 'BACKGROUND':
                        _bck = 'bgcolor="{}"'.format(arg.hexval().replace('0x', '#'))

        _lns = ''
        if hasattr(tbl, '_linecmds') and tbl._linecmds:
            for op, (si, sj), (ei, ej), weight, color, cap, dash, join, count, space in tbl._linecmds:
                if si < 0: si = tbl._ncols + si
                if ei < 0: ei = tbl._ncols + ei
                if sj < 0: sj = tbl._nrows + sj
                if ej < 0: ej = tbl._nrows + ej
                if si <= i and i <= ei and sj <= j and j <= ej:
                    if op == 'LINEABOVE':
                        _lns = 'style="border-top: {}px solid {}"'.format(weight, color.hexval().replace('0x', '#'))
        _colwidth = ''
        if hasattr(tbl, '_colWidths') and tbl._colWidths:
            if not tbl._colWidths[i] is None:
                _colwidth = 'WIDTH="{}px"'.format(tbl._colWidths[i])
        style = None
        try:
            style = tbl._cellStyles[j][i]
        except:
            pass
        if style:
            _content += '<td align="{}" {} {} {} {}>'.format(style.alignment, _span, _bck, _lns, _colwidth)
        else:
            _content += '<td  {} {} {} {}>'.format(_span, _bck, _lns, _colwidth)
        _content += self.renderFlowable(cell)
        _content += '</td>'
        return _content

    def renderFlowable(self, f):
        if f is None:
            return ''
        if type(f) == type(""):
            return f.replace('font size=', 'font id=')
        if type(f) in [type(0), type(0.0)]:
            return str(f)
        if isinstance(f, datetime):
            return str(f)
        if isinstance(f, date):
            return str(f)
        if type(f) == type([]):
            _content = ''
            for item in f:
                _content += self.renderFlowable(item)
            return _content
        if isinstance(f, Paragraph):
            return "<p>{}</p>\n".format(self.renderFlowable(f.text))
        if isinstance(f, Spacer):
            return "<hr color='transparent' width='{}px' size='{}px'></hr>\n".format(f.width, f.height)
        if isinstance(f, Image):
            with open(f.filename, "rb") as image_file:
                encoded_string = base64.b64encode(image_file.read()).decode()
                return """<img src='data:image/png;base64,{}' \n width='{}' height='' alt='invoice logo'>""".format(
                    encoded_string, f.drawWidth, f.drawHeight)
            return '<img src="">'
        if isinstance(f, Table):
            _content = "<table width='95%' align='CENTER'>"
            j = 0
            for row in f._cellvalues:
                _content += '<tr>'
                i = 0
                for cell in row:
                    _content += self.renderCell(f, cell, i, j)
                    i += 1
                _content += '</tr>\n'
                j += 1
            _content += "</table>"
            return _content
        return '<div>unknown {}</div>\n'.format(f)

    def build(self, flowables):
        self.filename = self.filename.replace('.pdf', '.htm')
        file = open(self.filename, "wb")

        file.write("""<html><head>
        <meta content='creator' name='{}'>
        <meta content='subject' name='{}'>
        <title>{}</title>
        <style type='text/css'>
        body {{font-family: -apple-system,BlinkMacSystemFont,Segoe UI,Roboto,Oxygen,Ubuntu,Fira Sans,Droid Sans,Helvetica Neue,sans-serif;}}
        hr {{border-width:0}}
        </style>
        </head><body>
        """.format(self.creator, self.subject, self.title).encode('utf-8'))
        _content = ''
        for item in flowables:
            _content += self.renderFlowable(item)
        file.write(_content.encode('utf-8'))
        file.write(b'</body></html>')
        file.close()


class InvoicePdf(object):
    _decimal = 2

    def __init__(self, data, filename, is_jurisdict=False):
        global DECIMAL
        DECIMAL = data.config.invoice_decimal_digits
        if DECIMAL is None:
            DECIMAL = 2
        format = data.client.invoice_format
        if not format:
            format = 'PDF'
        # format = 'HTML'
        self.filename = filename
        self.data = data
        self.logo = data.logo
        self.doc = None
        self.story = []
        self.frame = None
        self.t_style = ParagraphStyle(
            name='Tables', leftIndent=40, rightIndent=40)
        style_sheet = getSampleStyleSheet()
        self.style_sheet = style_sheet
        self.style_info = self.style_sheet["BodyText"]
        self.style_info.fontSize = 9
        self.style_info.name = "InfoText"
        self.style_sheet.add(self.style_info)
        self.style = style_sheet["BodyText"]
        self.t_style = ParagraphStyle(
            name='Tables', leftIndent=40, rightIndent=40)

        self.pagesize = A4
        if format == 'HTML':
            self.doc = BaseHtmlTemplate(
                self.filename, pagesize=self.pagesize, rightMargin=10, leftMargin=10)
        else:
            self.doc = BaseDocTemplate(
                self.filename, pagesize=self.pagesize, rightMargin=10, leftMargin=10)

        self.doc.title = 'Invoice # {} for {}'.format(self.data.invoice_number,
                                                      self.data.client.name if self.data.client else 'NO CLIENT')
        self.doc.creator = 'invoice_v5 ver. {} date {}. novvvster'.format(__version__, __date__)
        self.doc.subject = 'Billing period: {} '.format(self.data.billing_period)

        self.is_jurisdict = is_jurisdict
        self.table_style = [
            ('VALIGN', (0, 0), (-1, -1), 'TOP'),
            ('BACKGROUND', (0, 0), (-1, -1), colors.HexColor(TABLEBACK)),
            ('SPAN', (0, 0), (-1, 0)),
            ('LINEABOVE', (0, 1), (-1, 1), 0.5, colors.black),
            ('BACKGROUND', (0, 1), (-1, 1), colors.HexColor(HEADERBACK)),
            ('COLOR', (0, 1), (-1, 1), colors.HexColor(HEADERFORE)),
            # ('BACKGROUND', (0, 3), (-1, -1), colors.HexColor(ROWBACK)),
            ('INNERGRID', (0, 1), (-1, -1), 0.5, colors.white),
            ('ALIGN', (1, 2), (-1, -1), 'LEFT'),
            # ('SPAN', (0, -1), (1, -1)),
            ('BACKGROUND', (0, -1), (-1, -1), colors.HexColor(HEADERBACK)), ]

    def _add_page_number(self, canvas, doc):
        page_num = canvas.getPageNumber()
        text = "Page #%s" % page_num
        canvas.drawRightString(200 * mm, 20 * mm, text)
        # self.frame.drawBackground(canvas)

    @staticmethod
    def _get_total_by_column(table, column, column_type=float, round_digits=2):
        return round(
            sum([column_type(row[column]) for row in table]), round_digits)

    def p6(self, label):
        return self.p("<font size=6>{}</font>".format(label))

    def b8(self, label):
        return self.p("<b><font size=8>{}</font></b>".format(label))

    def p8(self, label):
        #return Paragraph(str(label),ParagraphStyle(name='p8',font_size=7,splitLongWords=0))
        return self.p("<font size=7>{}</font>".format(label))

    def b9(self, label):
        return self.p("<b><font size=9>{}</font></b>".format(label))

    def p9(self, label):
        return self.p("<font size=9>{}</font>".format(label))

    def b11(self, label):
        return self.p("<b><font size=11>{}</font></b>".format(label))

    def p11(self, label):
        return self.p("<font size=11>{}</font>".format(label))

    def p(self, content):
        # adding check in case the content is None
        if content == None:
            return Paragraph(" ", self.style)
        return Paragraph(content, self.style)

    def pt(self, content):
        # adding check in case the content is None
        if content == None:
            return Paragraph(" ", self.t_style)
        return Paragraph(content, self.t_style)

    def p_tpl(self):
        if self.data.pdf_tpl == None:
            return self.p(" ")
        return html_to_rl(self.data.pdf_tpl, self.style_sheet)

    def footer(self):
        p_name_1 = html_to_rl(self.data.company_info,
                              self.style_sheet) if self.data.company_info_location == 'bottom' else ""
        p_name_2 = html_to_rl(self.data.company_info,
                              self.style_sheet) if self.data.company_info_location == 'middle' else ""
        cl_d = self.data.json_content['client_data']
        p_bank = html_to_rl(cl_d['billing_info'] or '',
                            self.style_sheet) if cl_d['billing_info_location'] == 'bottom' else ""
        data = [
            [p_name_1, p_name_2, ""],
            [
                p_bank,
                "",
                ""
            ],
            ["", "", ""],
        ]
        return Table(
            data,
            style=[
                ('VALIGN', (0, 0), (-1, -1), 'TOP'),
                ('TOPPADDING', (0, 0), (-1, -1), 0),
                ('ALIGN', (0, 0), (-1, -1), 'LEFT'),
                ('BOTTOMPADDING', (0, 0), (-1, -1), 0),
            ])

    def header(self):
        def pl(label, data):
            return self.p("<para align=right><font size=9><b>{}|</b> {}</font></para>".format(label, data))
        def pr(label):
            return self.p("<para align=right>{}</para>".format(label))

        try:
            import PIL
            img = PIL.Image.open(self.logo)  # just try to identify
            if img.width>150:
                img.thumbnail((150, 150), PIL.Image.ANTIALIAS)
                img.save(self.logo)
        except:
            self.logo = settings.FILES['upload_to'] + '/logo.png'
            s = b'iVBORw0KGgoAAAANSUhEUgAAAAEAAAABCAYAAAAfFcSJAAAADUlEQVR42mP8z8BQDwAEhQGAhKmMIQAAAABJRU5ErkJggg=='
            with open(self.logo , 'wb') as f:
                f.write(base64.b64decode(s))
        logo = Image(self.logo)
        # logo.drawHeight = 0.5 * inch
        # logo.drawWidth = 1 * inch
        #logo.drawHeight = logo.drawHeight * inch / 150  # inch
        #logo.drawWidth = logo.drawWidth * inch / 150  # inch

        cl_d = self.data.json_content['client_data']
        client_data = [
            [pr('<b><font size = 9>Billed to </font></b>' + cl_d['company_name'])] ]
        if self.data.client_address:
            client_data=client_data+[[pr(cl_d['client_address'])]]
        client_data = client_data + [
            [pr('<b><font size = 9>Invoice# </font></b>' + str(cl_d['invoice_number']))],
            [pr('<b><font size = 9>Invoice </font>Date </b>' + str(cl_d['invoice_date'])[:10])],
            [pr('<b><font size = 9>Payment </font>Due Date </b>' + str(cl_d['payment_due_date']))],
            [pr('<b><font size = 9>Billing </font>Period </b>' + cl_d['billing_period'])],

        ]
        if self.data.client_invoice_zone:
            client_data = client_data +[[pr('<b><font size = 9>Invoice time zone </font></b>' + 'GMT '+cl_d['invoice_zone'])]]

        client_table = Table(
            client_data,
            style=[
                #('ALIGN', (0, 0), (0, -1), 'LEFT'),
                ('ALIGN', (0, 0), (0, -1), 'RIGHT'),
                ('VALIGN', (0, 0), (-1, -1), 'TOP'),
                ('TOPPADDING', (0, 0), (-1, -1), 0),
                ('BOTTOMPADDING', (0, 0), (-1, -1), 0),
            ]

        )
        p_name = html_to_rl(cl_d['company_info'] or '',
                            self.style_sheet) if cl_d['company_info_location'] == 'top' else ""
        p_bank = html_to_rl(cl_d['billing_info'] or '',
                            self.style_sheet) if cl_d['billing_info_location'] == 'top' else ""
        data = [
            [logo, ""],
            ["", ""],
            [p_name, client_table],
            ["", ""],
            [p_bank, ""],
        ]
        return Table(
            data,
            style=[
                ('ALIGN', (0, 1), (-1, -1), 'RIGHT'),
                ('VALIGN', (0, 0), (-1, -1), 'TOP'),
                ('TOPPADDING', (0, 0), (-1, -1), 0),
                ('BOTTOMPADDING', (0, 0), (-1, -1), 0),
            ]
            ,colWidths=[250,250]
        )

    def acc_summary(self, d):

        def row(label, txt):
            return [self.p9(label), self.p9(txt)]

        data = [
            [self.b11('Account Summary'), ''],
            [self.b9('Balance Information'), ''],
            # row('Previous Amount Due', d['past_due_amount']),
            row('Payments Received', d['payments_received']),
            row('Adjustments', d['adjustment']),
            # [self.b9('Past Due Balance'), self.b9(d['past_due_balance'])],
            [self.b9('Invoice Charges'), ''],
            row('Taxes and surcharges', d['tax']),
            row('Finance Charges', d['finance_charges']),
            row('Other Charges and Credits', d['other_charges']),
            row('Extra Charges', d['extra_charges']),
        ]
        if 'scc_charges' in self.data.json_content:
            data.append(row('Short Duration Surcharge', d['scc_charges']))
        if 'rec_charges' in self.data.json_content:
            data.append(row('Recurring Charges', d['rec_charges']))
        if 'non_rec_charges' in self.data.json_content:
            data.append(row('Non-Recurring Charges', d['non_rec_charges']))
        data = data + [
            row('Usage Charges', d['usage_charges']),
            [self.b9('Total Invoice Charges'), self.b9(d['total_charges'])],
            # [self.b9('Total Amount Due'), self.b9(d['total_amount'])],

        ]
        return Table(
            data,
            style=[
                ('ALIGN', (1, 1), (1, -1), 'LEFT'),
                ('BOTTOMPADDING', (0, 0), (1, 0), 4),
                ('TOPPADDING', (0, 1), (-1, -1), 0),
                ('BOTTOMPADDING', (0, 1), (-1, -1), 0),
                ('VALIGN', (0, 0), (-1, -1), 'TOP'),
                ('SPAN', (0, 0), (1, 0)),
                ('SPAN', (0, 1), (1, 1)),
                # ('SPAN', (0, 6), (1, 6)),
                ('LINEABOVE', (0, 1), (1, 1), 0.5, colors.black),
                ('BACKGROUND', (0, 0), (-1, -1), colors.HexColor(TABLEBACK)),
                ('BACKGROUND', (0, 1), (0, -1), colors.HexColor(HEADERBACK)),
                ('INNERGRID', (0, 1), (-1, -1), 0.5, colors.white),
            ])


    @property
    def lab_total(self):
        return self.p("<b><font size=9>TOTAL</font></b>")

    @property
    def lab_num_calls(self):
        return self.p("<b><font size=9># OF CALLS</font></b>")

    @property
    def lab_total_minutes(self):
        return self.p("<b><font size=9>TOTAL MINUTES</font></b>")

    @property
    def lab_charge(self):
        return self.p("<b><font size=9>CHARGE</font></b>")

    @property
    def lab_location(self):
        return self.p("<b><font size=9>LOCATION</font></b>")


    def payment_applied(self, d):
        data = [[self.b11('Summary of Payments'), ""],
                ["Payment ID", "Date", "Payment received ($)"]]

        for row in d['items']:
            if float(row['amount']):
                data.append([row['payment_id'], row['payment_time'] or '', '$' + znr(row['amount'])])

        if len(data) == 2:
            data.append([self.b11('No Payment is received during this billing period.'), ""])
            return KTable(
                data, style=[
                    ('VALIGN', (0, 0), (-1, -1), 'TOP'),
                    ('BACKGROUND', (0, 0), (-1, -1), colors.HexColor(TABLEBACK)),
                    ('SPAN', (0, 0), (-1, 0)),
                    ('SPAN', (0, -1), (-1, -1)),
                    ('LINEABOVE', (0, 1), (-1, 1), 0.5, colors.black),
                    ('LINEBELOW', (0, 1), (-1, 1), 0.5, colors.black),
                    ('BACKGROUND', (0, 1), (-1, 1), colors.HexColor(HEADERBACK)),
                    #('TEXTCOLOR', (0, 1), (-1, 1), colors.HexColor(HEADERFORE)),
                    # ('BACKGROUND', (0, 3), (-1, -1), colors.HexColor(ROWBACK)),
                    ('INNERGRID', (0, 1), (-1, 1), 0.5, colors.white),
                    ('ALIGN', (2, 2), (2, -1), 'LEFT'),
                ])
        else:
            data.append(['Total Payments', '', '$' + znr(d['total'])])
        return KTable(
            data, style=[
                ('VALIGN', (0, 0), (-1, -1), 'TOP'),
                ('BACKGROUND', (0, 0), (-1, -1), colors.HexColor(TABLEBACK)),
                ('SPAN', (0, 0), (-1, 0)),
                ('LINEABOVE', (0, 1), (-1, 1), 0.5, colors.black),
                ('BACKGROUND', (0, 1), (-1, 1), colors.HexColor(HEADERBACK)),
                #('TEXTCOLOR', (0, 1), (-1, 1), colors.HexColor(HEADERFORE)),
                # ('BACKGROUND', (0, 3), (-1, -1), colors.HexColor(ROWBACK)),
                ('INNERGRID', (0, 1), (-1, -1), 0.5, colors.white),
                ('ALIGN', (2, 2), (2, -1), 'LEFT'),
            ])

    def rec_charge(self, d):
        data = [[self.b11('Recurring Charge Details'), ""],
                ["Description", "Rate", "Quantity", "Amount ($)"]]

        for row in d['items']:
            if float(row['amount']):
                data.append([row['description'], znr(row['rate']), znr(row['quantity']), '$'+znr(row['amount'])])

        if len(data) == 2:
            data.append(['', '', ''])
        else:
            data.append(['Total Recurring Charges', '', '', '$'+znr(d['total'])])
        return KTable(
            data, style=[
                ('VALIGN', (0, 0), (-1, -1), 'TOP'),
                ('BACKGROUND', (0, 0), (-1, -1), colors.HexColor(TABLEBACK)),
                ('SPAN', (0, 0), (-1, 0)),
                ('LINEABOVE', (0, 1), (-1, 1), 0.5, colors.black),
                ('BACKGROUND', (0, 1), (-1, 1), colors.HexColor(HEADERBACK)),
                # ('BACKGROUND', (0, 3), (-1, -1), colors.HexColor(ROWBACK)),
                ('INNERGRID', (0, 1), (-1, -1), 0.5, colors.white),
                ('ALIGN', (1, 2), (3, -1), 'LEFT'),
            ])

    def non_rec_charge(self, d):
        data = [[self.b11('Non-Recurring Charge Details'), ""],
                ["Description", "Rate", "Quantity", "Amount ($)"]]

        for row in d['items']:
            if float(row['amount']):
                data.append([row['description'], znr(row['rate']), znr(row['quantity']), '$'+znr(row['amount'])])

        if len(data) == 2:
            data.append(['', '', ''])
        else:
            data.append(['Total Non-Recurring Charges', '', '', '$'+znr(d['total'])])
        return KTable(
            data, style=[
                ('VALIGN', (0, 0), (-1, -1), 'TOP'),
                ('BACKGROUND', (0, 0), (-1, -1), colors.HexColor(TABLEBACK)),
                ('SPAN', (0, 0), (-1, 0)),
                ('LINEABOVE', (0, 1), (-1, 1), 0.5, colors.black),
                ('BACKGROUND', (0, 1), (-1, 1), colors.HexColor(HEADERBACK)),
                # ('BACKGROUND', (0, 3), (-1, -1), colors.HexColor(ROWBACK)),
                ('INNERGRID', (0, 1), (-1, -1), 0.5, colors.white),
                ('ALIGN', (1, 2), (3, -1), 'LEFT'),
            ])

    def summary_of_charges(self, d):
        data = [[self.b11('Summary of charges'), ""],
                ["", "Total Calls", "Total Minutes", "Total Charges"]]

        for row in d['items']:
            data.append(['Minutes usage', zni(row['calls']), zni(row['minutes']), znr(row['amount'])])
        if len(data) == 2:
            data.append(['', '', ''])
        else:
            row = d['total']
            #data.append(['Total', zni(row['calls']), zni(row['minutes']), znr(row['amount'])])
            data.append(['Total', '', '', znr(row['amount'])])
        return KTable(
            data, style=[
                ('VALIGN', (0, 0), (-1, -1), 'TOP'),
                ('BACKGROUND', (0, 0), (-1, -1), colors.HexColor(TABLEBACK)),
                ('SPAN', (0, 0), (-1, 0)),
                ('LINEABOVE', (0, 1), (-1, 1), 0.5, colors.black),
                ('BACKGROUND', (0, 1), (-1, 1), colors.HexColor(HEADERBACK)),
                # ('BACKGROUND', (0, 3), (-1, -1), colors.HexColor(ROWBACK)),
                ('INNERGRID', (0, 1), (-1, -1), 0.5, colors.white),
                ('ALIGN', (1, 2), (3, -1), 'LEFT'),
            ])

    def usage_charge(self, d):
        data = [[self.b11('Usage Charge'), ""],
                ["Description", "Calls", "Minutes", "Amount ($)"]]

        return self._4_column_table('description', d, data)

    def trunk_detail(self, d):
        if d and d['items']:
            data = [[self.b11('Inbound Detail by Trunk (A-Z trunks)'), ""],
                    ["Trunk", "Calls", "Minutes", "Amount ($)"]]

            return self._4_column_table('trunk', d, data)
        else:
            return self.story.append(Spacer(self.doc.width, 30))

    def daily_summary(self, d):
        data = [[self.b11('Daily Summary'), ""],
                ["Date", "Calls", "Minutes", "Amount ($)"]]

        return self._4_column_table('date',d, data)

    def _jd_table(self,d,title,key):
        p8 = self.p8
        p6 = self.p6
        data = [[self.b11(title), ""],
                [key.capitalize(), "Interstate", "", "", "Intrastate", "", "", "IJ", "", "", "Total", "", ""],
                ["", "Calls", "Minute", "Cost ($)", "Calls", "Minute", "Cost ($)", "Calls", "Minute", "Cost ($)", "Calls",
                "Minute", "Cost ($)"]
                ]

        for row in d['items']:
            data.append([p6(row[key]),
                         p8(row['inter_calls']), p8(zni(row['inter_minutes'])), p8('$' + znr(row['inter_amount'])),
                         p8(row['intra_calls']), p8(zni(row['intra_minutes'])), p8('$' + znr(row['intra_amount'])),
                         p8(row['ij_calls']), p8(zni(row['ij_minutes'])), p8('$' + znr(row['ij_amount'])),
                         p8(row['total_calls']), p8(zni(row['total_minutes'])), p8('$' + znr(row['total_amount']))
                         ]
                        )

        if len(data) == 3:
            data.append(['', '', ''])
        else:
            row = d['total']
            data.append([p8('Total'),
                         p8(row['inter_calls']), p8(zni(row['inter_minutes'])), p8('$' + znr(row['inter_amount'])),
                         p8(row['intra_calls']), p8(zni(row['intra_minutes'])), p8('$' + znr(row['intra_amount'])),
                         p8(row['ij_calls']), p8(zni(row['ij_minutes'])), p8('$' + znr(row['ij_amount'])),
                         p8(row['total_calls']), p8(zni(row['total_minutes'])), p8('$' + znr(row['total_amount']))
                         ]
                        )
        return KTable(
            data, style=[
                ('VALIGN', (0, 0), (-1, -1), 'TOP'),
                ('BACKGROUND', (0, 0), (-1, -1), colors.HexColor(TABLEBACK)),
                ('SPAN', (0, 0), (-1, 0)),
                ('SPAN', (0, 1), (0, 2)),
                ('SPAN', (1, 1), (3, 1)), ('SPAN', (4, 1), (6, 1)), ('SPAN', (7, 1), (9, 1)),
                ('SPAN', (10, 1), (12, 1)),
                ('LINEABOVE', (0, 1), (-1, 1), 0.5, colors.black),
                ('BACKGROUND', (0, 1), (-1, 2), colors.HexColor(HEADERBACK)),
                # ('BACKGROUND', (0, 3), (-1, -1), colors.HexColor(ROWBACK)),
                ('INNERGRID', (0, 1), (-1, -1), 0.5, colors.white),
                ('ALIGN', (1, 2), (3, -1), 'LEFT'),
            ], colWidths=(46, 41, 41, 46, 41, 41, 46, 41, 41, 46, 41, 41, 46))

    def jd_trunk_detail(self, d):
        return self._jd_table(d,'Inbound Detail by Trunk (US trunks)','trunk')

    def jd_daily_summary(self, d):
        return self._jd_table(d, 'Daily Summary with Jurisdiction Breakdown', 'date')

    def prefix_summary(self, d):
        data = [[self.b11('Detail by Trunk Prefix'), ""],
                ["Trunk", "Prefix", "Calls", "Minutes", "Amount ($)"]]

        for row in d['items']:
            if float(row['amount']):
                data.append([row['trunk'], row['prefix'], zni(row['calls']), zni(row['minutes']),'$'+znr(row['amount'])])

        if len(data) == 2:
            data.append(['', '', ''])
        else:
            if 'total' in d:
                row = d['total']
                data.append(['Total Charge', '', zni(row['calls']), zni(row['minutes']), '$'+znr(row['amount'])])
        return KTable(
            data, style=[
                ('VALIGN', (0, 0), (-1, -1), 'TOP'),
                ('BACKGROUND', (0, 0), (-1, -1), colors.HexColor(TABLEBACK)),
                ('SPAN', (0, 0), (-1, 0)),
                ('LINEABOVE', (0, 1), (-1, 1), 0.5, colors.black),
                ('BACKGROUND', (0, 1), (-1, 1), colors.HexColor(HEADERBACK)),
                # ('BACKGROUND', (0, 3), (-1, -1), colors.HexColor(ROWBACK)),
                ('INNERGRID', (0, 1), (-1, -1), 0.5, colors.white),
                ('ALIGN', (1, 2), (4, -1), 'LEFT'),
            ])

    def code_name_summary(self, d):
        data = [[self.b11('Detail by Code Name'), ""],
                ["Country", "Code Name", "Calls", "Minutes", "Amount ($)"]]

        for row in d['items']:
            if float(row['amount']):
                data.append([row['country'], row['code_name'], zni(row['calls']), zni(row['minutes']), '$'+znr(row['amount'])])

        if len(data) == 2:
            data.append(['', '', ''])
        else:
            if 'total' in d:
                row = d['total']
                data.append(['Total Charge', '', zni(row['calls']), zni(row['minutes']), '$'+znr(row['amount'])])

        return KTable(
            data, style=[
                ('VALIGN', (0, 0), (-1, -1), 'TOP'),
                ('BACKGROUND', (0, 0), (-1, -1), colors.HexColor(TABLEBACK)),
                ('SPAN', (0, 0), (-1, 0)),
                ('LINEABOVE', (0, 1), (-1, 1), 0.5, colors.black),
                ('BACKGROUND', (0, 1), (-1, 1), colors.HexColor(HEADERBACK)),
                # ('BACKGROUND', (0, 3), (-1, -1), colors.HexColor(ROWBACK)),
                ('INNERGRID', (0, 1), (-1, -1), 0.5, colors.white),
                ('ALIGN', (1, 2), (4, -1), 'LEFT'),
            ])

    def country_summary(self, d):
        data = [[self.b11('Country Breakdown'), ""],
                ["Country", "Calls", "Minutes", "Amount ($)"]]
        return self._4_column_table('country', d, data)

    def top_100tn_summary(self, d):
        data = [[self.b11('Top 100 TNs'), ""],
                ["Number", "Calls", "Minutes", "Rate", "Amount ($)"]]

        for row in d['items']:
            if float(row['amount']):
                data.append([row['number'], zni(row['calls']), zni(row['minutes']), znr(row['rate']), '$'+znr(row['amount'])])

        if len(data) == 2:
            data.append(['', '', '', '', ''])
        else:
            row = d['total']
            data.append(['Total Charge',zni(row['calls']), zni(row['minutes']), znr(row['rate']), '$'+znr(row['amount'])])
        return KTable(
            data, style=[
                ('VALIGN', (0, 0), (-1, -1), 'TOP'),
                ('BACKGROUND', (0, 0), (-1, -1), colors.HexColor(TABLEBACK)),
                ('SPAN', (0, 0), (-1, 0)),
                ('LINEABOVE', (0, 1), (-1, 1), 0.5, colors.black),
                ('BACKGROUND', (0, 1), (-1, 1), colors.HexColor(HEADERBACK)),
                # ('BACKGROUND', (0, 3), (-1, -1), colors.HexColor(ROWBACK)),
                ('INNERGRID', (0, 1), (-1, -1), 0.5, colors.white),
                ('ALIGN', (1, 2), (3, -1), 'LEFT'),
            ])

    def inbound_summary(self, d):
        data = [[self.b11('Inbound Daily Summary'), ""],
                ["Date", "Calls", "Minutes", "Amount ($)"]]

        return self._4_column_table('date', d, data)

    def _4_column_table(self, key, d, data):
        for row in d['items']:
            if float(row['amount']):
                data.append([row[key], zni(row['calls']), zni(row['minutes']), '$'+znr(row['amount'])])
        if len(data) == 2:
            data.append(['', '', ''])
        else:
            row = d['total']
            data.append(['Total', zni(row['calls']), zni(row['minutes']),  '$'+znr(row['amount'])])
        return KTable(
            data, style=[
                ('VALIGN', (0, 0), (-1, -1), 'TOP'),
                ('BACKGROUND', (0, 0), (-1, -1), colors.HexColor(TABLEBACK)),
                ('SPAN', (0, 0), (-1, 0)),
                ('LINEABOVE', (0, 1), (-1, 1), 0.5, colors.black),
                ('BACKGROUND', (0, 1), (-1, 1), colors.HexColor(HEADERBACK)),
                # ('BACKGROUND', (0, 3), (-1, -1), colors.HexColor(ROWBACK)),
                ('INNERGRID', (0, 1), (-1, -1), 0.5, colors.white),
                ('ALIGN', (1, 2), (3, -1), 'LEFT'),
            ])

    def did_charge_summary(self, d):
        data = [[self.b11('DID Charge Summary'), "",""],
                ["Description", "Quantity", "Amount ($)"]]

        for row in d['items']:
            if float(row['charge']):
                data.append([str(row['name']).capitalize(), str(row['detail']), '$'+znr(row['charge'])])

        if len(data) == 2:
            data.append(['', '', ''])

        return KTable(
            data, style=[
                ('VALIGN', (0, 0), (-1, -1), 'TOP'),
                ('BACKGROUND', (0, 0), (-1, -1), colors.HexColor(TABLEBACK)),
                ('SPAN', (0, 0), (-1, 0)),
                ('LINEABOVE', (0, 1), (-1, 1), 0.5, colors.black),
                ('BACKGROUND', (0, 1), (-1, 1), colors.HexColor(HEADERBACK)),
                # ('BACKGROUND', (0, 3), (-1, -1), colors.HexColor(ROWBACK)),
                ('INNERGRID', (0, 1), (-1, -1), 0.5, colors.white),
                #('ALIGN', (1, 2), (3, -1), 'RIGHT'),
            ])

    def did_charge_breakdown(self, d):
        data = [[self.b11('DID Charge Breakdown'), ""],
                ["DID", "MRC", "NRC", "Minutes", "Attempts", "Call Charge", "Total"]]

        for row in d['items']:
            data.append([row['did'], znr(row['mrc']), znr(row['nrc']), zni(row['minutes']), zni(row['attempts']),
                         znr(row['call_charge']), znr(row['total'])])

        if len(data) == 2:
            data.append(['', '', ''])
        else:
            if 'total' in d:
                row = d['total']
                data.append(['Total Charge', znr(row['mrc']), znr(row['nrc']), zni(row['minutes']), zni(row['attempts']),
                             znr(row['call_charge']), znr(row['total'])])
        return KTable(
            data, style=[
                ('VALIGN', (0, 0), (-1, -1), 'TOP'),
                ('BACKGROUND', (0, 0), (-1, -1), colors.HexColor(TABLEBACK)),
                ('SPAN', (0, 0), (-1, 0)),
                ('LINEABOVE', (0, 1), (-1, 1), 0.5, colors.black),
                ('BACKGROUND', (0, 1), (-1, 1), colors.HexColor(HEADERBACK)),
                # ('BACKGROUND', (0, 3), (-1, -1), colors.HexColor(ROWBACK)),
                ('INNERGRID', (0, 1), (-1, -1), 0.5, colors.white),
                ('ALIGN', (1, 2), (3, -1), 'LEFT'),
            ])

    def port_usage_fee_per_port(self):
        data = [[self.b11('Port usage fee'), "", ""],
                ["Total Port Limit Per DID", "Fee Per Port", "Total"]
                ]
        for row in self.data.did_port_usage_info:
            data.append([zni(row.port_usage), znr(row.fee_per_port), znr(row.total_fee)])

        if len(data) == 2:
            data.append(['', '', ''])

        return Table(
            data, style=[
                ('VALIGN', (0, 0), (-1, -1), 'TOP'),
                ('BACKGROUND', (0, 0), (-1, -1), colors.HexColor(TABLEBACK)),
                ('SPAN', (0, 0), (-1, 0)),
                ('LINEABOVE', (0, 1), (-1, 1), 0.5, colors.black),
                ('BACKGROUND', (0, 1), (-1, 1), colors.HexColor(HEADERBACK)),
                # ('BACKGROUND', (0, 3), (-1, -1), colors.HexColor(ROWBACK)),
                ('INNERGRID', (0, 1), (-1, -1), 0.5, colors.white),
                ('ALIGN', (0, 2), (-1, -1), 'LEFT'),
                # ('SPAN', (0, -1), (1, -1)),
                ('BACKGROUND', (0, -1), (-1, -1), colors.HexColor(HEADERBACK)), ])

    def port_usage_fee_unlimited(self):
        data = [[self.b11('Port usage fee'), "", ""],
                ["DID Count", "Total Port Usage", "Fee Per Port", "Total"]
                ]
        for row in self.data.did_port_usage_info:
            data.append([zni(row.did_count), zni(row.port_usage), znr(row.fee_per_port),
                         znr(row.total_fee)])

        if len(data) == 2:
            data.append(['', '', '', ''])

        return Table(
            data, style=[
                ('VALIGN', (0, 0), (-1, -1), 'TOP'),
                ('BACKGROUND', (0, 0), (-1, -1), colors.HexColor(TABLEBACK)),
                ('SPAN', (0, 0), (-1, 0)),
                ('LINEABOVE', (0, 1), (-1, 1), 0.5, colors.black),
                ('BACKGROUND', (0, 1), (-1, 1), colors.HexColor(HEADERBACK)),
                # ('BACKGROUND', (0, 3), (-1, -1), colors.HexColor(ROWBACK)),
                ('INNERGRID', (0, 1), (-1, -1), 0.5, colors.white),
                ('ALIGN', (0, 2), (-1, -1), 'LEFT'),
                # ('SPAN', (0, -1), (1, -1)),
                ('BACKGROUND', (0, -1), (-1, -1), colors.HexColor(HEADERBACK)), ])

    def sms_summary(self):
        data = [[self.b11('Sms Daily Summary'), ""],
                ["Date", "Sent", "Received", "Cost"]
                ]
        for row in self.data.sms_summary:
            data.append([str(row.date)[:10], zni(row.sent), zni(row.received), znr(row.cost)])

        t = self.data.sms_summary

        data.append(
            ['', zni(sum(row.sent for row in t)), zni(sum(row.received for row in t)), znr(sum(row.cost for row in t))])
        return KTable(data, style=self.table_style)

    def scc_charges(self, d):
        data = [[self.b11('SCC Charges Summary'), "",""],
                ["Trunk", "Calls < {} sec".format(d['summary']['scc_below_sec']), "Total calls"]]
        if not d['summary']['scc_below_sec']:
            return None
        for row in d['items']:
            data.append([str(row['trunk_name']).capitalize(), zni(row['short_calls']), zni(row['calls'])])
        data.append(['Short Calls Rate', znr(d['summary']['scc_charge']),'' ])
        data.append(['SCC Charge', znr(d['summary']['scc_amount']),''])
        return KTable(
            data, style=[
                ('ALIGN', (1, 1), (-1, -1), 'LEFT'),
                ('VALIGN', (0, 0), (-1, -1), 'TOP'),
                ('BACKGROUND', (0, 0), (-1, -1), colors.HexColor(TABLEBACK)),
                ('SPAN', (0, 0), (-1, 0)),
                ('LINEABOVE', (0, 1), (-1, 1), 0.5, colors.black),
                ('BACKGROUND', (0, 1), (-1, 1), colors.HexColor(HEADERBACK)),
                ('BACKGROUND', (0, -2), (-0, -1), colors.HexColor(HEADERBACK)),
                ('SPAN', (1, -2), (-1, -2)),
                ('SPAN', (1, -1), (-1, -1)),
                # ('BACKGROUND', (0, 3), (-1, -1), colors.HexColor(ROWBACK)),
                ('INNERGRID', (0, 1), (-1, -1), 0.5, colors.white),
                #('ALIGN', (1, 2), (3, -1), 'RIGHT'),
            ])

    def body(self):
        "generate body"
        page_width, page_height = self.pagesize
        landscape = False
        for part in (  # 'client_data',
                'acc_summary',
                'payment_applied',
                'rec_charge',
                'non_rec_charge',
                'usage_charge',
                'trunk_detail',
                'daily_summary',
                'jd_trunk_detail',
                'jd_daily_summary',
                'prefix_summary',
                'code_name_summary',
                'country_summary',
                'top_100tn_summary',
                'inbound_summary',
                'did_charge_summary',
                'did_charge_breakdown',
                'summary_of_charges',
                'scc_charges'
        ):
            try:
                if part in self.data.json_content:
                    p = self.data.json_content[part]
                    if p and 'visible' in p and not p['visible']:
                        log.debug('part:{} skip no visible'.format(part))
                        print('part:{} skipno visible'.format(part))
                        continue
                    content = getattr(self, part)(self.data.json_content[part])

                    # if 'jd' in part and not landscape:
                    #     self.story.append(NextPageTemplate('landscape'))
                    #     self.story.append(PageBreak())
                    #     landscape=True
                    if content:
                        self.story.append(content)
                        self.story.append(Spacer(page_width, 30))
                    # if not 'jd' in part and landscape:
                    #     self.story.append(NextPageTemplate('portrait'))
                    #     self.story.append(PageBreak())
                    #     landscape = False
                    log.debug('part:{} success'.format(part))
                    print('part:{} success'.format(part))
                else:
                    log.debug('part:{} skip'.format(part))
                    print('part:{} skip'.format(part))
            except Exception as e:
                import traceback
                log.debug('part:{} error {}'.format(part, traceback.format_exc()))
                print('part:{} error {}'.format(part, traceback.format_exc()))
                pass
        self.story.append(Spacer(page_width, 30))

    def generate(self):
        page_width, page_height = self.pagesize
        self.frame = Frame(
            self.doc.leftMargin, self.doc.bottomMargin,
            self.doc.width, self.doc.height,
            leftPadding=10, rightPadding=10, topPadding=10,
            showBoundary=0)

        self.doc.addPageTemplates(
            [PageTemplate(id='portrait', onPage=self._add_page_number, frames=[self.frame], pagesize=letter),
             PageTemplate(id='landscape', onPage=self._add_page_number, frames=[self.frame],
                          pagesize=landscape(letter)),
             ])
        self.story.append(self.header())
        self.story.append(Spacer(page_width,     30))
        self.body()
        self.story.append(self.footer())
        self.doc.build(self.story)

        return True


class InvoicePdfOrig(InvoicePdf):
    pass


if __name__ == "__main__":
    from api_dnl.model import InvoiceSummary, init_db

    init_db()
    # inv = InvoiceSummary.get(1049)
    inv = InvoiceSummary.get(3401)
    doc = InvoicePdf(inv, '/home/novvv/tmp/' + inv.filename, False)
    doc.generate()

    print(inv.filename)
