from falcon_rest.db import orm, Column, fields, get_db
from sqlalchemy.sql import func, cast, select, alias, and_, or_
from sqlalchemy.ext.hybrid import hybrid_property, hybrid_method
from sqlalchemy.orm import backref, foreign, remote, column_property, synonym
from sqlalchemy import UniqueConstraint, column
from falcon_rest import schemes
from falcon_rest.logger import log
from falcon_rest.resources import resources
from api_dnl.base_model import DnlApiBaseModel
from api_dnl.fields.choice import ChoiceType
from api_dnl.models.user import User
# from api_dnl.model import SysSubMenu


class Role(DnlApiBaseModel):
    __tablename__ = 'role'

    role_id = Column(fields.Integer, primary_key=True)
    role_name = Column(fields.String(100), nullable=False, unique=True)
    view_pw = Column(fields.Boolean, default=False)
    default_sysfunc_id = Column(fields.Integer)
    del_able = Column(fields.Boolean, default=True)
    edit_able = Column(fields.Boolean, default=True)
    client_id = Column(fields.Integer)
    is_carriers = Column(fields.Boolean, default=True)
    is_transaction = Column(fields.Boolean, default=True)
    is_mutual_settlements = Column(fields.Boolean, default=True)
    is_invoices = Column(fields.Boolean, default=True)
    is_payment = Column(fields.Boolean, default=True)
    is_spam_report = Column(fields.Boolean, default=True)
    is_location_report = Column(fields.Boolean, default=True)
    is_origterm = Column(fields.Boolean, default=True)
    is_summary_report = Column(fields.Boolean, default=True)
    is_usage_report = Column(fields.Boolean, default=True)
    sync_list = Column(fields.Boolean, default=True)
    is_qos_report = Column(fields.Boolean, default=True)
    is_disconnect_cause = Column(fields.Boolean, default=True)
    is_billing_mismatch = Column(fields.Boolean, default=True)
    is_active_call = Column(fields.Boolean, default=True)
    is_termination_report = Column(fields.Boolean, default=True)
    is_rates_analysis = Column(fields.Boolean, default=True)
    is_call_simulation = Column(fields.Boolean, default=True)
    is_ingress_trunk_simulation = Column(fields.Boolean, default=True)
    is_egress_trunk_simulation = Column(fields.Boolean, default=True)
    is_sip_capture = Column(fields.Boolean, default=True)
    is_digit_mapping = Column(fields.Boolean, default=True)
    is_trunk = Column(fields.Boolean, default=True)
    is_dynamic_routing = Column(fields.Boolean, default=True)
    is_static_route_table = Column(fields.Boolean, default=True)
    is_block_list = Column(fields.Boolean, default=True)
    is_routing_plan = Column(fields.Boolean, default=True)
    is_active_web_session = Column(fields.Boolean, default=True)
    is_payment_term = Column(fields.Boolean, default=True)
    is_jurisdiction = Column(fields.Boolean, default=True)
    is_capicity = Column(fields.Boolean, default=True)
    is_rate_table = Column(fields.Boolean, default=True)
    is_code_deck = Column(fields.Boolean, default=True)
    is_time_profile = Column(fields.Boolean, default=True)
    is_currency = Column(fields.Boolean, default=True)
    is_task_schedulers = Column(fields.Boolean, default=True)
    is_mail_template = Column(fields.Boolean, default=True)
    is_role = Column(fields.Boolean, default=True)
    is_user = Column(fields.Boolean, default=True)
    is_change_password = Column(fields.Boolean, default=True)
    is_setting = Column(fields.Boolean, default=True)
    is_import_log = Column(fields.Boolean, default=True)
    is_export_log = Column(fields.Boolean, default=True)
    sync_backup = Column(fields.Boolean, default=True)
    is_lrn_setting = Column(fields.Boolean, default=True)
    active = Column(fields.Boolean, default=True)
    is_unpaid_bills = Column(fields.Boolean, default=True)
    is_service_charge = Column(fields.Boolean, default=True)
    is_voip_gateway = Column(fields.Boolean, default=True)
    is_trouble_shoot = Column(fields.Boolean, default=True)
    is_event = Column(fields.Boolean, default=True)
    is_ticket = Column(fields.Boolean, default=True)
    is_buy_select_country = Column(fields.Boolean, default=True)
    is_search_private_buy = Column(fields.Boolean, default=True)
    is_buy_confirm_order = Column(fields.Boolean, default=True)
    is_sell_select_country = Column(fields.Boolean, default=True)
    is_search_private_sell = Column(fields.Boolean, default=True)
    is_sell_confirm_order = Column(fields.Boolean, default=True)
    role_type = Column(fields.Integer, default=1)
    LANDING_PAGE = {
        0: 'QoS Report',
        1: 'Summary Report',
        2: 'Orig - Term Report',
        3: 'Carrier Management',
        4: 'Code Deck'
    }
    landing_page = Column(ChoiceType(LANDING_PAGE), default=3)
    default_mod = Column(fields.String(100), default='')
    default_mod2 = Column(fields.String(100), default='')
    delete_invoice = Column(fields.Boolean, default=True)
    delete_payment = Column(fields.Boolean, default=True)
    delete_credit = Column(fields.Boolean, default=True)
    reset_balance = Column(fields.Boolean, default=True)
    modify_credit_limit = Column(fields.Boolean, default=True)
    modify_min_profit = Column(fields.Boolean, default=True)
    users = orm.relationship(User, uselist=True, back_populates='role')
    role_privilegies = orm.relationship('RolePrivilege', back_populates='role', uselist=True, single_parent=True,
                                        cascade="all, delete-orphan")
    role_ui_privilegies = orm.relationship('RoleSubMenu', back_populates='role', uselist=True, single_parent=True,
                                        cascade="all, delete-orphan")

    @hybrid_property
    def _role_privilegies(self):
        privileges = RolePrivilege.filter(RolePrivilege.role_id == self.role_id).all()
        return [priv for priv in privileges]

    @_role_privilegies.setter
    def _role_privilegies(self, value):
        self.create_default_privileges()
        for i in value:
            if not self.role_id:
                privilege = list(filter(lambda j: j.system_function_id == i.system_function_id, self.role_privilegies))[0]
                privilege.readable = i.readable
                privilege.writable = i.writable
                privileges = list(filter(lambda j: j.system_function_id != i.system_function_id, self.role_privilegies))
                privileges += [privilege]
                self.role_privilegies = privileges
                continue
            privilege = RolePrivilege.filter(and_(RolePrivilege.system_function_id == i.system_function_id,
                                                  RolePrivilege.role_id == self.role_id))
            if privilege.first():
                privilege.update(dict(readable=i.readable, writable=i.writable, executable=i.executable), synchronize_session='fetch')
            else:
                system_function = SystemFunction.filter(SystemFunction.func_url == i.func_url).first()
                if system_function:
                    privilege = RolePrivilege(
                        role_id=self.role_id,
                        system_funtion_id=system_function.system_function_id
                    )
                    privilege.readable = i.readable
                    privilege.writable = i.writable
                    privilege.executable = i.executable
                    self.role_privilegies.append(privilege)

    @hybrid_property
    def _role_ui_privilegies(self):
        return self.role_ui_privilegies

    @_role_ui_privilegies.setter
    def _role_ui_privilegies(self, value):
        self.create_default_ui_privileges()
        for i in value:
            if not self.role_id:
                privilege = list(filter(lambda j: j.submenu_id == i.submenu_id, self.role_ui_privilegies))[0]
                privilege.readable = i.readable
                privilege.editable = i.editable
                privileges = list(filter(lambda j: j.submenu_id != i.submenu_id, self.role_ui_privilegies))
                privileges += [privilege]
                self.role_ui_privilegies = privileges
                continue
            privilege = RoleSubMenu.filter(and_(RoleSubMenu.submenu_id == i.submenu_id,
                                                  RoleSubMenu.role_id == self.role_id))
            if privilege.first():
                privilege.update(dict(readable=i.readable, writable=i.writable, executable=i.executable), synchronize_session='fetch')
            else:
                system_function = SysSubMenu.filter(SysSubMenu.func_url == i.func_url).first()
                if system_function:
                    privilege = RoleSubMenu(
                        role_id=self.role_id,
                        system_funtion_id=system_function.submenu_id
                    )
                    privilege.readable = i.readable
                    privilege.editable = i.editable
                    privilege.executable = i.executable
                    self.role_ui_privilegies.append(privilege)

    @hybrid_property
    def group_privilegies(self):
        groups = SysMainMenu.filter().all()
        group_privilegies = []
        for group in groups:
            function_ids = [i.id for i in group.submenus]
            privileges = RoleSubMenu.filter(and_(RoleSubMenu.role_id == self.role_id,
                                                   RoleSubMenu.submenu_id.in_(function_ids))).all()

            readable = list(set([privilege.readable for privilege in privileges]))
            if True in readable or int(self.role_id) == 0:
                readable = True
            else:
                readable = False

            group_privilegies += [{
                'id': group.id,
                'name': group.display_name,
                'readable': readable
            }]
        return group_privilegies

    @group_privilegies.setter
    def group_privilegies(self, value):
        for group in value:
            group_privilegie = SysMainMenu.get(group.id)
            function_ids = [i.id for i in group_privilegie.submenus]
            readable = getattr(group, 'readable', False)
            RoleSubMenu.filter(
                and_(RoleSubMenu.role_id == self.role_id, RoleSubMenu.submenu_id.in_(function_ids))). \
                update({'readable': readable},
                       synchronize_session='fetch')

    @hybrid_property
    def user_count(self):
        return len(self.users)

    @hybrid_property
    def privilege_count(self):
        return len(self.role_privilegies)

    def privilege_for_func(self, func_name):
        return [p for p in self.role_privilegies if p.system_function.func_name == func_name]

    @staticmethod
    def init():
        if not Role.get(0):
            r = Role(role_id=0, role_name='Administrator')
            try:
                r.save()
            except:
                r.session().rollback()
        if not Role.get(2):
            r = Role(role_id=2, role_name='Agent')
            try:
                r.save()
            except:
                r.session().rollback()
        if not Role.get(1):
            r = Role(role_id=1, role_name='Client')
            try:
                r.save()
            except:
                r.session().rollback()

    def create_default_privileges(self):
        for func in SystemFunction.query().all():
            priv = RolePrivilege.filter(RolePrivilege.role_id == self.role_id). \
                filter(RolePrivilege.system_function_id == func.system_function_id).first()
            if not priv:
                priv = RolePrivilege(role_id=self.role_id, system_function_id=func.system_function_id)
                self.role_privilegies.append(priv)
            if self.role_id == 0:
                if not priv.readable:
                    priv.readable = True
                if not priv.writable:
                    priv.writable = True
                priv.save()

    def create_default_ui_privileges(self):
        for func in SysSubMenu.query().all():
            priv = RoleSubMenu.filter(RoleSubMenu.role_id == self.role_id). \
                filter(RoleSubMenu.submenu_id == func.id).first()
            if not priv:
                priv = RoleSubMenu(role_id=self.role_id, submenu_id=func.id)
                self.role_ui_privilegies.append(priv)
            if self.role_id == 0:
                if not priv.readable:
                    priv.readable = True
                if not priv.writable:
                    priv.writable = True
                priv.save()


# User.role = orm.relationship(Role,uselist=False,back_populates='users',primaryjoin=(User.role_id == foreign(Role.role_id) ))
User.role_name = column_property(select([Role.role_name]).where(User.role_id == Role.role_id).correlate_except(Role))


class SystemFunctionGroup(DnlApiBaseModel):
    __tablename__ = 'system_function_group'
    id = Column(fields.Integer, primary_key=True)
    group_name = Column(fields.String(50), unique=True)
    description = Column(fields.Text)
    ordering = Column(fields.Integer)

    functions = orm.relationship('SystemFunction', back_populates='group', uselist=True, cascade="all")


class SystemFunction(DnlApiBaseModel):
    __tablename__ = 'system_function'
    FUNC_TYPE_DICT = {0: 'get', 1: 'post', 2: 'path'}
    system_function_id = Column(fields.Integer, primary_key=True)
    description = Column(fields.String(40))
    func_name = Column(fields.String(50), unique=True, index=True)
    is_read = Column(fields.Boolean, default=False)
    is_write = Column(fields.Boolean, default=False)
    is_exe = Column(fields.Boolean, default=False)
    func_url = Column(fields.String(), index=True, unique=True)
    func_type = Column(ChoiceType(FUNC_TYPE_DICT), nullable=False)  # Column(fields.Integer)
    key_118n = Column(fields.String(50))
    parent_id = Column(fields.Integer)
    image_name = Column(fields.String(50))
    develop_status = Column(fields.Integer, nullable=False)
    ordering = Column(fields.Integer)
    group_id = Column(fields.ForeignKey('system_function_group.id', ondelete='SET NULL'), nullable=True, index=True)

    role_privilegies = orm.relationship('RolePrivilege', back_populates='system_function', uselist=True)
    group = orm.relationship('SystemFunctionGroup', back_populates='functions', uselist=True)

    @hybrid_property
    def urls(self):
        try:
            return self.func_url.split(',')
        except:
            return []

    @urls.setter
    def urls(self, value):
        try:
            self.func_url = ','.join([str(x) for x in value])
        except:
            pass

    @classmethod
    def modules(cls):
        return cls.query().session.query(cls.image_name, func.count().label('usage_count')).group_by(
            cls.image_name).all()

    @classmethod
    def get_by_name(cls, name):
        return cls.filter(cls.func_name == name).first()

    @staticmethod
    def register_function(name, desc, path, meth, dev_stat=0, image=''):
        log.debug('REGISTER: {} {} {} {}'.format(name, path, meth, image))
        if meth == 'path':
            url = 'PATCH' + path
        else:
            url = meth.upper() + path
        srch = SystemFunction.filter(or_(and_(SystemFunction.func_url.like('%' + url + '%'), SystemFunction.func_url.notlike('%' + url + '/%')),
                                         SystemFunction.func_name == name)).all()
        if len(srch) == 0:
            obj = SystemFunction(func_name=name, develop_status=dev_stat, func_type=meth)
            # obj=SystemFunction.get(fid)
        else:
            obj = srch[0]
            for d in srch[1:]:
                d.delete()

        if obj and (url not in obj.urls):
            if desc:
                obj.description = desc
            u = obj.urls
            u.append(url)
            obj.urls = u
            if 'POST' in obj.func_url or 'PATCH' in obj.func_url or 'DELETE' in obj.func_url:
                obj.is_write = True
                # obj.is_exe=True
            if 'GET' in obj.func_url:
                obj.is_read = True
                # obj.is_write=False
                # obj.is_exe=False
            obj.func_type = meth
            obj.develop_status = dev_stat
            obj.image_name = image
            try:
                return obj.save()
            except Exception as e:
                log.debug(f"ROLLBACK |\t| FULL ERROR - {e}")
                log.debug('PARAMS: {} {} {} {}'.format(name, path, meth, image))
                obj.session().rollback()
        return None

    @staticmethod
    def init(routes):
        try:
            log.info('Initializing system functions...')
            # SystemFunction.filter(SystemFunction.system_function_id.isnot(None)).delete()
            # SystemFunction.query().update(dict(func_url=None),synchronize_session='fetch')
            for mod in routes:
                mod_routes = routes[mod]
                if 'routes' in mod_routes:
                    mod_routes = mod_routes['routes']
                for path in mod_routes:
                    if 'path' in path:
                        p = path['path']
                        method = path['method']
                        if method == 'patch' or method == 'delete':
                            method = 'path'
                        res = path['resource']
                        # name=res.__class__.__name__
                        description = res.comments if hasattr(res, 'comments') else res.description if hasattr(res,
                                                                                                            'description') else ''
                        develop_status = res.develop_status if hasattr(res, 'develop_status') else 0
                        if res and hasattr(res, 'has_info_operation') and res.has_info_operation and hasattr(res,
                                                                        'scheme_class_get') and res.scheme_class_get:
                            log.debug("Has info operation")
                            name = res.scheme_class_get.__name__.replace('Scheme', '')
                            SystemFunction.register_function(name, description, p, 'get', develop_status, image=mod)
                        if hasattr(res,
                                'has_modify_operation') and not getattr(res, 'has_modify_operation', True) and not getattr(res, 'has_delete_operation', True):
                            log.debug("Has modify operation")
                            continue
                        if res and hasattr(res, 'scheme_class') and res.scheme_class:
                            log.debug("Has scheme_class")
                            name = res.scheme_class.__name__.replace('SchemeModify', '').replace('SchemeGet', '').replace(
                                'Scheme', '')
                            SystemFunction.register_function(name, description, p, method, develop_status, image=mod)
                        if res and hasattr(res, 'cloud_method'):
                            log.debug("Has cloud method")
                            name = res.__class__.__name__
                            SystemFunction.register_function(name, description, p, method, develop_status, image=mod)
        except Exception as e:
            log.error('error when creating missing system functions {}'.format(str(e)))


class SystemModule(DnlApiBaseModel):
    __tablename__ = '#system_module'
    __table_args__ = {'prefixes': ['TEMPORARY']}

    @classmethod
    def query(cls):
        cls = SystemFunction
        return cls.query().session.query(cls.image_name, func.count().label('usage_count')).group_by(
            cls.image_name)

    image_name = Column(fields.String(50), primary_key=True)
    usage_count = Column(fields.Integer)


class RolePrivilege(DnlApiBaseModel):
    __tablename__ = 'role_privilege'
    __table_args__ = (
        UniqueConstraint('role_id', 'system_function_id'),
    )

    role_privilege_id = Column(fields.Integer, primary_key=True)
    role_id = Column(fields.ForeignKey('role.role_id'), nullable=False, index=True, autoincrement=True)
    system_function_id = Column(fields.ForeignKey('system_function.system_function_id'), nullable=False, index=True)
    writable = Column(fields.Boolean, nullable=False, default=False)
    readable = Column(fields.Boolean, nullable=False, default=False)
    executable = Column(fields.Boolean, nullable=False, default=False)
    reseller_id = Column(fields.Integer)
    system_function_group_id = column_property(
        select([SystemFunction.group_id]).where(
            SystemFunction.system_function_id == system_function_id).correlate_except(SystemFunction))

    role = orm.relationship(Role, back_populates='role_privilegies', uselist=False)
    system_function = orm.relationship(SystemFunction, back_populates='role_privilegies', uselist=False)

    @hybrid_property
    def role_name(self):
        return self.role.role_name

    @hybrid_property
    def func_name(self):
        return self.system_function.func_name

    @hybrid_property
    def image_name(self):
        return self.system_function.image_name

    @hybrid_property
    def func_url(self):
        try:
            return self.system_function.func_url
        except:
            return None

    @func_url.setter
    def func_url(self, value):
        f = SystemFunction.filter(SystemFunction.func_url == value).first()
        if f:
            self.system_function_id = f.system_function_id
        else:
            self.system_function_id = None

    def allow(self, perm):
        if not self.role.active:
            return False
        if perm == 'write' and self.writable and self.system_function.is_write:
            return True
        if perm == 'read' and self.readable and self.system_function.is_read:
            return True
        return False


class SysMainMenu(DnlApiBaseModel):
    __tablename__ = 'sys_main_menu'
    id = Column(fields.Integer, primary_key=True)
    display_name = Column(fields.String(50), unique=True, index=True)
    ordering = Column(fields.Integer)
    enabled = Column(fields.Boolean, default=True)

    submenus = orm.relationship('SysSubMenu', back_populates='main_menu', uselist=True, cascade="all")


class SysSubMenu(DnlApiBaseModel):
    __tablename__ = 'sys_submenu'
    id = Column(fields.Integer, primary_key=True)
    display_name = Column(fields.String(50), unique=True, index=True)
    web_url = Column(fields.String())
    ordering = Column(fields.Integer)
    is_readable = Column(fields.Boolean, default=True)
    is_editable = Column(fields.Boolean, default=True)
    is_executable = Column(fields.Boolean, default=True)
    enabled = Column(fields.Boolean, default=True)
    main_menu_id = Column(fields.ForeignKey('sys_main_menu.id', ondelete='SET NULL'), nullable=True, index=True)

    main_menu = orm.relationship('SysMainMenu', back_populates='submenus', uselist=True)


class RoleSubMenu(DnlApiBaseModel):
    __tablename__ = 'role_submenu_rel'
    __table_args__ = (
        UniqueConstraint('role_id', 'submenu_id'),
    )

    id = Column(fields.Integer, primary_key=True)
    role_id = Column(fields.ForeignKey('role.role_id'), nullable=False, index=True, autoincrement=True)
    submenu_id = Column(fields.ForeignKey('sys_submenu.id'), nullable=False, index=True)
    editable = Column(fields.Boolean, nullable=False, default=False)
    readable = Column(fields.Boolean, nullable=False, default=False)
    executable = Column(fields.Boolean, nullable=False, default=False)

    writable = synonym('editable')

    main_menu_id = column_property(
        select([SysSubMenu.main_menu_id]).where(
            SysSubMenu.id == submenu_id).correlate_except(SysSubMenu))
    enabled = column_property(
        select([SysSubMenu.enabled]).where(
            SysSubMenu.id == submenu_id).correlate_except(SysSubMenu))
    ordering = column_property(
        select([SysSubMenu.ordering]).where(
            SysSubMenu.id == submenu_id).correlate_except(SysSubMenu))
    display_name = column_property(
        select([SysSubMenu.display_name]).where(
            SysSubMenu.id == submenu_id).correlate_except(SysSubMenu))
    web_url = column_property(
        select([SysSubMenu.web_url]).where(
            SysSubMenu.id == submenu_id).correlate_except(SysSubMenu))

    role = orm.relationship(Role, back_populates='role_ui_privilegies', uselist=False)