from marshmallow import validate, pre_load
from marshmallow.fields import Bool, Str, Decimal, Int, Nested, DateTime, Float, List, UUID, Dict

from api_dnl import model
from api_dnl.fields import Choice
from api_dnl.scheme import BaseModelScheme, Emails, ClientPaymentScheme, PaymentGatewayHistoryGetScheme, \
    RateTableGetScheme, RateGetScheme, DidBillingRelGetScheme, \
    DidBillingPlanGetScheme,  PaymentGatewayHistoryScheme, _valid, ResourcePrefixGetScheme, \
    ResourceIpGetScheme, EmailLogGetScheme, ActualBalanceHistoryGetScheme, IngressTrunkScheme, IngressTrunkModifyScheme, \
    ResourceWithIpsScheme, ResourcePrefixScheme, ResourcePrefixGetScheme, ResourcePrefixModifyScheme, \
    ResourcePrefixWithIPScheme, ResourceWithIpAndAliasGetScheme

from api_dnl.scheme import IP_REGEXP, NAME_REGEXP, PREFIX_REGEXP, TIME_REGEXP, ValidationError, SuccessScheme
from api_dnl.utils.statisticapi2 import FIELD_MAP


class CdrQueryScheme(BaseModelScheme):
    fields = List(Str(validate=validate.OneOf(list(FIELD_MAP.keys()))))
    start_time = DateTime()
    end_time = DateTime()

    class Meta:
        model = model.Resource
        fields = ('start_time', 'end_time', 'fields')


def valid_cdr_detail_fields(value):
    try:
        fields = value.split(',')
        for f in fields:
            if not hasattr(model.CdrReportDetail, f):
                raise ValidationError('invalid field {}'.format(f))
    except:
        raise ValidationError('invalid fields')


class CdrReportDetailGetScheme(BaseModelScheme):
    report_time = DateTime()
    ingress_rate_table_id = Int(allow_none=True)

    # for query
    # fields = List(Str(validate=validate.OneOf(list(FIELD_MAP.keys()))))
    field = Str(validate=valid_cdr_detail_fields)
    start_time = Str()
    end_time = Str()
    release_cause = Int()
    ingress_id = Str()
    egress_id = Str()

    class Meta:
        model = model.CdrReportDetail
        fields = ()
        query_fields = ('start_time', 'end_time', 'field', 'release_cause', 'ingress_id', 'egress_id')


class CdrReportDetailEarliestGetScheme(BaseModelScheme):
    report_time = DateTime
    class Meta:
        model = model.CdrReportDetail
        fields = ()
        query_fields = ()


def valid_cdr_fields(value):
    try:
        fields = value.split(',')
        for f in fields:
            if not hasattr(model.ClientCdr, f):
                raise ValidationError('invalid field {}'.format(f))
    except:
        raise ValidationError('invalid field')


class CdrReportGetScheme(BaseModelScheme):
    # report_time = DateTime()
    # start = Int()
    # count = Int()
    start_time = Int()
    end_time = Int()
    field = Str(validate=valid_cdr_detail_fields)
    format = Str()
    tz = Int()
    call_tz = Int()
    non_zero = Int()
    egress_non_zero = Int()
    human_readable = Int()

    call_duration_min = Int()
    call_duration_max = Int()
    egress_cost_min = Float()
    egress_cost_max = Float()
    ingress_client_cost_min = Float()
    ingress_client_cost_max = Float()
    ingress_name = Str()
    egress_name = Str()
    ingress_client_name = Str()
    egress_client_name = Str()
    egress_rate_table_name = Str()
    dynamic_route_name = Str()
    static_route_name = Str()
    egress_trunk_trace = Str()

    release_cause_name = Str()
    trunk_type2 = Str()
    timezone_name = Str()

    egress_rate_type_name = Str()
    ingress_rate_type_name = Str()
    ingress_dnis_type_name = Str()
    orig_jur_type_name = Str()
    term_jur_type_name = Str()
    route_plan_name = Str()


    time = Str()
    start_time_of_date = Str()
    end_epoch = Str()
    answer_time_of_date = Str()
    release_tod = Str()
    ingress_rate_effective_date = Str()
    egress_rate_effective_date = Str()
    dnis_fuzzy = Str()
    orig_ani_fuzzy = Str()
    term_ani_fuzzy = Str()

    orig_shaken_status = Str()
    orig_shaken_lvl = Str()
    orig_shaken_ocn = Str()
    orig_shaken_subject = Str()
    term_shaken_status = Str()
    term_shaken_lvl = Str()
    term_shaken_ocn = Str()
    term_shaken_subject = Str()

    ingress_total_calls = Int()
    egress_total_calls = Int()

    dst_type = Str()
    src_type = Str()
    not_zero_calls = Int()
    ingress_duration = Int()
    ingress_call_cost = Float()
    egress_call_cost = Float()
    shaken_key_id = Int()

    class Meta:
        model = model.ClientCdr
        fields = ()
        query_fields = ('start_time', 'end_time', 'field', 'tz', 'human_readable', 'trunk_type2', 'o_trunk_type2',
                        'egress_non_zero', 'non_zero', 'dnis_fuzzy', 'orig_ani_fuzzy', 'term_ani_fuzzy', 'release_cause_name')


class CdrReportDailyGetScheme(BaseModelScheme):
    # report_time = DateTime()
    # start = Int()
    # count = Int()
    start_time = Int()
    end_time = Int()
    field = Str(validate=valid_cdr_detail_fields)
    format = Str()
    tz = Int()
    call_tz = Int()
    non_zero = Int()
    egress_non_zero = Int()
    human_readable = Int()
    o_trunk_type2 = Int()

    call_duration_min = Int()
    call_duration_max = Int()
    egress_cost_min = Float()
    egress_cost_max = Float()
    ingress_client_cost_min = Float()
    ingress_client_cost_max = Float()
    ingress_name = Str()
    egress_name = Str()
    ingress_client_name = Str()
    egress_client_name = Str()
    egress_rate_table_name = Str()
    dynamic_route_name = Str()
    static_route_name = Str()
    egress_trunk_trace = Str()

    release_cause_name = Str()
    trunk_type2 = Str()
    timezone_name = Str()

    egress_rate_type_name = Str()
    ingress_rate_type_name = Str()
    ingress_dnis_type_name = Str()
    orig_jur_type_name = Str()
    term_jur_type_name = Str()
    route_plan_name = Str()


    time = Str()
    start_time_of_date = Str()
    end_epoch = Str()
    answer_time_of_date = Str()
    release_tod = Str()
    ingress_rate_effective_date = Str()
    egress_rate_effective_date = Str()
    dnis_fuzzy = Str()

    orig_shaken_status = Str()
    orig_shaken_lvl = Str()
    orig_shaken_ocn = Str()
    orig_shaken_subject = Str()
    term_shaken_status = Int()
    term_shaken_lvl = Str()
    term_shaken_ocn = Str()
    term_shaken_subject = Str()

    class Meta:
        model = model.CdrReportDaily
        fields = ()
        query_fields = ('start_time', 'end_time', 'field', 'tz', 'human_readable', 'trunk_type2', 
                        'egress_non_zero', 'non_zero', 'release_cause_name', 'o_trunk_type2')


class CodeReportDailyGetScheme(BaseModelScheme):
    # report_time = DateTime()
    # start = Int()
    # count = Int()
    start_time = Int()
    report_time = Int()
    end_time = Int()
    field = Str(validate=valid_cdr_detail_fields)
    format = Str()
    tz = Int()
    human_readable = Int()

    ingress_id = Int()
    egress_id = Int()
    ingress_client_id = Int()
    egress_client_id = Int()
    code = Str()
    
    total_calls = Int()
    not_zero_calls = Int()
    ingress_total_call = Int()

    class Meta:
        model = model.CodeReportDaily
        fields = ()
        query_fields = ('start_time', 'end_time', 'field', 'tz', 'human_readable')


class CdrReportDetailGroupableGetScheme(CdrReportDailyGetScheme):
    # report_time = DateTime()
    # start = Int()
    # count = Int()
    start_time = Int()
    end_time = Int()
    field = Str(validate=valid_cdr_detail_fields)
    format = Str()
    tz = Int()

    class Meta:
        model = model.CdrReportDetail
        fields = ()
        query_fields = ('start_time', 'end_time', 'field', 'tz', 'human_readable',)


class ClientCdrReportDailyGroppedResponseScheme(SuccessScheme):
    total = Int()
    count = Int()
    _from = Int(dump_to='from', load_from="from")
    data = Nested('CdrReportDailyGetScheme', many=True)


class ClientCodeReportDailyGroppedResponseScheme(SuccessScheme):
    total = Int()
    count = Int()
    _from = Int(dump_to='from', load_from="from")
    data = Nested('CodeReportDailyGetScheme', many=True)

class ClientCdrReportGroppedResponseScheme(SuccessScheme):
    total = Int()
    count = Int()
    _from = Int(dump_to='from', load_from="from")
    data = Nested('CdrReportGetScheme', many=True)


class ObjectCreatedRequestIdScheme(SuccessScheme):
    request_id = Str()


# class CdrAsyncTaskSchemeGet(BaseModelScheme):
#     class Meta:
#         model = model.CdrAsyncTask
#         fields = ()


class CdrAsyncTaskScheme(CdrReportGetScheme):
    do_zip = Bool(default=False)
    class Meta:
        model = model.CdrAsyncTask
        # fields = ()
        exclude = ('request_id', 'request_client_id', 'filter', 'download_link',
                   'expiration_time', 'job_start_time', 'job_end_time', 'msg', 'progress', 'count', 'size',
                   'status', 'email', 'orig_file', 'mail_status', 'method', 'type')

    @classmethod
    def get_object_created_response(cls, **kwargs):
        from falcon_rest.responses import ObjectCreatedResponse
        kwargs['data_key'] = 'request_id'
        kwargs['scheme'] = ObjectCreatedRequestIdScheme
        return ObjectCreatedResponse(**kwargs)


class CdrAsyncTaskSchemeGet(BaseModelScheme):
    start_time = Int()
    end_time = Int()
    client_id = Int()
    o_trunk_type2 = Int()
    created_on = DateTime()
    download_link = Str(attribute='_download_link')
    class Meta:
        model = model.CdrAsyncTask
        search_fields = ('created_on', 'start_time', 'end_time', 'job_id', 'job_start_time', 'o_trunk_type2', 'client_id')
        query_fields = ('created_on_gt', 'created_on_lt', 'job_start_time_gt', 'job_start_time_lt')


class CdrAsyncTaskEmailScheme(BaseModelScheme):
    # request_id = Str(validate=[validate.Length(max=16),lambda value:_valid_unique('CdrDownloadTask','request_id',value)])
    request_id = Str(validate=[validate.Length(max=16), lambda value: _valid('CdrAsyncTask', 'request_id', value)])
    email = Emails()
    client_id = Int(allow_none=True, validate=lambda value: _valid('Client', 'client_id', value))
    direct = Nested('SendCdrDirectScheme', allow_none=True)

    @pre_load
    def fix_emails(self, data):
        if 'email' in data:
            if ';' not in data['email']:
                data['email'] = data['email'].replace(',', ';')
        return data

    class Meta:
        model = model.CdrAsyncTask
        fields = ('email', 'direct', 'client_id')


def validate_group(value):
    try:
        fields = value.split(',')
        for f in fields:
            if not hasattr(model.ClientCdr, f):
                raise ValidationError('invalid field {}'.format(f))
    except:
        raise ValidationError('invalid group')


def validate_step(value):
    if value in ['hour', 'day', 'week', 'month', 'year', 'all']:
        return True
    intv = int(value)
    return True


class CdrReportGroppedRequestScheme(BaseModelScheme):
    # for query
    # fields = List(Str(validate=validate.OneOf(list(FIELD_MAP.keys()))))
    # fields = Str(validate=valid_cdr_fields)
    start_time = Str(allow_none=False, required=True, description='search from specified time (allow negative values)')
    end_time = Str(allow_none=False, required=True,
                   description="search till time (end_time > start_time). Default: 0 do not limit search by time")
    method = Str(validate=validate.OneOf(['max', 'min', 'avrg', 'total']), default='total', allow_none=False,
                 required=True, description="'max', 'min', 'avrg', 'total'")
    # group = Str(validate=valid_cdr_fields)
    step = Str(validate=validate_step, default='all',
               description="granularity: 'hour', 'day', 'week', 'month', 'year', 'all', or amount of minutes.")
    sort = Str(description="sort results by field (allow multiple sort fields delimited by ','")
    order = Str(
        description="sorting order: 'asc', 'desc'. Default: descending order. Mandatory be multiple for multiple 'order'")
    is_hide_unauthorized = Bool()
    show_non_zero_only = Bool()
    tz = Str()
    o_trunk_type2 = Int()
    trunk_type2 = Int()
    class Meta:
        model = model.CdrReportDetail
        search_fields = ('start_time', 'end_time', 'method', 'step', 'sort', 'order', 'is_hide_unauthorized',
                         'show_non_zero_only', 'o_trunk_type2', 'trunk_type2', 'tz')


class CdrReportDetailInnerScheme(BaseModelScheme):
    ingress_call_cost = Float()
    egress_call_cost = Float()
    #calculated fields
    ingress_name = Str()
    egress_name = Str()
    ingress_client_name = Str()
    egress_client_name = Str()
    release_cause_name = Str()
    agent_name = Str()
    commission = Float()
    limited = Int()
    nrf_calls = Str()
    report_time = Int()
    asr = Float()
    qsr = Float()
    ingress_npr = Float()
    egress_npr = Float()
    SDP30_calls = Float()
    SDP6_calls = Float()
    port_fee = Float()
    ingress_call_dur = Float()
    egress_call_dur = Float()
    ingress_bill_dur = Float()
    egress_bill_dur = Float()
    not_zero_calls_12 = Float()
    not_zero_calls_18 = Float()
    not_zero_calls_24 = Float()

    class Meta:
        model = model.CdrReportDetail


class CdrReportGroppedResponseScheme(SuccessScheme):
    total = Int()
    count = Int()
    _from = Int(dump_to='from', load_from="from")
    data = Nested('CdrReportDetailInnerScheme', many=True)


class DidReportGroppedRequestScheme(CdrReportGroppedRequestScheme):
    trunk_type2 = Int()
    o_trunk_type2 = Int()
    class Meta:
        model = model.DidReport
        search_fields = ('start_time', 'end_time', 'method', 'step', 'group', 'fields', 'sort',
                         'order', 'show_non_zero_only', 'trunk_type2', 'agent_id', 'o_trunk_type2')


class DidReportInnerScheme(BaseModelScheme):
    ingress_name = Str()
    egress_name = Str()
    ingress_client_name = Str()
    egress_client_name = Str()
    agent_name = Str()
    agent_id = Int()
    mrc = Float()
    nrc = Float()
    port_fee = Float()
    report_time = Int()
    country = Str()
    state = Str()
    class Meta:
        model = model.DidReport


class DidReportGroppedResponseScheme(SuccessScheme):
    total = Int()
    count = Int()
    _from = Int(dump_to='from', load_from="from")
    data = Nested('DidReportInnerScheme', many=True)


# ==
class HostBasedReportGroppedRequestScheme(CdrReportGroppedRequestScheme):
    class Meta:
        model = model.HostBasedReport
        search_fields = ('start_time', 'end_time', 'method', 'step', 'group', 'fields', 'sort', 'order')


class HostBasedReportInnerScheme(CdrReportDetailInnerScheme):
    class Meta:
        model = model.HostBasedReport


class HostBasedReportGroppedResponseScheme(SuccessScheme):
    total = Int()
    count = Int()
    _from = Int(dump_to='from', load_from="from")
    data = Nested('HostBasedReportInnerScheme', many=True)


# region +++ReportTemplate+++
class ReportTemplateScheme(BaseModelScheme):
    id = Int()
    created_by = Str(validate=[validate.Length(max=40)])
    created_on = DateTime()
    template_name = Str(validate=[validate.Length(max=100), validate.Regexp(NAME_REGEXP)])
    period = Choice(required=True)
    params = Dict()
    #params = Nested(CdrReportGroppedRequestScheme,many=False)

    class Meta:
        model = model.ReportTemplate
        fields = ('template_name', 'params', 'period')


class ReportTemplateSchemeGet(ReportTemplateScheme):
    report_url = Str()

    class Meta:
        model = model.ReportTemplate
        fields = ('created_by', 'created_on', 'template_name', 'params', 'report_url', 'id')
        search_fields = ('id', 'created_by', 'template_name', 'params',)
        query_fields = ('created_on_gt', 'created_on_lt',)


class ReportTemplateSchemeModify(ReportTemplateScheme):
    pass


# endregion ---ReportTemplate---


class DailyCdrCloudUpdateScheme(BaseModelScheme):
    json_key = Str()
    bucket = Str()
    workdir = Str()
    non_zero_cdr_only = Bool()

    class Meta:
        model = model.DnlDailyCdrCloudCfg
        fields = ('json_key', 'bucket', 'workdir', 'non_zero_cdr_only',)
        search_fields = ('json_key', 'bucket', 'workdir', 'non_zero_cdr_only',)
        query_fields = ('json_key', 'bucket', 'workdir', 'non_zero_cdr_only',)


class DailyCdrCloudUpdateSchemeModify(DailyCdrCloudUpdateScheme):
    pass
