#!/usr/bin/python
# -*- coding: UTF-8 -*-
"""The common parts of CDR and PCAP files parser and loader."""
from os.path import join
import logging
import subprocess
import time
import hashlib


__author__ = 'Denis Trofimov <silaradost@ya.ru>'
__date__ = '28.03.2017'


class Timeout(Exception):

    """Exception raised by run_command() when the operation takes too long."""

    pass


def run_command(cmd, timeout=None, shell=True):
    """Execute string `cmd` as a shell command.

    :param cmd: A string containing the shell command.
    :param timeout: The time the command is allowed to spend.
    :param shell: Executes command in a shell if ``True``.
    :return: Like the subprocess.Popen.communicate() returns a tuple
    (stdoutdata, stderrdata)

    Executes string `cmd` as a shell command, allowing it only to
    take a certain amount of time `timeout`. Then `timeout` is hit
    raise `Timeout` exception. No exceptions are caught.

    """

    if timeout:
        if shell:
            logging.debug("Run in a shell the command `{0}` with the"
                          " timeout {1}sec.".format(cmd, timeout))
        else:
            logging.debug("Run the command `{0}` with the timeout {1}sec."
                          .format(cmd, timeout))
        timeout = float(timeout)
        _start_timer = time.time()
    else:
        if shell:
            logging.debug("Run in a shell the command `{0}`.".format(cmd))
        else:
            logging.debug("Run the command `{0}`.".format(cmd))

    if shell:
        _popen = subprocess.Popen(cmd,shell=True,stdout=subprocess.PIPE,
            stderr=subprocess.PIPE,universal_newlines=True)
    else:
        _popen = subprocess.Popen(cmd,shell=False,stdout=subprocess.PIPE,
            stderr=subprocess.PIPE,universal_newlines=True)

    if timeout:
        while _popen.poll() is None:
            if (time.time() - _start_timer) > timeout:
                _popen.terminate()
                raise Timeout("Timeout caused execution stop of the command "
                        "`{0}` . It took {1}sec but timeout is:{2}sec.".format(
                        cmd, time.time() - _start_timer, timeout))
            # No process is done, wait a bit and check again.
            time.sleep(.1)

    (stdout, stderr) = _popen.communicate()
    l_message = "OK done."
    if stdout or stderr:
        l_message += " Output:`{0}`, error:`{1}`.".format(stdout, stderr)
    logging.debug(l_message)
    return (stdout, stderr)


def compare_file_vs_hash(filename,hashfilename):
    """Compare a file with a MD5 hash file.

    :return: ``True`` if success, ``False`` otherwise.

    """
    with open(filename,'rb') as handler1, open(hashfilename,'r'
            ) as handler2:
        hash1 = hashlib.md5(handler1.read()).hexdigest()
        hash2 = handler2.readline().strip('\n')
        return (hash1 == hash2)


def compare_file_vs_file(filename1,filename2):
    """Compare two files by a MD5 hash.

    :return: ``True`` if success, ``False`` otherwise.

    """
    with open(filename1,'rb') as handler1, open(filename2,'rb') as handler2:
        hash1 = hashlib.md5(handler1.read()).hexdigest()
        hash2 = hashlib.md5(handler2.read()).hexdigest()
        return (hash1 == hash2)


def make_hash_file(filename,hashfilename):
    try:
        hash1 = hashlib.md5(open(filename,'rb').read()).hexdigest()
        hash_flag_file = open(hashfilename, 'w')
        hash_flag_file.write(hash1)
        hash_flag_file.close()
        return
    except Exception as e:
        logging.debug("make_hash_file :" + filename + " " + hashfilename+"  error: "+str(e))
        return


class Application():

    """A common application to be run by the daemon."""

    def __init__(self, name, log_dir='/var/tmp'):
        """Set up an instance.

        :param name: Script name for use with logging.
        :param log_dir: Folder to put log files.
        :return: ``None``.

        """
        self.name = name
        """daemon setUp"""
        self.stdin_path = '/dev/null'
        self.stdout_path = join(log_dir, self.name + '_stdout.log')
        self.stderr_path = join(log_dir, self.name + '_stderr.log')
        self.pidfile_path = join(log_dir, self.name + '.pid')
        self.pidfile_timeout = 5

    def run(self):
        """The part gets executed by daemon.

        To be redefined by child.

        """
        pass
