from urllib.parse import urlencode, parse_qsl
from datetime import datetime, timedelta
from pytz import UTC
from dateutil.parser import parse as parse_datetime
from sqlalchemy.exc import InternalError, IntegrityError, DataError
from sqlalchemy import and_, or_, inspect, alias, text, func, select, case, cast
from sqlalchemy.orm import aliased
from api_dnl import model
import falcon
from falcon_rest import conf
from falcon_rest.logger import log
from falcon_rest import schemes
from falcon_rest.responses import responses
from falcon_rest.responses import errors, ObjectCreatedResponse, SuccessResponseObjectInfo
from falcon_rest.resources.resources import swagger, ATTRIBUTE_ERROR_RE
from falcon_rest.helpers import check_permission
from api_dnl import settings
from api_dnl.resources import check_context, DnlResource,DnlList, ValidationError, DnlCreate, CustomGetAction, CustomPostAction, \
    CustomDeleteAction
from api_dnl.schemes.shaken import *
from api_dnl.scheme import SendCdrDirectScheme, ObjectCreatedMultipleScheme, ShakenStiSpConfScheme, ShakenStiSpConfSchemeModify, \
    ShakenStiSpConfSchemeGet
from api_dnl.view import DEFAULT_SECURITY, OperationErrorResponse
import json
from api_dnl.views.report import Report
import uuid
import requests

def generate_uuid_str():
    return lambda: str(uuid.uuid4())

# region +++C4ShakenConf+++

class C4ShakenConfManyCreate(DnlCreate):
    scheme_class = C4ShakenConfManyScheme
    model_class = model.C4ShakenConf
    entity = 'C4ShakenConf'
    path_parameters = ()
    security = (DEFAULT_SECURITY)
    restrict = ()
    additional_responses = (ObjectCreatedResponse(scheme=ObjectCreatedMultipleScheme),)

    def _on_post(self, req, resp, **kwargs):
        scheme = self.scheme_class()
        if not check_permission(self, req, 'create'):
            self.set_response(
                resp, responses.ForbiddenErrorResponse(data=errors.AuthErrors.Forbidden)
            )
            return
        try:
            if self.check_request_data(req, resp, scheme):
                ret = []
                for server in req.data['servers']:
                    singl_req = req
                    singl_req.data = server
                    create = C4ShakenConfCreate()
                    create.on_post(singl_req, resp, **kwargs)
                    if resp.status != falcon.HTTP_200:
                        return
                    object_id = json.loads(resp.body)['object_id']
                    ret.append(object_id)
                resp.data = {'success': True, 'object_id': ret}
                self.set_response(resp, ObjectCreatedResponse(scheme=ObjectCreatedMultipleScheme, data_key='object_ids',
                                                              data=ret))
                return True
        except Exception as e:
            self.set_response(resp, responses.OperationErrorResponse(e))
            return True


class C4ShakenConfManyResource(DnlResource):
    model_class = model.C4ShakenConf
    scheme_class = C4ShakenConfScheme
    scheme_class_get = C4ShakenConfManyScheme
    scheme_class_modify = C4ShakenConfManyScheme
    has_modify_operation = True
    has_delete_operation = False
    entity = 'C4ShakenConf'
    security = (DEFAULT_SECURITY)
    path_parameters = ()
    restrict = ()

    def get_path_parameters(self):
        return ({'name': 'id', 'description': 'id', 'type': 'string'},)

    def on_get(self, req, resp, **kwargs):
        scheme = self.scheme_class()
        if not check_permission(self, req, 'show'):
            self.set_response(
                resp, responses.ForbiddenErrorResponse(data=errors.AuthErrors.Forbidden)
            )
            return
        try:
            if self.check_request_data(req, resp, scheme):
                ret = []
                for id in kwargs['id'].split(','):
                    skwargs = kwargs
                    skwargs['id'] = id
                    get = C4ShakenConfResource()
                    get.on_get(req, resp, **skwargs)
                    if resp.status != falcon.HTTP_200:
                        return
                    object = json.loads(resp.body)['payload']
                    ret.append(object)
                resp.data = {'success': True, 'servers': ret}
                self.set_response(resp, SuccessResponseObjectInfo(data={'servers':ret}))
                return True
        except Exception as e:
            self.set_response(resp, responses.OperationErrorResponse(e))
            return True

    def on_patch(self, req, resp, **kwargs):
        scheme = self.scheme_class()
        if not check_permission(self, req, 'modify'):
            self.set_response(
                resp, responses.ForbiddenErrorResponse(data=errors.AuthErrors.Forbidden)
            )
            return
        try:
            if self.check_request_data(req, resp, scheme):
                ret = []
                for i in range(len(req.data['servers'])):
                    singl_req = req
                    singl_req.data = req.data['servers'][i]
                    skwargs = kwargs
                    skwargs['id'] = kwargs['id'].split(',')[i]
                    modify = C4ShakenConfResource()
                    modify._on_patch(singl_req, resp, **kwargs)
                    if resp.status != falcon.HTTP_200:
                        return
                    object = json.loads(resp.body)['payload']
                    ret.append(object)
                resp.data = {'success': True, 'servers': ret}
                self.set_response(resp, SuccessResponseObjectInfo(data={'servers':ret}))
                return True
        except Exception as e:
            self.set_response(resp, responses.OperationErrorResponse(e))
            return True


class C4ShakenConfCreate(DnlCreate):
    scheme_class = C4ShakenConfScheme
    model_class = model.C4ShakenConf
    entity = 'C4ShakenConf'
    path_parameters = ()
    security = (DEFAULT_SECURITY)
    restrict = ()

    def before_create(self, obj, **kwargs):
        user = self.get_user(self.req)
        # obj.created_by=user.name
        # obj.created_on=datetime.now(UTC)
        return obj


class C4ShakenConfResource(DnlResource):
    model_class = model.C4ShakenConf
    scheme_class = C4ShakenConfScheme
    scheme_class_get = C4ShakenConfSchemeGet
    scheme_class_modify = C4ShakenConfSchemeModify
    entity = 'C4ShakenConf'
    id_field = 'id'
    security = (DEFAULT_SECURITY)
    path_parameters = ()
    restrict = ()


class C4ShakenConfList(DnlList):
    scheme_class = C4ShakenConfSchemeGet
    model_class = model.C4ShakenConf
    entity_plural = 'C4ShakenConfs'
    path_parameters = ()
    security = (DEFAULT_SECURITY)
    restrict = ()

    def modify_query_from_filtering_for_list(self, filtering, **kwargs):
        filt, ret = super().modify_query_from_filtering_for_list(filtering, **kwargs)
        user = self.get_user(self.req)
        if not user.is_admin:
            cls = self.model_class

        return filt, ret


# endregion ---C4ShakenConf---

#region +++ShakenAniGroup+++
class ShakenAniGroupCreate(DnlCreate):
    scheme_class = ShakenAniGroupScheme
    model_class = model.ShakenAniGroup
    entity = 'ShakenAniGroup'
    path_parameters = ()
    security = (DEFAULT_SECURITY)
    restrict = ()
    def before_create(self, obj, **kwargs):
        user = self.get_user(self.req)
        obj.created_by=user.name
        obj.created_on=datetime.now(UTC)
        return obj
class ShakenAniGroupResource(DnlResource):
    model_class = model.ShakenAniGroup
    scheme_class = ShakenAniGroupScheme
    scheme_class_get = ShakenAniGroupSchemeGet
    scheme_class_modify = ShakenAniGroupSchemeModify
    entity = 'ShakenAniGroup'
    id_field = 'id'
    security = (DEFAULT_SECURITY)
    path_parameters = ()
    restrict = ()
class ShakenAniGroupList(DnlList):
    scheme_class = ShakenAniGroupSchemeGet
    model_class = model.ShakenAniGroup
    entity_plural = 'ShakenAniGroups'
    path_parameters = ()
    security = (DEFAULT_SECURITY)
    restrict = ()
    def modify_query_from_filtering_for_list(self, filtering, **kwargs):
        filt, ret = super().modify_query_from_filtering_for_list(filtering, **kwargs)
        user = self.get_user(self.req)
        if not user.is_admin:
            cls = self.model_class
            #ret = ret.filter(cls.pool_id != 0)#TODO:filter for user
        return filt, ret
#endregion ---ShakenAniGroup---

#region +++ShakenAniGroupList+++
class ShakenAniGroupListCreate(DnlCreate):
    scheme_class = ShakenAniGroupListScheme
    model_class = model.ShakenAniGroupList
    entity = 'ShakenAniGroupList'
    path_parameters = ()
    security = (DEFAULT_SECURITY)
    restrict = ()
    def before_create(self, obj, **kwargs):
        user = self.get_user(self.req)
        obj.created_by=user.name
        obj.created_on=datetime.now(UTC)
        return obj
class ShakenAniGroupListResource(DnlResource):
    model_class = model.ShakenAniGroupList
    scheme_class = ShakenAniGroupListScheme
    scheme_class_get = ShakenAniGroupListSchemeGet
    scheme_class_modify = ShakenAniGroupListSchemeModify
    entity = 'ShakenAniGroupList'
    id_field = 'id'
    security = (DEFAULT_SECURITY)
    path_parameters = ()
    restrict = ()
class ShakenAniGroupListList(DnlList):
    scheme_class = ShakenAniGroupListSchemeGet
    model_class = model.ShakenAniGroupList
    entity_plural = 'ShakenAniGroupLists'
    path_parameters = ()
    security = (DEFAULT_SECURITY)
    restrict = ()
    def modify_query_from_filtering_for_list(self, filtering, **kwargs):
        filt, ret = super().modify_query_from_filtering_for_list(filtering, **kwargs)
        user = self.get_user(self.req)
        if not user.is_admin:
            cls = self.model_class
            #ret = ret.filter(cls.pool_id != 0)#TODO:filter for user
        return filt, ret
#endregion ---ShakenAniGroupList---

#region +++ShakenAniGroupListRel+++
class ShakenAniGroupListRelCreate(DnlCreate):
    scheme_class = ShakenAniGroupListRelScheme
    model_class = model.ShakenAniGroupListRel
    entity = 'ShakenAniGroupListRel'
    unique_field = 'id'
    path_parameters = ()
    security = (DEFAULT_SECURITY)
    restrict = ()
    def before_create(self, obj, **kwargs):
        cls = model.ShakenAniGroupListRel
        user = self.get_user(self.req)
        l_obj = model.get_db().session.query(cls.id).order_by(cls.id.desc()).limit(1).first()
        obj.id = l_obj.id + 1 if l_obj else 1
        obj.created_by=user.name
        obj.created_on=datetime.now(UTC)
        return obj
class ShakenAniGroupListRelAdd(DnlCreate):
    scheme_class = ShakenAniGroupListRelAddScheme
    model_class = model.ShakenAniGroupListRel
    entity = 'ShakenAniGroupListRel'
    unique_field = 'id'
    path_parameters = ({'name':'id','type':'integer','description':'parent ani pool list id'},)
    security = (DEFAULT_SECURITY)
    restrict = ()
    def before_create(self, obj, id,**kwargs):
        pool=model.ShakenAniGroupList.get(id)
        if not pool:
            raise FileNotFoundError()
        cls = model.ShakenAniGroupListRel
        user = self.get_user(self.req)
        l_obj = model.get_db().session.query(cls.id).order_by(cls.id.desc()).limit(1).first()
        obj.id = l_obj.id + 1 if l_obj else 1
        obj.ani_group_list_id=id
        obj.created_by=user.name
        obj.created_on=datetime.now(UTC)
        return obj
class ResourceShakenAniGroupListRelAdd(DnlCreate):
    scheme_class = ShakenAniGroupListRelAddManyScheme
    unique_field = 'resource_id'
    entity = 'ShakenAniGroupListRel'
    path_parameters = ({'name':'resource_id','type':'integer','description':'parent resource id'},)
    security = (DEFAULT_SECURITY)
    restrict = ()
    def on_post(self, req, resp, **kwargs):
        user = self.get_user(req)
        resource_id = int(kwargs.pop('resource_id', 0))
        resource = model.Resource.get(resource_id)
        if not resource:
            self.set_response(resp, responses.ObjectNotFoundErrorResponse())
            return False
        if not resource.shaken_ani_group_list_id:
            shaken = model.ShakenAniGroupList()
            shaken.name = resource.alias
            shaken.created_by = user.name
            shaken_id = shaken.save()
            resource.shaken_ani_group_list_id = shaken_id
        kwargs['id'] = resource.shaken_ani_group_list_id
        attestation_settings = req.data.pop('attestation_settings', [])
        for attestation_setting in attestation_settings:
            req.data = attestation_setting
            ShakenAniGroupListRelAdd().on_post(req, resp, **kwargs)
class ShakenAniGroupListRelResource(DnlResource):
    model_class = model.ShakenAniGroupListRel
    scheme_class = ShakenAniGroupListRelScheme
    scheme_class_get = ShakenAniGroupListRelSchemeGet
    scheme_class_modify = ShakenAniGroupListRelSchemeModify
    entity = 'ShakenAniGroupListRel'
    id_field = 'id'
    security = (DEFAULT_SECURITY)
    path_parameters = ()
    restrict = ()

class ResourceShakenAniGroupListRelResource(ShakenAniGroupListRelResource):
    scheme_class_modify = ResourceShakenAniGroupListRelSchemeModify
    has_info_operation = False
    
class ShakenAniGroupListRelList(DnlList):
    scheme_class = ShakenAniGroupListRelSchemeGet
    model_class = model.ShakenAniGroupListRel
    entity_plural = 'ShakenAniGroupListRels'
    path_parameters = ()
    security = (DEFAULT_SECURITY)
    restrict = ()
    def modify_query_from_filtering_for_list(self, filtering, **kwargs):
        filt, ret = super().modify_query_from_filtering_for_list(filtering, **kwargs)
        user = self.get_user(self.req)
        if not user.is_admin:
            cls = self.model_class
            #ret = ret.filter(cls.pool_id != 0)#TODO:filter for user
        return filt, ret

# endregion ---ShakenAniGroupListRel---

class ShakenStiSpConfCreate(DnlCreate):
    scheme_class = ShakenStiSpConfScheme
    model_class = model.ShakenStiSpConf
    entity = 'ShakenStiSpConf'
    path_parameters = ()
    security = (DEFAULT_SECURITY)
    restrict = ()

    def get_spec(self):
        additional_responses = (
                                   responses.OperationErrorResponse(data=errors.CommonErrors.PermissionError,
                                                                    description='Can\'t create file'),
                               ) + self.additional_responses

        ext_body_parameters = (
            {'name': 'priv_key', 'in': 'formData', 'required': True, 'type': 'string'},
            {'name': 'passphrase', 'in': 'formData', 'required': False, 'type': 'string'},
            {'name': 'x5u', 'in': 'formData', "type": "string", 'required': False},
            {'name': 'default_key', 'in': 'formData', "type": "boolean"},
        )
        return swagger.specify.get_spec(
            method='post', description='Creates new {}'.format(self.entity.lower()),
            consumes=['multipart/form-data'],
            path_parameters=self.path_parameters,
            responses=(
                          # self.scheme_class.get_object_created_response(entity=self.entity),
                          responses.ObjectCreatedResponse(data_key='object_uuid', scheme=schemes.ObjectCreatedUuidAsPk),
                          responses.SuccessResponseJustOk()

                      ) + additional_responses + self.get_additional_responses(method='post'),
            security=self.get_security(method='post'),
            # body_parameters=('{} to create'.format(self.entity), self.scheme_class),
            ext_body_parameters=ext_body_parameters
        )

    def on_post(self, req, resp, **kwargs):
        default_key = req.data.get("default_key") == "true"
        session = model.get_db().session
        if not default_key and not session.query(model.ShakenStiSpConf).first():
            log.debug('No records in db, set default_key=true')
            default_key = True
        try:
            priv_key = req.data.get('priv_key')
            if not ShakenStiSpConfResource.validate_priv_key(priv_key):
                self.set_response(
                    resp,
                    responses.ValidationErrorResponse(data={'priv_key': 'bad value'})
                )
                return False
            x5u = req.data.get('x5u')
            if x5u is not None and not ShakenStiSpConfResource.validate_x5u(x5u):
                self.set_response(
                    resp,
                    responses.ValidationErrorResponse(data={'x5u': 'bad value'})
                )
                return False
            if default_key:
                sql = text("""
                    WITH reset AS (
                        UPDATE public.shaken_sti_sp_conf
                        SET default_key = NULL
                        WHERE default_key = TRUE
                    )
                    INSERT INTO public.shaken_sti_sp_conf (priv_key, passphrase, x5u, default_key)
                    VALUES (:priv_key, :passphrase, :x5u, TRUE)
                    RETURNING id;
                """)
                result = session.execute(sql, {
                    "priv_key": req.data.get("priv_key"),
                    "passphrase": req.data.get("passphrase"),
                    "x5u": req.data.get("x5u"),
                })
                new_id = result.scalar()

            else:
                sql = text("""
                    INSERT INTO public.shaken_sti_sp_conf (priv_key, passphrase, x5u, default_key)
                    VALUES (:priv_key, :passphrase, :x5u, NULL)
                    RETURNING id;
                """)
                result = session.execute(sql, {
                    "priv_key": req.data.get("priv_key"),
                    "passphrase": req.data.get("passphrase"),
                    "x5u": req.data.get("x5u"),
                })
                new_id = result.scalar()

            session.commit()
            self.set_response(resp, responses.SuccessResponse(
                data={
                    'object_id': new_id,
                },
                scheme=schemes.ObjectScheme
            ))
            return True

        except Exception as e:
            session.rollback()
            log.error('Error during processing: {}'.format(str(e)))
            self.set_response(resp, OperationErrorResponse(e))
            return False


class ShakenStiSpConfResource(DnlResource):
    model_class = model.ShakenStiSpConf
    scheme_class = ShakenStiSpConfScheme
    scheme_class_get = ShakenStiSpConfSchemeGet
    scheme_class_modify = ShakenStiSpConfSchemeModify
    entity = 'ShakenStiSpConf'
    id_field = 'id'
    security = (DEFAULT_SECURITY)
    restrict = ()


    def on_patch(self, req, resp, **kwargs):
        session = model.get_db().session
        default_key = req.data.get('default_key')
        object_id = kwargs['id']
        
        try:
            priv_key = req.data.get('priv_key')
            if priv_key and not self.validate_priv_key(priv_key):
                self.set_response(
                    resp,
                    responses.ValidationErrorResponse(data={'priv_key': 'bad value'})
                )
                return False
            x5u = req.data.get('x5u')
            if x5u is not None and not self.validate_x5u(x5u):
                self.set_response(
                    resp,
                    responses.ValidationErrorResponse(data={'x5u': 'bad value'})
                )
                return False

            total = session.query(model.ShakenStiSpConf).count()
            if total == 1 and default_key is False:
                self.set_response(
                    resp,
                    responses.ValidationErrorResponse(
                        data={'default_key': 'cannot unset default_key; only one record exists'}
                    )
                )
                return False

            if default_key:
                sql = text("""
                    WITH reset AS (
                        UPDATE public.shaken_sti_sp_conf
                        SET default_key = NULL
                        WHERE default_key = TRUE AND id != :id
                    )
                    UPDATE public.shaken_sti_sp_conf
                    SET priv_key = COALESCE(:priv_key, priv_key),
                        passphrase = COALESCE(:passphrase, passphrase),
                        x5u = COALESCE(:x5u, x5u),
                        default_key = TRUE
                    WHERE id = :id
                    RETURNING id;
                """)
            else:
                sql = text("""
                    UPDATE public.shaken_sti_sp_conf
                    SET priv_key = COALESCE(:priv_key, priv_key),
                        passphrase = COALESCE(:passphrase, passphrase),
                        x5u = COALESCE(:x5u, x5u)
                    WHERE id = :id
                    RETURNING id;
                """)

            result = session.execute(sql, {
                "id": object_id,
                "priv_key": req.data.get("priv_key"),
                "passphrase": req.data.get("passphrase"),
                "x5u": req.data.get("x5u"),
            })
            session.commit()

            updated_id = result.scalar()
            if updated_id:
                obj_data = self.get_object_data(resp, self.model_class, self.scheme_class_get, id=updated_id)
                self.set_response(resp, responses.SuccessResponseObjectInfo(data=obj_data))
                return True
            else:
                self.set_response(resp, responses.ObjectNotFoundErrorResponse())
                return False

        except Exception as e:
            session.rollback()
            log.error('Error during processing: {}'.format(str(e)))
            self.set_response(resp, OperationErrorResponse(e))
            return False


    def on_delete(self, req, resp, **kwargs):
        result = super(ShakenStiSpConfResource, self).on_delete(req, resp, **kwargs)

        if resp.status != 200 and resp.status != '200 OK':
            return result

        session = model.get_db().session

        try:
            has_default = (
                session.query(model.ShakenStiSpConf)
                .filter(model.ShakenStiSpConf.default_key == True)
                .first()
            )

            if has_default is None:
                last_row = (
                    session.query(model.ShakenStiSpConf)
                    .order_by(model.ShakenStiSpConf.id.desc())
                    .first()
                )

                if last_row:
                    last_row.default_key = True
                    session.commit()

        except Exception as e:
            session.rollback()
            log.error(f"Failed to restore default_key: {e}")

        return result

    @staticmethod
    def validate_priv_key(priv_key):
        try:
            if 'BEGIN EC PARAMETERS' in priv_key and 'END EC PARAMETERS' in priv_key and \
                    'BEGIN EC PRIVATE KEY' in priv_key and 'END EC PRIVATE KEY' in priv_key:
                return True
            else:
                log.warning('Private key markers not found or invalid format')
                return False
        except Exception as e:
            log.error('Error fetching priv key: {}'.format(str(e)))
            return False

    @staticmethod
    def validate_x5u(x5u):
        try:
            if x5u.startswith('http'):
                return True
            else:
                log.warning('Certificate not valid. {}'.format(x5u))
                return False
        except Exception as e:
            log.error('Error fetching certificate: {}'.format(str(e)))
            return False


class ShakenStiSpConfList(DnlList):
    scheme_class = ShakenStiSpConfSchemeGet
    model_class = model.ShakenStiSpConf
    entity_plural = 'ShakenStiSpConfs'
    path_parameters = ()
    security = (DEFAULT_SECURITY)
    restrict = ()



# region +++ShakenAniGroupRel+++


class ShakenAniGroupRelCreate(DnlCreate):
    scheme_class = ShakenAniGroupRelScheme
    model_class = model.ShakenAniGroupRel
    entity = 'ShakenAniGroupRel'
    path_parameters = ()
    security = (DEFAULT_SECURITY)
    restrict = ()

    def before_create(self, obj, **kwargs):
        user = self.get_user(self.req)
        obj.created_by = user.name
        #obj.created_on=datetime.now(UTC)
        return obj


class ShakenAniGroupRelResource(DnlResource):
    model_class = model.ShakenAniGroupRel
    scheme_class = ShakenAniGroupRelScheme
    scheme_class_get = ShakenAniGroupRelSchemeGet
    scheme_class_modify = ShakenAniGroupRelSchemeModify
    entity = 'ShakenAniGroupRel'
    id_field = 'group_id'
    security = (DEFAULT_SECURITY)
    path_parameters = ({'name':'did'},)
    restrict = ()
    has_modify_operation = False


class ShakenAniGroupRelList(DnlList):
    scheme_class = ShakenAniGroupRelSchemeGet
    model_class = model.ShakenAniGroupRel
    entity_plural = 'ShakenAniGroupRels'
    path_parameters = ()
    security = (DEFAULT_SECURITY)
    restrict = ()

    def modify_query_from_filtering_for_list(self, filtering, **kwargs):
        filt, ret = super().modify_query_from_filtering_for_list(filtering, **kwargs)
        user = self.get_user(self.req)
        if not user.is_admin:
            cls = self.model_class
            #ret = ret.filter(cls.pool_id != 0)#TODO:filter for user
        return filt, ret


class ShakenAniGroupRelNumbersDelete(CustomDeleteAction):
    model_class = model.ShakenAniGroup
    scheme_class = ShakenAniGroupSchemeGet
    entity = 'ShakenAniGroupRel'
    security = (DEFAULT_SECURITY)
    path_parameters = ({'name': 'group_id', 'description': 'ShakenAniGroup ids seperated by ,'},)
    scheme_class_get = ShakenAniGroupSchemeGet
    has_modify_operation = False

    def apply(self, obj, req, resp, **kwargs):
        from kombu.exceptions import OperationalError
        group_ids = kwargs['group_id']
        try:
            for group_id in group_ids.split(','):
                model.ShakenAniGroupRel.filter(model.ShakenAniGroupRel.group_id == group_id).delete(synchronize_session='fetch')
        except Exception as e:
            self.set_response(resp, OperationErrorResponse(e))
        self.set_response(resp, responses.SuccessResponseJustOk())
        return False
#endregion ---ShakenAniGroupRel---

# region +++ShakenAniGroupRelImportTask+++
class ShakenAniGroupRelImportTaskCreate(DnlCreate):
    scheme_class = ShakenAniGroupRelImportTaskScheme
    model_class = model.ShakenAniGroupRelImportTask
    entity = 'ShakenAniGroupRelImportTask'
    path_parameters = ()
    security = (DEFAULT_SECURITY)
    restrict = ()

    def get_spec(self):
        additional_responses = (
                                   responses.OperationErrorResponse(data=errors.CommonErrors.PermissionError,
                                                                    description='Can\'t create file'),
                               ) + self.additional_responses

        ext_body_parameters = (
            {'name': 'file', 'in': 'formData', 'description': 'File to upload', 'required': False, 'type': 'file'},
            {'name': 'group_id', 'in': 'formData', 'description': 'Specified group', 'required': False,
             'type': 'integer'},
            {'name': 'op_method', 'in': 'formData', "type": "string", "enum": ['Import', 'Number Range', 'Paste']},
            {'name': 'action', 'in': 'formData', "type": "string", "enum": ['Add', 'Delete']},
            {'name': 'data', 'in': 'formData', 'description': 'paste did list', 'required': False,
             'type': 'string'},
            {'name': 'first_number', 'in': 'formData', 'description': 'Start number range', 'required': False,
             'type': 'integer'},
            {'name': 'last_number', 'in': 'formData', 'description': 'End number range', 'required': False,
             'type': 'integer'},
        )
        return swagger.specify.get_spec(
            method='post', description='Creates new {}'.format(self.entity.lower()),
            consumes=['multipart/form-data'],
            path_parameters=self.path_parameters,
            responses=(
                          # self.scheme_class.get_object_created_response(entity=self.entity),
                          responses.ObjectCreatedResponse(data_key='object_uuid', scheme=schemes.ObjectCreatedUuidAsPk),
                          responses.SuccessResponseJustOk()

                      ) + additional_responses + self.get_additional_responses(method='post'),
            security=self.get_security(method='post'),
            # body_parameters=('{} to create'.format(self.entity), self.scheme_class),
            ext_body_parameters=ext_body_parameters
        )

    def before_create(self, obj, **kwargs):

        user = self.get_user(self.req)
        obj.operator_user = user.name
        obj.upload_file_path = settings.FILES['upload_to']
        obj.repeated_action = self.req_data.get('action', 'Add')
        obj.op_method = self.req_data.get('op_method', 'Import')
        obj.group_id = self.req_data.get('group_id', None)
        if obj.group_id is None or model.ShakenAniGroup.get(obj.group_id) is None:
            raise Exception('Not found any group id {}'.format(obj.group_id))
        file = None
        if obj.op_method == 'Import':
            file = self.req.files['file']
        data = ''
        if obj.op_method == 'Paste':
            data = self.req_data.get('data', '')

        obj.first_number = int(self.req_data.get('first_number', 0))
        obj.last_number = int(self.req_data.get('last_number', 0))
        if obj.op_method == 'Number Range':
            if obj.last_number < obj.first_number:
                tmp=obj.last_number
                obj.last_number = obj.first_number
                obj.first_number = tmp
            if obj.last_number - obj.first_number > 1000000:
                raise Exception("Too big number range {} {}".format(obj.last_number,obj.first_number))
            if obj.last_number == obj.first_number:
                raise Exception("Empty number range {} {}".format(obj.last_number, obj.first_number))
        else:
            if obj.last_number or  obj.first_number:
                raise Exception("Number range {} {} not used when op method=".format(obj.last_number, obj.first_number,obj.op_method))

        uuid = generate_uuid_str()()
        # obj.upload_orig_file = file.filename

        obj.upload_orig_file = uuid + '.csv'
        obj.upload_format_file = uuid + '.csv'
        file_name = obj.upload_file_path + '/' + obj.upload_format_file
        if obj.op_method == 'Import':
            obj.orig_name = file.filename
            b = file.readlines()
        elif obj.op_method == 'Number Range':
            obj.orig_name = 'range_{}_{}.csv'.format(obj.first_number,obj.last_number)
            b = range(obj.first_number,obj.last_number+1)
            b = [(str(i)+'\n').encode() for i in b]
        elif obj.op_method == 'Paste':
            obj.orig_name = 'pasted_data.csv'
            b = data.split(',')
            b = [(i+'\n').encode() for i in b if i.strip()]
        obj.num_records = len(b)
        log.info('Uploaded file rows:{}'.format(obj.num_records))
        local_file = open(file_name, 'wb')
        local_file.writelines(b)
        local_file.close()


        obj.status = 'Initial'
        obj.progress = 'Starting'
        return obj

    def after_create(self, object_id, req, resp, **kwargs):
        from kombu.exceptions import OperationalError
        from api_dnl.task.did_import_file import shaken_ani_group_rel_import
        try:
            ret = shaken_ani_group_rel_import.delay(object_id)
            log.debug('shaken_ani_group_rel_import scheduled {}'.format(ret))
        except OperationalError:
            log.debug('shaken_ani_group_rel_import start without celery')
            shaken_ani_group_rel_import(object_id)


class ShakenAniGroupRelImportTaskResource(DnlResource):
    model_class = model.ShakenAniGroupRelImportTask
    scheme_class = ShakenAniGroupRelImportTaskScheme
    scheme_class_get = ShakenAniGroupRelImportTaskSchemeGet
    entity = 'ShakenAniGroupRelImportTask'
    id_field = 'id'
    security = (DEFAULT_SECURITY)
    path_parameters = ()
    restrict = ()
    has_delete_operation = False
    has_modify_operation = False


class ShakenAniGroupRelImportTaskWarningsResource(ShakenAniGroupRelImportTaskResource):
    scheme_class_get = ShakenAniGroupRelImportTaskWarningsSchemeGet


class ShakenAniGroupRelImportTaskList(DnlList):
    scheme_class = ShakenAniGroupRelImportTaskSchemeGet
    model_class = model.ShakenAniGroupRelImportTask
    entity_plural = 'ShakenAniGroupRelImportTasks'
    path_parameters = ()
    security = (DEFAULT_SECURITY)
    restrict = ()

    def modify_query_from_filtering_for_list(self, filtering, **kwargs):
        filt, ret = super().modify_query_from_filtering_for_list(filtering, **kwargs)
        user = self.get_user(self.req)
        if not user.is_admin:
            cls = self.model_class
            ret = ret.filter(cls.operator_user==user.name)
        return filt, ret
    
    def on_get(self, req, resp, **kwargs):
        parsed_qs = dict(parse_qsl(req.query_string))
        order_by = None
        if 'order_by' in parsed_qs and parsed_qs['order_by'] in ('success', 'total', 'fail'):
            order_by = parsed_qs.pop('order_by')
            order_dir = parsed_qs.pop('order_dir', 'asc') == 'desc'
        super().on_get(req, resp, **kwargs)
        if order_by and 'payload' in req.data and 'items' in req.data['payload']:
            items = req.data['payload']['items']
            items.sort(key=lambda item: item[order_by], reversed=order_dir)
            req.data['payload']['items'] = items



# endregion ---ShakenAniGroupRelImportTask---

### end ###
