-- Class v6 tables:
--
-- Remote storage configuration:
--  - dnl_cloud_ftp_cfg - FTP connection settings
--  - dnl_cloud_sftp_cfg - SFTP connection settings
--  - dnl_cloud_gcloud_cfg - Google Cloud connection settings
--  - dnl_cloud_storages - List of all storage configurations
--
-- Apps configuration:
--  - dnl_cloud_dbman_cfg - DB manager configuration
--  - dnl_cloud_downloader_cfg - Data downloader configuration
--  - dnl_cloud_search_cfg - DB search configuration
--
-- (Read-only) Status, logging and versioning:
--  - dnl_cloud_log - log messages from all applications
--  - dnl_cloud_status - status of applications within multiple servers
--

CREATE TABLE IF NOT EXISTS dnl_cloud_storages (
  uuid text PRIMARY KEY CHECK ((TRIM(uuid) = '') IS FALSE),
  type character varying(16) CHECK(type = 'ftp' OR type = 'sftp' OR type = 'gcloud')
);

CREATE TABLE IF NOT EXISTS dnl_cloud_ftp_cfg (
  uuid text PRIMARY KEY REFERENCES dnl_cloud_storages(uuid) ON DELETE CASCADE,

-- Server bind address
  ip character varying(16) NOT NULL CHECK ((TRIM(ip) = '') IS FALSE),
  port integer NOT NULL CHECK (port > 0 AND port < 65536),

-- Authorization
  username text DEFAULT NULL,
  password text DEFAULT NULL,

-- Path to workdir
  workdir text DEFAULT NULL,

-- Use relative paths in QUOT commands (required for MS FTP servers)
  use_relative_paths boolean DEFAULT FALSE,
-- Path to .netrc file (authorization without login/password pair)
  netrc_path text DEFAULT NULL,

-- Enable additional log messages:
--  0 - nothing
--  1 - enable debug and verbose debug messages
--  2 - include FTP data trace
  debug integer DEFAULT NULL CHECK (debug >= 0 AND debug < 3),

-- Authorization method constraint
  CHECK ( ( TRIM(username) = 'anonymous' AND password IS NULL ) OR
	  ( (TRIM(username) = '') IS FALSE AND (TRIM(password) = '') IS FALSE ) OR
	  ( username IS NULL AND password IS NULL AND (TRIM(netrc_path) = '') IS FALSE )
	)
);

CREATE TABLE IF NOT EXISTS dnl_cloud_sftp_cfg (
  uuid text PRIMARY KEY REFERENCES dnl_cloud_storages(uuid) ON DELETE CASCADE,

-- Server bind address
  ip character varying(16) NOT NULL CHECK ((TRIM(ip) = '') IS FALSE),
  port integer NOT NULL CHECK (port > 0 AND port < 65536),

-- Authorization
  username text DEFAULT NULL,
  password text DEFAULT NULL,

-- Path to workdir
  workdir text DEFAULT NULL,

-- Authorization with private key
-- NOTE: app will try to extract public key from private, if pubkey_path is not provided
  pubkey_path text DEFAULT NULL,
  privkey_path text DEFAULT NULL,
  passphrase text DEFAULT NULL,

-- Enable additional log messages:
--  0 - nothing
--  1 - enable debug and verbose debug messages
--  2 - include SFTP data trace
  debug integer DEFAULT NULL CHECK (debug >= 0 AND debug < 3),

-- Authorization method constraint
  CHECK ( ( (TRIM(username) = '') IS FALSE AND (TRIM(password) = '') IS FALSE ) OR
	  ( username IS NULL AND password IS NULL AND (TRIM(privkey_path) = '') IS FALSE )
	)
);

CREATE TABLE IF NOT EXISTS dnl_cloud_gcloud_cfg (
  uuid text PRIMARY KEY REFERENCES dnl_cloud_storages(uuid) ON DELETE CASCADE,

-- Connection type: 'service', 'oauth2'
  type character varying(16) NOT NULL,

-- Bucket & path to workdir
  bucket text NOT NULL CHECK ((TRIM(bucket) = '') IS FALSE),
  workdir text DEFAULT NULL,

-- Service account configuration
  key_file_path text DEFAULT NULL,

-- Oauth2 configuration
  email text DEFAULT NULL,
  secret text DEFAULT NULL,
  token text DEFAULT NULL,

-- Enable additional log messages:
--  0 - nothing
--  1 - enable debug and verbose debug messages
--  2 - include HTTP data trace
  debug integer DEFAULT NULL CHECK (debug >= 0 AND debug < 3),

-- Authorization method constraint
  CHECK ( ( type = 'service' AND (TRIM(key_file_path) = '') IS FALSE ) OR
	  ( type = 'oauth2' AND (TRIM(email) = '') IS FALSE AND (TRIM(secret) = '') IS FALSE AND (TRIM(token) = '') IS FALSE )
	)
);

CREATE TABLE IF NOT EXISTS dnl_cloud_dbman_cfg (
-- Server ID
  server_name character varying(256) PRIMARY KEY,

-- Link to remote storage configuration
-- Default (NULL): do not connect remote storage.
  cdr_storage_cfg_uuid text DEFAULT NULL REFERENCES dnl_cloud_storages(uuid),
  pcap_storage_cfg_uuid text DEFAULT NULL REFERENCES dnl_cloud_storages(uuid),

-- Compress CDR files older than specified amount of days
-- for CDR and PCAP files respectively.
-- Default (NULL or 0): do not compress local files.
  keep_cdr integer DEFAULT NULL CHECK (keep_cdr > 0),
  keep_pcap integer DEFAULT NULL CHECK (keep_pcap > 0),

-- Remove local archives older than specified amount of days
-- for CDR and PCAP files respectively.
-- Default (NULL or 0): do not remove local archives.
  keep_cdr_archives integer DEFAULT NULL CHECK (keep_cdr_archives > 0),
  keep_pcap_archives integer DEFAULT NULL CHECK (keep_pcap_archives > 0),

-- Remove remote archives older than specified amount of days
-- for CDR and PCAP files respectively.
-- Default (NULL or 0): do not remove remote archives.
  keep_cdr_remote integer DEFAULT NULL CHECK (keep_cdr_remote > 0),
  keep_pcap_remote integer DEFAULT NULL CHECK (keep_pcap_remote > 0),

-- Upload temporary CDR backup every hour
  hourly_cdr_backup boolean NOT NULL DEFAULT FALSE,

--
-- HTTP Server
--

-- Bind address
-- Default (NULL): 127.0.0.1
  server_ip character varying(16) DEFAULT NULL,

-- Server port
-- Default (NULL): 33500
  server_port integer DEFAULT NULL CHECK (server_port > 0 AND server_port < 65536),

-- Enable SSL
  server_use_ssl boolean NOT NULL DEFAULT FALSE,

-- Firewall
-- Set comma-separated list of patterns to allow or to block.
-- NOTE: naturally you cannot have both lists enabled.
-- Default (NULL): allow all connections
  server_ip_whitelist text DEFAULT NULL,
  server_ip_blacklist text DEFAULT NULL,

-- Maximum amount of simultaneous connections
-- Default (NULL): unlimited
  server_con_limit integer DEFAULT NULL CHECK (server_con_limit > 0),

-- Maximum amount of simultaneous connections from the same IP address.
-- Default (NULL): unlimited
  server_per_ip_con_limit integer DEFAULT NULL CHECK (server_per_ip_con_limit > 0),

-- Connection inactivity timeout (in seconds)
-- Default (NULL): unlimited
  server_con_timeout integer DEFAULT NULL CHECK (server_con_timeout > 0),

--
-- Advanced settings
--

-- Verbosity level
--  -1 - Disable log messages
--   0 - INFO, WARNING, and ERROR messages
--   1 - DEBUG messages
--   2 - Verbose DEBUG messages
--
-- NOTE: this setting cannot _decrease_ logging level for syslogd
-- and logfile output, which shall be set from the command-line arguments.
-- But nevertheless it may _increase_ it.
-- The logic behind logger module is following:
--  - For PostgreSQL logging: use `log_level` from this table
--  - For syslogd and logfile: use max value of `log_level` and command-line arguments.
  log_level integer DEFAULT 0 CHECK (log_level >= -1 AND log_level < 3),

-- Log files location.
-- Default (NULL): /workdir/log
  log_dir_path text DEFAULT NULL,

-- Disable log file creation
  disable_logfile boolean DEFAULT FALSE,

-- Enable logging to dnl_cloud_log table
  enable_postgresql_log boolean DEFAULT FALSE,

-- Path to directory to keep temporary files.
-- Default (NULL): /workdir/tmp
  tmp_dir_path text DEFAULT NULL,

-- When set, daemon will create temporary archives needed for uploading
-- right in the CDR and PCAP directories for CDR and PCAP backup operations
-- respectively.
-- Otherwise `tmp_dir_path` will be used for storing temporary files.
  disable_tmp_dir boolean DEFAULT FALSE,

-- Amount of seconds to wait after midnight, before waking up the backup thread
-- NOTE: 0 value will run thread at 00:00:59. The minimum delay is required
-- to let switch finish updating latest data.
-- Default (NULL): 10800 - 03:00 AM
  startup_delay integer DEFAULT NULL CHECK (startup_delay >= 0 AND startup_delay < 86400),

-- Run backup thread immediately on start.
-- Otherwise run as scheduled in startup_delay.
  run_on_start boolean NOT NULL DEFAULT FALSE,

-- HTTP server configuration constraint
  CHECK ( (server_ip_whitelist IS NOT NULL AND server_ip_blacklist IS NOT NULL) IS FALSE )
);

CREATE TABLE IF NOT EXISTS dnl_cloud_downloader_cfg (
-- Server ID
  server_name character varying(256) PRIMARY KEY,

--
-- HTTP Server
--

-- Bind address
-- Default (NULL): 127.0.0.1
  server_ip character varying(16) DEFAULT NULL,

-- Server port
-- Default (NULL): 33501
  server_port integer DEFAULT NULL CHECK (server_port > 0 AND server_port < 65536),

-- Enable SSL
  server_use_ssl boolean NOT NULL DEFAULT FALSE,

-- Firewall
-- Set comma-separated list of patterns to allow or to block.
-- NOTE: naturally you cannot have both lists enabled.
-- Default (NULL): allow all connections
  server_ip_whitelist text DEFAULT NULL,
  server_ip_blacklist text DEFAULT NULL,

-- Maximum amount of simultaneous connections
-- Default (NULL): unlimited
  server_con_limit integer DEFAULT NULL CHECK (server_con_limit > 0),

-- Maximum amount of simultaneous connections from the same IP address.
-- Default (NULL): unlimited
  server_per_ip_con_limit integer DEFAULT NULL CHECK (server_per_ip_con_limit > 0),

-- Connection inactivity timeout (in seconds)
-- Default (NULL): unlimited
  server_con_timeout integer DEFAULT NULL CHECK (server_con_timeout > 0),

--
-- Advanced settings
--

-- Verbosity level
--  -1 - Disable log messages
--   0 - INFO, WARNING, and ERROR messages
--   1 - DEBUG messages
--   2 - Verbose DEBUG messages
--
-- NOTE: this setting cannot _decrease_ logging level for syslogd
-- and logfile output, which shall be set from the command-line arguments.
-- But nevertheless it may _increase_ it.
-- The logic behind logger module is following:
--  - For PostgreSQL logging: use `log_level` from this table
--  - For syslogd and logfile: use max value of `log_level` and command-line arguments.
  log_level integer DEFAULT 0 CHECK (log_level >= -1 AND log_level < 3),

-- Log files location.
-- Default (NULL): /workdir/log
  log_dir_path text DEFAULT NULL,

-- Disable log file creation
  disable_logfile boolean DEFAULT FALSE,

-- Enable logging to dnl_cloud_log table
  enable_postgresql_log boolean DEFAULT FALSE,

-- Path to directory to keep temporary files.
-- Default (NULL): /workdir/tmp
  tmp_dir_path text DEFAULT NULL,

-- When set, daemon will create temporary archives when sending local uncompressed files
-- right in the CDR and PCAP directories for CDR and PCAP operations
-- respectively.
-- Otherwise `tmp_dir_path` will be used for storing temporary files.
  disable_tmp_dir boolean DEFAULT FALSE,

-- HTTP server configuration constraint
  CHECK ( (server_ip_whitelist IS NOT NULL AND server_ip_blacklist IS NOT NULL) IS FALSE )
);

CREATE TABLE IF NOT EXISTS dnl_cloud_search_cfg (
-- Server ID
  server_name character varying(256) PRIMARY KEY,

-- Time-window margin in seconds.
-- This value is used to expand pcap search time-window,
-- which is necessary to ensure, that call has ended.
-- Default (NULL): 60 seconds (recommended value).
  pcap_time_margin integer DEFAULT NULL CHECK (pcap_time_margin >= 0),

-- Request expiration time _after completion_ in seconds.
-- Expired requests will be removed from the database
-- automatically with all the data.
-- If set to 0, daemon will not cleanup requests.
-- Default (NULL): 604800 (7 days).
  req_exp_time integer DEFAULT NULL CHECK (req_exp_time >= 0),

-- Maximum amount of simultaneous processing threads
-- Default (NULL): 0 - unlimited
  threads_max integer DEFAULT NULL CHECK (threads_max >= 0),

--
-- HTTP Server
--

-- Bind address
-- Default (NULL): 127.0.0.1
  server_ip character varying(16) DEFAULT NULL,

-- Server port
-- Default (NULL): 33502
  server_port integer DEFAULT NULL CHECK (server_port > 0 AND server_port < 65536),

-- Enable SSL
  server_use_ssl boolean NOT NULL DEFAULT FALSE,

-- Firewall
-- Set comma-separated list of patterns to allow or to block.
-- NOTE: naturally you cannot have both lists enabled.
-- Default (NULL): allow all connections
  server_ip_whitelist text DEFAULT NULL,
  server_ip_blacklist text DEFAULT NULL,

-- Maximum amount of simultaneous connections
-- Default (NULL): unlimited
  server_con_limit integer DEFAULT NULL CHECK (server_con_limit > 0),

-- Maximum amount of simultaneous connections from the same IP address.
-- Default (NULL): unlimited
  server_per_ip_con_limit integer DEFAULT NULL CHECK (server_per_ip_con_limit > 0),

-- Connection inactivity timeout (in seconds)
-- Default (NULL): unlimited
  server_con_timeout integer DEFAULT NULL CHECK (server_con_timeout > 0),

--
-- Advanced settings
--

-- Verbosity level
--  -1 - Disable log messages
--   0 - INFO, WARNING, and ERROR messages
--   1 - DEBUG messages
--   2 - Verbose DEBUG messages
--
-- NOTE: this setting cannot _decrease_ logging level for syslogd
-- and logfile output, which shall be set from the command-line arguments.
-- But nevertheless it may _increase_ it.
-- The logic behind logger module is following:
--  - For PostgreSQL logging: use `log_level` from this table
--  - For syslogd and logfile: use max value of `log_level` and command-line arguments.
  log_level integer DEFAULT 0 CHECK (log_level >= -1 AND log_level < 3),

-- Log files location.
-- Default (NULL): /workdir/log
  log_dir_path text DEFAULT NULL,

-- Disable log file creation
  disable_logfile boolean DEFAULT FALSE,

-- Enable logging to dnl_cloud_log table
  enable_postgresql_log boolean DEFAULT FALSE,

-- Path to directory to keep temporary files.
-- Default (NULL): /workdir/tmp
  tmp_dir_path text DEFAULT NULL,

-- HTTP server configuration constraint
  CHECK ( (server_ip_whitelist IS NOT NULL AND server_ip_blacklist IS NOT NULL) IS FALSE )
);

CREATE TABLE IF NOT EXISTS dnl_cloud_log (
    id SERIAL PRIMARY KEY,
    time timestamp NOT NULL DEFAULT now(),

-- Server ID
    server_name character varying(256) NOT NULL,

-- Application type
    instance character varying(128) NOT NULL,

-- Logging level (PANIC, INFO, WARNING, ERROR, DEBUG, ...)
    level character varying(16) NOT NULL CHECK ((TRIM(level) = '') IS FALSE),
-- Log message
    msg text NOT NULL
);

CREATE TABLE IF NOT EXISTS dnl_cloud_status (
-- Server ID
    server_name character varying(256) NOT NULL,

-- Application type
    instance character varying(128) NOT NULL,

-- Current operation in progress
    operation text DEFAULT NULL,

-- HTTP server status
    is_online boolean NOT NULL DEFAULT FALSE,

-- Status update time
    upd_time timestamp NOT NULL DEFAULT now(),

-- App bind address
    server_ip character varying(16) NOT NULL CHECK ((TRIM(server_ip) = '') IS FALSE),
    server_port integer DEFAULT NULL CHECK (server_port > 0 AND server_port < 65536),
    server_use_ssl boolean NOT NULL,

-- List of connected switches
    switch_list text DEFAULT NULL,

-- Primary key
    PRIMARY KEY (server_name, instance)
);
