PRAGMA foreign_keys = ON;

DROP TABLE IF EXISTS db_info;
DROP TABLE IF EXISTS backup_files;
DROP TABLE IF EXISTS local_files;
DROP TABLE IF EXISTS remote_files;
DROP TABLE IF EXISTS data_index;
DROP TABLE IF EXISTS switch_connections;
DROP TABLE IF EXISTS storage_connections;
DROP TABLE IF EXISTS search_requests;


-- Database information
--
-- Each new update shall increment version number by 1.
-- On start check db version and upgrade database if necessary.

CREATE TABLE db_info (
  id BOOLEAN NOT NULL PRIMARY KEY DEFAULT TRUE CHECK (id),
  version INTEGER NOT NULL DEFAULT 3
);
INSERT INTO db_info DEFAULT VALUES;


-- List of switches:
-- Every item in files db must reference the corresponding switch.

CREATE TABLE switch_connections (
  uuid TEXT PRIMARY KEY NOT NULL CHECK ((TRIM(uuid) = '') IS FALSE),

-- Identification for search
  name TEXT UNIQUE NOT NULL CHECK ((TRIM(name) = '') IS FALSE),

-- Working directories
  cdr_dir TEXT UNIQUE NOT NULL CHECK ((TRIM(cdr_dir) = '') IS FALSE),
  pcap_dir TEXT UNIQUE NOT NULL CHECK ((TRIM(pcap_dir) = '') IS FALSE),

-- Mark if was auto-detected
  detected BOOLEAN NOT NULL DEFAULT FALSE
);


-- List of remote storages
-- Every item in remote files db must reference corresponding storage.
-- The 'uuid' field references storage from PostgreSQL structures.

CREATE TABLE storage_connections (
  uuid TEXT PRIMARY KEY NOT NULL CHECK ((TRIM(uuid) = '') IS FALSE)
);


-- Data index
-- Every file (local or remote) must reference corresponding index.

CREATE TABLE data_index (
  id INTEGER PRIMARY KEY,

-- Switch
  switch_uuid TEXT NOT NULL REFERENCES switch_connections(uuid) ON DELETE CASCADE,

-- Type of file
  type CHARACTER VARYING(16) NOT NULL CHECK (type = 'cdr' OR type = 'pcap'),

-- Timestamp
  ts INTEGER NOT NULL CHECK (ts > 0),

-- Backup
  has_backup BOOLEAN NOT NULL DEFAULT FALSE,

  UNIQUE (switch_uuid, type, ts)
);


-- Local files

CREATE TABLE local_files (
  id INTEGER PRIMARY KEY,
  index_id INTEGER NOT NULL REFERENCES data_index(id) ON DELETE CASCADE,

-- Path to file (from switch cdr/pcap directory)
  path TEXT UNIQUE NOT NULL CHECK ((TRIM(path) = '') IS FALSE),

-- Compression flag (use NULL as FALSE)
  compressed BOOLEAN DEFAULT NULL CHECK (compressed),

-- Flag for re-indexing
  updated BOOLEAN DEFAULT NULL
);

CREATE TRIGGER local_files_on_delete AFTER DELETE ON local_files
BEGIN
DELETE FROM data_index
WHERE id = OLD.index_id
AND NOT EXISTS(SELECT 1 FROM local_files WHERE index_id = OLD.index_id)
AND NOT EXISTS(SELECT 1 FROM remote_files WHERE index_id = OLD.index_id);
END;


-- Remote files

CREATE TABLE remote_files (
  id INTEGER PRIMARY KEY,
  index_id INTEGER NOT NULL REFERENCES data_index(id) ON DELETE CASCADE,

-- Remote storage
  storage_uuid TEXT NOT NULL REFERENCES storage_connections(uuid) ON DELETE CASCADE,

-- Path to file (from storage workdir)
  path TEXT NOT NULL CHECK ((TRIM(path) = '') IS FALSE),

-- Flag for re-indexing
  updated BOOLEAN DEFAULT NULL,

  UNIQUE (index_id, storage_uuid, path)
);

CREATE TRIGGER remote_files_on_insert AFTER INSERT ON remote_files
BEGIN
UPDATE data_index SET has_backup = TRUE WHERE id = NEW.index_id;
END;

CREATE TRIGGER remote_files_on_delete AFTER DELETE ON remote_files
BEGIN
UPDATE data_index SET has_backup = FALSE WHERE id = OLD.index_id;
DELETE FROM data_index WHERE id = OLD.index_id AND NOT EXISTS(SELECT 1 FROM local_files WHERE index_id = OLD.index_id);
END;


-- Hourly CDR backup files
-- Hourly backup does not have to reference main data index,
-- nor local, nor remote files. It is completely independent.
-- Reference to data index (if exists) can be built from timestamp:
-- data_index.ts = ts - ts % 86400

CREATE TABLE backup_files (
  id INTEGER PRIMARY KEY,

-- Switch
  switch_uuid TEXT NOT NULL REFERENCES switch_connections(uuid) ON DELETE CASCADE,

-- Remote storage
  storage_uuid TEXT NOT NULL REFERENCES storage_connections(uuid) ON DELETE CASCADE,

-- Timestamp (aligned to hours)
  ts INTEGER NOT NULL CHECK (ts > 0),

-- Path to file (from remote storage workdir)
  path TEXT NOT NULL CHECK ((TRIM(path) = '') IS FALSE),

-- Flag for re-indexing
  updated BOOLEAN DEFAULT NULL,

  UNIQUE (switch_uuid, storage_uuid, path)
);


-- Search requests

CREATE TABLE search_requests (
  uuid TEXT PRIMARY KEY NOT NULL CHECK ((TRIM(uuid) = '') IS FALSE),

-- Server
  server_name TEXT DEFAULT NULL CHECK (server_name IS NULL OR (TRIM(server_name) = '') IS FALSE),

-- Type of request
  type CHARACTER VARYING(16) NOT NULL CHECK (type = 'cdr' OR type = 'pcap'),

-- Timestamps
  created INTEGER NOT NULL CHECK (created > 0),
  started INTEGER DEFAULT NULL CHECK (started > 0),
  finished INTEGER DEFAULT NULL CHECK (finished > 0),
  exp_time INTEGER DEFAULT NULL CHECK (exp_time > 0),

-- Status of request
  status CHARACTER VARYING(16) DEFAULT NULL CHECK (status = 'created' OR status = 'processing' OR status = 'complete' OR status = 'failed'),
  error TEXT DEFAULT NULL,
  is_empty BOOLEAN DEFAULT NULL,
  is_downloaded BOOLEAN DEFAULT NULL,

-- Input arguments
  arg_switch_name TEXT DEFAULT NULL CHECK (server_name IS NOT NULL OR (TRIM(arg_switch_name) = '') IS FALSE),
  arg_start_time INTEGER DEFAULT NULL CHECK (arg_start_time > 0),
  arg_end_time INTEGER DEFAULT NULL CHECK (arg_end_time > 0),
  arg_fields TEXT DEFAULT NULL CHECK (type = 'cdr' OR arg_fields IS NULL),
  arg_ani TEXT DEFAULT NULL,
  arg_dnis TEXT DEFAULT NULL,
  arg_ingress_trunk_id TEXT DEFAULT NULL CHECK (type = 'cdr' OR arg_ingress_trunk_id IS NULL),
  arg_egress_trunk_id TEXT DEFAULT NULL CHECK (type = 'cdr' OR arg_egress_trunk_id IS NULL),
  arg_ingress_call_id TEXT DEFAULT NULL CHECK (type = 'cdr' OR arg_ingress_call_id IS NULL),
  arg_egress_call_id TEXT DEFAULT NULL CHECK (type = 'cdr' OR arg_egress_call_id IS NULL),
  arg_call_id TEXT DEFAULT NULL CHECK (type = 'cdr' OR (TRIM(arg_call_id) = '') IS FALSE),

-- Output
  out_path TEXT DEFAULT NULL CHECK (out_path IS NULL OR (TRIM(out_path) = '') IS FALSE),
  out_filename TEXT DEFAULT NULL CHECK (out_filename IS NULL OR (TRIM(out_filename) = '') IS FALSE)
);
