#!/usr/bin/bash

# DNL Cloud Tool deployment script

# Target: prod (default) | debug
target="prod"
# Overwrite existing files
force=""
# Print debug messages
verbose=""
# Remove existing files
uninstall=""
# Path to the installation directory
base="/opt/denovo_v6/dnl_cloud_tool"
# Database name
dbname="class4_dnl"

conf=""

print_help() {
    echo "DNL Cloud Tool deployment script"
    echo ""
    echo "./deploy.sh [prod|debug] [/opt/denovo_v6]"
    echo "   -dbname <name> - database name"
    echo "   -f   overwrite existing configuration/service files"
    echo "   -h   help"
    echo "   -u   uninstall"
    echo "   -v   verbose logging"
    echo ""
    echo "Examples:"
    echo ""
    echo "Deploy production version to the default path"
    echo "   ./deploy.sh -vf"
    echo ""
    echo "Deploy debug version:"
    echo "   ./deploy.sh -vf debug"
    echo "   ./deploy.sh -vf debug /tmp/denovo_v6 -dbname class4_tmp"
    echo ""
    echo "Uninstall"
    echo "   ./deploy.sh -u"
    echo "   ./deploy.sh -u debug /tmp/denovo_v6 -dbname class4_tmp"
}

prod_target() {
    target="prod"
    base="/opt/denovo_v6/dnl_cloud_tool"
    dbname="class4_dnl"
}

debug_target() {
    target="debug"
    base="/opt/dbg_denovo_v6/dnl_cloud_tool"
    dbname="class4_debug"
}

argparser() {
    wait_dbname=""
    force_dbname=""
    for arg in "$@"; do
        if [ -n "$wait_dbname" ]; then
            force_dbname="$arg"
            wait_dbname=""
            continue
        fi
        case "$arg" in
            -h|-help|--help)
                print_help
                exit 0
                ;;
            -dbname)
                wait_dbname="true"
                continue
                ;;
            -*)
                for ((i = 1; i < ${#arg}; i++)); do
                    opt="${arg:i:1}"
                    case "$opt" in
                        f)
                            force="true"
                            ;;
                        v)
                            verbose="true"
                            ;;
                        u)
                            uninstall="true"
                            ;;
                        *)
                            echo "Invalid option: '-$opt'"
                            exit 1
                            ;;
                    esac
                done
                ;;
            prod)
                prod_target
                ;;
            debug)
                debug_target
                ;;
            *)
                base="$arg/dnl_cloud_tool"
                ;;
        esac
    done

    if [ -n "$wait_dbname" ]; then
        echo "Missing database name"
        exit 1
    fi

    if [ -n "$force_dbname" ]; then
        dbname="$force_dbname"
    fi
}

debug() {
    if [ -n "$verbose" ]; then
        echo -n "-- "
        echo "$@"
    fi
}

check_root() {
    if [ "$USER" != "root" ]; then
        echo "Must be run as root!"
        exit 1
    fi
}

# Update configuration parameter
# update_config "param" "value"
set_config() {
    debug "Setting '$1' = '$2'"
    sed -i -e "s|^#\? *$1[ =].*\$|$1 = $2|g" "$conf"
}

deploy_config() {
    conf="$base/conf/dnl_cloud_tool.conf"

    if [ -f "$conf" ] && [ -z "$force" ]; then
        debug "Configuration already exists"
        return
    fi
    debug "Deploying config"
    mkdir -p "$base/conf" || exit 1
    cp -f "conf/dnl_cloud_tool.conf" "$conf" || exit 1
    set_config "db_name" "$dbname"
}

deploy_binaries() {
    debug "Deploying binaries"
    mkdir -p "$base" || exit 1
    cp -f "bin/dnl_cloud_dbman" "$base/" || exit 1
    cp -f "bin/dnl_cloud_downloader" "$base/" || exit 1
    cp -f "bin/dnl_cloud_search" "$base/" || exit 1
}

update_service_paths() {
    orig="/opt/denovo_v6/dnl_cloud_tool"
    sed -i -e "s|$orig|$base|g" "$*"
}

deploy_systemd() {
    if [ -z "$force" ]; then
        return
    fi
    debug "Deploying systemd target"
    cp -f "systemd/dnl_cloud_dbman.service" "/etc/systemd/system/" || exit 1
    update_service_paths "/etc/systemd/system/dnl_cloud_dbman.service"
    cp -f "systemd/dnl_cloud_downloader.service" "/etc/systemd/system/" || exit 1
    update_service_paths "/etc/systemd/system/dnl_cloud_downloader.service"
    cp -f "systemd/dnl_cloud_search.service" "/etc/systemd/system/" || exit 1
    update_service_paths "/etc/systemd/system/dnl_cloud_search.service"
    systemctl daemon-reload
}

remove_all() {
    echo "Removing target '$target'"
    rm -f "/etc/systemd/system/dnl_cloud_dbman.service"
    rm -f "/etc/systemd/system/dnl_cloud_downloader.service"
    rm -f "/etc/systemd/system/dnl_cloud_search.service"
    rm -rf "$base"
}

#
# Main
#

check_root
argparser "$@"

if [ -n "$uninstall" ]; then
    remove_all
    exit 0
fi

# Deploy
deploy_config
deploy_binaries
deploy_systemd
